/**
 * Copyright (c) 2013, 2017 itemis AG (http://www.itemis.eu) and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.xtend.core.macro;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Singleton;
import java.io.Closeable;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Collections;
import org.apache.log4j.Logger;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.xtend.core.macro.FilteringClassLoader;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtend.lib.macro.TransformationContext;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.resource.XtextResourceSet;
import org.eclipse.xtext.util.internal.AlternateJdkLoader;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pure;

@Singleton
@SuppressWarnings("all")
public class ProcessorInstanceForJvmTypeProvider {
  @Accessors
  public static class ProcessorClassloaderAdapter extends AdapterImpl {
    private ClassLoader classLoader;
    
    public ProcessorClassloaderAdapter(final ClassLoader classLoader) {
      this.classLoader = classLoader;
    }
    
    @Override
    public boolean isAdapterForType(final Object type) {
      return Objects.equal(type, ProcessorInstanceForJvmTypeProvider.ProcessorClassloaderAdapter.class);
    }
    
    @Override
    public void unsetTarget(final Notifier oldTarget) {
      this.discard();
    }
    
    @Override
    public void setTarget(final Notifier newTarget) {
      if ((newTarget == null)) {
        this.discard();
      }
    }
    
    public ClassLoader discard() {
      ClassLoader _xifexpression = null;
      if ((this.classLoader instanceof Closeable)) {
        ClassLoader _xtrycatchfinallyexpression = null;
        try {
          ClassLoader _xblockexpression = null;
          {
            ((Closeable) this.classLoader).close();
            _xblockexpression = this.classLoader = null;
          }
          _xtrycatchfinallyexpression = _xblockexpression;
        } catch (final Throwable _t) {
          if (_t instanceof IOException) {
            final IOException e = (IOException)_t;
            ProcessorInstanceForJvmTypeProvider.logger.error(e.getMessage(), e);
          } else {
            throw Exceptions.sneakyThrow(_t);
          }
        }
        _xifexpression = _xtrycatchfinallyexpression;
      }
      return _xifexpression;
    }
    
    @Pure
    public ClassLoader getClassLoader() {
      return this.classLoader;
    }
    
    public void setClassLoader(final ClassLoader classLoader) {
      this.classLoader = classLoader;
    }
  }
  
  private static final Logger logger = Logger.getLogger(ProcessorInstanceForJvmTypeProvider.class);
  
  /**
   * @return an instance of the given JvmType
   */
  public Object getProcessorInstance(final JvmType type) {
    try {
      ClassLoader _classLoader = this.getClassLoader(type);
      Class<?> _loadClass = null;
      if (_classLoader!=null) {
        _loadClass=_classLoader.loadClass(type.getIdentifier());
      }
      final Class<?> loadClass = _loadClass;
      Constructor<?> _declaredConstructor = null;
      if (loadClass!=null) {
        _declaredConstructor=loadClass.getDeclaredConstructor();
      }
      Object _newInstance = null;
      if (_declaredConstructor!=null) {
        _newInstance=_declaredConstructor.newInstance();
      }
      return _newInstance;
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception e = (Exception)_t;
        String _identifier = type.getIdentifier();
        String _plus = ("Problem during instantiation of " + _identifier);
        String _plus_1 = (_plus + " : ");
        String _message = e.getMessage();
        String _plus_2 = (_plus_1 + _message);
        throw new IllegalStateException(_plus_2, e);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  public ClassLoader getClassLoader(final EObject ctx) {
    final ResourceSet resourceSet = ctx.eResource().getResourceSet();
    final ProcessorInstanceForJvmTypeProvider.ProcessorClassloaderAdapter adapter = IterableExtensions.<ProcessorInstanceForJvmTypeProvider.ProcessorClassloaderAdapter>head(Iterables.<ProcessorInstanceForJvmTypeProvider.ProcessorClassloaderAdapter>filter(resourceSet.eAdapters(), ProcessorInstanceForJvmTypeProvider.ProcessorClassloaderAdapter.class));
    if ((adapter != null)) {
      return adapter.getClassLoader();
    }
    boolean _matched = false;
    if (resourceSet instanceof XtextResourceSet) {
      _matched=true;
      final Object classLoaderCtx = ((XtextResourceSet)resourceSet).getClasspathURIContext();
      ClassLoader _switchResult_1 = null;
      boolean _matched_1 = false;
      if (classLoaderCtx instanceof ClassLoader) {
        _matched_1=true;
        _switchResult_1 = ((ClassLoader)classLoaderCtx);
      }
      if (!_matched_1) {
        if (classLoaderCtx instanceof Class) {
          _matched_1=true;
          _switchResult_1 = ((Class<?>)classLoaderCtx).getClassLoader();
        }
      }
      final ClassLoader jvmTypeLoader = _switchResult_1;
      ClassLoader _xifexpression = null;
      if ((jvmTypeLoader instanceof URLClassLoader)) {
        URLClassLoader _xblockexpression = null;
        {
          final ArrayList<URL> urls = CollectionLiterals.<URL>newArrayList();
          URL[] _uRLs = ((URLClassLoader)jvmTypeLoader).getURLs();
          Iterables.<URL>addAll(urls, ((Iterable<? extends URL>)Conversions.doWrapArray(_uRLs)));
          final ClassLoader bootClassloader = ((URLClassLoader)jvmTypeLoader).getParent();
          if ((bootClassloader instanceof AlternateJdkLoader)) {
            URL[] _uRLs_1 = ((AlternateJdkLoader)bootClassloader).getURLs();
            Iterables.<URL>addAll(urls, ((Iterable<? extends URL>)Conversions.doWrapArray(_uRLs_1)));
          }
          ClassLoader _classLoader = TransformationContext.class.getClassLoader();
          final FilteringClassLoader filtered = new FilteringClassLoader(_classLoader, Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList("org.eclipse.xtext.xbase.lib", "org.eclipse.xtend.lib", "org.eclipse.xtend2.lib", "com.google.common")));
          _xblockexpression = new URLClassLoader(((URL[])Conversions.unwrapArray(urls, URL.class)), filtered);
        }
        _xifexpression = _xblockexpression;
      } else {
        _xifexpression = jvmTypeLoader;
      }
      final ClassLoader processorClassLoader = _xifexpression;
      if ((processorClassLoader != null)) {
        EList<Adapter> _eAdapters = ((XtextResourceSet)resourceSet).eAdapters();
        ProcessorInstanceForJvmTypeProvider.ProcessorClassloaderAdapter _processorClassloaderAdapter = new ProcessorInstanceForJvmTypeProvider.ProcessorClassloaderAdapter(processorClassLoader);
        _eAdapters.add(_processorClassloaderAdapter);
        return processorClassLoader;
      }
    }
    ProcessorInstanceForJvmTypeProvider.logger.info("No class loader configured. Trying with this class classloader.");
    return this.getClass().getClassLoader();
  }
}
