/* ========================================================================
 * PlantUML : a free UML diagram generator
 * ========================================================================
 *
 * (C) Copyright 2009-2020, Arnaud Roques
 *
 * Project Info:  http://plantuml.com
 * 
 * If you like this project or if you find it useful, you can support us at:
 * 
 * http://plantuml.com/patreon (only 1$ per month!)
 * http://plantuml.com/paypal
 * 
 * This file is part of PlantUML.
 *
 * THE ACCOMPANYING PROGRAM IS PROVIDED UNDER THE TERMS OF THIS ECLIPSE PUBLIC
 * LICENSE ("AGREEMENT"). [Eclipse Public License - v 1.0]
 * 
 * ANY USE, REPRODUCTION OR DISTRIBUTION OF THE PROGRAM CONSTITUTES
 * RECIPIENT'S ACCEPTANCE OF THIS AGREEMENT.
 * 
 * You may obtain a copy of the License at
 * 
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 *
 * Original Author:  Arnaud Roques
 */
package net.sourceforge.plantuml.creole;

import java.awt.geom.Dimension2D;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import net.sourceforge.plantuml.Dimension2DDouble;
import net.sourceforge.plantuml.graphic.HorizontalAlignment;
import net.sourceforge.plantuml.graphic.HtmlColor;
import net.sourceforge.plantuml.graphic.StringBounder;
import net.sourceforge.plantuml.ugraphic.UChangeBackColor;
import net.sourceforge.plantuml.ugraphic.UChangeColor;
import net.sourceforge.plantuml.ugraphic.UGraphic;
import net.sourceforge.plantuml.ugraphic.ULine;
import net.sourceforge.plantuml.ugraphic.URectangle;
import net.sourceforge.plantuml.ugraphic.UTranslate;

public class AtomTable implements Atom {

	class Line {
		private final List<Atom> cells = new ArrayList<Atom>();
		private final List<HtmlColor> cellsBackColor = new ArrayList<HtmlColor>();
		private final HtmlColor lineBackColor;

		private Line(HtmlColor lineBackColor) {
			this.lineBackColor = lineBackColor;
		}

		public void add(Atom cell, HtmlColor cellBackColor) {
			cells.add(cell);
			cellsBackColor.add(cellBackColor);
		}

		public int size() {
			return cells.size();
		}

		@Override
		public String toString() {
			return super.toString() + " " + cells.size();
		}
	}

	private final List<Line> lines = new ArrayList<Line>();
	private final Map<Atom, Position> positions = new HashMap<Atom, Position>();
	private final HtmlColor lineColor;

	public AtomTable(HtmlColor lineColor) {
		this.lineColor = lineColor;
	}

	public Dimension2D calculateDimension(StringBounder stringBounder) {
		initMap(stringBounder);
		final double width = getEndingX(getNbCols() - 1);
		final double height = getEndingY(getNbLines() - 1);
		return new Dimension2DDouble(width, height);
	}

	public double getStartingAltitude(StringBounder stringBounder) {
		return 0;
	}

	public void drawU(UGraphic ug) {
		initMap(ug.getStringBounder());
		for (int i = 0; i < getNbLines(); i++) {
			final Line line = lines.get(i);
			if (line.lineBackColor != null) {
				final double y1 = getStartingY(i);
				final double y2 = getStartingY(i + 1);
				final double x1 = getStartingX(0);
				final double x2 = getStartingX(getNbCols());
				ug.apply(new UChangeColor(null)).apply(new UChangeBackColor(line.lineBackColor))
						.apply(new UTranslate(x1, y1)).draw(new URectangle(x2 - x1, y2 - y1));
			}
			for (int j = 0; j < getNbCols(); j++) {
				if (j >= line.cells.size()) {
					continue;
				}
				final Atom cell = line.cells.get(j);
				HorizontalAlignment align = HorizontalAlignment.LEFT;
				if (cell instanceof SheetBlock1) {
					align = ((SheetBlock1) cell).getCellAlignment();
				}
				final HtmlColor cellBackColor = line.cellsBackColor.get(j);
				final double x1 = getStartingX(j);
				final double x2 = getStartingX(j + 1);
				final double cellWidth = x2 - x1;
				if (cellBackColor != null) {
					final double y1 = getStartingY(i);
					final double y2 = getStartingY(i + 1);
					ug.apply(new UChangeColor(null)).apply(new UChangeBackColor(cellBackColor))
							.apply(new UTranslate(x1, y1)).draw(new URectangle(x2 - x1, y2 - y1));
				}
				final Position pos = positions.get(cell);
				final Dimension2D dimCell = cell.calculateDimension(ug.getStringBounder());
				final double dx;
				if (align == HorizontalAlignment.RIGHT) {
					dx = cellWidth - dimCell.getWidth();
				} else {
					dx = 0;
				}
				cell.drawU(ug.apply(pos.getTranslate().compose(new UTranslate(dx, 0))));
			}
		}
		ug = ug.apply(new UChangeColor(lineColor));
		final ULine hline = new ULine(getEndingX(getNbCols() - 1), 0);
		for (int i = 0; i <= getNbLines(); i++) {
			ug.apply(new UTranslate(0, getStartingY(i))).draw(hline);
		}
		final ULine vline = new ULine(0, getEndingY(getNbLines() - 1));
		for (int i = 0; i <= getNbCols(); i++) {
			ug.apply(new UTranslate(getStartingX(i), 0)).draw(vline);
		}

	}

	private void initMap(StringBounder stringBounder) {
		if (positions.size() > 0) {
			return;
		}
		for (Line line : lines) {
			for (Atom cell : line.cells) {
				final Dimension2D dim = cell.calculateDimension(stringBounder);
				final Position pos = new Position(0, 0, dim);
				positions.put(cell, pos);
			}
		}
		for (int i = 0; i < lines.size(); i++) {
			for (int j = 0; j < lines.get(i).size(); j++) {
				final Atom cell = lines.get(i).cells.get(j);
				final Dimension2D dim = cell.calculateDimension(stringBounder);
				final double x = getStartingX(j);
				final double y = getStartingY(i);
				final Position pos = new Position(x, y, dim);
				positions.put(cell, pos);
			}
		}
	}

	private double getStartingX(int col) {
		double result = 0;
		for (int i = 0; i < col; i++) {
			result += getColWidth(i);
		}
		return result;
	}

	private double getEndingX(int col) {
		double result = 0;
		for (int i = 0; i <= col; i++) {
			result += getColWidth(i);
		}
		return result;
	}

	private double getStartingY(int line) {
		double result = 0;
		for (int i = 0; i < line; i++) {
			result += getLineHeight(i);
		}
		return result;
	}

	private double getEndingY(int line) {
		double result = 0;
		for (int i = 0; i <= line; i++) {
			result += getLineHeight(i);
		}
		return result;
	}

	private double getColWidth(int col) {
		double result = 0;
		for (int i = 0; i < getNbLines(); i++) {
			final Position position = getPosition(i, col);
			if (position == null) {
				continue;
			}
			final double width = position.getWidth();
			result = Math.max(result, width);
		}
		return result;
	}

	private double getLineHeight(int line) {
		double result = 0;
		for (int i = 0; i < getNbCols(); i++) {
			final Position position = getPosition(line, i);
			if (position == null) {
				continue;
			}
			final double height = position.getHeight();
			result = Math.max(result, height);
		}
		return result;
	}

	private Position getPosition(int line, int col) {
		if (line >= lines.size()) {
			return null;
		}
		final Line l = lines.get(line);
		if (col >= l.cells.size()) {
			return null;
		}
		final Atom atom = l.cells.get(col);
		return positions.get(atom);
	}

	private int getNbCols() {
		return lines.get(0).size();
	}

	private int getNbLines() {
		return lines.size();
	}

	private Line lastLine() {
		return lines.get(lines.size() - 1);
	}

	public void addCell(Atom cell, HtmlColor cellBackColor) {
		lastLine().add(cell, cellBackColor);
		positions.clear();
	}

	public void newLine(HtmlColor lineBackColor) {
		lines.add(new Line(lineBackColor));
	}
}
