/* ========================================================================
 * PlantUML : a free UML diagram generator
 * ========================================================================
 *
 * (C) Copyright 2009-2020, Arnaud Roques
 *
 * Project Info:  http://plantuml.com
 * 
 * If you like this project or if you find it useful, you can support us at:
 * 
 * http://plantuml.com/patreon (only 1$ per month!)
 * http://plantuml.com/paypal
 * 
 * This file is part of PlantUML.
 *
 * THE ACCOMPANYING PROGRAM IS PROVIDED UNDER THE TERMS OF THIS ECLIPSE PUBLIC
 * LICENSE ("AGREEMENT"). [Eclipse Public License - v 1.0]
 * 
 * ANY USE, REPRODUCTION OR DISTRIBUTION OF THE PROGRAM CONSTITUTES
 * RECIPIENT'S ACCEPTANCE OF THIS AGREEMENT.
 * 
 * You may obtain a copy of the License at
 * 
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 *
 * Original Author:  Arnaud Roques
 */
package net.sourceforge.plantuml.preproc;

import java.io.IOException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicLong;

import net.sourceforge.plantuml.CharSequence2;
import net.sourceforge.plantuml.Log;

public abstract class ReadLineInstrumented implements ReadLine {

	private static final boolean TRACE = false;

	private static ConcurrentMap<Class, AtomicLong> durations = new ConcurrentHashMap<Class, AtomicLong>();
	private static ConcurrentMap<Class, AtomicLong> maxes = new ConcurrentHashMap<Class, AtomicLong>();

	private long current = 0;

	private AtomicLong get(ConcurrentMap<Class, AtomicLong> source) {
		AtomicLong result = source.get(getClass());
		if (result == null) {
			result = new AtomicLong();
			source.put(getClass(), result);
		}
		return result;
	}

	public final CharSequence2 readLine() throws IOException {
		if (TRACE == false) {
			return readLineInst();
		}
		final long now = System.currentTimeMillis();
		try {
			return readLineInst();
		} finally {
			final long time = System.currentTimeMillis() - now;
			current += time;
			get(durations).addAndGet(time);
		}
	}

	@Override
	public String toString() {
		return super.toString() + " current=" + current;
	}

	abstract CharSequence2 readLineInst() throws IOException;

	public final void close() throws IOException {
		if (TRACE) {
			if (current > get(maxes).get()) {
				get(maxes).set(current);
			}
			Log.info("DURATION::" + getClass() + " duration= " + get(durations).get() + " current=" + current + " max="
					+ get(maxes).get());
		}
		closeInst();
	}

	abstract void closeInst() throws IOException;

}
