/* ========================================================================
 * PlantUML : a free UML diagram generator
 * ========================================================================
 *
 * (C) Copyright 2009-2020, Arnaud Roques
 *
 * Project Info:  http://plantuml.com
 * 
 * If you like this project or if you find it useful, you can support us at:
 * 
 * http://plantuml.com/patreon (only 1$ per month!)
 * http://plantuml.com/paypal
 * 
 * This file is part of PlantUML.
 *
 * THE ACCOMPANYING PROGRAM IS PROVIDED UNDER THE TERMS OF THIS ECLIPSE PUBLIC
 * LICENSE ("AGREEMENT"). [Eclipse Public License - v 1.0]
 * 
 * ANY USE, REPRODUCTION OR DISTRIBUTION OF THE PROGRAM CONSTITUTES
 * RECIPIENT'S ACCEPTANCE OF THIS AGREEMENT.
 * 
 * You may obtain a copy of the License at
 * 
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 *
 * Original Author:  Arnaud Roques
 */
package net.sourceforge.plantuml.project3;

import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Set;

public class TaskImpl implements Task, LoadPlanable {

	private final TaskCode code;
	private final Solver3 solver;
	private final Set<Resource> resources = new LinkedHashSet<Resource>();
	private final LoadPlanable defaultPlan;

	public TaskImpl(TaskCode code, LoadPlanable defaultPlan) {
		this.code = code;
		this.defaultPlan = defaultPlan;
		this.solver = new Solver3(this);
		setStart(new InstantDay(0));
		setLoad(LoadInDays.inDay(1));
	}

	public int getLoadAt(Instant instant) {
		LoadPlanable plan1 = defaultPlan;
		if (resources.size() > 0) {
			plan1 = PlanUtils.minOf(plan1, getRessourcePlan());
		}
		return PlanUtils.minOf(getLoad(), plan1).getLoadAt(instant);
	}

	public int loadForResource(Resource res, Instant i) {
		if (resources.contains(res) && i.compareTo(getStart()) >= 0 && i.compareTo(getEnd()) <= 0) {
			if (res.getLoadAt(i) == 0) {
				return 0;
			}
			int size = 0;
			for (Resource r : resources) {
				if (r.getLoadAt(i) > 0) {
					size++;
				}
			}
			return getLoadAt(i) / size;
		}
		return 0;
	}

	private LoadPlanable getRessourcePlan() {
		if (resources.size() == 0) {
			throw new IllegalStateException();
		}
		return new LoadPlanable() {

			public int getLoadAt(Instant instant) {
				int result = 0;
				for (Resource res : resources) {
					result += res.getLoadAt(instant);
				}
				return result;
			}
		};
	}

	public String getPrettyDisplay() {
		if (resources.size() > 0) {
			final StringBuilder result = new StringBuilder(code.getSimpleDisplay());
			result.append(" ");
			for (Iterator<Resource> it = resources.iterator(); it.hasNext();) {
				result.append("{");
				result.append(it.next().getName());
				result.append("}");
				if (it.hasNext()) {
					result.append(" ");
				}
			}
			return result.toString();
		}
		return code.getSimpleDisplay();
	}

	@Override
	public String toString() {
		return code.toString();
	}

	public String debug() {
		return "" + getStart() + " ---> " + getEnd() + "   [" + getLoad() + "]";
	}

	public TaskCode getCode() {
		return code;
	}

	public Instant getStart() {
		Instant result = (Instant) solver.getData(TaskAttribute.START);
		while (getLoadAt(result) == 0) {
			result = result.increment();
		}
		return result;
	}

	public Instant getEnd() {
		return (Instant) solver.getData(TaskAttribute.END);
	}

	public Load getLoad() {
		return (Load) solver.getData(TaskAttribute.LOAD);
	}

	public void setLoad(Load load) {
		solver.setData(TaskAttribute.LOAD, load);
	}

	public void setStart(Instant start) {
		solver.setData(TaskAttribute.START, start);
	}

	public void setEnd(Instant end) {
		solver.setData(TaskAttribute.END, end);
	}

	private TaskDraw taskDraw;
	private ComplementColors colors;

	public void setTaskDraw(TaskDraw taskDraw) {
		taskDraw.setColors(colors);
		this.taskDraw = taskDraw;
	}

	public TaskDraw getTaskDraw() {
		return taskDraw;
	}

	public void setColors(ComplementColors colors) {
		this.colors = colors;
	}

	public void addResource(Resource resource) {
		this.resources.add(resource);
	}

}
