/**
 * Copyright (c) 2017, 2020 itemis AG and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Tamas Miklossy (itemis AG) - initial API and implementation
 *     Zoey Gerrit Prigge         - Generalized dependent attribute method
 *                                  to use with recordBased Node shapes (bug #454629)
 *                                - include getAllAttributesSameName/Value (bug #548911)
 */
package org.eclipse.gef.dot.internal.language;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import java.util.LinkedList;
import java.util.List;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.gef.dot.internal.DotAttributes;
import org.eclipse.gef.dot.internal.language.dot.AttrList;
import org.eclipse.gef.dot.internal.language.dot.AttrStmt;
import org.eclipse.gef.dot.internal.language.dot.Attribute;
import org.eclipse.gef.dot.internal.language.dot.AttributeType;
import org.eclipse.gef.dot.internal.language.dot.DotGraph;
import org.eclipse.gef.dot.internal.language.dot.EdgeRhs;
import org.eclipse.gef.dot.internal.language.dot.EdgeRhsNode;
import org.eclipse.gef.dot.internal.language.dot.EdgeStmtNode;
import org.eclipse.gef.dot.internal.language.dot.NodeId;
import org.eclipse.gef.dot.internal.language.dot.NodeStmt;
import org.eclipse.gef.dot.internal.language.dot.Stmt;
import org.eclipse.gef.dot.internal.language.dot.Subgraph;
import org.eclipse.gef.dot.internal.language.terminals.ID;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * This class provides helper methods for walking the DOT abstract syntax tree.
 */
@SuppressWarnings("all")
public class DotAstHelper {
  public static int getNumberOfNodes(final DotGraph dotGraph) {
    final Function1<NodeId, ID> _function = (NodeId it) -> {
      return it.getName();
    };
    return IterableExtensions.<ID>toSet(ListExtensions.<NodeId, ID>map(EcoreUtil2.<NodeId>eAllOfType(dotGraph, NodeId.class), _function)).size();
  }
  
  public static int getNumberOfEdges(final DotGraph dotGraph) {
    return EcoreUtil2.<EdgeStmtNode>eAllOfType(dotGraph, EdgeStmtNode.class).size();
  }
  
  public static NodeId getNodeId(final NodeId nodeId) {
    Object _xblockexpression = null;
    {
      final DotGraph dotGraph = EcoreUtil2.<DotGraph>getContainerOfType(nodeId, DotGraph.class);
      Iterable<NodeStmt> _allNodeStatements = DotAstHelper.getAllNodeStatements(dotGraph);
      for (final NodeStmt nodeStmt : _allNodeStatements) {
        if (((nodeStmt.getNode() != null) && Objects.equal(nodeStmt.getNode().getName(), nodeId.getName()))) {
          return nodeStmt.getNode();
        }
      }
      _xblockexpression = null;
    }
    return ((NodeId)_xblockexpression);
  }
  
  /**
   * Collects all nodeId EObjects having the same name as the baseNodeId
   */
  public static List<NodeId> getAllNodeIds(final NodeId baseNodeId) {
    LinkedList<NodeId> _xblockexpression = null;
    {
      final LinkedList<NodeId> result = CollectionLiterals.<NodeId>newLinkedList();
      final DotGraph dotGraph = EcoreUtil2.<DotGraph>getContainerOfType(baseNodeId, DotGraph.class);
      Iterable<NodeStmt> _filter = Iterables.<NodeStmt>filter(dotGraph.getStmts(), NodeStmt.class);
      for (final NodeStmt nodeStmt : _filter) {
        {
          final NodeId nodeId = nodeStmt.getNode();
          if ((((nodeId != null) && Objects.equal(nodeId.getName(), baseNodeId.getName())) && (!Objects.equal(nodeId, baseNodeId)))) {
            result.add(nodeId);
          }
        }
      }
      Iterable<EdgeStmtNode> _filter_1 = Iterables.<EdgeStmtNode>filter(dotGraph.getStmts(), EdgeStmtNode.class);
      for (final EdgeStmtNode edgeStmtNode : _filter_1) {
        {
          NodeId nodeId = edgeStmtNode.getNode();
          if ((((nodeId != null) && Objects.equal(nodeId.getName(), baseNodeId.getName())) && (nodeId != baseNodeId))) {
            result.add(nodeId);
          }
          final EdgeRhs edgeRHS = IterableExtensions.<EdgeRhs>head(edgeStmtNode.getEdgeRHS());
          if ((edgeRHS instanceof EdgeRhsNode)) {
            nodeId = ((EdgeRhsNode)edgeRHS).getNode();
            if ((((nodeId != null) && Objects.equal(nodeId.getName(), baseNodeId.getName())) && (nodeId != baseNodeId))) {
              result.add(nodeId);
            }
          }
        }
      }
      _xblockexpression = result;
    }
    return _xblockexpression;
  }
  
  public static LinkedList<Attribute> getAllAttributesSameValue(final Attribute comparator) {
    LinkedList<Attribute> _xblockexpression = null;
    {
      final LinkedList<Attribute> result = CollectionLiterals.<Attribute>newLinkedList();
      final DotGraph dotGraph = EcoreUtil2.<DotGraph>getContainerOfType(comparator, DotGraph.class);
      final Object comparatorParsed = DotAttributes.parsed(comparator);
      final Function1<Attribute, Boolean> _function = (Attribute e) -> {
        boolean _xblockexpression_1 = false;
        {
          final Object parsed = DotAttributes.parsed(e);
          boolean _xifexpression = false;
          if (((parsed instanceof EObject) && (comparatorParsed instanceof EObject))) {
            _xifexpression = EcoreUtil2.equals(((EObject) parsed), ((EObject) comparatorParsed));
          } else {
            _xifexpression = Objects.equal(parsed, comparatorParsed);
          }
          _xblockexpression_1 = _xifexpression;
        }
        return Boolean.valueOf(_xblockexpression_1);
      };
      Iterable<Attribute> _filter = IterableExtensions.<Attribute>filter(EcoreUtil2.<Attribute>getAllContentsOfType(dotGraph, Attribute.class), _function);
      for (final Attribute candidate : _filter) {
        result.add(candidate);
      }
      _xblockexpression = result;
    }
    return _xblockexpression;
  }
  
  public static LinkedList<Attribute> getAllAttributesSameName(final Attribute comparator) {
    LinkedList<Attribute> _xblockexpression = null;
    {
      final LinkedList<Attribute> result = CollectionLiterals.<Attribute>newLinkedList();
      final DotGraph dotGraph = EcoreUtil2.<DotGraph>getContainerOfType(comparator, DotGraph.class);
      ID _name = null;
      if (comparator!=null) {
        _name=comparator.getName();
      }
      String _value = null;
      if (_name!=null) {
        _value=_name.toValue();
      }
      final String comparatorName = _value;
      final Function1<Attribute, Boolean> _function = (Attribute e) -> {
        ID _name_1 = null;
        if (e!=null) {
          _name_1=e.getName();
        }
        String _value_1 = null;
        if (_name_1!=null) {
          _value_1=_name_1.toValue();
        }
        return Boolean.valueOf(Objects.equal(_value_1, comparatorName));
      };
      Iterable<Attribute> _filter = IterableExtensions.<Attribute>filter(EcoreUtil2.<Attribute>getAllContentsOfType(dotGraph, Attribute.class), _function);
      for (final Attribute candidate : _filter) {
        result.add(candidate);
      }
      _xblockexpression = result;
    }
    return _xblockexpression;
  }
  
  /**
   * Returns the color scheme attribute value that is set for the given
   * attribute.
   * 
   * @param attribute
   *            The attribute to determine the color scheme attribute value
   *            for.
   * @return The color scheme value that is set for the given attribute, or
   *         null if it cannot be determined.
   */
  public static String getColorSchemeAttributeValue(final Attribute attribute) {
    return DotAstHelper.getDependedOnAttributeValue(attribute, DotAttributes.COLORSCHEME__GCNE);
  }
  
  /**
   * Returns an attribute value specified by attributeName that is set for given attribute
   * 
   * @param dependentAttribute
   * 			The attribute to determine a depending value for.
   * @param attributeName
   * 			The name of the attribute that the dependentAttribute depends on.
   * @return The attribute value set for the attribute specified by attributeName
   */
  public static String getDependedOnAttributeValue(final Attribute dependentAttribute, final String attributeName) {
    Object _xblockexpression = null;
    {
      final EdgeStmtNode edgeStmtNode = EcoreUtil2.<EdgeStmtNode>getContainerOfType(dependentAttribute, EdgeStmtNode.class);
      if ((edgeStmtNode != null)) {
        ID dependedOnValue = DotAstHelper.getAttributeValue(edgeStmtNode.getAttrLists(), attributeName);
        if ((dependedOnValue != null)) {
          return dependedOnValue.toValue();
        }
        dependedOnValue = DotAstHelper.getGlobalDependedOnValue(edgeStmtNode, AttributeType.EDGE, attributeName);
        if ((dependedOnValue != null)) {
          return dependedOnValue.toValue();
        }
      }
      final NodeStmt nodeStmt = EcoreUtil2.<NodeStmt>getContainerOfType(dependentAttribute, NodeStmt.class);
      if ((nodeStmt != null)) {
        ID dependedOnValue_1 = DotAstHelper.getAttributeValue(nodeStmt.getAttrLists(), attributeName);
        if ((dependedOnValue_1 != null)) {
          return dependedOnValue_1.toValue();
        }
        dependedOnValue_1 = DotAstHelper.getGlobalDependedOnValue(nodeStmt, AttributeType.NODE, attributeName);
        if ((dependedOnValue_1 != null)) {
          return dependedOnValue_1.toValue();
        }
      }
      final AttrStmt attrStmt = EcoreUtil2.<AttrStmt>getContainerOfType(dependentAttribute, AttrStmt.class);
      if ((attrStmt != null)) {
        final ID dependedOnValue_2 = DotAstHelper.getAttributeValue(attrStmt.getAttrLists(), attributeName);
        if ((dependedOnValue_2 != null)) {
          return dependedOnValue_2.toValue();
        }
      }
      final Subgraph subgraph = EcoreUtil2.<Subgraph>getContainerOfType(dependentAttribute, Subgraph.class);
      if ((subgraph != null)) {
        final ID dependedOnValue_3 = DotAstHelper.getAttributeValue(subgraph, attributeName);
        if ((dependedOnValue_3 != null)) {
          return dependedOnValue_3.toValue();
        }
      }
      final DotGraph dotGraph = EcoreUtil2.<DotGraph>getContainerOfType(dependentAttribute, DotGraph.class);
      if ((dotGraph != null)) {
        ID dependedOnValue_4 = DotAstHelper.getAttributeValueAll(dotGraph, attributeName);
        if ((dependedOnValue_4 != null)) {
          return dependedOnValue_4.toValue();
        }
        dependedOnValue_4 = DotAstHelper.getGlobalDependedOnValue(dotGraph, AttributeType.GRAPH, attributeName);
        if ((dependedOnValue_4 != null)) {
          return dependedOnValue_4.toValue();
        }
      }
      _xblockexpression = null;
    }
    return ((String)_xblockexpression);
  }
  
  private static ID getGlobalDependedOnValue(final EObject eObject, final AttributeType attributeType, final String attributeName) {
    Object _xblockexpression = null;
    {
      final Subgraph subgraph = EcoreUtil2.<Subgraph>getContainerOfType(eObject, Subgraph.class);
      if ((subgraph != null)) {
        final ID value = DotAstHelper.getAttributeValue(subgraph.getStmts(), attributeType, attributeName);
        if ((value != null)) {
          return value;
        }
      }
      final DotGraph dotGraph = EcoreUtil2.<DotGraph>getContainerOfType(eObject, DotGraph.class);
      if ((dotGraph != null)) {
        final ID value_1 = DotAstHelper.getAttributeValue(dotGraph.getStmts(), attributeType, attributeName);
        if ((value_1 != null)) {
          return value_1;
        }
      }
      _xblockexpression = null;
    }
    return ((ID)_xblockexpression);
  }
  
  private static ID getAttributeValue(final EList<Stmt> stmts, final AttributeType attributeType, final String attributeName) {
    Object _xblockexpression = null;
    {
      for (final Stmt stmt : stmts) {
        if ((stmt instanceof AttrStmt)) {
          AttributeType _type = ((AttrStmt)stmt).getType();
          boolean _equals = Objects.equal(_type, attributeType);
          if (_equals) {
            return DotAstHelper.getAttributeValue(((AttrStmt)stmt).getAttrLists(), attributeName);
          }
        }
      }
      _xblockexpression = null;
    }
    return ((ID)_xblockexpression);
  }
  
  public static ID getAttributeValue(final DotGraph graph, final String name) {
    Object _xblockexpression = null;
    {
      EList<Stmt> _stmts = graph.getStmts();
      for (final Stmt stmt : _stmts) {
        {
          ID _switchResult = null;
          boolean _matched = false;
          if (stmt instanceof Attribute) {
            _matched=true;
            _switchResult = DotAstHelper.getAttributeValue(((Attribute)stmt), name);
          }
          final ID value = _switchResult;
          if ((value != null)) {
            return value;
          }
        }
      }
      _xblockexpression = null;
    }
    return ((ID)_xblockexpression);
  }
  
  public static ID getAttributeValueAll(final DotGraph graph, final String name) {
    Object _xblockexpression = null;
    {
      EList<Stmt> _stmts = graph.getStmts();
      for (final Stmt stmt : _stmts) {
        {
          ID _switchResult = null;
          boolean _matched = false;
          if (stmt instanceof AttrStmt) {
            _matched=true;
            _switchResult = DotAstHelper.getAttributeValue(((AttrStmt)stmt).getAttrLists(), name);
          }
          if (!_matched) {
            if (stmt instanceof Attribute) {
              _matched=true;
              _switchResult = DotAstHelper.getAttributeValue(((Attribute)stmt), name);
            }
          }
          final ID value = _switchResult;
          if ((value != null)) {
            return value;
          }
        }
      }
      _xblockexpression = null;
    }
    return ((ID)_xblockexpression);
  }
  
  public static ID getAttributeValue(final Subgraph subgraph, final String name) {
    Object _xblockexpression = null;
    {
      EList<Stmt> _stmts = subgraph.getStmts();
      for (final Stmt stmt : _stmts) {
        {
          ID _switchResult = null;
          boolean _matched = false;
          if (stmt instanceof Attribute) {
            _matched=true;
            _switchResult = DotAstHelper.getAttributeValue(((Attribute)stmt), name);
          }
          final ID value = _switchResult;
          if ((value != null)) {
            return value;
          }
        }
      }
      _xblockexpression = null;
    }
    return ((ID)_xblockexpression);
  }
  
  /**
   * Returns the value of the first attribute with the give name or
   * <code>null</code> if no attribute could be found.
   * 
   * @param attrLists
   *            The {@link AttrList}s to search.
   * @param name
   *            The name of the attribute whose value is to be retrieved.
   * @return The attribute value or <code>null</code> in case the attribute
   *         could not be found.
   */
  public static ID getAttributeValue(final List<AttrList> attrLists, final String name) {
    Object _xblockexpression = null;
    {
      for (final AttrList attrList : attrLists) {
        {
          final ID value = DotAstHelper.getAttributeValue(attrList, name);
          if ((value != null)) {
            return value;
          }
        }
      }
      _xblockexpression = null;
    }
    return ((ID)_xblockexpression);
  }
  
  private static ID getAttributeValue(final AttrList attrList, final String name) {
    final Function1<Attribute, Boolean> _function = (Attribute it) -> {
      String _value = it.getName().toValue();
      return Boolean.valueOf(Objects.equal(_value, name));
    };
    Attribute _findFirst = IterableExtensions.<Attribute>findFirst(attrList.getAttributes(), _function);
    ID _value = null;
    if (_findFirst!=null) {
      _value=_findFirst.getValue();
    }
    return _value;
  }
  
  private static ID getAttributeValue(final Attribute attribute, final String name) {
    Object _xblockexpression = null;
    {
      boolean _equals = attribute.getName().toValue().equals(name);
      if (_equals) {
        return attribute.getValue();
      }
      _xblockexpression = null;
    }
    return ((ID)_xblockexpression);
  }
  
  /**
   * Collects all node statements residing in the dot graph or in its subgraphs.
   */
  private static Iterable<NodeStmt> getAllNodeStatements(final DotGraph dotGraph) {
    Iterable<NodeStmt> _xblockexpression = null;
    {
      final Iterable<NodeStmt> nodeStamentsInDotGraph = Iterables.<NodeStmt>filter(dotGraph.getStmts(), NodeStmt.class);
      final LinkedList<NodeStmt> nodeStatementsInSubgraphs = CollectionLiterals.<NodeStmt>newLinkedList();
      final Consumer<Subgraph> _function = (Subgraph it) -> {
        List<NodeStmt> _allNodeStatementsInSubgraph = DotAstHelper.getAllNodeStatementsInSubgraph(it);
        Iterables.<NodeStmt>addAll(nodeStatementsInSubgraphs, _allNodeStatementsInSubgraph);
      };
      Iterables.<Subgraph>filter(dotGraph.getStmts(), Subgraph.class).forEach(_function);
      _xblockexpression = Iterables.<NodeStmt>concat(nodeStamentsInDotGraph, nodeStatementsInSubgraphs);
    }
    return _xblockexpression;
  }
  
  private static List<NodeStmt> getAllNodeStatementsInSubgraph(final Subgraph subgraph) {
    List<NodeStmt> _xblockexpression = null;
    {
      final List<NodeStmt> nodeStatementInSubgraph = IterableExtensions.<NodeStmt>toList(Iterables.<NodeStmt>filter(subgraph.getStmts(), NodeStmt.class));
      final Consumer<Subgraph> _function = (Subgraph it) -> {
        List<NodeStmt> _allNodeStatementsInSubgraph = DotAstHelper.getAllNodeStatementsInSubgraph(it);
        Iterables.<NodeStmt>addAll(nodeStatementInSubgraph, _allNodeStatementsInSubgraph);
      };
      Iterables.<Subgraph>filter(subgraph.getStmts(), Subgraph.class).forEach(_function);
      _xblockexpression = nodeStatementInSubgraph;
    }
    return _xblockexpression;
  }
}
