"use strict";
/* --------------------------------------------------------------------------------------------
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for license information.
 * ------------------------------------------------------------------------------------------ */
Object.defineProperty(exports, "__esModule", { value: true });
const assert = require("assert");
const zlib = require("zlib");
const messages_1 = require("../../common/messages");
const encoding_1 = require("../../common/encoding");
const main_1 = require("../../node/main");
const stream_1 = require("stream");
const util_1 = require("util");
const buffer_1 = require("buffer");
function assertDefined(value) {
    assert.ok(value !== undefined && value !== null);
}
const TestWritable = function () {
    function TestWritable() {
        stream_1.Writable.call(this);
    }
    util_1.inherits(TestWritable, stream_1.Writable);
    TestWritable.prototype._write = function (chunk, encoding, done) {
        const toAdd = (typeof chunk === 'string')
            ? buffer_1.Buffer.from(chunk, encoding)
            : chunk;
        if (this.data === undefined) {
            this.data = toAdd;
        }
        else {
            this.data = buffer_1.Buffer.concat([this.data, toAdd]);
        }
        done();
    };
    return TestWritable;
}();
const gzipEncoder = {
    name: 'gzip',
    encode: async (input) => {
        return new Promise((resolve, reject) => {
            zlib.gzip(input, (error, buffer) => {
                if (error) {
                    reject(error);
                }
                else {
                    resolve(buffer);
                }
            });
        });
    }
};
const gzipDecoder = {
    name: 'gzip',
    decode: async (buffer) => {
        return new Promise((resolve, reject) => {
            zlib.gunzip(buffer, (error, value) => {
                if (error) {
                    reject(error);
                }
                else {
                    resolve(value);
                }
            });
        });
    }
};
suite('Messages', () => {
    test('Writing', async () => {
        const writable = new TestWritable();
        const writer = new main_1.StreamMessageWriter(writable, 'ascii');
        const request = {
            jsonrpc: '2.0',
            id: 1,
            method: 'example'
        };
        await writer.write(request);
        writable.end();
        assertDefined(writable.data);
        assert.ok(writable.data.equals(buffer_1.Buffer.from('Content-Length: 43\r\n\r\n{"jsonrpc":"2.0","id":1,"method":"example"}', 'ascii')));
    });
    test('Reading', (done) => {
        const readable = new stream_1.Readable();
        new main_1.StreamMessageReader(readable).listen((msg) => {
            const message = msg;
            assert.equal(message.id, 1);
            assert.equal(message.method, 'example');
            done();
        });
        readable.push('Content-Length: 43\r\n\r\n{"jsonrpc":"2.0","id":1,"method":"example"}');
        readable.push(null);
    });
    test('Read partial', (done) => {
        const readable = new stream_1.Readable();
        const reader = new main_1.StreamMessageReader(readable);
        reader.partialMessageTimeout = 100;
        const partOne = 'Content-Length: 43\r\n\r\n';
        const partTwo = '{"jsonrpc":"2.0","id":1,"method":"example"}';
        reader.listen((msg) => {
            const message = msg;
            assert.equal(message.id, 1);
            assert.equal(message.method, 'example');
            setTimeout(() => {
                done();
            }, 200);
        });
        reader.onPartialMessage((_info) => {
            setTimeout(() => {
                readable.push(partTwo);
                readable.push(null);
            }, 20);
        });
        readable.push(partOne);
    });
    test('Basic Zip / Unzip', async () => {
        const msg = { jsonrpc: '2.0', id: 1, method: 'example' };
        const zipped = await gzipEncoder.encode(buffer_1.Buffer.from(JSON.stringify(msg), 'utf8'));
        assert.strictEqual(buffer_1.Buffer.from(zipped).toString('base64'), 'H4sIAAAAAAAAA6tWyirOzysqSFayUjLSM1DSUcpMUbIy1FHKTS3JyAcylVIrEnMLclKVagH7JiWtKwAAAA==');
        const unzipped = JSON.parse(buffer_1.Buffer.from(await gzipDecoder.decode(zipped)).toString('utf-8'));
        assert.strictEqual(unzipped.id, 1);
        assert.strictEqual(unzipped.method, 'example');
    });
    test('Encode', (done) => {
        const writable = new TestWritable();
        const writer = new main_1.StreamMessageWriter(writable, {
            charset: 'utf-8',
            contentEncoder: gzipEncoder,
        });
        const request = {
            jsonrpc: '2.0',
            id: 1,
            method: 'example'
        };
        writer.write(request).then(() => {
            writable.end();
            assertDefined(writable.data);
            const readable = new stream_1.Readable();
            const reader = new main_1.StreamMessageReader(readable, {
                charset: 'utf-8',
                contentDecoder: gzipDecoder
            });
            reader.listen((message) => {
                if (!messages_1.isRequestMessage(message)) {
                    throw new Error(`No request message`);
                }
                assert.equal(message.id, 1);
                assert.equal(message.method, 'example');
                done();
            });
            readable.push(writable.data);
            readable.push(null);
        });
    });
    test('Decode', (done) => {
        const readable = new stream_1.Readable();
        const reader = new main_1.StreamMessageReader(readable, {
            charset: 'utf-8',
            contentDecoder: gzipDecoder
        });
        reader.listen((message) => {
            if (!messages_1.isRequestMessage(message)) {
                throw new Error(`No request message`);
            }
            assert.equal(message.id, 1);
            assert.equal(message.method, 'example');
            done();
        });
        const payload = buffer_1.Buffer.concat([
            buffer_1.Buffer.from('Content-Encoding: gzip\r\nContent-Length: 61\r\n\r\n', 'ascii'),
            zlib.gzipSync(buffer_1.Buffer.from('{"jsonrpc":"2.0","id":1,"method":"example"}', 'utf8'))
        ]);
        readable.push(payload);
        readable.push(null);
    });
    test('Generate Accept Encoding', () => {
        assert.deepStrictEqual(encoding_1.Encodings.getEncodingHeaderValue([{ name: 'gzip' }]), 'gzip');
        assert.deepStrictEqual(encoding_1.Encodings.getEncodingHeaderValue([{ name: 'gzip' }, { name: 'compress' }]), 'gzip;q=1, compress;q=0');
        assert.deepStrictEqual(encoding_1.Encodings.getEncodingHeaderValue([{ name: 'gzip' }, { name: 'compress' }, { name: 'deflate' }]), 'gzip;q=1, compress;q=0.5, deflate;q=0');
        assert.deepStrictEqual(encoding_1.Encodings.getEncodingHeaderValue([{ name: 'gzip' }, { name: 'compress' }, { name: 'deflate' }, { name: 'br' }]), 'gzip;q=1, compress;q=0.7, deflate;q=0.4, br;q=0.1');
    });
});
//# sourceMappingURL=messages.test.js.map