/**
 * Copyright (c) 2010-2012, Mark Czotter, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Mark Czotter - initial API and implementation
 */
package org.eclipse.viatra.query.patternlanguage.emf.jvmmodel;

import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import com.google.inject.Inject;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.viatra.query.patternlanguage.emf.helper.PatternLanguageHelper;
import org.eclipse.viatra.query.patternlanguage.emf.jvmmodel.BodyCodeGenerator;
import org.eclipse.viatra.query.patternlanguage.emf.jvmmodel.EMFPatternLanguageJvmModelInferrerUtil;
import org.eclipse.viatra.query.patternlanguage.emf.jvmmodel.JavadocInferrer;
import org.eclipse.viatra.query.patternlanguage.emf.types.ITypeInferrer;
import org.eclipse.viatra.query.patternlanguage.emf.types.ITypeSystem;
import org.eclipse.viatra.query.patternlanguage.emf.util.EMFJvmTypesBuilder;
import org.eclipse.viatra.query.patternlanguage.emf.util.EMFPatternLanguageGeneratorConfig;
import org.eclipse.viatra.query.patternlanguage.emf.util.IErrorFeedback;
import org.eclipse.viatra.query.patternlanguage.emf.validation.IssueCodes;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Annotation;
import org.eclipse.viatra.query.patternlanguage.emf.vql.CallableRelation;
import org.eclipse.viatra.query.patternlanguage.emf.vql.ExecutionType;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Modifiers;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Parameter;
import org.eclipse.viatra.query.patternlanguage.emf.vql.ParameterDirection;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Pattern;
import org.eclipse.viatra.query.patternlanguage.emf.vql.PatternBody;
import org.eclipse.viatra.query.patternlanguage.emf.vql.PatternCall;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Type;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Variable;
import org.eclipse.viatra.query.runtime.api.IQuerySpecification;
import org.eclipse.viatra.query.runtime.api.ViatraQueryEngine;
import org.eclipse.viatra.query.runtime.api.impl.BaseGeneratedEMFPQuery;
import org.eclipse.viatra.query.runtime.api.impl.BaseGeneratedEMFQuerySpecification;
import org.eclipse.viatra.query.runtime.api.impl.BaseGeneratedEMFQuerySpecificationWithGenericMatcher;
import org.eclipse.viatra.query.runtime.matchers.backend.QueryEvaluationHint;
import org.eclipse.viatra.query.runtime.matchers.context.IInputKey;
import org.eclipse.viatra.query.runtime.matchers.psystem.PBody;
import org.eclipse.viatra.query.runtime.matchers.psystem.annotations.PAnnotation;
import org.eclipse.viatra.query.runtime.matchers.psystem.annotations.ParameterReference;
import org.eclipse.viatra.query.runtime.matchers.psystem.queries.PParameter;
import org.eclipse.viatra.query.runtime.matchers.psystem.queries.PParameterDirection;
import org.eclipse.viatra.query.runtime.matchers.psystem.queries.PProblem;
import org.eclipse.viatra.query.runtime.matchers.psystem.queries.PVisibility;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtend2.lib.StringConcatenationClient;
import org.eclipse.xtext.common.types.JvmAnnotationReference;
import org.eclipse.xtext.common.types.JvmConstructor;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmField;
import org.eclipse.xtext.common.types.JvmFormalParameter;
import org.eclipse.xtext.common.types.JvmGenericType;
import org.eclipse.xtext.common.types.JvmMember;
import org.eclipse.xtext.common.types.JvmOperation;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.common.types.JvmUnknownTypeReference;
import org.eclipse.xtext.common.types.JvmVisibility;
import org.eclipse.xtext.diagnostics.Severity;
import org.eclipse.xtext.serializer.impl.Serializer;
import org.eclipse.xtext.xbase.XExpression;
import org.eclipse.xtext.xbase.jvmmodel.JvmAnnotationReferenceBuilder;
import org.eclipse.xtext.xbase.jvmmodel.JvmTypeReferenceBuilder;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * {@link IQuerySpecification} implementation inferrer.
 * 
 * @author Mark Czotter
 * @noreference
 */
@SuppressWarnings("all")
public class PatternQuerySpecificationClassInferrer {
  @Inject
  @Extension
  private EMFJvmTypesBuilder _eMFJvmTypesBuilder;
  
  @Inject
  @Extension
  private EMFPatternLanguageJvmModelInferrerUtil util;
  
  @Inject
  @Extension
  private JavadocInferrer _javadocInferrer;
  
  @Inject
  @Extension
  private ITypeInferrer typeInferrer;
  
  @Inject
  private ITypeSystem typeSystem;
  
  @Inject
  private IErrorFeedback feedback;
  
  @Inject
  private Serializer serializer;
  
  @Extension
  private JvmTypeReferenceBuilder builder;
  
  @Extension
  private JvmAnnotationReferenceBuilder annBuilder;
  
  /**
   * Infers the {@link IQuerySpecification} implementation class from {@link Pattern}.
   */
  public JvmDeclaredType inferQuerySpecificationClass(final Pattern pattern, final boolean isPrelinkingPhase, final String querySpecificationPackageName, final JvmType matcherClass, final JvmTypeReferenceBuilder builder, final JvmAnnotationReferenceBuilder annBuilder, final EMFPatternLanguageGeneratorConfig config) {
    this.builder = builder;
    this.annBuilder = annBuilder;
    final Procedure1<JvmGenericType> _function = (JvmGenericType it) -> {
      it.setPackageName(querySpecificationPackageName);
      this._eMFJvmTypesBuilder.setDocumentation(it, this._javadocInferrer.javadocQuerySpecificationClass(pattern).toString());
      it.setFinal(true);
      if ((this.util.isPublic(pattern) && (config.getMatcherGenerationStrategy() != EMFPatternLanguageGeneratorConfig.MatcherGenerationStrategy.USE_GENERIC))) {
        EList<JvmTypeReference> _superTypes = it.getSuperTypes();
        JvmTypeReference _typeRef = this.builder.typeRef(BaseGeneratedEMFQuerySpecification.class, this.builder.typeRef(matcherClass));
        this._eMFJvmTypesBuilder.<JvmTypeReference>operator_add(_superTypes, _typeRef);
      } else {
        EList<JvmTypeReference> _superTypes_1 = it.getSuperTypes();
        JvmTypeReference _typeRef_1 = this.builder.typeRef(BaseGeneratedEMFQuerySpecificationWithGenericMatcher.class);
        this._eMFJvmTypesBuilder.<JvmTypeReference>operator_add(_superTypes_1, _typeRef_1);
      }
      this._eMFJvmTypesBuilder.setFileHeader(it, this.util.getFileComment(pattern));
    };
    final JvmGenericType querySpecificationClass = this._eMFJvmTypesBuilder.toClass(pattern, 
      this.util.querySpecificationClassName(pattern, config.getMatcherGenerationStrategy()), _function);
    return querySpecificationClass;
  }
  
  public void initializeSpecification(final JvmDeclaredType querySpecificationClass, final Pattern pattern, final JvmType matcherClass, final JvmType matchClass, final EMFPatternLanguageGeneratorConfig config) {
    try {
      final boolean withPatternSpecificMatcher = (this.util.isPublic(pattern) && 
        (config.getMatcherGenerationStrategy() != EMFPatternLanguageGeneratorConfig.MatcherGenerationStrategy.USE_GENERIC));
      this.inferQuerySpecificationMethods(querySpecificationClass, pattern, matcherClass, matchClass, withPatternSpecificMatcher);
      this.inferQuerySpecificationInnerClasses(querySpecificationClass, pattern, withPatternSpecificMatcher);
      this.inferExpressions(querySpecificationClass, pattern);
    } catch (final Throwable _t) {
      if (_t instanceof IllegalStateException) {
        final IllegalStateException ex = (IllegalStateException)_t;
        this.feedback.reportError(pattern, ex.getMessage(), IssueCodes.OTHER_ISSUE, Severity.ERROR, 
          IErrorFeedback.JVMINFERENCE_ERROR_TYPE);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  /**
   * Infers methods for QuerySpecification class based on the input 'pattern'.
   */
  public boolean inferQuerySpecificationMethods(final JvmDeclaredType querySpecificationClass, final Pattern pattern, final JvmType matcherClass, final JvmType matchClass, final boolean withPatternSpecificMatcher) {
    boolean _xblockexpression = false;
    {
      EList<JvmMember> _members = querySpecificationClass.getMembers();
      final Procedure1<JvmConstructor> _function = (JvmConstructor it) -> {
        it.setVisibility(JvmVisibility.PRIVATE);
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("super(");
            String _querySpecificationPQueryClassName = PatternQuerySpecificationClassInferrer.this.util.querySpecificationPQueryClassName(pattern);
            _builder.append(_querySpecificationPQueryClassName);
            _builder.append(".INSTANCE);");
            _builder.newLineIfNotEmpty();
          }
        };
        this._eMFJvmTypesBuilder.setBody(it, _client);
      };
      JvmConstructor _constructor = this._eMFJvmTypesBuilder.toConstructor(pattern, _function);
      this._eMFJvmTypesBuilder.<JvmConstructor>operator_add(_members, _constructor);
      EList<JvmMember> _members_1 = querySpecificationClass.getMembers();
      final Procedure1<JvmOperation> _function_1 = (JvmOperation it) -> {
        it.setVisibility(JvmVisibility.PUBLIC);
        it.setStatic(true);
        this._eMFJvmTypesBuilder.setDocumentation(it, this._javadocInferrer.javadocQuerySpecificationInstanceMethod(pattern).toString());
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("try{");
            _builder.newLine();
            _builder.append("    ");
            _builder.append("return ");
            String _querySpecificationHolderClassName = PatternQuerySpecificationClassInferrer.this.util.querySpecificationHolderClassName(pattern);
            _builder.append(_querySpecificationHolderClassName, "    ");
            _builder.append(".INSTANCE;");
            _builder.newLineIfNotEmpty();
            _builder.append("} catch (");
            _builder.append(ExceptionInInitializerError.class);
            _builder.append(" err) {");
            _builder.newLineIfNotEmpty();
            _builder.append("    ");
            _builder.append("throw processInitializerError(err);");
            _builder.newLine();
            _builder.append("}");
            _builder.newLine();
          }
        };
        this._eMFJvmTypesBuilder.setBody(it, _client);
      };
      JvmOperation _method = this._eMFJvmTypesBuilder.toMethod(pattern, "instance", this.builder.typeRef(querySpecificationClass), _function_1);
      this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_1, _method);
      boolean _xifexpression = false;
      if (withPatternSpecificMatcher) {
        boolean _xblockexpression_1 = false;
        {
          EList<JvmMember> _members_2 = querySpecificationClass.getMembers();
          final Procedure1<JvmOperation> _function_2 = (JvmOperation it) -> {
            it.setVisibility(JvmVisibility.PROTECTED);
            EList<JvmAnnotationReference> _annotations = it.getAnnotations();
            JvmAnnotationReference _annotationRef = this.annBuilder.annotationRef(Override.class);
            this._eMFJvmTypesBuilder.<JvmAnnotationReference>operator_add(_annotations, _annotationRef);
            EList<JvmFormalParameter> _parameters = it.getParameters();
            JvmFormalParameter _parameter = this._eMFJvmTypesBuilder.toParameter(pattern, "engine", this.builder.typeRef(ViatraQueryEngine.class));
            this._eMFJvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _parameter);
            StringConcatenationClient _client = new StringConcatenationClient() {
              @Override
              protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                _builder.append("return ");
                _builder.append(matcherClass);
                _builder.append(".on(engine);");
              }
            };
            this._eMFJvmTypesBuilder.setBody(it, _client);
          };
          JvmOperation _method_1 = this._eMFJvmTypesBuilder.toMethod(pattern, "instantiate", this.builder.typeRef(matcherClass), _function_2);
          this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_2, _method_1);
          EList<JvmMember> _members_3 = querySpecificationClass.getMembers();
          final Procedure1<JvmOperation> _function_3 = (JvmOperation it) -> {
            it.setVisibility(JvmVisibility.PUBLIC);
            EList<JvmAnnotationReference> _annotations = it.getAnnotations();
            JvmAnnotationReference _annotationRef = this.annBuilder.annotationRef(Override.class);
            this._eMFJvmTypesBuilder.<JvmAnnotationReference>operator_add(_annotations, _annotationRef);
            StringConcatenationClient _client = new StringConcatenationClient() {
              @Override
              protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                _builder.append("return ");
                _builder.append(matcherClass);
                _builder.append(".create();");
              }
            };
            this._eMFJvmTypesBuilder.setBody(it, _client);
          };
          JvmOperation _method_2 = this._eMFJvmTypesBuilder.toMethod(pattern, "instantiate", this.builder.typeRef(matcherClass), _function_3);
          this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_3, _method_2);
          EList<JvmMember> _members_4 = querySpecificationClass.getMembers();
          final Procedure1<JvmOperation> _function_4 = (JvmOperation it) -> {
            it.setVisibility(JvmVisibility.PUBLIC);
            EList<JvmAnnotationReference> _annotations = it.getAnnotations();
            JvmAnnotationReference _annotationRef = this.annBuilder.annotationRef(Override.class);
            this._eMFJvmTypesBuilder.<JvmAnnotationReference>operator_add(_annotations, _annotationRef);
            StringConcatenationClient _client = new StringConcatenationClient() {
              @Override
              protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                _builder.append("return ");
                _builder.append(matchClass);
                _builder.append(".newEmptyMatch();");
              }
            };
            this._eMFJvmTypesBuilder.setBody(it, _client);
          };
          JvmOperation _method_3 = this._eMFJvmTypesBuilder.toMethod(pattern, "newEmptyMatch", this.builder.typeRef(matchClass), _function_4);
          this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_4, _method_3);
          EList<JvmMember> _members_5 = querySpecificationClass.getMembers();
          final Procedure1<JvmOperation> _function_5 = (JvmOperation it) -> {
            it.setVisibility(JvmVisibility.PUBLIC);
            EList<JvmAnnotationReference> _annotations = it.getAnnotations();
            JvmAnnotationReference _annotationRef = this.annBuilder.annotationRef(Override.class);
            this._eMFJvmTypesBuilder.<JvmAnnotationReference>operator_add(_annotations, _annotationRef);
            EList<JvmFormalParameter> _parameters = it.getParameters();
            JvmFormalParameter _parameter = this._eMFJvmTypesBuilder.toParameter(pattern, "parameters", this._eMFJvmTypesBuilder.addArrayTypeDimension(this.builder.typeRef(Object.class)));
            this._eMFJvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, _parameter);
            it.setVarArgs(true);
            StringConcatenationClient _client = new StringConcatenationClient() {
              @Override
              protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                _builder.append("return ");
                _builder.append(matchClass);
                _builder.append(".newMatch(");
                {
                  EList<Variable> _parameters = pattern.getParameters();
                  boolean _hasElements = false;
                  for(final Variable p : _parameters) {
                    if (!_hasElements) {
                      _hasElements = true;
                    } else {
                      _builder.appendImmediate(", ", "");
                    }
                    _builder.append("(");
                    String _qualifiedName = PatternQuerySpecificationClassInferrer.this.util.calculateType(p).getQualifiedName();
                    _builder.append(_qualifiedName);
                    _builder.append(") parameters[");
                    int _indexOf = pattern.getParameters().indexOf(p);
                    _builder.append(_indexOf);
                    _builder.append("]");
                  }
                }
                _builder.append(");");
              }
            };
            this._eMFJvmTypesBuilder.setBody(it, _client);
          };
          JvmOperation _method_4 = this._eMFJvmTypesBuilder.toMethod(pattern, "newMatch", this.builder.typeRef(matchClass), _function_5);
          _xblockexpression_1 = this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_5, _method_4);
        }
        _xifexpression = _xblockexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  public ParameterDirection direction(final Variable variable) {
    ParameterDirection _xblockexpression = null;
    {
      if ((variable instanceof Parameter)) {
        ((Parameter)variable).getDirection();
      }
      _xblockexpression = ParameterDirection.INOUT;
    }
    return _xblockexpression;
  }
  
  public StringConcatenationClient directionLiteral(final Variable variable) {
    StringConcatenationClient _client = new StringConcatenationClient() {
      @Override
      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
        _builder.append(PParameterDirection.class);
        _builder.append(".");
        String _name = PatternQuerySpecificationClassInferrer.this.direction(variable).name();
        _builder.append(_name);
      }
    };
    return _client;
  }
  
  public boolean inferPQueryMembers(final JvmDeclaredType pQueryClass, final Pattern pattern) {
    boolean _xblockexpression = false;
    {
      EList<JvmMember> _members = pQueryClass.getMembers();
      final Procedure1<JvmField> _function = (JvmField it) -> {
        it.setFinal(true);
        it.setStatic(true);
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("new ");
            String _querySpecificationPQueryClassName = PatternQuerySpecificationClassInferrer.this.util.querySpecificationPQueryClassName(pattern);
            _builder.append(_querySpecificationPQueryClassName);
            _builder.append("()");
          }
        };
        this._eMFJvmTypesBuilder.setInitializer(it, _client);
      };
      JvmField _field = this._eMFJvmTypesBuilder.toField(pattern, "INSTANCE", 
        this.builder.typeRef(pQueryClass), _function);
      this._eMFJvmTypesBuilder.<JvmField>operator_add(_members, _field);
      EList<Variable> _parameters = pattern.getParameters();
      for (final Variable parameter : _parameters) {
        EList<JvmMember> _members_1 = pQueryClass.getMembers();
        final Procedure1<JvmField> _function_1 = (JvmField it) -> {
          it.setFinal(true);
          it.setVisibility(JvmVisibility.PRIVATE);
          this._eMFJvmTypesBuilder.setInitializer(it, this.parameterInstantiation(parameter));
        };
        JvmField _field_1 = this._eMFJvmTypesBuilder.toField(pattern, this.util.getPParameterName(parameter.getName()), this.builder.typeRef(PParameter.class), _function_1);
        this._eMFJvmTypesBuilder.<JvmField>operator_add(_members_1, _field_1);
      }
      EList<JvmMember> _members_2 = pQueryClass.getMembers();
      final Procedure1<JvmField> _function_2 = (JvmField it) -> {
        it.setFinal(true);
        it.setVisibility(JvmVisibility.PRIVATE);
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append(Arrays.class);
            _builder.append(".asList(");
            {
              EList<Variable> _parameters = pattern.getParameters();
              boolean _hasElements = false;
              for(final Variable param : _parameters) {
                if (!_hasElements) {
                  _hasElements = true;
                } else {
                  _builder.appendImmediate(", ", "");
                }
                String _pParameterName = PatternQuerySpecificationClassInferrer.this.util.getPParameterName(param.getName());
                _builder.append(_pParameterName);
              }
            }
            _builder.append(")");
          }
        };
        this._eMFJvmTypesBuilder.setInitializer(it, _client);
      };
      JvmField _field_2 = this._eMFJvmTypesBuilder.toField(pattern, "parameters", this.builder.typeRef(List.class, this.builder.typeRef(PParameter.class)), _function_2);
      this._eMFJvmTypesBuilder.<JvmField>operator_add(_members_2, _field_2);
      final Function1<CallableRelation, Boolean> _function_3 = (CallableRelation call) -> {
        return Boolean.valueOf((!(call instanceof PatternCall)));
      };
      final Function1<CallableRelation, Boolean> _function_4 = (CallableRelation it) -> {
        return Boolean.valueOf(PatternLanguageHelper.isNonSimpleConstraint(it));
      };
      final Procedure1<CallableRelation> _function_5 = (CallableRelation call) -> {
        final LinkedHashMap<String, IInputKey> embeddedParameters = PatternLanguageHelper.getParameterVariables(call, this.typeSystem, this.typeInferrer);
        EList<JvmMember> _members_3 = pQueryClass.getMembers();
        String _string = Integer.toString(call.hashCode());
        String _plus = ("EmbeddedQuery" + _string);
        final Procedure1<JvmGenericType> _function_6 = (JvmGenericType it) -> {
          Set<Map.Entry<String, IInputKey>> _entrySet = embeddedParameters.entrySet();
          for (final Map.Entry<String, IInputKey> entry : _entrySet) {
            {
              final String variableName = entry.getKey();
              EList<JvmMember> _members_4 = it.getMembers();
              final Procedure1<JvmField> _function_7 = (JvmField it_1) -> {
                it_1.setFinal(true);
                it_1.setVisibility(JvmVisibility.PRIVATE);
                this._eMFJvmTypesBuilder.setInitializer(it_1, this.parameterInstantiation(variableName, this.getTypeString(entry.getValue(), call), entry.getValue()));
              };
              JvmField _field_3 = this._eMFJvmTypesBuilder.toField(call, this.util.getPParameterName(variableName), this.builder.typeRef(PParameter.class), _function_7);
              this._eMFJvmTypesBuilder.<JvmField>operator_add(_members_4, _field_3);
            }
          }
          EList<JvmMember> _members_4 = it.getMembers();
          final Procedure1<JvmField> _function_7 = (JvmField it_1) -> {
            it_1.setFinal(true);
            it_1.setVisibility(JvmVisibility.PRIVATE);
            StringConcatenationClient _client = new StringConcatenationClient() {
              @Override
              protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                _builder.append(Arrays.class);
                _builder.append(".asList(");
                {
                  Set<String> _keySet = embeddedParameters.keySet();
                  boolean _hasElements = false;
                  for(final String paramName : _keySet) {
                    if (!_hasElements) {
                      _hasElements = true;
                    } else {
                      _builder.appendImmediate(", ", "");
                    }
                    String _pParameterName = PatternQuerySpecificationClassInferrer.this.util.getPParameterName(paramName);
                    _builder.append(_pParameterName);
                  }
                }
                _builder.append(")");
              }
            };
            this._eMFJvmTypesBuilder.setInitializer(it_1, _client);
          };
          JvmField _field_3 = this._eMFJvmTypesBuilder.toField(pattern, "embeddedParameters", this.builder.typeRef(List.class, this.builder.typeRef(PParameter.class)), _function_7);
          this._eMFJvmTypesBuilder.<JvmField>operator_add(_members_4, _field_3);
          it.setVisibility(JvmVisibility.PRIVATE);
          EList<JvmTypeReference> _superTypes = it.getSuperTypes();
          JvmTypeReference _typeRef = this.builder.typeRef(BaseGeneratedEMFPQuery.class);
          this._eMFJvmTypesBuilder.<JvmTypeReference>operator_add(_superTypes, _typeRef);
          EList<JvmMember> _members_5 = it.getMembers();
          final Procedure1<JvmConstructor> _function_8 = (JvmConstructor it_1) -> {
            StringConcatenationClient _client = new StringConcatenationClient() {
              @Override
              protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                _builder.append("super(");
                _builder.append(PVisibility.class);
                _builder.append(".EMBEDDED);");
              }
            };
            this._eMFJvmTypesBuilder.setBody(it_1, _client);
          };
          JvmConstructor _constructor = this._eMFJvmTypesBuilder.toConstructor(call, _function_8);
          this._eMFJvmTypesBuilder.<JvmConstructor>operator_add(_members_5, _constructor);
          EList<JvmMember> _members_6 = it.getMembers();
          final Procedure1<JvmOperation> _function_9 = (JvmOperation it_1) -> {
            EList<JvmAnnotationReference> _annotations = it_1.getAnnotations();
            JvmAnnotationReference _annotationRef = this.annBuilder.annotationRef(Override.class);
            this._eMFJvmTypesBuilder.<JvmAnnotationReference>operator_add(_annotations, _annotationRef);
            StringConcatenationClient _client = new StringConcatenationClient() {
              @Override
              protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                _builder.append("return ");
                String _simpleName = pQueryClass.getSimpleName();
                _builder.append(_simpleName);
                _builder.append(".this.getFullyQualifiedName() + \"$");
                int _hashCode = call.hashCode();
                _builder.append(_hashCode);
                _builder.append("\";");
              }
            };
            this._eMFJvmTypesBuilder.setBody(it_1, _client);
          };
          JvmOperation _method = this._eMFJvmTypesBuilder.toMethod(call, "getFullyQualifiedName", this.builder.typeRef(String.class), _function_9);
          this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_6, _method);
          EList<JvmMember> _members_7 = it.getMembers();
          final Procedure1<JvmOperation> _function_10 = (JvmOperation it_1) -> {
            EList<JvmAnnotationReference> _annotations = it_1.getAnnotations();
            JvmAnnotationReference _annotationRef = this.annBuilder.annotationRef(Override.class);
            this._eMFJvmTypesBuilder.<JvmAnnotationReference>operator_add(_annotations, _annotationRef);
            StringConcatenationClient _client = new StringConcatenationClient() {
              @Override
              protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                _builder.append("return embeddedParameters;");
              }
            };
            this._eMFJvmTypesBuilder.setBody(it_1, _client);
          };
          JvmOperation _method_1 = this._eMFJvmTypesBuilder.toMethod(call, "getParameters", this.builder.typeRef(List.class, this.builder.typeRef(PParameter.class)), _function_10);
          this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_7, _method_1);
          EList<JvmMember> _members_8 = it.getMembers();
          final Procedure1<JvmOperation> _function_11 = (JvmOperation it_1) -> {
            EList<JvmAnnotationReference> _annotations = it_1.getAnnotations();
            JvmAnnotationReference _annotationRef = this.annBuilder.annotationRef(Override.class);
            this._eMFJvmTypesBuilder.<JvmAnnotationReference>operator_add(_annotations, _annotationRef);
            try {
              StringConcatenationClient _client = new StringConcatenationClient() {
                @Override
                protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                  StringConcatenationClient _inferSingleConstraintBody = PatternQuerySpecificationClassInferrer.this.inferSingleConstraintBody(pattern, call);
                  _builder.append(_inferSingleConstraintBody);
                  _builder.newLineIfNotEmpty();
                  _builder.append("return ");
                  _builder.append(Collections.class);
                  _builder.append(".singleton(body);");
                  _builder.newLineIfNotEmpty();
                }
              };
              this._eMFJvmTypesBuilder.setBody(it_1, _client);
            } catch (final Throwable _t) {
              if (_t instanceof Exception) {
                final Exception e = (Exception)_t;
                StringConcatenationClient _client_1 = new StringConcatenationClient() {
                  @Override
                  protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                    _builder.append("addError(new ");
                    _builder.append(PProblem.class);
                    _builder.append("(\"Inconsistent pattern definition threw exception ");
                    String _simpleName = e.getClass().getSimpleName();
                    _builder.append(_simpleName);
                    _builder.append("  with message: ");
                    String _escapeToQuotedString = PatternQuerySpecificationClassInferrer.this.util.escapeToQuotedString(e.getMessage());
                    _builder.append(_escapeToQuotedString);
                    _builder.append("\"));");
                    _builder.newLineIfNotEmpty();
                    _builder.append("return bodies;");
                    _builder.newLine();
                  }
                };
                this._eMFJvmTypesBuilder.setBody(it_1, _client_1);
              } else {
                throw Exceptions.sneakyThrow(_t);
              }
            }
          };
          JvmOperation _method_2 = this._eMFJvmTypesBuilder.toMethod(pattern, "doGetContainedBodies", this.builder.typeRef(Set.class, this.builder.typeRef(PBody.class)), _function_11);
          this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_8, _method_2);
        };
        JvmGenericType _class = this._eMFJvmTypesBuilder.toClass(call, _plus, _function_6);
        this._eMFJvmTypesBuilder.<JvmGenericType>operator_add(_members_3, _class);
      };
      IteratorExtensions.<CallableRelation>forEach(IteratorExtensions.<CallableRelation>filter(IteratorExtensions.<CallableRelation>filter(Iterators.<CallableRelation>filter(pattern.eAllContents(), CallableRelation.class), _function_3), _function_4), _function_5);
      EList<JvmMember> _members_3 = pQueryClass.getMembers();
      final Procedure1<JvmConstructor> _function_6 = (JvmConstructor it) -> {
        it.setVisibility(JvmVisibility.PRIVATE);
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("super(");
            _builder.append(PVisibility.class);
            _builder.append(".");
            PVisibility _calculatePVisibility = PatternLanguageHelper.calculatePVisibility(pattern);
            _builder.append(_calculatePVisibility);
            _builder.append(");");
          }
        };
        this._eMFJvmTypesBuilder.setBody(it, _client);
      };
      JvmConstructor _constructor = this._eMFJvmTypesBuilder.toConstructor(pattern, _function_6);
      this._eMFJvmTypesBuilder.<JvmConstructor>operator_add(_members_3, _constructor);
      EList<JvmMember> _members_4 = pQueryClass.getMembers();
      final Procedure1<JvmOperation> _function_7 = (JvmOperation it) -> {
        it.setVisibility(JvmVisibility.PUBLIC);
        EList<JvmAnnotationReference> _annotations = it.getAnnotations();
        JvmAnnotationReference _annotationRef = this.annBuilder.annotationRef(Override.class);
        this._eMFJvmTypesBuilder.<JvmAnnotationReference>operator_add(_annotations, _annotationRef);
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("return \"");
            String _fullyQualifiedName = PatternLanguageHelper.getFullyQualifiedName(pattern);
            _builder.append(_fullyQualifiedName);
            _builder.append("\";");
            _builder.newLineIfNotEmpty();
          }
        };
        this._eMFJvmTypesBuilder.setBody(it, _client);
      };
      JvmOperation _method = this._eMFJvmTypesBuilder.toMethod(pattern, "getFullyQualifiedName", this.builder.typeRef(String.class), _function_7);
      this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_4, _method);
      EList<JvmMember> _members_5 = pQueryClass.getMembers();
      final Procedure1<JvmOperation> _function_8 = (JvmOperation it) -> {
        it.setVisibility(JvmVisibility.PUBLIC);
        EList<JvmAnnotationReference> _annotations = it.getAnnotations();
        JvmAnnotationReference _annotationRef = this.annBuilder.annotationRef(Override.class);
        this._eMFJvmTypesBuilder.<JvmAnnotationReference>operator_add(_annotations, _annotationRef);
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("return ");
            _builder.append(Arrays.class);
            _builder.append(".asList(");
            {
              EList<Variable> _parameters = pattern.getParameters();
              boolean _hasElements = false;
              for(final Variable param : _parameters) {
                if (!_hasElements) {
                  _hasElements = true;
                } else {
                  _builder.appendImmediate(",", "");
                }
                _builder.append("\"");
                String _name = param.getName();
                _builder.append(_name);
                _builder.append("\"");
              }
            }
            _builder.append(");");
          }
        };
        this._eMFJvmTypesBuilder.setBody(it, _client);
      };
      JvmOperation _method_1 = this._eMFJvmTypesBuilder.toMethod(pattern, "getParameterNames", this.builder.typeRef(List.class, this.builder.typeRef(String.class)), _function_8);
      this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_5, _method_1);
      EList<JvmMember> _members_6 = pQueryClass.getMembers();
      final Procedure1<JvmOperation> _function_9 = (JvmOperation it) -> {
        it.setVisibility(JvmVisibility.PUBLIC);
        EList<JvmAnnotationReference> _annotations = it.getAnnotations();
        JvmAnnotationReference _annotationRef = this.annBuilder.annotationRef(Override.class);
        this._eMFJvmTypesBuilder.<JvmAnnotationReference>operator_add(_annotations, _annotationRef);
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("return parameters;");
          }
        };
        this._eMFJvmTypesBuilder.setBody(it, _client);
      };
      JvmOperation _method_2 = this._eMFJvmTypesBuilder.toMethod(pattern, "getParameters", this.builder.typeRef(List.class, this.builder.typeRef(PParameter.class)), _function_9);
      this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_6, _method_2);
      EList<JvmMember> _members_7 = pQueryClass.getMembers();
      final Procedure1<JvmOperation> _function_10 = (JvmOperation it) -> {
        it.setVisibility(JvmVisibility.PUBLIC);
        EList<JvmAnnotationReference> _annotations = it.getAnnotations();
        JvmAnnotationReference _annotationRef = this.annBuilder.annotationRef(Override.class);
        this._eMFJvmTypesBuilder.<JvmAnnotationReference>operator_add(_annotations, _annotationRef);
        try {
          StringConcatenationClient _client = new StringConcatenationClient() {
            @Override
            protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
              StringConcatenationClient _inferQueryEvaluationHints = PatternQuerySpecificationClassInferrer.this.inferQueryEvaluationHints(pattern);
              _builder.append(_inferQueryEvaluationHints);
              _builder.newLineIfNotEmpty();
              _builder.append(Set.class);
              _builder.append("<");
              _builder.append(PBody.class);
              _builder.append("> bodies = new ");
              _builder.append(LinkedHashSet.class);
              _builder.append("<>();");
              _builder.newLineIfNotEmpty();
              StringConcatenationClient _inferBodies = PatternQuerySpecificationClassInferrer.this.inferBodies(pattern);
              _builder.append(_inferBodies);
              _builder.newLineIfNotEmpty();
              StringConcatenationClient _inferAnnotations = PatternQuerySpecificationClassInferrer.this.inferAnnotations(pattern);
              _builder.append(_inferAnnotations);
              _builder.newLineIfNotEmpty();
              _builder.append("return bodies;");
              _builder.newLine();
            }
          };
          this._eMFJvmTypesBuilder.setBody(it, _client);
        } catch (final Throwable _t) {
          if (_t instanceof Exception) {
            final Exception e = (Exception)_t;
            StringConcatenationClient _client_1 = new StringConcatenationClient() {
              @Override
              protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
                _builder.append("addError(new ");
                _builder.append(PProblem.class);
                _builder.append("(\"Inconsistent pattern definition threw exception ");
                String _simpleName = e.getClass().getSimpleName();
                _builder.append(_simpleName);
                _builder.append("  with message: ");
                String _escapeToQuotedString = PatternQuerySpecificationClassInferrer.this.util.escapeToQuotedString(e.getMessage());
                _builder.append(_escapeToQuotedString);
                _builder.append("\"));");
                _builder.newLineIfNotEmpty();
                _builder.append("return bodies;");
                _builder.newLine();
              }
            };
            this._eMFJvmTypesBuilder.setBody(it, _client_1);
          } else {
            throw Exceptions.sneakyThrow(_t);
          }
        }
      };
      JvmOperation _method_3 = this._eMFJvmTypesBuilder.toMethod(pattern, "doGetContainedBodies", this.builder.typeRef(Set.class, this.builder.typeRef(PBody.class)), _function_10);
      _xblockexpression = this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_7, _method_3);
    }
    return _xblockexpression;
  }
  
  public ExecutionType getRequestedExecutionType(final Pattern pattern) {
    ExecutionType _xblockexpression = null;
    {
      final Modifiers modifier = pattern.getModifiers();
      ExecutionType _xifexpression = null;
      if ((modifier != null)) {
        _xifexpression = modifier.getExecution();
      } else {
        _xifexpression = ExecutionType.UNSPECIFIED;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  public StringConcatenationClient inferQueryEvaluationHints(final Pattern pattern) {
    StringConcatenationClient _xblockexpression = null;
    {
      QueryEvaluationHint.BackendRequirement _switchResult = null;
      ExecutionType _requestedExecutionType = this.getRequestedExecutionType(pattern);
      if (_requestedExecutionType != null) {
        switch (_requestedExecutionType) {
          case INCREMENTAL:
            _switchResult = QueryEvaluationHint.BackendRequirement.DEFAULT_CACHING;
            break;
          case SEARCH:
            _switchResult = QueryEvaluationHint.BackendRequirement.DEFAULT_SEARCH;
            break;
          case UNSPECIFIED:
            _switchResult = QueryEvaluationHint.BackendRequirement.UNSPECIFIED;
            break;
          default:
            break;
        }
      }
      final QueryEvaluationHint.BackendRequirement requirement = _switchResult;
      StringConcatenationClient _client = new StringConcatenationClient() {
        @Override
        protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
          _builder.append("setEvaluationHints(new ");
          _builder.append(QueryEvaluationHint.class);
          _builder.append("(null, ");
          _builder.append(QueryEvaluationHint.BackendRequirement.class);
          _builder.append(".");
          _builder.append(requirement);
          _builder.append("));");
        }
      };
      _xblockexpression = _client;
    }
    return _xblockexpression;
  }
  
  public StringConcatenationClient inferBodies(final Pattern pattern) throws IllegalStateException {
    StringConcatenationClient _client = new StringConcatenationClient() {
      @Override
      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
        {
          EList<PatternBody> _bodies = pattern.getBodies();
          for(final PatternBody body : _bodies) {
            _builder.newLineIfNotEmpty();
            _builder.append("{");
            _builder.newLine();
            _builder.append("    ");
            _builder.append("PBody body = new PBody(this);");
            _builder.newLine();
            _builder.append("    ");
            BodyCodeGenerator _bodyCodeGenerator = new BodyCodeGenerator(pattern, body, PatternQuerySpecificationClassInferrer.this.util, PatternQuerySpecificationClassInferrer.this.feedback, PatternQuerySpecificationClassInferrer.this.serializer, PatternQuerySpecificationClassInferrer.this.builder, PatternQuerySpecificationClassInferrer.this.typeInferrer, PatternQuerySpecificationClassInferrer.this.typeSystem);
            _builder.append(_bodyCodeGenerator, "    ");
            _builder.newLineIfNotEmpty();
            _builder.append("    ");
            _builder.append("bodies.add(body);");
            _builder.newLine();
            _builder.append("}");
            _builder.newLine();
          }
        }
      }
    };
    return _client;
  }
  
  public StringConcatenationClient inferSingleConstraintBody(final Pattern parentPattern, final CallableRelation call) throws IllegalStateException {
    StringConcatenationClient _client = new StringConcatenationClient() {
      @Override
      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
        _builder.append("PBody body = new PBody(this);");
        _builder.newLine();
        BodyCodeGenerator _bodyCodeGenerator = new BodyCodeGenerator(parentPattern, call, PatternQuerySpecificationClassInferrer.this.util, PatternQuerySpecificationClassInferrer.this.feedback, PatternQuerySpecificationClassInferrer.this.serializer, PatternQuerySpecificationClassInferrer.this.builder, PatternQuerySpecificationClassInferrer.this.typeInferrer, PatternQuerySpecificationClassInferrer.this.typeSystem);
        _builder.append(_bodyCodeGenerator);
        _builder.newLineIfNotEmpty();
      }
    };
    return _client;
  }
  
  /**
   * Infers inner class for QuerySpecification class based on the input 'pattern'.
   */
  public boolean inferQuerySpecificationInnerClasses(final JvmDeclaredType querySpecificationClass, final Pattern pattern, final boolean withPatternSpecificMatcher) {
    boolean _xblockexpression = false;
    {
      EList<JvmMember> _members = querySpecificationClass.getMembers();
      final Procedure1<JvmGenericType> _function = (JvmGenericType it) -> {
        it.setVisibility(JvmVisibility.PRIVATE);
        it.setStatic(true);
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Inner class allowing the singleton instance of {@link ");
        JvmType _findInferredSpecification = this.util.findInferredSpecification(pattern);
        _builder.append(_findInferredSpecification);
        _builder.append("} to be created ");
        _builder.newLineIfNotEmpty();
        _builder.append("    ");
        _builder.append("<b>not</b> at the class load time of the outer class, ");
        _builder.newLine();
        _builder.append("    ");
        _builder.append("but rather at the first call to {@link ");
        JvmType _findInferredSpecification_1 = this.util.findInferredSpecification(pattern);
        _builder.append(_findInferredSpecification_1, "    ");
        _builder.append("#instance()}.");
        _builder.newLineIfNotEmpty();
        _builder.newLine();
        _builder.append("<p> This workaround is required e.g. to support recursion.");
        _builder.newLine();
        this._eMFJvmTypesBuilder.setDocumentation(it, _builder.toString());
        EList<JvmMember> _members_1 = it.getMembers();
        final Procedure1<JvmField> _function_1 = (JvmField it_1) -> {
          it_1.setFinal(true);
          it_1.setStatic(true);
          StringConcatenationClient _client = new StringConcatenationClient() {
            @Override
            protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
              _builder.append("new ");
              JvmType _findInferredSpecification = PatternQuerySpecificationClassInferrer.this.util.findInferredSpecification(pattern);
              _builder.append(_findInferredSpecification);
              _builder.append("()");
            }
          };
          this._eMFJvmTypesBuilder.setInitializer(it_1, _client);
        };
        JvmField _field = this._eMFJvmTypesBuilder.toField(pattern, "INSTANCE", this.builder.typeRef(querySpecificationClass), _function_1);
        this._eMFJvmTypesBuilder.<JvmField>operator_add(_members_1, _field);
        EList<JvmMember> _members_2 = it.getMembers();
        final Procedure1<JvmField> _function_2 = (JvmField it_1) -> {
          it_1.setFinal(true);
          it_1.setStatic(true);
          StringConcatenationClient _client = new StringConcatenationClient() {
            @Override
            protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
              _builder.append("ensureInitialized()");
            }
          };
          this._eMFJvmTypesBuilder.setInitializer(it_1, _client);
          StringConcatenation _builder_1 = new StringConcatenation();
          _builder_1.append("Statically initializes the query specification <b>after</b> the field {@link #INSTANCE} is assigned.");
          _builder_1.newLine();
          _builder_1.append("This initialization order is required to support indirect recursion.");
          _builder_1.newLine();
          _builder_1.newLine();
          _builder_1.append("<p> The static initializer is defined using a helper field to work around limitations of the code generator.");
          _builder_1.newLine();
          this._eMFJvmTypesBuilder.setDocumentation(it_1, _builder_1.toString());
        };
        JvmField _field_1 = this._eMFJvmTypesBuilder.toField(pattern, "STATIC_INITIALIZER", this.builder.typeRef(Object.class), _function_2);
        this._eMFJvmTypesBuilder.<JvmField>operator_add(_members_2, _field_1);
        EList<JvmMember> _members_3 = it.getMembers();
        final Procedure1<JvmOperation> _function_3 = (JvmOperation it_1) -> {
          it_1.setVisibility(JvmVisibility.PUBLIC);
          it_1.setStatic(true);
          StringConcatenationClient _client = new StringConcatenationClient() {
            @Override
            protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
              _builder.append("INSTANCE.ensureInitializedInternal();");
              _builder.newLine();
              _builder.append("return null;");
              _builder.newLine();
            }
          };
          this._eMFJvmTypesBuilder.setBody(it_1, _client);
        };
        JvmOperation _method = this._eMFJvmTypesBuilder.toMethod(pattern, "ensureInitialized", this.builder.typeRef(Object.class), _function_3);
        this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members_3, _method);
      };
      JvmGenericType _class = this._eMFJvmTypesBuilder.toClass(pattern, this.util.querySpecificationHolderClassName(pattern), _function);
      this._eMFJvmTypesBuilder.<JvmGenericType>operator_add(_members, _class);
      EList<JvmMember> _members_1 = querySpecificationClass.getMembers();
      final Procedure1<JvmGenericType> _function_1 = (JvmGenericType it) -> {
        it.setVisibility(JvmVisibility.PRIVATE);
        it.setStatic(true);
        EList<JvmTypeReference> _superTypes = it.getSuperTypes();
        JvmTypeReference _typeRef = this.builder.typeRef(BaseGeneratedEMFPQuery.class);
        this._eMFJvmTypesBuilder.<JvmTypeReference>operator_add(_superTypes, _typeRef);
        this.inferPQueryMembers(it, pattern);
      };
      JvmGenericType _class_1 = this._eMFJvmTypesBuilder.toClass(pattern, this.util.querySpecificationPQueryClassName(pattern), _function_1);
      _xblockexpression = this._eMFJvmTypesBuilder.<JvmGenericType>operator_add(_members_1, _class_1);
    }
    return _xblockexpression;
  }
  
  public void inferExpressions(final JvmDeclaredType querySpecificationClass, final Pattern pattern) {
    final Function1<PatternBody, Collection<XExpression>> _function = (PatternBody it) -> {
      return PatternLanguageHelper.getAllTopLevelXBaseExpressions(it);
    };
    final Consumer<XExpression> _function_1 = (XExpression ex) -> {
      EList<JvmMember> _members = querySpecificationClass.getMembers();
      final Procedure1<JvmOperation> _function_2 = (JvmOperation it) -> {
        it.setVisibility(JvmVisibility.PRIVATE);
        it.setStatic(true);
        List<Variable> _variables = this.util.variables(ex);
        for (final Variable variable : _variables) {
          {
            final JvmFormalParameter parameter = this._eMFJvmTypesBuilder.toParameter(variable, variable.getName(), this.util.calculateType(variable));
            EList<JvmFormalParameter> _parameters = it.getParameters();
            this._eMFJvmTypesBuilder.<JvmFormalParameter>operator_add(_parameters, parameter);
          }
        }
        this._eMFJvmTypesBuilder.setBody(it, ex);
      };
      JvmOperation _method = this._eMFJvmTypesBuilder.toMethod(ex, this.util.expressionMethodName(ex), this._eMFJvmTypesBuilder.inferredType(ex), _function_2);
      this._eMFJvmTypesBuilder.<JvmOperation>operator_add(_members, _method);
    };
    Iterables.<XExpression>concat(ListExtensions.<PatternBody, Collection<XExpression>>map(pattern.getBodies(), _function)).forEach(_function_1);
  }
  
  public StringConcatenationClient parameterInstantiation(final Variable variable) {
    StringConcatenationClient _xblockexpression = null;
    {
      final String clazz = this.getTypeString(variable);
      final Type type = variable.getType();
      StringConcatenationClient _xifexpression = null;
      if (((type == null) || (!this.typeSystem.isValidType(type)))) {
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("new PParameter(\"");
            String _name = variable.getName();
            _builder.append(_name);
            _builder.append("\", \"");
            _builder.append(clazz);
            _builder.append("\", (");
            _builder.append(IInputKey.class);
            _builder.append(")null, ");
            StringConcatenationClient _directionLiteral = PatternQuerySpecificationClassInferrer.this.directionLiteral(variable);
            _builder.append(_directionLiteral);
            _builder.append(")");
          }
        };
        _xifexpression = _client;
      } else {
        final IInputKey declaredInputKey = this.typeSystem.extractTypeDescriptor(type);
        return this.parameterInstantiation(variable, clazz, declaredInputKey);
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  public StringConcatenationClient parameterInstantiation(final Variable variable, final String clazz, final IInputKey type) {
    StringConcatenationClient _client = new StringConcatenationClient() {
      @Override
      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
        _builder.append("new PParameter(\"");
        String _name = variable.getName();
        _builder.append(_name);
        _builder.append("\", \"");
        _builder.append(clazz);
        _builder.append("\", ");
        StringConcatenationClient _serializeInputKey = PatternQuerySpecificationClassInferrer.this.util.serializeInputKey(type, true);
        _builder.append(_serializeInputKey);
        _builder.append(", ");
        StringConcatenationClient _directionLiteral = PatternQuerySpecificationClassInferrer.this.directionLiteral(variable);
        _builder.append(_directionLiteral);
        _builder.append(")");
      }
    };
    return _client;
  }
  
  public StringConcatenationClient parameterInstantiation(final String variableName, final String clazz, final IInputKey type) {
    StringConcatenationClient _client = new StringConcatenationClient() {
      @Override
      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
        _builder.append("new PParameter(\"");
        _builder.append(variableName);
        _builder.append("\", \"");
        _builder.append(clazz);
        _builder.append("\", ");
        StringConcatenationClient _serializeInputKey = PatternQuerySpecificationClassInferrer.this.util.serializeInputKey(type, true);
        _builder.append(_serializeInputKey);
        _builder.append(", ");
        _builder.append(PParameterDirection.class);
        _builder.append(".");
        String _name = PParameterDirection.INOUT.name();
        _builder.append(_name);
        _builder.append(")");
      }
    };
    return _client;
  }
  
  public String getTypeString(final Variable variable) {
    final JvmTypeReference ref = this.typeInferrer.getJvmType(variable, variable);
    String _xifexpression = null;
    if (((ref == null) || (ref instanceof JvmUnknownTypeReference))) {
      _xifexpression = "";
    } else {
      _xifexpression = ref.getType().getQualifiedName();
    }
    return _xifexpression;
  }
  
  public String getTypeString(final IInputKey type, final EObject context) {
    final JvmTypeReference ref = this.typeSystem.toJvmTypeReference(type, context);
    String _xifexpression = null;
    if (((ref == null) || (ref instanceof JvmUnknownTypeReference))) {
      _xifexpression = "";
    } else {
      _xifexpression = ref.getType().getQualifiedName();
    }
    return _xifexpression;
  }
  
  public StringConcatenationClient inferAnnotations(final Pattern pattern) {
    StringConcatenationClient _client = new StringConcatenationClient() {
      @Override
      protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
        {
          EList<Annotation> _annotations = pattern.getAnnotations();
          for(final Annotation annotation : _annotations) {
            _builder.append("{");
            _builder.newLine();
            _builder.append("    ");
            _builder.append(PAnnotation.class, "    ");
            _builder.append(" annotation = new ");
            _builder.append(PAnnotation.class, "    ");
            _builder.append("(\"");
            String _name = annotation.getName();
            _builder.append(_name, "    ");
            _builder.append("\");");
            _builder.newLineIfNotEmpty();
            {
              Set<Map.Entry<String, Object>> _entries = PatternLanguageHelper.evaluateAnnotationParametersWithMultiplicity(annotation).entries();
              for(final Map.Entry<String, Object> attribute : _entries) {
                _builder.append("    ");
                _builder.append("annotation.addAttribute(\"");
                String _key = attribute.getKey();
                _builder.append(_key, "    ");
                _builder.append("\", ");
                StringConcatenationClient _outputAnnotationParameter = PatternQuerySpecificationClassInferrer.this.outputAnnotationParameter(attribute.getValue());
                _builder.append(_outputAnnotationParameter, "    ");
                _builder.append(");");
                _builder.newLineIfNotEmpty();
              }
            }
            _builder.append("    ");
            _builder.append("addAnnotation(annotation);");
            _builder.newLine();
            _builder.append("}");
            _builder.newLine();
          }
        }
      }
    };
    return _client;
  }
  
  public StringConcatenationClient outputAnnotationParameter(final Object value) {
    StringConcatenationClient _switchResult = null;
    boolean _matched = false;
    if (value instanceof List) {
      _matched=true;
      StringConcatenationClient _client = new StringConcatenationClient() {
        @Override
        protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
          _builder.append(Arrays.class);
          _builder.append(".asList(new Object[] {");
          _builder.newLineIfNotEmpty();
          {
            boolean _hasElements = false;
            for(final Object item : ((List<?>)value)) {
              if (!_hasElements) {
                _hasElements = true;
              } else {
                _builder.appendImmediate(", ", "                    ");
              }
              _builder.append("                    ");
              StringConcatenationClient _outputAnnotationParameter = PatternQuerySpecificationClassInferrer.this.outputAnnotationParameter(item);
              _builder.append(_outputAnnotationParameter, "                    ");
              _builder.newLineIfNotEmpty();
            }
          }
          _builder.append("                    ");
          _builder.append("})");
        }
      };
      _switchResult = _client;
    }
    if (!_matched) {
      if (value instanceof ParameterReference) {
        _matched=true;
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("new ");
            _builder.append(ParameterReference.class);
            _builder.append("(\"");
            String _name = ((ParameterReference)value).getName();
            _builder.append(_name);
            _builder.append("\")");
          }
        };
        _switchResult = _client;
      }
    }
    if (!_matched) {
      if (value instanceof String) {
        _matched=true;
        StringConcatenationClient _client = new StringConcatenationClient() {
          @Override
          protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
            _builder.append("\"");
            _builder.append(((String)value));
            _builder.append("\"");
          }
        };
        _switchResult = _client;
      }
    }
    if (!_matched) {
      StringConcatenationClient _client = new StringConcatenationClient() {
        @Override
        protected void appendTo(StringConcatenationClient.TargetStringConcatenation _builder) {
          String _string = value.toString();
          _builder.append(_string);
        }
      };
      _switchResult = _client;
    }
    return _switchResult;
  }
}
