"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getInstructions = exports.getEmptyInstructions = exports.getDisassembledInstruction = void 0;
const mi_1 = require("../mi");
const calculateMemoryOffset_1 = require("./calculateMemoryOffset");
const isHexString_1 = require("./isHexString");
/**
 * Converts the MIDataDisassembleAsmInsn object to DebugProtocol.DisassembledInstruction
 *
 * @param asmInstruction
 * 		MI instruction object
 * @return
 * 		Returns the DebugProtocol.DisassembledInstruction object
 */
const getDisassembledInstruction = (asmInstruction) => {
    let symbol;
    if (asmInstruction['func-name'] && asmInstruction.offset) {
        symbol = `${asmInstruction['func-name']}+${asmInstruction.offset}`;
    }
    else if (asmInstruction['func-name']) {
        symbol = asmInstruction['func-name'];
    }
    else {
        symbol = undefined;
    }
    return Object.assign({ address: asmInstruction.address, instructionBytes: asmInstruction.opcodes, instruction: asmInstruction.inst }, (symbol ? { symbol } : {}));
};
exports.getDisassembledInstruction = getDisassembledInstruction;
/**
 * Returns a sequence of empty instructions to fill the gap in DisassembleRequest
 *
 * @param startAddress
 * 		The starting address of the sequence
 * @param count
 * 		The number of the instructions to return back
 * @param step
 * 		Memory step to calculate the next instructions address. It can be negative.
 * @return
 * 		Returns sequence of empty instructions
 */
const getEmptyInstructions = (startAddress, count, step) => {
    const badDisInsn = (address) => ({
        address,
        instruction: 'failed to retrieve instruction',
        presentationHint: 'invalid',
    });
    const list = [];
    let address = startAddress;
    for (let ix = 0; ix < count; ix++) {
        if (step < 0) {
            address = (0, calculateMemoryOffset_1.calculateMemoryOffset)(address, step);
            list.unshift(badDisInsn(address));
        }
        else {
            list.push(badDisInsn(address));
            address = (0, calculateMemoryOffset_1.calculateMemoryOffset)(address, step);
        }
    }
    return list;
};
exports.getEmptyInstructions = getEmptyInstructions;
/**
 * Gets the instructions from the memory according to the given reference values.
 *
 * For example:
 * If you like to return 100 instructions starting from the 0x00001F00 address,
 * you can use the method like below:
 *
 * const instructions = await memoryReference('0x00001F00', 100);
 *
 * To return lower memory areas, (handling the negative offset),
 * you can use negative length value:
 *
 * const instructions = await memoryReference('0x00001F00', -100);
 *
 * Method returns the expected length of the instructions, if cannot read expected
 * length (can be due to memory bounds), empty instructions will be filled.
 *
 * @param gdb
 * 		GDB Backend instance
 * @param memoryReference
 * 		Starting memory address for the operation
 * @param length
 * 		The count of the instructions to fetch, can be negative if wanted to return negative offset
 * @return
 * 		Returns the given amount of instructions
 */
const getInstructions = (gdb, memoryReference, length) => __awaiter(void 0, void 0, void 0, function* () {
    const list = [];
    const meanSizeOfInstruction = 4;
    const isReverseFetch = length < 0;
    const absLength = Math.abs(length);
    const formatMemoryAddress = (offset) => {
        return `(${memoryReference})${offset < 0 ? '-' : '+'}${Math.abs(offset)}`;
    };
    const sendDataDisassembleWrapper = (lower, upper) => __awaiter(void 0, void 0, void 0, function* () {
        if (lower === upper) {
            return [];
        }
        const list = [];
        const result = yield (0, mi_1.sendDataDisassemble)(gdb, formatMemoryAddress(lower), formatMemoryAddress(upper));
        for (const asmInsn of result.asm_insns) {
            const line = asmInsn.line
                ? parseInt(asmInsn.line, 10)
                : undefined;
            const location = {
                name: asmInsn.file,
                path: asmInsn.fullname,
            };
            for (const asmLine of asmInsn.line_asm_insn) {
                list.push(Object.assign(Object.assign({}, (0, exports.getDisassembledInstruction)(asmLine)), { location,
                    line }));
            }
        }
        return list;
    });
    const target = { lower: 0, higher: 0 };
    const recalculateTargetBounds = (length) => {
        if (isReverseFetch) {
            target.higher = target.lower;
            target.lower += length * meanSizeOfInstruction;
            // Limit the lower bound to not to cross negative memory address area
            if ((0, isHexString_1.isHexString)(memoryReference) &&
                BigInt(memoryReference) + BigInt(target.lower) < 0) {
                // Lower and Upper bounds are in number type
                target.lower = Number(memoryReference) * -1;
            }
        }
        else {
            target.lower = target.higher;
            target.higher += length * meanSizeOfInstruction;
        }
    };
    const remainingLength = () => Math.sign(length) * Math.max(absLength - list.length, 0);
    const pushToList = (instructions) => {
        if (isReverseFetch) {
            list.unshift(...instructions);
        }
        else {
            list.push(...instructions);
        }
    };
    try {
        while (absLength > list.length) {
            recalculateTargetBounds(remainingLength());
            const result = yield sendDataDisassembleWrapper(target.lower, target.higher);
            if (result.length === 0) {
                // If cannot retrieve more instructions, break the loop, go to catch
                // and fill the remaining instructions with empty instruction information
                break;
            }
            pushToList(result);
        }
    }
    catch (e) {
        // If error occured in the first iteration and no items can be read
        // throw the original error, otherwise continue and fill the empty instructions.
        if (list.length === 0) {
            throw e;
        }
    }
    if (absLength < list.length) {
        if (length < 0) {
            // Remove the heading, if necessary
            list.splice(0, list.length - absLength);
        }
        else {
            // Remove the tail, if necessary
            list.splice(absLength, list.length - absLength);
        }
    }
    // Fill with empty instructions in case couldn't read desired length
    if (absLength > list.length) {
        if (list.length === 0) {
            // In case of memory read error, where no instructions read before you cannot be sure about the memory offsets
            // Avoid sending empty instructions, which is overriding the previous disassembled instructions in the VSCode
            // Instead, send error message and fail the request.
            throw new Error(`Cannot retrieve instructions!`);
        }
        const lastMemoryAddress = list[isReverseFetch ? 0 : list.length - 1].address;
        const emptyInstuctions = (0, exports.getEmptyInstructions)(lastMemoryAddress, absLength - list.length, Math.sign(length) * 2);
        pushToList(emptyInstuctions);
    }
    return list;
});
exports.getInstructions = getInstructions;
//# sourceMappingURL=disassembly.js.map