/*******************************************************************************
 * Copyright 2005-2011, CHISEL Group, University of Victoria, Victoria, BC,
 * Canada. All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License v1.0 which
 * accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: The Chisel Group, University of Victoria Mateusz Matela
 * <mateusz.matela@gmail.com> - Adapt Zest to changes in layout -
 * https://bugs.eclipse.org/bugs/show_bug.cgi?id=283179
 ******************************************************************************/
package org.eclipse.gef4.zest.core.viewers;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;

import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.gef4.layout.LayoutAlgorithm;
import org.eclipse.gef4.zest.core.viewers.internal.IStylingGraphModelFactory;
import org.eclipse.gef4.zest.core.widgets.GraphWidget;
import org.eclipse.gef4.zest.core.widgets.GraphConnection;
import org.eclipse.gef4.zest.core.widgets.GraphContainer;
import org.eclipse.gef4.zest.core.widgets.GraphItem;
import org.eclipse.gef4.zest.core.widgets.GraphNode;
import org.eclipse.gef4.zest.core.widgets.ZestStyles;
import org.eclipse.gef4.zest.core.widgets.custom.CGraphNode;
import org.eclipse.swt.SWT;
import org.eclipse.swt.SWTError;
import org.eclipse.swt.widgets.Widget;

/**
 * Abstraction of graph viewers to implement functionality used by all of them.
 * Not intended to be implemented by clients. Use one of the provided children
 * instead.
 * 
 * @author Del Myers
 * @since 2.0
 */
public abstract class AbstractStructuredGraphViewer extends
		AbstractZoomableViewer {
	/**
	 * Contains top-level styles for the entire graph. Set in the constructor. *
	 */
	private int graphStyle;

	/**
	 * Contains node-level styles for the graph. Set in setNodeStyle(). Defaults
	 * are used in the constructor.
	 */
	private int nodeStyle;

	/**
	 * Contains arc-level styles for the graph. Set in setConnectionStyle().
	 * Defaults are used in the constructor.
	 */
	private int connectionStyle;

	private HashMap<Object, GraphItem> nodesMap = new HashMap<Object, GraphItem>();
	private HashMap<Object, GraphItem> connectionsMap = new HashMap<Object, GraphItem>();

	/**
	 * A simple graph comparator that orders graph elements based on their type
	 * (connection or node), and their unique object identification.
	 */
	private class SimpleGraphComparator implements Comparator<GraphItem> {
		TreeSet<String> storedStrings;

		/**
		 * 
		 */
		public SimpleGraphComparator() {
			this.storedStrings = new TreeSet<String>();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
		 */
		public int compare(GraphItem arg0, GraphItem arg1) {
			if (arg0 instanceof GraphNode && arg1 instanceof GraphConnection) {
				return 1;
			} else if (arg0 instanceof GraphConnection
					&& arg1 instanceof GraphNode) {
				return -1;
			}
			if (arg0.equals(arg1)) {
				return 0;
			}
			return getObjectString(arg0).compareTo(getObjectString(arg1));
		}

		private String getObjectString(Object o) {
			String s = o.getClass().getName() + "@"
					+ Integer.toHexString(o.hashCode());
			while (storedStrings.contains(s)) {
				s = s + 'X';
			}
			return s;
		}
	}

	protected AbstractStructuredGraphViewer(int graphStyle) {
		this.graphStyle = graphStyle;
		this.connectionStyle = SWT.NONE;
		this.nodeStyle = SWT.NONE;

	}

	/**
	 * Sets the default style for nodes in this graph. Note: if an input is set
	 * on the viewer, a ZestException will be thrown.
	 * 
	 * @param nodeStyle
	 *            the style for the nodes.
	 * @see #ZestStyles
	 */
	public void setNodeStyle(int nodeStyle) {
		if (getInput() != null) {
			throw new SWTError(SWT.ERROR_UNSPECIFIED);
		}
		this.nodeStyle = nodeStyle;
	}

	/**
	 * Sets the default style for connections in this graph. Note: if an input
	 * is set on the viewer, a ZestException will be thrown.
	 * 
	 * @param connectionStyle
	 *            the style for the connections.
	 * @see #ZestStyles
	 */
	public void setConnectionStyle(int connectionStyle) {
		if (getInput() != null) {
			throw new SWTError(SWT.ERROR_UNSPECIFIED);
		}
		if (!ZestStyles.validateConnectionStyle(connectionStyle)) {
			throw new SWTError(SWT.ERROR_INVALID_ARGUMENT);
		}
		this.connectionStyle = connectionStyle;
	}

	/**
	 * Returns the style set for the graph
	 * 
	 * @return The style set of the graph
	 */
	public int getGraphStyle() {
		return graphStyle;
	}

	/**
	 * Returns the style set for the nodes.
	 * 
	 * @return the style set for the nodes.
	 */
	public int getNodeStyle() {
		return nodeStyle;
	}

	public GraphWidget getGraphControl() {
		return (GraphWidget) getControl();
	}

	/**
	 * @return the connection style.
	 */
	public int getConnectionStyle() {
		return connectionStyle;
	}

	/**
	 * Sets the layout algorithm for this viewer. Subclasses may place
	 * restrictions on the algorithms that it accepts.
	 * 
	 * @param algorithm
	 *            the layout algorithm
	 * @param run
	 *            true if the layout algorithm should be run immediately. This
	 *            is a hint.
	 */
	public abstract void setLayoutAlgorithm(LayoutAlgorithm algorithm,
			boolean run);

	/**
	 * Gets the current layout algorithm.
	 * 
	 * @return the current layout algorithm.
	 */
	protected abstract LayoutAlgorithm getLayoutAlgorithm();

	/**
	 * Equivalent to setLayoutAlgorithm(algorithm, false).
	 * 
	 * @param algorithm
	 */
	public void setLayoutAlgorithm(LayoutAlgorithm algorithm) {
		setLayoutAlgorithm(algorithm, false);
	}

	public Object[] getNodeElements() {
		return this.nodesMap.keySet().toArray();
	}

	public Object[] getConnectionElements() {
		return this.connectionsMap.keySet().toArray();
	}

	/**
	 * @noreference This method is not intended to be referenced by clients.
	 * @return
	 */
	public HashMap<Object, GraphItem> getNodesMap() {
		return this.nodesMap;
	}

	/**
	 * 
	 * @param element
	 * @return
	 * @noreference This method is not intended to be referenced by clients.
	 */
	public GraphNode addGraphModelContainer(Object element) {
		GraphNode node = this.getGraphModelNode(element);
		if (node == null) {
			node = new GraphContainer((GraphWidget) getControl(), SWT.NONE);
			this.nodesMap.put(element, node);
			node.setData(element);
		}
		return node;
	}

	/**
	 * 
	 * @param container
	 * @param element
	 * @return
	 * @noreference This method is not intended to be referenced by clients.
	 */
	public GraphNode addGraphModelNode(GraphContainer container, Object element) {
		GraphNode node = this.getGraphModelNode(element);
		if (node == null) {
			node = new GraphNode(container, SWT.NONE);
			this.nodesMap.put(element, node);
			node.setData(element);
		}
		return node;
	}

	/**
	 * 
	 * @param element
	 * @param figure
	 * @return
	 * @noreference This method is not intended to be referenced by clients.
	 */
	public GraphNode addGraphModelNode(Object element, IFigure figure) {
		GraphNode node = this.getGraphModelNode(element);
		if (node == null) {
			if (figure != null) {
				node = new CGraphNode((GraphWidget) getControl(), SWT.NONE, figure);
				this.nodesMap.put(element, node);
				node.setData(element);
			} else {
				node = new GraphNode((GraphWidget) getControl(), SWT.NONE);
				this.nodesMap.put(element, node);
				node.setData(element);
			}
		}
		return node;
	}

	/**
	 * 
	 * @param element
	 * @param source
	 * @param target
	 * @return
	 * @noreference This method is not intended to be referenced by clients.
	 */
	public GraphConnection addGraphModelConnection(Object element,
			GraphNode source, GraphNode target) {
		GraphConnection connection = this.getGraphModelConnection(element);
		if (connection == null) {
			connection = new GraphConnection((GraphWidget) getControl(), SWT.NONE,
					source, target);
			this.connectionsMap.put(element, connection);
			connection.setData(element);
		}
		return connection;

	}

	/**
	 * 
	 * @param obj
	 * @return
	 * @noreference This method is not intended to be referenced by clients.
	 */
	public GraphConnection getGraphModelConnection(Object obj) {
		GraphConnection connection = (GraphConnection) this.connectionsMap
				.get(obj);
		return (connection != null && !connection.isDisposed()) ? connection
				: null;
	}

	/**
	 * 
	 * @param obj
	 * @return
	 * @noreference This method is not intended to be referenced by clients.
	 */
	public GraphNode getGraphModelNode(Object obj) {
		GraphNode node = (GraphNode) this.nodesMap.get(obj);
		return (node != null && !node.isDisposed()) ? node : null;
	}

	/**
	 * @param obj
	 * @noreference This method is not intended to be referenced by clients.
	 */
	public void removeGraphModelConnection(Object obj) {
		GraphConnection connection = (GraphConnection) connectionsMap.get(obj);
		if (connection != null) {
			connectionsMap.remove(obj);
			if (!connection.isDisposed()) {
				connection.dispose();
			}
		}
	}

	/**
	 * @noreference This method is not intended to be referenced by clients.
	 */
	public void removeGraphModelNode(Object obj) {
		GraphNode node = (GraphNode) nodesMap.get(obj);
		if (node != null) {
			nodesMap.remove(obj);
			if (!node.isDisposed()) {
				node.dispose();
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.jface.viewers.StructuredViewer#internalRefresh(java.lang.
	 * Object)
	 */
	protected void internalRefresh(Object element) {
		if (getInput() == null) {
			return;
		}
		if (element == getInput()) {
			getFactory().refreshGraph(getGraphControl());
		} else {
			getFactory().refresh(getGraphControl(), element);
		}
		// After all the items are loaded, we call update to ensure drawing.
		// This way the damaged area does not get too big if we start
		// adding and removing more nodes
		getGraphControl().getLightweightSystem().getUpdateManager()
				.performUpdate();
	}

	protected void doUpdateItem(Widget item, Object element, boolean fullMap) {
		if (item == getGraphControl()) {
			getFactory().update(getNodesArray(getGraphControl()));
			getFactory().update(getConnectionsArray(getGraphControl()));
		} else if (item instanceof GraphItem) {
			getFactory().update((GraphItem) item);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.jface.viewers.StructuredViewer#doFindInputItem(java.lang.
	 * Object)
	 */
	protected Widget doFindInputItem(Object element) {

		if (element == getInput() && element instanceof Widget) {
			return (Widget) element;
		}
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.jface.viewers.StructuredViewer#doFindItem(java.lang.Object)
	 */
	protected Widget doFindItem(Object element) {
		Widget node = nodesMap.get(element);
		Widget connection = connectionsMap.get(element);
		return (node != null) ? node : connection;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.viewers.StructuredViewer#getSelectionFromWidget()
	 */
	protected List<Object> getSelectionFromWidget() {
		List internalSelection = getWidgetSelection();
		LinkedList<Object> externalSelection = new LinkedList<Object>();
		for (Iterator i = internalSelection.iterator(); i.hasNext();) {
			Object data = ((GraphItem) i.next()).getData();
			if (data != null) {
				externalSelection.add(data);
			}
		}
		return externalSelection;
	}

	protected GraphItem[] /* GraphItem */findItems(List l) {
		if (l == null) {
			return new GraphItem[0];
		}

		ArrayList<GraphItem> list = new ArrayList<GraphItem>();
		Iterator iterator = l.iterator();

		while (iterator.hasNext()) {
			GraphItem w = (GraphItem) findItem(iterator.next());
			list.add(w);
		}
		return list.toArray(new GraphItem[list.size()]);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.jface.viewers.StructuredViewer#setSelectionToWidget(java.
	 * util.List, boolean)
	 */
	protected void setSelectionToWidget(List l, boolean reveal) {
		GraphWidget control = (GraphWidget) getControl();
		List<GraphItem> selection = new LinkedList<GraphItem>();
		for (Iterator i = l.iterator(); i.hasNext();) {
			Object obj = i.next();
			GraphNode node = (GraphNode) nodesMap.get(obj);
			GraphConnection conn = (GraphConnection) connectionsMap.get(obj);
			if (node != null) {
				selection.add(node);
			}
			if (conn != null) {
				selection.add(conn);
			}
		}
		control.setSelection(selection.toArray(new GraphNode[selection.size()]));
	}

	/**
	 * Gets the internal model elements that are selected.
	 * 
	 * @return
	 */
	protected List getWidgetSelection() {
		GraphWidget control = (GraphWidget) getControl();
		return control.getSelection();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.viewers.Viewer#inputChanged(java.lang.Object,
	 * java.lang.Object)
	 */
	protected void inputChanged(Object input, Object oldInput) {
		IStylingGraphModelFactory factory = getFactory();
		factory.setConnectionStyle(getConnectionStyle());
		factory.setNodeStyle(getNodeStyle());

		// Save the old map so we can set the size and position of any nodes
		// that are the same
		Map<Object, GraphItem> oldNodesMap = nodesMap;
		GraphWidget graph = (GraphWidget) getControl();
		graph.setSelection(new GraphNode[0]);

		Iterator<GraphItem> iterator = nodesMap.values().iterator();
		while (iterator.hasNext()) {
			GraphNode node = (GraphNode) iterator.next();
			if (!node.isDisposed()) {
				node.dispose();
			}
		}

		iterator = connectionsMap.values().iterator();
		while (iterator.hasNext()) {
			GraphConnection connection = (GraphConnection) iterator.next();
			if (!connection.isDisposed()) {
				connection.dispose();
			}
		}

		nodesMap = new HashMap<Object, GraphItem>();
		connectionsMap = new HashMap<Object, GraphItem>();

		graph = factory.createGraphModel(graph);

		((GraphWidget) getControl()).setNodeStyle(getNodeStyle());
		((GraphWidget) getControl()).setConnectionStyle(getConnectionStyle());

		// check if any of the pre-existing nodes are still present
		// in this case we want them to keep the same location & size
		for (Iterator<Object> iter = oldNodesMap.keySet().iterator(); iter
				.hasNext();) {
			Object data = iter.next();
			GraphNode newNode = (GraphNode) nodesMap.get(data);
			if (newNode != null) {
				GraphNode oldNode = (GraphNode) oldNodesMap.get(data);
				newNode.setLocation(oldNode.getLocation().x,
						oldNode.getLocation().y);
				if (oldNode.isSizeFixed()) {
					newNode.setSize(oldNode.getSize().width,
							oldNode.getSize().height);
				}
			}
		}
	}

	/**
	 * Returns the factory used to create the model. This must not be called
	 * before the content provider is set.
	 * 
	 * @return
	 * @noreference This method is not intended to be referenced by clients.
	 * @nooverride This method is not intended to be re-implemented or extended
	 *             by clients.
	 */
	protected abstract IStylingGraphModelFactory getFactory();

	protected void filterVisuals() {
		if (getGraphControl() == null) {
			return;
		}
		Object[] filtered = getFilteredChildren(getInput());
		SimpleGraphComparator comparator = new SimpleGraphComparator();
		TreeSet<GraphItem> filteredElements = new TreeSet<GraphItem>(comparator);
		TreeSet<GraphItem> unfilteredElements = new TreeSet<GraphItem>(
				comparator);
		List connections = getGraphControl().getConnections();
		List nodes = getGraphControl().getNodes();
		if (filtered.length == 0) {
			// set everything to invisible.
			// @tag zest.bug.156528-Filters.check : should we only filter out
			// the nodes?
			for (Iterator i = connections.iterator(); i.hasNext();) {
				GraphConnection c = (GraphConnection) i.next();
				c.setVisible(false);
			}
			for (Iterator i = nodes.iterator(); i.hasNext();) {
				GraphNode n = (GraphNode) i.next();
				n.setVisible(false);
			}
			return;
		}
		for (Iterator i = connections.iterator(); i.hasNext();) {
			GraphConnection c = (GraphConnection) i.next();
			if (c.getData() != null) {
				unfilteredElements.add(c);
			}
		}
		for (Iterator i = nodes.iterator(); i.hasNext();) {
			GraphNode n = (GraphNode) i.next();
			if (n.getData() != null) {
				unfilteredElements.add(n);
			}
		}
		for (int i = 0; i < filtered.length; i++) {
			GraphItem modelElement = connectionsMap.get(filtered[i]);
			if (modelElement == null) {
				modelElement = nodesMap.get(filtered[i]);
			}
			if (modelElement != null) {
				filteredElements.add(modelElement);
			}
		}
		unfilteredElements.removeAll(filteredElements);
		// set all the elements that did not pass the filters to invisible, and
		// all the elements that passed to visible.
		while (unfilteredElements.size() > 0) {
			GraphItem i = unfilteredElements.first();
			i.setVisible(false);
			unfilteredElements.remove(i);
		}
		while (filteredElements.size() > 0) {
			GraphItem i = (GraphItem) filteredElements.first();
			i.setVisible(true);
			filteredElements.remove(i);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.jface.viewers.StructuredViewer#getRawChildren(java.lang.Object
	 * )
	 */
	protected Object[] getRawChildren(Object parent) {
		if (parent == getInput()) {
			// get the children from the model.
			LinkedList<Object> children = new LinkedList<Object>();
			if (getGraphControl() != null) {
				List connections = getGraphControl().getConnections();
				List nodes = getGraphControl().getNodes();
				for (Iterator i = connections.iterator(); i.hasNext();) {
					GraphConnection c = (GraphConnection) i.next();
					if (c.getData() != null) {
						children.add(c.getData());
					}
				}
				for (Iterator i = nodes.iterator(); i.hasNext();) {
					GraphNode n = (GraphNode) i.next();
					if (n.getData() != null) {
						children.add(n.getData());
					}
				}
				return children.toArray();
			}
		}
		return super.getRawChildren(parent);
	}

	/**
	 * 
	 */
	public void reveal(Object element) {
		Widget[] items = this.findItems(element);

		Point location = null;
		for (int i = 0; i < items.length; i++) {
			Widget item = items[i];
			if (item instanceof GraphNode) {
				GraphNode graphModelNode = (GraphNode) item;
				graphModelNode.highlight();
				location = getTopLeftBoundary(graphModelNode.getLocation(),
						location);
			} else if (item instanceof GraphConnection) {
				GraphConnection graphModelConnection = (GraphConnection) item;
				graphModelConnection.highlight();
				location = getTopLeftBoundary(graphModelConnection.getSource()
						.getLocation(), location);
				location = getTopLeftBoundary(graphModelConnection
						.getDestination().getLocation(), location);
			}
		}
		// Scrolling to new location
		if (location != null) {
			getGraphControl().scrollSmoothTo(location.x, location.y);
		}
	}

	/**
	 * Calculates the top left corner of the boundary of a single point or two
	 * points.
	 * 
	 * @param point1
	 *            must not be null
	 * @param point2
	 *            may be null - in that case, point1 is returned
	 */
	private Point getTopLeftBoundary(Point point1, Point point2) {
		if (point2 != null) {
			return new Point(Math.min(point1.x, point2.x), Math.min(point1.y,
					point2.y));
		} else {
			return point1;
		}
	}

	public void unReveal(Object element) {
		Widget[] items = this.findItems(element);
		for (int i = 0; i < items.length; i++) {
			Widget item = items[i];
			if (item instanceof GraphNode) {
				GraphNode graphModelNode = (GraphNode) item;
				graphModelNode.unhighlight();
			} else if (item instanceof GraphConnection) {
				GraphConnection graphModelConnection = (GraphConnection) item;
				graphModelConnection.unhighlight();
			}
		}
	}

	/**
	 * Applies the viewers layouts.
	 * 
	 */
	public abstract void applyLayout();

	/**
	 * Removes the given connection object from the layout algorithm and the
	 * model.
	 * 
	 * @param connection
	 */
	public void removeRelationship(Object connection) {
		GraphConnection relationship = (GraphConnection) connectionsMap
				.get(connection);

		if (relationship != null) {
			// remove the relationship from the model
			relationship.dispose();
			connectionsMap.remove(connection);
		}
	}

	/**
	 * Creates a new node and adds it to the graph. If it already exists nothing
	 * happens.
	 * 
	 * @param newNode
	 */
	public void addNode(Object element) {
		if (nodesMap.get(element) == null) {
			// create the new node
			getFactory().createNode(getGraphControl(), element);

		}
	}

	/**
	 * Removes the given element from the layout algorithm and the model.
	 * 
	 * @param element
	 *            The node element to remove.
	 */
	public void removeNode(Object element) {
		GraphNode node = (GraphNode) nodesMap.get(element);

		if (node != null) {
			// remove the node and it's connections from the model
			node.dispose();
			nodesMap.remove(element);
		}
	}

	/**
	 * Creates a new relationship between the source node and the destination
	 * node. If either node doesn't exist then it will be created.
	 * 
	 * @param connection
	 *            The connection data object.
	 * @param srcNode
	 *            The source node data object.
	 * @param destNode
	 *            The destination node data object.
	 */
	public void addRelationship(Object connection, Object srcNode,
			Object destNode) {
		// create the new relationship
		IStylingGraphModelFactory modelFactory = getFactory();
		modelFactory.createConnection(getGraphControl(), connection, srcNode,
				destNode);

	}

	/**
	 * Adds a new relationship given the connection. It will use the content
	 * provider to determine the source and destination nodes.
	 * 
	 * @param connection
	 *            The connection data object.
	 */
	public void addRelationship(Object connection) {
		IStylingGraphModelFactory modelFactory = getFactory();
		if (connectionsMap.get(connection) == null) {
			if (modelFactory.getContentProvider() instanceof IGraphContentProvider) {
				IGraphContentProvider content = ((IGraphContentProvider) modelFactory
						.getContentProvider());
				Object source = content.getSource(connection);
				Object dest = content.getDestination(connection);
				// create the new relationship
				modelFactory.createConnection(getGraphControl(), connection,
						source, dest);
			} else {
				throw new UnsupportedOperationException();
			}
		}
	}

	/**
	 * Converts the list of GraphModelConnection objects into an array and
	 * returns it.
	 * 
	 * @return GraphModelConnection[]
	 */
	protected GraphConnection[] getConnectionsArray(GraphWidget graph) {
		GraphConnection[] connsArray = new GraphConnection[graph
				.getConnections().size()];
		connsArray = (GraphConnection[]) graph.getConnections().toArray(
				connsArray);
		return connsArray;
	}

	/**
	 * Converts the list of GraphModelNode objects into an array an returns it.
	 * 
	 * @return GraphModelNode[]
	 */
	protected GraphNode[] getNodesArray(GraphWidget graph) {
		GraphNode[] nodesArray = new GraphNode[graph.getNodes().size()];
		nodesArray = (GraphNode[]) graph.getNodes().toArray(nodesArray);
		return nodesArray;
	}

}
