Example of code that triggers a warning:

```ada
with System;
procedure P (A : System.Address) is
   Y : Integer with Address => A;
begin
   null;
end P;
```

This warning is issued on the use of an `Address` aspect or representation
clause on a variable, when the value of the `Address` expression is
_imprecisely supported_ as defined in [section 7.3.7.2 of SPARK Reference
Manual], because it is not a reference to the `Address` attribute on a part of
a standalone object or constant.

In such a case, GNATprove cannot follow potential aliasing between the variable
whose address is specified, and other variables in the program. As a result, if
the variable is aliased, the user should verify that writes through the
variable are taken into account by GNATprove when its aliases are read, and
similarly that writes through its aliases are taken into account by GNATprove
when the variable is read. In addition, such writes should not lead to reads of
invalid values. GNATprove outputs additional explanations to the warning
message, listing the assumptions made by analysis in each particular case:

- "no concurrent accesses to non-atomic object": GNATprove assumes here that,
  as the variable is not atomic, it is not accessed concurrently.

- "correct volatile properties": GNATprove assumes here that, as the variable
  is not volatile or does not have all the volatile properties set, these
  choices are correct with respect to uses of the variable in the program.

- "no writes to or through a potential alias": GNATprove assumes here that
  there are no writes to the variable or its potential aliases whose effects
  would not be visible to the analysis.

- "valid reads": GNATprove assumes here that no write through potential aliases
  can lead to reading an invalid value for the variable.

A possible fix is to mark such variables as volatile with the property
`Async_Writers`, so that GNATprove assumes that the value of the variable can
change at any time. For a local variable inside a subprogram like `Y` above,
directly marking the variable as volatile is not allowed in SPARK, so it must
be moved first to library level.

If the variable is not aliased, or after applying the required fixes above, the
warning can be justified with pragma `Warnings` with value `Off`:

```ada
with System;
procedure P (A : System.Address) is
   pragma Warnings (GNATprove, Off,
     "address specification on ""Y"" is imprecisely supported: assuming no concurrent accesses to non-atomic object and no writes to or through a potential alias");
   Y : Integer with Address => A;
   pragma Warnings (GNATprove, On,
     "address specification on ""Y"" is imprecisely supported: assuming no concurrent accesses to non-atomic object and no writes to or through a potential alias");
begin
   null;
end P;

[section 7.3.7.2 of SPARK Reference Manual]:
https://docs.adacore.com/live/wave/spark2014/html/spark2014_ug/en/source/how_to_use_gnatprove_in_a_team.html#complete-list-of-assumptions
