// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief B -> c l nu moments
  class CLEOII_2004_I647287 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(CLEOII_2004_I647287);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // projections
      declare(UnstableParticles(Cuts::abspid==511 ||
				Cuts::abspid==521),"UFS");
      // spin weighted average D and D* mass
      _mD2 = sqr(0.125*(1.86966+1.86484)+0.375*(2.00685+2.01026));
      // histograms
      for(unsigned int ix=0;ix<2;++ix) {
	for(unsigned int iy=0;iy<4;++iy) {
	  if(iy%2==1) book(_p[ix][iy],"TMP/p_"+toString(ix)+"_"+toString(iy),
			   refData(1,1+iy,1+ix));
	  else        book(_p[ix][iy],1,+1+iy,1+ix);
	}
      }
      book(_p_dist,2,1,1);
    }

    void findDecayProducts(Particle parent, Particles & em, Particles & ep,
			   Particles & nue, Particles & nueBar, bool & charm) {
      for(const Particle & p : parent.children()) {
	if(PID::isCharmHadron(p.pid())) {
	  charm=true;
	}
	else if(p.pid() == PID::EMINUS || p.pid()==PID::MUON) {
	  em.push_back(p);
	}
	else if(p.pid() == PID::EPLUS || p.pid()==PID::ANTIMUON) {
	  ep.push_back(p);
	}
	else if(p.pid() == PID::NU_E  || p.pid()==PID::NU_MU) {
	  nue.push_back(p);
	}
	else if(p.pid() == PID::NU_EBAR || p.pid()==PID::NU_MUBAR) {
	  nueBar.push_back(p);
	}
	else if(PID::isBottomHadron(p.pid())) {
	  findDecayProducts(p,em,ep,nue,nueBar,charm);
	}
	else if(!PID::isHadron(p.pid())) {
	  findDecayProducts(p,em,ep,nue,nueBar,charm);
	}
      }
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const double Ecut[2]={1.,1.5};
      // find and loop over Upslion(4S)
      for (const Particle& p : apply<UnstableParticles>(event, "UFS").particles()) {
	if(p.children().empty() ||
	   (p.children().size()==1 && p.children()[1].abspid()==p.abspid()))
	  continue;
	// find decay products
	bool charm = false;
	Particles em,ep,nue,nueBar;
	findDecayProducts(p,em,ep,nue,nueBar,charm);
	if(!charm) continue;
	FourMomentum pl,pnu;
	if(em.size()==1 && nueBar.size()==1 && em[0].pid()+1==-nueBar[0].pid()) {
	  pl  = em[0].momentum();
	  pnu = nueBar[0].momentum();
	}
	else if(ep.size()==1 && nue.size()==1 && nue[0].pid()==-ep[0].pid()+1) {
	  pl  = ep[0].momentum();
	  pnu = nue[0].momentum();
       	}
	else
	  continue;
	// boost to rest frame
	LorentzTransform boost = LorentzTransform::mkFrameTransformFromBeta(p.momentum().betaVec());
	double q2 = (pl+pnu).mass2();
	FourMomentum pX = boost.transform(p.momentum()-pl-pnu);
	pl = boost.transform(pl);
	double mX2  = pX.mass2();
	for(unsigned int ix=0;ix<2;++ix) {
	  if(pl.E()>Ecut[ix]) {
	    _p[ix][0]->fill(0.5,mX2-_mD2);
	    _p[ix][1]->fill(0.5,sqr(mX2));
	    _p[ix][2]->fill(0.5,q2);
	    _p[ix][3]->fill(0.5,sqr(q2));
	  }
	}
	for(const auto & bin : _p_dist->bins())
	  if(bin.xMin()<pl.E()) _p_dist->fill(bin.xMid(),mX2-_mD2);
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      for(unsigned int ix=0;ix<2;++ix) {
	// compute <(mx2-<mx2>)^2> = <mx4>-<mx2>^2
	Scatter2DPtr tmp;
	book(tmp,1,2,1+ix);
	for(unsigned int iy=0;iy<_p[ix][0]->numBins();++iy) {
	  double x  = 0.5*(_p[ix][0]->xEdges()[iy+1]+_p[ix][0]->xEdges()[iy]);
	  double dx = 0.5*(_p[ix][0]->xEdges()[iy+1]-_p[ix][0]->xEdges()[iy]);
	  double value = _p[ix][1]->bins()[iy].mean()-sqr(_p[ix][0]->bins()[iy].mean()+_mD2);
	  double error = value*sqrt(sqr(_p[ix][1]->bins()[iy].relErr())+4.*sqr(_p[ix][0]->bins()[iy].stdErr()/(_p[ix][0]->bins()[iy].mean()+_mD2)));
	  tmp->addPoint(x,value,make_pair(dx,dx),make_pair(error,error));
	}
	// compute <(q2-<q2>)^2> = <q4>-<q2>^2
	book(tmp,1,4,1+ix);
	for(unsigned int iy=0;iy<_p[ix][2]->numBins();++iy) {
	  double x  = 0.5*(_p[ix][2]->xEdges()[iy+1]+_p[ix][2]->xEdges()[iy]);
	  double dx = 0.5*(_p[ix][2]->xEdges()[iy+1]-_p[ix][2]->xEdges()[iy]);
	  double value = _p[ix][3]->bins()[iy].mean()-sqr(_p[ix][2]->bins()[iy].mean());
	  double error = value*sqrt(sqr(_p[ix][3]->bins()[iy].relErr())+4.*sqr(_p[ix][2]->bins()[iy].relErr()));
	  tmp->addPoint(x,value,make_pair(dx,dx),make_pair(error,error));
	}
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    Profile1DPtr _p[2][4],_p_dist;
    double _mD2;
    /// @}


  };


  RIVET_DECLARE_PLUGIN(CLEOII_2004_I647287);

}
