pub const ext = @import("ext.zig");
const gcr = @This();

const std = @import("std");
const compat = @import("compat");
const gio = @import("gio2");
const gobject = @import("gobject2");
const glib = @import("glib2");
const gmodule = @import("gmodule2");
const gck = @import("gck2");
/// Represents a chain of certificates, normally used to
/// validate the trust in a certificate. An X.509 certificate chain has one
/// endpoint certificate (the one for which trust is being verified) and then
/// in turn the certificate that issued each previous certificate in the chain.
///
/// This functionality is for building of certificate chains not for validating
/// them. Use your favorite crypto library to validate trust in a certificate
/// chain once its built.
///
/// The order of certificates in the chain should be first the endpoint
/// certificates and then the signing certificates.
///
/// Create a new certificate chain with `CertificateChain.new` and then
/// add the certificates with `CertificateChain.add`.
///
/// You can then use `CertificateChain.build` to build the remainder of
/// the chain. This will lookup missing certificates in PKCS`@"11"` modules and
/// also check that each certificate in the chain is the signer of the previous
/// one. If a trust anchor, pinned certificate, or self-signed certificate is
/// found, then the chain is considered built. Any extra certificates are
/// removed from the chain.
///
/// Once the certificate chain has been built, you can access its status
/// through `CertificateChain.getStatus`. The status signifies whether
/// the chain is anchored on a trust root, self-signed, incomplete etc. See
/// `CertificateChainStatus` for information on the various statuses.
///
/// It's important to understand that the building of a certificate chain is
/// merely the first step towards verifying trust in a certificate.
pub const CertificateChain = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gcr.CertificateChainClass;
    f_parent: gobject.Object,
    f_pv: ?*gcr.CertificateChainPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The length of the certificate chain.
        pub const length = struct {
            pub const name = "length";

            pub const Type = c_uint;
        };

        /// The certificate chain status. See `gcr.CertificateChainStatus`
        pub const status = struct {
            pub const name = "status";

            pub const Type = @compileError("no type information available");
        };
    };

    pub const signals = struct {};

    /// Create a new `gcr.CertificateChain`.
    extern fn gcr_certificate_chain_new() *gcr.CertificateChain;
    pub const new = gcr_certificate_chain_new;

    /// Add `certificate` to the chain. The order of certificates in the chain are
    /// important. The first certificate should be the endpoint certificate, and
    /// then come the signers (certificate authorities) each in turn. If a root
    /// certificate authority is present, it should come last.
    ///
    /// Adding a certificate an already built chain (see
    /// `gcr.CertificateChain.build`) resets the type of the certificate chain
    /// to `GCR_CERTIFICATE_CHAIN_UNKNOWN`
    extern fn gcr_certificate_chain_add(p_self: *CertificateChain, p_certificate: *gcr.Certificate) void;
    pub const add = gcr_certificate_chain_add;

    /// Complete a certificate chain. Once a certificate chain has been built
    /// its status can be examined.
    ///
    /// This operation will lookup missing certificates in PKCS`@"11"`
    /// modules and also that each certificate in the chain is the signer of the
    /// previous one. If a trust anchor, pinned certificate, or self-signed certificate
    /// is found, then the chain is considered built. Any extra certificates are
    /// removed from the chain.
    ///
    /// It's important to understand that building of a certificate chain does not
    /// constitute verifying that chain. This is merely the first step towards
    /// trust verification.
    ///
    /// The `purpose` is a string like `GCR_PURPOSE_CLIENT_AUTH` and is the purpose
    /// for which the certificate chain will be used. Trust anchors are looked up
    /// for this purpose. This argument is required.
    ///
    /// The `peer` is usually the host name of the peer whith which this certificate
    /// chain is being used. It is used to look up pinned certificates that have
    /// been stored for this peer. If `NULL` then no pinned certificates will
    /// be considered.
    ///
    /// If the `GCR_CERTIFICATE_CHAIN_NO_LOOKUPS` flag is specified then no
    /// lookups for anchors or pinned certificates are done, and the resulting chain
    /// will be neither anchored or pinned. Additionally no missing certificate
    /// authorities are looked up in PKCS`@"11"`
    ///
    /// This call will block, see `gcr.CertificateChain.buildAsync` for the
    /// asynchronous version.
    extern fn gcr_certificate_chain_build(p_self: *CertificateChain, p_purpose: [*:0]const u8, p_peer: ?[*:0]const u8, p_flags: gcr.CertificateChainFlags, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const build = gcr_certificate_chain_build;

    /// Complete a certificate chain. Once a certificate chain has been built
    /// its status can be examined.
    ///
    /// This will lookup missing certificates in PKCS`@"11"`
    /// modules and also that each certificate in the chain is the signer of the
    /// previous one. If a trust anchor, pinned certificate, or self-signed certificate
    /// is found, then the chain is considered built. Any extra certificates are
    /// removed from the chain.
    ///
    /// It's important to understand that building of a certificate chain does not
    /// constitute verifying that chain. This is merely the first step towards
    /// trust verification.
    ///
    /// The `purpose` is a string like `GCR_PURPOSE_CLIENT_AUTH` and is the purpose
    /// for which the certificate chain will be used. Trust anchors are looked up
    /// for this purpose. This argument is required.
    ///
    /// The `peer` is usually the host name of the peer whith which this certificate
    /// chain is being used. It is used to look up pinned certificates that have
    /// been stored for this peer. If `NULL` then no pinned certificates will
    /// be considered.
    ///
    /// If the `GCR_CERTIFICATE_CHAIN_NO_LOOKUPS` flag is specified then no
    /// lookups for anchors or pinned certificates are done, and the resulting chain
    /// will be neither anchored or pinned. Additionally no missing certificate
    /// authorities are looked up in PKCS`@"11"`
    ///
    /// When the operation is finished, `callback` will be called. You can then call
    /// `gcr.CertificateChain.buildFinish` to get the result of the operation.
    extern fn gcr_certificate_chain_build_async(p_self: *CertificateChain, p_purpose: [*:0]const u8, p_peer: ?[*:0]const u8, p_flags: gcr.CertificateChainFlags, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const buildAsync = gcr_certificate_chain_build_async;

    /// Finishes an asynchronous operation started by
    /// `gcr.CertificateChain.buildAsync`.
    extern fn gcr_certificate_chain_build_finish(p_self: *CertificateChain, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const buildFinish = gcr_certificate_chain_build_finish;

    /// If the certificate chain has been built and is of status
    /// `GCR_CERTIFICATE_CHAIN_ANCHORED`, then this will return the anchor
    /// certificate that was found. This is not necessarily a root certificate
    /// authority. If an intermediate certificate authority in the chain was
    /// found to be anchored, then that certificate will be returned.
    ///
    /// If an anchor is returned it does not mean that the certificate chain has
    /// been verified, but merely that an anchor has been found.
    extern fn gcr_certificate_chain_get_anchor(p_self: *CertificateChain) *gcr.Certificate;
    pub const getAnchor = gcr_certificate_chain_get_anchor;

    /// Get a certificate in the chain. It is an error to call this function
    /// with an invalid index.
    extern fn gcr_certificate_chain_get_certificate(p_self: *CertificateChain, p_index: c_uint) *gcr.Certificate;
    pub const getCertificate = gcr_certificate_chain_get_certificate;

    /// Get the endpoint certificate in the chain. This is always the first
    /// certificate in the chain. The endpoint certificate cannot be anchored.
    extern fn gcr_certificate_chain_get_endpoint(p_self: *CertificateChain) *gcr.Certificate;
    pub const getEndpoint = gcr_certificate_chain_get_endpoint;

    /// Get the length of the certificate chain.
    extern fn gcr_certificate_chain_get_length(p_self: *CertificateChain) c_uint;
    pub const getLength = gcr_certificate_chain_get_length;

    /// Get the status of a certificate chain. If the certificate chain has not
    /// been built, then the status will be `GCR_CERTIFICATE_CHAIN_UNKNOWN`.
    ///
    /// A status of `GCR_CERTIFICATE_CHAIN_ANCHORED` does not mean that the
    /// certificate chain has been verified, but merely that an anchor has been
    /// found.
    extern fn gcr_certificate_chain_get_status(p_self: *CertificateChain) gcr.CertificateChainStatus;
    pub const getStatus = gcr_certificate_chain_get_status;

    extern fn gcr_certificate_chain_get_type() usize;
    pub const getGObjectType = gcr_certificate_chain_get_type;

    extern fn g_object_ref(p_self: *gcr.CertificateChain) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcr.CertificateChain) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CertificateChain, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const CertificateField = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gcr.CertificateFieldClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const label = struct {
            pub const name = "label";

            pub const Type = ?[*:0]u8;
        };

        pub const section = struct {
            pub const name = "section";

            pub const Type = ?*gcr.CertificateSection;
        };

        pub const value = struct {
            pub const name = "value";

            pub const Type = ?*gobject.Value;
        };
    };

    pub const signals = struct {};

    /// Get the display label of the field.
    extern fn gcr_certificate_field_get_label(p_self: *CertificateField) [*:0]const u8;
    pub const getLabel = gcr_certificate_field_get_label;

    /// Get the parent `gcr.CertificateSection`.
    extern fn gcr_certificate_field_get_section(p_self: *CertificateField) *gcr.CertificateSection;
    pub const getSection = gcr_certificate_field_get_section;

    /// Get the value of the field.
    ///
    /// The `value` will have been initialized to the `GType` the value should be
    /// provided in.
    extern fn gcr_certificate_field_get_value(p_self: *CertificateField, p_value: *gobject.Value) c_int;
    pub const getValue = gcr_certificate_field_get_value;

    /// Get the type associated with the value.
    extern fn gcr_certificate_field_get_value_type(p_self: *CertificateField) usize;
    pub const getValueType = gcr_certificate_field_get_value_type;

    extern fn gcr_certificate_field_get_type() usize;
    pub const getGObjectType = gcr_certificate_field_get_type;

    extern fn g_object_ref(p_self: *gcr.CertificateField) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcr.CertificateField) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CertificateField, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An object that allows creation of certificate requests. A certificate
/// request is sent to a certificate authority to request an X.509 certificate.
///
/// Use `CertificateRequest.prepare` to create a blank certificate
/// request for a given private key. Set the common name on the certificate
/// request with `CertificateRequest.setCn`, and then sign the request
/// with `CertificateRequest.completeAsync`.
pub const CertificateRequest = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gcr.CertificateRequestClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The format of the certificate request.
        pub const format = struct {
            pub const name = "format";

            pub const Type = @compileError("no type information available");
        };

        /// The private key that this certificate request is for.
        pub const private_key = struct {
            pub const name = "private-key";

            pub const Type = ?*gck.Object;
        };
    };

    pub const signals = struct {};

    /// Check whether `CertificateRequest` is capable of creating a request
    /// for the given `private_key`.
    extern fn gcr_certificate_request_capable(p_private_key: *gck.Object, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const capable = gcr_certificate_request_capable;

    /// Asynchronously check whether `CertificateRequest` is capable of
    /// creating a request for the given `private_key`.
    extern fn gcr_certificate_request_capable_async(p_private_key: *gck.Object, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const capableAsync = gcr_certificate_request_capable_async;

    /// Get the result for asynchronously check whether `CertificateRequest` is
    /// capable of creating a request for the given `private_key`.
    extern fn gcr_certificate_request_capable_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const capableFinish = gcr_certificate_request_capable_finish;

    /// Create a new certificate request, in the given format for the private key.
    extern fn gcr_certificate_request_prepare(p_format: gcr.CertificateRequestFormat, p_private_key: *gck.Object) *gcr.CertificateRequest;
    pub const prepare = gcr_certificate_request_prepare;

    /// Complete and sign a certificate request, so that it can be encoded
    /// and sent to a certificate authority.
    ///
    /// This call may block as it signs the request using the private key.
    extern fn gcr_certificate_request_complete(p_self: *CertificateRequest, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const complete = gcr_certificate_request_complete;

    /// Asynchronously complete and sign a certificate request, so that it can
    /// be encoded and sent to a certificate authority.
    ///
    /// This call will return immediately and complete later.
    extern fn gcr_certificate_request_complete_async(p_self: *CertificateRequest, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const completeAsync = gcr_certificate_request_complete_async;

    /// Finish an asynchronous operation to complete and sign a certificate
    /// request.
    extern fn gcr_certificate_request_complete_finish(p_self: *CertificateRequest, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const completeFinish = gcr_certificate_request_complete_finish;

    /// Encode the certificate request. It must have been completed with
    /// `CertificateRequest.complete` or
    /// `CertificateRequest.completeAsync`.
    ///
    /// If `textual` is `FALSE`, the output is a DER encoded certificate request.
    ///
    /// If `textual` is `TRUE`, the output is encoded as text. For PKCS`@"10"` requests
    /// this is done using the OpenSSL style PEM encoding.
    extern fn gcr_certificate_request_encode(p_self: *CertificateRequest, p_textual: c_int, p_length: *usize) [*]u8;
    pub const encode = gcr_certificate_request_encode;

    /// Get the format of this certificate request.
    extern fn gcr_certificate_request_get_format(p_self: *CertificateRequest) gcr.CertificateRequestFormat;
    pub const getFormat = gcr_certificate_request_get_format;

    /// Get the private key this certificate request is for.
    extern fn gcr_certificate_request_get_private_key(p_self: *CertificateRequest) *gck.Object;
    pub const getPrivateKey = gcr_certificate_request_get_private_key;

    /// Set the common name encoded in the certificate request.
    extern fn gcr_certificate_request_set_cn(p_self: *CertificateRequest, p_cn: [*:0]const u8) void;
    pub const setCn = gcr_certificate_request_set_cn;

    extern fn gcr_certificate_request_get_type() usize;
    pub const getGObjectType = gcr_certificate_request_get_type;

    extern fn g_object_ref(p_self: *gcr.CertificateRequest) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcr.CertificateRequest) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CertificateRequest, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const CertificateSection = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gcr.CertificateSectionClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        pub const fields = struct {
            pub const name = "fields";

            pub const Type = ?*gio.ListModel;
        };

        pub const flags = struct {
            pub const name = "flags";

            pub const Type = @compileError("no type information available");
        };

        pub const label = struct {
            pub const name = "label";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Get the list of all the fields in this section.
    extern fn gcr_certificate_section_get_fields(p_self: *CertificateSection) *gio.ListModel;
    pub const getFields = gcr_certificate_section_get_fields;

    /// Get the flags.
    extern fn gcr_certificate_section_get_flags(p_self: *CertificateSection) gcr.CertificateSectionFlags;
    pub const getFlags = gcr_certificate_section_get_flags;

    /// Get the displayable label of the section.
    extern fn gcr_certificate_section_get_label(p_self: *CertificateSection) [*:0]const u8;
    pub const getLabel = gcr_certificate_section_get_label;

    extern fn gcr_certificate_section_get_type() usize;
    pub const getGObjectType = gcr_certificate_section_get_type;

    extern fn g_object_ref(p_self: *gcr.CertificateSection) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcr.CertificateSection) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *CertificateSection, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A parser for parsing various types of files or data.
///
/// A `GcrParser` can parse various certificate and key files such as OpenSSL
/// PEM files, DER encoded certifictes, PKCS`@"8"` keys and so on. Each various
/// format is identified by a value in the `DataFormat` enumeration.
///
/// In order to parse data, a new parser is created with `gcr.Parser.new` and
/// then the `Parser.signals.authenticate` and `Parser.signals.parsed` signals
/// should be connected to. Data is then fed to the parser via
/// `Parser.parseData` or `Parser.parseStream`.
///
/// During the `Parser.signals.parsed` signal the attributes that make up the
/// currently parsed item can be retrieved using the
/// `Parser.getParsedAttributes` function.
pub const Parser = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gcr.ParserClass;
    f_parent: gobject.Object,
    f_pv: ?*gcr.ParserPrivate,

    pub const virtual_methods = struct {
        /// The default handler for the authenticate signal.
        pub const authenticate = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_count: c_int) c_int {
                return gobject.ext.as(Parser.Class, p_class).f_authenticate.?(gobject.ext.as(Parser, p_self), p_count);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_count: c_int) callconv(.C) c_int) void {
                gobject.ext.as(Parser.Class, p_class).f_authenticate = @ptrCast(p_implementation);
            }
        };

        /// The default handler for the parsed signal.
        pub const parsed = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Parser.Class, p_class).f_parsed.?(gobject.ext.as(Parser, p_self));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Parser.Class, p_class).f_parsed = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// Get the attributes that make up the currently parsed item. This is
        /// generally only valid during a `Parser.signals.parsed` signal.
        pub const parsed_attributes = struct {
            pub const name = "parsed-attributes";

            pub const Type = ?*gck.Attributes;
        };

        /// The description of the type of the currently parsed item. This is generally
        /// only valid during a `Parser.signals.parsed` signal.
        pub const parsed_description = struct {
            pub const name = "parsed-description";

            pub const Type = ?[*:0]u8;
        };

        /// The label of the currently parsed item. This is generally
        /// only valid during a `Parser.signals.parsed` signal.
        pub const parsed_label = struct {
            pub const name = "parsed-label";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {
        /// This signal is emitted when an item needs to be unlocked or decrypted before
        /// it can be parsed. The `count` argument specifies the number of times
        /// the signal has been emitted for a given item. This can be used to
        /// display a message saying the previous password was incorrect.
        ///
        /// Typically the `gcr.Parser.addPassword` function is called in
        /// response to this signal.
        ///
        /// If `FALSE` is returned, then the authentication was not handled. If
        /// no handlers return `TRUE` then the item is not parsed and an error
        /// with the code `GCR_ERROR_CANCELLED` will be raised.
        pub const authenticate = struct {
            pub const name = "authenticate";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), p_count: c_int, P_Data) callconv(.C) c_int, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Parser, p_instance))),
                    gobject.signalLookup("authenticate", Parser.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };

        /// This signal is emitted when an item is sucessfully parsed. To access
        /// the information about the item use the `gcr.Parser.getParsedLabel`,
        /// `gcr.Parser.getParsedAttributes` and `gcr.Parser.getParsedDescription`
        /// functions.
        pub const parsed = struct {
            pub const name = "parsed";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Parser, p_instance))),
                    gobject.signalLookup("parsed", Parser.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Create a new `gcr.Parser`
    extern fn gcr_parser_new() *gcr.Parser;
    pub const new = gcr_parser_new;

    /// Add a password to the set of passwords to try when parsing locked or encrypted
    /// items. This is usually called from the `Parser.signals.authenticate` signal.
    extern fn gcr_parser_add_password(p_self: *Parser, p_password: ?[*:0]const u8) void;
    pub const addPassword = gcr_parser_add_password;

    /// Disable parsing of the given format. Use `GCR_FORMAT_ALL` to disable all the formats.
    extern fn gcr_parser_format_disable(p_self: *Parser, p_format: gcr.DataFormat) void;
    pub const formatDisable = gcr_parser_format_disable;

    /// Enable parsing of the given format. Use `GCR_FORMAT_ALL` to enable all the formats.
    extern fn gcr_parser_format_enable(p_self: *Parser, p_format: gcr.DataFormat) void;
    pub const formatEnable = gcr_parser_format_enable;

    /// Check whether the given format is supported by the parser.
    extern fn gcr_parser_format_supported(p_self: *Parser, p_format: gcr.DataFormat) c_int;
    pub const formatSupported = gcr_parser_format_supported;

    /// Get the filename of the parser item.
    extern fn gcr_parser_get_filename(p_self: *Parser) [*:0]const u8;
    pub const getFilename = gcr_parser_get_filename;

    /// Get the currently parsed item
    extern fn gcr_parser_get_parsed(p_self: *Parser) *gcr.Parsed;
    pub const getParsed = gcr_parser_get_parsed;

    /// Get the attributes which make up the currently parsed item. This is generally
    /// only valid during the `Parser.signals.parsed` signal.
    extern fn gcr_parser_get_parsed_attributes(p_self: *Parser) ?*gck.Attributes;
    pub const getParsedAttributes = gcr_parser_get_parsed_attributes;

    /// Get the raw data block that represents this parsed object. This is only
    /// valid during the `Parser.signals.parsed` signal.
    extern fn gcr_parser_get_parsed_block(p_self: *Parser, p_n_block: *usize) ?[*]const u8;
    pub const getParsedBlock = gcr_parser_get_parsed_block;

    /// Get the raw data block that represents this parsed object. This is only
    /// valid during the `Parser.signals.parsed` signal.
    extern fn gcr_parser_get_parsed_bytes(p_self: *Parser) *glib.Bytes;
    pub const getParsedBytes = gcr_parser_get_parsed_bytes;

    /// Get a description for the type of the currently parsed item. This is generally
    /// only valid during the `Parser.signals.parsed` signal.
    extern fn gcr_parser_get_parsed_description(p_self: *Parser) ?[*:0]const u8;
    pub const getParsedDescription = gcr_parser_get_parsed_description;

    /// Get the format of the raw data block that represents this parsed object.
    /// This corresponds with the data returned from
    /// `Parser.getParsedBlock`.
    ///
    /// This is only valid during the `Parser.signals.parsed` signal.
    extern fn gcr_parser_get_parsed_format(p_self: *Parser) gcr.DataFormat;
    pub const getParsedFormat = gcr_parser_get_parsed_format;

    /// Get the label of the currently parsed item. This is generally only valid
    /// during the `Parser.signals.parsed` signal.
    extern fn gcr_parser_get_parsed_label(p_self: *Parser) ?[*:0]const u8;
    pub const getParsedLabel = gcr_parser_get_parsed_label;

    /// Parse the data. The `Parser.signals.parsed` and
    /// `Parser.signals.authenticate` signals may fire during the parsing.
    extern fn gcr_parser_parse_bytes(p_self: *Parser, p_data: *glib.Bytes, p_error: ?*?*glib.Error) c_int;
    pub const parseBytes = gcr_parser_parse_bytes;

    /// Parse the data. The `Parser.signals.parsed` and `Parser.signals.authenticate`
    /// signals may fire during the parsing.
    ///
    /// A copy of the data will be made. Use `Parser.parseBytes` to avoid
    /// this.
    extern fn gcr_parser_parse_data(p_self: *Parser, p_data: [*]const u8, p_n_data: usize, p_error: ?*?*glib.Error) c_int;
    pub const parseData = gcr_parser_parse_data;

    /// Parse items from the data in a `gio.InputStream`. This function may block while
    /// reading from the input stream. Use `Parser.parseStreamAsync` for
    /// a non-blocking variant.
    ///
    /// The `Parser.signals.parsed` and `Parser.signals.authenticate` signals
    /// may fire during the parsing.
    extern fn gcr_parser_parse_stream(p_self: *Parser, p_input: *gio.InputStream, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const parseStream = gcr_parser_parse_stream;

    /// Parse items from the data in a `gio.InputStream`. This function completes
    /// asyncronously and doesn't block.
    ///
    /// The `Parser.signals.parsed` and `Parser.signals.authenticate` signals
    /// may fire during the parsing.
    extern fn gcr_parser_parse_stream_async(p_self: *Parser, p_input: *gio.InputStream, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const parseStreamAsync = gcr_parser_parse_stream_async;

    /// Complete an operation to parse a stream.
    extern fn gcr_parser_parse_stream_finish(p_self: *Parser, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const parseStreamFinish = gcr_parser_parse_stream_finish;

    /// Sets the filename of the parser item.
    extern fn gcr_parser_set_filename(p_self: *Parser, p_filename: ?[*:0]const u8) void;
    pub const setFilename = gcr_parser_set_filename;

    extern fn gcr_parser_get_type() usize;
    pub const getGObjectType = gcr_parser_get_type;

    extern fn g_object_ref(p_self: *gcr.Parser) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcr.Parser) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Parser, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A certificate loaded from a PKCS`@"11"` storage.
/// It is also a valid `gck.Object` and can be used as such.
///
/// Use `gcr.Pkcs11Certificate.lookupIssuer` to lookup the issuer of a given
/// certificate in the PKCS`@"11"` store.
///
/// Various common PKCS`@"11"` certificate attributes are automatically loaded and
/// are available via `gcr.Pkcs11Certificate.getAttributes`.
pub const Pkcs11Certificate = extern struct {
    pub const Parent = gck.Object;
    pub const Implements = [_]type{gcr.Certificate};
    pub const Class = gcr.Pkcs11CertificateClass;
    f_parent: gck.Object,
    f_pv: ?*gcr.Pkcs11CertificatePrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Automatically loaded attributes for this certificate.
        pub const attributes = struct {
            pub const name = "attributes";

            pub const Type = ?*gck.Attributes;
        };
    };

    pub const signals = struct {};

    /// Lookup a the issuer of a `certificate` in the PKCS`@"11"` storage. The
    /// lookup is done using the issuer DN of the certificate. No certificate chain
    /// verification is done. Use a crypto library to make trust decisions.
    ///
    /// This call may block, see `gcr.Pkcs11Certificate.lookupIssuer` for the
    /// non-blocking version.
    ///
    /// Will return `NULL` if no issuer certificate is found. Use `error` to determine
    /// if an error occurred.
    extern fn gcr_pkcs11_certificate_lookup_issuer(p_certificate: *gcr.Certificate, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gcr.Certificate;
    pub const lookupIssuer = gcr_pkcs11_certificate_lookup_issuer;

    /// Lookup a the issuer of a `certificate` in the PKCS`@"11"` storage. The
    /// lookup is done using the issuer DN of the certificate. No certificate chain
    /// verification is done. Use a crypto library to make trust decisions.
    ///
    /// When the operation is finished, callback will be called. You can then call
    /// `gcr.Pkcs11Certificate.lookupIssuerFinish` to get the result of the
    /// operation.
    extern fn gcr_pkcs11_certificate_lookup_issuer_async(p_certificate: *gcr.Certificate, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const lookupIssuerAsync = gcr_pkcs11_certificate_lookup_issuer_async;

    /// Finishes an asynchronous operation started by
    /// `gcr.Pkcs11Certificate.lookupIssuerAsync`.
    ///
    /// Will return `NULL` if no issuer certificate is found. Use `error` to determine
    /// if an error occurred.
    extern fn gcr_pkcs11_certificate_lookup_issuer_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*gcr.Certificate;
    pub const lookupIssuerFinish = gcr_pkcs11_certificate_lookup_issuer_finish;

    /// Lookup a certificate in the PKCS`@"11"` storage by the given URI.
    ///
    /// This call may block, see `gcr.Pkcs11Certificate.newFromUriAsync` for the
    /// non-blocking version.
    ///
    /// Will return `NULL` if no certificate is found. Use `error` to determine
    /// if an error occurred.
    extern fn gcr_pkcs11_certificate_new_from_uri(p_pkcs11_uri: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gcr.Certificate;
    pub const newFromUri = gcr_pkcs11_certificate_new_from_uri;

    /// Lookup a certificate in the PKCS`@"11"` storage by the given URI.
    ///
    /// When the operation is finished, callback will be called. You can then call
    /// `gcr.Pkcs11Certificate.newFromUriFinish` to get the result of the
    /// operation.
    extern fn gcr_pkcs11_certificate_new_from_uri_async(p_pkcs11_uri: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const newFromUriAsync = gcr_pkcs11_certificate_new_from_uri_async;

    /// Finishes an asynchronous operation started by
    /// `gcr.Pkcs11Certificate.newFromUriAsync`.
    ///
    /// Will return `NULL` if no certificate is found. Use `error` to determine
    /// if an error occurred.
    extern fn gcr_pkcs11_certificate_new_from_uri_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*gcr.Certificate;
    pub const newFromUriFinish = gcr_pkcs11_certificate_new_from_uri_finish;

    /// Access the automatically loaded attributes for this certificate.
    extern fn gcr_pkcs11_certificate_get_attributes(p_self: *Pkcs11Certificate) *gck.Attributes;
    pub const getAttributes = gcr_pkcs11_certificate_get_attributes;

    extern fn gcr_pkcs11_certificate_get_type() usize;
    pub const getGObjectType = gcr_pkcs11_certificate_get_type;

    extern fn g_object_ref(p_self: *gcr.Pkcs11Certificate) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcr.Pkcs11Certificate) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Pkcs11Certificate, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Allows exchange of secrets between two processes on the same system without
/// exposing those secrets to things like loggers, non-pageable memory etc.
///
/// This does not protect against active attacks like MITM attacks.
///
/// Each side creates a secret exchange object, and one of the sides calls
/// `SecretExchange.begin`. This creates a string, which should be passed
/// to the other side. Each side passes the strings it receives into
/// `SecretExchange.receive`.
///
/// In order to send a reply (either with or without a secret) use
/// `SecretExchange.send`. A side must have successfully called
/// `SecretExchange.receive` before it can use
/// `SecretExchange.send`.
///
/// The secret exchange objects can be used for multiple iterations of the
/// conversation, or for just one request/reply. The only limitation being that
/// the initial request cannot contain a secret.
///
/// Caveat: Information about the approximate length (rounded up to the nearest
/// 16 bytes) may be leaked. If this is considered inacceptable, do not use
/// `SecretExchange`.
pub const SecretExchange = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gcr.SecretExchangeClass;
    f_parent: gobject.Object,
    f_pv: ?*gcr.SecretExchangePrivate,

    pub const virtual_methods = struct {
        pub const decrypt_transport_data = struct {
            pub fn call(p_class: anytype, p_exchange: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_allocator: gck.Allocator, p_cipher_text: *const u8, p_n_cipher_text: usize, p_parameter: *const u8, p_n_parameter: usize, p_plain_text: **u8, p_n_plain_text: *usize) c_int {
                return gobject.ext.as(SecretExchange.Class, p_class).f_decrypt_transport_data.?(gobject.ext.as(SecretExchange, p_exchange), p_allocator, p_cipher_text, p_n_cipher_text, p_parameter, p_n_parameter, p_plain_text, p_n_plain_text);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_exchange: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_allocator: gck.Allocator, p_cipher_text: *const u8, p_n_cipher_text: usize, p_parameter: *const u8, p_n_parameter: usize, p_plain_text: **u8, p_n_plain_text: *usize) callconv(.C) c_int) void {
                gobject.ext.as(SecretExchange.Class, p_class).f_decrypt_transport_data = @ptrCast(p_implementation);
            }
        };

        pub const derive_transport_key = struct {
            pub fn call(p_class: anytype, p_exchange: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_peer: *const u8, p_n_peer: usize) c_int {
                return gobject.ext.as(SecretExchange.Class, p_class).f_derive_transport_key.?(gobject.ext.as(SecretExchange, p_exchange), p_peer, p_n_peer);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_exchange: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_peer: *const u8, p_n_peer: usize) callconv(.C) c_int) void {
                gobject.ext.as(SecretExchange.Class, p_class).f_derive_transport_key = @ptrCast(p_implementation);
            }
        };

        pub const encrypt_transport_data = struct {
            pub fn call(p_class: anytype, p_exchange: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_allocator: gck.Allocator, p_plain_text: *const u8, p_n_plain_text: usize, p_parameter: **u8, p_n_parameter: *usize, p_cipher_text: **u8, p_n_cipher_text: *usize) c_int {
                return gobject.ext.as(SecretExchange.Class, p_class).f_encrypt_transport_data.?(gobject.ext.as(SecretExchange, p_exchange), p_allocator, p_plain_text, p_n_plain_text, p_parameter, p_n_parameter, p_cipher_text, p_n_cipher_text);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_exchange: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_allocator: gck.Allocator, p_plain_text: *const u8, p_n_plain_text: usize, p_parameter: **u8, p_n_parameter: *usize, p_cipher_text: **u8, p_n_cipher_text: *usize) callconv(.C) c_int) void {
                gobject.ext.as(SecretExchange.Class, p_class).f_encrypt_transport_data = @ptrCast(p_implementation);
            }
        };

        pub const generate_exchange_key = struct {
            pub fn call(p_class: anytype, p_exchange: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_scheme: [*:0]const u8, p_public_key: **u8, p_n_public_key: *usize) c_int {
                return gobject.ext.as(SecretExchange.Class, p_class).f_generate_exchange_key.?(gobject.ext.as(SecretExchange, p_exchange), p_scheme, p_public_key, p_n_public_key);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_exchange: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_scheme: [*:0]const u8, p_public_key: **u8, p_n_public_key: *usize) callconv(.C) c_int) void {
                gobject.ext.as(SecretExchange.Class, p_class).f_generate_exchange_key = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The protocol being used for the exchange.
        ///
        /// Will be `NULL` if no protocol was specified when creating this object,
        /// and either `SecretExchange.begin` or `SecretExchange.receive`
        /// have not been called successfully.
        pub const protocol = struct {
            pub const name = "protocol";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Create a new secret exchange object.
    ///
    /// Specify a protocol of `NULL` to allow any protocol. This is especially
    /// relevant on the side of the exchange that does not call
    /// `SecretExchange.begin`, that is the originator. Currently the only
    /// protocol supported is `GCR_SECRET_EXCHANGE_PROTOCOL_1`.
    extern fn gcr_secret_exchange_new(p_protocol: ?[*:0]const u8) *gcr.SecretExchange;
    pub const new = gcr_secret_exchange_new;

    /// Begin the secret exchange. The resulting string should be sent to the other
    /// side of the exchange. The other side should use `SecretExchange.receive`
    /// to process the string.
    extern fn gcr_secret_exchange_begin(p_self: *SecretExchange) [*:0]u8;
    pub const begin = gcr_secret_exchange_begin;

    /// Will return `NULL` if no protocol was specified, and either
    /// `SecretExchange.begin` or `SecretExchange.receive` have not
    /// been called successfully.
    extern fn gcr_secret_exchange_get_protocol(p_self: *SecretExchange) [*:0]const u8;
    pub const getProtocol = gcr_secret_exchange_get_protocol;

    /// Returns the last secret received. If no secret has yet been received this
    /// will return `NULL`. The string is owned by the `gcr.SecretExchange` object
    /// and will be valid until the next time that `gcr.SecretExchange.receive`
    /// is called on this object, or the object is destroyed.
    ///
    /// Depending on the secret passed into the other side of the secret exchange,
    /// the result may be a binary string. It does however have a null terminator,
    /// so if you're certain that it is does not contain arbitrary binary data,
    /// it can be used as a string.
    extern fn gcr_secret_exchange_get_secret(p_self: *SecretExchange, p_secret_len: ?*usize) [*]const u8;
    pub const getSecret = gcr_secret_exchange_get_secret;

    /// Receive a string from the other side of secret exchange. This string will
    /// have been created by `SecretExchange.begin` or
    /// `SecretExchange.send`.
    ///
    /// After this call completes successfully the value returned from
    /// `gcr.SecretExchange.getSecret` will have changed.
    extern fn gcr_secret_exchange_receive(p_self: *SecretExchange, p_exchange: [*:0]const u8) c_int;
    pub const receive = gcr_secret_exchange_receive;

    /// Send a reply to the other side of the secret exchange, optionally sending a
    /// secret.
    ///
    /// `SecretExchange.receive` must have been successfully called at least
    /// once on this object. In other words this object must have received data
    /// from the other side of the secret exchange, before we can send a secret.
    extern fn gcr_secret_exchange_send(p_self: *SecretExchange, p_secret: ?[*:0]const u8, p_secret_len: isize) [*:0]u8;
    pub const send = gcr_secret_exchange_send;

    extern fn gcr_secret_exchange_get_type() usize;
    pub const getGObjectType = gcr_secret_exchange_get_type;

    extern fn g_object_ref(p_self: *gcr.SecretExchange) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcr.SecretExchange) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SecretExchange, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An implementation of `Certificate` which loads a certificate from DER
/// data already located in memory.
///
/// To create an object, use the `SimpleCertificate.new` or
/// `SimpleCertificate.newStatic` functions.
pub const SimpleCertificate = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{gcr.Certificate};
    pub const Class = gcr.SimpleCertificateClass;
    f_parent: gobject.Object,
    f_pv: ?*gcr.SimpleCertificatePrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Create a new `gcr.SimpleCertificate` for the raw DER data. The `data` memory is
    /// copied so you can dispose of it after this function returns.
    extern fn gcr_simple_certificate_new(p_data: [*]const u8, p_n_data: usize) *gcr.SimpleCertificate;
    pub const new = gcr_simple_certificate_new;

    /// Create a new `gcr.SimpleCertificate` for the raw DER data. The `data` memory is
    /// not copied and must persist until the `gcr.SimpleCertificate` object is
    /// destroyed.
    extern fn gcr_simple_certificate_new_static(p_data: [*]const u8, p_n_data: usize) *gcr.SimpleCertificate;
    pub const newStatic = gcr_simple_certificate_new_static;

    extern fn gcr_simple_certificate_get_type() usize;
    pub const getGObjectType = gcr_simple_certificate_get_type;

    extern fn g_object_ref(p_self: *gcr.SimpleCertificate) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcr.SimpleCertificate) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SimpleCertificate, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// When used as the setup function while spawning an ssh command like ssh-add
/// or ssh, this allows callbacks for passwords on the provided interaction.
pub const SshAskpass = opaque {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gcr.SshAskpassClass;
    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The interaction used to prompt for passwords.
        pub const interaction = struct {
            pub const name = "interaction";

            pub const Type = ?*gio.TlsInteraction;
        };
    };

    pub const signals = struct {};

    /// Use this function as a callback setup function passed to `glib.spawnSync`,
    /// `glib.spawnAsync`, `glib.spawnAsyncWithPipes`.
    extern fn gcr_ssh_askpass_child_setup(p_askpass: ?*anyopaque) void;
    pub const childSetup = gcr_ssh_askpass_child_setup;

    /// Create a new GcrSshAskpass object which can be used to spawn an
    /// ssh command and prompt for any necessary passwords.
    ///
    /// Use the `gcr.SshAskpass.childSetup` function as a callback with
    /// `glib.spawnSync`, `glib.spawnAsync` or `glib.spawnAsyncWithPipes`.
    extern fn gcr_ssh_askpass_new(p_interaction: *gio.TlsInteraction) *gcr.SshAskpass;
    pub const new = gcr_ssh_askpass_new;

    /// Get the interaction associated with this object.
    extern fn gcr_ssh_askpass_get_interaction(p_self: *SshAskpass) *gio.TlsInteraction;
    pub const getInteraction = gcr_ssh_askpass_get_interaction;

    extern fn gcr_ssh_askpass_get_type() usize;
    pub const getGObjectType = gcr_ssh_askpass_get_type;

    extern fn g_object_ref(p_self: *gcr.SshAskpass) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcr.SshAskpass) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SshAskpass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A `Prompt` implementation which calls to the system prompter to
/// display prompts in a system modal fashion.
///
/// Since the system prompter usually only displays one prompt at a time, you
/// may have to wait for the prompt to be displayed. Use `SystemPrompt.open`
/// or a related function to open a prompt. Since this can take a long time, you
/// should always check that the prompt is still needed after it is opened. A
/// previous prompt may have already provided the information needed and you
/// may no longer need to prompt.
///
/// Use `SystemPrompt.close` to close the prompt when you're done with it.
pub const SystemPrompt = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{ gcr.Prompt, gio.AsyncInitable, gio.Initable };
    pub const Class = gcr.SystemPromptClass;
    f_parent: gobject.Object,
    f_pv: ?*gcr.SystemPromptPrivate,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// The DBus bus name of the prompter to use for prompting, or `NULL`
        /// for the default prompter.
        pub const bus_name = struct {
            pub const name = "bus-name";

            pub const Type = ?[*:0]u8;
        };

        /// The `gcr.SecretExchange` to use when transferring passwords. A default
        /// secret exchange will be used if this is not set.
        pub const secret_exchange = struct {
            pub const name = "secret-exchange";

            pub const Type = ?*gcr.SecretExchange;
        };

        /// The timeout in seconds to wait when opening the prompt.
        pub const timeout_seconds = struct {
            pub const name = "timeout-seconds";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    extern fn gcr_system_prompt_error_get_domain() glib.Quark;
    pub const errorGetDomain = gcr_system_prompt_error_get_domain;

    /// Opens a system prompt with the default prompter.
    ///
    /// Most system prompters only allow showing one prompt at a time, and if
    /// another prompt is shown then this method will block for up to
    /// `timeout_seconds` seconds. If `timeout_seconds` is equal to -1, then this
    /// will block indefinitely until the prompt can be opened. If `timeout_seconds`
    /// expires, then this function will fail with a `GCR_SYSTEM_PROMPT_IN_PROGRESS`
    /// error.
    extern fn gcr_system_prompt_open(p_timeout_seconds: c_int, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gcr.SystemPrompt;
    pub const open = gcr_system_prompt_open;

    /// Asynchronously open a system prompt with the default system prompter.
    ///
    /// Most system prompters only allow showing one prompt at a time, and if
    /// another prompt is shown then this method will block for up to
    /// `timeout_seconds` seconds. If `timeout_seconds` is equal to -1, then this
    /// will block indefinitely until the prompt can be opened. If `timeout_seconds`
    /// expires, then this operation will fail with a `GCR_SYSTEM_PROMPT_IN_PROGRESS`
    /// error.
    extern fn gcr_system_prompt_open_async(p_timeout_seconds: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const openAsync = gcr_system_prompt_open_async;

    /// Complete an operation to asynchronously open a system prompt.
    extern fn gcr_system_prompt_open_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?*gcr.SystemPrompt;
    pub const openFinish = gcr_system_prompt_open_finish;

    /// Opens a system prompt. If prompter_name is `NULL`, then the default
    /// system prompter is used.
    ///
    /// Most system prompters only allow showing one prompt at a time, and if
    /// another prompt is shown then this method will block for up to
    /// `timeout_seconds` seconds. If `timeout_seconds` is equal to -1, then this
    /// will block indefinitely until the prompt can be opened. If `timeout_seconds`
    /// expires, then this function will fail with a `GCR_SYSTEM_PROMPT_IN_PROGRESS`
    /// error.
    extern fn gcr_system_prompt_open_for_prompter(p_prompter_name: ?[*:0]const u8, p_timeout_seconds: c_int, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*gcr.SystemPrompt;
    pub const openForPrompter = gcr_system_prompt_open_for_prompter;

    /// Opens a system prompt asynchronously. If prompter_name is `NULL`, then the
    /// default system prompter is used.
    ///
    /// Most system prompters only allow showing one prompt at a time, and if
    /// another prompt is shown then this method will block for up to
    /// `timeout_seconds` seconds. If `timeout_seconds` is equal to -1, then this
    /// will block indefinitely until the prompt can be opened. If `timeout_seconds`
    /// expires, then this operation will fail with a `GCR_SYSTEM_PROMPT_IN_PROGRESS`
    /// error.
    extern fn gcr_system_prompt_open_for_prompter_async(p_prompter_name: ?[*:0]const u8, p_timeout_seconds: c_int, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const openForPrompterAsync = gcr_system_prompt_open_for_prompter_async;

    /// Close this prompt. After calling this function, no further prompts will
    /// succeed on this object. The prompt object is not unreferenced by this
    /// function, and you must unreference it once done.
    ///
    /// This call may block, use the `gcr.SystemPrompt.closeAsync` to perform
    /// this action indefinitely.
    ///
    /// Whether or not this function returns `TRUE`, the system prompt object is
    /// still closed and may not be further used.
    extern fn gcr_system_prompt_close(p_self: *SystemPrompt, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const close = gcr_system_prompt_close;

    /// Close this prompt asynchronously. After calling this function, no further
    /// methods may be called on this object. The prompt object is not unreferenced
    /// by this function, and you must unreference it once done.
    ///
    /// This call returns immediately and completes asynchronously.
    extern fn gcr_system_prompt_close_async(p_self: *SystemPrompt, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const closeAsync = gcr_system_prompt_close_async;

    /// Complete operation to close this prompt.
    ///
    /// Whether or not this function returns `TRUE`, the system prompt object is
    /// still closed and may not be further used.
    extern fn gcr_system_prompt_close_finish(p_self: *SystemPrompt, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const closeFinish = gcr_system_prompt_close_finish;

    /// Get the current `SecretExchange` used to transfer secrets in this prompt.
    extern fn gcr_system_prompt_get_secret_exchange(p_self: *SystemPrompt) *gcr.SecretExchange;
    pub const getSecretExchange = gcr_system_prompt_get_secret_exchange;

    extern fn gcr_system_prompt_get_type() usize;
    pub const getGObjectType = gcr_system_prompt_get_type;

    extern fn g_object_ref(p_self: *gcr.SystemPrompt) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcr.SystemPrompt) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SystemPrompt, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A prompter used by implementations of system prompts.
///
/// This is a D-Bus service which is rarely implemented. Use `SystemPrompt`
/// to display system prompts.
///
/// The system prompter service responds to D-Bus requests to create system
/// prompts and creates `gcr.Prompt` type objects to display those prompts.
///
/// Pass the GType of the implementation of `Prompt` to
/// `SystemPrompter.new`.
pub const SystemPrompter = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = gcr.SystemPrompterClass;
    f_parent: gobject.Object,
    f_pv: ?*gcr.SystemPrompterPrivate,

    pub const virtual_methods = struct {
        /// default handler for the `gcr.SystemPrompter.signals.new`-prompt signal
        pub const new_prompt = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) *gcr.Prompt {
                return gobject.ext.as(SystemPrompter.Class, p_class).f_new_prompt.?(gobject.ext.as(SystemPrompter, p_self));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) *gcr.Prompt) void {
                gobject.ext.as(SystemPrompter.Class, p_class).f_new_prompt = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The mode for this prompter.
        ///
        /// Most system prompters only display one prompt at a time and therefore
        /// return `GCR_SYSTEM_PROMPTER_SINGLE`.
        pub const mode = struct {
            pub const name = "mode";

            pub const Type = @compileError("no type information available");
        };

        /// The `gobject.Type` for prompts created by this prompter. This must be a
        /// `gcr.Prompt` implementation.
        pub const prompt_type = struct {
            pub const name = "prompt-type";

            pub const Type = usize;
        };

        /// Whether the prompter is prompting or not.
        pub const prompting = struct {
            pub const name = "prompting";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {
        /// Signal emitted to create a new prompt when needed.
        ///
        /// The default implementation of this signal creates a prompt of the type
        /// `gcr.SystemPrompter.getPromptType`.
        pub const new_prompt = struct {
            pub const name = "new-prompt";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) *gcr.Prompt, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(SystemPrompter, p_instance))),
                    gobject.signalLookup("new-prompt", SystemPrompter.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Create a new system prompter service. This prompter won't do anything unless
    /// you connect to its signals and show appropriate prompts.
    ///
    /// If `prompt_type` is zero, then the new-prompt signal must be handled and
    /// return a valid prompt object implementing the `gcr.Prompt` interface.
    ///
    /// If `prompt_type` is non-zero then the `gobject.Type` must implement the `gcr.Prompt`
    /// interface.
    extern fn gcr_system_prompter_new(p_mode: gcr.SystemPrompterMode, p_prompt_type: usize) *gcr.SystemPrompter;
    pub const new = gcr_system_prompter_new;

    /// Get the mode for this prompter.
    ///
    /// Most system prompters only display one prompt at a time and therefore
    /// return `GCR_SYSTEM_PROMPTER_SINGLE`.
    extern fn gcr_system_prompter_get_mode(p_self: *SystemPrompter) gcr.SystemPrompterMode;
    pub const getMode = gcr_system_prompter_get_mode;

    /// Get the `gobject.Type` for prompts created by this prompter.
    ///
    /// The returned `gobject.Type` will be a `gcr.Prompt` implementation.
    extern fn gcr_system_prompter_get_prompt_type(p_self: *SystemPrompter) usize;
    pub const getPromptType = gcr_system_prompter_get_prompt_type;

    /// Get whether prompting or not.
    extern fn gcr_system_prompter_get_prompting(p_self: *SystemPrompter) c_int;
    pub const getPrompting = gcr_system_prompter_get_prompting;

    /// Register this system prompter on the DBus `connection`.
    ///
    /// This makes the prompter available for clients to call. The prompter will
    /// remain registered until `gcr.SystemPrompter.unregister` is called, or the
    /// prompter is unreferenced.
    extern fn gcr_system_prompter_register(p_self: *SystemPrompter, p_connection: *gio.DBusConnection) void;
    pub const register = gcr_system_prompter_register;

    /// Unregister this system prompter on the DBus `connection`.
    ///
    /// The prompter must have previously been registered with `gcr.SystemPrompter.register`.
    ///
    /// If `wait` is set then this function will wait until all prompts have been closed
    /// or cancelled. This is usually only used by tests.
    extern fn gcr_system_prompter_unregister(p_self: *SystemPrompter, p_wait: c_int) void;
    pub const unregister = gcr_system_prompter_unregister;

    extern fn gcr_system_prompter_get_type() usize;
    pub const getGObjectType = gcr_system_prompter_get_type;

    extern fn g_object_ref(p_self: *gcr.SystemPrompter) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcr.SystemPrompter) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *SystemPrompter, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An interface that represents an X.509 certificate.
///
/// Objects can implement this interface to make a certificate usable with the
/// GCR library.
///
/// Various methods are available to parse out relevant bits of the certificate.
/// However no verification of the validity of a certificate is done here. Use
/// your favorite crypto library to do this.
///
/// You can use `SimpleCertificate` to simply load a certificate for which
/// you already have the raw certificate data.
///
/// The `gcr.Certificate` interface has several properties that must be implemented.
/// You can use a mixin to implement these properties if desired. See the
/// `gcr.certificateMixinClassInit` and `gcr.certificateMixinGetProperty`
/// functions.
pub const Certificate = opaque {
    pub const Prerequisites = [_]type{gobject.Object};
    pub const Iface = gcr.CertificateIface;
    pub const virtual_methods = struct {
        /// Gets the raw DER data for an X.509 certificate.
        pub const get_der_data = struct {
            pub fn call(p_class: anytype, p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_n_data: *usize) [*]const u8 {
                return gobject.ext.as(Certificate.Iface, p_class).f_get_der_data.?(gobject.ext.as(Certificate, p_self), p_n_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_self: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_n_data: *usize) callconv(.C) [*]const u8) void {
                gobject.ext.as(Certificate.Iface, p_class).f_get_der_data = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// A readable description for this certificate
        pub const description = struct {
            pub const name = "description";

            pub const Type = ?[*:0]u8;
        };

        /// The expiry date of the certificate
        pub const expiry_date = struct {
            pub const name = "expiry-date";

            pub const Type = ?*glib.DateTime;
        };

        /// Common name part of the certificate issuer
        pub const issuer_name = struct {
            pub const name = "issuer-name";

            pub const Type = ?[*:0]u8;
        };

        /// A readable label for this certificate.
        pub const label = struct {
            pub const name = "label";

            pub const Type = ?[*:0]u8;
        };

        pub const subject_name = struct {
            pub const name = "subject-name";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Initialize the certificate mixin for the class. This mixin implements the
    /// various required properties for the certificate.
    ///
    /// Call this function near the end of your derived class_init function. The
    /// derived class must implement the `gcr.Certificate` interface.
    extern fn gcr_certificate_mixin_class_init(p_object_class: *gobject.ObjectClass) void;
    pub const mixinClassInit = gcr_certificate_mixin_class_init;

    /// Implementation to get various required certificate properties. This should
    /// be called from your derived class get_property function, or used as a
    /// get_property virtual function.
    ///
    /// Example of use as called from derived class get_property function:
    ///
    /// <informalexample><programlisting>
    /// static void
    /// my_get_property (GObject *obj, guint prop_id, GValue *value, GParamSpec *pspec)
    /// {
    ///     switch (prop_id) {
    ///
    ///     ...
    ///
    ///     default:
    ///         gcr_certificate_mixin_get_property (obj, prop_id, value, pspec);
    ///         break;
    ///     }
    /// }
    /// </programlisting></informalexample>
    ///
    /// Example of use as get_property function:
    ///
    /// <informalexample><programlisting>
    /// static void
    /// my_class_init (MyClass *klass)
    /// {
    ///     GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
    ///     gobject_class->get_property = gcr_certificate_mixin_get_property;
    ///
    ///     ...
    /// }
    /// </programlisting></informalexample>
    extern fn gcr_certificate_mixin_get_property(p_obj: *gobject.Object, p_prop_id: c_uint, p_value: *gobject.Value, p_pspec: *gobject.ParamSpec) void;
    pub const mixinGetProperty = gcr_certificate_mixin_get_property;

    /// Get the basic constraints for the certificate if present. If `FALSE` is
    /// returned then no basic constraints are present and the `is_ca` and
    /// `path_len` arguments are not changed.
    extern fn gcr_certificate_get_basic_constraints(p_self: *Certificate, p_is_ca: ?*c_int, p_path_len: ?*c_int) c_int;
    pub const getBasicConstraints = gcr_certificate_get_basic_constraints;

    /// Gets the raw DER data for an X.509 certificate.
    extern fn gcr_certificate_get_der_data(p_self: *Certificate, p_n_data: *usize) [*]const u8;
    pub const getDerData = gcr_certificate_get_der_data;

    /// Get the expiry date of this certificate.
    extern fn gcr_certificate_get_expiry_date(p_self: *Certificate) ?*glib.DateTime;
    pub const getExpiryDate = gcr_certificate_get_expiry_date;

    /// Calculate the fingerprint for this certificate.
    ///
    /// The caller should free the returned data using `glib.free` when
    /// it is no longer required.
    extern fn gcr_certificate_get_fingerprint(p_self: *Certificate, p_type: glib.ChecksumType, p_n_length: *usize) ?[*]u8;
    pub const getFingerprint = gcr_certificate_get_fingerprint;

    /// Calculate the fingerprint for this certificate, and return it
    /// as a hex string.
    ///
    /// The caller should free the returned data using `glib.free` when
    /// it is no longer required.
    extern fn gcr_certificate_get_fingerprint_hex(p_self: *Certificate, p_type: glib.ChecksumType) ?[*:0]u8;
    pub const getFingerprintHex = gcr_certificate_get_fingerprint_hex;

    /// Get the list of sections from the certificate that can be shown to the user
    /// interface.
    extern fn gcr_certificate_get_interface_elements(p_self: *Certificate) *glib.List;
    pub const getInterfaceElements = gcr_certificate_get_interface_elements;

    /// Get the issued date of this certificate.
    extern fn gcr_certificate_get_issued_date(p_self: *Certificate) ?*glib.DateTime;
    pub const getIssuedDate = gcr_certificate_get_issued_date;

    /// Get the common name of the issuer of this certificate.
    ///
    /// The string returned should be freed by the caller when no longer
    /// required.
    extern fn gcr_certificate_get_issuer_cn(p_self: *Certificate) ?[*:0]u8;
    pub const getIssuerCn = gcr_certificate_get_issuer_cn;

    /// Get the full issuer DN of the certificate as a (mostly)
    /// readable string.
    ///
    /// The string returned should be freed by the caller when no longer
    /// required.
    extern fn gcr_certificate_get_issuer_dn(p_self: *Certificate) ?[*:0]u8;
    pub const getIssuerDn = gcr_certificate_get_issuer_dn;

    /// Get a name to represent the issuer of this certificate.
    ///
    /// This will try to lookup the common name, orianizational unit,
    /// organization in that order.
    extern fn gcr_certificate_get_issuer_name(p_self: *Certificate) ?[*:0]u8;
    pub const getIssuerName = gcr_certificate_get_issuer_name;

    /// Get a part of the DN of the issuer of this certificate.
    ///
    /// Examples of a `part` might be the 'OU' (organizational unit)
    /// or the 'CN' (common name). Only the value of that part
    /// of the DN is returned.
    ///
    /// The string returned should be freed by the caller when no longer
    /// required.
    extern fn gcr_certificate_get_issuer_part(p_self: *Certificate, p_part: [*:0]const u8) ?[*:0]u8;
    pub const getIssuerPart = gcr_certificate_get_issuer_part;

    /// Get the raw DER data for the issuer DN of the certificate.
    ///
    /// The data should be freed by using `glib.free` when no longer required.
    extern fn gcr_certificate_get_issuer_raw(p_self: *Certificate, p_n_data: *usize) ?[*]u8;
    pub const getIssuerRaw = gcr_certificate_get_issuer_raw;

    /// Get the key size in bits of the public key represented
    /// by this certificate.
    extern fn gcr_certificate_get_key_size(p_self: *Certificate) c_uint;
    pub const getKeySize = gcr_certificate_get_key_size;

    /// Get the raw binary serial number of the certificate.
    ///
    /// The caller should free the returned data using `glib.free` when
    /// it is no longer required.
    extern fn gcr_certificate_get_serial_number(p_self: *Certificate, p_n_length: *usize) ?[*]u8;
    pub const getSerialNumber = gcr_certificate_get_serial_number;

    /// Get the serial number of the certificate as a hex string.
    ///
    /// The caller should free the returned data using `glib.free` when
    /// it is no longer required.
    extern fn gcr_certificate_get_serial_number_hex(p_self: *Certificate) ?[*:0]u8;
    pub const getSerialNumberHex = gcr_certificate_get_serial_number_hex;

    /// Get the common name of the subject of this certificate.
    ///
    /// The string returned should be freed by the caller when no longer
    /// required.
    extern fn gcr_certificate_get_subject_cn(p_self: *Certificate) ?[*:0]u8;
    pub const getSubjectCn = gcr_certificate_get_subject_cn;

    /// Get the full subject DN of the certificate as a (mostly)
    /// readable string.
    ///
    /// The string returned should be freed by the caller when no longer
    /// required.
    extern fn gcr_certificate_get_subject_dn(p_self: *Certificate) ?[*:0]u8;
    pub const getSubjectDn = gcr_certificate_get_subject_dn;

    /// Get a name to represent the subject of this certificate.
    ///
    /// This will try to lookup the common name, orianizational unit,
    /// organization in that order.
    extern fn gcr_certificate_get_subject_name(p_self: *Certificate) ?[*:0]u8;
    pub const getSubjectName = gcr_certificate_get_subject_name;

    /// Get a part of the DN of the subject of this certificate.
    ///
    /// Examples of a `part` might be the 'OU' (organizational unit)
    /// or the 'CN' (common name). Only the value of that part
    /// of the DN is returned.
    ///
    /// The string returned should be freed by the caller when no longer
    /// required.
    extern fn gcr_certificate_get_subject_part(p_self: *Certificate, p_part: [*:0]const u8) ?[*:0]u8;
    pub const getSubjectPart = gcr_certificate_get_subject_part;

    /// Get the raw DER data for the subject DN of the certificate.
    ///
    /// The data should be freed by using `glib.free` when no longer required.
    extern fn gcr_certificate_get_subject_raw(p_self: *Certificate, p_n_data: *usize) ?[*]u8;
    pub const getSubjectRaw = gcr_certificate_get_subject_raw;

    /// Get the version of the X.509 certificate.
    extern fn gcr_certificate_get_version(p_self: *Certificate) c_ulong;
    pub const getVersion = gcr_certificate_get_version;

    /// Check if `issuer` could be the issuer of this certificate. This is done by
    /// comparing the relevant subject and issuer fields. No signature check is
    /// done. Proper verification of certificates must be done via a crypto
    /// library.
    extern fn gcr_certificate_is_issuer(p_self: *Certificate, p_issuer: *gcr.Certificate) c_int;
    pub const isIssuer = gcr_certificate_is_issuer;

    /// Implementers of the `gcr.Certificate` mixin should call this function to notify
    /// when the certificate has changed to emit notifications on the various
    /// properties.
    extern fn gcr_certificate_mixin_emit_notify(p_self: *Certificate) void;
    pub const mixinEmitNotify = gcr_certificate_mixin_emit_notify;

    extern fn gcr_certificate_get_type() usize;
    pub const getGObjectType = gcr_certificate_get_type;

    extern fn g_object_ref(p_self: *gcr.Certificate) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcr.Certificate) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Certificate, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// This is an interface implemented by a caller performing an import. It allows
/// the importer to ask the caller for further information about the import.
///
/// It must be implemented on a derived class of `gio.TlsInteraction`
pub const ImportInteraction = opaque {
    pub const Prerequisites = [_]type{gio.TlsInteraction};
    pub const Iface = gcr.ImportInteractionInterface;
    pub const virtual_methods = struct {
        /// Supplement attributes before import. This means prompting the user for
        /// things like labels and the like. The needed attributes will have been passed
        /// to `gcr.ImportInteraction.supplementPrep`.
        ///
        /// This method prompts the user and fills in the attributes. If the user or
        /// cancellable cancels the operation the error should be set with `G_IO_ERROR_CANCELLED`.
        pub const supplement = struct {
            pub fn call(p_class: anytype, p_interaction: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_builder: *gck.Builder, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) gio.TlsInteractionResult {
                return gobject.ext.as(ImportInteraction.Iface, p_class).f_supplement.?(gobject.ext.as(ImportInteraction, p_interaction), p_builder, p_cancellable, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_interaction: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_builder: *gck.Builder, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) gio.TlsInteractionResult) void {
                gobject.ext.as(ImportInteraction.Iface, p_class).f_supplement = @ptrCast(p_implementation);
            }
        };

        /// Asynchronously supplement attributes before import. This means prompting the
        /// user for things like labels and the like. The needed attributes will have
        /// been passed to `gcr.ImportInteraction.supplementPrep`.
        ///
        /// This method prompts the user and fills in the attributes.
        pub const supplement_async = struct {
            pub fn call(p_class: anytype, p_interaction: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_builder: *gck.Builder, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(ImportInteraction.Iface, p_class).f_supplement_async.?(gobject.ext.as(ImportInteraction, p_interaction), p_builder, p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_interaction: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_builder: *gck.Builder, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(ImportInteraction.Iface, p_class).f_supplement_async = @ptrCast(p_implementation);
            }
        };

        /// Complete operation to asynchronously supplement attributes before import.
        ///
        /// If the user or cancellable cancels the operation the error should be set
        /// with `G_IO_ERROR_CANCELLED`.
        pub const supplement_finish = struct {
            pub fn call(p_class: anytype, p_interaction: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) gio.TlsInteractionResult {
                return gobject.ext.as(ImportInteraction.Iface, p_class).f_supplement_finish.?(gobject.ext.as(ImportInteraction, p_interaction), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_interaction: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) gio.TlsInteractionResult) void {
                gobject.ext.as(ImportInteraction.Iface, p_class).f_supplement_finish = @ptrCast(p_implementation);
            }
        };

        /// Prepare for supplementing the given attributes before import. This means
        /// prompting the user for things like labels and the like. The attributes
        /// will contain attributes for values that the importer needs, either empty
        /// or prefilled with suggested values.
        ///
        /// This method does not prompt the user, but rather just prepares the
        /// interaction that these are the attributes that are needed.
        pub const supplement_prep = struct {
            pub fn call(p_class: anytype, p_interaction: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_builder: *gck.Builder) void {
                return gobject.ext.as(ImportInteraction.Iface, p_class).f_supplement_prep.?(gobject.ext.as(ImportInteraction, p_interaction), p_builder);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_interaction: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_builder: *gck.Builder) callconv(.C) void) void {
                gobject.ext.as(ImportInteraction.Iface, p_class).f_supplement_prep = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {};

    pub const signals = struct {};

    /// Supplement attributes before import. This means prompting the user for
    /// things like labels and the like. The needed attributes will have been passed
    /// to `gcr.ImportInteraction.supplementPrep`.
    ///
    /// This method prompts the user and fills in the attributes. If the user or
    /// cancellable cancels the operation the error should be set with `G_IO_ERROR_CANCELLED`.
    extern fn gcr_import_interaction_supplement(p_interaction: *ImportInteraction, p_builder: *gck.Builder, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) gio.TlsInteractionResult;
    pub const supplement = gcr_import_interaction_supplement;

    /// Asynchronously supplement attributes before import. This means prompting the
    /// user for things like labels and the like. The needed attributes will have
    /// been passed to `gcr.ImportInteraction.supplementPrep`.
    ///
    /// This method prompts the user and fills in the attributes.
    extern fn gcr_import_interaction_supplement_async(p_interaction: *ImportInteraction, p_builder: *gck.Builder, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const supplementAsync = gcr_import_interaction_supplement_async;

    /// Complete operation to asynchronously supplement attributes before import.
    ///
    /// If the user or cancellable cancels the operation the error should be set
    /// with `G_IO_ERROR_CANCELLED`.
    extern fn gcr_import_interaction_supplement_finish(p_interaction: *ImportInteraction, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) gio.TlsInteractionResult;
    pub const supplementFinish = gcr_import_interaction_supplement_finish;

    /// Prepare for supplementing the given attributes before import. This means
    /// prompting the user for things like labels and the like. The attributes
    /// will contain attributes for values that the importer needs, either empty
    /// or prefilled with suggested values.
    ///
    /// This method does not prompt the user, but rather just prepares the
    /// interaction that these are the attributes that are needed.
    extern fn gcr_import_interaction_supplement_prep(p_interaction: *ImportInteraction, p_builder: *gck.Builder) void;
    pub const supplementPrep = gcr_import_interaction_supplement_prep;

    extern fn gcr_import_interaction_get_type() usize;
    pub const getGObjectType = gcr_import_interaction_get_type;

    extern fn g_object_ref(p_self: *gcr.ImportInteraction) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcr.ImportInteraction) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *ImportInteraction, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An interface which allows importing of certificates and keys. Each importer
/// is registered with a set of PKCS`@"11"` attributes to match stuff that it can
/// import.
///
/// An importer gets passed a `Parser` and accesses the currently parsed
/// item. To create a set of importers that can import the currently parsed
/// item in a parser, use `Importer.createForParsed`. The list of
/// importers returned has the parsed item queued for import.
///
/// To queue additional items with a importer use
/// `Importer.queueForParsed`.  In addition you can try and queue an
/// additional item with a set of importers using the
/// `Importer.queueAndFilterForParsed`.
///
/// To start the import, use `Importer.importAsync`.
pub const Importer = opaque {
    pub const Prerequisites = [_]type{gobject.Object};
    pub const Iface = gcr.ImporterInterface;
    pub const virtual_methods = struct {
        /// Import the queued items in the importer. This function returns immediately
        /// and completes asynchronously.
        pub const import_async = struct {
            pub fn call(p_class: anytype, p_importer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(Importer.Iface, p_class).f_import_async.?(gobject.ext.as(Importer, p_importer), p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_importer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(Importer.Iface, p_class).f_import_async = @ptrCast(p_implementation);
            }
        };

        /// Complete an asynchronous operation to import queued items.
        pub const import_finish = struct {
            pub fn call(p_class: anytype, p_importer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int {
                return gobject.ext.as(Importer.Iface, p_class).f_import_finish.?(gobject.ext.as(Importer, p_importer), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_importer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) c_int) void {
                gobject.ext.as(Importer.Iface, p_class).f_import_finish = @ptrCast(p_implementation);
            }
        };

        /// Queues an additional item to be imported. The parsed item is represented
        /// by the state of the `Parser` at the time of calling this method.
        ///
        /// If the parsed item is incompatible with the importer, then this will
        /// fail and the item will not be queued.
        pub const queue_for_parsed = struct {
            pub fn call(p_class: anytype, p_importer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_parsed: *gcr.Parsed) c_int {
                return gobject.ext.as(Importer.Iface, p_class).f_queue_for_parsed.?(gobject.ext.as(Importer, p_importer), p_parsed);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_importer: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_parsed: *gcr.Parsed) callconv(.C) c_int) void {
                gobject.ext.as(Importer.Iface, p_class).f_queue_for_parsed = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The interaction for the importer.
        pub const interaction = struct {
            pub const name = "interaction";

            pub const Type = ?*gio.TlsInteraction;
        };

        /// The label for the importer.
        pub const label = struct {
            pub const name = "label";

            pub const Type = ?[*:0]u8;
        };

        /// The URI of the location imported to.
        pub const uri = struct {
            pub const name = "uri";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {};

    /// Create a set of importers which can import this parsed item.
    /// The parsed item is represented by the state of the GcrParser at the
    /// time of calling this method.
    extern fn gcr_importer_create_for_parsed(p_parsed: *gcr.Parsed) *glib.List;
    pub const createForParsed = gcr_importer_create_for_parsed;

    /// Queues an additional item to be imported in all compattible importers
    /// in the set. The parsed item is represented by the state of the `gcr.Parser`
    /// at the time of calling this method.
    ///
    /// If the parsed item is incompatible with an importer, then that the item
    /// will not be queued on that importer.
    extern fn gcr_importer_queue_and_filter_for_parsed(p_importers: *glib.List, p_parsed: *gcr.Parsed) *glib.List;
    pub const queueAndFilterForParsed = gcr_importer_queue_and_filter_for_parsed;

    /// Register an importer to handle parsed items that match the given attributes.
    extern fn gcr_importer_register(p_importer_type: usize, p_attrs: *gck.Attributes) void;
    pub const register = gcr_importer_register;

    /// Register built-in PKCS`@"11"` and GnuPG importers.
    extern fn gcr_importer_register_well_known() void;
    pub const registerWellKnown = gcr_importer_register_well_known;

    /// Get the interaction used to prompt the user when needed by this
    /// importer.
    extern fn gcr_importer_get_interaction(p_importer: *Importer) ?*gio.TlsInteraction;
    pub const getInteraction = gcr_importer_get_interaction;

    /// Import the queued items in the importer. This function returns immediately
    /// and completes asynchronously.
    extern fn gcr_importer_import_async(p_importer: *Importer, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const importAsync = gcr_importer_import_async;

    /// Complete an asynchronous operation to import queued items.
    extern fn gcr_importer_import_finish(p_importer: *Importer, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
    pub const importFinish = gcr_importer_import_finish;

    /// Queues an additional item to be imported. The parsed item is represented
    /// by the state of the `Parser` at the time of calling this method.
    ///
    /// If the parsed item is incompatible with the importer, then this will
    /// fail and the item will not be queued.
    extern fn gcr_importer_queue_for_parsed(p_importer: *Importer, p_parsed: *gcr.Parsed) c_int;
    pub const queueForParsed = gcr_importer_queue_for_parsed;

    /// Set the interaction used to prompt the user when needed by this
    /// importer.
    extern fn gcr_importer_set_interaction(p_importer: *Importer, p_interaction: *gio.TlsInteraction) void;
    pub const setInteraction = gcr_importer_set_interaction;

    extern fn gcr_importer_get_type() usize;
    pub const getGObjectType = gcr_importer_get_type;

    extern fn g_object_ref(p_self: *gcr.Importer) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcr.Importer) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Importer, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A prompt displayed to the user. It is an interface with various
/// implementations.
///
/// Various properties are set on the prompt, and then the prompt is displayed
/// the various prompt methods like `Prompt.passwordRun`.
///
/// A `GcrPrompt` may be used to display multiple related prompts. Most
/// implementions do not hide the window between display of multiple related
/// prompts, and the `gcr.Prompt` must be closed or destroyed in order to make
/// it go away. This allows the user to see that the prompts are related.
///
/// Use `GcrPromptDialog` (part of gcr-ui) to create an in-process GTK+ dialog
/// prompt. Use `SystemPrompt` to create a system prompt in a prompter
/// process.
///
/// The prompt implementation will always display the `Prompt.properties.message`
/// property, but may choose not to display the `Prompt.properties.description` or
/// `Prompt.properties.title` properties.
pub const Prompt = opaque {
    pub const Prerequisites = [_]type{gobject.Object};
    pub const Iface = gcr.PromptInterface;
    pub const virtual_methods = struct {
        /// close a prompt
        pub const prompt_close = struct {
            pub fn call(p_class: anytype, p_prompt: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) void {
                return gobject.ext.as(Prompt.Iface, p_class).f_prompt_close.?(gobject.ext.as(Prompt, p_prompt));
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_prompt: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance) callconv(.C) void) void {
                gobject.ext.as(Prompt.Iface, p_class).f_prompt_close = @ptrCast(p_implementation);
            }
        };

        /// Prompts for confirmation asking a cancel/continue style question.
        /// Set the various properties on the prompt before calling this method to
        /// represent the question correctly.
        ///
        /// This method will return immediately and complete asynchronously.
        pub const prompt_confirm_async = struct {
            pub fn call(p_class: anytype, p_prompt: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(Prompt.Iface, p_class).f_prompt_confirm_async.?(gobject.ext.as(Prompt, p_prompt), p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_prompt: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(Prompt.Iface, p_class).f_prompt_confirm_async = @ptrCast(p_implementation);
            }
        };

        /// Complete an operation to prompt for confirmation.
        ///
        /// `GCR_PROMPT_REPLY_CONTINUE` will be returned if the user confirms the prompt. The
        /// return value will also be `GCR_PROMPT_REPLY_CANCEL` if the user cancels or if
        /// an error occurs. Check the `error` argument to tell the difference.
        pub const prompt_confirm_finish = struct {
            pub fn call(p_class: anytype, p_prompt: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) gcr.PromptReply {
                return gobject.ext.as(Prompt.Iface, p_class).f_prompt_confirm_finish.?(gobject.ext.as(Prompt, p_prompt), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_prompt: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) gcr.PromptReply) void {
                gobject.ext.as(Prompt.Iface, p_class).f_prompt_confirm_finish = @ptrCast(p_implementation);
            }
        };

        /// Prompts for password. Set the various properties on the prompt before calling
        /// this method to explain which password should be entered.
        ///
        /// This method will return immediately and complete asynchronously.
        pub const prompt_password_async = struct {
            pub fn call(p_class: anytype, p_prompt: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void {
                return gobject.ext.as(Prompt.Iface, p_class).f_prompt_password_async.?(gobject.ext.as(Prompt, p_prompt), p_cancellable, p_callback, p_user_data);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_prompt: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void) void {
                gobject.ext.as(Prompt.Iface, p_class).f_prompt_password_async = @ptrCast(p_implementation);
            }
        };

        /// Complete an operation to prompt for a password.
        ///
        /// A password will be returned if the user enters a password successfully.
        /// The returned password is valid until the next time a method is called
        /// to display another prompt.
        ///
        /// `NULL` will be returned if the user cancels or if an error occurs. Check the
        /// `error` argument to tell the difference.
        pub const prompt_password_finish = struct {
            pub fn call(p_class: anytype, p_prompt: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?[*:0]const u8 {
                return gobject.ext.as(Prompt.Iface, p_class).f_prompt_password_finish.?(gobject.ext.as(Prompt, p_prompt), p_result, p_error);
            }

            pub fn implement(p_class: anytype, p_implementation: *const fn (p_prompt: *compat.typeInfo(@TypeOf(p_class)).pointer.child.Instance, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) ?[*:0]const u8) void {
                gobject.ext.as(Prompt.Iface, p_class).f_prompt_password_finish = @ptrCast(p_implementation);
            }
        };
    };

    pub const properties = struct {
        /// The string handle of the caller's window.
        ///
        /// The caller window indicates to the prompt which window is prompting the
        /// user. The prompt may choose to ignore this information or use it in whatever
        /// way it sees fit.
        ///
        /// In X11, this will be a stringified version of the XWindow handle; in
        /// Wayland this is the result of an export using the XDG foreign
        /// protocol.
        pub const caller_window = struct {
            pub const name = "caller-window";

            pub const Type = ?[*:0]u8;
        };

        /// The label for the cancel button in the prompt.
        pub const cancel_label = struct {
            pub const name = "cancel-label";

            pub const Type = ?[*:0]u8;
        };

        /// Whether the additional choice is chosen or not.
        ///
        /// The additional choice would have been setup using `gcr.Prompt.properties.choice`-label.
        pub const choice_chosen = struct {
            pub const name = "choice-chosen";

            pub const Type = c_int;
        };

        /// The label for the additional choice.
        ///
        /// If this is a non-`NULL` value then an additional boolean choice will be
        /// displayed by the prompt allowing the user to select or deselect it.
        ///
        /// If `NULL`, then no additional choice is displayed.
        ///
        /// The initial value of the choice can be set with `gcr.Prompt.properties.choice`-chosen.
        pub const choice_label = struct {
            pub const name = "choice-label";

            pub const Type = ?[*:0]u8;
        };

        /// The label for the continue button in the prompt.
        pub const continue_label = struct {
            pub const name = "continue-label";

            pub const Type = ?[*:0]u8;
        };

        /// The detailed description of the prompt.
        ///
        /// A prompt implementation may choose not to display this detailed description.
        /// The prompt message should contain relevant information.
        pub const description = struct {
            pub const name = "description";

            pub const Type = ?[*:0]u8;
        };

        /// The prompt message for the user.
        ///
        /// A prompt implementation should always display this message.
        pub const message = struct {
            pub const name = "message";

            pub const Type = ?[*:0]u8;
        };

        /// Whether the prompt will prompt for a new password.
        ///
        /// This will cause the prompt implementation to ask the user to confirm the
        /// password and/or display other relevant user interface for creating a new
        /// password.
        pub const password_new = struct {
            pub const name = "password-new";

            pub const Type = c_int;
        };

        /// Indication of the password strength.
        ///
        /// Prompts will return a zero value if the password is empty, and a value
        /// greater than zero if the password has any characters.
        ///
        /// This is only valid after a successful prompt for a password.
        pub const password_strength = struct {
            pub const name = "password-strength";

            pub const Type = c_int;
        };

        /// The title of the prompt.
        ///
        /// A prompt implementation may choose not to display the prompt title. The
        /// `gcr.Prompt.properties.message` should contain relevant information.
        pub const title = struct {
            pub const name = "title";

            pub const Type = ?[*:0]u8;
        };

        /// A prompt warning displayed on the prompt, or `NULL` for no warning.
        ///
        /// This is a warning like "The password is incorrect." usually displayed to the
        /// user about a previous 'unsuccessful' prompt.
        pub const warning = struct {
            pub const name = "warning";

            pub const Type = ?[*:0]u8;
        };
    };

    pub const signals = struct {
        /// Action signal fired when the prompt is to be closed. After the default
        /// handler has run, the prompt is closed. The various prompting methods
        /// will return results as if the user dismissed the prompt.
        ///
        /// You can use the `Prompt.close` method to emit this signal.
        pub const prompt_close = struct {
            pub const name = "prompt-close";

            pub fn connect(p_instance: anytype, comptime P_Data: type, p_callback: *const fn (@TypeOf(p_instance), P_Data) callconv(.C) void, p_data: P_Data, p_options: gobject.ext.ConnectSignalOptions(P_Data)) c_ulong {
                return gobject.signalConnectClosureById(
                    @ptrCast(@alignCast(gobject.ext.as(Prompt, p_instance))),
                    gobject.signalLookup("prompt-close", Prompt.getGObjectType()),
                    glib.quarkFromString(p_options.detail orelse null),
                    gobject.CClosure.new(@ptrCast(p_callback), p_data, @ptrCast(p_options.destroyData)),
                    @intFromBool(p_options.after),
                );
            }
        };
    };

    /// Closes the prompt so that in can no longer be used to prompt. The various
    /// prompt methods will return results as if the user dismissed the prompt.
    ///
    /// The prompt may also be closed by the implementor of the prompt object.
    ///
    /// This emits the `Prompt.signals.prompt_close` signal on the prompt object.
    extern fn gcr_prompt_close(p_prompt: *Prompt) void;
    pub const close = gcr_prompt_close;

    /// Prompts for confirmation asking a cancel/continue style question.
    /// Set the various properties on the prompt before calling this function to
    /// represent the question correctly.
    ///
    /// This method will block until the a response is returned from the prompter.
    ///
    /// `GCR_PROMPT_REPLY_CONTINUE` will be returned if the user confirms the prompt. The
    /// return value will also be `GCR_PROMPT_REPLY_CANCEL` if the user cancels or if
    /// an error occurs. Check the `error` argument to tell the difference.
    extern fn gcr_prompt_confirm(p_prompt: *Prompt, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) gcr.PromptReply;
    pub const confirm = gcr_prompt_confirm;

    /// Prompts for confirmation asking a cancel/continue style question.
    /// Set the various properties on the prompt before calling this method to
    /// represent the question correctly.
    ///
    /// This method will return immediately and complete asynchronously.
    extern fn gcr_prompt_confirm_async(p_prompt: *Prompt, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const confirmAsync = gcr_prompt_confirm_async;

    /// Complete an operation to prompt for confirmation.
    ///
    /// `GCR_PROMPT_REPLY_CONTINUE` will be returned if the user confirms the prompt. The
    /// return value will also be `GCR_PROMPT_REPLY_CANCEL` if the user cancels or if
    /// an error occurs. Check the `error` argument to tell the difference.
    extern fn gcr_prompt_confirm_finish(p_prompt: *Prompt, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) gcr.PromptReply;
    pub const confirmFinish = gcr_prompt_confirm_finish;

    /// Prompts for confirmation asking a cancel/continue style question.
    /// Set the various properties on the prompt before calling this function to
    /// represent the question correctly.
    ///
    /// This method will block until the a response is returned from the prompter
    /// and will run a main loop similar to a ``gtk_dialog_run``. The application
    /// will remain responsive but care must be taken to handle reentrancy issues.
    ///
    /// `GCR_PROMPT_REPLY_CONTINUE` will be returned if the user confirms the prompt. The
    /// return value will also be `GCR_PROMPT_REPLY_CANCEL` if the user cancels or if
    /// an error occurs. Check the `error` argument to tell the difference.
    extern fn gcr_prompt_confirm_run(p_prompt: *Prompt, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) gcr.PromptReply;
    pub const confirmRun = gcr_prompt_confirm_run;

    /// Get the string handle of the caller's window.
    ///
    /// The caller window indicates to the prompt which window is prompting the
    /// user. The prompt may choose to ignore this information or use it in whatever
    /// way it sees fit.
    extern fn gcr_prompt_get_caller_window(p_prompt: *Prompt) [*:0]u8;
    pub const getCallerWindow = gcr_prompt_get_caller_window;

    /// Get the label for the cancel button.
    ///
    /// This is the button that results in a `GCR_PROMPT_REPLY_CANCEL` reply
    /// from the prompt.
    extern fn gcr_prompt_get_cancel_label(p_prompt: *Prompt) [*:0]u8;
    pub const getCancelLabel = gcr_prompt_get_cancel_label;

    /// Get whether the additional choice was chosen or not.
    ///
    /// The additional choice would have been setup using
    /// `gcr.Prompt.setChoiceLabel`.
    extern fn gcr_prompt_get_choice_chosen(p_prompt: *Prompt) c_int;
    pub const getChoiceChosen = gcr_prompt_get_choice_chosen;

    /// Get the label for the additional choice.
    ///
    /// This will be `NULL` if no additional choice is being displayed.
    extern fn gcr_prompt_get_choice_label(p_prompt: *Prompt) [*:0]u8;
    pub const getChoiceLabel = gcr_prompt_get_choice_label;

    /// Get the label for the continue button.
    ///
    /// This is the button that results in a `GCR_PROMPT_REPLY_CONTINUE` reply
    /// from the prompt.
    extern fn gcr_prompt_get_continue_label(p_prompt: *Prompt) [*:0]u8;
    pub const getContinueLabel = gcr_prompt_get_continue_label;

    /// Get the detailed description of the prompt.
    ///
    /// A prompt implementation may choose not to display this detailed description.
    /// The prompt message should contain relevant information.
    extern fn gcr_prompt_get_description(p_prompt: *Prompt) [*:0]u8;
    pub const getDescription = gcr_prompt_get_description;

    /// Gets the prompt message for the user.
    ///
    /// A prompt implementation should always display this message.
    extern fn gcr_prompt_get_message(p_prompt: *Prompt) [*:0]u8;
    pub const getMessage = gcr_prompt_get_message;

    /// Get whether the prompt will prompt for a new password.
    ///
    /// This will cause the prompt implementation to ask the user to confirm the
    /// password and/or display other relevant user interface for creating a new
    /// password.
    extern fn gcr_prompt_get_password_new(p_prompt: *Prompt) c_int;
    pub const getPasswordNew = gcr_prompt_get_password_new;

    /// Get indication of the password strength.
    ///
    /// Prompts will return a zero value if the password is empty, and a value
    /// greater than zero if the password has any characters.
    ///
    /// This is only valid after a successful prompt for a password.
    extern fn gcr_prompt_get_password_strength(p_prompt: *Prompt) c_int;
    pub const getPasswordStrength = gcr_prompt_get_password_strength;

    /// Gets the title of the prompt.
    ///
    /// A prompt implementation may choose not to display the prompt title. The
    /// prompt message should contain relevant information.
    extern fn gcr_prompt_get_title(p_prompt: *Prompt) [*:0]u8;
    pub const getTitle = gcr_prompt_get_title;

    /// Get a prompt warning displayed on the prompt.
    ///
    /// This is a warning like "The password is incorrect." usually displayed to the
    /// user about a previous 'unsuccessful' prompt.
    ///
    /// If this string is `NULL` then no warning is displayed.
    extern fn gcr_prompt_get_warning(p_prompt: *Prompt) [*:0]u8;
    pub const getWarning = gcr_prompt_get_warning;

    /// Prompts for password. Set the various properties on the prompt before calling
    /// this method to explain which password should be entered.
    ///
    /// This method will block until the a response is returned from the prompter.
    ///
    /// A password will be returned if the user enters a password successfully.
    /// The returned password is valid until the next time a method is called
    /// to display another prompt.
    ///
    /// `NULL` will be returned if the user cancels or if an error occurs. Check the
    /// `error` argument to tell the difference.
    extern fn gcr_prompt_password(p_prompt: *Prompt, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?[*:0]const u8;
    pub const password = gcr_prompt_password;

    /// Prompts for password. Set the various properties on the prompt before calling
    /// this method to explain which password should be entered.
    ///
    /// This method will return immediately and complete asynchronously.
    extern fn gcr_prompt_password_async(p_prompt: *Prompt, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
    pub const passwordAsync = gcr_prompt_password_async;

    /// Complete an operation to prompt for a password.
    ///
    /// A password will be returned if the user enters a password successfully.
    /// The returned password is valid until the next time a method is called
    /// to display another prompt.
    ///
    /// `NULL` will be returned if the user cancels or if an error occurs. Check the
    /// `error` argument to tell the difference.
    extern fn gcr_prompt_password_finish(p_prompt: *Prompt, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) ?[*:0]const u8;
    pub const passwordFinish = gcr_prompt_password_finish;

    /// Prompts for password. Set the various properties on the prompt before calling
    /// this method to explain which password should be entered.
    ///
    /// This method will block until the a response is returned from the prompter
    /// and will run a main loop similar to a `gtk_dialog_run`. The application
    /// will remain responsive but care must be taken to handle reentrancy issues.
    ///
    /// A password will be returned if the user enters a password successfully.
    /// The returned password is valid until the next time a method is called
    /// to display another prompt.
    ///
    /// `NULL` will be returned if the user cancels or if an error occurs. Check the
    /// `error` argument to tell the difference.
    extern fn gcr_prompt_password_run(p_prompt: *Prompt, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?[*:0]const u8;
    pub const passwordRun = gcr_prompt_password_run;

    /// Reset the contents and properties of the prompt.
    extern fn gcr_prompt_reset(p_prompt: *Prompt) void;
    pub const reset = gcr_prompt_reset;

    /// Set the string handle of the caller's window.
    ///
    /// The caller window indicates to the prompt which window is prompting the
    /// user. The prompt may choose to ignore this information or use it in whatever
    /// way it sees fit.
    extern fn gcr_prompt_set_caller_window(p_prompt: *Prompt, p_window_id: [*:0]const u8) void;
    pub const setCallerWindow = gcr_prompt_set_caller_window;

    /// Set the label for the continue button.
    ///
    /// This is the button that results in a `GCR_PROMPT_REPLY_CANCEL` reply
    /// from the prompt.
    extern fn gcr_prompt_set_cancel_label(p_prompt: *Prompt, p_cancel_label: [*:0]const u8) void;
    pub const setCancelLabel = gcr_prompt_set_cancel_label;

    /// Set whether the additional choice is chosen or not.
    ///
    /// The additional choice should be set up using `gcr.Prompt.setChoiceLabel`.
    extern fn gcr_prompt_set_choice_chosen(p_prompt: *Prompt, p_chosen: c_int) void;
    pub const setChoiceChosen = gcr_prompt_set_choice_chosen;

    /// Set the label for the additional choice.
    ///
    /// If this is a non-`NULL` value then an additional boolean choice will be
    /// displayed by the prompt allowing the user to select or deselect it.
    ///
    /// The initial value of the choice can be set with the
    /// `gcr.Prompt.setChoiceLabel` method.
    ///
    /// If this is `NULL`, then no additional choice is being displayed.
    extern fn gcr_prompt_set_choice_label(p_prompt: *Prompt, p_choice_label: ?[*:0]const u8) void;
    pub const setChoiceLabel = gcr_prompt_set_choice_label;

    /// Set the label for the continue button.
    ///
    /// This is the button that results in a `GCR_PROMPT_REPLY_CONTINUE` reply
    /// from the prompt.
    extern fn gcr_prompt_set_continue_label(p_prompt: *Prompt, p_continue_label: [*:0]const u8) void;
    pub const setContinueLabel = gcr_prompt_set_continue_label;

    /// Set the detailed description of the prompt.
    ///
    /// A prompt implementation may choose not to display this detailed description.
    /// Use `gcr.Prompt.setMessage` to set a general message containing relevant
    /// information.
    extern fn gcr_prompt_set_description(p_prompt: *Prompt, p_description: [*:0]const u8) void;
    pub const setDescription = gcr_prompt_set_description;

    /// Sets the prompt message for the user.
    ///
    /// A prompt implementation should always display this message.
    extern fn gcr_prompt_set_message(p_prompt: *Prompt, p_message: [*:0]const u8) void;
    pub const setMessage = gcr_prompt_set_message;

    /// Set whether the prompt will prompt for a new password.
    ///
    /// This will cause the prompt implementation to ask the user to confirm the
    /// password and/or display other relevant user interface for creating a new
    /// password.
    extern fn gcr_prompt_set_password_new(p_prompt: *Prompt, p_new_password: c_int) void;
    pub const setPasswordNew = gcr_prompt_set_password_new;

    /// Sets the title of the prompt.
    ///
    /// A prompt implementation may choose not to display the prompt title. The
    /// prompt message should contain relevant information.
    extern fn gcr_prompt_set_title(p_prompt: *Prompt, p_title: [*:0]const u8) void;
    pub const setTitle = gcr_prompt_set_title;

    /// Set a prompt warning displayed on the prompt.
    ///
    /// This is a warning like "The password is incorrect." usually displayed to the
    /// user about a previous 'unsuccessful' prompt.
    ///
    /// If this string is `NULL` then no warning is displayed.
    extern fn gcr_prompt_set_warning(p_prompt: *Prompt, p_warning: ?[*:0]const u8) void;
    pub const setWarning = gcr_prompt_set_warning;

    extern fn gcr_prompt_get_type() usize;
    pub const getGObjectType = gcr_prompt_get_type;

    extern fn g_object_ref(p_self: *gcr.Prompt) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *gcr.Prompt) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Prompt, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The class for `gcr.CertificateChain`.
pub const CertificateChainClass = extern struct {
    pub const Instance = gcr.CertificateChain;

    /// The parent class
    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *CertificateChainClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const CertificateChainPrivate = opaque {};

pub const CertificateFieldClass = extern struct {
    pub const Instance = gcr.CertificateField;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *CertificateFieldClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// The interface that implementors of `gcr.Certificate` must implement.
pub const CertificateIface = extern struct {
    pub const Instance = gcr.Certificate;

    /// the parent interface type
    f_parent: gobject.TypeInterface,
    /// a method which returns the RAW der data of the certificate
    f_get_der_data: ?*const fn (p_self: *gcr.Certificate, p_n_data: *usize) callconv(.C) [*]const u8,
    f_dummy1: ?*anyopaque,
    f_dummy2: ?*anyopaque,
    f_dummy3: ?*anyopaque,
    f_dummy5: ?*anyopaque,
    f_dummy6: ?*anyopaque,
    f_dummy7: ?*anyopaque,
    f_dummy8: ?*anyopaque,

    pub fn as(p_instance: *CertificateIface, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const CertificateRequestClass = extern struct {
    pub const Instance = gcr.CertificateRequest;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *CertificateRequestClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const CertificateSectionClass = extern struct {
    pub const Instance = gcr.CertificateSection;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *CertificateSectionClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Interface implemented by implementations of `ImportInteraction`.
pub const ImportInteractionInterface = extern struct {
    pub const Instance = gcr.ImportInteraction;

    /// parent interface
    f_parent: gobject.TypeInterface,
    /// method which prepares for supplementing the given attributes before import
    f_supplement_prep: ?*const fn (p_interaction: *gcr.ImportInteraction, p_builder: *gck.Builder) callconv(.C) void,
    /// method which synchronously supplements attributes before import
    f_supplement: ?*const fn (p_interaction: *gcr.ImportInteraction, p_builder: *gck.Builder, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) gio.TlsInteractionResult,
    /// method which asynchronously supplements attributes before import
    f_supplement_async: ?*const fn (p_interaction: *gcr.ImportInteraction, p_builder: *gck.Builder, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void,
    /// method which completes `supplement_async`
    f_supplement_finish: ?*const fn (p_interaction: *gcr.ImportInteraction, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) gio.TlsInteractionResult,
    f_reserved: [6]*anyopaque,

    pub fn as(p_instance: *ImportInteractionInterface, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// Interface implemented for a `gcr.Importer`.
pub const ImporterInterface = extern struct {
    pub const Instance = gcr.Importer;

    /// parent interface
    f_parent: gobject.TypeInterface,
    /// implementation of `gcr.importerCreateForParsed`, required
    f_create_for_parsed: ?*const fn (p_parsed: *gcr.Parsed) callconv(.C) *glib.List,
    /// implementation of `gcr.Importer.queueForParsed`, required
    f_queue_for_parsed: ?*const fn (p_importer: *gcr.Importer, p_parsed: *gcr.Parsed) callconv(.C) c_int,
    /// implementation of `Importer.importAsync`, required
    f_import_async: ?*const fn (p_importer: *gcr.Importer, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void,
    /// implementation of `Importer.importFinish`
    f_import_finish: ?*const fn (p_importer: *gcr.Importer, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) c_int,
    f_reserved: [14]*anyopaque,

    pub fn as(p_instance: *ImporterInterface, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A parsed item parsed by a `gcr.Parser`.
pub const Parsed = opaque {
    /// Unreferences a parsed item which was referenced with `gcr.Parsed.ref`
    extern fn gcr_parsed_unref(p_parsed: ?*anyopaque) void;
    pub const unref = gcr_parsed_unref;

    /// Get the attributes which make up the parsed item.
    extern fn gcr_parsed_get_attributes(p_parsed: *Parsed) ?*gck.Attributes;
    pub const getAttributes = gcr_parsed_get_attributes;

    /// Get the raw data block for the parsed item.
    extern fn gcr_parsed_get_bytes(p_parsed: *Parsed) *glib.Bytes;
    pub const getBytes = gcr_parsed_get_bytes;

    /// Get the raw data block for the parsed item.
    extern fn gcr_parsed_get_data(p_parsed: *Parsed, p_n_data: *usize) ?[*]const u8;
    pub const getData = gcr_parsed_get_data;

    /// Get the descirption for a parsed item.
    extern fn gcr_parsed_get_description(p_parsed: *Parsed) ?[*:0]const u8;
    pub const getDescription = gcr_parsed_get_description;

    /// Get the filename of the parsed item.
    extern fn gcr_parsed_get_filename(p_parsed: *Parsed) [*:0]const u8;
    pub const getFilename = gcr_parsed_get_filename;

    /// Get the format of the parsed item.
    extern fn gcr_parsed_get_format(p_parsed: *Parsed) gcr.DataFormat;
    pub const getFormat = gcr_parsed_get_format;

    /// Get the label for the parsed item.
    extern fn gcr_parsed_get_label(p_parsed: *Parsed) ?[*:0]const u8;
    pub const getLabel = gcr_parsed_get_label;

    /// Add a reference to a parsed item. An item may not be shared across threads
    /// until it has been referenced at least once.
    extern fn gcr_parsed_ref(p_parsed: *Parsed) *gcr.Parsed;
    pub const ref = gcr_parsed_ref;

    extern fn gcr_parsed_get_type() usize;
    pub const getGObjectType = gcr_parsed_get_type;
};

/// The class for `gcr.Parser`
pub const ParserClass = extern struct {
    pub const Instance = gcr.Parser;

    /// The parent class
    f_parent_class: gobject.ObjectClass,
    /// The default handler for the authenticate signal.
    f_authenticate: ?*const fn (p_self: *gcr.Parser, p_count: c_int) callconv(.C) c_int,
    /// The default handler for the parsed signal.
    f_parsed: ?*const fn (p_self: *gcr.Parser) callconv(.C) void,

    pub fn as(p_instance: *ParserClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const ParserPrivate = opaque {};

pub const Pkcs11CertificateClass = extern struct {
    pub const Instance = gcr.Pkcs11Certificate;

    f_parent_class: gck.ObjectClass,

    pub fn as(p_instance: *Pkcs11CertificateClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const Pkcs11CertificatePrivate = opaque {};

/// The interface for implementing `Prompt`.
pub const PromptInterface = extern struct {
    pub const Instance = gcr.Prompt;

    /// parent interface
    f_parent_iface: gobject.TypeInterface,
    /// begin a password prompt
    f_prompt_password_async: ?*const fn (p_prompt: *gcr.Prompt, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void,
    /// complete a password prompt
    f_prompt_password_finish: ?*const fn (p_prompt: *gcr.Prompt, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) ?[*:0]const u8,
    /// begin a confirm prompt
    f_prompt_confirm_async: ?*const fn (p_prompt: *gcr.Prompt, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) callconv(.C) void,
    /// complete a confirm prompt
    f_prompt_confirm_finish: ?*const fn (p_prompt: *gcr.Prompt, p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) callconv(.C) gcr.PromptReply,
    /// close a prompt
    f_prompt_close: ?*const fn (p_prompt: *gcr.Prompt) callconv(.C) void,

    pub fn as(p_instance: *PromptInterface, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const SecretExchangeClass = extern struct {
    pub const Instance = gcr.SecretExchange;

    f_parent_class: gobject.ObjectClass,
    f_generate_exchange_key: ?*const fn (p_exchange: *gcr.SecretExchange, p_scheme: [*:0]const u8, p_public_key: **u8, p_n_public_key: *usize) callconv(.C) c_int,
    f_derive_transport_key: ?*const fn (p_exchange: *gcr.SecretExchange, p_peer: *const u8, p_n_peer: usize) callconv(.C) c_int,
    f_encrypt_transport_data: ?*const fn (p_exchange: *gcr.SecretExchange, p_allocator: gck.Allocator, p_plain_text: *const u8, p_n_plain_text: usize, p_parameter: **u8, p_n_parameter: *usize, p_cipher_text: **u8, p_n_cipher_text: *usize) callconv(.C) c_int,
    f_decrypt_transport_data: ?*const fn (p_exchange: *gcr.SecretExchange, p_allocator: gck.Allocator, p_cipher_text: *const u8, p_n_cipher_text: usize, p_parameter: *const u8, p_n_parameter: usize, p_plain_text: **u8, p_n_plain_text: *usize) callconv(.C) c_int,
    f_dummy: [6]*anyopaque,

    pub fn as(p_instance: *SecretExchangeClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const SecretExchangePrivate = opaque {};

pub const SimpleCertificateClass = extern struct {
    pub const Instance = gcr.SimpleCertificate;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *SimpleCertificateClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const SimpleCertificatePrivate = opaque {};

pub const SshAskpassClass = extern struct {
    pub const Instance = gcr.SshAskpass;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *SshAskpassClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const SystemPromptClass = extern struct {
    pub const Instance = gcr.SystemPrompt;

    f_parent_class: gobject.ObjectClass,

    pub fn as(p_instance: *SystemPromptClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const SystemPromptPrivate = opaque {};

/// The class for `gcr.SystemPrompter`.
pub const SystemPrompterClass = extern struct {
    pub const Instance = gcr.SystemPrompter;

    /// parent class
    f_parent_class: gobject.ObjectClass,
    /// default handler for the `gcr.SystemPrompter.signals.new`-prompt signal
    f_new_prompt: ?*const fn (p_self: *gcr.SystemPrompter) callconv(.C) *gcr.Prompt,
    f_padding: [7]*anyopaque,

    pub fn as(p_instance: *SystemPrompterClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const SystemPrompterPrivate = opaque {};

/// The status of a built certificate chain. Will be set to
/// `GCR_CERTIFICATE_CHAIN_UNKNOWN` for certificate chains that have not been
/// built.
pub const CertificateChainStatus = enum(c_int) {
    unknown = 0,
    incomplete = 1,
    distrusted = 2,
    selfsigned = 3,
    pinned = 4,
    anchored = 5,
    _,
};

/// The format of a certificate request. Currently only PKCS`@"10"` is supported.
pub const CertificateRequestFormat = enum(c_int) {
    certificate_request_pkcs10 = 1,
    _,
};

/// Values responding to error codes for parsing and serializing data.
pub const DataError = enum(c_int) {
    failure = -1,
    unrecognized = 1,
    cancelled = 2,
    locked = 3,
    _,
};

/// The various format identifiers.
pub const DataFormat = enum(c_int) {
    all = -1,
    invalid = 0,
    der_private_key = 100,
    der_private_key_rsa = 101,
    der_private_key_dsa = 102,
    der_private_key_ec = 103,
    der_subject_public_key = 150,
    der_certificate_x509 = 200,
    der_pkcs7 = 300,
    der_pkcs8 = 400,
    der_pkcs8_plain = 401,
    der_pkcs8_encrypted = 402,
    der_pkcs10 = 450,
    der_spkac = 455,
    base64_spkac = 456,
    der_pkcs12 = 500,
    openssh_public = 600,
    openpgp_packet = 700,
    openpgp_armor = 701,
    pem = 1000,
    pem_private_key_rsa = 1001,
    pem_private_key_dsa = 1002,
    pem_certificate_x509 = 1003,
    pem_pkcs7 = 1004,
    pem_pkcs8_plain = 1005,
    pem_pkcs8_encrypted = 1006,
    pem_pkcs12 = 1007,
    pem_private_key = 1008,
    pem_pkcs10 = 1009,
    pem_private_key_ec = 1010,
    pem_public_key = 1011,
    _,
};

/// Various replies returned by `Prompt.confirm` and friends.
pub const PromptReply = enum(c_int) {
    cancel = 0,
    @"continue" = 1,
    _,
};

/// No error returned by the `gcr.SystemPrompt` is suitable for display or
/// to the user.
///
/// If the system prompter can only show one prompt at a time, and there is
/// already a prompt being displayed, and the timeout waiting to open the
/// prompt expires, then `GCR_SYSTEM_PROMPT_IN_PROGRESS` is returned.
pub const SystemPromptError = enum(c_int) {
    system_prompt_in_progress = 1,
    _,
};

/// The mode for the system prompter. Most system prompters can only show
/// one prompt at a time and would use the `GCR_SYSTEM_PROMPTER_SINGLE` mode.
pub const SystemPrompterMode = enum(c_int) {
    single = 0,
    multiple = 1,
    _,
};

/// Flags to be used with the `gcr.CertificateChain.build` operation.
pub const CertificateChainFlags = packed struct(c_uint) {
    no_lookups: bool = false,
    _padding1: bool = false,
    _padding2: bool = false,
    _padding3: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_none: CertificateChainFlags = @bitCast(@as(c_uint, 0));
    const flags_no_lookups: CertificateChainFlags = @bitCast(@as(c_uint, 1));
};

pub const CertificateSectionFlags = packed struct(c_uint) {
    important: bool = false,
    _padding1: bool = false,
    _padding2: bool = false,
    _padding3: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_none: CertificateSectionFlags = @bitCast(@as(c_uint, 0));
    const flags_important: CertificateSectionFlags = @bitCast(@as(c_uint, 1));
};

extern fn gcr_data_error_get_domain() glib.Quark;
pub const dataErrorGetDomain = gcr_data_error_get_domain;

/// Create a key fingerprint for a certificate, public key or private key.
/// Note that this is not a fingerprint of certificate data, which you would
/// use `gcr.Certificate.getFingerprint` for.
extern fn gcr_fingerprint_from_attributes(p_attrs: *gck.Attributes, p_checksum_type: glib.ChecksumType, p_n_fingerprint: *usize) ?[*]u8;
pub const fingerprintFromAttributes = gcr_fingerprint_from_attributes;

/// Create a key fingerprint for a DER encoded subjectPublicKeyInfo. The
/// fingerprint is created so that it will be identical for a key and its
/// corresponding certificate.
///
/// Note that in the case of certificates this is not a fingerprint of the
/// actual certificate data, but rather of the public key contained in a
/// certificate.
extern fn gcr_fingerprint_from_subject_public_key_info(p_key_info: [*]const u8, p_n_key_info: usize, p_checksum_type: glib.ChecksumType, p_n_fingerprint: *usize) ?[*]u8;
pub const fingerprintFromSubjectPublicKeyInfo = gcr_fingerprint_from_subject_public_key_info;

/// Disconnect the mock prompter
extern fn gcr_mock_prompter_disconnect() void;
pub const mockPrompterDisconnect = gcr_mock_prompter_disconnect;

/// Queue an expected response on the mock prompter.
///
/// Expects any prompt, and closes the prompt when it gets it.
extern fn gcr_mock_prompter_expect_close() void;
pub const mockPrompterExpectClose = gcr_mock_prompter_expect_close;

/// Queue an expected response on the mock prompter.
///
/// Expects a confirmation prompt, and then cancels that prompt.
extern fn gcr_mock_prompter_expect_confirm_cancel() void;
pub const mockPrompterExpectConfirmCancel = gcr_mock_prompter_expect_confirm_cancel;

/// Queue an expected response on the mock prompter.
///
/// Expects a confirmation prompt, and then confirms that prompt by
/// simulating a click on the ok button.
///
/// Additional property pairs for the prompt can be added in the argument
/// list, in the same way that you would with `gobject.Object.new`.
///
/// If the "choice-chosen" property is specified then that value will be
/// set on the prompt as if the user had changed the value.
///
/// All other properties will be checked against the prompt, and an error
/// will occur if they do not match the value set on the prompt.
extern fn gcr_mock_prompter_expect_confirm_ok(p_first_property_name: [*:0]const u8, ...) void;
pub const mockPrompterExpectConfirmOk = gcr_mock_prompter_expect_confirm_ok;

/// Queue an expected response on the mock prompter.
///
/// Expects a password prompt, and then cancels that prompt.
extern fn gcr_mock_prompter_expect_password_cancel() void;
pub const mockPrompterExpectPasswordCancel = gcr_mock_prompter_expect_password_cancel;

/// Queue an expected response on the mock prompter.
///
/// Expects a password prompt, and returns `password` as if the user had entered
/// it and clicked the ok button.
///
/// Additional property pairs for the prompt can be added in the argument
/// list, in the same way that you would with `gobject.Object.new`.
///
/// If the "choice-chosen" property is specified then that value will be
/// set on the prompt as if the user had changed the value.
///
/// All other properties will be checked against the prompt, and an error
/// will occur if they do not match the value set on the prompt.
extern fn gcr_mock_prompter_expect_password_ok(p_password: [*:0]const u8, p_first_property_name: [*:0]const u8, ...) void;
pub const mockPrompterExpectPasswordOk = gcr_mock_prompter_expect_password_ok;

/// Get the delay in milliseconds before the mock prompter completes
/// an expected prompt.
extern fn gcr_mock_prompter_get_delay_msec() c_uint;
pub const mockPrompterGetDelayMsec = gcr_mock_prompter_get_delay_msec;

/// Check if the mock prompter is expecting a response. This will be `TRUE`
/// when one of the <literal>gcr_mock_prompter_expect_xxx<!-- -->()</literal>
/// functions have been used to queue an expected prompt, but that prompt
/// response has not be 'used' yet.
extern fn gcr_mock_prompter_is_expecting() c_int;
pub const mockPrompterIsExpecting = gcr_mock_prompter_is_expecting;

/// Check if the mock prompter is showing any prompts.
extern fn gcr_mock_prompter_is_prompting() c_int;
pub const mockPrompterIsPrompting = gcr_mock_prompter_is_prompting;

/// Set the delay in milliseconds before the mock prompter completes
/// an expected prompt.
extern fn gcr_mock_prompter_set_delay_msec(p_delay_msec: c_uint) void;
pub const mockPrompterSetDelayMsec = gcr_mock_prompter_set_delay_msec;

/// Start the mock prompter. This is often used from the
/// <literal>setup<!-- -->()</literal> function of tests.
///
/// Starts the mock prompter in an additional thread. Use the returned DBus bus
/// name with `gcr.SystemPrompt.openForPrompter` to connect to this prompter.
extern fn gcr_mock_prompter_start() [*:0]const u8;
pub const mockPrompterStart = gcr_mock_prompter_start;

/// Stop the mock prompter. This is often used from the
/// <literal>teardown<!-- -->()</literal> function of tests.
extern fn gcr_mock_prompter_stop() void;
pub const mockPrompterStop = gcr_mock_prompter_stop;

/// Add a `gck.Module` to the list of PKCS`@"11"` modules that are used by the
/// GCR library.
///
/// It is not normally necessary to call this function. The available
/// PKCS`@"11"` modules installed on the system are automatically loaded
/// by the GCR library.
extern fn gcr_pkcs11_add_module(p_module: *gck.Module) void;
pub const pkcs11AddModule = gcr_pkcs11_add_module;

/// Initialize a PKCS`@"11"` module and add it to the modules that are
/// used by the GCR library. Note that is an error to initialize the same
/// PKCS`@"11"` module twice.
///
/// It is not normally necessary to call this function. The available
/// PKCS`@"11"` modules installed on the system are automatically loaded
/// by the GCR library.
extern fn gcr_pkcs11_add_module_from_file(p_module_path: [*:0]const u8, p_unused: ?*anyopaque, p_error: ?*?*glib.Error) c_int;
pub const pkcs11AddModuleFromFile = gcr_pkcs11_add_module_from_file;

/// List all the PKCS`@"11"` modules that are used by the GCR library.
/// Each module is a `gck.Module` object.
///
/// An empty list of modules will be returned if `pkcs11SetModules`,
/// or `pkcs11Initialize` has not yet run.
extern fn gcr_pkcs11_get_modules() *glib.List;
pub const pkcs11GetModules = gcr_pkcs11_get_modules;

/// List all the PKCS`@"11"` slots that are used by the GCR library for lookup
/// of trust assertions. Each slot is a `gck.Slot` object.
///
/// This will return an empty list if the `pkcs11Initialize` function has
/// not yet been called.
extern fn gcr_pkcs11_get_trust_lookup_slots() *glib.List;
pub const pkcs11GetTrustLookupSlots = gcr_pkcs11_get_trust_lookup_slots;

/// Get the PKCS`@"11"` URIs that are used to identify which slots to use for
/// lookup trust assertions.
extern fn gcr_pkcs11_get_trust_lookup_uris() ?[*][*:0]const u8;
pub const pkcs11GetTrustLookupUris = gcr_pkcs11_get_trust_lookup_uris;

/// Selects an appropriate PKCS`@"11"` slot to store trust assertions. The slot
/// to use is normally configured automatically by the system.
///
/// This will only return a valid result after the `pkcs11Initialize`
/// method has been called.
///
/// When done with the `gck.Slot`, use `gobject.Object.unref` to release it.
extern fn gcr_pkcs11_get_trust_store_slot() ?*gck.Slot;
pub const pkcs11GetTrustStoreSlot = gcr_pkcs11_get_trust_store_slot;

/// Get the PKCS`@"11"` URI that is used to identify which slot to use for
/// storing trust storage.
extern fn gcr_pkcs11_get_trust_store_uri() ?[*:0]const u8;
pub const pkcs11GetTrustStoreUri = gcr_pkcs11_get_trust_store_uri;

/// Asynchronously initialize the registered PKCS`@"11"` modules.
extern fn gcr_pkcs11_initialize(p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
pub const pkcs11Initialize = gcr_pkcs11_initialize;

/// Asynchronously initialize the registered PKCS`@"11"` modules.
extern fn gcr_pkcs11_initialize_async(p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
pub const pkcs11InitializeAsync = gcr_pkcs11_initialize_async;

/// Complete the asynchronous operation to initialize the registered PKCS`@"11"`
/// modules.
extern fn gcr_pkcs11_initialize_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
pub const pkcs11InitializeFinish = gcr_pkcs11_initialize_finish;

/// Set the list of PKCS`@"11"` modules that are used by the GCR library.
/// Each module in the list is a `gck.Module` object.
///
/// It is not normally necessary to call this function. The available
/// PKCS`@"11"` modules installed on the system are automatically loaded
/// by the GCR library.
extern fn gcr_pkcs11_set_modules(p_modules: *glib.List) void;
pub const pkcs11SetModules = gcr_pkcs11_set_modules;

/// Set the PKCS`@"11"` URIs that are used to identify which slots to use for
/// lookup of trust assertions.
///
/// It is not normally necessary to call this function. The relevant
/// PKCS`@"11"` slots are automatically configured by the GCR library.
extern fn gcr_pkcs11_set_trust_lookup_uris(p_pkcs11_uris: ?*[*:0]const u8) void;
pub const pkcs11SetTrustLookupUris = gcr_pkcs11_set_trust_lookup_uris;

/// Set the PKCS`@"11"` URI that is used to identify which slot to use for
/// storing trust assertions.
///
/// It is not normally necessary to call this function. The relevant
/// PKCS`@"11"` slot is automatically configured by the GCR library.
extern fn gcr_pkcs11_set_trust_store_uri(p_pkcs11_uri: ?[*:0]const u8) void;
pub const pkcs11SetTrustStoreUri = gcr_pkcs11_set_trust_store_uri;

/// Allocate a block of non-pageable memory.
///
/// If non-pageable memory cannot be allocated then normal memory will be
/// returned.
extern fn gcr_secure_memory_alloc(p_size: usize) ?*anyopaque;
pub const secureMemoryAlloc = gcr_secure_memory_alloc;

/// Free a block of non-pageable memory.
///
/// Glib memory is also freed correctly when passed to this function. If called
/// with a `NULL` pointer then no action is taken.
extern fn gcr_secure_memory_free(p_memory: ?*anyopaque) void;
pub const secureMemoryFree = gcr_secure_memory_free;

/// Check if a pointer is in non-pageable memory allocated by.
extern fn gcr_secure_memory_is_secure(p_memory: ?*anyopaque) c_int;
pub const secureMemoryIsSecure = gcr_secure_memory_is_secure;

/// Reallocate a block of non-pageable memory.
///
/// Glib memory is also reallocated correctly. If called with a null pointer,
/// then a new block of memory is allocated. If called with a zero size,
/// then the block of memory is freed.
///
/// If non-pageable memory cannot be allocated then normal memory will be
/// returned.
extern fn gcr_secure_memory_realloc(p_memory: ?*anyopaque, p_size: usize) ?*anyopaque;
pub const secureMemoryRealloc = gcr_secure_memory_realloc;

/// Copy a string into non-pageable memory. If the input string is `NULL`, then
/// `NULL` will be returned.
extern fn gcr_secure_memory_strdup(p_string: ?[*:0]const u8) [*:0]u8;
pub const secureMemoryStrdup = gcr_secure_memory_strdup;

/// Free a string, whether securely allocated using these functions or not.
/// This will also clear out the contents of the string so they do not
/// remain in memory.
extern fn gcr_secure_memory_strfree(p_string: ?[*:0]u8) void;
pub const secureMemoryStrfree = gcr_secure_memory_strfree;

/// Allocate a block of non-pageable memory.
///
/// If non-pageable memory cannot be allocated, then `NULL` is returned.
extern fn gcr_secure_memory_try_alloc(p_size: usize) ?*anyopaque;
pub const secureMemoryTryAlloc = gcr_secure_memory_try_alloc;

/// Reallocate a block of non-pageable memory.
///
/// Glib memory is also reallocated correctly when passed to this function.
/// If called with a null pointer, then a new block of memory is allocated.
/// If called with a zero size, then the block of memory is freed.
///
/// If memory cannot be allocated, `NULL` is returned and the original block
/// of memory remains intact.
extern fn gcr_secure_memory_try_realloc(p_memory: ?*anyopaque, p_size: usize) ?*anyopaque;
pub const secureMemoryTryRealloc = gcr_secure_memory_try_realloc;

/// Add a pinned `certificate` for connections to `peer` for `purpose`. A pinned
/// certificate overrides all other certificate verification and should be
/// used with care.
///
/// If the same pinned certificate already exists, then this operation
/// does not add another, and succeeds without error.
///
/// This call may block, see `gcr.trustAddPinnedCertificateAsync` for the
/// non-blocking version.
extern fn gcr_trust_add_pinned_certificate(p_certificate: *gcr.Certificate, p_purpose: [*:0]const u8, p_peer: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
pub const trustAddPinnedCertificate = gcr_trust_add_pinned_certificate;

/// Add a pinned certificate for communication with `peer` for `purpose`. A pinned
/// certificate overrides all other certificate verification and should be used
/// with care.
///
/// If the same pinned certificate already exists, then this operation
/// does not add another, and succeeds without error.
///
/// When the operation is finished, callback will be called. You can then call
/// `gcr.trustAddPinnedCertificateFinish` to get the result of the
/// operation.
extern fn gcr_trust_add_pinned_certificate_async(p_certificate: *gcr.Certificate, p_purpose: [*:0]const u8, p_peer: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
pub const trustAddPinnedCertificateAsync = gcr_trust_add_pinned_certificate_async;

/// Finishes an asynchronous operation started by
/// `gcr.trustAddPinnedCertificateAsync`.
extern fn gcr_trust_add_pinned_certificate_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
pub const trustAddPinnedCertificateFinish = gcr_trust_add_pinned_certificate_finish;

/// Check if the `certificate` is a trust anchor for the given `purpose`. A trust
/// anchor is used to verify the signatures on other certificates when verifying
/// a certificate chain. Also known as a trusted certificate authority.
///
/// This call may block, see `gcr.trustIsCertificateAnchoredAsync` for
/// the non-blocking version.
///
/// In the case of an error, `FALSE` is also returned. Check `error` to detect
/// if an error occurred.
extern fn gcr_trust_is_certificate_anchored(p_certificate: *gcr.Certificate, p_purpose: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
pub const trustIsCertificateAnchored = gcr_trust_is_certificate_anchored;

/// Check if the `certificate` is a trust anchor for the given `purpose`. A trust
/// anchor is used to verify the signatures on other certificates when verifying
/// a certificate chain. Also known as a trusted certificate authority.
///
/// When the operation is finished, callback will be called. You can then call
/// `gcr.trustIsCertificateAnchoredFinish` to get the result of the operation.
extern fn gcr_trust_is_certificate_anchored_async(p_certificate: *gcr.Certificate, p_purpose: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
pub const trustIsCertificateAnchoredAsync = gcr_trust_is_certificate_anchored_async;

/// Finishes an asynchronous operation started by
/// `gcr.trustIsCertificateAnchoredAsync`.
///
/// In the case of an error, `FALSE` is also returned. Check `error` to detect
/// if an error occurred.
extern fn gcr_trust_is_certificate_anchored_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
pub const trustIsCertificateAnchoredFinish = gcr_trust_is_certificate_anchored_finish;

/// Checks whether the certificate that can be uniquely identified with the
/// given `serial_nr` and `issuer` is marked as distrusted (for example by the
/// user, or because it's part of a CRL).
///
/// Since we can't directly use `Certificate` to fetch these values, you
/// need to call these with the raw serial number and issuer as provided by the
/// PKCS`@"11"` fields `CKA_SERIAL_NR` and `CKA_ISSUER`.
extern fn gcr_trust_is_certificate_distrusted(p_serial_nr: [*]u8, p_serial_nr_len: usize, p_issuer: [*]u8, p_issuer_len: usize, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
pub const trustIsCertificateDistrusted = gcr_trust_is_certificate_distrusted;

/// Asynchronously checks whether the certificate that can be uniquely
/// identified with the given `serial_nr` and `issuer` is marked as distrusted
/// (for example by the user, or because it's part of a CRL).
///
/// Since we can't directly use `Certificate` to fetch these values, you
/// need to call these with the raw serial number and issuer as provided by the
/// PKCS`@"11"` fields `CKA_SERIAL_NR` and `CKA_ISSUER`.
///
/// When the operation is finished, `callback` will be called. You can then call
/// `trustIsCertificateDistrustedFinish` to get the result of the
/// operation.
extern fn gcr_trust_is_certificate_distrusted_async(p_serial_nr: [*]u8, p_serial_nr_len: usize, p_issuer: [*]u8, p_issuer_len: usize, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
pub const trustIsCertificateDistrustedAsync = gcr_trust_is_certificate_distrusted_async;

/// Finishes an asynchronous operation started by
/// `trustIsCertificateDistrustedAsync`.
///
/// In the case of an error, `FALSE` is also returned. Check `error` to detect
/// if an error occurred.
extern fn gcr_trust_is_certificate_distrusted_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
pub const trustIsCertificateDistrustedFinish = gcr_trust_is_certificate_distrusted_finish;

/// Check if `certificate` is pinned for `purpose` to communicate with `peer`.
/// A pinned certificate overrides all other certificate verification.
///
/// This call may block, see `gcr.trustIsCertificatePinnedAsync` for the
/// non-blocking version.
///
/// In the case of an error, `FALSE` is also returned. Check `error` to detect
/// if an error occurred.
extern fn gcr_trust_is_certificate_pinned(p_certificate: *gcr.Certificate, p_purpose: [*:0]const u8, p_peer: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
pub const trustIsCertificatePinned = gcr_trust_is_certificate_pinned;

/// Check if `certificate` is pinned for `purpose` to communicate with `peer`. A
/// pinned certificate overrides all other certificate verification.
///
/// When the operation is finished, callback will be called. You can then call
/// `gcr.trustIsCertificatePinnedFinish` to get the result of the
/// operation.
extern fn gcr_trust_is_certificate_pinned_async(p_certificate: *gcr.Certificate, p_purpose: [*:0]const u8, p_peer: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
pub const trustIsCertificatePinnedAsync = gcr_trust_is_certificate_pinned_async;

/// Finishes an asynchronous operation started by
/// `gcr.trustIsCertificatePinnedAsync`.
///
/// In the case of an error, `FALSE` is also returned. Check `error` to detect
/// if an error occurred.
extern fn gcr_trust_is_certificate_pinned_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
pub const trustIsCertificatePinnedFinish = gcr_trust_is_certificate_pinned_finish;

/// Remove a pinned certificate for communication with `peer` for `purpose`.
///
/// If the same pinned certificate does not exist, or was already removed,
/// then this operation succeeds without error.
///
/// This call may block, see `gcr.trustRemovePinnedCertificateAsync` for the
/// non-blocking version.
extern fn gcr_trust_remove_pinned_certificate(p_certificate: *gcr.Certificate, p_purpose: [*:0]const u8, p_peer: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
pub const trustRemovePinnedCertificate = gcr_trust_remove_pinned_certificate;

/// Remove a pinned certificate for communication with `peer` for `purpose`.
///
/// If the same pinned certificate does not exist, or was already removed,
/// then this operation succeeds without error.
///
/// When the operation is finished, callback will be called. You can then call
/// `gcr.trustRemovePinnedCertificateFinish` to get the result of the
/// operation.
extern fn gcr_trust_remove_pinned_certificate_async(p_certificate: *gcr.Certificate, p_purpose: [*:0]const u8, p_peer: [*:0]const u8, p_cancellable: ?*gio.Cancellable, p_callback: ?gio.AsyncReadyCallback, p_user_data: ?*anyopaque) void;
pub const trustRemovePinnedCertificateAsync = gcr_trust_remove_pinned_certificate_async;

/// Finishes an asynchronous operation started by
/// `gcr.trustRemovePinnedCertificateAsync`.
extern fn gcr_trust_remove_pinned_certificate_finish(p_result: *gio.AsyncResult, p_error: ?*?*glib.Error) c_int;
pub const trustRemovePinnedCertificateFinish = gcr_trust_remove_pinned_certificate_finish;

/// The major version number of the Gcr library.
pub const MAJOR_VERSION = 4;
/// The micro version number of the Gcr library.
pub const MICRO_VERSION = 0;
/// The minor version number of the Gcr library.
pub const MINOR_VERSION = 3;
/// The purpose used to verify the client certificate in a TLS connection.
pub const PURPOSE_CLIENT_AUTH = "1.3.6.1.5.5.7.3.2";
/// The purpose used to verify certificate used for the signature on signed code.
pub const PURPOSE_CODE_SIGNING = "1.3.6.1.5.5.7.3.3";
/// The purpose used to verify certificates that are used in email communication
/// such as S/MIME.
pub const PURPOSE_EMAIL = "1.3.6.1.5.5.7.3.4";
/// The purpose used to verify the server certificate in a TLS connection. This
/// is the most common purpose in use.
pub const PURPOSE_SERVER_AUTH = "1.3.6.1.5.5.7.3.1";
/// The current secret exchange protocol. Key agreement is done using DH with the
/// 1536 bit IKE parameter group. Keys are derived using SHA256 with HKDF. The
/// transport encryption is done with 128 bit AES.
pub const SECRET_EXCHANGE_PROTOCOL_1 = "sx-aes-1";
pub const UNLOCK_OPTION_ALWAYS = "always";
pub const UNLOCK_OPTION_IDLE = "idle";
pub const UNLOCK_OPTION_SESSION = "session";
pub const UNLOCK_OPTION_TIMEOUT = "timeout";
