/* vim: set ro : */

/*
 * libsyd: syd /dev/syd interface implementation
 * Copyright (c) 2023, 2024 Ali Polatel <alip@chesswob.org>
 * SPDX-License-Identifier: LGPL-3.0
 */


#ifndef LIBSYD_3_SYD_H
#define LIBSYD_3_SYD_H

/* THIS IS A GENERATED FILE. DO NOT EDIT THIS FILE DIRECTLY */


#include <stdarg.h>
#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>
#include <stdlib.h>


/**
 * `action_t` type represents possible sandboxing action values.
 */
typedef uint8_t action_t;

/**
 * `lock_state_t_t` type represents possible states for the sandbox lock.
 */
typedef uint8_t lock_state_t;

/**
 * Deny system call, warn and abort the offending process.
 */
#define ACTION_ABORT 6

/**
 * Allow system call.
 */
#define ACTION_ALLOW 0

/**
 * Deny system call and warn.
 */
#define ACTION_DENY 3

/**
 * Warn, and exit Syd immediately with deny errno as exit value.
 */
#define ACTION_EXIT 8

/**
 * Deny system call silently.
 */
#define ACTION_FILTER 2

/**
 * Deny system call, warn and kill the offending process.
 */
#define ACTION_KILL 7

/**
 * Deny system call, warn and panic the current Syd thread.
 */
#define ACTION_PANIC 4

/**
 * Deny system call, warn and stop the offending process.
 */
#define ACTION_STOP 5

/**
 * Allow system call and warn.
 */
#define ACTION_WARN 1

/**
 * The sandbox lock is set to on for all processes except the initial
 * process (syd exec child).
 */
#define LOCK_EXEC 1

/**
 * The sandbox lock is off, allowing all sandbox commands.
 */
#define LOCK_OFF 0

/**
 * The sandbox lock is on, disallowing all sandbox commands.
 */
#define LOCK_ON 2

/**
 * Performs a syd API check
 *
 * The caller is advised to perform this check before
 * calling any other syd API calls.
 *
 * Returns API number on success, negated errno on failure.
 */
int syd_api(void);

/**
 * Adds to the given actionlist of chattr sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_chattr_add(action_t action, const char *glob);

/**
 * Removes the first instance from the end of the given actionlist of
 * chattr sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_chattr_del(action_t action, const char *glob);

/**
 * Removes all matching patterns from the given actionlist of chattr sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_chattr_rem(action_t action, const char *glob);

/**
 * Adds to the given actionlist of chdir sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_chdir_add(action_t action, const char *glob);

/**
 * Removes the first instance from the end of the given actionlist of
 * chdir sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_chdir_del(action_t action, const char *glob);

/**
 * Removes all matching patterns from the given actionlist of chdir sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_chdir_rem(action_t action, const char *glob);

/**
 * Performs an lstat system call on the file "/dev/syd".
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_check(void);

/**
 * Adds to the given actionlist of chgrp sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_chgrp_add(action_t action, const char *glob);

/**
 * Removes the first instance from the end of the given actionlist of
 * chgrp sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_chgrp_del(action_t action, const char *glob);

/**
 * Removes all matching patterns from the given actionlist of chgrp sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_chgrp_rem(action_t action, const char *glob);

/**
 * Adds to the given actionlist of chmod sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_chmod_add(action_t action, const char *glob);

/**
 * Removes the first instance from the end of the given actionlist of
 * chmod sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_chmod_del(action_t action, const char *glob);

/**
 * Removes all matching patterns from the given actionlist of chmod sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_chmod_rem(action_t action, const char *glob);

/**
 * Adds to the given actionlist of chown sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_chown_add(action_t action, const char *glob);

/**
 * Removes the first instance from the end of the given actionlist of
 * chown sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_chown_del(action_t action, const char *glob);

/**
 * Removes all matching patterns from the given actionlist of chown sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_chown_rem(action_t action, const char *glob);

/**
 * Adds to the given actionlist of chroot sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_chroot_add(action_t action, const char *glob);

/**
 * Removes the first instance from the end of the given actionlist of
 * chroot sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_chroot_del(action_t action, const char *glob);

/**
 * Removes all matching patterns from the given actionlist of chroot sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_chroot_rem(action_t action, const char *glob);

/**
 * Adds to the given actionlist of create sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_create_add(action_t action, const char *glob);

/**
 * Removes the first instance from the end of the given actionlist of
 * create sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_create_del(action_t action, const char *glob);

/**
 * Removes all matching patterns from the given actionlist of create sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_create_rem(action_t action, const char *glob);

/**
 * Set the default action for IP blocklist violations.
 */
int syd_default_block(action_t action);

/**
 * Set the default action for Chattr Sandboxing.
 */
int syd_default_chattr(action_t action);

/**
 * Set the default action for Chdir Sandboxing.
 */
int syd_default_chdir(action_t action);

/**
 * Set the default action for Chgrp Sandboxing.
 */
int syd_default_chgrp(action_t action);

/**
 * Set the default action for Chmod Sandboxing.
 */
int syd_default_chmod(action_t action);

/**
 * Set the default action for Chown Sandboxing.
 */
int syd_default_chown(action_t action);

/**
 * Set the default action for Chroot Sandboxing.
 */
int syd_default_chroot(action_t action);

/**
 * Set the default action for Create Sandboxing.
 */
int syd_default_create(action_t action);

/**
 * Set the default action for Delete Sandboxing.
 */
int syd_default_delete(action_t action);

/**
 * Set the default action for Exec Sandboxing.
 */
int syd_default_exec(action_t action);

/**
 * Set the default action for Force Sandboxing.
 */
int syd_default_force(action_t action);

/**
 * Set the default action for Filesystem sandboxing.
 */
int syd_default_fs(action_t action);

/**
 * Set the default action for Ioctl Sandboxing.
 */
int syd_default_ioctl(action_t action);

/**
 * Set the default action for Memory Sandboxing.
 */
int syd_default_mem(action_t action);

/**
 * Set the default action for Mkbdev Sandboxing.
 */
int syd_default_mkbdev(action_t action);

/**
 * Set the default action for Mkcdev Sandboxing.
 */
int syd_default_mkcdev(action_t action);

/**
 * Set the default action for Mkdir Sandboxing.
 */
int syd_default_mkdir(action_t action);

/**
 * Set the default action for Mkfifo Sandboxing.
 */
int syd_default_mkfifo(action_t action);

/**
 * Set the default action for Mktemp Sandboxing.
 */
int syd_default_mktemp(action_t action);

/**
 * Set the default action for Network Sandboxing.
 */
int syd_default_net(action_t action);

/**
 * Set the default action for PID Sandboxing.
 */
int syd_default_pid(action_t action);

/**
 * Set the default action for Read Sandboxing.
 */
int syd_default_read(action_t action);

/**
 * Set the default action for Readdir Sandboxing.
 */
int syd_default_readdir(action_t action);

/**
 * Set the default action for Rename Sandboxing.
 */
int syd_default_rename(action_t action);

/**
 * Set the default action for Rmdir Sandboxing.
 */
int syd_default_rmdir(action_t action);

/**
 * Set the default action for SegvGuard
 */
int syd_default_segvguard(action_t action);

/**
 * Set the default action for Stat Sandboxing.
 */
int syd_default_stat(action_t action);

/**
 * Set the default action for Symlink Sandboxing.
 */
int syd_default_symlink(action_t action);

/**
 * Set the default action for TPE Sandboxing.
 */
int syd_default_tpe(action_t action);

/**
 * Set the default action for Truncate Sandboxing.
 */
int syd_default_truncate(action_t action);

/**
 * Set the default action for Utime Sandboxing.
 */
int syd_default_utime(action_t action);

/**
 * Set the default action for Walk Sandboxing.
 */
int syd_default_walk(action_t action);

/**
 * Set the default action for Write Sandboxing.
 */
int syd_default_write(action_t action);

/**
 * Adds to the given actionlist of delete sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_delete_add(action_t action, const char *glob);

/**
 * Removes the first instance from the end of the given actionlist of
 * delete sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_delete_del(action_t action, const char *glob);

/**
 * Removes all matching patterns from the given actionlist of delete sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_delete_rem(action_t action, const char *glob);

/**
 * Disable chattr sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_chattr(void);

/**
 * Disable chdir sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_chdir(void);

/**
 * Disable chgrp sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_chgrp(void);

/**
 * Disable chmod sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_chmod(void);

/**
 * Disable chown sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_chown(void);

/**
 * Disable chroot sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_chroot(void);

/**
 * Disable create sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_create(void);

/**
 * Disable delete sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_delete(void);

/**
 * Disable exec sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_exec(void);

/**
 * Disable force sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_force(void);

/**
 * Disable Filesystem sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_fs(void);

/**
 * Disable ioctl sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_ioctl(void);

/**
 * Disable memory sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_mem(void);

/**
 * Disable mkbdev sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_mkbdev(void);

/**
 * Disable mkcdev sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_mkcdev(void);

/**
 * Disable mkdir sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_mkdir(void);

/**
 * Disable mkfifo sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_mkfifo(void);

/**
 * Disable mktemp sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_mktemp(void);

/**
 * Disable net sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_net(void);

/**
 * Disable PID sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_pid(void);

/**
 * Disable read sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_read(void);

/**
 * Disable readdir sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_readdir(void);

/**
 * Disable rename sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_rename(void);

/**
 * Disable rmdir sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_rmdir(void);

/**
 * Disable stat sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_stat(void);

/**
 * Disable symlink sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_symlink(void);

/**
 * Disable TPE sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_tpe(void);

/**
 * Disable truncate sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_truncate(void);

/**
 * Disable utime sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_utime(void);

/**
 * Disable walk sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_walk(void);

/**
 * Disable write sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_disable_write(void);

/**
 * Enable chattr sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_chattr(void);

/**
 * Enable chdir sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_chdir(void);

/**
 * Enable chgrp sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_chgrp(void);

/**
 * Enable chmod sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_chmod(void);

/**
 * Enable chown sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_chown(void);

/**
 * Enable chroot sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_chroot(void);

/**
 * Enable create sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_create(void);

/**
 * Enable delete sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_delete(void);

/**
 * Enable exec sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_exec(void);

/**
 * Enable force sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_force(void);

/**
 * Enable Filesystem sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_fs(void);

/**
 * Enable ioctl sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_ioctl(void);

/**
 * Enable memory sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_mem(void);

/**
 * Enable mkbdev sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_mkbdev(void);

/**
 * Enable mkcdev sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_mkcdev(void);

/**
 * Enable mkdir sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_mkdir(void);

/**
 * Enable mkfifo sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_mkfifo(void);

/**
 * Enable mktemp sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_mktemp(void);

/**
 * Enable net sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_net(void);

/**
 * Enable PID sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_pid(void);

/**
 * Enable read sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_read(void);

/**
 * Enable readdir sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_readdir(void);

/**
 * Enable rename sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_rename(void);

/**
 * Enable rmdir sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_rmdir(void);

/**
 * Enable stat sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_stat(void);

/**
 * Enable symlink sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_symlink(void);

/**
 * Enable TPE sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_tpe(void);

/**
 * Enable truncate sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_truncate(void);

/**
 * Enable utime sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_utime(void);

/**
 * Enable walk sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_walk(void);

/**
 * Enable write sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_enable_write(void);

/**
 * Checks if chattr sandboxing is enabled.
 *
 * Returns true if chattr sandboxing is enabled, false otherwise.
 */
bool syd_enabled_chattr(void);

/**
 * Checks if chdir sandboxing is enabled.
 *
 * Returns true if chdir sandboxing is enabled, false otherwise.
 */
bool syd_enabled_chdir(void);

/**
 * Checks if chgrp sandboxing is enabled.
 *
 * Returns true if chgrp sandboxing is enabled, false otherwise.
 */
bool syd_enabled_chgrp(void);

/**
 * Checks if chmod sandboxing is enabled.
 *
 * Returns true if chmod sandboxing is enabled, false otherwise.
 */
bool syd_enabled_chmod(void);

/**
 * Checks if chown sandboxing is enabled.
 *
 * Returns true if chown sandboxing is enabled, false otherwise.
 */
bool syd_enabled_chown(void);

/**
 * Checks if chroot sandboxing is enabled.
 *
 * Returns true if chroot sandboxing is enabled, false otherwise.
 */
bool syd_enabled_chroot(void);

/**
 * Checks if create sandboxing is enabled.
 *
 * Returns true if create sandboxing is enabled, false otherwise.
 */
bool syd_enabled_create(void);

/**
 * Checks if crypt sandboxing is enabled.
 *
 * Returns true if crypt sandboxing is enabled, false otherwise.
 */
bool syd_enabled_crypt(void);

/**
 * Checks if delete sandboxing is enabled.
 *
 * Returns true if delete sandboxing is enabled, false otherwise.
 */
bool syd_enabled_delete(void);

/**
 * Checks if exec sandboxing is enabled.
 *
 * Returns true if exec sandboxing is enabled, false otherwise.
 */
bool syd_enabled_exec(void);

/**
 * Checks if force sandboxing is enabled.
 *
 * Returns true if force sandboxing is enabled, false otherwise.
 */
bool syd_enabled_force(void);

/**
 * Checks if Filesystem sandboxing is enabled.
 *
 * Returns true if Filesystem sandboxing is enabled, false otherwise.
 */
bool syd_enabled_fs(void);

/**
 * Checks if ioctl sandboxing is enabled.
 *
 * Returns true if ioctl sandboxing is enabled, false otherwise.
 */
bool syd_enabled_ioctl(void);

/**
 * Checks if lock sandboxing is enabled.
 *
 * Returns true if lock sandboxing is enabled, false otherwise.
 */
bool syd_enabled_lock(void);

/**
 * Checks if memory sandboxing is enabled.
 *
 * Returns true if memory sandboxing is enabled, false otherwise.
 */
bool syd_enabled_mem(void);

/**
 * Checks if mkbdev sandboxing is enabled.
 *
 * Returns true if mkbdev sandboxing is enabled, false otherwise.
 */
bool syd_enabled_mkbdev(void);

/**
 * Checks if mkcdev sandboxing is enabled.
 *
 * Returns true if mkcdev sandboxing is enabled, false otherwise.
 */
bool syd_enabled_mkcdev(void);

/**
 * Checks if mkdir sandboxing is enabled.
 *
 * Returns true if mkdir sandboxing is enabled, false otherwise.
 */
bool syd_enabled_mkdir(void);

/**
 * Checks if mkfifo sandboxing is enabled.
 *
 * Returns true if mkfifo sandboxing is enabled, false otherwise.
 */
bool syd_enabled_mkfifo(void);

/**
 * Checks if mktemp sandboxing is enabled.
 *
 * Returns true if mktemp sandboxing is enabled, false otherwise.
 */
bool syd_enabled_mktemp(void);

/**
 * Checks if net sandboxing is enabled.
 *
 * Returns true if net sandboxing is enabled, false otherwise.
 */
bool syd_enabled_net(void);

/**
 * Checks if PID sandboxing is enabled.
 *
 * Returns true if PID sandboxing is enabled, false otherwise.
 */
bool syd_enabled_pid(void);

/**
 * Checks if proxy sandboxing is enabled.
 *
 * Returns true if proxy sandboxing is enabled, false otherwise.
 */
bool syd_enabled_proxy(void);

/**
 * Checks if read sandboxing is enabled.
 *
 * Returns true if read sandboxing is enabled, false otherwise.
 */
bool syd_enabled_read(void);

/**
 * Checks if readdir sandboxing is enabled.
 *
 * Returns true if readdir sandboxing is enabled, false otherwise.
 */
bool syd_enabled_readdir(void);

/**
 * Checks if rename sandboxing is enabled.
 *
 * Returns true if rename sandboxing is enabled, false otherwise.
 */
bool syd_enabled_rename(void);

/**
 * Checks if rmdir sandboxing is enabled.
 *
 * Returns true if rmdir sandboxing is enabled, false otherwise.
 */
bool syd_enabled_rmdir(void);

/**
 * Checks if stat sandboxing is enabled.
 *
 * Returns true if stat sandboxing is enabled, false otherwise.
 */
bool syd_enabled_stat(void);

/**
 * Checks if symlink sandboxing is enabled.
 *
 * Returns true if symlink sandboxing is enabled, false otherwise.
 */
bool syd_enabled_symlink(void);

/**
 * Checks if TPE sandboxing is enabled.
 *
 * Returns true if TPE sandboxing is enabled, false otherwise.
 */
bool syd_enabled_tpe(void);

/**
 * Checks if truncate sandboxing is enabled.
 *
 * Returns true if truncate sandboxing is enabled, false otherwise.
 */
bool syd_enabled_truncate(void);

/**
 * Checks if utime sandboxing is enabled.
 *
 * Returns true if utime sandboxing is enabled, false otherwise.
 */
bool syd_enabled_utime(void);

/**
 * Checks if walk sandboxing is enabled.
 *
 * Returns true if walk sandboxing is enabled, false otherwise.
 */
bool syd_enabled_walk(void);

/**
 * Checks if write sandboxing is enabled.
 *
 * Returns true if write sandboxing is enabled, false otherwise.
 */
bool syd_enabled_write(void);

/**
 * Execute a command outside the sandbox without sandboxing
 *
 * # Safety
 *
 * This function is marked `unsafe` because it dereferences raw
 * pointers, which is inherently unsafe in Rust.
 *
 * The caller must ensure the following conditions are met to safely
 * use this function:
 *
 * 1. The `file` pointer must point to a valid, null-terminated C-style
 *    string.
 *
 * 2. The `argv` pointer must point to an array of pointers, where each
 *    pointer refers to a valid, null-terminated C-style string. The
 *    last pointer in the array must be null, indicating the end of the
 *    array.
 *
 * 3. The memory pointed to by `file` and `argv` must remain valid for
 *    the duration of the call.
 *
 * Failing to uphold these guarantees can lead to undefined behavior,
 * including memory corruption and data races.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_exec(const char *file, const char *const *argv);

/**
 * Adds to the given actionlist of exec sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_exec_add(action_t action, const char *glob);

/**
 * Removes the first instance from the end of the given actionlist of
 * exec sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_exec_del(action_t action, const char *glob);

/**
 * Removes all matching patterns from the given actionlist of exec sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_exec_rem(action_t action, const char *glob);

/**
 * Adds an entry to the Integrity Force map for Force Sandboxing.
 *
 * # Safety
 *
 * This function is marked `unsafe` because it dereferences raw
 * pointers, which is inherently unsafe in Rust.
 *
 * The caller must ensure the following conditions are met to safely
 * use this function:
 *
 * 1. The `path` pointer must point to a valid, null-terminated C-style
 *    string.
 * 2. The `hash` pointer must point to a valid, null-terminated C-style
 *    string.
 */
int syd_force_add(const char *path, const char *hash, action_t action);

/**
 * Clears the Integrity Force map for Force Sandboxing.
 */
int syd_force_clr(void);

/**
 * Removes an entry from the Integrity Force map for Force Sandboxing.
 * # Safety
 *
 * This function is marked `unsafe` because it dereferences raw
 * pointers, which is inherently unsafe in Rust.
 *
 * The caller must ensure the following conditions are met to safely
 * use this function:
 *
 * 1. The `path` pointer must point to a valid, null-terminated C-style
 *    string.
 */
int syd_force_del(const char *path);

/**
 * Adds to the given actionlist of Filesystem sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_fs_add(action_t action, const char *name);

/**
 * Removes the first instance from the end of the given actionlist of
 * Filesystem sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_fs_del(action_t action, const char *name);

/**
 * Removes all matching patterns from the given actionlist of Filesystem sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_fs_rem(action_t action, const char *name);

/**
 * Adds a request to the _ioctl_(2) denylist.
 */
int syd_ioctl_deny(uint64_t request);

/**
 * Causes syd to read configuration from the given file descriptor.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_load(int fd);

/**
 * Sets the state of the sandbox lock.
 *
 * state: The desired state of the sandbox lock.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_lock(lock_state_t state);

/**
 * Set syd maximum per-process memory usage limit for memory sandboxing.
 *
 * parse-size crate is used to parse the value so formatted strings are OK.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_mem_max(const char *size);

/**
 * Set syd maximum per-process virtual memory usage limit for memory sandboxing.
 *
 * parse-size crate is used to parse the value so formatted strings are OK.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_mem_vm_max(const char *size);

/**
 * Adds to the given actionlist of mkbdev sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_mkbdev_add(action_t action, const char *glob);

/**
 * Removes the first instance from the end of the given actionlist of
 * mkbdev sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_mkbdev_del(action_t action, const char *glob);

/**
 * Removes all matching patterns from the given actionlist of mkbdev sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_mkbdev_rem(action_t action, const char *glob);

/**
 * Adds to the given actionlist of mkcdev sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_mkcdev_add(action_t action, const char *glob);

/**
 * Removes the first instance from the end of the given actionlist of
 * mkcdev sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_mkcdev_del(action_t action, const char *glob);

/**
 * Removes all matching patterns from the given actionlist of mkcdev sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_mkcdev_rem(action_t action, const char *glob);

/**
 * Adds to the given actionlist of mkdir sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_mkdir_add(action_t action, const char *glob);

/**
 * Removes the first instance from the end of the given actionlist of
 * mkdir sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_mkdir_del(action_t action, const char *glob);

/**
 * Removes all matching patterns from the given actionlist of mkdir sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_mkdir_rem(action_t action, const char *glob);

/**
 * Adds to the given actionlist of mkfifo sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_mkfifo_add(action_t action, const char *glob);

/**
 * Removes the first instance from the end of the given actionlist of
 * mkfifo sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_mkfifo_del(action_t action, const char *glob);

/**
 * Removes all matching patterns from the given actionlist of mkfifo sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_mkfifo_rem(action_t action, const char *glob);

/**
 * Adds to the given actionlist of mktemp sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_mktemp_add(action_t action, const char *glob);

/**
 * Removes the first instance from the end of the given actionlist of
 * mktemp sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_mktemp_del(action_t action, const char *glob);

/**
 * Removes all matching patterns from the given actionlist of mktemp sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_mktemp_rem(action_t action, const char *glob);

/**
 * Adds to the given actionlist of net/bind sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_net_bind_add(action_t action, const char *glob);

/**
 * Removes the first instance from the end of the given actionlist of
 * net/bind sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_net_bind_del(action_t action, const char *glob);

/**
 * Removes all matching patterns from the given actionlist of net/bind sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_net_bind_rem(action_t action, const char *glob);

/**
 * Adds to the given actionlist of net/connect sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_net_connect_add(action_t action, const char *glob);

/**
 * Removes the first instance from the end of the given actionlist of
 * net/connect sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_net_connect_del(action_t action, const char *glob);

/**
 * Removes all matching patterns from the given actionlist of net/connect sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_net_connect_rem(action_t action, const char *glob);

/**
 * Adds to the given actionlist of net/link sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_net_link_add(action_t action, const char *family);

/**
 * Removes the first instance from the end of the given actionlist of
 * net/link sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_net_link_del(action_t action, const char *family);

/**
 * Removes all matching patterns from the given actionlist of net/link sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_net_link_rem(action_t action, const char *family);

/**
 * Adds to the given actionlist of net/sendfd sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_net_sendfd_add(action_t action, const char *glob);

/**
 * Removes the first instance from the end of the given actionlist of
 * net/sendfd sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_net_sendfd_del(action_t action, const char *glob);

/**
 * Removes all matching patterns from the given actionlist of net/sendfd sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_net_sendfd_rem(action_t action, const char *glob);

/**
 * Causes syd to exit immediately with code 127
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_panic(void);

/**
 * Set syd maximum process id limit for PID sandboxing
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_pid_max(size_t size);

/**
 * Adds to the given actionlist of read sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_read_add(action_t action, const char *glob);

/**
 * Removes the first instance from the end of the given actionlist of
 * read sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_read_del(action_t action, const char *glob);

/**
 * Removes all matching patterns from the given actionlist of read sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_read_rem(action_t action, const char *glob);

/**
 * Adds to the given actionlist of readdir sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_readdir_add(action_t action, const char *glob);

/**
 * Removes the first instance from the end of the given actionlist of
 * readdir sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_readdir_del(action_t action, const char *glob);

/**
 * Removes all matching patterns from the given actionlist of readdir sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_readdir_rem(action_t action, const char *glob);

/**
 * Adds to the given actionlist of rename sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_rename_add(action_t action, const char *glob);

/**
 * Removes the first instance from the end of the given actionlist of
 * rename sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_rename_del(action_t action, const char *glob);

/**
 * Removes all matching patterns from the given actionlist of rename sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_rename_rem(action_t action, const char *glob);

/**
 * Causes syd to reset sandboxing to the default state.
 * Allowlists, denylists and filters are going to be cleared.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_reset(void);

/**
 * Adds to the given actionlist of rmdir sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_rmdir_add(action_t action, const char *glob);

/**
 * Removes the first instance from the end of the given actionlist of
 * rmdir sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_rmdir_del(action_t action, const char *glob);

/**
 * Removes all matching patterns from the given actionlist of rmdir sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_rmdir_rem(action_t action, const char *glob);

/**
 * Specify SegvGuard entry expiry timeout in seconds.
 * Setting this timeout to 0 effectively disables SegvGuard.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_segvguard_expiry(uint64_t timeout);

/**
 * Specify SegvGuard max number of crashes before suspension.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_segvguard_maxcrashes(uint8_t max);

/**
 * Specify SegvGuard entry suspension timeout in seconds.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_segvguard_suspension(uint64_t timeout);

/**
 * Adds to the given actionlist of stat sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_stat_add(action_t action, const char *glob);

/**
 * Removes the first instance from the end of the given actionlist of
 * stat sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_stat_del(action_t action, const char *glob);

/**
 * Removes all matching patterns from the given actionlist of stat sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_stat_rem(action_t action, const char *glob);

/**
 * Adds to the given actionlist of symlink sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_symlink_add(action_t action, const char *glob);

/**
 * Removes the first instance from the end of the given actionlist of
 * symlink sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_symlink_del(action_t action, const char *glob);

/**
 * Removes all matching patterns from the given actionlist of symlink sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_symlink_rem(action_t action, const char *glob);

/**
 * Adds to the given actionlist of truncate sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_truncate_add(action_t action, const char *glob);

/**
 * Removes the first instance from the end of the given actionlist of
 * truncate sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_truncate_del(action_t action, const char *glob);

/**
 * Removes all matching patterns from the given actionlist of truncate sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_truncate_rem(action_t action, const char *glob);

/**
 * Adds to the given actionlist of utime sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_utime_add(action_t action, const char *glob);

/**
 * Removes the first instance from the end of the given actionlist of
 * utime sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_utime_del(action_t action, const char *glob);

/**
 * Removes all matching patterns from the given actionlist of utime sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_utime_rem(action_t action, const char *glob);

/**
 * Adds to the given actionlist of walk sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_walk_add(action_t action, const char *glob);

/**
 * Removes the first instance from the end of the given actionlist of
 * walk sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_walk_del(action_t action, const char *glob);

/**
 * Removes all matching patterns from the given actionlist of walk sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_walk_rem(action_t action, const char *glob);

/**
 * Adds to the given actionlist of write sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_write_add(action_t action, const char *glob);

/**
 * Removes the first instance from the end of the given actionlist of
 * write sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_write_del(action_t action, const char *glob);

/**
 * Removes all matching patterns from the given actionlist of write sandboxing.
 *
 * Returns 0 on success, negated errno on failure.
 */
int syd_write_rem(action_t action, const char *glob);

#endif  /* LIBSYD_3_SYD_H */
