% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/feature_search.R
\name{feature_search}
\alias{feature_search}
\title{General Feature Searching Engine}
\usage{
feature_search(
  attributes,
  fun,
  data,
  mode = c("greedy", "exhaustive"),
  type = c("forward", "backward"),
  sizes = 1:length(attributes),
  parallel = TRUE,
  ...
)
}
\arguments{
\item{attributes}{A character vector with attributes' names to be used to extract the most valuable features.}

\item{fun}{A function (evaluator) to be used to score features' sets at each iteration of the algorithm passed via \code{mode}.
See Examples.}

\item{data}{A data set for \code{fun} function (evaluator).}

\item{mode}{A character that determines which search algorithm to perform. Defualt is \code{"greedy"}.}

\item{type}{Used when \code{mode = "greedy"} - whether to use the
\code{backward} or the \code{forward} multiple-way search. Default is \code{"forward"}.}

\item{sizes}{Used when \code{mode = "exhaustive"} - a vector of sizes
of attributes subsets.}

\item{parallel}{Allow parallelization.}

\item{\dots}{Other arguments passed to \link[foreach]{foreach} function.}
}
\value{
A list with following components
\itemize{
  \item best - a \link{data.frame} with the best subset and it's score (1 - feature used, 0 - feature not used),
  \item all - a \link{data.frame} with all checked features' subsets and their score (1 - feature used, 0 - feature not used),
  \item data - the data used in the feature selection,
  \item fun - the evaluator used to compute the score of importance for features' subsets,
  \item call - an origin call of the \code{feature_search},
  \item mode - the mode used in the call.
}
}
\description{
A convenience wrapper for \code{greedy} and \code{exhaustive} feature selection algorithms that
extract valuable attributes depending on the evaluation method (called evaluator). This function
is a reimplementation of \pkg{FSelector}'s \link[FSelector]{exhaustive.search} and \link[FSelector]{greedy.search}.
}
\details{
The evaluator function passed with \code{fun} is used to determine
the importance score of current features' subset.
The score is used in a multiple-way (backward or forward) \code{greedy}
algorithm as a stopping moment or as a selection criterion
in the \code{exhaustive} search that checks all possible
attributes' subset combinations (of sizes passed in \code{sizes}).
}
\note{
Note that score depends on the evaluator you provide in the \code{fun} parameter.
}
\examples{

# Enable parallelization in examples
\dontrun{
 library(doParallel)
 cl <- makeCluster(2)
 registerDoParallel(cl)
}
# Close at the end
# stopCluster(cl) #nolint
# registerDoSEQ() #nolint

if(require("rpart")) {
# 1) Evaluator from FSelector package.
evaluator <- function(subset, data, dependent = names(iris)[5]) {
  library(rpart)
  k <- 5
  splits <- runif(nrow(data))
  results <- sapply(1:k, function(i) {
    test.idx <- (splits >= (i - 1) / k) & (splits < i / k)
    train.idx <- !test.idx
    test <- data[test.idx, , drop = FALSE]
    train <- data[train.idx, , drop = FALSE]
    tree <- rpart(to_formula(subset, dependent), train)
    error.rate <- sum(test[[dependent]] != predict(tree, test, type = "c")) /
    nrow(test)
    return(1 - error.rate)
  })
  return(mean(results))
}

set.seed(123)
# Default greedy search.
system.time(
  feature_search(attributes = names(iris)[-5],
                 fun = evaluator,
                 data = iris)
)
system.time(
  feature_search(attributes = names(iris)[-5],
                 fun = evaluator,
                 data = iris,
                 parallel = FALSE)
)

# Optional exhaustive search.
system.time(
  feature_search(attributes = names(iris)[-5],
                 fun = evaluator,
                 data = iris,
                 mode = "exhaustive")
)
system.time(
  feature_search(attributes = names(iris)[-5],
                 fun = evaluator,
                 data = iris,
                 mode = "exhaustive",
                 parallel = FALSE)
)
}

# 2) Maximize R^2 statistics in the linear regression model/problem.

evaluator_R2_lm <- function(attributes, data, dependent = names(iris)[1]) {
  summary(
    lm(to_formula(attributes, dependent), data = data)
  )$r.squared
}

feature_search(attributes = names(iris)[-1],
               fun = evaluator_R2_lm, data = iris,
               mode = "exhaustive")

# 3) Optimize BIC crietion in generalized linear model.
# Aim of Bayesian approach it to identify the model with the highest
# probability of being the true model. - Kuha 2004
if(require("MASS")) {
utils::data(anorexia, package = "MASS")

evaluator_BIC_glm <- function(attributes, data, dependent = "Postwt") {
  extractAIC(
    fit = glm(to_formula(attributes, dependent), family = gaussian,
              data = data),
    k = log(nrow(data))
  )[2]
}

feature_search(attributes = c("Prewt", "Treat", "offset(Prewt)"),
               fun = evaluator_BIC_glm,
               data = anorexia,
               mode = "exhaustive")
}
# Close parallelization
\dontrun{
stopCluster(cl)
registerDoSEQ()
}
}
\author{
Zygmunt Zawadzki \email{zygmunt@zstat.pl}

Krzysztof Slomczynski \email{krzysztofslomczynski@gmail.com}
}
