% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/setupPerryPlot.R
\name{setupPerryPlot}
\alias{setupPerryPlot}
\alias{setupPerryPlot.perry}
\alias{setupPerryPlot.perrySelect}
\alias{setupPerryPlot.perryTuning}
\title{Set up a plot of resampling-based prediction error results}
\usage{
setupPerryPlot(object, ...)

\method{setupPerryPlot}{perry}(
  object,
  which = c("box", "density", "dot"),
  select = NULL,
  seFactor = NA,
  ...
)

\method{setupPerryPlot}{perrySelect}(
  object,
  which = c("box", "density", "dot", "line"),
  subset = NULL,
  select = NULL,
  seFactor = object$seFactor,
  ...
)

\method{setupPerryPlot}{perryTuning}(object, ...)
}
\arguments{
\item{object}{an object inheriting from class \code{"perry"} or
\code{"perrySelect"} that contains prediction error results.}

\item{\dots}{for the \code{"perryTuning"} method, additional arguments to
be passed down to the \code{"perrySelect"} method.  For the other methods,
additional arguments are currently ignored.}

\item{which}{a character string specifying the type of plot to
prepare.  Possible values are \code{"box"} to prepare a box plot,
\code{"density"} to prepare a smooth density plot, \code{"dot"} to prepare
a dot plot, or \code{"line"} to prepare a plot of the (average) results for
each model as a connected line (for objects inheriting from class
\code{"perrySelect"}).  Note that the first two plots are only meaningful
in case of repeated resampling.  The default is to use \code{"box"} in case
of repeated resampling and \code{"dot"} otherwise.}

\item{select}{a character, integer or logical vector indicating the columns
of prediction error results to be prepared for plotting.}

\item{seFactor}{a numeric value giving the multiplication factor of the
standard error for displaying error bars in dot plots or line plots.  Error
bars in those plots can be suppressed by setting this to \code{NA}.}

\item{subset}{a character, integer or logical vector indicating the subset
of models to be prepared for plotting.}
}
\value{
An object of class \code{"setupPerryPlot"} with the following
components:
\describe{
  \item{\code{data}}{a data frame containing the following columns:
  \describe{
    \item{\code{Fit}}{a vector or factor containing the identifiers of the
    models.}
    \item{\code{Name}}{a factor containing the names of the predictor error
    results (not returned in case of only one column of prediction error
    results with the default name).}
    \item{\code{PE}}{the estimated prediction errors.}
    \item{\code{Lower}}{the lower end points of the error bars (only
    returned if possible to compute).}
    \item{\code{Upper}}{the upper end points of the error bars (only
    returned if possible to compute).}
  }
  }
  \item{\code{which}}{a character string specifying the type of plot.}
  \item{\code{grouped}}{a logical indicating whether density plots should
  be grouped due to multiple model fits (only returned in case of density
  plots for the \code{"perrySelect"} and \code{"perryTuning"} methods).}
  \item{\code{includeSE}}{a logical indicating whether error bars based on
  standard errors are available (only returned in case of dot plots or line
  plots).}
  \item{\code{mapping}}{default aesthetic mapping for the plots.}
  \item{\code{facets}}{default faceting formula for the plots (not returned
  in case of only one column of prediction error results with the default
  name).}
  \item{\code{tuning}}{a data frame containing the grid of tuning parameter
  values for which the prediction error was estimated (only returned for the
  \code{"perryTuning"} method).}
}
}
\description{
Extract and prepare the relevant information for a plot of results of
resampling-based prediction error measures.
}
\note{
Duplicate indices in \code{subset} or \code{select} are removed such
that all models and prediction error results are unique.
}
\examples{
library("perryExamples")
data("coleman")
set.seed(1234)  # set seed for reproducibility

## set up folds for cross-validation
folds <- cvFolds(nrow(coleman), K = 5, R = 10)

## compare LS, MM and LTS regression

# perform cross-validation for an LS regression model
fitLm <- lm(Y ~ ., data = coleman)
cvLm <- perry(fitLm, splits = folds,
              cost = rtmspe, trim = 0.1)

# perform cross-validation for an MM regression model
fitLmrob <- lmrob(Y ~ ., data = coleman, maxit.scale = 500)
cvLmrob <- perry(fitLmrob, splits = folds,
                 cost = rtmspe, trim = 0.1)

# perform cross-validation for an LTS regression model
fitLts <- ltsReg(Y ~ ., data = coleman)
cvLts <- perry(fitLts, splits = folds,
               cost = rtmspe, trim = 0.1)

# combine results into one object
cv <- perrySelect(LS = cvLm, MM = cvLmrob, LTS = cvLts,
                  .selectBest = "min")
cv

## convert MM regression results to data frame for plotting
# all replications for box plot
cvLmrobBox <- setupPerryPlot(cvLmrob, which = "box")
perryPlot(cvLmrobBox)
# aggregated results for dot plot
cvLmrobDot <- setupPerryPlot(cvLmrob, which = "dot", seFactor = 2)
perryPlot(cvLmrobDot)

## convert combined results to data frame for plotting
# all replications for box plot
cvBox <- setupPerryPlot(cv, which = "box")
perryPlot(cvBox)
# aggregated results for dot plot
cvDot <- setupPerryPlot(cv, which = "dot", seFactor = 2)
perryPlot(cvDot)
}
\seealso{
\code{\link{perryPlot}},

\code{\link{perryFit}}, \code{\link{perrySelect}},
\code{\link{perryTuning}},

\code{\link[ggplot2]{ggplot}}, \code{\link[ggplot2]{autoplot}},
\code{\link[graphics]{plot}}
}
\author{
Andreas Alfons
}
\keyword{utilities}
