/*******************************************************************************
* E.S.O. - VLT project
*
* "@(#) $Id: FpilExtras.c 77237 2003-01-31 05:42:32Z flamemgr $"
*
* who       when      what
* --------  --------  ----------------------------------------------
* rsc/aka   28/01/03  PPRS 8650: FpilFibreCombos: mode names made unique.
*/

/*
 *                       F p i l  E x t r a s . c
 *
 *  This file is a temporary repository for a number of new Fpil routines
 *  that were found to be needed during the conversion of configure to
 *  handle OzPoz. In time they should all be implemented properly and
 *  invoked indirectly like other Fpil routines, and properly separated
 *  into the code specific to the different instruments. But for the moment
 *  it's easier to have them all in one place like this.
 */
 
#include "fpil.h"

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE
#define FALSE 0
#endif

#ifdef DEBUG
#undef DEBUG
#endif
#define DEBUG printf

/*  These type definitions are those used by the FLAMES verison of 
 *  FpilConfWhichSpec(). They have to match the FIBRE_TYPE_xxx
 *  #defines in fpcolFpil.c.
 */
 
#define FLAMES_FIBRE_TYPE_GUIDE   0
#define FLAMES_FIBRE_TYPE_IFU     1
#define FLAMES_FIBRE_TYPE_IFU_SKY 2
#define FLAMES_FIBRE_TYPE_MEDUSA  3
#define FLAMES_FIBRE_TYPE_UVES    4
#define FLAMES_FIBRE_TYPE_ARGUS   5
#define FLAMES_FIBRE_TYPE_TEST    6
#define FLAMES_FIBRE_TYPE_UVES_7  7
#define FLAMES_FIBRE_TYPE_UVES_8  8
#define FLAMES_FIBRE_TYPE_NONE    9
#define FLAMES_FIBRE_TYPE_UNKN   10

/*  The disabled target bit is chosen so that it doesn't get in the way of 
 *  normal alphabetic characters.
 */
 
#define DISABLED_TARGET_BIT 0x80

/*  The broken pivot bit is chosen so that it doesn't get in the way of 
 *  normal fibre type index numbers. (Strictly speaking, both this an
 *  DISABLED_TARGET_BIT are bit masks, not bits.)
 */
 
#define BROKEN_FIBRE_BIT 0x80

/*  The value of FLAMES_YAXIS_POS should be the same as that used for
 *  the first telescope parameter (ARGUS_POS) in the file fpcolFpil.c.
 *  Having to do this is a bit inelegant, but there isn't an easy way to 
 *  get this value otherwise. (Modified 23rd Oct 2002, KS - was 90 degrees,
 *  but fpcolFpil.c was changed since then..)
 */
 
#define FLAMES_YAXIS_POS 0.0

 
/*  ------------------------------------------------------------------------  */

/*
 *+			        F p i l  E n c o d e  T a r g e t  T y p e

 * Function name:
      FpilEncodeTargetType

 * Function:
       Checks a target type string for validity and encodes it.

 * Description:
       The fibre configuration routines allow a target object to be specified
       using a string that is supposed to describe its type in such a way that
       one can tell from this which pivots (ie which fibres) can be assigned 
       to it. For somewhat historical reasons, this is encoded into two
       separate single characters, one described as the 'type' and the other
       described as the 'spectrograph'. In the original 2dF code, the type was
       one of 'P' 'S' or 'F' for a program object, sky, and a fiducial
       respectively, and the spectrograph was 'A' or 'B'. Other instruments
       such as OzPoz allow a greater variety of target types, but it is still
       possible to represent them all using this two character scheme, even 
       if the two characters are no longer always quite so intuitive in their
       meanings. These are now used only by Fpil routines such as this and
       FpilTargetPivotCompatible(). Note that targets classed as 'guide' 
       targets are not allocated to spectrographs and the spect field should
       not be used for these. This implies that a target can be classed as
       a guide or not on the basis of it 'type' character, irrespective of the
       value of its 'spectrograph' character.
       
       This routine takes string describing a target and returns the two
       characters used to encode its type. It returns true if the string is
       valid and false (ie zero) if it was invalid.

       Invokes the instrument specific virtual encode target type routine
       using the specified arguments.

 * Language:
      C

 * Call:
      (int) = FpilEncodeTargetType(inst, targetString, targetType, 
                                                              targetSpect)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) targetString    (char *)    String describing the target type.
      (<) targetType      (char *)    Address of the single character set to
                                      the 'type' character describing the
                                      target.
      (<) targetSpect     (char *)    Address of the single character set to
                                      the 'spect' character describing the
                                      target.

 * Returned value:
      True (ie non-zero) if the type string was valid, false (ie zero) if
      it was invalid.

 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     12-Jul-2000 -  KS - Original version
     16-Jan-2001 -  KS - Added support for IFU Sky targets, as code 'J',
                         and ARGUS Sky targets, as code 'A'.
     30-Oct-2001 -  KS - 'M' is now allowed for Medusa targets, and is handled
                         as if 'P' had been specified. Now survives being
                         called with targetString null.
 */
 
extern int FpilEncodeTargetType (
    const FpilType instrument,
    char* targetString,
    char* targetType,
    char* targetSpect)
{
    int isFlames;
    char typeChar;
    char spectChar;
    int knownType;
    
    /*  Initial values  */
    
    knownType = FALSE;
    *targetSpect = '\0';
    *targetType = '\0';
    isFlames = !strcmp(FpilGetInstName(instrument),"FLAMES");
    
    /*  For the moment, this version has the 2dF and 6dF and OzPoz (FLAMES)
     *  cases hard coded. 2dF and 6dF both accept 'S', 'F' or 'P' for
     *  the first character of the type string, and 'A' or 'B' for the
     *  second. OzPoz accepts a single character, which can be one of
     *  S,F,P,I,J,T,U,G,A,M. Note that we don't allow a 'disabled target' type
     *  to be specified at this point.
     */
    
    typeChar = ' ';
    if (targetString) typeChar = targetString[0];
    if (islower((int)typeChar)) typeChar = toupper(typeChar);
    if (typeChar == 'F') {
        *targetType = 'F';
        knownType = TRUE;
    } else if (typeChar == 'S') {
        *targetType = 'S';
        knownType = TRUE;
    } else if (typeChar == 'P') {
        *targetType = 'P';
        knownType = TRUE;
    } else {
        if (isFlames) {
            if (typeChar == 'G') {
                *targetType = 'G';
                knownType = TRUE;
            } else if (typeChar == 'U') {
                *targetType = 'U';
                knownType = TRUE;
            } else if (typeChar == 'I') {
                *targetType = 'I';
                knownType = TRUE;
            } else if (typeChar == 'J') {
                *targetType = 'J';
                knownType = TRUE;
            } else if (typeChar == 'A') {
                *targetType = 'A';
                knownType = TRUE;
            } else if (typeChar == 'T') {
                *targetType = 'T';
                knownType = TRUE;
            } else if (typeChar == 'M') {
            
               /*  The 'M' type is the preferred way of specifying a Medusa
                *  target, but the original version of the code used 'P'. We
                *  handle 'M' by treating it as 'P' - it would be nicer to
                *  handle 'P' by treating it as 'M' and changing all the 
                *  other routines in this file to use 'M' rather than 'P',
                */

                *targetType = 'P';
                knownType = TRUE;
            }
        }
    }
    
    if (knownType) {
        spectChar = targetString[1]; 
        if (islower((int)spectChar)) spectChar = toupper(spectChar);
       
        if (isFlames) {
            if (spectChar != '\0') knownType = FALSE;
        } else {
            if ((spectChar == 'A') || (spectChar == 'B') || 
                                             (spectChar == '\0')) {
                *targetSpect = spectChar;
            } else {
                knownType = FALSE;
            }
        }
    }
    
    return (knownType);
}

/*  ------------------------------------------------------------------------  */

/*
 *+			      F p i l  D e c o d e  T a r g e t  T y p e

 * Function name:
      FpilDecodeTargetType

 * Function:
       Decodes the encoded chracters describing a target back to one string.

 * Description:
       The fibre configuration routines allow a target object to be specified
       using a string that is supposed to describe its type in such a way that
       one can tell from this which pivots (ie which fibres) can be assigned 
       to it. For somewhat historical reasons, this is encoded into two
       separate single characters, one described as the 'type' and the other
       described as the 'spectrograph'. See the comments to the routine
       FpilEncodeTargetType() for more details. This routine reverses the
       process performed by FpilEncodeTargetType(), taking the two
       characters used to encode a target's type and returning a single
       string that will be equivalent to that originally passed to
       FpilEncodeTargetType(). It returns true if the characters were
       valid and false (ie zero) if they were invalid. It also returns a
       string that provides a less cryptic description of the target that
       can be used for informative dialogues and the like. In some cases,
       the configure code only has the target type to work with (usually
       because the information has been obtained from the "objects" structure
       which does not include the 'spect' information), and this routine can
       be used in this case, a call with the 'spectKnown' parameter set false
       causing the spect code to be ignored.

       Invokes the instrument specific virtual decode target type routine
       using the specified arguments.

 * Language:
      C

 * Call:
      (int) = FpilDecodeTargetType(inst, targetType, targetSpect, spectKnown,
                                                  targetString, targetText)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) targetType      (char)      The single character set to the 'type' 
                                      character describing the target.
      (>) targetSpect     (char)      The single character set to the 'spect' 
                                      character describing the target.
      (>) spectKnown      (int)       True if the 'spect' character should be
                                      used. If false, targetSpect is ignored.
      (<) targetString    (char *)    String describing the target type. There
                                      should be at least 32 characters available
                                      at this address. The string is relatively
                                      cryptic, and attempts to reproduce that
                                      originally supplied in a target list.
      (<) targetText      (char *)    String describing the target type. There
                                      should be at least 64 characters available
                                      at this address. The string is intended
                                      to be a more readable description than 
                                      that returned in targetString.

 * Returned value:
      True (ie non-zero) if the type string was valid, false (ie zero) if
      it was invalid.

 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     12-Jul-2000 -  KS - Original version
     16-Jan-2001 -  KS - Added support for IFU Sky targets, as code 'J',
                         and ARGUS Sky targets, as code 'A'.
     30-Oct-2001 -  KS - Use 'M' rather than 'P' for Medusa targets.
 */
 
extern int FpilDecodeTargetType (
    const FpilType instrument,
    char  targetType,
    char  targetSpect,
    int   spectKnown,
    char* targetString,
    char* targetText)
{
    int disabled;
    int isFlames;
    int knownType;
    
    /*  Initial values  */
    
    knownType = FALSE;
    isFlames = !strcmp(FpilGetInstName(instrument),"FLAMES");
    
    /*  In most cases, the string we return is a single character followed
     *  by a terminating nul, so we set that up here first.  We also set up
     *  a null description.
     */
     
    targetString[0] = ' ';
    targetString[1] = '\0';
    targetText[0] = '\0';
    
    /*  For the moment, version has the 2dF and 6dF and OzPoz (FLAMES)
     *  cases hard coded. 2dF and 6dF both accept 'S', 'F' or 'P' for
     *  the first character of the type string, and 'A' or 'B' for the
     *  second. OzPoz accepts a single character, which can be one of
     *  S,F,P,I,J,T,U,G,A.
     */
     
    if (isFlames) {
           
        /*  For OzPoz the string can be the same as the type character. We
         *  have to allow for 'disabled target' types, so we mask off the 
         *  disabled target flag before making the comparisons. We remember
         *  the disabled flag for the full description, however. If a target
         *  is disabled, we assume this is because it is obscured by the Guide
         *  probe - at present this is the only reason a target is disabled.
         *  OzPoz doesn't use the 'spect' code at all. We prefer 'M' for
         *  Medusa targets, although the internal code uses 'P'.
         */
        
        disabled = (targetType & DISABLED_TARGET_BIT);
        targetType &= ~DISABLED_TARGET_BIT;
        
        knownType = TRUE;
        switch (targetType) {
            case 'S': {
                strcpy(targetText,"Sky");
                break;
            }
            case 'F': {
                strcpy(targetText,"Guide star");
                break;
            }
            case 'M':
            case 'P': {
                strcpy(targetText,"Medusa target");
                break;
            }
            case 'I': {
                strcpy(targetText,"IFU target");
                break;
            }
            case 'J': {
                strcpy(targetText,"IFU sky target");
                break;
            }
            case 'T': {
                strcpy(targetText,"Test target");
                break;
            }
            case 'U': {
                strcpy(targetText,"UVES target");
                break;
            }
            case 'G': {
                strcpy(targetText,"VLT guide star");
                break;
            }
            case 'A': {
                strcpy(targetText,"ARGUS sky target");
                break;
            }
            case 'X': {
                strcpy(targetText,"Unallocated");
                break;
            }
            default : {
                knownType = FALSE;
            }
        }
        if (knownType) {
            targetString[0] = targetType;
            if (targetType == 'P') targetString[0] = 'M';
            if (disabled) strcat (targetText," (obscured) ");
        }
                              
    } else {
    
        /*  For 2dF and 6dF, the first character of the string is the same
         *  as the type character, with an 'A' or 'B' appended if the
         *  spectrograph character is one of these. Otherwise a spect
         *  character of zero is allowed, but anything else is an error.
         */

        knownType = TRUE;
        switch (targetType) {
            case 'S' : {
                strcpy (targetText,"Sky");
                break;
            }         
            case 'F' : {
                strcpy (targetText,"Guide star");
                break;
            }         
            case 'P' : {
                strcpy (targetText,"Program object");
                break;
            }
            default : {
                knownType = FALSE;
            }
        }
        if (knownType) {       
            targetString[0] = targetType;
            if (spectKnown) {
                if ((targetSpect == 'A') || (targetSpect == 'B')) {
                    targetString[1] = targetSpect;
                    targetString[2] = '\0';
                    if (targetSpect == 'A') {
                        strcat (targetText," (A)");
                    } else {
                        strcat (targetText," (B)");
                    }
                } else if (targetSpect != 0) {
                    knownType = FALSE;
                }
            }
        }
        
    }
    
    return (knownType);
}

/*  ------------------------------------------------------------------------  */

/*
 *+			        F p i l  I s  T a r g e t  G u i d e

 * Function name:
      FpilIsTargetGuide

 * Function:
       Sees if the target type character refers to a guide object.

 * Description:
       The fibre configuration routines allow a target object to be specified
       using a string that is supposed to describe its type in such a way that
       one can tell from this which pivots (ie which fibres) can be assigned 
       to it. For somewhat historical reasons, this is encoded into two
       separate single characters, one described as the 'type' and the other
       described as the 'spectrograph'. In some cases the code specifically
       needs to know if a target (identified by these two characters) refers
       corresponds to a guide object or not. This classification into guide
       and non-guide targets can always be done on the basis of the 'type'
       character alone.
       
       This routine takes the character describing a target type and returns
       true if it refers to a guide object. It returns false if the character
       is invalid or does not refer to a guide target. The passed character
       should have been returned by FpilEncodeTargetType().

       Invokes the instrument specific virtual guide target type routine
       using the specified arguments.

 * Language:
      C

 * Call:
      (int) = FpilIsTargetGuide(inst, targetType)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) targetType      (char)      The single character describing the
                                      target type as set by the routine
                                      FpilEncodeTargetType().
 
 * Returned value:
      True (ie non-zero) if the type and spect characters were valid and 
      described a guide target, false otherwise.

 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     12-Jul-2000 -  KS - Original version
 */
 
extern int FpilIsTargetGuide (
    const FpilType instrument,
    char targetType)
{
    /*  All supported instruments use 'F' for a guide target, so for the 
     *  moment we can cheat and use that.  FLAMES also allows 'G' for
     *  a VLT guide star, best treated as a guide for this purpose, and
     *  no other instrument has type 'G' objects. We have to allow for 
     *  'disabled target' types, so we mask off the disabled target flag 
     *  before making the comparisons.
     */
         
   targetType &= ~DISABLED_TARGET_BIT;
     
   return ((targetType == 'F') || (targetType == 'G'));
}

/*  ------------------------------------------------------------------------  */

/*
 *+			        F p i l  I s  T a r g e t  T e l  G u i d e

 * Function name:
      FpilIsTargetTelGuide

 * Function:
       Sees if the target type character refers to a telescope guider object.

 * Description:
       The fibre configuration routines allow a target object to be specified
       using a string that is supposed to describe its type in such a way that
       one can tell from this which pivots (ie which fibres) can be assigned 
       to it. For somewhat historical reasons, this is encoded into two
       separate single characters, one described as the 'type' and the other
       described as the 'spectrograph'.  Some instruments have the concept of
       a 'telescope guide object', an object that is not to be allocated a
       fibre, but is to be used by a guide probe that is part of the telescope
       system. (This is the case for the VLT.)  Sometimes the code specifically
       needs to know if a target (identified by these two characters) refers
       corresponds to a guide object or not. This classification can always be
       done on the basis of the 'type' character alone.
       
       This routine takes the character describing a target type and returns
       true if it refers to a guide object. It returns false if the character
       is invalid or does not refer to a guide target. The passed character
       should have been returned by FpilEncodeTargetType().

       Invokes the instrument specific virtual guide target type routine
       using the specified arguments.

 * Language:
      C

 * Call:
      (int) = FpilIsTargetTelGuide(inst, targetType)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) targetType      (char)      The single character describing the
                                      target type as set by the routine
                                      FpilEncodeTargetType().
 
 * Returned value:
      True (ie non-zero) if the type and spect characters were valid and 
      described a telescope guider target, false otherwise.

 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     16-Aug-2000 -  KS - Original version
 */
 
extern int FpilIsTargetTelGuide (
    const FpilType instrument,
    char targetType)
{
    /*  FLAMES is the only instrument that supports such a concept, and it
     *  uses 'G' for such targets. Since no other instrument uses 'G' we can
     *  base our test purely on that.
     */
         
   targetType &= ~DISABLED_TARGET_BIT;
     
   return (targetType == 'G');
}

/*  ------------------------------------------------------------------------  */

/*
 *+			        F p i l  I s  T a r g e t  S k y

 * Function name:
      FpilIsTargetSky

 * Function:
       Sees if the target type character refers to a sky object.

 * Description:
       The fibre configuration routines allow a target object to be specified
       using a string that is supposed to describe its type in such a way that
       one can tell from this which pivots (ie which fibres) can be assigned 
       to it. For somewhat historical reasons, this is encoded into two
       separate single characters, one described as the 'type' and the other
       described as the 'spectrograph'. In some cases the code specifically
       needs to know if a target (identified by these two characters) refers
       corresponds to a sky object or not. This classification into sky
       and non-sky targets can always be done on the basis of the 'type'
       chracter alone.
       
       This routine takes the character describing a target type and returns
       true if it refers to a sky object. It returns false if the character
       is invalid or does not refer to a guide target. The passed character
       should have been returned by FpilEncodeTargetType().

       Invokes the instrument specific virtual sky target type routine
       using the specified arguments.

 * Language:
      C

 * Call:
      (int) = FpilIsTargetSky(inst, targetType)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) targetType      (char)      The single character describing the
                                      target type as set by the routine
                                      FpilEncodeTargetType().
 
 * Returned value:
      True (ie non-zero) if the type and spect characters were valid and 
      described a sky target, false otherwise.

 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     12-Jul-2000 -  KS - Original version
     16-Jan-2001 -  KS - Added support for IFU Sky targets, as code 'J',
                         and ARGUS Sky targets, as code 'A'.
 */
 
extern int FpilIsTargetSky (
    const FpilType instrument,
    char targetType)
{
    int returnValue = 0;
    
    /*  All supported instruments use 'S' for a sky target, and
     *  FLAMES also uses 'J' and 'A' for sky targets.  We have to allow for 
     *  'disabled target' types, so we mask off the disabled target flag 
     *  before making the comparison.
     */
         
   targetType &= ~DISABLED_TARGET_BIT;
   if (targetType == 'S') {
       returnValue = 1;
   } else {
       int isFlames;
       isFlames = !strcmp(FpilGetInstName(instrument),"FLAMES");
       if (isFlames) {
          if ((targetType == 'A') || (targetType == 'J')) returnValue = 1;
       }
   }
   return (returnValue);
}

/*  ------------------------------------------------------------------------  */

/*
 *+			        F p i l  T a r g e t  S k y  T y p e

 * Function name:
      FpilTargetSkyType

 * Function:
       Returns the characters used to indicate a target sky object.

 * Description:
       The fibre configuration routines allow a target object to be specified
       using a string that is supposed to describe its type in such a way that
       one can tell from this which pivots (ie which fibres) can be assigned 
       to it. For somewhat historical reasons, this is encoded into two
       separate single characters, one described as the 'type' and the other
       described as the 'spectrograph'. In some cases the code needs to 
       create a new sky object automatically, and so needs to be able to set
       the type and spectrograph characters appropriately. This routine
       returns the values to be used in this case.
       
       Invokes the instrument specific virtual sky target type routine
       using the specified arguments.

 * Language:
      C

 * Call:
      (void) = FpilTargetSkyType(inst, targetType, spectTy)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (<) targetType      (char *)    Address of the single character set to
                                      the 'type' character describing the
                                      target.
      (<) targetSpect     (char *)    Address of the single character set to
                                      the 'spect' character describing the
                                      target.
 
 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     12-Jul-2000 -  KS - Original version
 */
 
extern void FpilTargetSkyType (
    const FpilType instrument,
    char* targetType,
    char* spectType)
{
    /*  All supported instruments use 'S' for a sky target, so for the 
     *  moment we can cheat and use that.  We set it as an enabled target (ie
     *  we don't set the 'target disabled' flag) and set the spectrograph 
     *  character to nul.
     */
    
    *targetType = 'S';
    *spectType = 0;     
     
}

/*  ------------------------------------------------------------------------  */

/*
 *+			        F p i l  G e t  P i v o t  T e x t

 * Function name:
      FpilGetPivotText

 * Function:
       Returns a text string describing a given pivot type.

 * Description:
       A pivot can be described using a single integer that is supposed to 
       define its fibre type. See FpilConfWhichSpec() for more details. This
       routine is passed such an integer fibre type code and returns the 
       address of a text string that can be used to describe a pivot type 
       specified in this way.
        
       It is expected that this will mainly be used to generate messages 
       about the type.
       
       Invokes the instrument specific virtual pivot text routine
       using the specified arguments.

 * Language:
      C

 * Call:
      (char*) = FpilGetPivotText(inst, pivotType)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst         (FpilType)     The instrument as returned 
                                      by FpilInit(3).
      (>) pivotType    (int)          The integer code describing the
                                      pivot fibre type as set by the routine
                                      FpilConfWhichSpec().
 * Returns:
      A pointer to a static text string describing the pivot type. This
      is not invalidated by a later call to this routine.
 
 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     16-Nov-2001 -  KS - Original version
 */
 
extern char* FpilGetPivotText (
    const FpilType instrument,
    int pivotType)
{
    /*  Local variables */
    
    int isFlames;                        /* True if instrument is FLAMES */
    char* Text;                          /* Returned text address */
    
    /*  We clear out the broken fibre bit - we ignore it */
    
    Text = "Unknown";
    pivotType &= ~BROKEN_FIBRE_BIT;
    isFlames = !strcmp(FpilGetInstName(instrument),"FLAMES");
    if (isFlames) {
    
        /*  For FLAMES, we expect the pivot type to be one of the 
         *  FLAMES_FIBRE_TYPE_Xxxx codes.
         */
         
        switch (pivotType) {
            case FLAMES_FIBRE_TYPE_GUIDE: {
                Text = "Guide";
                break;
            }
            case FLAMES_FIBRE_TYPE_IFU: {
                Text = "IFU";
                break;
            }
            case FLAMES_FIBRE_TYPE_IFU_SKY: {
                Text = "IFU sky";
                break;
            }
            case FLAMES_FIBRE_TYPE_MEDUSA: {
                Text = "Medusa";
                break;
            }
            case FLAMES_FIBRE_TYPE_UVES_7:
            case FLAMES_FIBRE_TYPE_UVES_8:
            case FLAMES_FIBRE_TYPE_UVES: {
                Text = "UVES";
                break;
            }
            case FLAMES_FIBRE_TYPE_ARGUS: {
                Text = "Argus sky";
                break;
            }
            case FLAMES_FIBRE_TYPE_TEST: {
                Text = "Test";
                break;
            }
            case FLAMES_FIBRE_TYPE_NONE: {
                Text = "None-existent";
                break;
            }
            case FLAMES_FIBRE_TYPE_UNKN: {
                Text = "Unknown";
                break;
            }
        }
        
     } else {
     
         /*  For the moment, for 2dF and 6dF we do what the old code does,
          *  which is distinguish only between guide and non-guide cases.
          *  We assume that guide pivots are all of type code 0. When this
          *  is incorporated properly into the 2dF and 6dF specific routines
          *  this should be done more correctly.
          */
         
         if (pivotType == 0) {
            Text = "Guide";
         } else {
            Text = "Spectrograph";
         }
    }
    
    return Text;
}

/*  ------------------------------------------------------------------------  */

/*
 *+			        F p i l  P i v o t  T y p e  M a t c h

 * Function name:
      FpilPivotTypeMatch

 * Function:
       Determines if two pivot types are compatible for allocation purposes.

 * Description:
       A pivot can be described using a single integer that is supposed to 
       define its fibre type. See FpilConfWhichSpec() for more details. 
       FLAMES has the peculiarity that the UVES fibres are treated as being
       of more than one type, which is needed in order to eliminate some of
       them from certain configurations, but in most cases can be ignored. 
       This routine determines whether two fibre types can be regarded as 
       equivalent for configuration purposes. (It really exists simply so that
       the three FLMAES UVES types can be treated as one, but there may
       be other instruments that might make use of this later.)
       
       Invokes the instrument specific virtual pivot match routine
       using the specified arguments.

 * Language:
      C

 * Call:
      (int) = FpilPivotTypeMatch(inst, pivotType, secondType)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst         (FpilType)     The instrument as returned 
                                      by FpilInit(3).
      (>) pivotType    (int)          The integer code describing the first
                                      pivot fibre type as set by the routine
                                      FpilConfWhichSpec().
      (>) secondType   (int)          The integer code describing the second
                                      pivot fibre type as set by the routine
                                      FpilConfWhichSpec().
 
 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     16-Nov-2001 -  KS - Original version
 */
 
extern int FpilPivotTypeMatch (
    const FpilType instrument,
    int pivotType,
    int secondType)
{
    /*  Local variables */
    
    int isFlames;                        /* True if instrument is FLAMES */
    int match;                           /* Returned match flag */
    
    /*  We clear out the broken fibre bits - we ignore them */
    
    pivotType &= ~BROKEN_FIBRE_BIT;
    secondType &= ~BROKEN_FIBRE_BIT;
    
    /*  Obviously, if the two are the same, they match. If they don't we have
     *  to look a little deeper.
     */

    match = (pivotType == secondType);
    
    if (!match) {
        isFlames = !strcmp(FpilGetInstName(instrument),"FLAMES");
        if (isFlames) {

            /*  For FLAMES, we expect the pivot types to be one of the 
             *  FLAMES_FIBRE_TYPE_Xxxx codes. The only case of interest are
             *  the UVES codes.
             */

             if ((pivotType == FLAMES_FIBRE_TYPE_UVES_7) || 
                     (pivotType == FLAMES_FIBRE_TYPE_UVES_8) ||
                          (pivotType == FLAMES_FIBRE_TYPE_UVES)) {
                 if ((secondType == FLAMES_FIBRE_TYPE_UVES_7) || 
                         (secondType == FLAMES_FIBRE_TYPE_UVES_8) ||
                               (secondType == FLAMES_FIBRE_TYPE_UVES)) {
                     match = TRUE;
                 }
             }
        }
    }
    return match;
}

/*  ------------------------------------------------------------------------  */

/*
 *+			       F p i l  P i v o t  S h a r e s  S k y

 * Function name:
      FpilPivotSharesSky

 * Function:
       Determines whether a pivot type is suitable for both sky and objects.

 * Description:
       A pivot can be described using a single integer that is supposed to 
       define its fibre type. See  FpilConfWhichSpec() for more details.  
       Some pivot types are exclusively for sky targets, some are exclusively
       for non-sky targets. Other types are shared between sky and non-sky 
       targets, and so need special consideration when sky targets are 
       assigned. This routine is used to determine whether or not a given
       pivot type is used for both sky and non-sky targets.
       
       Invokes the instrument specific virtual pivot text routine
       using the specified arguments.

 * Language:
      C

 * Call:
      (int) = FpilPivotSharesSky(inst, pivotType)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst         (FpilType)     The instrument as returned 
                                      by FpilInit(3).
      (>) pivotType    (int)          The integer code describing the
                                      pivot fibre type as set by the routine
                                      FpilConfWhichSpec().
 * Returns:
      True if this pivot type shares sky and non-sky targets.
 
 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     16-Nov-2001 -  KS - Original version
 */
 
extern int FpilPivotSharesSky (
    const FpilType instrument,
    int pivotType)
{
    /*  Local variables */
    
    int isFlames;                        /* True if instrument is FLAMES */
    int Shares;                          /* Returned value */
    
    /*  We clear out the broken fibre bit - we ignore it */
    
    Shares = TRUE;
    pivotType &= ~BROKEN_FIBRE_BIT;
    isFlames = !strcmp(FpilGetInstName(instrument),"FLAMES");
    if (isFlames) {
    
        /*  For FLAMES, we expect the pivot type to be one of the 
         *  FLAMES_FIBRE_TYPE_Xxxx codes. The guide fibres, IFU fibres,
         *  and ARGUS sky fibre types are not shared, all the rest are.
         */
        
        if ((pivotType == FLAMES_FIBRE_TYPE_IFU) ||
            (pivotType == FLAMES_FIBRE_TYPE_IFU_SKY) ||
            (pivotType == FLAMES_FIBRE_TYPE_GUIDE) ||
            (pivotType == FLAMES_FIBRE_TYPE_ARGUS)) {
            Shares = FALSE;
        }
    } else {
            
         /*  For the moment, for 2dF and 6dF we do what the old code does,
          *  which is distinguish only between guide and non-guide cases.
          *  We assume that guide pivots are all of type code 0. 
          */
         
         if (pivotType == 0) Shares = FALSE;
    }
    
    return Shares;
}

/*  ------------------------------------------------------------------------  */

/*
 *+			 T a r g e t  /  P i v o t  C o m p a t a b i l i t y
 *
 *  These arrays are used to provide a compatability matrix for FLAMES
 *  targets and pivots. They are used by FpilCompatiblePivotTypes()
 *  and FpilTargetPivotCompatible().
 */
 

   static int FlamesPivotTypes[] = {
        FLAMES_FIBRE_TYPE_GUIDE,FLAMES_FIBRE_TYPE_IFU,
        FLAMES_FIBRE_TYPE_IFU_SKY,
        FLAMES_FIBRE_TYPE_MEDUSA, FLAMES_FIBRE_TYPE_UVES,
        FLAMES_FIBRE_TYPE_ARGUS, FLAMES_FIBRE_TYPE_TEST };
   static char FlamesTargetTypes[] = 
          {'F','S','P','G','U','I','T','A','J'};
   static int FlamesCompatibility[] = {
       /*  'F','S','P','G','U','I','T','A','J'    <- Target types */
            1,  0,  0,  0,  0,  0,  0,  0,  0,    /* GUIDE pivots */
            0,  0,  0,  0,  0,  1,  1,  0,  0,    /* IFU pivots */
            0,  1,  0,  0,  0,  0,  1,  0,  1,    /* IFU sky pivots */
            0,  1,  1,  0,  0,  0,  1,  0,  0,    /* MEDUSA pivots */
            0,  1,  0,  0,  1,  0,  1,  0,  0,    /* UVES pivots */
            0,  1,  0,  0,  0,  0,  1,  1,  0,    /* ARGUS pivots */
            0,  1,  1,  0,  1,  1,  1,  1,  1     /* Test pivots */
       };

/*  ------------------------------------------------------------------------  */

/*
 *+			F p i l  T a r g e t  P i v o t  C o m p a t i b l e

 * Function name:
      FpilTargetPivotCompatible

 * Function:
       Sees if a pivot of given type can be assigned to a target of given type.

 * Description:
       The fibre configuration routines allow a target object to be specified
       using a string that is supposed to describe its type in such a way that
       one can tell from this which pivots (ie which fibres) can be assigned 
       to it. Similarly a pivot can be described using a single integer that
       is supposed to define its fibre type. See FpilConfWhichSpec() for more
       details.
             
       This routine takes the two characters describing a target and the single
       integer describing a fibre type and returns true if that pivot can be 
       assigned to that target. The passed characters should have been returned
       by FpilEncodeTargetType() and FpilConfWhichSpec().

       Invokes the instrument specific virtual compatibility type routine
       using the specified arguments.

 * Language:
      C

 * Call:
      (int) = FpilTargetPivotCompatible(inst, targetType, targetSpect,
                                                             pivotType)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) targetType      (char)      The single character describing the
                                      target type as set by the routine
                                      FpilEncodeTargetType().
      (>) targetSpect     (char)      The single character providing 
                                      supplementary information about the target
                                      type as set by FpilEncodeTargetType().
      (>) pivotType       (int)       The integer code describing the
                                      pivot fibre type as set by the routine
                                      FpilConfWhichSpec().
 
 * Returned value:
      True (ie non-zero) if the type and spect codes were valid and 
      describe a compatible target and pivot, false otherwise.

 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     12-Jul-2000 -  KS - Original version
     16-Jan-2001 -  KS - Added support for IFU Sky targets, as code 'J',
                         and ARGUS Sky targets, as code 'A'.
     14-Aug-2001 -  KS - Made test pivots compatible with all non-guide
                         target types.
     18-Aug-2001 -  KS - gcc -Wall erroneously complained about TargetIndex
                         and PivotIndex being used uninitialised. Fixed.
     28-Aug-2001 -  KS - Added support for UVES_7 and UVES_8 types.
                         
 * Note:
      The reason test pivots are not made compatible with guide targets is
      that the allocation code in places has to decide whether to look up
      a target in the unallocated objects structure or the unallocated
      guide targets structure. It makes the choice on the basis of the
      type of fibre, assuming that a guide target can only be allocated to
      a guide fibre. This all goes wrong if a non-guide fibre gets allocated
      to a guide target. It's not a good reason, but there it is. (KS).
     
 */
  
extern int FpilTargetPivotCompatible (
    const FpilType instrument,
    char targetType,
    char targetSpect,
    int  pivotType)
{
    int isFlames;
    int Compatible = FALSE;
    
    isFlames = !strcmp(FpilGetInstName(instrument),"FLAMES");

    if (isFlames) {
    
        /*  Flames targets can be disabled. If the disabled bit is set,
         *  then the target is not compatible. Otherwise, we use a
         *  compatibility matrix to handle the various possible combinations
         *  of target and pivot types.
         */
    
        if (((targetType & DISABLED_TARGET_BIT) == 0) &&
                    ((pivotType & BROKEN_FIBRE_BIT) == 0)) {
        
             int NPivotTypes = sizeof(FlamesPivotTypes)/sizeof(int);
             int NTargetTypes = sizeof(FlamesTargetTypes)/sizeof(char);
         
             int PivotOK;
             int TargetOK;
         
             int TargetIndex;
             int PivotIndex;
             int I;
         
             /*  Actually, the pivot types returned by FpilConfWhichSpec()
              *  are integers and we could just use them as the PivotIndex,
              *  but this acts as an additional check and makes the code a
              *  little more obvious - it also makes it easier to change the
              *  allocation of indices to fibre tyes. Note that the list of
              *  types deliberately doesn't include 'none' and 'unknown' -
              *  if we get one of those, it's just regarded as incompatible
              *  with anything. We explicitly treat all three UVES types 
              *  alike - UVES_7 and UVES_8 have the same compatibility
              *  rules as the ordinary UVES fibres.
              */
             
             if ((pivotType == FLAMES_FIBRE_TYPE_UVES_7) ||
                    (pivotType == FLAMES_FIBRE_TYPE_UVES_8)) {
                 pivotType = FLAMES_FIBRE_TYPE_UVES;
             }
             PivotOK = FALSE;
             PivotIndex = -1;
             for (I = 0; I < NPivotTypes; I++) {
                 if (FlamesPivotTypes[I] == pivotType) {
                     PivotIndex = I;
                     PivotOK = TRUE;
                     break;
                 }
             }
             TargetOK = FALSE;
             TargetIndex = -1;
             for (I = 0; I < NTargetTypes; I++) {
                 if (FlamesTargetTypes[I] == targetType) {
                     TargetIndex = I;
                     TargetOK = TRUE;
                     break;
                 }
             }
             if (PivotOK && TargetOK) {
                 Compatible = FlamesCompatibility[(PivotIndex * NTargetTypes)
                                                                + TargetIndex];
             }
         }
         
     } else {
     
         /*  For the moment, for 2dF and 6dF we do what the old code does,
          *  which is distinguish only between guide and non-guide cases.
          *  We assume that guide pivots are all of type code 0. When this
          *  is incorporated properly into the 2dF and 6dF specific routines
          *  this should be done more correctly.
          */
          
         Compatible = FALSE;
         if ((targetType == 'F') && (pivotType == 0)) Compatible = TRUE;
         if ((targetType != 'F') && (pivotType != 0)) Compatible = TRUE;
    }
    
    return Compatible;
}

/*  ------------------------------------------------------------------------  */

/*
 *+			F p i l  C o m p a t i b l e  P i v o t  T y p e s

 * Function name:
      FpilCompatiblePivotTypes

 * Function:
       Returns list of pivot types compatible with a given target type.

 * Description:
       The fibre configuration routines allow a target object to be specified
       using a string that is supposed to describe its type in such a way that
       one can tell from this which pivots (ie which fibres) can be assigned 
       to it. Similarly a pivot can be described using a single integer that
       is supposed to define its fibre type. See FpilConfWhichSpec() for 
       more details.
             
       This routine takes the two characters describing a target and returns
       an array of the integers describing a pivot type giving the set of
       pivot types compatible with that target type. The passed target type
       characters should have been returned by FpilEncodeTargetType() and 
       the returned pivot type integers match those returned by 
       FpilConfWhichSpec().

       Invokes the instrument specific virtual compatibility type routine
       using the specified arguments.

 * Language:
      C

 * Call:
      FpilCompatiblePivotTypes (inst, maxPivTypes, targetType, targetSpect,
                                                    numPivTypes, pivTypes )

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) maxPivTypes     (int)       The maximum number of pivot types
                                      that can be returned - ie the size of
                                      the array passed as pivTypes.
      (>) targetType      (char)      The single character describing the
                                      target type as set by the routine
                                      FpilEncodeTargetType().
      (>) targetSpect     (char)      The single character providing 
                                      supplementary information about the target
                                      type as set by FpilEncodeTargetType().
      (<) numPivTypes     (int*)      Set to the number of compatible
                                      pivot types.
      (<) pivTypes        (int[])     The integer codes describing the
                                      compatible pivot fibre types as set by
                                      the routine FpilConfWhichSpec().
 
 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     03-Aug-2002 -  KS - Original version
     01-Oct-2002 -  KS - Corrected number of fibre types for 6dF.
                         
 * Note:
      The same argument about compatability of test pivots applies as for
      FpilTargetPivotCompatible(). See comments for that routine.
     
 */
  
extern void FpilCompatiblePivotTypes (
    const FpilType instrument,
    int maxPivTypes,
    char targetType,
    char targetSpect,
    int *numPivTypes,
    int *pivTypes)
{
    int isFlames;
    int is2dF;
    
    *numPivTypes = 0;
    
    isFlames = !strcmp(FpilGetInstName(instrument),"FLAMES");

    if (isFlames) {
    
        /*  Flames targets can be disabled. If the disabled bit is set,
         *  then the target is not compatible with any pivot types. Otherwise,
         *  we use the compatibility matrix.
         */
    
        if ((targetType & DISABLED_TARGET_BIT) == 0) {
        
            int NPivotTypes = sizeof(FlamesPivotTypes)/sizeof(int);
            int NTargetTypes = sizeof(FlamesTargetTypes)/sizeof(char);
            int NTypes = 0;
         
            int TargetOK;
         
            int TargetIndex;
            int PivotIndex;
            int I;
         
            /*  Work out the target index used in the compatability matrix
             *  given the target type code.
             */
             
            TargetOK = FALSE;
            TargetIndex = -1;
            for (I = 0; I < NTargetTypes; I++) {
                if (FlamesTargetTypes[I] == targetType) {
                    TargetIndex = I;
                    TargetOK = TRUE;
                    break;
                }
            }
            
            /*  Then load up the list of pivot types from those in the
             *  matrix that are compatible with this target type. Allow
             *  for the different UVES types, which are not explicitly
             *  included in the matrix.
             */
             
            if (TargetOK) {
                for (PivotIndex = 0; PivotIndex < NPivotTypes; PivotIndex++) {
                    if (FlamesCompatibility[(PivotIndex * NTargetTypes)
                                                          + TargetIndex]) {
                        if (NTypes < maxPivTypes) {
                            int PivType = FlamesPivotTypes[PivotIndex];
                            pivTypes[NTypes++] = PivType;
                            if ((PivType == FLAMES_FIBRE_TYPE_UVES) && 
                                          ((NTypes + 2) < maxPivTypes)) {
                                pivTypes[NTypes++] = FLAMES_FIBRE_TYPE_UVES_7;
                                pivTypes[NTypes++] = FLAMES_FIBRE_TYPE_UVES_8;
                            }
                        }
                    }
                }
                *numPivTypes = NTypes;
            }
        }
         
    } else {
     
        /*  For the moment, for 2dF and 6dF we do what the old code does,
         *  which is distinguish only between guide and non-guide cases.
         *  We assume that guide pivots are all of type code 0 and the
         *  spectrograph pivots are of types 1 and 2. (Or, for 6dF, just
         *  type 1.)
         */
         
         is2dF = !strcmp(FpilGetInstName(instrument),"2dF");
          
        if (targetType == 'F') {
            if (maxPivTypes > 0) {
                *numPivTypes = 1;
                pivTypes[0] = 0;
            }
        } else {
            if (!is2dF) {
                if (maxPivTypes > 0) {
                    *numPivTypes = 1;
                    pivTypes[0] = 1;
                }
            } else {
                if (maxPivTypes > 1) {
                    *numPivTypes = 2;
                    pivTypes[0] = 1;
                    pivTypes[1] = 2;
                }
            }
        }
    }
    
}



/*  ------------------------------------------------------------------------  */

/*
 *+			            F p i l  D r a w  T a r g e t 

 * Function name:
      FpilTDrawTarget

 * Function:
       Returns a Tcl/Tk canvas option string to draw a particular target.

 * Description:
       The fibre configuration routines allow a target object to be specified
       using a string that is supposed to describe its type in such a way that
       one can tell from this which pivots (ie which fibres) can be assigned 
       to it. When such an object is drawn on the mimic display it is done 
       using a canvas object. This routine returns the argument string for a
       canvas create command that can be used to draw an object of this type
       at a specified point.
       
       This routine takes the two characters describing a target, the center
       coordinates of the target and the current zoom factor being used for
       the display, and returns the arguments to be used when drawing that 
       target. The passed characters should have been returned by 
       FpilEncodeTargetType().

       Invokes the instrument specific draw target type routine
       using the specified arguments.

 * Language:
      C

 * Call:
      (void) = FpilDrawTarget(inst, targetType, targetSpect, x, y, zoom,
                                   tscale, radius, offset, useGrey, arguments)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) targetType      (char)      The single character describing the
                                      target type as set by the routine
                                      FpilEncodeTargetType().
      (>) targetSpect     (char)      The single character providing 
                                      supplementary information about the target
                                      type as set by FpilEncodeTargetType().
      (>) x               (int)       The X-coordinate of the center of the
                                      target in microns.
      (>) y               (int)       The Y-coordinate of the center of the
                                      target in microns.
      (>) zoom            (float)     The current zoom factor.
      (>) tscale          (float)     A scaling factor to apply to the sizes
                                      of the targets. 1.0 is normal.
      (>) radius          (int)       The radius of the circle drawn on the
                                      mimic display.
      (>) offset          (int)       The offset applied to the circle on the
                                      mimic display.
      (>) useGrey         (short)     A flag which, if non-zero, indicates that
                                      colours should all be grey shades.
      (<) arguments       (char*)     Address of the argument string to use to
                                      draw the target. The string should be at
                                      least 80 characters long.
 
 * Returned value:
      The argument string for a Tcl/Tk canvas create command.

 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     19-Jul-2000 -  KS - Original version
      4-Jun-2001 -  KS - Modified thickness and colour of VLT guide stars.
     12-Jun-2001 -  KS - Added useGrey argument.
     20-Jun-2001 -  KS - Added tscale argument.
 */
 
/*
 *  Object sizes.  These sizes are the sizes of objects of different
 *  types in microns.  Of course these aren't real sizes, just how
 *  we represent each object type. These are the default sizes.
 */

#define OBJECT_SIZE_FIDUCIAL 4200   /* For fiducial objects */
#define OBJECT_SIZE_PROGRAM  3000   /* For normal program objects */
#define OBJECT_SIZE_GUIDE    5800   /* For (VLT) guide objects */
 
extern void FpilDrawTarget (
    const FpilType instrument,
    char  targetType,
    char  targetSpect,
    int   x,
    int   y,
    float zoom,
    float tscale,
    int   radius,
    int   offset,
    short useGrey,
    char* arguments)
{
    int    i;
    double scale;
    int    disabledTarget;
    float  size;
    float  xb[2],yb[2];
    float  xc,yc;
    
    /*  We handle disabled types by clearing the disabled flag and
     *  treating the target as if it were enabled. But we remember
     *  the disabling for later.
     */
     
    disabledTarget = 
          (targetType & DISABLED_TARGET_BIT);    /* Was disabled flag set ? */
    targetType &= ~DISABLED_TARGET_BIT;          /* Clear any disabled flag */
      
    /*  We can use the same code for all supported instruments at the 
     *  moment.  We ignore the spectrograph type and simply treat the 
     *  two types of guide object 'F' and 'G' separately.
     */
   
    switch (targetType)
    {
        case 'F':
            size = OBJECT_SIZE_FIDUCIAL;
            break;
        case 'G':
            size = OBJECT_SIZE_GUIDE;
            break;
        default:
            size = OBJECT_SIZE_PROGRAM;
            break;
     }
     size *= tscale;
     
     /*  Now work out the x and y limits for the target on the field plate.
      *  Note that we keep the final drawn size of the target the same no
      *  matter what the zoom factor, by dividing by zoom at this point.
      */
     
     xc = (float)x; 
     yc = (float)y; 
     xb[0] = xc-size/zoom;
     yb[0] = yc-size/zoom;
     xb[1] = xc+size/zoom;
     yb[1] = yc+size/zoom;
     
     /*  Now we convert to display values */
     
     scale = (double)radius/(double)FpilGetFieldRadiusIncPark(instrument);
     for (i=0;i<2;i++)
     {                  
         xb[i] = (scale * xb[i] + (double)offset) * zoom + 0.5;
         yb[i] = (-1.0 * scale * yb[i] + (double)offset) * zoom + 0.5;
     }
     xc = (scale * xc + (double)offset) * zoom + 0.5;
     yc = (-1.0 * scale * yc + (double)offset) * zoom + 0.5;
     
     /*  Now we have all we need to create the argument string, which again
      *  depends only on the target type. At this point we could use
      *  disabledTarget to highlight disabled targets in some way, should
      *  we care to.
      */
   
     if (targetType == 'P') {
     
         /*  'Normal' program targets  */
         
         sprintf(arguments,"oval %f %f %f %f",xb[0],yb[0],xb[1],yb[1]);
         
     } else if (targetType == 'F') {
     
         /*  Guide stars - but not VLT guide probe targets  */
         
         if (useGrey) {
             sprintf(arguments,
               "rect %f %f %f %f -outline grey40",xb[0],yb[0],xb[1],yb[1]);
         } else {
             sprintf(arguments,
               "rect %f %f %f %f -outline maroon",xb[0],yb[0],xb[1],yb[1]);
         }
               
     } else if (targetType == 'G') {
     
         /*  VLT guide probe targets  */
         
         if (useGrey) {
             sprintf(arguments,"oval %f %f %f %f -outline grey20 -width 2",
                                                    xb[0],yb[0],xb[1],yb[1]);
         } else {
             sprintf(arguments,"oval %f %f %f %f -outline blue -width 2",
                                                    xb[0],yb[0],xb[1],yb[1]);
         }
                                                    
     } else if (targetType == 'S' || targetType == 'J' || targetType == 'A') {
     
         /*  Sky targets of all types  */
         
         if (useGrey) {
             sprintf(arguments,
                  "poly %f %f %f %f %f %f %f %f -outline grey60",
                   xb[0],yc,xc,yb[0],xb[1],yc,xc,yb[1]);
         } else {
             sprintf(arguments,
                  "poly %f %f %f %f %f %f %f %f -outline DeepSkyBlue",
                   xb[0],yc,xc,yb[0],xb[1],yc,xc,yb[1]);
         }
               
     } else  {
     
         /*  Anything else */
         
         sprintf(arguments,"oval %f %f %f %f",xb[0],yb[0],xb[1],yb[1]);
     }
}

/*  ------------------------------------------------------------------------  */
 
/*
 *+			      F p i l  U n a l l o c a t e d  T y p e

 * Function name:
      FpilUnallocatedType

 * Function:
       Returns a character identifying an unallocated pivot.

 * Description:
       The original 2dF version of the configuration program used the
       character 'U' to indicate an unallocated pivot. Since the encoding of
       pivot types has now been abstracted into the Fpil library, the code 
       can no longer assume that 'U' is not a character used to describe a 
       pivot type (FLAMES, for example, has UVES fibres, naturally encoded 
       as 'U'). So this routine returns a character that can be used 
       unambiguously to indicate an unallocated pivot.
       
       Invokes the instrument specific unallocated chracter type routine
       using the specified arguments.

 * Language:
      C

 * Call:
      (char) = FpilUnallocatedType(inst)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
 
 * Returned value:
      The character to be used to indicate an unallocated pivot.

 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     12-Jul-2000 -  KS - Original version
     17-Oct-2002 -  KS - 2dF/6dF now use 'U' for compatability with
                         observing software.
 */
 
extern char FpilUnallocatedType (
    const FpilType instrument)
{
    int isFlames;
    char UnallocChar;
    
    /*  For 2dF and 6dF, we use the traditional 'U'. This is necessary for
     *  compatability with the observing software, since this code is
     *  included in the SDS format files used.
     */
     
    UnallocChar = 'U';
    isFlames = !strcmp(FpilGetInstName(instrument),"FLAMES");
    if (isFlames) {
    
       /*  FLAMES cannot use 'U', because this clashes with UVES, so we
        *  use 'X' instead.
        */
        
       UnallocChar = 'X';
    }
   
    return (UnallocChar);
}

/*  ------------------------------------------------------------------------  */
 
/*
 *+			      F p i l  D i s a b l e d  T a r g e t  T y p e

 * Function name:
      FpilDisabledTargetType

 * Function:
       Returns the 'disabled' type character for a given target type.

 * Description:
       Some instruments, such as FLAMES, have to allow for the possibility
       that a target is disabled for some reason. For reasons of program
       structure, this is best done by having 'disabled' types that
       correspond to the ordinary target types supported by the instrument.
       (This is because the program only maintains a single field that
       needs to contain both the target type - as specified in the input
       files - and the disabled/enabled information.)
       
       This routine takes the character that encodes a given target type
       (as returned by FpilEncodeTargetType()) and returns the disabled
       version of that target code character. If the passed character is
       invalid or represents a disabled type already, the character is
       returned unchanged.
       
       Invokes the instrument specific disabled target type routine
       using the specified arguments.

 * Language:
      C

 * Call:
      (char) = FpilDisabledTargetType(inst,targetType)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) targetType      (char)      The single character describing the
                                      target type as set by the routine
                                      FpilEncodeTargetType().
 
 * Returned value:
      The character indicating a disabled target of the specified type.

 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     27-Jul-2000 -  KS - Original version
 */
 
extern char FpilDisabledTargetType (
    const FpilType instrument,
    char targetType)
{
    /*  Strictly, we ought to check for FLAMES here, but for the moment
     *  we'll just set the disabled target flag.
     */

    return (targetType |= DISABLED_TARGET_BIT);
}

/*  ------------------------------------------------------------------------  */
 
/*
 *+			      F p i l  E n a b l e d  T a r g e t  T y p e

 * Function name:
      FpilEnabledTargetType

 * Function:
       Returns the 'enabled' type character for a given target type.

 * Description:
       Some instruments, such as FLAMES, have to allow for the possibility
       that a target is disabled for some reason. For reasons of program
       structure, this is best done by having 'disabled' types that
       correspond to the ordinary target types supported by the instrument.
       (This is because the program only maintains a single field that
       needs to contain both the target type - as specified in the input
       files - and the disabled/enabled information.)
       
       This routine takes the character that encodes a given target type
       (as returned by FpilEncodeTargetType()) and returns the enabled
       version of that target code character. If the passed character is
       invalid or represents an enabled type already, the character is
       returned unchanged. In other words, this reverses the changes (if any)
       made to a target type by FpilDisabledTargetType().
       
       Invokes the instrument specific enabled target type routine
       using the specified arguments.

 * Language:
      C

 * Call:
      (char) = FpilEnabledTargetType(inst,targetType)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) targetType      (char)      The single character describing the
                                      target type as set by the routine
                                      FpilEncodeTargetType().
 
 * Returned value:
      The character indicating an enabled target of the specified type.

 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     27-Jul-2000 -  KS - Original version
 */
 
extern char FpilEnabledTargetType (
    const FpilType instrument,
    char targetType)
{
    /*  Strictly, we ought to check for FLAMES here, but for the moment
     *  we'll just clear the disabled target flag.
     */

    return (targetType &= ~DISABLED_TARGET_BIT);
}

/*  ------------------------------------------------------------------------  */

/*
 *+			      F p i l  B r o k e n  F i b r e  T y p e

 * Function name:
      FpilBrokenFibreType

 * Function:
       Returns the 'broken' type code for a given pivot/fibre type.

 * Description:
       The routine FpilConfWhichSpec() returns an integer code value
       that identifies the fibre type associated with the a specified pivot.
       This does not contain any information as to whether or not the 
       fibre in question is broken (or should be treated as broken). This
       information is held separately in the instrument configuration
       structures. Some of the configuration code (mostly the user interface
       part) uses the integer returned by FpilConfWhichSpec() as an index,
       and ignores the question of broken fibres. However, the actual
       configuration algorithms need to be able to combine both the pivot
       type code and the information as to whether or not the fibre is
       broken into one single number. This routine takes the index number
       returned by FpilConfWhichSpec() and modifies it to indicate that the
       fibre is broken (all it does is set a bit in the number). Note that
       the resulting code should not be used as an index, as it will not be
       in the normal range of 0..(Number of fibre types).
       
       This routine takes the integer that identifies a given pivot/fibre type
       (as returned by FpilConfWhichSpec()) and returns the broken
       version of that integer code. If the passed code value is
       invalid or represents a broken type already, the value is
       returned unchanged.
       
       Invokes the instrument specific broken fibre type routine
       using the specified arguments.

 * Language:
      C

 * Call:
      (int) = FpilBrokenFibreType(inst,pivotType)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) pivotType       (int)       The single character describing the
                                      pivot/fibre type as set by the routine
                                      FpilConfWhichSpec().
 
 * Returned value:
      The character indicating a disabled pivot/fibre of the specified type.

 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     27-Jul-2000 -  KS - Original version
 */
 
extern int FpilBrokenFibreType (
    const FpilType instrument,
    int pivotType)
{
    /*  Strictly, we ought to check for FLAMES here, but for the moment
     *  we'll just set the broken pivot flag.
     */

    return (pivotType |= BROKEN_FIBRE_BIT);
}

/*  ------------------------------------------------------------------------  */

/*
 *+			        F p i l  I s  F i b r e  B r o k e n

 * Function name:
      FpilIsFibreBroken

 * Function:
       Sees if a pivot fibre type indicates a broken fibre.

 * Description:
       This routine takes the fibre type code describing a pivot/fibre type
       and returns true if it indicates a broken fibre.  The passed fibre type
       code should have been returned by FpilConfWhichSpect() originally and
       this routine is really telling us if FpilBrokenFibreType() has been
       used to modify the code. See FpilBrokenFibreType() for more details.

       Invokes the instrument specific virtual broken fibre type routine
       using the specified arguments.

 * Language:
      C

 * Call:
      (int) = FpilIsFibreBroken (inst, fibreType)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) fibreType       (int)       The single integer code describing the
                                      fibre type.
 
 * Returned value:
      True (ie non-zero) if the type code indicates a broken fibre, false
      otherwise.

 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     12-Jul-2000 -  KS - Original version
 */
 
extern int FpilIsFibreBroken (
    const FpilType instrument,
    int fibreType)
{
    /*  We just look to see if the broken fibre bit is set in the code.
     */
           
   return (fibreType & BROKEN_FIBRE_BIT);
}

/*  ------------------------------------------------------------------------  */

/*
 *+			   F p i l  F i b r e  R e a l l y  E x i s t s

 * Function name:
      FpilFibreReallyExists

 * Function:
       Sees if a pivot fibre type indicates a real, existing, fibre.

 * Description:
       This routine takes the fibre type code describing a pivot/fibre type.
       Some instruments (FLAMES) have instrument descriptions that include
       fibres that don't actually exist. They are there just as placeholders
       to keep the fibre numbering consistent. These fibres are incompatible
       with any target type, of course, so they don't get allocated, but
       sometimes - particularly if you want to check for collisions with
       parked fibres (and these remain permanently 'parked'!) then you need
       to know if the fibre is really there.

       Invokes the instrument specific virtual existing fibre type routine
       using the specified arguments.

 * Language:
      C

 * Call:
      (int) = FpilFibreReallyExists (inst, fibreType)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) fibreType       (int)       The single integer code describing the
                                      fibre type.
 
 * Returned value:
      True (ie non-zero) if the type code indicates a real fibre, false
      otherwise.

 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     12-Jul-2000 -  KS - Original version
 */
 
extern int FpilFibreReallyExists (
    const FpilType instrument,
    int fibreType)
{
    int isFlames;
    int exists = TRUE;
    
    /*  Only FLAMES supports non-existent fibres */
    
    isFlames = !strcmp(FpilGetInstName(instrument),"FLAMES");
    if (isFlames) {

        /*  Clip out the broken fibre bit, and see if we have a type classed as
         *  'none' - this is how WhichSpec() ends up classifying non-existent
         *  FLAMES fibres.
         */
       
        fibreType &= ~BROKEN_FIBRE_BIT;
        exists = (fibreType != FLAMES_FIBRE_TYPE_NONE);
    }
    return (exists);
}

/*  ------------------------------------------------------------------------  */

/*
 *+			        F p i l  I s  F i b r e  G u i d e

 * Function name:
      FpilIsFibreGuide

 * Function:
       Sees if a pivot fibre type indicates a guide fibre.

 * Description:
       This routine takes the fibre type code describing a pivot/fibre type
       and returns true if it indicates a guide fibre.  The passed fibre type
       code should have been returned by FpilConfWhichSpect() originally.
       Note that this routine ignores the question of broken fibres. If
       FpilConfWhichSpect() returned a code corresponding to a guide fibre,
       then this routine will return true irrespective of whether the routine
       FpilBrokenFibreType() has been called for the type or not.

       Invokes the instrument specific virtual broken fibre type routine
       using the specified arguments.

 * Language:
      C

 * Call:
      (int) = FpilIsFibreGuide (inst, fibreType)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) fibreType       (int)       The single integer code describing the
                                      fibre type.
 
 * Returned value:
      True (ie non-zero) if the type code indicates a broken fibre, false
      otherwise.

 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     12-Jul-2000 -  KS - Original version
 */
 
extern int FpilIsFibreGuide (
    const FpilType instrument,
    int fibreType)
{
    /*  As it happens, at present, all supported instruments use zero
     *  as the code for a guide fibre, so for the moment we can cheat and use
     *  that. But this is NOT a safe assumption. For FLAMES, for example,
     *  we should be testing against FLAMES_FIBRE_TYPE_GUIDE (which at the
     *  moment does happen to be zero).
     */
     
   return (fibreType == 0);
}

/*  ------------------------------------------------------------------------  */

/*
 *+			        F p i l  F i b r e  C o l o u r

 * Function name:
       FpilFibreColour

 * Function:
       Returns a colour to use to draw a fibre, based on its type.

 * Description:
       This routine takes the fibre type code describing a pivot/fibre type
       and returns a colour that can be used to draw it.  The passed fibre type
       code should have been returned by FpilConfWhichSpect() originally, and
       may have subsequently been modified using FpilBrokenFibreType().

       Invokes the instrument specific virtual fibre colour type routine
       using the specified arguments.

 * Language:
      C

 * Call:
      (void) = FpilFibreColour (inst, fibreType, fibreCombo, 
                                                   useGrey, colour, nChar)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) fibreType       (int)       The single integer code describing the
                                      fibre type.
      (>) fibreCombo      (int)       The index for the current fibre 
                                      combination. If -1, no combination has
                                      been selected, values from 0 up are
                                      combination Ids as passed to
                                      FpilComboFibreCodes().
      (>) useGrey         (short)     A flag which, if non-zero, indicates that
                                      colours should all be grey shades.
      (<) colour          (char*)     Address of string into which the colour
                                      to use is to be written.
      (>) nChar           (int)       The number of bytes available at the
                                      address given by colour.
 
 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     12-Jul-2000 -  KS - Original version
     15-Jan-2001 -  KS - Added support for IFU Sky fibres.
     12-Jun-2001 -  KS - Added useGrey argument.
     28-Aug-2001 -  KS - Added support for UVES_7 and UVES_8 types.
 */
 
extern void FpilFibreColour (
    const FpilType instrument,
    int fibreType,
    int fibreCombo,
    short useGrey,
    char* colour,
    int nChar)
{
    int isFlames;
    isFlames = !strcmp(FpilGetInstName(instrument),"FLAMES");
    
    /*  For the moment, we treat FLAMES separately from 2dF and 6dF */
    
    if (isFlames) {
    
        /*  For FLAMES, we use a pretty arbitrary set of colours and allow
         *  for broken fibres.
         */
         
        if (fibreType & BROKEN_FIBRE_BIT) {
            if (useGrey) strncpy (colour,"grey95",nChar);
            else strncpy (colour,"Gray",nChar);
        } else {
        
            if (fibreCombo < 0) {
            
               /*  This is the case where no fibre combination has been
                *  selected. The choice of colours is pretty arbitrary.
                */
        
                switch (fibreType) {
                    case FLAMES_FIBRE_TYPE_GUIDE: {
                        if (useGrey) strncpy (colour,"grey70",nChar);
                        else strncpy (colour,"Green",nChar);
                        break;
                    }
                    case FLAMES_FIBRE_TYPE_IFU: {
                        if (useGrey) strncpy (colour,"grey50",nChar);
                        else strncpy (colour,"Orange",nChar);
                        break;
                    }
                    case FLAMES_FIBRE_TYPE_IFU_SKY: {
                        if (useGrey) strncpy (colour,"grey30",nChar);
                        else strncpy (colour,"Yellow",nChar);
                        break;
                    }
                    case FLAMES_FIBRE_TYPE_MEDUSA: {
                        if (useGrey) strncpy (colour,"grey40",nChar);
                        else strncpy (colour,"LightBlue",nChar);
                        break;
                    }
                    case FLAMES_FIBRE_TYPE_UVES: 
                    case FLAMES_FIBRE_TYPE_UVES_7: 
                    case FLAMES_FIBRE_TYPE_UVES_8: {
                        if (useGrey) strncpy (colour,"grey80",nChar);
                        else strncpy (colour,"Wheat",nChar);
                        break;
                    }
                    case FLAMES_FIBRE_TYPE_ARGUS: {
                        if (useGrey) strncpy (colour,"grey20",nChar);
                        else strncpy (colour,"Cyan",nChar);
                        break;
                    }
                    case FLAMES_FIBRE_TYPE_TEST: {
                        if (useGrey) strncpy (colour,"grey10",nChar);
                        else strncpy (colour,"Pink",nChar);
                        break;
                    }
                    case FLAMES_FIBRE_TYPE_NONE: {
                        if (useGrey) strncpy (colour,"grey90",nChar);
                        else strncpy (colour,"Gray",nChar);
                        break;
                    }
                    default : {
                        if (useGrey) strncpy (colour,"grey95",nChar);
                        else strncpy (colour,"Gray",nChar);
                        break;
                    }
                }           /*  End switch on fibre type */
                
            } else {
            
                /*  This is the case where a fibre combination has been
                 *  selected. We colour the primary type red and the 
                 *  secondary type a light blue. Guide fibres are green
                 *  and all others are gray.
                 */
                 
                if (fibreType == FLAMES_FIBRE_TYPE_GUIDE) {
                    if (useGrey) strncpy (colour,"grey90",nChar);
                    else strncpy (colour,"Green",nChar);
                } else {
                
                    int numTypeCodes;
                    int TypeCodes[3];
                    
                    TypeCodes[0] = TypeCodes[1] = TypeCodes[2] = -1;
                    
                    FpilComboFibreCodes (instrument,fibreCombo,
                                            3,&numTypeCodes,TypeCodes);
                    if (fibreType == TypeCodes[0]) {
                        if (useGrey) strncpy (colour,"grey55",nChar);
                        else strncpy (colour,"Red",nChar);
                    } else if (fibreType == TypeCodes[1]) {
                        if (useGrey) strncpy (colour,"grey75",nChar);
                        else strncpy (colour,"LightBlue",nChar);
                    } else {
                        if (useGrey) strncpy (colour,"grey95",nChar);
                        else strncpy (colour,"Gray",nChar);
                    }
                }
            }
        }
           
    } else {
    
        /*  For 2dF and 6dF, we make broken fibres grey, guide fibres
         *  green and everything else blue. We assume the guide fibres
         *  all have a type code of zero.
         */
         
        if (fibreType & BROKEN_FIBRE_BIT) {
            strncpy (colour,"Gray",nChar);
        } else {
            if (fibreType == 0) {
                if (useGrey) strncpy (colour,"grey95",nChar);
                else strncpy (colour,"Green",nChar);
            } else {
                if (useGrey) strncpy (colour,"grey55",nChar);
                else strncpy (colour,"LightBlue",nChar);
            }
        }
    }
    
    /*  Make sure the colour string is properly terminated */
    
    colour[nChar - 1] = '\0';
}


/*  ------------------------------------------------------------------------  */

/*
 *+			          F p i l  T a r g e t  C r i t e r i a

 * Function name:
       FpilTargetCriteria

 * Function:
       Return a set of criteria used to distinguish between target types.

 * Description:
       To assist in the 'highlighting' of various targets according to 
       various criteria, this routine allows the Fpil layer to define a
       set of instrument-dependent criteria for selecting different
       target types. The criteria can be anything the Fpil instrument
       dependent routines want, although for each one it must be possible
       to give a simple 'yes/no' response to the question 'does a target
       with this type and spect code (returned by FpilEncodeTargetType())
       meet this criterion?

 * Language:
      C

 * Call:
      (void) = FpilTargetCriteria (inst, numCriteria, criteria)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst         (FpilType)      The instrument as returned 
                                       by FpilInit(3).
      (<) numCriteria  (unsigned int*) The number of instrument-dependent
                                       criteria.
      (<) criteria     (char*[])       Receives the address of an array of
                                       strings each containing a nul-terminated
                                       criterion string.
 
 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     21-Aug-2000 -  KS - Original version
     16-Jan-2001 -  KS - Added support for IFU Sky fibres.
     01-Oct-2002 -  KS - 6dF only has FLAIR, not two spectrographs.
 */

static const char* const FlamesTargetCriteria[] =
    {"Disabled","IFU","IFU sky","UVES","Medusa","Argus sky","Sky",
                                                    "FACB guide","VLT guider"};
static const char* const TwodfTargetCriteria[] =
    {"Guide","Sky","Spect A","Spect B"};
    
static const char* const SixdfTargetCriteria[] =
    {"Guide","Sky","FLAIR"};
    
extern void FpilTargetCriteria (
    const FpilType instrument,
    unsigned int* numCriteria,
    const char* const **criteria)
{
    int isFlames;
    int is2dF;
    isFlames = !strcmp(FpilGetInstName(instrument),"FLAMES");
    
    /*  For the moment, we treat FLAMES separately from 2dF and 6dF */
    
    if (isFlames) {
           
       *numCriteria = sizeof(FlamesTargetCriteria)/sizeof(char*);
       *criteria = FlamesTargetCriteria;  
           
    } else {
    
       is2dF = !strcmp(FpilGetInstName(instrument),"2dF");
       
       if (!is2dF) {
           *numCriteria = sizeof(SixdfTargetCriteria)/sizeof(char*);
           *criteria = SixdfTargetCriteria;
       } else {
           *numCriteria = sizeof(TwodfTargetCriteria)/sizeof(char*);
           *criteria = TwodfTargetCriteria;
       }
    }
    
}

/*  ------------------------------------------------------------------------  */

/*
 *+			    F p i l  T a r g e t  C r i t e r i a  M e t

 * Function name:
       FpilTargetCriteriaMet

 * Function:
       Returns the number of criteria a given target type meets.

 * Description:
       This routine assumes that FpilTargetCriteria() has been called to
       supply a set of criteria for selecting targets on the basis of
       their type and spect codes (as returned by FpilEncodeTargetType()).
       It is passed an array of flags showing which of these criteria
       have been selected together with the character codes giving type 
       and spect for a given target type. (In some cases, the spect code
       will not be known to the calling layers of the code, and there is
       an additional parameter that can be used to show this.) This
       routine then returns the number of flagged criteria that are met
       by a target of the given type.
       
       The numCriteria parameter should match that supplied by the routine
       FpilTargetCriteria(), and the elements of the flags array correspond
       to the elements of the criteria array returned by FpilTargetCriteria().

 * Language:
      C

 * Call:
      (void) = FpilTargetCriteria (inst, flags, numCriteria, targetType,
                                                targetSpect, spectKnown)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst         (FpilType)      The instrument as returned 
                                       by FpilInit(3).
      (>) flags        (int[])         Array of flags indicating which of the
                                       criteria should be used.
      (>) numCriteria  (unsigned int*) The number of instrument-dependent
                                       criteria.
      (>) targetType   (char)          The single character set to the 'type' 
                                       character describing the target.
      (>) targetSpect  (char)          The single character set to the 'spect' 
                                       character describing the target.
      (>) spectKnown   (int)           True if the 'spect' character should be
                                       used. If false, targetSpect is ignored.
 
 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     21-Aug-2000 -  KS - Original version
     16-Jan-2001 -  KS - Added support for IFU Sky fibres.
     18-Aug-2001 -  KS - gcc -Wall erroneously complained about TargetIndex
                         being used uninitialised. Fixed.
     01-Oct-2002 -  KS - 6dF only has one spectrograph.
 */


extern int FpilTargetCriteriaMet (
    const FpilType instrument,
    const int* flags,
    int numCriteria,
    char targetType,
    char targetSpect,
    int spectKnown)
{
    int criteriaMet;
    short isFlames;
    short is2dF;
    short disabled;

    /*  Initial values  */
    
    criteriaMet = 0;
    disabled = FALSE;
    
    /*  See which instrument we are using */
    
    isFlames = !strcmp(FpilGetInstName(instrument),"FLAMES");

    if (isFlames) {
    
        /*  For FLAMES, this code uses a criterion matched matrix. This is
         *  actually over the top, since as it happens each target type
         *  matches one and only one criterion. (With the exception of 'Test'
         *  targets which don't match anything, and the 'Disabled' criterion
         *  which is handled separately at the end of the tests, but is 
         *  included in the matrix so as not to confuse the indexing, since
         *  it is the first criterion used.) This code would allow one target
         *  type to satisfy more than one criterion, if anyone wanted 
         *  such a test - we could have a criterion that was 'any of IFU,
         *  UVES or Medusa', for example. Note that the order of rows in 
         *  this matrix must match the order of criteria returned by
         *  FpilTargetCriteria(), and that the disabled test at the very end
         *  explicitly assumes that Disabled is criterion zero.
         */
            
        static char FlamesTargetTypes[] = {'F','S','P','G','U','I','T','A','J'};
        static int FlamesCriteriaMet[] = {
           /*  'F','S','P','G','U','I','T','A','J'      <- Target types */
                0,  0,  0,  0,  0,  0,  0,  0,  0,     /* Disabled */
                0,  0,  0,  0,  0,  1,  0,  0,  0,     /* IFU  */
                0,  0,  0,  0,  0,  0,  0,  0,  1,     /* IFU sky */
                0,  0,  0,  0,  1,  0,  0,  0,  0,     /* UVES */
                0,  0,  1,  0,  0,  0,  0,  0,  0,     /* Medusa */
                0,  0,  0,  0,  0,  0,  0,  1,  0,     /* Argus */
                0,  1,  0,  0,  0,  0,  0,  0,  0,     /* Sky */
                1,  0,  0,  0,  0,  0,  0,  0,  0,     /* FACB guide */
                0,  0,  0,  1,  0,  0,  0,  0,  0      /* VLT guider */
           };

        int NTargetTypes = sizeof(FlamesTargetTypes)/sizeof(char);

        int TargetOK;

        int TargetIndex;
        int I;
        int Icrit;
        
        /*  If the target type is disabled, we ignore that flag for the
         *  initial tests - by clearing it - but remember that it was set
         *  for the final test against the disabled criterion.
         */

        if (targetType & DISABLED_TARGET_BIT) {

            disabled = TRUE;
            targetType &= ~DISABLED_TARGET_BIT;
        }

        /*  Actually, the pivot types returned by FpilConfWhichSpec()
         *  are integers and we could just use them as the PivotIndex,
         *  but this acts as an additional check and makes the code a
         *  little more obvious - it also makes it easier to change the
         *  allocation of indices to fibre tyes.
         */

        TargetOK = FALSE;
        TargetIndex = -1;
        for (I = 0; I < NTargetTypes; I++) {
            if (FlamesTargetTypes[I] == targetType) {
                TargetIndex = I;
                TargetOK = TRUE;
                break;
            }
        }
        
        /*  The target type is valid, see how many criteria it matches,
         *  using the table.  Then see if the 'Disabled' criterion is 
         *  in use (the zeroth index) and test that explicitly.
         */
         
        if (TargetOK) {
            for (Icrit = 1; Icrit < numCriteria; Icrit++) {
                if (flags[Icrit]) {
                    if (FlamesCriteriaMet[
                           (Icrit * NTargetTypes) + TargetIndex]) {
                        criteriaMet++;
                    }
                }
            }
            if (disabled && flags[0]) criteriaMet++;

         }

     } else {
     
         is2dF = !strcmp(FpilGetInstName(instrument),"2dF");
         
         if (is2dF) {
         
             /*  For 2dF we assume flags[0] .. flags[3] refer
              *  to guide, sky, spect A and spect B, and test accordingly.
              *  If the target spectrograph isn't known, then we allow
              *  either spectrograph criterion to count against a program
              *  object.
              */
        
             criteriaMet = 0;
             if ((targetType == 'F') && (flags[0])) criteriaMet++;
             if ((targetType == 'S') && (flags[1])) criteriaMet++;
             if (targetType == 'P') {
                 if (spectKnown) {
                    if (flags[2] || flags[3]) criteriaMet++;
                 } else {
                    if (flags[2] && (targetSpect == 'A')) criteriaMet++;
                    if (flags[3] && (targetSpect == 'B')) criteriaMet++;
                 }
             }
         } else {
         
             /*  6dF is like 2dF, except that it only has one spectrograph */
             
             criteriaMet = 0;
             if ((targetType == 'F') && (flags[0])) criteriaMet++;
             if ((targetType == 'S') && (flags[1])) criteriaMet++;
             if (targetType == 'P' && (flags[2])) criteriaMet++;             
         }
     }
     
     return (criteriaMet);

}

/*  ------------------------------------------------------------------------  */

/*
 *+			    F p i l  P A F  T y p e s

 * Function name:
       FpilPAFTypes

 * Function:
       Returns the number of PAF types supported by the instrument.

 * Description:
       This routine is really only relevant if FLAMES is being used, 
       since the PAF output format is supported only for ESO purposes.
       However, in principle other instruments might use PAF format.
       For PAF purposes, certain fibre types are collected together,
       for example, both IFU fibres and Medusa fibres are considered
       to be Giraffe fibres - that's we way the output format was
       defined. So this routine returns the number of such PAF type
       classifications supported. It also returns for each type a string
       describing the type and a 4-letter abbreviation actually used in
       the hierarchical names in the PAF file.

 * Language:
      C

 * Call:
      (void) = FpilPAFTypes (inst, maxTypes, numTypes, typeNames,
                                                               abbrNames)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst         (FpilType)      The instrument as returned 
                                       by FpilInit(3).
      (>) maxTypes     (int)           The maximum number of PAF types that
                                       the caller can support - ie the
                                       dimension of the typeNames array.
      (<) numTypes     (int*)          The actual number of PAF types that
                                       the system defines for this instrument.
      (<) typeNames    (char*[])       Each element of this array is set to a
                                       pointer to a string describing one of
                                       the PAF types.
      (<) abbrNames    (char*[])       Each element of this array is set to a
                                       pointer to a string describing the
                                       abbreviated PAF name used for the type.
 
 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     21-Aug-2000 -  KS - Original version
 */

extern void FpilPAFTypes (
    const FpilType instrument,
    int maxTypes,
    int *numTypes,
    char* typeNames[],
    char* abbrNames[])
{
    short isFlames;
    
    /*  See which instrument we are using */
    
    isFlames = !strcmp(FpilGetInstName(instrument),"FLAMES");

    if (isFlames) {
    
        /*  FLAMES categorises fibres for PAF purposes as ARGUS,
         *  Giraffe (both Medusa and IFU fibres) and UVES. 
         */
         
        *numTypes = 3;
        if (maxTypes >= 1) {
            typeNames[0] = "GIRAFFE";
            abbrNames[0] = "GIRA";
        }
        if (maxTypes >= 2) {
            typeNames[1] = "UVES";
            abbrNames[1] = "UVES";
        }
        if (maxTypes >= 3) {
            typeNames[2] = "ARGUS";
            abbrNames[2] = "ARGS";
        }
    } else {
    
        /*  6dF and 2dF really aren't relevant here, so we return no
         *  categories at all.
         */
       
        *numTypes = 0;
    }
}

/*  ------------------------------------------------------------------------  */

/*
 *+			    F p i l  P A F  F i b r e  C o d e s

 * Function name:
       FpilPAFFibreCodes

 * Function:
       Returns the fibre codes corresponding to each PAF type.

 * Description:
       This routine is really only relevant if FLAMES is being used, 
       since the PAF output format is supported only for ESO purposes.
       However, in principle other instruments might use PAF format.
       For PAF purposes, certain fibre types are collected together,
       for example, both IFU fibres and Medusa fibres are considered
       to be Giraffe fibres - that's we way the output format was
       defined. This routine is passed a PAF fibre type - just an index
       from zero up to one less than the number of PAF types returned
       by a call to FpilPAFTypes(). It returns the number of fibre types
       that are classified as matching this PAF type, and an array
       giving the fibre type codes in question. For example, the PAF
       'Giraffe' type is matched by fibres that are Medusa, IFU or IFU
       sky.

 * Language:
      C

 * Call:
      (void) = FpilPAFFibreCodes (inst, PAFType, maxTypeCodes, numTypeCodes,
                                                             TypeCodes)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst         (FpilType)      The instrument as returned 
                                       by FpilInit(3).
      (>) PAFType      (int)           The PAF type code.
      (>) maxTypeCodes (int)           The maximum number of fibre types 
                                       matching this PAF type that the caller
                                       can support - ie the dimension of the 
                                       TypeCodes array.
      (<) numTypeCodes (int*)          The actual number of fibre types that
                                       the correspond to this PAF type.
      (<) TypeCodes    (int[])         Each element of this array is set to a
                                       fibre type code corresponding to this PAF
                                       type.
 
 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     21-Aug-2000 -  KS - Original version
     30-Oct-2001 -  KS - TEST fibres are now included in Giraffe classification.
 */

 
extern void FpilPAFFibreCodes (
    const FpilType instrument,
    int PAFType,
    int maxTypeCodes,
    int *numTypeCodes,
    int TypeCodes[])
{
    short isFlames;
    
    /*  See which instrument we are using */
    
    isFlames = !strcmp(FpilGetInstName(instrument),"FLAMES");

    if (isFlames) {
    
        /*  FLAMES categorises fibres for PAF purposes as 
         *  Giraffe (both Medusa and IFU fibres), UVES and ARGUS,
         *  with PAF types of 0,1 and 2 respectively.
         */
        
        if (PAFType == 0) {
        
            /*  Giraffe has Medusa and IFU and IFU sky fibres. It also has
             *  TEST fibres - they aren't really Giraffe fibres, but it's
             *  convenient to treat them as if they are.
             */
           
            *numTypeCodes = 4;
            if (maxTypeCodes >= 1) TypeCodes[0] = FLAMES_FIBRE_TYPE_MEDUSA;
            if (maxTypeCodes >= 2) TypeCodes[1] = FLAMES_FIBRE_TYPE_IFU;
            if (maxTypeCodes >= 3) TypeCodes[2] = FLAMES_FIBRE_TYPE_IFU_SKY;
            if (maxTypeCodes >= 4) TypeCodes[3] = FLAMES_FIBRE_TYPE_TEST;
           
        } else if (PAFType == 1) {

            /*  UVES has all the UVES fibre types */
           
            *numTypeCodes = 3;
            if (maxTypeCodes >= 1) TypeCodes[0] = FLAMES_FIBRE_TYPE_UVES;
            if (maxTypeCodes >= 2) TypeCodes[1] = FLAMES_FIBRE_TYPE_UVES_7;
            if (maxTypeCodes >= 3) TypeCodes[2] = FLAMES_FIBRE_TYPE_UVES_8;
           
        } else if (PAFType == 2) {

            /*  ARGUS just has ARGUS fibres */
           
            *numTypeCodes = 1;
            if (maxTypeCodes >= 1) TypeCodes[0] = FLAMES_FIBRE_TYPE_ARGUS;
           
        } else {
        
            /*  This is an invalid code */
            
            *numTypeCodes = 0;
        }
        
    } else {
    
        /*  6dF and 2dF don't really have PAF types, so we don't return
         *  any codes.
         */
         
        *numTypeCodes = 0;
    }
    
}

/*  ------------------------------------------------------------------------  */

/*
 *+			    F p i l  F i b r e  C o m b o s

 * Function name:
       FpilFibreCombos

 * Function:
       Returns the number of fibre combinations supported by the instrument.

 * Description:
       For FLAMES in particuar, with its large number of different fibre
       types, there is a requirement to be able present the user with a
       number of fibre combinations that can be configured - for example,
       Medusa fibres with IFU fibres but no ARGUS fibres. The user can then
       select one combination and only the appropriate fibre types will be
       configured. The same concept could apply to other instruments.
       This rouine returns the number of such combinations for a given
       instrument, and the strings describing these combinations. The
       mode keyword is formatted as a set of keywords that is intended to
       be taken apart by the PAF file output code, separated by slashes -
       it has the form FILE/INS.MODE/INS.GIRAF.MODE/INS.UVES.SLIT, where each
       slash-separated part is the value of the named keyword - ie the
       middle part is the keyword value for INS.GIRAF.MODE, as specified
       by ESO. The first, shown here as FILE, is to be used to form the file
       name. 
        
 * Language:
      C

 * Call:
      (void) = FpilFibreCombos (inst, maxCombos, numCombos, comboNames,
                                                               modeNames)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst         (FpilType)      The instrument as returned 
                                       by FpilInit(3).
      (>) maxCombos    (int)           The maximum number ofcombinations that
                                       the caller can support - ie the
                                       dimension of the comboNames array.
      (<) numCombos    (int*)          The actual number of combinations that
                                       the system defines for this instrument.
      (<) comboNames   (char*[])       Each element of this array is set to a
                                       pointer to a string describing one of
                                       the possible combinations.
      (<) modeNames    (char*[])       Each element of this array is set to a
                                       pointer to a string giving the value
                                       used (if any) for a 'mode' keyword
                                       associated with the combination.
 
 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
      3-Feb-2001 -  KS - Original version
      8-Oct-2001 -  KS - Added combined modes with simulation.
      1-Aug-2002 -  KS - Mode name format changed.
      2-Aug-2002 -  KS - Null mode strings now represented by '-'
      21-Nov-2002 - AK/FP  - Order and Names paranalized 
      15-Dec-2002 - AK - UVES6 added to all combined modes
 */

extern void FpilFibreCombos (
    const FpilType instrument,
    int maxCombos,
    int *numCombos,
    char* comboNames[],
    char* modeNames[])
{
    short isFlames;
    
    /*  See which instrument we are using */
    
    isFlames = !strcmp(FpilGetInstName(instrument),"FLAMES");

    if (isFlames) {
    
        /*  FLAMES allows the following combinations. 
         */
         
        *numCombos = 15;
        if (maxCombos >= 1) {
            comboNames[0] = "Medusa";
            modeNames[0] = "GIRMED/GIRAF/MED/-";
        }
        if (maxCombos >= 2) {
            comboNames[1] = "Medusa + UVES (8 fibres) [580,860nm]";
            modeNames[1] = "COMMED8/COM/MED/8FIB";
        }
        if (maxCombos >= 3) {
            comboNames[2] = "Medusa + UVES (7 fibres + 1 calibration) [580nm]";
            modeNames[2] = "COMMED7SIM/COM/MED/7+1FIB";
        }
        if (maxCombos >= 4) {
            comboNames[3] = "Medusa + UVES (6 fibres) [520nm]";
            modeNames[3] = "COMMED6/COM/MED/6FIB";
        }
        if (maxCombos >= 5) {
            comboNames[4] = "IFU";
            modeNames[4] = "GIRIFU/GIRAF/IFU/-";
        }
        if (maxCombos >= 6) {
            comboNames[5] = "IFU + UVES (8 fibres) [580,860nm]";
            modeNames[5] = "COMIFU8/COM/IFU/8FIB";
        }
        if (maxCombos >= 7) {
            comboNames[6] = "IFU + UVES (7 fibres + 1 calibration) [580nm]";
            modeNames[6] = "COMIFU7SIM/COM/IFU/7+1FIB";
        }
        if (maxCombos >= 8) {
            comboNames[7] = "IFU + UVES (6 fibres) [520nm]";
            modeNames[7] = "COMIFU6/COM/IFU/6FIB";
        }
        if (maxCombos >= 9) {
            comboNames[8] = "ARGUS sky";
            modeNames[8] = "GIRARG/GIRAF/ARG/-";
        }
        if (maxCombos >= 10) {
            comboNames[9] = "ARGUS sky + UVES (8 fibres) [580,860nm]";
            modeNames[9] = "COMARG8/COM/ARG/8FIB";
        }
        if (maxCombos >= 11) {
            comboNames[10] = "ARGUS sky + UVES (7 fibres + 1 calibration) [580nm]";
            modeNames[10] = "COMARG7SIM/COM/ARG/7+1FIB";
        }
        if (maxCombos >= 12) {
            comboNames[11] = "ARGUS sky + UVES (6 fibres) [520nm]";
            modeNames[11] = "COMARG6/COM/ARG/6FIB";
        }
        if (maxCombos >= 13) {
            comboNames[12] = "UVES (8 fibres) [580,860nm]";
            modeNames[12] = "UVES8/UVES/-/8FIB";
        }
        if (maxCombos >= 14) {
            comboNames[13] = "UVES (7 fibres + 1 calibration) [580nm]";
            modeNames[13] = "UVES7SIM/UVES/-/7+1FIB";
        }
        if (maxCombos >= 15) {
            comboNames[14] = "UVES (6 fibres) [520nm]";
            modeNames[14] = "UVES6/UVES/-/6FIB";
        }
    } else {
    
        short is2dF;
        int iCombo;
        
        /*  For the moment, for 6dF and 2dF we allow 'all fibres', 'guide
         *  only', 'spectrograph only' and for 2dF we allow the additional
         *  'spectrograph 1' and 'spectrograph 2' options. There are no
         *  particular mode keywords defined, so we return a blank string
         *  for all cases.
         */
      
        for (iCombo = 0; iCombo < maxCombos; iCombo++) {
            modeNames[iCombo] = "";
        }
       
        *numCombos = 3;
        if (maxCombos >= 1) {
            comboNames[0] = "All fibres";
        }
        if (maxCombos >= 2) {
            comboNames[1] = "Guide fibres only";
        }
        if (maxCombos >= 3) {
            comboNames[2] = "Spectrograph fibres only";
        }
        is2dF = !strcmp(FpilGetInstName(instrument),"2dF");
        if (is2dF) {
            if (maxCombos >= 4) {
                comboNames[3] = "Spectrograph 1 only";
            }
            if (maxCombos >= 5) {
                comboNames[4] = "Spectrograph 2 only";
            }
            if (maxCombos >= 6) {
                comboNames[5] = "Spectrograph 1 + guide fibres";
            }
            if (maxCombos >= 7) {
                comboNames[6] = "Spectrograph 2 + guide fibres";
            }
            *numCombos = 7;
        }
    }
}

/*  ------------------------------------------------------------------------  */

/*
 *+			    F p i l  C o m b o  F i b r e  C o d e s

 * Function name:
       FpilComboFibreCodes

 * Function:
       Returns the fibre codes corresponding to each fibre combination.

 * Description:
       This routine is passed a combination identifying number - just an
       index from zero up to one less than the number of combinations returned
       by a call to FpilFibreCombos(). It returns the number of fibre types
       that are classified as forming this combination, and an array
       giving the fibre type codes in question. For example, the combination
       'UVES + ARGUS' type is matched by fibres that are UVES or ARGUS.

 * Language:
      C

 * Call:
      (void) = FpilComboFibreCodes (inst, comboId, maxTypeCodes, numTypeCodes,
                                                             TypeCodes)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst         (FpilType)      The instrument as returned 
                                       by FpilInit(3).
      (>) comboId      (int)           The combination Id number.
      (>) maxTypeCodes (int)           The maximum number of fibre types 
                                       matching this combination that the 
                                       caller can support - ie the dimension 
                                       of the TypeCodes array.
      (<) numTypeCodes (int*)          The actual number of fibre types that
                                       form this combination.
      (<) TypeCodes    (int[])         Each element of this array is set to a
                                       fibre type code included in this
                                       combination PAF.
 
 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
      3-Feb-2001 -  KS - Original version
     28-Aug-2001 -  KS - Added support for UVES_7 and UVES_8 types.
      8-Oct-2001 -  KS - Added combinations that include simultaneous
                         calibration.
     15-Dec-2002 -  AK - Added combinations that include UVES 6 fibres			 
 */

 
extern void FpilComboFibreCodes (
    const FpilType instrument,
    int comboId,
    int maxTypeCodes,
    int *numTypeCodes,
    int TypeCodes[])
{
    /*  In retrospect, this code is far too messy. The interface should
     *  just return a pointer to an array that lists the codes, rather than
     *  setting elements of a passed array. In any case, there should be
     *  a set of tables driving it, rather than all this messy explicit code -
     *  all I can say is that when I did this first I thought there'd only
     *  be one or two different combinations..
     */
     
    short isFlames;
    
    /*  See which instrument we are using */
    
    isFlames = !strcmp(FpilGetInstName(instrument),"FLAMES");

    if (isFlames) {
    
        /*  FLAMES has a number of combinations - see the code for the
         *  routine FpilFibreCombos() for the list. This routine has to
         *  use the same order. Note that all combinations include guide
         *  fibres.
         */
        
        if (comboId == 0) {
        
            /*  Medusa only */
           
            *numTypeCodes = 2;
            if (maxTypeCodes >= 1) TypeCodes[0] = FLAMES_FIBRE_TYPE_MEDUSA;
            if (maxTypeCodes >= 2) TypeCodes[1] = FLAMES_FIBRE_TYPE_GUIDE;
           
        } else if (comboId == 1) {

            /*  'Medusa + UVES' has Medusa and all the UVES fibres.
             */
           
            *numTypeCodes = 5;
            if (maxTypeCodes >= 1) TypeCodes[0] = FLAMES_FIBRE_TYPE_MEDUSA;
            if (maxTypeCodes >= 2) TypeCodes[1] = FLAMES_FIBRE_TYPE_UVES;
            if (maxTypeCodes >= 3) TypeCodes[2] = FLAMES_FIBRE_TYPE_UVES_7;
            if (maxTypeCodes >= 4) TypeCodes[3] = FLAMES_FIBRE_TYPE_UVES_8;
            if (maxTypeCodes >= 5) TypeCodes[4] = FLAMES_FIBRE_TYPE_GUIDE;
           
        } else if (comboId == 2) {

            /*  'Medusa + UVES + calibration' has Medusa and all the UVES 
             *  fibres except for the 8th calibration fibre.
             */
           
            *numTypeCodes = 4;
            if (maxTypeCodes >= 1) TypeCodes[0] = FLAMES_FIBRE_TYPE_MEDUSA;
            if (maxTypeCodes >= 2) TypeCodes[1] = FLAMES_FIBRE_TYPE_UVES;
            if (maxTypeCodes >= 3) TypeCodes[2] = FLAMES_FIBRE_TYPE_UVES_7;
            if (maxTypeCodes >= 4) TypeCodes[3] = FLAMES_FIBRE_TYPE_GUIDE;
           
        } else if (comboId == 3) {

            /*  'Medusa + UVES 6 fibres has Medusa and the
             *  lower 6 UVES fibres
             */
           
            *numTypeCodes = 3;
            if (maxTypeCodes >= 1) TypeCodes[0] = FLAMES_FIBRE_TYPE_MEDUSA;
            if (maxTypeCodes >= 2) TypeCodes[1] = FLAMES_FIBRE_TYPE_UVES;
            if (maxTypeCodes >= 3) TypeCodes[2] = FLAMES_FIBRE_TYPE_GUIDE;
           
        } else if (comboId == 4) {

            /*  IFU includes both IFU and IFU sky fibres */
           
            *numTypeCodes = 3;
            if (maxTypeCodes >= 1) TypeCodes[0] = FLAMES_FIBRE_TYPE_IFU;
            if (maxTypeCodes >= 2) TypeCodes[1] = FLAMES_FIBRE_TYPE_IFU_SKY;
            if (maxTypeCodes >= 3) TypeCodes[2] = FLAMES_FIBRE_TYPE_GUIDE;
            
        } else if (comboId == 5) {

            /*  'IFU + UVES' has IFU and IFU sky and all the UVES fibres.
             */
           
            *numTypeCodes = 6;
            if (maxTypeCodes >= 1) TypeCodes[0] = FLAMES_FIBRE_TYPE_IFU;
            if (maxTypeCodes >= 2) TypeCodes[1] = FLAMES_FIBRE_TYPE_IFU_SKY;
            if (maxTypeCodes >= 3) TypeCodes[2] = FLAMES_FIBRE_TYPE_UVES;
            if (maxTypeCodes >= 4) TypeCodes[3] = FLAMES_FIBRE_TYPE_UVES_7;
            if (maxTypeCodes >= 5) TypeCodes[4] = FLAMES_FIBRE_TYPE_UVES_8;
            if (maxTypeCodes >= 6) TypeCodes[5] = FLAMES_FIBRE_TYPE_GUIDE;
           
        } else if (comboId == 6) {

            /*  'IFU + UVES + calibration' has IFU and IFU sky and all the 
             *  UVES fibres except for the 8th calibration fibre.
             */
           
            *numTypeCodes = 5;
            if (maxTypeCodes >= 1) TypeCodes[0] = FLAMES_FIBRE_TYPE_IFU;
            if (maxTypeCodes >= 2) TypeCodes[1] = FLAMES_FIBRE_TYPE_IFU_SKY;
            if (maxTypeCodes >= 3) TypeCodes[2] = FLAMES_FIBRE_TYPE_UVES;
            if (maxTypeCodes >= 4) TypeCodes[3] = FLAMES_FIBRE_TYPE_UVES_7;
            if (maxTypeCodes >= 5) TypeCodes[4] = FLAMES_FIBRE_TYPE_GUIDE;
           
        } else if (comboId == 7) {

            /*  'IFU + UVES 6 fibres has IFU and IFU sky and the 
             *  lower 6 UVES fibres 
             */
           
            *numTypeCodes = 4;
            if (maxTypeCodes >= 1) TypeCodes[0] = FLAMES_FIBRE_TYPE_IFU;
            if (maxTypeCodes >= 2) TypeCodes[1] = FLAMES_FIBRE_TYPE_IFU_SKY;
            if (maxTypeCodes >= 3) TypeCodes[2] = FLAMES_FIBRE_TYPE_UVES;
            if (maxTypeCodes >= 4) TypeCodes[3] = FLAMES_FIBRE_TYPE_GUIDE;
           
        } else if (comboId == 8) {

            /*  ARGUS just has ARGUS fibres */
           
            *numTypeCodes = 2;
            if (maxTypeCodes >= 1) TypeCodes[0] = FLAMES_FIBRE_TYPE_ARGUS;
            if (maxTypeCodes >= 2) TypeCodes[1] = FLAMES_FIBRE_TYPE_GUIDE;
           
        } else if (comboId == 9) {

            /*  'ARGUS + UVES' has ARGUS and all the UVES fibres.
             */
           
            *numTypeCodes = 5;
            if (maxTypeCodes >= 1) TypeCodes[0] = FLAMES_FIBRE_TYPE_ARGUS;
            if (maxTypeCodes >= 2) TypeCodes[1] = FLAMES_FIBRE_TYPE_UVES;
            if (maxTypeCodes >= 3) TypeCodes[2] = FLAMES_FIBRE_TYPE_UVES_7;
            if (maxTypeCodes >= 4) TypeCodes[3] = FLAMES_FIBRE_TYPE_UVES_8;
            if (maxTypeCodes >= 5) TypeCodes[4] = FLAMES_FIBRE_TYPE_GUIDE;
           
        } else if (comboId == 10) {

            /*  'ARGUS + UVES + calibration' has ARGUS and all the UVES fibres
             *  except for the 8th calibration fibre.
             */
           
            *numTypeCodes = 4;
            if (maxTypeCodes >= 1) TypeCodes[0] = FLAMES_FIBRE_TYPE_ARGUS;
            if (maxTypeCodes >= 2) TypeCodes[1] = FLAMES_FIBRE_TYPE_UVES;
            if (maxTypeCodes >= 3) TypeCodes[2] = FLAMES_FIBRE_TYPE_UVES_7;
            if (maxTypeCodes >= 4) TypeCodes[3] = FLAMES_FIBRE_TYPE_GUIDE;
           
        } else if (comboId == 11) {

            /*  'ARGUS + UVES 6 fibres has ARGUS and all the UVES lower 6 fibres
             */
           
            *numTypeCodes = 3;
            if (maxTypeCodes >= 1) TypeCodes[0] = FLAMES_FIBRE_TYPE_ARGUS;
            if (maxTypeCodes >= 2) TypeCodes[1] = FLAMES_FIBRE_TYPE_UVES;
            if (maxTypeCodes >= 3) TypeCodes[2] = FLAMES_FIBRE_TYPE_GUIDE;
           
        } else if (comboId == 12) {

            /*  'UVES - all 8 fibres' has all the UVES fibre types.
             */
           
            *numTypeCodes = 4;
            if (maxTypeCodes >= 1) TypeCodes[0] = FLAMES_FIBRE_TYPE_UVES;
            if (maxTypeCodes >= 2) TypeCodes[1] = FLAMES_FIBRE_TYPE_GUIDE;
            if (maxTypeCodes >= 3) TypeCodes[2] = FLAMES_FIBRE_TYPE_UVES_7;
            if (maxTypeCodes >= 4) TypeCodes[3] = FLAMES_FIBRE_TYPE_UVES_8;
           
        } else if (comboId == 13) {

            /*  'UVES - 7 fibres + calibration' has two of the UVES fibre
             *  types - the lower six and the 7th.
             */
           
            *numTypeCodes = 3;
            if (maxTypeCodes >= 1) TypeCodes[0] = FLAMES_FIBRE_TYPE_UVES;
            if (maxTypeCodes >= 2) TypeCodes[1] = FLAMES_FIBRE_TYPE_GUIDE;
            if (maxTypeCodes >= 3) TypeCodes[2] = FLAMES_FIBRE_TYPE_UVES_7;
           
        } else if (comboId == 14) {

            /*  'UVES - 6 fibres' only uses the lower 6 fibres.
             */
           
            *numTypeCodes = 2;
            if (maxTypeCodes >= 1) TypeCodes[0] = FLAMES_FIBRE_TYPE_UVES;
            if (maxTypeCodes >= 2) TypeCodes[1] = FLAMES_FIBRE_TYPE_GUIDE;
           
        } else {
        
            /*  This is an invalid code */
            
            *numTypeCodes = 0;
        }
        
    } else {
    
        /*  6dF and 2dF distinguish between spectrograph and guide fibres,
         *  and 2dF supports selection of the two spectrographs.  In both
         *  cases, the codes used are easy - 0 is guide fibres, and the
         *  spectrographs are type codes 1 and 2 (6dF only has type 1).
         */
         
        short is2dF;
        
        is2dF = !strcmp(FpilGetInstName(instrument),"2dF");
        
        if (is2dF) {
        
            if (comboId == 0) {
        
                /*  All fibres */
           
                *numTypeCodes = 3;
                if (maxTypeCodes >= 1) TypeCodes[0] = 0;
                if (maxTypeCodes >= 2) TypeCodes[1] = 1;
                if (maxTypeCodes >= 3) TypeCodes[2] = 2;
           
            } else if (comboId == 1) {

                /*  Just guide fibres */
           
                *numTypeCodes = 1;
                if (maxTypeCodes >= 1) TypeCodes[0] = 0;
            
            } else if (comboId == 2) {

                /*  Both spectrographs */
           
                *numTypeCodes = 2;
                if (maxTypeCodes >= 1) TypeCodes[0] = 1;
                if (maxTypeCodes >= 2) TypeCodes[1] = 2;
           
            } else if (comboId == 3) {

                /*  Spectrograph 1 only */
                
                *numTypeCodes = 1;
                if (maxTypeCodes >= 1) TypeCodes[0] = 1;
           
            } else if (comboId == 4) {

                /*  Spectrograph 2 only */
                
                *numTypeCodes = 1;
                if (maxTypeCodes >= 1) TypeCodes[0] = 2;
                
            } else if (comboId == 5) {

                /*  Spectrograph 1 + guide */
                
                *numTypeCodes = 2;
                if (maxTypeCodes >= 1) TypeCodes[0] = 0;
                if (maxTypeCodes >= 2) TypeCodes[1] = 1;
                
            } else if (comboId == 6) {

                /*  Spectrograph 2 + guide */
                
                *numTypeCodes = 2;
                if (maxTypeCodes >= 1) TypeCodes[0] = 0;
                if (maxTypeCodes >= 2) TypeCodes[1] = 2;
                
            } else {
            
                /*  Invalid code */
                
                *numTypeCodes = 0;
            }
            
        } else {
        
            /*  This is the 6dF case */
            
            if (comboId == 0) {
        
                /*  All fibres */
           
                *numTypeCodes = 2;
                if (maxTypeCodes >= 1) TypeCodes[0] = 0;
                if (maxTypeCodes >= 2) TypeCodes[1] = 1;
           
            } else if (comboId == 1) {

                /*  Just guide fibres */
           
                *numTypeCodes = 1;
                if (maxTypeCodes >= 1) TypeCodes[0] = 0;
            
            } else if (comboId == 2) {

                /*  Just the spectrograph */
           
                *numTypeCodes = 1;
                if (maxTypeCodes >= 1) TypeCodes[0] = 1;
           
            } else {
            
                /*  Invalid code */
                
                *numTypeCodes = 0;
            }
        }
    }
    
}


/*  ------------------------------------------------------------------------  */

/*
 *+			   F p i l  S e t  P o s i t i o n  A n g l e

 * Function name:
      FpilSetPositionAngle

 * Function:
       Sets the position angle for an instrument.

 * Description:
       This routine is passed the address of the Model parameters used
       by the Fpil conversion routines, and - if appropriate for the
       instrument - modifies these for the specified position angle.

       Invokes the instrument specific virtual existing fibre type routine
       using the specified arguments.

 * Language:
      C

 * Call:
      FpilSetPositionAngle (inst, useDefault, posangle, params)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) useDefault      (short)     True if the default value is to be used.
      (>) posangle        (double)    The position angle, in radians.
      (!) params          (double*)   The model parameters.
 
 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     13-Aug-2001 -  KS - Original version
 */
 
extern void FpilSetPositionAngle (
    const FpilType instrument,
    short useDefault,
    double posangle,
    double *params)
{
    int isFlames;
    
    /*  Only FLAMES supports specification of the position angle, and
     *  does so by setting the first of the model parameters to the 
     *  position angle in radians.
     */
    
    isFlames = !strcmp(FpilGetInstName(instrument),"FLAMES");
    if (isFlames) {
       if (useDefault) {
          params[0] = FLAMES_YAXIS_POS;
       } else {
          params[0] = posangle;
       }
    }
}

/*  ------------------------------------------------------------------------  */

/*  Static variables used by the wavelength combination routines:
 *  Modified: 7-Nov-2002 - KS - Guide wavelength was .55, now .73
 *           12-Nov-2002 - KS - 2dF guide wavelength .60.
 */
 
#define WAVE 0.60
#define WAVE_FLAMES_GUIDE 0.73
#define WAVE_GUIDE 0.60
#define WAVE_TEL_GUIDE 0.65

static int gFibreComboId = 0;
static int gWavelengthComboId = 0;
static double gObsWavelength = WAVE;    /* Wavelength in microns for main 
                                         * target objects (non-guide objects) */
static double gTelGuideWavelength =     /* Wavelength in microns for */ 
                       WAVE_TEL_GUIDE;  /* telescope guide targets */
                       
/*  ------------------------------------------------------------------------  */

/*
 *+        F p i l  W a v e l e n g t h  C o m b o s

 * Function name:
      FpilWavelengthCombos

 * Function:
       Gets the number of wavelength combinations for an instrument.

 * Description:
       For some purposes, we need to select a wavelength combination for
       and instrument. This allows us to check a configuration at a
       variety of wavelengths for different fibre types - for example,
       for FLAMES we can check with Medusa fibres at one wavelength 
       extreme and UVES fibres at a different extreme.

       Invokes the instrument specific wavelength combination type routine
       using the specified arguments.
       
 * Language:
      C

 * Call:
      FpilWavelengthCombos (inst, maxCombos, numCombos, comboNames)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst         (FpilType)      The instrument as returned 
                                       by FpilInit(3).
      (>) maxCombos    (int)           The maximum number ofcombinations that
                                       the caller can support - ie the
                                       dimension of the comboNames array.
      (<) numCombos    (int*)          The actual number of combinations that
                                       the system defines for this instrument.
      (<) comboNames   (char*[])       Each element of this array is set to a
                                       pointer to a string describing one of
                                       the possible combinations.
 
 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable. Also,
      since the number of wavelength combinations depends on the current
      fibre combination, this must have been set using FpilSetFibreCombo().

 * Support: Keith Shortridge, AAO

 *-

 * History:
     14-Mar-2002 -  KS - Original version
     18-Sep-2002 -  KS - Fixed bug in 2dF/6dF code.
 */
 
extern void FpilWavelengthCombos (
    const FpilType instrument,
    int maxCombos,
    int *numCombos,
    char* comboNames[])
{
    short isFlames;
    int fibreComboId = gFibreComboId;
    static char CurrentWavelengthString[64];
    
    sprintf (CurrentWavelengthString,"%f nm",gObsWavelength * 1000.0);
    
    /*  See which instrument we are using */
    
    isFlames = !strcmp(FpilGetInstName(instrument),"FLAMES");

    if (isFlames) {
    
        /*  The combinations allowed by FLAMES depend on the fibre
         *  combination in use. If this is a combination that includes
         *  UVES and some other fibre, then we have to support more
         *  wavelength combinations. This code has to be changed if changes
         *  are made to the fibre combination routines. Fibre combo Ids
         *  greater than 6 are all mixed wavelength combinations. If we
         *  don't know the fibre combination, we assume a mixed combo.
         */
        
        short MixedCombo = FALSE;
          
        if (fibreComboId >= 6) MixedCombo = TRUE;
        
        if (MixedCombo) {
        
           *numCombos = 7;
           if (maxCombos >= 1) {
               comboNames[0] = CurrentWavelengthString;
           }
           if (maxCombos >= 2) {
               comboNames[1] = "300nm + UVES=520nm";
           }
           if (maxCombos >= 3) {
               comboNames[2] = "300nm + UVES=580nm";
           }
           if (maxCombos >= 4) {
               comboNames[3] = "300nm + UVES=860nm";
           }
           if (maxCombos >= 5) {
               comboNames[4] = "1000nm + UVES=520nm";
           }
           if (maxCombos >= 6) {
               comboNames[5] = "1000nm + UVES=580nm";
           }
           if (maxCombos >= 7) {
               comboNames[6] = "1000nm + UVES=860nm";
           }
           
        } else {
        
           /*  If the fibre combination is a pure UVES combination,
            *  the three UVES wavelengths should be tried. Otherwise,
            *  the two limiting wavelengths should be tried.
            *  Fibre combination Ids between 3 and 5 are the pure
            *  UVES combinations.
            */
           
           if (fibreComboId >= 3) {
           
              *numCombos = 4;
              if (maxCombos >= 1) {
                  comboNames[0] = CurrentWavelengthString;
              }
              if (maxCombos >= 2) {
                  comboNames[1] = "520nm";
              }
              if (maxCombos >= 3) {
                  comboNames[2] = "580nm";
              }
              if (maxCombos >= 4) {
                  comboNames[3] = "860nm";
              }
             
           } else {
           
              *numCombos = 3;
              if (maxCombos >= 1) {
                 comboNames[0] = CurrentWavelengthString;
              }
              if (maxCombos >= 2) {
                 comboNames[1] = "300nm";
              }
              if (maxCombos >= 3) {
                 comboNames[2] = "1000nm";
              }
           }
        }
           
    } else {
    
        /*  6dF and 2dF only need to consider the maximum and minimum
         *  wavelengths.
         */
    
        *numCombos = 3;
        if (maxCombos >= 1) {
            comboNames[0] = CurrentWavelengthString;
        }
        if (maxCombos >= 2) {
            comboNames[1] = "300nm";
        }
        if (maxCombos >= 3) {
            comboNames[2] = "1000nm";
        }
    }
}

/*  ------------------------------------------------------------------------  */

/*
 *+        F p i l  S e t  W a v e l e n g t h  C o m b o 

 * Function name:
      FpilSetWavelengthCombo

 * Function:
       Sets the current wavelength combination id.

 * Description:
       For some purposes, we need to select a wavelength combination for
       an instrument. This routine allows us to select one of these
       wavelength combinations, and this will be used when 
       FpilGetFibreWavelength() is called to determine the wavelength to
       use for a given fibre type.

       Invokes the instrument specific set wavelength combination routine
       using the specified arguments.
       
 * Language:
      C

 * Call:
      FpilSetWavelengthCombo (inst, comboId)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst         (FpilType)      The instrument as returned 
                                       by FpilInit(3).
      (>) comboId (int)                The current wavelength combination -
                                       should be a number between 0 and the
                                       maximum number returned by
                                       FpilWavelengthCombos() - 1. Note that
                                       combination 0 always refers to the
                                       actual observed wavelength specified
                                       by FpilSetObsWavelength().
 
 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     14-Mar-2002 -  KS - Original version
 */
  
extern void FpilSetWavelengthCombo (
    const FpilType instrument,
    int comboId)
{
    gWavelengthComboId = comboId;
}

/*  ------------------------------------------------------------------------  */

/*
 *+        F p i l  S e t  F i b r e  C o m b o 

 * Function name:
      FpilSetFibreCombo

 * Function:
       Sets the current fibre combination id.

 * Description:
       For some purposes, we need to select a fibre combination for
       an instrument. This routine allows us to select one of these
       fibre combinations, and this will be used when 
       FpilGetFibreWavelength() is called to determine the wavelength to
       use for a given fibre type.

       Invokes the instrument specific set fibre combination routine
       using the specified arguments.
       
 * Language:
      C

 * Call:
      FpilSetFibreCombo (inst, comboId)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst         (FpilType)      The instrument as returned 
                                       by FpilInit(3).
      (>) fibreComboId (int)           The current fibre combination - the Id
                                       number for the current fibre combo as
                                       used by FpilComboFibreCodes(). If
                                       unknown, should be passed as -1.
 
 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     14-Mar-2002 -  KS - Original version
 */
 
 
extern void FpilSetFibreCombo (
    const FpilType instrument,
    int fibreComboId)
{
    gFibreComboId = fibreComboId;
}

/*  ------------------------------------------------------------------------  */

/*
 *+        F p i l  S e t  O b s  W a v e l e n g t h 

 * Function name:
      FpilSetObsWavelength

 * Function:
       Sets the current observing wavelength.

 * Description:
       For some purposes, we need to select a wavlength combination for
       and instrument. Some of these combinations make use of the 'current'
       observing wavelength. This routine allows this wavelength to
       be set, and this will be used when FpilGetFibreWavelength() is 
       called to determine the wavelength to use for a given fibre type.

       Invokes the instrument specific set obs wavelength routine
       using the specified arguments.
       
 * Language:
      C

 * Call:
      FpilSetObsWavelength (inst, wavelength)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst         (FpilType)      The instrument as returned 
                                       by FpilInit(3).
      (>) wavelength   (double)        The current selected observing
                                       wavelength, in microns.
 
 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     14-Mar-2002 -  KS - Original version
 */
 
 
extern void FpilSetObsWavelength (
    const FpilType instrument,
    double wavelength)
{
    gObsWavelength = wavelength;
}

/*  ------------------------------------------------------------------------  */

/*
 *+        F p i l  G e t  T a r g e t  W a v e l e n g t h 

 * Function name:
      FpilGetTargetWavelength

 * Function:
       Gets the wavelength for a given target type and wavelength combination.

 * Description:
       For some purposes, we need to select a wavelength combination for
       and instrument. If we are using such a combination, this routine
       returns the wavelength to be used for a fibre of the specified type.
       Note that the wavelength combination used is that set by 
       FpilSetWavelengthCombo(), and is not passed to this routine - it
       was too awkward to get it passed, unfortunately.
       
       Invokes the instrument specific get fibre wavelength routine
       using the specified arguments.
       
 * Language:
      C

 * Call:
      FpilGetTargetWavelength (inst, targetType)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst         (FpilType)      The instrument as returned 
                                       by FpilInit(3).
      (>) targetType   (char)          The single character describing the
                                       target type as set by the routine
                                       FpilEncodeTargetType().
 
 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     14-Mar-2002 -  KS - Original version
     12-Nov-2002 -  KS - Distinguishes between instruments for guide stars.
 */
 
extern double FpilGetFibreWavelength (
    const FpilType instrument,
    char targetType)
{
    short isFlames;
    double Wavelength = 0.0;
    
    targetType &= ~DISABLED_TARGET_BIT;
       
    /*  See which instrument we are using */
    
    isFlames = !strcmp(FpilGetInstName(instrument),"FLAMES");
    
    /*  Some cases we can deal with quickly. Guide fibres and telescope
     *  guide targets are always at the standard wavelengths. However,
     *  the guide fibre wavelength is different for FLAMES and 2dF/6dF.
     *  Only FLAMES has telescope guide targets.
     */
    
    Wavelength = 0.0;
    if (FpilIsTargetGuide(instrument,targetType)) {
       if (FpilIsTargetTelGuide(instrument,targetType)) {
          Wavelength = gTelGuideWavelength;
       } else {
          if (isFlames) {
              Wavelength = WAVE_FLAMES_GUIDE;
          } else {
              Wavelength = WAVE_GUIDE;
          }
       }
    }
    
    if (Wavelength <= 0.0) {
    
       if (isFlames) {
       
          /*  For FLAMES, we have to distinguish between UVES and
           *  non-UVES targets.
           */
          
          short MixedCombo = FALSE;
          
          short IsUVES = (targetType == 'U');
           
          if (gFibreComboId >= 6) MixedCombo = TRUE;
        
          if (MixedCombo) {
        
             if (gWavelengthComboId == 0) {
                Wavelength = gObsWavelength;
             }
             if (gWavelengthComboId == 1) {
                if (IsUVES) Wavelength = 0.520;
                else Wavelength = 0.300;
             }
             if (gWavelengthComboId == 2) {
                if (IsUVES) Wavelength = 0.580;
                else Wavelength = 0.300;
             }
             if (gWavelengthComboId == 3) {
                if (IsUVES) Wavelength = 0.860;
                else Wavelength = 0.300;
             }
             if (gWavelengthComboId == 4) {
                if (IsUVES) Wavelength = 0.520;
                else Wavelength = 1.000;
             }
             if (gWavelengthComboId == 5) {
                if (IsUVES) Wavelength = 0.580;
                else Wavelength = 1.000;
             }
             if (gWavelengthComboId == 6) {
                if (IsUVES) Wavelength = 0.860;
                else Wavelength = 1.000;
             }
           
          } else {
        
             /*  If the fibre combination is a pure UVES combination,
              *  the three UVES wavelengths should be tried. Otherwise,
              *  the two limiting wavelengths should be tried.
              *  Fibre combination Ids between 3 and 5 are the pure
              *  UVES combinations.
              */
           
             if (gFibreComboId >= 3) {
           
                if (gWavelengthComboId == 0) {
                   Wavelength = gObsWavelength;
                }
                if (gWavelengthComboId == 1) {
                   Wavelength = 0.520;
                }
                if (gWavelengthComboId == 2) {
                   Wavelength = 0.580;
                }
                if (gWavelengthComboId == 3) {
                   Wavelength = 0.860;
                }
             
             } else {
           
                if (gWavelengthComboId == 0) {
                   Wavelength = gObsWavelength;
                }
                if (gWavelengthComboId == 1) {
                   Wavelength = 0.300;
                }
                if (gWavelengthComboId == 2) {
                   Wavelength = 1.000;
                }
             }
          }
          
       } else {
       
          /*  For 6dF and 2dF, we only need to consider the observing
           *  or the two limit wavelengths, no matter what the target type.
           */
           
          if (gWavelengthComboId == 0) {
             Wavelength = gObsWavelength;
          }
          if (gWavelengthComboId == 1) {
             Wavelength = 0.300;
          }
          if (gWavelengthComboId == 2) {
             Wavelength = 1.000;
          }
       }
    }
    return Wavelength;
}

/*  ------------------------------------------------------------------------  */

/*
 *+        F p i l  G e t  P o i n t i n g  W a v e l e n g t h 

 * Function name:
      FpilGetPointingWavelength

 * Function:
       Gets the wavelength used to point the telescope.

 * Description:
       For some purposes, we need to select a wavelength combination for
       and instrument.  FpilGetFibreWavelength() returns the wavelength
       used for a specific target type, but this routine returns the
       wavelength 'used to point the telescope' - this terminology comes
       from an e-mail from Tony Farrell, but the important thing is that
       it is the value that has to be passed as the first wavelength
       parameter in a call to FpilModelCvtInit().
       
       Invokes the instrument specific get fibre wavelength routine
       using the specified arguments.
       
 * Language:
      C

 * Call:
      FpilGetPointingWavelength (inst)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst         (FpilType)      The instrument as returned 
                                       by FpilInit(3).
 
 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     17-Oct-2002 -  KS - Original version
      7-Nov-2002 -  KS - Distinction is not between combined modes and
                        single instrument, but between UVES and non-UVES.
 */
 
extern double FpilGetPointingWavelength (
    const FpilType instrument)
{
    short isFlames;
    double Wavelength = 0.0;
        
    /*  See which instrument we are using */
    
    isFlames = !strcmp(FpilGetInstName(instrument),"FLAMES");

    if (isFlames) {
       
       /*  For FLAMES, we have to distinguish between pure UVES and
        *  combined modes.
        */

       short UVESOnly = FALSE;

       if ((gFibreComboId >= 3) && (gFibreComboId <= 5)) UVESOnly = TRUE;

       if (UVESOnly) {
       
          /*  If the fibre combination is a pure UVES combination,
           *  the wavelength should be a UVES wavelength. We can't
           *  actually be sure that the main observing wavelength
           *  will have been set properly for UVES, and in any case
           *  we're supposed to be able to handle any actual wavelength,
           *  so we use, slightly arbitrarily, the .580 micron value
           *  unless the main wavelength has explicitly been set to
           *  one of the other UVES wavelengths, in which case we
           *  assume this is the one to use.
           */

          if ((gObsWavelength >= 0.5199) && (gObsWavelength <= 0.5201)) {
             Wavelength = gObsWavelength;
          } else if ((gObsWavelength >= 0.8599) && (gObsWavelength <= 0.8601)) {
             Wavelength = gObsWavelength;
          } else {
             Wavelength = 0.580;
          }
          
      } else {
      
          /*  In a combined or Giraffe mode, the pointing wavelength is
           *  the main, ie the Giraffe, wavelength.
           */

          Wavelength = gObsWavelength;
      }
          
    } else {

       /*  For 6dF and 2dF, we always use the current wavelength.
        */

       Wavelength = gObsWavelength;
       
    }
    
    return Wavelength;
}

/*  ------------------------------------------------------------------------  */

/*
 *+            F p i l  G e t  F i b r e  L e n g t h 

 * Function name:
      FpilGetFibreLength

 * Function:
       Gets the maximum length for a given fibre type.

 * Description:
       FpilGetMaxExtension() returns the maximum extension for any
       fibre for a given instrument, but some instruments have
       different maximum extensions for different fibre types. This
       routine returns the maximum extension given a fibre type code.
       
       Invokes the instrument specific get fibre wavelength routine
       using the specified arguments.
       
 * Language:
      C

 * Call:
      FpilGetFibreLength (inst, fibreType)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst         (FpilType)      The instrument as returned 
                                       by FpilInit(3).
      (>) fibreType    (int)           The type code for the fibre type
                                       in question.
  
 * Returns: The maximum extension for fibres of this type.
 
 * Include files: fpil.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Keith Shortridge, AAO

 *-

 * History:
     05-Nov-2002 -  KS - Original version
 */

#include "fpcolInstDescr.h"

extern unsigned long FpilGetFibreLength (
    const FpilType instrument,
    int fibreType)
{
    short isFlames;
    unsigned long FibreLength;
    
    /*  In almost all cases, we return the same value as that given
     *  by FpilGetMaxExtension().
     */
     
    FibreLength = FpilGetMaxExtension (instrument);
        
    /*  See which instrument we are using */
    
    isFlames = !strcmp(FpilGetInstName(instrument),"FLAMES");

    if (isFlames) {
       
        /*  For FLAMES, fibre bundles - FACBs and IFUs have a different
         *  length.
         */
        
        if ((fibreType == FLAMES_FIBRE_TYPE_GUIDE) ||
             (fibreType == FLAMES_FIBRE_TYPE_IFU) ||
                (fibreType == FLAMES_FIBRE_TYPE_IFU_SKY)) {
            FibreLength = fpcolINST_BUNDLE_EXT;
        }
        
    }
        
    return FibreLength;
}

/*
         1         2         3         4         5         6         7         8
12345678901234567890123456789012345678901234567890123456789012345678901234567890
*/
