module cubedag_link_type
  use cubelist_types
  use cubedag_parameters
  use cubedag_object
  use cubedag_messaging
  !
  ! The cubedag_link_t is a collection of list_object_t, but an extended
  ! version which provides a get_id() method.
  type, extends(list_t) :: cubedag_link_t
    ! No more components. Avoid adding arrays of size %n otherwise you
    ! will have to provide specific (de)allocation, copy, etc routines
  contains
    procedure, public  :: repr     => cubedag_link_repr
    procedure, public  :: entrynum => cubedag_link_entrynum
    procedure, public  :: object   => cubedag_link_object_ptr
    procedure, public  :: resolve  => cubedag_link_resolve
    procedure, public  :: write    => cubedag_link_write
    procedure, public  :: read     => cubedag_link_read
  end type cubedag_link_t

  integer(kind=entr_k), parameter :: root_id=0
  character(len=*), parameter :: form_lk='(A,T26,I20,1X,A)'  ! Link_t

  public :: cubedag_link_t,dag_object_t
  private

contains

  subroutine cubedag_link_repr(link,prefix,str)
    !-------------------------------------------------------------------
    ! Create a one-line representation of the list
    !-------------------------------------------------------------------
    class(cubedag_link_t), intent(in)    :: link
    character(len=*),      intent(in)    :: prefix
    character(len=*),      intent(inout) :: str
    ! Local
    character(len=*), parameter :: rname='LINK>REPR'
    integer(kind=entr_k) :: jent
    integer(kind=4) :: nc,mlen
    character(len=10) :: tmp
    class(list_object_t), pointer :: tot
    !
    str = prefix
    nc = len_trim(prefix)
    mlen= len(str)
    if (link%n.le.0) then
      write(str(nc+1:),'(A6)')  '<none>'
    else
      do jent=1,link%n
        tot => link%list(jent)%p
        select type (tot)
        class is (dag_object_t)
          write(tmp,'(I0,A1)')  tot%get_id(),','
        class default
          call cubedag_message(seve%e,rname,'Internal error: object has wrong class')
        end select
        str = str(1:nc)//tmp
        nc = len_trim(str)
        if (nc.eq.mlen) then  ! List too long, string exhausted
          str(nc-1:nc) = '..'
          exit
        elseif (jent.eq.link%n) then  ! Last element, strip off trailing coma
          str(nc:nc) = ' '
        endif
      enddo
    endif
  end subroutine cubedag_link_repr
  !
  subroutine cubedag_link_entrynum(optx,id,found_id,found_num,error)
    !-------------------------------------------------------------------
    ! Resolve the entry number corresponding to the given id by looking
    ! in the given index. This resolution is based on two strong
    ! assumptions:
    !  1) the identifier is UNIQUE,
    !  2) the identifier list is SORTED
    ! No error is raised if ID is not found in the index. Return the
    ! nearest found instead (i.e. the requested ID is found if
    ! ID.eq.FOUND_ID). Let the caller decide if it is an error or not.
    !-------------------------------------------------------------------
    class(cubedag_link_t), intent(in)    :: optx
    integer(kind=iden_l),  intent(in)    :: id
    integer(kind=iden_l),  intent(out)   :: found_id
    integer(kind=entr_k),  intent(out)   :: found_num
    logical,               intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='LINK>ENTRYNUM'
    integer(kind=entr_k) :: inf,mid,sup
    class(dag_object_t), pointer :: dot
    !
    ! Sanity
    found_id = 0
    found_num = 0
    if (optx%n.le.0)  return
    !
    ! Dichotomic search
    dot => cubedag_object_ptr(optx%list(1)%p,error)
    if (error)  return
    if (dot%get_id().eq.id) then
      found_id = id
      found_num = 1
      return
    endif
    !
    inf = 1
    sup = optx%n
    do while (sup.gt.inf+1)
      mid = (inf+sup)/2  ! Integer division
      dot => cubedag_object_ptr(optx%list(mid)%p,error)
      if (error)  return
      if (dot%get_id().lt.id) then
        inf = mid
      else
        sup = mid
      endif
    enddo
    !
    dot => cubedag_object_ptr(optx%list(sup)%p,error)
    if (error)  return
    found_id = dot%get_id()  ! Success if found_id.eq.id
    found_num = sup
  end subroutine cubedag_link_entrynum
  !
  function cubedag_link_object_ptr(link,id,error)
    !-------------------------------------------------------------------
    ! Search for the object #id in the input list and return a pointer
    ! to it
    !-------------------------------------------------------------------
    class(dag_object_t), pointer :: cubedag_link_object_ptr ! Associated on return
    class(cubedag_link_t), intent(in)    :: link
    integer(kind=iden_l),  intent(in)    :: id
    logical,               intent(inout) :: error
    !
    integer(kind=iden_l) :: found_id
    integer(kind=entr_k) :: ient
    class(dag_object_t), pointer :: tmp
    character(len=*), parameter :: rname='LINK>OBJECT>PTR'
    !
    call link%entrynum(id,found_id,ient,error)
    if (error)  return
    if (found_id.ne.id) then
      call cubedag_message(seve%e,rname,'No such identifier in DAG')
      error = .true.
      return
    endif
    !
    ! Note: spurious crash with gfortran 14.1.0 need this two-steps
    ! association...
    tmp => cubedag_object_ptr(link%list(ient)%p,error)
    if (error)  return
    cubedag_link_object_ptr => tmp
  end function cubedag_link_object_ptr
  !
  subroutine cubedag_link_resolve(link,optx,ids,error)
    !-------------------------------------------------------------------
    ! Resolve the cross-links (from IDs to pointer) for the given list
    ! of IDs. Ids are searched in the input index (optx) and their
    ! associated objects are saved in the 'link' list.
    !-------------------------------------------------------------------
    class(cubedag_link_t), intent(inout) :: link
    type(cubedag_link_t),  intent(in)    :: optx
    integer(kind=iden_l),  intent(in)    :: ids(:)
    logical,               intent(inout) :: error
    !
    integer(kind=list_k) :: il
    class(dag_object_t), pointer :: dot
    !
    do il=1,link%n
      dot => optx%object(ids(il),error)
      if (error)  return
      link%list(il)%p => dot
    enddo
  end subroutine cubedag_link_resolve
  !
  subroutine cubedag_link_write(link,lun,name,error)
    use gkernel_interfaces
    !-------------------------------------------------------------------
    ! Write the cubedag_link_t to output file
    !-------------------------------------------------------------------
    class(cubedag_link_t), intent(in)    :: link
    integer(kind=4),       intent(in)    :: lun
    character(len=*),      intent(in)    :: name
    logical,               intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='LINK>WRITE'
    integer(kind=entr_k) :: il
    integer(kind=4) :: ic,nc,ier
    character(len=:), allocatable :: buf,tmp
    character(len=mess_l) :: mess
    class(list_object_t), pointer :: tot
    !
    if (link%n.le.0) then
      write(lun,form_lk) name,link%n
    else
      ic = 0
      allocate(character(100)::buf,stat=ier)
      if (failed_allocate(rname,'char buffer',ier,error)) return
      do il=1,link%n
        if (len(buf).lt.ic+21) then
          tmp = buf(1:ic)  ! Implicit (re)allocation
          deallocate(buf)
          allocate(character(2*ic)::buf,stat=ier)
          if (failed_allocate(rname,'char buffer',ier,error)) return
          buf(1:ic) = tmp
        endif
        if (.not.associated(link%list(il)%p)) then  ! Sanity check
          write(mess,'(3(A,I0))')  &
            'Internal error: pointer to node #',il,'/',link%n,' is not associated'
          call cubedag_message(seve%e,rname,mess)
          error = .true.
          return
        endif
        tot => link%list(il)%p
        select type (tot)
        class is (dag_object_t)
          write(buf(ic+1:ic+20),'(I0,A1)')  tot%get_id(),' '
        class default
          call cubedag_message(seve%e,rname,'Internal error: object has wrong class')
        end select
        nc = len_trim(buf(ic+1:ic+20))+1
        ic = ic+nc
      enddo
      write(lun,form_lk) name,link%n,buf(1:ic)
    endif
    !
  end subroutine cubedag_link_write

  subroutine cubedag_link_read(link,lun,nshift,ids,error)
    use gkernel_interfaces
    !-------------------------------------------------------------------
    ! Read the cubedag_link_t from input file
    !-------------------------------------------------------------------
    class(cubedag_link_t),             intent(inout) :: link
    integer(kind=4),                   intent(in)    :: lun
    integer(kind=iden_l),              intent(in)    :: nshift  ! Node ID shift
    integer(kind=iden_l), allocatable, intent(out)   :: ids(:)
    logical,                           intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='LINK>READ'
    character(len=12) :: key
    character(len=:), allocatable :: buf
    integer(kind=entr_k) :: nl,il
    integer(kind=4) :: i1,i2,nc,ier
    !
    read(lun,form_lk) key,nl
    if (nl.gt.0) then
      ! Try to read in a long-enough buffer
      nc = 32
      do
        allocate(character(nc)::buf,stat=ier)
        if (failed_allocate(rname,'char buffer',ier,error)) return
        backspace(lun)   ! Backspace in formatted file is not standard!
        read(lun,form_lk) key,nl,buf
        if (buf(nc-1:nc).eq.' ')  then
          ! 2 last chars are blank => ok, no number missed
          exit
        endif
        deallocate(buf)
        nc = 2*nc
      enddo
      !
      ! Now decode the line
      allocate(ids(nl),stat=ier)
      if (failed_allocate(rname,'ids',ier,error)) return
      il = 0
      i1 = 1
      i2 = 1
      do while (il.lt.nl)
        if (buf(i2+1:i2+1).eq.' ') then
          il = il+1
          read(buf(i1:i2),*)  ids(il)
          if (ids(il).ne.root_id) then
            ! Root id is never renumbered (always shared between all DAGs)
            ids(il) = ids(il) + nshift
          endif
          i1 = i2+2
          i2 = i1
        else
          i2 = i2+1
        endif
      enddo
    endif
    !
    ! Reallocate the link list so that it is ready to store its objects
    ! during resolution of ids
    if (nl.gt.0) then
      call link%reallocate(nl,error)
      if (error)  return
    endif
    link%n = nl
  end subroutine cubedag_link_read

end module cubedag_link_type
