package action

import (
	"strings"

	"github.com/carapace-sh/carapace"
	"github.com/carapace-sh/carapace-bin/pkg/actions/net/http"
	"github.com/carapace-sh/carapace-bin/pkg/actions/tools/aws"
	"github.com/carapace-sh/carapace-bin/pkg/actions/tools/docker"
	"github.com/carapace-sh/carapace-bin/pkg/actions/tools/jaeger"
	"github.com/carapace-sh/carapace/pkg/style"
)

func ActionParameters() carapace.Action {
	return carapace.ActionCallback(func(c carapace.Context) carapace.Action {
		if len(c.Args) > 0 {
			if a, ok := flagValues(c.Args[len(c.Args)-1]); ok {
				return a
			}
		}

		if len(c.Args) == 0 {
			return carapace.Batch(
				actionFlagNames(),
				carapace.ActionValuesDescribed(
					"--help", "show help",
					"healthcheck", "execute healthcheck",
					"version", "show version",
				),
			).Invoke(c).Merge().ToA()
		}
		return actionFlagNames()
	})
}

func actionFlagNames() carapace.Action {
	return carapace.ActionCallback(func(c carapace.Context) carapace.Action {
		flags := map[string]string{
			"--accesslog":                                      "Access log settings.",
			"--accesslog.bufferingsize":                        "Number of access log lines to process in a buffered way.",
			"--accesslog.fields.defaultmode":                   "Default mode for fields: keep | drop",
			"--accesslog.fields.headers.defaultmode":           "Default mode for fields: keep | drop | redact",
			"--accesslog.fields.headers.names.<name>":          "Override mode for headers",
			"--accesslog.fields.names.<name>":                  "Override mode for fields",
			"--accesslog.filepath":                             "Access log file path. Stdout is used when omitted or empty.",
			"--accesslog.filters.minduration":                  "Keep access logs when request took longer than the specified duration.",
			"--accesslog.filters.retryattempts":                "Keep access logs when at least one retry happened.",
			"--accesslog.filters.statuscodes":                  "Keep access logs with status codes in the specified range.",
			"--accesslog.format":                               "Access log format: json | common",
			"--api":                                            "Enable api/dashboard.",
			"--api.dashboard":                                  "Activate dashboard.",
			"--api.debug":                                      "Enable additional endpoints for debugging and profiling.",
			"--api.insecure":                                   "Activate API directly on the entryPoint named traefik.",
			"--certificatesresolvers.<name>":                   "Certificates resolvers configuration.",
			"--certificatesresolvers.<name>.acme.caserver":     "CA server to use.",
			"--certificatesresolvers.<name>.acme.dnschallenge": "Activate DNS-01 Challenge.",
			"--certificatesresolvers.<name>.acme.dnschallenge.delaybeforecheck":        "Assume DNS propagates after a delay in seconds rather than finding and querying nameservers.",
			"--certificatesresolvers.<name>.acme.dnschallenge.disablepropagationcheck": "Disable the DNS propagation checks before notifying ACME that the DNS challenge is ready. [not recommended]",
			"--certificatesresolvers.<name>.acme.dnschallenge.provider":                "Use a DNS-01 based challenge provider rather than HTTPS.",
			"--certificatesresolvers.<name>.acme.dnschallenge.resolvers":               "Use following DNS servers to resolve the FQDN authority.",
			"--certificatesresolvers.<name>.acme.eab.hmacencoded":                      "Base64 encoded HMAC key from External CA.",
			"--certificatesresolvers.<name>.acme.eab.kid":                              "Key identifier from External CA.",
			"--certificatesresolvers.<name>.acme.email":                                "Email address used for registration.",
			"--certificatesresolvers.<name>.acme.httpchallenge":                        "Activate HTTP-01 Challenge.",
			"--certificatesresolvers.<name>.acme.httpchallenge.entrypoint":             "HTTP challenge EntryPoint",
			"--certificatesresolvers.<name>.acme.keytype":                              "KeyType used for generating certificate private key. Allow value 'EC256', 'EC384', 'RSA2048', 'RSA4096', 'RSA8192'.",
			"--certificatesresolvers.<name>.acme.preferredchain":                       "Preferred chain to use.",
			"--certificatesresolvers.<name>.acme.storage":                              "Storage to use.",
			"--certificatesresolvers.<name>.acme.tlschallenge":                         "Activate TLS-ALPN-01 Challenge.",
			"--configfile":                                                       "Configuration file to use. If specified all other flags are ignored.",
			"--entrypoints.<name>":                                               "Entry points definition.",
			"--entrypoints.<name>.address":                                       "Entry point address.",
			"--entrypoints.<name>.enablehttp3":                                   "Enable HTTP3.",
			"--entrypoints.<name>.forwardedheaders.insecure":                     "Trust all forwarded headers.",
			"--entrypoints.<name>.forwardedheaders.trustedips":                   "Trust only forwarded headers from selected IPs.",
			"--entrypoints.<name>.http":                                          "HTTP configuration.",
			"--entrypoints.<name>.http.middlewares":                              "Default middlewares for the routers linked to the entry point.",
			"--entrypoints.<name>.http.redirections.entrypoint.permanent":        "Applies a permanent redirection.",
			"--entrypoints.<name>.http.redirections.entrypoint.priority":         "Priority of the generated router.",
			"--entrypoints.<name>.http.redirections.entrypoint.scheme":           "Scheme used for the redirection.",
			"--entrypoints.<name>.http.redirections.entrypoint.to":               "Targeted entry point of the redirection.",
			"--entrypoints.<name>.http.tls":                                      "Default TLS configuration for the routers linked to the entry point.",
			"--entrypoints.<name>.http.tls.certresolver":                         "Default certificate resolver for the routers linked to the entry point.",
			"--entrypoints.<name>.http.tls.domains":                              "Default TLS domains for the routers linked to the entry point.",
			"--entrypoints.<name>.http.tls.domains[n].main":                      "Default subject name.",
			"--entrypoints.<name>.http.tls.domains[n].sans":                      "Subject alternative names.",
			"--entrypoints.<name>.http.tls.options":                              "Default TLS options for the routers linked to the entry point.",
			"--entrypoints.<name>.proxyprotocol":                                 "Proxy-Protocol configuration.",
			"--entrypoints.<name>.proxyprotocol.insecure":                        "Trust all.",
			"--entrypoints.<name>.proxyprotocol.trustedips":                      "Trust only selected IPs.",
			"--entrypoints.<name>.transport.lifecycle.gracetimeout":              "Duration to give active requests a chance to finish before Traefik stops.",
			"--entrypoints.<name>.transport.lifecycle.requestacceptgracetimeout": "Duration to keep accepting requests before Traefik initiates the graceful shutdown procedure.",
			"--entrypoints.<name>.transport.respondingtimeouts.idletimeout":      "IdleTimeout is the maximum amount duration an idle (keep-alive) connection will remain idle before closing itself. If zero, no timeout is set.",
			"--entrypoints.<name>.transport.respondingtimeouts.readtimeout":      "ReadTimeout is the maximum duration for reading the entire request, including the body. If zero, no timeout is set.",
			"--entrypoints.<name>.transport.respondingtimeouts.writetimeout":     "WriteTimeout is the maximum duration before timing out writes of the response. If zero, no timeout is set.",
			"--entrypoints.<name>.udp.timeout":                                   "Timeout defines how long to wait on an idle session before releasing the related resources.",
			"--experimental.http3":                                               "Enable HTTP3.",
			"--experimental.kubernetesgateway":                                   "Allow the Kubernetes gateway api provider usage.",
			"--experimental.localplugins.<name>":                                 "Local plugins configuration.",
			"--experimental.localplugins.<name>.modulename":                      "plugin's module name.",
			"--experimental.plugins.<name>":                                      "Plugins configuration.",
			"--experimental.plugins.<name>.modulename":                           "plugin's module name.",
			"--experimental.plugins.<name>.version":                              "plugin's version.",
			"--global.checknewversion":                                           "Periodically check if a new version has been released.",
			"--global.sendanonymoususage":                                        "Periodically send anonymous usage statistics. If the option is not specified, it will be enabled by default.",
			"--hostresolver":                                                     "Enable CNAME Flattening.",
			"--hostresolver.cnameflattening":                                     "A flag to enable/disable CNAME flattening",
			"--hostresolver.resolvconfig":                                        "resolv.conf used for DNS resolving",
			"--hostresolver.resolvdepth":                                         "The maximal depth of DNS recursive resolving",
			"--log":                                                              "Traefik log settings.",
			"--log.filepath":                                                     "Traefik log file path. Stdout is used when omitted or empty.",
			"--log.format":                                                       "Traefik log format: json | common",
			"--log.level":                                                        "Log level set to traefik logs.",
			"--metrics.datadog":                                                  "Datadog metrics exporter type.",
			"--metrics.datadog.addentrypointslabels":                             "Enable metrics on entry points.",
			"--metrics.datadog.address":                                          "Datadog's address.",
			"--metrics.datadog.addrouterslabels":                                 "Enable metrics on routers.",
			"--metrics.datadog.addserviceslabels":                                "Enable metrics on services.",
			"--metrics.datadog.pushinterval":                                     "Datadog push interval.",
			"--metrics.influxdb":                                                 "InfluxDB metrics exporter type.",
			"--metrics.influxdb.addentrypointslabels":                            "Enable metrics on entry points.",
			"--metrics.influxdb.address":                                         "InfluxDB address.",
			"--metrics.influxdb.addrouterslabels":                                "Enable metrics on routers.",
			"--metrics.influxdb.addserviceslabels":                               "Enable metrics on services.",
			"--metrics.influxdb.database":                                        "InfluxDB database used when protocol is http.",
			"--metrics.influxdb.password":                                        "InfluxDB password (only with http).",
			"--metrics.influxdb.protocol":                                        "InfluxDB address protocol (udp or http).",
			"--metrics.influxdb.pushinterval":                                    "InfluxDB push interval.",
			"--metrics.influxdb.retentionpolicy":                                 "InfluxDB retention policy used when protocol is http.",
			"--metrics.influxdb.username":                                        "InfluxDB username (only with http).",
			"--metrics.prometheus":                                               "Prometheus metrics exporter type.",
			"--metrics.prometheus.addentrypointslabels":                          "Enable metrics on entry points.",
			"--metrics.prometheus.addrouterslabels":                              "Enable metrics on routers.",
			"--metrics.prometheus.addserviceslabels":                             "Enable metrics on services.",
			"--metrics.prometheus.buckets":                                       "Buckets for latency metrics.",
			"--metrics.prometheus.entrypoint":                                    "EntryPoint",
			"--metrics.prometheus.manualrouting":                                 "Manual routing",
			"--metrics.statsd":                                                   "StatsD metrics exporter type.",
			"--metrics.statsd.addentrypointslabels":                              "Enable metrics on entry points.",
			"--metrics.statsd.address":                                           "StatsD address.",
			"--metrics.statsd.addrouterslabels":                                  "Enable metrics on routers.",
			"--metrics.statsd.addserviceslabels":                                 "Enable metrics on services.",
			"--metrics.statsd.prefix":                                            "Prefix to use for metrics collection.",
			"--metrics.statsd.pushinterval":                                      "StatsD push interval.",
			"--pilot.dashboard":                                                  "Enable Traefik Pilot in the dashboard.",
			"--pilot.token":                                                      "Traefik Pilot token.",
			"--ping":                                                             "Enable ping.",
			"--ping.entrypoint":                                                  "EntryPoint",
			"--ping.manualrouting":                                               "Manual routing",
			"--ping.terminatingstatuscode":                                       "Terminating status code",
			"--providers.consul":                                                 "Enable Consul backend with default settings.",
			"--providers.consul.endpoints":                                       "KV store endpoints",
			"--providers.consul.password":                                        "KV Password",
			"--providers.consul.rootkey":                                         "Root key used for KV store",
			"--providers.consul.tls.ca":                                          "TLS CA",
			"--providers.consul.tls.caoptional":                                  "TLS CA.Optional",
			"--providers.consul.tls.cert":                                        "TLS cert",
			"--providers.consul.tls.insecureskipverify":                          "TLS insecure skip verify",
			"--providers.consul.tls.key":                                         "TLS key",
			"--providers.consul.username":                                        "KV Username",
			"--providers.consulcatalog":                                          "Enable ConsulCatalog backend with default settings.",
			"--providers.consulcatalog.cache":                                    "Use local agent caching for catalog reads.",
			"--providers.consulcatalog.connectaware":                             "Enable Consul Connect support.",
			"--providers.consulcatalog.connectbydefault":                         "Consider every service as Connect capable by default.",
			"--providers.consulcatalog.constraints":                              "Constraints is an expression that Traefik matches against the container's labels to determine whether to create any route for that container.",
			"--providers.consulcatalog.defaultrule":                              "Default rule.",
			"--providers.consulcatalog.endpoint.address":                         "The address of the Consul server",
			"--providers.consulcatalog.endpoint.datacenter":                      "Data center to use. If not provided, the default agent data center is used",
			"--providers.consulcatalog.endpoint.endpointwaittime":                "WaitTime limits how long a Watch will block. If not provided, the agent default values will be used",
			"--providers.consulcatalog.endpoint.httpauth.password":               "Basic Auth password",
			"--providers.consulcatalog.endpoint.httpauth.username":               "Basic Auth username",
			"--providers.consulcatalog.endpoint.scheme":                          "The URI scheme for the Consul server",
			"--providers.consulcatalog.endpoint.tls.ca":                          "TLS CA",
			"--providers.consulcatalog.endpoint.tls.caoptional":                  "TLS CA.Optional",
			"--providers.consulcatalog.endpoint.tls.cert":                        "TLS cert",
			"--providers.consulcatalog.endpoint.tls.insecureskipverify":          "TLS insecure skip verify",
			"--providers.consulcatalog.endpoint.tls.key":                         "TLS key",
			"--providers.consulcatalog.endpoint.token":                           "Token is used to provide a per-request ACL token which overrides the agent's default token",
			"--providers.consulcatalog.exposedbydefault":                         "Expose containers by default.",
			"--providers.consulcatalog.prefix":                                   "Prefix for consul service tags. Default 'traefik'",
			"--providers.consulcatalog.refreshinterval":                          "Interval for check Consul API. Default 15s",
			"--providers.consulcatalog.requireconsistent":                        "Forces the read to be fully consistent.",
			"--providers.consulcatalog.servicename":                              "Name of the Traefik service in Consul Catalog (needs to be registered via the orchestrator or manually).",
			"--providers.consulcatalog.stale":                                    "Use stale consistency for catalog reads.",
			"--providers.docker":                                                 "Enable Docker backend with default settings.",
			"--providers.docker.constraints":                                     "Constraints is an expression that Traefik matches against the container's labels to determine whether to create any route for that container.",
			"--providers.docker.defaultrule":                                     "Default rule.",
			"--providers.docker.endpoint":                                        "Docker server endpoint. Can be a tcp or a unix socket endpoint.",
			"--providers.docker.exposedbydefault":                                "Expose containers by default.",
			"--providers.docker.httpclienttimeout":                               "Client timeout for HTTP connections.",
			"--providers.docker.network":                                         "Default Docker network used.",
			"--providers.docker.swarmmode":                                       "Use Docker on Swarm Mode.",
			"--providers.docker.swarmmoderefreshseconds":                         "Polling interval for swarm mode.",
			"--providers.docker.tls.ca":                                          "TLS CA",
			"--providers.docker.tls.caoptional":                                  "TLS CA.Optional",
			"--providers.docker.tls.cert":                                        "TLS cert",
			"--providers.docker.tls.insecureskipverify":                          "TLS insecure skip verify",
			"--providers.docker.tls.key":                                         "TLS key",
			"--providers.docker.usebindportip":                                   "Use the ip address from the bound port, rather than from the inner network.",
			"--providers.docker.watch":                                           "Watch Docker Swarm events.",
			"--providers.ecs":                                                    "Enable AWS ECS backend with default settings.",
			"--providers.ecs.accesskeyid":                                        "The AWS credentials access key to use for making requests",
			"--providers.ecs.autodiscoverclusters":                               "Auto discover cluster",
			"--providers.ecs.clusters":                                           "ECS Clusters name",
			"--providers.ecs.constraints":                                        "Constraints is an expression that Traefik matches against the container's labels to determine whether to create any route for that container.",
			"--providers.ecs.defaultrule":                                        "Default rule.",
			"--providers.ecs.exposedbydefault":                                   "Expose services by default",
			"--providers.ecs.refreshseconds":                                     "Polling interval (in seconds)",
			"--providers.ecs.region":                                             "The AWS region to use for requests",
			"--providers.ecs.secretaccesskey":                                    "The AWS credentials access key to use for making requests",
			"--providers.etcd":                                                   "Enable Etcd backend with default settings.",
			"--providers.etcd.endpoints":                                         "KV store endpoints",
			"--providers.etcd.password":                                          "KV Password",
			"--providers.etcd.rootkey":                                           "Root key used for KV store",
			"--providers.etcd.tls.ca":                                            "TLS CA",
			"--providers.etcd.tls.caoptional":                                    "TLS CA.Optional",
			"--providers.etcd.tls.cert":                                          "TLS cert",
			"--providers.etcd.tls.insecureskipverify":                            "TLS insecure skip verify",
			"--providers.etcd.tls.key":                                           "TLS key",
			"--providers.etcd.username":                                          "KV Username",
			"--providers.file.debugloggeneratedtemplate":                         "Enable debug logging of generated configuration template.",
			"--providers.file.directory":                                         "Load dynamic configuration from one or more .yml or .toml files in a directory.",
			"--providers.file.filename":                                          "Load dynamic configuration from a file.",
			"--providers.file.watch":                                             "Watch provider.",
			"--providers.http":                                                   "Enable HTTP backend with default settings.",
			"--providers.http.endpoint":                                          "Load configuration from this endpoint.",
			"--providers.http.pollinterval":                                      "Polling interval for endpoint.",
			"--providers.http.polltimeout":                                       "Polling timeout for endpoint.",
			"--providers.http.tls.ca":                                            "TLS CA",
			"--providers.http.tls.caoptional":                                    "TLS CA.Optional",
			"--providers.http.tls.cert":                                          "TLS cert",
			"--providers.http.tls.insecureskipverify":                            "TLS insecure skip verify",
			"--providers.http.tls.key":                                           "TLS key",
			"--providers.kubernetescrd":                                          "Enable Kubernetes backend with default settings.",
			"--providers.kubernetescrd.allowcrossnamespace":                      "Allow cross namespace resource reference.",
			"--providers.kubernetescrd.allowexternalnameservices":                "Allow ExternalName services.",
			"--providers.kubernetescrd.certauthfilepath":                         "Kubernetes certificate authority file path (not needed for in-cluster client).",
			"--providers.kubernetescrd.endpoint":                                 "Kubernetes server endpoint (required for external cluster client).",
			"--providers.kubernetescrd.ingressclass":                             "Value of kubernetes.io/ingress.class annotation to watch for.",
			"--providers.kubernetescrd.labelselector":                            "Kubernetes label selector to use.",
			"--providers.kubernetescrd.namespaces":                               "Kubernetes namespaces.",
			"--providers.kubernetescrd.throttleduration":                         "Ingress refresh throttle duration",
			"--providers.kubernetescrd.token":                                    "Kubernetes bearer token (not needed for in-cluster client).",
			"--providers.kubernetesgateway":                                      "Enable Kubernetes gateway api provider with default settings.",
			"--providers.kubernetesgateway.certauthfilepath":                     "Kubernetes certificate authority file path (not needed for in-cluster client).",
			"--providers.kubernetesgateway.endpoint":                             "Kubernetes server endpoint (required for external cluster client).",
			"--providers.kubernetesgateway.labelselector":                        "Kubernetes label selector to select specific GatewayClasses.",
			"--providers.kubernetesgateway.namespaces":                           "Kubernetes namespaces.",
			"--providers.kubernetesgateway.throttleduration":                     "Kubernetes refresh throttle duration",
			"--providers.kubernetesgateway.token":                                "Kubernetes bearer token (not needed for in-cluster client).",
			"--providers.kubernetesingress":                                      "Enable Kubernetes backend with default settings.",
			"--providers.kubernetesingress.allowemptyservices":                   "Allow creation of services without endpoints.",
			"--providers.kubernetesingress.allowexternalnameservices":            "Allow ExternalName services.",
			"--providers.kubernetesingress.certauthfilepath":                     "Kubernetes certificate authority file path (not needed for in-cluster client).",
			"--providers.kubernetesingress.endpoint":                             "Kubernetes server endpoint (required for external cluster client).",
			"--providers.kubernetesingress.ingressclass":                         "Value of kubernetes.io/ingress.class annotation or IngressClass name to watch for.",
			"--providers.kubernetesingress.ingressendpoint.hostname":             "Hostname used for Kubernetes Ingress endpoints.",
			"--providers.kubernetesingress.ingressendpoint.ip":                   "IP used for Kubernetes Ingress endpoints.",
			"--providers.kubernetesingress.ingressendpoint.publishedservice":     "Published Kubernetes Service to copy status from.",
			"--providers.kubernetesingress.labelselector":                        "Kubernetes Ingress label selector to use.",
			"--providers.kubernetesingress.namespaces":                           "Kubernetes namespaces.",
			"--providers.kubernetesingress.throttleduration":                     "Ingress refresh throttle duration",
			"--providers.kubernetesingress.token":                                "Kubernetes bearer token (not needed for in-cluster client).",
			"--providers.marathon":                                               "Enable Marathon backend with default settings.",
			"--providers.marathon.basic.httpbasicauthuser":                       "Basic authentication User.",
			"--providers.marathon.basic.httpbasicpassword":                       "Basic authentication Password.",
			"--providers.marathon.constraints":                                   "Constraints is an expression that Traefik matches against the application's labels to determine whether to create any route for that application.",
			"--providers.marathon.dcostoken":                                     "DCOSToken for DCOS environment, This will override the Authorization header.",
			"--providers.marathon.defaultrule":                                   "Default rule.",
			"--providers.marathon.dialertimeout":                                 "Set a dialer timeout for Marathon.",
			"--providers.marathon.endpoint":                                      "Marathon server endpoint. You can also specify multiple endpoint for Marathon.",
			"--providers.marathon.exposedbydefault":                              "Expose Marathon apps by default.",
			"--providers.marathon.forcetaskhostname":                             "Force to use the task's hostname.",
			"--providers.marathon.keepalive":                                     "Set a TCP Keep Alive time.",
			"--providers.marathon.respectreadinesschecks":                        "Filter out tasks with non-successful readiness checks during deployments.",
			"--providers.marathon.responseheadertimeout":                         "Set a response header timeout for Marathon.",
			"--providers.marathon.tls.ca":                                        "TLS CA",
			"--providers.marathon.tls.caoptional":                                "TLS CA.Optional",
			"--providers.marathon.tls.cert":                                      "TLS cert",
			"--providers.marathon.tls.insecureskipverify":                        "TLS insecure skip verify",
			"--providers.marathon.tls.key":                                       "TLS key",
			"--providers.marathon.tlshandshaketimeout":                           "Set a TLS handshake timeout for Marathon.",
			"--providers.marathon.trace":                                         "Display additional provider logs.",
			"--providers.marathon.watch":                                         "Watch provider.",
			"--providers.plugin.<name>":                                          "Plugins configuration.",
			"--providers.providersthrottleduration":                              "Backends throttle duration: minimum duration between 2 events from providers before applying a new configuration. It avoids unnecessary reloads if multiples events are sent in a short amount of time.",
			"--providers.rancher":                                                "Enable Rancher backend with default settings.",
			"--providers.rancher.constraints":                                    "Constraints is an expression that Traefik matches against the container's labels to determine whether to create any route for that container.",
			"--providers.rancher.defaultrule":                                    "Default rule.",
			"--providers.rancher.enableservicehealthfilter":                      "Filter services with unhealthy states and inactive states.",
			"--providers.rancher.exposedbydefault":                               "Expose containers by default.",
			"--providers.rancher.intervalpoll":                                   "Poll the Rancher metadata service every 'rancher.refreshseconds' (less accurate).",
			"--providers.rancher.prefix":                                         "Prefix used for accessing the Rancher metadata service.",
			"--providers.rancher.refreshseconds":                                 "Defines the polling interval in seconds.",
			"--providers.rancher.watch":                                          "Watch provider.",
			"--providers.redis":                                                  "Enable Redis backend with default settings.",
			"--providers.redis.endpoints":                                        "KV store endpoints",
			"--providers.redis.password":                                         "KV Password",
			"--providers.redis.rootkey":                                          "Root key used for KV store",
			"--providers.redis.tls.ca":                                           "TLS CA",
			"--providers.redis.tls.caoptional":                                   "TLS CA.Optional",
			"--providers.redis.tls.cert":                                         "TLS cert",
			"--providers.redis.tls.insecureskipverify":                           "TLS insecure skip verify",
			"--providers.redis.tls.key":                                          "TLS key",
			"--providers.redis.username":                                         "KV Username",
			"--providers.rest":                                                   "Enable Rest backend with default settings.",
			"--providers.rest.insecure":                                          "Activate REST Provider directly on the entryPoint named traefik.",
			"--providers.zookeeper":                                              "Enable ZooKeeper backend with default settings.",
			"--providers.zookeeper.endpoints":                                    "KV store endpoints",
			"--providers.zookeeper.password":                                     "KV Password",
			"--providers.zookeeper.rootkey":                                      "Root key used for KV store",
			"--providers.zookeeper.tls.ca":                                       "TLS CA",
			"--providers.zookeeper.tls.caoptional":                               "TLS CA.Optional",
			"--providers.zookeeper.tls.cert":                                     "TLS cert",
			"--providers.zookeeper.tls.insecureskipverify":                       "TLS insecure skip verify",
			"--providers.zookeeper.tls.key":                                      "TLS key",
			"--providers.zookeeper.username":                                     "KV Username",
			"--serverstransport.forwardingtimeouts.dialtimeout":                  "The amount of time to wait until a connection to a backend server can be established. If zero, no timeout exists.",
			"--serverstransport.forwardingtimeouts.idleconntimeout":              "The maximum period for which an idle HTTP keep-alive connection will remain open before closing itself",
			"--serverstransport.forwardingtimeouts.responseheadertimeout":        "The amount of time to wait for a server's response headers after fully writing the request (including its body, if any). If zero, no timeout exists.",
			"--serverstransport.insecureskipverify":                              "Disable SSL certificate verification.",
			"--serverstransport.maxidleconnsperhost":                             "If non-zero, controls the maximum idle (keep-alive) to keep per-host. If zero, DefaultMaxIdleConnsPerHost is used",
			"--serverstransport.rootcas":                                         "Add cert file for self-signed certificate.",
			"--tracing":                                                          "OpenTracing configuration.",
			"--tracing.datadog":                                                  "Settings for Datadog.",
			"--tracing.datadog.bagageprefixheadername":                           "Specifies the header name prefix that will be used to store baggage items in a map.",
			"--tracing.datadog.debug":                                            "Enable Datadog debug.",
			"--tracing.datadog.globaltag":                                        "Key:Value tag to be set on all the spans.",
			"--tracing.datadog.localagenthostport":                               "Set datadog-agent's host:port that the reporter will used.",
			"--tracing.datadog.parentidheadername":                               "Specifies the header name that will be used to store the parent ID.",
			"--tracing.datadog.prioritysampling":                                 "Enable priority sampling. When using distributed tracing, this option must be enabled in order to get all the parts of a distributed trace sampled.",
			"--tracing.datadog.samplingpriorityheadername":                       "Specifies the header name that will be used to store the sampling priority.",
			"--tracing.datadog.traceidheadername":                                "Specifies the header name that will be used to store the trace ID.",
			"--tracing.elastic":                                                  "Settings for Elastic.",
			"--tracing.elastic.secrettoken":                                      "Set the token used to connect to Elastic APM Server.",
			"--tracing.elastic.serverurl":                                        "Set the URL of the Elastic APM server.",
			"--tracing.elastic.serviceenvironment":                               "Set the name of the environment Traefik is deployed in, e.g. 'production' or 'staging'.",
			"--tracing.haystack":                                                 "Settings for Haystack.",
			"--tracing.haystack.baggageprefixheadername":                         "Specifies the header name prefix that will be used to store baggage items in a map.",
			"--tracing.haystack.globaltag":                                       "Key:Value tag to be set on all the spans.",
			"--tracing.haystack.localagenthost":                                  "Set haystack-agent's host that the reporter will used.",
			"--tracing.haystack.localagentport":                                  "Set haystack-agent's port that the reporter will used.",
			"--tracing.haystack.parentidheadername":                              "Specifies the header name that will be used to store the parent ID.",
			"--tracing.haystack.spanidheadername":                                "Specifies the header name that will be used to store the span ID.",
			"--tracing.haystack.traceidheadername":                               "Specifies the header name that will be used to store the trace ID.",
			"--tracing.instana":                                                  "Settings for Instana.",
			"--tracing.instana.localagenthost":                                   "Set instana-agent's host that the reporter will used.",
			"--tracing.instana.localagentport":                                   "Set instana-agent's port that the reporter will used.",
			"--tracing.instana.loglevel":                                         "Set instana-agent's log level. ('error','warn','info','debug')",
			"--tracing.jaeger":                                                   "Settings for Jaeger.",
			"--tracing.jaeger.collector.endpoint":                                "Instructs reporter to send spans to jaeger-collector at this URL.",
			"--tracing.jaeger.collector.password":                                "Password for basic http authentication when sending spans to jaeger-collector.",
			"--tracing.jaeger.collector.user":                                    "User for basic http authentication when sending spans to jaeger-collector.",
			"--tracing.jaeger.disableattemptreconnecting":                        "Disable the periodic re-resolution of the agent's hostname and reconnection if there was a change.",
			"--tracing.jaeger.gen128bit":                                         "Generate 128 bit span IDs.",
			"--tracing.jaeger.localagenthostport":                                "Set jaeger-agent's host:port that the reporter will used.",
			"--tracing.jaeger.propagation":                                       "Which propagation format to use (jaeger/b3).",
			"--tracing.jaeger.samplingparam":                                     "Set the sampling parameter.",
			"--tracing.jaeger.samplingserverurl":                                 "Set the sampling server url.",
			"--tracing.jaeger.samplingtype":                                      "Set the sampling type.",
			"--tracing.jaeger.tracecontextheadername":                            "Set the header to use for the trace-id.",
			"--tracing.servicename":                                              "Set the name for this service.",
			"--tracing.spannamelimit":                                            "Set the maximum character limit for Span names (default 0 = no limit).",
			"--tracing.zipkin":                                                   "Settings for Zipkin.",
			"--tracing.zipkin.httpendpoint":                                      "HTTP Endpoint to report traces to.",
			"--tracing.zipkin.id128bit":                                          "Use Zipkin 128 bit root span IDs.",
			"--tracing.zipkin.samespan":                                          "Use Zipkin SameSpan RPC style traces.",
			"--tracing.zipkin.samplerate":                                        "The rate between 0.0 and 1.0 of requests to trace.",
		}

		vals := make([]string, 0)
		for key, value := range flags {
			replacement := ""
			if index := strings.Index(key, "<name>"); index != -1 && strings.HasPrefix(c.Value, key[:index]) {
				replacement = strings.Split(c.Value[index:], ".")[0]
			}
			vals = append(vals, strings.Replace(key, "<name>", replacement, -1), value)
		}

		if strings.HasPrefix(c.Value, "--accesslog.fields.headers.names.") {
			return http.ActionRequestHeaderNames().Invoke(c).Prefix("--accesslog.fields.headers.names.").ToMultiPartsA(".")
		}
		if strings.HasPrefix(c.Value, "--accesslog.fields.names.") {
			return ActionAccessLogFieldNames().Invoke(c).Prefix("--accesslog.fields.names.").ToMultiPartsA(".")
		}
		return carapace.ActionValuesDescribed(vals...).Invoke(c).ToMultiPartsA(".")
	})
}

func flagValues(name string) (acn carapace.Action, ok bool) {
	a := map[string]carapace.Action{
		"--accesslog.bufferingsize":                                         carapace.ActionValues(),
		"--accesslog.fields.defaultmode":                                    carapace.ActionValues("keep", "drop", "redact"),
		"--accesslog.fields.headers.defaultmode":                            carapace.ActionValues("keep", "drop", "redact"),
		"--accesslog.fields.headers.names.<name>":                           carapace.ActionValues("keep", "drop", "redact"),
		"--accesslog.fields.names.<name>":                                   carapace.ActionValues("keep", "drop", "redact"),
		"--accesslog.filepath":                                              carapace.ActionFiles(),
		"--accesslog.filters.minduration":                                   carapace.ActionValues(),
		"--accesslog.filters.statuscodes":                                   actionStatusCodeRanges(),
		"--accesslog.format":                                                carapace.ActionValues("json", "common"),
		"--certificatesresolvers.<name>.acme.caserver":                      carapace.ActionValues(),
		"--certificatesresolvers.<name>.acme.dnschallenge.delaybeforecheck": carapace.ActionValues(),
		"--certificatesresolvers.<name>.acme.dnschallenge.provider":         carapace.ActionValues(),
		"--certificatesresolvers.<name>.acme.dnschallenge.resolvers":        carapace.ActionValues(),
		"--certificatesresolvers.<name>.acme.eab.hmacencoded":               carapace.ActionValues(),
		"--certificatesresolvers.<name>.acme.eab.kid":                       carapace.ActionValues(),
		"--certificatesresolvers.<name>.acme.email":                         carapace.ActionValues(),
		"--certificatesresolvers.<name>.acme.httpchallenge.entrypoint":      carapace.ActionValues(),
		"--certificatesresolvers.<name>.acme.keytype":                       carapace.ActionValues("EC256", "EC384", "RSA2048", "RSA4096", "RSA8192"),
		"--certificatesresolvers.<name>.acme.preferredchain":                carapace.ActionValues(),
		"--certificatesresolvers.<name>.acme.storage":                       carapace.ActionFiles(),
		"--configfile":                                                       carapace.ActionFiles(),
		"--entrypoints.<name>.address":                                       carapace.ActionValues(),
		"--entrypoints.<name>.forwardedheaders.trustedips":                   carapace.ActionValues(),
		"--entrypoints.<name>.http":                                          carapace.ActionValues(),
		"--entrypoints.<name>.http.middlewares":                              carapace.ActionValues(),
		"--entrypoints.<name>.http.redirections.entrypoint.priority":         carapace.ActionValues(),
		"--entrypoints.<name>.http.redirections.entrypoint.scheme":           carapace.ActionValues(),
		"--entrypoints.<name>.http.redirections.entrypoint.to":               carapace.ActionValues(),
		"--entrypoints.<name>.http.tls.certresolver":                         carapace.ActionValues(),
		"--entrypoints.<name>.http.tls.domains":                              carapace.ActionValues(),
		"--entrypoints.<name>.http.tls.domains[n].main":                      carapace.ActionValues(),
		"--entrypoints.<name>.http.tls.domains[n].sans":                      carapace.ActionValues(),
		"--entrypoints.<name>.http.tls.options":                              carapace.ActionValues(),
		"--entrypoints.<name>.proxyprotocol.trustedips":                      carapace.ActionValues(),
		"--entrypoints.<name>.transport.lifecycle.gracetimeout":              carapace.ActionValues(),
		"--entrypoints.<name>.transport.lifecycle.requestacceptgracetimeout": carapace.ActionValues(),
		"--entrypoints.<name>.transport.respondingtimeouts.idletimeout":      carapace.ActionValues(),
		"--entrypoints.<name>.transport.respondingtimeouts.readtimeout":      carapace.ActionValues(),
		"--entrypoints.<name>.transport.respondingtimeouts.writetimeout":     carapace.ActionValues(),
		"--entrypoints.<name>.udp.timeout":                                   carapace.ActionValues(),
		"--experimental.localplugins.<name>.modulename":                      carapace.ActionValues(),
		"--experimental.plugins.<name>.modulename":                           carapace.ActionValues(),
		"--experimental.plugins.<name>.version":                              carapace.ActionValues(),
		"--hostresolver.resolvconfig":                                        carapace.ActionFiles(),
		"--hostresolver.resolvdepth":                                         carapace.ActionValues(),
		"--log.filepath":                                                     carapace.ActionFiles(),
		"--log.format":                                                       carapace.ActionValues("json", "common"),
		"--log.level":                                                        carapace.ActionValues(),
		"--metrics.datadog.address":                                          carapace.ActionValues(),
		"--metrics.datadog.pushinterval":                                     carapace.ActionValues(),
		"--metrics.influxdb.address":                                         carapace.ActionValues(),
		"--metrics.influxdb.database":                                        carapace.ActionValues(),
		"--metrics.influxdb.password":                                        carapace.ActionValues(),
		"--metrics.influxdb.protocol":                                        carapace.ActionValues("udp", "http"),
		"--metrics.influxdb.pushinterval":                                    carapace.ActionValues(),
		"--metrics.influxdb.retentionpolicy":                                 carapace.ActionValues(),
		"--metrics.influxdb.username":                                        carapace.ActionValues(),
		"--metrics.prometheus.buckets":                                       carapace.ActionValues(),
		"--metrics.prometheus.entrypoint":                                    carapace.ActionValues(),
		"--metrics.statsd.address":                                           carapace.ActionValues(),
		"--metrics.statsd.prefix":                                            carapace.ActionValues(),
		"--metrics.statsd.pushinterval":                                      carapace.ActionValues(),
		"--pilot.token":                                                      carapace.ActionValues(),
		"--ping.entrypoint":                                                  carapace.ActionValues(),
		"--ping.terminatingstatuscode":                                       http.ActionStatusCodes(),
		"--providers.consul.endpoints":                                       carapace.ActionValues(),
		"--providers.consul.password":                                        carapace.ActionValues(),
		"--providers.consul.rootkey":                                         carapace.ActionValues(),
		"--providers.consul.tls.ca":                                          carapace.ActionFiles(),
		"--providers.consul.tls.cert":                                        carapace.ActionFiles(),
		"--providers.consul.tls.key":                                         carapace.ActionFiles(),
		"--providers.consul.username":                                        carapace.ActionValues(),
		"--providers.consulcatalog.constraints":                              carapace.ActionValues(),
		"--providers.consulcatalog.defaultrule":                              carapace.ActionValues(),
		"--providers.consulcatalog.endpoint.address":                         carapace.ActionValues(),
		"--providers.consulcatalog.endpoint.datacenter":                      carapace.ActionValues(),
		"--providers.consulcatalog.endpoint.endpointwaittime":                carapace.ActionValues(),
		"--providers.consulcatalog.endpoint.httpauth.password":               carapace.ActionValues(),
		"--providers.consulcatalog.endpoint.httpauth.username":               carapace.ActionValues(),
		"--providers.consulcatalog.endpoint.scheme":                          carapace.ActionValues(),
		"--providers.consulcatalog.endpoint.tls.ca":                          carapace.ActionFiles(),
		"--providers.consulcatalog.endpoint.tls.cert":                        carapace.ActionFiles(),
		"--providers.consulcatalog.endpoint.tls.key":                         carapace.ActionFiles(),
		"--providers.consulcatalog.endpoint.token":                           carapace.ActionValues(),
		"--providers.consulcatalog.prefix":                                   carapace.ActionValues(),
		"--providers.consulcatalog.refreshinterval":                          carapace.ActionValues(),
		"--providers.consulcatalog.servicename":                              carapace.ActionValues(),
		"--providers.docker.constraints":                                     carapace.ActionValues(),
		"--providers.docker.defaultrule":                                     carapace.ActionValues(),
		"--providers.docker.endpoint":                                        carapace.ActionValues(),
		"--providers.docker.httpclienttimeout":                               carapace.ActionValues(),
		"--providers.docker.network":                                         docker.ActionNetworks(),
		"--providers.docker.swarmmoderefreshseconds":                         carapace.ActionValues(),
		"--providers.docker.tls.ca":                                          carapace.ActionFiles(),
		"--providers.docker.tls.cert":                                        carapace.ActionFiles(),
		"--providers.docker.tls.key":                                         carapace.ActionFiles(),
		"--providers.ecs.accesskeyid":                                        carapace.ActionValues(),
		"--providers.ecs.clusters":                                           carapace.ActionValues(),
		"--providers.ecs.constraints":                                        carapace.ActionValues(),
		"--providers.ecs.defaultrule":                                        carapace.ActionValues(),
		"--providers.ecs.refreshseconds":                                     carapace.ActionValues(),
		"--providers.ecs.region":                                             aws.ActionRegions(),
		"--providers.ecs.secretaccesskey":                                    carapace.ActionValues(),
		"--providers.etcd.endpoints":                                         carapace.ActionValues(),
		"--providers.etcd.password":                                          carapace.ActionValues(),
		"--providers.etcd.rootkey":                                           carapace.ActionValues(),
		"--providers.etcd.tls.ca":                                            carapace.ActionFiles(),
		"--providers.etcd.tls.cert":                                          carapace.ActionFiles(),
		"--providers.etcd.tls.key":                                           carapace.ActionFiles(),
		"--providers.etcd.username":                                          carapace.ActionValues(),
		"--providers.file.directory":                                         carapace.ActionDirectories(),
		"--providers.file.filename":                                          carapace.ActionFiles(),
		"--providers.http.endpoint":                                          carapace.ActionValues(),
		"--providers.http.pollinterval":                                      carapace.ActionValues(),
		"--providers.http.polltimeout":                                       carapace.ActionValues(),
		"--providers.http.tls.ca":                                            carapace.ActionFiles(),
		"--providers.http.tls.cert":                                          carapace.ActionFiles(),
		"--providers.http.tls.key":                                           carapace.ActionFiles(),
		"--providers.kubernetescrd.certauthfilepath":                         carapace.ActionFiles(),
		"--providers.kubernetescrd.endpoint":                                 carapace.ActionValues(),
		"--providers.kubernetescrd.ingressclass":                             carapace.ActionValues(),
		"--providers.kubernetescrd.labelselector":                            carapace.ActionValues(),
		"--providers.kubernetescrd.namespaces":                               carapace.ActionValues(),
		"--providers.kubernetescrd.throttleduration":                         carapace.ActionValues(),
		"--providers.kubernetescrd.token":                                    carapace.ActionValues(),
		"--providers.kubernetesgateway.certauthfilepath":                     carapace.ActionFiles(),
		"--providers.kubernetesgateway.endpoint":                             carapace.ActionValues(),
		"--providers.kubernetesgateway.labelselector":                        carapace.ActionValues(),
		"--providers.kubernetesgateway.namespaces":                           carapace.ActionValues(),
		"--providers.kubernetesgateway.throttleduration":                     carapace.ActionValues(),
		"--providers.kubernetesgateway.token":                                carapace.ActionValues(),
		"--providers.kubernetesingress.certauthfilepath":                     carapace.ActionFiles(),
		"--providers.kubernetesingress.endpoint":                             carapace.ActionValues(),
		"--providers.kubernetesingress.ingressclass":                         carapace.ActionValues(),
		"--providers.kubernetesingress.ingressendpoint.hostname":             carapace.ActionValues(),
		"--providers.kubernetesingress.ingressendpoint.ip":                   carapace.ActionValues(),
		"--providers.kubernetesingress.ingressendpoint.publishedservice":     carapace.ActionValues(),
		"--providers.kubernetesingress.labelselector":                        carapace.ActionValues(),
		"--providers.kubernetesingress.namespaces":                           carapace.ActionValues(),
		"--providers.kubernetesingress.throttleduration":                     carapace.ActionValues(),
		"--providers.kubernetesingress.token":                                carapace.ActionValues(),
		"--providers.marathon.basic.httpbasicauthuser":                       carapace.ActionValues(),
		"--providers.marathon.basic.httpbasicpassword":                       carapace.ActionValues(),
		"--providers.marathon.constraints":                                   carapace.ActionValues(),
		"--providers.marathon.dcostoken":                                     carapace.ActionValues(),
		"--providers.marathon.defaultrule":                                   carapace.ActionValues(),
		"--providers.marathon.dialertimeout":                                 carapace.ActionValues(),
		"--providers.marathon.endpoint":                                      carapace.ActionValues(),
		"--providers.marathon.keepalive":                                     carapace.ActionValues(),
		"--providers.marathon.responseheadertimeout":                         carapace.ActionValues(),
		"--providers.marathon.tls.ca":                                        carapace.ActionValues(),
		"--providers.marathon.tls.cert":                                      carapace.ActionFiles(),
		"--providers.marathon.tls.key":                                       carapace.ActionFiles(),
		"--providers.marathon.tlshandshaketimeout":                           carapace.ActionFiles(),
		"--providers.plugin.<name>":                                          carapace.ActionValues(),
		"--providers.providersthrottleduration":                              carapace.ActionValues(),
		"--providers.rancher.constraints":                                    carapace.ActionValues(),
		"--providers.rancher.defaultrule":                                    carapace.ActionValues(),
		"--providers.rancher.prefix":                                         carapace.ActionValues(),
		"--providers.rancher.refreshseconds":                                 carapace.ActionValues(),
		"--providers.redis.endpoints":                                        carapace.ActionValues(),
		"--providers.redis.password":                                         carapace.ActionValues(),
		"--providers.redis.rootkey":                                          carapace.ActionValues(),
		"--providers.redis.tls.ca":                                           carapace.ActionFiles(),
		"--providers.redis.tls.cert":                                         carapace.ActionFiles(),
		"--providers.redis.tls.key":                                          carapace.ActionFiles(),
		"--providers.redis.username":                                         carapace.ActionValues(),
		"--providers.zookeeper.endpoints":                                    carapace.ActionValues(),
		"--providers.zookeeper.password":                                     carapace.ActionValues(),
		"--providers.zookeeper.rootkey":                                      carapace.ActionValues(),
		"--providers.zookeeper.tls.ca":                                       carapace.ActionFiles(),
		"--providers.zookeeper.tls.cert":                                     carapace.ActionFiles(),
		"--providers.zookeeper.tls.key":                                      carapace.ActionFiles(),
		"--providers.zookeeper.username":                                     carapace.ActionValues(),
		"--serverstransport.forwardingtimeouts.dialtimeout":                  carapace.ActionValues(),
		"--serverstransport.forwardingtimeouts.idleconntimeout":              carapace.ActionValues(),
		"--serverstransport.forwardingtimeouts.responseheadertimeout":        carapace.ActionValues(),
		"--serverstransport.maxidleconnsperhost":                             carapace.ActionValues(),
		"--serverstransport.rootcas":                                         carapace.ActionValues(),
		"--tracing.datadog.bagageprefixheadername":                           carapace.ActionValues(),
		"--tracing.datadog.globaltag":                                        carapace.ActionValues(),
		"--tracing.datadog.localagenthostport":                               carapace.ActionValues(),
		"--tracing.datadog.parentidheadername":                               carapace.ActionValues(),
		"--tracing.datadog.samplingpriorityheadername":                       carapace.ActionValues(),
		"--tracing.datadog.traceidheadername":                                carapace.ActionValues(),
		"--tracing.elastic.secrettoken":                                      carapace.ActionValues(),
		"--tracing.elastic.serverurl":                                        carapace.ActionValues(),
		"--tracing.elastic.serviceenvironment":                               carapace.ActionValues(),
		"--tracing.haystack.baggageprefixheadername":                         carapace.ActionValues(),
		"--tracing.haystack.globaltag":                                       carapace.ActionValues(),
		"--tracing.haystack.localagenthost":                                  carapace.ActionValues(),
		"--tracing.haystack.localagentport":                                  carapace.ActionValues(),
		"--tracing.haystack.parentidheadername":                              carapace.ActionValues(),
		"--tracing.haystack.spanidheadername":                                carapace.ActionValues(),
		"--tracing.haystack.traceidheadername":                               carapace.ActionValues(),
		"--tracing.instana.localagenthost":                                   carapace.ActionValues(),
		"--tracing.instana.localagentport":                                   carapace.ActionValues(),
		"--tracing.instana.loglevel":                                         carapace.ActionValues("error", "warn", "info", "debug").StyleF(style.ForLogLevel),
		"--tracing.jaeger.collector.endpoint":                                carapace.ActionValues(),
		"--tracing.jaeger.collector.password":                                carapace.ActionValues(),
		"--tracing.jaeger.collector.user":                                    carapace.ActionValues(),
		"--tracing.jaeger.localagenthostport":                                carapace.ActionValues(),
		"--tracing.jaeger.propagation":                                       carapace.ActionValues(),
		"--tracing.jaeger.samplingparam":                                     carapace.ActionValues(),
		"--tracing.jaeger.samplingserverurl":                                 carapace.ActionValues(),
		"--tracing.jaeger.samplingtype":                                      jaeger.ActionSamplingTypes(),
		"--tracing.jaeger.tracecontextheadername":                            carapace.ActionValues(),
		"--tracing.servicename":                                              carapace.ActionValues(),
		"--tracing.spannamelimit":                                            carapace.ActionValues(),
		"--tracing.zipkin.httpendpoint":                                      carapace.ActionValues(),
		"--tracing.zipkin.samplerate":                                        carapace.ActionValues(),
	}

	replaced := make(map[string]carapace.Action)
	for key, value := range a {
		replacement := ""
		if index := strings.Index(key, "<name>"); index != -1 && strings.HasPrefix(name, key[:index]) {
			replacement = strings.Split(name[index:], ".")[0]
		}
		replaced[strings.Replace(key, "<name>", replacement, -1)] = value
	}

	acn, ok = replaced[name]
	return
}

func actionStatusCodeRanges() carapace.Action {
	return carapace.ActionMultiParts(",", func(c carapace.Context) carapace.Action {
		return carapace.ActionMultiParts("-", func(c carapace.Context) carapace.Action {
			if len(c.Parts) < 2 {
				return http.ActionStatusCodes().NoSpace()
			}
			return carapace.ActionValues()
		})
	})
}
