// Core Grafana history https://github.com/grafana/grafana/blob/v11.0.0-preview/public/app/plugins/datasource/prometheus/components/monaco-query-field/monaco-completion-provider/index.ts
import { TimeRange } from '@grafana/data';
import type { Monaco, monacoTypes } from '@grafana/ui';

import { CompletionType, getCompletions } from './completions';
import { DataProvider } from './data_provider';
import { getSituation } from './situation';
import { NeverCaseError } from './util';

export type TriggerType = 'partial' | 'full';

export type MonacoQueryFieldLocalState = {
  isManualTriggerRequested: boolean;
};

const TRIGGER_CHARACTERS = ['{', ',', '[', '(', '=', '~', ' ', '"'];
const MIN_WORD_LENGTH_FOR_FULL_COMPLETIONS = 3;

export function getSuggestOptions(): monacoTypes.editor.ISuggestOptions {
  return {
    // monaco-editor sometimes provides suggestions automatically, i am not
    // sure based on what, seems to be by analyzing the words already
    // written.
    // to try it out:
    // - enter `go_goroutines{job~`
    // - have the cursor at the end of the string
    // - press ctrl-enter
    // - you will get two suggestions
    // those were not provided by grafana, they are offered automatically.
    // i want to remove those. the only way i found is:
    // - every suggestion-item has a `kind` attribute,
    //   that controls the icon to the left of the suggestion.
    // - items auto-generated by monaco have `kind` set to `text`.
    // - we make sure grafana-provided suggestions do not have `kind` set to `text`.
    // - and then we tell monaco not to show suggestions of kind `text`
    showWords: false,
  };
}

function getMonacoCompletionItemKind(type: CompletionType, monaco: Monaco): monacoTypes.languages.CompletionItemKind {
  switch (type) {
    case 'DURATION':
      return monaco.languages.CompletionItemKind.Unit;
    case 'FUNCTION':
      return monaco.languages.CompletionItemKind.Variable;
    case 'HISTORY':
      return monaco.languages.CompletionItemKind.Snippet;
    case 'LABEL_NAME':
      return monaco.languages.CompletionItemKind.Enum;
    case 'LABEL_VALUE':
      return monaco.languages.CompletionItemKind.EnumMember;
    case 'METRIC_NAME':
      return monaco.languages.CompletionItemKind.Constructor;
    default:
      throw new NeverCaseError(type);
  }
}

function getTriggerType(
  word: monacoTypes.editor.IWordAtPosition | null,
  model: monacoTypes.editor.ITextModel,
  position: monacoTypes.Position,
  state: MonacoQueryFieldLocalState
): TriggerType {
  // Manual trigger (Ctrl+Space) - always full completions
  if (state.isManualTriggerRequested) {
    return 'full';
  }

  const charBeforeCursor = model.getValueInRange({
    startLineNumber: position.lineNumber,
    endLineNumber: position.lineNumber,
    startColumn: Math.max(1, position.column - 1),
    endColumn: position.column,
  });

  if (TRIGGER_CHARACTERS.includes(charBeforeCursor)) {
    return 'full';
  }

  // For typed words of sufficient length, use full completions
  if (word && word.word.length >= MIN_WORD_LENGTH_FOR_FULL_COMPLETIONS) {
    return 'full';
  }

  return 'partial';
}

export function getCompletionProvider(
  monaco: Monaco,
  dataProvider: DataProvider,
  timeRange: TimeRange
): { provider: monacoTypes.languages.CompletionItemProvider; state: MonacoQueryFieldLocalState } {
  const state: MonacoQueryFieldLocalState = {
    isManualTriggerRequested: false,
  };

  const provideCompletionItems = (
    model: monacoTypes.editor.ITextModel,
    position: monacoTypes.Position
  ): monacoTypes.languages.ProviderResult<monacoTypes.languages.CompletionList> => {
    const word = model.getWordAtPosition(position);
    const range =
      word != null
        ? monaco.Range.lift({
            startLineNumber: position.lineNumber,
            endLineNumber: position.lineNumber,
            startColumn: word.startColumn,
            endColumn: word.endColumn,
          })
        : monaco.Range.fromPositions(position);

    // Set input range for data provider
    dataProvider.monacoSettings.setInputInRange(model.getValueInRange(range));

    // Get adjusted position for cursor/selection handling
    const adjustedPosition = getAdjustedPosition(position);
    const offset = model.getOffsetAt(adjustedPosition);
    const situation = getSituation(model.getValue(), offset);

    // Early exit if no situation detected
    if (situation === null) {
      return Promise.resolve({ suggestions: [], incomplete: false });
    }

    const triggerType: TriggerType = getTriggerType(word, model, position, state);

    return getCompletions(situation, dataProvider, timeRange, word?.word, triggerType).then((items) => {
      // Monaco by-default alphabetically orders the items.
      // We use a number-as-string sortkey to maintain our custom order
      const maxIndexDigits = items.length > 0 ? items.length.toString().length : 1;
      const suggestions: monacoTypes.languages.CompletionItem[] = items.map((item, index) => ({
        kind: getMonacoCompletionItemKind(item.type, monaco),
        label: item.label,
        insertText: item.insertText,
        insertTextRules: item.insertTextRules,
        detail: item.detail,
        documentation: item.documentation,
        sortText: index.toString().padStart(maxIndexDigits, '0'), // to force the order we have
        range,
        command: item.triggerOnInsert
          ? {
              id: 'editor.action.triggerSuggest',
              title: '',
            }
          : undefined,
      }));

      return { suggestions };
    });
  };

  // Helper function to handle position adjustment for selection
  function getAdjustedPosition(position: monacoTypes.Position): { column: number; lineNumber: number } {
    let adjustedColumn = position.column;

    // Check to see if the browser supports window.getSelection()
    if (window.getSelection) {
      const selectedText = window.getSelection()?.toString();
      // If the user has selected text, adjust the cursor position to be at the start of the selection
      if (selectedText && selectedText.length > 0) {
        adjustedColumn = Math.max(1, adjustedColumn - selectedText.length);
      }
    }

    return {
      column: adjustedColumn,
      lineNumber: position.lineNumber,
    };
  }

  return {
    provider: {
      triggerCharacters: TRIGGER_CHARACTERS,
      provideCompletionItems,
    },
    state,
  };
}
