// Copyright (c) 2010, Andrei Vieru. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package lzma

import (
	"log"
	"os"
)

type lzmaTest struct {
	descr string
	level int
	size  bool
	raw   string
	lzma  []byte
	err   error
}

var lzmaTests = []lzmaTest{
	// fmt -w 80 -s file
	// cat file | sed s/\\\\/\\\\\\\\/g | sed s/\"/\\\\\"/g | sed s/^/\"/ | sed s/$/\\\\n\"\ +/
	// hexdump -Cv file.lzma | awk '{for (i=2; i<18; i++) {printf("0x%s, ", $i); if (i==9) printf("\n");} printf("\n")}'
	{
		"empty test",
		3,
		true,
		"",
		[]byte{
			0x5d, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00,
		},
		nil,
	},
	{
		"empty (with size == -1) test",
		3,
		false,
		"",
		[]byte{
			0x5d, 0x00, 0x00, 0x10, 0x00, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x83, 0xff,
			0xfb, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00,
		},
		nil,
	},
	{
		"hello world test",
		3,
		true,
		"hello world\n",
		[]byte{
			0x5d, 0x00, 0x00, 0x10, 0x00, 0x0c, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x34, 0x19,
			0x49, 0xee, 0x8d, 0xe9, 0x17, 0x89, 0x3a, 0x33,
			0x5f, 0xfc, 0xac, 0xf7, 0x20, 0x00,
		},
		nil,
	},
	{
		"hello world (with size == -1) test",
		3,
		false,
		"hello world\n",
		[]byte{
			0x5d, 0x00, 0x00, 0x10, 0x00, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x34, 0x19,
			0x49, 0xee, 0x8d, 0xe9, 0x17, 0x89, 0x3a, 0x33,
			0x5f, 0xfc, 0xb2, 0x09, 0x82, 0x2f, 0xff, 0xfd,
			0xe2, 0x80, 0x00,
		},
		nil,
	},
	{
		"text test",
		3,
		true,
		"Two named types are identical if their type names originate in the same type\n" +
			"declaration (§Declarations and scope). A named and an unnamed type are never\n" +
			"identical. Two unnamed types are identical if the corresponding type literals\n" +
			"have the same literal structure and corresponding components have identical\n" +
			"types. In detail:\n" +
			"\n" +
			"    * Two array types are identical if they have identical element types and\n" +
			"the same array length.\n" +
			"    * Two slice types are identical if they have identical element types.\n" +
			"    * Two struct types are identical if they have the same sequence of fields,\n" +
			"and if corresponding fields have the same names and identical types. Two\n" +
			"anonymous fields are considered to have the same name.\n" +
			"    * Two pointer types are identical if they have identical base types.\n" +
			"    * Two function types are identical if they have the same number of\n" +
			"parameters and result values and if corresponding parameter and result types\n" +
			"are identical. All \"...\" parameters without a specified type are defined to\n" +
			"have identical type. All \"...\" parameters with specified identical type T are\n" +
			"defined to have identical type. Parameter and result names are not required to\n" +
			"match.\n" +
			"    * Two interface types are identical if they have the same set of methods\n" +
			"with the same names and identical function types. The order of the methods is\n" +
			"irrelevant.\n" +
			"    * Two map types are identical if they have identical key and value types.\n" +
			"    * Two channel types are identical if they have identical value types and\n" +
			"the same direction.\n",
		[]byte{
			0x5d, 0x00, 0x00, 0x10, 0x00, 0xe8, 0x05, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2a, 0x1d,
			0xc9, 0xe2, 0x03, 0x0c, 0x4e, 0x75, 0xc8, 0xee,
			0x65, 0x97, 0xae, 0x0a, 0x7b, 0x0a, 0x66, 0xfa,
			0x4a, 0x94, 0xe6, 0x32, 0x2f, 0x9b, 0x44, 0x8c,
			0xaf, 0xf6, 0xf4, 0x67, 0xe2, 0x8a, 0x8b, 0x2f,
			0x26, 0x41, 0xf6, 0x64, 0xe2, 0xf6, 0x00, 0xbb,
			0x1f, 0x4a, 0xb1, 0xca, 0xa6, 0xf0, 0x54, 0x9e,
			0xa5, 0x2d, 0xe7, 0x6e, 0x6a, 0x49, 0x84, 0xac,
			0xf5, 0x52, 0xcf, 0x57, 0xcd, 0xa7, 0x84, 0xe9,
			0x27, 0xdc, 0x3d, 0xf3, 0xf4, 0x1a, 0x8d, 0x98,
			0xba, 0xcc, 0x47, 0xca, 0x1b, 0x38, 0xa6, 0x8c,
			0x96, 0x24, 0x14, 0x8d, 0x1c, 0x81, 0x6b, 0xc0,
			0x1c, 0x69, 0xa1, 0x79, 0xa2, 0x20, 0x10, 0x6c,
			0x0c, 0x56, 0x40, 0x1f, 0x3f, 0x32, 0x4a, 0x4f,
			0x1f, 0x2a, 0x29, 0xf0, 0x23, 0x99, 0xbe, 0xc7,
			0x49, 0x7e, 0xc7, 0x60, 0xa3, 0x54, 0x9c, 0x06,
			0x06, 0xb4, 0xc4, 0x4d, 0x07, 0xd5, 0xef, 0xa0,
			0xcd, 0xac, 0x8f, 0x54, 0xf9, 0x15, 0x15, 0xa0,
			0xb5, 0xf1, 0x7d, 0xde, 0xeb, 0xe0, 0x41, 0x9a,
			0x3e, 0x5a, 0x66, 0xd2, 0xfc, 0x08, 0x00, 0xe8,
			0xf2, 0xc9, 0x04, 0x94, 0x7e, 0x0b, 0x06, 0x56,
			0x9d, 0xe7, 0x1f, 0x48, 0x21, 0x4d, 0x18, 0x21,
			0xa3, 0x5f, 0x69, 0x7a, 0xed, 0x4d, 0x2d, 0xb4,
			0x3e, 0x34, 0xcf, 0x9e, 0xae, 0x11, 0x2d, 0x15,
			0xbc, 0x2c, 0xd7, 0xef, 0x1c, 0xd5, 0xd8, 0xd3,
			0xc2, 0x4f, 0xcb, 0x2c, 0xb1, 0xa3, 0x85, 0x0a,
			0xad, 0x44, 0x68, 0x42, 0xab, 0x40, 0x58, 0xa2,
			0x9b, 0xdf, 0xbd, 0xa2, 0x10, 0xc7, 0x2e, 0x5a,
			0x1d, 0x07, 0xe4, 0xb5, 0xa6, 0xa8, 0xb3, 0x12,
			0x44, 0x64, 0x18, 0xaf, 0xa7, 0x72, 0x05, 0x29,
			0x48, 0xc6, 0x78, 0xef, 0x7f, 0x54, 0xe9, 0xb3,
			0xae, 0x10, 0x51, 0x11, 0x91, 0xf6, 0x96, 0x40,
			0x6a, 0xf1, 0xc8, 0x3d, 0x46, 0x86, 0x2e, 0xd3,
			0xa1, 0xdc, 0x38, 0x97, 0x11, 0x49, 0x4b, 0x03,
			0x91, 0xa5, 0xed, 0x53, 0x3f, 0xd5, 0x87, 0x04,
			0x25, 0xf0, 0xe2, 0x19, 0x55, 0x99, 0x6b, 0xad,
			0xa2, 0x9c, 0x7a, 0xcd, 0xc9, 0x34, 0x18, 0x8f,
			0x4c, 0xaa, 0xed, 0xd3, 0x49, 0x98, 0xd9, 0x67,
			0xbe, 0x41, 0x85, 0xbf, 0x09, 0x22, 0x16, 0xa0,
			0x81, 0x71, 0x38, 0xdb, 0xb2, 0xd6, 0xfe, 0x2b,
			0xf4, 0x03, 0xf8, 0xd1, 0x00, 0x16, 0x5e, 0x77,
			0x69, 0xd4, 0x28, 0x8f, 0x94, 0x4d, 0x58, 0x4f,
			0xae, 0x6b, 0xb1, 0x09, 0x85, 0x71, 0xd7, 0x3a,
			0x4b, 0xea, 0xd2, 0x70, 0xbb, 0xa2, 0x20, 0x85,
			0x2d, 0xbd, 0x57, 0x3f, 0x81, 0x3e, 0xe4, 0xa2,
			0x43, 0x3e, 0xee, 0x04, 0xbe, 0x42, 0x51, 0xaa,
			0xad, 0xda, 0x53, 0x87, 0xeb, 0xc0, 0x9a, 0xde,
			0xa2, 0x7e, 0x19, 0x4f, 0xa7, 0xdf, 0x23, 0x96,
			0xdd, 0xe3, 0x18, 0x0f, 0xc0, 0x48, 0xfd, 0x9f,
			0x13, 0xc3, 0x3c, 0x3a, 0x7b, 0x7c, 0xa8, 0x89,
			0x6a, 0xd6, 0x68, 0xdd, 0x3b, 0xc6, 0xc6, 0x44,
			0x79, 0xc5, 0x59, 0x1c, 0x23, 0xa9, 0x8b, 0x00,
			0x62, 0xeb, 0x3a, 0x51, 0x14, 0x5e, 0x63, 0x79,
			0x97, 0x60, 0x62, 0x13, 0x49, 0x41, 0x06, 0xf4,
			0x5f, 0xe7, 0x9b, 0xd6, 0x51, 0x31, 0x6e, 0x7d,
			0x5e, 0xe8, 0x72, 0xce, 0x5e, 0xd0, 0xa7, 0x9d,
			0xa3, 0xa9, 0x92, 0xa8, 0x2f, 0x78, 0x00, 0x92,
			0x4f, 0xf0, 0x21, 0xb6, 0x74, 0xfb, 0x3c, 0xe2,
			0x60, 0xdf, 0x82, 0x09, 0x07, 0xb3, 0x68, 0x5b,
			0xe2, 0x49, 0xeb, 0x81, 0x98, 0x23, 0x19, 0xdc,
			0x2c, 0xa1, 0xad, 0x95, 0xc0, 0x9a, 0x48, 0xcd,
			0xa8, 0xe4, 0xdf, 0xbb, 0xca, 0xd4, 0x47, 0x46,
			0xe2, 0xdb, 0x5c, 0xe2, 0xdb, 0xb2, 0x32, 0x44,
			0x87, 0x0a, 0x49, 0x99, 0xc6, 0x32, 0x75, 0xc6,
			0xe7, 0x1b, 0xf1, 0x71, 0x74, 0x32, 0x4a, 0x17,
			0x2f, 0xe8, 0x00,
		},
		nil,
	},
	{
		"text test with size == -1",
		3,
		false,
		"Two named types are identical if their type names originate in the same type\n" +
			"declaration (§Declarations and scope). A named and an unnamed type are never\n" +
			"identical. Two unnamed types are identical if the corresponding type literals\n" +
			"have the same literal structure and corresponding components have identical\n" +
			"types. In detail:\n" +
			"\n" +
			"    * Two array types are identical if they have identical element types and\n" +
			"the same array length.\n" +
			"    * Two slice types are identical if they have identical element types.\n" +
			"    * Two struct types are identical if they have the same sequence of fields,\n" +
			"and if corresponding fields have the same names and identical types. Two\n" +
			"anonymous fields are considered to have the same name.\n" +
			"    * Two pointer types are identical if they have identical base types.\n" +
			"    * Two function types are identical if they have the same number of\n" +
			"parameters and result values and if corresponding parameter and result types\n" +
			"are identical. All \"...\" parameters without a specified type are defined to\n" +
			"have identical type. All \"...\" parameters with specified identical type T are\n" +
			"defined to have identical type. Parameter and result names are not required to\n" +
			"match.\n" +
			"    * Two interface types are identical if they have the same set of methods\n" +
			"with the same names and identical function types. The order of the methods is\n" +
			"irrelevant.\n" +
			"    * Two map types are identical if they have identical key and value types.\n" +
			"    * Two channel types are identical if they have identical value types and\n" +
			"the same direction.\n",
		[]byte{
			0x5d, 0x00, 0x00, 0x10, 0x00, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x2a, 0x1d,
			0xc9, 0xe2, 0x03, 0x0c, 0x4e, 0x75, 0xc8, 0xee,
			0x65, 0x97, 0xae, 0x0a, 0x7b, 0x0a, 0x66, 0xfa,
			0x4a, 0x94, 0xe6, 0x32, 0x2f, 0x9b, 0x44, 0x8c,
			0xaf, 0xf6, 0xf4, 0x67, 0xe2, 0x8a, 0x8b, 0x2f,
			0x26, 0x41, 0xf6, 0x64, 0xe2, 0xf6, 0x00, 0xbb,
			0x1f, 0x4a, 0xb1, 0xca, 0xa6, 0xf0, 0x54, 0x9e,
			0xa5, 0x2d, 0xe7, 0x6e, 0x6a, 0x49, 0x84, 0xac,
			0xf5, 0x52, 0xcf, 0x57, 0xcd, 0xa7, 0x84, 0xe9,
			0x27, 0xdc, 0x3d, 0xf3, 0xf4, 0x1a, 0x8d, 0x98,
			0xba, 0xcc, 0x47, 0xca, 0x1b, 0x38, 0xa6, 0x8c,
			0x96, 0x24, 0x14, 0x8d, 0x1c, 0x81, 0x6b, 0xc0,
			0x1c, 0x69, 0xa1, 0x79, 0xa2, 0x20, 0x10, 0x6c,
			0x0c, 0x56, 0x40, 0x1f, 0x3f, 0x32, 0x4a, 0x4f,
			0x1f, 0x2a, 0x29, 0xf0, 0x23, 0x99, 0xbe, 0xc7,
			0x49, 0x7e, 0xc7, 0x60, 0xa3, 0x54, 0x9c, 0x06,
			0x06, 0xb4, 0xc4, 0x4d, 0x07, 0xd5, 0xef, 0xa0,
			0xcd, 0xac, 0x8f, 0x54, 0xf9, 0x15, 0x15, 0xa0,
			0xb5, 0xf1, 0x7d, 0xde, 0xeb, 0xe0, 0x41, 0x9a,
			0x3e, 0x5a, 0x66, 0xd2, 0xfc, 0x08, 0x00, 0xe8,
			0xf2, 0xc9, 0x04, 0x94, 0x7e, 0x0b, 0x06, 0x56,
			0x9d, 0xe7, 0x1f, 0x48, 0x21, 0x4d, 0x18, 0x21,
			0xa3, 0x5f, 0x69, 0x7a, 0xed, 0x4d, 0x2d, 0xb4,
			0x3e, 0x34, 0xcf, 0x9e, 0xae, 0x11, 0x2d, 0x15,
			0xbc, 0x2c, 0xd7, 0xef, 0x1c, 0xd5, 0xd8, 0xd3,
			0xc2, 0x4f, 0xcb, 0x2c, 0xb1, 0xa3, 0x85, 0x0a,
			0xad, 0x44, 0x68, 0x42, 0xab, 0x40, 0x58, 0xa2,
			0x9b, 0xdf, 0xbd, 0xa2, 0x10, 0xc7, 0x2e, 0x5a,
			0x1d, 0x07, 0xe4, 0xb5, 0xa6, 0xa8, 0xb3, 0x12,
			0x44, 0x64, 0x18, 0xaf, 0xa7, 0x72, 0x05, 0x29,
			0x48, 0xc6, 0x78, 0xef, 0x7f, 0x54, 0xe9, 0xb3,
			0xae, 0x10, 0x51, 0x11, 0x91, 0xf6, 0x96, 0x40,
			0x6a, 0xf1, 0xc8, 0x3d, 0x46, 0x86, 0x2e, 0xd3,
			0xa1, 0xdc, 0x38, 0x97, 0x11, 0x49, 0x4b, 0x03,
			0x91, 0xa5, 0xed, 0x53, 0x3f, 0xd5, 0x87, 0x04,
			0x25, 0xf0, 0xe2, 0x19, 0x55, 0x99, 0x6b, 0xad,
			0xa2, 0x9c, 0x7a, 0xcd, 0xc9, 0x34, 0x18, 0x8f,
			0x4c, 0xaa, 0xed, 0xd3, 0x49, 0x98, 0xd9, 0x67,
			0xbe, 0x41, 0x85, 0xbf, 0x09, 0x22, 0x16, 0xa0,
			0x81, 0x71, 0x38, 0xdb, 0xb2, 0xd6, 0xfe, 0x2b,
			0xf4, 0x03, 0xf8, 0xd1, 0x00, 0x16, 0x5e, 0x77,
			0x69, 0xd4, 0x28, 0x8f, 0x94, 0x4d, 0x58, 0x4f,
			0xae, 0x6b, 0xb1, 0x09, 0x85, 0x71, 0xd7, 0x3a,
			0x4b, 0xea, 0xd2, 0x70, 0xbb, 0xa2, 0x20, 0x85,
			0x2d, 0xbd, 0x57, 0x3f, 0x81, 0x3e, 0xe4, 0xa2,
			0x43, 0x3e, 0xee, 0x04, 0xbe, 0x42, 0x51, 0xaa,
			0xad, 0xda, 0x53, 0x87, 0xeb, 0xc0, 0x9a, 0xde,
			0xa2, 0x7e, 0x19, 0x4f, 0xa7, 0xdf, 0x23, 0x96,
			0xdd, 0xe3, 0x18, 0x0f, 0xc0, 0x48, 0xfd, 0x9f,
			0x13, 0xc3, 0x3c, 0x3a, 0x7b, 0x7c, 0xa8, 0x89,
			0x6a, 0xd6, 0x68, 0xdd, 0x3b, 0xc6, 0xc6, 0x44,
			0x79, 0xc5, 0x59, 0x1c, 0x23, 0xa9, 0x8b, 0x00,
			0x62, 0xeb, 0x3a, 0x51, 0x14, 0x5e, 0x63, 0x79,
			0x97, 0x60, 0x62, 0x13, 0x49, 0x41, 0x06, 0xf4,
			0x5f, 0xe7, 0x9b, 0xd6, 0x51, 0x31, 0x6e, 0x7d,
			0x5e, 0xe8, 0x72, 0xce, 0x5e, 0xd0, 0xa7, 0x9d,
			0xa3, 0xa9, 0x92, 0xa8, 0x2f, 0x78, 0x00, 0x92,
			0x4f, 0xf0, 0x21, 0xb6, 0x74, 0xfb, 0x3c, 0xe2,
			0x60, 0xdf, 0x82, 0x09, 0x07, 0xb3, 0x68, 0x5b,
			0xe2, 0x49, 0xeb, 0x81, 0x98, 0x23, 0x19, 0xdc,
			0x2c, 0xa1, 0xad, 0x95, 0xc0, 0x9a, 0x48, 0xcd,
			0xa8, 0xe4, 0xdf, 0xbb, 0xca, 0xd4, 0x47, 0x46,
			0xe2, 0xdb, 0x5c, 0xe2, 0xdb, 0xb2, 0x32, 0x44,
			0x87, 0x0a, 0x49, 0x99, 0xc6, 0x32, 0x75, 0xc6,
			0xe7, 0x1b, 0xf1, 0x71, 0x74, 0x32, 0x4a, 0x8d,
			0xf2, 0x0a, 0x64, 0xff, 0xe4, 0x0d, 0xf4, 0xa2,
		},
		nil,
	},
	{
		"hello world test with corrupt lc, lp, pb in header",
		3,
		true,
		"hello world\n",
		[]byte{
			0xfe, 0x00, 0x00, 0x10, 0x00, 0x0c, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x34, 0x19,
			0x49, 0xee, 0x8d, 0xe9, 0x17, 0x89, 0x3a, 0x33,
			0x5f, 0xfc, 0xac, 0xf7, 0x20, 0x00,
		},
		headerError,
	},
}

type lzmaBenchmark struct {
	descr string
	level int
	raw   []byte
	lzma  []byte
}

func readFile(filename string) []byte {
	file, err := os.ReadFile(filename)
	if err != nil {
		log.Fatalf("%v", err)
	}
	return file
}

var bench = lzmaBenchmark{
	descr: "text bench with size == -1",
	level: 3,
	raw:   readFile("data/data.txt"),
	lzma:  readFile("data/data.eos.l3.lzma"),
}
