/*
Copyright 2023 The Flux authors

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package awskms

import (
	"fmt"

	"github.com/aws/aws-sdk-go-v2/credentials"
	"sigs.k8s.io/yaml"
)

// LoadStaticCredentialsFromYAML parses the given YAML and returns a
// credentials.StaticCredentialsProvider that can be used to authenticate with
// AWS, or an error if the YAML could not be parsed.
func LoadStaticCredentialsFromYAML(b []byte) (credentials.StaticCredentialsProvider, error) {
	d := struct {
		AccessKeyID     string `json:"aws_access_key_id"`
		SecretAccessKey string `json:"aws_secret_access_key"`
		SessionToken    string `json:"aws_session_token"`
	}{}
	if err := yaml.Unmarshal(b, &d); err != nil {
		return credentials.StaticCredentialsProvider{}, fmt.Errorf("failed to unmarshal AWS credentials file: %w", err)
	}
	return credentials.NewStaticCredentialsProvider(d.AccessKeyID, d.SecretAccessKey, d.SessionToken), nil
}
