module imager_interfaces_none
  interface
    subroutine gdf_uv_size(huvl,nc)
      use image_def
      !
      type(gildas), intent(inout) :: huvl
      integer, intent(in) :: nc(2)
    end subroutine gdf_uv_size
  end interface
  !
  interface
    subroutine imager_write_data(head,data,error)
      use image_def
      use gkernel_types
      use gbl_message
      use clean_types
      !*
      !   Utility routine to write a Data Set in various formats.
      ! Supported formats are GILDAS, FITS, and also CSV fro 2-D Tables
      !!
      type(gildas), intent(inout) :: head  !! Data header
      real, intent(in) :: data(*)          !! Data buffer - Contiguous array
      logical, intent(inout) :: error      !! Logical error flag
    end subroutine imager_write_data
  end interface
  !
  interface
    subroutine table_to_csv(head,data,error)
      use image_def
      !*
      !   Utility routine
      !
      ! Write GILDAS Table to CSV file. Supports Real(8) and
      ! Real(4) tables.
      !!
      type(gildas), intent(inout) :: head
      !! Table Header
      real, intent(in) :: data(*)
      !! Data buffer -- contiguous, no type checking
      logical, intent(inout) :: error
    end subroutine table_to_csv
  end interface
  !
  interface
    subroutine table_to_csv4(nx,ny,file,data)
      !*
      !   Utility routine
      !
      ! Write Real(4) 2-D array Table to CSV file. 
      !!
      integer, intent(in) :: nx   !! First dimension
      integer, intent(in) :: ny   !! Second dimension
      character(len=*), intent(inout) :: file !! File name
      real, intent(in) :: data(nx,ny)   !! Array
    end subroutine table_to_csv4
  end interface
  !
  interface
    subroutine table_to_csv8(nx,ny,file,data)
      !*
      !   Utility routine
      !
      ! Write Real(8) 2-D array Table to CSV file. 
      !!
      integer, intent(in) :: nx   !! First dimension
      integer, intent(in) :: ny   !! Second dimension
      character(len=*), intent(inout) :: file !! File name
      real(kind=8), intent(in) :: data(nx,ny) !! Array
    end subroutine table_to_csv8
  end interface
  !
  interface
    subroutine uv_addnoise(line,comm,error)
      use gildas_def
      use gbl_message
      use clean_types
      use clean_arrays
      !---------------------------------------------------------------------
      !*
      ! IMAGER -- Support routine for command SIMULATE\ADD_NOISE
      ! 
      !   ADD_NOISE Value [Bandwidth [Wcol]] /FILE InputFile OutputFile
      !!
      !---------------------------------------------------------------------
      character(len=*), intent(in) :: line  !! Command line
      character(len=*), intent(in) :: comm  !! Command name
      logical, intent(inout) :: error       !! Error flag
    end subroutine uv_addnoise
  end interface
  !
  interface
    subroutine new_dirty_beam
      use clean_def
      use clean_arrays
      use clean_beams
      !
      !! IMAGER -- BEAM Utility  
      !! Called when a new dirty map is computed by command UV_MAP
      !
    end subroutine new_dirty_beam
  end interface
  !
  interface
    subroutine define_imager_vars (error)
      use gildas_def
      use clean_def
      use clean_arrays
      use clean_types
      use clean_default
      use clean_support
      use clean_beams
      use grid_control
      !----------------------------------------------------
      !! IMAGER -- Define internal SIC variables
      !----------------------------------------------------
      logical :: error    !! Logical error flag
    end subroutine  define_imager_vars
  end interface
  !
  interface
    subroutine big_wait(n,b)
      real, intent(inout) :: b
      integer, intent(in) :: n
    end subroutine big_wait
  end interface
  !
  interface
    subroutine uv_grid_comm(line,comm,error)
      use clean_arrays
      use clean_default
      !! Test routine -- not yet used
      character(len=*), intent(in) :: comm
      character(len=*), intent(in) :: line
      logical, intent(out) :: error
    end subroutine uv_grid_comm
  end interface
  !
  interface
    subroutine map_message_set_id(id)
      use gbl_message
      !---------------------------------------------------------------------
      !*
      ! IMAGER -- Message Utility  
      ! Alter library id into input id. Should be called by the library
      ! which wants to share its id with the current one.
      !!
      !---------------------------------------------------------------------
      integer, intent(in) :: id  !! Package ID
    end subroutine map_message_set_id
  end interface
  !
  interface
    subroutine message_attribute(cattr)
      use gbl_ansicodes
      !! IMAGER -- Message Utility -- Write color or bold modifier to terminal
      character(len=*), intent(in) :: cattr  !! Attribute modifier
    end subroutine message_attribute
  end interface
  !
  interface
    subroutine sub_modify_scale(head,rdata,rsize,scale)
      use image_def
      !
      !*
      !   IMAGER -- GILDAS Utility --  Change the Unit of data cube
      ! 
      !     Support routine for SPECIFY UNIT
      !!
      type(gildas), intent(inout) :: head   !! Cube Header
      integer(kind=index_length), intent(in) :: rsize  !! Data length
      real, intent(inout) :: rdata(rsize)   !! Data array
      real, intent(in) :: scale             !! Scale factor
    end subroutine sub_modify_scale
  end interface
  !
  interface
    subroutine sub_modify_blank(head,rdata,rsize,bval)
      use image_def
      !*
      !   IMAGER -- GILDAS Utility --  Change the Cube blanking value
      ! 
      !     Support routine for SPECIFY BLANKING
      !!
      type(gildas), intent(inout) :: head               !! Cube Header
      integer(kind=index_length), intent(in) :: rsize   !! Data length
      real, intent(inout) :: rdata(rsize)               !! Data array
      real, intent(in) :: bval                          !! New blanking
    end subroutine sub_modify_blank
  end interface
  !
  interface
    subroutine cct_fft_size(mx,my,nx,ny)
      !*
      ! IMAGER -- Utility routine 
      !   Get best FFT size given the data size
      !!
      integer, intent(in) :: mx,my   !! Data size
      integer, intent(out) :: nx,ny  !! FFT Size
    end subroutine cct_fft_size
  end interface
  !
  interface
    subroutine maxcmp(nx,ny,wl,a)
      !! Maximum of real part of a 2-D complex array
      integer, intent(in) :: nx,ny    !! Array size
      real, intent(in) :: wl(2,nx,ny) !! Complex array
      real, intent(out) :: a    !! Max of real part
    end subroutine maxcmp
  end interface
  !
  interface
    subroutine userc (nv,coords,xref,xval,xinc,yref,yval,yinc)
      integer, intent(in) :: nv                     !
      real(4), intent(out) :: coords(2,nv)          !
      real(8), intent(in) :: xref                   !
      real(8), intent(in) :: xval                   !
      real(8), intent(in) :: xinc                   !
      real(8), intent(in) :: yref                   !
      real(8), intent(in) :: yval                   !
      real(8), intent(in) :: yinc                   !
    end subroutine userc
  end interface
  !
  interface
    subroutine mos_primary (prim,primary,bsize)
      use image_def
      use gkernel_types
      !--------------------------------------------------------------
      !*
      ! IMAGER -- Support routine for MOSAIC
      !
      ! Mosaic and Primary beam support.
      ! Compute a Gaussian primary beam
      !!
      !--------------------------------------------------------------
      type(gildas), intent(in) :: prim         !! Primary beam header
      real, intent(inout) :: primary(prim%gil%dim(1),prim%gil%dim(2)) !! Values
      real, intent(in) :: bsize                !! Primary beam size in radian
    end subroutine mos_primary
  end interface
  !
  interface
    subroutine mos_inverse (n,weight,thre)
      !! IMAGER -- Returns the inverse for the weights for Mosaic
      integer, intent(in) :: n          !! Size of problem
      real, intent(inout) :: weight(n)  !! Weight to be inversed
      real, intent(in) :: thre          !! Threhsold
    end subroutine mos_inverse
  end interface
  !
  interface
    subroutine mos_addsq (n,nf,weight,lobe)
      !! IMAGER -- Increment weights by Square of primary beam
      integer, intent(in) :: n                      !! Size of arrays
      integer, intent(in) :: nf                     !! Number of fields
      real, intent(inout) :: weight(n)              !! Total weight
      real, intent(in) :: lobe(nf,n)                !! Primary beam array
    end subroutine mos_addsq
  end interface
  !
  interface
    subroutine mrc_clear
    end subroutine mrc_clear
  end interface
  !
  interface
    subroutine douvext(jc,nv,visi,jx,jy,umin,umax,vmin,vmax)
      !---------------------------------------------------------------------
      !*
      ! IMAGER -- Utility routine
      !
      !   Compute extrema of UV coverage. No assumption,
      !   but may return 0 if all Negative or all Positive
      !!
      !---------------------------------------------------------------------
      integer, intent(in) :: jc         !! Size of a visibility
      integer, intent(in) :: nv         !! Number of visibilities
      integer, intent(in) :: jx         !! X coordinate pointer
      integer, intent(in) :: jy         !! Y coordinate pointer
      real, intent(in) :: visi(jc,nv)   !! Visibilities
      real, intent(out) :: umin,umax    !! U extrema
      real, intent(out) :: vmin,vmax    !! V extrema
    end subroutine douvext
  end interface
  !
  interface
    subroutine uv_free_buffers
      use clean_def
      use clean_arrays
      !---------------------------------------------------------------------
      !*
      ! IMAGER -- Utility routine
      !
      !   Deallocate all UV buffers.
      !!
      !---------------------------------------------------------------------
    end subroutine uv_free_buffers
  end interface
  !
  interface
    subroutine uv_buffer_finduv(code) 
      use clean_arrays
      !! IMAGER -- Find code of current buffer 
      character(len=1), intent(out) :: code
    end subroutine uv_buffer_finduv
  end interface
  !
  interface
    subroutine uv_buffer_resetuv(code) 
      use clean_arrays
      !! IMAGER -- Reset current buffer to the specified code
      character(len=1), intent(in) :: code  !! Code (i,r,s or space) of current buffer
    end subroutine uv_buffer_resetuv
  end interface
  !
  interface
    subroutine discard_uvdata(error)
      use clean_def
      use clean_arrays
      use clean_default
      use clean_types
      !! IMAGER -- Discard all UV data
      logical, intent(inout) :: error
    end subroutine discard_uvdata
  end interface
  !
  interface
    subroutine set_buffer_modified(bufnam)
      use clean_types
      !! IMAGER -- Indicate the named buffer has been modified
      character(len=*), intent(in) :: bufnam  !! Buffer name 
    end subroutine set_buffer_modified
  end interface
  !
  interface
    function continuum_emg(x, amp, mu, sig, lamb)
      !! IMAGER -- Implementation of an Exponentially modified Gaussian function
      real(4) :: continuum_emg ! intent(out)
      real(4), intent(in) :: x
      ! Input value
      real(4), intent(in) :: amp 
      !! Amplitude of function
      real(4), intent(in) :: mu  
      !! Position
      real(4), intent(in) :: sig 
      !! Width
      real(4), intent(in) :: lamb 
    end function continuum_emg
  end interface
  !
  interface
    function continuum_gauss(x, amp, mu, sig)
      real(4) :: continuum_gauss ! intent(out)
      !! IMAGER -- Implementation of a Gaussian function 
      real(4), intent(in) :: x
      ! Input value
      real(4), intent(in) :: amp 
      !! Amplitude of function
      real(4), intent(in) :: mu  
      !! Position
      real(4), intent(in) :: sig 
    end function continuum_gauss
  end interface
  !
  interface
    subroutine parabola(x,y,a,b,c) 
      !----------------------------------------------------------------------
      !!	Solve for Y = a + b*x() + c*x()^2
      !----------------------------------------------------------------------
      real, intent(in) :: x(3)
      real, intent(in) :: y(3)
      real, intent(out) :: a,b,c
    end subroutine parabola
  end interface
  !
  interface
    subroutine diagonal(m,n,r,c)
      integer :: m                      !! First dimension
      integer :: n                      !! Second dimension (=< M)
      real(kind=8) :: r(m,n)            !! Matrix
      real(kind=8) :: c(n)              !! Diagonal vector
    end subroutine diagonal
  end interface
  !
  interface
    subroutine fitfcn(iflag,m,nvpar,x,f,fjac,ljc)
      use gildas_def
      use uvfit_data
      use gbl_message
      !$ use omp_lib
      !---------------------------------------------------------------------
      !! IMAGER -- Support routine for UV_FIT  
      !! FITFCN is called by DNLS1E
      !---------------------------------------------------------------------
      integer, intent(in) :: iflag          !! Code for print out
      integer, intent(in) :: m              !! Number of data points
      integer, intent(in) :: nvpar          !! Number of variables
      real(8), intent(in) :: x(nvpar)       !! Variable parameters
      real(8), intent(out) :: f(m)          !! Function value at each point
      integer, intent(in) :: ljc            !! First dimension of FJAC
      real(8), intent(out) :: fjac(ljc,nvpar) !! Partial derivatives
    end subroutine fitfcn
  end interface
  !
  interface
    subroutine getvisi(n,uvriw,k,u,v,r,i,w)
      !---------------------------------------------------------------------
      !*
      ! IMAGER --  Support routine for UV_FIT
      !
      !     Obtain U, V, and the visibility from the array UVRIW.
      !!
      !---------------------------------------------------------------------
      integer, intent(in) :: n         !! Number of visibilities
      real, intent(in) :: uvriw(5,n)   !! Visibilities
      integer, intent(in) :: k         !! Desired visibility number
      real(8), intent(out) ::  u       !! U coordinate
      real(8), intent(out) ::  v       !! V coordinate
      real(8), intent(out) ::  r       !! Real part
      real(8), intent(out) ::  i       !! Imaginary part
      real(8), intent(out) ::  w       !! Weight
    end subroutine getvisi
  end interface
  !
  interface
    subroutine outpar(id,kfunc,pars,errs)
      !---------------------------------------------------------------------
      !*
      ! IMAGER -- Support routine for UV_FIT
      !
      ! Process fitted parameters to be saved, according to fitted functions
      !!
      !---------------------------------------------------------------------
      integer, intent(in) :: id    !! Function code
      integer, intent(in) :: kfunc !! Number of parameters
      real(8), intent(inout) :: pars(kfunc)  !! Parameters
      real(8), intent(inout) :: errs(kfunc)  !! Uncertainties
    end subroutine outpar
  end interface
  !
  interface
    function zant(i,r)
      !*
      ! Return the apparent number of Antenna #i when reference
      ! antenna is #r. Since #r is not in the list of antennas for
      ! which a solution is to be searched, this number
      ! is #i for #i < #r, and #i-1 for #i > #r
      !!
      integer :: zant                   ! intent(out)
      integer, intent(in) :: i  !! Apparent number of antenna #i
      integer, intent(in) :: r  !! Reference antenna number
    end function zant
  end interface
  !
  interface
    subroutine mth_dpotrf (name, uplo, n, a, lda, error)
      character(len=*) :: name           !
      character(len=*) :: uplo           !
      integer :: n                       !
      integer :: lda                     !
      real(8) :: a(lda,*)                !
      logical :: error                   !
    end subroutine mth_dpotrf
  end interface
  !
  interface
    subroutine mth_dpotrs (name, uplo, n, nrhs, a, lda, b, ldb, info )
      character(len=*) :: name           !
      character(len=*) :: uplo           !
      integer :: n                       !
      integer :: nrhs                    !
      integer :: lda                     !
      real(8) :: a(lda,*)                !
      integer :: ldb                     !
      real(8) :: b(ldb,*)                !
      integer :: info
      !
      !  Purpose
      !  =======
      !
      !  DPOTRS solves a system of linear equations A*X = B with a symmetric
      !  positive definite matrix A using the Cholesky factorization
      !  A = U**T*U or A = L*L**T computed by DPOTRF.
      !
      !  Arguments
      !  =========
      !
      !  UPLO    (input) CHARACTER*1
      !          = 'U':  Upper triangle of A is stored;
      !          = 'L':  Lower triangle of A is stored.
      !
      !  N       (input) INTEGER
      !          The order of the matrix A.  N >= 0.
      !
      !  NRHS    (input) INTEGER
      !          The number of right hand sides, i.e., the number of columns
      !          of the matrix B.  NRHS >= 0.
      !
      !  A       (input) DOUBLE PRECISION array, dimension (LDA,N)
      !          The triangular factor U or L from the Cholesky factorization
      !          A = U**T*U or A = L*L**T, as computed by DPOTRF.
      !
      !  LDA     (input) INTEGER
      !          The leading dimension of the array A.  LDA >= max(1,N).
      !
      !  B       (input/output) DOUBLE PRECISION array, dimension (LDB,NRHS)
      !          On entry, the right hand side matrix B.
      !          On exit, the solution matrix X.
      !
      !  LDB     (input) INTEGER
      !          The leading dimension of the array B.  LDB >= max(1,N).
      !
      !  INFO    (output) INTEGER
      !          = 0:  successful exit
      !          < 0:  if INFO = -i, the i-th argument had an illegal value
      !
      logical :: error
      !
      ! Call LAPACK routine
    end subroutine mth_dpotrs
  end interface
  !
  interface
    subroutine mth_dpbtrf (name, uplo, n, kd, ab, ldab, error)
      character(len=*) :: name             !
      character(len=*) :: uplo             !
      integer :: n                         !
      integer :: kd                        !
      integer :: ldab                      !
      real(8) :: ab(ldab,*)                !
      logical :: error                     !
    end subroutine mth_dpbtrf
  end interface
  !
  interface
    subroutine mth_dpbtrs (name, uplo, n, kd, nrhs, ab, ldab, b, ldb, error)
      character(len=*) :: name             !
      character(len=*) :: uplo             !
      integer :: n                         !
      integer :: kd                        !
      integer :: nrhs                      !
      integer :: ldab                      !
      real(8) :: ab(ldab,*)                !
      integer :: ldb                       !
      real(8) :: b(ldb,*)                  !
      logical :: error                     !
    end subroutine mth_dpbtrs
  end interface
  !
  interface
    subroutine mth_fail (fac,prog,ifail,error)
      use gbl_message
      !---------------------------------------------------------------------
      ! Error handling routine
      !---------------------------------------------------------------------
      character(len=*) :: fac           !
      character(len=*) :: prog          !
      integer :: ifail                  !
      logical :: error                  !
    end subroutine mth_fail
  end interface
  !
  interface
    subroutine overlap(a,n,b,m,c,k)
      !
      ! Find the intersection of ensembles A and B
      !
      integer, intent(in) :: n        !! Size of Ensemble A
      integer, intent(in) :: a(n)     !! First Ensemble
      integer, intent(in) :: m        !! Size of Ensemble B
      integer, intent(in) :: b(m)     !! Second Ensemble
      integer, intent(inout) :: k     !! Size of Intersection
      integer, intent(inout) :: c(*)  !! Ensemble Intersection
    end subroutine overlap
  end interface
  !
  interface
    subroutine union(a,n,b,m,c,k)
      !
      !! Find the Union of ensembles A and B
      !
      integer, intent(in) :: n        !! Size of Ensemble A
      integer, intent(in) :: a(n)     !! First Ensemble
      integer, intent(in) :: m        !! Size of Ensemble B
      integer, intent(in) :: b(m)     !! Second Ensemble
      integer, intent(inout) :: k     !! Size of Union
      integer, intent(inout) :: c(*)  !! Ensemble Union
    end subroutine union
  end interface
  !
  interface
    subroutine do_fft (nx,ny,nf,fft,work)
      !$ use omp_lib
      !! IMAGER -- Compute a Cube from its FFT
      !
      integer, intent(in)  :: nx                       !! X size
      integer, intent(in)  :: ny                       !! Y size
      integer, intent(in)  :: nf                       !! Z size
      complex, intent(inout) :: fft(nx,ny,nf)          !! FFT cube
      real, intent(inout)  :: work(2*max(nx,ny))       !! Work space
    end subroutine do_fft
  end interface
  !
  interface
    subroutine uvdebug_pointers(name,text,huv)
      use image_def
      use gbl_message
      !! IMAGER - Debug the Mosaic offset pointers
      character(len=*), intent(in) :: name    !! Caller name
      character(len=*), intent(in) :: text    !! Buffers name
      type (gildas), intent(in) :: huv        !! UV header
    end subroutine uvdebug_pointers
  end interface
  !
  interface
    subroutine uvshort_dotrans (a,b,n,m)
      !
      ! Utility routine -- Output table "b" is table "a" transposed in line/column order
      !
      integer, intent(in) :: n,m
      real, intent(in) :: a(n,m)
      real, intent(out) ::  b(m,n)
    end subroutine uvshort_dotrans
  end interface
  !
  interface
    subroutine uvshort_prmult(z,f,nx,ny)
      !*
      ! IMAGER -- Support for UV_SHORT
      !
      ! Update z(nx,ny) table with z*f, calculate in uv plane
      ! Used to multiply by interferometer primary beam
      !
      integer, intent(in) :: nx, ny           !! Problem size
      complex, intent(inout) :: z(nx, ny)     !! Complex values
      real, intent(in) :: f(nx,ny)            !! Multiplication factors 
    end subroutine uvshort_prmult
  end interface
  !
  interface
    subroutine uvshort_dopoint(sdt,data,nd,np,xcol,ycol,old,new)
      use image_def
      use gkernel_types
      !---------------------------------------------------------------------
      !*
      ! IMAGER  --  Support routine for command UV_SHORT
      !
      ! Recompute data(nd,np) xcol and ycol coordinates values
      ! in case of changing reference position a0 d0 from old to new
      !!
      !---------------------------------------------------------------------
      type(gildas), intent(in) :: sdt      !! Single Dish Table
      integer, intent(in) :: nd            !! Size of a spectrum
      integer, intent(in) :: np            !! Number of spectra
      integer, intent(in) :: xcol          !! RA offset pointer
      integer, intent(in) :: ycol          !! Dec offset pointer
      real, intent(inout) :: data(nd,np)   !! Spectra
      real(8), intent(in) :: old(2)        !! Old RA and Dec center
      real(8), intent(in) :: new(2)        !! New Ra and Dec center
    end subroutine uvshort_dopoint
  end interface
  !
  interface
    subroutine uvfit_comm(line,error)
      use gbl_message
      use clean_default
      use clean_arrays
      !---------------------------------------------------------------------
      !*
      ! IMAGER -- Support for command UV_FIT   
      !   UV_FIT [Func1 ... FuncN] [/QUIET] 
      !     [/WIDGET Nfunc] [/SAVE File] [/RESULT]
      !     [/UVRANGE Min [Max]]
      !     [/CONTINUUM [Field] [/INDEX Sp] [/RANGE Min Max TYPE]]
      !---------------------------------------------------------------------
      character(len=*), intent(inout) :: line !! Command line
      logical, intent(inout) :: error         !! Error flag
    end subroutine uvfit_comm
  end interface
  !
  interface
    subroutine  uvfit_model(ifunc,kfunc,uu,vv,x,y,dy)
      !---------------------------------------------------------------------
      !*
      ! IMAGER --  Support routine for UV_FIT
      !
      ! Compute a model function.
      !!
      !---------------------------------------------------------------------
      integer, intent(in) :: ifunc       !! Fuction type
      integer, intent(in) :: kfunc       !! Number of parameters
      real(8), intent(in) :: uu          !! U coordinate in arcsec^-1
      real(8), intent(in) :: vv          !! V coordinate in arcsec^-1
      real(8), intent(in) :: x(kfunc)    !! Parameters
      real(8), intent(inout) :: y(2)        !! Function (real, imag.)
      real(8), intent(inout) :: dy(2,kfunc) !! Partial derivatives.
    end subroutine uvfit_model
  end interface
  !
  interface
    function z_exp(x)
      !---------------------------------------------------------------------
      ! protect again underflows in exp(x)
      !---------------------------------------------------------------------
      real(8) :: z_exp                   !! Underflowed Exponential
      real(8) :: x                       !! Argument    
    end function z_exp
  end interface
  !
  interface
    subroutine sub_uvfit_results(line,error)
      use uvfit_data
      use gbl_message
      !---------------------------------------------------------------------
      !* 
      ! IMAGER --  Support for command 
      !     UV_FIT [Func1 ... FuncN] /RESULT
      !
      ! Only works for  
      !  - all functions  
      !  - 1 channel only  
      ! Test functionality only. May disappear at any time...
      !---------------------------------------------------------------------
      character(len=*), intent(in) :: line  !! Command line
      logical, intent(inout) :: error       !! Error flag
    end subroutine sub_uvfit_results
  end interface
  !
  interface
    subroutine get_uvfit_model(nvisi,uu,vv,rr,ii,error)
      use uvfit_data
      !$ use omp_lib
      !---------------------------------------------------------------------
      !* 
      ! IMAGER -- Support for command   
      !     UV_FIT [Func1 ... FuncN] /RESULT
      !!
      !---------------------------------------------------------------------
      integer, intent(in) :: nvisi    !! Number of visibilities
      real, intent(in) :: uu(nvisi)   !! U coordinates
      real, intent(in) :: vv(nvisi)   !! V coordinates
      real, intent(out) :: rr(nvisi)  !! Real part
      real, intent(out) :: ii(nvisi)  !! Imaginary part
      logical, intent(inout) :: error !! Error flag
    end subroutine get_uvfit_model
  end interface
  !
  interface
    subroutine moments_comm(line,error)
      use gkernel_types
      use iso_c_binding
      use clean_arrays
      use moment_arrays
      use gbl_message
      !---------------------------------------------------------------------
      !*
      ! IMAGER -- Support for command MOMENTS  
      !       MOMENTS  [SMOOTH]  
      !         [/MASK]  
      !         [/METHOD Mean Peak Parabolic]  
      !         [/RANGE Min Max TYPE] [/THRESHOLD Min]  
      ! or  
      !         MOMENTS /CUTS Threshold [Regions]  
      !!
      !---------------------------------------------------------------------
      character(len=*), intent(in) :: line  !! Command line
      logical, intent(inout) :: error       !! Logical error flag
    end subroutine moments_comm
  end interface
  !
  interface
    subroutine moments_cuts(line,error)
      use clean_arrays
      use moment_arrays
      use gbl_message
      !---------------------------------------------------------------------
      !*
      ! IMAGER  -- Support for command MOMENTS /CUTS Threshold [Regions]
      !!
      !---------------------------------------------------------------------
      character(len=*), intent(in) :: line  !! Command line
      logical, intent(inout) :: error       !! Logical error flag
    end subroutine moments_cuts
  end interface
  !
  interface
    subroutine sub_moments(horigin,line,error)
      use clean_arrays
      use moment_arrays
      use gbl_message
      !---------------------------------------------------------------------
      !*
      ! IMAGER -- Support for command  
      !       MOMENT  
      !         [/MASK]  
      !         [/METHOD Mean Peak Parabolic]  
      !         [/RANGE Min Max TYPE] [/THRESHOLD Min]  
      !
      !	A dedicated routine to compute the moment of an input data cube.
      ! Produces any or all of 4 following images:  
      !           1) the integrated intensity map  
      !           2) the Peak brightness  
      !           3) The mean velocity  
      !           4) The line width  
      !	Method:  
      !     AREA      Integrated intensity in Velocity range  
      !     PEAK      Peak intensity in Velocity range  
      !     VELOCITY  Velocity of peak    
      !     WIDTH     2nd moment: line width
      !
      ! The peak intensity is obtained either as a per-channel basis,
      ! or a parabolic fit around the maximum. The Velocity is obtained
      ! in a similar way, or intensity weighted.
      !     
      !        For each spectrum, make a gate in velocity and clip based on
      !       intensity. Sum up weighted velocity and mean intensity.
      !!
      !---------------------------------------------------------------------
      type(gildas), intent(in) :: horigin   !! Input image, CLEAN or SKY
      character(len=*), intent(in) :: line  !! Command line
      logical, intent(inout) :: error       !! Logical error flag
    end subroutine sub_moments
  end interface
  !
  interface
    subroutine set_velo(nv,av,ref,val,inc)
      use gildas_def
      !! IMAGER -- Utility routine : Compute velocity sampling
      integer(kind=index_length), intent(in)  :: nv  !! Number of channels
      real(kind=4), intent(out) :: av(nv)       !! Velocities
      real(kind=8), intent(in)  :: ref,val,inc  !! Reference, Value and Increment
    end subroutine set_velo
  end interface
  !
  interface
    subroutine catalog_init(imol,error)
      !
      !! IMAGER --  Initialize or Update catalog variables
      integer, intent(inout) :: imol  !! Number of molecules
      logical, intent(out) :: error   !! Logical error flag
    end subroutine catalog_init
  end interface
  !
  interface
    subroutine uvsort_check_fields(hiuv,ioff,joff,sorted)
      use image_def
      !
      type(gildas), intent(in) :: hiuv  !! Input UV header
      integer, intent(in) :: ioff       !! X column pointer
      integer, intent(in) :: joff       !! Y column pointer
      logical, intent(out) :: sorted    !! Is data sorted ?
    end subroutine uvsort_check_fields
  end interface
  !
  interface
    subroutine big_stupid_uvsort(uvdata,uvtri,type,error)
      use image_def
      use gbl_message
      !----------------------------------------------------------------------
      ! IMAGER -- Sort an input UV table
      !
      !       Add CC code to implement precession
      !----------------------------------------------------------------------
      character(len=*), intent(in) :: type
      character(len=*), intent(in) :: uvdata,uvtri
      logical, intent(out) :: error
    end subroutine big_stupid_uvsort
  end interface
  !
  interface
    subroutine color_comm(line,error)
      !---------------------------------------------------------------------
      !*
      ! IMAGER -- Support for command COLOR Span HueMin HueMax
      !
      !   calls "@ p_color.ima" to derive a lut
      !   with unsaturated colours up to Span around 0,
      !   and a Hue range between Min and Max 
      !!
      !---------------------------------------------------------------------
      character(len=*), intent(in) :: line
      logical, intent(inout) :: error
    end subroutine color_comm
  end interface
  !
  interface
    subroutine howto_comm(line,error,icode)
      use gbl_message
      !---------------------------------------------------------------------
      !! IMAGER -- Support for command HOW_TO
      !---------------------------------------------------------------------
      character(len=*), intent(inout) :: line  !! Command line
      logical, intent(out) :: error            !! Logical error flag
      integer, intent(inout) :: icode  !! Operation code: 0 examine ambiguities, 1 use unambiguous result
    end subroutine howto_comm
  end interface
  !
  interface
    subroutine extract_real (c,mx,my,r,nx,ny)  
      !-----------------------------------------------------------------
      !!     Extract a Real array from a larger Complex array
      !! while shuffling the quarters 
      !-----------------------------------------------------------------
      integer nx,ny                        !! size of input array
      real r(nx,ny)                        !! input real array
      integer mx,my                        !! size of output array
      complex c(mx,my)                     !! output complex array
    end subroutine extract_real
  end interface
  !
  interface
    subroutine scale_factor(nx,ny,cbig,chigh,rmask,fscale,flux) 
      !*
      !  Compute best fit scaling factor for overlapping regions
      ! of visibilities.
      !
      integer, intent(in) :: nx,ny
      !! FFT of image size
      complex, intent(in) :: cbig(nx,ny)
      !! First FFT
      complex, intent(in) :: chigh(nx,ny)
      !! Second FFT
      real, intent(in) :: rmask(nx,ny)
      !! Masked regions
      real, intent(out) :: fscale
      !! Scale factor
      real, intent(out) :: flux
    end subroutine scale_factor
  end interface
  !
  interface
    subroutine init_display 
      !! Initialize the DISPLAY language from VIEWER
    end subroutine init_display
  end interface
  !
  interface
    subroutine save_result(error)
      use gildas_def
      use clean_types
      use gbl_message
      !-------------------------------------------------------------------
      !! IMAGER - Exit routine: ask user whether to save data or not
      !-------------------------------------------------------------------
      logical error !! Logical error flag
    end subroutine save_result
  end interface
  !
  interface
    subroutine imager_version
      use gbl_ansicodes
      use gkernel_types
      !
      !! IMAGER -- Test last IMAGER usage
      !
    end subroutine imager_version
  end interface
  !
  interface
    subroutine get_sys_code(in,out,sys_code,error)
      use gbl_format
      use image_def
      use gbl_message
      !---------------------------------------------------------------------
      !*
      ! GILDAS  Utility
      !
      ! Get old-system to new-system conversion code.
      ! Trap illegal/impossible conversions.
      !
      ! Caution: ICRS not yet supported here
      !---------------------------------------------------------------------
      type(gildas),    intent(in)  :: in          !! Input header
      type(gildas),    intent(in)  :: out         !! Desired output header
      integer(kind=4), intent(out) :: sys_code    !! Conversion code
      logical,         intent(out) :: error       !! Logical error flag
    end subroutine get_sys_code
  end interface
  !
  interface
    subroutine combine_comm(line,error)
      !! IMAGER -- Dispatch routine for COMBINE
      character(len=*), intent(in) :: line
      logical, intent(inout) :: error
    end subroutine combine_comm
  end interface
  !
  interface
    subroutine map_combine_comm(line,error)
      !! IMAGER -- Dispatch routine for MAP_COMBINE
      character(len=*), intent(in) :: line
      logical, intent(inout) :: error
    end subroutine map_combine_comm
  end interface
  !
  interface
    subroutine sub_combine(hx,hy,hz,n,m,dx,dy,dz,b,iblank,code,ay,ty,az,tz,c,error)
      use gkernel_types
      !----------------------------------------------------------------------
      !*
      ! IMAGER  -- Support routine for commands  
      ! COMBINE Out CODE In1 In2 /FACTOR A1 A2 /THRESHOLD T1 T2 /BLANK Bval
      !!
      !----------------------------------------------------------------------
      character(len=*), intent(inout) :: code    !! Combination code
      integer(kind=index_length), intent(in) :: n  !! Size of image
      integer(kind=index_length), intent(in) :: m  !! Number of channels
      type(gildas), intent(inout) :: hx    !! Output header
      type(gildas), intent(in) :: hy    !! Input Cube header
      type(gildas), intent(in) :: hz    !! Input Cube or Plane header
      !
      real, intent(in) :: dx(n,m)       !! Output data
      real, intent(in) :: dy(n,m)       !! Input data
      real, intent(in) :: dz(n)         !! Input data
      real, intent(in) :: c             !! Offset
      real, intent(in) :: ay            !! Y Factor
      real, intent(in) :: ty            !! Y Threshold
      real, intent(in) :: az            !! Z Factor
      real, intent(in) :: tz            !! Z Threshold
      real, intent(in) :: b  !! Output blanking
      integer(kind=index_length), intent(inout) :: iblank  !! Number of blanked pixels
      logical, intent(out) :: error  ! Logical error flag
    end subroutine sub_combine
  end interface
  !
  interface
    subroutine add002(z,y,x,n,m,bz,ez,az,tz,by,ey,ay,ty,bx,c,iblank)
      !$ use omp_lib
      use gkernel_types
      !---------------------------------------------------------------------
      !! GDF	-- Internal routine  
      !!	  Linear combination of input arrays  
      !!	X = Ay*Y + Az*Z + C 
      !---------------------------------------------------------------------
      integer(kind=index_length), intent(in) :: n
      integer(kind=index_length), intent(in) :: m
      real :: z(n)                      !! Plane array
      real :: y(n,m)                    !! Input Cube
      real :: x(n,m)                    !! Output Cube
      real :: bz                        !! Z Blanking value
      real :: ez                        !! Z Tolerance on blanking
      real :: az                        !! Z factor
      real :: tz                        !! Z threshold
      real :: by                        !! Y Blanking value
      real :: ey                        !! Y Tolerance on blanking
      real :: ay                        !! Y factor
      real :: ty                        !! Y Threshold
      real :: bx                        !! X Blanking
      real :: c                         !! Constant Offset
      integer(kind=index_length), intent(inout) :: iblank !! Number of bad pixels
    end subroutine add002
  end interface
  !
  interface
    subroutine div002(z,y,x,n,m,bz,ez,az,tz,by,ey,ay,ty,bx,c,iblank)
      !$ use omp_lib
      use gkernel_types
      !---------------------------------------------------------------------
      !! GDF	-- Internal routine
      !!	  Division of 2 input arrays  
      !!	X = Ay*Y / Az*Z + C  
      !---------------------------------------------------------------------
      integer(kind=index_length), intent(in) :: n
      integer(kind=index_length), intent(in) :: m
      real :: z(n)                      !! Plane array
      real :: y(n,m)                    !! Input Cube
      real :: x(n,m)                    !! Output Cube
      real :: bz                        !! Z Blanking value
      real :: ez                        !! Z Tolerance on blanking
      real :: az                        !! Z factor
      real :: tz                        !! Z threshold
      real :: by                        !! Y Blanking value
      real :: ey                        !! Y Tolerance on blanking
      real :: ay                        !! Y factor
      real :: ty                        !! Y Threshold
      real :: bx                        !! X Blanking
      real :: c                         !! Constant Offset
      integer(kind=index_length), intent(inout) :: iblank !! Number of bad pixels
    end subroutine div002
  end interface
  !
  interface
    subroutine mul002(z,y,x,n,m,bz,ez,az,tz,by,ey,ay,ty,bx,c,iblank)
      !$ use omp_lib
      use gkernel_types
      !---------------------------------------------------------------------
      !! GDF	-- Internal routine  
      !!	Multiplication of input arrays  
      !!  X = Ay*Y * Az*Z + C
      !---------------------------------------------------------------------
      integer(kind=index_length), intent(in) :: n
      integer(kind=index_length), intent(in) :: m
      real :: z(n)                      !! Plane array
      real :: y(n,m)                    !! Input Cube
      real :: x(n,m)                    !! Output Cube
      real :: bz                        !! Z Blanking value
      real :: ez                        !! Z Tolerance on blanking
      real :: az                        !! Z factor
      real :: tz                        !! Z threshold
      real :: by                        !! Y Blanking value
      real :: ey                        !! Y Tolerance on blanking
      real :: ay                        !! Y factor
      real :: ty                        !! Y Threshold
      real :: bx                        !! X Blanking
      real :: c                         !! Constant Offset
      integer(kind=index_length), intent(inout) :: iblank !! Number of bad pixels
    end subroutine mul002
  end interface
  !
  interface
    subroutine opt002(z,y,x,n,m,bz,ez,az,tz,by,ey,ay,ty,bx,c,iblank)
      !$ use omp_lib
      use gkernel_types
      !---------------------------------------------------------------------
      !! GDF --	Internal routine  
      !!	Optical depth from input arrays  
      !!	X = - LOG ( Ay*Y / Az*Z + C )  
      !---------------------------------------------------------------------
      integer(kind=index_length), intent(in) :: n
      integer(kind=index_length), intent(in) :: m
      real :: z(n)                      !! Plane array
      real :: y(n,m)                    !! Input Cube
      real :: x(n,m)                    !! Output Cube
      real :: bz                        !! Z Blanking value
      real :: ez                        !! Z Tolerance on blanking
      real :: az                        !! Z factor
      real :: tz                        !! Z threshold
      real :: by                        !! Y Blanking value
      real :: ey                        !! Y Tolerance on blanking
      real :: ay                        !! Y factor
      real :: ty                        !! Y Threshold
      real :: bx                        !! X Blanking
      real :: c                         !! Constant Offset
      integer(kind=index_length), intent(inout) :: iblank !! Number of bad pixels
    end subroutine opt002
  end interface
  !
  interface
    subroutine explore_comm(line,comm,error)
      use gkernel_types
      use clean_default
      use gbl_message
      !---------------------------------------------------------------------
      !*
      ! VIEWER -- Support for command  
      ! EXPLORE [DataCube] [/ADD Ux Uy [Ibox]] [/NOWAIT]
      !
      !    Call the "explore" scripts that allow to display from a data cube:  
      ! - a 2-D image (channel or integrated area)  
      ! - spectra at selected positions around it  
      !
      ! The display is organized in a Keypad layout  
      !     7 8 9  
      !     4 5 6  
      !     1 2 3  
      !!
      !---------------------------------------------------------------------
      character(len=*), intent(in) :: line  !! Command line
      character(len=*), intent(in) :: comm  !! Command name
      logical, intent(out) :: error  !! Logical error flag
    end subroutine explore_comm
  end interface
  !
  interface
    subroutine dump_memory(error)
      use clean_default
      use gbl_message
      !
      !*
      ! IMAGER -- Support for DUMP MEMORY: Printout memory usage.
      !
      ! There is no generic code to retrieve the memory use.
      !
      ! On Linux, one could scan directly the /proc/self/status   file.
      !
      ! Most efficient would be to call the C Posix standard "getrusage"
      ! but that only returns the Maximum used memory, not the current one.
      ! 
      ! Other methods are system dependent, and require to fork a child, 
      ! which duplicates the memory content and can lead to Crashes !...
      !!
      logical, intent(out) :: error  !! Logical error flag
      interface
        subroutine getSize(val) bind(c, name='getSize')
          import :: c_long
          integer(kind=c_long) :: val
        end
      end interface
    end subroutine dump_memory
  end interface
  !
  interface
    subroutine sub_cct_collect(sx,sy,sz,ncx,ncy,ncz,ny,nz,x,y,z)
      use gildas_def
      use clean_def
      !! Concatenate CCT Tables
      integer, intent(in) :: sx  !! Size of a Clean Component (3 or 4)
      integer, intent(in) :: sy  !! Size of Y Clean component
      integer, intent(in) :: sz  !! Size of Z Clean component
      integer(kind=address_length), intent(in) :: ncx  !! Number of channels in X (output) CCT table
      integer(kind=address_length), intent(in) :: ncy  !! Number of channels in Y CCT Table
      integer(kind=address_length), intent(in) :: ncz  !! Number of channels in Z CCT Table
      integer(kind=address_length), intent(in) :: ny   !! Number of components in Y table
      integer(kind=address_length), intent(in) :: nz   !! Number of compoents in Z table
      real, intent(in) :: y(sy,ncy,ny)  !! Y CCT table
      real, intent(in) :: z(sz,ncz,nz)  !! Z CCT table
      real, intent(out) :: x(sx,ncx,ny+nz)  !! Combined CCT table
    end subroutine sub_cct_collect
  end interface
  !
  interface
    function mth_bessj1 (x)
      !-------------------------------------------------------------------
      !! Compute Bessel function J1
      !-------------------------------------------------------------------
      real(8) :: mth_bessj1       
      real(8), intent(in) :: x 
    end function mth_bessj1
  end interface
  !
  interface
    subroutine imager_pack_set(pack)
      use gpack_def
      !! Declare the IMAGER package
      type(gpack_info_t), intent(out) :: pack
    end subroutine imager_pack_set
  end interface
  !
  interface
    subroutine imager_pack_init(gpack_id,error)
      use sic_def
      !! Initialize the IMAGER program
      integer :: gpack_id   !! Package ID
      logical :: error      !! Logical error flag
    end subroutine imager_pack_init
  end interface
  !
  interface
    subroutine imager_pack_on_exit(error)
      !! Clean up buffers and exit
      logical :: error    !! Logical error flag
    end subroutine imager_pack_on_exit
  end interface
  !
  interface
    subroutine omp_setup
      use omp_control
      !$ use omp_lib
      !
    end subroutine omp_setup
  end interface
  !
  interface
    function ompget_debug_code()
      use omp_control
      logical :: ompget_debug_code    ! Intent(out)
    end function ompget_debug_code
  end interface
  !
  interface
    subroutine cdaten(cdate,ndate,error)
      use gbl_message
      !---------------------------------------------------------------------
      !  Convert the date '01-JAN-1984' in integers for year, month, and day
      !---------------------------------------------------------------------
      character(len=*), intent(in)    :: cdate     !! String date
      integer(kind=4),  intent(out)   :: ndate(3)  !! Integer values
      logical,          intent(inout) :: error     !! Logical error flag
    end subroutine cdaten
  end interface
  !
  interface
    subroutine ndatec(ndate,cdate,error)
      use gbl_message
      !------------------------------------------------------------------------
      !  Convert the integers for year, month, and day into date '01-JAN-1984'
      !------------------------------------------------------------------------
      integer(kind=4),  intent(in)    :: ndate(3)  !! Year Month Day
      character(len=*), intent(out)   :: cdate     !! DD-MMM-YYYY
      logical,          intent(inout) :: error     !! Logical error flag
    end subroutine ndatec
  end interface
  !
  interface
    subroutine ctimen(ctime,ntime,error)
      use gbl_message
      !---------------------------------------------------------------------
      ! Converts the TIME '12:34:56.1234' in integers for Hours, Minutes,
      ! Seconds, and milliSeconds
      !---------------------------------------------------------------------
      character(len=*), intent(in)    :: ctime     !! Time string
      integer(kind=4),  intent(out)   :: ntime(4)  !! H, M, S and milliSec
      logical,          intent(inout) :: error     !! Logical error flag
    end subroutine ctimen
  end interface
  !
  interface
    subroutine ntimec(ntime,ctime,error)
      use gbl_message
      !---------------------------------------------------------------------
      ! Converts the integers for Hours, Minutes, Seconds, milliSeconds to
      ! formatted TIME
      !---------------------------------------------------------------------
      character(len=*), intent(out)   :: ctime     !
      integer(kind=4),  intent(in)    :: ntime(4)  !
      logical,          intent(inout) :: error     !
    end subroutine ntimec
  end interface
  !
  interface
    subroutine utc(date)
      use gbl_message
      !---------------------------------------------------------------------
      !  Lecture de l'heure du systeme par appel des routines systeme
      ! SIC_DATE en sortie : DATE date gregorienne (year, month, day, hour,
      ! minute, second)
      !---------------------------------------------------------------------
      integer(kind=4), intent(out) :: date(7)  !! Gregorian day
    end subroutine utc
  end interface
  !
  interface
    subroutine astro_message(seve,comm,mess)
      !! Remap to MAP_MESSAGE
      integer, intent(in) :: seve
      character(len=*), intent(in) :: comm
      character(len=*), intent(in) :: mess
    end subroutine astro_message
  end interface
  !
  interface
    subroutine read_and_map(name,head,file,error)
      use gkernel_types
      use gbl_message
      !*
      ! IMAGER -- Utility routine 
      !   Read a data cube and define an associated SIC variable
      !!
      character(len=*), intent(in) :: name      !! SIC Variable name
      character(len=*), intent(inout) :: file   !! Data file
      type(gildas), intent(inout) :: head       !! Data cube header
      logical, intent(out) :: error             !! Logical error flag
    end subroutine read_and_map
  end interface
  !
  interface
    subroutine plait_command(line,comm,error)
      use image_def
      use gbl_message
      !--------------------------------------------------------------------
      !*
      ! IMAGER -- Support for PLAIT command  
      ! PLAIT Output In1 Angle1 Length1 In2 Angle2 Length2 ...
      !
      ! Elaborated on original ideas from Ch. Nieten, 
      !   see https://articles.adsabs.harvard.edu/pdf/2000ASPC..217...72N
      ! and the NOD2 package, namely the "Basket Weaving" technique described
      ! in Emerson & Graeve 1988, 
      !   see https://articles.adsabs.harvard.edu/pdf/1988A%26A...190..353E.
      !
      ! Initial coding P. Hily-Blant feb-2005  
      ! Documented and Cleaned    Stéphane Guilloteau May-2023
      !---------------------------------------------------------------------
      character(len=*), intent(in) :: line  !! Command line
      character(len=*), intent(in) :: comm  !! Command name
      logical :: error                      !! Logical error flag
    end subroutine plait_command
  end interface
  !
  interface
    subroutine gclean_init
      use gclean_mod
      !! IMAGER - CLEAN Method GAUSS -- GCLEAN command  
      !!  Initialize scale arrays
    end subroutine gclean_init
  end interface
  !
  interface
    subroutine cossin(phase,rcos,rsin)
      !-------------------------------------------------------
      !*
      ! Semi-Fast,  Semi-accurate Sin/Cos pair computation
      ! using (not yet clever) interpolation from a precise
      ! loop-up table
      !
      ! A solution using Taylor expansion and symmetries
      ! would be faster and more accurate. The speed is
      ! unfortunately controlled by the Modulo function.
      !    Might become faster if a simple test is made
      ! to avoid this Modulo.  To be developped as needed.
      !! 
      !-------------------------------------------------------
      real(8), intent(inout) :: phase  !! Phase value (radian)
      real(8), intent(out) :: rcos     !! Approximate Cosine
      real(8), intent(out) :: rsin     !! Approximate Sine
    end subroutine cossin
  end interface
  !
  interface
    subroutine no_logstack
      !* 
      ! IMAGER -- Ensure that previous command is not written in the 
      !  Stack or LogFile.
      !  Some otherwise "normal" commands require this after some
      !  interactive use
      !!
    end subroutine no_logstack
  end interface
  !
end module imager_interfaces_none
