//
//  Utilities.swift
//  mas
//
//  Created by Andrew Naylor on 14/09/2016.
//  Copyright © 2016 Andrew Naylor. All rights reserved.
//

import Foundation

// A collection of output formatting helpers

/// Terminal Control Sequence Indicator.
private let csi = "\u{001B}["

private var standardError = FileHandleTextOutputStream(FileHandle.standardError)

private struct FileHandleTextOutputStream: TextOutputStream {
    private let fileHandle: FileHandle

    init(_ fileHandle: FileHandle) {
        self.fileHandle = fileHandle
    }

    /// Appends the given string to the stream.
    func write(_ string: String) {
        guard let data = string.data(using: .utf8) else {
            return
        }
        fileHandle.write(data)
    }
}

/// Prints a message to stdout prefixed with a blue arrow.
func printInfo(_ message: String) {
    guard isatty(fileno(stdout)) != 0 else {
        print("==> \(message)")
        return
    }

    // Blue bold arrow, Bold text
    print("\(csi)1;34m==>\(csi)0m \(csi)1m\(message)\(csi)0m")
}

/// Prints a message to stderr prefixed with "Warning:" underlined in yellow.
func printWarning(_ message: String) {
    guard isatty(fileno(stderr)) != 0 else {
        print("Warning: \(message)", to: &standardError)
        return
    }

    // Yellow, underlined "Warning:" prefix
    print("\(csi)4;33mWarning:\(csi)0m \(message)", to: &standardError)
}

/// Prints a message to stderr prefixed with "Error:" underlined in red.
func printError(_ message: String) {
    guard isatty(fileno(stderr)) != 0 else {
        print("Error: \(message)", to: &standardError)
        return
    }

    // Red, underlined "Error:" prefix
    print("\(csi)4;31mError:\(csi)0m \(message)", to: &standardError)
}

/// Flushes stdout.
func clearLine() {
    guard isatty(fileno(stdout)) != 0 else {
        return
    }

    print("\(csi)2K\(csi)0G", terminator: "")
    fflush(stdout)
}
