//===----------------------------------------------------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

// <complex>

// template<class T>    complex<T>           proj(const complex<T>&);
//                      complex<long double> proj(long double);
//                      complex<double>      proj(double);
// template<Integral T> complex<double>      proj(T);
//                      complex<float>       proj(float);

#include "support/test_complex.h"

#include "../cases.h"

template <class T>
void
test(T x, ::std::enable_if_t<std::is_integral_v<T>>* = 0)
{
    static_assert((std::is_same_v<decltype(dpl::proj(x)), dpl::complex<double> >));
    assert(dpl::proj(x) == dpl::proj(dpl::complex<double>(x, 0)));
}

template <class T>
void
test(T x, ::std::enable_if_t<std::is_floating_point_v<T>>* = 0)
{
    static_assert((std::is_same_v<decltype(dpl::proj(x)), dpl::complex<T> >));
    assert(dpl::proj(x) == dpl::proj(dpl::complex<T>(x, 0)));
}

template <class T>
void
test(T x, ::std::enable_if_t<!std::is_integral_v<T> && !std::is_floating_point_v<T>>* = 0)
{
    static_assert((std::is_same_v<decltype(dpl::proj(x)), dpl::complex<T> >));
    assert(dpl::proj(x) == dpl::proj(dpl::complex<T>(x, 0)));
}

template <class T>
void
test()
{
    test<T>(0);
    test<T>(1);
    test<T>(10);
}

ONEDPL_TEST_NUM_MAIN
{
    test<float>();
    IF_DOUBLE_SUPPORT(test<double>())
    IF_LONG_DOUBLE_SUPPORT(test<long double>())
    // This check required to avoid code with dpl::complex<double> instantiation
    // when double type not supported on device
    IF_DOUBLE_SUPPORT(test<int>())
    IF_DOUBLE_SUPPORT(test<unsigned>())
    IF_DOUBLE_SUPPORT(test<long long>())

  return 0;
}
