import { IDisposable } from '@lumino/disposable';
import { Poll } from '@lumino/polling';
import { ISignal } from '@lumino/signaling';
import { Builder } from './builder';
import { Contents } from './contents';
import { Event } from './event';
import { Kernel } from './kernel';
import { KernelSpec } from './kernelspec';
import { NbConvert } from './nbconvert';
import { ServerConnection } from './serverconnection';
import { Session } from './session';
import { Setting } from './setting';
import { Terminal } from './terminal';
import { User } from './user';
import { Workspace } from './workspace';
/**
 * A Jupyter services manager.
 */
export declare class ServiceManager implements ServiceManager.IManager {
    /**
     * Construct a new services provider.
     */
    constructor(options?: Partial<ServiceManager.IOptions>);
    /**
     * A signal emitted when there is a connection failure with the kernel.
     */
    get connectionFailure(): ISignal<this, Error>;
    /**
     * Test whether the service manager is disposed.
     */
    get isDisposed(): boolean;
    /**
     * Dispose of the resources used by the manager.
     */
    dispose(): void;
    /**
     * The server settings of the manager.
     */
    readonly serverSettings: ServerConnection.ISettings;
    /**
     * Get the session manager instance.
     */
    readonly sessions: Session.IManager;
    /**
     * Get the kernel manager instance.
     */
    readonly kernels: Kernel.IManager;
    /**
     * Get the kernelspec manager instance.
     */
    readonly kernelspecs: KernelSpec.IManager;
    /**
     * Get the setting manager instance.
     */
    readonly settings: Setting.IManager;
    /**
     * The builder for the manager.
     */
    readonly builder: Builder.IManager;
    /**
     * Get the contents manager instance.
     */
    readonly contents: Contents.IManager;
    /**
     * The event manager instance.
     */
    readonly events: Event.IManager;
    /**
     * Get the terminal manager instance.
     */
    readonly terminals: Terminal.IManager;
    /**
     * Get the user manager instance.
     */
    readonly user: User.IManager;
    /**
     * Get the workspace manager instance.
     */
    readonly workspaces: Workspace.IManager;
    /**
     * Get the nbconvert manager instance.
     */
    readonly nbconvert: NbConvert.IManager;
    /**
     * Test whether the manager is ready.
     */
    get isReady(): boolean;
    /**
     * A promise that fulfills when the manager is ready.
     */
    get ready(): Promise<void>;
    private _onConnectionFailure;
    private _isDisposed;
    private _readyPromise;
    private _connectionFailure;
    private _isReady;
}
/**
 * The namespace for `ServiceManager` statics.
 */
export declare namespace ServiceManager {
    /**
     * A service manager interface.
     */
    export interface IManager extends IDisposable, IManagers {
        /**
         * Test whether the manager is ready.
         */
        readonly isReady: boolean;
        /**
         * A promise that fulfills when the manager is initially ready.
         */
        readonly ready: Promise<void>;
        /**
         * A signal emitted when there is a connection failure with the server.
         */
        readonly connectionFailure: ISignal<IManager, Error>;
    }
    /**
     * The options used to create a service manager.
     */
    export interface IOptions extends IManagers {
        /**
         * The default drive for the contents manager.
         */
        readonly defaultDrive: Contents.IDrive;
        /**
         * When the manager stops polling the API. Defaults to `when-hidden`.
         */
        standby: Poll.Standby | (() => boolean | Poll.Standby);
    }
    /**
     * The managers provided by the service manager.
     */
    interface IManagers {
        /**
         * The builder for the manager.
         *
         * @deprecated will be removed in JupyterLab v5
         */
        readonly builder: Builder.IManager;
        /**
         * The contents manager for the manager.
         */
        readonly contents: Contents.IManager;
        /**
         * The events service manager.
         */
        readonly events: Event.IManager;
        /**
         * A promise that fulfills when the manager is initially ready.
         */
        readonly ready: Promise<void>;
        /**
         * The server settings of the manager.
         */
        readonly serverSettings: ServerConnection.ISettings;
        /**
         * The session manager for the manager.
         */
        readonly sessions: Session.IManager;
        /**
         * The kernel manager of the manager.
         */
        readonly kernels: Kernel.IManager;
        /**
         * The kernelspec manager for the manager.
         */
        readonly kernelspecs: KernelSpec.IManager;
        /**
         * The setting manager for the manager.
         */
        readonly settings: Setting.IManager;
        /**
         * The terminals manager for the manager.
         */
        readonly terminals: Terminal.IManager;
        /**
         * The user manager for the manager.
         */
        readonly user: User.IManager;
        /**
         * The workspace manager for the manager.
         */
        readonly workspaces: Workspace.IManager;
        /**
         * The nbconvert manager for the manager.
         */
        readonly nbconvert: NbConvert.IManager;
    }
    export {};
}
