#Copyright 2009 Diego Duclos
#
#This program is free software: you can redistribute it and/or modify
#it under the terms of the GNU General Public License as published by
#the Free Software Foundation, either version 3 of the License, or
#(at your option) any later version.
#
#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.
#
#You should have received a copy of the GNU General Public License
#along with this program.  If not, see <http://www.gnu.org/licenses/>.

from window import window
from model import marketGroup, item, character, eveapi, category, group, graphic
from gui.errorWindow import errorWindow
from gui.inputWindow import inputWindow
from gui.itemStatsWindow import itemStatsWindow
from gui import pixbufLoader
from gui.pluggable import pluggable
import gtk
import gtk.gdk
import copy

class characterEditorWindow(window):
    def __init__(self, master):
        self.master = master
        signalsDict = {'exitRequest' : self.exit,
                       'delete' : self.exit,
                       'getSkills' : self.getSkills,
                       'apiConnectRequest' : self.apiConnect,
                       'importCharacterRequest' : self.importCharacter,
                       'editCharRequest' : self.editChar,
                       'newCharRequest' : self.newChar,
                       'deleteCharRequest' : self.deleteChar,
                       'confirm' : self.confirm,
                       'activeCharacterChanged' : self.activeCharacterChanged,
                       'searchSkill' : self.searchSkill,
                       'skillClicked' : self.skillClicked,
                       'clearRequest' : self.clearCurrent,}
        
        window.__init__(self, "characterEditor", signalsDict)
        self.initPluggable()

        #Get a copy we can work on without changing the original list
        #We'll write back to the original when the user clicks ok
        self.tmpCharDict = copy.deepcopy(self.master.characters)
        
        self.initSkillTab()
        self.initCharacterList()
        self.initApiCharList()
        self.initSkillLevelMenu()
        self.initAllSections()
        self.window.show()
        
    #We don't care about args, really
    def exit(self, *stuff):
        self.window.destroy()
        return False
    
    def confirm(self, source):
        self.master.characters = self.tmpCharDict
        self.master.updateCharacters()
        self.exit()
        
    def toggleGroup(self, widget, event):
        widget.do_button_press_event(widget, event)
        stuff = widget.get_path_at_pos(int(event.x), int(event.y))
        if stuff == None: 
            widget.get_selection().unselect_all()
            return
        if event.type == gtk.gdk._2BUTTON_PRESS and event.button == 1:
            store, selected_rows = widget.get_selection().get_selected_rows()
            if selected_rows:
                selected_row = selected_rows[0]
                if widget.row_expanded(selected_row):
                    widget.collapse_row(selected_row)
                else: widget.expand_row(selected_row, False)
                
    def initCharacterList(self):
        store = gtk.ListStore(str, object)
        self.cmbCharacter.set_model(store)
        nameRenderer = gtk.CellRendererText()
        self.cmbCharacter.pack_start(nameRenderer, True)
        self.cmbCharacter.add_attribute(nameRenderer, 'text', 0)
        store.set_sort_column_id(0, gtk.SORT_ASCENDING)
        for name, char in self.tmpCharDict.iteritems():
            store.append((name, char))
        
        if self.master.activeCharacter:
            iter = store.get_iter_first()
            while iter:
                char = store.get(iter, 1)[0]
                if char.name == self.master.activeCharacter:
                    path = store.get_path(iter)[0]
                    self.cmbCharacter.set_active(path)
                    return
                iter = store.iter_next(iter)
            
        self.cmbCharacter.set_active(0)
    
    def initSkillTab(self):
        #Let's start with getting the skills market group
        self.tvwSkills.connect("button-press-event", self.toggleGroup)
        skillsCat = category.getCategory(name = "Skill")
        self.skillGroups = group.getGroupsByCategory(skillsCat.ID)
        store = gtk.TreeStore(gtk.gdk.Pixbuf, str, str, str, object)
        store.set_sort_column_id(1, gtk.SORT_ASCENDING)
        self.tvwSkills.set_model(store)
        colSkill = gtk.TreeViewColumn('Skill')
        colLevel = gtk.TreeViewColumn('Level')
        imgRenderer = gtk.CellRendererPixbuf()
        skillRenderer = gtk.CellRendererText()
        levelRenderer = gtk.CellRendererText()
        #Add stuff to the first column (image + skill name)
        colSkill.pack_start(imgRenderer, False)
        colSkill.pack_start(skillRenderer, False)
        colSkill.add_attribute(imgRenderer, 'pixbuf', 0)
        colSkill.add_attribute(skillRenderer, 'text', 1)
        #add stuff to the second column (skill level)
        colLevel.pack_start(levelRenderer, False)
        colLevel.add_attribute(levelRenderer, 'text', 2)
        #Add the cols to the view
        self.tvwSkills.append_column(colSkill)
        self.tvwSkills.append_column(colLevel)
        #Prevent drag & drop
        self.tvwSkills.set_reorderable(False)
        #Set tooltip column
        self.tvwSkills.set_tooltip_column(3)
        #Put stuff in the store
        skillPixbuf = pixbufLoader.getPixbuf("skill")
        for grp in self.skillGroups:
            if grp.published == 1:
                iter = store.append(None, (skillPixbuf, grp.name, None, None, grp))
                store.append(iter, (None, 'Something went horribly wrong.\nYou should check the console for details', None, None, None))
           
    def initApiCharList(self):
        store = gtk.ListStore(str, str, str, int)
        self.cmbApiCharacters.set_model(store)
        nameRenderer = gtk.CellRendererText()
        self.cmbApiCharacters.pack_start(nameRenderer, True)
        self.cmbApiCharacters.add_attribute(nameRenderer, 'text', 0)
        store.set_sort_column_id(0, gtk.SORT_ASCENDING)
        if not self.tmpCharDict:
            self.btnConnect.set_sensitive(False)
     
    def getSkills(self, treeview, iter, path):
        #get character
        charStore = self.cmbCharacter.get_model()
        charPath = self.cmbCharacter.get_active()
        if charPath != -1 and charPath != None:
            charIter = charStore.get_iter(charPath)
            char = charStore.get(charIter, 1)[0]
            charSkills = char.skills
        else:
            char = None
            charSkills = None
        
        store = treeview.get_model()
        grp = store.get(iter, 4)[0]
        iterChild = store.iter_nth_child(iter, 0)
        waiter = store.get(iterChild, 4)[0]
        if waiter == None:
            skills = item.getItemsByGroup(group = grp)
            skillPixbuf = pixbufLoader.getPixbuf("skill")
            for skill in skills:
                if skill.published == 1:
                    if charSkills != None:
                        try:
                            level = charSkills[skill]
                        except KeyError:
                            level = 0
                    else:
                        level = 0
                    store.append(iter, (skillPixbuf, skill.name, level, skill.description, skill))
                
            store.remove(iterChild)
        
    def initSkillLevelMenu(self):
        self.skillMenu = gtk.Menu()
        self.skillLevelMenu = gtk.Menu()
        skillBookPixbuf = pixbufLoader.getPixbuf("skill")
        for i in range(6):
            box = gtk.HBox()
            menuItem = gtk.MenuItem()
            skillBookImage = gtk.Image()
            skillBookImage.set_from_pixbuf(skillBookPixbuf)
            lbl = gtk.Label()
            lbl.set_alignment(0, 0.5)
            lbl.set_text("Set to Level " + str(i))
            box.pack_start(skillBookImage, False, False)
            box.pack_start(lbl, True, True)
            menuItem.add(box)
            menuItem.connect("activate", self.setSkillLevel, i)
            self.skillLevelMenu.append(menuItem)
        self.skillLevelMenu.append(gtk.SeparatorMenuItem())
        
        for menu in (self.skillMenu, self.skillLevelMenu):
            menuItem = gtk.MenuItem()
            lbl = gtk.Label("Skill Stats")
            lbl.set_alignment(0, 0.5)
            menuItem.add(lbl)
            menuItem.connect("activate", self.viewStats)
            menu.add(menuItem)
            
    def skillClicked(self, widget, event):
        widget.do_button_press_event(widget, event)
        if event.button == 3:
            store, selected_rows = self.tvwSkills.get_selection().get_selected_rows()
            if not selected_rows: return
            charStore = self.cmbCharacter.get_model()
            charPath = self.cmbCharacter.get_active()
            if charPath == -1 or charPath == None: return
            charIter = charStore.get_iter(charPath)
            char = charStore.get(charIter, 1)[0]
            iter = store.get_iter(selected_rows[0])
            elem = store.get(iter, 4)[0]
            if isinstance(elem, item.item):
                if char.name == "All 0" or char.name == "All 5": menu = self.skillMenu
                else: menu = self.skillLevelMenu
                menu.show_all()
                menu.popup(None, None, None, event.button, event.time)

    def viewStats(self, *stuff):
        store, selected_rows = self.tvwSkills.get_selection().get_selected_rows()
        if selected_rows:
            iter = store.get_iter(selected_rows[0])
            skill = store.get(iter, 4)[0]
            itemStatsWindow(skill, pixbufLoader.getPixbuf("skill"))
        
    def setSkillLevel(self, menuItem, level):
        charStore = self.cmbCharacter.get_model()
        charPath = self.cmbCharacter.get_active()
        charIter = charStore.get_iter(charPath)
        char = charStore.get(charIter, 1)[0]
        charSkills = char.skills
        store, selected_rows = self.tvwSkills.get_selection().get_selected_rows()
        selected_row = selected_rows[0]
        iter = store.get_iter(selected_row)
        skill = store.get(iter, 4)[0]
        store.set(iter, 2, level)
        charSkills[skill] = level
        
    def apiConnect(self, source):
        userID = self.entryUserID.get_text()
        apiKey = self.entryApiKey.get_text()
        if userID == None or userID == "" or userID == None or userID == "":
            errorWindow(self, "Both userId and apiKey must be filled in.")
            return
        try:
            charList = character.getCharacterList(userID, apiKey)
        except Exception, e:
            errorWindow(self, e.args[0])
            return
        
        store = self.cmbApiCharacters.get_model()
        store.clear()
        for char in charList:
            store.append((char[1], userID, apiKey, char[0]))
            
        #Make stuff visible
        self.characterListBox.show_all()
        self.cmbApiCharacters.set_active(0)
            
    def newChar(self, source):
        inputWindow(self, "Name: ", self.addChar)
        
    def addChar(self, name):
        if name == "" or name == None or name in self.tmpCharDict:
            return False
        
        char = character.character(name)
        self.tmpCharDict[name] = char
        store = self.cmbCharacter.get_model()
        iter = store.append((name, char))
        path = store.get_path(iter)
        self.cmbCharacter.set_active(path[0])
        self.btnConnect.set_sensitive(True)
        return True
    
    def editChar(self, source):
        inputWindow(self, "New name: ", self.renameChar)
    
    def renameChar(self, newName):
        if newName == "" or newName == None or newName in self.tmpCharDict:
            return False
    
        store = self.cmbCharacter.get_model()
        path = self.cmbCharacter.get_active()
        iter = store.get_iter(path)
        oldName, char = store.get(iter, 0, 1)
        del self.tmpCharDict[oldName]
        self.tmpCharDict[newName] = char
        char.name = newName
        store.set(iter, 0, newName)
        
        return True
    
    def deleteChar(self, source):
        path = self.cmbCharacter.get_active()
        if path == None or path == -1:
            return
        
        store = self.cmbCharacter.get_model()
        iter = store.get_iter(path)
        name = store.get(iter, 0)[0]
        del self.tmpCharDict[name]
        store.remove(iter)
        if path == 0:
            self.cmbCharacter.set_active(0)
        else:
            self.cmbCharacter.set_active(path - 1)
            
        #Check if there are any characters left, if not we disable api connect
        if not self.tmpCharDict:
            self.btnConnect.set_sensitive(False)
            
        self.activeCharacterChanged()
    
    def importCharacter(self, source):
        apiStore = self.cmbApiCharacters.get_model()
        apiPath = self.cmbApiCharacters.get_active()
        apiIter = apiStore.get_iter(apiPath)
        userID, apiKey, charID = apiStore.get(apiIter, 1, 2, 3)
        store = self.cmbCharacter.get_model()
        path = self.cmbCharacter.get_active()
        iter = store.get_iter(path)
        char = store.get(iter, 1)[0]
        char.userID = userID
        char.apiKey = apiKey
        char.charID = charID
        char.getSkills()
        self.characterListBox.hide_all()
        self.rebuildSkills()
    
    def activeCharacterChanged(self, *stuff):
        path = self.cmbCharacter.get_active()
                
        if path == None or path == -1:
            skills = None
            char = None
        else:
            #Get the active character and his skills
            store = self.cmbCharacter.get_model()
            iter = store.get_iter(path)
            char = store.get(iter, 1)[0]
            
        #Hide the character selection box in the API tab
        self.characterListBox.hide_all()
        #rebuild skills
        self.rebuildSkills()
        #Set user id & api key entry fields
        self.fillApiEntries(char)
        #Check if there's any characters left, if not we need to gray out rename
        if len(self.tmpCharDict) == 0 or char == None or char.name == "All 0" or char.name == "All 5":
            self.btnEditChar.set_sensitive(False)
            self.btnDeleteChar.set_sensitive(False)
            self.btnConnect.set_sensitive(False)
        else:
            self.btnEditChar.set_sensitive(True)
            self.btnDeleteChar.set_sensitive(True)
            self.btnConnect.set_sensitive(True)
    def fillApiEntries(self, char):
        box = self.cmbCharacter            
        if char != None:
            self.entryUserID.set_text(char.userID or "")
            self.entryApiKey.set_text(char.apiKey or "")
        else:
            self.entryUserID.set_text("")
            self.entryApiKey.set_text("")
    
    def rebuildSkills(self):
        path = self.cmbCharacter.get_active()
                
        if path == None or path == -1:
            skills = None
            char = None
        else:
            #Get the active character and his skills
            store = self.cmbCharacter.get_model()
            iter = store.get_iter(path)
            char = store.get(iter, 1)[0]
            skills = char.skills
            
        store = self.tvwSkills.get_model()
        #Store is None when we get called as the window is created
        if store != None:
            #Set all loaded skills to their new levels
            iter = store.get_iter_root()
            while True:
                if iter == None:
                    break
                childIter = store.iter_children(iter)
                while True:
                    if childIter == None:
                        break
                    
                    skill = store.get(childIter, 4)[0]
                    if skill == None:
                        break
                    else:
                        if skills == None:
                            level = 0
                        else:
                            try:
                                level = skills[skill]
                            except KeyError:
                                level = 0
                            
                        store.set(childIter, 2, level)
                        childIter = store.iter_next(childIter)
                    
                iter = store.iter_next(iter)
    
    def searchSkill(self, *stuff):
        search = self.entrySearchSkill.get_text().encode('ascii', 'replace')
        self.search(self.tvwSkills, search)
    
    def search(self, view, search):
        store = view.get_model()
        results = item.searchItem(search)
        #Look for the first match that matches a skillgroup
        for i in results:
            iter = store.get_iter_root()
            while True:
                if iter == None:
                    break

                path = store.get_path(iter)
                grp = store.get_value(iter, 4)
                #Check if the group we're currently on matches the item's group
                if grp.ID == i.groupID:
                    #We found the group, now look better
                    #expand the row the item is in
                    #This will query sqlite for the contents of the group if it hasn't before.
                    #This querying is the main bottleneck of the algorithm (75% of time spent)
                    view.expand_row(path, False)
                    iterChild = store.iter_children(iter)
                    while True:
                        if iterChild == None: break
                        skill = store.get_value(iterChild, 4)
                        if skill.name == i.name:
                            #Omg we found the item, finaly done.
                            pathChild = store.get_path(iterChild)
                            view.get_selection().select_path(pathChild)
                            return
                        
                        iterChild = store.iter_next(iterChild)
                        
                    view.collapse_row(path)
                    return
                    
                iter = store.iter_next(iter)
    
    def clearCurrent(self, source):
        page = self.nb.get_current_page()
        #Skills
        if page == 0:
            self.clearSkills()
        if page == 1:
            self.clearPluggedInImplants()
        if page == 2:
            self.entryUserID.set_text("")
            self.entryApiKey.set_text("")
            self.characterListBox.hide_all()
            charStore = self.cmbCharacter.get_model()
            charPath = self.cmbCharacter.get_active()
            if charPath == -1 or charPath == None:
                return
            
            charIter = charStore.get_iter(charPath)
            char = charStore.get(charIter, 1)[0]
            char.userID = None
            char.apiKey = None
                
    def clearSkills(self):
        charStore = self.cmbCharacter.get_model()
        charPath = self.cmbCharacter.get_active()
        if charPath == -1 or charPath == None:
            return
        
        charIter = charStore.get_iter(charPath)
        char = charStore.get(charIter, 1)[0]
        char.skills = {}
            
        store = self.tvwSkills.get_model()
        iter = store.get_iter_root()
        while True:
            if iter == None:
                break
            
            childIter = store.iter_children(iter)
            while True:
                if childIter == None:
                    break
                
                skill = store.get(childIter, 4)[0]
                if skill == None:
                    break
                else:
                    store.set(childIter, 2, 0)
                    
                childIter = store.iter_next(childIter)
                
            iter = store.iter_next(iter)  
            
characterEditorWindow = pluggable(characterEditorWindow)
