#Copyright 2009 Diego Duclos
#
#This program is free software: you can redistribute it and/or modify
#it under the terms of the GNU General Public License as published by
#the Free Software Foundation, either version 3 of the License, or
#(at your option) any later version.
#
#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.
#
#You should have received a copy of the GNU General Public License
#along with this program.  If not, see <http://www.gnu.org/licenses/>.
from model import category
from model import group
from model import item
from gui.window import window
from gui.errorWindow import errorWindow
from gui.itemStatsWindow import itemStatsWindow
from gui import pixbufLoader
try:
    import pygtk
    pygtk.require("2.0")
except:
    pass
import gtk
from gtk import gdk
from gui.pluggable import pluggable
import re

class shipBrowserWindow(window):
    def __init__(self, master):
        self.master = master
        signalsDict = {'exitRequest' : self.exit,
                       'delete' : self.delete,
                       'rowLookup' : self.rowLookup,
                       'loadDescription' : self.loadDescription,
                       'checkCollapse' : self.checkCollapse,
                       'confirm' : self.confirm,
                       'filtersChanged' : self.filtersChanged,
                       'shipClicked' : self.shipClicked}
        
        window.__init__(self, "shipBrowser", signalsDict)
        self.initPluggable()
        #Set tree store and filter
        self.treeStore = gtk.TreeStore(gdk.Pixbuf, str, object)
        self.treeFilter = self.treeStore.filter_new()
        self.treeFilter.set_visible_func(self.checkVisibility)
        
        self.tvwShips.set_model(self.treeFilter)
        
        #Add column to the view
        treeViewCol = gtk.TreeViewColumn('Ships')
        self.tvwShips.append_column(treeViewCol)
        #Cells
        treeViewCellText = gtk.CellRendererText()
        treeViewCellImage = gtk.CellRendererPixbuf()
        
        #Add the cell to the view, allow text to expand but image not
        treeViewCol.pack_start(treeViewCellImage, False)
        treeViewCol.pack_start(treeViewCellText, True)
        
        treeViewCol.add_attribute(treeViewCellImage, 'pixbuf', 0)
        treeViewCol.add_attribute(treeViewCellText, 'text', 1)
        
        self.treeStore.set_sort_func(1, sortByRaceAndName, None)
        self.treeStore.set_sort_column_id(1, gtk.SORT_DESCENDING)
        
        #Don't let users drag & drop
        self.tvwShips.set_reorderable(False)
        
        #Get all the filter checkboxes and check them all
        self.filterBoxes = {'amarr' : self.get_widget("tgbAmarr"),
                            'caldari' : self.get_widget("tgbCaldari"),
                            'gallente' : self.get_widget("tgbGallente"),
                            'minmatar' : self.get_widget("tgbMinmatar"),
                            'ore' : self.get_widget("tgbOre"),
                            'serpentis' : self.get_widget("tgbSerpentis"),
                            'angel' : self.get_widget("tgbAngel"),
                            'blood' : self.get_widget("tgbBlood"),
                            'sansha' : self.get_widget("tgbSansha"),
                            'guristas' : self.get_widget("tgbGuristas")}
        
        for box in self.filterBoxes.itervalues():
            box.set_active(True)
        
        #Set tooltips
        self.setTooltips(**{"tgbAmarr" : "Amarr Empire",
                            "tgbCaldari" : "Caldari State",
                            "tgbGallente" : "Gallente Federation",
                            "tgbMinmatar" : "Minmatar Republic",
                            "tgbOre" : "ORE",
                            "tgbSerpentis" : "Serpentis",
                            "tgbAngel" : "Angel Cartel",
                            "tgbBlood" : "The Blood Raider Covenant",
                            "tgbSansha" : "Sansha's Nation",
                            "tgbGuristas" : "Guristas Pirates"})
        
        #Fetch the ships market group and all it's children groups
        shipCategory = category.getCategory(name="Ship")
        self.shipGroupList = group.getGroupsByCategory(shipCategory.ID)
        shipPixbuf = pixbufLoader.getPixbuf("ships")
        for grp in self.shipGroupList:
            if grp.published == 1 or grp.name == "Rookie ship":
                parent = self.treeStore.append(None, (shipPixbuf, grp.name, grp))
                self.treeStore.append(parent, (None, 'Something went horribly wrong.\nYou should check the console for details', None))
        self.initShipMenu()
        self.initAllSections()
        self.window.show()
        
    def initShipMenu(self):
        menu = gtk.Menu()
        self.shipMenu = menu
        menuItem = gtk.MenuItem()
        lblEntry = gtk.Label()
        lblEntry.set_alignment(0, 0.5)
        lblEntry.set_text("Ship Stats")
        menuItem.add(lblEntry)
        menuItem.set_right_justified(False)
        menuItem.connect("activate", self.menuClicked)
        menu.add(menuItem)
    
    def menuClicked(self, widget):
        store, selected_rows = self.tvwShips.get_selection().get_selected_rows()
        iterF = self.treeFilter.get_iter(selected_rows[0])
        iter = self.treeFilter.convert_iter_to_child_iter(iterF)
        icon, selected = self.treeStore.get(iter, 0, 2)
        itemStatsWindow(selected, icon)
    
    def shipClicked(self, widget, event):
        widget.do_button_press_event(widget, event)
        stuff = widget.get_path_at_pos(int(event.x), int(event.y))
        if stuff == None: 
            widget.get_selection().unselect_all()
            return
        store, selected_rows = self.tvwShips.get_selection().get_selected_rows()
        if selected_rows:
            selected_row = selected_rows[0]
            iterF = self.treeFilter.get_iter(selected_row)
            iter = self.treeFilter.convert_iter_to_child_iter(iterF)
            selected = self.treeStore.get(iter, 2)[0]
            if event.button == 1 and event.type == gtk.gdk._2BUTTON_PRESS:
                if isinstance(selected, group.group):
                    if self.tvwShips.row_expanded(selected_row):
                        self.tvwShips.collapse_row(selected_row)
                    else: self.tvwShips.expand_row(selected_row, False)
                else:
                    self.confirm()
            elif event.button == 3 and isinstance(selected, item.item):
                self.shipMenu.show_all()
                self.shipMenu.popup(None, None, None, event.button, event.time)

            
    def exit(self, widget):
        self.window.destroy()
        
    def delete(self, widget, source):
        return False
        
    def main(self):
        gtk.main()
    
    def filtersChanged(self, elem):
        self.treeFilter.refilter()
        self.loadDescription(self.tvwShips)
    
    def checkVisibility(self, model, iter):
        elem = model.get(iter, 2)[0]
        if isinstance(elem, item.item):
            race = elem.race
            if race == "unknown": return False
            else: return self.filterBoxes[race].get_active()
        else:
            return True
        
    def rowLookup(self, treeview, iter, path):
        #Get the first child of the iter we got, this will either be a ship [in which case we do nothin'] or it will be a waiter
        iter = self.treeFilter.convert_iter_to_child_iter(iter)
        iterChild = self.treeStore.iter_nth_child(iter, 0)
        if iterChild == None:
            return None
        
        firstChildValue = self.treeStore.get(iterChild, 2)[0]
        grp = self.treeStore.get(iter, 2)[0]
        if firstChildValue == None:
            items = item.getItemsByGroup(group=grp)
            for i in items:
                if (i.published == 1 or i.group.name == "Rookie ship") and i.race != "unknown":
                    img = pixbufLoader.getPixbuf(i.race)
                    self.treeStore.append(iter, [img, i.name, i])
                    
            self.treeStore.remove(iterChild)
            
    def checkCollapse(self, treeview, iter, path):
        iter = self.treeFilter.convert_iter_to_child_iter(iter)
        store, selected_rows = treeview.get_selection().get_selected_rows()
        iterF = self.treeFilter.get_iter(selected_rows[0])
        iter = self.treeFilter.convert_iter_to_child_iter(iterF)
        self.treeStore.is_ancestor(iter, iter)
        if self.treeStore.is_ancestor(iter, iter):
            treeview.get_selection().select_path(self.treeStore.get_path(iter))
            self.loadDescription(treeview)
        
        return False
    
    def loadDescription(self, treeview):
        store, selected_rows = treeview.get_selection().get_selected_rows()
        if not selected_rows:
            self.clearDescription()
            return
        iterF = self.treeFilter.get_iter(selected_rows[0])
        iter = self.treeFilter.convert_iter_to_child_iter(iterF)
        selected = self.treeStore.get(iter, 2)[0]
        if isinstance(selected, item.item):
            description = re.sub(r'<[^>]*?>', '', selected.description).replace("\\\"", "\"") 
            buffer = gtk.TextBuffer()
            buffer.set_text(description)
            self.txvDescription.set_buffer(buffer)
        else:
            self.clearDescription()
    
    def clearDescription(self):
        buffer = gtk.TextBuffer()
        buffer.set_text("")
        self.txvDescription.set_buffer(buffer)
    #This can be triggered from the following:
    # - the OK button's clicked event
    # - the treeviews row-activated event
    # - the entry's activated event
    #So let's not rely on their data but figure stuff on our own
    def confirm(self, *stuff):
        store, selected_rows = self.tvwShips.get_selection().get_selected_rows()
        if not selected_rows:
            errorWindow(self, "A ship must be selected.")
            return
        
        iterF = self.treeFilter.get_iter(selected_rows[0])
        iter = self.treeFilter.convert_iter_to_child_iter(iterF)
        ship = self.treeStore.get(iter, 2)[0]
        fits = self.master.fits
        if isinstance(ship, item.item):
            fitName = self.entryFitName.get_text()
            if fitName != None and fitName != "":
                if ship.group in fits and \
                   ship in fits[ship.group] and \
                   fitName in fits[ship.group][ship]:
                    errorWindow(self, "A setup with this name already exists.")
                    return
                
                self.master.newShip(fitName, ship)
                self.window.destroy()
            else:
                errorWindow(self, "Setup name may not be empty.")
                return
        else:
            errorWindow(self, "A ship must be selected.")
            return
                

raceOrder = {'amarr' : 1,
             'caldari': 2,
             'gallente' : 3,
             'minmatar' : 4,
             'ore' : 5,
             'serpentis' : 6,
             'angel' : 7,
             'blood' : 8,
             'sansha' : 9,
             'guristas' : 10}

def sortByRaceAndName(treemodel, iter1, iter2, user_data):
    elem1 = treemodel.get(iter1, 2)[0]
    elem2 = treemodel.get(iter2, 2)[0]
    if elem1 == None and elem2 == None:
        return 0
    elif elem1 == None:
        return -1
    elif elem2 == None:
        return 1
    
    if isinstance(elem1, group.group):
        if elem1.name > elem2.name:
            return -1
        elif elem1.name == elem2.name:
            return 0
        else:
            return 1
        
    elif isinstance(elem1, item.item):
        race1 = elem1.race
        race2 = elem2.race
        if race1 == "unknown": raceOrder1 = -1
        else: raceOrder1 = raceOrder[race1]
        if race2 == "unknown": raceOrder2 = -1
        else: raceOrder2 = raceOrder[race2]
        if raceOrder1 > raceOrder2:
            return -1
        elif raceOrder2 > raceOrder1:
            return 1
        else:
            if elem1.name > elem2.name:
                return -1
            elif elem1.name == elem2.name:
                return 0
            else:
                return 1

shipBrowserWindow = pluggable(shipBrowserWindow)
