#Copyright 2009 Diego Duclos
#
#This program is free software: you can redistribute it and/or modify
#it under the terms of the GNU General Public License as published by
#the Free Software Foundation, either version 3 of the License, or
#(at your option) any later version.
#
#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.
#
#You should have received a copy of the GNU General Public License
#along with this program.  If not, see <http://www.gnu.org/licenses/>.
from model import attribute, dataFolder, character, item, group
import module, drone, projectedEffect
import math
import copy
import itertools
import re
import xml.dom
import sys
import traceback

slotTypes = ["high", "medium", "low", "rig", "subsystem"]

slotShortNameMap = {"high" : "hi",
                    "low" : "low",
                    "medium" : "med",
                    "rig" : "rig",
                    "subsystem" : "subsystem"}

slotNameMap = {'high' : 'hiSlots',
               'medium' : 'medSlots',
               'low' : 'lowSlots',
               'rig' : 'rigSlots',
               'subsystem' : 'maxSubSystems'}

modStates = {-1 : 'offline',
             0 : 'inactive',
             1: 'active',
             2: 'overheated'}

STATE_OFFLINE = -1
STATE_INACTIVE = 0
STATE_ACTIVE = 1
STATE_OVERLOADED = 2

defaultDamagePattern = {'thermal' : 0.25,
                        'kinetic' : 0.25,
                        'explosive' : 0.25,
                        'em' : 0.25}

PEAK_RECHARGE = -(math.sqrt(2) - 2 ) / 2

class fitting(object):
    """
    The class used to represent a fit.
    """
    def __init__(self, ship = None, modules = None, character = None, drones = None, projEffects = None, implants = None, boosters = None, gang = None, factorInReload = False):
        '''
        Constructor
        @param ship: the ship this fit is using
        @param modules: a list of tuples, each containing a module and it's ammo
        @param character: the character using this fit
        '''
        self.ship = ship
        self.modules = modules or []
        self.drones = drones or []
        self.character = character
        self.boosters = boosters or {}
        self.implants = implants or {}
        self.blockedItems = set()                    
        self.gang = gang or []
        self.projEffects = projEffects or []
        if ship: self.setShip(ship)
        self.factorInReload = factorInReload
        
    def setShip(self, ship):
        self.ship = ship
        ship.attributes["_maxLockedTargetsSkill"] = attribute.basicAttribute(self.ship, "_maxLockedTargetsSkill", 2)
        ship.attributes["_droneControlRange"] = attribute.basicAttribute(self.ship, "_droneControlRange", 20000)
        ship.attributes["_maxActiveDrones"] = attribute.basicAttribute(self.ship, "_maxActiveDrones", 0)
                
    def getNumModulesAttributeValue(self, attribute, value, minState = STATE_OFFLINE):
        '''
        Get the number of modules fitted to the ship that have a certain attribute set to a certain value
        @param attribute: the attribute to look for
        @param value: the value to look for
        '''
        num = 0
        if self.modules != None:
            for module in self.modules:
                item = module.getItem()
                try:
                    if item.attributes[attribute].getModifiedValue() == value and module.getState() >= minState:
                        num = num + 1
                except KeyError:
                    pass
                
        return num
    
    def getDronesAttributeSum(self, attribute, activeOnly):
        '''
        Get the sum of an attribute on all drones, this can be used to get the combined resource usage of drones
        @param attribute: the attribute to get the sum for
        '''
        sum = 0
        if self.drones != None:
            for drone in self.drones:
                if activeOnly: amount = drone.getAmountActive()
                else: amount = drone.getAmountTotal()
                sum += ((drone.getItem().getModifiedAttribute(attribute) or 0) * amount)
                
        return sum    
    
    def getModuleAttributeSum(self, attribute, minState = STATE_OFFLINE):
        '''
        Get the sum of an attribute on all modules, this can be used to get the combined resource usage of modules
        @param attribute: the attribute to get the sum for
        '''
        sum = 0
        if self.modules != None:
            for module in self.modules:
                if module.getState() >= minState:
                    sum = sum + (module.getItem().getModifiedAttribute(attribute) or 0)
            
        return sum    
    
    def calculateCapUsage(self):
        '''
        Calculates the capacitor recharge rate and puts it in _peakCapRecharge
        '''
        capacity = self.ship.attributes["capacitorCapacity"].getModifiedValue()
        rechargeTime = self.ship.attributes['rechargeRate'].getModifiedValue() / 1000.0
        #Let's calculate cap usage now
        sum = 0
        for module in self.modules:
            if module.getState() >= STATE_ACTIVE:
                item = module.getItem()
                speed = item.getModifiedAttribute("speed") or item.getModifiedAttribute("duration")
                if speed:
                    capUsage = - (item.getModifiedAttribute("capacitorNeed") or 0) / (speed / 1000.0)
                    sum += capUsage
                    item.attributes["_capUsage"] = attribute.basicAttribute(item, "_capUsage", None, capUsage)

        #Figure stability / discharge time
        capDrain = self.ship.getModifiedAttribute("_drain") or 0
        dischargeRate = -self.getModuleAttributeSum("_capUsage", STATE_ACTIVE) - capDrain
        capBoost = self.getModuleAttributeSum("_capBoost", STATE_ACTIVE) + (self.ship.getModifiedAttribute("_capBoost") or 0)
        peakRecharge = calcCapRechForGivenCap(capacity * PEAK_RECHARGE, capacity, rechargeTime) + capBoost
        self.ship.attributes['_peakCapRecharge'] = attribute.basicAttribute(self.ship, "_peakCapRecharge", None, peakRecharge)
        currRecharge = 0
        currCap = capacity
        if peakRecharge > dischargeRate:
            low = PEAK_RECHARGE
            high = 1.0
            capStable = True
            while True:
                oldRecharge = currRecharge
                mid = (low + high) / 2.0
                currRecharge = calcCapRechForGivenCap(capacity * mid, capacity, rechargeTime) + capBoost
                diff = abs(oldRecharge - currRecharge)
                if diff <= 0.001:
                    capState = mid * 100
                    break
                else:
                    if currRecharge > dischargeRate:
                        low = mid
                    else:
                        high = mid
        else:
            capStable = False
            for i in itertools.count(1):
                rechargeRate = calcCapRechForGivenCap(currCap, capacity, rechargeTime) + capBoost
                currCap = currCap - dischargeRate + rechargeRate
                if currCap <= 0:
                    capState = i
                    break
        
        self.ship.attributes["_capStable"] = attribute.basicAttribute(self.ship, "_capStable", None, capStable)
        self.ship.attributes["_capState"] = attribute.basicAttribute(self.ship, "_capState", None, capState)
            
    def calculateTankEhp(self, damagePattern = defaultDamagePattern):
        '''
        Calculates the effective HP of this fit and put it in _tankEhp
        @param damagePattern: the damage pattern attackers are using
        '''
        tankEhp = 0
        #shield first, then armor, then structure
        totalDamage = float(sum(damagePattern.values()))
        for (prefix, attr) in [('shield', 'shieldCapacity'), ('armor', 'armorHP'), ('', 'hp')]:
            rawCapacity = self.ship.attributes[attr].getModifiedValue()
            specificDivider = 0
            for damageType, amount in damagePattern.iteritems():
                if prefix != '':
                    damageType = damageType[0].capitalize() + damageType[1:]
                
                resonance = self.ship.attributes[prefix + damageType + 'DamageResonance'].getModifiedValue()
                specificDivider += amount / totalDamage * resonance
            
            if specificDivider == 0: specificDivider = 1    
            currentLayerEhp = rawCapacity / specificDivider
            
            attrName = "_" + prefix.capitalize() + "Ehp"
            self.ship.attributes[attrName] = attribute.basicAttribute(self.ship, attrName, None, currentLayerEhp)
            tankEhp = tankEhp + currentLayerEhp
        
        self.ship.attributes['_totalEhp'] = attribute.basicAttribute(self.ship, '_totalEhp', None, tankEhp)
        
    def calculateTankRegen(self, damagePattern = defaultDamagePattern):
        '''
        Calculate how much DPS can be tanked by this fit and put it in _tankRegen
        @param damagePattern: the damage pattern attackers are using
        '''
        #Passive shields
        rechargeTime = self.ship.attributes['shieldRechargeRate'].getModifiedValue() / 1000.0
        capacity = self.ship.attributes['shieldCapacity'].getModifiedValue()
        #Shield follows the same rule as cap does
        rawRecharge = calcCapRechForGivenCap(capacity * PEAK_RECHARGE, capacity, rechargeTime)
        self.ship.attributes["_passiveShieldRawRecharge"] = attribute.basicAttribute(self.ship, "_passiveShieldRawRecharge", None, rawRecharge)
        pShieldRegen = self.calculateSpecificRecharge("shield", rawRecharge, damagePattern)
        self.ship.attributes["_passiveShieldRegen"] = attribute.basicAttribute(self.ship, "_passiveShieldRegen", None, pShieldRegen)
        for (tankType, resonanceType) in (("shield", "shield"), ("armor", "armor"), ("hull", "")):
            rawAttrName = "_" + tankType + "RawRecharge"
            attrName = "_" + tankType + "Regen"
            specificRecharge = 0
            for module in self.modules:
                if module.getState() >= STATE_ACTIVE:
                    item = module.getItem()
                    rawRecharge = item.getModifiedAttribute(rawAttrName)
                    if rawRecharge:
                        modSpecificRecharge = self.calculateSpecificRecharge(resonanceType, rawRecharge, damagePattern)
                        item.attributes[attrName] = attribute.basicAttribute(item, attrName, None, modSpecificRecharge)
                        specificRecharge += modSpecificRecharge
                        
            shipRawRecharge = self.ship.getModifiedAttribute(rawAttrName) or 0
            shipSpecificRecharge = self.calculateSpecificRecharge(resonanceType, shipRawRecharge, damagePattern)
            specificRecharge += shipSpecificRecharge
            self.ship.attributes[attrName] = attribute.basicAttribute(self.ship, attrName, None, specificRecharge)
    
    def calculateSpecificRecharge(self, prefix, rawRecharge, damagePattern):
        totalDamage = float(sum(damagePattern.values()))
        specificDivider = 0
        for damageType, amount in damagePattern.iteritems():
            if prefix: damageType = damageType.capitalize()
            resonance = self.ship.getModifiedAttribute(prefix + damageType + 'DamageResonance')
            specificDivider += amount / totalDamage * resonance
        
        if specificDivider == 0: specificDivider = 1
        return rawRecharge / specificDivider
                    
    def calculateSustainedTankRegen(self):
        reppers = []
        recharge = self.ship.getModifiedAttribute("_peakCapRecharge")
        drain = self.ship.getModifiedAttribute("_drain") or 0
        capLeft = recharge + self.getModuleAttributeSum("_capUsage", STATE_ACTIVE) + drain
        #Figure all the modules that are useful
        for tankType in ("shield", "armor", "hull"):
            reinfAttrName = "_" + tankType + "Regen"
            for module in self.modules:
                if module.getState() >= STATE_ACTIVE:
                    item = module.getItem()
                    reinf = item.getModifiedAttribute(reinfAttrName)
                    if reinf:
                        cap = -item.getModifiedAttribute("_capUsage")
                        capLeft += cap
                        reppers.append((tankType, reinf, cap))
                        
        reppers.sort(cmp = lambda x, y: cmp(y[1] / y[2], x[1] / x[2]))
        #Now we have a sorted list of reppers, go through them
        sustainedRegen = {'shield' : 0,
                          'armor' : 0,
                          'hull' : 0}
        for type, reinf, cap in reppers:
            if capLeft <= 0:
                break
            
            activity = min(1, capLeft / cap)
            capLeft -= cap
            sustainedRegen[type] += activity * reinf
        for type, regen in sustainedRegen.iteritems():
            attrName = "_" + type + "SustainedRegen"
            self.ship.attributes[attrName] = \
                attribute.basicAttribute(self.ship, attrName, None, regen, 1)
        
    def calculatePerSecond(self, target, containerName, activeOnly = False): 
        if self.modules != None:
            for module in self.modules:
                if module.getState() >= STATE_ACTIVE or not activeOnly:
                    item = module.getItem()
                    rof = (item.getModifiedAttribute("speed") or item.getModifiedAttribute("duration") or 0) / 1000.0
                    if rof:
                        delay = (item.getModifiedAttribute("moduleReactivationDelay") or 0) / 1000.0
                        if delay: rof += delay
                        dps = (item.getModifiedAttribute(target) or 0) / rof
                        item.attributes[containerName] = attribute.basicAttribute(item, containerName, None, dps)
            
        if self.drones != None:
            for drone in self.drones:
                item = drone.getItem()
                if drone.getAmountActive() > 0 or not activeOnly:
                    try:
                        speed = item.getModifiedAttribute("speed") or item.getModifiedAttribute("duration")
                        dps = (item.getModifiedAttribute(target) or 0) / (speed / 1000.0)
                        item.attributes[containerName] = attribute.basicAttribute(item, containerName, None, dps)
                    except: pass
                    
    def calculateVolley(self, targetResistance = { 'thermal' : 0, 'kinetic' : 0, 'explosive' : 0, 'em' : 0}):
        '''
        Calculate volley information of this fit and puts it in _volley
        @param targetResistance: the resistance the target of attacks has
        '''
        if self.modules != None:
            for module in self.modules:
                if module.getState() >= STATE_ACTIVE:
                    #The try block ensures that modules without ammo or modules that are not weapon don't crash the whole thing
                    try:
                        item = module.getItem()
                        volley = self.getVolleyItem(item, module.getAmmo(), targetResistance)
                        item.attributes['_volley'] = attribute.basicAttribute(item, '_volley', None, volley)
                    except (KeyError, AttributeError): pass

        if self.drones != None:
            for drone in self.drones:
                it = drone.getItem()
                #Used by fighter bombers
                ammo = drone.getAmmo()
                if ammo != None:
                    it.attributes["speed"] = it.attributes["missileLaunchDuration"]
                    volley = self.getVolleyItem(it, ammo, targetResistance)
                #All other drones
                else: volley = self.getVolleyItem(it, it, targetResistance)
                it.attributes['_volley'] = attribute.basicAttribute(it, '_volley', None, volley)

    def getVolleyItem(self, item, ammo, targetResistance):
        damageMultiplier = item.getModifiedAttribute("damageMultiplier") or 1
        damage = {}
        for damageType in targetResistance.iterkeys():
            try:
                damage[damageType] = ammo.getModifiedAttribute(damageType + 'Damage') or 0
            except AttributeError:
                damage[damageType] = item.getModifiedAttribute(damageType + 'Damage') or 0
                
        volley = 0
        for type, amount in damage.iteritems():
            volley = volley + amount * (1 - targetResistance[type]) * damageMultiplier
            
        return volley
    
    def calculateAlignTime(self):
        mass = self.ship.getModifiedAttribute("mass")
        agility = self.ship.getModifiedAttribute("agility")
        alignTime = -math.log(0.25) * mass *  agility / 1000000.0
        self.ship.attributes["_alignTime"] = attribute.basicAttribute(self.ship, '_alignTime', None, alignTime)
    
    def calculateLockTime(self, signature):
        scanRes = self.ship.getModifiedAttribute("scanResolution")
        if scanRes == 0: return 0
        else:
            #Not all versions of python have asinh available
            try:
                return 40000.0 / (scanRes * math.pow(math.asinh(signature), 2))
            except:
                return 0
            
    def calculateModifiedAttributes(self, calculatedFits = None):
        '''
        modify the attributes of this ship with the effects of the ship itself, the modules on it,
        the ammo in the modules, the character's skills and the implants on the character
        '''
        #Clear blocked modules
        self.blockedItems.clear()
        #Runtime dict to store when to call what
        order = []
        runTime = {}
        if self.character == None: charName = None
        else: charName = self.character.name
        calculatedFits = calculatedFits or {self: [charName]}
        
        def processFit(fit, character = None):
            if fit not in calculatedFits: calculatedFits[fit] = []
            charName = character and character.name or None
            if not charName in calculatedFits[fit]:
                calculatedFits[fit].append(charName)
                if fit == self:
                    oldChar = fit.character
                    fit.character = None
                    fitCopy = copy.deepcopy(fit)
                    fit.character = oldChar
                    calculatedFits[fitCopy] = copy.copy(calculatedFits[fit])
                    calculatedFits[fitCopy].append(fit.character and fit.character.name or None)
                    fit = fitCopy

                fit.character = character
                fit.calculateModifiedAttributes(calculatedFits)
                
        def addRuntime(name):
            order.append(name)
            runTime[name] = {"late": [],
                              "normal": [],
                              "early" : []}
        
        def addToRuntime(name, prio, stuff):
            runTime[name][prio or "normal"].append(stuff)
            
        #Skills stuff
        addRuntime("skill")
        if self.character:
            if self.character.skills:
                for skill, level in self.character.skills.iteritems():
                    skill.clearModifiedAttributes()
                    for skillEffect in skill.effects:
                        if skillEffect.func and "normal" in skillEffect.type:
                            addToRuntime("skill", skillEffect.runTime,
                                         (skillEffect, {"level" : level}))
        
        addRuntime("ship")
        self.ship.clearModifiedAttributes()
        #Ship hull effects
        for shipEffect in self.ship.effects:
            if shipEffect.func:
                if "normal" in shipEffect.type:
                    addToRuntime("ship", shipEffect.runTime, (shipEffect, None))
        
        #Loop through drones
        addRuntime("drone")
        addRuntime("droneCharge")
        if self.drones:
            for droneObject in self.drones:
                droneItem, droneCharge = droneObject.getItem(), droneObject.getAmmo()
                droneItem.clearModifiedAttributes()
                for droneEffect in droneItem.effects:
                    if droneEffect.func and "normal" in droneEffect.type:
                        addToRuntime("drone", droneEffect.runTime, (droneEffect, None))
                if droneCharge:
                    droneCharge.clearModifiedAttributes()
                    if droneCharge.effects:
                        for droneChargeEffect in droneCharge.effects:
                            if droneChargeEffect.func and "normal" in droneChargeEffect.type:
                                addToRuntime("droneCharge", droneChargeEffect.runTime, (droneChargeEffect, {"containerModule" : droneItem}))

        #Loop through boosters
        addRuntime("booster")
        for booster, activeSideEffects in self.boosters.itervalues():
            booster.clearModifiedAttributes()
            for boosterEffect in booster.effects:
                if boosterEffect.func and \
                  (not "boosterSideEffect" in boosterEffect.type or \
                   boosterEffect.name in activeSideEffects):
                    addToRuntime("booster", boosterEffect.runTime, (boosterEffect, None))
                    
        #Figure when to call implants
        addRuntime("implant")
        for implant in self.implants.itervalues():
            implant.clearModifiedAttributes()
            for implantEffect in implant.effects:
                if implantEffect.func and "normal" in implantEffect.type:
                    addToRuntime("implant", implantEffect.runTime, (implantEffect, None))
                        
        #Figure what modules and ammo
        addRuntime("overload")
        addRuntime("ammo")
        addRuntime("module")
        if self.modules:
            for mod in self.modules:
                item, state, ammo = mod.getItem(), mod.getState(), mod.getAmmo()
                if state >= STATE_OFFLINE:
                    item.ammo = ammo
                    if ammo:
                        ammo.clearModifiedAttributes()
                        if ammo.effects:
                            for ammoEffect in ammo.effects:
                                if ammoEffect.func and "normal" in ammoEffect.type:
                                    addToRuntime("ammo", ammoEffect.runTime,
                                                 (ammoEffect, {"containerModule" : item}))
                    
                    item.clearModifiedAttributes()
                    for modEffect in item.effects:
                        if modEffect.func:
                            if "normal" in modEffect.type or modEffect.type == ("active",):
                                addToRuntime("module", modEffect.runTime,
                                             (modEffect, {"state" : state}))
                            elif "overload" in modEffect.type:
                                addToRuntime("overload", modEffect.runTime,
                                             (modEffect, {"state": state}))
        
        #Gang stuff
        addRuntime("gang")
        
        self.gangMods = {}
        self.gangSkills = {}
        
        def handleModule(mod, state = None):
            newBonus = math.fabs(mod.getModifiedAttribute("commandBonus") or 0)
            currentBonus = mod.name in self.gangMods and \
                self.gangMods[mod.name][0].getModifiedAttribute("commandBonus") or 0 
            #print newBonus
            if (state >= STATE_ACTIVE or state == None) and newBonus > currentBonus:
                self.gangMods[mod.name] = (mod, state)
                
        for fit, char, enabled in self.gang:
            if enabled:
                processFit(fit, char)
                handleModule(fit.ship)
                for mod in fit.modules: handleModule(mod.getItem(), mod.getState())
                for skill, level in char.skills.iteritems():
                    if not skill in self.gangSkills or level > self.gangSkills[skill]["level"]:
                        self.gangSkills[skill] = {"level": level}
                        
                for implant in fit.implants.itervalues(): self.gangMods[implant.name] = (implant, None)
        
        
        def process(mod, overrides):
            for gangEffect in mod.effects:
                if gangEffect.func and "gang" in gangEffect.type:
                    addToRuntime("gang", gangEffect.runTime, (gangEffect, overrides))
                    
        for mod, state in self.gangMods.itervalues():
            if state: process(mod, {"state": state})
            else: process(mod, None)    
        
        for skill, overrides in self.gangSkills.iteritems(): process(skill, overrides)
        
        #Figure projected effects
        addRuntime("projected")
        def process(mod, amount):
            item = mod.getItem()
            if "projected" in item.getType():
                for projEffect in item.effects:
                    if projEffect.func and "projected" in projEffect.type:
                        addToRuntime("projected", projEffect.runTime,
                                     (projEffect, {"state": mod.getState()}, amount))
                
        for projectedEffect in self.projEffects:
            source = projectedEffect.getSource()
            if isinstance(source, module.module):
                process(source, projectedEffect.getAmountActive())
            elif isinstance(source, fitting):
                processFit(source)
                for mod in source.modules:
                    process(mod, projectedEffect.getAmountActive())
                    
        for time in ("early", "normal", "late"):
            for layer in order:
                effects = runTime[layer][time]
                for stuff in effects:
                    try:
                        effect = stuff[0]
                        overrides = stuff[1] or {}
                        if len(filter(lambda n: n != "active", effect.type)) > 1:
                            overrides["activeLayer"] = layer
                        if len(stuff) >= 2: rest = stuff[2:]
                        if layer == "projected":
                            for i in range(rest[0]):
                                effect.func(effect, self, **overrides)
                        else:
                            if overrides:
                                effect.func(effect, self, **overrides)
                            else:
                                effect.func(effect, self)
                    except Exception:
                        sys.stderr.write("error while processing " + effect.name + "\n")
                        traceback.print_exc()
            
        if self.factorInReload:
            self.factorReloadTime()
    
    def factorReloadTime(self):
        for module in self.modules:
            item, ammo = module.getItem(), module.getAmmo()
            if ammo and item.getModifiedAttribute("_ammoConsumed"):
                if "duration" in item.attributes: rateAttr = item.attributes["duration"]
                else: rateAttr = item.attributes["speed"]
                rate = rateAttr.getModifiedValue()
                chargeSize = ammo.getModifiedAttribute("volume")
                containerSize = item.getModifiedAttribute("capacity")
                cyclesTillEmpty = int(containerSize / chargeSize)
                emptyTime = cyclesTillEmpty * rate + 10000
                rate = emptyTime / cyclesTillEmpty
                rateAttr.overrideValue = rate
    
    def block(self, item): self.blockedItems.add(item)
        
    def disableImpossibleMods(self):
        groups = {}
        changes = False
        for i in range(len(self.modules)):
            module = self.modules[i]
            item, state = module.getItem(), module.getState()
            if state >= STATE_ACTIVE:
                ##Check if the module has been blocked
                if item in self.blockedItems:
                    module.setState(STATE_INACTIVE)
                    changes = True
                
                ##Check if the module is over it's max limit
                maxGroupActive = item.getModifiedAttribute("maxGroupActive")
                if maxGroupActive != None:
                    if not item.group.name in groups: groups[item.group.name] = 0
                    groups[item.group.name] += 1
                    if groups[item.group.name] > maxGroupActive:
                        module.setState(STATE_INACTIVE)
                        changes = True
                        
        return changes
    
    def canAddModule(self, mod):
        '''
        Returns wether the passed module can be fit on this fit
        @param module: the module to check for
        '''
        moduleSlot = mod.getSlot()
        if self.getSlotsLeft(moduleSlot) <= 0: return False
        it = mod.getItem()
        
        #Check if the item is restricted to a certain ship
        fitsOn = it.getModifiedAttribute("fitsToShipType")
        if fitsOn != None and fitsOn != self.ship.ID: return False
        #more checking for different attributes doing the same thing
        for i in range(4):
            fitsOn = it.getModifiedAttribute("canFitShipType%d" % i)
            if fitsOn != None and fitsOn != self.ship.ID: return False
            
        #Check if the module is restricted to certain ship group(s)
        fitsToShipGroups = []
        for i in range(5):
            fitsOnGroup = it.getModifiedAttribute("canFitShipGroup" + str(i + 1))
            if fitsOnGroup: fitsToShipGroups.append(fitsOnGroup)
        if fitsToShipGroups and not self.ship.groupID in fitsToShipGroups: return False
        
        #If the mod is a subsystem, don't let two subs using the same slot fit
        subSysSlot = it.getModifiedAttribute("subSystemSlot")
        if subSysSlot:
            for currMod in self.modules:
                currSysSlot = currMod.getItem().getModifiedAttribute("subSystemSlot")
                if currSysSlot == subSysSlot: return False
                
        #Check rig sizes
        rigSize = it.getModifiedAttribute("rigSize")
        if rigSize:
            rigFits = rigSize == self.ship.getModifiedAttribute("rigSize")
            if rigFits == False: return rigFits
        
        #Check max group fitted
        maxGroupFitted= it.getModifiedAttribute("maxGroupFitted")
        num = 0
        if maxGroupFitted:
            for currMod in self.modules:
                if currMod.getItem().group == it.group: num += 1
                    
            if num >= maxGroupFitted: return False
        
        return True
    
    def addBooster(self, module, activeSideEffects = None):
        slot = module.getModifiedAttribute("boosterness")
        if slot == None: raise ValueError("Passed item is not a booster")
        self.boosters[slot] = (module, activeSideEffects or [])
        
    def addImplant(self, module):
        slot = module.getModifiedAttribute("implantness")
        if slot == None: raise ValueError("Passed item is not an implant")
        self.implants[slot] = module
        
    def addModule(self, mod):
        '''
        @param module: the module to add
        '''
        #Check if we got any dummies,  if so replace it
        for i in range(len(self.modules)):
            currMod = self.modules[i]
            item = currMod.getItem()
            if item.ID == 0 and currMod.getSlot() == mod.getSlot():
                self.modules[i] = mod
                return
        
        #Seems we didn't find a dummy
        self.modules.append(mod)
    
    def addDrone(self, dr):
        droneI = None
        for currDrone in self.drones:
            if currDrone.getItem() == dr:
                droneI = currDrone
                break
        
        if droneI: droneI.setAmountTotal(droneI.getAmountTotal() + 1)    
        else: self.drones.append(drone.drone(dr, self))
    
    def removeDrone(self, drone, removeActive):
        if drone.getAmountTotal() > 1:
            if removeActive: drone.setAmountActive(drone.getAmountActive() - 1)
            drone.setAmountTotal(drone.getAmountTotal() - 1)
            return False
        else:
            self.drones.remove(drone)
            return True
    
    def getNumActiveDrones(self):
        total = 0
        for drone in self.drones: total += drone.getAmountActive()
        return total
    
    def getSlotsLeft(self, type):
        try:
            total = self.ship.getModifiedAttribute(slotNameMap[type]) or 0
            used = self.getNumModulesAttributeValue("_slot", type)
            return total - used
        except KeyError:
            return 0
        
    def getCharSkill(self, name):
        if self.character == None: return None, 0
        else: return self.character.getSkill(name)
            
    def __getstate__(self):
        modules = []
        drones = {}
        for mod in self.modules:
            it = mod.getItem()
            if it.ID == 0: moduleID = mod.getSlot()
            else: moduleID = it.ID
            
            ammo = mod.getAmmo()
            if ammo: ammoID = ammo.ID
            else: ammoID = None
                
            modules.append((moduleID, mod.getState(), ammoID))
            
        for drone in self.drones:
            drones[drone.getItem().ID] = (drone.getAmountTotal(), drone.getAmountActive())

        projEffects = {}
        if self.projEffects:
            for projectedEffect in self.projEffects:
                source = projectedEffect.getSource()
                if isinstance(source, module.module): source = (source.getItem().ID, source.getState())
                projEffects[source] = (projectedEffect.getAmountTotal(), projectedEffect.getAmountActive())
        
        boosters = {}
        if self.boosters:
            for slot, (mod, activeSideEffects) in self.boosters.iteritems():
                boosters[slot] = (mod.ID, activeSideEffects)
                
        implants = {}
        if self.implants:
            for slot, mod in self.implants.iteritems():
                implants[slot] = mod.ID
                
        factorInReload = self.factorInReload
        
        return {'ship' : self.ship.ID,
                'modules' : modules,
                'drones' : drones,
                'projEffects' : projEffects,
                'implants' : implants,
                'boosters' : boosters,
                'gang' : self.gang,
                'factorInReload' : factorInReload}
        
    def __setstate__(self, state):
        ship = item.getItem(state["ship"])
        modules = []
        #See why we're keeping this cache a little lower, in the gang section
        mods = {}
        for (modID, stateID, ammoID) in state["modules"]:
            if isinstance(modID, str): mod = generateDummy(modID)
            else:
                it = item.getItem(modID)
                mods[modID] = it
                if ammoID: ammo = item.getItem(ammoID)
                else: ammo = None
                mod = module.module(it, self, stateID, ammo)
            modules.append(mod)
            
        drones = []
        for droneID, (amount, active) in state["drones"].iteritems():
            #Convert old format to new one
            if isinstance(active, bool):
                if active: active = amount
                else: active = 0

            dr = drone.drone(item.getItem(ID = droneID), self, amount, active)
            drones.append(dr)
            
        if "projEffects" in state:
            projEffects = []
            for mod, args in state["projEffects"].iteritems():
                amount, active = args[0:2]
                if isinstance(active, bool):
                    if active: active = amount
                    else: active = 0
                if isinstance(mod, int):
                    mod = module.module(item.getItem(ID = mod), self, STATE_ACTIVE)
                elif isinstance(mod, tuple): mod = module.module(item.getItem(ID = mod[0]), self, mod[1])
                elif isinstance(mod, item.item): mod = module.module(mod, self, STATE_ACTIVE)
                projEffects.append(projectedEffect.projectedEffect(mod, self, amount, active))
        else:
            projEffects = None
            
        if "factorInReload" in state:
            factorInReload = state["factorInReload"]
        else:
            factorInReload = False
    
        if "implants" in state:
            implants = {}
            for slot, modID in state["implants"].iteritems():
                mod = item.getItem(ID = modID)
                implants[slot] = mod
        else:
            implants = None
                
        if "boosters" in state:
            boosters = {}
            for slot, (modID, activeSideEffects) in state["boosters"].iteritems():
                mod = item.getItem(ID = modID)
                boosters[slot] = (mod, activeSideEffects)
        else:
            boosters = None
        
        if "gang" in state: gang = state["gang"]
        else: gang = None
        self.__init__(ship, modules, None, drones, projEffects, implants, boosters, gang, factorInReload)        
    
    def importEft(self, eftString, shipType = None):
        eftString = eftString.strip()
        typeNameRe = re.compile("\\[(.*), (.*)\\]")
        lines = re.split('[\n\r]+', eftString)
        if shipType == None:
            shipType, fitName = re.match(typeNameRe, lines[0]).groups()
        else:
            fitName = re.match('\[(.*)\]', lines[0]).group(1)
        
        ship = item.getItem(name = shipType)
        if ship == None:
            return None
        self.setShip(ship)
        
        for i in range(1, len(lines)):
            line = lines[i]
            modAmmo = line.split(",")
            modDrone = modAmmo[0].split(" x")
            if len(modAmmo) == 2: ammoName = modAmmo[1].strip()
            else: ammoName = None
            modName = modDrone[0].strip()
            if len(modDrone) == 2: droneAmount = modDrone[1].strip()
            else: droneAmount = None
            mod = item.getItem(name=modName)
            ##Temporary workaround for EFT not having new rig names
            if not mod: mod = item.getItem(name="Large " + modName)
            ##Workaround for items with " x" in them
            if not mod: mod = item.getItem(name= modAmmo[0])
            if mod:
                if ammoName:
                    ammo = item.getItem(name=ammoName)
                else:
                    ammo = None
                if mod.group.category.name == "Drone":
                    if droneAmount is None: droneAmount = 1
                    droneAmount = int(droneAmount)
                    self.drones.append(drone.drone(mod, self, droneAmount, 0))
                else: self.modules.append(module.module(mod, self, STATE_ACTIVE, ammo))
                
        return fitName
    
    def exportEft(self, name):
        export = "[%s, %s]\n" % (self.ship.name, name)
        stuff = {}
        for module in self.modules:
            slot = module.getSlot()
            if not slot in stuff: stuff[slot] = []
            curr = module.getItem().name
            if module.getAmmo():
                curr += ", %s" % module.getAmmo().name
            curr += "\n"
            stuff[slot].append(curr)
        
        for slot in slotTypes:
            if slot in stuff:
                export += "\n"
                for curr in stuff[slot]:
                    export += curr
                    
        export += "\n\n"
        for drone in self.drones:
            export += "%s x%s\n" % (drone.getItem().name, drone.getAmountTotal())
            
        return export

    def exportXml(self, name):        
        doc = xml.dom.minidom.Document()
        fittings = doc.createElement("fittings")
        doc.appendChild(fittings)
        fitting = doc.createElement("fitting")
        fitting.setAttribute("name", name)
        fittings.appendChild(fitting)
        description = doc.createElement("description")
        description.setAttribute("value", "")
        fitting.appendChild(description)
        shipType = doc.createElement("shipType")
        shipType.setAttribute("value", self.ship.name)
        fitting.appendChild(shipType)
        
        slotNum = {}
        for module in self.modules:
            item = module.getItem()
            slot = slotShortNameMap[module.getSlot()]
            if not slot in slotNum: slotNum[slot] = 0
            slotId = slotNum[slot]
            slotNum[slot] += 1
            hardware = doc.createElement("hardware")
            hardware.setAttribute("type", item.name)
            hardware.setAttribute("slot", "%s slot %d" % (slot, slotId))
            fitting.appendChild(hardware)
            
        for drone in self.drones:
            hardware = doc.createElement("hardware")
            hardware.setAttribute("qty", "%d" % drone.getAmountTotal())
            hardware.setAttribute("slot", "drone bay")
            hardware.setAttribute("type", drone.getItem().name)
            fitting.appendChild(hardware)
            
        return doc.toprettyxml()
    
    def exportStats(self, sections = None):
        stats = []
        
        resistances = "Format: EM Therm Kin Exp EHP"
        for layer, attrPrefix in (('shield', 'shield'), ('armor', 'armor'), ('hull', '')):
            resistances += "\n%s:" % layer.capitalize()
            for damageType in ('Em', 'Thermal', 'Kinetic', 'Explosive'):
                attrName = attrPrefix + damageType + 'DamageResonance'
                attrName = attrName[0].lower() + attrName[1:]
                resistances += " %.2f%%" % ((1 - self.ship.getModifiedAttribute(attrName)) * 100)
                
            ehp = self.ship.getModifiedAttribute("_" + attrPrefix.capitalize() + "Ehp")
            resistances += " %.0f" % ehp
        resistances += "\nTotal EHP: %.0f" % self.ship.getModifiedAttribute("_totalEhp")
        stats.append(("Resistances", resistances))
        
        tank = "Format: Burst Sustained\n"
        for tankType in ("passiveShield", "shield", "armor", "hull"):
            tank += "%s: " % tankType
            for subType in ("", "Sustained"):
                if subType == "Sustained" and tankType == "passiveShield":
                    continue
                attrName = "_" + tankType + subType + "Regen"
                t = self.ship.getModifiedAttribute(attrName) or 0
                tank += "%.2f " % t
            tank += "\n"
        stats.append(("Tank", tank))
        
        miningYield = self.getModuleAttributeSum("_miningYield") + self.getDronesAttributeSum("_miningYield", True)
        shipDps = self.getModuleAttributeSum("_dps")
        if shipDps or not miningYield:
            droneDps = self.getDronesAttributeSum("_dps", True)
            volley = self.getModuleAttributeSum("_volley")
            dps = "Ship DPS: %.2f\nDrone DPS: %.2f\nTotal DPS: %.2f\nShip Volley: %.0f" \
                % (shipDps, droneDps, shipDps + droneDps, volley)
            stats.append(("Firepower", dps))
        else:
            volley = self.getModuleAttributeSum("miningAmount")
            droneMps = self.getDronesAttributeSum("_miningYield", True)
            mps = "Ship MPS: %.2f\nDrone MPS: %.2f\nTotal MPS: %.2f\nShip Volley: %.0f" \
                % (miningYield, droneMps, miningYield + droneMps, volley)
            stats.append(("Minind Yield", mps))
        
        drain = self.ship.getModifiedAttribute("_drain") or 0
        capacity = self.ship.getModifiedAttribute("capacitorCapacity")
        recharge = self.ship.getModifiedAttribute("_peakCapRecharge")
        discharge = self.getModuleAttributeSum("_capUsage") + drain
        stable = self.ship.getModifiedAttribute("_capStable")
        state = self.ship.getModifiedAttribute("_capState")
        capacitor = "Capacity: %.0f\nState: %s\nRecharge: +%.1f\nDischarge: %.1f" \
            % (capacity, (stable == True and "Stable at %.1f%%" or "Lasts %.0f sec") % state, recharge, discharge)
        stats.append(("Capacitor", capacitor))
        
        maxTargets = min(self.ship.getModifiedAttribute("_maxLockedTargetsSkill"),
                     self.ship.getModifiedAttribute("maxLockedTargets"))
        
        scanResolution = self.ship.getModifiedAttribute("scanResolution")
        scanStr = 0
        t = None
        for scanType in ("Magnetometric", "Ladar", "Radar", "Gravimetric"):
            currScanStr = self.ship.getModifiedAttribute("scan" + scanType + "Strength")
            if currScanStr > scanStr:
                scanStr = currScanStr
                t = scanType
        targetRange = self.ship.getModifiedAttribute("maxTargetRange") / 1000
        targeting = "Targets: %d\nRange: %.2fkm\nScan Res: %.1f\nSensor str: %.0f\nSensor Type: %s" \
            % (maxTargets, targetRange, scanResolution, scanStr, t)
        stats.append(("Targeting", targeting))
        
        speed = self.ship.getModifiedAttribute("maxVelocity")
        speed = speed > 1000 and "%.3f km/s" % (speed / 1000) or "%.1f m/s" % speed
        cargo = self.ship.getModifiedAttribute("capacity")
        cargo = cargo > 1000 and "%.3f km3" % (cargo / 1000) or "%.1f m3" % cargo
        alignTime = self.ship.getModifiedAttribute("_alignTime")
        sigRadius = self.ship.getModifiedAttribute("signatureRadius")
        misc = "Speed: %s\nAlign Time: %.1f sec\nCargo Capacity: %s\nSignature Radius: %.0f m" \
            % (speed, alignTime, cargo, sigRadius)
        stats.append(("Misc", misc))
        statsString = ""
        for key, val in stats:
            if sections == None or key in sections:
                if statsString != "": statsString += "\n" * 2
                statsString += "%s\n%s\n%s" % (key, "=" * len(key), val)
        return statsString
        
    
    def importXml(self, text):
        doc = xml.dom.minidom.parseString(text)
        fittings = doc.getElementsByTagName("fittings").item(0)
        fitting = fittings.getElementsByTagName("fitting").item(0)
        name = fitting.getAttribute("name")
        shipType = fitting.getElementsByTagName("shipType").item(0)
        shipType = shipType.getAttribute("value")
        ship = item.getItem(name = shipType)
        self.setShip(ship)
        hardwares = fitting.getElementsByTagName("hardware")
        for hardware in hardwares:
            moduleName = hardware.getAttribute("type")
            quantity = hardware.getAttribute("qty")
            mod = item.getItem(name=moduleName)
            if mod:
                if mod.group.category.name == "Drone":
                    self.drones.append(drone.drone(mod, self, int(quantity), 0))
                else:
                    self.modules.append(module.module(mod, self, STATE_ACTIVE, None))
                
        return name
    
def generateDummy(slot):
    name = slot.capitalize() + " Slot"
    dummy = item.item(0, name, name)
    dummy.baseAttributes = {}
    dummy.attributes["_slot"] = attribute.basicAttribute(dummy, "_slot", slot, None)
    return module.module(dummy)


#Make sure to return the same dict across calls
savedData = None
def loadFits():
    '''
    Load fitting data from hard disk (location: ~/.pyfa/fits)
    '''
    global savedData
    if savedData == None:
        packedData = dataFolder.pickleFileToObj("fits")
        savedData = {}
        if packedData:
            for groupStuff, ships in packedData.iteritems():
                if isinstance(groupStuff, int): grp = group.getGroup(ID = groupStuff)
                else:
                    if groupStuff == "Supercarrier": grp = group.getGroup(ID = 659)
                    else: grp = group.getGroup(name=groupStuff)
                savedData[grp] = {}
                for shipID, fits in ships.iteritems():
                    ship = item.getItem(ID = shipID)
                    savedData[grp][ship] = {}
                    for fitName, fit in fits.iteritems():
                        savedData[grp][ship][fitName] = fit
        
    return savedData

def saveFits(fits):
    '''
    Save fitting data to hard disk (location: ~/.pyfa/fits)
    '''
    packedFits = {}
    for group, ships in fits.iteritems():
        packedFits[group.ID] = {}
        for ship, fits in ships.iteritems():
            packedFits[group.ID][ship.ID] = {}
            for fitName, fit in fits.iteritems():
                packedFits[group.ID][ship.ID][fitName] = fit
                
    return dataFolder.pickleObjToFile(packedFits, "fits")

def calcCapRechForGivenCap(capacitorState, capacitorCapacity, rechargeTime):
    '''
    Calculate the capacitor recharge rate for the passed capacitor state.
    @param capacitorState: the capacitor's current charge
    @param capacitorCapacity: the capacitor's capacity
    @param rechargeTime: the capacitor's recharge time
    '''
    return capacitorCapacity * ((4.9678 / rechargeTime) *
              (1 - (capacitorState / capacitorCapacity)) *
              math.sqrt((2 * (capacitorState / capacitorCapacity)) -
                        pow((capacitorState / capacitorCapacity), 2)))
