#Copyright 2009 Diego Duclos
#
#This program is free software: you can redistribute it and/or modify
#it under the terms of the GNU General Public License as published by
#the Free Software Foundation, either version 3 of the License, or
#(at your option) any later version.
#
#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.
#
#You should have received a copy of the GNU General Public License
#along with this program.  If not, see <http://www.gnu.org/licenses/>.

import sql
import category, graphic
from model.cache import cached_property, cached_query

baseQuery = '''
SELECT groupID,
categoryID,
groupName,
description,
published,
graphicID,
iconID,
dataID
FROM invgroups
'''
searchGroupQueryByName = baseQuery + " WHERE groupName = ?"
searchGroupQueryByID = baseQuery + " WHERE groupID = ?"
searchGroupQueryByCategoryID = baseQuery + " WHERE categoryID = ?"

class group(object):
    """
    The class used to represent a group
    """
    @cached_property
    def icon(self): return graphic.getIconName(self.iconID)
    
    @cached_property
    def category(self): return category.getCategory(ID = self.categoryID)
    
    def __init__(self, (ID, categoryID, name, description, published, graphicID, iconID, dataID)):
        '''
        Constructor
        @param ID: the ID of this group [from database]
        @param categoryID: the ID of the category of this group [from database]
        @param name: the name of this group [from database]
        @param description: the description of this group [from database]
        @param published: whether this group has been published [from database]
        @param graphicID: the graphicID of this group [from database]
        @param iconID: the iconID of this group [from database]
        @param dataID: the dataID of this group [from database]
        '''
        self.iconID = iconID
        self.ID = ID
        self.published = published
        self.categoryID = categoryID
        self.name = name
        self.description = description
    
    def __str__(self, level = 0):
        '''
        Returns a textual representation of this group
        @param level:
        '''
        formatting = ((level * 2) * " ") + "{0:15} = {1}\n"
        newline = ((level * 2) * " ") + "\n"
        str = formatting.format("ID", self.ID) + formatting.format("name", self.name) + formatting.format("description", self.description)
        return str
    
    def __eq__(self, other): return type(other) == type(self) and self.ID == other.ID
    def __ne__(self, other): return not self.__eq__(other)
    def __hash__(self):
        return self.ID
    
def getGroupsByCategory(ID = None):
    '''
    Get all groups that belong to a certain category
    @param ID: the ID of the category to get groups for
    '''
    c = cached_query(searchGroupQueryByCategoryID, (ID,))
    groupsList = []
    for row in c:
        grp = group(row)
        groupsList.append(grp)
    
    return groupsList
    
def getGroup(ID = None, name = None):
    '''
    Get a group from database. Either ID or name need to be specified NOT BOTH
    @param ID: the ID of the group to get
    @param name: the name of the group to get
    @param buildCategory: wether to build category info after fetching
    '''
    if ID != None and name == None: c = cached_query(searchGroupQueryByID, (ID,))
    elif name != None: c = cached_query(searchGroupQueryByName, (name,))
    else: return None
    row = c[0]
    if row == None: return None
    return group(row)