
///////////////////////////////////////////////////////////
//                                                       //
//                         SAGA                          //
//                                                       //
//      System for Automated Geoscientific Analyses      //
//                                                       //
//                     Tool Library                      //
//                                                       //
//                       io_gdal                         //
//                                                       //
//-------------------------------------------------------//
//                                                       //
//                    ogr_import.cpp                     //
//                                                       //
//            Copyright (C) 2008 O. Conrad               //
//                                                       //
//-------------------------------------------------------//
//                                                       //
// This file is part of 'SAGA - System for Automated     //
// Geoscientific Analyses'. SAGA is free software; you   //
// can redistribute it and/or modify it under the terms  //
// of the GNU General Public License as published by the //
// Free Software Foundation, either version 2 of the     //
// License, or (at your option) any later version.       //
//                                                       //
// SAGA is distributed in the hope that it will be       //
// useful, but WITHOUT ANY WARRANTY; without even the    //
// implied warranty of MERCHANTABILITY or FITNESS FOR A  //
// PARTICULAR PURPOSE. See the GNU General Public        //
// License for more details.                             //
//                                                       //
// You should have received a copy of the GNU General    //
// Public License along with this program; if not, see   //
// <http://www.gnu.org/licenses/>.                       //
//                                                       //
//-------------------------------------------------------//
//                                                       //
//    e-mail:     oconrad@saga-gis.de                    //
//                                                       //
//    contact:    Olaf Conrad                            //
//                Bundesstr. 55                          //
//                D-20146 Hamburg                        //
//                Germany                                //
//                                                       //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
#include "ogr_import.h"


///////////////////////////////////////////////////////////
//                                                       //
//                                                       //
//                                                       //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
COGR_Import::COGR_Import(void)
{
	Set_Name	(_TL("Import Shapes"));

	Set_Author	("O.Conrad (c) 2008");

	Add_Reference("GDAL/OGR contributors", "2019",
		"GDAL/OGR Geospatial Data Abstraction software Library",
		"A translator library for raster and vector geospatial data formats. Open Source Geospatial Foundation.",
		SG_T("https://gdal.org"), SG_T("Link")
	);

	Set_Description(_TW(
		"The \"OGR Vector Data Import\" tool imports vector data from various file/database formats using the "
		"\"Geospatial Data Abstraction Library\" (GDAL) by Frank Warmerdam. "
	));

	Add_Description(CSG_String::Format("\nGDAL %s:%s\n", _TL("Version"), SG_Get_OGR_Drivers().Get_Version().c_str()));

	Add_Description(_TL("Following vector formats are currently supported:"), false);
	Add_Description(CSG_String::Format("\n<table border=\"1\"><tr><th>%s</th><th>%s</th><th>%s</th></tr>",
		_TL("Name"), _TL("ID"), _TL("Extension")), false
	);

	CSG_String Filter[2]; CSG_Table Drivers;
	Drivers.Add_Field("Name"     , SG_DATATYPE_String);
	Drivers.Add_Field("ID"       , SG_DATATYPE_String);
	Drivers.Add_Field("Extension", SG_DATATYPE_String);

	for(int i=0; i<SG_Get_OGR_Drivers().Get_Count(); i++)
    {
		if( SG_Get_OGR_Drivers().is_Vector(i) && SG_Get_OGR_Drivers().Can_Read(i) )
		{
			CSG_Table_Record &Driver = *Drivers.Add_Record();
			Driver.Set_Value(0, SG_Get_OGR_Drivers().Get_Name       (i));
			Driver.Set_Value(1, SG_Get_OGR_Drivers().Get_Description(i));
			Driver.Set_Value(2, SG_Get_OGR_Drivers().Get_Extension  (i));
		}
	}

	Drivers.Sort(0);

	for(sLong i=0; i<Drivers.Get_Count(); i++)
	{
		CSG_String Name(Drivers[i].asString(0)), ID(Drivers[i].asString(1)), Extension(Drivers[i].asString(2));

		Add_Description("<tr><td>" + Name + "</td><td>" + ID  + "</td><td>" + Extension + "</td></tr>", false);

		if( !Extension.is_Empty() )
		{
			Extension.Replace("/", ";");

			Filter[0] += "|" + Name + "|*." + Extension;
			Filter[1] += Filter[1].is_Empty() ? "*." : ";*.";
			Filter[1] += Extension;
		}
	}

	Add_Description("</table>", false);

	Filter[0].Prepend(CSG_String::Format("%s|%s|%s|*.*", _TL("Recognized Files"), Filter[1].c_str(), _TL("All Files")));

	//-----------------------------------------------------
	Parameters.Add_Shapes_List("",
		"SHAPES"   , _TL("Shapes"),
		_TL(""),
		PARAMETER_OUTPUT
	);

	Parameters.Add_FilePath("",
		"FILES"    , _TL("Files"),
		_TL(""),
		Filter[0], NULL, false, false, true
	);

	Parameters.Add_Choice("",
		"GEOM_TYPE", _TL("Geometry Type"),
		_TL("Some OGR drivers are unable to determine the geometry type automatically, please choose the appropriate one in this case"),
		SG_Get_OGR_WKB_Type_Choices(), 0
	);
}


///////////////////////////////////////////////////////////
//                                                       //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
bool COGR_Import::On_Execute(void)
{
	CSG_Strings Files;

	if( !Parameters("FILES")->asFilePath()->Get_FilePaths(Files) )
	{
		return( false );
	}

	//-----------------------------------------------------
	Parameters("SHAPES")->asShapesList()->Del_Items();

	for(int iFile=0; iFile<Files.Get_Count(); iFile++)
	{
		Message_Fmt("\n%s: %s", _TL("loading"), Files[iFile].c_str());

		CSG_OGR_DataSet DataSource;

		if( !DataSource.Create(Files[iFile]) )
		{
			Message_Add(_TL("could not open data source"));
		}
		else if( DataSource.Get_Count() <= 0 )
		{
			Message_Add(_TL("no layers in data source"));
		}
		else for(int iLayer=0; iLayer<DataSource.Get_Count(); iLayer++)
		{
			CSG_Shapes *pShapes = DataSource.Read(iLayer, Parameters("GEOM_TYPE")->asInt());

			if( pShapes )
			{
				Parameters("SHAPES")->asShapesList()->Add_Item(pShapes);

				CSG_String Name = pShapes->Get_Name();

				pShapes->Get_MetaData().Add_Child("GDAL_DRIVER", DataSource.Get_DriverID());
				pShapes->Set_File_Name(Files[iFile]);
				pShapes->Set_Description(DataSource.Get_Description(iLayer));

				if( Name.is_Empty() )
				{
					pShapes->Set_Name(SG_File_Get_Name(Files[iFile], false) + (DataSource.Get_Count() == 1 ? CSG_String("") : CSG_String::Format(" [%d]", 1 + iLayer)));
				}
				else
				{
					pShapes->Set_Name(Name);
				}
			}
		}
	}

	//-----------------------------------------------------
	return( Parameters("SHAPES")->asShapesList()->Get_Item_Count() > 0 );
}


///////////////////////////////////////////////////////////
//                                                       //
//                                                       //
//                                                       //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
