/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2016, Hewlett Packard Enterprise Development LP
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_H
#define BLIS_H


// Allow C++ users to include this header file in their source code. However,
// we make the extern "C" conditional on whether we're using a C++ compiler,
// since regular C compilers don't understand the extern "C" construct.
#ifdef __cplusplus
extern "C" {
#endif

// NOTE: PLEASE DON'T CHANGE THE ORDER IN WHICH HEADERS ARE INCLUDED UNLESS
// YOU ARE SURE THAT IT DOESN'T BREAK INTER-HEADER MACRO DEPENDENCIES.

// -- configure definitions --

// NOTE: bli_config.h must be included before any other BLIS header. It is
// bootstrapped by ./configure and does not depend on later headers. Moreover
// these configuration variables are necessary to change some default behaviors
// (e.g. disable OS detection in bli_system.h in case of --disable-system).
// begin bli_config.h
#line 1 ".//bli_config.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_CONFIG_H
#define BLIS_CONFIG_H

// Enabled configuration "family" (config_name)
#define BLIS_FAMILY_GENERIC


// Enabled sub-configurations (config_list)
#define BLIS_CONFIG_GENERIC


// Enabled kernel sets (kernel_list)
#define BLIS_KERNELS_GENERIC


#define BLIS_VERSION_STRING "1.0"

#if 1
#define BLIS_ENABLE_SYSTEM
#else
#define BLIS_DISABLE_SYSTEM
#endif

#if 1
#define BLIS_ENABLE_TLS
#else
#define BLIS_DISABLE_TLS
#endif

#if 0
#define BLIS_ENABLE_OPENMP
#if 0
#define BLIS_ENABLE_OPENMP_AS_DEFAULT
#endif
#endif

#if 1
#define BLIS_ENABLE_PTHREADS
#if 1
#define BLIS_ENABLE_PTHREADS_AS_DEFAULT
#endif
#endif

#if 0
#define BLIS_ENABLE_HPX
#if 0
#define BLIS_ENABLE_HPX_AS_DEFAULT
#endif
#endif

#if 1
#define BLIS_ENABLE_JRIR_SLAB
#endif

#if 0
#define BLIS_ENABLE_JRIR_RR
#endif

#if 0
#define BLIS_ENABLE_JRIR_TLB
#endif

#if 1
#define BLIS_ENABLE_PBA_POOLS
#else
#define BLIS_DISABLE_PBA_POOLS
#endif

#if 1
#define BLIS_ENABLE_SBA_POOLS
#else
#define BLIS_DISABLE_SBA_POOLS
#endif

#if 0
#define BLIS_ENABLE_MEM_TRACING
#else
#define BLIS_DISABLE_MEM_TRACING
#endif

#if 0 == 64
#define BLIS_INT_TYPE_SIZE 64
#elif 0 == 32
#define BLIS_INT_TYPE_SIZE 32
#else
// determine automatically
#endif

#if 32 == 64
#define BLIS_BLAS_INT_TYPE_SIZE 64
#elif 32 == 32
#define BLIS_BLAS_INT_TYPE_SIZE 32
#else
// determine automatically
#endif

#ifndef BLIS_ENABLE_BLAS
#ifndef BLIS_DISABLE_BLAS
#if 1
#define BLIS_ENABLE_BLAS
#else
#define BLIS_DISABLE_BLAS
#endif
#endif
#endif

#ifndef BLIS_ENABLE_CBLAS
#ifndef BLIS_DISABLE_CBLAS
#if 0
#define BLIS_ENABLE_CBLAS
#else
#define BLIS_DISABLE_CBLAS
#endif
#endif
#endif

#ifndef BLIS_ENABLE_MIXED_DT
#ifndef BLIS_DISABLE_MIXED_DT
#if 1
#define BLIS_ENABLE_MIXED_DT
#else
#define BLIS_DISABLE_MIXED_DT
#endif
#endif
#endif

#ifndef BLIS_ENABLE_MIXED_DT_EXTRA_MEM
#ifndef BLIS_DISABLE_MIXED_DT_EXTRA_MEM
#if 1
#define BLIS_ENABLE_MIXED_DT_EXTRA_MEM
#else
#define BLIS_DISABLE_MIXED_DT_EXTRA_MEM
#endif
#endif
#endif

#if 1
#define BLIS_ENABLE_SUP_HANDLING
#else
#define BLIS_DISABLE_SUP_HANDLING
#endif

#if 0
#define BLIS_ENABLE_MEMKIND
#else
#define BLIS_DISABLE_MEMKIND
#endif

#if 1
#define BLIS_ENABLE_TRSM_PREINVERSION
#else
#define BLIS_DISABLE_TRSM_PREINVERSION
#endif

#if 1
#define BLIS_ENABLE_PRAGMA_OMP_SIMD
#else
#define BLIS_DISABLE_PRAGMA_OMP_SIMD
#endif

#if 0
#define BLIS_ENABLE_SANDBOX
#else
#define BLIS_DISABLE_SANDBOX
#endif

#if 1
#define BLIS_ENABLE_SHARED
#else
#define BLIS_DISABLE_SHARED
#endif

#if 0
#define BLIS_ENABLE_COMPLEX_RETURN_INTEL
#else
#define BLIS_DISABLE_COMPLEX_RETURN_INTEL
#endif


#endif
// end bli_config.h
#line 58 "./frame/include/blis.h"


// -- System and language-related headers --

// NOTE: bli_system.h header must be included before bli_config_macro_defs.h.
// begin bli_system.h
#line 1 "./frame/include//bli_system.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SYSTEM_H
#define BLIS_SYSTEM_H

// NOTE: If not yet defined, we define _POSIX_C_SOURCE to make sure that
// various parts of POSIX are defined and made available.
#ifndef _POSIX_C_SOURCE
#define _POSIX_C_SOURCE 200809L
#endif

#include <stdio.h> // skipped
#include <stdlib.h> // skipped
#include <math.h> // skipped
#include <string.h> // skipped
#include <stdarg.h> // skipped
#include <float.h> // skipped
#include <errno.h> // skipped
#include <ctype.h> // skipped

// Determine the compiler (hopefully) and define conveniently named macros
// accordingly.
#if   defined(__ICC) || defined(__INTEL_COMPILER)
  #define BLIS_ICC
#elif defined(__clang__)
  #define BLIS_CLANG
#elif defined(__GNUC__)
  #define BLIS_GCC
#endif

// Determine if we are on a 64-bit or 32-bit architecture.
#if defined(_M_X64) || defined(__x86_64) || defined(__aarch64__) || \
    defined(_ARCH_PPC64) || defined(__s390x__) || defined(_LP64)
  #define BLIS_ARCH_64
#else
  #define BLIS_ARCH_32
#endif

// Determine the target operating system.
#if defined(BLIS_ENABLE_SYSTEM)
  #if defined(_WIN32) || defined(__CYGWIN__)
    #define BLIS_OS_WINDOWS 1
  #elif defined(__gnu_hurd__)
    #define BLIS_OS_GNU 1
  #elif defined(__APPLE__) || defined(__MACH__)
    #define BLIS_OS_OSX 1
  #elif defined(__ANDROID__)
    #define BLIS_OS_ANDROID 1
  #elif defined(__linux__)
    #define BLIS_OS_LINUX 1
  #elif defined(__bgq__)
    #define BLIS_OS_BGQ 1
  #elif defined(__bg__)
    #define BLIS_OS_BGP 1
  #elif defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__) || \
        defined(__bsdi__) || defined(__DragonFly__) || \
        defined(__FreeBSD_kernel__) || defined(__HAIKU__)
    #define BLIS_OS_BSD 1
  #elif defined(EMSCRIPTEN)
    #define BLIS_OS_EMSCRIPTEN
  #else
    #error "Cannot determine operating system"
  #endif
#else // #if defined(BLIS_DISABLE_SYSTEM)
  #define BLIS_OS_NONE
#endif

// A few changes that may be necessary in Windows environments.
#if BLIS_OS_WINDOWS

  // Include Windows header file.
  #define WIN32_LEAN_AND_MEAN
  #define VC_EXTRALEAN
#include <windows.h> // skipped

  #if !defined(__clang__) && !defined(__GNUC__)
    // Undefine attribute specifiers in Windows.
    #define __attribute__(x)

    // Undefine restrict.
    #define restrict
  #endif

#endif

// time.h provides clock_gettime().
#if BLIS_OS_WINDOWS
#include <time.h> // skipped
#elif BLIS_OS_OSX
#include <mach/mach_time.h> // skipped
#else
  //#include <sys/time.h>

#include <time.h> // skipped
#endif


#endif
// end bli_system.h
#line 63 "./frame/include/blis.h"

// begin bli_lang_defs.h
#line 1 "./frame/include//bli_lang_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_LANG_DEFS_H
#define BLIS_LANG_DEFS_H


// -- Undefine restrict for C++ and C89/90 --

#ifdef __cplusplus
  // Language is C++; define restrict as nothing.
  #ifndef restrict
  #define restrict
  #endif
#elif __STDC_VERSION__ >= 199901L
  // Language is C99 (or later); do nothing since restrict is recognized.
#else
  // Language is pre-C99; define restrict as nothing.
  #ifndef restrict
  #define restrict
  #endif
#endif


// -- Define typeof() operator if using non-GNU compiler --

#ifndef __GNUC__
  #define typeof __typeof__
#else
  #ifndef typeof
  #define typeof __typeof__
  #endif
#endif


// -- BLIS Thread Local Storage Keyword --

// __thread for TLS is supported by GCC, CLANG, ICC, and IBMC.
// There is a small risk here as __GNUC__ can also be defined by some other
// compiler (other than ICC and CLANG which we know define it) that
// doesn't support __thread, as __GNUC__ is not quite unique to GCC.
// But the possibility of someone using such non-main-stream compiler
// for building BLIS is low.
#if defined(BLIS_ENABLE_TLS) && ( defined(__GNUC__)  || \
                                  defined(__clang__) || \
                                  defined(__ICC)     || \
                                  defined(__IBMC__) )
  #define BLIS_THREAD_LOCAL __thread
#else
  #define BLIS_THREAD_LOCAL
#endif


// -- BLIS constructor/destructor function attribute --

// __attribute__((constructor/destructor)) is supported by GCC only.
// There is a small risk here as __GNUC__ can also be defined by some other
// compiler (other than ICC and CLANG which we know define it) that
// doesn't support this, as __GNUC__ is not quite unique to GCC.
// But the possibility of someone using such non-main-stream compiler
// for building BLIS is low.

#if defined(__ICC) || defined(__INTEL_COMPILER)
  // ICC defines __GNUC__ but doesn't support this
  #define BLIS_ATTRIB_CTOR
  #define BLIS_ATTRIB_DTOR
#elif defined(__clang__)
  // CLANG supports __attribute__, but its documentation doesn't
  // mention support for constructor/destructor. Compiling with
  // clang and testing shows that it does support.
  #define BLIS_ATTRIB_CTOR __attribute__((constructor))
  #define BLIS_ATTRIB_DTOR __attribute__((destructor))
#elif defined(__GNUC__)
  #define BLIS_ATTRIB_CTOR __attribute__((constructor))
  #define BLIS_ATTRIB_DTOR __attribute__((destructor))
#else
  #define BLIS_ATTRIB_CTOR
  #define BLIS_ATTRIB_DTOR
#endif


#endif
// end bli_lang_defs.h
#line 64 "./frame/include/blis.h"



// -- configure default definitions --

// begin bli_config_macro_defs.h
#line 1 "./frame/include//bli_config_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_CONFIG_MACRO_DEFS_H
#define BLIS_CONFIG_MACRO_DEFS_H

// NOTE: This file should ONLY contain processing of macros that are set by
// configure and output into bli_config.h. Any other macro processing --
// especially such as for those macros that are expected to be optionally
// set within a configuration's bli_family_<conf>.h header -- MUST be placed
// in bli_kernel_macro_defs.h instead. The reason: bli_arch_config.h (which
// #includes the configuration's bli_family_<conf>.h header) is #included
// much later in blis.h than this file (bli_config_macro_defs.h), and so any
// macros set in bli_family_<conf>.h would have no effect on the processing
// that happens below.


// -- INTEGER PROPERTIES -------------------------------------------------------

// The bit size of the integer type used to track values such as dimensions,
// strides, diagonal offsets. A value of 32 results in BLIS using 32-bit signed
// integers while 64 results in 64-bit integers. Any other value results in use
// of the C99 type "long int". Note that this ONLY affects integers used
// internally within BLIS as well as those exposed in the native BLAS-like BLIS
// interface.
#ifndef BLIS_INT_TYPE_SIZE
  #ifdef BLIS_ARCH_64
    #define BLIS_INT_TYPE_SIZE 64
  #else
    #define BLIS_INT_TYPE_SIZE 32
  #endif
#endif


// -- FLOATING-POINT PROPERTIES ------------------------------------------------

// Enable use of built-in C99 "float complex" and "double complex" types and
// associated overloaded operations and functions? Disabling results in
// scomplex and dcomplex being defined in terms of simple structs.
// NOTE: AVOID USING THIS FEATURE. IT IS PROBABLY BROKEN.
#ifdef BLIS_ENABLE_C99_COMPLEX
  // No additional definitions needed.
#else
  // Default behavior is disabled.
#endif


// -- MEMORY SUBSYSTEM PROPERTIES ----------------------------------------------

// Size of a cache line (in bytes).
#ifndef BLIS_CACHE_LINE_SIZE
#define BLIS_CACHE_LINE_SIZE 64
#endif


// -- MULTITHREADING -----------------------------------------------------------

// Enable caching of queried cntx_t pointers in the gks?
#ifdef BLIS_DISABLE_GKS_CACHING
  #undef BLIS_ENABLE_GKS_CACHING
#else
  // Default behavior is enabled.
  #define BLIS_ENABLE_GKS_CACHING
#endif


// -- MULTITHREADING -----------------------------------------------------------

// Enable multithreading via POSIX threads.
#ifdef BLIS_ENABLE_PTHREADS
  // No additional definitions needed.
#else
  // Default behavior is disabled.
#endif

// Enable multithreading via OpenMP.
#ifdef BLIS_ENABLE_OPENMP
  // No additional definitions needed.
#else
  // Default behavior is disabled.
#endif

// Enable multithreading via HPX.
#ifdef BLIS_ENABLE_HPX
  // No additional definitions needed.
#else
  // Default behavior is disabled.
#endif

// Here, we define BLIS_ENABLE_MULTITHREADING if either OpenMP
// or pthreads are enabled. This macro is useful in situations when
// we want to detect use of either OpenMP or pthreads, or both (as
// opposed to neither being used).
#if defined ( BLIS_ENABLE_OPENMP )   || \
    defined ( BLIS_ENABLE_PTHREADS ) || \
    defined ( BLIS_ENABLE_HPX )
  #define BLIS_ENABLE_MULTITHREADING
#endif

// Enable the use of prime numbers of threads when requesting automatic thread
// factorization. When disabled, requesting a prime number of threads will
// result in a reduction (by one) of the number of threads, provided that the
// prime number exceeds a minimum threshold (see below).
#ifdef BLIS_ENABLE_AUTO_PRIME_NUM_THREADS
  #undef BLIS_DISABLE_AUTO_PRIME_NUM_THREADS
#else
  // Default behavior is disabled.
  #undef  BLIS_DISABLE_AUTO_PRIME_NUM_THREADS // In case user explicitly disabled.
  #define BLIS_DISABLE_AUTO_PRIME_NUM_THREADS
#endif

// Set the maximum requested number of threads that BLIS will accept from the
// user that may be prime. If a larger prime number of threads is requested,
// it will be reduced by one to allow for more efficient thread factorizations.
// This value will only be used if BLIS_ENABLE_AUTO_PRIME_NUM_THREADS is defined.
#ifndef BLIS_NT_MAX_PRIME
  #define BLIS_NT_MAX_PRIME 11
#endif


// -- MIXED DATATYPE SUPPORT ---------------------------------------------------

// Enable mixed datatype support?
#ifdef BLIS_DISABLE_MIXED_DT
  #undef BLIS_ENABLE_GEMM_MD
#else
  // Default behavior is enabled.
  #define BLIS_ENABLE_GEMM_MD
#endif

// Enable memory-intensive optimizations for mixed datatype support?
#ifdef BLIS_DISABLE_MIXED_DT_EXTRA_MEM
  #undef BLIS_ENABLE_GEMM_MD_EXTRA_MEM
#else
  // Default behavior is enabled.
  #define BLIS_ENABLE_GEMM_MD_EXTRA_MEM
#endif


// -- MISCELLANEOUS OPTIONS ----------------------------------------------------

// Do NOT require the cross-blocksize constraints. That is, do not enforce
// MC % NR = 0 and NC % MR = 0 in bli_kernel_macro_defs.h. These are ONLY
// needed when implementing trsm_r by allowing the right-hand matrix B to
// be triangular.
#ifndef BLIS_RELAX_MCNR_NCMR_CONSTRAINTS
  #define BLIS_RELAX_MCNR_NCMR_CONSTRAINTS
#endif


// -- BLAS COMPATIBILITY LAYER -------------------------------------------------

// Enable the BLAS compatibility layer?
#ifdef BLIS_DISABLE_BLAS
  #undef BLIS_ENABLE_BLAS
#else
  // Default behavior is enabled.
  #undef  BLIS_ENABLE_BLAS // In case user explicitly enabled.
  #define BLIS_ENABLE_BLAS
#endif

// The bit size of the integer type used to track values such as dimensions and
// leading dimensions (ie: column strides) within the BLAS compatibility layer.
// A value of 32 results in the compatibility layer using 32-bit signed integers
// while 64 results in 64-bit integers. Any other value results in use of the
// C99 type "long int". Note that this ONLY affects integers used within the
// BLAS compatibility layer.
#ifndef BLIS_BLAS_INT_TYPE_SIZE
  #define BLIS_BLAS_INT_TYPE_SIZE 32
#endif

// By default, the level-3 BLAS routines are implemented by directly calling
// the BLIS object API. Alternatively, they may first call the typed BLIS
// API, which will then call the object API.
//#define BLIS_BLAS3_CALLS_TAPI
#ifdef BLIS_BLAS3_CALLS_TAPI
  #undef  BLIS_BLAS3_CALLS_OAPI
#else
  // Default behavior is to call object API directly.
  #undef  BLIS_BLAS3_CALLS_OAPI // In case user explicitly enabled.
  #define BLIS_BLAS3_CALLS_OAPI
#endif


// -- CBLAS COMPATIBILITY LAYER ------------------------------------------------

// Enable the CBLAS compatibility layer?
// NOTE: Enabling CBLAS will automatically enable the BLAS compatibility layer
// regardless of whether or not it was explicitly enabled above. Furthermore,
// the CBLAS compatibility layer will use the integer type size definition
// specified above when defining the size of its own integers (regardless of
// whether the BLAS layer was enabled directly or indirectly).
#ifdef BLIS_ENABLE_CBLAS
  // No additional definitions needed.
#else
  // Default behavior is disabled.
#endif


// -- SHARED LIBRARY SYMBOL EXPORT ---------------------------------------------

// When building shared libraries, we can control which symbols are exported for
// linking by external applications. BLIS annotates all function prototypes that
// are meant to be "public" with BLIS_EXPORT_BLIS (with BLIS_EXPORT_BLAS playing
// a similar role for BLAS compatibility routines). Which symbols are exported
// is controlled by the default symbol visibility, as specifed by the gcc option
// -fvisibility=[default|hidden]. The default for this option is 'default', or,
// "public", which, if allowed to stand, causes all symbols in BLIS to be
// linkable from the outside. But when compiling with -fvisibility=hidden, all
// symbols start out hidden (that is, restricted only for internal use by BLIS),
// with that setting overridden only for function prototypes or variable
// declarations that are annotated with BLIS_EXPORT_BLIS.

#ifndef BLIS_EXPORT
  #if !defined(BLIS_ENABLE_SHARED)
    #define BLIS_EXPORT
  #else
    #if defined(_WIN32) || defined(__CYGWIN__)
      #ifdef BLIS_IS_BUILDING_LIBRARY
        #define BLIS_EXPORT __declspec(dllexport)
      #else
        #define BLIS_EXPORT __declspec(dllimport)
      #endif
    #elif defined(__GNUC__) && __GNUC__ >= 4
      #define BLIS_EXPORT __attribute__ ((visibility ("default")))
    #else
      #define BLIS_EXPORT
    #endif
  #endif
#endif

#define BLIS_EXPORT_BLIS  BLIS_EXPORT
#define BLIS_EXPORT_BLAS  BLIS_EXPORT
#define BLIS_EXPORT_ADDON BLIS_EXPORT


// -- OVERRIDABLE (WEAK) SYMBOLS -----------------------------------------------

// On Linux, functions called from a shared library can be overriden by the main
// program simply by providing a new definition. However, macOS uses a "two-level
// namespace" which causes calls to shared library functions to be tied to the
// library and not overridable. As a workaround, certain symbols can be defined
// as "weak" and are given lower preference during linking.
#ifndef BLIS_OVERRIDABLE
#if BLIS_OS_OSX
#define BLIS_OVERRIDABLE __attribute__((weak))
#else
#define BLIS_OVERRIDABLE
#endif
#endif


// -- STATIC INLINE FUNCTIONS --------------------------------------------------

// C and C++ have different semantics for defining "inline" functions. In C,
// the keyword phrase "static inline" accomplishes this, though the "inline"
// is optional. In C++, the "inline" keyword is required and obviates "static"
// altogether. Why does this matter? While BLIS is compiled in C99, blis.h may
// be #included by a source file that is compiled with C++.
#ifdef __cplusplus
  #define BLIS_INLINE inline
#else
  //#define BLIS_INLINE static inline
  #define BLIS_INLINE static
#endif


#endif

// end bli_config_macro_defs.h
#line 69 "./frame/include/blis.h"



// -- Common BLIS definitions --

// begin bli_type_defs.h
#line 1 "./frame/include//bli_type_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2016, Hewlett Packard Enterprise Development LP
   Copyright (C) 2020, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_TYPE_DEFS_H
#define BLIS_TYPE_DEFS_H


//
// -- BLIS basic types ---------------------------------------------------------
//

#ifdef __cplusplus
  // For C++, include stdint.h.
#include <cstdint> // skipped
#elif __STDC_VERSION__ >= 199901L
  // For C99 (or later), include stdint.h.
#include <stddef.h> // skipped
#include <stdint.h> // skipped
#include <stdbool.h> // skipped
#else
  // When stdint.h is not available, manually typedef the types we will use.
  #ifdef _WIN32
    typedef          __int32  int32_t;
    typedef unsigned __int32 uint32_t;
    typedef          __int64  int64_t;
    typedef unsigned __int64 uint64_t;
  #else
    #error "Attempting to compile on pre-C99 system without stdint.h."
  #endif
#endif

// -- General-purpose integers --

// If BLAS integers are 64 bits, mandate that BLIS integers also be 64 bits.
// NOTE: This cpp guard will only meaningfully change BLIS's behavior on
// systems where the BLIS integer size would have been automatically selected
// to be 32 bits, since explicit selection of 32 bits is prohibited at
// configure-time (and explicit or automatic selection of 64 bits is fine
// and would have had the same result).
#if BLIS_BLAS_INT_TYPE_SIZE == 64
  #undef  BLIS_INT_TYPE_SIZE
  #define BLIS_INT_TYPE_SIZE 64
#endif

// Define integer types depending on what size integer was requested.
#if   BLIS_INT_TYPE_SIZE == 32
typedef           int32_t  gint_t;
typedef          uint32_t guint_t;
#elif BLIS_INT_TYPE_SIZE == 64
typedef           int64_t  gint_t;
typedef          uint64_t guint_t;
#else
typedef   signed long int  gint_t;
typedef unsigned long int guint_t;
#endif

// -- Boolean type --

// NOTE: bool_t is no longer used and has been replaced with C99's bool type.
//typedef bool bool_t;

// BLIS uses TRUE and FALSE macro constants as possible boolean values, but we
// define these macros in terms of true and false, respectively, which are
// defined by C99 in stdbool.h.
#ifndef TRUE
  #define TRUE  true
#endif

#ifndef FALSE
  #define FALSE false
#endif

// -- Special-purpose integers --

// This cpp guard provides a temporary hack to allow libflame
// interoperability with BLIS.
#ifndef _DEFINED_DIM_T
#define _DEFINED_DIM_T
typedef   gint_t dim_t;      // dimension type
#endif
typedef   gint_t inc_t;      // increment/stride type
typedef   gint_t doff_t;     // diagonal offset type
typedef  guint_t siz_t;      // byte size type
typedef uint32_t objbits_t;  // object information bit field

// -- Real types --

// Define the number of floating-point types supported, and the size of the
// largest type.
#define BLIS_NUM_FP_TYPES   4
#define BLIS_MAX_TYPE_SIZE  sizeof(dcomplex)

// There are some places where we need to use sizeof() inside of a C
// preprocessor #if conditional, and so here we define the various sizes
// for those purposes.
#define BLIS_SIZEOF_S      4  // sizeof(float)
#define BLIS_SIZEOF_D      8  // sizeof(double)
#define BLIS_SIZEOF_C      8  // sizeof(scomplex)
#define BLIS_SIZEOF_Z      16 // sizeof(dcomplex)

// -- Complex types --

#ifdef BLIS_ENABLE_C99_COMPLEX

	#if __STDC_VERSION__ >= 199901L
#include <complex.h> // skipped

		// Typedef official complex types to BLIS complex type names.
		typedef  float complex scomplex;
		typedef double complex dcomplex;
	#else
		#error "Configuration requested C99 complex types, but C99 does not appear to be supported."
	#endif

#else // ifndef BLIS_ENABLE_C99_COMPLEX

	// This cpp guard provides a temporary hack to allow libflame
	// interoperability with BLIS.
	#ifndef _DEFINED_SCOMPLEX
	#define _DEFINED_SCOMPLEX
	typedef struct scomplex
	{
		float  real;
		float  imag;
	} scomplex;
	#endif

	// This cpp guard provides a temporary hack to allow libflame
	// interoperability with BLIS.
	#ifndef _DEFINED_DCOMPLEX
	#define _DEFINED_DCOMPLEX
	typedef struct dcomplex
	{
		double real;
		double imag;
	} dcomplex;
	#endif

#endif // BLIS_ENABLE_C99_COMPLEX

// -- Atom type --

// Note: atom types are used to hold "bufferless" scalar object values. Note
// that it needs to be as large as the largest possible scalar value we might
// want to hold. Thus, for now, it is a dcomplex.
typedef dcomplex atom_t;

// -- Fortran-77 types --

// Note: These types are typically only used by BLAS compatibility layer, but
// we must define them even when the compatibility layer isn't being built
// because they also occur in bli_slamch() and bli_dlamch().

// Define f77_int depending on what size of integer was requested.
#if   BLIS_BLAS_INT_TYPE_SIZE == 32
typedef int32_t   f77_int;
#elif BLIS_BLAS_INT_TYPE_SIZE == 64
typedef int64_t   f77_int;
#else
typedef long int  f77_int;
#endif

typedef char      f77_char;
typedef float     f77_float;
typedef double    f77_double;
typedef scomplex  f77_scomplex;
typedef dcomplex  f77_dcomplex;

// -- Misc. function pointer types --

// Note: This type should be used in any situation where the address of a
// *function* will be conveyed or stored prior to it being typecast back
// to the correct function type. It does not need to be used when conveying
// or storing the address of *data* (such as an array of float or double).
//typedef void (*void_fp)( void );
typedef void* void_fp;

// Typedef function pointer types for malloc() and free() substitutes.
typedef void* (*malloc_ft)( size_t size );
typedef void  (*free_ft)  ( void*  p    );


//
// -- BLIS info bit field offsets ----------------------------------------------
//

/*
  info field description

  bit(s)   purpose
  -------  -------
   2 ~ 0   Stored numerical datatype
           - 0: domain    (0 == real, 1 == complex)
           - 1: precision (0 == single, 1 == double)
           - 2: special   (100 = int; 101 = const)
       3   Transposition required [during pack]?
       4   Conjugation required [during pack]?
   7 ~ 5   Part of matrix stored:
           - 5: strictly upper triangular
           - 6: diagonal
           - 7: strictly lower triangular
       8   Implicit unit diagonal?
       9   Invert diagonal required [during pack]?
  12 ~ 10  Target numerical datatype
           - 10: domain    (0 == real, 1 == complex)
           - 11: precision (0 == single, 1 == double)
           - 12: used to encode integer, constant types
  15 ~ 13  Execution numerical datatype
           - 13: domain    (0 == real, 1 == complex)
           - 14: precision (0 == single, 1 == double)
           - 15: used to encode integer, constant types
  22 ~ 16  Packed type/status
           - 0 0000 00: not packed
           - 1 0000 00: packed (unspecified; by rows, columns, or vector)
           - 1 0000 00: packed by rows
           - 1 0000 01: packed by columns
           - 1 0000 10: packed by row panels
           - 1 0000 11: packed by column panels
           - 1 0001 10: packed by 1m expanded row panels
           - 1 0001 11: packed by 1m expanded column panels
           - 1 0010 10: packed by 1m reordered row panels
           - 1 0010 11: packed by 1m reordered column panels
       23  Packed panel order if upper-stored
           - 0 == forward order if upper
           - 1 == reverse order if upper
       24  Packed panel order if lower-stored
           - 0 == forward order if lower
           - 1 == reverse order if lower
  26 ~ 25  Packed buffer type
           - 0 == block of A
           - 1 == panel of B
           - 2 == panel of C
           - 3 == general use
  28 ~ 27  Structure type
           - 0 == general
           - 1 == Hermitian
           - 2 == symmetric
           - 3 == triangular
  31 ~ 29  Computation numerical datatype
           - 29: domain    (0 == real, 1 == complex)
           - 30: precision (0 == single, 1 == double)
           - 31: used to encode integer, constant types

  info2 field description

  bit(s)   purpose
  -------  -------
    2 ~ 0  Scalar storage numerical datatype
           -  0: domain    (0 == real, 1 == complex)
           -  1: precision (0 == single, 1 == double)
           -  2: used to encode integer, constant types
*/

// info
#define BLIS_DATATYPE_SHIFT                0
#define   BLIS_DOMAIN_SHIFT                0
#define   BLIS_PRECISION_SHIFT             1
#define BLIS_CONJTRANS_SHIFT               3
#define   BLIS_TRANS_SHIFT                 3
#define   BLIS_CONJ_SHIFT                  4
#define BLIS_UPLO_SHIFT                    5
#define   BLIS_UPPER_SHIFT                 5
#define   BLIS_DIAG_SHIFT                  6
#define   BLIS_LOWER_SHIFT                 7
#define BLIS_UNIT_DIAG_SHIFT               8
#define BLIS_INVERT_DIAG_SHIFT             9
#define BLIS_TARGET_DT_SHIFT               10
#define   BLIS_TARGET_DOMAIN_SHIFT         10
#define   BLIS_TARGET_PREC_SHIFT           11
#define BLIS_EXEC_DT_SHIFT                 13
#define   BLIS_EXEC_DOMAIN_SHIFT           13
#define   BLIS_EXEC_PREC_SHIFT             14
#define BLIS_PACK_SCHEMA_SHIFT             16
#define   BLIS_PACK_RC_SHIFT               16
#define   BLIS_PACK_PANEL_SHIFT            17
#define   BLIS_PACK_FORMAT_SHIFT           18
#define   BLIS_PACK_SHIFT                  22
#define BLIS_PACK_REV_IF_UPPER_SHIFT       23
#define BLIS_PACK_REV_IF_LOWER_SHIFT       24
#define BLIS_PACK_BUFFER_SHIFT             25
#define BLIS_STRUC_SHIFT                   27
#define BLIS_COMP_DT_SHIFT                 29
#define   BLIS_COMP_DOMAIN_SHIFT           29
#define   BLIS_COMP_PREC_SHIFT             30

// info2
#define BLIS_SCALAR_DT_SHIFT                0
#define   BLIS_SCALAR_DOMAIN_SHIFT          0
#define   BLIS_SCALAR_PREC_SHIFT            1

//
// -- BLIS info bit field masks ------------------------------------------------
//

// info
#define BLIS_DATATYPE_BITS                 ( 0x7  << BLIS_DATATYPE_SHIFT )
#define   BLIS_DOMAIN_BIT                  ( 0x1  << BLIS_DOMAIN_SHIFT )
#define   BLIS_PRECISION_BIT               ( 0x1  << BLIS_PRECISION_SHIFT )
#define BLIS_CONJTRANS_BITS                ( 0x3  << BLIS_CONJTRANS_SHIFT )
#define   BLIS_TRANS_BIT                   ( 0x1  << BLIS_TRANS_SHIFT )
#define   BLIS_CONJ_BIT                    ( 0x1  << BLIS_CONJ_SHIFT )
#define BLIS_UPLO_BITS                     ( 0x7  << BLIS_UPLO_SHIFT )
#define   BLIS_UPPER_BIT                   ( 0x1  << BLIS_UPPER_SHIFT )
#define   BLIS_DIAG_BIT                    ( 0x1  << BLIS_DIAG_SHIFT )
#define   BLIS_LOWER_BIT                   ( 0x1  << BLIS_LOWER_SHIFT )
#define BLIS_UNIT_DIAG_BIT                 ( 0x1  << BLIS_UNIT_DIAG_SHIFT )
#define BLIS_INVERT_DIAG_BIT               ( 0x1  << BLIS_INVERT_DIAG_SHIFT )
#define BLIS_TARGET_DT_BITS                ( 0x7  << BLIS_TARGET_DT_SHIFT )
#define   BLIS_TARGET_DOMAIN_BIT           ( 0x1  << BLIS_TARGET_DOMAIN_SHIFT )
#define   BLIS_TARGET_PREC_BIT             ( 0x1  << BLIS_TARGET_PREC_SHIFT )
#define BLIS_EXEC_DT_BITS                  ( 0x7  << BLIS_EXEC_DT_SHIFT )
#define   BLIS_EXEC_DOMAIN_BIT             ( 0x1  << BLIS_EXEC_DOMAIN_SHIFT )
#define   BLIS_EXEC_PREC_BIT               ( 0x1  << BLIS_EXEC_PREC_SHIFT )
#define BLIS_PACK_SCHEMA_BITS              ( 0x7F << BLIS_PACK_SCHEMA_SHIFT )
#define   BLIS_PACK_RC_BIT                 ( 0x1  << BLIS_PACK_RC_SHIFT )
#define   BLIS_PACK_PANEL_BIT              ( 0x1  << BLIS_PACK_PANEL_SHIFT )
#define   BLIS_PACK_FORMAT_BITS            ( 0xF  << BLIS_PACK_FORMAT_SHIFT )
#define   BLIS_PACK_BIT                    ( 0x1  << BLIS_PACK_SHIFT )
#define BLIS_PACK_REV_IF_UPPER_BIT         ( 0x1  << BLIS_PACK_REV_IF_UPPER_SHIFT )
#define BLIS_PACK_REV_IF_LOWER_BIT         ( 0x1  << BLIS_PACK_REV_IF_LOWER_SHIFT )
#define BLIS_PACK_BUFFER_BITS              ( 0x3  << BLIS_PACK_BUFFER_SHIFT )
#define BLIS_STRUC_BITS                    ( 0x3  << BLIS_STRUC_SHIFT )
#define BLIS_COMP_DT_BITS                  ( 0x7  << BLIS_COMP_DT_SHIFT )
#define   BLIS_COMP_DOMAIN_BIT             ( 0x1  << BLIS_COMP_DOMAIN_SHIFT )
#define   BLIS_COMP_PREC_BIT               ( 0x1  << BLIS_COMP_PREC_SHIFT )

// info2
#define BLIS_SCALAR_DT_BITS                ( 0x7  << BLIS_SCALAR_DT_SHIFT )
#define   BLIS_SCALAR_DOMAIN_BIT           ( 0x1  << BLIS_SCALAR_DOMAIN_SHIFT )
#define   BLIS_SCALAR_PREC_BIT             ( 0x1  << BLIS_SCALAR_PREC_SHIFT )


//
// -- BLIS enumerated type value definitions -----------------------------------
//

#define BLIS_BITVAL_REAL                      0x0
#define BLIS_BITVAL_COMPLEX                   BLIS_DOMAIN_BIT
#define BLIS_BITVAL_SINGLE_PREC               0x0
#define BLIS_BITVAL_DOUBLE_PREC               BLIS_PRECISION_BIT
#define   BLIS_BITVAL_FLOAT_TYPE              0x0
#define   BLIS_BITVAL_SCOMPLEX_TYPE           BLIS_DOMAIN_BIT
#define   BLIS_BITVAL_DOUBLE_TYPE             BLIS_PRECISION_BIT
#define   BLIS_BITVAL_DCOMPLEX_TYPE         ( BLIS_DOMAIN_BIT | BLIS_PRECISION_BIT )
#define   BLIS_BITVAL_INT_TYPE                0x04
#define   BLIS_BITVAL_CONST_TYPE              0x05
#define BLIS_BITVAL_NO_TRANS                  0x0
#define BLIS_BITVAL_TRANS                     BLIS_TRANS_BIT
#define BLIS_BITVAL_NO_CONJ                   0x0
#define BLIS_BITVAL_CONJ                      BLIS_CONJ_BIT
#define BLIS_BITVAL_CONJ_TRANS              ( BLIS_CONJ_BIT | BLIS_TRANS_BIT )
#define BLIS_BITVAL_ZEROS                     0x0
#define BLIS_BITVAL_UPPER                   ( BLIS_UPPER_BIT | BLIS_DIAG_BIT )
#define BLIS_BITVAL_LOWER                   ( BLIS_LOWER_BIT | BLIS_DIAG_BIT )
#define BLIS_BITVAL_DENSE                     BLIS_UPLO_BITS
#define BLIS_BITVAL_NONUNIT_DIAG              0x0
#define BLIS_BITVAL_UNIT_DIAG                 BLIS_UNIT_DIAG_BIT
#define BLIS_BITVAL_INVERT_DIAG               BLIS_INVERT_DIAG_BIT
#define BLIS_BITVAL_NOT_PACKED                0x0
#define   BLIS_BITVAL_1E                    ( 0x1  << BLIS_PACK_FORMAT_SHIFT )
#define   BLIS_BITVAL_1R                    ( 0x2  << BLIS_PACK_FORMAT_SHIFT )
#define   BLIS_BITVAL_PACKED_UNSPEC         ( BLIS_PACK_BIT                                                            )
#define   BLIS_BITVAL_PACKED_ROWS           ( BLIS_PACK_BIT                                                            )
#define   BLIS_BITVAL_PACKED_COLUMNS        ( BLIS_PACK_BIT                                         | BLIS_PACK_RC_BIT )
#define   BLIS_BITVAL_PACKED_ROW_PANELS     ( BLIS_PACK_BIT                   | BLIS_PACK_PANEL_BIT                    )
#define   BLIS_BITVAL_PACKED_COL_PANELS     ( BLIS_PACK_BIT                   | BLIS_PACK_PANEL_BIT | BLIS_PACK_RC_BIT )
#define   BLIS_BITVAL_PACKED_ROW_PANELS_1E  ( BLIS_PACK_BIT | BLIS_BITVAL_1E  | BLIS_PACK_PANEL_BIT                    )
#define   BLIS_BITVAL_PACKED_COL_PANELS_1E  ( BLIS_PACK_BIT | BLIS_BITVAL_1E  | BLIS_PACK_PANEL_BIT | BLIS_PACK_RC_BIT )
#define   BLIS_BITVAL_PACKED_ROW_PANELS_1R  ( BLIS_PACK_BIT | BLIS_BITVAL_1R  | BLIS_PACK_PANEL_BIT                    )
#define   BLIS_BITVAL_PACKED_COL_PANELS_1R  ( BLIS_PACK_BIT | BLIS_BITVAL_1R  | BLIS_PACK_PANEL_BIT | BLIS_PACK_RC_BIT )
#define BLIS_BITVAL_PACK_FWD_IF_UPPER         0x0
#define BLIS_BITVAL_PACK_REV_IF_UPPER         BLIS_PACK_REV_IF_UPPER_BIT
#define BLIS_BITVAL_PACK_FWD_IF_LOWER         0x0
#define BLIS_BITVAL_PACK_REV_IF_LOWER         BLIS_PACK_REV_IF_LOWER_BIT
#define BLIS_BITVAL_BUFFER_FOR_A_BLOCK        0x0
#define BLIS_BITVAL_BUFFER_FOR_B_PANEL      ( 0x1 << BLIS_PACK_BUFFER_SHIFT )
#define BLIS_BITVAL_BUFFER_FOR_C_PANEL      ( 0x2 << BLIS_PACK_BUFFER_SHIFT )
#define BLIS_BITVAL_BUFFER_FOR_GEN_USE      ( 0x3 << BLIS_PACK_BUFFER_SHIFT )
#define BLIS_BITVAL_GENERAL                   0x0
#define BLIS_BITVAL_HERMITIAN               ( 0x1 << BLIS_STRUC_SHIFT )
#define BLIS_BITVAL_SYMMETRIC               ( 0x2 << BLIS_STRUC_SHIFT )
#define BLIS_BITVAL_TRIANGULAR              ( 0x3 << BLIS_STRUC_SHIFT )


//
// -- BLIS enumerated type definitions -----------------------------------------
//

// -- Operational parameter types --

typedef enum
{
	BLIS_NO_TRANSPOSE      = 0x0,
	BLIS_TRANSPOSE         = BLIS_BITVAL_TRANS,
	BLIS_CONJ_NO_TRANSPOSE = BLIS_BITVAL_CONJ,
	BLIS_CONJ_TRANSPOSE    = BLIS_BITVAL_CONJ_TRANS
} trans_t;

typedef enum
{
	BLIS_NO_CONJUGATE      = 0x0,
	BLIS_CONJUGATE         = BLIS_BITVAL_CONJ
} conj_t;

typedef enum
{
	BLIS_ZEROS             = BLIS_BITVAL_ZEROS,
	BLIS_LOWER             = BLIS_BITVAL_LOWER,
	BLIS_UPPER             = BLIS_BITVAL_UPPER,
	BLIS_DENSE             = BLIS_BITVAL_DENSE
} uplo_t;

typedef enum
{
	BLIS_LEFT              = 0x0,
	BLIS_RIGHT
} side_t;

typedef enum
{
	BLIS_NONUNIT_DIAG      = 0x0,
	BLIS_UNIT_DIAG         = BLIS_BITVAL_UNIT_DIAG
} diag_t;

typedef enum
{
	BLIS_NO_INVERT_DIAG    = 0x0,
	BLIS_INVERT_DIAG       = BLIS_BITVAL_INVERT_DIAG
} invdiag_t;

typedef enum
{
	BLIS_GENERAL           = BLIS_BITVAL_GENERAL,
	BLIS_HERMITIAN         = BLIS_BITVAL_HERMITIAN,
	BLIS_SYMMETRIC         = BLIS_BITVAL_SYMMETRIC,
	BLIS_TRIANGULAR        = BLIS_BITVAL_TRIANGULAR
} struc_t;


// -- Data type --

typedef enum
{
	BLIS_FLOAT             = BLIS_BITVAL_FLOAT_TYPE,
	BLIS_DOUBLE            = BLIS_BITVAL_DOUBLE_TYPE,
	BLIS_SCOMPLEX          = BLIS_BITVAL_SCOMPLEX_TYPE,
	BLIS_DCOMPLEX          = BLIS_BITVAL_DCOMPLEX_TYPE,
	BLIS_INT               = BLIS_BITVAL_INT_TYPE,
	BLIS_CONSTANT          = BLIS_BITVAL_CONST_TYPE,
	BLIS_DT_LO             = BLIS_FLOAT,
	BLIS_DT_HI             = BLIS_DCOMPLEX
} num_t;

typedef enum
{
	BLIS_REAL              = BLIS_BITVAL_REAL,
	BLIS_COMPLEX           = BLIS_BITVAL_COMPLEX
} dom_t;

typedef enum
{
	BLIS_SINGLE_PREC       = BLIS_BITVAL_SINGLE_PREC,
	BLIS_DOUBLE_PREC       = BLIS_BITVAL_DOUBLE_PREC
} prec_t;


// -- Pack schema type --

typedef enum
{
	BLIS_NOT_PACKED            = BLIS_BITVAL_NOT_PACKED,
	BLIS_PACKED_UNSPEC         = BLIS_BITVAL_PACKED_UNSPEC,
	BLIS_PACKED_VECTOR         = BLIS_BITVAL_PACKED_UNSPEC,
	BLIS_PACKED_ROWS           = BLIS_BITVAL_PACKED_ROWS,
	BLIS_PACKED_COLUMNS        = BLIS_BITVAL_PACKED_COLUMNS,
	BLIS_PACKED_ROW_PANELS     = BLIS_BITVAL_PACKED_ROW_PANELS,
	BLIS_PACKED_COL_PANELS     = BLIS_BITVAL_PACKED_COL_PANELS,
	BLIS_PACKED_ROW_PANELS_1E  = BLIS_BITVAL_PACKED_ROW_PANELS_1E,
	BLIS_PACKED_COL_PANELS_1E  = BLIS_BITVAL_PACKED_COL_PANELS_1E,
	BLIS_PACKED_ROW_PANELS_1R  = BLIS_BITVAL_PACKED_ROW_PANELS_1R,
	BLIS_PACKED_COL_PANELS_1R  = BLIS_BITVAL_PACKED_COL_PANELS_1R
} pack_t;

// We combine row and column packing into one "type", and we start
// with BLIS_PACKED_ROW_PANELS, _COLUMN_PANELS.
#define BLIS_NUM_PACK_SCHEMA_TYPES 3


// -- Pack order type --

typedef enum
{
	BLIS_PACK_FWD_IF_UPPER = BLIS_BITVAL_PACK_FWD_IF_UPPER,
	BLIS_PACK_REV_IF_UPPER = BLIS_BITVAL_PACK_REV_IF_UPPER,

	BLIS_PACK_FWD_IF_LOWER = BLIS_BITVAL_PACK_FWD_IF_LOWER,
	BLIS_PACK_REV_IF_LOWER = BLIS_BITVAL_PACK_REV_IF_LOWER
} packord_t;


// -- Pack buffer type --

typedef enum
{
	BLIS_BUFFER_FOR_A_BLOCK = BLIS_BITVAL_BUFFER_FOR_A_BLOCK,
	BLIS_BUFFER_FOR_B_PANEL = BLIS_BITVAL_BUFFER_FOR_B_PANEL,
	BLIS_BUFFER_FOR_C_PANEL = BLIS_BITVAL_BUFFER_FOR_C_PANEL,
	BLIS_BUFFER_FOR_GEN_USE = BLIS_BITVAL_BUFFER_FOR_GEN_USE
} packbuf_t;


// -- Partitioning direction --

typedef enum
{
	BLIS_FWD,
	BLIS_BWD
} dir_t;


// -- Subpartition type --

typedef enum
{
	BLIS_SUBPART0,
	BLIS_SUBPART1,
	BLIS_SUBPART2,
	BLIS_SUBPART1AND0,
	BLIS_SUBPART1AND2,
	BLIS_SUBPART1A,
	BLIS_SUBPART1B,
	BLIS_SUBPART00,
	BLIS_SUBPART10,
	BLIS_SUBPART20,
	BLIS_SUBPART01,
	BLIS_SUBPART11,
	BLIS_SUBPART21,
	BLIS_SUBPART02,
	BLIS_SUBPART12,
	BLIS_SUBPART22
} subpart_t;


// -- Matrix dimension type --

typedef enum
{
	BLIS_M = 0,
	BLIS_N = 1
} mdim_t;


// -- Machine parameter types --

typedef enum
{
	BLIS_MACH_EPS = 0,
	BLIS_MACH_SFMIN,
	BLIS_MACH_BASE,
	BLIS_MACH_PREC,
	BLIS_MACH_NDIGMANT,
	BLIS_MACH_RND,
	BLIS_MACH_EMIN,
	BLIS_MACH_RMIN,
	BLIS_MACH_EMAX,
	BLIS_MACH_RMAX,
	BLIS_MACH_EPS2
} machval_t;

#define BLIS_NUM_MACH_PARAMS   11
#define BLIS_MACH_PARAM_FIRST  BLIS_MACH_EPS
#define BLIS_MACH_PARAM_LAST   BLIS_MACH_EPS2


// -- Induced method types --

typedef enum
{
	BLIS_1M        = 0,
	BLIS_NAT,
	BLIS_IND_FIRST = 0,
	BLIS_IND_LAST  = BLIS_NAT
} ind_t;

#define BLIS_NUM_IND_METHODS (BLIS_NAT+1)

// These are used in bli_l3_*_oapi.c to construct the ind_t values from
// the induced method substrings that go into function names.
#define bli_1m   BLIS_1M
#define bli_nat  BLIS_NAT


// -- Threading implementation type --

typedef enum
{
	BLIS_SINGLE = 0,
	BLIS_OPENMP,
	BLIS_POSIX,
	BLIS_HPX,

	// BLIS_NUM_THREAD_IMPLS must be last!
	BLIS_NUM_THREAD_IMPLS

} timpl_t;


// -- Kernel ID types --

typedef enum
{
	// l1v kernels
	BLIS_ADDV_KER,
	BLIS_AMAXV_KER,
	BLIS_AXPBYV_KER,
	BLIS_AXPYV_KER,
	BLIS_COPYV_KER,
	BLIS_DOTV_KER,
	BLIS_DOTXV_KER,
	BLIS_INVERTV_KER,
	BLIS_INVSCALV_KER,
	BLIS_SCALV_KER,
	BLIS_SCAL2V_KER,
	BLIS_SETV_KER,
	BLIS_SUBV_KER,
	BLIS_SWAPV_KER,
	BLIS_XPBYV_KER,
	BLIS_AXPY2V_KER,
	BLIS_DOTAXPYV_KER,

	// l1f kernels
	BLIS_AXPYF_KER,
	BLIS_DOTXF_KER,
	BLIS_DOTXAXPYF_KER,

	// pack kernels
	BLIS_PACKM_MRXK_KER,
	BLIS_PACKM_NRXK_KER,
	BLIS_PACKM_MRXK_1ER_KER,
	BLIS_PACKM_NRXK_1ER_KER,
	BLIS_PACKM_MRXMR_DIAG_KER,
	BLIS_PACKM_NRXNR_DIAG_KER,
	BLIS_PACKM_MRXMR_DIAG_1ER_KER,
	BLIS_PACKM_NRXNR_DIAG_1ER_KER,

	// unpack kernels
	BLIS_UNPACKM_MRXK_KER,
	BLIS_UNPACKM_NRXK_KER,

	// l3 native kernels
	BLIS_GEMM_UKR,
	BLIS_GEMMTRSM_L_UKR,
	BLIS_GEMMTRSM_U_UKR,
	BLIS_TRSM_L_UKR,
	BLIS_TRSM_U_UKR,

	// l3 virtual kernels
	BLIS_GEMM_VIR_UKR,
	BLIS_GEMMTRSM_L_VIR_UKR,
	BLIS_GEMMTRSM_U_VIR_UKR,
	BLIS_TRSM_L_VIR_UKR,
	BLIS_TRSM_U_VIR_UKR,

	// gemmsup kernels
	BLIS_GEMMSUP_RRR_UKR,
	BLIS_GEMMSUP_RRC_UKR,
	BLIS_GEMMSUP_RCR_UKR,
	BLIS_GEMMSUP_RCC_UKR,
	BLIS_GEMMSUP_CRR_UKR,
	BLIS_GEMMSUP_CRC_UKR,
	BLIS_GEMMSUP_CCR_UKR,
	BLIS_GEMMSUP_CCC_UKR,
	BLIS_GEMMSUP_XXX_UKR,

	// BLIS_NUM_UKRS must be last!
	BLIS_NUM_UKRS
} ukr_t;


typedef enum
{
    // l3 kernel row preferences
	BLIS_GEMM_UKR_ROW_PREF,
	BLIS_GEMMTRSM_L_UKR_ROW_PREF,
	BLIS_GEMMTRSM_U_UKR_ROW_PREF,
	BLIS_TRSM_L_UKR_ROW_PREF,
	BLIS_TRSM_U_UKR_ROW_PREF,

    // gemmsup kernel row preferences
	BLIS_GEMMSUP_RRR_UKR_ROW_PREF,
	BLIS_GEMMSUP_RRC_UKR_ROW_PREF,
	BLIS_GEMMSUP_RCR_UKR_ROW_PREF,
	BLIS_GEMMSUP_RCC_UKR_ROW_PREF,
	BLIS_GEMMSUP_CRR_UKR_ROW_PREF,
	BLIS_GEMMSUP_CRC_UKR_ROW_PREF,
	BLIS_GEMMSUP_CCR_UKR_ROW_PREF,
	BLIS_GEMMSUP_CCC_UKR_ROW_PREF,
	BLIS_GEMMSUP_XXX_UKR_ROW_PREF,

    // BLIS_NUM_UKR_PREFS must be last!
    BLIS_NUM_UKR_PREFS
} ukr_pref_t;


typedef enum
{
	BLIS_REFERENCE_UKERNEL = 0,
	BLIS_VIRTUAL_UKERNEL,
	BLIS_OPTIMIZED_UKERNEL,
	BLIS_NOTAPPLIC_UKERNEL
} kimpl_t;

#define BLIS_NUM_UKR_IMPL_TYPES 4


#if 0
typedef enum
{
	// RV = row-stored, contiguous vector-loading
	// RG = row-stored, non-contiguous gather-loading
	// CV = column-stored, contiguous vector-loading
	// CG = column-stored, non-contiguous gather-loading

	// RD = row-stored, dot-based
	// CD = col-stored, dot-based

	// RC = row-stored, column-times-column
	// CR = column-stored, row-times-row

	// GX = general-stored generic implementation

	BLIS_GEMMSUP_RV_UKR = 0,
	BLIS_GEMMSUP_RG_UKR,
	BLIS_GEMMSUP_CV_UKR,
	BLIS_GEMMSUP_CG_UKR,

	BLIS_GEMMSUP_RD_UKR,
	BLIS_GEMMSUP_CD_UKR,

	BLIS_GEMMSUP_RC_UKR,
	BLIS_GEMMSUP_CR_UKR,

	BLIS_GEMMSUP_GX_UKR,
} l3sup_t;

#define BLIS_NUM_LEVEL3_SUP_UKRS 9
#endif


typedef enum
{
	// 3-operand storage combinations
	BLIS_RRR = 0,
	BLIS_RRC, // 1
	BLIS_RCR, // 2
	BLIS_RCC, // 3
	BLIS_CRR, // 4
	BLIS_CRC, // 5
	BLIS_CCR, // 6
	BLIS_CCC, // 7
	BLIS_XXX, // 8

#if 0
	BLIS_RRG,
	BLIS_RCG,
	BLIS_RGR,
	BLIS_RGC,
	BLIS_RGG,
	BLIS_CRG,
	BLIS_CCG,
	BLIS_CGR,
	BLIS_CGC,
	BLIS_CGG,
	BLIS_GRR,
	BLIS_GRC,
	BLIS_GRG,
	BLIS_GCR,
	BLIS_GCC,
	BLIS_GCG,
	BLIS_GGR,
	BLIS_GGC,
	BLIS_GGG,
#endif
} stor3_t;

#define BLIS_NUM_3OP_RC_COMBOS 9
//#define BLIS_NUM_3OP_RCG_COMBOS 27


#if 0
typedef enum
{
	BLIS_JC_IDX = 0,
	BLIS_PC_IDX,
	BLIS_IC_IDX,
	BLIS_JR_IDX,
	BLIS_IR_IDX,
	BLIS_PR_IDX
} thridx_t;
#endif

#define BLIS_NUM_LOOPS 6


// -- Operation ID type --

typedef enum
{
//
// NOTE: If/when additional type values are added to this enum,
// you must either:
// - keep the level-3 values (starting with _GEMM) beginning at
//   index 0; or
// - if the value range is moved such that it does not begin at
//   index 0, implement something like a BLIS_OPID_LEVEL3_RANGE_START
//   value that can be subtracted from the opid_t value to map it
//   to a zero-based range.
// This is needed because these level-3 opid_t values are used in
// bli_l3_ind.c to index into arrays.
//
	BLIS_GEMM = 0,
	BLIS_GEMMT,
	BLIS_HEMM,
	BLIS_HERK,
	BLIS_HER2K,
	BLIS_SYMM,
	BLIS_SYRK,
	BLIS_SYR2K,
	BLIS_TRMM3,
	BLIS_TRMM,
	BLIS_TRSM,

	BLIS_NOID
} opid_t;

#define BLIS_NUM_LEVEL3_OPS 11


// -- Blocksize ID type --

typedef enum
{
	// NOTE: the level-3 blocksizes MUST be indexed starting at zero.
	// At one point, we made this assumption in bli_cntx_set_blkszs()
	// and friends.
	BLIS_KR,
	BLIS_MR,
	BLIS_NR,
	BLIS_MC,
	BLIS_KC,
	BLIS_NC,

	// broadcast factors for packing
	BLIS_BBM,
	BLIS_BBN,

	// level-2 blocksizes
	BLIS_M2, // level-2 blocksize in m dimension
	BLIS_N2, // level-2 blocksize in n dimension

	// level-1f blocksizes
	BLIS_AF, // level-1f axpyf fusing factor
	BLIS_DF, // level-1f dotxf fusing factor
	BLIS_XF, // level-1f dotxaxpyf fusing factor

	// gemmsup thresholds
	BLIS_MT, // level-3 small/unpacked matrix threshold in m dimension
	BLIS_NT, // level-3 small/unpacked matrix threshold in n dimension
	BLIS_KT, // level-3 small/unpacked matrix threshold in k dimension

	// gemmsup block sizes
	BLIS_KR_SUP,
	BLIS_MR_SUP,
	BLIS_NR_SUP,
	BLIS_MC_SUP,
	BLIS_KC_SUP,
	BLIS_NC_SUP,

	// BLIS_NO_PART (= BLIS_NUM_BLKSZS) must be last!
	BLIS_NO_PART, // used as a placeholder when blocksizes are not applicable,
	              // such as when characterizing a packm operation.
	BLIS_NUM_BLKSZS = BLIS_NO_PART
} bszid_t;


// -- Architecture ID type --

// NOTE: This typedef enum must be kept up-to-date with the arch_t
// string array in bli_arch.c. Whenever values are added/inserted
// OR if values are rearranged, be sure to update the string array
// in bli_arch.c.

typedef enum
{
	// NOTE: The C language standard guarantees that the first enum value
	// starts at 0.

	// Intel
	BLIS_ARCH_SKX,
	BLIS_ARCH_KNL,
	BLIS_ARCH_KNC,
	BLIS_ARCH_HASWELL,
	BLIS_ARCH_SANDYBRIDGE,
	BLIS_ARCH_PENRYN,

	// AMD
	BLIS_ARCH_ZEN3,
	BLIS_ARCH_ZEN2,
	BLIS_ARCH_ZEN,
	BLIS_ARCH_EXCAVATOR,
	BLIS_ARCH_STEAMROLLER,
	BLIS_ARCH_PILEDRIVER,
	BLIS_ARCH_BULLDOZER,

	// ARM-SVE
	BLIS_ARCH_ARMSVE,
	BLIS_ARCH_A64FX,

	// ARM-NEON (4 pipes x 128-bit vectors)
	BLIS_ARCH_ALTRAMAX,
	BLIS_ARCH_ALTRA,
	BLIS_ARCH_FIRESTORM,

	// ARM (2 pipes x 128-bit vectors)
	BLIS_ARCH_THUNDERX2,
	BLIS_ARCH_CORTEXA57,
	BLIS_ARCH_CORTEXA53,

	// ARM 32-bit (vintage)
	BLIS_ARCH_CORTEXA15,
	BLIS_ARCH_CORTEXA9,

	// IBM/Power
	BLIS_ARCH_POWER10,
	BLIS_ARCH_POWER9,
	BLIS_ARCH_POWER7,
	BLIS_ARCH_BGQ,

	// RISC-V
	BLIS_ARCH_RV32I,
	BLIS_ARCH_RV64I,
	BLIS_ARCH_RV32IV,
	BLIS_ARCH_RV64IV,

	// SiFive
	BLIS_ARCH_SIFIVE_X280,

	// Generic architecture/configuration
	BLIS_ARCH_GENERIC,

	// The total number of defined architectures. This must be last in the
	// list of enums since its definition assumes that the previous enum
	// value (BLIS_ARCH_GENERIC) is given index num_archs-1.
	BLIS_NUM_ARCHS

} arch_t;


//
// -- BLIS misc. structure types -----------------------------------------------
//

// This header must be included here (or earlier) because definitions it
// provides are needed in the pool_t and related structs.
// begin bli_pthread.h
#line 1 "./frame/thread//bli_pthread.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2018, Southern Methodist University
   Copyright (C) 2018, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_PTHREAD_H
#define BLIS_PTHREAD_H

// -- Type and macro definitions -----------------------------------------------

#if defined(BLIS_DISABLE_SYSTEM)

// This branch defines a pthread-like API, bli_pthread_*(), and implements it
// in terms of "dummy" code that doesn't depend on POSIX threads or any other
// threading mechanism. See issue #454 to see the use case that prompted this
// feature.
// NOTE: THIS CODE DOES NOT IMPLEMENT THREADING AND IS NOT THREAD-SAFE!

// -- pthread types --

typedef int bli_pthread_t;
typedef int bli_pthread_attr_t;
typedef int bli_pthread_mutex_t;
typedef int bli_pthread_mutexattr_t;
typedef int bli_pthread_cond_t;
typedef int bli_pthread_condattr_t;
typedef int bli_pthread_once_t;

typedef int bli_pthread_barrier_t;
typedef int bli_pthread_barrierattr_t;

// -- pthreads macros --

#define BLIS_PTHREAD_MUTEX_INITIALIZER 0
#define BLIS_PTHREAD_COND_INITIALIZER  0
#define BLIS_PTHREAD_ONCE_INIT         0

#elif defined(_MSC_VER) // !defined(BLIS_DISABLE_SYSTEM)

// This branch defines a pthread-like API, bli_pthread_*(), and implements it
// in terms of Windows API calls.

// -- pthread types --

typedef struct
{
    HANDLE handle;
    void* retval;
} bli_pthread_t;
typedef void bli_pthread_attr_t;
typedef SRWLOCK bli_pthread_mutex_t;
typedef void bli_pthread_mutexattr_t;
typedef CONDITION_VARIABLE bli_pthread_cond_t;
typedef void bli_pthread_condattr_t;
typedef INIT_ONCE bli_pthread_once_t;
typedef struct
{
    bli_pthread_mutex_t mutex;
    bli_pthread_cond_t  cond;
    int                 count;
    int                 tripCount;
} bli_pthread_barrier_t;
typedef void bli_pthread_barrierattr_t;

// -- pthreads macros --

#define BLIS_PTHREAD_MUTEX_INITIALIZER SRWLOCK_INIT
#define BLIS_PTHREAD_ONCE_INIT         INIT_ONCE_STATIC_INIT
#define BLIS_PTHREAD_COND_INITIALIZER  CONDITION_VARIABLE_INIT

#else // !defined(BLIS_DISABLE_SYSTEM) && !defined(_MSC_VER)

#include <pthread.h> // skipped

// This branch defines a pthreads-like API, bli_pthreads_*(), and implements it
// in terms of the corresponding pthreads_*() types, macros, and function calls. 

// -- pthread types --

typedef pthread_t           bli_pthread_t;
typedef pthread_attr_t      bli_pthread_attr_t;
typedef pthread_mutex_t     bli_pthread_mutex_t;
typedef pthread_mutexattr_t bli_pthread_mutexattr_t;
typedef pthread_cond_t      bli_pthread_cond_t;
typedef pthread_condattr_t  bli_pthread_condattr_t;
typedef pthread_once_t      bli_pthread_once_t;

#if defined(__APPLE__)

// For OS X, we must define the barrier types ourselves since Apple does
// not implement barriers in their variant of pthreads.

typedef void bli_pthread_barrierattr_t;

typedef struct
{
	bli_pthread_mutex_t mutex;
	bli_pthread_cond_t  cond;
	int                 count;
	int                 tripCount;
} bli_pthread_barrier_t;

#else

// For other non-Windows OSes (primarily Linux), we can define the barrier
// types in terms of existing pthreads barrier types since we expect they
// will be provided by the pthreads implementation.

typedef pthread_barrier_t      bli_pthread_barrier_t;
typedef pthread_barrierattr_t  bli_pthread_barrierattr_t;

#endif

// -- pthreads macros --

#define BLIS_PTHREAD_MUTEX_INITIALIZER PTHREAD_MUTEX_INITIALIZER
#define BLIS_PTHREAD_COND_INITIALIZER  PTHREAD_COND_INITIALIZER
#define BLIS_PTHREAD_ONCE_INIT         PTHREAD_ONCE_INIT

#endif

// -- Function definitions -----------------------------------------------------

// -- pthread_create(), pthread_join() --

BLIS_EXPORT_BLIS int bli_pthread_create
     (
       bli_pthread_t*            thread,
       const bli_pthread_attr_t* attr,
       void*                   (*start_routine)(void*),
       void*                     arg
     );

BLIS_EXPORT_BLIS int bli_pthread_join
     (
       bli_pthread_t thread,
       void**        retval
     );

// -- pthread_mutex_*() --

BLIS_EXPORT_BLIS int bli_pthread_mutex_init
     (
       bli_pthread_mutex_t*           mutex,
       const bli_pthread_mutexattr_t* attr
     );

BLIS_EXPORT_BLIS int bli_pthread_mutex_destroy
     (
       bli_pthread_mutex_t* mutex
     );

BLIS_EXPORT_BLIS int bli_pthread_mutex_lock
     (
       bli_pthread_mutex_t* mutex
     );

BLIS_EXPORT_BLIS int bli_pthread_mutex_trylock
     (
       bli_pthread_mutex_t* mutex
     );

BLIS_EXPORT_BLIS int bli_pthread_mutex_unlock
     (
       bli_pthread_mutex_t* mutex
     );

// -- pthread_cond_*() --

BLIS_EXPORT_BLIS int bli_pthread_cond_init
     (
       bli_pthread_cond_t*           cond,
       const bli_pthread_condattr_t* attr
     );

BLIS_EXPORT_BLIS int bli_pthread_cond_destroy
     (
       bli_pthread_cond_t* cond
     );

BLIS_EXPORT_BLIS int bli_pthread_cond_wait
     (
       bli_pthread_cond_t*  cond,
       bli_pthread_mutex_t* mutex
     );

BLIS_EXPORT_BLIS int bli_pthread_cond_broadcast
     (
       bli_pthread_cond_t* cond
     );

// -- pthread_once() --

BLIS_EXPORT_BLIS void bli_pthread_once
     (
       bli_pthread_once_t* once,
       void              (*init)(void)
     );

#if 0
// NOTE: This part of the API is disabled because (1) we don't actually need
// _self() or _equal() yet, and (2) when we do try to include these functions,
// AppVeyor for some reason fails on all the Windows/clang builds with the
// error:
//    libblis.a(bli_pthread.o) : error LNK2019: unresolved external symbol
//     __imp_CompareObjectHandles referenced in function bli_pthread_equal

// -- pthread_self() --

BLIS_EXPORT_BLIS bli_pthread_t bli_pthread_self
     (
       void
     );

// -- pthread_equal() --

BLIS_EXPORT_BLIS int bli_pthread_equal
     (
       bli_pthread_t t1,
       bli_pthread_t t2
     );
#endif

// -- pthread_barrier_*() --

BLIS_EXPORT_BLIS int bli_pthread_barrier_init
     (
       bli_pthread_barrier_t*           barrier,
       const bli_pthread_barrierattr_t* attr,
       unsigned int                     count
     );

BLIS_EXPORT_BLIS int bli_pthread_barrier_destroy
     (
       bli_pthread_barrier_t* barrier
     );

BLIS_EXPORT_BLIS int bli_pthread_barrier_wait
     (
       bli_pthread_barrier_t* barrier
     );

// -- Non-standard extensions --------------------------------------------------

// -- pthread_switch --

typedef struct
{
    int                 status;
    bli_pthread_mutex_t mutex;
} bli_pthread_switch_t;

#define BLIS_PTHREAD_SWITCH_INIT { .status = 0, \
                                   .mutex  = BLIS_PTHREAD_MUTEX_INITIALIZER }

int bli_pthread_switch_on
     (
       bli_pthread_switch_t* sw,
       int                 (*init)(void)
     );

int bli_pthread_switch_off
     (
       bli_pthread_switch_t* sw,
       int                 (*deinit)(void)
     );

#endif // BLIS_PTHREAD_H
// end bli_pthread.h
#line 999 "./frame/include//bli_type_defs.h"


// -- Pool block type --

typedef struct
{
	void*     buf;
	siz_t     block_size;

} pblk_t;


// -- Pool type --

typedef struct
{
	void*     block_ptrs;
	dim_t     block_ptrs_len;

	dim_t     top_index;
	dim_t     num_blocks;

	siz_t     block_size;
	siz_t     align_size;
	siz_t     offset_size;

	malloc_ft malloc_fp;
	free_ft   free_fp;

} pool_t;


// -- Array type --

typedef struct
{
	void*     buf;

	siz_t     num_elem;
	siz_t     elem_size;

} array_t;


// -- Locked pool-of-arrays-of-pools type --

typedef struct
{
	bli_pthread_mutex_t mutex;
	pool_t              pool;

	siz_t               def_array_len;

} apool_t;


// -- packing block allocator: Locked set of pools type --

typedef struct pba_s
{
	pool_t              pools[3];
	bli_pthread_mutex_t mutex;

	// These fields are used for general-purpose allocation.
	siz_t               align_size;
	malloc_ft           malloc_fp;
	free_ft             free_fp;

} pba_t;


// -- Memory object type --

typedef struct mem_s
{
	pblk_t    pblk;
	packbuf_t buf_type;
	pool_t*   pool;
	siz_t     size;
} mem_t;


// -- Control tree node type --

struct cntl_s
{
	// Basic fields (usually required).
	opid_t         family;
	bszid_t        bszid;
	void_fp        var_func;
	struct cntl_s* sub_prenode;
	struct cntl_s* sub_node;
	void*          params;
};
typedef struct cntl_s cntl_t;


// -- Blocksize object type --

typedef struct blksz_s
{
	// Primary blocksize values.
	dim_t  v[BLIS_NUM_FP_TYPES];

	// Blocksize extensions.
	dim_t  e[BLIS_NUM_FP_TYPES];

} blksz_t;


// -- Function pointer object type --

typedef struct func_s
{
	// Kernel function address.
	void_fp ptr[BLIS_NUM_FP_TYPES];

} func_t;


// -- Multi-boolean object type --

typedef struct mbool_s
{
	bool v[BLIS_NUM_FP_TYPES];

} mbool_t;


// -- Auxiliary kernel info type --

// Note: This struct is used by macro-kernels to package together extra
// parameter values that may be of use to the micro-kernel without
// cluttering up the micro-kernel interface itself.

typedef struct
{
	// The pack schemas of A and B.
	pack_t schema_a;
	pack_t schema_b;

	// Pointers to the micro-panels of A and B which will be used by the
	// next call to the micro-kernel.
	const void* a_next;
	const void* b_next;

	// The imaginary strides of A and B.
	inc_t is_a;
	inc_t is_b;

	// The panel strides of A and B.
	// NOTE: These are only used in situations where iteration over the
	// micropanels takes place in part within the kernel code (e.g. sup
	// millikernels).
	inc_t ps_a;
	inc_t ps_b;

	// The type to convert to on output.
	//num_t  dt_on_output;

	// (Virtual) microkernel address and additional parameters.
	void_fp ukr;
	const void* params;

} auxinfo_t;


// -- Global scalar constant data struct --

// Note: This struct is used only when statically initializing the
// global scalar constants in bli_const.c.
typedef struct constdata_s
{
	float    s;
	double   d;
	scomplex c;
	dcomplex z;
	gint_t   i;

} constdata_t;


//
// -- BLIS object type definitions ---------------------------------------------
//

// Forward declarations for function pointer types
struct obj_s;
struct cntx_s;
struct rntm_s;
struct thrinfo_s;

typedef void (*obj_pack_fn_t)
    (
      const struct obj_s*     a,
            struct obj_s*     ap,
      const struct cntx_s*    cntx,
      const struct cntl_s*    cntl,
            struct thrinfo_s* thread
    );

typedef void (*obj_ker_fn_t)
    (
      const struct obj_s*     a,
      const struct obj_s*     b,
      const struct obj_s*     c,
      const struct cntx_s*    cntx,
      const struct cntl_s*    cntl,
            struct thrinfo_s* thread
    );

typedef struct obj_s
{
	// Basic fields
	struct obj_s* root;

	dim_t         off[2];
	dim_t         dim[2];
	doff_t        diag_off;

	objbits_t     info;
	objbits_t     info2;
	siz_t         elem_size;

	void*         buffer;
	inc_t         rs;
	inc_t         cs;
	inc_t         is;

	// Bufferless scalar storage
	atom_t        scalar;

	// Pack-related fields
	dim_t         m_padded; // m dimension of matrix, including any padding
	dim_t         n_padded; // n dimension of matrix, including any padding
	inc_t         ps;       // panel stride (distance to next panel)
	inc_t         pd;       // panel dimension (the "width" of a panel:
	                        // usually MR or NR)
	dim_t         m_panel;  // m dimension of a "full" panel
	dim_t         n_panel;  // n dimension of a "full" panel

	// User-customizable fields
	obj_pack_fn_t pack_fn;
	void*         pack_params;
	obj_ker_fn_t  ker_fn;
	void*         ker_params;

} obj_t;

// Pre-initializors. Things that must be set afterwards:
// - root object pointer
// - info bitfields: dt, target_dt, exec_dt, comp_dt
// - info2 bitfields: scalar_dt
// - elem_size
// - dims, strides
// - buffer
// - internal scalar buffer (must always set imaginary component)

#define BLIS_OBJECT_INITIALIZER \
{ \
	.root        = NULL, \
\
	.off         = { 0, 0 }, \
	.dim         = { 0, 0 }, \
	.diag_off    = 0, \
\
	.info        = 0x0 | BLIS_BITVAL_DENSE      | \
	                     BLIS_BITVAL_GENERAL, \
	.info2       = 0x0, \
	.elem_size   = sizeof( float ), /* this is changed later. */ \
\
	.buffer      = NULL, \
	.rs          = 0, \
	.cs          = 0, \
	.is          = 1,  \
\
	.scalar      = { 0.0, 0.0 }, \
\
	.m_padded    = 0, \
	.n_padded    = 0, \
	.ps          = 0, \
	.pd          = 0, \
	.m_panel     = 0, \
	.n_panel     = 0, \
\
	.pack_fn     = NULL, \
	.pack_params = NULL, \
	.ker_fn      = NULL, \
	.ker_params  = NULL  \
}

#define BLIS_OBJECT_INITIALIZER_1X1 \
{ \
	.root        = NULL, \
\
	.off         = { 0, 0 }, \
	.dim         = { 1, 1 }, \
	.diag_off    = 0, \
\
	.info        = 0x0 | BLIS_BITVAL_DENSE      | \
	                     BLIS_BITVAL_GENERAL, \
	.info2       = 0x0, \
	.elem_size   = sizeof( float ), /* this is changed later. */ \
\
	.buffer      = NULL, \
	.rs          = 0, \
	.cs          = 0, \
	.is          = 1,  \
\
	.scalar      = { 0.0, 0.0 }, \
\
	.m_padded    = 0, \
	.n_padded    = 0, \
	.ps          = 0, \
	.pd          = 0, \
	.m_panel     = 0, \
	.n_panel     = 0, \
\
	.pack_fn     = NULL, \
	.pack_params = NULL, \
	.ker_fn      = NULL, \
	.ker_params  = NULL  \
}

// Define these macros here since they must be updated if contents of
// obj_t changes.

BLIS_INLINE void bli_obj_init_full_shallow_copy_of( const obj_t* a, obj_t* b )
{
	b->root        = a->root;

	b->off[0]      = a->off[0];
	b->off[1]      = a->off[1];
	b->dim[0]      = a->dim[0];
	b->dim[1]      = a->dim[1];
	b->diag_off    = a->diag_off;

	b->info        = a->info;
	b->info2       = a->info2;
	b->elem_size   = a->elem_size;

	b->buffer      = a->buffer;
	b->rs          = a->rs;
	b->cs          = a->cs;
	b->is          = a->is;

	b->scalar      = a->scalar;

	//b->pack_mem    = a->pack_mem;
	b->m_padded    = a->m_padded;
	b->n_padded    = a->n_padded;
	b->ps          = a->ps;
	b->pd          = a->pd;
	b->m_panel     = a->m_panel;
	b->n_panel     = a->n_panel;

	b->pack_fn     = a->pack_fn;
	b->pack_params = a->pack_params;
	b->ker_fn      = a->ker_fn;
	b->ker_params  = a->ker_params;
}

BLIS_INLINE void bli_obj_init_subpart_from( const obj_t* a, obj_t* b )
{
	b->root        = a->root;

	b->off[0]      = a->off[0];
	b->off[1]      = a->off[1];
	// Avoid copying m and n since they will be overwritten.
	//b->dim[0]      = a->dim[0];
	//b->dim[1]      = a->dim[1];
	b->diag_off    = a->diag_off;

	b->info        = a->info;
	b->info2       = a->info2;
	b->elem_size   = a->elem_size;

	b->buffer      = a->buffer;
	b->rs          = a->rs;
	b->cs          = a->cs;
	b->is          = a->is;

	b->scalar      = a->scalar;

	// Avoid copying pack_mem entry.
	// FGVZ: You should probably make sure this is right.
	//b->pack_mem    = a->pack_mem;
	b->m_padded    = a->m_padded;
	b->n_padded    = a->n_padded;
	b->ps          = a->ps;
	b->pd          = a->pd;
	b->m_panel     = a->m_panel;
	b->n_panel     = a->n_panel;

	b->pack_fn     = a->pack_fn;
	b->pack_params = a->pack_params;
	b->ker_fn      = a->ker_fn;
	b->ker_params  = a->ker_params;
}

// Initializors for global scalar constants.
// NOTE: These must remain cpp macros since they are initializor
// expressions, not functions.

#define bli_obj_init_const( buffer0 ) \
{ \
	.root      = NULL, \
\
	.off       = { 0, 0 }, \
	.dim       = { 1, 1 }, \
	.diag_off  = 0, \
\
	.info      = 0x0 | BLIS_BITVAL_CONST_TYPE | \
	                   BLIS_BITVAL_DENSE      | \
	                   BLIS_BITVAL_GENERAL, \
	.info2     = 0x0, \
	.elem_size = sizeof( constdata_t ), \
\
	.buffer    = buffer0, \
	.rs        = 1, \
	.cs        = 1, \
	.is        = 1  \
}

#define bli_obj_init_constdata( val ) \
{ \
	.s =           ( float  )val, \
	.d =           ( double )val, \
	.c = { .real = ( float  )val, .imag = 0.0f }, \
	.z = { .real = ( double )val, .imag = 0.0 }, \
	.i =           ( gint_t )val, \
}

#define bli_obj_init_constdata_ri( valr, vali ) \
{ \
	.s =           ( float  )valr, \
	.d =           ( double )valr, \
	.c = { .real = ( float  )valr, .imag = ( float  )vali }, \
	.z = { .real = ( double )valr, .imag = ( double )vali }, \
	.i =           ( gint_t )valr, \
}


// -- Context type --

typedef struct cntx_s
{
	blksz_t   blkszs[ BLIS_NUM_BLKSZS ];
	bszid_t   bmults[ BLIS_NUM_BLKSZS ];

	func_t    ukrs[ BLIS_NUM_UKRS ];
	mbool_t   ukr_prefs[ BLIS_NUM_UKR_PREFS ];

	void_fp   l3_sup_handlers[ BLIS_NUM_LEVEL3_OPS ];

	ind_t     method;

} cntx_t;


// -- Runtime type --

// NOTE: The order of these fields must be kept consistent with the definition
// of the BLIS_RNTM_INITIALIZER macro in bli_rntm.h.

typedef struct rntm_s
{
	// "External" fields: these may be queried by the end-user.
	timpl_t   thread_impl;

	bool      auto_factor;

	dim_t     num_threads;
	dim_t     thrloop[ BLIS_NUM_LOOPS ];
	bool      pack_a; // enable/disable packing of left-hand matrix A.
	bool      pack_b; // enable/disable packing of right-hand matrix B.
	bool      l3_sup; // enable/disable small matrix handling in level-3 ops.
} rntm_t;


// -- Error types --

typedef enum
{
	BLIS_NO_ERROR_CHECKING = 0,
	BLIS_FULL_ERROR_CHECKING
} errlev_t;

typedef enum
{
	// Generic error codes
	BLIS_SUCCESS                               = (  -1),
	BLIS_FAILURE                               = (  -2),

	BLIS_ERROR_CODE_MIN                        = (  -9),

	// General errors
	BLIS_INVALID_ERROR_CHECKING_LEVEL          = ( -10),
	BLIS_UNDEFINED_ERROR_CODE                  = ( -11),
	BLIS_NULL_POINTER                          = ( -12),
	BLIS_NOT_YET_IMPLEMENTED                   = ( -13),

	// Parameter-specific errors
	BLIS_INVALID_SIDE                          = ( -20),
	BLIS_INVALID_UPLO                          = ( -21),
	BLIS_INVALID_TRANS                         = ( -22),
	BLIS_INVALID_CONJ                          = ( -23),
	BLIS_INVALID_DIAG                          = ( -24),
	BLIS_INVALID_MACHVAL                       = ( -25),
	BLIS_EXPECTED_NONUNIT_DIAG                 = ( -26),

	// Datatype-specific errors
	BLIS_INVALID_DATATYPE                      = ( -30),
	BLIS_EXPECTED_FLOATING_POINT_DATATYPE      = ( -31),
	BLIS_EXPECTED_NONINTEGER_DATATYPE          = ( -32),
	BLIS_EXPECTED_NONCONSTANT_DATATYPE         = ( -33),
	BLIS_EXPECTED_REAL_DATATYPE                = ( -34),
	BLIS_EXPECTED_INTEGER_DATATYPE             = ( -35),
	BLIS_INCONSISTENT_DATATYPES                = ( -36),
	BLIS_EXPECTED_REAL_PROJ_OF                 = ( -37),
	BLIS_EXPECTED_REAL_VALUED_OBJECT           = ( -38),
	BLIS_INCONSISTENT_PRECISIONS               = ( -39),

	// Dimension-specific errors
	BLIS_NONCONFORMAL_DIMENSIONS               = ( -40),
	BLIS_EXPECTED_SCALAR_OBJECT                = ( -41),
	BLIS_EXPECTED_VECTOR_OBJECT                = ( -42),
	BLIS_UNEQUAL_VECTOR_LENGTHS                = ( -43),
	BLIS_EXPECTED_SQUARE_OBJECT                = ( -44),
	BLIS_UNEXPECTED_OBJECT_LENGTH              = ( -45),
	BLIS_UNEXPECTED_OBJECT_WIDTH               = ( -46),
	BLIS_UNEXPECTED_VECTOR_DIM                 = ( -47),
	BLIS_UNEXPECTED_DIAG_OFFSET                = ( -48),
	BLIS_NEGATIVE_DIMENSION                    = ( -49),

	// Stride-specific errors
	BLIS_INVALID_ROW_STRIDE                    = ( -50),
	BLIS_INVALID_COL_STRIDE                    = ( -51),
	BLIS_INVALID_DIM_STRIDE_COMBINATION        = ( -52),

	// Structure-specific errors
	BLIS_EXPECTED_GENERAL_OBJECT               = ( -60),
	BLIS_EXPECTED_HERMITIAN_OBJECT             = ( -61),
	BLIS_EXPECTED_SYMMETRIC_OBJECT             = ( -62),
	BLIS_EXPECTED_TRIANGULAR_OBJECT            = ( -63),

	// Storage-specific errors
	BLIS_EXPECTED_UPPER_OR_LOWER_OBJECT        = ( -70),

	// Partitioning-specific errors
	BLIS_INVALID_3x1_SUBPART                   = ( -80),
	BLIS_INVALID_1x3_SUBPART                   = ( -81),
	BLIS_INVALID_3x3_SUBPART                   = ( -82),

	// Control tree-specific errors
	BLIS_UNEXPECTED_NULL_CONTROL_TREE          = ( -90),

	// Packing-specific errors
	BLIS_PACK_SCHEMA_NOT_SUPPORTED_FOR_UNPACK  = (-100),

	// Buffer-specific errors
	BLIS_EXPECTED_NONNULL_OBJECT_BUFFER        = (-110),

	// Memory errors
	BLIS_MALLOC_RETURNED_NULL                  = (-120),

	// Internal memory pool errors
	BLIS_INVALID_PACKBUF                       = (-130),
	BLIS_EXHAUSTED_CONTIG_MEMORY_POOL          = (-131),
	BLIS_INSUFFICIENT_STACK_BUF_SIZE           = (-132),
	BLIS_ALIGNMENT_NOT_POWER_OF_TWO            = (-133),
	BLIS_ALIGNMENT_NOT_MULT_OF_PTR_SIZE        = (-134),

	// Object-related errors
	BLIS_EXPECTED_OBJECT_ALIAS                 = (-140),

	// Architecture-related errors
	BLIS_INVALID_ARCH_ID                       = (-150),
	BLIS_UNINITIALIZED_GKS_CNTX                = (-151),
	BLIS_INVALID_UKR_ID                        = (-152),

	// Blocksize-related errors
	BLIS_MC_DEF_NONMULTIPLE_OF_MR              = (-160),
	BLIS_MC_MAX_NONMULTIPLE_OF_MR              = (-161),
	BLIS_NC_DEF_NONMULTIPLE_OF_NR              = (-162),
	BLIS_NC_MAX_NONMULTIPLE_OF_NR              = (-163),
	BLIS_KC_DEF_NONMULTIPLE_OF_KR              = (-164),
	BLIS_KC_MAX_NONMULTIPLE_OF_KR              = (-165),

	BLIS_ERROR_CODE_MAX                        = (-170)
} err_t;

#endif
// end bli_type_defs.h
#line 74 "./frame/include/blis.h"

// begin bli_macro_defs.h
#line 1 "./frame/include//bli_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_MACRO_DEFS_H
#define BLIS_MACRO_DEFS_H


// -- Concatenation macros --

#define BLIS_FUNC_PREFIX_STR       "bli"

// We add an extra layer the definitions of these string-pasting macros
// because sometimes it is needed if, for example, one of the PASTE
// macros is invoked with an "op" argument that is itself a macro.

#define PASTEMAC0_(op)             bli_ ## op
#define PASTEMAC0(op)              PASTEMAC0_(op)

#define PASTEMAC_(ch,op)           bli_ ## ch  ## op
#define PASTEMAC(ch,op)            PASTEMAC_(ch,op)

#define PASTEMAC2_(ch1,ch2,op)     bli_ ## ch1 ## ch2 ## op
#define PASTEMAC2(ch1,ch2,op)      PASTEMAC2_(ch1,ch2,op)

#define PASTEMAC3_(ch1,ch2,ch3,op) bli_ ## ch1 ## ch2 ## ch3 ## op
#define PASTEMAC3(ch1,ch2,ch3,op)  PASTEMAC3_(ch1,ch2,ch3,op)

#define PASTEMAC4_(ch1,ch2,ch3,ch4,op) bli_ ## ch1 ## ch2 ## ch3 ## ch4 ## op
#define PASTEMAC4(ch1,ch2,ch3,ch4,op)  PASTEMAC4_(ch1,ch2,ch3,ch4,op)

#define PASTEMAC5_(ch1,ch2,ch3,ch4,ch5,op) bli_ ## ch1 ## ch2 ## ch3 ## ch4 ## ch5 ## op
#define PASTEMAC5(ch1,ch2,ch3,ch4,ch5,op)  PASTEMAC5_(ch1,ch2,ch3,ch4,ch5,op)

#define PASTEMAC6_(ch1,ch2,ch3,ch4,ch5,ch6,op) bli_ ## ch1 ## ch2 ## ch3 ## ch4 ## ch5 ## ch6 ## op
#define PASTEMAC6(ch1,ch2,ch3,ch4,ch5,ch6,op)  PASTEMAC6_(ch1,ch2,ch3,ch4,ch5,ch6,op)

#define PASTEBLACHK_(op)           bla_ ## op ## _check
#define PASTEBLACHK(op)            PASTEBLACHK_(op)

#define PASTECH0_(op)              op
#define PASTECH0(op)               PASTECH0_(op)

#define PASTECH_(ch,op)            ch ## op
#define PASTECH(ch,op)             PASTECH_(ch,op)

#define PASTECH2_(ch1,ch2,op)      ch1 ## ch2 ## op
#define PASTECH2(ch1,ch2,op)       PASTECH2_(ch1,ch2,op)

#define PASTECH3_(ch1,ch2,ch3,op)  ch1 ## ch2 ## ch3 ## op
#define PASTECH3(ch1,ch2,ch3,op)   PASTECH3_(ch1,ch2,ch3,op)

#define MKSTR(s1)                  #s1
#define STRINGIFY_INT( s )         MKSTR( s )

// Fortran-77 name-mangling macros.
#define PASTEF770(name)                                      name ## _
#define PASTEF77(ch1,name)                     ch1        ## name ## _
#define PASTEF772(ch1,ch2,name)                ch1 ## ch2 ## name ## _
#define PASTEF773(ch1,ch2,ch3,name)     ch1 ## ch2 ## ch3 ## name ## _

// -- Include other groups of macros

// begin bli_genarray_macro_defs.h
#line 1 "./frame/include//bli_genarray_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_GENARRAY_MACRO_DEFS_H
#define BLIS_GENARRAY_MACRO_DEFS_H


// -- Macros to generate function arrays ---------------------------------------

// -- "Smart" one-operand macro --

#define GENARRAY_FPA(tname,opname) \
\
static tname PASTECH(opname,_fpa)[BLIS_NUM_FP_TYPES] = \
{ \
	( tname )PASTEMAC(s,opname), \
	( tname )PASTEMAC(c,opname), \
	( tname )PASTEMAC(d,opname), \
	( tname )PASTEMAC(z,opname)  \
}

// -- "Smart" one-operand macro (with integer support) --

#define GENARRAY_FPA_I(tname,opname) \
\
static tname PASTECH(opname,_fpa)[BLIS_NUM_FP_TYPES+1] = \
{ \
	( tname )PASTEMAC(s,opname), \
	( tname )PASTEMAC(c,opname), \
	( tname )PASTEMAC(d,opname), \
	( tname )PASTEMAC(z,opname), \
	( tname )PASTEMAC(i,opname)  \
}

// -- "Smart" two-operand macro --

#define GENARRAY_FPA2(tname,op) \
\
static tname PASTECH(op,_fpa2)[BLIS_NUM_FP_TYPES][BLIS_NUM_FP_TYPES] = \
{ \
	{ ( tname )PASTEMAC2(s,s,op), ( tname )PASTEMAC2(s,c,op), ( tname )PASTEMAC2(s,d,op), ( tname )PASTEMAC2(s,z,op) }, \
	{ ( tname )PASTEMAC2(c,s,op), ( tname )PASTEMAC2(c,c,op), ( tname )PASTEMAC2(c,d,op), ( tname )PASTEMAC2(c,z,op) }, \
	{ ( tname )PASTEMAC2(d,s,op), ( tname )PASTEMAC2(d,c,op), ( tname )PASTEMAC2(d,d,op), ( tname )PASTEMAC2(d,z,op) }, \
	{ ( tname )PASTEMAC2(z,s,op), ( tname )PASTEMAC2(z,c,op), ( tname )PASTEMAC2(z,d,op), ( tname )PASTEMAC2(z,z,op) }  \
}

// -- "Smart" two-operand macro --

/*
#define GENARRAY2_VFP(arrayname,op) \
\
arrayname[BLIS_NUM_FP_TYPES][BLIS_NUM_FP_TYPES] = \
{ \
	{ PASTEMAC2(s,s,op), PASTEMAC2(s,c,op), PASTEMAC2(s,d,op), PASTEMAC2(s,z,op) }, \
	{ PASTEMAC2(c,s,op), PASTEMAC2(c,c,op), PASTEMAC2(c,d,op), PASTEMAC2(c,z,op) }, \
	{ PASTEMAC2(d,s,op), PASTEMAC2(d,c,op), PASTEMAC2(d,d,op), PASTEMAC2(d,z,op) }, \
	{ PASTEMAC2(z,s,op), PASTEMAC2(z,c,op), PASTEMAC2(z,d,op), PASTEMAC2(z,z,op) }  \
}
*/



// -- One-operand macro --

#define GENARRAY(arrayname,op) \
\
arrayname[BLIS_NUM_FP_TYPES] = \
{ \
	PASTEMAC(s,op), \
	PASTEMAC(c,op), \
	PASTEMAC(d,op), \
	PASTEMAC(z,op)  \
}

#define GENARRAY_I(arrayname,op) \
\
arrayname[BLIS_NUM_FP_TYPES+1] = \
{ \
	PASTEMAC(s,op), \
	PASTEMAC(c,op), \
	PASTEMAC(d,op), \
	PASTEMAC(z,op), \
	PASTEMAC(i,op)  \
}

/*
#define GENARRAYR(arrayname,op) \
\
arrayname[BLIS_NUM_FP_TYPES][BLIS_NUM_FP_TYPES] = \
{ \
	{ PASTEMAC2(s,s,op), NULL,              PASTEMAC2(s,d,op), NULL,             }, \
	{ PASTEMAC2(c,s,op), NULL,              PASTEMAC2(c,d,op), NULL,             }, \
	{ PASTEMAC2(d,s,op), NULL,              PASTEMAC2(d,d,op), NULL,             }, \
	{ PASTEMAC2(z,s,op), NULL,              PASTEMAC2(z,d,op), NULL,             }  \
}
*/



// -- One-operand macro (with custom prefix) --

#define GENARRAY_PREF(arrayname,prefix,op) \
\
arrayname[BLIS_NUM_FP_TYPES] = \
{ \
	PASTECH2(prefix,s,op), \
	PASTECH2(prefix,c,op), \
	PASTECH2(prefix,d,op), \
	PASTECH2(prefix,z,op)  \
}



// -- Two-operand macros --


#define GENARRAY2_ALL(arrayname,op) \
\
arrayname[BLIS_NUM_FP_TYPES][BLIS_NUM_FP_TYPES] = \
{ \
	{ PASTEMAC2(s,s,op), PASTEMAC2(s,c,op), PASTEMAC2(s,d,op), PASTEMAC2(s,z,op) }, \
	{ PASTEMAC2(c,s,op), PASTEMAC2(c,c,op), PASTEMAC2(c,d,op), PASTEMAC2(c,z,op) }, \
	{ PASTEMAC2(d,s,op), PASTEMAC2(d,c,op), PASTEMAC2(d,d,op), PASTEMAC2(d,z,op) }, \
	{ PASTEMAC2(z,s,op), PASTEMAC2(z,c,op), PASTEMAC2(z,d,op), PASTEMAC2(z,z,op) }  \
}


#define GENARRAY2_EXT(arrayname,op) \
\
arrayname[BLIS_NUM_FP_TYPES][BLIS_NUM_FP_TYPES] = \
{ \
	{ PASTEMAC2(s,s,op), PASTEMAC2(s,c,op), NULL,              NULL,             }, \
	{ PASTEMAC2(c,s,op), PASTEMAC2(c,c,op), NULL,              NULL,             }, \
	{ NULL,              NULL,              PASTEMAC2(d,d,op), PASTEMAC2(d,z,op) }, \
	{ NULL,              NULL,              PASTEMAC2(z,d,op), PASTEMAC2(z,z,op) }  \
}


#define GENARRAY2_MIN(arrayname,op) \
\
arrayname[BLIS_NUM_FP_TYPES][BLIS_NUM_FP_TYPES] = \
{ \
	{ PASTEMAC2(s,s,op), NULL,              NULL,              NULL,             }, \
	{ NULL,              PASTEMAC2(c,c,op), NULL,              NULL,             }, \
	{ NULL,              NULL,              PASTEMAC2(d,d,op), NULL,             }, \
	{ NULL,              NULL,              NULL,              PASTEMAC2(z,z,op) }  \
}


// -- Three-operand macros --


#define GENARRAY3_ALL(arrayname,op) \
\
arrayname[BLIS_NUM_FP_TYPES][BLIS_NUM_FP_TYPES][BLIS_NUM_FP_TYPES] = \
{ \
	{ \
	{ PASTEMAC3(s,s,s,op), PASTEMAC3(s,s,c,op), PASTEMAC3(s,s,d,op), PASTEMAC3(s,s,z,op) }, \
	{ PASTEMAC3(s,c,s,op), PASTEMAC3(s,c,c,op), PASTEMAC3(s,c,d,op), PASTEMAC3(s,c,z,op) }, \
	{ PASTEMAC3(s,d,s,op), PASTEMAC3(s,d,c,op), PASTEMAC3(s,d,d,op), PASTEMAC3(s,d,z,op) }, \
	{ PASTEMAC3(s,z,s,op), PASTEMAC3(s,z,c,op), PASTEMAC3(s,z,d,op), PASTEMAC3(s,z,z,op) }  \
	}, \
	{ \
	{ PASTEMAC3(c,s,s,op), PASTEMAC3(c,s,c,op), PASTEMAC3(c,s,d,op), PASTEMAC3(c,s,z,op) }, \
	{ PASTEMAC3(c,c,s,op), PASTEMAC3(c,c,c,op), PASTEMAC3(c,c,d,op), PASTEMAC3(c,c,z,op) }, \
	{ PASTEMAC3(c,d,s,op), PASTEMAC3(c,d,c,op), PASTEMAC3(c,d,d,op), PASTEMAC3(c,d,z,op) }, \
	{ PASTEMAC3(c,z,s,op), PASTEMAC3(c,z,c,op), PASTEMAC3(c,z,d,op), PASTEMAC3(c,z,z,op) }  \
	}, \
	{ \
	{ PASTEMAC3(d,s,s,op), PASTEMAC3(d,s,c,op), PASTEMAC3(d,s,d,op), PASTEMAC3(d,s,z,op) }, \
	{ PASTEMAC3(d,c,s,op), PASTEMAC3(d,c,c,op), PASTEMAC3(d,c,d,op), PASTEMAC3(d,c,z,op) }, \
	{ PASTEMAC3(d,d,s,op), PASTEMAC3(d,d,c,op), PASTEMAC3(d,d,d,op), PASTEMAC3(d,d,z,op) }, \
	{ PASTEMAC3(d,z,s,op), PASTEMAC3(d,z,c,op), PASTEMAC3(d,z,d,op), PASTEMAC3(d,z,z,op) }  \
	}, \
	{ \
	{ PASTEMAC3(z,s,s,op), PASTEMAC3(z,s,c,op), PASTEMAC3(z,s,d,op), PASTEMAC3(z,s,z,op) }, \
	{ PASTEMAC3(z,c,s,op), PASTEMAC3(z,c,c,op), PASTEMAC3(z,c,d,op), PASTEMAC3(z,c,z,op) }, \
	{ PASTEMAC3(z,d,s,op), PASTEMAC3(z,d,c,op), PASTEMAC3(z,d,d,op), PASTEMAC3(z,d,z,op) }, \
	{ PASTEMAC3(z,z,s,op), PASTEMAC3(z,z,c,op), PASTEMAC3(z,z,d,op), PASTEMAC3(z,z,z,op) }  \
	} \
}


#define GENARRAY3_EXT(arrayname,op) \
\
arrayname[BLIS_NUM_FP_TYPES][BLIS_NUM_FP_TYPES][BLIS_NUM_FP_TYPES] = \
{ \
	{ \
	{ PASTEMAC3(s,s,s,op), PASTEMAC3(s,s,c,op), NULL,                NULL,               }, \
	{ PASTEMAC3(s,c,s,op), PASTEMAC3(s,c,c,op), NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }  \
	}, \
	{ \
	{ PASTEMAC3(c,s,s,op), PASTEMAC3(c,s,c,op), NULL,                NULL,               }, \
	{ PASTEMAC3(c,c,s,op), PASTEMAC3(c,c,c,op), NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }  \
	}, \
	{ \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                PASTEMAC3(d,d,d,op), PASTEMAC3(d,d,z,op) }, \
	{ NULL,                NULL,                PASTEMAC3(d,z,d,op), PASTEMAC3(d,z,z,op) }  \
	}, \
	{ \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                PASTEMAC3(z,d,d,op), PASTEMAC3(z,d,z,op) }, \
	{ NULL,                NULL,                PASTEMAC3(z,z,d,op), PASTEMAC3(z,z,z,op) }  \
	} \
}


#define GENARRAY3_MIN(arrayname,op) \
\
arrayname[BLIS_NUM_FP_TYPES][BLIS_NUM_FP_TYPES][BLIS_NUM_FP_TYPES] = \
{ \
	{ \
	{ PASTEMAC3(s,s,s,op), NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }  \
	}, \
	{ \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                PASTEMAC3(c,c,c,op), NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }  \
	}, \
	{ \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                PASTEMAC3(d,d,d,op), NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }  \
	}, \
	{ \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                NULL,               }, \
	{ NULL,                NULL,                NULL,                PASTEMAC3(z,z,z,op) }  \
	} \
}


#endif
// end bli_genarray_macro_defs.h
#line 96 "./frame/include//bli_macro_defs.h"

// begin bli_gentdef_macro_defs.h
#line 1 "./frame/include//bli_gentdef_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_GENTDEF_MACRO_DEFS_H
#define BLIS_GENTDEF_MACRO_DEFS_H

//
// -- MACROS TO INSERT TYPEDEF-GENERATING MACROS -------------------------------
//


// -- function typedef macro (both typed and void) --

#define INSERT_GENTDEF( opname ) \
\
GENTDEF( float,    s, opname, _ft ) \
GENTDEF( double,   d, opname, _ft ) \
GENTDEF( scomplex, c, opname, _ft ) \
GENTDEF( dcomplex, z, opname, _ft ) \
\
GENTDEF( void,     s, opname, _vft ) \
GENTDEF( void,     d, opname, _vft ) \
GENTDEF( void,     c, opname, _vft ) \
GENTDEF( void,     z, opname, _vft ) \
\
GENTDEF( void,      , opname, _vft )

// -- function typedef macro (both typed and void) with real projection --

#define INSERT_GENTDEFR( opname ) \
\
GENTDEFR( float,    float,    s, s, opname, _ft ) \
GENTDEFR( double,   double,   d, d, opname, _ft ) \
GENTDEFR( scomplex, float,    c, s, opname, _ft ) \
GENTDEFR( dcomplex, double,   z, d, opname, _ft ) \
\
GENTDEFR( void,     void,     s, s, opname, _vft ) \
GENTDEFR( void,     void,     d, d, opname, _vft ) \
GENTDEFR( void,     void,     c, s, opname, _vft ) \
GENTDEFR( void,     void,     z, d, opname, _vft ) \
\
GENTDEFR( void,     void,      ,  , opname, _vft )


#endif
// end bli_gentdef_macro_defs.h
#line 97 "./frame/include//bli_macro_defs.h"

// begin bli_gentfunc_macro_defs.h
#line 1 "./frame/include//bli_gentfunc_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


#ifndef BLIS_GENTFUNC_MACRO_DEFS_H
#define BLIS_GENTFUNC_MACRO_DEFS_H

//
// -- MACROS TO INSERT FUNCTION-GENERATING MACROS ------------------------------
//



// -- Macros for generating BLAS routines --------------------------------------


// -- Basic one-operand macro --


#define INSERT_GENTFUNC_BLAS( blasname, blisname ) \
\
GENTFUNC( float,    s, blasname, blisname ) \
GENTFUNC( double,   d, blasname, blisname ) \
GENTFUNC( scomplex, c, blasname, blisname ) \
GENTFUNC( dcomplex, z, blasname, blisname )


// -- Basic one-operand macro with real domain only --


#define INSERT_GENTFUNCRO_BLAS( blasname, blisname ) \
\
GENTFUNCRO( float,    s, blasname, blisname ) \
GENTFUNCRO( double,   d, blasname, blisname )


// -- Basic one-operand macro with complex domain only and real projection --


#define INSERT_GENTFUNCCO_BLAS( blasname, blisname ) \
\
GENTFUNCCO( scomplex, float,  c, s, blasname, blisname ) \
GENTFUNCCO( dcomplex, double, z, d, blasname, blisname )


// -- Basic one-operand macro with conjugation (real funcs only, used only for dot, ger) --


#define INSERT_GENTFUNCDOTR_BLAS( blasname, blisname ) \
\
GENTFUNCDOT( float,    s,  , BLIS_NO_CONJUGATE, blasname, blisname ) \
GENTFUNCDOT( double,   d,  , BLIS_NO_CONJUGATE, blasname, blisname )


// -- Basic one-operand macro with conjugation (complex funcs only, used only for dot, ger) --


#define INSERT_GENTFUNCDOTC_BLAS( blasname, blisname ) \
\
GENTFUNCDOT( scomplex, c, c, BLIS_CONJUGATE,    blasname, blisname ) \
GENTFUNCDOT( scomplex, c, u, BLIS_NO_CONJUGATE, blasname, blisname ) \
GENTFUNCDOT( dcomplex, z, c, BLIS_CONJUGATE,    blasname, blisname ) \
GENTFUNCDOT( dcomplex, z, u, BLIS_NO_CONJUGATE, blasname, blisname )


// -- Basic one-operand macro with conjugation (used only for dot, ger) --


#define INSERT_GENTFUNCDOT_BLAS( blasname, blisname ) \
\
INSERT_GENTFUNCDOTR_BLAS( blasname, blisname ) \
INSERT_GENTFUNCDOTC_BLAS( blasname, blisname )


// -- Basic one-operand macro with real projection --


#define INSERT_GENTFUNCR_BLAS( rblasname, cblasname, blisname ) \
\
GENTFUNCR( float,    float,  s, s, rblasname, blisname ) \
GENTFUNCR( double,   double, d, d, rblasname, blisname ) \
GENTFUNCR( scomplex, float,  c, s, cblasname, blisname ) \
GENTFUNCR( dcomplex, double, z, d, cblasname, blisname )


// -- Alternate two-operand macro (one char for complex, one for real proj) --


#define INSERT_GENTFUNCR2_BLAS( blasname, blisname ) \
\
GENTFUNCR2( float,    float,  s,  , blasname, blisname ) \
GENTFUNCR2( double,   double, d,  , blasname, blisname ) \
GENTFUNCR2( scomplex, float,  c, s, blasname, blisname ) \
GENTFUNCR2( dcomplex, double, z, d, blasname, blisname )


// -- Extended two-operand macro (used only for scal) --


#define INSERT_GENTFUNCSCAL_BLAS( blasname, blisname ) \
\
GENTFUNCSCAL( float,    float,    s,  , blasname, blisname ) \
GENTFUNCSCAL( double,   double,   d,  , blasname, blisname ) \
GENTFUNCSCAL( scomplex, scomplex, c,  , blasname, blisname ) \
GENTFUNCSCAL( dcomplex, dcomplex, z,  , blasname, blisname ) \
GENTFUNCSCAL( scomplex, float,    c, s, blasname, blisname ) \
GENTFUNCSCAL( dcomplex, double,   z, d, blasname, blisname )




// -- Macros for functions with one operand ------------------------------------


// -- Basic one-operand macro --

#define INSERT_GENTFUNC_BASIC( ... ) \
\
GENTFUNC( float,    s, __VA_ARGS__ ) \
GENTFUNC( double,   d, __VA_ARGS__ ) \
GENTFUNC( scomplex, c, __VA_ARGS__ ) \
GENTFUNC( dcomplex, z, __VA_ARGS__ )



// -- Basic one-operand with real projection --

#define INSERT_GENTFUNCR_BASIC( ... ) \
\
GENTFUNCR( float,    float,  s, s, __VA_ARGS__ ) \
GENTFUNCR( double,   double, d, d, __VA_ARGS__ ) \
GENTFUNCR( scomplex, float,  c, s, __VA_ARGS__ ) \
GENTFUNCR( dcomplex, double, z, d, __VA_ARGS__ )



// -- Basic one-operand macro with real domain only --

#define INSERT_GENTFUNCRO_BASIC( ... ) \
\
GENTFUNCRO( float,  s, __VA_ARGS__ ) \
GENTFUNCRO( double, d, __VA_ARGS__ )

// -- Basic one-operand macro with complex domain only --

#define INSERT_GENTFUNCCO_BASIC( ... ) \
\
GENTFUNCCO( scomplex, c, __VA_ARGS__ ) \
GENTFUNCCO( dcomplex, z, __VA_ARGS__ )

// -- Basic one-operand macro with real domain only and complex projection --

#define INSERT_GENTFUNCRO( ... ) \
\
GENTFUNCRO( float,  scomplex, s, c, __VA_ARGS__ ) \
GENTFUNCRO( double, dcomplex, d, z, __VA_ARGS__ )

// -- Basic one-operand macro with complex domain only and real projection --

#define INSERT_GENTFUNCCO( ... ) \
\
GENTFUNCCO( scomplex, float,  c, s, __VA_ARGS__ ) \
GENTFUNCCO( dcomplex, double, z, d, __VA_ARGS__ )



// -- Basic one-operand macro with integer instance --

#define INSERT_GENTFUNC_BASIC_I( ... ) \
\
GENTFUNC( float,    s, __VA_ARGS__ ) \
GENTFUNC( double,   d, __VA_ARGS__ ) \
GENTFUNC( scomplex, c, __VA_ARGS__ ) \
GENTFUNC( dcomplex, z, __VA_ARGS__ ) \
GENTFUNC( gint_t,   i, __VA_ARGS__ )



// -- Basic one-operand with integer projection --

#define INSERT_GENTFUNCI_BASIC( ... ) \
\
GENTFUNCI( float,    gint_t, s, i, __VA_ARGS__ ) \
GENTFUNCI( double,   gint_t, d, i, __VA_ARGS__ ) \
GENTFUNCI( scomplex, gint_t, c, i, __VA_ARGS__ ) \
GENTFUNCI( dcomplex, gint_t, z, i, __VA_ARGS__ )



// -- Basic one-operand with real and integer projections --

#define INSERT_GENTFUNCRI_BASIC( ... ) \
\
GENTFUNCRI( float,    float,  gint_t, s, s, i, __VA_ARGS__ ) \
GENTFUNCRI( double,   double, gint_t, d, d, i, __VA_ARGS__ ) \
GENTFUNCRI( scomplex, float,  gint_t, c, s, i, __VA_ARGS__ ) \
GENTFUNCRI( dcomplex, double, gint_t, z, d, i, __VA_ARGS__ )




// -- Macros for functions with two primary operands ---------------------------


// -- Basic two-operand macro --

#define INSERT_GENTFUNC2_BASIC( ... ) \
\
GENTFUNC2( float,    float,    s, s, __VA_ARGS__ ) \
GENTFUNC2( double,   double,   d, d, __VA_ARGS__ ) \
GENTFUNC2( scomplex, scomplex, c, c, __VA_ARGS__ ) \
GENTFUNC2( dcomplex, dcomplex, z, z, __VA_ARGS__ )



// -- Mixed domain two-operand macro --

#define INSERT_GENTFUNC2_MIX_D( ... ) \
\
GENTFUNC2( float,    scomplex, s, c, __VA_ARGS__ ) \
GENTFUNC2( scomplex, float,    c, s, __VA_ARGS__ ) \
\
GENTFUNC2( double,   dcomplex, d, z, __VA_ARGS__ ) \
GENTFUNC2( dcomplex, double,   z, d, __VA_ARGS__ )



// -- Mixed precision two-operand macro --

#define INSERT_GENTFUNC2_MIX_P( ... ) \
\
GENTFUNC2( float,    double,   s, d, __VA_ARGS__ ) \
GENTFUNC2( float,    dcomplex, s, z, __VA_ARGS__ ) \
\
GENTFUNC2( double,   float,    d, s, __VA_ARGS__ ) \
GENTFUNC2( double,   scomplex, d, c, __VA_ARGS__ ) \
\
GENTFUNC2( scomplex, double,   c, d, __VA_ARGS__ ) \
GENTFUNC2( scomplex, dcomplex, c, z, __VA_ARGS__ ) \
\
GENTFUNC2( dcomplex, float,    z, s, __VA_ARGS__ ) \
GENTFUNC2( dcomplex, scomplex, z, c, __VA_ARGS__ )



// -- Mixed domain/precision (all) two-operand macro --

#define INSERT_GENTFUNC2_MIX_DP( ... ) \
\
GENTFUNC2( float,    double,   s, d, __VA_ARGS__ ) \
GENTFUNC2( float,    scomplex, s, c, __VA_ARGS__ ) \
GENTFUNC2( float,    dcomplex, s, z, __VA_ARGS__ ) \
\
GENTFUNC2( double,   float,    d, s, __VA_ARGS__ ) \
GENTFUNC2( double,   scomplex, d, c, __VA_ARGS__ ) \
GENTFUNC2( double,   dcomplex, d, z, __VA_ARGS__ ) \
\
GENTFUNC2( scomplex, float,    c, s, __VA_ARGS__ ) \
GENTFUNC2( scomplex, double,   c, d, __VA_ARGS__ ) \
GENTFUNC2( scomplex, dcomplex, c, z, __VA_ARGS__ ) \
\
GENTFUNC2( dcomplex, float,    z, s, __VA_ARGS__ ) \
GENTFUNC2( dcomplex, double,   z, d, __VA_ARGS__ ) \
GENTFUNC2( dcomplex, scomplex, z, c, __VA_ARGS__ )



// -- Basic two-operand with real projection of second operand --

#define INSERT_GENTFUNC2R_BASIC( ... ) \
\
GENTFUNC2R( float,    float,    float,    s, s, s, __VA_ARGS__ ) \
GENTFUNC2R( double,   double,   double,   d, d, d, __VA_ARGS__ ) \
GENTFUNC2R( scomplex, scomplex, float,    c, c, s, __VA_ARGS__ ) \
GENTFUNC2R( dcomplex, dcomplex, double,   z, z, d, __VA_ARGS__ )



// -- Mixed domain two-operand with real projection of second operand --

#define INSERT_GENTFUNC2R_MIX_D( ... ) \
\
GENTFUNC2R( float,    scomplex, float,    s, c, s, __VA_ARGS__ ) \
GENTFUNC2R( scomplex, float,    float,    c, s, s, __VA_ARGS__ ) \
\
GENTFUNC2R( double,   dcomplex, double,   d, z, d, __VA_ARGS__ ) \
GENTFUNC2R( dcomplex, double,   double,   z, d, d, __VA_ARGS__ )



// -- Mixed precision two-operand with real projection of second operand --

#define INSERT_GENTFUNC2R_MIX_P( ... ) \
\
GENTFUNC2R( float,    double,   double,   s, d, d, __VA_ARGS__ ) \
GENTFUNC2R( float,    dcomplex, double,   s, z, d, __VA_ARGS__ ) \
\
GENTFUNC2R( double,   float,    float,    d, s, s, __VA_ARGS__ ) \
GENTFUNC2R( double,   scomplex, float,    d, c, s, __VA_ARGS__ ) \
\
GENTFUNC2R( scomplex, double,   double,   c, d, d, __VA_ARGS__ ) \
GENTFUNC2R( scomplex, dcomplex, double,   c, z, d, __VA_ARGS__ ) \
\
GENTFUNC2R( dcomplex, float,    float,    z, s, s, __VA_ARGS__ ) \
GENTFUNC2R( dcomplex, scomplex, float,    z, c, s, __VA_ARGS__ )



// -- Mixed domain/precision (all) two-operand macro with real projection of second operand --

#define INSERT_GENTFUNC2R_MIX_DP( ... ) \
\
GENTFUNC2R( float,    double,   double,   s, d, d, __VA_ARGS__ ) \
GENTFUNC2R( float,    scomplex, float,    s, c, s, __VA_ARGS__ ) \
GENTFUNC2R( float,    dcomplex, double,   s, z, d, __VA_ARGS__ ) \
\
GENTFUNC2R( double,   float,    float,    d, s, s, __VA_ARGS__ ) \
GENTFUNC2R( double,   scomplex, float,    d, c, s, __VA_ARGS__ ) \
GENTFUNC2R( double,   dcomplex, double,   d, z, d, __VA_ARGS__ ) \
\
GENTFUNC2R( scomplex, float,    float,    c, s, s, __VA_ARGS__ ) \
GENTFUNC2R( scomplex, double,   double,   c, d, d, __VA_ARGS__ ) \
GENTFUNC2R( scomplex, dcomplex, double,   c, z, d, __VA_ARGS__ ) \
\
GENTFUNC2R( dcomplex, float,    float,    z, s, s, __VA_ARGS__ ) \
GENTFUNC2R( dcomplex, double,   double,   z, d, d, __VA_ARGS__ ) \
GENTFUNC2R( dcomplex, scomplex, float,    z, c, s, __VA_ARGS__ )




// -- Macros for functions with three primary operands -------------------------


// -- Basic three-operand macro --

#define INSERT_GENTFUNC3_BASIC( ... ) \
\
GENTFUNC3( float,    float,    float,    s, s, s, __VA_ARGS__ ) \
GENTFUNC3( double,   double,   double,   d, d, d, __VA_ARGS__ ) \
GENTFUNC3( scomplex, scomplex, scomplex, c, c, c, __VA_ARGS__ ) \
GENTFUNC3( dcomplex, dcomplex, dcomplex, z, z, z, __VA_ARGS__ )



// -- Mixed domain three-operand macro --

#define INSERT_GENTFUNC3_MIX_D( ... ) \
\
GENTFUNC3( float,    float,    scomplex, s, s, c, __VA_ARGS__ ) \
GENTFUNC3( float,    scomplex, float,    s, c, s, __VA_ARGS__ ) \
GENTFUNC3( float,    scomplex, scomplex, s, c, c, __VA_ARGS__ ) \
\
GENTFUNC3( double,   double,   dcomplex, d, d, z, __VA_ARGS__ ) \
GENTFUNC3( double,   dcomplex, double,   d, z, d, __VA_ARGS__ ) \
GENTFUNC3( double,   dcomplex, dcomplex, d, z, z, __VA_ARGS__ ) \
\
GENTFUNC3( scomplex, float,    float,    c, s, s, __VA_ARGS__ ) \
GENTFUNC3( scomplex, float,    scomplex, c, s, c, __VA_ARGS__ ) \
GENTFUNC3( scomplex, scomplex, float,    c, c, s, __VA_ARGS__ ) \
\
GENTFUNC3( dcomplex, double,   double,   z, d, d, __VA_ARGS__ ) \
GENTFUNC3( dcomplex, double,   dcomplex, z, d, z, __VA_ARGS__ ) \
GENTFUNC3( dcomplex, dcomplex, double,   z, z, d, __VA_ARGS__ )



// -- Mixed precision three-operand macro --

#define INSERT_GENTFUNC3_MIX_P( ... ) \
\
GENTFUNC3( float,    float,    double,   s, s, d, __VA_ARGS__ ) \
GENTFUNC3( float,    float,    dcomplex, s, s, z, __VA_ARGS__ ) \
\
GENTFUNC3( float,    double,   float,    s, d, s, __VA_ARGS__ ) \
GENTFUNC3( float,    double,   double,   s, d, d, __VA_ARGS__ ) \
GENTFUNC3( float,    double,   scomplex, s, d, c, __VA_ARGS__ ) \
GENTFUNC3( float,    double,   dcomplex, s, d, z, __VA_ARGS__ ) \
\
GENTFUNC3( float,    scomplex, double,   s, c, d, __VA_ARGS__ ) \
GENTFUNC3( float,    scomplex, dcomplex, s, c, z, __VA_ARGS__ ) \
\
GENTFUNC3( float,    dcomplex, float,    s, z, s, __VA_ARGS__ ) \
GENTFUNC3( float,    dcomplex, double,   s, z, d, __VA_ARGS__ ) \
GENTFUNC3( float,    dcomplex, scomplex, s, z, c, __VA_ARGS__ ) \
GENTFUNC3( float,    dcomplex, dcomplex, s, z, z, __VA_ARGS__ ) \
\
\
GENTFUNC3( double,   float,    float,    d, s, s, __VA_ARGS__ ) \
GENTFUNC3( double,   float,    double,   d, s, d, __VA_ARGS__ ) \
GENTFUNC3( double,   float,    scomplex, d, s, c, __VA_ARGS__ ) \
GENTFUNC3( double,   float,    dcomplex, d, s, z, __VA_ARGS__ ) \
\
GENTFUNC3( double,   double,   float,    d, d, s, __VA_ARGS__ ) \
GENTFUNC3( double,   double,   scomplex, d, d, c, __VA_ARGS__ ) \
\
GENTFUNC3( double,   scomplex, float,    d, c, s, __VA_ARGS__ ) \
GENTFUNC3( double,   scomplex, double,   d, c, d, __VA_ARGS__ ) \
GENTFUNC3( double,   scomplex, scomplex, d, c, c, __VA_ARGS__ ) \
GENTFUNC3( double,   scomplex, dcomplex, d, c, z, __VA_ARGS__ ) \
\
GENTFUNC3( double,   dcomplex, float,    d, z, s, __VA_ARGS__ ) \
GENTFUNC3( double,   dcomplex, scomplex, d, z, c, __VA_ARGS__ ) \
\
\
GENTFUNC3( scomplex, float,    double,   c, s, d, __VA_ARGS__ ) \
GENTFUNC3( scomplex, float,    dcomplex, c, s, z, __VA_ARGS__ ) \
\
GENTFUNC3( scomplex, double,   float,    c, d, s, __VA_ARGS__ ) \
GENTFUNC3( scomplex, double,   double,   c, d, d, __VA_ARGS__ ) \
GENTFUNC3( scomplex, double,   scomplex, c, d, c, __VA_ARGS__ ) \
GENTFUNC3( scomplex, double,   dcomplex, c, d, z, __VA_ARGS__ ) \
\
GENTFUNC3( scomplex, scomplex, double,   c, c, d, __VA_ARGS__ ) \
GENTFUNC3( scomplex, scomplex, dcomplex, c, c, z, __VA_ARGS__ ) \
\
GENTFUNC3( scomplex, dcomplex, float,    c, z, s, __VA_ARGS__ ) \
GENTFUNC3( scomplex, dcomplex, double,   c, z, d, __VA_ARGS__ ) \
GENTFUNC3( scomplex, dcomplex, scomplex, c, z, c, __VA_ARGS__ ) \
GENTFUNC3( scomplex, dcomplex, dcomplex, c, z, z, __VA_ARGS__ ) \
\
\
GENTFUNC3( dcomplex, float,    float,    z, s, s, __VA_ARGS__ ) \
GENTFUNC3( dcomplex, float,    double,   z, s, d, __VA_ARGS__ ) \
GENTFUNC3( dcomplex, float,    scomplex, z, s, c, __VA_ARGS__ ) \
GENTFUNC3( dcomplex, float,    dcomplex, z, s, z, __VA_ARGS__ ) \
\
GENTFUNC3( dcomplex, double,   float,    z, d, s, __VA_ARGS__ ) \
GENTFUNC3( dcomplex, double,   scomplex, z, d, c, __VA_ARGS__ ) \
\
GENTFUNC3( dcomplex, scomplex, float,    z, c, s, __VA_ARGS__ ) \
GENTFUNC3( dcomplex, scomplex, double,   z, c, d, __VA_ARGS__ ) \
GENTFUNC3( dcomplex, scomplex, scomplex, z, c, c, __VA_ARGS__ ) \
GENTFUNC3( dcomplex, scomplex, dcomplex, z, c, z, __VA_ARGS__ ) \
\
GENTFUNC3( dcomplex, dcomplex, float,    z, z, s, __VA_ARGS__ ) \
GENTFUNC3( dcomplex, dcomplex, scomplex, z, z, c, __VA_ARGS__ )



// -- Basic three-operand with union of operands 1 and 2 --

#define INSERT_GENTFUNC3U12_BASIC( ... ) \
\
GENTFUNC3U12( float,    float,    float,    float,    s, s, s, s, __VA_ARGS__ ) \
GENTFUNC3U12( double,   double,   double,   double,   d, d, d, d, __VA_ARGS__ ) \
GENTFUNC3U12( scomplex, scomplex, scomplex, scomplex, c, c, c, c, __VA_ARGS__ ) \
GENTFUNC3U12( dcomplex, dcomplex, dcomplex, dcomplex, z, z, z, z, __VA_ARGS__ )



// -- Mixed domain three-operand with union of operands 1 and 2 --

#define INSERT_GENTFUNC3U12_MIX_D( ... ) \
\
GENTFUNC3U12( float,    float,    scomplex, float,    s, s, c, s, __VA_ARGS__ ) \
GENTFUNC3U12( float,    scomplex, float,    scomplex, s, c, s, c, __VA_ARGS__ ) \
GENTFUNC3U12( float,    scomplex, scomplex, scomplex, s, c, c, c, __VA_ARGS__ ) \
\
GENTFUNC3U12( double,   double,   dcomplex, double,   d, d, z, d, __VA_ARGS__ ) \
GENTFUNC3U12( double,   dcomplex, double,   dcomplex, d, z, d, z, __VA_ARGS__ ) \
GENTFUNC3U12( double,   dcomplex, dcomplex, dcomplex, d, z, z, z, __VA_ARGS__ ) \
\
GENTFUNC3U12( scomplex, float,    float,    scomplex, c, s, s, c, __VA_ARGS__ ) \
GENTFUNC3U12( scomplex, float,    scomplex, scomplex, c, s, c, c, __VA_ARGS__ ) \
GENTFUNC3U12( scomplex, scomplex, float,    scomplex, c, c, s, c, __VA_ARGS__ ) \
\
GENTFUNC3U12( dcomplex, double,   double,   dcomplex, z, d, d, z, __VA_ARGS__ ) \
GENTFUNC3U12( dcomplex, double,   dcomplex, dcomplex, z, d, z, z, __VA_ARGS__ ) \
GENTFUNC3U12( dcomplex, dcomplex, double,   dcomplex, z, z, d, z, __VA_ARGS__ )



// -- Mixed precision three-operand with union of operands 1 and 2 --

#define INSERT_GENTFUNC3U12_MIX_P( ... ) \
\
GENTFUNC3U12( float,    float,    double,   float,    s, s, d, s, __VA_ARGS__ ) \
GENTFUNC3U12( float,    float,    dcomplex, float,    s, s, z, s, __VA_ARGS__ ) \
\
GENTFUNC3U12( float,    double,   float,    double,   s, d, s, d, __VA_ARGS__ ) \
GENTFUNC3U12( float,    double,   double,   double,   s, d, d, d, __VA_ARGS__ ) \
GENTFUNC3U12( float,    double,   scomplex, double,   s, d, c, d, __VA_ARGS__ ) \
GENTFUNC3U12( float,    double,   dcomplex, double,   s, d, z, d, __VA_ARGS__ ) \
\
GENTFUNC3U12( float,    scomplex, double,   scomplex, s, c, d, c, __VA_ARGS__ ) \
GENTFUNC3U12( float,    scomplex, dcomplex, scomplex, s, c, z, c, __VA_ARGS__ ) \
\
GENTFUNC3U12( float,    dcomplex, float,    dcomplex, s, z, s, z, __VA_ARGS__ ) \
GENTFUNC3U12( float,    dcomplex, double,   dcomplex, s, z, d, z, __VA_ARGS__ ) \
GENTFUNC3U12( float,    dcomplex, scomplex, dcomplex, s, z, c, z, __VA_ARGS__ ) \
GENTFUNC3U12( float,    dcomplex, dcomplex, dcomplex, s, z, z, z, __VA_ARGS__ ) \
\
\
GENTFUNC3U12( double,   float,    float,    double,   d, s, s, d, __VA_ARGS__ ) \
GENTFUNC3U12( double,   float,    double,   double,   d, s, d, d, __VA_ARGS__ ) \
GENTFUNC3U12( double,   float,    scomplex, double,   d, s, c, d, __VA_ARGS__ ) \
GENTFUNC3U12( double,   float,    dcomplex, double,   d, s, z, d, __VA_ARGS__ ) \
\
GENTFUNC3U12( double,   double,   float,    double,   d, d, s, d, __VA_ARGS__ ) \
GENTFUNC3U12( double,   double,   scomplex, double,   d, d, c, d, __VA_ARGS__ ) \
\
GENTFUNC3U12( double,   scomplex, float,    dcomplex, d, c, s, z, __VA_ARGS__ ) \
GENTFUNC3U12( double,   scomplex, double,   dcomplex, d, c, d, z, __VA_ARGS__ ) \
GENTFUNC3U12( double,   scomplex, scomplex, dcomplex, d, c, c, z, __VA_ARGS__ ) \
GENTFUNC3U12( double,   scomplex, dcomplex, dcomplex, d, c, z, z, __VA_ARGS__ ) \
\
GENTFUNC3U12( double,   dcomplex, float,    dcomplex, d, z, s, z, __VA_ARGS__ ) \
GENTFUNC3U12( double,   dcomplex, scomplex, dcomplex, d, z, c, z, __VA_ARGS__ ) \
\
\
GENTFUNC3U12( scomplex, float,    double,   scomplex, c, s, d, c, __VA_ARGS__ ) \
GENTFUNC3U12( scomplex, float,    dcomplex, scomplex, c, s, z, c, __VA_ARGS__ ) \
\
GENTFUNC3U12( scomplex, double,   float,    dcomplex, c, d, s, z, __VA_ARGS__ ) \
GENTFUNC3U12( scomplex, double,   double,   dcomplex, c, d, d, z, __VA_ARGS__ ) \
GENTFUNC3U12( scomplex, double,   scomplex, dcomplex, c, d, c, z, __VA_ARGS__ ) \
GENTFUNC3U12( scomplex, double,   dcomplex, dcomplex, c, d, z, z, __VA_ARGS__ ) \
\
GENTFUNC3U12( scomplex, scomplex, double,   scomplex, c, c, d, c, __VA_ARGS__ ) \
GENTFUNC3U12( scomplex, scomplex, dcomplex, scomplex, c, c, z, c, __VA_ARGS__ ) \
\
GENTFUNC3U12( scomplex, dcomplex, float,    dcomplex, c, z, s, z, __VA_ARGS__ ) \
GENTFUNC3U12( scomplex, dcomplex, double,   dcomplex, c, z, d, z, __VA_ARGS__ ) \
GENTFUNC3U12( scomplex, dcomplex, scomplex, dcomplex, c, z, c, z, __VA_ARGS__ ) \
GENTFUNC3U12( scomplex, dcomplex, dcomplex, dcomplex, c, z, z, z, __VA_ARGS__ ) \
\
\
GENTFUNC3U12( dcomplex, float,    float,    dcomplex, z, s, s, z, __VA_ARGS__ ) \
GENTFUNC3U12( dcomplex, float,    double,   dcomplex, z, s, d, z, __VA_ARGS__ ) \
GENTFUNC3U12( dcomplex, float,    scomplex, dcomplex, z, s, c, z, __VA_ARGS__ ) \
GENTFUNC3U12( dcomplex, float,    dcomplex, dcomplex, z, s, z, z, __VA_ARGS__ ) \
\
GENTFUNC3U12( dcomplex, double,   float,    dcomplex, z, d, s, z, __VA_ARGS__ ) \
GENTFUNC3U12( dcomplex, double,   scomplex, dcomplex, z, d, c, z, __VA_ARGS__ ) \
\
GENTFUNC3U12( dcomplex, scomplex, float,    dcomplex, z, c, s, z, __VA_ARGS__ ) \
GENTFUNC3U12( dcomplex, scomplex, double,   dcomplex, z, c, d, z, __VA_ARGS__ ) \
GENTFUNC3U12( dcomplex, scomplex, scomplex, dcomplex, z, c, c, z, __VA_ARGS__ ) \
GENTFUNC3U12( dcomplex, scomplex, dcomplex, dcomplex, z, c, z, z, __VA_ARGS__ ) \
\
GENTFUNC3U12( dcomplex, dcomplex, float,    dcomplex, z, z, s, z, __VA_ARGS__ ) \
GENTFUNC3U12( dcomplex, dcomplex, scomplex, dcomplex, z, z, c, z, __VA_ARGS__ )


#endif
// end bli_gentfunc_macro_defs.h
#line 98 "./frame/include//bli_macro_defs.h"

// begin bli_gentprot_macro_defs.h
#line 1 "./frame/include//bli_gentprot_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


#ifndef BLIS_GENTPROT_MACRO_DEFS_H
#define BLIS_GENTPROT_MACRO_DEFS_H

//
// -- MACROS TO INSERT PROTOTYPE-GENERATING MACROS -----------------------------
//



// -- Macros for generating BLAS routines --------------------------------------


// -- Basic one-operand macro --


#define INSERT_GENTPROT_BLAS( blasname ) \
\
GENTPROT( float,    s, blasname ) \
GENTPROT( double,   d, blasname ) \
GENTPROT( scomplex, c, blasname ) \
GENTPROT( dcomplex, z, blasname )


// -- Basic one-operand macro with real domain only --


#define INSERT_GENTPROTRO_BLAS( blasname ) \
\
GENTPROTRO( float,    s, blasname ) \
GENTPROTRO( double,   d, blasname )


// -- Basic one-operand macro with complex domain only and real projection --


#define INSERT_GENTPROTCO_BLAS( blasname ) \
\
GENTPROTCO( scomplex, float,  c, s, blasname ) \
GENTPROTCO( dcomplex, double, z, d, blasname )


// -- Basic one-operand macro with conjugation (real funcs only, used only for dot, ger) --


#define INSERT_GENTPROTDOTR_BLAS( blasname ) \
\
GENTPROTDOT( float,    s,  , blasname ) \
GENTPROTDOT( double,   d,  , blasname )


// -- Basic one-operand macro with conjugation (complex funcs only, used only for dot, ger) --


#define INSERT_GENTPROTDOTC_BLAS( blasname ) \
\
GENTPROTDOT( scomplex, c, c, blasname ) \
GENTPROTDOT( scomplex, c, u, blasname ) \
GENTPROTDOT( dcomplex, z, c, blasname ) \
GENTPROTDOT( dcomplex, z, u, blasname )


// -- Basic one-operand macro with conjugation (used only for dot, ger) --


#define INSERT_GENTPROTDOT_BLAS( blasname ) \
\
INSERT_GENTPROTDOTR_BLAS( blasname ) \
INSERT_GENTPROTDOTC_BLAS( blasname )


// -- Basic one-operand macro with real projection --


#define INSERT_GENTPROTR_BLAS( rblasname, cblasname ) \
\
GENTPROTR( float,    float,  s, s, rblasname ) \
GENTPROTR( double,   double, d, d, rblasname ) \
GENTPROTR( scomplex, float,  c, s, cblasname ) \
GENTPROTR( dcomplex, double, z, d, cblasname )


// -- Alternate two-operand macro (one char for complex, one for real proj) --


#define INSERT_GENTPROTR2_BLAS( blasname ) \
\
GENTPROTR2( float,    float,   , s, blasname ) \
GENTPROTR2( double,   double,  , d, blasname ) \
GENTPROTR2( scomplex, float,  c, s, blasname ) \
GENTPROTR2( dcomplex, double, z, d, blasname )


// -- Extended two-operand macro (used only for scal) --


#define INSERT_GENTPROTSCAL_BLAS( blasname ) \
\
GENTPROTSCAL( float,    float,     , s, blasname ) \
GENTPROTSCAL( double,   double,    , d, blasname ) \
GENTPROTSCAL( scomplex, scomplex,  , c, blasname ) \
GENTPROTSCAL( dcomplex, dcomplex,  , z, blasname ) \
GENTPROTSCAL( float,    scomplex, s, c, blasname ) \
GENTPROTSCAL( double,   dcomplex, d, z, blasname )




// -- Macros for functions with one operand ------------------------------------


// -- Basic one-operand macro --

#define INSERT_GENTPROT_BASIC( ... ) \
\
GENTPROT( float,    s, __VA_ARGS__ ) \
GENTPROT( double,   d, __VA_ARGS__ ) \
GENTPROT( scomplex, c, __VA_ARGS__ ) \
GENTPROT( dcomplex, z, __VA_ARGS__ )



// -- Basic one-operand with real projection --

#define INSERT_GENTPROTR_BASIC( ... ) \
\
GENTPROTR( float,    float,  s, s, __VA_ARGS__ ) \
GENTPROTR( double,   double, d, d, __VA_ARGS__ ) \
GENTPROTR( scomplex, float,  c, s, __VA_ARGS__ ) \
GENTPROTR( dcomplex, double, z, d, __VA_ARGS__ )



// -- Basic one-operand macro with complex domain only and real projection --

#define INSERT_GENTPROTCO_BASIC( ... ) \
\
GENTPROTCO( scomplex, float,  c, s, __VA_ARGS__ ) \
GENTPROTCO( dcomplex, double, z, d, __VA_ARGS__ )



// -- Basic one-operand macro with integer instance --

#define INSERT_GENTPROT_BASIC_I( ... ) \
\
GENTPROT( float,    s, __VA_ARGS__ ) \
GENTPROT( double,   d, __VA_ARGS__ ) \
GENTPROT( scomplex, c, __VA_ARGS__ ) \
GENTPROT( dcomplex, z, __VA_ARGS__ ) \
GENTPROT( gint_t,   i, __VA_ARGS__ )



// -- Basic one-operand with integer projection --

#define INSERT_GENTPROTI_BASIC( ... ) \
\
GENTPROTI( float,    gint_t, s, i, __VA_ARGS__ ) \
GENTPROTI( double,   gint_t, d, i, __VA_ARGS__ ) \
GENTPROTI( scomplex, gint_t, c, i, __VA_ARGS__ ) \
GENTPROTI( dcomplex, gint_t, z, i, __VA_ARGS__ )



// -- Basic one-operand with real and integer projections --

#define INSERT_GENTPROTRI_BASIC( funcname ) \
\
GENTPROTRI( float,    float,  gint_t, s, s, i, funcname ) \
GENTPROTRI( double,   double, gint_t, d, d, i, funcname ) \
GENTPROTRI( scomplex, float,  gint_t, c, s, i, funcname ) \
GENTPROTRI( dcomplex, double, gint_t, z, d, i, funcname )




// -- Macros for functions with two primary operands ---------------------------


// -- Basic two-operand macro --

#define INSERT_GENTPROT2_BASIC( ... ) \
\
GENTPROT2( float,    float,    s, s, __VA_ARGS__ ) \
GENTPROT2( double,   double,   d, d, __VA_ARGS__ ) \
GENTPROT2( scomplex, scomplex, c, c, __VA_ARGS__ ) \
GENTPROT2( dcomplex, dcomplex, z, z, __VA_ARGS__ )



// -- Mixed domain two-operand macro --

#define INSERT_GENTPROT2_MIX_D( ... ) \
\
GENTPROT2( float,    scomplex, s, c, __VA_ARGS__ ) \
GENTPROT2( scomplex, float,    c, s, __VA_ARGS__ ) \
\
GENTPROT2( double,   dcomplex, d, z, __VA_ARGS__ ) \
GENTPROT2( dcomplex, double,   z, d, __VA_ARGS__ )



// -- Mixed precision two-operand macro --

#define INSERT_GENTPROT2_MIX_P( ... ) \
\
GENTPROT2( float,    double,   s, d, __VA_ARGS__ ) \
GENTPROT2( float,    dcomplex, s, z, __VA_ARGS__ ) \
\
GENTPROT2( double,   float,    d, s, __VA_ARGS__ ) \
GENTPROT2( double,   scomplex, d, c, __VA_ARGS__ ) \
\
GENTPROT2( scomplex, double,   c, d, __VA_ARGS__ ) \
GENTPROT2( scomplex, dcomplex, c, z, __VA_ARGS__ ) \
\
GENTPROT2( dcomplex, float,    z, s, __VA_ARGS__ ) \
GENTPROT2( dcomplex, scomplex, z, c, __VA_ARGS__ ) \



// -- Mixed domain/precision (all) two-operand macro --

#define INSERT_GENTPROT2_MIX_DP( ... ) \
\
GENTPROT2( float,    double,   s, d, __VA_ARGS__ ) \
GENTPROT2( float,    scomplex, s, c, __VA_ARGS__ ) \
GENTPROT2( float,    dcomplex, s, z, __VA_ARGS__ ) \
\
GENTPROT2( double,   float,    d, s, __VA_ARGS__ ) \
GENTPROT2( double,   scomplex, d, c, __VA_ARGS__ ) \
GENTPROT2( double,   dcomplex, d, z, __VA_ARGS__ ) \
\
GENTPROT2( scomplex, float,    c, s, __VA_ARGS__ ) \
GENTPROT2( scomplex, double,   c, d, __VA_ARGS__ ) \
GENTPROT2( scomplex, dcomplex, c, z, __VA_ARGS__ ) \
\
GENTPROT2( dcomplex, float,    z, s, __VA_ARGS__ ) \
GENTPROT2( dcomplex, double,   z, d, __VA_ARGS__ ) \
GENTPROT2( dcomplex, scomplex, z, c, __VA_ARGS__ )



// -- Basic two-operand with real projection of first operand --

#define INSERT_GENTPROT2R_BASIC( ... ) \
\
GENTPROT2R( float,    float,    float,    s, s, s, __VA_ARGS__ ) \
GENTPROT2R( double,   double,   double,   d, d, d, __VA_ARGS__ ) \
GENTPROT2R( scomplex, scomplex, float,    c, c, s, __VA_ARGS__ ) \
GENTPROT2R( dcomplex, dcomplex, double,   z, z, d, __VA_ARGS__ )



// -- Mixed domain two-operand with real projection of first operand --

#define INSERT_GENTPROT2R_MIX_D( ... ) \
\
GENTPROT2R( float,    scomplex, float,    s, c, s, __VA_ARGS__ ) \
GENTPROT2R( scomplex, float,    float,    c, s, s, __VA_ARGS__ ) \
\
GENTPROT2R( double,   dcomplex, double,   d, z, d, __VA_ARGS__ ) \
GENTPROT2R( dcomplex, double,   double,   z, d, d, __VA_ARGS__ )



// -- Mixed precision two-operand with real projection of first operand --

#define INSERT_GENTPROT2R_MIX_P( ... ) \
\
GENTPROT2R( float,    double,   float,    s, d, s, __VA_ARGS__ ) \
GENTPROT2R( float,    dcomplex, float,    s, z, s, __VA_ARGS__ ) \
\
GENTPROT2R( double,   float,    double,   d, s, d, __VA_ARGS__ ) \
GENTPROT2R( double,   scomplex, double,   d, c, d, __VA_ARGS__ ) \
\
GENTPROT2R( scomplex, double,   float,    c, d, s, __VA_ARGS__ ) \
GENTPROT2R( scomplex, dcomplex, float,    c, z, s, __VA_ARGS__ ) \
\
GENTPROT2R( dcomplex, float,    double,   z, s, d, __VA_ARGS__ ) \
GENTPROT2R( dcomplex, scomplex, double,   z, c, d, __VA_ARGS__ )



// -- Macros for functions with three primary operands -------------------------


// -- Basic three-operand macro --


#define INSERT_GENTPROT3_BASIC( funcname ) \
\
GENTPROT3( float,    float,    float,    s, s, s, funcname ) \
GENTPROT3( double,   double,   double,   d, d, d, funcname ) \
GENTPROT3( scomplex, scomplex, scomplex, c, c, c, funcname ) \
GENTPROT3( dcomplex, dcomplex, dcomplex, z, z, z, funcname )


// -- Mixed domain three-operand macro --


#define INSERT_GENTPROT3_MIX_D( funcname ) \
\
GENTPROT3( float,    float,    scomplex, s, s, c, funcname ) \
GENTPROT3( float,    scomplex, float,    s, c, s, funcname ) \
GENTPROT3( float,    scomplex, scomplex, s, c, c, funcname ) \
\
GENTPROT3( double,   double,   dcomplex, d, d, z, funcname ) \
GENTPROT3( double,   dcomplex, double,   d, z, d, funcname ) \
GENTPROT3( double,   dcomplex, dcomplex, d, z, z, funcname ) \
\
GENTPROT3( scomplex, float,    float,    c, s, s, funcname ) \
GENTPROT3( scomplex, float,    scomplex, c, s, c, funcname ) \
GENTPROT3( scomplex, scomplex, float,    c, c, s, funcname ) \
\
GENTPROT3( dcomplex, double,   double,   z, d, d, funcname ) \
GENTPROT3( dcomplex, double,   dcomplex, z, d, z, funcname ) \
GENTPROT3( dcomplex, dcomplex, double,   z, z, d, funcname )


// -- Mixed precision three-operand macro --


#define INSERT_GENTPROT3_MIX_P( funcname ) \
\
GENTPROT3( float,    float,    double,   s, s, d, funcname ) \
GENTPROT3( float,    float,    dcomplex, s, s, z, funcname ) \
\
GENTPROT3( float,    double,   float,    s, d, s, funcname ) \
GENTPROT3( float,    double,   double,   s, d, d, funcname ) \
GENTPROT3( float,    double,   scomplex, s, d, c, funcname ) \
GENTPROT3( float,    double,   dcomplex, s, d, z, funcname ) \
\
GENTPROT3( float,    scomplex, double,   s, c, d, funcname ) \
GENTPROT3( float,    scomplex, dcomplex, s, c, z, funcname ) \
\
GENTPROT3( float,    dcomplex, float,    s, z, s, funcname ) \
GENTPROT3( float,    dcomplex, double,   s, z, d, funcname ) \
GENTPROT3( float,    dcomplex, scomplex, s, z, c, funcname ) \
GENTPROT3( float,    dcomplex, dcomplex, s, z, z, funcname ) \
\
\
GENTPROT3( double,   float,    float,    d, s, s, funcname ) \
GENTPROT3( double,   float,    double,   d, s, d, funcname ) \
GENTPROT3( double,   float,    scomplex, d, s, c, funcname ) \
GENTPROT3( double,   float,    dcomplex, d, s, z, funcname ) \
\
GENTPROT3( double,   double,   float,    d, d, s, funcname ) \
GENTPROT3( double,   double,   scomplex, d, d, c, funcname ) \
\
GENTPROT3( double,   scomplex, float,    d, c, s, funcname ) \
GENTPROT3( double,   scomplex, double,   d, c, d, funcname ) \
GENTPROT3( double,   scomplex, scomplex, d, c, c, funcname ) \
GENTPROT3( double,   scomplex, dcomplex, d, c, z, funcname ) \
\
GENTPROT3( double,   dcomplex, float,    d, z, s, funcname ) \
GENTPROT3( double,   dcomplex, scomplex, d, z, c, funcname ) \
\
\
GENTPROT3( scomplex, float,    double,   c, s, d, funcname ) \
GENTPROT3( scomplex, float,    dcomplex, c, s, z, funcname ) \
\
GENTPROT3( scomplex, double,   float,    c, d, s, funcname ) \
GENTPROT3( scomplex, double,   double,   c, d, d, funcname ) \
GENTPROT3( scomplex, double,   scomplex, c, d, c, funcname ) \
GENTPROT3( scomplex, double,   dcomplex, c, d, z, funcname ) \
\
GENTPROT3( scomplex, scomplex, double,   c, c, d, funcname ) \
GENTPROT3( scomplex, scomplex, dcomplex, c, c, z, funcname ) \
\
GENTPROT3( scomplex, dcomplex, float,    c, z, s, funcname ) \
GENTPROT3( scomplex, dcomplex, double,   c, z, d, funcname ) \
GENTPROT3( scomplex, dcomplex, scomplex, c, z, c, funcname ) \
GENTPROT3( scomplex, dcomplex, dcomplex, c, z, z, funcname ) \
\
\
GENTPROT3( dcomplex, float,    float,    z, s, s, funcname ) \
GENTPROT3( dcomplex, float,    double,   z, s, d, funcname ) \
GENTPROT3( dcomplex, float,    scomplex, z, s, c, funcname ) \
GENTPROT3( dcomplex, float,    dcomplex, z, s, z, funcname ) \
\
GENTPROT3( dcomplex, double,   float,    z, d, s, funcname ) \
GENTPROT3( dcomplex, double,   scomplex, z, d, c, funcname ) \
\
GENTPROT3( dcomplex, scomplex, float,    z, c, s, funcname ) \
GENTPROT3( dcomplex, scomplex, double,   z, c, d, funcname ) \
GENTPROT3( dcomplex, scomplex, scomplex, z, c, c, funcname ) \
GENTPROT3( dcomplex, scomplex, dcomplex, z, c, z, funcname ) \
\
GENTPROT3( dcomplex, dcomplex, float,    z, z, s, funcname ) \
GENTPROT3( dcomplex, dcomplex, scomplex, z, z, c, funcname ) \



// -- Basic three-operand with union of operands 1 and 2 --


#define INSERT_GENTPROT3U12_BASIC( funcname ) \
\
GENTPROT3U12( float,    float,    float,    float,    s, s, s, s, funcname ) \
GENTPROT3U12( double,   double,   double,   double,   d, d, d, d, funcname ) \
GENTPROT3U12( scomplex, scomplex, scomplex, scomplex, c, c, c, c, funcname ) \
GENTPROT3U12( dcomplex, dcomplex, dcomplex, dcomplex, z, z, z, z, funcname )


// -- Mixed domain three-operand with union of operands 1 and 2 --


#define INSERT_GENTPROT3U12_MIX_D( funcname ) \
\
GENTPROT3U12( float,    float,    scomplex, float,    s, s, c, s, funcname ) \
GENTPROT3U12( float,    scomplex, float,    scomplex, s, c, s, c, funcname ) \
GENTPROT3U12( float,    scomplex, scomplex, scomplex, s, c, c, c, funcname ) \
\
GENTPROT3U12( double,   double,   dcomplex, double,   d, d, z, d, funcname ) \
GENTPROT3U12( double,   dcomplex, double,   dcomplex, d, z, d, z, funcname ) \
GENTPROT3U12( double,   dcomplex, dcomplex, dcomplex, d, z, z, z, funcname ) \
\
GENTPROT3U12( scomplex, float,    float,    scomplex, c, s, s, c, funcname ) \
GENTPROT3U12( scomplex, float,    scomplex, scomplex, c, s, c, c, funcname ) \
GENTPROT3U12( scomplex, scomplex, float,    scomplex, c, c, s, c, funcname ) \
\
GENTPROT3U12( dcomplex, double,   double,   dcomplex, z, d, d, z, funcname ) \
GENTPROT3U12( dcomplex, double,   dcomplex, dcomplex, z, d, z, z, funcname ) \
GENTPROT3U12( dcomplex, dcomplex, double,   dcomplex, z, z, d, z, funcname )


// -- Mixed precision three-operand with union of operands 1 and 2 --


#define INSERT_GENTPROT3U12_MIX_P( funcname ) \
\
GENTPROT3U12( float,    float,    double,   float,    s, s, d, s, funcname ) \
GENTPROT3U12( float,    float,    dcomplex, float,    s, s, z, s, funcname ) \
\
GENTPROT3U12( float,    double,   float,    double,   s, d, s, d, funcname ) \
GENTPROT3U12( float,    double,   double,   double,   s, d, d, d, funcname ) \
GENTPROT3U12( float,    double,   scomplex, double,   s, d, c, d, funcname ) \
GENTPROT3U12( float,    double,   dcomplex, double,   s, d, z, d, funcname ) \
\
GENTPROT3U12( float,    scomplex, double,   scomplex, s, c, d, c, funcname ) \
GENTPROT3U12( float,    scomplex, dcomplex, scomplex, s, c, z, c, funcname ) \
\
GENTPROT3U12( float,    dcomplex, float,    dcomplex, s, z, s, z, funcname ) \
GENTPROT3U12( float,    dcomplex, double,   dcomplex, s, z, d, z, funcname ) \
GENTPROT3U12( float,    dcomplex, scomplex, dcomplex, s, z, c, z, funcname ) \
GENTPROT3U12( float,    dcomplex, dcomplex, dcomplex, s, z, z, z, funcname ) \
\
\
GENTPROT3U12( double,   float,    float,    double,   d, s, s, d, funcname ) \
GENTPROT3U12( double,   float,    double,   double,   d, s, d, d, funcname ) \
GENTPROT3U12( double,   float,    scomplex, double,   d, s, c, d, funcname ) \
GENTPROT3U12( double,   float,    dcomplex, double,   d, s, z, d, funcname ) \
\
GENTPROT3U12( double,   double,   float,    double,   d, d, s, d, funcname ) \
GENTPROT3U12( double,   double,   scomplex, double,   d, d, c, d, funcname ) \
\
GENTPROT3U12( double,   scomplex, float,    dcomplex, d, c, s, z, funcname ) \
GENTPROT3U12( double,   scomplex, double,   dcomplex, d, c, d, z, funcname ) \
GENTPROT3U12( double,   scomplex, scomplex, dcomplex, d, c, c, z, funcname ) \
GENTPROT3U12( double,   scomplex, dcomplex, dcomplex, d, c, z, z, funcname ) \
\
GENTPROT3U12( double,   dcomplex, float,    dcomplex, d, z, s, z, funcname ) \
GENTPROT3U12( double,   dcomplex, scomplex, dcomplex, d, z, c, z, funcname ) \
\
\
GENTPROT3U12( scomplex, float,    double,   scomplex, c, s, d, c, funcname ) \
GENTPROT3U12( scomplex, float,    dcomplex, scomplex, c, s, z, c, funcname ) \
\
GENTPROT3U12( scomplex, double,   float,    dcomplex, c, d, s, z, funcname ) \
GENTPROT3U12( scomplex, double,   double,   dcomplex, c, d, d, z, funcname ) \
GENTPROT3U12( scomplex, double,   scomplex, dcomplex, c, d, c, z, funcname ) \
GENTPROT3U12( scomplex, double,   dcomplex, dcomplex, c, d, z, z, funcname ) \
\
GENTPROT3U12( scomplex, scomplex, double,   scomplex, c, c, d, c, funcname ) \
GENTPROT3U12( scomplex, scomplex, dcomplex, scomplex, c, c, z, c, funcname ) \
\
GENTPROT3U12( scomplex, dcomplex, float,    dcomplex, c, z, s, z, funcname ) \
GENTPROT3U12( scomplex, dcomplex, double,   dcomplex, c, z, d, z, funcname ) \
GENTPROT3U12( scomplex, dcomplex, scomplex, dcomplex, c, z, c, z, funcname ) \
GENTPROT3U12( scomplex, dcomplex, dcomplex, dcomplex, c, z, z, z, funcname ) \
\
\
GENTPROT3U12( dcomplex, float,    float,    dcomplex, z, s, s, z, funcname ) \
GENTPROT3U12( dcomplex, float,    double,   dcomplex, z, s, d, z, funcname ) \
GENTPROT3U12( dcomplex, float,    scomplex, dcomplex, z, s, c, z, funcname ) \
GENTPROT3U12( dcomplex, float,    dcomplex, dcomplex, z, s, z, z, funcname ) \
\
GENTPROT3U12( dcomplex, double,   float,    dcomplex, z, d, s, z, funcname ) \
GENTPROT3U12( dcomplex, double,   scomplex, dcomplex, z, d, c, z, funcname ) \
\
GENTPROT3U12( dcomplex, scomplex, float,    dcomplex, z, c, s, z, funcname ) \
GENTPROT3U12( dcomplex, scomplex, double,   dcomplex, z, c, d, z, funcname ) \
GENTPROT3U12( dcomplex, scomplex, scomplex, dcomplex, z, c, c, z, funcname ) \
GENTPROT3U12( dcomplex, scomplex, dcomplex, dcomplex, z, c, z, z, funcname ) \
\
GENTPROT3U12( dcomplex, dcomplex, float,    dcomplex, z, z, s, z, funcname ) \
GENTPROT3U12( dcomplex, dcomplex, scomplex, dcomplex, z, z, c, z, funcname )


#endif
// end bli_gentprot_macro_defs.h
#line 99 "./frame/include//bli_macro_defs.h"


// begin bli_misc_macro_defs.h
#line 1 "./frame/include//bli_misc_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_MISC_MACRO_DEFS_H
#define BLIS_MISC_MACRO_DEFS_H


// -- Miscellaneous macros --

// min, max, abs
// NOTE: These must remain macros since we don't know the types of a and b.

#define bli_min( a, b )  ( (a) < (b) ? (a) : (b) )
#define bli_max( a, b )  ( (a) > (b) ? (a) : (b) )
#define bli_abs( a )     ( (a) <= 0 ? -(a) : (a) )

// fmin, fmax, fabs
// NOTE: These must remain macros since we don't know the types of a and b.

#define bli_fmin( a, b ) bli_min( a, b )
#define bli_fmax( a, b ) bli_max( a, b )
#define bli_fabs( a )    ( (a) <= 0.0 ? -(a) : (a) )

// fminabs, fmaxabs
// NOTE: These must remain macros since we don't know the types of a and b.

#define bli_fminabs( a, b ) \
\
	bli_fmin( bli_fabs( a ), \
	          bli_fabs( b ) )

#define bli_fmaxabs( a, b ) \
\
	bli_fmax( bli_fabs( a ), \
	          bli_fabs( b ) )

// round

BLIS_INLINE double bli_round( double a )
{
	return round( a );
}

// round_to_mult

BLIS_INLINE guint_t bli_round_to_mult( guint_t val, guint_t mult )
{
	return ( guint_t )
	       ( ( ( ( guint_t )val +
	             ( guint_t )mult / 2
	           ) / mult
	         ) * mult
	       );
}

// isnan, isinf
// NOTE: These must remain macros, since isinf() and isnan() are macros
// (defined in math.h) that likely depend on the type of the argument 'a'
// below.

#define bli_isinf( a )  isinf( a )
#define bli_isnan( a )  isnan( a )

// is_odd, is_even

BLIS_INLINE bool bli_is_odd( gint_t a )
{
	return ( bool )( a % 2 == 1 );
}

BLIS_INLINE bool bli_is_even( gint_t a )
{
	return ( bool )( a % 2 == 0 );
}

// swap_dims

BLIS_INLINE void bli_swap_dims( dim_t* dim1, dim_t* dim2 )
{
	dim_t temp = *dim1;
	*dim1 = *dim2;
	*dim2 = temp;
}

// swap_incs

BLIS_INLINE void bli_swap_incs( inc_t* inc1, inc_t* inc2 )
{
	inc_t temp = *inc1;
	*inc1 = *inc2;
	*inc2 = temp;
}

// toggle_bool

BLIS_INLINE void bli_toggle_bool( bool* b )
{
	if ( *b == TRUE ) *b = FALSE;
	else              *b = TRUE;
}

// return datatype for char

#define bli_stype ( BLIS_FLOAT    )
#define bli_dtype ( BLIS_DOUBLE   )
#define bli_ctype ( BLIS_SCOMPLEX )
#define bli_ztype ( BLIS_DCOMPLEX )

// return C type for char

#define bli_sctype  float
#define bli_dctype  double
#define bli_cctype  scomplex
#define bli_zctype  dcomplex

// return real proj of C type for char

#define bli_sctyper  float
#define bli_dctyper  double
#define bli_cctyper  float
#define bli_zctyper  double


// return default format specifier for char

// NOTE: These must remain macros due to the way they are used to initialize
// local char arrays.

#define bli_sformatspec() "%9.2e"
#define bli_dformatspec() "%9.2e"
#define bli_cformatspec() "%9.2e + %9.2e "
#define bli_zformatspec() "%9.2e + %9.2e "
#define bli_iformatspec() "%6d"


// Sentinel constant used to indicate the end of a variable argument function
// (See bli_cntx.c)

#define BLIS_VA_END  (-1)


// Static assertion compatible with any version of C/C++
#define bli_static_assert(cond) while(0){struct s {int STATIC_ASSERT_FAILED : !!(cond);};}

#endif
// end bli_misc_macro_defs.h
#line 101 "./frame/include//bli_macro_defs.h"

// begin bli_edge_case_macro_defs.h
#line 1 "./frame/include//bli_edge_case_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2021, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_EDGE_CASE_MACRO_DEFS_H
#define BLIS_EDGE_CASE_MACRO_DEFS_H

//
// Macros for edge-case handling within gemm microkernels.
//

// -- Setup helper macros --

#define GEMM_UKR_SETUP_CT_PRE(ch,mr,nr,row_major,alignment) \
\
	const PASTEMAC(ch,ctype)* restrict _beta   = beta; \
	      PASTEMAC(ch,ctype)* restrict _c      = c; \
	const inc_t                        _rs_c   = rs_c; \
	const inc_t                        _cs_c   = cs_c; \
	      PASTEMAC(ch,ctype)           _ct[ BLIS_STACK_BUF_MAX_SIZE \
	                                        / sizeof( PASTEMAC(ch,ctype) ) \
	                                      ] __attribute__((aligned(alignment))); \
	const inc_t                        _rs_ct  = row_major ? nr :  1; \
	const inc_t                        _cs_ct  = row_major ?  1 : mr;

#define GEMM_UKR_SETUP_CT_POST(ch) \
\
	PASTEMAC(ch,ctype) _zero; \
	PASTEMAC(ch,set0s)( _zero ); \
	\
	if ( _use_ct ) \
	{ \
		c = _ct; \
		rs_c = _rs_ct; \
		cs_c = _cs_ct; \
		beta = &_zero; \
	}

// -- Setup macros --

#define GEMM_UKR_SETUP_CT(ch,mr,nr,row_major) \
\
	/* Scenario 1: the ukernel contains assembly-level support only for its
	   IO preference (e.g. only row-oriented or only column-oriented IO).
	   Use a temporary microtile for the other two cases as well as edge
	   cases. */ \
	GEMM_UKR_SETUP_CT_PRE(ch,mr,nr,row_major,1); \
	const bool _use_ct = ( row_major ? cs_c != 1 : rs_c != 1 ) || \
	                     m != mr || n != nr; \
	GEMM_UKR_SETUP_CT_POST(ch);

#define GEMM_UKR_SETUP_CT_AMBI(ch,mr,nr,row_major) \
\
	/* Scenario 2: the ukernel contains assembly-level support for its IO
	   preference as well as its opposite via in-register transpose
	   (e.g. both row- and column-oriented IO). Use a temporary microtile
	   for the general stride case as well as edge cases. */ \
	GEMM_UKR_SETUP_CT_PRE(ch,mr,nr,row_major,1); \
	const bool _use_ct = ( cs_c != 1 && rs_c != 1 ) || \
	                     m != mr || n != nr; \
	GEMM_UKR_SETUP_CT_POST(ch);

#define GEMM_UKR_SETUP_CT_ANY(ch,mr,nr,row_major) \
\
	/* Scenario 3: Similar to (2) where the assembly region also supports
	   general stride I0. Use a temporary microtile only for edge cases. */ \
	GEMM_UKR_SETUP_CT_PRE(ch,mr,nr,row_major,1); \
	const bool _use_ct = ( m != mr || n != nr ); \
	GEMM_UKR_SETUP_CT_POST(ch);

#define GEMM_UKR_SETUP_CT_ALIGNED(ch,mr,nr,row_major,alignment) \
\
	/* Scenario 4: Similar to (1), but uses temporary microtile to handle
	   cases where the pointer to the C microtile is not aligned. */ \
	GEMM_UKR_SETUP_CT_PRE(ch,mr,nr,row_major,alignment); \
	const bool _use_ct = ( row_major ? cs_c != 1 : rs_c != 1 ) || \
	                     m != mr || n != nr || \
	                     ( (uintptr_t)_c % alignment ) || \
	                     ( ( ( row_major ? _rs_c : _cs_c )*sizeof( PASTEMAC(ch,ctype) ) ) % alignment ); \
	GEMM_UKR_SETUP_CT_POST(ch);

// -- Flush macros --

#define GEMM_UKR_FLUSH_CT(ch) \
\
	/* If we actually used the temporary microtile, accumulate it to the output
	   microtile. */ \
	if ( _use_ct ) \
	{ \
		PASTEMAC(ch,xpbys_mxn) \
		( \
		  m, n, \
		  _ct, _rs_ct, _cs_ct, \
		  _beta, \
		  _c,  _rs_c,  _cs_c \
		); \
	} \


//
// Macros for edge-case handling within gemmtrsm microkernels.
//

// -- Setup helper macros --

#define GEMMTRSM_UKR_SETUP_CT_PRE(ch,mr,nr,row_major,alignment) \
\
	      PASTEMAC(ch,ctype)* restrict _c      = c11; \
	const inc_t                        _rs_c   = rs_c; \
	const inc_t                        _cs_c   = cs_c; \
	      PASTEMAC(ch,ctype)           _ct[ BLIS_STACK_BUF_MAX_SIZE \
	                                        / sizeof( PASTEMAC(ch,ctype) ) \
	                                      ] __attribute__((aligned(alignment))); \
	const inc_t                        _rs_ct  = row_major ? nr :  1; \
	const inc_t                        _cs_ct  = row_major ?  1 : mr;

#define GEMMTRSM_UKR_SETUP_CT_POST(ch) \
\
	if ( _use_ct ) \
	{ \
		c11 = _ct; \
		rs_c = _rs_ct; \
		cs_c = _cs_ct; \
	}

// -- Setup macros --

#define GEMMTRSM_UKR_SETUP_CT(ch,mr,nr,row_major) \
\
	/* Scenario 1: the ukernel contains assembly-level support only for its
	   IO preference (e.g. only row-oriented or only column-oriented IO).
	   Use a temporary microtile for the other two cases as well as edge
	   cases. */ \
	GEMMTRSM_UKR_SETUP_CT_PRE(ch,mr,nr,row_major,1); \
	const bool _use_ct = ( row_major ? cs_c != 1 : rs_c != 1 ) || \
	                     m != mr || n != nr; \
	GEMMTRSM_UKR_SETUP_CT_POST(ch);

#define GEMMTRSM_UKR_SETUP_CT_AMBI(ch,mr,nr,row_major) \
\
	/* Scenario 2: the ukernel contains assembly-level support for its IO
	   preference as well as its opposite via in-register transpose
	   (e.g. both row- and column-oriented IO). Use a temporary microtile
	   for the general stride case as well as edge cases. */ \
	GEMMTRSM_UKR_SETUP_CT_PRE(ch,mr,nr,row_major,1); \
	const bool _use_ct = ( cs_c != 1 && rs_c != 1 ) || \
	                     m != mr || n != nr; \
	GEMMTRSM_UKR_SETUP_CT_POST(ch);

#define GEMMTRSM_UKR_SETUP_CT_ANY(ch,mr,nr,row_major) \
\
	/* Scenario 3: Similar to (2) where the assembly region also supports
	   general stride I0. Use a temporary microtile only for edge cases. */ \
	GEMMTRSM_UKR_SETUP_CT_PRE(ch,mr,nr,row_major,1); \
	const bool _use_ct = ( m != mr || n != nr ); \
	GEMMTRSM_UKR_SETUP_CT_POST(ch);

#define GEMMTRSM_UKR_SETUP_CT_ALIGNED(ch,mr,nr,row_major,alignment) \
\
	/* Scenario 4: Similar to (1), but uses temporary microtile to handle
	   cases where the pointer to the C microtile is not aligned. */ \
	GEMMTRSM_UKR_SETUP_CT_PRE(ch,mr,nr,row_major,alignment); \
	const bool _use_ct = ( row_major ? cs_c != 1 : rs_c != 1 ) || \
	                     m != mr || n != nr || \
	                     ( (uintptr_t)_c % alignment ) || \
	                     ( ( ( row_major ? _rs_c : _cs_c )*sizeof( PASTEMAC(ch,ctype) ) ) % alignment ); \
	GEMMTRSM_UKR_SETUP_CT_POST(ch);

// -- Flush macros --

#define GEMMTRSM_UKR_FLUSH_CT(ch) \
\
	/* If we actually used the temporary microtile, use it to overwrite the
	   output microtile. Used by trsm. */ \
	if ( _use_ct ) \
	{ \
		PASTEMAC(ch,copys_mxn) \
		( \
		  m, n, \
		  _ct, _rs_ct, _cs_ct, \
		  _c,  _rs_c,  _cs_c \
		); \
	} \


#endif

// end bli_edge_case_macro_defs.h
#line 102 "./frame/include//bli_macro_defs.h"

// begin bli_param_macro_defs.h
#line 1 "./frame/include//bli_param_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_PARAM_MACRO_DEFS_H
#define BLIS_PARAM_MACRO_DEFS_H


// -- Parameter query macros --

// buffer

BLIS_INLINE bool bli_is_aligned_to( siz_t p, siz_t size )
{
	return ( bool )
	       ( p % size == 0 );
}

BLIS_INLINE bool bli_is_unaligned_to( siz_t p, siz_t size )
{
	return ( bool )
	       ( p % size != 0 );
}

BLIS_INLINE siz_t bli_offset_past_alignment( siz_t p, siz_t size )
{
	return ( siz_t )
	       ( p % size );
}


// datatype

BLIS_INLINE bool bli_is_float( num_t dt )
{
	return ( bool )
	       ( dt == BLIS_FLOAT );
}

BLIS_INLINE bool bli_is_double( num_t dt )
{
	return ( bool )
	       ( dt == BLIS_DOUBLE );
}

BLIS_INLINE bool bli_is_scomplex( num_t dt )
{
	return ( bool )
	       ( dt == BLIS_SCOMPLEX );
}

BLIS_INLINE bool bli_is_dcomplex( num_t dt )
{
	return ( bool )
	       ( dt == BLIS_DCOMPLEX );
}

BLIS_INLINE bool bli_is_constant( num_t dt )
{
	return ( bool )
	       ( dt == BLIS_CONSTANT );
}

BLIS_INLINE bool bli_is_int( num_t dt )
{
	return ( bool )
	       ( dt == BLIS_INT );
}

BLIS_INLINE bool bli_is_real( num_t dt )
{
	return ( bool )
	       ( bli_is_float( dt ) ||
	                   bli_is_double( dt ) );
}

BLIS_INLINE bool bli_is_complex( num_t dt )
{
	return ( bool )
	       ( bli_is_scomplex( dt ) ||
	                   bli_is_dcomplex( dt ) );
}

BLIS_INLINE bool bli_is_single_prec( num_t dt )
{
	return ( bool )
	       ( bli_is_float( dt ) ||
	                   bli_is_scomplex( dt ) );
}

BLIS_INLINE bool bli_is_double_prec( num_t dt )
{
	return ( bool )
	       ( bli_is_double( dt ) ||
	                   bli_is_dcomplex( dt ) );
}

BLIS_INLINE dom_t bli_dt_domain( num_t dt )
{
	return ( dom_t )
	       ( dt & BLIS_DOMAIN_BIT );
}

BLIS_INLINE bool bli_dt_dom_is_real( num_t dt )
{
	return ( bool )
	       ( ( dt & BLIS_DOMAIN_BIT ) == BLIS_REAL );
}

BLIS_INLINE bool bli_dt_dom_is_complex( num_t dt )
{
	return ( bool )
	       ( ( dt & BLIS_DOMAIN_BIT ) == BLIS_COMPLEX );
}

BLIS_INLINE prec_t bli_dt_prec( num_t dt )
{
	return ( prec_t )
	       ( dt & BLIS_PRECISION_BIT );
}

BLIS_INLINE bool bli_dt_prec_is_single( num_t dt )
{
	return ( bool )
	       ( ( dt & BLIS_PRECISION_BIT ) == BLIS_SINGLE_PREC );
}

BLIS_INLINE bool bli_dt_prec_is_double( num_t dt )
{
	return ( bool )
	       ( ( dt & BLIS_PRECISION_BIT ) == BLIS_DOUBLE_PREC );
}

BLIS_INLINE num_t bli_dt_proj_to_real( num_t dt )
{
	return ( num_t )
	       ( dt & ~BLIS_BITVAL_COMPLEX );
}

BLIS_INLINE num_t bli_dt_proj_to_complex( num_t dt )
{
	return ( num_t )
	       ( dt | BLIS_BITVAL_COMPLEX );
}

BLIS_INLINE num_t bli_dt_proj_to_single_prec( num_t dt )
{
	return ( num_t )
	       ( dt & ~BLIS_BITVAL_DOUBLE_PREC );
}

BLIS_INLINE num_t bli_dt_proj_to_double_prec( num_t dt )
{
	return ( num_t )
	       ( dt | BLIS_BITVAL_DOUBLE_PREC );
}


// trans

BLIS_INLINE bool bli_is_notrans( trans_t trans )
{
	return ( bool )
	       ( trans == BLIS_NO_TRANSPOSE );
}

BLIS_INLINE bool bli_is_trans( trans_t trans )
{
	return ( bool )
	       ( trans == BLIS_TRANSPOSE );
}

BLIS_INLINE bool bli_is_conjnotrans( trans_t trans )
{
	return ( bool )
	       ( trans == BLIS_CONJ_NO_TRANSPOSE );
}

BLIS_INLINE bool bli_is_conjtrans( trans_t trans )
{
	return ( bool )
	       ( trans == BLIS_CONJ_TRANSPOSE );
}

BLIS_INLINE bool bli_does_notrans( trans_t trans )
{
	return ( bool )
	       ( (~trans & BLIS_TRANS_BIT ) == BLIS_BITVAL_TRANS );
}

BLIS_INLINE bool bli_does_trans( trans_t trans )
{
	return ( bool )
	       ( ( trans & BLIS_TRANS_BIT ) == BLIS_BITVAL_TRANS );
}

BLIS_INLINE bool bli_does_noconj( trans_t trans )
{
	return ( bool )
	       ( (~trans & BLIS_CONJ_BIT ) == BLIS_BITVAL_CONJ );
}

BLIS_INLINE bool bli_does_conj( trans_t trans )
{
	return ( bool )
	       ( ( trans & BLIS_CONJ_BIT ) == BLIS_BITVAL_CONJ );
}

BLIS_INLINE trans_t bli_extract_trans( trans_t trans )
{
	return ( trans_t )
	       ( trans & BLIS_TRANS_BIT );
}

BLIS_INLINE conj_t bli_extract_conj( trans_t trans )
{
	return ( conj_t )
	       ( trans & BLIS_CONJ_BIT );
}

BLIS_INLINE trans_t bli_trans_toggled( trans_t trans )
{
	return ( trans_t )
	       ( trans ^ BLIS_TRANS_BIT );
}

BLIS_INLINE trans_t bli_trans_toggled_conj( trans_t trans )
{
	return ( trans_t )
	       ( trans ^ BLIS_CONJ_BIT );
}

BLIS_INLINE trans_t bli_apply_trans( trans_t transapp, trans_t trans )
{
	return ( trans_t )
	       ( trans ^ transapp );
}

BLIS_INLINE void bli_toggle_trans( trans_t* trans )
{
	*trans = bli_trans_toggled( *trans );
}


// side

BLIS_INLINE bool bli_is_left( side_t side )
{
	return ( bool )
	       ( side == BLIS_LEFT );
}

BLIS_INLINE bool bli_is_right( side_t side )
{
	return ( bool )
	       ( side == BLIS_RIGHT );
}

BLIS_INLINE side_t bli_side_toggled( side_t side )
{
	return ( bli_is_left( side ) ? BLIS_RIGHT : BLIS_LEFT );
}

BLIS_INLINE void bli_toggle_side( side_t* side )
{
	*side = bli_side_toggled( *side );
}


// uplo

BLIS_INLINE bool bli_is_lower( uplo_t uplo )
{
	return ( bool )
	       ( uplo == BLIS_LOWER );
}

BLIS_INLINE bool bli_is_upper( uplo_t uplo )
{
	return ( bool )
	       ( uplo == BLIS_UPPER );
}

BLIS_INLINE bool bli_is_upper_or_lower( uplo_t uplo )
{
	return ( bool )
	       ( bli_is_upper( uplo ) ||
	         bli_is_lower( uplo ) );
}

BLIS_INLINE bool bli_is_dense( uplo_t uplo )
{
	return ( bool )
	       ( uplo == BLIS_DENSE );
}

BLIS_INLINE bool bli_is_zeros( uplo_t uplo )
{
	return ( bool )
	       ( uplo == BLIS_ZEROS );
}

BLIS_INLINE uplo_t bli_uplo_toggled( uplo_t uplo )
{
	return ( uplo_t )
	       ( bli_is_upper_or_lower( uplo )
	         ? ( ( uplo ^ BLIS_LOWER_BIT ) ^ BLIS_UPPER_BIT )
	         :     uplo
	       );
}

BLIS_INLINE void bli_toggle_uplo( uplo_t* uplo )
{
	*uplo = bli_uplo_toggled( *uplo );
}


// structure

BLIS_INLINE bool bli_is_general( struc_t struc )
{
	return ( bool )
	       ( struc == BLIS_GENERAL );
}

BLIS_INLINE bool bli_is_hermitian( struc_t struc )
{
	return ( bool )
	       ( struc == BLIS_HERMITIAN );
}

BLIS_INLINE bool bli_is_symmetric( struc_t struc )
{
	return ( bool )
	       ( struc == BLIS_SYMMETRIC );
}

BLIS_INLINE bool bli_is_triangular( struc_t struc )
{
	return ( bool )
	       ( struc == BLIS_TRIANGULAR );
}

BLIS_INLINE bool bli_is_herm_or_symm( struc_t struc )
{
	return ( bool )
	       ( bli_is_hermitian( struc ) ||
	         bli_is_symmetric( struc ) );
}


// conj

BLIS_INLINE bool bli_is_noconj( conj_t conj )
{
	return ( bool )
	       ( conj == BLIS_NO_CONJUGATE );
}

BLIS_INLINE bool bli_is_conj( conj_t conj )
{
	return ( bool )
	       ( conj == BLIS_CONJUGATE );
}

BLIS_INLINE conj_t bli_conj_toggled( conj_t conj )
{
	return ( conj_t )
	       ( conj ^ BLIS_CONJ_BIT );
}

BLIS_INLINE conj_t bli_apply_conj( conj_t conjapp, conj_t conj )
{
	return ( conj_t )
	       ( conj ^ conjapp );
}

BLIS_INLINE void bli_toggle_conj( conj_t* conj )
{
	*conj = bli_conj_toggled( *conj );
}


// diag

BLIS_INLINE bool bli_is_nonunit_diag( diag_t diag )
{
	return ( bool )
	       ( diag == BLIS_NONUNIT_DIAG );
}

BLIS_INLINE bool bli_is_unit_diag( diag_t diag )
{
	return ( bool )
	       ( diag == BLIS_UNIT_DIAG );
}


// err_t-related

BLIS_INLINE bool bli_is_success( err_t err )
{
	return ( bool )
	       ( err == BLIS_SUCCESS );
}

BLIS_INLINE bool bli_is_failure( err_t err )
{
	return ( bool )
	       ( err != BLIS_SUCCESS );
}


// dimension-related

BLIS_INLINE bool bli_zero_dim1( dim_t m )
{
	return ( bool )
	       ( m == 0 );
}

BLIS_INLINE bool bli_zero_dim2( dim_t m, dim_t n )
{
	return ( bool )
	       ( m == 0 || n == 0 );
}

BLIS_INLINE bool bli_zero_dim3( dim_t m, dim_t n, dim_t k )
{
	return ( bool )
	       ( m == 0 || n == 0 || k == 0 );
}

BLIS_INLINE bool bli_nonzero_dim( dim_t m )
{
	return ( bool )
	       ( m > 0 );
}

BLIS_INLINE bool bli_vector_dim( dim_t m, dim_t n )
{
	return ( bool )
	       ( m == 1 ? n : m );
}

BLIS_INLINE bool bli_is_vector( dim_t m, dim_t n )
{
	return ( bool )
	       ( m == 1 || n == 1 );
}

BLIS_INLINE bool bli_is_row_vector( dim_t m, dim_t n )
{
	return ( bool )
	       ( m == 1 );
}

BLIS_INLINE bool bli_is_col_vector( dim_t m, dim_t n )
{
	return ( bool )
	       ( n == 1 );
}

BLIS_INLINE void bli_set_dim_with_side( side_t side, dim_t m, dim_t n, dim_t* dim )
{
	if ( bli_is_left( side ) ) *dim = m;
	else                       *dim = n;
}

BLIS_INLINE void bli_set_dims_with_trans( trans_t trans, dim_t m, dim_t n, dim_t* mt, dim_t* nt )
{
	if ( bli_does_notrans( trans ) ) { *mt = m; *nt = n; }
	else                             { *mt = n; *nt = m; }
}

BLIS_INLINE void bli_set_dims_incs_with_trans( trans_t trans,
                                          dim_t  m,  dim_t  n,  inc_t  rs,  inc_t  cs,
                                          dim_t* mt, dim_t* nt, inc_t* rst, inc_t* cst )
{
	if ( bli_does_notrans( trans ) ) { *mt = m; *nt = n; *rst = rs; *cst = cs; }
	else                             { *mt = n; *nt = m; *rst = cs; *cst = rs; }
}


// blocksize-related

BLIS_INLINE dim_t bli_determine_blocksize_dim_f( dim_t i, dim_t dim, dim_t b_alg )
{
	return ( dim_t )
	       ( bli_min( b_alg, dim - i ) );
}

BLIS_INLINE dim_t bli_determine_blocksize_dim_b( dim_t i, dim_t dim, dim_t b_alg )
{
	return ( dim_t )
	       ( i == 0 && dim % b_alg != 0 ? dim % b_alg
	                                    : b_alg );
}


// stride-related

BLIS_INLINE inc_t bli_vector_inc( trans_t trans, dim_t m, dim_t n, inc_t rs, inc_t cs )
{
	return ( inc_t )
	       ( bli_does_notrans( trans ) ? ( m == 1 ? cs : rs )
	                                   : ( m == 1 ? rs : cs ) );
}

BLIS_INLINE bool bli_is_row_stored( inc_t rs, inc_t cs )
{
	return ( bool )
	       ( bli_abs( cs ) == 1 );
}

BLIS_INLINE bool bli_is_col_stored( inc_t rs, inc_t cs )
{
	return ( bool )
	       ( bli_abs( rs ) == 1 );
}

BLIS_INLINE bool bli_is_row_stored_f( dim_t m, dim_t n, inc_t rs, inc_t cs )
{
	return ( bool )
	       ( cs == 1 && ( rs > 1 || n == 1 ) );
}

BLIS_INLINE bool bli_is_col_stored_f( dim_t m, dim_t n, inc_t rs, inc_t cs )
{
	return ( bool )
	       ( rs == 1 && ( cs > 1 || m == 1 ) );
}

BLIS_INLINE bool bli_is_gen_stored( inc_t rs, inc_t cs )
{
	return ( bool )
	       ( bli_abs( rs ) != 1 &&
	         bli_abs( cs ) != 1 );
}

BLIS_INLINE bool bli_is_row_tilted( dim_t m, dim_t n, inc_t rs, inc_t cs )
{
	return ( bool )
	       ( bli_abs( cs ) == bli_abs( rs )
	         ? n < m
	         : bli_abs( cs ) < bli_abs( rs ) );
}

BLIS_INLINE bool bli_is_col_tilted( dim_t m, dim_t n, inc_t rs, inc_t cs )
{
	return ( bool )
	       ( bli_abs( rs ) == bli_abs( cs )
	         ? m < n
	         : bli_abs( rs ) < bli_abs( cs ) );
}

BLIS_INLINE bool bli_has_nonunit_inc1( inc_t s1 )
{
	return ( bool )
	       ( s1 != 1 );
}

BLIS_INLINE bool bli_has_nonunit_inc2( inc_t s1, inc_t s2 )
{
	return ( bool )
	       ( s1 != 1 || s2 != 1 );
}

BLIS_INLINE bool bli_has_nonunit_inc3( inc_t s1, inc_t s2, inc_t s3 )
{
	return ( bool )
	       ( s1 != 1 || s2 != 1 || s3 != 1 );
}


// diag offset-related

BLIS_INLINE void bli_negate_diag_offset( doff_t* diagoff )
{
	*diagoff = -(*diagoff);
}

BLIS_INLINE void bli_shift_diag_offset_to_grow_uplo( uplo_t uplo, doff_t* diagoff )
{
	if      ( bli_is_upper( uplo ) ) *diagoff -= 1;
	else if ( bli_is_lower( uplo ) ) *diagoff += 1;
}

BLIS_INLINE void bli_shift_diag_offset_to_shrink_uplo( uplo_t uplo, doff_t* diagoff )
{
	if      ( bli_is_upper( uplo ) ) *diagoff += 1;
	else if ( bli_is_lower( uplo ) ) *diagoff -= 1;
}

BLIS_INLINE doff_t bli_diag_offset_with_trans( trans_t trans, doff_t diagoff )
{
	return ( doff_t )
	       ( bli_does_trans( trans ) ? -diagoff : diagoff );
}

BLIS_INLINE bool bli_is_strictly_above_diag( doff_t diagoff, trans_t trans, dim_t m, dim_t n )
{
	return ( bool )
	       ( bli_does_trans( trans )
	         ? ( ( doff_t )n <= -diagoff )
	         : ( ( doff_t )m <= -diagoff ) );
}

BLIS_INLINE bool bli_is_strictly_below_diag( doff_t diagoff, trans_t trans, dim_t m, dim_t n )
{
	return ( bool )
	       ( bli_does_trans( trans )
	         ? ( ( doff_t )m <=  diagoff )
	         : ( ( doff_t )n <=  diagoff ) );
}

BLIS_INLINE bool bli_is_outside_diag( doff_t diagoff, trans_t trans, dim_t m, dim_t n )
{
	return ( bool )
	       ( bli_is_strictly_above_diag( diagoff, trans, m, n ) ||
	         bli_is_strictly_below_diag( diagoff, trans, m, n ) );
}

BLIS_INLINE bool bli_is_stored_subpart( doff_t diagoff, trans_t trans, uplo_t uplo, dim_t m, dim_t n )
{
	return ( bool )
	( ( bli_is_upper( uplo ) && bli_is_strictly_above_diag( diagoff, trans, m, n ) ) ||
	  ( bli_is_lower( uplo ) && bli_is_strictly_below_diag( diagoff, trans, m, n ) ) );
}

BLIS_INLINE bool bli_is_unstored_subpart( doff_t diagoff, trans_t trans, uplo_t uplo, dim_t m, dim_t n )
{
	return ( bool )
	( ( bli_is_upper( uplo ) && bli_is_strictly_below_diag( diagoff, trans, m, n ) ) ||
	  ( bli_is_lower( uplo ) && bli_is_strictly_above_diag( diagoff, trans, m, n ) ) );
}

BLIS_INLINE bool bli_is_strictly_above_diag_n( doff_t diagoff, dim_t m, dim_t n )
{
	return ( bool )
	       ( ( doff_t )m <= -diagoff );
}

BLIS_INLINE bool bli_is_strictly_below_diag_n( doff_t diagoff, dim_t m, dim_t n )
{
	return ( bool )
	       ( ( doff_t )n <=  diagoff );
}

BLIS_INLINE bool bli_intersects_diag_n( doff_t diagoff, dim_t m, dim_t n )
{
	return ( bool )
	       ( !bli_is_strictly_above_diag_n( diagoff, m, n ) &&
	         !bli_is_strictly_below_diag_n( diagoff, m, n ) );
}

BLIS_INLINE bool bli_is_outside_diag_n( doff_t diagoff, dim_t m, dim_t n )
{
	return ( bool )
	       ( bli_is_strictly_above_diag_n( diagoff, m, n ) ||
	         bli_is_strictly_below_diag_n( diagoff, m, n ) );
}

BLIS_INLINE bool bli_is_stored_subpart_n( doff_t diagoff, uplo_t uplo, dim_t m, dim_t n )
{
	return ( bool )
	( ( bli_is_upper( uplo ) && bli_is_strictly_above_diag_n( diagoff, m, n ) ) ||
	  ( bli_is_lower( uplo ) && bli_is_strictly_below_diag_n( diagoff, m, n ) ) );
}

BLIS_INLINE bool bli_is_unstored_subpart_n( doff_t diagoff, uplo_t uplo, dim_t m, dim_t n )
{
	return ( bool )
	( ( bli_is_upper( uplo ) && bli_is_strictly_below_diag_n( diagoff, m, n ) ) ||
	  ( bli_is_lower( uplo ) && bli_is_strictly_above_diag_n( diagoff, m, n ) ) );
}


// pruning-related

BLIS_INLINE void bli_prune_unstored_region_top_l( doff_t* diagoff, dim_t* m, dim_t* n, dim_t* offm_inc )
{
	*offm_inc = 0;

	// If the diagonal intersects the left side of the matrix,
	// ignore the area above that intersection.
	if ( *diagoff < 0 )
	{
		*m        = *m + *diagoff;
		*offm_inc =    - *diagoff;
		*diagoff  = 0;
	}
}

BLIS_INLINE void bli_prune_unstored_region_right_l( doff_t* diagoff, dim_t* m, dim_t* n, dim_t* offn_inc )
{
	*offn_inc = 0;

	// If the diagonal intersects the bottom side of the matrix,
	// ignore the area to the right of that intersection.
	if ( *n > *diagoff + *m )
	{
		*n = *diagoff + *m;
	}
}

BLIS_INLINE void bli_prune_unstored_region_left_u( doff_t* diagoff, dim_t* m, dim_t* n, dim_t* offn_inc )
{
	*offn_inc = 0;

	// If the diagonal intersects the top side of the matrix,
	// ignore the area to the left of that intersection.
	if ( *diagoff > 0 )
	{
		*n        = *n - *diagoff;
		*offn_inc =    + *diagoff;
		*diagoff  = 0;
	}
}

BLIS_INLINE void bli_prune_unstored_region_bottom_u( doff_t* diagoff, dim_t* m, dim_t* n, dim_t* offm_inc )
{
	*offm_inc = 0;

	// If the diagonal intersects the right side of the matrix,
	// ignore the area below that intersection.
	if ( *m > -(*diagoff) + *n )
	{
		*m = -(*diagoff) + *n;
	}
}


// thread range-related

BLIS_INLINE void bli_rotate180_trapezoid( doff_t* diagoff, uplo_t* uplo, dim_t* m, dim_t* n )
{
	*diagoff = *n - *diagoff - *m;
	bli_toggle_uplo( uplo );
}

BLIS_INLINE void bli_reflect_about_diag( doff_t* diagoff, uplo_t* uplo, dim_t* m, dim_t* n )
{
	bli_swap_dims( m, n );
	bli_negate_diag_offset( diagoff );
	bli_toggle_uplo( uplo );
}

// we don't know the type of a, so this must be a macro
// rs_a and cs_a must be variables and not expressions
#define bli_reflect_to_stored_part( diagoff, a, rs_a, cs_a ) \
do { \
	a += ( diagoff ) * ( cs_a - rs_a ); \
	bli_swap_incs( &rs_a, &cs_a ); \
} while (0) \

BLIS_INLINE void bli_reverse_index_direction( dim_t n, dim_t* start, dim_t* end )
{
	dim_t start2 = n - *start;
	dim_t end2   = n - *end;
	*start = end2;
	*end   = start2;
}


// mdim_t-related

BLIS_INLINE bool bli_is_m_dim( mdim_t mdim )
{
	return ( bool )
	       ( mdim == BLIS_M );
}

BLIS_INLINE bool bli_is_n_dim( mdim_t mdim )
{
	return ( bool )
	       ( mdim == BLIS_N );
}

BLIS_INLINE mdim_t bli_dim_toggled( mdim_t mdim )
{
	return ( mdim_t )
	       ( mdim == BLIS_M ? BLIS_N : BLIS_M );
}

BLIS_INLINE void bli_toggle_dim( mdim_t* mdim )
{
	*mdim = bli_dim_toggled( *mdim );
}


// stor3_t-related

BLIS_INLINE stor3_t bli_stor3_from_strides( inc_t rs_c, inc_t cs_c,
                                       inc_t rs_a, inc_t cs_a,
                                       inc_t rs_b, inc_t cs_b  )
{
	// If any matrix is general-stored, return the stor3_t id for the
	// general-purpose sup microkernel.
	if ( bli_is_gen_stored( rs_c, cs_c ) ||
	     bli_is_gen_stored( rs_a, cs_a ) ||
	     bli_is_gen_stored( rs_b, cs_b ) ) return BLIS_XXX;

	// Otherwise, compute and return the stor3_t id as follows.
	const bool c_is_col = bli_is_col_stored( rs_c, cs_c );
	const bool a_is_col = bli_is_col_stored( rs_a, cs_a );
	const bool b_is_col = bli_is_col_stored( rs_b, cs_b );

	return ( stor3_t )( 4 * c_is_col +
	                    2 * a_is_col +
	                    1 * b_is_col );
}

BLIS_INLINE stor3_t bli_stor3_trans( stor3_t id )
{
#if 1
	stor3_t map[ BLIS_NUM_3OP_RC_COMBOS ]
	=
	{
	  ( stor3_t )7,  // BLIS_RRR = 0  ->  BLIS_CCC = 7
	  ( stor3_t )5,  // BLIS_RRC = 1  ->  BLIS_CRC = 5
	  ( stor3_t )6,  // BLIS_RCR = 2  ->  BLIS_CCR = 6
	  ( stor3_t )4,  // BLIS_RCC = 3  ->  BLIS_CRR = 4
	  ( stor3_t )3,  // BLIS_CRR = 4  ->  BLIS_RCC = 3
	  ( stor3_t )1,  // BLIS_CRC = 5  ->  BLIS_RRC = 1
	  ( stor3_t )2,  // BLIS_CCR = 6  ->  BLIS_RCR = 2
	  ( stor3_t )0,  // BLIS_CCC = 7  ->  BLIS_RRR = 0
	};

	return map[id];
#else
	return   ( ( id & 0x4 ) ^ 0x4 )        | // flip c bit
	       ( ( ( id & 0x1 ) ^ 0x1 ) << 1 ) | // flip b bit and move to a position
	       ( ( ( id & 0x2 ) ^ 0x2 ) >> 1 );  // flip a bit and move to b position
#endif
}

BLIS_INLINE ukr_t bli_stor3_ukr( stor3_t id )
{
	switch ( id )
	{
		case BLIS_RRR: return BLIS_GEMMSUP_RRR_UKR;
		case BLIS_RRC: return BLIS_GEMMSUP_RRC_UKR;
		case BLIS_RCR: return BLIS_GEMMSUP_RCR_UKR;
		case BLIS_RCC: return BLIS_GEMMSUP_RCC_UKR;
		case BLIS_CRR: return BLIS_GEMMSUP_CRR_UKR;
		case BLIS_CRC: return BLIS_GEMMSUP_CRC_UKR;
		case BLIS_CCR: return BLIS_GEMMSUP_CCR_UKR;
		case BLIS_CCC: return BLIS_GEMMSUP_CCC_UKR;
		default: return BLIS_GEMMSUP_XXX_UKR;
	}
}

BLIS_INLINE stor3_t bli_stor3_transa( stor3_t id )
{
#if 0
	stor3_t map[ BLIS_NUM_3OP_RC_COMBOS ]
	=
	{
	  ( stor3_t )1,  // BLIS_RRR = 0  ->  BLIS_RRC = 1
	  ( stor3_t )0,  // BLIS_RRC = 1  ->  BLIS_RRR = 0
	  ( stor3_t )3,  // BLIS_RCR = 2  ->  BLIS_RCC = 3
	  ( stor3_t )2,  // BLIS_RCC = 3  ->  BLIS_RCR = 2
	  ( stor3_t )5,  // BLIS_CRR = 4  ->  BLIS_CRC = 5
	  ( stor3_t )4,  // BLIS_CRC = 5  ->  BLIS_CRR = 4
	  ( stor3_t )7,  // BLIS_CCR = 6  ->  BLIS_CCC = 7
	  ( stor3_t )6,  // BLIS_CCC = 7  ->  BLIS_CCR = 6
	};

	return map[id];
#else
	return ( stor3_t )( id ^ 0x1 );
#endif
}

BLIS_INLINE stor3_t bli_stor3_transb( stor3_t id )
{
#if 0
	stor3_t map[ BLIS_NUM_3OP_RC_COMBOS ]
	=
	{
	  ( stor3_t )2,  // BLIS_RRR = 0  ->  BLIS_RCR = 2
	  ( stor3_t )3,  // BLIS_RRC = 1  ->  BLIS_RCC = 3
	  ( stor3_t )0,  // BLIS_RCR = 2  ->  BLIS_RRR = 0
	  ( stor3_t )1,  // BLIS_RCC = 3  ->  BLIS_RRC = 1
	  ( stor3_t )6,  // BLIS_CRR = 4  ->  BLIS_CCR = 6
	  ( stor3_t )7,  // BLIS_CRC = 5  ->  BLIS_CCC = 7
	  ( stor3_t )4,  // BLIS_CCR = 6  ->  BLIS_CRR = 4
	  ( stor3_t )5,  // BLIS_CCC = 7  ->  BLIS_CRC = 5
	};

	return map[id];
#else
	return ( stor3_t )( id ^ 0x2 );
#endif
}


// index-related

BLIS_INLINE bool bli_is_edge_f( dim_t i, dim_t n_iter, dim_t n_left )
{
	return ( bool )
	       ( i == n_iter - 1 && n_left != 0 );
}

BLIS_INLINE bool bli_is_not_edge_f( dim_t i, dim_t n_iter, dim_t n_left )
{
	return ( bool )
	       ( i != n_iter - 1 || n_left == 0 );
}

BLIS_INLINE bool bli_is_edge_b( dim_t i, dim_t n_iter, dim_t n_left )
{
	return ( bool )
	       ( i == 0 && n_left != 0 );
}

BLIS_INLINE bool bli_is_not_edge_b( dim_t i, dim_t n_iter, dim_t n_left )
{
	return ( bool )
	       ( i != 0 || n_left == 0 );
}

BLIS_INLINE bool bli_is_last_iter_sl( dim_t i, dim_t end_iter )
{
	return ( bool )
	       ( i == end_iter - 1 );
}

BLIS_INLINE bool bli_is_last_iter_rr( dim_t i, dim_t end_iter, dim_t tid, dim_t nth )
{
	return ( bool )
	       ( i == end_iter - 1 - ( ( end_iter - tid - 1 ) % nth ) );
}

BLIS_INLINE bool bli_is_last_iter_slrr( dim_t i, dim_t end_iter, dim_t tid, dim_t nth )
{
#ifdef BLIS_ENABLE_JRIR_RR
	return bli_is_last_iter_rr( i, end_iter, tid, nth );
#else // ifdef ( _SLAB || _TLB )
	return bli_is_last_iter_sl( i, end_iter );
#endif
}

BLIS_INLINE bool bli_is_last_iter_l( dim_t i, dim_t end_iter, dim_t tid, dim_t nth )
{
	return bli_is_last_iter_slrr( i, end_iter, tid, nth );
}

BLIS_INLINE bool bli_is_last_iter_u( doff_t diagoff, dim_t mr, dim_t nr, inc_t inc )
{
	return bli_is_strictly_below_diag_n( diagoff + inc*mr, mr, nr );
}

BLIS_INLINE bool bli_is_last_iter_tlb_l( dim_t i, dim_t end_iter )
{
	return bli_is_last_iter_sl( i, end_iter );
}

BLIS_INLINE bool bli_is_last_iter_tlb_u( doff_t diagoff, dim_t mr, dim_t nr )
{
	return bli_is_strictly_below_diag_n( diagoff + 1*mr, mr, nr );
}

BLIS_INLINE bool bli_is_my_iter_sl( dim_t i, dim_t st, dim_t en )
{
	return ( st <= i && i < en );
}

BLIS_INLINE bool bli_is_my_iter_rr( dim_t i, dim_t work_id, dim_t n_way )
{
	return ( i % n_way == work_id % n_way );
}

BLIS_INLINE bool bli_is_my_iter( dim_t i, dim_t st, dim_t en, dim_t work_id, dim_t n_way )
{
	// NOTE: This function is (as of this writing) only called from packm.
	// If the structure of the cpp macros below is ever changed, make sure
	// it is still consistent with that of bli_thread_range_slrr() since
	// these functions are used together in packm.

#ifdef BLIS_ENABLE_JRIR_RR
	return bli_is_my_iter_rr( i, work_id, n_way );
#else // ifdef ( _SLAB || _TLB )
	return bli_is_my_iter_sl( i, st, en );
#endif
}


// packbuf_t-related

BLIS_INLINE guint_t bli_packbuf_index( packbuf_t buf_type )
{
	return ( guint_t )
	       ( ( buf_type & BLIS_PACK_BUFFER_BITS ) >> BLIS_PACK_BUFFER_SHIFT );
}

// pack_t-related

BLIS_INLINE bool bli_is_packed( pack_t schema )
{
	return ( bool )
	       ( schema & BLIS_PACK_BIT );
}

BLIS_INLINE bool bli_is_row_packed( pack_t schema )
{
	return ( bool )
	       ( ( schema & BLIS_PACK_RC_BIT ) == ( BLIS_BITVAL_PACKED_UNSPEC ^
	                                            BLIS_BITVAL_PACKED_ROWS ) );
}

BLIS_INLINE bool bli_is_col_packed( pack_t schema )
{
	return ( bool )
	       ( ( schema & BLIS_PACK_RC_BIT ) == ( BLIS_BITVAL_PACKED_UNSPEC ^
	                                            BLIS_BITVAL_PACKED_COLUMNS ) );
}

BLIS_INLINE bool bli_is_panel_packed( pack_t schema )
{
	return ( bool )
	       ( schema & BLIS_PACK_PANEL_BIT );
}

BLIS_INLINE bool bli_is_1r_packed( pack_t schema )
{
	return ( bool )
	       ( ( schema & BLIS_PACK_FORMAT_BITS ) == BLIS_BITVAL_1R );
}

BLIS_INLINE bool bli_is_1e_packed( pack_t schema )
{
	return ( bool )
	       ( ( schema & BLIS_PACK_FORMAT_BITS ) == BLIS_BITVAL_1E );
}

BLIS_INLINE bool bli_is_1m_packed( pack_t schema )
{
	return ( bool )
	       ( bli_is_1r_packed( schema ) ||
	         bli_is_1e_packed( schema ) );
}

BLIS_INLINE bool bli_is_nat_packed( pack_t schema )
{
	return ( bool )
	       ( ( schema & BLIS_PACK_FORMAT_BITS ) == 0 );
}

BLIS_INLINE bool bli_is_ind_packed( pack_t schema )
{
	return ( bool )
	       ( ( schema & BLIS_PACK_FORMAT_BITS ) != 0 );
}

BLIS_INLINE guint_t bli_pack_schema_index( pack_t schema )
{
	return ( guint_t )
	       ( ( schema & BLIS_PACK_FORMAT_BITS ) >> BLIS_PACK_FORMAT_SHIFT );
}


// Set dimensions, increments, effective uplo/diagoff, etc for ONE matrix
// argument.

BLIS_INLINE void bli_set_dims_incs_uplo_1m
     (
       doff_t  diagoffa, diag_t diaga,
       uplo_t  uploa,    dim_t  m,          dim_t  n,      inc_t  rs_a, inc_t  cs_a,
       uplo_t* uplo_eff, dim_t* n_elem_max, dim_t* n_iter, inc_t* inca, inc_t* lda,
       dim_t*  ij0,      dim_t* n_shift
     )
{
	// This is to prevent the compiler from warning about uninitialized
	// variables.
	*ij0     = 0;
	*n_shift = 0;

	// If matrix A is entirely "unstored", that is, if either:
	// - A is lower-stored and entirely above the diagonal, or
	// - A is upper-stored and entirely below the diagonal
	// then we mark the storage as implicitly zero.
	if ( bli_is_unstored_subpart( diagoffa, BLIS_NO_TRANSPOSE, uploa, m, n ) )
	{
		*uplo_eff = BLIS_ZEROS;
	}
	else
	{
		doff_t diagoffa_use_ = diagoffa;
		doff_t diagoff_eff_;
		dim_t  n_iter_max_;

		if ( bli_is_unit_diag( diaga ) )
			bli_shift_diag_offset_to_shrink_uplo( uploa, &diagoffa_use_ );

		// If matrix A is entirely "stored", that is, if either:
		// - A is upper-stored and entirely above the diagonal, or
		// - A is lower-stored and entirely below the diagonal
		// then we mark the storage as dense.
		if ( bli_is_stored_subpart( diagoffa_use_, BLIS_NO_TRANSPOSE, uploa, m, n ) )
			uploa = BLIS_DENSE;

		n_iter_max_  = n;
		*n_elem_max   = m;
		*inca         = rs_a;
		*lda          = cs_a;
		*uplo_eff     = uploa;
		diagoff_eff_ = diagoffa_use_;

		if ( bli_is_row_tilted( *n_elem_max, n_iter_max_, *inca, *lda ) )
		{
			bli_swap_dims( &n_iter_max_, n_elem_max );
			bli_swap_incs( inca, lda );
			bli_toggle_uplo( uplo_eff );
			bli_negate_diag_offset( &diagoff_eff_ );
		}

		if ( bli_is_dense( *uplo_eff ) )
		{
			*n_iter = n_iter_max_;
		}
		else if ( bli_is_upper( *uplo_eff ) )
		{
			if ( diagoff_eff_ < 0 )
			{
				*ij0        = 0;
				*n_shift    = -diagoff_eff_;
				*n_elem_max = bli_min( *n_elem_max, *n_shift + bli_min( m, n ) );
				*n_iter     = n_iter_max_;
			}
			else
			{
				*ij0        = diagoff_eff_;
				*n_shift    = 0;
				*n_iter     = n_iter_max_ - diagoff_eff_;
			}
		}
		else // if ( bli_is_lower( *uplo_eff ) )
		{
			if ( diagoff_eff_ < 0 )
			{
				*ij0        = -diagoff_eff_;
				*n_shift    = 0;
				*n_elem_max = *n_elem_max + diagoff_eff_;
				*n_iter     = bli_min( *n_elem_max, bli_min( m, n ) );
			}
			else
			{
				*ij0        = 0;
				*n_shift    = diagoff_eff_;
				*n_iter     = bli_min( n_iter_max_, *n_shift + bli_min( m, n ) );
			}
		}
	}
}

// Set dimensions, increments, effective uplo/diagoff, etc for ONE matrix
// argument (without column-wise stride optimization).

BLIS_INLINE void bli_set_dims_incs_uplo_1m_noswap
     (
       doff_t  diagoffa, diag_t diaga,
       uplo_t  uploa,    dim_t  m,          dim_t  n,      inc_t  rs_a, inc_t  cs_a,
       uplo_t* uplo_eff, dim_t* n_elem_max, dim_t* n_iter, inc_t* inca, inc_t* lda,
       dim_t*  ij0,      dim_t* n_shift
     )
{
	// This is to prevent the compiler from warning about uninitialized
	// variables.
	*ij0     = 0;
	*n_shift = 0;

	// If matrix A is entirely "unstored", that is, if either:
	// - A is lower-stored and entirely above the diagonal, or
	// - A is upper-stored and entirely below the diagonal
	// then we mark the storage as implicitly zero.
	if ( bli_is_unstored_subpart( diagoffa, BLIS_NO_TRANSPOSE, uploa, m, n ) )
	{
		*uplo_eff = BLIS_ZEROS;
	}
	else
	{
		doff_t diagoffa_use_ = diagoffa;
		doff_t diagoff_eff_;
		dim_t  n_iter_max_;

		if ( bli_is_unit_diag( diaga ) )
			bli_shift_diag_offset_to_shrink_uplo( uploa, &diagoffa_use_ );

		// If matrix A is entirely "stored", that is, if either:
		// - A is upper-stored and entirely above the diagonal, or
		// - A is lower-stored and entirely below the diagonal
		// then we mark the storage as dense.
		if ( bli_is_stored_subpart( diagoffa_use_, BLIS_NO_TRANSPOSE, uploa, m, n ) )
			uploa = BLIS_DENSE;

		n_iter_max_  = n;
		*n_elem_max   = m;
		*inca         = rs_a;
		*lda          = cs_a;
		*uplo_eff     = uploa;
		diagoff_eff_ = diagoffa_use_;

		if ( bli_is_dense( *uplo_eff ) )
		{
			*n_iter = n_iter_max_;
		}
		else if ( bli_is_upper( *uplo_eff ) )
		{
			if ( diagoff_eff_ < 0 )
			{
				*ij0        = 0;
				*n_shift    = -diagoff_eff_;
				*n_elem_max = bli_min( *n_elem_max, *n_shift + bli_min( m, n ) );
				*n_iter     = n_iter_max_;
			}
			else
			{
				*ij0        = diagoff_eff_;
				*n_shift    = 0;
				*n_iter     = n_iter_max_ - diagoff_eff_;
			}
		}
		else // if ( bli_is_lower( *uplo_eff ) )
		{
			if ( diagoff_eff_ < 0 )
			{
				*ij0        = -diagoff_eff_;
				*n_shift    = 0;
				*n_elem_max = *n_elem_max + diagoff_eff_;
				*n_iter     = bli_min( *n_elem_max, bli_min( m, n ) );
			}
			else
			{
				*ij0        = 0;
				*n_shift    = diagoff_eff_;
				*n_iter     = bli_min( n_iter_max_, *n_shift + bli_min( m, n ) );
			}
		}
	}
}

// Set dimensions and increments for TWO matrix arguments.

BLIS_INLINE void bli_set_dims_incs_2m
     (
       trans_t transa,
       dim_t  m,      dim_t  n,      inc_t  rs_a, inc_t  cs_a,
                                     inc_t  rs_b, inc_t  cs_b,
       dim_t* n_elem, dim_t* n_iter, inc_t* inca, inc_t* lda,
                                     inc_t* incb, inc_t* ldb
     )
{
	{
		*n_iter = n;
		*n_elem = m;
		*inca   = rs_a;
		*lda    = cs_a;
		*incb   = rs_b;
		*ldb    = cs_b;

		if ( bli_does_trans( transa ) )
		{
			bli_swap_incs( inca, lda );
		}

		if ( bli_is_row_tilted( *n_elem, *n_iter, *incb, *ldb ) &&
		     bli_is_row_tilted( *n_elem, *n_iter, *inca, *lda ) )
		{
			bli_swap_dims( n_iter, n_elem );
			bli_swap_incs( inca, lda );
			bli_swap_incs( incb, ldb );
		}
	}
}

// Set dimensions, increments, effective uplo/diagoff, etc for TWO matrix
// arguments.

BLIS_INLINE void bli_set_dims_incs_uplo_2m
     (
       doff_t  diagoffa, diag_t diaga, trans_t transa,
       uplo_t  uploa,    dim_t  m,          dim_t  n,      inc_t  rs_a, inc_t  cs_a,
                                                           inc_t  rs_b, inc_t  cs_b,
       uplo_t* uplo_eff, dim_t* n_elem_max, dim_t* n_iter, inc_t* inca, inc_t* lda,
                                                           inc_t* incb, inc_t* ldb,
       dim_t*  ij0,      dim_t* n_shift
     )
{
	// This is to prevent the compiler from warning about uninitialized
	// variables.
	*ij0     = 0;
	*n_shift = 0;

	// If matrix A is entirely "unstored", that is, if either:
	// - A is lower-stored and entirely above the diagonal, or
	// - A is upper-stored and entirely below the diagonal
	// then we mark the storage as implicitly zero.
	if ( bli_is_unstored_subpart( diagoffa, transa, uploa, m, n ) )
	{
		*uplo_eff = BLIS_ZEROS;
	}
	else
	{
		doff_t diagoffa_use_ = diagoffa;
		doff_t diagoff_eff_;
		dim_t  n_iter_max_;

		if ( bli_is_unit_diag( diaga ) )
			bli_shift_diag_offset_to_shrink_uplo( uploa, &diagoffa_use_ );

		// If matrix A is entirely "stored", that is, if either:
		// - A is upper-stored and entirely above the diagonal, or
		// - A is lower-stored and entirely below the diagonal
		// then we mark the storage as dense.
		if ( bli_is_stored_subpart( diagoffa_use_, transa, uploa, m, n ) )
			uploa = BLIS_DENSE;

		n_iter_max_   = n;
		*n_elem_max   = m;
		*inca         = rs_a;
		*lda          = cs_a;
		*incb         = rs_b;
		*ldb          = cs_b;
		*uplo_eff     = uploa;
		diagoff_eff_ = diagoffa_use_;

		if ( bli_does_trans( transa ) )
		{
			bli_swap_incs( inca, lda );
			bli_toggle_uplo( uplo_eff );
			bli_negate_diag_offset( &diagoff_eff_ );
		}

		if ( bli_is_row_tilted( *n_elem_max, n_iter_max_, *incb, *ldb ) &&
		     bli_is_row_tilted( *n_elem_max, n_iter_max_, *inca, *lda ) )
		{
			bli_swap_dims( &n_iter_max_, n_elem_max );
			bli_swap_incs( inca, lda );
			bli_swap_incs( incb, ldb );
			bli_toggle_uplo( uplo_eff );
			bli_negate_diag_offset( &diagoff_eff_ );
		}

		if ( bli_is_dense( *uplo_eff ) )
		{
			*n_iter = n_iter_max_;
		}
		else if ( bli_is_upper( *uplo_eff ) )
		{
			if ( diagoff_eff_ < 0 )
			{
				*ij0        = 0;
				*n_shift    = -diagoff_eff_;
				*n_elem_max = bli_min( *n_elem_max, *n_shift + bli_min( m, n ) );
				*n_iter     = n_iter_max_;
			}
			else
			{
				*ij0        = diagoff_eff_;
				*n_shift    = 0;
				*n_iter     = n_iter_max_ - diagoff_eff_;
			}
		}
		else // if ( bli_is_lower( *uplo_eff ) )
		{
			if ( diagoff_eff_ < 0 )
			{
				*ij0        = -diagoff_eff_;
				*n_shift    = 0;
				*n_elem_max = *n_elem_max + diagoff_eff_;
				*n_iter     = bli_min( *n_elem_max, bli_min( m, n ) );
			}
			else
			{
				*ij0        = 0;
				*n_shift    = diagoff_eff_;
				*n_iter     = bli_min( n_iter_max_, *n_shift + bli_min( m, n ) );
			}
		}
	}
}

// Set dimensions, increments, etc for ONE matrix argument when operating
// on the diagonal.

BLIS_INLINE void bli_set_dims_incs_1d
     (
       doff_t diagoffx,
       dim_t  m,    dim_t  n,      inc_t  rs_x, inc_t  cs_x,
       dim_t* offx, dim_t* n_elem, inc_t* incx
     )
{
	if ( diagoffx < 0 )
	{
		*n_elem = bli_min( m - ( dim_t )(-diagoffx), n );
		*offx   = ( dim_t )(-diagoffx) * rs_x;
	}
	else
	{
		*n_elem = bli_min( n - ( dim_t )( diagoffx), m );
		*offx   = ( dim_t )( diagoffx) * cs_x;
	}

	*incx = rs_x + cs_x; \
}

// Set dimensions, increments, etc for TWO matrix arguments when operating
// on diagonals.
BLIS_INLINE void bli_set_dims_incs_2d
     (
       doff_t diagoffx, trans_t transx,
       dim_t  m, dim_t  n, inc_t  rs_x, inc_t  cs_x,
                           inc_t  rs_y, inc_t  cs_y,
       dim_t* offx, dim_t* offy, dim_t* n_elem,
       inc_t* incx, inc_t* incy
     )
{
	doff_t diagoffy_ = bli_diag_offset_with_trans( transx, diagoffx );

	if ( diagoffx < 0 ) *offx = -diagoffx * rs_x;
	else                *offx =  diagoffx * cs_x;

	if ( diagoffy_ < 0 )
	{
		*n_elem = bli_min( m - ( dim_t )(-diagoffy_), n );
		*offy   = -diagoffy_ * rs_y;
	}
	else
	{
		*n_elem = bli_min( n - ( dim_t )( diagoffy_), m );
		*offy   = diagoffy_ * cs_y;
	}

	*incx = rs_x + cs_x;
	*incy = rs_y + cs_y;
}


#endif
// end bli_param_macro_defs.h
#line 103 "./frame/include//bli_macro_defs.h"

// begin bli_obj_macro_defs.h
#line 1 "./frame/include//bli_obj_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2016, Hewlett Packard Enterprise Development LP
   Copyright (C) 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_OBJ_MACRO_DEFS_H
#define BLIS_OBJ_MACRO_DEFS_H


// -- Object query/modification macros --

// Info query

BLIS_INLINE num_t bli_obj_dt( const obj_t* obj )
{
	return ( num_t )
	       ( obj->info & BLIS_DATATYPE_BITS );
}

BLIS_INLINE bool bli_obj_is_float( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_dt( obj ) == BLIS_BITVAL_FLOAT_TYPE );
}

BLIS_INLINE bool bli_obj_is_double( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_dt( obj ) == BLIS_BITVAL_DOUBLE_TYPE );
}

BLIS_INLINE bool bli_obj_is_scomplex( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_dt( obj ) == BLIS_BITVAL_SCOMPLEX_TYPE );
}

BLIS_INLINE bool bli_obj_is_dcomplex( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_dt( obj ) == BLIS_BITVAL_DCOMPLEX_TYPE );
}

BLIS_INLINE bool bli_obj_is_int( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_dt( obj ) == BLIS_BITVAL_INT_TYPE );
}

BLIS_INLINE bool bli_obj_is_const( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_dt( obj ) == BLIS_BITVAL_CONST_TYPE );
}

BLIS_INLINE dom_t bli_obj_domain( const obj_t* obj )
{
	return ( dom_t )
	       ( obj->info & BLIS_DOMAIN_BIT );
}

BLIS_INLINE prec_t bli_obj_prec( const obj_t* obj )
{
	return ( prec_t )
	       ( obj->info & BLIS_PRECISION_BIT );
}

BLIS_INLINE bool bli_obj_is_single_prec( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_prec( obj ) == BLIS_BITVAL_SINGLE_PREC );
}

BLIS_INLINE bool bli_obj_is_double_prec( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_prec( obj ) == BLIS_BITVAL_DOUBLE_PREC );
}

BLIS_INLINE num_t bli_obj_dt_proj_to_single_prec( const obj_t* obj )
{
	return ( num_t )
	       ( bli_obj_dt( obj ) & ~BLIS_BITVAL_SINGLE_PREC );
}

BLIS_INLINE num_t bli_obj_dt_proj_to_double_prec( const obj_t* obj )
{
	return ( num_t )
	       ( bli_obj_dt( obj ) | BLIS_BITVAL_DOUBLE_PREC );
}

BLIS_INLINE bool bli_obj_is_real( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_domain( obj ) == BLIS_BITVAL_REAL &&
	         !bli_obj_is_const( obj ) );
}

BLIS_INLINE bool bli_obj_is_complex( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_domain( obj ) == BLIS_BITVAL_COMPLEX &&
	         !bli_obj_is_const( obj ) );
}

BLIS_INLINE num_t bli_obj_dt_proj_to_real( const obj_t* obj )
{
	return ( num_t )
	       ( bli_obj_dt( obj ) & ~BLIS_BITVAL_COMPLEX );
}

BLIS_INLINE num_t bli_obj_dt_proj_to_complex( const obj_t* obj )
{
	return ( num_t )
	       ( bli_obj_dt( obj ) | BLIS_BITVAL_COMPLEX );
}

BLIS_INLINE num_t bli_obj_target_dt( const obj_t* obj )
{
	return ( num_t )
	       ( ( obj->info & BLIS_TARGET_DT_BITS ) >> BLIS_TARGET_DT_SHIFT );
}

BLIS_INLINE dom_t bli_obj_target_domain( const obj_t* obj )
{
	return ( dom_t )
	       ( ( obj->info & BLIS_TARGET_DOMAIN_BIT ) >> BLIS_TARGET_DT_SHIFT );
}

BLIS_INLINE prec_t bli_obj_target_prec( const obj_t* obj )
{
	return ( prec_t )
	       ( ( obj->info & BLIS_TARGET_PREC_BIT ) >> BLIS_TARGET_DT_SHIFT );
}

BLIS_INLINE num_t bli_obj_exec_dt( const obj_t* obj )
{
	return ( num_t )
	       ( ( obj->info & BLIS_EXEC_DT_BITS ) >> BLIS_EXEC_DT_SHIFT );
}

BLIS_INLINE dom_t bli_obj_exec_domain( const obj_t* obj )
{
	return ( dom_t )
	       ( ( obj->info & BLIS_EXEC_DOMAIN_BIT ) >> BLIS_EXEC_DT_SHIFT );
}

BLIS_INLINE prec_t bli_obj_exec_prec( const obj_t* obj )
{
	return ( prec_t )
	       ( ( obj->info & BLIS_EXEC_PREC_BIT ) >> BLIS_EXEC_DT_SHIFT );
}

BLIS_INLINE num_t bli_obj_comp_dt( const obj_t* obj )
{
	return ( num_t )
	       ( ( obj->info & BLIS_COMP_DT_BITS ) >> BLIS_COMP_DT_SHIFT );
}

BLIS_INLINE dom_t bli_obj_comp_domain( const obj_t* obj )
{
	return ( dom_t )
	       ( ( obj->info & BLIS_COMP_DOMAIN_BIT ) >> BLIS_COMP_DT_SHIFT );
}

BLIS_INLINE prec_t bli_obj_comp_prec( const obj_t* obj )
{
	return ( prec_t )
	       ( ( obj->info & BLIS_COMP_PREC_BIT ) >> BLIS_COMP_DT_SHIFT );
}

// NOTE: This function queries info2.
BLIS_INLINE num_t bli_obj_scalar_dt( const obj_t* obj )
{
	return ( num_t )
	       ( ( obj->info2 & BLIS_SCALAR_DT_BITS ) >> BLIS_SCALAR_DT_SHIFT );
}

// NOTE: This function queries info2.
BLIS_INLINE dom_t bli_obj_scalar_domain( const obj_t* obj )
{
	return ( dom_t )
	       ( ( obj->info2 & BLIS_SCALAR_DOMAIN_BIT ) >> BLIS_SCALAR_DT_SHIFT );
}

// NOTE: This function queries info2.
BLIS_INLINE prec_t bli_obj_scalar_prec( const obj_t* obj )
{
	return ( prec_t )
	       ( ( obj->info2 & BLIS_SCALAR_PREC_BIT ) >> BLIS_SCALAR_DT_SHIFT );
}

BLIS_INLINE trans_t bli_obj_conjtrans_status( const obj_t* obj )
{
	return ( trans_t )
	       ( obj->info & BLIS_CONJTRANS_BITS );
}

BLIS_INLINE trans_t bli_obj_onlytrans_status( const obj_t* obj )
{
	return ( trans_t )
	       ( obj->info & BLIS_TRANS_BIT );
}

BLIS_INLINE bool bli_obj_has_trans( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_onlytrans_status( obj ) == BLIS_BITVAL_TRANS );
}

BLIS_INLINE bool bli_obj_has_notrans( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_onlytrans_status( obj ) == BLIS_BITVAL_NO_TRANS );
}

BLIS_INLINE conj_t bli_obj_conj_status( const obj_t* obj )
{
	return ( conj_t )
	       ( obj->info & BLIS_CONJ_BIT );
}

BLIS_INLINE bool bli_obj_has_conj( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_conj_status( obj ) == BLIS_BITVAL_CONJ );
}

BLIS_INLINE bool bli_obj_has_noconj( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_conj_status( obj ) == BLIS_BITVAL_NO_CONJ );
}

BLIS_INLINE uplo_t bli_obj_uplo( const obj_t* obj )
{
	return ( uplo_t )
	       ( obj->info & BLIS_UPLO_BITS );
}

BLIS_INLINE bool bli_obj_is_upper( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_uplo( obj ) == BLIS_BITVAL_UPPER );
}

BLIS_INLINE bool bli_obj_is_lower( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_uplo( obj ) == BLIS_BITVAL_LOWER );
}

BLIS_INLINE bool bli_obj_is_upper_or_lower( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_is_upper( obj ) ||
	         bli_obj_is_lower( obj ) );
}

BLIS_INLINE bool bli_obj_is_dense( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_uplo( obj ) == BLIS_BITVAL_DENSE );
}

BLIS_INLINE bool bli_obj_is_zeros( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_uplo( obj ) == BLIS_BITVAL_ZEROS );
}

BLIS_INLINE diag_t bli_obj_diag( const obj_t* obj )
{
	return ( diag_t )
	       ( obj->info & BLIS_UNIT_DIAG_BIT );
}

BLIS_INLINE bool bli_obj_has_nonunit_diag( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_diag( obj ) == BLIS_BITVAL_NONUNIT_DIAG );
}

BLIS_INLINE bool bli_obj_has_unit_diag( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_diag( obj ) == BLIS_BITVAL_UNIT_DIAG );
}

BLIS_INLINE bool bli_obj_has_inverted_diag( const obj_t* obj )
{
	return ( bool )
	       ( ( obj->info & BLIS_INVERT_DIAG_BIT ) == BLIS_BITVAL_INVERT_DIAG );
}

BLIS_INLINE bool bli_obj_is_pack_rev_if_upper( const obj_t* obj )
{
	return ( bool )
	       ( ( obj->info & BLIS_PACK_REV_IF_UPPER_BIT ) == BLIS_BITVAL_PACK_REV_IF_UPPER );
}

BLIS_INLINE bool bli_obj_is_pack_rev_if_lower( const obj_t* obj )
{
	return ( bool )
	       ( ( obj->info & BLIS_PACK_REV_IF_LOWER_BIT ) == BLIS_BITVAL_PACK_REV_IF_LOWER );
}

BLIS_INLINE pack_t bli_obj_pack_schema( const obj_t* obj )
{
	return ( pack_t )
	       ( obj->info & BLIS_PACK_SCHEMA_BITS );
}

BLIS_INLINE bool bli_obj_is_packed( const obj_t* obj )
{
	return ( bool )
	       ( obj->info & BLIS_PACK_BIT );
}

BLIS_INLINE bool bli_obj_is_row_packed( const obj_t* obj )
{
	return ( bool )
	       ( ( obj->info & BLIS_PACK_RC_BIT ) == ( BLIS_BITVAL_PACKED_UNSPEC ^
	                                               BLIS_BITVAL_PACKED_ROWS    ) );
}

BLIS_INLINE bool bli_obj_is_col_packed( const obj_t* obj )
{
	return ( bool )
	       ( ( obj->info & BLIS_PACK_RC_BIT ) == ( BLIS_BITVAL_PACKED_UNSPEC ^
	                                               BLIS_BITVAL_PACKED_COLUMNS ) );
}

BLIS_INLINE bool bli_obj_is_panel_packed( const obj_t* obj )
{
	return ( bool )
	       ( obj->info & BLIS_PACK_PANEL_BIT );
}

BLIS_INLINE packbuf_t bli_obj_pack_buffer_type( const obj_t* obj )
{
	return ( packbuf_t )
	       ( obj->info & BLIS_PACK_BUFFER_BITS );
}

BLIS_INLINE struc_t bli_obj_struc( const obj_t* obj )
{
	return ( struc_t )
	       ( obj->info & BLIS_STRUC_BITS );
}

BLIS_INLINE bool bli_obj_is_general( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_struc( obj ) == BLIS_BITVAL_GENERAL );
}

BLIS_INLINE bool bli_obj_is_hermitian( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_struc( obj ) == BLIS_BITVAL_HERMITIAN );
}

BLIS_INLINE bool bli_obj_is_symmetric( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_struc( obj ) == BLIS_BITVAL_SYMMETRIC );
}

BLIS_INLINE bool bli_obj_is_triangular( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_struc( obj ) == BLIS_BITVAL_TRIANGULAR );
}

// Info modification

BLIS_INLINE void bli_obj_apply_trans( trans_t trans, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( obj->info ^ trans );
}

BLIS_INLINE void bli_obj_apply_conj( conj_t conj, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( obj->info ^ conj );
}

BLIS_INLINE void bli_obj_set_conjtrans( trans_t trans, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_CONJTRANS_BITS ) | trans );
}

BLIS_INLINE void bli_obj_set_onlytrans( trans_t trans, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_TRANS_BIT ) | trans );
}

BLIS_INLINE void bli_obj_set_conj( conj_t conj, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_CONJ_BIT ) | conj );
}

BLIS_INLINE void bli_obj_set_uplo( uplo_t uplo, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_UPLO_BITS ) | uplo );
}

BLIS_INLINE void bli_obj_set_diag( diag_t diag, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_UNIT_DIAG_BIT ) | diag );
}

BLIS_INLINE void bli_obj_set_invert_diag( invdiag_t invdiag, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_INVERT_DIAG_BIT ) | invdiag );
}

BLIS_INLINE void bli_obj_set_dt( num_t dt, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_DATATYPE_BITS ) | dt );
}

BLIS_INLINE void bli_obj_set_target_dt( num_t dt, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_TARGET_DT_BITS ) |
	              ( dt << BLIS_TARGET_DT_SHIFT ) );
}

BLIS_INLINE void bli_obj_set_target_domain( dom_t dt, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_TARGET_DOMAIN_BIT ) |
	              ( dt << BLIS_TARGET_DT_SHIFT ) );
}

BLIS_INLINE void bli_obj_set_target_prec( prec_t dt, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_TARGET_PREC_BIT ) |
	              ( dt << BLIS_TARGET_DT_SHIFT ) );
}

BLIS_INLINE void bli_obj_set_exec_dt( num_t dt, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_EXEC_DT_BITS ) |
	              ( dt << BLIS_EXEC_DT_SHIFT ) );
}

BLIS_INLINE void bli_obj_set_exec_domain( dom_t dt, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_EXEC_DOMAIN_BIT ) |
	              ( dt << BLIS_EXEC_DT_SHIFT ) );
}

BLIS_INLINE void bli_obj_set_exec_prec( prec_t dt, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_EXEC_PREC_BIT ) |
	              ( dt << BLIS_EXEC_DT_SHIFT ) );
}

BLIS_INLINE void bli_obj_set_comp_dt( num_t dt, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_COMP_DT_BITS ) |
	              ( dt << BLIS_COMP_DT_SHIFT ) );
}

BLIS_INLINE void bli_obj_set_comp_domain( dom_t dt, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_COMP_DOMAIN_BIT ) |
	              ( dt << BLIS_COMP_DT_SHIFT ) );
}

BLIS_INLINE void bli_obj_set_comp_prec( prec_t dt, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_COMP_PREC_BIT ) |
	              ( dt << BLIS_COMP_DT_SHIFT ) );
}

// NOTE: This function queries and modifies info2.
BLIS_INLINE void bli_obj_set_scalar_dt( num_t dt, obj_t* obj )
{
	obj->info2 = ( objbits_t )
	             ( ( obj->info2 & ~BLIS_SCALAR_DT_BITS ) |
	               ( dt << BLIS_SCALAR_DT_SHIFT ) );
}

// NOTE: This function queries and modifies info2.
BLIS_INLINE void bli_obj_set_scalar_domain( dom_t dt, obj_t* obj )
{
	obj->info2 = ( objbits_t )
	             ( ( obj->info2 & ~BLIS_SCALAR_DOMAIN_BIT ) |
	               ( dt << BLIS_SCALAR_DT_SHIFT ) );
}

// NOTE: This function queries and modifies info2.
BLIS_INLINE void bli_obj_set_scalar_prec( prec_t dt, obj_t* obj )
{
	obj->info2 = ( objbits_t )
	             ( ( obj->info2 & ~BLIS_SCALAR_PREC_BIT ) |
	               ( dt << BLIS_SCALAR_DT_SHIFT ) );
}

BLIS_INLINE void bli_obj_set_pack_schema( pack_t schema, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_PACK_SCHEMA_BITS ) | schema );
}

BLIS_INLINE void bli_obj_set_pack_order_if_upper( packord_t ordif, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_PACK_REV_IF_UPPER_BIT ) | ordif );
}

BLIS_INLINE void bli_obj_set_pack_order_if_lower( packord_t ordif, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_PACK_REV_IF_LOWER_BIT ) | ordif );
}

// NOTE: The packbuf_t bitfield in the obj_t is currently unused. Instead,
// packbuf_t is stored/used from the context in order to support various
// induced methods. (Though ideally the packbuf_t field would only be
// present in the control tree).
BLIS_INLINE void bli_obj_set_pack_buffer_type( packbuf_t buf_type, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_PACK_BUFFER_BITS ) | buf_type );
}

BLIS_INLINE void bli_obj_set_struc( struc_t struc, obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( ( obj->info & ~BLIS_STRUC_BITS ) | struc );
}

BLIS_INLINE void bli_obj_toggle_trans( obj_t* obj )
{
	bli_obj_apply_trans( BLIS_TRANSPOSE, obj );
}

BLIS_INLINE void bli_obj_toggle_conj( obj_t* obj )
{
	bli_obj_apply_conj( BLIS_CONJUGATE, obj );
}

BLIS_INLINE void bli_obj_toggle_uplo( obj_t* obj )
{
	obj->info = ( objbits_t )
	            ( obj->info ^ BLIS_LOWER_BIT ) ^ BLIS_UPPER_BIT;
}

// Root matrix query

BLIS_INLINE obj_t* bli_obj_root( const obj_t* obj )
{
	return ( obj_t* )( obj->root );
}

BLIS_INLINE bool bli_obj_root_is_general( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_is_general( bli_obj_root( obj ) ) );
}

BLIS_INLINE bool bli_obj_root_is_hermitian( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_is_hermitian( bli_obj_root( obj ) ) );
}

BLIS_INLINE bool bli_obj_root_is_symmetric( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_is_symmetric( bli_obj_root( obj ) ) );
}

BLIS_INLINE bool bli_obj_root_is_triangular( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_is_triangular( bli_obj_root( obj ) ) );
}

BLIS_INLINE bool bli_obj_root_is_herm_or_symm( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_is_hermitian( bli_obj_root( obj ) ) ||
	         bli_obj_is_symmetric( bli_obj_root( obj ) ) );
}

BLIS_INLINE bool bli_obj_root_is_upper( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_is_upper( bli_obj_root( obj ) ) );
}

BLIS_INLINE bool bli_obj_root_is_lower( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_is_lower( bli_obj_root( obj ) ) );
}

// Root matrix modification

BLIS_INLINE void bli_obj_set_as_root( obj_t* obj )
{
	obj->root = obj;
}

// Diagonal offset query

BLIS_INLINE doff_t bli_obj_diag_offset( const obj_t* obj )
{
	return ( doff_t )
	       ( obj->diag_off );
}

BLIS_INLINE doff_t bli_obj_diag_offset_after_trans( const obj_t* obj )
{
	return ( doff_t )
	       ( bli_obj_has_trans( obj ) ? -bli_obj_diag_offset( obj )
	                                  :  bli_obj_diag_offset( obj ) );
}

// Diagonal offset modification

BLIS_INLINE void bli_obj_set_diag_offset( doff_t offset, obj_t* obj )
{
	obj->diag_off = ( doff_t )offset;
}

BLIS_INLINE void bli_obj_negate_diag_offset( obj_t* obj )
{
	obj->diag_off = -(obj->diag_off);
}

BLIS_INLINE void bli_obj_inc_diag_offset( doff_t offset, obj_t* obj )
{
	obj->diag_off += ( doff_t )offset;
}

// Dimension query

BLIS_INLINE dim_t bli_obj_length( const obj_t* obj )
{
	return ( obj->dim[ BLIS_M ] );
}

BLIS_INLINE dim_t bli_obj_width( const obj_t* obj )
{
	return ( obj->dim[ BLIS_N ] );
}

BLIS_INLINE dim_t bli_obj_dim( mdim_t mdim, const obj_t* obj )
{
	return ( obj->dim[ mdim ] );
}

BLIS_INLINE dim_t bli_obj_min_dim( const obj_t* obj )
{
	return bli_min( bli_obj_length( obj ),
	                bli_obj_width( obj ) );
}

BLIS_INLINE dim_t bli_obj_max_dim( const obj_t* obj )
{
	return bli_max( bli_obj_length( obj ),
	                bli_obj_width( obj ) );
}

BLIS_INLINE dim_t bli_obj_length_after_trans( const obj_t* obj )
{
	return ( bli_obj_has_trans( obj ) ? bli_obj_width( obj )
	                                  : bli_obj_length( obj ) );
}

BLIS_INLINE dim_t bli_obj_width_after_trans( const obj_t* obj )
{
	return ( bli_obj_has_trans( obj ) ? bli_obj_length( obj )
	                                  : bli_obj_width( obj ) );
}

BLIS_INLINE bool bli_obj_is_1x1( const obj_t* x )
{
	return ( bool )
	       ( bli_obj_length( x ) == 1 &&
	         bli_obj_width( x ) == 1 );
}

// Stride/increment query

BLIS_INLINE inc_t bli_obj_row_stride( const obj_t* obj )
{
	return ( obj->rs );
}

BLIS_INLINE inc_t bli_obj_col_stride( const obj_t* obj )
{
	return ( obj->cs );
}

BLIS_INLINE inc_t bli_obj_imag_stride( const obj_t* obj )
{
	return ( obj->is );
}

BLIS_INLINE inc_t bli_obj_row_stride_mag( const obj_t* obj )
{
	return ( inc_t )
	       ( bli_abs( obj->rs ) );
}

BLIS_INLINE inc_t bli_obj_col_stride_mag( const obj_t* obj )
{
	return ( inc_t )
	       ( bli_abs( obj->cs ) );
}

BLIS_INLINE inc_t bli_obj_imag_stride_mag( const obj_t* obj )
{
	return ( inc_t )
	       ( bli_abs( obj->is ) );
}

// Note: The purpose of these functions is to obtain the length and width
// of the smallest submatrices of an object that could still encompass
// the stored data above (if obj is upper) or below (if obj is lower)
// the diagonal.
BLIS_INLINE dim_t bli_obj_length_stored( const obj_t* obj )
{
	return ( dim_t )
	       ( bli_obj_is_upper( obj )
	         ? bli_min( bli_obj_length( obj ),
	                    bli_obj_width( obj )  - bli_obj_diag_offset( obj ) )
	         : bli_min( bli_obj_length( obj ),
	                    bli_obj_length( obj ) + bli_obj_diag_offset( obj ) )
	       );
}

BLIS_INLINE dim_t bli_obj_width_stored( const obj_t* obj )
{
	return ( dim_t )
	       ( bli_obj_is_lower( obj )
	         ? bli_min( bli_obj_width( obj ),
	                    bli_obj_length( obj ) + bli_obj_diag_offset( obj ) )
	         : bli_min( bli_obj_width( obj ),
	                    bli_obj_width( obj )  - bli_obj_diag_offset( obj ) )
	       );
}

BLIS_INLINE dim_t bli_obj_length_stored_after_trans( const obj_t* obj )
{
	return ( bli_obj_has_trans( obj ) ? bli_obj_width_stored( obj )
	                                  : bli_obj_length_stored( obj ) );
}

BLIS_INLINE dim_t bli_obj_width_stored_after_trans( const obj_t* obj )
{
	return ( bli_obj_has_trans( obj ) ? bli_obj_length_stored( obj )
	                                  : bli_obj_width_stored( obj ) );
}

BLIS_INLINE dim_t bli_obj_vector_dim( const obj_t* x )
{
	return ( bli_obj_length( x ) == 1 ? bli_obj_width( x )
	                                  : bli_obj_length( x ) );
}

BLIS_INLINE inc_t bli_obj_vector_inc( const obj_t* x )
{
	return ( bli_obj_is_1x1( x ) ? 1 :
	         ( bli_obj_length( x ) == 1 ? bli_obj_col_stride( x )
	                                    : bli_obj_row_stride( x ) )
	       );
}

BLIS_INLINE bool bli_obj_is_vector( const obj_t* x )
{
	return ( bool )
	       ( bli_obj_length( x ) == 1 ||
	         bli_obj_width(  x ) == 1 );
}

BLIS_INLINE bool bli_obj_is_row_vector( const obj_t* x )
{
	return ( bool )
	       ( bli_obj_length( x ) == 1 );
}

BLIS_INLINE bool bli_obj_is_col_vector( const obj_t* x )
{
	return ( bool )
	       ( bli_obj_width( x ) == 1 );
}

BLIS_INLINE bool bli_obj_has_zero_dim( const obj_t* x )
{
	return ( bool )
	       ( bli_obj_length( x ) == 0 ||
	         bli_obj_width(  x ) == 0 );
}

// Dimension modification

BLIS_INLINE void bli_obj_set_length( dim_t m, obj_t* obj )
{
	obj->dim[ BLIS_M ] = m;
}

BLIS_INLINE void bli_obj_set_width( dim_t n, obj_t* obj )
{
	obj->dim[ BLIS_N ] = n;
}

BLIS_INLINE void bli_obj_set_dim( mdim_t mdim, dim_t dim_val, obj_t* obj )
{
	obj->dim[ mdim ] = dim_val;
}

BLIS_INLINE void bli_obj_set_dims( dim_t m, dim_t n, obj_t* obj )
{
	bli_obj_set_length( m, obj );
	bli_obj_set_width( n, obj );
}

BLIS_INLINE void bli_obj_set_dims_with_trans( trans_t trans, dim_t m, dim_t n, obj_t* obj )
{
	if ( bli_does_notrans( trans ) )
	{
		bli_obj_set_length( m, obj );
		bli_obj_set_width( n, obj );
	}
	else // if ( bli_does_trans( trans ) )
	{
		bli_obj_set_length( n, obj );
		bli_obj_set_width( m, obj );
	}
}

// Stride/increment predicates

//
// NOTE: The following two macros differ from their non-obj counterparts
// in that they do not identify m x 1 and 1 x n objects as row-stored and
// column-stored, respectively, which is needed when considering packed
// objects. But this is okay, since none of the invocations of these
// "obj" macros are used on packed matrices.
//

BLIS_INLINE bool bli_obj_is_row_stored( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_col_stride_mag( obj ) == 1 );
}

BLIS_INLINE bool bli_obj_is_col_stored( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_row_stride_mag( obj ) == 1 );
}

BLIS_INLINE bool bli_obj_is_gen_stored( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_row_stride_mag( obj ) != 1 &&
	         bli_obj_col_stride_mag( obj ) != 1 );
}

BLIS_INLINE bool bli_obj_is_row_tilted( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_col_stride_mag( obj ) < bli_obj_row_stride_mag( obj ) );
}

BLIS_INLINE bool bli_obj_is_col_tilted( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_row_stride_mag( obj ) < bli_obj_col_stride_mag( obj ) );
}

// Stride/increment modification

BLIS_INLINE void bli_obj_set_row_stride( inc_t rs, obj_t* obj )
{
	obj->rs = rs;
}

BLIS_INLINE void bli_obj_set_col_stride( inc_t cs, obj_t* obj )
{
	obj->cs = cs;
}

BLIS_INLINE void bli_obj_set_strides( inc_t rs, inc_t cs, obj_t* obj )
{
	bli_obj_set_row_stride( rs, obj );
	bli_obj_set_col_stride( cs, obj );
}

BLIS_INLINE void bli_obj_set_imag_stride( inc_t is, obj_t* obj )
{
	obj->is = is;
}

// Offset query

BLIS_INLINE dim_t bli_obj_row_off( const obj_t* obj )
{
	return ( obj->off[ BLIS_M ] );
}

BLIS_INLINE dim_t bli_obj_col_off( const obj_t* obj )
{
	return ( obj->off[ BLIS_N ] );
}

BLIS_INLINE dim_t bli_obj_off( mdim_t mdim, const obj_t* obj )
{
	return ( obj->off[ mdim ] );
}

// Offset modification

BLIS_INLINE void bli_obj_set_off( mdim_t mdim, dim_t offset, obj_t* obj )
{
	obj->off[ mdim ] = offset;
}

BLIS_INLINE void bli_obj_set_offs( dim_t offm, dim_t offn, obj_t* obj )
{
	bli_obj_set_off( BLIS_M, offm, obj );
	bli_obj_set_off( BLIS_N, offn, obj );
}

BLIS_INLINE void bli_obj_inc_off( mdim_t mdim, dim_t offset, obj_t* obj )
{
	obj->off[ mdim ] += offset;
}

BLIS_INLINE void bli_obj_inc_offs( dim_t offm, dim_t offn, obj_t* obj )
{
	bli_obj_inc_off( BLIS_M, offm, obj );
	bli_obj_inc_off( BLIS_N, offn, obj );
}

// Diagonal offset predicates

BLIS_INLINE bool bli_obj_is_strictly_above_diag( const obj_t* obj )
{
	return ( bool )
	       ( ( doff_t )bli_obj_length( obj ) <= -bli_obj_diag_offset( obj ) );
}

BLIS_INLINE bool bli_obj_is_strictly_below_diag( const obj_t* obj )
{
	return ( bool )
	       ( ( doff_t )bli_obj_width( obj ) <= bli_obj_diag_offset( obj ) );
}

BLIS_INLINE bool bli_obj_is_outside_diag( const obj_t* obj )
{
	return ( bool )
	       ( bli_obj_is_strictly_above_diag( obj ) ||
	         bli_obj_is_strictly_below_diag( obj ) );
}

BLIS_INLINE bool bli_obj_intersects_diag( const obj_t* obj )
{
	return ( bool )
	       ( !bli_obj_is_strictly_above_diag( obj ) &&
	         !bli_obj_is_strictly_below_diag( obj ) );
}

BLIS_INLINE bool bli_obj_is_unstored_subpart( const obj_t* obj )
{
	return ( bool )
	       ( ( bli_obj_root_is_lower( obj ) && bli_obj_is_strictly_above_diag( obj ) ) ||
	         ( bli_obj_root_is_upper( obj ) && bli_obj_is_strictly_below_diag( obj ) ) );
}

// Buffer address query

BLIS_INLINE void* bli_obj_buffer( const obj_t* obj )
{
	return ( void* )
	       ( obj->buffer );
}

// Buffer address modification

BLIS_INLINE void bli_obj_set_buffer( void* p, obj_t* obj )
{
	obj->buffer = p;
}

// Bufferless scalar field query

BLIS_INLINE void* bli_obj_internal_scalar_buffer( const obj_t* obj )
{
	return ( void* )
	       ( &( obj->scalar ) );
}

// Bufferless scalar field modification

BLIS_INLINE void bli_obj_copy_internal_scalar( const obj_t* a, obj_t* b )
{
	b->scalar = a->scalar;
}

// Element size query

BLIS_INLINE siz_t bli_obj_elem_size( const obj_t* obj )
{
	return ( siz_t )
	       ( obj->elem_size );
}

// Element size modification

BLIS_INLINE void bli_obj_set_elem_size( siz_t size, obj_t* obj )
{
	obj->elem_size = size;
}

// Packed matrix info query

BLIS_INLINE dim_t bli_obj_padded_length( const obj_t* obj )
{
	return ( obj->m_padded );
}

BLIS_INLINE dim_t bli_obj_padded_width( const obj_t* obj )
{
	return ( obj->n_padded );
}

// Packed matrix info modification

BLIS_INLINE void bli_obj_set_padded_length( dim_t m, obj_t* obj )
{
	obj->m_padded = m;
}

BLIS_INLINE void bli_obj_set_padded_width( dim_t n, obj_t* obj )
{
	obj->n_padded = n;
}

BLIS_INLINE void bli_obj_set_padded_dims( dim_t m, dim_t n, obj_t* obj )
{
	bli_obj_set_padded_length( m, obj );
	bli_obj_set_padded_width( n, obj );
}

// Packed panel info query

BLIS_INLINE dim_t bli_obj_panel_length( const obj_t* obj )
{
	return ( obj->m_panel );
}

BLIS_INLINE dim_t bli_obj_panel_width( const obj_t* obj )
{
	return ( obj->n_panel );
}

BLIS_INLINE inc_t bli_obj_panel_dim( const obj_t* obj )
{
	return ( obj->pd );
}

BLIS_INLINE inc_t bli_obj_panel_stride( const obj_t* obj )
{
	return ( obj->ps );
}

// Packed panel info modification

BLIS_INLINE void bli_obj_set_panel_length( dim_t m, obj_t* obj )
{
	obj->m_panel = m;
}

BLIS_INLINE void bli_obj_set_panel_width( dim_t n, obj_t* obj )
{
	obj->n_panel = n;
}

BLIS_INLINE void bli_obj_set_panel_dims( dim_t m, dim_t n, obj_t* obj )
{
	bli_obj_set_panel_length( m, obj );
	bli_obj_set_panel_width( n, obj );
}

BLIS_INLINE void bli_obj_set_panel_dim( inc_t pd, obj_t* obj )
{
	obj->pd = pd;
}

BLIS_INLINE void bli_obj_set_panel_stride( inc_t ps, obj_t* obj )
{
	obj->ps = ps;
}

// stor3_t-related

BLIS_INLINE stor3_t bli_obj_stor3_from_strides( const obj_t* c, const obj_t* a, const obj_t* b )
{
	const inc_t rs_c = bli_obj_row_stride( c );
	const inc_t cs_c = bli_obj_col_stride( c );

	inc_t rs_a, cs_a;
	inc_t rs_b, cs_b;

	if ( bli_obj_has_notrans( a ) )
	{
		rs_a = bli_obj_row_stride( a );
		cs_a = bli_obj_col_stride( a );
	}
	else
	{
		rs_a = bli_obj_col_stride( a );
		cs_a = bli_obj_row_stride( a );
	}

	if ( bli_obj_has_notrans( b ) )
	{
		rs_b = bli_obj_row_stride( b );
		cs_b = bli_obj_col_stride( b );
	}
	else
	{
		rs_b = bli_obj_col_stride( b );
		cs_b = bli_obj_row_stride( b );
	}

	return bli_stor3_from_strides( rs_c, cs_c,
	                               rs_a, cs_a,
	                               rs_b, cs_b  );
}


// -- User-provided information macros --

// Function pointer query

BLIS_INLINE obj_pack_fn_t bli_obj_pack_fn( const obj_t* obj )
{
	return obj->pack_fn;
}

BLIS_INLINE void* bli_obj_pack_params( const obj_t* obj )
{
	return obj->pack_params;
}

BLIS_INLINE obj_ker_fn_t bli_obj_ker_fn( const obj_t* obj )
{
	return obj->ker_fn;
}

BLIS_INLINE void* bli_obj_ker_params( const obj_t* obj )
{
	return obj->ker_params;
}

// Function pointer modification

BLIS_INLINE void bli_obj_set_pack_fn( obj_pack_fn_t pack_fn, obj_t* obj )
{
	obj->pack_fn = pack_fn;
}

BLIS_INLINE void bli_obj_set_pack_params( void* params, obj_t* obj )
{
	obj->pack_params = params;
}

BLIS_INLINE void bli_obj_set_ker_fn( obj_ker_fn_t ker_fn, obj_t* obj )
{
	obj->ker_fn = ker_fn;
}

BLIS_INLINE void bli_obj_set_ker_params( void* params, obj_t* obj )
{
	obj->ker_params = params;
}


// -- Initialization-related macros --

// Finish the initialization started by the matrix-specific static initializer
// (e.g. BLIS_OBJECT_INITIALIZER)
// NOTE: This is intended only for use in the BLAS compatibility API and typed
// BLIS API.

BLIS_INLINE void bli_obj_init_finish( num_t dt, dim_t m, dim_t n, void* p, inc_t rs, inc_t cs, obj_t* obj )
{
	bli_obj_set_as_root( obj );

	bli_obj_set_dt( dt, obj );
	bli_obj_set_target_dt( dt, obj );
	bli_obj_set_exec_dt( dt, obj );
	bli_obj_set_comp_dt( dt, obj );

	bli_obj_set_dims( m, n, obj );
	bli_obj_set_strides( rs, cs, obj );

	siz_t elem_size = sizeof( float );
	if ( bli_dt_prec_is_double( dt ) ) elem_size *= 2;
	if ( bli_dt_dom_is_complex( dt ) ) elem_size *= 2;
	bli_obj_set_elem_size( elem_size, obj );

	bli_obj_set_buffer( p, obj );

	bli_obj_set_scalar_dt( dt, obj );
	void* s = bli_obj_internal_scalar_buffer( obj );

	if      ( bli_dt_prec_is_single( dt ) ) { (( scomplex* )s)->real = 1.0F;
	                                          (( scomplex* )s)->imag = 0.0F; }
	else if ( bli_dt_prec_is_double( dt ) ) { (( dcomplex* )s)->real = 1.0;
	                                          (( dcomplex* )s)->imag = 0.0; }
}

// Finish the initialization started by the 1x1-specific static initializer
// (e.g. BLIS_OBJECT_INITIALIZER_1X1)
// NOTE: This is intended only for use in the BLAS compatibility API and typed
// BLIS API.

BLIS_INLINE void bli_obj_init_finish_1x1( num_t dt, void* p, obj_t* obj )
{
	bli_obj_set_as_root( obj );

	bli_obj_set_dt( dt, obj );

	bli_obj_set_buffer( p, obj );
}

// -- Miscellaneous object macros --

// Toggle the region referenced (or "stored").

BLIS_INLINE void bli_obj_toggle_region_ref( obj_t* obj )
{
	if      ( bli_obj_is_upper( obj ) ) bli_obj_inc_diag_offset( -1, obj );
	else if ( bli_obj_is_lower( obj ) ) bli_obj_inc_diag_offset(  1, obj );

	bli_obj_toggle_uplo( obj );
}

BLIS_INLINE void bli_obj_toggle_uplo_if_trans( trans_t trans, obj_t* obj )
{
	if ( bli_does_trans( trans ) &&
	     bli_obj_is_upper_or_lower( obj ) )
	{
		bli_obj_toggle_uplo( obj );
		bli_obj_negate_diag_offset( obj );
	}
}

// Initialize object with default properties (info field).

BLIS_INLINE void bli_obj_set_defaults( obj_t* obj )
{
	obj->info = 0x0;
	obj->info = obj->info | BLIS_BITVAL_DENSE | BLIS_BITVAL_GENERAL;
}

// Acquire buffer at object's submatrix offset (offset-aware buffer query).

BLIS_INLINE void* bli_obj_buffer_at_off( const obj_t* obj )
{
	return ( void* )
	       (
	         ( ( char* )( bli_obj_buffer   ( obj ) ) +
	           ( dim_t )( bli_obj_elem_size( obj ) ) *
	                      ( bli_obj_col_off( obj ) * bli_obj_col_stride( obj ) +
	                        bli_obj_row_off( obj ) * bli_obj_row_stride( obj )
	                      )
	         )
	       );
}

// Acquire buffer from BLIS_CONSTANT object.

BLIS_INLINE const void* bli_obj_buffer_for_const( num_t dt, const obj_t* obj )
{
	void* p;

	if      ( dt == BLIS_FLOAT    ) p = &((( constdata_t* )bli_obj_buffer( obj ))->s);
	else if ( dt == BLIS_DOUBLE   ) p = &((( constdata_t* )bli_obj_buffer( obj ))->d);
	else if ( dt == BLIS_SCOMPLEX ) p = &((( constdata_t* )bli_obj_buffer( obj ))->c);
	else if ( dt == BLIS_DCOMPLEX ) p = &((( constdata_t* )bli_obj_buffer( obj ))->z);
	else                            p = &((( constdata_t* )bli_obj_buffer( obj ))->i);

	return p;
}

// Acquire buffer from scalar (1x1) object, including BLIS_CONSTANT objects.

BLIS_INLINE void* bli_obj_buffer_for_1x1( num_t dt, const obj_t* obj )
{
	return ( void* )
	       ( bli_obj_is_const( obj ) ? bli_obj_buffer_for_const( dt, obj )
	                                 : bli_obj_buffer_at_off( obj )
	       );
}

// Adjust the pointer based on current offsets, zero the offsets, and then
// set the current object as the root. For obj_t's with at least one non-zero
// offset, this effectively makes the obj_t "forget" that it was ever a view
// into a larger matrix.

BLIS_INLINE void bli_obj_reset_origin( obj_t* obj )
{
	bli_obj_set_buffer( bli_obj_buffer_at_off( obj ), obj );
	bli_obj_set_offs( 0, 0, obj );
	bli_obj_set_as_root( obj );
}

// Make a full alias (shallow copy).

BLIS_INLINE void bli_obj_alias_to( const obj_t* a, obj_t* b )
{
	bli_obj_init_full_shallow_copy_of( a, b );
}

// Check if two objects are aliases of one another.

BLIS_INLINE bool bli_obj_is_alias_of( const obj_t* a, const obj_t* b )
{
	return ( bool )
	       ( bli_obj_buffer( a ) == bli_obj_buffer( b ) );
}


// Create an alias with a trans value applied.
// (Note: trans may include a conj component.)

BLIS_INLINE void bli_obj_alias_with_trans( trans_t trans, const obj_t* a, obj_t* b )
{
	bli_obj_alias_to( a, b );
	bli_obj_apply_trans( trans, b );
}

// Create an alias with a conj value applied.

BLIS_INLINE void bli_obj_alias_with_conj( conj_t conja, const obj_t* a, obj_t* b )
{
	bli_obj_alias_to( a, b );
	bli_obj_apply_conj( conja, b );
}

// Alias only the real part.

BLIS_INLINE void bli_obj_real_part( const obj_t* c, obj_t* r )
{
	bli_obj_alias_to( c, r );

	if ( bli_obj_is_complex( c ) )
	{
		// Change the datatypes.
		const num_t dt_stor_r = bli_dt_proj_to_real( bli_obj_dt( c )        );
		const num_t dt_targ_r = bli_dt_proj_to_real( bli_obj_target_dt( c ) );
		const num_t dt_exec_r = bli_dt_proj_to_real( bli_obj_exec_dt( c )   );
		const num_t dt_comp_r = bli_dt_proj_to_real( bli_obj_comp_dt( c )   );
		bli_obj_set_dt(        dt_stor_r, r );
		bli_obj_set_target_dt( dt_targ_r, r );
		bli_obj_set_exec_dt(   dt_exec_r, r );
		bli_obj_set_comp_dt(   dt_comp_r, r );

		// Don't touch the attached scalar datatype.

		// Update the element size.
		siz_t es_c = bli_obj_elem_size( c );
		bli_obj_set_elem_size( es_c/2, r );

		// Update the strides.
		inc_t rs_c = bli_obj_row_stride( c );
		inc_t cs_c = bli_obj_col_stride( c );
		bli_obj_set_strides( 2*rs_c, 2*cs_c, r );

		// Buffer is left unchanged.
	}
}

// Alias only the imaginary part.

BLIS_INLINE void bli_obj_imag_part( const obj_t* c, obj_t* i )
{
	if ( bli_obj_is_complex( c ) )
	{
		bli_obj_alias_to( c, i );

		// Change the datatype.
		const num_t dt_stor_r = bli_dt_proj_to_real( bli_obj_dt( c )        );
		const num_t dt_targ_r = bli_dt_proj_to_real( bli_obj_target_dt( c ) );
		const num_t dt_exec_r = bli_dt_proj_to_real( bli_obj_exec_dt( c )   );
		const num_t dt_comp_r = bli_dt_proj_to_real( bli_obj_comp_dt( c )   );
		bli_obj_set_dt(        dt_stor_r, i );
		bli_obj_set_target_dt( dt_targ_r, i );
		bli_obj_set_exec_dt(   dt_exec_r, i );
		bli_obj_set_comp_dt(   dt_comp_r, i );

		// Don't touch the attached scalar datatype.

		// Update the element size.
		siz_t es_c = bli_obj_elem_size( c );
		bli_obj_set_elem_size( es_c/2, i );

		// Update the strides.
		inc_t rs_c = bli_obj_row_stride( c );
		inc_t cs_c = bli_obj_col_stride( c );
		bli_obj_set_strides( 2*rs_c, 2*cs_c, i );

		// Update the buffer.
		inc_t is_c = bli_obj_imag_stride( c );
		char* p    = ( char* )bli_obj_buffer_at_off( c );
		bli_obj_set_buffer( p + is_c * es_c/2, i );
	}
}

// Given a 1x1 object, acquire an address to the buffer depending on whether
// the object is a BLIS_CONSTANT, and also set a datatype associated with the
// chosen buffer (possibly using an auxiliary datatype if the object is
// BLIS_CONSTANT).

BLIS_INLINE void bli_obj_scalar_set_dt_buffer( const obj_t* obj, num_t dt_aux, num_t* dt, void** buf )
{
	if ( bli_obj_is_const( obj ) )
	{
		*dt  = dt_aux;
		*buf = bli_obj_buffer_for_1x1( dt_aux, obj );
	}
	else
	{
		*dt  = bli_obj_dt( obj );
		*buf = bli_obj_buffer_at_off( obj );
	}
}

// Swap all object fields (metadata/properties).

BLIS_INLINE void bli_obj_swap( obj_t* a, obj_t* b )
{
	bool a_root_is_self = ( bli_obj_root( a ) == a );
	bool b_root_is_self = ( bli_obj_root( b ) == b );

	obj_t t = *b; *b = *a; *a = t;

	if ( a_root_is_self ) bli_obj_set_as_root( b );
	if ( b_root_is_self ) bli_obj_set_as_root( a );
}

// Swap object pack schemas.

BLIS_INLINE void bli_obj_swap_pack_schemas( obj_t* a, obj_t* b )
{
	const pack_t schema_a = bli_obj_pack_schema( a );
	const pack_t schema_b = bli_obj_pack_schema( b );

	bli_obj_set_pack_schema( schema_b, a );
	bli_obj_set_pack_schema( schema_a, b );
}

// Induce a transposition on an object: swap dimensions, increments, and
// offsets, then clear the trans bit.

BLIS_INLINE void bli_obj_induce_trans( obj_t* obj )
{
	// Induce transposition among basic fields.
	dim_t  m        = bli_obj_length( obj );
	dim_t  n        = bli_obj_width( obj );
	inc_t  rs       = bli_obj_row_stride( obj );
	inc_t  cs       = bli_obj_col_stride( obj );
	dim_t  offm     = bli_obj_row_off( obj );
	dim_t  offn     = bli_obj_col_off( obj );
	doff_t diag_off = bli_obj_diag_offset( obj );

	bli_obj_set_dims( n, m, obj );
	bli_obj_set_strides( cs, rs, obj );
	bli_obj_set_offs( offn, offm, obj );
	bli_obj_set_diag_offset( -diag_off, obj );

	if ( bli_obj_is_upper_or_lower( obj ) )
		bli_obj_toggle_uplo( obj );

	// Induce transposition among packed fields.
	dim_t  m_padded = bli_obj_padded_length( obj );
	dim_t  n_padded = bli_obj_padded_width( obj );
	dim_t  m_panel  = bli_obj_panel_length( obj );
	dim_t  n_panel  = bli_obj_panel_width( obj );

	bli_obj_set_padded_dims( n_padded, m_padded, obj );
	bli_obj_set_panel_dims( n_panel, m_panel, obj );

	// Note that this macro DOES NOT touch the transposition bit! If
	// the calling code is using this function to handle an object whose
	// transposition bit is set prior to computation, that code needs
	// to manually clear or toggle the bit, via
	// bli_obj_set_onlytrans() or bli_obj_toggle_trans(),
	// respectively.
}

BLIS_INLINE void bli_obj_induce_fast_trans( obj_t* obj )
{
	// NOTE: This function is only used in situations where the matrices
	// are guaranteed to not have structure or be packed.

	// Induce transposition among basic fields.
	dim_t  m        = bli_obj_length( obj );
	dim_t  n        = bli_obj_width( obj );
	inc_t  rs       = bli_obj_row_stride( obj );
	inc_t  cs       = bli_obj_col_stride( obj );
	dim_t  offm     = bli_obj_row_off( obj );
	dim_t  offn     = bli_obj_col_off( obj );

	bli_obj_set_dims( n, m, obj );
	bli_obj_set_strides( cs, rs, obj );
	bli_obj_set_offs( offn, offm, obj );

	// Note that this macro DOES NOT touch the transposition bit! If
	// the calling code is using this function to handle an object whose
	// transposition bit is set prior to computation, that code needs
	// to manually clear or toggle the bit, via
	// bli_obj_set_onlytrans() or bli_obj_toggle_trans(),
	// respectively.
}

// Sometimes we need to "reflect" a partition because the data we want is
// actually stored on the other side of the diagonal. The nuts and bolts of
// this macro look a lot like an induced transposition, except that the row
// and column strides are left unchanged (which, of course, drastically
// changes the effect of the macro).

BLIS_INLINE void bli_obj_reflect_about_diag( obj_t* obj )
{
	dim_t  m        = bli_obj_length( obj );
	dim_t  n        = bli_obj_width( obj );
	dim_t  offm     = bli_obj_row_off( obj );
	dim_t  offn     = bli_obj_col_off( obj );
	doff_t diag_off = bli_obj_diag_offset( obj );

	bli_obj_set_dims( n, m, obj );
	bli_obj_set_offs( offn, offm, obj );
	bli_obj_set_diag_offset( -diag_off, obj );

	bli_obj_toggle_trans( obj );
}


#endif
// end bli_obj_macro_defs.h
#line 104 "./frame/include//bli_macro_defs.h"

// begin bli_complex_macro_defs.h
#line 1 "./frame/include//bli_complex_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_COMPLEX_MACRO_DEFS_H
#define BLIS_COMPLEX_MACRO_DEFS_H


// -- Real and imaginary accessor macros --


#define bli_sreal( x )  ( x )
#define bli_simag( x )  ( 0.0F )
#define bli_dreal( x )  ( x )
#define bli_dimag( x )  ( 0.0 )


#ifndef BLIS_ENABLE_C99_COMPLEX


#define bli_creal( x )  ( (x).real )
#define bli_cimag( x )  ( (x).imag )
#define bli_zreal( x )  ( (x).real )
#define bli_zimag( x )  ( (x).imag )


#else // ifdef BLIS_ENABLE_C99_COMPLEX


#define bli_creal( x )  ( crealf(x) )
#define bli_cimag( x )  ( cimagf(x) )
#define bli_zreal( x )  ( creal(x) )
#define bli_zimag( x )  ( cimag(x) )


#endif // BLIS_ENABLE_C99_COMPLEX


#endif

// end bli_complex_macro_defs.h
#line 105 "./frame/include//bli_macro_defs.h"

// begin bli_scalar_macro_defs.h
#line 1 "./frame/include//bli_scalar_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SCALAR_MACRO_DEFS_H
#define BLIS_SCALAR_MACRO_DEFS_H



// -- Assignment/Accessor macros --

// NOTE: This macro is defined first since some of the other scalar macros
// use it to abstract away the method used to assign complex values (ie:
// whether fields of a struct are set directly or whether native C99
// assignment is used).

// begin bli_sets.h
#line 1 "./frame/include/level0//bli_sets.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SETS_H
#define BLIS_SETS_H

// sets

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of y.

#define bli_sssets( xr, xi, y )  { (y) = (xr); }
#define bli_dssets( xr, xi, y )  { (y) = (xr); }
#define bli_cssets( xr, xi, y )  { (y) = (xr); }
#define bli_zssets( xr, xi, y )  { (y) = (xr); }
#define bli_issets( xr, xi, y )  { (y) = (xr); }

#define bli_sdsets( xr, xi, y )  { (y) = (xr); }
#define bli_ddsets( xr, xi, y )  { (y) = (xr); }
#define bli_cdsets( xr, xi, y )  { (y) = (xr); }
#define bli_zdsets( xr, xi, y )  { (y) = (xr); }
#define bli_idsets( xr, xi, y )  { (y) = (xr); }

#ifndef BLIS_ENABLE_C99_COMPLEX 

#define bli_scsets( xr, xi, y ) { bli_creal(y) = (xr); bli_cimag(y) = (xi); }
#define bli_dcsets( xr, xi, y ) { bli_creal(y) = (xr); bli_cimag(y) = (xi); }
#define bli_ccsets( xr, xi, y ) { bli_creal(y) = (xr); bli_cimag(y) = (xi); }
#define bli_zcsets( xr, xi, y ) { bli_creal(y) = (xr); bli_cimag(y) = (xi); }
#define bli_icsets( xr, xi, y ) { bli_creal(y) = (xr); bli_cimag(y) = (xi); }

#define bli_szsets( xr, xi, y ) { bli_zreal(y) = (xr); bli_zimag(y) = (xi); }
#define bli_dzsets( xr, xi, y ) { bli_zreal(y) = (xr); bli_zimag(y) = (xi); }
#define bli_czsets( xr, xi, y ) { bli_zreal(y) = (xr); bli_zimag(y) = (xi); }
#define bli_zzsets( xr, xi, y ) { bli_zreal(y) = (xr); bli_zimag(y) = (xi); }
#define bli_izsets( xr, xi, y ) { bli_zreal(y) = (xr); bli_zimag(y) = (xi); }

#else // ifdef BLIS_ENABLE_C99_COMPLEX

#define bli_scsets( xr, xi, y )  { (y) = (xr) + (xi) * (I); }
#define bli_dcsets( xr, xi, y )  { (y) = (xr) + (xi) * (I); }
#define bli_ccsets( xr, xi, y )  { (y) = (xr) + (xi) * (I); }
#define bli_zcsets( xr, xi, y )  { (y) = (xr) + (xi) * (I); }

#define bli_szsets( xr, xi, y )  { (y) = (xr) + (xi) * (I); }
#define bli_dzsets( xr, xi, y )  { (y) = (xr) + (xi) * (I); }
#define bli_czsets( xr, xi, y )  { (y) = (xr) + (xi) * (I); }
#define bli_zzsets( xr, xi, y )  { (y) = (xr) + (xi) * (I); }

#endif // BLIS_ENABLE_C99_COMPLEX

#define bli_sisets( xr, xi, y ) { (y) = bli_sreal(xr); }
#define bli_disets( xr, xi, y ) { (y) = bli_dreal(xr); }
#define bli_cisets( xr, xi, y ) { (y) = bli_creal(xr); }
#define bli_zisets( xr, xi, y ) { (y) = bli_zreal(xr); }
#define bli_iisets( xr, xi, y ) { (y) =          (xr); }


#define bli_ssets( xr, xi, y )  bli_sssets( xr, xi, y )
#define bli_dsets( xr, xi, y )  bli_ddsets( xr, xi, y )
#define bli_csets( xr, xi, y )  bli_scsets( xr, xi, y )
#define bli_zsets( xr, xi, y )  bli_dzsets( xr, xi, y )
#define bli_isets( xr, xi, y )  bli_disets( xr, xi, y )


#endif

// end bli_sets.h
#line 48 "./frame/include//bli_scalar_macro_defs.h"


// NOTE: These macros are not used by other scalar macros, but they are
// related to those defined in bli_sets.h, and so we #include them here.

// begin bli_setrs.h
#line 1 "./frame/include/level0//bli_setrs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SETRS_H
#define BLIS_SETRS_H

// setrs

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of y.

#define bli_sssetrs( xr, y )  { (y) = (xr); }
#define bli_dssetrs( xr, y )  { (y) = (xr); }

#define bli_sdsetrs( xr, y )  { (y) = (xr); }
#define bli_ddsetrs( xr, y )  { (y) = (xr); }

#ifndef BLIS_ENABLE_C99_COMPLEX

#define bli_scsetrs( xr, y )  { bli_creal(y) = (xr); }
#define bli_dcsetrs( xr, y )  { bli_creal(y) = (xr); }

#define bli_szsetrs( xr, y )  { bli_zreal(y) = (xr); }
#define bli_dzsetrs( xr, y )  { bli_zreal(y) = (xr); }

#else // ifdef BLIS_ENABLE_C99_COMPLEX

#define bli_scsetrs( xr, y )  { (y) = (xr) + bli_cimag(y) * (I); }
#define bli_dcsetrs( xr, y )  { (y) = (xr) + bli_cimag(y) * (I); }

#define bli_szsetrs( xr, y )  { (y) = (xr) + bli_zimag(y) * (I); }
#define bli_dzsetrs( xr, y )  { (y) = (xr) + bli_zimag(y) * (I); }

#endif // BLIS_ENABLE_C99_COMPLEX


#define bli_ssetrs( xr, y )  bli_sssetrs( xr, y )
#define bli_dsetrs( xr, y )  bli_ddsetrs( xr, y )
#define bli_csetrs( xr, y )  bli_scsetrs( xr, y )
#define bli_zsetrs( xr, y )  bli_dzsetrs( xr, y )


#endif

// end bli_setrs.h
#line 53 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_setis.h
#line 1 "./frame/include/level0//bli_setis.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SETIS_H
#define BLIS_SETIS_H

// setis

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of y.

#define bli_sssetis( xi, y )  { ; }
#define bli_dssetis( xi, y )  { ; }

#define bli_sdsetis( xi, y )  { ; }
#define bli_ddsetis( xi, y )  { ; }

#ifndef BLIS_ENABLE_C99_COMPLEX

#define bli_scsetis( xi, y )  { bli_cimag(y) = (xi); }
#define bli_dcsetis( xi, y )  { bli_cimag(y) = (xi); }

#define bli_szsetis( xi, y )  { bli_zimag(y) = (xi); }
#define bli_dzsetis( xi, y )  { bli_zimag(y) = (xi); }

#else // ifdef BLIS_ENABLE_C99_COMPLEX

#define bli_scsetis( xi, y )  { (y) = bli_creal(y) + (xi) * (I); }
#define bli_dcsetis( xi, y )  { (y) = bli_creal(y) + (xi) * (I); }

#define bli_szsetis( xi, y )  { (y) = bli_zreal(y) + (xi) * (I); }
#define bli_dzsetis( xi, y )  { (y) = bli_zreal(y) + (xi) * (I); }

#endif // BLIS_ENABLE_C99_COMPLEX


#define bli_ssetis( xi, y )  bli_sssetis( xi, y )
#define bli_dsetis( xi, y )  bli_ddsetis( xi, y )
#define bli_csetis( xi, y )  bli_scsetis( xi, y )
#define bli_zsetis( xi, y )  bli_dzsetis( xi, y )


#endif

// end bli_setis.h
#line 54 "./frame/include//bli_scalar_macro_defs.h"


// NOTE: This macro also needs to be defined early on since it determines
// how real and imaginary components are accessed (ie: whether the fields
// of a struct are read directly or whether native C99 functions are used.)

// begin bli_gets.h
#line 1 "./frame/include/level0//bli_gets.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_GETS_H
#define BLIS_GETS_H

// gets

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of y.


#define bli_ssgets( x, yr, yi )  { (yr) = bli_sreal(x); (yi) = bli_simag(x); }
#define bli_dsgets( x, yr, yi )  { (yr) = bli_dreal(x); (yi) = bli_dimag(x); }
#define bli_csgets( x, yr, yi )  { (yr) = bli_creal(x); (yi) = bli_cimag(x); }
#define bli_zsgets( x, yr, yi )  { (yr) = bli_zreal(x); (yi) = bli_zimag(x); }
#define bli_isgets( x, yr, yi )  { (yr) = ( float )(x); (yi) = 0.0F; }

#define bli_sdgets( x, yr, yi )  { (yr) = bli_sreal(x); (yi) = bli_simag(x); }
#define bli_ddgets( x, yr, yi )  { (yr) = bli_dreal(x); (yi) = bli_dimag(x); }
#define bli_cdgets( x, yr, yi )  { (yr) = bli_creal(x); (yi) = bli_cimag(x); }
#define bli_zdgets( x, yr, yi )  { (yr) = bli_zreal(x); (yi) = bli_zimag(x); }
#define bli_idgets( x, yr, yi )  { (yr) = ( double )(x); (yi) = 0.0; }

#define bli_scgets( x, yr, yi )  { (yr) = bli_sreal(x); (yi) = bli_simag(x); }
#define bli_dcgets( x, yr, yi )  { (yr) = bli_dreal(x); (yi) = bli_dimag(x); }
#define bli_ccgets( x, yr, yi )  { (yr) = bli_creal(x); (yi) = bli_cimag(x); }
#define bli_zcgets( x, yr, yi )  { (yr) = bli_zreal(x); (yi) = bli_zimag(x); }
#define bli_icgets( x, yr, yi )  { (yr) = ( float )(x); (yi) = 0.0F; }

#define bli_szgets( x, yr, yi )  { (yr) = bli_sreal(x); (yi) = bli_simag(x); }
#define bli_dzgets( x, yr, yi )  { (yr) = bli_dreal(x); (yi) = bli_dimag(x); }
#define bli_czgets( x, yr, yi )  { (yr) = bli_creal(x); (yi) = bli_cimag(x); }
#define bli_zzgets( x, yr, yi )  { (yr) = bli_zreal(x); (yi) = bli_zimag(x); }
#define bli_izgets( x, yr, yi )  { (yr) = ( double )(x); (yi) = 0.0; }

#define bli_sigets( x, yr, yi )  { (yr) = bli_sreal(x); (yi) = 0; }
#define bli_digets( x, yr, yi )  { (yr) = bli_dreal(x); (yi) = 0; }
#define bli_cigets( x, yr, yi )  { (yr) = bli_creal(x); (yi) = 0; }
#define bli_zigets( x, yr, yi )  { (yr) = bli_zreal(x); (yi) = 0; }
#define bli_iigets( x, yr, yi )  { (yr) =          (x); (yi) = 0; }


#define bli_sgets( x, yr, yi )  bli_ssgets( x, yr, yi )
#define bli_dgets( x, yr, yi )  bli_ddgets( x, yr, yi )
#define bli_cgets( x, yr, yi )  bli_csgets( x, yr, yi )
#define bli_zgets( x, yr, yi )  bli_zdgets( x, yr, yi )
#define bli_igets( x, yr, yi )  bli_idgets( x, yr, yi )


#endif
// end bli_gets.h
#line 60 "./frame/include//bli_scalar_macro_defs.h"



// -- Scalar constant initialization macros --

// begin bli_constants.h
#line 1 "./frame/include/level0//bli_constants.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_CONSTANTS_H
#define BLIS_CONSTANTS_H

// return pointers to constants

// 1

#define bli_s1 \
\
	( ( float*    ) bli_obj_buffer_for_const( BLIS_FLOAT,    &BLIS_ONE ) )

#define bli_d1 \
\
	( ( double*   ) bli_obj_buffer_for_const( BLIS_DOUBLE,   &BLIS_ONE ) )

#define bli_c1 \
\
	( ( scomplex* ) bli_obj_buffer_for_const( BLIS_SCOMPLEX, &BLIS_ONE ) )

#define bli_z1 \
\
	( ( dcomplex* ) bli_obj_buffer_for_const( BLIS_DCOMPLEX, &BLIS_ONE ) )

#define bli_i1 \
\
	( ( gint_t*   ) bli_obj_buffer_for_const( BLIS_INT,      &BLIS_ONE ) )

// 0

#define bli_s0 \
\
	( ( float*    ) bli_obj_buffer_for_const( BLIS_FLOAT,    &BLIS_ZERO ) )

#define bli_d0 \
\
	( ( double*   ) bli_obj_buffer_for_const( BLIS_DOUBLE,   &BLIS_ZERO ) )

#define bli_c0 \
\
	( ( scomplex* ) bli_obj_buffer_for_const( BLIS_SCOMPLEX, &BLIS_ZERO ) )

#define bli_z0 \
\
	( ( dcomplex* ) bli_obj_buffer_for_const( BLIS_DCOMPLEX, &BLIS_ZERO ) )

#define bli_i0 \
\
	( ( gint_t*   ) bli_obj_buffer_for_const( BLIS_INT,      &BLIS_ZERO ) )

// -1

#define bli_sm1 \
\
	( ( float*    ) bli_obj_buffer_for_const( BLIS_FLOAT,    &BLIS_MINUS_ONE ) )

#define bli_dm1 \
\
	( ( double*   ) bli_obj_buffer_for_const( BLIS_DOUBLE,   &BLIS_MINUS_ONE ) )

#define bli_cm1 \
\
	( ( scomplex* ) bli_obj_buffer_for_const( BLIS_SCOMPLEX, &BLIS_MINUS_ONE ) )

#define bli_zm1 \
\
	( ( dcomplex* ) bli_obj_buffer_for_const( BLIS_DCOMPLEX, &BLIS_MINUS_ONE ) )

#define bli_im1 \
\
	( ( gint_t*   ) bli_obj_buffer_for_const( BLIS_INT,      &BLIS_MINUS_ONE ) )


#endif

// end bli_constants.h
#line 65 "./frame/include//bli_scalar_macro_defs.h"



// -- Separated scalar macros (separated real/imaginary values) --

// begin bli_absq2ris.h
#line 1 "./frame/include/level0/ri//bli_absq2ris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_ABSQ2RIS_H
#define BLIS_ABSQ2RIS_H

// absq2ris

#define bli_sabsq2ris( ar, ai, br, bi ) \
{ \
	(br) = (ar) * (ar); \
}

#define bli_dabsq2ris( ar, ai, br, bi ) \
{ \
	(br) = (ar) * (ar); \
}

#define bli_cabsq2ris( ar, ai, br, bi ) \
{ \
	(br) = (ar) * (ar) + (ai) * (ai); \
	(bi) = 0.0F; \
}

#define bli_zabsq2ris( ar, ai, br, bi ) \
{ \
	(br) = (ar) * (ar) + (ai) * (ai); \
	(bi) = 0.0; \
}

#endif

// end bli_absq2ris.h
#line 70 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_abval2ris.h
#line 1 "./frame/include/level0/ri//bli_abval2ris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_ABVAL2RIS_H
#define BLIS_ABVAL2RIS_H

// abval2ris

#define bli_sabval2ris( xr, xi, ar, ai ) \
{ \
	(ar)       = fabsf(xr); \
}

#define bli_dabval2ris( xr, xi, ar, ai ) \
{ \
	(ar)       = fabs(xr); \
}

#define bli_cabval2ris( xr, xi, ar, ai ) \
{ \
	float  s   = bli_fmaxabs( (xr), (xi) ); \
	float  mag; \
	if ( s == 0.0F ) mag = 0.0F; \
	else \
	{ \
		mag = sqrtf( s ) * \
		      sqrtf( ( (xr) / s ) * (xr) + \
		             ( (xi) / s ) * (xi) ); \
	} \
	(ar)       = mag; \
	(ai)       = 0.0F; \
}

#define bli_zabval2ris( xr, xi, ar, ai ) \
{ \
	double s   = bli_fmaxabs( (xr), (xi) ); \
	double mag; \
	if ( s == 0.0 ) mag = 0.0; \
	else \
	{ \
		mag = sqrt( s ) * \
		      sqrt( ( (xr) / s ) * (xr) + \
		            ( (xi) / s ) * (xi) ); \
	} \
	(ar)       = mag; \
	(ai)       = 0.0; \
}

#endif
// end bli_abval2ris.h
#line 72 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_addris.h
#line 1 "./frame/include/level0/ri//bli_addris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_ADDRIS_H
#define BLIS_ADDRIS_H

// addris

#define bli_saddris( ar, ai, xr, xi ) \
{ \
	(xr) = (xr) + (ar); \
}

#define bli_daddris( ar, ai, xr, xi ) \
{ \
	(xr) = (xr) + (ar); \
}

#define bli_caddris( ar, ai, xr, xi ) \
{ \
	(xr) = (xr) + (ar); \
	(xi) = (xi) + (ai); \
}

#define bli_zaddris( ar, ai, xr, xi ) \
{ \
	(xr) = (xr) + (ar); \
	(xi) = (xi) + (ai); \
}

#endif

// end bli_addris.h
#line 74 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_addjris.h
#line 1 "./frame/include/level0/ri//bli_addjris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_ADDJRIS_H
#define BLIS_ADDJRIS_H

// addjris

#define bli_saddjris( ar, ai, xr, xi )  bli_saddris( (ar), -(ai), (xr), (xi) )
#define bli_daddjris( ar, ai, xr, xi )  bli_daddris( (ar), -(ai), (xr), (xi) )
#define bli_caddjris( ar, ai, xr, xi )  bli_caddris( (ar), -(ai), (xr), (xi) )
#define bli_zaddjris( ar, ai, xr, xi )  bli_zaddris( (ar), -(ai), (xr), (xi) )

#endif

// end bli_addjris.h
#line 75 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_add3ris.h
#line 1 "./frame/include/level0/ri//bli_add3ris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_ADD3RIS_H
#define BLIS_ADD3RIS_H

// add3ris

#define bli_sadd3ris( ar, ai, br, bi, cr, ci ) \
{ \
	(cr) = (ar) + (br); \
}

#define bli_dadd3ris( ar, ai, br, bi, cr, ci ) \
{ \
	(cr) = (ar) + (br); \
}

#define bli_cadd3ris( ar, ai, br, bi, cr, ci ) \
{ \
	(cr) = (ar) + (br); \
	(ci) = (ai) + (bi); \
}

#define bli_zadd3ris( ar, ai, br, bi, cr, ci ) \
{ \
	(cr) = (ar) + (br); \
	(ci) = (ai) + (bi); \
}

#endif

// end bli_add3ris.h
#line 77 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_axpbyris.h
#line 1 "./frame/include/level0/ri//bli_axpbyris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_AXPBYRIS_H
#define BLIS_AXPBYRIS_H

// axpbyris

#define bli_rxaxpbyris( ar, ai, xr, xi, br, bi, yr, yi ) \
{ \
    (yr) = (ar) * (xr) + (br) * (yr); \
}

#define bli_cxaxpbyris( ar, ai, xr, xi, br, bi, yr, yi ) \
{ \
    const __typeof__(yr) yt_r = (ar) * (xr) - (ai) * (xi) + (br) * (yr) - (bi) * (yi); \
    const __typeof__(yi) yt_i = (ai) * (xr) + (ar) * (xi) + (bi) * (yr) + (br) * (yi); \
    (yr) = yt_r; \
    (yi) = yt_i; \
}

// Notes:
// - The first char encodes the type of a.
// - The second char encodes the type of x.
// - The third char encodes the type of b.
// - The fourth char encodes the type of y.

// -- (axby) = (??ss) ----------------------------------------------------------

#define bli_ssssxpbyris  bli_rxxpbyris
#define bli_dsssxpbyris  bli_rxxpbyris
#define bli_csssxpbyris  bli_rxxpbyris
#define bli_zsssxpbyris  bli_rxxpbyris

#define bli_sdssxpbyris  bli_rxxpbyris
#define bli_ddssxpbyris  bli_rxxpbyris
#define bli_cdssxpbyris  bli_rxxpbyris
#define bli_zdssxpbyris  bli_rxxpbyris

#define bli_scssxpbyris  bli_rxxpbyris
#define bli_dcssxpbyris  bli_rxxpbyris
#define bli_ccssxpbyris  bli_rxxpbyris
#define bli_zcssxpbyris  bli_rxxpbyris

#define bli_szssxpbyris  bli_rxxpbyris
#define bli_dzssxpbyris  bli_rxxpbyris
#define bli_czssxpbyris  bli_rxxpbyris
#define bli_zzssxpbyris  bli_rxxpbyris

// NOTE: This series needs to be finished for all other char values for (by), but
// not until something in BLIS actually needs mixed-datatype axpbyris.


#define bli_saxpbyris    bli_ssssaxpbyris
#define bli_daxpbyris    bli_ddddaxpbyris
#define bli_caxpbyris    bli_ccccaxpbyris
#define bli_zaxpbyris    bli_zzzzaxpbyris

#endif

// end bli_axpbyris.h
#line 79 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_axpbyjris.h
#line 1 "./frame/include/level0/ri//bli_axpbyjris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_AXPBYJRIS_H
#define BLIS_AXPBYJRIS_H

// axpbyjris

#define bli_rxaxpbyjris( ar, ai, xr, xi, br, bi, yr, yi ) \
{ \
    (yr) = (ar) * (xr) + (br) * (yr); \
}

#define bli_cxaxpbyjris( ar, ai, xr, xi, br, bi, yr, yi ) \
{ \
    const __typeof__(yr) yt_r = (ar) * (xr) + (ai) * (xi) + (br) * (yr) - (bi) * (yi); \
    const __typeof__(yi) yt_i = (ai) * (xr) - (ar) * (xi) + (bi) * (yr) + (br) * (yi); \
    (yr) = yt_r; \
    (yi) = yt_i; \
}

// Notes:
// - The first char encodes the type of a.
// - The second char encodes the type of x.
// - The third char encodes the type of b.
// - The fourth char encodes the type of y.

// -- (axby) = (??ss) ----------------------------------------------------------

#define bli_ssssxpbyjris  bli_rxxpbyjris
#define bli_dsssxpbyjris  bli_rxxpbyjris
#define bli_csssxpbyjris  bli_rxxpbyjris
#define bli_zsssxpbyjris  bli_rxxpbyjris

#define bli_sdssxpbyjris  bli_rxxpbyjris
#define bli_ddssxpbyjris  bli_rxxpbyjris
#define bli_cdssxpbyjris  bli_rxxpbyjris
#define bli_zdssxpbyjris  bli_rxxpbyjris

#define bli_scssxpbyjris  bli_rxxpbyjris
#define bli_dcssxpbyjris  bli_rxxpbyjris
#define bli_ccssxpbyjris  bli_rxxpbyjris
#define bli_zcssxpbyjris  bli_rxxpbyjris

#define bli_szssxpbyjris  bli_rxxpbyjris
#define bli_dzssxpbyjris  bli_rxxpbyjris
#define bli_czssxpbyjris  bli_rxxpbyjris
#define bli_zzssxpbyjris  bli_rxxpbyjris

// NOTE: This series needs to be finished for all other char values for (by), but
// not until something in BLIS actually needs mixed-datatype axpbyjris.


#define bli_saxpbyjris    bli_ssssaxpbyjris
#define bli_daxpbyjris    bli_ddddaxpbyjris
#define bli_caxpbyjris    bli_ccccaxpbyjris
#define bli_zaxpbyjris    bli_zzzzaxpbyjris

#endif

// end bli_axpbyjris.h
#line 80 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_axpyris.h
#line 1 "./frame/include/level0/ri//bli_axpyris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_AXPYRIS_H
#define BLIS_AXPYRIS_H

// axpyris

#define bli_rxaxpyris( ar, ai, xr, xi, yr, yi ) \
{ \
	(yr) += (ar) * (xr); \
}

#define bli_cxaxpyris( ar, ai, xr, xi, yr, yi ) \
{ \
	(yr) += (ar) * (xr) - (ai) * (xi); \
	(yi) += (ai) * (xr) + (ar) * (xi); \
}

#define bli_roaxpyris( ar, ai, xr, xi, yr, yi ) \
{ \
	(yr) += (ar) * (xr) - (ai) * (xi); \
}

#define bli_craxpyris( ar, ai, xr, xi, yr, yi ) \
{ \
	(yr) += (ar) * (xr); \
	(yi) += (ar) * (xi); \
}

#define bli_rcaxpyris( ar, ai, xr, xi, yr, yi ) \
{ \
	(yr) += (ar) * (xr); \
	(yi) += (ai) * (xr); \
}

// Notes:
// - The first char encodes the type of a.
// - The second char encodes the type of x.
// - The third char encodes the type of y.

// -- (axy) = (??s) ------------------------------------------------------------

#define bli_sssaxpyris  bli_rxaxpyris
#define bli_dssaxpyris  bli_rxaxpyris
#define bli_cssaxpyris  bli_rxaxpyris
#define bli_zssaxpyris  bli_rxaxpyris

#define bli_sdsaxpyris  bli_rxaxpyris
#define bli_ddsaxpyris  bli_rxaxpyris
#define bli_cdsaxpyris  bli_rxaxpyris
#define bli_zdsaxpyris  bli_rxaxpyris

#define bli_scsaxpyris  bli_rxaxpyris
#define bli_dcsaxpyris  bli_rxaxpyris
#define bli_ccsaxpyris  bli_roaxpyris
#define bli_zcsaxpyris  bli_roaxpyris

#define bli_szsaxpyris  bli_rxaxpyris
#define bli_dzsaxpyris  bli_rxaxpyris
#define bli_czsaxpyris  bli_roaxpyris
#define bli_zzsaxpyris  bli_roaxpyris

// -- (axy) = (??d) ------------------------------------------------------------

#define bli_ssdaxpyris  bli_rxaxpyris
#define bli_dsdaxpyris  bli_rxaxpyris
#define bli_csdaxpyris  bli_rxaxpyris
#define bli_zsdaxpyris  bli_rxaxpyris

#define bli_sddaxpyris  bli_rxaxpyris
#define bli_dddaxpyris  bli_rxaxpyris
#define bli_cddaxpyris  bli_rxaxpyris
#define bli_zddaxpyris  bli_rxaxpyris

#define bli_scdaxpyris  bli_rxaxpyris
#define bli_dcdaxpyris  bli_rxaxpyris
#define bli_ccdaxpyris  bli_roaxpyris
#define bli_zcdaxpyris  bli_roaxpyris

#define bli_szdaxpyris  bli_rxaxpyris
#define bli_dzdaxpyris  bli_rxaxpyris
#define bli_czdaxpyris  bli_roaxpyris
#define bli_zzdaxpyris  bli_roaxpyris

// -- (axy) = (??c) ------------------------------------------------------------

#define bli_sscaxpyris  bli_rxaxpyris
#define bli_dscaxpyris  bli_rxaxpyris
#define bli_cscaxpyris  bli_rcaxpyris
#define bli_zscaxpyris  bli_rcaxpyris

#define bli_sdcaxpyris  bli_rxaxpyris
#define bli_ddcaxpyris  bli_rxaxpyris
#define bli_cdcaxpyris  bli_rcaxpyris
#define bli_zdcaxpyris  bli_rcaxpyris

#define bli_sccaxpyris  bli_craxpyris
#define bli_dccaxpyris  bli_craxpyris
#define bli_cccaxpyris  bli_cxaxpyris
#define bli_zccaxpyris  bli_cxaxpyris

#define bli_szcaxpyris  bli_craxpyris
#define bli_dzcaxpyris  bli_craxpyris
#define bli_czcaxpyris  bli_cxaxpyris
#define bli_zzcaxpyris  bli_cxaxpyris

// -- (axy) = (??z) ------------------------------------------------------------

#define bli_sszaxpyris  bli_rxaxpyris
#define bli_dszaxpyris  bli_rxaxpyris
#define bli_cszaxpyris  bli_rcaxpyris
#define bli_zszaxpyris  bli_rcaxpyris

#define bli_sdzaxpyris  bli_rxaxpyris
#define bli_ddzaxpyris  bli_rxaxpyris
#define bli_cdzaxpyris  bli_rcaxpyris
#define bli_zdzaxpyris  bli_rcaxpyris

#define bli_sczaxpyris  bli_craxpyris
#define bli_dczaxpyris  bli_craxpyris
#define bli_cczaxpyris  bli_cxaxpyris
#define bli_zczaxpyris  bli_cxaxpyris

#define bli_szzaxpyris  bli_craxpyris
#define bli_dzzaxpyris  bli_craxpyris
#define bli_czzaxpyris  bli_cxaxpyris
#define bli_zzzaxpyris  bli_cxaxpyris



#define bli_saxpyris    bli_sssaxpyris
#define bli_daxpyris    bli_dddaxpyris
#define bli_caxpyris    bli_cccaxpyris
#define bli_zaxpyris    bli_zzzaxpyris

#endif

// end bli_axpyris.h
#line 82 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_axpyjris.h
#line 1 "./frame/include/level0/ri//bli_axpyjris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_AXPYJRIS_H
#define BLIS_AXPYJRIS_H

// axpyjris

#define bli_rxaxpyjris( ar, ai, xr, xi, yr, yi ) \
{ \
	(yr) += (ar) * (xr); \
}

#define bli_cxaxpyjris( ar, ai, xr, xi, yr, yi ) \
{ \
	(yr) += (ar) * (xr) + (ai) * (xi); \
	(yi) += (ai) * (xr) - (ar) * (xi); \
}

#define bli_roaxpyjris( ar, ai, xr, xi, yr, yi ) \
{ \
	(yr) += (ar) * (xr) + (ai) * (xi); \
}

#define bli_craxpyjris( ar, ai, xr, xi, yr, yi ) \
{ \
	(yr) += (ar) *  (xr); \
	(yi) += (ar) * -(xi); \
}

#define bli_rcaxpyjris( ar, ai, xr, xi, yr, yi ) \
{ \
	(yr) += (ar) * (xr); \
	(yi) += (ai) * (xr); \
}

// Notes:
// - The first char encodes the type of a.
// - The second char encodes the type of x.
// - The third char encodes the type of y.

// -- (axy) = (??s) ------------------------------------------------------------

#define bli_sssaxpyjris  bli_rxaxpyjris
#define bli_dssaxpyjris  bli_rxaxpyjris
#define bli_cssaxpyjris  bli_rxaxpyjris
#define bli_zssaxpyjris  bli_rxaxpyjris

#define bli_sdsaxpyjris  bli_rxaxpyjris
#define bli_ddsaxpyjris  bli_rxaxpyjris
#define bli_cdsaxpyjris  bli_rxaxpyjris
#define bli_zdsaxpyjris  bli_rxaxpyjris

#define bli_scsaxpyjris  bli_rxaxpyjris
#define bli_dcsaxpyjris  bli_rxaxpyjris
#define bli_ccsaxpyjris  bli_roaxpyjris
#define bli_zcsaxpyjris  bli_roaxpyjris

#define bli_szsaxpyjris  bli_rxaxpyjris
#define bli_dzsaxpyjris  bli_rxaxpyjris
#define bli_czsaxpyjris  bli_roaxpyjris
#define bli_zzsaxpyjris  bli_roaxpyjris

// -- (axy) = (??d) ------------------------------------------------------------

#define bli_ssdaxpyjris  bli_rxaxpyjris
#define bli_dsdaxpyjris  bli_rxaxpyjris
#define bli_csdaxpyjris  bli_rxaxpyjris
#define bli_zsdaxpyjris  bli_rxaxpyjris

#define bli_sddaxpyjris  bli_rxaxpyjris
#define bli_dddaxpyjris  bli_rxaxpyjris
#define bli_cddaxpyjris  bli_rxaxpyjris
#define bli_zddaxpyjris  bli_rxaxpyjris

#define bli_scdaxpyjris  bli_rxaxpyjris
#define bli_dcdaxpyjris  bli_rxaxpyjris
#define bli_ccdaxpyjris  bli_roaxpyjris
#define bli_zcdaxpyjris  bli_roaxpyjris

#define bli_szdaxpyjris  bli_rxaxpyjris
#define bli_dzdaxpyjris  bli_rxaxpyjris
#define bli_czdaxpyjris  bli_roaxpyjris
#define bli_zzdaxpyjris  bli_roaxpyjris

// -- (axy) = (??c) ------------------------------------------------------------

#define bli_sscaxpyjris  bli_rxaxpyjris
#define bli_dscaxpyjris  bli_rxaxpyjris
#define bli_cscaxpyjris  bli_rcaxpyjris
#define bli_zscaxpyjris  bli_rcaxpyjris

#define bli_sdcaxpyjris  bli_rxaxpyjris
#define bli_ddcaxpyjris  bli_rxaxpyjris
#define bli_cdcaxpyjris  bli_rcaxpyjris
#define bli_zdcaxpyjris  bli_rcaxpyjris

#define bli_sccaxpyjris  bli_craxpyjris
#define bli_dccaxpyjris  bli_craxpyjris
#define bli_cccaxpyjris  bli_cxaxpyjris
#define bli_zccaxpyjris  bli_cxaxpyjris

#define bli_szcaxpyjris  bli_craxpyjris
#define bli_dzcaxpyjris  bli_craxpyjris
#define bli_czcaxpyjris  bli_cxaxpyjris
#define bli_zzcaxpyjris  bli_cxaxpyjris

// -- (axy) = (??z) ------------------------------------------------------------

#define bli_sszaxpyjris  bli_rxaxpyjris
#define bli_dszaxpyjris  bli_rxaxpyjris
#define bli_cszaxpyjris  bli_rcaxpyjris
#define bli_zszaxpyjris  bli_rcaxpyjris

#define bli_sdzaxpyjris  bli_rxaxpyjris
#define bli_ddzaxpyjris  bli_rxaxpyjris
#define bli_cdzaxpyjris  bli_rcaxpyjris
#define bli_zdzaxpyjris  bli_rcaxpyjris

#define bli_sczaxpyjris  bli_craxpyjris
#define bli_dczaxpyjris  bli_craxpyjris
#define bli_cczaxpyjris  bli_cxaxpyjris
#define bli_zczaxpyjris  bli_cxaxpyjris

#define bli_szzaxpyjris  bli_craxpyjris
#define bli_dzzaxpyjris  bli_craxpyjris
#define bli_czzaxpyjris  bli_cxaxpyjris
#define bli_zzzaxpyjris  bli_cxaxpyjris



#define bli_saxpyjris    bli_sssaxpyjris
#define bli_daxpyjris    bli_dddaxpyjris
#define bli_caxpyjris    bli_cccaxpyjris
#define bli_zaxpyjris    bli_zzzaxpyjris

#endif

// end bli_axpyjris.h
#line 83 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_axmyris.h
#line 1 "./frame/include/level0/ri//bli_axmyris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_AXMYRIS_H
#define BLIS_AXMYRIS_H

// axmyris

#define bli_saxmyris( ar, ai, xr, xi, yr, yi ) \
{ \
	(yr) -= (ar) * (xr); \
}

#define bli_daxmyris( ar, ai, xr, xi, yr, yi ) \
{ \
	(yr) -= (ar) * (xr); \
}

#define bli_caxmyris( ar, ai, xr, xi, yr, yi ) \
{ \
	(yr) -= (ar) * (xr) - (ai) * (xi); \
	(yi) -= (ai) * (xr) + (ar) * (xi); \
}

#define bli_zaxmyris( ar, ai, xr, xi, yr, yi ) \
{ \
	(yr) -= (ar) * (xr) - (ai) * (xi); \
	(yi) -= (ai) * (xr) + (ar) * (xi); \
}

#define bli_scaxmyris( ar, ai, xr, xi, yr, yi ) \
{ \
	(yr) -= (ar) * (xr); \
	(yi) -= (ar) * (xi); \
}

#define bli_dzaxmyris( ar, ai, xr, xi, yr, yi ) \
{ \
	(yr) -= (ar) * (xr); \
	(yi) -= (ar) * (xi); \
}

#endif

// end bli_axmyris.h
#line 85 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_conjris.h
#line 1 "./frame/include/level0/ri//bli_conjris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_CONJRIS_H
#define BLIS_CONJRIS_H

// conjris

#define bli_sconjris( xr, xi ) \
{ \
	; \
}

#define bli_dconjris( xr, xi ) \
{ \
	; \
}

#define bli_cconjris( xr, xi ) \
{ \
	(xi) = -(xi); \
}

#define bli_zconjris( xr, xi ) \
{ \
	(xi) = -(xi); \
}

#endif

// end bli_conjris.h
#line 87 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_copyris.h
#line 1 "./frame/include/level0/ri//bli_copyris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_COPYRIS_H
#define BLIS_COPYRIS_H

// copyris

#define bli_scopyris( ar, ai, br, bi ) \
{ \
	(br) = (ar); \
    ( void )ai; ( void )bi; \
}

#define bli_dcopyris( ar, ai, br, bi ) \
{ \
	(br) = (ar); \
    ( void )ai; ( void )bi; \
}

#define bli_ccopyris( ar, ai, br, bi ) \
{ \
	(br) = (ar); \
	(bi) = (ai); \
}

#define bli_zcopyris( ar, ai, br, bi ) \
{ \
	(br) = (ar); \
	(bi) = (ai); \
}

#define bli_sscopyris( ar, ai, br, bi )  { bli_scopyris( ar, 0.0F, br, bi ); ( void )ai; }
#define bli_dscopyris( ar, ai, br, bi )  { bli_scopyris( ar, 0.0,  br, bi ); ( void )ai; }
#define bli_cscopyris( ar, ai, br, bi )  bli_scopyris( ar, ai,   br, bi )
#define bli_zscopyris( ar, ai, br, bi )  bli_scopyris( ar, ai,   br, bi )

#define bli_sdcopyris( ar, ai, br, bi )  { bli_dcopyris( ar, 0.0F, br, bi ); ( void )ai; }
#define bli_ddcopyris( ar, ai, br, bi )  { bli_dcopyris( ar, 0.0,  br, bi ); ( void )ai; }
#define bli_cdcopyris( ar, ai, br, bi )  bli_dcopyris( ar, ai,   br, bi )
#define bli_zdcopyris( ar, ai, br, bi )  bli_dcopyris( ar, ai,   br, bi )

#define bli_sccopyris( ar, ai, br, bi )  { bli_ccopyris( ar, 0.0F, br, bi ); ( void )ai; }
#define bli_dccopyris( ar, ai, br, bi )  { bli_ccopyris( ar, 0.0,  br, bi ); ( void )ai; }
#define bli_cccopyris( ar, ai, br, bi )  bli_ccopyris( ar, ai,   br, bi )
#define bli_zccopyris( ar, ai, br, bi )  bli_ccopyris( ar, ai,   br, bi )

#define bli_szcopyris( ar, ai, br, bi )  { bli_zcopyris( ar, 0.0F, br, bi ); ( void )ai; }
#define bli_dzcopyris( ar, ai, br, bi )  { bli_zcopyris( ar, 0.0,  br, bi ); ( void )ai; }
#define bli_czcopyris( ar, ai, br, bi )  bli_zcopyris( ar, ai,   br, bi )
#define bli_zzcopyris( ar, ai, br, bi )  bli_zcopyris( ar, ai,   br, bi )

#endif
// end bli_copyris.h
#line 89 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_copyjris.h
#line 1 "./frame/include/level0/ri//bli_copyjris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_COPYJRIS_H
#define BLIS_COPYJRIS_H

// copyjris

#define bli_scopyjris( ar, ai, br, bi )  bli_scopyris( (ar), -(ai), (br), (bi) )
#define bli_dcopyjris( ar, ai, br, bi )  bli_dcopyris( (ar), -(ai), (br), (bi) )
#define bli_ccopyjris( ar, ai, br, bi )  bli_ccopyris( (ar), -(ai), (br), (bi) )
#define bli_zcopyjris( ar, ai, br, bi )  bli_zcopyris( (ar), -(ai), (br), (bi) )

#define bli_sscopyjris( ar, ai, br, bi )  bli_scopyjris( ar, 0.0F, br, bi )
#define bli_dscopyjris( ar, ai, br, bi )  bli_scopyjris( ar, 0.0,  br, bi )
#define bli_cscopyjris( ar, ai, br, bi )  bli_scopyjris( ar, ai,   br, bi )
#define bli_zscopyjris( ar, ai, br, bi )  bli_scopyjris( ar, ai,   br, bi )

#define bli_sdcopyjris( ar, ai, br, bi )  bli_dcopyjris( ar, 0.0F, br, bi )
#define bli_ddcopyjris( ar, ai, br, bi )  bli_dcopyjris( ar, 0.0,  br, bi )
#define bli_cdcopyjris( ar, ai, br, bi )  bli_dcopyjris( ar, ai,   br, bi )
#define bli_zdcopyjris( ar, ai, br, bi )  bli_dcopyjris( ar, ai,   br, bi )

#define bli_sccopyjris( ar, ai, br, bi )  bli_ccopyjris( ar, 0.0F, br, bi )
#define bli_dccopyjris( ar, ai, br, bi )  bli_ccopyjris( ar, 0.0,  br, bi )
#define bli_cccopyjris( ar, ai, br, bi )  bli_ccopyjris( ar, ai,   br, bi )
#define bli_zccopyjris( ar, ai, br, bi )  bli_ccopyjris( ar, ai,   br, bi )

#define bli_szcopyjris( ar, ai, br, bi )  bli_zcopyjris( ar, 0.0F, br, bi )
#define bli_dzcopyjris( ar, ai, br, bi )  bli_zcopyjris( ar, 0.0,  br, bi )
#define bli_czcopyjris( ar, ai, br, bi )  bli_zcopyjris( ar, ai,   br, bi )
#define bli_zzcopyjris( ar, ai, br, bi )  bli_zcopyjris( ar, ai,   br, bi )

#endif

// end bli_copyjris.h
#line 90 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_copycjris.h
#line 1 "./frame/include/level0/ri//bli_copycjris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_COPYCJRIS_H
#define BLIS_COPYCJRIS_H

// copycjris

#define bli_scopycjris( conj, xr, xi, yr, yi ) \
{ \
	bli_scopyris( (xr), (xi), (yr), (yi) ); \
}

#define bli_dcopycjris( conj, xr, xi, yr, yi ) \
{ \
	bli_dcopyris( (xr), (xi), (yr), (yi) ); \
}

#define bli_ccopycjris( conj, xr, xi, yr, yi ) \
{ \
	(yr) =                          (xr); \
	(yi) = ( bli_is_conj( conj ) ? -(xi) \
	                             :  (xi) ); \
}

#define bli_zcopycjris( conj, xr, xi, yr, yi ) \
{ \
	(yr) =                          (xr); \
	(yi) = ( bli_is_conj( conj ) ? -(xi) \
	                             :  (xi) ); \
}

#define bli_icopycjris( conj, xr, xi, yr, yi ) \
{ \
	bli_icopyris( (xr), (xi), (yr), (yi) ); \
}

#endif
// end bli_copycjris.h
#line 91 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_eqris.h
#line 1 "./frame/include/level0/ri//bli_eqris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_EQRIS_H
#define BLIS_EQRIS_H


// eqris (passed by value)

#define bli_seqris( ar, ai, br, bi )  ( (ar) == (br) )
#define bli_deqris( ar, ai, br, bi )  ( (ar) == (br) )
#define bli_ceqris( ar, ai, br, bi )  ( (ar) == (br) && (ai) == (bi) )
#define bli_zeqris( ar, ai, br, bi )  ( (ar) == (br) && (ai) == (bi) )
#define bli_ieqris( ar, ai, br, bi )  ( (ar) == (br) )


// eq1ris

#define bli_seq1ris( ar, ai )  bli_seqris( (ar), (ai), 1.0F, 0.0F )
#define bli_deq1ris( ar, ai )  bli_deqris( (ar), (ai), 1.0,  0.0  )
#define bli_ceq1ris( ar, ai )  bli_ceqris( (ar), (ai), 1.0F, 0.0F )
#define bli_zeq1ris( ar, ai )  bli_zeqris( (ar), (ai), 1.0,  0.0  )
#define bli_ieq1ris( ar, ai )  bli_ieqris( (ar), (ai), 1,    0    )


// eq0ris

#define bli_seq0ris( ar, ai )  bli_seqris( (ar), (ai), 0.0F, 0.0F )
#define bli_deq0ris( ar, ai )  bli_deqris( (ar), (ai), 0.0,  0.0  )
#define bli_ceq0ris( ar, ai )  bli_ceqris( (ar), (ai), 0.0F, 0.0F )
#define bli_zeq0ris( ar, ai )  bli_zeqris( (ar), (ai), 0.0,  0.0  )
#define bli_ieq0ris( ar, ai )  bli_ieqris( (ar), (ai), 0,    0    )


// eqm1ris

#define bli_seqm1ris( ar, ai )  bli_seqris( (ar), (ai), -1.0F, 0.0F )
#define bli_deqm1ris( ar, ai )  bli_deqris( (ar), (ai), -1.0,  0.0  )
#define bli_ceqm1ris( ar, ai )  bli_ceqris( (ar), (ai), -1.0F, 0.0F )
#define bli_zeqm1ris( ar, ai )  bli_zeqris( (ar), (ai), -1.0,  0.0  )
#define bli_ieqm1ris( ar, ai )  bli_ieqris( (ar), (ai), -1,    0    )



#endif
// end bli_eqris.h
#line 93 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_invertris.h
#line 1 "./frame/include/level0/ri//bli_invertris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_INVERTRIS_H
#define BLIS_INVERTRIS_H

// invertris

#define bli_sinvertris( xr, xi ) \
{ \
	(xr) = 1.0F / (xr); \
}

#define bli_dinvertris( xr, xi ) \
{ \
	(xr) = 1.0  / (xr); \
}

#define bli_cinvertris( xr, xi ) \
{ \
	float  s    = bli_fmaxabs( (xr), (xi) ); \
	float  xr_s = (xr) / s; \
	float  xi_s = (xi) / s; \
	float  temp = ( xr_s * (xr) + xi_s * (xi) ); \
	(xr)        =  xr_s / temp; \
	(xi)        = -xi_s / temp; \
}

#define bli_zinvertris( xr, xi ) \
{ \
	double s    = bli_fmaxabs( (xr), (xi) ); \
	double xr_s = (xr) / s; \
	double xi_s = (xi) / s; \
	double temp = ( xr_s * (xr) + xi_s * (xi) ); \
	(xr)        =  xr_s / temp; \
	(xi)        = -xi_s / temp; \
}

#endif
// end bli_invertris.h
#line 95 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_invscalris.h
#line 1 "./frame/include/level0/ri//bli_invscalris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_INVSCALRIS_H
#define BLIS_INVSCALRIS_H

// invscalris

#define bli_sinvscalris( ar, ai, xr, xi ) \
{ \
	(xr) /= (ar); \
}

#define bli_dinvscalris( ar, ai, xr, xi ) \
{ \
	(xr) /= (ar); \
}

#define bli_cinvscalris( ar, ai, xr, xi ) \
{ \
	float  s     = bli_fmaxabs( (ar), (ai) ); \
	float  ar_s  = (ar) / s; \
	float  ai_s  = (ai) / s; \
	float  xrt   = (xr); \
	float  temp  = ( ar_s * (ar) + ai_s * (ai) ); \
	(xr)         = ( (xrt) * ar_s + (xi)  * ai_s ) / temp; \
	(xi)         = ( (xi)  * ar_s - (xrt) * ai_s ) / temp; \
}

#define bli_zinvscalris( ar, ai, xr, xi ) \
{ \
	double s     = bli_fmaxabs( (ar), (ai) ); \
	double ar_s  = (ar) / s; \
	double ai_s  = (ai) / s; \
	double xrt   = (xr); \
	double temp  = ( ar_s * (ar) + ai_s * (ai) ); \
	(xr)         = ( (xrt) * ar_s + (xi)  * ai_s ) / temp; \
	(xi)         = ( (xi)  * ar_s - (xrt) * ai_s ) / temp; \
}

#define bli_scinvscalris( ar, ai, xr, xi ) \
{ \
	(xr) /= (ar); \
	(xi) /= (ar); \
}

#define bli_dzinvscalris( ar, ai, xr, xi ) \
{ \
	(xr) /= (ar); \
	(xi) /= (ar); \
}

#endif
// end bli_invscalris.h
#line 97 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_invscaljris.h
#line 1 "./frame/include/level0/ri//bli_invscaljris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_INVSCALJRIS_H
#define BLIS_INVSCALJRIS_H

// invscaljris

#define bli_sinvscaljris( ar, ai, xr, xi )  bli_sinvscalris( (ar), -(ai), (xr), (xi) )
#define bli_dinvscaljris( ar, ai, xr, xi )  bli_dinvscalris( (ar), -(ai), (xr), (xi) )
#define bli_cinvscaljris( ar, ai, xr, xi )  bli_cinvscalris( (ar), -(ai), (xr), (xi) )
#define bli_zinvscaljris( ar, ai, xr, xi )  bli_zinvscalris( (ar), -(ai), (xr), (xi) )

#define bli_scinvscaljris( ar, ai, xr, xi )  bli_scinvscalris( (ar), -(ai), (xr), (xi) )
#define bli_dzinvscaljris( ar, ai, xr, xi )  bli_dzinvscalris( (ar), -(ai), (xr), (xi) )

#endif

// end bli_invscaljris.h
#line 98 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_neg2ris.h
#line 1 "./frame/include/level0/ri//bli_neg2ris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_NEG2RIS_H
#define BLIS_NEG2RIS_H

// neg2ris

#define bli_sneg2ris( ar, ai, br, bi ) \
{ \
	(br) = -(ar); \
}

#define bli_dneg2ris( ar, ai, br, bi ) \
{ \
	(br) = -(ar); \
}

#define bli_cneg2ris( ar, ai, br, bi ) \
{ \
	(br) = -(ar); \
	(bi) = -(ai); \
}

#define bli_zneg2ris( ar, ai, br, bi ) \
{ \
	(br) = -(ar); \
	(bi) = -(ai); \
}

#endif

// end bli_neg2ris.h
#line 100 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_scalris.h
#line 1 "./frame/include/level0/ri//bli_scalris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SCALRIS_H
#define BLIS_SCALRIS_H

// scalris

#define bli_sscalris( ar, ai, xr, xi ) \
{ \
	(xr)      = (ar) * (xr); \
}

#define bli_dscalris( ar, ai, xr, xi ) \
{ \
	(xr)      = (ar) * (xr); \
}

#define bli_cscalris( ar, ai, xr, xi ) \
{ \
	float  yr = (ar) * (xr) - (ai) * (xi); \
	float  yi = (ai) * (xr) + (ar) * (xi); \
	(xr) = yr; \
	(xi) = yi; \
}

#define bli_zscalris( ar, ai, xr, xi ) \
{ \
	double yr = (ar) * (xr) - (ai) * (xi); \
	double yi = (ai) * (xr) + (ar) * (xi); \
	(xr) = yr; \
	(xi) = yi; \
}

#define bli_scscalris( ar, ai, xr, xi ) \
{ \
	(xr)      = (ar) * (xr); \
	(xi)      = (ar) * (xi); \
}

#define bli_dzscalris( ar, ai, xr, xi ) \
{ \
	(xr)      = (ar) * (xr); \
	(xi)      = (ar) * (xi); \
}

#endif

// end bli_scalris.h
#line 102 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_scaljris.h
#line 1 "./frame/include/level0/ri//bli_scaljris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SCALJRIS_H
#define BLIS_SCALJRIS_H

// scaljris

#define bli_sscaljris( ar, ai, xr, xi )  bli_sscalris( (ar), -(ai), (xr), (xi) )
#define bli_dscaljris( ar, ai, xr, xi )  bli_dscalris( (ar), -(ai), (xr), (xi) )
#define bli_cscaljris( ar, ai, xr, xi )  bli_cscalris( (ar), -(ai), (xr), (xi) )
#define bli_zscaljris( ar, ai, xr, xi )  bli_zscalris( (ar), -(ai), (xr), (xi) )

#define bli_scscaljris( ar, ai, xr, xi )  bli_scscalris( (ar), -(ai), (xr), (xi) )
#define bli_dzscaljris( ar, ai, xr, xi )  bli_dzscalris( (ar), -(ai), (xr), (xi) )

#endif

// end bli_scaljris.h
#line 103 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_scalcjris.h
#line 1 "./frame/include/level0/ri//bli_scalcjris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SCALCJRIS_H
#define BLIS_SCALCJRIS_H

// scalcjris

#define bli_sscalcjris( conj, ar, ai, xr, xi ) \
{ \
	bli_sscalris( (ar), (ai), (xr), (xi) ); \
}

#define bli_dscalcjris( conj, ar, ai, xr, xi ) \
{ \
	bli_dscalris( (ar), (ai), (xr), (xi) ); \
}

#define bli_cscalcjris( conj, ar, ai, xr, xi ) \
{ \
	if ( bli_is_conj( conj ) ) { bli_cscaljris( (ar), (ai), (xr), (xi) ); } \
	else                       { bli_cscalris(  (ar), (ai), (xr), (xi) ); } \
}

#define bli_zscalcjris( conj, ar, ai, xr, xi ) \
{ \
	if ( bli_is_conj( conj ) ) { bli_zscaljris( (ar), (ai), (xr), (xi) ); } \
	else                       { bli_zscalris(  (ar), (ai), (xr), (xi) ); } \
}

#define bli_iscalcjris( conj, ar, ai, xr, xi ) \
{ \
	bli_iscalris( (ar), (xi), (xr), (xi) ); \
}

#define bli_scscalcjris( conj, ar, ai, xr, xi ) \
{ \
	bli_scscalris( (ar), (ai), (xr), (xi) ); \
}

#define bli_dzscalcjris( conj, ar, ai, xr, xi ) \
{ \
	bli_dzscalris( (ar), (ai), (xr), (xi) ); \
}

#endif
// end bli_scalcjris.h
#line 104 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_scal2ris.h
#line 1 "./frame/include/level0/ri//bli_scal2ris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SCAL2RIS_H
#define BLIS_SCAL2RIS_H

// scal2ris

#define bli_rxscal2ris( ar, ai, xr, xi, yr, yi ) \
{ \
	(yr) = (ar) * (xr); \
    ( void )ai; ( void )xi; ( void )yi; \
}

#define bli_cxscal2ris( ar, ai, xr, xi, yr, yi ) \
{ \
	(yr) = (ar) * (xr) - (ai) * (xi); \
	(yi) = (ai) * (xr) + (ar) * (xi); \
}

#define bli_roscal2ris( ar, ai, xr, xi, yr, yi ) \
{ \
	(yr) = (ar) * (xr) - (ai) * (xi); \
    ( void )yi; \
}

#define bli_crscal2ris( ar, ai, xr, xi, yr, yi ) \
{ \
	(yr) = (ar) * (xr); \
	(yi) = (ar) * (xi); \
    ( void )ai; \
}

#define bli_rcscal2ris( ar, ai, xr, xi, yr, yi ) \
{ \
	(yr) = (ar) * (xr); \
	(yi) = (ai) * (xr); \
    ( void )xi; \
}

// Notes:
// - The first char encodes the type of a.
// - The second char encodes the type of x.
// - The third char encodes the type of y.

// -- (axy) = (??s) ------------------------------------------------------------

#define bli_sssscal2ris  bli_rxscal2ris
#define bli_dssscal2ris  bli_rxscal2ris
#define bli_cssscal2ris  bli_rxscal2ris
#define bli_zssscal2ris  bli_rxscal2ris

#define bli_sdsscal2ris  bli_rxscal2ris
#define bli_ddsscal2ris  bli_rxscal2ris
#define bli_cdsscal2ris  bli_rxscal2ris
#define bli_zdsscal2ris  bli_rxscal2ris

#define bli_scsscal2ris  bli_rxscal2ris
#define bli_dcsscal2ris  bli_rxscal2ris
#define bli_ccsscal2ris  bli_roscal2ris
#define bli_zcsscal2ris  bli_roscal2ris

#define bli_szsscal2ris  bli_rxscal2ris
#define bli_dzsscal2ris  bli_rxscal2ris
#define bli_czsscal2ris  bli_roscal2ris
#define bli_zzsscal2ris  bli_roscal2ris

// -- (axy) = (??d) ------------------------------------------------------------

#define bli_ssdscal2ris  bli_rxscal2ris
#define bli_dsdscal2ris  bli_rxscal2ris
#define bli_csdscal2ris  bli_rxscal2ris
#define bli_zsdscal2ris  bli_rxscal2ris

#define bli_sddscal2ris  bli_rxscal2ris
#define bli_dddscal2ris  bli_rxscal2ris
#define bli_cddscal2ris  bli_rxscal2ris
#define bli_zddscal2ris  bli_rxscal2ris

#define bli_scdscal2ris  bli_rxscal2ris
#define bli_dcdscal2ris  bli_rxscal2ris
#define bli_ccdscal2ris  bli_roscal2ris
#define bli_zcdscal2ris  bli_roscal2ris

#define bli_szdscal2ris  bli_rxscal2ris
#define bli_dzdscal2ris  bli_rxscal2ris
#define bli_czdscal2ris  bli_roscal2ris
#define bli_zzdscal2ris  bli_roscal2ris

// -- (axy) = (??c) ------------------------------------------------------------

#define bli_sscscal2ris  bli_rxscal2ris
#define bli_dscscal2ris  bli_rxscal2ris
#define bli_cscscal2ris  bli_rcscal2ris
#define bli_zscscal2ris  bli_rcscal2ris

#define bli_sdcscal2ris  bli_rxscal2ris
#define bli_ddcscal2ris  bli_rxscal2ris
#define bli_cdcscal2ris  bli_rcscal2ris
#define bli_zdcscal2ris  bli_rcscal2ris

#define bli_sccscal2ris  bli_crscal2ris
#define bli_dccscal2ris  bli_crscal2ris
#define bli_cccscal2ris  bli_cxscal2ris
#define bli_zccscal2ris  bli_cxscal2ris

#define bli_szcscal2ris  bli_crscal2ris
#define bli_dzcscal2ris  bli_crscal2ris
#define bli_czcscal2ris  bli_cxscal2ris
#define bli_zzcscal2ris  bli_cxscal2ris

// -- (axy) = (??z) ------------------------------------------------------------

#define bli_sszscal2ris  bli_rxscal2ris
#define bli_dszscal2ris  bli_rxscal2ris
#define bli_cszscal2ris  bli_rcscal2ris
#define bli_zszscal2ris  bli_rcscal2ris

#define bli_sdzscal2ris  bli_rxscal2ris
#define bli_ddzscal2ris  bli_rxscal2ris
#define bli_cdzscal2ris  bli_rcscal2ris
#define bli_zdzscal2ris  bli_rcscal2ris

#define bli_sczscal2ris  bli_crscal2ris
#define bli_dczscal2ris  bli_crscal2ris
#define bli_cczscal2ris  bli_cxscal2ris
#define bli_zczscal2ris  bli_cxscal2ris

#define bli_szzscal2ris  bli_crscal2ris
#define bli_dzzscal2ris  bli_crscal2ris
#define bli_czzscal2ris  bli_cxscal2ris
#define bli_zzzscal2ris  bli_cxscal2ris



#define bli_sscal2ris    bli_sssscal2ris
#define bli_dscal2ris    bli_dddscal2ris
#define bli_cscal2ris    bli_cccscal2ris
#define bli_zscal2ris    bli_zzzscal2ris

#endif

// end bli_scal2ris.h
#line 106 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_scal2jris.h
#line 1 "./frame/include/level0/ri//bli_scal2jris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SCAL2JRIS_H
#define BLIS_SCAL2JRIS_H

// scal2jris

#define bli_rxscal2jris( ar, ai, xr, xi, yr, yi ) \
{ \
	(yr) = (ar) * (xr); \
    ( void )ai; ( void )xi; ( void )yi; \
}

#define bli_cxscal2jris( ar, ai, xr, xi, yr, yi ) \
{ \
	(yr) = (ar) * (xr) + (ai) * (xi); \
	(yi) = (ai) * (xr) - (ar) * (xi); \
}

#define bli_roscal2jris( ar, ai, xr, xi, yr, yi ) \
{ \
	(yr) = (ar) * (xr) + (ai) * (xi); \
    ( void )yi; \
}

#define bli_crscal2jris( ar, ai, xr, xi, yr, yi ) \
{ \
	(yr) = (ar) *  (xr); \
	(yi) = (ar) * -(xi); \
    ( void )ai; \
}

#define bli_rcscal2jris( ar, ai, xr, xi, yr, yi ) \
{ \
	(yr) = (ar) * (xr); \
	(yi) = (ai) * (xr); \
    ( void )xi; \
}

// Notes:
// - The first char encodes the type of a.
// - The second char encodes the type of x.
// - The third char encodes the type of y.

// -- (axy) = (??s) ------------------------------------------------------------

#define bli_sssscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_dssscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_cssscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_zssscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )

#define bli_sdsscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_ddsscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_cdsscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_zdsscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )

#define bli_scsscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_dcsscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_ccsscal2jris( ar, ai, xr, xi, yr, yi ) bli_roscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_zcsscal2jris( ar, ai, xr, xi, yr, yi ) bli_roscal2jris( ar, ai, xr, xi, yr, yi )

#define bli_szsscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_dzsscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_czsscal2jris( ar, ai, xr, xi, yr, yi ) bli_roscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_zzsscal2jris( ar, ai, xr, xi, yr, yi ) bli_roscal2jris( ar, ai, xr, xi, yr, yi )

// -- (axy) = (??d) ------------------------------------------------------------

#define bli_ssdscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_dsdscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_csdscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_zsdscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )

#define bli_sddscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_dddscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_cddscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_zddscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )

#define bli_scdscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_dcdscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_ccdscal2jris( ar, ai, xr, xi, yr, yi ) bli_roscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_zcdscal2jris( ar, ai, xr, xi, yr, yi ) bli_roscal2jris( ar, ai, xr, xi, yr, yi )

#define bli_szdscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_dzdscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_czdscal2jris( ar, ai, xr, xi, yr, yi ) bli_roscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_zzdscal2jris( ar, ai, xr, xi, yr, yi ) bli_roscal2jris( ar, ai, xr, xi, yr, yi )

// -- (axy) = (??c) ------------------------------------------------------------

#define bli_sscscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_dscscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_cscscal2jris( ar, ai, xr, xi, yr, yi ) bli_rcscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_zscscal2jris( ar, ai, xr, xi, yr, yi ) bli_rcscal2jris( ar, ai, xr, xi, yr, yi )

#define bli_sdcscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_ddcscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_cdcscal2jris( ar, ai, xr, xi, yr, yi ) bli_rcscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_zdcscal2jris( ar, ai, xr, xi, yr, yi ) bli_rcscal2jris( ar, ai, xr, xi, yr, yi )

#define bli_sccscal2jris( ar, ai, xr, xi, yr, yi ) bli_crscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_dccscal2jris( ar, ai, xr, xi, yr, yi ) bli_crscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_cccscal2jris( ar, ai, xr, xi, yr, yi ) bli_cxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_zccscal2jris( ar, ai, xr, xi, yr, yi ) bli_cxscal2jris( ar, ai, xr, xi, yr, yi )

#define bli_szcscal2jris( ar, ai, xr, xi, yr, yi ) bli_crscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_dzcscal2jris( ar, ai, xr, xi, yr, yi ) bli_crscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_czcscal2jris( ar, ai, xr, xi, yr, yi ) bli_cxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_zzcscal2jris( ar, ai, xr, xi, yr, yi ) bli_cxscal2jris( ar, ai, xr, xi, yr, yi )

// -- (axy) = (??z) ------------------------------------------------------------

#define bli_sszscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_dszscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_cszscal2jris( ar, ai, xr, xi, yr, yi ) bli_rcscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_zszscal2jris( ar, ai, xr, xi, yr, yi ) bli_rcscal2jris( ar, ai, xr, xi, yr, yi )

#define bli_sdzscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_ddzscal2jris( ar, ai, xr, xi, yr, yi ) bli_rxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_cdzscal2jris( ar, ai, xr, xi, yr, yi ) bli_rcscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_zdzscal2jris( ar, ai, xr, xi, yr, yi ) bli_rcscal2jris( ar, ai, xr, xi, yr, yi )

#define bli_sczscal2jris( ar, ai, xr, xi, yr, yi ) bli_crscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_dczscal2jris( ar, ai, xr, xi, yr, yi ) bli_crscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_cczscal2jris( ar, ai, xr, xi, yr, yi ) bli_cxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_zczscal2jris( ar, ai, xr, xi, yr, yi ) bli_cxscal2jris( ar, ai, xr, xi, yr, yi )

#define bli_szzscal2jris( ar, ai, xr, xi, yr, yi ) bli_crscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_dzzscal2jris( ar, ai, xr, xi, yr, yi ) bli_crscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_czzscal2jris( ar, ai, xr, xi, yr, yi ) bli_cxscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_zzzscal2jris( ar, ai, xr, xi, yr, yi ) bli_cxscal2jris( ar, ai, xr, xi, yr, yi )



#define bli_sscal2jris( ar, ai, xr, xi, yr, yi ) bli_sssscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_dscal2jris( ar, ai, xr, xi, yr, yi ) bli_dddscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_cscal2jris( ar, ai, xr, xi, yr, yi ) bli_cccscal2jris( ar, ai, xr, xi, yr, yi )
#define bli_zscal2jris( ar, ai, xr, xi, yr, yi ) bli_zzzscal2jris( ar, ai, xr, xi, yr, yi )

#endif

// end bli_scal2jris.h
#line 107 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_set0ris.h
#line 1 "./frame/include/level0/ri//bli_set0ris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SET0RIS_H
#define BLIS_SET0RIS_H

// set0ris

#define bli_sset0ris( xr, xi ) bli_scopyris( 0.0F, 0.0F, xr, xi )
#define bli_dset0ris( xr, xi ) bli_dcopyris( 0.0 , 0.0 , xr, xi )
#define bli_cset0ris( xr, xi ) bli_ccopyris( 0.0F, 0.0F, xr, xi )
#define bli_zset0ris( xr, xi ) bli_zcopyris( 0.0 , 0.0 , xr, xi )

#endif

// end bli_set0ris.h
#line 109 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_sqrt2ris.h
#line 1 "./frame/include/level0/ri//bli_sqrt2ris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SQRT2RIS_H
#define BLIS_SQRT2RIS_H

// sqrt2ris

#define bli_ssqrt2ris( xr, xi, ar, ai ) \
{ \
	(ar)       = sqrtf( (xr) ); \
}

#define bli_dsqrt2ris( xr, xi, ar, ai ) \
{ \
	(ar)       = sqrt( (xr) ); \
}

#define bli_csqrt2ris( xr, xi, ar, ai ) \
{ \
	float  s   = bli_fmaxabs( (xr), (xi) ); \
	float  mag; \
	if ( s == 0.0F ) mag = 0.0F; \
	else \
	{ \
		mag = sqrtf( s ) * \
		      sqrtf( ( (xr) / s ) * (xr) + \
		             ( (xi) / s ) * (xi) ); \
	} \
\
	(ar)       = sqrtf( ( mag + (xr) ) / 2.0F ); \
	(ai)       = sqrtf( ( mag - (xi) ) / 2.0F ); \
}

#define bli_zsqrt2ris( xr, xi, ar, ai ) \
{ \
	double s   = bli_fmaxabs( (xr), (xi) ); \
	double mag; \
	if ( s == 0.0 ) mag = 0.0; \
	else \
	{ \
		mag = sqrt( s ) * \
		      sqrt( ( (xr) / s ) * (xr) + \
		            ( (xi) / s ) * (xi) ); \
	} \
\
	(ar)       = sqrt( ( mag + (xr) ) / 2.0 ); \
	(ai)       = sqrt( ( mag - (xi) ) / 2.0 ); \
}

#define bli_scsqrt2ris( xr, xi, ar, ai ) \
{ \
	(ar)       = sqrtf( (xr) ); \
	(ai)       = 0.0F; \
}

#define bli_dzsqrt2ris( xr, xi, ar, ai ) \
{ \
	(ar)       = sqrt( (xr) ); \
	(ai)       = 0.0; \
}

#endif

// end bli_sqrt2ris.h
#line 111 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_subris.h
#line 1 "./frame/include/level0/ri//bli_subris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SUBRIS_H
#define BLIS_SUBRIS_H

// subris

#define bli_ssubris( ar, ai, xr, xi ) \
{ \
	(xr) = (xr) - (ar); \
}

#define bli_dsubris( ar, ai, xr, xi ) \
{ \
	(xr) = (xr) - (ar); \
}

#define bli_csubris( ar, ai, xr, xi ) \
{ \
	(xr) = (xr) - (ar); \
	(xi) = (xi) - (ai); \
}

#define bli_zsubris( ar, ai, xr, xi ) \
{ \
	(xr) = (xr) - (ar); \
	(xi) = (xi) - (ai); \
}

#endif

// end bli_subris.h
#line 113 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_subjris.h
#line 1 "./frame/include/level0/ri//bli_subjris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SUBJRIS_H
#define BLIS_SUBJRIS_H

// subjris

#define bli_ssubjris( ar, ai, xr, xi )  bli_ssubris( (ar), -(ai), (xr), (xi) )
#define bli_dsubjris( ar, ai, xr, xi )  bli_dsubris( (ar), -(ai), (xr), (xi) )
#define bli_csubjris( ar, ai, xr, xi )  bli_csubris( (ar), -(ai), (xr), (xi) )
#define bli_zsubjris( ar, ai, xr, xi )  bli_zsubris( (ar), -(ai), (xr), (xi) )

#endif

// end bli_subjris.h
#line 114 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_swapris.h
#line 1 "./frame/include/level0/ri//bli_swapris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SWAPRIS_H
#define BLIS_SWAPRIS_H

// swapris

#define bli_sswapris( ar, ai, br, bi ) \
{ \
	float tr, ti; \
\
	bli_scopyris( (br), (bi), (tr), (ti) ); \
	bli_scopyris( (ar), (ai), (br), (bi) ); \
	bli_scopyris( (tr), (ti), (ar), (ai) ); \
}

#define bli_dswapris( ar, ai, br, bi ) \
{ \
	double tr, ti; \
\
	bli_dcopyris( (br), (bi), (tr), (ti) ); \
	bli_dcopyris( (ar), (ai), (br), (bi) ); \
	bli_dcopyris( (tr), (ti), (ar), (ai) ); \
}

#define bli_cswapris( ar, ai, br, bi ) \
{ \
	scomplex tr, ti; \
\
	bli_ccopyris( (br), (bi), (tr), (ti) ); \
	bli_ccopyris( (ar), (ai), (br), (bi) ); \
	bli_ccopyris( (tr), (ti), (ar), (ai) ); \
}

#define bli_zswapris( ar, ai, br, bi ) \
{ \
	dcomplex tr, ti; \
\
	bli_zcopyris( (br), (bi), (tr), (ti) ); \
	bli_zcopyris( (ar), (ai), (br), (bi) ); \
	bli_zcopyris( (tr), (ti), (ar), (ai) ); \
}

#endif

// end bli_swapris.h
#line 116 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_xpbyris.h
#line 1 "./frame/include/level0/ri//bli_xpbyris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_XPBYRIS_H
#define BLIS_XPBYRIS_H

// xpbyris

#define bli_rxxpbyris( xr, xi, br, bi, yr, yi ) \
{ \
	(yr) = (xr) + (br) * (yr); \
}

#define bli_cxxpbyris( xr, xi, br, bi, yr, yi ) \
{ \
	const __typeof__(yr) yt_r = (xr) + (br) * (yr) - (bi) * (yi); \
	const __typeof__(yi) yt_i = (xi) + (bi) * (yr) + (br) * (yi); \
	(yr) = yt_r; \
	(yi) = yt_i; \
}

#define bli_crxpbyris( xr, xi, br, bi, yr, yi ) \
{ \
	const __typeof__(yr) yt_r = (xr) + (br) * (yr); \
	const __typeof__(yi) yt_i = (xi) + (br) * (yi); \
	(yr) = yt_r; \
	(yi) = yt_i; \
}

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of b.
// - The third char encodes the type of y.

// -- (xby) = (??s) ------------------------------------------------------------

#define bli_sssxpbyris  bli_rxxpbyris
#define bli_dssxpbyris  bli_rxxpbyris
#define bli_cssxpbyris  bli_rxxpbyris
#define bli_zssxpbyris  bli_rxxpbyris

#define bli_sdsxpbyris  bli_rxxpbyris
#define bli_ddsxpbyris  bli_rxxpbyris
#define bli_cdsxpbyris  bli_rxxpbyris
#define bli_zdsxpbyris  bli_rxxpbyris

#define bli_scsxpbyris  bli_rxxpbyris
#define bli_dcsxpbyris  bli_rxxpbyris
#define bli_ccsxpbyris  bli_rxxpbyris
#define bli_zcsxpbyris  bli_rxxpbyris

#define bli_szsxpbyris  bli_rxxpbyris
#define bli_dzsxpbyris  bli_rxxpbyris
#define bli_czsxpbyris  bli_rxxpbyris
#define bli_zzsxpbyris  bli_rxxpbyris

// -- (xby) = (??d) ------------------------------------------------------------

#define bli_ssdxpbyris  bli_rxxpbyris
#define bli_dsdxpbyris  bli_rxxpbyris
#define bli_csdxpbyris  bli_rxxpbyris
#define bli_zsdxpbyris  bli_rxxpbyris

#define bli_sddxpbyris  bli_rxxpbyris
#define bli_dddxpbyris  bli_rxxpbyris
#define bli_cddxpbyris  bli_rxxpbyris
#define bli_zddxpbyris  bli_rxxpbyris

#define bli_scdxpbyris  bli_rxxpbyris
#define bli_dcdxpbyris  bli_rxxpbyris
#define bli_ccdxpbyris  bli_rxxpbyris
#define bli_zcdxpbyris  bli_rxxpbyris

#define bli_szdxpbyris  bli_rxxpbyris
#define bli_dzdxpbyris  bli_rxxpbyris
#define bli_czdxpbyris  bli_rxxpbyris
#define bli_zzdxpbyris  bli_rxxpbyris

// -- (xby) = (??c) ------------------------------------------------------------

#define bli_sscxpbyris  bli_rxxpbyris
#define bli_dscxpbyris  bli_rxxpbyris
#define bli_cscxpbyris  bli_crxpbyris
#define bli_zscxpbyris  bli_crxpbyris

#define bli_sdcxpbyris  bli_rxxpbyris
#define bli_ddcxpbyris  bli_rxxpbyris
#define bli_cdcxpbyris  bli_crxpbyris
#define bli_zdcxpbyris  bli_crxpbyris

#define bli_sccxpbyris  bli_cxxpbyris
#define bli_dccxpbyris  bli_cxxpbyris
#define bli_cccxpbyris  bli_cxxpbyris
#define bli_zccxpbyris  bli_cxxpbyris

#define bli_szcxpbyris  bli_cxxpbyris
#define bli_dzcxpbyris  bli_cxxpbyris
#define bli_czcxpbyris  bli_cxxpbyris
#define bli_zzcxpbyris  bli_cxxpbyris

// -- (xby) = (??z) ------------------------------------------------------------

#define bli_sszxpbyris  bli_rxxpbyris
#define bli_dszxpbyris  bli_rxxpbyris
#define bli_cszxpbyris  bli_crxpbyris
#define bli_zszxpbyris  bli_crxpbyris

#define bli_sdzxpbyris  bli_rxxpbyris
#define bli_ddzxpbyris  bli_rxxpbyris
#define bli_cdzxpbyris  bli_crxpbyris
#define bli_zdzxpbyris  bli_crxpbyris

#define bli_sczxpbyris  bli_cxxpbyris
#define bli_dczxpbyris  bli_cxxpbyris
#define bli_cczxpbyris  bli_cxxpbyris
#define bli_zczxpbyris  bli_cxxpbyris

#define bli_szzxpbyris  bli_cxxpbyris
#define bli_dzzxpbyris  bli_cxxpbyris
#define bli_czzxpbyris  bli_cxxpbyris
#define bli_zzzxpbyris  bli_cxxpbyris



#define bli_sxpbyris    bli_sssxpbyris
#define bli_dxpbyris    bli_dddxpbyris
#define bli_cxpbyris    bli_cccxpbyris
#define bli_zxpbyris    bli_zzzxpbyris

#endif

// end bli_xpbyris.h
#line 118 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_xpbyjris.h
#line 1 "./frame/include/level0/ri//bli_xpbyjris.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_XPBYJRIS_H
#define BLIS_XPBYJRIS_H

// xpbyjris

#define bli_rxxpbyjris( xr, xi, br, bi, yr, yi ) \
{ \
	(yr) = (xr) + (br) * (yr); \
}

#define bli_cxxpbyjris( xr, xi, br, bi, yr, yi ) \
{ \
	const __typeof__(yr) yt_r =  (xr) + (br) * (yr) - (bi) * (yi); \
	const __typeof__(yi) yt_i = -(xi) + (bi) * (yr) + (br) * (yi); \
	(yr) = yt_r; \
	(yi) = yt_i; \
}

#define bli_crxpbyjris( xr, xi, br, bi, yr, yi ) \
{ \
	const __typeof__(yr) yt_r =  (xr) + (br) * (yr); \
	const __typeof__(yi) yt_i = -(xi) + (br) * (yi); \
	(yr) = yt_r; \
	(yi) = yt_i; \
}

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of b.
// - The third char encodes the type of y.

// -- (xby) = (??s) ------------------------------------------------------------

#define bli_sssxpbyjris  bli_rxxpbyjris
#define bli_dssxpbyjris  bli_rxxpbyjris
#define bli_cssxpbyjris  bli_rxxpbyjris
#define bli_zssxpbyjris  bli_rxxpbyjris

#define bli_sdsxpbyjris  bli_rxxpbyjris
#define bli_ddsxpbyjris  bli_rxxpbyjris
#define bli_cdsxpbyjris  bli_rxxpbyjris
#define bli_zdsxpbyjris  bli_rxxpbyjris

#define bli_scsxpbyjris  bli_rxxpbyjris
#define bli_dcsxpbyjris  bli_rxxpbyjris
#define bli_ccsxpbyjris  bli_rxxpbyjris
#define bli_zcsxpbyjris  bli_rxxpbyjris

#define bli_szsxpbyjris  bli_rxxpbyjris
#define bli_dzsxpbyjris  bli_rxxpbyjris
#define bli_czsxpbyjris  bli_rxxpbyjris
#define bli_zzsxpbyjris  bli_rxxpbyjris

// -- (xby) = (??d) ------------------------------------------------------------

#define bli_ssdxpbyjris  bli_rxxpbyjris
#define bli_dsdxpbyjris  bli_rxxpbyjris
#define bli_csdxpbyjris  bli_rxxpbyjris
#define bli_zsdxpbyjris  bli_rxxpbyjris

#define bli_sddxpbyjris  bli_rxxpbyjris
#define bli_dddxpbyjris  bli_rxxpbyjris
#define bli_cddxpbyjris  bli_rxxpbyjris
#define bli_zddxpbyjris  bli_rxxpbyjris

#define bli_scdxpbyjris  bli_rxxpbyjris
#define bli_dcdxpbyjris  bli_rxxpbyjris
#define bli_ccdxpbyjris  bli_rxxpbyjris
#define bli_zcdxpbyjris  bli_rxxpbyjris

#define bli_szdxpbyjris  bli_rxxpbyjris
#define bli_dzdxpbyjris  bli_rxxpbyjris
#define bli_czdxpbyjris  bli_rxxpbyjris
#define bli_zzdxpbyjris  bli_rxxpbyjris

// -- (xby) = (??c) ------------------------------------------------------------

#define bli_sscxpbyjris  bli_rxxpbyjris
#define bli_dscxpbyjris  bli_rxxpbyjris
#define bli_cscxpbyjris  bli_crxpbyjris
#define bli_zscxpbyjris  bli_crxpbyjris

#define bli_sdcxpbyjris  bli_rxxpbyjris
#define bli_ddcxpbyjris  bli_rxxpbyjris
#define bli_cdcxpbyjris  bli_crxpbyjris
#define bli_zdcxpbyjris  bli_crxpbyjris

#define bli_sccxpbyjris  bli_cxxpbyjris
#define bli_dccxpbyjris  bli_cxxpbyjris
#define bli_cccxpbyjris  bli_cxxpbyjris
#define bli_zccxpbyjris  bli_cxxpbyjris

#define bli_szcxpbyjris  bli_cxxpbyjris
#define bli_dzcxpbyjris  bli_cxxpbyjris
#define bli_czcxpbyjris  bli_cxxpbyjris
#define bli_zzcxpbyjris  bli_cxxpbyjris

// -- (xby) = (??z) ------------------------------------------------------------

#define bli_sszxpbyjris  bli_rxxpbyjris
#define bli_dszxpbyjris  bli_rxxpbyjris
#define bli_cszxpbyjris  bli_crxpbyjris
#define bli_zszxpbyjris  bli_crxpbyjris

#define bli_sdzxpbyjris  bli_rxxpbyjris
#define bli_ddzxpbyjris  bli_rxxpbyjris
#define bli_cdzxpbyjris  bli_crxpbyjris
#define bli_zdzxpbyjris  bli_crxpbyjris

#define bli_sczxpbyjris  bli_cxxpbyjris
#define bli_dczxpbyjris  bli_cxxpbyjris
#define bli_cczxpbyjris  bli_cxxpbyjris
#define bli_zczxpbyjris  bli_cxxpbyjris

#define bli_szzxpbyjris  bli_cxxpbyjris
#define bli_dzzxpbyjris  bli_cxxpbyjris
#define bli_czzxpbyjris  bli_cxxpbyjris
#define bli_zzzxpbyjris  bli_cxxpbyjris



#define bli_sxpbyjris    bli_sssxpbyjris
#define bli_dxpbyjris    bli_dddxpbyjris
#define bli_cxpbyjris    bli_cccxpbyjris
#define bli_zxpbyjris    bli_zzzxpbyjris

#endif

// end bli_xpbyjris.h
#line 119 "./frame/include//bli_scalar_macro_defs.h"


// Inlined scalar macros in loops
// begin bli_scal2ris_mxn.h
#line 1 "./frame/include/level0/ri//bli_scal2ris_mxn.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SCAL2RIS_MXN_H
#define BLIS_SCAL2RIS_MXN_H

// scal2ris_mxn

BLIS_INLINE void bli_cscal2ris_mxn
     (
       const conj_t       conjx,
       const dim_t        m,
       const dim_t        n,
       scomplex* restrict alpha,
       scomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
       scomplex* restrict y, const inc_t rs_y, const inc_t cs_y, const inc_t is_y
     )
{
	float*  restrict alpha_r = ( float*  )alpha; \
	float*  restrict alpha_i = ( float*  )alpha + 1; \
	float*  restrict x_r     = ( float*  )x; \
	float*  restrict x_i     = ( float*  )x + 1; \
	float*  restrict y_r     = ( float*  )y; \
	float*  restrict y_i     = ( float*  )y + is_y; \
	const dim_t      incx2   = 2*rs_x; \
	const dim_t      ldx2    = 2*cs_x; \

	/* Treat the micro-panel as panel_dim x panel_len and column-stored
	   (unit row stride). */ \

	if ( bli_is_conj( conjx ) )
	{
		for ( dim_t j = 0; j < n; ++j )
		for ( dim_t i = 0; i < m; ++i )
		{
			float*  restrict chi11_r = x_r + (i  )*incx2 + (j  )*ldx2;
			float*  restrict chi11_i = x_i + (i  )*incx2 + (j  )*ldx2;
			float*  restrict psi11_r = y_r + (i  )*1     + (j  )*cs_y;
			float*  restrict psi11_i = y_i + (i  )*1     + (j  )*cs_y;

			bli_cscal2jris
			(
			  *alpha_r,
			  *alpha_i,
			  *chi11_r,
			  *chi11_i,
			  *psi11_r,
			  *psi11_i
			);
		}
	}
	else /* if ( bli_is_noconj( conjx ) ) */
	{
		for ( dim_t j = 0; j < n; ++j )
		for ( dim_t i = 0; i < m; ++i )
		{
			float*  restrict chi11_r = x_r + (i  )*incx2 + (j  )*ldx2;
			float*  restrict chi11_i = x_i + (i  )*incx2 + (j  )*ldx2;
			float*  restrict psi11_r = y_r + (i  )*1     + (j  )*cs_y;
			float*  restrict psi11_i = y_i + (i  )*1     + (j  )*cs_y;

			bli_cscal2ris
			(
			  *alpha_r,
			  *alpha_i,
			  *chi11_r,
			  *chi11_i,
			  *psi11_r,
			  *psi11_i
			);
		}
	}
}

BLIS_INLINE void bli_zscal2ris_mxn
     (
       const conj_t       conjx,
       const dim_t        m,
       const dim_t        n,
       dcomplex* restrict alpha,
       dcomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
       dcomplex* restrict y, const inc_t rs_y, const inc_t cs_y, const inc_t is_y
     )
{
	double* restrict alpha_r = ( double* )alpha; \
	double* restrict alpha_i = ( double* )alpha + 1; \
	double* restrict x_r     = ( double* )x; \
	double* restrict x_i     = ( double* )x + 1; \
	double* restrict y_r     = ( double* )y; \
	double* restrict y_i     = ( double* )y + is_y; \
	const dim_t      incx2   = 2*rs_x; \
	const dim_t      ldx2    = 2*cs_x; \

	/* Treat the micro-panel as panel_dim x panel_len and column-stored
	   (unit row stride). */ \

	if ( bli_is_conj( conjx ) )
	{
		for ( dim_t j = 0; j < n; ++j )
		for ( dim_t i = 0; i < m; ++i )
		{
			double* restrict chi11_r = x_r + (i  )*incx2 + (j  )*ldx2;
			double* restrict chi11_i = x_i + (i  )*incx2 + (j  )*ldx2;
			double* restrict psi11_r = y_r + (i  )*1     + (j  )*cs_y;
			double* restrict psi11_i = y_i + (i  )*1     + (j  )*cs_y;

			bli_zscal2jris
			(
			  *alpha_r,
			  *alpha_i,
			  *chi11_r,
			  *chi11_i,
			  *psi11_r,
			  *psi11_i
			);
		}
	}
	else /* if ( bli_is_noconj( conjx ) ) */
	{
		for ( dim_t j = 0; j < n; ++j )
		for ( dim_t i = 0; i < m; ++i )
		{
			double* restrict chi11_r = x_r + (i  )*incx2 + (j  )*ldx2;
			double* restrict chi11_i = x_i + (i  )*incx2 + (j  )*ldx2;
			double* restrict psi11_r = y_r + (i  )*1     + (j  )*cs_y;
			double* restrict psi11_i = y_i + (i  )*1     + (j  )*cs_y;

			bli_zscal2ris
			(
			  *alpha_r,
			  *alpha_i,
			  *chi11_r,
			  *chi11_i,
			  *psi11_r,
			  *psi11_i
			);
		}
	}
}


#endif
// end bli_scal2ris_mxn.h
#line 122 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_scalris_mxn_uplo.h
#line 1 "./frame/include/level0/ri//bli_scalris_mxn_uplo.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SCALRIS_MXN_UPLO_H
#define BLIS_SCALRIS_MXN_UPLO_H

// scalris_mxn_u

#define bli_cscalris_mxn_u( diagoff, m, n, ar, ai, xr, xi, rs_x, cs_x ) \
{ \
	dim_t _i, _j; \
\
	for ( _j = 0; _j < n; ++_j ) \
	for ( _i = 0; _i < m; ++_i ) \
	{ \
		if ( (doff_t)_j - (doff_t)_i >= diagoff ) \
		{ \
			bli_cscalris( *(ar), \
			              *(ai), \
			              *((xr) + _i*rs_x + _j*cs_x), \
			              *((xi) + _i*rs_x + _j*cs_x) ); \
		} \
	} \
}

#define bli_zscalris_mxn_u( diagoff, m, n, ar, ai, xr, xi, rs_x, cs_x ) \
{ \
	dim_t _i, _j; \
\
	for ( _j = 0; _j < n; ++_j ) \
	for ( _i = 0; _i < m; ++_i ) \
	{ \
		if ( (doff_t)_j - (doff_t)_i >= diagoff ) \
		{ \
			bli_zscalris( *(ar), \
			              *(ai), \
			              *((xr) + _i*rs_x + _j*cs_x), \
			              *((xi) + _i*rs_x + _j*cs_x) ); \
		} \
	} \
}

// scalris_mxn_l

#define bli_cscalris_mxn_l( diagoff, m, n, ar, ai, xr, xi, rs_x, cs_x ) \
{ \
	dim_t _i, _j; \
\
	for ( _j = 0; _j < n; ++_j ) \
	for ( _i = 0; _i < m; ++_i ) \
	{ \
		if ( (doff_t)_j - (doff_t)_i <= diagoff ) \
		{ \
			bli_cscalris( *(ar), \
			              *(ai), \
			              *((xr) + _i*rs_x + _j*cs_x), \
			              *((xi) + _i*rs_x + _j*cs_x) ); \
		} \
	} \
}

#define bli_zscalris_mxn_l( diagoff, m, n, ar, ai, xr, xi, rs_x, cs_x ) \
{ \
	dim_t _i, _j; \
\
	for ( _j = 0; _j < n; ++_j ) \
	for ( _i = 0; _i < m; ++_i ) \
	{ \
		if ( (doff_t)_j - (doff_t)_i <= diagoff ) \
		{ \
			bli_zscalris( *(ar), \
			              *(ai), \
			              *((xr) + _i*rs_x + _j*cs_x), \
			              *((xi) + _i*rs_x + _j*cs_x) ); \
		} \
	} \
}

#endif
// end bli_scalris_mxn_uplo.h
#line 123 "./frame/include//bli_scalar_macro_defs.h"



// -- Conventional scalar macros (paired real/imaginary values) --

// begin bli_absq2s.h
#line 1 "./frame/include/level0//bli_absq2s.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_ABSQR2_H
#define BLIS_ABSQR2_H

// absq2s

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of a.

#define bli_ssabsq2s( x, a )              bli_sabsq2ris( bli_sreal(x), bli_simag(x), bli_sreal(a), 0.0F         )
#define bli_dsabsq2s( x, a )              bli_dabsq2ris( bli_dreal(x), bli_dimag(x), bli_sreal(a), 0.0F         )
#define bli_csabsq2s( x, a ) { float ti;  bli_cabsq2ris( bli_creal(x), bli_cimag(x), bli_sreal(a), ti           ); ( void )ti; }
#define bli_zsabsq2s( x, a ) { float ti;  bli_zabsq2ris( bli_zreal(x), bli_zimag(x), bli_sreal(a), ti           ); ( void )ti; }

#define bli_sdabsq2s( x, a )              bli_sabsq2ris( bli_sreal(x), bli_simag(x), bli_dreal(a), 0.0          )
#define bli_ddabsq2s( x, a )              bli_dabsq2ris( bli_dreal(x), bli_dimag(x), bli_dreal(a), 0.0          )
#define bli_cdabsq2s( x, a ) { double ti; bli_cabsq2ris( bli_creal(x), bli_cimag(x), bli_dreal(a), ti           ); ( void )ti; }
#define bli_zdabsq2s( x, a ) { double ti; bli_zabsq2ris( bli_zreal(x), bli_zimag(x), bli_dreal(a), ti           ); ( void )ti; }

#ifndef BLIS_ENABLE_C99_COMPLEX

#define bli_scabsq2s( x, a )              bli_sabsq2ris( bli_sreal(x), bli_simag(x), bli_creal(a), bli_cimag(a) )
#define bli_dcabsq2s( x, a )              bli_dabsq2ris( bli_dreal(x), bli_dimag(x), bli_creal(a), bli_cimag(a) )
#define bli_ccabsq2s( x, a )              bli_cabsq2ris( bli_creal(x), bli_cimag(x), bli_creal(a), bli_cimag(a) )
#define bli_zcabsq2s( x, a )              bli_zabsq2ris( bli_zreal(x), bli_zimag(x), bli_creal(a), bli_cimag(a) )

#define bli_szabsq2s( x, a )              bli_sabsq2ris( bli_sreal(x), bli_simag(x), bli_zreal(a), bli_zimag(a) )
#define bli_dzabsq2s( x, a )              bli_dabsq2ris( bli_dreal(x), bli_dimag(x), bli_zreal(a), bli_zimag(a) )
#define bli_czabsq2s( x, a )              bli_cabsq2ris( bli_creal(x), bli_cimag(x), bli_zreal(a), bli_zimag(a) )
#define bli_zzabsq2s( x, a )              bli_zabsq2ris( bli_zreal(x), bli_zimag(x), bli_zreal(a), bli_zimag(a) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

#define bli_scabsq2s( x, a )  bli_scsets(          (x) *          (x), 0.0, (a) )
#define bli_dcabsq2s( x, a )  bli_dcsets(          (x) *          (x), 0.0, (a) )
#define bli_ccabsq2s( x, a )  bli_ccsets( bli_creal(x) * bli_creal(x) + \
                                          bli_cimag(x) * bli_cimag(x), 0.0, (a) )
#define bli_zcabsq2s( x, a )  bli_zcsets( bli_zreal(x) * bli_zreal(x) + \
                                          bli_zimag(x) * bli_zimag(x), 0.0, (a) )

#define bli_szabsq2s( x, a )  bli_szsets(          (x) *          (x), 0.0, (a) )
#define bli_dzabsq2s( x, a )  bli_dzsets(          (x) *          (x), 0.0, (a) )
#define bli_czabsq2s( x, a )  bli_czsets( bli_creal(x) * bli_creal(x) + \
                                          bli_cimag(x) * bli_cimag(x), 0.0, (a) )
#define bli_zzabsq2s( x, a )  bli_zzsets( bli_zreal(x) * bli_zreal(x) + \
                                          bli_zimag(x) * bli_zimag(x), 0.0, (a) )

#endif // BLIS_ENABLE_C99_COMPLEX

#define bli_sabsq2s( x, a )  bli_ssabsq2s( x, a )
#define bli_dabsq2s( x, a )  bli_ddabsq2s( x, a )
#define bli_cabsq2s( x, a )  bli_ccabsq2s( x, a )
#define bli_zabsq2s( x, a )  bli_zzabsq2s( x, a )


#endif
// end bli_absq2s.h
#line 128 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_abval2s.h
#line 1 "./frame/include/level0//bli_abval2s.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_ABVAL2S_H
#define BLIS_ABVAL2S_H

// abval2s

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of a.

#ifndef BLIS_ENABLE_C99_COMPLEX

#define bli_ssabval2s( x, a )              bli_sabval2ris( bli_sreal(x), bli_simag(x), bli_sreal(a), 0.0F         )
#define bli_dsabval2s( x, a )              bli_dabval2ris( bli_dreal(x), bli_dimag(x), bli_sreal(a), 0.0F         )
#define bli_csabval2s( x, a ) { float ti;  bli_cabval2ris( bli_creal(x), bli_cimag(x), bli_sreal(a), ti           ); ( void )ti; }
#define bli_zsabval2s( x, a ) { float ti;  bli_zabval2ris( bli_zreal(x), bli_zimag(x), bli_sreal(a), ti           ); ( void )ti; }

#define bli_sdabval2s( x, a )              bli_sabval2ris( bli_sreal(x), bli_simag(x), bli_dreal(a), 0.0          )
#define bli_ddabval2s( x, a )              bli_dabval2ris( bli_dreal(x), bli_dimag(x), bli_dreal(a), 0.0          )
#define bli_cdabval2s( x, a ) { double ti; bli_cabval2ris( bli_creal(x), bli_cimag(x), bli_dreal(a), ti           ); ( void )ti; }
#define bli_zdabval2s( x, a ) { double ti; bli_zabval2ris( bli_zreal(x), bli_zimag(x), bli_dreal(a), ti           ); ( void )ti; }

#define bli_scabval2s( x, a )              bli_sabval2ris( bli_sreal(x), bli_simag(x), bli_creal(a), bli_cimag(a) )
#define bli_dcabval2s( x, a )              bli_dabval2ris( bli_dreal(x), bli_dimag(x), bli_creal(a), bli_cimag(a) )
#define bli_ccabval2s( x, a )              bli_cabval2ris( bli_creal(x), bli_cimag(x), bli_creal(a), bli_cimag(a) )
#define bli_zcabval2s( x, a )              bli_zabval2ris( bli_zreal(x), bli_zimag(x), bli_creal(a), bli_cimag(a) )

#define bli_szabval2s( x, a )              bli_sabval2ris( bli_sreal(x), bli_simag(x), bli_zreal(a), bli_zimag(a) )
#define bli_dzabval2s( x, a )              bli_dabval2ris( bli_dreal(x), bli_dimag(x), bli_zreal(a), bli_zimag(a) )
#define bli_czabval2s( x, a )              bli_cabval2ris( bli_creal(x), bli_cimag(x), bli_zreal(a), bli_zimag(a) )
#define bli_zzabval2s( x, a )              bli_zabval2ris( bli_zreal(x), bli_zimag(x), bli_zreal(a), bli_zimag(a) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

#define bli_ssabval2s( x, a )  bli_sssets( fabsf(x), 0.0, (a) )
#define bli_dsabval2s( x, a )  bli_dssets( fabs (x), 0.0, (a) )
#define bli_csabval2s( x, a )  bli_cssets( cabsf(x), 0.0, (a) )
#define bli_zsabval2s( x, a )  bli_zssets( cabs (x), 0.0, (a) )

#define bli_sdabval2s( x, a )  bli_sdsets( fabsf(x), 0.0, (a) )
#define bli_ddabval2s( x, a )  bli_ddsets( fabs (x), 0.0, (a) )
#define bli_cdabval2s( x, a )  bli_cdsets( cabsf(x), 0.0, (a) )
#define bli_zdabval2s( x, a )  bli_zdsets( cabs (x), 0.0, (a) )

#define bli_scabval2s( x, a )  bli_scsets( fabsf(x), 0.0, (a) )
#define bli_dcabval2s( x, a )  bli_dcsets( fabs (x), 0.0, (a) )
#define bli_ccabval2s( x, a )  bli_ccsets( cabsf(x), 0.0, (a) )
#define bli_zcabval2s( x, a )  bli_zcsets( cabs (x), 0.0, (a) )

#define bli_szabval2s( x, a )  bli_szsets( fabsf(x), 0.0, (a) )
#define bli_dzabval2s( x, a )  bli_dzsets( fabs (x), 0.0, (a) )
#define bli_czabval2s( x, a )  bli_czsets( cabsf(x), 0.0, (a) )
#define bli_zzabval2s( x, a )  bli_zzsets( cabs (x), 0.0, (a) )

#endif // BLIS_ENABLE_C99_COMPLEX


#define bli_sabval2s( x, a )  bli_ssabval2s( x, a )
#define bli_dabval2s( x, a )  bli_ddabval2s( x, a )
#define bli_cabval2s( x, a )  bli_ccabval2s( x, a )
#define bli_zabval2s( x, a )  bli_zzabval2s( x, a )


#endif
// end bli_abval2s.h
#line 130 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_adds.h
#line 1 "./frame/include/level0//bli_adds.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_ADDS_H
#define BLIS_ADDS_H

// adds

// Notes:
// - The first char encodes the type of a.
// - The second char encodes the type of y.

#define bli_ssadds( a, y )  bli_saddris( bli_sreal(a), bli_simag(a), bli_sreal(y), bli_simag(y) )
#define bli_dsadds( a, y )  bli_saddris( bli_dreal(a), bli_dimag(a), bli_sreal(y), bli_simag(y) )
#define bli_csadds( a, y )  bli_saddris( bli_creal(a), bli_cimag(a), bli_sreal(y), bli_simag(y) )
#define bli_zsadds( a, y )  bli_saddris( bli_zreal(a), bli_zimag(a), bli_sreal(y), bli_simag(y) )

#define bli_sdadds( a, y )  bli_daddris( bli_sreal(a), bli_simag(a), bli_dreal(y), bli_dimag(y) )
#define bli_ddadds( a, y )  bli_daddris( bli_dreal(a), bli_dimag(a), bli_dreal(y), bli_dimag(y) )
#define bli_cdadds( a, y )  bli_daddris( bli_creal(a), bli_cimag(a), bli_dreal(y), bli_dimag(y) )
#define bli_zdadds( a, y )  bli_daddris( bli_zreal(a), bli_zimag(a), bli_dreal(y), bli_dimag(y) )

#ifndef BLIS_ENABLE_C99_COMPLEX

#define bli_scadds( a, y )  bli_caddris( bli_sreal(a), bli_simag(a), bli_creal(y), bli_cimag(y) )
#define bli_dcadds( a, y )  bli_caddris( bli_dreal(a), bli_dimag(a), bli_creal(y), bli_cimag(y) )
#define bli_ccadds( a, y )  bli_caddris( bli_creal(a), bli_cimag(a), bli_creal(y), bli_cimag(y) )
#define bli_zcadds( a, y )  bli_caddris( bli_zreal(a), bli_zimag(a), bli_creal(y), bli_cimag(y) )

#define bli_szadds( a, y )  bli_zaddris( bli_sreal(a), bli_simag(a), bli_zreal(y), bli_zimag(y) )
#define bli_dzadds( a, y )  bli_zaddris( bli_dreal(a), bli_dimag(a), bli_zreal(y), bli_zimag(y) )
#define bli_czadds( a, y )  bli_zaddris( bli_creal(a), bli_cimag(a), bli_zreal(y), bli_zimag(y) )
#define bli_zzadds( a, y )  bli_zaddris( bli_zreal(a), bli_zimag(a), bli_zreal(y), bli_zimag(y) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

#define bli_scadds( a, y )  { (y) += (a); }
#define bli_dcadds( a, y )  { (y) += (a); }
#define bli_ccadds( a, y )  { (y) += (a); }
#define bli_zcadds( a, y )  { (y) += (a); }

#define bli_szadds( a, y )  { (y) += (a); }
#define bli_dzadds( a, y )  { (y) += (a); }
#define bli_czadds( a, y )  { (y) += (a); }
#define bli_zzadds( a, y )  { (y) += (a); }

#endif // BLIS_ENABLE_C99_COMPLEX


#define bli_sadds( a, y )  bli_ssadds( a, y )
#define bli_dadds( a, y )  bli_ddadds( a, y )
#define bli_cadds( a, y )  bli_ccadds( a, y )
#define bli_zadds( a, y )  bli_zzadds( a, y )


#endif

// end bli_adds.h
#line 132 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_addjs.h
#line 1 "./frame/include/level0//bli_addjs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_ADDJS_H
#define BLIS_ADDJS_H

// addjs

// Notes:
// - The first char encodes the type of a.
// - The second char encodes the type of y.

#define bli_ssaddjs( a, y )  bli_saddjris( bli_sreal(a), bli_simag(a), bli_sreal(y), bli_simag(y) )
#define bli_dsaddjs( a, y )  bli_saddjris( bli_dreal(a), bli_dimag(a), bli_sreal(y), bli_simag(y) )
#define bli_csaddjs( a, y )  bli_saddjris( bli_creal(a), bli_cimag(a), bli_sreal(y), bli_simag(y) )
#define bli_zsaddjs( a, y )  bli_saddjris( bli_zreal(a), bli_zimag(a), bli_sreal(y), bli_simag(y) )

#define bli_sdaddjs( a, y )  bli_daddjris( bli_sreal(a), bli_simag(a), bli_dreal(y), bli_dimag(y) )
#define bli_ddaddjs( a, y )  bli_daddjris( bli_dreal(a), bli_dimag(a), bli_dreal(y), bli_dimag(y) )
#define bli_cdaddjs( a, y )  bli_daddjris( bli_creal(a), bli_cimag(a), bli_dreal(y), bli_dimag(y) )
#define bli_zdaddjs( a, y )  bli_daddjris( bli_zreal(a), bli_zimag(a), bli_dreal(y), bli_dimag(y) )

#ifndef BLIS_ENABLE_C99_COMPLEX

#define bli_scaddjs( a, y )  bli_caddjris( bli_sreal(a), bli_simag(a), bli_creal(y), bli_cimag(y) )
#define bli_dcaddjs( a, y )  bli_caddjris( bli_dreal(a), bli_dimag(a), bli_creal(y), bli_cimag(y) )
#define bli_ccaddjs( a, y )  bli_caddjris( bli_creal(a), bli_cimag(a), bli_creal(y), bli_cimag(y) )
#define bli_zcaddjs( a, y )  bli_caddjris( bli_zreal(a), bli_zimag(a), bli_creal(y), bli_cimag(y) )

#define bli_szaddjs( a, y )  bli_zaddjris( bli_sreal(a), bli_simag(a), bli_zreal(y), bli_zimag(y) )
#define bli_dzaddjs( a, y )  bli_zaddjris( bli_dreal(a), bli_dimag(a), bli_zreal(y), bli_zimag(y) )
#define bli_czaddjs( a, y )  bli_zaddjris( bli_creal(a), bli_cimag(a), bli_zreal(y), bli_zimag(y) )
#define bli_zzaddjs( a, y )  bli_zaddjris( bli_zreal(a), bli_zimag(a), bli_zreal(y), bli_zimag(y) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

#define bli_scaddjs( a, y )  { (y) +=      (a); }
#define bli_dcaddjs( a, y )  { (y) +=      (a); }
#define bli_ccaddjs( a, y )  { (y) += conjf(a); }
#define bli_zcaddjs( a, y )  { (y) += conj (a); }

#define bli_szaddjs( a, y )  { (y) +=      (a); }
#define bli_dzaddjs( a, y )  { (y) +=      (a); }
#define bli_czaddjs( a, y )  { (y) += conjf(a); }
#define bli_zzaddjs( a, y )  { (y) += conj (a); }

#endif // BLIS_ENABLE_C99_COMPLEX


#define bli_saddjs( a, y )  bli_ssaddjs( a, y )
#define bli_daddjs( a, y )  bli_ddaddjs( a, y )
#define bli_caddjs( a, y )  bli_ccaddjs( a, y )
#define bli_zaddjs( a, y )  bli_zzaddjs( a, y )


#endif

// end bli_addjs.h
#line 133 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_add3s.h
#line 1 "./frame/include/level0//bli_add3s.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_ADD3S_H
#define BLIS_ADD3S_H

// add3s

// Notes:
// - The first char encodes the type of a.
// - The second char encodes the type of b.
// - The third char encodes the type of c.


// -- (axy) = (??s) ------------------------------------------------------------

#define bli_sssadd3s( a, b, c )  bli_sadd3ris( bli_sreal(a), bli_simag(a), bli_sreal(b), bli_simag(b), bli_sreal(c), bli_simag(c) )
#define bli_dssadd3s( a, b, c )  bli_sadd3ris( bli_dreal(a), bli_dimag(a), bli_sreal(b), bli_simag(b), bli_sreal(c), bli_simag(c) )
#define bli_cssadd3s( a, b, c )  bli_sadd3ris( bli_creal(a), bli_cimag(a), bli_sreal(b), bli_simag(b), bli_sreal(c), bli_simag(c) )
#define bli_zssadd3s( a, b, c )  bli_sadd3ris( bli_zreal(a), bli_zimag(a), bli_sreal(b), bli_simag(b), bli_sreal(c), bli_simag(c) )

#define bli_sdsadd3s( a, b, c )  bli_sadd3ris( bli_sreal(a), bli_simag(a), bli_dreal(b), bli_dimag(b), bli_sreal(c), bli_simag(c) )
#define bli_ddsadd3s( a, b, c )  bli_sadd3ris( bli_dreal(a), bli_dimag(a), bli_dreal(b), bli_dimag(b), bli_sreal(c), bli_simag(c) )
#define bli_cdsadd3s( a, b, c )  bli_sadd3ris( bli_creal(a), bli_cimag(a), bli_dreal(b), bli_dimag(b), bli_sreal(c), bli_simag(c) )
#define bli_zdsadd3s( a, b, c )  bli_sadd3ris( bli_zreal(a), bli_zimag(a), bli_dreal(b), bli_dimag(b), bli_sreal(c), bli_simag(c) )

#define bli_scsadd3s( a, b, c )  bli_sadd3ris( bli_sreal(a), bli_simag(a), bli_creal(b), bli_cimag(b), bli_sreal(c), bli_simag(c) )
#define bli_dcsadd3s( a, b, c )  bli_sadd3ris( bli_dreal(a), bli_dimag(a), bli_creal(b), bli_cimag(b), bli_sreal(c), bli_simag(c) )
#define bli_ccsadd3s( a, b, c )  bli_sadd3ris( bli_creal(a), bli_cimag(a), bli_creal(b), bli_cimag(b), bli_sreal(c), bli_simag(c) )
#define bli_zcsadd3s( a, b, c )  bli_sadd3ris( bli_zreal(a), bli_zimag(a), bli_creal(b), bli_cimag(b), bli_sreal(c), bli_simag(c) )

#define bli_szsadd3s( a, b, c )  bli_sadd3ris( bli_sreal(a), bli_simag(a), bli_zreal(b), bli_zimag(b), bli_sreal(c), bli_simag(c) )
#define bli_dzsadd3s( a, b, c )  bli_sadd3ris( bli_dreal(a), bli_dimag(a), bli_zreal(b), bli_zimag(b), bli_sreal(c), bli_simag(c) )
#define bli_czsadd3s( a, b, c )  bli_sadd3ris( bli_creal(a), bli_cimag(a), bli_zreal(b), bli_zimag(b), bli_sreal(c), bli_simag(c) )
#define bli_zzsadd3s( a, b, c )  bli_sadd3ris( bli_zreal(a), bli_zimag(a), bli_zreal(b), bli_zimag(b), bli_sreal(c), bli_simag(c) )

// -- (axy) = (??d) ------------------------------------------------------------

#define bli_ssdadd3s( a, b, c )  bli_dadd3ris( bli_sreal(a), bli_simag(a), bli_sreal(b), bli_simag(b), bli_dreal(c), bli_dimag(c) )
#define bli_dsdadd3s( a, b, c )  bli_dadd3ris( bli_dreal(a), bli_dimag(a), bli_sreal(b), bli_simag(b), bli_dreal(c), bli_dimag(c) )
#define bli_csdadd3s( a, b, c )  bli_dadd3ris( bli_creal(a), bli_cimag(a), bli_sreal(b), bli_simag(b), bli_dreal(c), bli_dimag(c) )
#define bli_zsdadd3s( a, b, c )  bli_dadd3ris( bli_zreal(a), bli_zimag(a), bli_sreal(b), bli_simag(b), bli_dreal(c), bli_dimag(c) )

#define bli_sddadd3s( a, b, c )  bli_dadd3ris( bli_sreal(a), bli_simag(a), bli_dreal(b), bli_dimag(b), bli_dreal(c), bli_dimag(c) )
#define bli_dddadd3s( a, b, c )  bli_dadd3ris( bli_dreal(a), bli_dimag(a), bli_dreal(b), bli_dimag(b), bli_dreal(c), bli_dimag(c) )
#define bli_cddadd3s( a, b, c )  bli_dadd3ris( bli_creal(a), bli_cimag(a), bli_dreal(b), bli_dimag(b), bli_dreal(c), bli_dimag(c) )
#define bli_zddadd3s( a, b, c )  bli_dadd3ris( bli_zreal(a), bli_zimag(a), bli_dreal(b), bli_dimag(b), bli_dreal(c), bli_dimag(c) )

#define bli_scdadd3s( a, b, c )  bli_dadd3ris( bli_sreal(a), bli_simag(a), bli_creal(b), bli_cimag(b), bli_dreal(c), bli_dimag(c) )
#define bli_dcdadd3s( a, b, c )  bli_dadd3ris( bli_dreal(a), bli_dimag(a), bli_creal(b), bli_cimag(b), bli_dreal(c), bli_dimag(c) )
#define bli_ccdadd3s( a, b, c )  bli_dadd3ris( bli_creal(a), bli_cimag(a), bli_creal(b), bli_cimag(b), bli_dreal(c), bli_dimag(c) )
#define bli_zcdadd3s( a, b, c )  bli_dadd3ris( bli_zreal(a), bli_zimag(a), bli_creal(b), bli_cimag(b), bli_dreal(c), bli_dimag(c) )

#define bli_szdadd3s( a, b, c )  bli_dadd3ris( bli_sreal(a), bli_simag(a), bli_zreal(b), bli_zimag(b), bli_dreal(c), bli_dimag(c) )
#define bli_dzdadd3s( a, b, c )  bli_dadd3ris( bli_dreal(a), bli_dimag(a), bli_zreal(b), bli_zimag(b), bli_dreal(c), bli_dimag(c) )
#define bli_czdadd3s( a, b, c )  bli_dadd3ris( bli_creal(a), bli_cimag(a), bli_zreal(b), bli_zimag(b), bli_dreal(c), bli_dimag(c) )
#define bli_zzdadd3s( a, b, c )  bli_dadd3ris( bli_zreal(a), bli_zimag(a), bli_zreal(b), bli_zimag(b), bli_dreal(c), bli_dimag(c) )

#ifndef BLIS_ENABLE_C99_COMPLEX

// -- (axy) = (??c) ------------------------------------------------------------

#define bli_sscadd3s( a, b, c )  bli_sadd3ris( bli_sreal(a), bli_simag(a), bli_sreal(b), bli_simag(b), bli_creal(c), bli_cimag(c) )
#define bli_dscadd3s( a, b, c )  bli_sadd3ris( bli_dreal(a), bli_dimag(a), bli_sreal(b), bli_simag(b), bli_creal(c), bli_cimag(c) )
#define bli_cscadd3s( a, b, c )  bli_cadd3ris( bli_creal(a), bli_cimag(a), bli_sreal(b), bli_simag(b), bli_creal(c), bli_cimag(c) )
#define bli_zscadd3s( a, b, c )  bli_cadd3ris( bli_zreal(a), bli_zimag(a), bli_sreal(b), bli_simag(b), bli_creal(c), bli_cimag(c) )

#define bli_sdcadd3s( a, b, c )  bli_sadd3ris( bli_sreal(a), bli_simag(a), bli_dreal(b), bli_dimag(b), bli_creal(c), bli_cimag(c) )
#define bli_ddcadd3s( a, b, c )  bli_sadd3ris( bli_dreal(a), bli_dimag(a), bli_dreal(b), bli_dimag(b), bli_creal(c), bli_cimag(c) )
#define bli_cdcadd3s( a, b, c )  bli_cadd3ris( bli_creal(a), bli_cimag(a), bli_dreal(b), bli_dimag(b), bli_creal(c), bli_cimag(c) )
#define bli_zdcadd3s( a, b, c )  bli_cadd3ris( bli_zreal(a), bli_zimag(a), bli_dreal(b), bli_dimag(b), bli_creal(c), bli_cimag(c) )

#define bli_sccadd3s( a, b, c )  bli_cadd3ris( bli_sreal(a), bli_simag(a), bli_creal(b), bli_cimag(b), bli_creal(c), bli_cimag(c) )
#define bli_dccadd3s( a, b, c )  bli_cadd3ris( bli_dreal(a), bli_dimag(a), bli_creal(b), bli_cimag(b), bli_creal(c), bli_cimag(c) )
#define bli_cccadd3s( a, b, c )  bli_cadd3ris( bli_creal(a), bli_cimag(a), bli_creal(b), bli_cimag(b), bli_creal(c), bli_cimag(c) )
#define bli_zccadd3s( a, b, c )  bli_cadd3ris( bli_zreal(a), bli_zimag(a), bli_creal(b), bli_cimag(b), bli_creal(c), bli_cimag(c) )

#define bli_szcadd3s( a, b, c )  bli_cadd3ris( bli_sreal(a), bli_simag(a), bli_zreal(b), bli_zimag(b), bli_creal(c), bli_cimag(c) )
#define bli_dzcadd3s( a, b, c )  bli_cadd3ris( bli_dreal(a), bli_dimag(a), bli_zreal(b), bli_zimag(b), bli_creal(c), bli_cimag(c) )
#define bli_czcadd3s( a, b, c )  bli_cadd3ris( bli_creal(a), bli_cimag(a), bli_zreal(b), bli_zimag(b), bli_creal(c), bli_cimag(c) )
#define bli_zzcadd3s( a, b, c )  bli_cadd3ris( bli_zreal(a), bli_zimag(a), bli_zreal(b), bli_zimag(b), bli_creal(c), bli_cimag(c) )

// -- (axy) = (??z) ------------------------------------------------------------

#define bli_sszadd3s( a, b, c )  bli_dadd3ris( bli_sreal(a), bli_simag(a), bli_sreal(b), bli_simag(b), bli_zreal(c), bli_zimag(c) )
#define bli_dszadd3s( a, b, c )  bli_dadd3ris( bli_dreal(a), bli_dimag(a), bli_sreal(b), bli_simag(b), bli_zreal(c), bli_zimag(c) )
#define bli_cszadd3s( a, b, c )  bli_zadd3ris( bli_creal(a), bli_cimag(a), bli_sreal(b), bli_simag(b), bli_zreal(c), bli_zimag(c) )
#define bli_zszadd3s( a, b, c )  bli_zadd3ris( bli_zreal(a), bli_zimag(a), bli_sreal(b), bli_simag(b), bli_zreal(c), bli_zimag(c) )

#define bli_sdzadd3s( a, b, c )  bli_dadd3ris( bli_sreal(a), bli_simag(a), bli_dreal(b), bli_dimag(b), bli_zreal(c), bli_zimag(c) )
#define bli_ddzadd3s( a, b, c )  bli_dadd3ris( bli_dreal(a), bli_dimag(a), bli_dreal(b), bli_dimag(b), bli_zreal(c), bli_zimag(c) )
#define bli_cdzadd3s( a, b, c )  bli_zadd3ris( bli_creal(a), bli_cimag(a), bli_dreal(b), bli_dimag(b), bli_zreal(c), bli_zimag(c) )
#define bli_zdzadd3s( a, b, c )  bli_zadd3ris( bli_zreal(a), bli_zimag(a), bli_dreal(b), bli_dimag(b), bli_zreal(c), bli_zimag(c) )

#define bli_sczadd3s( a, b, c )  bli_zadd3ris( bli_sreal(a), bli_simag(a), bli_creal(b), bli_cimag(b), bli_zreal(c), bli_zimag(c) )
#define bli_dczadd3s( a, b, c )  bli_zadd3ris( bli_dreal(a), bli_dimag(a), bli_creal(b), bli_cimag(b), bli_zreal(c), bli_zimag(c) )
#define bli_cczadd3s( a, b, c )  bli_zadd3ris( bli_creal(a), bli_cimag(a), bli_creal(b), bli_cimag(b), bli_zreal(c), bli_zimag(c) )
#define bli_zczadd3s( a, b, c )  bli_zadd3ris( bli_zreal(a), bli_zimag(a), bli_creal(b), bli_cimag(b), bli_zreal(c), bli_zimag(c) )

#define bli_szzadd3s( a, b, c )  bli_zadd3ris( bli_sreal(a), bli_simag(a), bli_zreal(b), bli_zimag(b), bli_zreal(c), bli_zimag(c) )
#define bli_dzzadd3s( a, b, c )  bli_zadd3ris( bli_dreal(a), bli_dimag(a), bli_zreal(b), bli_zimag(b), bli_zreal(c), bli_zimag(c) )
#define bli_czzadd3s( a, b, c )  bli_zadd3ris( bli_creal(a), bli_cimag(a), bli_zreal(b), bli_zimag(b), bli_zreal(c), bli_zimag(c) )
#define bli_zzzadd3s( a, b, c )  bli_zadd3ris( bli_zreal(a), bli_zimag(a), bli_zreal(b), bli_zimag(b), bli_zreal(c), bli_zimag(c) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

// -- (axy) = (??c) ------------------------------------------------------------

#define bli_sscadd3s( a, b, c )  { (c) = (a) + (b); }
#define bli_dscadd3s( a, b, c )  { (c) = (a) + (b); }
#define bli_cscadd3s( a, b, c )  { (c) = (a) + (b); }
#define bli_zscadd3s( a, b, c )  { (c) = (a) + (b); }

#define bli_sdcadd3s( a, b, c )  { (c) = (a) + (b); }
#define bli_ddcadd3s( a, b, c )  { (c) = (a) + (b); }
#define bli_cdcadd3s( a, b, c )  { (c) = (a) + (b); }
#define bli_zdcadd3s( a, b, c )  { (c) = (a) + (b); }

#define bli_sccadd3s( a, b, c )  { (c) = (a) + (b); }
#define bli_dccadd3s( a, b, c )  { (c) = (a) + (b); }
#define bli_cccadd3s( a, b, c )  { (c) = (a) + (b); }
#define bli_zccadd3s( a, b, c )  { (c) = (a) + (b); }

#define bli_szcadd3s( a, b, c )  { (c) = (a) + (b); }
#define bli_dzcadd3s( a, b, c )  { (c) = (a) + (b); }
#define bli_czcadd3s( a, b, c )  { (c) = (a) + (b); }
#define bli_zzcadd3s( a, b, c )  { (c) = (a) + (b); }

// -- (axy) = (??z) ------------------------------------------------------------

#define bli_sszadd3s( a, b, c )  { (c) = (a) + (b); }
#define bli_dszadd3s( a, b, c )  { (c) = (a) + (b); }
#define bli_cszadd3s( a, b, c )  { (c) = (a) + (b); }
#define bli_zszadd3s( a, b, c )  { (c) = (a) + (b); }

#define bli_sdzadd3s( a, b, c )  { (c) = (a) + (b); }
#define bli_ddzadd3s( a, b, c )  { (c) = (a) + (b); }
#define bli_cdzadd3s( a, b, c )  { (c) = (a) + (b); }
#define bli_zdzadd3s( a, b, c )  { (c) = (a) + (b); }

#define bli_sczadd3s( a, b, c )  { (c) = (a) + (b); }
#define bli_dczadd3s( a, b, c )  { (c) = (a) + (b); }
#define bli_cczadd3s( a, b, c )  { (c) = (a) + (b); }
#define bli_zczadd3s( a, b, c )  { (c) = (a) + (b); }

#define bli_szzadd3s( a, b, c )  { (c) = (a) + (b); }
#define bli_dzzadd3s( a, b, c )  { (c) = (a) + (b); }
#define bli_czzadd3s( a, b, c )  { (c) = (a) + (b); }
#define bli_zzzadd3s( a, b, c )  { (c) = (a) + (b); }

#endif // BLIS_ENABLE_C99_COMPLEX


#define bli_sadd3s( a, b, c )  bli_sssadd3s( a, b, c )
#define bli_dadd3s( a, b, c )  bli_dddadd3s( a, b, c )
#define bli_cadd3s( a, b, c )  bli_cccadd3s( a, b, c )
#define bli_zadd3s( a, b, c )  bli_zzzadd3s( a, b, c )


#endif

// end bli_add3s.h
#line 135 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_axpbys.h
#line 1 "./frame/include/level0//bli_axpbys.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_AXPBYS_H
#define BLIS_AXPBYS_H

// axpbys

// Notes:
// - The first char encodes the type of a.
// - The second char encodes the type of x.
// - The third char encodes the type of b.
// - The fourth char encodes the type of y.

// -- (axby) = (???s) ----------------------------------------------------------

#define bli_ssssaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_dsssaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_csssaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_zsssaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_sdssaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_ddssaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_cdssaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_zdssaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_scssaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_dcssaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_ccssaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_zcssaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_szssaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_dzssaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_czssaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_zzssaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )

#define bli_ssdsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_dsdsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_csdsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zsdsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_sddsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_dddsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_cddsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zddsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_scdsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_dcdsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_ccdsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zcdsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_szdsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_dzdsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_czdsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zzdsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )

#define bli_sscsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_dscsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_cscsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zscsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_sdcsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_ddcsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_cdcsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zdcsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_sccsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_dccsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_cccsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zccsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_szcsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_dzcsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_czcsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zzcsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )

#define bli_sszsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_dszsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_cszsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zszsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_sdzsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_ddzsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_cdzsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zdzsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_sczsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_dczsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_cczsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zczsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_szzsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_dzzsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_czzsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zzzsaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )

// -- (axby) = (???d) ----------------------------------------------------------

#define bli_sssdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dssdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_cssdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zssdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_sdsdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_ddsdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_cdsdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zdsdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_scsdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dcsdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_ccsdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zcsdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_szsdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dzsdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_czsdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zzsdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )

#define bli_ssddaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dsddaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_csddaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zsddaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_sdddaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_ddddaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_cdddaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zdddaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_scddaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dcddaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_ccddaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zcddaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_szddaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dzddaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_czddaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zzddaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )

#define bli_sscdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dscdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_cscdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zscdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_sdcdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_ddcdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_cdcdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zdcdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_sccdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dccdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_cccdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zccdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_szcdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dzcdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_czcdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zzcdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )

#define bli_sszdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dszdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_cszdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zszdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_sdzdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_ddzdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_cdzdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zdzdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_sczdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dczdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_cczdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zczdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_szzdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dzzdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_czzdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zzzdaxpbys( a, x, b, y )  bli_rxaxpbyris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )

#ifndef BLIS_ENABLE_C99_COMPLEX

// -- (axby) = (???c) ----------------------------------------------------------

#define bli_ssscaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_dsscaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_csscaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_zsscaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_sdscaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_ddscaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_cdscaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_zdscaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_scscaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_dcscaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_ccscaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_zcscaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_szscaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_dzscaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_czscaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_zzscaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )

#define bli_ssdcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_dsdcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_csdcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zsdcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_sddcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_dddcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_cddcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zddcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_scdcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_dcdcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_ccdcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zcdcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_szdcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_dzdcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_czdcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zzdcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )

#define bli_ssccaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_dsccaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_csccaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zsccaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_sdccaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_ddccaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_cdccaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zdccaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_scccaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_dcccaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_ccccaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zcccaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_szccaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_dzccaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_czccaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zzccaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )

#define bli_sszcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_dszcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_cszcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zszcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_sdzcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_ddzcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_cdzcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zdzcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_sczcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_dczcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_cczcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zczcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_szzcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_dzzcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_czzcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zzzcaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )

// -- (axby) = (???z) ----------------------------------------------------------

#define bli_ssszaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dsszaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_csszaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zsszaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_sdszaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_ddszaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_cdszaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zdszaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_scszaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dcszaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_ccszaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zcszaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_szszaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dzszaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_czszaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zzszaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )

#define bli_ssdzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dsdzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_csdzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zsdzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_sddzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dddzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_cddzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zddzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_scdzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dcdzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_ccdzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zcdzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_szdzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dzdzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_czdzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zzdzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )

#define bli_ssczaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dsczaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_csczaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zsczaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_sdczaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_ddczaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_cdczaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zdczaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_scczaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dcczaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_ccczaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zcczaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_szczaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dzczaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_czczaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zzczaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )

#define bli_sszzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dszzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_cszzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zszzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_sdzzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_ddzzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_cdzzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zdzzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_sczzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dczzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_cczzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zczzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_szzzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dzzzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_czzzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zzzzaxpbys( a, x, b, y )  bli_cxaxpbyris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

// -- (axby) = (???c) ----------------------------------------------------------

#define bli_ssscaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_dsscaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_csscaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zsscaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_sdscaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_ddscaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_cdscaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zdscaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_scscaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_dcscaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_ccscaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zcscaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_szscaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_dzscaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_czscaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zzscaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }

#define bli_ssdcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_dsdcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_csdcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zsdcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_sddcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_dddcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_cddcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zddcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_scdcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_dcdcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_ccdcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zcdcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_szdcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_dzdcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_czdcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zzdcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }

#define bli_ssccaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_dsccaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_csccaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zsccaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_sdccaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_ddccaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_cdccaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zdccaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_scccaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_dcccaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_ccccaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zcccaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_szccaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_dzccaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_czccaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zzccaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }

#define bli_sszcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_dszcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_cszcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zszcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_sdzcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_ddzcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_cdzcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zdzcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_sczcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_dczcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_cczcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zczcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_szzcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_dzzcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_czzcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zzzcaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }

// -- (axby) = (???z) ----------------------------------------------------------

#define bli_ssszaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_dsszaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_csszaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zsszaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_sdszaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_ddszaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_cdszaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zdszaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_scszaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_dcszaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_ccszaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zcszaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_szszaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_dzszaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_czszaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zzszaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }

#define bli_ssdzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_dsdzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_csdzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zsdzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_sddzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_dddzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_cddzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zddzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_scdzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_dcdzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_ccdzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zcdzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_szdzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_dzdzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_czdzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zzdzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }

#define bli_ssczaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_dsczaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_csczaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zsczaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_sdczaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_ddczaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_cdczaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zdczaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_scczaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_dcczaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_ccczaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zcczaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_szczaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_dzczaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_czczaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zzczaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }

#define bli_sszzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_dszzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_cszzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zszzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_sdzzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_ddzzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_cdzzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zdzzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_sczzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_dczzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_cczzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zczzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_szzzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_dzzzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_czzzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }
#define bli_zzzzaxpbys( a, x, b, y )  { (y) = (a) * (x) + (b) * (y); }

#endif // BLIS_ENABLE_C99_COMPLEX


#define bli_saxpbys( a, x, b, y )  bli_ssssaxpbys( a, x, b, y )
#define bli_daxpbys( a, x, b, y )  bli_ddddaxpbys( a, x, b, y )
#define bli_caxpbys( a, x, b, y )  bli_ccccaxpbys( a, x, b, y )
#define bli_zaxpbys( a, x, b, y )  bli_zzzzaxpbys( a, x, b, y )


#endif

// end bli_axpbys.h
#line 137 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_axpbyjs.h
#line 1 "./frame/include/level0//bli_axpbyjs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_AXPBYJS_H
#define BLIS_AXPBYJS_H

// axpbyjs

// Notes:
// - The first char encodes the type of a.
// - The second char encodes the type of x.
// - The third char encodes the type of b.
// - The fourth char encodes the type of y.

// -- (axby) = (???s) ----------------------------------------------------------

#define bli_ssssaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_dsssaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_csssaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_zsssaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_sdssaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_ddssaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_cdssaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_zdssaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_scssaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_dcssaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_ccssaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_zcssaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_szssaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_dzssaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_czssaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_zzssaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )

#define bli_ssdsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_dsdsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_csdsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zsdsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_sddsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_dddsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_cddsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zddsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_scdsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_dcdsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_ccdsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zcdsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_szdsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_dzdsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_czdsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zzdsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )

#define bli_sscsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_dscsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_cscsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zscsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_sdcsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_ddcsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_cdcsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zdcsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_sccsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_dccsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_cccsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zccsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_szcsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_dzcsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_czcsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zzcsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )

#define bli_sszsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_dszsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_cszsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zszsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_sdzsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_ddzsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_cdzsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zdzsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_sczsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_dczsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_cczsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zczsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_szzsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_dzzsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_czzsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zzzsaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )

// -- (axby) = (???d) ----------------------------------------------------------

#define bli_sssdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dssdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_cssdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zssdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_sdsdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_ddsdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_cdsdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zdsdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_scsdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dcsdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_ccsdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zcsdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_szsdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dzsdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_czsdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zzsdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )

#define bli_ssddaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dsddaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_csddaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zsddaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_sdddaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_ddddaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_cdddaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zdddaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_scddaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dcddaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_ccddaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zcddaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_szddaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dzddaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_czddaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zzddaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )

#define bli_sscdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dscdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_cscdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zscdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_sdcdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_ddcdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_cdcdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zdcdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_sccdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dccdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_cccdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zccdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_szcdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dzcdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_czcdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zzcdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )

#define bli_sszdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dszdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_cszdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zszdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_sdzdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_ddzdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_cdzdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zdzdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_sczdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dczdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_cczdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zczdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_szzdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dzzdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_czzdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zzzdaxpbyjs( a, x, b, y )  bli_rxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )

#ifndef BLIS_ENABLE_C99_COMPLEX

// -- (axby) = (???c) ----------------------------------------------------------

#define bli_ssscaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_dsscaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_csscaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_zsscaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_sdscaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_ddscaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_cdscaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_zdscaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_scscaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_dcscaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_ccscaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_zcscaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_szscaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_dzscaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_czscaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_zzscaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )

#define bli_ssdcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_dsdcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_csdcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zsdcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_sddcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_dddcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_cddcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zddcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_scdcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_dcdcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_ccdcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zcdcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_szdcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_dzdcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_czdcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zzdcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )

#define bli_ssccaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_dsccaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_csccaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zsccaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_sdccaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_ddccaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_cdccaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zdccaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_scccaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_dcccaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_ccccaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zcccaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_szccaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_dzccaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_czccaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zzccaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )

#define bli_sszcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_dszcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_cszcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zszcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_sdzcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_ddzcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_cdzcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zdzcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_sczcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_dczcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_cczcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zczcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_szzcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_dzzcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_czzcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zzzcaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )

// -- (axby) = (???z) ----------------------------------------------------------

#define bli_ssszaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dsszaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_csszaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zsszaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_sdszaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_ddszaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_cdszaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zdszaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_scszaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dcszaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_ccszaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zcszaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_szszaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dzszaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_czszaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zzszaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )

#define bli_ssdzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dsdzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_csdzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zsdzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_sddzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dddzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_cddzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zddzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_scdzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dcdzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_ccdzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zcdzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_szdzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dzdzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_czdzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zzdzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )

#define bli_ssczaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dsczaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_csczaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zsczaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_sdczaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_ddczaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_cdczaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zdczaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_scczaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dcczaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_ccczaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zcczaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_szczaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dzczaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_czczaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zzczaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )

#define bli_sszzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dszzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_cszzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zszzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_sdzzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_ddzzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_cdzzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zdzzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_sczzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dczzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_cczzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zczzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_szzzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dzzzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_czzzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zzzzaxpbyjs( a, x, b, y )  bli_cxaxpbyjris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

// -- (axby) = (???c) ----------------------------------------------------------

#define bli_ssscaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_dsscaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_csscaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_zsscaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_sdscaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_ddscaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_cdscaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_zdscaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_scscaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_dcscaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_ccscaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_zcscaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_szscaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }
#define bli_dzscaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }
#define bli_czscaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }
#define bli_zzscaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }

#define bli_ssdcaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_dsdcaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_csdcaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_zsdcaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_sddcaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_dddcaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_cddcaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_zddcaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_scdcaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_dcdcaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_ccdcaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_zcdcaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_szdcaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }
#define bli_dzdcaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }
#define bli_czdcaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }
#define bli_zzdcaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }

#define bli_ssccaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_dsccaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_csccaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_zsccaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_sdccaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_ddccaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_cdccaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_zdccaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_scccaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_dcccaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_ccccaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_zcccaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_szccaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }
#define bli_dzccaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }
#define bli_czccaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }
#define bli_zzccaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }

#define bli_sszcaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_dszcaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_cszcaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_zszcaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_sdzcaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_ddzcaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_cdzcaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_zdzcaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_sczcaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_dczcaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_cczcaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_zczcaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_szzcaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }
#define bli_dzzcaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }
#define bli_czzcaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }
#define bli_zzzcaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }

// -- (axby) = (???z) ----------------------------------------------------------

#define bli_ssszaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_dsszaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_csszaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_zsszaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_sdszaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_ddszaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_cdszaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_zdszaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_scszaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_dcszaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_ccszaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_zcszaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_szszaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }
#define bli_dzszaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }
#define bli_czszaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }
#define bli_zzszaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }

#define bli_ssdzaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_dsdzaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_csdzaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_zsdzaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_sddzaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_dddzaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_cddzaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_zddzaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_scdzaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_dcdzaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_ccdzaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_zcdzaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_szdzaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }
#define bli_dzdzaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }
#define bli_czdzaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }
#define bli_zzdzaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }

#define bli_ssczaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_dsczaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_csczaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_zsczaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_sdczaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_ddczaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_cdczaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_zdczaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_scczaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_dcczaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_ccczaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_zcczaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_szczaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }
#define bli_dzczaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }
#define bli_czczaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }
#define bli_zzczaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }

#define bli_sszzaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_dszzaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_cszzaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_zszzaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_sdzzaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_ddzzaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_cdzzaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_zdzzaxpbyjs( a, x, b, y )  { (y) = (a) *      (x) + (b) * (y); }
#define bli_sczzaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_dczzaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_cczzaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_zczzaxpbyjs( a, x, b, y )  { (y) = (a) * conjf(x) + (b) * (y); }
#define bli_szzzaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }
#define bli_dzzzaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }
#define bli_czzzaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }
#define bli_zzzzaxpbyjs( a, x, b, y )  { (y) = (a) *  conj(x) + (b) * (y); }

#endif // BLIS_ENABLE_C99_COMPLEX


#define bli_saxpbyjs( a, x, b, y )  bli_ssssaxpbyjs( a, x, b, y )
#define bli_daxpbyjs( a, x, b, y )  bli_ddddaxpbyjs( a, x, b, y )
#define bli_caxpbyjs( a, x, b, y )  bli_ccccaxpbyjs( a, x, b, y )
#define bli_zaxpbyjs( a, x, b, y )  bli_zzzzaxpbyjs( a, x, b, y )


#endif

// end bli_axpbyjs.h
#line 138 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_axpys.h
#line 1 "./frame/include/level0//bli_axpys.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_AXPYS_H
#define BLIS_AXPYS_H

// axpys

// Notes:
// - The first char encodes the type of a.
// - The second char encodes the type of x.
// - The third char encodes the type of y.


// -- (axy) = (??s) ------------------------------------------------------------

#define bli_sssaxpys( a, x, y )  bli_saxpyris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_sreal(y), bli_simag(y) )
#define bli_dssaxpys( a, x, y )  bli_saxpyris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_sreal(y), bli_simag(y) )
#define bli_cssaxpys( a, x, y )  bli_saxpyris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_sreal(y), bli_simag(y) )
#define bli_zssaxpys( a, x, y )  bli_saxpyris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_sreal(y), bli_simag(y) )

#define bli_sdsaxpys( a, x, y )  bli_saxpyris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_sreal(y), bli_simag(y) )
#define bli_ddsaxpys( a, x, y )  bli_saxpyris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(y), bli_simag(y) )
#define bli_cdsaxpys( a, x, y )  bli_saxpyris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(y), bli_simag(y) )
#define bli_zdsaxpys( a, x, y )  bli_saxpyris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(y), bli_simag(y) )

#define bli_scsaxpys( a, x, y )  bli_saxpyris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_sreal(y), bli_simag(y) )
#define bli_dcsaxpys( a, x, y )  bli_saxpyris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_sreal(y), bli_simag(y) )
#define bli_ccsaxpys( a, x, y )  bli_saxpyris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_sreal(y), bli_simag(y) )
#define bli_zcsaxpys( a, x, y )  bli_saxpyris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_sreal(y), bli_simag(y) )

#define bli_szsaxpys( a, x, y )  bli_saxpyris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_sreal(y), bli_simag(y) )
#define bli_dzsaxpys( a, x, y )  bli_saxpyris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(y), bli_simag(y) )
#define bli_czsaxpys( a, x, y )  bli_saxpyris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(y), bli_simag(y) )
#define bli_zzsaxpys( a, x, y )  bli_saxpyris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(y), bli_simag(y) )

// -- (axy) = (??d) ------------------------------------------------------------

#define bli_ssdaxpys( a, x, y )  bli_daxpyris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_dreal(y), bli_dimag(y) )
#define bli_dsdaxpys( a, x, y )  bli_daxpyris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_dreal(y), bli_dimag(y) )
#define bli_csdaxpys( a, x, y )  bli_daxpyris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_dreal(y), bli_dimag(y) )
#define bli_zsdaxpys( a, x, y )  bli_daxpyris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_dreal(y), bli_dimag(y) )

#define bli_sddaxpys( a, x, y )  bli_daxpyris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_dddaxpys( a, x, y )  bli_daxpyris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_cddaxpys( a, x, y )  bli_daxpyris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_zddaxpys( a, x, y )  bli_daxpyris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(y), bli_dimag(y) )

#define bli_scdaxpys( a, x, y )  bli_daxpyris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_dcdaxpys( a, x, y )  bli_daxpyris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_ccdaxpys( a, x, y )  bli_daxpyris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_zcdaxpys( a, x, y )  bli_daxpyris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_dreal(y), bli_dimag(y) )

#define bli_szdaxpys( a, x, y )  bli_daxpyris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_dzdaxpys( a, x, y )  bli_daxpyris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_czdaxpys( a, x, y )  bli_daxpyris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_zzdaxpys( a, x, y )  bli_daxpyris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(y), bli_dimag(y) )

#ifndef BLIS_ENABLE_C99_COMPLEX

// -- (axy) = (??c) ------------------------------------------------------------

#define bli_sscaxpys( a, x, y )  bli_saxpyris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_creal(y), bli_cimag(y) )
#define bli_dscaxpys( a, x, y )  bli_saxpyris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_creal(y), bli_cimag(y) )
#define bli_cscaxpys( a, x, y )  bli_caxpyris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_creal(y), bli_cimag(y) )
#define bli_zscaxpys( a, x, y )  bli_caxpyris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_creal(y), bli_cimag(y) )

#define bli_sdcaxpys( a, x, y )  bli_saxpyris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_creal(y), bli_cimag(y) )
#define bli_ddcaxpys( a, x, y )  bli_saxpyris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_creal(y), bli_cimag(y) )
#define bli_cdcaxpys( a, x, y )  bli_caxpyris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_creal(y), bli_cimag(y) )
#define bli_zdcaxpys( a, x, y )  bli_caxpyris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_creal(y), bli_cimag(y) )

#define bli_sccaxpys( a, x, y )  bli_scaxpyris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_creal(y), bli_cimag(y) )
#define bli_dccaxpys( a, x, y )  bli_scaxpyris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_creal(y), bli_cimag(y) )
#define bli_cccaxpys( a, x, y )   bli_caxpyris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_creal(y), bli_cimag(y) )
#define bli_zccaxpys( a, x, y )   bli_caxpyris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_creal(y), bli_cimag(y) )

#define bli_szcaxpys( a, x, y )  bli_scaxpyris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_creal(y), bli_cimag(y) )
#define bli_dzcaxpys( a, x, y )  bli_scaxpyris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_creal(y), bli_cimag(y) )
#define bli_czcaxpys( a, x, y )   bli_caxpyris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_creal(y), bli_cimag(y) )
#define bli_zzcaxpys( a, x, y )   bli_caxpyris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_creal(y), bli_cimag(y) )

// -- (axy) = (??z) ------------------------------------------------------------

#define bli_sszaxpys( a, x, y )  bli_daxpyris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_zreal(y), bli_zimag(y) )
#define bli_dszaxpys( a, x, y )  bli_daxpyris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_zreal(y), bli_zimag(y) )
#define bli_cszaxpys( a, x, y )  bli_zaxpyris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_zreal(y), bli_zimag(y) )
#define bli_zszaxpys( a, x, y )  bli_zaxpyris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_zreal(y), bli_zimag(y) )

#define bli_sdzaxpys( a, x, y )  bli_daxpyris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_ddzaxpys( a, x, y )  bli_daxpyris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_cdzaxpys( a, x, y )  bli_zaxpyris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_zdzaxpys( a, x, y )  bli_zaxpyris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(y), bli_zimag(y) )

#define bli_sczaxpys( a, x, y )  bli_dzaxpyris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_dczaxpys( a, x, y )  bli_dzaxpyris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_cczaxpys( a, x, y )   bli_zaxpyris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_zczaxpys( a, x, y )   bli_zaxpyris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_zreal(y), bli_zimag(y) )

#define bli_szzaxpys( a, x, y )  bli_dzaxpyris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_dzzaxpys( a, x, y )  bli_dzaxpyris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_czzaxpys( a, x, y )   bli_zaxpyris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_zzzaxpys( a, x, y )   bli_zaxpyris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(y), bli_zimag(y) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

// -- (axy) = (??c) ------------------------------------------------------------

#define bli_sscaxpys( a, x, y )  { (y) += (a) * (x); }
#define bli_dscaxpys( a, x, y )  { (y) += (a) * (x); }
#define bli_cscaxpys( a, x, y )  { (y) += (a) * (x); }
#define bli_zscaxpys( a, x, y )  { (y) += (a) * (x); }

#define bli_sdcaxpys( a, x, y )  { (y) += (a) * (x); }
#define bli_ddcaxpys( a, x, y )  { (y) += (a) * (x); }
#define bli_cdcaxpys( a, x, y )  { (y) += (a) * (x); }
#define bli_zdcaxpys( a, x, y )  { (y) += (a) * (x); }

#define bli_sccaxpys( a, x, y )  { (y) += (a) * (x); }
#define bli_dccaxpys( a, x, y )  { (y) += (a) * (x); }
#define bli_cccaxpys( a, x, y )  { (y) += (a) * (x); }
#define bli_zccaxpys( a, x, y )  { (y) += (a) * (x); }

#define bli_szcaxpys( a, x, y )  { (y) += (a) * (x); }
#define bli_dzcaxpys( a, x, y )  { (y) += (a) * (x); }
#define bli_czcaxpys( a, x, y )  { (y) += (a) * (x); }
#define bli_zzcaxpys( a, x, y )  { (y) += (a) * (x); }

// -- (axy) = (??z) ------------------------------------------------------------

#define bli_sszaxpys( a, x, y )  { (y) += (a) * (x); }
#define bli_dszaxpys( a, x, y )  { (y) += (a) * (x); }
#define bli_cszaxpys( a, x, y )  { (y) += (a) * (x); }
#define bli_zszaxpys( a, x, y )  { (y) += (a) * (x); }

#define bli_sdzaxpys( a, x, y )  { (y) += (a) * (x); }
#define bli_ddzaxpys( a, x, y )  { (y) += (a) * (x); }
#define bli_cdzaxpys( a, x, y )  { (y) += (a) * (x); }
#define bli_zdzaxpys( a, x, y )  { (y) += (a) * (x); }

#define bli_sczaxpys( a, x, y )  { (y) += (a) * (x); }
#define bli_dczaxpys( a, x, y )  { (y) += (a) * (x); }
#define bli_cczaxpys( a, x, y )  { (y) += (a) * (x); }
#define bli_zczaxpys( a, x, y )  { (y) += (a) * (x); }

#define bli_szzaxpys( a, x, y )  { (y) += (a) * (x); }
#define bli_dzzaxpys( a, x, y )  { (y) += (a) * (x); }
#define bli_czzaxpys( a, x, y )  { (y) += (a) * (x); }
#define bli_zzzaxpys( a, x, y )  { (y) += (a) * (x); }

#endif // BLIS_ENABLE_C99_COMPLEX


#define bli_saxpys( a, x, y )  bli_sssaxpys( a, x, y )
#define bli_daxpys( a, x, y )  bli_dddaxpys( a, x, y )
#define bli_caxpys( a, x, y )  bli_cccaxpys( a, x, y )
#define bli_zaxpys( a, x, y )  bli_zzzaxpys( a, x, y )


#endif

// end bli_axpys.h
#line 140 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_axpyjs.h
#line 1 "./frame/include/level0//bli_axpyjs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_AXPYJS_H
#define BLIS_AXPYJS_H

// axpyjs

// Notes:
// - The first char encodes the type of a.
// - The second char encodes the type of x.
// - The third char encodes the type of y.


// -- (axy) = (??s) ------------------------------------------------------------

#define bli_sssaxpyjs( a, x, y )  bli_saxpyjris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_sreal(y), bli_simag(y) )
#define bli_dssaxpyjs( a, x, y )  bli_saxpyjris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_sreal(y), bli_simag(y) )
#define bli_cssaxpyjs( a, x, y )  bli_saxpyjris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_sreal(y), bli_simag(y) )
#define bli_zssaxpyjs( a, x, y )  bli_saxpyjris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_sreal(y), bli_simag(y) )

#define bli_sdsaxpyjs( a, x, y )  bli_saxpyjris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_sreal(y), bli_simag(y) )
#define bli_ddsaxpyjs( a, x, y )  bli_saxpyjris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(y), bli_simag(y) )
#define bli_cdsaxpyjs( a, x, y )  bli_saxpyjris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(y), bli_simag(y) )
#define bli_zdsaxpyjs( a, x, y )  bli_saxpyjris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(y), bli_simag(y) )

#define bli_scsaxpyjs( a, x, y )  bli_saxpyjris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_sreal(y), bli_simag(y) )
#define bli_dcsaxpyjs( a, x, y )  bli_saxpyjris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_sreal(y), bli_simag(y) )
#define bli_ccsaxpyjs( a, x, y )  bli_saxpyjris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_sreal(y), bli_simag(y) )
#define bli_zcsaxpyjs( a, x, y )  bli_saxpyjris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_sreal(y), bli_simag(y) )

#define bli_szsaxpyjs( a, x, y )  bli_saxpyjris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_sreal(y), bli_simag(y) )
#define bli_dzsaxpyjs( a, x, y )  bli_saxpyjris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(y), bli_simag(y) )
#define bli_czsaxpyjs( a, x, y )  bli_saxpyjris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(y), bli_simag(y) )
#define bli_zzsaxpyjs( a, x, y )  bli_saxpyjris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(y), bli_simag(y) )

// -- (axy) = (??d) ------------------------------------------------------------

#define bli_ssdaxpyjs( a, x, y )  bli_daxpyjris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_dreal(y), bli_dimag(y) )
#define bli_dsdaxpyjs( a, x, y )  bli_daxpyjris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_dreal(y), bli_dimag(y) )
#define bli_csdaxpyjs( a, x, y )  bli_daxpyjris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_dreal(y), bli_dimag(y) )
#define bli_zsdaxpyjs( a, x, y )  bli_daxpyjris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_dreal(y), bli_dimag(y) )

#define bli_sddaxpyjs( a, x, y )  bli_daxpyjris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_dddaxpyjs( a, x, y )  bli_daxpyjris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_cddaxpyjs( a, x, y )  bli_daxpyjris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_zddaxpyjs( a, x, y )  bli_daxpyjris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(y), bli_dimag(y) )

#define bli_scdaxpyjs( a, x, y )  bli_daxpyjris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_dcdaxpyjs( a, x, y )  bli_daxpyjris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_ccdaxpyjs( a, x, y )  bli_daxpyjris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_zcdaxpyjs( a, x, y )  bli_daxpyjris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_dreal(y), bli_dimag(y) )

#define bli_szdaxpyjs( a, x, y )  bli_daxpyjris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_dzdaxpyjs( a, x, y )  bli_daxpyjris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_czdaxpyjs( a, x, y )  bli_daxpyjris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_zzdaxpyjs( a, x, y )  bli_daxpyjris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(y), bli_dimag(y) )

#ifndef BLIS_ENABLE_C99_COMPLEX

// -- (axy) = (??c) ------------------------------------------------------------

#define bli_sscaxpyjs( a, x, y )  bli_saxpyjris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_creal(y), bli_cimag(y) )
#define bli_dscaxpyjs( a, x, y )  bli_saxpyjris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_creal(y), bli_cimag(y) )
#define bli_cscaxpyjs( a, x, y )  bli_caxpyjris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_creal(y), bli_cimag(y) )
#define bli_zscaxpyjs( a, x, y )  bli_caxpyjris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_creal(y), bli_cimag(y) )

#define bli_sdcaxpyjs( a, x, y )  bli_saxpyjris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_creal(y), bli_cimag(y) )
#define bli_ddcaxpyjs( a, x, y )  bli_saxpyjris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_creal(y), bli_cimag(y) )
#define bli_cdcaxpyjs( a, x, y )  bli_caxpyjris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_creal(y), bli_cimag(y) )
#define bli_zdcaxpyjs( a, x, y )  bli_caxpyjris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_creal(y), bli_cimag(y) )

#define bli_sccaxpyjs( a, x, y )  bli_scaxpyjris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_creal(y), bli_cimag(y) )
#define bli_dccaxpyjs( a, x, y )  bli_scaxpyjris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_creal(y), bli_cimag(y) )
#define bli_cccaxpyjs( a, x, y )   bli_caxpyjris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_creal(y), bli_cimag(y) )
#define bli_zccaxpyjs( a, x, y )   bli_caxpyjris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_creal(y), bli_cimag(y) )

#define bli_szcaxpyjs( a, x, y )  bli_scaxpyjris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_creal(y), bli_cimag(y) )
#define bli_dzcaxpyjs( a, x, y )  bli_scaxpyjris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_creal(y), bli_cimag(y) )
#define bli_czcaxpyjs( a, x, y )   bli_caxpyjris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_creal(y), bli_cimag(y) )
#define bli_zzcaxpyjs( a, x, y )   bli_caxpyjris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_creal(y), bli_cimag(y) )

// -- (axy) = (??z) ------------------------------------------------------------

#define bli_sszaxpyjs( a, x, y )  bli_daxpyjris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_zreal(y), bli_zimag(y) )
#define bli_dszaxpyjs( a, x, y )  bli_daxpyjris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_zreal(y), bli_zimag(y) )
#define bli_cszaxpyjs( a, x, y )  bli_zaxpyjris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_zreal(y), bli_zimag(y) )
#define bli_zszaxpyjs( a, x, y )  bli_zaxpyjris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_zreal(y), bli_zimag(y) )

#define bli_sdzaxpyjs( a, x, y )  bli_daxpyjris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_ddzaxpyjs( a, x, y )  bli_daxpyjris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_cdzaxpyjs( a, x, y )  bli_zaxpyjris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_zdzaxpyjs( a, x, y )  bli_zaxpyjris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(y), bli_zimag(y) )

#define bli_sczaxpyjs( a, x, y )  bli_dzaxpyjris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_dczaxpyjs( a, x, y )  bli_dzaxpyjris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_cczaxpyjs( a, x, y )   bli_zaxpyjris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_zczaxpyjs( a, x, y )   bli_zaxpyjris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_zreal(y), bli_zimag(y) )

#define bli_szzaxpyjs( a, x, y )  bli_dzaxpyjris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_dzzaxpyjs( a, x, y )  bli_dzaxpyjris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_czzaxpyjs( a, x, y )   bli_zaxpyjris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_zzzaxpyjs( a, x, y )   bli_zaxpyjris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(y), bli_zimag(y) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

// -- (axy) = (??c) ------------------------------------------------------------

#define bli_sscaxpyjs( a, x, y )  { (y) += (a) * (x); }
#define bli_dscaxpyjs( a, x, y )  { (y) += (a) * (x); }
#define bli_cscaxpyjs( a, x, y )  { (y) += (a) * (x); }
#define bli_zscaxpyjs( a, x, y )  { (y) += (a) * (x); }

#define bli_sdcaxpyjs( a, x, y )  { (y) += (a) * (x); }
#define bli_ddcaxpyjs( a, x, y )  { (y) += (a) * (x); }
#define bli_cdcaxpyjs( a, x, y )  { (y) += (a) * (x); }
#define bli_zdcaxpyjs( a, x, y )  { (y) += (a) * (x); }

#define bli_sccaxpyjs( a, x, y )  { (y) += (a) * conjf(x); }
#define bli_dccaxpyjs( a, x, y )  { (y) += (a) * conjf(x); }
#define bli_cccaxpyjs( a, x, y )  { (y) += (a) * conjf(x); }
#define bli_zccaxpyjs( a, x, y )  { (y) += (a) * conjf(x); }

#define bli_szcaxpyjs( a, x, y )  { (y) += (a) * conj(x); }
#define bli_dzcaxpyjs( a, x, y )  { (y) += (a) * conj(x); }
#define bli_czcaxpyjs( a, x, y )  { (y) += (a) * conj(x); }
#define bli_zzcaxpyjs( a, x, y )  { (y) += (a) * conj(x); }

// -- (axy) = (??z) ------------------------------------------------------------

#define bli_sszaxpyjs( a, x, y )  { (y) += (a) * (x); }
#define bli_dszaxpyjs( a, x, y )  { (y) += (a) * (x); }
#define bli_cszaxpyjs( a, x, y )  { (y) += (a) * (x); }
#define bli_zszaxpyjs( a, x, y )  { (y) += (a) * (x); }

#define bli_sdzaxpyjs( a, x, y )  { (y) += (a) * (x); }
#define bli_ddzaxpyjs( a, x, y )  { (y) += (a) * (x); }
#define bli_cdzaxpyjs( a, x, y )  { (y) += (a) * (x); }
#define bli_zdzaxpyjs( a, x, y )  { (y) += (a) * (x); }

#define bli_sczaxpyjs( a, x, y )  { (y) += (a) * conjf(x); }
#define bli_dczaxpyjs( a, x, y )  { (y) += (a) * conjf(x); }
#define bli_cczaxpyjs( a, x, y )  { (y) += (a) * conjf(x); }
#define bli_zczaxpyjs( a, x, y )  { (y) += (a) * conjf(x); }

#define bli_szzaxpyjs( a, x, y )  { (y) += (a) * conj(x); }
#define bli_dzzaxpyjs( a, x, y )  { (y) += (a) * conj(x); }
#define bli_czzaxpyjs( a, x, y )  { (y) += (a) * conj(x); }
#define bli_zzzaxpyjs( a, x, y )  { (y) += (a) * conj(x); }

#endif // BLIS_ENABLE_C99_COMPLEX


#define bli_saxpyjs( a, x, y )  bli_sssaxpyjs( a, x, y )
#define bli_daxpyjs( a, x, y )  bli_dddaxpyjs( a, x, y )
#define bli_caxpyjs( a, x, y )  bli_cccaxpyjs( a, x, y )
#define bli_zaxpyjs( a, x, y )  bli_zzzaxpyjs( a, x, y )


#endif

// end bli_axpyjs.h
#line 141 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_axmys.h
#line 1 "./frame/include/level0//bli_axmys.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_AXMYS_H
#define BLIS_AXMYS_H

// axmys

// Notes:
// - The first char encodes the type of a.
// - The second char encodes the type of x.
// - The third char encodes the type of y.


// -- (axy) = (??s) ------------------------------------------------------------

#define bli_sssaxmys( a, x, y )  bli_saxmyris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_sreal(y), bli_simag(y) )
#define bli_dssaxmys( a, x, y )  bli_saxmyris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_sreal(y), bli_simag(y) )
#define bli_cssaxmys( a, x, y )  bli_saxmyris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_sreal(y), bli_simag(y) )
#define bli_zssaxmys( a, x, y )  bli_saxmyris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_sreal(y), bli_simag(y) )

#define bli_sdsaxmys( a, x, y )  bli_saxmyris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_sreal(y), bli_simag(y) )
#define bli_ddsaxmys( a, x, y )  bli_saxmyris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(y), bli_simag(y) )
#define bli_cdsaxmys( a, x, y )  bli_saxmyris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(y), bli_simag(y) )
#define bli_zdsaxmys( a, x, y )  bli_saxmyris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(y), bli_simag(y) )

#define bli_scsaxmys( a, x, y )  bli_saxmyris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_sreal(y), bli_simag(y) )
#define bli_dcsaxmys( a, x, y )  bli_saxmyris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_sreal(y), bli_simag(y) )
#define bli_ccsaxmys( a, x, y )  bli_saxmyris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_sreal(y), bli_simag(y) )
#define bli_zcsaxmys( a, x, y )  bli_saxmyris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_sreal(y), bli_simag(y) )

#define bli_szsaxmys( a, x, y )  bli_saxmyris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_sreal(y), bli_simag(y) )
#define bli_dzsaxmys( a, x, y )  bli_saxmyris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(y), bli_simag(y) )
#define bli_czsaxmys( a, x, y )  bli_saxmyris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(y), bli_simag(y) )
#define bli_zzsaxmys( a, x, y )  bli_saxmyris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(y), bli_simag(y) )

// -- (axy) = (??d) ------------------------------------------------------------

#define bli_ssdaxmys( a, x, y )  bli_daxmyris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_dreal(y), bli_dimag(y) )
#define bli_dsdaxmys( a, x, y )  bli_daxmyris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_dreal(y), bli_dimag(y) )
#define bli_csdaxmys( a, x, y )  bli_daxmyris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_dreal(y), bli_dimag(y) )
#define bli_zsdaxmys( a, x, y )  bli_daxmyris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_dreal(y), bli_dimag(y) )

#define bli_sddaxmys( a, x, y )  bli_daxmyris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_dddaxmys( a, x, y )  bli_daxmyris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_cddaxmys( a, x, y )  bli_daxmyris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_zddaxmys( a, x, y )  bli_daxmyris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(y), bli_dimag(y) )

#define bli_scdaxmys( a, x, y )  bli_daxmyris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_dcdaxmys( a, x, y )  bli_daxmyris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_ccdaxmys( a, x, y )  bli_daxmyris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_zcdaxmys( a, x, y )  bli_daxmyris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_dreal(y), bli_dimag(y) )

#define bli_szdaxmys( a, x, y )  bli_daxmyris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_dzdaxmys( a, x, y )  bli_daxmyris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_czdaxmys( a, x, y )  bli_daxmyris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_zzdaxmys( a, x, y )  bli_daxmyris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(y), bli_dimag(y) )

#ifndef BLIS_ENABLE_C99_COMPLEX

// -- (axy) = (??c) ------------------------------------------------------------

#define bli_sscaxmys( a, x, y )  bli_saxmyris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_creal(y), bli_cimag(y) )
#define bli_dscaxmys( a, x, y )  bli_saxmyris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_creal(y), bli_cimag(y) )
#define bli_cscaxmys( a, x, y )  bli_caxmyris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_creal(y), bli_cimag(y) )
#define bli_zscaxmys( a, x, y )  bli_caxmyris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_creal(y), bli_cimag(y) )

#define bli_sdcaxmys( a, x, y )  bli_saxmyris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_creal(y), bli_cimag(y) )
#define bli_ddcaxmys( a, x, y )  bli_saxmyris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_creal(y), bli_cimag(y) )
#define bli_cdcaxmys( a, x, y )  bli_caxmyris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_creal(y), bli_cimag(y) )
#define bli_zdcaxmys( a, x, y )  bli_caxmyris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_creal(y), bli_cimag(y) )

#define bli_sccaxmys( a, x, y )  bli_scaxmyris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_creal(y), bli_cimag(y) )
#define bli_dccaxmys( a, x, y )  bli_scaxmyris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_creal(y), bli_cimag(y) )
#define bli_cccaxmys( a, x, y )   bli_caxmyris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_creal(y), bli_cimag(y) )
#define bli_zccaxmys( a, x, y )   bli_caxmyris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_creal(y), bli_cimag(y) )

#define bli_szcaxmys( a, x, y )  bli_scaxmyris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_creal(y), bli_cimag(y) )
#define bli_dzcaxmys( a, x, y )  bli_scaxmyris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_creal(y), bli_cimag(y) )
#define bli_czcaxmys( a, x, y )   bli_caxmyris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_creal(y), bli_cimag(y) )
#define bli_zzcaxmys( a, x, y )   bli_caxmyris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_creal(y), bli_cimag(y) )

// -- (axy) = (??z) ------------------------------------------------------------

#define bli_sszaxmys( a, x, y )  bli_daxmyris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_zreal(y), bli_zimag(y) )
#define bli_dszaxmys( a, x, y )  bli_daxmyris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_zreal(y), bli_zimag(y) )
#define bli_cszaxmys( a, x, y )  bli_zaxmyris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_zreal(y), bli_zimag(y) )
#define bli_zszaxmys( a, x, y )  bli_zaxmyris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_zreal(y), bli_zimag(y) )

#define bli_sdzaxmys( a, x, y )  bli_daxmyris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_ddzaxmys( a, x, y )  bli_daxmyris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_cdzaxmys( a, x, y )  bli_zaxmyris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_zdzaxmys( a, x, y )  bli_zaxmyris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(y), bli_zimag(y) )

#define bli_sczaxmys( a, x, y )  bli_dzaxmyris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_dczaxmys( a, x, y )  bli_dzaxmyris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_cczaxmys( a, x, y )   bli_zaxmyris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_zczaxmys( a, x, y )   bli_zaxmyris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_zreal(y), bli_zimag(y) )

#define bli_szzaxmys( a, x, y )  bli_dzaxmyris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_dzzaxmys( a, x, y )  bli_dzaxmyris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_czzaxmys( a, x, y )   bli_zaxmyris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_zzzaxmys( a, x, y )   bli_zaxmyris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(y), bli_zimag(y) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

// -- (axy) = (??c) ------------------------------------------------------------

#define bli_sscaxmys( a, x, y )  { (y) -= (a) * (x); }
#define bli_dscaxmys( a, x, y )  { (y) -= (a) * (x); }
#define bli_cscaxmys( a, x, y )  { (y) -= (a) * (x); }
#define bli_zscaxmys( a, x, y )  { (y) -= (a) * (x); }

#define bli_sdcaxmys( a, x, y )  { (y) -= (a) * (x); }
#define bli_ddcaxmys( a, x, y )  { (y) -= (a) * (x); }
#define bli_cdcaxmys( a, x, y )  { (y) -= (a) * (x); }
#define bli_zdcaxmys( a, x, y )  { (y) -= (a) * (x); }

#define bli_sccaxmys( a, x, y )  { (y) -= (a) * (x); }
#define bli_dccaxmys( a, x, y )  { (y) -= (a) * (x); }
#define bli_cccaxmys( a, x, y )  { (y) -= (a) * (x); }
#define bli_zccaxmys( a, x, y )  { (y) -= (a) * (x); }

#define bli_szcaxmys( a, x, y )  { (y) -= (a) * (x); }
#define bli_dzcaxmys( a, x, y )  { (y) -= (a) * (x); }
#define bli_czcaxmys( a, x, y )  { (y) -= (a) * (x); }
#define bli_zzcaxmys( a, x, y )  { (y) -= (a) * (x); }

// -- (axy) = (??z) ------------------------------------------------------------

#define bli_sszaxmys( a, x, y )  { (y) -= (a) * (x); }
#define bli_dszaxmys( a, x, y )  { (y) -= (a) * (x); }
#define bli_cszaxmys( a, x, y )  { (y) -= (a) * (x); }
#define bli_zszaxmys( a, x, y )  { (y) -= (a) * (x); }

#define bli_sdzaxmys( a, x, y )  { (y) -= (a) * (x); }
#define bli_ddzaxmys( a, x, y )  { (y) -= (a) * (x); }
#define bli_cdzaxmys( a, x, y )  { (y) -= (a) * (x); }
#define bli_zdzaxmys( a, x, y )  { (y) -= (a) * (x); }

#define bli_sczaxmys( a, x, y )  { (y) -= (a) * (x); }
#define bli_dczaxmys( a, x, y )  { (y) -= (a) * (x); }
#define bli_cczaxmys( a, x, y )  { (y) -= (a) * (x); }
#define bli_zczaxmys( a, x, y )  { (y) -= (a) * (x); }

#define bli_szzaxmys( a, x, y )  { (y) -= (a) * (x); }
#define bli_dzzaxmys( a, x, y )  { (y) -= (a) * (x); }
#define bli_czzaxmys( a, x, y )  { (y) -= (a) * (x); }
#define bli_zzzaxmys( a, x, y )  { (y) -= (a) * (x); }

#endif // BLIS_ENABLE_C99_COMPLEX


#define bli_saxmys( a, x, y )  bli_sssaxmys( a, x, y )
#define bli_daxmys( a, x, y )  bli_dddaxmys( a, x, y )
#define bli_caxmys( a, x, y )  bli_cccaxmys( a, x, y )
#define bli_zaxmys( a, x, y )  bli_zzzaxmys( a, x, y )


#endif

// end bli_axmys.h
#line 143 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_conjs.h
#line 1 "./frame/include/level0//bli_conjs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_CONJS_H
#define BLIS_CONJS_H

// conjs

#define bli_sconjs( x )  bli_sconjris( bli_sreal(x), bli_simag(x) )
#define bli_dconjs( x )  bli_dconjris( bli_dreal(x), bli_dimag(x) )

#ifndef BLIS_ENABLE_C99_COMPLEX

#define bli_cconjs( x )  bli_cconjris( bli_creal(x), bli_cimag(x) )
#define bli_zconjs( x )  bli_zconjris( bli_zreal(x), bli_zimag(x) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

#define bli_cconjs( x )  { (x) = conjf(x); }
#define bli_zconjs( x )  { (x) = conj (x); }

#endif // BLIS_ENABLE_C99_COMPLEX


#endif

// end bli_conjs.h
#line 145 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_copys.h
#line 1 "./frame/include/level0//bli_copys.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_COPYS_H
#define BLIS_COPYS_H

// copys

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of y.

#define bli_sscopys( x, y )  bli_scopyris( bli_sreal(x), bli_simag(x), bli_sreal(y), bli_simag(y) )
#define bli_dscopys( x, y )  bli_scopyris( bli_dreal(x), bli_dimag(x), bli_sreal(y), bli_simag(y) )
#define bli_cscopys( x, y )  bli_scopyris( bli_creal(x), bli_cimag(x), bli_sreal(y), bli_simag(y) )
#define bli_zscopys( x, y )  bli_scopyris( bli_zreal(x), bli_zimag(x), bli_sreal(y), bli_simag(y) )

#define bli_sdcopys( x, y )  bli_dcopyris( bli_sreal(x), bli_simag(x), bli_dreal(y), bli_dimag(y) )
#define bli_ddcopys( x, y )  bli_dcopyris( bli_dreal(x), bli_dimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_cdcopys( x, y )  bli_dcopyris( bli_creal(x), bli_cimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_zdcopys( x, y )  bli_dcopyris( bli_zreal(x), bli_zimag(x), bli_dreal(y), bli_dimag(y) )

// NOTE: Use of ccopyris() means the imaginary part of y will be overwritten with zero.
#define bli_sccopys( x, y )  bli_ccopyris( bli_sreal(x), bli_simag(x), bli_creal(y), bli_cimag(y) )
#define bli_dccopys( x, y )  bli_ccopyris( bli_dreal(x), bli_dimag(x), bli_creal(y), bli_cimag(y) )
#define bli_cccopys( x, y )  bli_ccopyris( bli_creal(x), bli_cimag(x), bli_creal(y), bli_cimag(y) )
#define bli_zccopys( x, y )  bli_ccopyris( bli_zreal(x), bli_zimag(x), bli_creal(y), bli_cimag(y) )

// NOTE: Use of zcopyris() means the imaginary part of y will be overwritten with zero.
#define bli_szcopys( x, y )  bli_zcopyris( bli_sreal(x), bli_simag(x), bli_zreal(y), bli_zimag(y) )
#define bli_dzcopys( x, y )  bli_zcopyris( bli_dreal(x), bli_dimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_czcopys( x, y )  bli_zcopyris( bli_creal(x), bli_cimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_zzcopys( x, y )  bli_zcopyris( bli_zreal(x), bli_zimag(x), bli_zreal(y), bli_zimag(y) )


#define bli_iicopys( x, y )  { (y) = ( gint_t ) (x); }


#define bli_scopys( x, y )  bli_sscopys( x, y )
#define bli_dcopys( x, y )  bli_ddcopys( x, y )
#define bli_ccopys( x, y )  bli_cccopys( x, y )
#define bli_zcopys( x, y )  bli_zzcopys( x, y )
#define bli_icopys( x, y )  bli_iicopys( x, y )


#endif

// end bli_copys.h
#line 147 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_copyjs.h
#line 1 "./frame/include/level0//bli_copyjs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_COPYJS_H
#define BLIS_COPYJS_H

// copyjs

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of y.

#define bli_sscopyjs( x, y )  bli_scopyjris( bli_sreal(x), bli_simag(x), bli_sreal(y), bli_simag(y) )
#define bli_dscopyjs( x, y )  bli_scopyjris( bli_dreal(x), bli_dimag(x), bli_sreal(y), bli_simag(y) )
#define bli_cscopyjs( x, y )  bli_scopyjris( bli_creal(x), bli_cimag(x), bli_sreal(y), bli_simag(y) )
#define bli_zscopyjs( x, y )  bli_scopyjris( bli_zreal(x), bli_zimag(x), bli_sreal(y), bli_simag(y) )

#define bli_sdcopyjs( x, y )  bli_dcopyjris( bli_sreal(x), bli_simag(x), bli_dreal(y), bli_dimag(y) )
#define bli_ddcopyjs( x, y )  bli_dcopyjris( bli_dreal(x), bli_dimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_cdcopyjs( x, y )  bli_dcopyjris( bli_creal(x), bli_cimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_zdcopyjs( x, y )  bli_dcopyjris( bli_zreal(x), bli_zimag(x), bli_dreal(y), bli_dimag(y) )

#ifndef BLIS_ENABLE_C99_COMPLEX

#define bli_sccopyjs( x, y )  bli_ccopyjris( bli_sreal(x), bli_simag(x), bli_creal(y), bli_cimag(y) )
#define bli_dccopyjs( x, y )  bli_ccopyjris( bli_dreal(x), bli_dimag(x), bli_creal(y), bli_cimag(y) )
#define bli_cccopyjs( x, y )  bli_ccopyjris( bli_creal(x), bli_cimag(x), bli_creal(y), bli_cimag(y) )
#define bli_zccopyjs( x, y )  bli_ccopyjris( bli_zreal(x), bli_zimag(x), bli_creal(y), bli_cimag(y) )

#define bli_szcopyjs( x, y )  bli_zcopyjris( bli_sreal(x), bli_simag(x), bli_zreal(y), bli_zimag(y) )
#define bli_dzcopyjs( x, y )  bli_zcopyjris( bli_dreal(x), bli_dimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_czcopyjs( x, y )  bli_zcopyjris( bli_creal(x), bli_cimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_zzcopyjs( x, y )  bli_zcopyjris( bli_zreal(x), bli_zimag(x), bli_zreal(y), bli_zimag(y) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

#define bli_sccopyjs( x, y )  { (y) =      (x); }
#define bli_dccopyjs( x, y )  { (y) =      (x); }
#define bli_cccopyjs( x, y )  { (y) = conjf(x); }
#define bli_zccopyjs( x, y )  { (y) = conj (x); }

#define bli_szcopyjs( x, y )  { (y) =      (x); }
#define bli_dzcopyjs( x, y )  { (y) =      (x); }
#define bli_czcopyjs( x, y )  { (y) = conjf(x); }
#define bli_zzcopyjs( x, y )  { (y) = conj (x); }

#endif // BLIS_ENABLE_C99_COMPLEX


#define bli_iicopyjs( x, y )  { (y) = ( gint_t ) (x); }


#define bli_scopyjs( x, y )  bli_sscopyjs( x, y )
#define bli_dcopyjs( x, y )  bli_ddcopyjs( x, y )
#define bli_ccopyjs( x, y )  bli_cccopyjs( x, y )
#define bli_zcopyjs( x, y )  bli_zzcopyjs( x, y )
#define bli_icopyjs( x, y )  bli_iicopyjs( x, y )


#endif

// end bli_copyjs.h
#line 148 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_copycjs.h
#line 1 "./frame/include/level0//bli_copycjs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_COPYCJS_H
#define BLIS_COPYCJS_H

// copycjs

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of y.

#define bli_sscopycjs( conjx, x, y )  bli_scopycjris( conjx, bli_sreal(x), bli_simag(x), bli_sreal(y), bli_simag(y) )
#define bli_dscopycjs( conjx, x, y )  bli_scopycjris( conjx, bli_dreal(x), bli_dimag(x), bli_sreal(y), bli_simag(y) )
#define bli_cscopycjs( conjx, x, y )  bli_scopycjris( conjx, bli_creal(x), bli_cimag(x), bli_sreal(y), bli_simag(y) )
#define bli_zscopycjs( conjx, x, y )  bli_scopycjris( conjx, bli_zreal(x), bli_zimag(x), bli_sreal(y), bli_simag(y) )

#define bli_sdcopycjs( conjx, x, y )  bli_dcopycjris( conjx, bli_sreal(x), bli_simag(x), bli_dreal(y), bli_dimag(y) )
#define bli_ddcopycjs( conjx, x, y )  bli_dcopycjris( conjx, bli_dreal(x), bli_dimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_cdcopycjs( conjx, x, y )  bli_dcopycjris( conjx, bli_creal(x), bli_cimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_zdcopycjs( conjx, x, y )  bli_dcopycjris( conjx, bli_zreal(x), bli_zimag(x), bli_dreal(y), bli_dimag(y) )

#ifndef BLIS_ENABLE_C99_COMPLEX

#define bli_sccopycjs( conjx, x, y )  bli_ccopycjris( conjx, bli_sreal(x), bli_simag(x), bli_creal(y), bli_cimag(y) )
#define bli_dccopycjs( conjx, x, y )  bli_ccopycjris( conjx, bli_dreal(x), bli_dimag(x), bli_creal(y), bli_cimag(y) )
#define bli_cccopycjs( conjx, x, y )  bli_ccopycjris( conjx, bli_creal(x), bli_cimag(x), bli_creal(y), bli_cimag(y) )
#define bli_zccopycjs( conjx, x, y )  bli_ccopycjris( conjx, bli_zreal(x), bli_zimag(x), bli_creal(y), bli_cimag(y) )

#define bli_szcopycjs( conjx, x, y )  bli_zcopycjris( conjx, bli_sreal(x), bli_simag(x), bli_zreal(y), bli_zimag(y) )
#define bli_dzcopycjs( conjx, x, y )  bli_zcopycjris( conjx, bli_dreal(x), bli_dimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_czcopycjs( conjx, x, y )  bli_zcopycjris( conjx, bli_creal(x), bli_cimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_zzcopycjs( conjx, x, y )  bli_zcopycjris( conjx, bli_zreal(x), bli_zimag(x), bli_zreal(y), bli_zimag(y) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

#define bli_sccopycjs( conjx, x, y )  { (y) = (x); }
#define bli_dccopycjs( conjx, x, y )  { (y) = (x); }
#define bli_cccopycjs( conjx, x, y )  { (y) = ( bli_is_conj( conjx ) ? conjf(x) : (x) ); }
#define bli_zccopycjs( conjx, x, y )  { (y) = ( bli_is_conj( conjx ) ? conj (x) : (x) ); }

#define bli_szcopycjs( conjx, x, y )  { (y) = (x); }
#define bli_dzcopycjs( conjx, x, y )  { (y) = (x); }
#define bli_czcopycjs( conjx, x, y )  { (y) = ( bli_is_conj( conjx ) ? conjf(x) : (x) ); }
#define bli_zzcopycjs( conjx, x, y )  { (y) = ( bli_is_conj( conjx ) ? conj (x) : (x) ); }

#endif // BLIS_ENABLE_C99_COMPLEX


#define bli_iicopycjs( conjx, x, y )  { (y) = ( gint_t ) (x); }


#define bli_scopycjs( conjx, x, y )  bli_sscopycjs( conjx, x, y )
#define bli_dcopycjs( conjx, x, y )  bli_ddcopycjs( conjx, x, y )
#define bli_ccopycjs( conjx, x, y )  bli_cccopycjs( conjx, x, y )
#define bli_zcopycjs( conjx, x, y )  bli_zzcopycjs( conjx, x, y )
#define bli_icopycjs( conjx, x, y )  bli_iicopycjs( conjx, x, y )


#endif

// end bli_copycjs.h
#line 149 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_copynzs.h
#line 1 "./frame/include/level0//bli_copynzs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_COPYNZS_H
#define BLIS_COPYNZS_H

// copynzs

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of y.

#define bli_sscopynzs( x, y )  bli_scopyris( bli_sreal(x), bli_simag(x), bli_sreal(y), bli_simag(y) )
#define bli_dscopynzs( x, y )  bli_scopyris( bli_dreal(x), bli_dimag(x), bli_sreal(y), bli_simag(y) )
#define bli_cscopynzs( x, y )  bli_scopyris( bli_creal(x), bli_cimag(x), bli_sreal(y), bli_simag(y) )
#define bli_zscopynzs( x, y )  bli_scopyris( bli_zreal(x), bli_zimag(x), bli_sreal(y), bli_simag(y) )

#define bli_sdcopynzs( x, y )  bli_dcopyris( bli_sreal(x), bli_simag(x), bli_dreal(y), bli_dimag(y) )
#define bli_ddcopynzs( x, y )  bli_dcopyris( bli_dreal(x), bli_dimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_cdcopynzs( x, y )  bli_dcopyris( bli_creal(x), bli_cimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_zdcopynzs( x, y )  bli_dcopyris( bli_zreal(x), bli_zimag(x), bli_dreal(y), bli_dimag(y) )

// NOTE: Use of scopyris() is so we don't touch the imaginary part of y.
#define bli_sccopynzs( x, y )  bli_scopyris( bli_sreal(x), bli_simag(x), bli_creal(y), bli_cimag(y) )
#define bli_dccopynzs( x, y )  bli_scopyris( bli_dreal(x), bli_dimag(x), bli_creal(y), bli_cimag(y) )
#define bli_cccopynzs( x, y )  bli_ccopyris( bli_creal(x), bli_cimag(x), bli_creal(y), bli_cimag(y) )
#define bli_zccopynzs( x, y )  bli_ccopyris( bli_zreal(x), bli_zimag(x), bli_creal(y), bli_cimag(y) )

// NOTE: Use of dcopyris() is so we don't touch the imaginary part of y.
#define bli_szcopynzs( x, y )  bli_dcopyris( bli_sreal(x), bli_simag(x), bli_zreal(y), bli_zimag(y) )
#define bli_dzcopynzs( x, y )  bli_dcopyris( bli_dreal(x), bli_dimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_czcopynzs( x, y )  bli_zcopyris( bli_creal(x), bli_cimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_zzcopynzs( x, y )  bli_zcopyris( bli_zreal(x), bli_zimag(x), bli_zreal(y), bli_zimag(y) )


#define bli_iicopynzs( x, y )  { (y) = ( gint_t ) (x); }


#define bli_scopynzs( x, y )  bli_sscopynzs( x, y )
#define bli_dcopynzs( x, y )  bli_ddcopynzs( x, y )
#define bli_ccopynzs( x, y )  bli_cccopynzs( x, y )
#define bli_zcopynzs( x, y )  bli_zzcopynzs( x, y )
#define bli_icopynzs( x, y )  bli_iicopynzs( x, y )


#endif

// end bli_copynzs.h
#line 151 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_copyjnzs.h
#line 1 "./frame/include/level0//bli_copyjnzs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_COPYJNZS_H
#define BLIS_COPYJNZS_H

// copyjnzs

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of y.

#define bli_sscopyjnzs( x, y )  bli_scopyjris( bli_sreal(x), bli_simag(x), bli_sreal(y), bli_simag(y) )
#define bli_dscopyjnzs( x, y )  bli_scopyjris( bli_dreal(x), bli_dimag(x), bli_sreal(y), bli_simag(y) )
#define bli_cscopyjnzs( x, y )  bli_scopyjris( bli_creal(x), bli_cimag(x), bli_sreal(y), bli_simag(y) )
#define bli_zscopyjnzs( x, y )  bli_scopyjris( bli_zreal(x), bli_zimag(x), bli_sreal(y), bli_simag(y) )

#define bli_sdcopyjnzs( x, y )  bli_dcopyjris( bli_sreal(x), bli_simag(x), bli_dreal(y), bli_dimag(y) )
#define bli_ddcopyjnzs( x, y )  bli_dcopyjris( bli_dreal(x), bli_dimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_cdcopyjnzs( x, y )  bli_dcopyjris( bli_creal(x), bli_cimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_zdcopyjnzs( x, y )  bli_dcopyjris( bli_zreal(x), bli_zimag(x), bli_dreal(y), bli_dimag(y) )

// NOTE: Use of scopyjris() (implemented in terms of scopyris()), is so we
// don't touch the imaginary part of y.
#define bli_sccopyjnzs( x, y )  bli_scopyjris( bli_sreal(x), bli_simag(x), bli_creal(y), bli_cimag(y) )
#define bli_dccopyjnzs( x, y )  bli_scopyjris( bli_dreal(x), bli_dimag(x), bli_creal(y), bli_cimag(y) )
#define bli_cccopyjnzs( x, y )  bli_ccopyjris( bli_creal(x), bli_cimag(x), bli_creal(y), bli_cimag(y) )
#define bli_zccopyjnzs( x, y )  bli_ccopyjris( bli_zreal(x), bli_zimag(x), bli_creal(y), bli_cimag(y) )

// NOTE: Use of dcopyjris() (implemented in terms of dcopyris()), is so we
// don't touch the imaginary part of y.
#define bli_szcopyjnzs( x, y )  bli_dcopyjris( bli_sreal(x), bli_simag(x), bli_zreal(y), bli_zimag(y) )
#define bli_dzcopyjnzs( x, y )  bli_dcopyjris( bli_dreal(x), bli_dimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_czcopyjnzs( x, y )  bli_zcopyjris( bli_creal(x), bli_cimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_zzcopyjnzs( x, y )  bli_zcopyjris( bli_zreal(x), bli_zimag(x), bli_zreal(y), bli_zimag(y) )


#define bli_iicopyjnzs( x, y )  { (y) = ( gint_t ) (x); }


#define bli_scopyjnzs( x, y )  bli_sscopyjnzs( x, y )
#define bli_dcopyjnzs( x, y )  bli_ddcopyjnzs( x, y )
#define bli_ccopyjnzs( x, y )  bli_cccopyjnzs( x, y )
#define bli_zcopyjnzs( x, y )  bli_zzcopyjnzs( x, y )
#define bli_icopyjnzs( x, y )  bli_iicopyjnzs( x, y )


#endif

// end bli_copyjnzs.h
#line 152 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_dots.h
#line 1 "./frame/include/level0//bli_dots.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_DOTS_H
#define BLIS_DOTS_H

// dots

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of y.
// - The third char encodes the type of rho.


#define bli_sssdots( x, y, a )  bli_sssaxpys( x, y, a )
#define bli_dssdots( x, y, a )  bli_dssaxpys( x, y, a )
#define bli_cssdots( x, y, a )  bli_cssaxpys( x, y, a )
#define bli_zssdots( x, y, a )  bli_zssaxpys( x, y, a )

#define bli_sdsdots( x, y, a )  bli_sdsaxpys( x, y, a )
#define bli_ddsdots( x, y, a )  bli_ddsaxpys( x, y, a )
#define bli_cdsdots( x, y, a )  bli_cdsaxpys( x, y, a )
#define bli_zdsdots( x, y, a )  bli_zdsaxpys( x, y, a )

#define bli_scsdots( x, y, a )  bli_scsaxpys( x, y, a )
#define bli_dcsdots( x, y, a )  bli_dcsaxpys( x, y, a )
#define bli_ccsdots( x, y, a )  bli_ccsaxpys( x, y, a )
#define bli_zcsdots( x, y, a )  bli_zcsaxpys( x, y, a )

#define bli_szsdots( x, y, a )  bli_szsaxpys( x, y, a )
#define bli_dzsdots( x, y, a )  bli_dzsaxpys( x, y, a )
#define bli_czsdots( x, y, a )  bli_czsaxpys( x, y, a )
#define bli_zzsdots( x, y, a )  bli_zzsaxpys( x, y, a )



#define bli_ssddots( x, y, a )  bli_ssdaxpys( x, y, a )
#define bli_dsddots( x, y, a )  bli_dsdaxpys( x, y, a )
#define bli_csddots( x, y, a )  bli_csdaxpys( x, y, a )
#define bli_zsddots( x, y, a )  bli_zsdaxpys( x, y, a )

#define bli_sdddots( x, y, a )  bli_sddaxpys( x, y, a )
#define bli_ddddots( x, y, a )  bli_dddaxpys( x, y, a )
#define bli_cdddots( x, y, a )  bli_cddaxpys( x, y, a )
#define bli_zdddots( x, y, a )  bli_zddaxpys( x, y, a )

#define bli_scddots( x, y, a )  bli_scdaxpys( x, y, a )
#define bli_dcddots( x, y, a )  bli_dcdaxpys( x, y, a )
#define bli_ccddots( x, y, a )  bli_ccdaxpys( x, y, a )
#define bli_zcddots( x, y, a )  bli_zcdaxpys( x, y, a )

#define bli_szddots( x, y, a )  bli_szdaxpys( x, y, a )
#define bli_dzddots( x, y, a )  bli_dzdaxpys( x, y, a )
#define bli_czddots( x, y, a )  bli_czdaxpys( x, y, a )
#define bli_zzddots( x, y, a )  bli_zzdaxpys( x, y, a )



#define bli_sscdots( x, y, a )  bli_sscaxpys( x, y, a )
#define bli_dscdots( x, y, a )  bli_dscaxpys( x, y, a )
#define bli_cscdots( x, y, a )  bli_cscaxpys( x, y, a )
#define bli_zscdots( x, y, a )  bli_zscaxpys( x, y, a )

#define bli_sdcdots( x, y, a )  bli_sdcaxpys( x, y, a )
#define bli_ddcdots( x, y, a )  bli_ddcaxpys( x, y, a )
#define bli_cdcdots( x, y, a )  bli_cdcaxpys( x, y, a )
#define bli_zdcdots( x, y, a )  bli_zdcaxpys( x, y, a )

#define bli_sccdots( x, y, a )  bli_sccaxpys( x, y, a )
#define bli_dccdots( x, y, a )  bli_dccaxpys( x, y, a )
#define bli_cccdots( x, y, a )  bli_cccaxpys( x, y, a )
#define bli_zccdots( x, y, a )  bli_zccaxpys( x, y, a )

#define bli_szcdots( x, y, a )  bli_szcaxpys( x, y, a )
#define bli_dzcdots( x, y, a )  bli_dzcaxpys( x, y, a )
#define bli_czcdots( x, y, a )  bli_czcaxpys( x, y, a )
#define bli_zzcdots( x, y, a )  bli_zzcaxpys( x, y, a )



#define bli_sszdots( x, y, a )  bli_sszaxpys( x, y, a )
#define bli_dszdots( x, y, a )  bli_dszaxpys( x, y, a )
#define bli_cszdots( x, y, a )  bli_cszaxpys( x, y, a )
#define bli_zszdots( x, y, a )  bli_zszaxpys( x, y, a )

#define bli_sdzdots( x, y, a )  bli_sdzaxpys( x, y, a )
#define bli_ddzdots( x, y, a )  bli_ddzaxpys( x, y, a )
#define bli_cdzdots( x, y, a )  bli_cdzaxpys( x, y, a )
#define bli_zdzdots( x, y, a )  bli_zdzaxpys( x, y, a )

#define bli_sczdots( x, y, a )  bli_sczaxpys( x, y, a )
#define bli_dczdots( x, y, a )  bli_dczaxpys( x, y, a )
#define bli_cczdots( x, y, a )  bli_cczaxpys( x, y, a )
#define bli_zczdots( x, y, a )  bli_zczaxpys( x, y, a )

#define bli_szzdots( x, y, a )  bli_szzaxpys( x, y, a )
#define bli_dzzdots( x, y, a )  bli_dzzaxpys( x, y, a )
#define bli_czzdots( x, y, a )  bli_czzaxpys( x, y, a )
#define bli_zzzdots( x, y, a )  bli_zzzaxpys( x, y, a )



#define bli_sdots( x, y, a )  bli_sssdots( x, y, a )
#define bli_ddots( x, y, a )  bli_ddddots( x, y, a )
#define bli_cdots( x, y, a )  bli_cccdots( x, y, a )
#define bli_zdots( x, y, a )  bli_zzzdots( x, y, a )


#endif

// end bli_dots.h
#line 154 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_dotjs.h
#line 1 "./frame/include/level0//bli_dotjs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_DOTJS_H
#define BLIS_DOTJS_H

// dotjs

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of y.
// - The third char encodes the type of rho.
// - x is used in conjugated form.


#define bli_sssdotjs( x, y, a )  bli_sssaxpyjs( y, x, a )
#define bli_dssdotjs( x, y, a )  bli_sdsaxpyjs( y, x, a )
#define bli_cssdotjs( x, y, a )  bli_scsaxpyjs( y, x, a )
#define bli_zssdotjs( x, y, a )  bli_szsaxpyjs( y, x, a )

#define bli_sdsdotjs( x, y, a )  bli_dssaxpyjs( y, x, a )
#define bli_ddsdotjs( x, y, a )  bli_ddsaxpyjs( y, x, a )
#define bli_cdsdotjs( x, y, a )  bli_dcsaxpyjs( y, x, a )
#define bli_zdsdotjs( x, y, a )  bli_dzsaxpyjs( y, x, a )

#define bli_scsdotjs( x, y, a )  bli_cssaxpyjs( y, x, a )
#define bli_dcsdotjs( x, y, a )  bli_cdsaxpyjs( y, x, a )
#define bli_ccsdotjs( x, y, a )  bli_ccsaxpyjs( y, x, a )
#define bli_zcsdotjs( x, y, a )  bli_czsaxpyjs( y, x, a )

#define bli_szsdotjs( x, y, a )  bli_zssaxpyjs( y, x, a )
#define bli_dzsdotjs( x, y, a )  bli_zdsaxpyjs( y, x, a )
#define bli_czsdotjs( x, y, a )  bli_zcsaxpyjs( y, x, a )
#define bli_zzsdotjs( x, y, a )  bli_zzsaxpyjs( y, x, a )


#define bli_ssddotjs( x, y, a )  bli_ssdaxpyjs( y, x, a )
#define bli_dsddotjs( x, y, a )  bli_sddaxpyjs( y, x, a )
#define bli_csddotjs( x, y, a )  bli_scdaxpyjs( y, x, a )
#define bli_zsddotjs( x, y, a )  bli_szdaxpyjs( y, x, a )

#define bli_sdddotjs( x, y, a )  bli_dsdaxpyjs( y, x, a )
#define bli_ddddotjs( x, y, a )  bli_dddaxpyjs( y, x, a )
#define bli_cdddotjs( x, y, a )  bli_dcdaxpyjs( y, x, a )
#define bli_zdddotjs( x, y, a )  bli_dzdaxpyjs( y, x, a )

#define bli_scddotjs( x, y, a )  bli_csdaxpyjs( y, x, a )
#define bli_dcddotjs( x, y, a )  bli_cddaxpyjs( y, x, a )
#define bli_ccddotjs( x, y, a )  bli_ccdaxpyjs( y, x, a )
#define bli_zcddotjs( x, y, a )  bli_czdaxpyjs( y, x, a )

#define bli_szddotjs( x, y, a )  bli_zsdaxpyjs( y, x, a )
#define bli_dzddotjs( x, y, a )  bli_zddaxpyjs( y, x, a )
#define bli_czddotjs( x, y, a )  bli_zcdaxpyjs( y, x, a )
#define bli_zzddotjs( x, y, a )  bli_zzdaxpyjs( y, x, a )


#define bli_sscdotjs( x, y, a )  bli_sscaxpyjs( y, x, a )
#define bli_dscdotjs( x, y, a )  bli_sdcaxpyjs( y, x, a )
#define bli_cscdotjs( x, y, a )  bli_sccaxpyjs( y, x, a )
#define bli_zscdotjs( x, y, a )  bli_szcaxpyjs( y, x, a )

#define bli_sdcdotjs( x, y, a )  bli_dscaxpyjs( y, x, a )
#define bli_ddcdotjs( x, y, a )  bli_ddcaxpyjs( y, x, a )
#define bli_cdcdotjs( x, y, a )  bli_dccaxpyjs( y, x, a )
#define bli_zdcdotjs( x, y, a )  bli_dzcaxpyjs( y, x, a )

#define bli_sccdotjs( x, y, a )  bli_cscaxpyjs( y, x, a )
#define bli_dccdotjs( x, y, a )  bli_cdcaxpyjs( y, x, a )
#define bli_cccdotjs( x, y, a )  bli_cccaxpyjs( y, x, a )
#define bli_zccdotjs( x, y, a )  bli_czcaxpyjs( y, x, a )

#define bli_szcdotjs( x, y, a )  bli_zscaxpyjs( y, x, a )
#define bli_dzcdotjs( x, y, a )  bli_zdcaxpyjs( y, x, a )
#define bli_czcdotjs( x, y, a )  bli_zccaxpyjs( y, x, a )
#define bli_zzcdotjs( x, y, a )  bli_zzcaxpyjs( y, x, a )


#define bli_sszdotjs( x, y, a )  bli_sszaxpyjs( y, x, a )
#define bli_dszdotjs( x, y, a )  bli_sdzaxpyjs( y, x, a )
#define bli_cszdotjs( x, y, a )  bli_sczaxpyjs( y, x, a )
#define bli_zszdotjs( x, y, a )  bli_szzaxpyjs( y, x, a )

#define bli_sdzdotjs( x, y, a )  bli_dszaxpyjs( y, x, a )
#define bli_ddzdotjs( x, y, a )  bli_ddzaxpyjs( y, x, a )
#define bli_cdzdotjs( x, y, a )  bli_dczaxpyjs( y, x, a )
#define bli_zdzdotjs( x, y, a )  bli_dzzaxpyjs( y, x, a )

#define bli_sczdotjs( x, y, a )  bli_cszaxpyjs( y, x, a )
#define bli_dczdotjs( x, y, a )  bli_cdzaxpyjs( y, x, a )
#define bli_cczdotjs( x, y, a )  bli_cczaxpyjs( y, x, a )
#define bli_zczdotjs( x, y, a )  bli_czzaxpyjs( y, x, a )

#define bli_szzdotjs( x, y, a )  bli_zszaxpyjs( y, x, a )
#define bli_dzzdotjs( x, y, a )  bli_zdzaxpyjs( y, x, a )
#define bli_czzdotjs( x, y, a )  bli_zczaxpyjs( y, x, a )
#define bli_zzzdotjs( x, y, a )  bli_zzzaxpyjs( y, x, a )





#define bli_sdotjs( x, y, a )  bli_sssdotjs( x, y, a )
#define bli_ddotjs( x, y, a )  bli_ddddotjs( x, y, a )
#define bli_cdotjs( x, y, a )  bli_cccdotjs( x, y, a )
#define bli_zdotjs( x, y, a )  bli_zzzdotjs( x, y, a )


#endif

// end bli_dotjs.h
#line 155 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_eq.h
#line 1 "./frame/include/level0//bli_eq.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_EQ_H
#define BLIS_EQ_H


// eq (passed by value)

#define bli_seq( a, b )  ( (a) == (b) )
#define bli_deq( a, b )  ( (a) == (b) )

#ifndef BLIS_ENABLE_C99_COMPLEX

#define bli_ceq( a, b )  ( ( bli_creal(a) == bli_creal(b) ) && ( bli_cimag(a) == bli_cimag(b) ) )
#define bli_zeq( a, b )  ( ( bli_zreal(a) == bli_zreal(b) ) && ( bli_zimag(a) == bli_zimag(b) ) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

#define bli_ceq( a, b )  ( (a) == (b) )
#define bli_zeq( a, b )  ( (a) == (b) )

#endif // BLIS_ENABLE_C99_COMPLEX

#define bli_ieq( a, b )  ( (a) == (b) )



// eqtori (passed by value)

#define bli_seqtori( a, br, bi )  ( (a) == (br) )
#define bli_deqtori( a, br, bi )  ( (a) == (br) )

#ifndef BLIS_ENABLE_C99_COMPLEX

#define bli_ceqtori( a, br, bi )  ( ( bli_creal(a) == (br) ) && ( bli_cimag(a) == (bi) ) )
#define bli_zeqtori( a, br, bi )  ( ( bli_zreal(a) == (br) ) && ( bli_zimag(a) == (bi) ) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

#define bli_ceqtori( a, br, bi )  ( (a) == (br) + (bi) * (I) )
#define bli_zeqtori( a, br, bi )  ( (a) == (br) + (bi) * (I) )

#endif // BLIS_ENABLE_C99_COMPLEX



// eqa (passed by address)

#define bli_seqa( a, b )  bli_seq( *(( float*    )(a)), *(( float*    )(b)) )
#define bli_deqa( a, b )  bli_deq( *(( double*   )(a)), *(( double*   )(b)) )
#define bli_ceqa( a, b )  bli_ceq( *(( scomplex* )(a)), *(( scomplex* )(b)) )
#define bli_zeqa( a, b )  bli_zeq( *(( dcomplex* )(a)), *(( dcomplex* )(b)) )
#define bli_ieqa( a, b )  bli_ieq( *(( gint_t*   )(a)), *(( gint_t*   )(b)) )



// eq1

#define bli_seq1( a )  bli_seqtori( (a), 1.0F, 0.0F )
#define bli_deq1( a )  bli_deqtori( (a), 1.0,  0.0  )
#define bli_ceq1( a )  bli_ceqtori( (a), 1.0F, 0.0F )
#define bli_zeq1( a )  bli_zeqtori( (a), 1.0,  0.0  )
#define bli_ieq1( a )  bli_ieq    ( (a), 1          )



// eq0

#define bli_seq0( a )  bli_seqtori( (a), 0.0F, 0.0F )
#define bli_deq0( a )  bli_deqtori( (a), 0.0,  0.0  )
#define bli_ceq0( a )  bli_ceqtori( (a), 0.0F, 0.0F )
#define bli_zeq0( a )  bli_zeqtori( (a), 0.0,  0.0  )
#define bli_ieq0( a )  bli_ieq    ( (a), 0          )



// eqm1

#define bli_seqm1( a )  bli_seqtori( (a), -1.0F, 0.0F )
#define bli_deqm1( a )  bli_deqtori( (a), -1.0,  0.0  )
#define bli_ceqm1( a )  bli_ceqtori( (a), -1.0F, 0.0F )
#define bli_zeqm1( a )  bli_zeqtori( (a), -1.0,  0.0  )
#define bli_ieqm1( a )  bli_ieq    ( (a), -1          )



#endif
// end bli_eq.h
#line 157 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_lt.h
#line 1 "./frame/include/level0//bli_lt.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_LT_H
#define BLIS_LT_H


// lt (passed by value)

#define bli_slt( a, b )  (          (a) <          (b) )
#define bli_dlt( a, b )  (          (a) <          (b) )
#define bli_clt( a, b )  ( bli_creal(a) < bli_creal(b) )
#define bli_zlt( a, b )  ( bli_zreal(a) < bli_zreal(b) )
#define bli_ilt( a, b )  (          (a) <          (b) )

// lt0

#define bli_slt0( a )  (          (a) < 0.0F )
#define bli_dlt0( a )  (          (a) < 0.0  )
#define bli_clt0( a )  ( bli_creal(a) < 0.0F )
#define bli_zlt0( a )  ( bli_zreal(a) < 0.0  )

// gt (passed by value)

#define bli_sgt( a, b )  (          (a) >          (b) )
#define bli_dgt( a, b )  (          (a) >          (b) )
#define bli_cgt( a, b )  ( bli_creal(a) > bli_creal(b) )
#define bli_zgt( a, b )  ( bli_zreal(a) > bli_zreal(b) )
#define bli_igt( a, b )  (          (a) >          (b) )

// gt0

#define bli_sgt0( a )  (          (a) > 0.0F )
#define bli_dgt0( a )  (          (a) > 0.0  )
#define bli_cgt0( a )  ( bli_creal(a) > 0.0F )
#define bli_zgt0( a )  ( bli_zreal(a) > 0.0  )



#endif
// end bli_lt.h
#line 158 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_lte.h
#line 1 "./frame/include/level0//bli_lte.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_LTE_H
#define BLIS_LTE_H


// lte (passed by value)

#define bli_slte( a, b )  (          (a) <=          (b) )
#define bli_dlte( a, b )  (          (a) <=          (b) )
#define bli_clte( a, b )  ( bli_creal(a) <= bli_creal(b) )
#define bli_zlte( a, b )  ( bli_zreal(a) <= bli_zreal(b) )
#define bli_ilte( a, b )  (          (a) <=          (b) )

// lte0

#define bli_slte0( a )  (          (a) <= 0.0F )
#define bli_dlte0( a )  (          (a) <= 0.0  )
#define bli_clte0( a )  ( bli_creal(a) <= 0.0F )
#define bli_zlte0( a )  ( bli_zreal(a) <= 0.0  )

// gte (passed by value)

#define bli_sgte( a, b )  (          (a) >=          (b) )
#define bli_dgte( a, b )  (          (a) >=          (b) )
#define bli_cgte( a, b )  ( bli_creal(a) >= bli_creal(b) )
#define bli_zgte( a, b )  ( bli_zreal(a) >= bli_zreal(b) )
#define bli_igte( a, b )  (          (a) >=          (b) )

// gte0

#define bli_sgte0( a )  (          (a) >= 0.0F )
#define bli_dgte0( a )  (          (a) >= 0.0  )
#define bli_cgte0( a )  ( bli_creal(a) >= 0.0F )
#define bli_zgte0( a )  ( bli_zreal(a) >= 0.0  )



#endif
// end bli_lte.h
#line 159 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_fprints.h
#line 1 "./frame/include/level0//bli_fprints.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_FPRINTS_H
#define BLIS_FPRINTS_H

// prints

#define bli_sfprints( file, spec, x ) \
{ \
	fprintf( file, spec, (x) ); \
}
#define bli_dfprints( file, spec, x ) \
{ \
	fprintf( file, spec, (x) ); \
}
#define bli_cfprints( file, spec, x ) \
{ \
	fprintf( file, spec, bli_creal(x) ); \
	fprintf( file, " + " ); \
	fprintf( file, spec, bli_cimag(x) ); \
	fprintf( file, " " ); \
}
#define bli_zfprints( file, spec, x ) \
{ \
	fprintf( file, spec, bli_zreal(x) ); \
	fprintf( file, " + " ); \
	fprintf( file, spec, bli_zimag(x) ); \
	fprintf( file, " " ); \
}
#define bli_ifprints( file, spec, x ) \
{ \
	fprintf( file, spec, (x) ); \
}


#endif
// end bli_fprints.h
#line 161 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_inverts.h
#line 1 "./frame/include/level0//bli_inverts.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_INVERTS_H
#define BLIS_INVERTS_H

// inverts

// Notes:
// - The first char encodes the type of x.

#define bli_sinverts( x )  bli_sinvertris( bli_sreal(x), bli_simag(x) )
#define bli_dinverts( x )  bli_dinvertris( bli_dreal(x), bli_dimag(x) )

#ifndef BLIS_ENABLE_C99_COMPLEX

#define bli_cinverts( x )  bli_cinvertris( bli_creal(x), bli_cimag(x) )
#define bli_zinverts( x )  bli_zinvertris( bli_zreal(x), bli_zimag(x) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

#define bli_cinverts( x )  { (x) = 1.0F / (x); }
#define bli_zinverts( x )  { (x) = 1.0  / (x); }

#endif // BLIS_ENABLE_C99_COMPLEX


#endif

// end bli_inverts.h
#line 163 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_invscals.h
#line 1 "./frame/include/level0//bli_invscals.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_INVSCALS_H
#define BLIS_INVSCALS_H

// invscals

// Notes:
// - The first char encodes the type of a.
// - The second char encodes the type of y.

#define bli_ssinvscals( a, y )  bli_sinvscalris( bli_sreal(a), bli_simag(a), bli_sreal(y), bli_simag(y) )
#define bli_dsinvscals( a, y )  bli_sinvscalris( bli_dreal(a), bli_dimag(a), bli_sreal(y), bli_simag(y) )
#define bli_csinvscals( a, y )  bli_sinvscalris( bli_creal(a), bli_cimag(a), bli_sreal(y), bli_simag(y) )
#define bli_zsinvscals( a, y )  bli_sinvscalris( bli_zreal(a), bli_zimag(a), bli_sreal(y), bli_simag(y) )

#define bli_sdinvscals( a, y )  bli_dinvscalris( bli_sreal(a), bli_simag(a), bli_dreal(y), bli_dimag(y) )
#define bli_ddinvscals( a, y )  bli_dinvscalris( bli_dreal(a), bli_dimag(a), bli_dreal(y), bli_dimag(y) )
#define bli_cdinvscals( a, y )  bli_dinvscalris( bli_creal(a), bli_cimag(a), bli_dreal(y), bli_dimag(y) )
#define bli_zdinvscals( a, y )  bli_dinvscalris( bli_zreal(a), bli_zimag(a), bli_dreal(y), bli_dimag(y) )

#ifndef BLIS_ENABLE_C99_COMPLEX

#define bli_scinvscals( a, y )  bli_scinvscalris( bli_sreal(a), bli_simag(a), bli_creal(y), bli_cimag(y) )
#define bli_dcinvscals( a, y )  bli_scinvscalris( bli_dreal(a), bli_dimag(a), bli_creal(y), bli_cimag(y) )
#define bli_ccinvscals( a, y )   bli_cinvscalris( bli_creal(a), bli_cimag(a), bli_creal(y), bli_cimag(y) )
#define bli_zcinvscals( a, y )   bli_cinvscalris( bli_zreal(a), bli_zimag(a), bli_creal(y), bli_cimag(y) )

#define bli_szinvscals( a, y )  bli_dzinvscalris( bli_sreal(a), bli_simag(a), bli_zreal(y), bli_zimag(y) )
#define bli_dzinvscals( a, y )  bli_dzinvscalris( bli_dreal(a), bli_dimag(a), bli_zreal(y), bli_zimag(y) )
#define bli_czinvscals( a, y )   bli_zinvscalris( bli_creal(a), bli_cimag(a), bli_zreal(y), bli_zimag(y) )
#define bli_zzinvscals( a, y )   bli_zinvscalris( bli_zreal(a), bli_zimag(a), bli_zreal(y), bli_zimag(y) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

#define bli_scinvscals( a, y )  { (y) /= (a); }
#define bli_dcinvscals( a, y )  { (y) /= (a); }
#define bli_ccinvscals( a, y )  { (y) /= (a); }
#define bli_zcinvscals( a, y )  { (y) /= (a); }

#define bli_szinvscals( a, y )  { (y) /= (a); }
#define bli_dzinvscals( a, y )  { (y) /= (a); }
#define bli_czinvscals( a, y )  { (y) /= (a); }
#define bli_zzinvscals( a, y )  { (y) /= (a); }

#endif // BLIS_ENABLE_C99_COMPLEX


#define bli_sinvscals( a, y )  bli_ssinvscals( a, y )
#define bli_dinvscals( a, y )  bli_ddinvscals( a, y )
#define bli_cinvscals( a, y )  bli_ccinvscals( a, y )
#define bli_zinvscals( a, y )  bli_zzinvscals( a, y )


#endif

// end bli_invscals.h
#line 165 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_invscaljs.h
#line 1 "./frame/include/level0//bli_invscaljs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_INVSCALJS_H
#define BLIS_INVSCALJS_H

// invscaljs

// Notes:
// - The first char encodes the type of a.
// - The second char encodes the type of y.

#define bli_ssinvscaljs( a, y )  bli_sinvscaljris( bli_sreal(a), bli_simag(a), bli_sreal(y), bli_simag(y) )
#define bli_dsinvscaljs( a, y )  bli_sinvscaljris( bli_dreal(a), bli_dimag(a), bli_sreal(y), bli_simag(y) )
#define bli_csinvscaljs( a, y )  bli_sinvscaljris( bli_creal(a), bli_cimag(a), bli_sreal(y), bli_simag(y) )
#define bli_zsinvscaljs( a, y )  bli_sinvscaljris( bli_zreal(a), bli_zimag(a), bli_sreal(y), bli_simag(y) )

#define bli_sdinvscaljs( a, y )  bli_dinvscaljris( bli_sreal(a), bli_simag(a), bli_dreal(y), bli_dimag(y) )
#define bli_ddinvscaljs( a, y )  bli_dinvscaljris( bli_dreal(a), bli_dimag(a), bli_dreal(y), bli_dimag(y) )
#define bli_cdinvscaljs( a, y )  bli_dinvscaljris( bli_creal(a), bli_cimag(a), bli_dreal(y), bli_dimag(y) )
#define bli_zdinvscaljs( a, y )  bli_dinvscaljris( bli_zreal(a), bli_zimag(a), bli_dreal(y), bli_dimag(y) )

#ifndef BLIS_ENABLE_C99_COMPLEX

#define bli_scinvscaljs( a, y )  bli_scinvscaljris( bli_sreal(a), bli_simag(a), bli_creal(y), bli_cimag(y) )
#define bli_dcinvscaljs( a, y )  bli_scinvscaljris( bli_dreal(a), bli_dimag(a), bli_creal(y), bli_cimag(y) )
#define bli_ccinvscaljs( a, y )   bli_cinvscaljris( bli_creal(a), bli_cimag(a), bli_creal(y), bli_cimag(y) )
#define bli_zcinvscaljs( a, y )   bli_cinvscaljris( bli_zreal(a), bli_zimag(a), bli_creal(y), bli_cimag(y) )

#define bli_szinvscaljs( a, y )  bli_dzinvscaljris( bli_sreal(a), bli_simag(a), bli_zreal(y), bli_zimag(y) )
#define bli_dzinvscaljs( a, y )  bli_dzinvscaljris( bli_dreal(a), bli_dimag(a), bli_zreal(y), bli_zimag(y) )
#define bli_czinvscaljs( a, y )   bli_zinvscaljris( bli_creal(a), bli_cimag(a), bli_zreal(y), bli_zimag(y) )
#define bli_zzinvscaljs( a, y )   bli_zinvscaljris( bli_zreal(a), bli_zimag(a), bli_zreal(y), bli_zimag(y) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

#define bli_scinvscaljs( a, y )  { (y) /=      (a); }
#define bli_dcinvscaljs( a, y )  { (y) /=      (a); }
#define bli_ccinvscaljs( a, y )  { (y) /= conjf(a); }
#define bli_zcinvscaljs( a, y )  { (y) /= conj (a); }

#define bli_szinvscaljs( a, y )  { (y) /=      (a); }
#define bli_dzinvscaljs( a, y )  { (y) /=      (a); }
#define bli_czinvscaljs( a, y )  { (y) /= conjf(a); }
#define bli_zzinvscaljs( a, y )  { (y) /= conj (a); }

#endif // BLIS_ENABLE_C99_COMPLEX


#define bli_sinvscaljs( a, y )  bli_ssinvscaljs( a, y )
#define bli_dinvscaljs( a, y )  bli_ddinvscaljs( a, y )
#define bli_cinvscaljs( a, y )  bli_ccinvscaljs( a, y )
#define bli_zinvscaljs( a, y )  bli_zzinvscaljs( a, y )


#endif

// end bli_invscaljs.h
#line 166 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_neg2s.h
#line 1 "./frame/include/level0//bli_neg2s.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_NEG2S_H
#define BLIS_NEG2S_H

// neg2s

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of y.

#define bli_ssneg2s( x, y )  bli_sneg2ris( bli_sreal(x), bli_simag(x), bli_sreal(y), bli_simag(y) )
#define bli_dsneg2s( x, y )  bli_sneg2ris( bli_dreal(x), bli_dimag(x), bli_sreal(y), bli_simag(y) )
#define bli_csneg2s( x, y )  bli_sneg2ris( bli_creal(x), bli_cimag(x), bli_sreal(y), bli_simag(y) )
#define bli_zsneg2s( x, y )  bli_sneg2ris( bli_zreal(x), bli_zimag(x), bli_sreal(y), bli_simag(y) )

#define bli_sdneg2s( x, y )  bli_dneg2ris( bli_sreal(x), bli_simag(x), bli_dreal(y), bli_dimag(y) )
#define bli_ddneg2s( x, y )  bli_dneg2ris( bli_dreal(x), bli_dimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_cdneg2s( x, y )  bli_dneg2ris( bli_creal(x), bli_cimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_zdneg2s( x, y )  bli_dneg2ris( bli_zreal(x), bli_zimag(x), bli_dreal(y), bli_dimag(y) )

#ifndef BLIS_ENABLE_C99_COMPLEX

#define bli_scneg2s( x, y )  bli_cneg2ris( bli_sreal(x), bli_simag(x), bli_creal(y), bli_cimag(y) )
#define bli_dcneg2s( x, y )  bli_cneg2ris( bli_dreal(x), bli_dimag(x), bli_creal(y), bli_cimag(y) )
#define bli_ccneg2s( x, y )  bli_cneg2ris( bli_creal(x), bli_cimag(x), bli_creal(y), bli_cimag(y) )
#define bli_zcneg2s( x, y )  bli_cneg2ris( bli_zreal(x), bli_zimag(x), bli_creal(y), bli_cimag(y) )

#define bli_szneg2s( x, y )  bli_zneg2ris( bli_sreal(x), bli_simag(x), bli_zreal(y), bli_zimag(y) )
#define bli_dzneg2s( x, y )  bli_zneg2ris( bli_dreal(x), bli_dimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_czneg2s( x, y )  bli_zneg2ris( bli_creal(x), bli_cimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_zzneg2s( x, y )  bli_zneg2ris( bli_zreal(x), bli_zimag(x), bli_zreal(y), bli_zimag(y) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

#define bli_scneg2s( x, y )  { (y) = -(x); }
#define bli_dcneg2s( x, y )  { (y) = -(x); }
#define bli_ccneg2s( x, y )  { (y) = -(x); }
#define bli_zcneg2s( x, y )  { (y) = -(x); }

#define bli_szneg2s( x, y )  { (y) = -(x); }
#define bli_dzneg2s( x, y )  { (y) = -(x); }
#define bli_czneg2s( x, y )  { (y) = -(x); }
#define bli_zzneg2s( x, y )  { (y) = -(x); }

#endif // BLIS_ENABLE_C99_COMPLEX


#define bli_sneg2s( x, y )  bli_ssneg2s( x, y )
#define bli_dneg2s( x, y )  bli_ddneg2s( x, y )
#define bli_cneg2s( x, y )  bli_ccneg2s( x, y )
#define bli_zneg2s( x, y )  bli_zzneg2s( x, y )


#endif

// end bli_neg2s.h
#line 168 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_rands.h
#line 1 "./frame/include/level0//bli_rands.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_RANDS_H
#define BLIS_RANDS_H

// rands


#define bli_srands( a ) \
{ \
	(a) = ( float  ) (   ( double ) rand() / \
	                   ( ( double ) RAND_MAX / 2.0 ) \
	                 ) - 1.0F; \
}
#define bli_drands( a ) \
{ \
	(a) = ( double ) (   ( double ) rand() / \
	                   ( ( double ) RAND_MAX / 2.0 ) \
	                 ) - 1.0; \
}
#define bli_crands( a ) \
{ \
	float  ar, ai; \
\
	bli_srands( ar ); \
	bli_srands( ai ); \
\
	bli_csets( ar, ai, (a) ); \
}
#define bli_zrands( a ) \
{ \
	double ar, ai; \
\
	bli_drands( ar ); \
	bli_drands( ai ); \
\
	bli_zsets( ar, ai, (a) ); \
}


#endif

// end bli_rands.h
#line 170 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_randnp2s.h
#line 1 "./frame/include/level0//bli_randnp2s.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_RANDNP2S_H
#define BLIS_RANDNP2S_H

// randnp2s


#define bli_srandnp2s( a ) \
{ \
	bli_drandnp2s( a ); \
}

#if 0
#define bli_drandnp2s_prev( a ) \
{ \
	const double m_max  = 3.0; \
	const double m_max2 = m_max + 2.0; \
	double       t; \
	double       r_val; \
\
	/* Compute a narrow-range power of two.

	   For the purposes of commentary, we'll assume that m_max = 4. This
	   represents the largest power of two we will use to generate the
	   random numbers. */ \
\
	/* Generate a random real number t on the interval: [0.0, 6.0]. */ \
	t = ( ( double ) rand() / ( double ) RAND_MAX ) * m_max2; \
\
	/* Modify t to guarantee that is never equal to the upper bound of
	   the interval (in this case, 6.0). */ \
	if ( t == m_max2 ) t = t - 1.0; \
\
	/* Transform the interval into the set of integers, {0,1,2,3,4,5}. */ \
	t = floor( t ); \
\
	/* Map values of t == 0 to a final value of 0. */ \
	if ( t == 0.0 ) r_val = 0.0; \
	else \
	{ \
		/* This case handles values of t = {1,2,3,4,5}. */ \
\
		double s_exp, s_val; \
\
		/* Compute two random numbers to determine the signs of the
		   exponent and the end result. */ \
		PASTEMAC(d,rands)( s_exp ); \
		PASTEMAC(d,rands)( s_val ); \
\
		/* Compute r_val = 2^s where s = +/-(t-1) = {-4,-3,-2,-1,0,1,2,3,4}. */ \
		if ( s_exp < 0.0 ) r_val = pow( 2.0, -(t - 1.0) ); \
		else               r_val = pow( 2.0,   t - 1.0  ); \
\
		/* If our sign value is negative, our random power of two will
		   be negative. */ \
		if ( s_val < 0.0 ) r_val = -r_val; \
	} \
\
	/* Normalize by the largest possible positive value. */ \
	r_val = r_val / pow( 2.0, m_max ); \
\
	/* r_val = 0, or +/-{2^-4, 2^-3, 2^-2, 2^-1, 2^0, 2^1, 2^2, 2^3, 2^4}. */ \
	/* NOTE: For single-precision macros, this assignment results in typecast
	   down to float. */ \
	a = r_val; \
}
#endif

#define bli_drandnp2s( a ) \
{ \
	const double m_max  = 6.0; \
	const double m_max2 = m_max + 2.0; \
	double       t; \
	double       r_val; \
\
	/* Compute a narrow-range power of two.

	   For the purposes of commentary, we'll assume that m_max = 4. This
	   represents the largest power of two we will use to generate the
	   random numbers. */ \
\
	do \
	{ \
		/* Generate a random real number t on the interval: [0.0, 6.0]. */ \
		t = ( ( double ) rand() / ( double ) RAND_MAX ) * m_max2; \
\
		/* Transform the interval into the set of integers, {0,1,2,3,4,5}.
		   Note that 6 is prohibited by the loop guard below. */ \
		t = floor( t ); \
	} \
	/* If t is ever equal to m_max2, we re-randomize. The guard against
	   m_max2 < t is for sanity and shouldn't happen, unless perhaps there
	   is weirdness in the typecasting to double when computing t above. */ \
	while ( m_max2 <= t ); \
\
	/* Map values of t == 0 to a final value of 0. */ \
	if ( t == 0.0 ) r_val = 0.0; \
	else \
	{ \
		/* This case handles values of t = {1,2,3,4,5}. */ \
\
		double s_val; \
\
		/* Compute r_val = 2^s where s = -(t-1) = {-4,-3,-2,-1,0}. */ \
		r_val = pow( 2.0, -(t - 1.0) ); \
\
		/* Compute a random number to determine the sign of the final
		   result. */ \
		PASTEMAC(d,rands)( s_val ); \
\
		/* If our sign value is negative, our random power of two will
		   be negative. */ \
		if ( s_val < 0.0 ) r_val = -r_val; \
	} \
\
	/* r_val = 0, or +/-{2^0, 2^-1, 2^-2, 2^-3, 2^-4}. */ \
	/* NOTE: For single-precision macros, this assignment results in typecast
	   down to float. */ \
	a = r_val; \
}
#define bli_crandnp2s( a ) \
{ \
	float  ar, ai; \
\
	bli_srandnp2s( ar ); \
	bli_srandnp2s( ai ); \
\
	bli_csets( ar, ai, (a) ); \
}
#define bli_zrandnp2s( a ) \
{ \
	double ar, ai; \
\
	bli_drandnp2s( ar ); \
	bli_drandnp2s( ai ); \
\
	bli_zsets( ar, ai, (a) ); \
}


#endif

// end bli_randnp2s.h
#line 171 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_scals.h
#line 1 "./frame/include/level0//bli_scals.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SCALS_H
#define BLIS_SCALS_H

// scals

// Notes:
// - The first char encodes the type of a.
// - The second char encodes the type of y.

#define bli_ssscals( a, y )  bli_sscalris( bli_sreal(a), bli_simag(a), bli_sreal(y), bli_simag(y) )
#define bli_dsscals( a, y )  bli_sscalris( bli_dreal(a), bli_dimag(a), bli_sreal(y), bli_simag(y) )
#define bli_csscals( a, y )  bli_sscalris( bli_creal(a), bli_cimag(a), bli_sreal(y), bli_simag(y) )
#define bli_zsscals( a, y )  bli_sscalris( bli_zreal(a), bli_zimag(a), bli_sreal(y), bli_simag(y) )

#define bli_sdscals( a, y )  bli_dscalris( bli_sreal(a), bli_simag(a), bli_dreal(y), bli_dimag(y) )
#define bli_ddscals( a, y )  bli_dscalris( bli_dreal(a), bli_dimag(a), bli_dreal(y), bli_dimag(y) )
#define bli_cdscals( a, y )  bli_dscalris( bli_creal(a), bli_cimag(a), bli_dreal(y), bli_dimag(y) )
#define bli_zdscals( a, y )  bli_dscalris( bli_zreal(a), bli_zimag(a), bli_dreal(y), bli_dimag(y) )

#ifndef BLIS_ENABLE_C99_COMPLEX

#define bli_scscals( a, y )  bli_scscalris( bli_sreal(a), bli_simag(a), bli_creal(y), bli_cimag(y) )
#define bli_dcscals( a, y )  bli_scscalris( bli_dreal(a), bli_dimag(a), bli_creal(y), bli_cimag(y) )
#define bli_ccscals( a, y )   bli_cscalris( bli_creal(a), bli_cimag(a), bli_creal(y), bli_cimag(y) )
#define bli_zcscals( a, y )   bli_cscalris( bli_zreal(a), bli_zimag(a), bli_creal(y), bli_cimag(y) )

#define bli_szscals( a, y )  bli_dzscalris( bli_sreal(a), bli_simag(a), bli_zreal(y), bli_zimag(y) )
#define bli_dzscals( a, y )  bli_dzscalris( bli_dreal(a), bli_dimag(a), bli_zreal(y), bli_zimag(y) )
#define bli_czscals( a, y )   bli_zscalris( bli_creal(a), bli_cimag(a), bli_zreal(y), bli_zimag(y) )
#define bli_zzscals( a, y )   bli_zscalris( bli_zreal(a), bli_zimag(a), bli_zreal(y), bli_zimag(y) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

#define bli_scscals( a, y )  { (y) *= (a); }
#define bli_dcscals( a, y )  { (y) *= (a); }
#define bli_ccscals( a, y )  { (y) *= (a); }
#define bli_zcscals( a, y )  { (y) *= (a); }

#define bli_szscals( a, y )  { (y) *= (a); }
#define bli_dzscals( a, y )  { (y) *= (a); }
#define bli_czscals( a, y )  { (y) *= (a); }
#define bli_zzscals( a, y )  { (y) *= (a); }

#endif // BLIS_ENABLE_C99_COMPLEX


#define bli_sscals( a, y )  bli_ssscals( a, y )
#define bli_dscals( a, y )  bli_ddscals( a, y )
#define bli_cscals( a, y )  bli_ccscals( a, y )
#define bli_zscals( a, y )  bli_zzscals( a, y )


#endif

// end bli_scals.h
#line 173 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_scaljs.h
#line 1 "./frame/include/level0//bli_scaljs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SCALJS_H
#define BLIS_SCALJS_H

// scaljs

// Notes:
// - The first char encodes the type of a.
// - The second char encodes the type of y.

#define bli_ssscaljs( a, y )  bli_sscaljris( bli_sreal(a), bli_simag(a), bli_sreal(y), bli_simag(y) )
#define bli_dsscaljs( a, y )  bli_sscaljris( bli_dreal(a), bli_dimag(a), bli_sreal(y), bli_simag(y) )
#define bli_csscaljs( a, y )  bli_sscaljris( bli_creal(a), bli_cimag(a), bli_sreal(y), bli_simag(y) )
#define bli_zsscaljs( a, y )  bli_sscaljris( bli_zreal(a), bli_zimag(a), bli_sreal(y), bli_simag(y) )

#define bli_sdscaljs( a, y )  bli_dscaljris( bli_sreal(a), bli_simag(a), bli_dreal(y), bli_dimag(y) )
#define bli_ddscaljs( a, y )  bli_dscaljris( bli_dreal(a), bli_dimag(a), bli_dreal(y), bli_dimag(y) )
#define bli_cdscaljs( a, y )  bli_dscaljris( bli_creal(a), bli_cimag(a), bli_dreal(y), bli_dimag(y) )
#define bli_zdscaljs( a, y )  bli_dscaljris( bli_zreal(a), bli_zimag(a), bli_dreal(y), bli_dimag(y) )

#ifndef BLIS_ENABLE_C99_COMPLEX

#define bli_scscaljs( a, y )  bli_scscaljris( bli_sreal(a), bli_simag(a), bli_creal(y), bli_cimag(y) )
#define bli_dcscaljs( a, y )  bli_scscaljris( bli_dreal(a), bli_dimag(a), bli_creal(y), bli_cimag(y) )
#define bli_ccscaljs( a, y )   bli_cscaljris( bli_creal(a), bli_cimag(a), bli_creal(y), bli_cimag(y) )
#define bli_zcscaljs( a, y )   bli_cscaljris( bli_zreal(a), bli_zimag(a), bli_creal(y), bli_cimag(y) )

#define bli_szscaljs( a, y )  bli_dzscaljris( bli_sreal(a), bli_simag(a), bli_zreal(y), bli_zimag(y) )
#define bli_dzscaljs( a, y )  bli_dzscaljris( bli_dreal(a), bli_dimag(a), bli_zreal(y), bli_zimag(y) )
#define bli_czscaljs( a, y )   bli_zscaljris( bli_creal(a), bli_cimag(a), bli_zreal(y), bli_zimag(y) )
#define bli_zzscaljs( a, y )   bli_zscaljris( bli_zreal(a), bli_zimag(a), bli_zreal(y), bli_zimag(y) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

#define bli_scscaljs( a, y )  { (y) *=      (a); }
#define bli_dcscaljs( a, y )  { (y) *=      (a); }
#define bli_ccscaljs( a, y )  { (y) *= conjf(a); }
#define bli_zcscaljs( a, y )  { (y) *= conj (a); }

#define bli_szscaljs( a, y )  { (y) *=      (a); }
#define bli_dzscaljs( a, y )  { (y) *=      (a); }
#define bli_czscaljs( a, y )  { (y) *= conjf(a); }
#define bli_zzscaljs( a, y )  { (y) *= conj (a); }

#endif // BLIS_ENABLE_C99_COMPLEX


#define bli_sscaljs( a, y )  bli_ssscaljs( a, y )
#define bli_dscaljs( a, y )  bli_ddscaljs( a, y )
#define bli_cscaljs( a, y )  bli_ccscaljs( a, y )
#define bli_zscaljs( a, y )  bli_zzscaljs( a, y )


#endif

// end bli_scaljs.h
#line 174 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_scalcjs.h
#line 1 "./frame/include/level0//bli_scalcjs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SCALCJS_H
#define BLIS_SCALCJS_H

// scalcjs

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of y.

#define bli_ssscalcjs( conjx, x, y )  bli_sscalcjris( conjx, bli_sreal(x), bli_simag(x), bli_sreal(y), bli_simag(y) )
#define bli_dsscalcjs( conjx, x, y )  bli_sscalcjris( conjx, bli_dreal(x), bli_dimag(x), bli_sreal(y), bli_simag(y) )
#define bli_csscalcjs( conjx, x, y )  bli_sscalcjris( conjx, bli_creal(x), bli_cimag(x), bli_sreal(y), bli_simag(y) )
#define bli_zsscalcjs( conjx, x, y )  bli_sscalcjris( conjx, bli_zreal(x), bli_zimag(x), bli_sreal(y), bli_simag(y) )

#define bli_sdscalcjs( conjx, x, y )  bli_dscalcjris( conjx, bli_sreal(x), bli_simag(x), bli_dreal(y), bli_dimag(y) )
#define bli_ddscalcjs( conjx, x, y )  bli_dscalcjris( conjx, bli_dreal(x), bli_dimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_cdscalcjs( conjx, x, y )  bli_dscalcjris( conjx, bli_creal(x), bli_cimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_zdscalcjs( conjx, x, y )  bli_dscalcjris( conjx, bli_zreal(x), bli_zimag(x), bli_dreal(y), bli_dimag(y) )

#ifndef BLIS_ENABLE_C99_COMPLEX

#define bli_scscalcjs( conjx, x, y )  bli_scscalcjris( conjx, bli_sreal(x), bli_simag(x), bli_creal(y), bli_cimag(y) )
#define bli_dcscalcjs( conjx, x, y )  bli_scscalcjris( conjx, bli_dreal(x), bli_dimag(x), bli_creal(y), bli_cimag(y) )
#define bli_ccscalcjs( conjx, x, y )   bli_cscalcjris( conjx, bli_creal(x), bli_cimag(x), bli_creal(y), bli_cimag(y) )
#define bli_zcscalcjs( conjx, x, y )   bli_cscalcjris( conjx, bli_zreal(x), bli_zimag(x), bli_creal(y), bli_cimag(y) )

#define bli_szscalcjs( conjx, x, y )  bli_dzscalcjris( conjx, bli_sreal(x), bli_simag(x), bli_zreal(y), bli_zimag(y) )
#define bli_dzscalcjs( conjx, x, y )  bli_dzscalcjris( conjx, bli_dreal(x), bli_dimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_czscalcjs( conjx, x, y )   bli_zscalcjris( conjx, bli_creal(x), bli_cimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_zzscalcjs( conjx, x, y )   bli_zscalcjris( conjx, bli_zreal(x), bli_zimag(x), bli_zreal(y), bli_zimag(y) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

#define bli_scscalcjs( conjx, x, y )  { (y) *= (x); }
#define bli_dcscalcjs( conjx, x, y )  { (y) *= (x); }
#define bli_ccscalcjs( conjx, x, y )  { (y) *= ( bli_is_conj( conjx ) ? conjf(x) : (x) ); }
#define bli_zcscalcjs( conjx, x, y )  { (y) *= ( bli_is_conj( conjx ) ? conj (x) : (x) ); }

#define bli_szscalcjs( conjx, x, y )  { (y) *= (x); }
#define bli_dzscalcjs( conjx, x, y )  { (y) *= (x); }
#define bli_czscalcjs( conjx, x, y )  { (y) *= ( bli_is_conj( conjx ) ? conjf(x) : (x) ); }
#define bli_zzscalcjs( conjx, x, y )  { (y) *= ( bli_is_conj( conjx ) ? conj (x) : (x) ); }

#endif // BLIS_ENABLE_C99_COMPLEX


#define bli_sscalcjs( conjx, x, y )  bli_ssscalcjs( conjx, x, y )
#define bli_dscalcjs( conjx, x, y )  bli_ddscalcjs( conjx, x, y )
#define bli_cscalcjs( conjx, x, y )  bli_ccscalcjs( conjx, x, y )
#define bli_zscalcjs( conjx, x, y )  bli_zzscalcjs( conjx, x, y )


#endif

// end bli_scalcjs.h
#line 175 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_scal2s.h
#line 1 "./frame/include/level0//bli_scal2s.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SCAL2S_H
#define BLIS_SCAL2S_H

// scal2s

// Notes:
// - The first char encodes the type of a.
// - The second char encodes the type of x.
// - The third char encodes the type of y.

// -- (axy) = (??s) ------------------------------------------------------------

#define bli_sssscal2s( a, x, y )  bli_rxscal2ris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_sreal(y), bli_simag(y) )
#define bli_dssscal2s( a, x, y )  bli_rxscal2ris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_sreal(y), bli_simag(y) )
#define bli_cssscal2s( a, x, y )  bli_rxscal2ris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_sreal(y), bli_simag(y) )
#define bli_zssscal2s( a, x, y )  bli_rxscal2ris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_sreal(y), bli_simag(y) )

#define bli_sdsscal2s( a, x, y )  bli_rxscal2ris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_sreal(y), bli_simag(y) )
#define bli_ddsscal2s( a, x, y )  bli_rxscal2ris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(y), bli_simag(y) )
#define bli_cdsscal2s( a, x, y )  bli_rxscal2ris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(y), bli_simag(y) )
#define bli_zdsscal2s( a, x, y )  bli_rxscal2ris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(y), bli_simag(y) )

#define bli_scsscal2s( a, x, y )  bli_rxscal2ris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_sreal(y), bli_simag(y) )
#define bli_dcsscal2s( a, x, y )  bli_rxscal2ris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_sreal(y), bli_simag(y) )
#define bli_ccsscal2s( a, x, y )  bli_roscal2ris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_sreal(y), bli_simag(y) )
#define bli_zcsscal2s( a, x, y )  bli_roscal2ris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_sreal(y), bli_simag(y) )

#define bli_szsscal2s( a, x, y )  bli_rxscal2ris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_sreal(y), bli_simag(y) )
#define bli_dzsscal2s( a, x, y )  bli_rxscal2ris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(y), bli_simag(y) )
#define bli_czsscal2s( a, x, y )  bli_roscal2ris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(y), bli_simag(y) )
#define bli_zzsscal2s( a, x, y )  bli_roscal2ris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(y), bli_simag(y) )

// -- (axy) = (??d) ------------------------------------------------------------

#define bli_ssdscal2s( a, x, y )  bli_rxscal2ris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_dreal(y), bli_dimag(y) )
#define bli_dsdscal2s( a, x, y )  bli_rxscal2ris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_dreal(y), bli_dimag(y) )
#define bli_csdscal2s( a, x, y )  bli_rxscal2ris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_dreal(y), bli_dimag(y) )
#define bli_zsdscal2s( a, x, y )  bli_rxscal2ris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_dreal(y), bli_dimag(y) )

#define bli_sddscal2s( a, x, y )  bli_rxscal2ris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_dddscal2s( a, x, y )  bli_rxscal2ris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_cddscal2s( a, x, y )  bli_rxscal2ris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_zddscal2s( a, x, y )  bli_rxscal2ris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(y), bli_dimag(y) )

#define bli_scdscal2s( a, x, y )  bli_rxscal2ris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_dcdscal2s( a, x, y )  bli_rxscal2ris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_ccdscal2s( a, x, y )  bli_roscal2ris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_zcdscal2s( a, x, y )  bli_roscal2ris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_dreal(y), bli_dimag(y) )

#define bli_szdscal2s( a, x, y )  bli_rxscal2ris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_dzdscal2s( a, x, y )  bli_rxscal2ris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_czdscal2s( a, x, y )  bli_roscal2ris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_zzdscal2s( a, x, y )  bli_roscal2ris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(y), bli_dimag(y) )

#ifndef BLIS_ENABLE_C99_COMPLEX

// -- (axy) = (??c) ------------------------------------------------------------

#define bli_sscscal2s( a, x, y )  bli_rxscal2ris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_creal(y), bli_cimag(y) )
#define bli_dscscal2s( a, x, y )  bli_rxscal2ris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_creal(y), bli_cimag(y) )
#define bli_cscscal2s( a, x, y )  bli_rcscal2ris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_creal(y), bli_cimag(y) )
#define bli_zscscal2s( a, x, y )  bli_rcscal2ris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_creal(y), bli_cimag(y) )

#define bli_sdcscal2s( a, x, y )  bli_rxscal2ris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_creal(y), bli_cimag(y) )
#define bli_ddcscal2s( a, x, y )  bli_rxscal2ris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_creal(y), bli_cimag(y) )
#define bli_cdcscal2s( a, x, y )  bli_rcscal2ris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_creal(y), bli_cimag(y) )
#define bli_zdcscal2s( a, x, y )  bli_rcscal2ris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_creal(y), bli_cimag(y) )

#define bli_sccscal2s( a, x, y )  bli_crscal2ris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_creal(y), bli_cimag(y) )
#define bli_dccscal2s( a, x, y )  bli_crscal2ris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_creal(y), bli_cimag(y) )
#define bli_cccscal2s( a, x, y )  bli_cxscal2ris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_creal(y), bli_cimag(y) )
#define bli_zccscal2s( a, x, y )  bli_cxscal2ris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_creal(y), bli_cimag(y) )

#define bli_szcscal2s( a, x, y )  bli_crscal2ris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_creal(y), bli_cimag(y) )
#define bli_dzcscal2s( a, x, y )  bli_crscal2ris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_creal(y), bli_cimag(y) )
#define bli_czcscal2s( a, x, y )  bli_cxscal2ris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_creal(y), bli_cimag(y) )
#define bli_zzcscal2s( a, x, y )  bli_cxscal2ris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_creal(y), bli_cimag(y) )

// -- (axy) = (??z) ------------------------------------------------------------

#define bli_sszscal2s( a, x, y )  bli_rxscal2ris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_zreal(y), bli_zimag(y) )
#define bli_dszscal2s( a, x, y )  bli_rxscal2ris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_zreal(y), bli_zimag(y) )
#define bli_cszscal2s( a, x, y )  bli_rcscal2ris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_zreal(y), bli_zimag(y) )
#define bli_zszscal2s( a, x, y )  bli_rcscal2ris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_zreal(y), bli_zimag(y) )

#define bli_sdzscal2s( a, x, y )  bli_rxscal2ris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_ddzscal2s( a, x, y )  bli_rxscal2ris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_cdzscal2s( a, x, y )  bli_rcscal2ris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_zdzscal2s( a, x, y )  bli_rcscal2ris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(y), bli_zimag(y) )

#define bli_sczscal2s( a, x, y )  bli_crscal2ris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_dczscal2s( a, x, y )  bli_crscal2ris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_cczscal2s( a, x, y )  bli_cxscal2ris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_zczscal2s( a, x, y )  bli_cxscal2ris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_zreal(y), bli_zimag(y) )

#define bli_szzscal2s( a, x, y )  bli_crscal2ris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_dzzscal2s( a, x, y )  bli_crscal2ris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_czzscal2s( a, x, y )  bli_cxscal2ris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_zzzscal2s( a, x, y )  bli_cxscal2ris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(y), bli_zimag(y) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

// -- (axy) = (??c) ------------------------------------------------------------

#define bli_sscscal2s( a, x, y )  { (y) = (a) * (x); }
#define bli_dscscal2s( a, x, y )  { (y) = (a) * (x); }
#define bli_cscscal2s( a, x, y )  { (y) = (a) * (x); }
#define bli_zscscal2s( a, x, y )  { (y) = (a) * (x); }

#define bli_sdcscal2s( a, x, y )  { (y) = (a) * (x); }
#define bli_ddcscal2s( a, x, y )  { (y) = (a) * (x); }
#define bli_cdcscal2s( a, x, y )  { (y) = (a) * (x); }
#define bli_zdcscal2s( a, x, y )  { (y) = (a) * (x); }

#define bli_sccscal2s( a, x, y )  { (y) = (a) * (x); }
#define bli_dccscal2s( a, x, y )  { (y) = (a) * (x); }
#define bli_cccscal2s( a, x, y )  { (y) = (a) * (x); }
#define bli_zccscal2s( a, x, y )  { (y) = (a) * (x); }

#define bli_szcscal2s( a, x, y )  { (y) = (a) * (x); }
#define bli_dzcscal2s( a, x, y )  { (y) = (a) * (x); }
#define bli_czcscal2s( a, x, y )  { (y) = (a) * (x); }
#define bli_zzcscal2s( a, x, y )  { (y) = (a) * (x); }

// -- (axy) = (??z) ------------------------------------------------------------

#define bli_sszscal2s( a, x, y )  { (y) = (a) * (x); }
#define bli_dszscal2s( a, x, y )  { (y) = (a) * (x); }
#define bli_cszscal2s( a, x, y )  { (y) = (a) * (x); }
#define bli_zszscal2s( a, x, y )  { (y) = (a) * (x); }

#define bli_sdzscal2s( a, x, y )  { (y) = (a) * (x); }
#define bli_ddzscal2s( a, x, y )  { (y) = (a) * (x); }
#define bli_cdzscal2s( a, x, y )  { (y) = (a) * (x); }
#define bli_zdzscal2s( a, x, y )  { (y) = (a) * (x); }

#define bli_sczscal2s( a, x, y )  { (y) = (a) * (x); }
#define bli_dczscal2s( a, x, y )  { (y) = (a) * (x); }
#define bli_cczscal2s( a, x, y )  { (y) = (a) * (x); }
#define bli_zczscal2s( a, x, y )  { (y) = (a) * (x); }

#define bli_szzscal2s( a, x, y )  { (y) = (a) * (x); }
#define bli_dzzscal2s( a, x, y )  { (y) = (a) * (x); }
#define bli_czzscal2s( a, x, y )  { (y) = (a) * (x); }
#define bli_zzzscal2s( a, x, y )  { (y) = (a) * (x); }

#endif // BLIS_ENABLE_C99_COMPLEX


#define bli_sscal2s( a, x, y )  bli_sssscal2s( a, x, y )
#define bli_dscal2s( a, x, y )  bli_dddscal2s( a, x, y )
#define bli_cscal2s( a, x, y )  bli_cccscal2s( a, x, y )
#define bli_zscal2s( a, x, y )  bli_zzzscal2s( a, x, y )


#endif

// end bli_scal2s.h
#line 177 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_scal2js.h
#line 1 "./frame/include/level0//bli_scal2js.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SCAL2JS_H
#define BLIS_SCAL2JS_H

// scal2js

// Notes:
// - The first char encodes the type of a.
// - The second char encodes the type of x.
// - The third char encodes the type of y.


// -- (axy) = (??s) ------------------------------------------------------------

#define bli_sssscal2js( a, x, y )  bli_rxscal2jris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_sreal(y), bli_simag(y) )
#define bli_dssscal2js( a, x, y )  bli_rxscal2jris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_sreal(y), bli_simag(y) )
#define bli_cssscal2js( a, x, y )  bli_rxscal2jris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_sreal(y), bli_simag(y) )
#define bli_zssscal2js( a, x, y )  bli_rxscal2jris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_sreal(y), bli_simag(y) )

#define bli_sdsscal2js( a, x, y )  bli_rxscal2jris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_sreal(y), bli_simag(y) )
#define bli_ddsscal2js( a, x, y )  bli_rxscal2jris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(y), bli_simag(y) )
#define bli_cdsscal2js( a, x, y )  bli_rxscal2jris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(y), bli_simag(y) )
#define bli_zdsscal2js( a, x, y )  bli_rxscal2jris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_sreal(y), bli_simag(y) )

#define bli_scsscal2js( a, x, y )  bli_rxscal2jris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_sreal(y), bli_simag(y) )
#define bli_dcsscal2js( a, x, y )  bli_rxscal2jris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_sreal(y), bli_simag(y) )
#define bli_ccsscal2js( a, x, y )  bli_roscal2jris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_sreal(y), bli_simag(y) )
#define bli_zcsscal2js( a, x, y )  bli_roscal2jris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_sreal(y), bli_simag(y) )

#define bli_szsscal2js( a, x, y )  bli_rxscal2jris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_sreal(y), bli_simag(y) )
#define bli_dzsscal2js( a, x, y )  bli_rxscal2jris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(y), bli_simag(y) )
#define bli_czsscal2js( a, x, y )  bli_roscal2jris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(y), bli_simag(y) )
#define bli_zzsscal2js( a, x, y )  bli_roscal2jris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_sreal(y), bli_simag(y) )

// -- (axy) = (??d) ------------------------------------------------------------

#define bli_ssdscal2js( a, x, y )  bli_rxscal2jris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_dreal(y), bli_dimag(y) )
#define bli_dsdscal2js( a, x, y )  bli_rxscal2jris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_dreal(y), bli_dimag(y) )
#define bli_csdscal2js( a, x, y )  bli_rxscal2jris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_dreal(y), bli_dimag(y) )
#define bli_zsdscal2js( a, x, y )  bli_rxscal2jris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_dreal(y), bli_dimag(y) )

#define bli_sddscal2js( a, x, y )  bli_rxscal2jris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_dddscal2js( a, x, y )  bli_rxscal2jris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_cddscal2js( a, x, y )  bli_rxscal2jris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_zddscal2js( a, x, y )  bli_rxscal2jris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_dreal(y), bli_dimag(y) )

#define bli_scdscal2js( a, x, y )  bli_rxscal2jris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_dcdscal2js( a, x, y )  bli_rxscal2jris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_ccdscal2js( a, x, y )  bli_roscal2jris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_zcdscal2js( a, x, y )  bli_roscal2jris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_dreal(y), bli_dimag(y) )

#define bli_szdscal2js( a, x, y )  bli_rxscal2jris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_dzdscal2js( a, x, y )  bli_rxscal2jris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_czdscal2js( a, x, y )  bli_roscal2jris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(y), bli_dimag(y) )
#define bli_zzdscal2js( a, x, y )  bli_roscal2jris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_dreal(y), bli_dimag(y) )

#ifndef BLIS_ENABLE_C99_COMPLEX

// -- (axy) = (??c) ------------------------------------------------------------

#define bli_sscscal2js( a, x, y )  bli_rxscal2jris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_creal(y), bli_cimag(y) )
#define bli_dscscal2js( a, x, y )  bli_rxscal2jris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_creal(y), bli_cimag(y) )
#define bli_cscscal2js( a, x, y )  bli_rcscal2jris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_creal(y), bli_cimag(y) )
#define bli_zscscal2js( a, x, y )  bli_rcscal2jris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_creal(y), bli_cimag(y) )

#define bli_sdcscal2js( a, x, y )  bli_rxscal2jris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_creal(y), bli_cimag(y) )
#define bli_ddcscal2js( a, x, y )  bli_rxscal2jris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_creal(y), bli_cimag(y) )
#define bli_cdcscal2js( a, x, y )  bli_rcscal2jris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_creal(y), bli_cimag(y) )
#define bli_zdcscal2js( a, x, y )  bli_rcscal2jris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_creal(y), bli_cimag(y) )

#define bli_sccscal2js( a, x, y )  bli_crscal2jris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_creal(y), bli_cimag(y) )
#define bli_dccscal2js( a, x, y )  bli_crscal2jris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_creal(y), bli_cimag(y) )
#define bli_cccscal2js( a, x, y )  bli_cxscal2jris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_creal(y), bli_cimag(y) )
#define bli_zccscal2js( a, x, y )  bli_cxscal2jris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_creal(y), bli_cimag(y) )

#define bli_szcscal2js( a, x, y )  bli_crscal2jris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_creal(y), bli_cimag(y) )
#define bli_dzcscal2js( a, x, y )  bli_crscal2jris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_creal(y), bli_cimag(y) )
#define bli_czcscal2js( a, x, y )  bli_cxscal2jris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_creal(y), bli_cimag(y) )
#define bli_zzcscal2js( a, x, y )  bli_cxscal2jris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_creal(y), bli_cimag(y) )

// -- (axy) = (??z) ------------------------------------------------------------

#define bli_sszscal2js( a, x, y )  bli_rxscal2jris( bli_sreal(a), bli_simag(a), bli_sreal(x), bli_simag(x), bli_zreal(y), bli_zimag(y) )
#define bli_dszscal2js( a, x, y )  bli_rxscal2jris( bli_dreal(a), bli_dimag(a), bli_sreal(x), bli_simag(x), bli_zreal(y), bli_zimag(y) )
#define bli_cszscal2js( a, x, y )  bli_rcscal2jris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), bli_zreal(y), bli_zimag(y) )
#define bli_zszscal2js( a, x, y )  bli_rcscal2jris( bli_zreal(a), bli_zimag(a), bli_sreal(x), bli_simag(x), bli_zreal(y), bli_zimag(y) )

#define bli_sdzscal2js( a, x, y )  bli_rxscal2jris( bli_sreal(a), bli_simag(a), bli_dreal(x), bli_dimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_ddzscal2js( a, x, y )  bli_rxscal2jris( bli_dreal(a), bli_dimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_cdzscal2js( a, x, y )  bli_rcscal2jris( bli_creal(a), bli_cimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_zdzscal2js( a, x, y )  bli_rcscal2jris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), bli_zreal(y), bli_zimag(y) )

#define bli_sczscal2js( a, x, y )  bli_crscal2jris( bli_sreal(a), bli_simag(a), bli_creal(x), bli_cimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_dczscal2js( a, x, y )  bli_crscal2jris( bli_dreal(a), bli_dimag(a), bli_creal(x), bli_cimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_cczscal2js( a, x, y )  bli_cxscal2jris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_zczscal2js( a, x, y )  bli_cxscal2jris( bli_zreal(a), bli_zimag(a), bli_creal(x), bli_cimag(x), bli_zreal(y), bli_zimag(y) )

#define bli_szzscal2js( a, x, y )  bli_crscal2jris( bli_sreal(a), bli_simag(a), bli_zreal(x), bli_zimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_dzzscal2js( a, x, y )  bli_crscal2jris( bli_dreal(a), bli_dimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_czzscal2js( a, x, y )  bli_cxscal2jris( bli_creal(a), bli_cimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(y), bli_zimag(y) )
#define bli_zzzscal2js( a, x, y )  bli_cxscal2jris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), bli_zreal(y), bli_zimag(y) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

// -- (axy) = (??c) ------------------------------------------------------------

#define bli_sscscal2js( a, x, y )  { (y) = (a) * (x); }
#define bli_dscscal2js( a, x, y )  { (y) = (a) * (x); }
#define bli_cscscal2js( a, x, y )  { (y) = (a) * (x); }
#define bli_zscscal2js( a, x, y )  { (y) = (a) * (x); }

#define bli_sdcscal2js( a, x, y )  { (y) = (a) * (x); }
#define bli_ddcscal2js( a, x, y )  { (y) = (a) * (x); }
#define bli_cdcscal2js( a, x, y )  { (y) = (a) * (x); }
#define bli_zdcscal2js( a, x, y )  { (y) = (a) * (x); }

#define bli_sccscal2js( a, x, y )  { (y) = (a) * conjf(x); }
#define bli_dccscal2js( a, x, y )  { (y) = (a) * conjf(x); }
#define bli_cccscal2js( a, x, y )  { (y) = (a) * conjf(x); }
#define bli_zccscal2js( a, x, y )  { (y) = (a) * conjf(x); }

#define bli_szcscal2js( a, x, y )  { (y) = (a) * conj(x); }
#define bli_dzcscal2js( a, x, y )  { (y) = (a) * conj(x); }
#define bli_czcscal2js( a, x, y )  { (y) = (a) * conj(x); }
#define bli_zzcscal2js( a, x, y )  { (y) = (a) * conj(x); }

// -- (axy) = (??z) ------------------------------------------------------------

#define bli_sszscal2js( a, x, y )  { (y) = (a) * (x); }
#define bli_dszscal2js( a, x, y )  { (y) = (a) * (x); }
#define bli_cszscal2js( a, x, y )  { (y) = (a) * (x); }
#define bli_zszscal2js( a, x, y )  { (y) = (a) * (x); }

#define bli_sdzscal2js( a, x, y )  { (y) = (a) * (x); }
#define bli_ddzscal2js( a, x, y )  { (y) = (a) * (x); }
#define bli_cdzscal2js( a, x, y )  { (y) = (a) * (x); }
#define bli_zdzscal2js( a, x, y )  { (y) = (a) * (x); }

#define bli_sczscal2js( a, x, y )  { (y) = (a) * conjf(x); }
#define bli_dczscal2js( a, x, y )  { (y) = (a) * conjf(x); }
#define bli_cczscal2js( a, x, y )  { (y) = (a) * conjf(x); }
#define bli_zczscal2js( a, x, y )  { (y) = (a) * conjf(x); }

#define bli_szzscal2js( a, x, y )  { (y) = (a) * conj(x); }
#define bli_dzzscal2js( a, x, y )  { (y) = (a) * conj(x); }
#define bli_czzscal2js( a, x, y )  { (y) = (a) * conj(x); }
#define bli_zzzscal2js( a, x, y )  { (y) = (a) * conj(x); }

#endif // BLIS_ENABLE_C99_COMPLEX


#define bli_sscal2js( a, x, y )  bli_sssscal2js( a, x, y )
#define bli_dscal2js( a, x, y )  bli_dddscal2js( a, x, y )
#define bli_cscal2js( a, x, y )  bli_cccscal2js( a, x, y )
#define bli_zscal2js( a, x, y )  bli_zzzscal2js( a, x, y )


#endif

// end bli_scal2js.h
#line 178 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_set0s.h
#line 1 "./frame/include/level0//bli_set0s.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SET0S_H
#define BLIS_SET0S_H

#define bli_sset0s( a )  bli_ssets( 0.0F, 0.0F, (a) )
#define bli_dset0s( a )  bli_dsets( 0.0 , 0.0 , (a) )
#define bli_cset0s( a )  bli_csets( 0.0F, 0.0F, (a) )
#define bli_zset0s( a )  bli_zsets( 0.0 , 0.0 , (a) )

#endif

// end bli_set0s.h
#line 180 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_set1s.h
#line 1 "./frame/include/level0//bli_set1s.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SET1S_H
#define BLIS_SET1S_H

#define bli_sset1s( a )  bli_ssets( 1.0F, 0.0F, (a) )
#define bli_dset1s( a )  bli_dsets( 1.0 , 0.0 , (a) )
#define bli_cset1s( a )  bli_csets( 1.0F, 0.0F, (a) )
#define bli_zset1s( a )  bli_zsets( 1.0 , 0.0 , (a) )

#endif

// end bli_set1s.h
#line 182 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_seti0s.h
#line 1 "./frame/include/level0//bli_seti0s.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SETI0S_H
#define BLIS_SETI0S_H

#define bli_sseti0s( a )  bli_ssetis( 0.0F, (a) )
#define bli_dseti0s( a )  bli_dsetis( 0.0 , (a) )
#define bli_cseti0s( a )  bli_csetis( 0.0F, (a) )
#define bli_zseti0s( a )  bli_zsetis( 0.0 , (a) )

#endif

// end bli_seti0s.h
#line 184 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_sqrt2s.h
#line 1 "./frame/include/level0//bli_sqrt2s.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SQRT2S_H
#define BLIS_SQRT2S_H

// sqrt2s

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of a.

#ifndef BLIS_ENABLE_C99_COMPLEX

#define bli_sssqrt2s( x, a )  bli_ssqrt2ris( bli_sreal(x), bli_simag(x), bli_sreal(a), bli_simag(a) )
#define bli_dssqrt2s( x, a )  bli_ssqrt2ris( bli_dreal(x), bli_dimag(x), bli_sreal(a), bli_simag(a) )
#define bli_cssqrt2s( x, a )  bli_ssqrt2ris( bli_creal(x), bli_cimag(x), bli_sreal(a), bli_simag(a) )
#define bli_zssqrt2s( x, a )  bli_ssqrt2ris( bli_zreal(x), bli_zimag(x), bli_sreal(a), bli_simag(a) )

#define bli_sdsqrt2s( x, a )  bli_dsqrt2ris( bli_sreal(x), bli_simag(x), bli_dreal(a), bli_dimag(a) )
#define bli_ddsqrt2s( x, a )  bli_dsqrt2ris( bli_dreal(x), bli_dimag(x), bli_dreal(a), bli_dimag(a) )
#define bli_cdsqrt2s( x, a )  bli_dsqrt2ris( bli_creal(x), bli_cimag(x), bli_dreal(a), bli_dimag(a) )
#define bli_zdsqrt2s( x, a )  bli_dsqrt2ris( bli_zreal(x), bli_zimag(x), bli_dreal(a), bli_dimag(a) )

#define bli_scsqrt2s( x, a )  bli_scsqrt2ris( bli_sreal(x), bli_simag(x), bli_creal(a), bli_cimag(a) )
#define bli_dcsqrt2s( x, a )  bli_scsqrt2ris( bli_dreal(x), bli_dimag(x), bli_creal(a), bli_cimag(a) )
#define bli_ccsqrt2s( x, a )   bli_csqrt2ris( bli_creal(x), bli_cimag(x), bli_creal(a), bli_cimag(a) )
#define bli_zcsqrt2s( x, a )   bli_csqrt2ris( bli_zreal(x), bli_zimag(x), bli_creal(a), bli_cimag(a) )

#define bli_szsqrt2s( x, a )  bli_dzsqrt2ris( bli_sreal(x), bli_simag(x), bli_zreal(a), bli_zimag(a) )
#define bli_dzsqrt2s( x, a )  bli_dzsqrt2ris( bli_dreal(x), bli_dimag(x), bli_zreal(a), bli_zimag(a) )
#define bli_czsqrt2s( x, a )   bli_zsqrt2ris( bli_creal(x), bli_cimag(x), bli_zreal(a), bli_zimag(a) )
#define bli_zzsqrt2s( x, a )   bli_zsqrt2ris( bli_zreal(x), bli_zimag(x), bli_zreal(a), bli_zimag(a) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

#define bli_sssqrt2s( x, a )  { (a) = ( float    )            sqrtf( (x) )  ; }
#define bli_dssqrt2s( x, a )  { (a) = ( float    )            sqrt ( (x) )  ; }
#define bli_cssqrt2s( x, a )  { (a) = ( float    )bli_creal( csqrtf( (x) ) ); }
#define bli_zssqrt2s( x, a )  { (a) = ( float    )bli_zreal( csqrt ( (x) ) ); }

#define bli_sdsqrt2s( x, a )  { (a) = ( double   )            sqrtf( (x) )  ; }
#define bli_ddsqrt2s( x, a )  { (a) = ( double   )            sqrt ( (x) )  ; }
#define bli_cdsqrt2s( x, a )  { (a) = ( double   )bli_creal( csqrtf( (x) ) ); }
#define bli_zdsqrt2s( x, a )  { (a) = ( double   )bli_zreal( csqrt ( (x) ) ); }

#define bli_scsqrt2s( x, a )  { (a) = ( scomplex )            sqrtf( (x) )  ; }
#define bli_dcsqrt2s( x, a )  { (a) = ( scomplex )            sqrt ( (x) )  ; }
#define bli_ccsqrt2s( x, a )  { (a) = ( scomplex )           csqrtf( (x) )  ; }
#define bli_zcsqrt2s( x, a )  { (a) = ( scomplex )           csqrt ( (x) )  ; }

#define bli_szsqrt2s( x, a )  { (a) = ( dcomplex )            sqrtf( (x) )  ; }
#define bli_dzsqrt2s( x, a )  { (a) = ( dcomplex )            sqrt ( (x) )  ; }
#define bli_czsqrt2s( x, a )  { (a) = ( dcomplex )           csqrtf( (x) )  ; }
#define bli_zzsqrt2s( x, a )  { (a) = ( dcomplex )           csqrt ( (x) )  ; }

#endif // BLIS_ENABLE_C99_COMPLEX


#define bli_ssqrt2s( x, a )  bli_sssqrt2s( x, a )
#define bli_dsqrt2s( x, a )  bli_ddsqrt2s( x, a )
#define bli_csqrt2s( x, a )  bli_ccsqrt2s( x, a )
#define bli_zsqrt2s( x, a )  bli_zzsqrt2s( x, a )


#endif
// end bli_sqrt2s.h
#line 186 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_subs.h
#line 1 "./frame/include/level0//bli_subs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SUBS_H
#define BLIS_SUBS_H

// subs

// Notes:
// - The first char encodes the type of a.
// - The second char encodes the type of y.

#define bli_sssubs( a, y )  bli_ssubris( bli_sreal(a), bli_simag(a), bli_sreal(y), bli_simag(y) )
#define bli_dssubs( a, y )  bli_ssubris( bli_dreal(a), bli_dimag(a), bli_sreal(y), bli_simag(y) )
#define bli_cssubs( a, y )  bli_ssubris( bli_creal(a), bli_cimag(a), bli_sreal(y), bli_simag(y) )
#define bli_zssubs( a, y )  bli_ssubris( bli_zreal(a), bli_zimag(a), bli_sreal(y), bli_simag(y) )

#define bli_sdsubs( a, y )  bli_dsubris( bli_sreal(a), bli_simag(a), bli_dreal(y), bli_dimag(y) )
#define bli_ddsubs( a, y )  bli_dsubris( bli_dreal(a), bli_dimag(a), bli_dreal(y), bli_dimag(y) )
#define bli_cdsubs( a, y )  bli_dsubris( bli_creal(a), bli_cimag(a), bli_dreal(y), bli_dimag(y) )
#define bli_zdsubs( a, y )  bli_dsubris( bli_zreal(a), bli_zimag(a), bli_dreal(y), bli_dimag(y) )

#ifndef BLIS_ENABLE_C99_COMPLEX

#define bli_scsubs( a, y )  bli_csubris( bli_sreal(a), bli_simag(a), bli_creal(y), bli_cimag(y) )
#define bli_dcsubs( a, y )  bli_csubris( bli_dreal(a), bli_dimag(a), bli_creal(y), bli_cimag(y) )
#define bli_ccsubs( a, y )  bli_csubris( bli_creal(a), bli_cimag(a), bli_creal(y), bli_cimag(y) )
#define bli_zcsubs( a, y )  bli_csubris( bli_zreal(a), bli_zimag(a), bli_creal(y), bli_cimag(y) )

#define bli_szsubs( a, y )  bli_zsubris( bli_sreal(a), bli_simag(a), bli_zreal(y), bli_zimag(y) )
#define bli_dzsubs( a, y )  bli_zsubris( bli_dreal(a), bli_dimag(a), bli_zreal(y), bli_zimag(y) )
#define bli_czsubs( a, y )  bli_zsubris( bli_creal(a), bli_cimag(a), bli_zreal(y), bli_zimag(y) )
#define bli_zzsubs( a, y )  bli_zsubris( bli_zreal(a), bli_zimag(a), bli_zreal(y), bli_zimag(y) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

#define bli_scsubs( a, y )  { (y) -= (a); }
#define bli_dcsubs( a, y )  { (y) -= (a); }
#define bli_ccsubs( a, y )  { (y) -= (a); }
#define bli_zcsubs( a, y )  { (y) -= (a); }

#define bli_szsubs( a, y )  { (y) -= (a); }
#define bli_dzsubs( a, y )  { (y) -= (a); }
#define bli_czsubs( a, y )  { (y) -= (a); }
#define bli_zzsubs( a, y )  { (y) -= (a); }

#endif // BLIS_ENABLE_C99_COMPLEX


#define bli_ssubs( a, y )  bli_sssubs( a, y )
#define bli_dsubs( a, y )  bli_ddsubs( a, y )
#define bli_csubs( a, y )  bli_ccsubs( a, y )
#define bli_zsubs( a, y )  bli_zzsubs( a, y )


#endif

// end bli_subs.h
#line 188 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_subjs.h
#line 1 "./frame/include/level0//bli_subjs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SUBJS_H
#define BLIS_SUBJS_H

// subjs

// Notes:
// - The first char encodes the type of a.
// - The second char encodes the type of y.

#define bli_sssubjs( a, y )  bli_ssubjris( bli_sreal(a), bli_simag(a), bli_sreal(y), bli_simag(y) )
#define bli_dssubjs( a, y )  bli_ssubjris( bli_dreal(a), bli_dimag(a), bli_sreal(y), bli_simag(y) )
#define bli_cssubjs( a, y )  bli_ssubjris( bli_creal(a), bli_cimag(a), bli_sreal(y), bli_simag(y) )
#define bli_zssubjs( a, y )  bli_ssubjris( bli_zreal(a), bli_zimag(a), bli_sreal(y), bli_simag(y) )

#define bli_sdsubjs( a, y )  bli_dsubjris( bli_sreal(a), bli_simag(a), bli_dreal(y), bli_dimag(y) )
#define bli_ddsubjs( a, y )  bli_dsubjris( bli_dreal(a), bli_dimag(a), bli_dreal(y), bli_dimag(y) )
#define bli_cdsubjs( a, y )  bli_dsubjris( bli_creal(a), bli_cimag(a), bli_dreal(y), bli_dimag(y) )
#define bli_zdsubjs( a, y )  bli_dsubjris( bli_zreal(a), bli_zimag(a), bli_dreal(y), bli_dimag(y) )

#ifndef BLIS_ENABLE_C99_COMPLEX

#define bli_scsubjs( a, y )  bli_csubjris( bli_sreal(a), bli_simag(a), bli_creal(y), bli_cimag(y) )
#define bli_dcsubjs( a, y )  bli_csubjris( bli_dreal(a), bli_dimag(a), bli_creal(y), bli_cimag(y) )
#define bli_ccsubjs( a, y )  bli_csubjris( bli_creal(a), bli_cimag(a), bli_creal(y), bli_cimag(y) )
#define bli_zcsubjs( a, y )  bli_csubjris( bli_zreal(a), bli_zimag(a), bli_creal(y), bli_cimag(y) )

#define bli_szsubjs( a, y )  bli_zsubjris( bli_sreal(a), bli_simag(a), bli_zreal(y), bli_zimag(y) )
#define bli_dzsubjs( a, y )  bli_zsubjris( bli_dreal(a), bli_dimag(a), bli_zreal(y), bli_zimag(y) )
#define bli_czsubjs( a, y )  bli_zsubjris( bli_creal(a), bli_cimag(a), bli_zreal(y), bli_zimag(y) )
#define bli_zzsubjs( a, y )  bli_zsubjris( bli_zreal(a), bli_zimag(a), bli_zreal(y), bli_zimag(y) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

#define bli_scsubjs( a, y )  { (y) -=      (a); }
#define bli_dcsubjs( a, y )  { (y) -=      (a); }
#define bli_ccsubjs( a, y )  { (y) -= conjf(a); }
#define bli_zcsubjs( a, y )  { (y) -= conj (a); }

#define bli_szsubjs( a, y )  { (y) -=      (a); }
#define bli_dzsubjs( a, y )  { (y) -=      (a); }
#define bli_czsubjs( a, y )  { (y) -= conjf(a); }
#define bli_zzsubjs( a, y )  { (y) -= conj (a); }

#endif // BLIS_ENABLE_C99_COMPLEX


#define bli_ssubjs( a, y )  bli_sssubjs( a, y )
#define bli_dsubjs( a, y )  bli_ddsubjs( a, y )
#define bli_csubjs( a, y )  bli_ccsubjs( a, y )
#define bli_zsubjs( a, y )  bli_zzsubjs( a, y )


#endif

// end bli_subjs.h
#line 189 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_swaps.h
#line 1 "./frame/include/level0//bli_swaps.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SWAPS_H
#define BLIS_SWAPS_H

// swaps

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of y.


#define bli_ssswaps( x, y ) \
{ \
	float    w; \
	bli_sscopys( (y), (w) ); \
	bli_sscopys( (x), (y) ); \
	bli_sscopys( (w), (x) ); \
}
#define bli_dsswaps( x, y ) \
{ \
	double   w; \
	bli_sdcopys( (y), (w) ); \
	bli_dscopys( (x), (y) ); \
	bli_ddcopys( (w), (x) ); \
}
#define bli_csswaps( x, y ) \
{ \
	scomplex w; \
	bli_sccopys( (y), (w) ); \
	bli_cscopys( (x), (y) ); \
	bli_cccopys( (w), (x) ); \
}
#define bli_zsswaps( x, y ) \
{ \
	dcomplex w; \
	bli_szcopys( (y), (w) ); \
	bli_zscopys( (x), (y) ); \
	bli_zzcopys( (w), (x) ); \
}


#define bli_sdswaps( x, y ) \
{ \
	float    w; \
	bli_dscopys( (y), (w) ); \
	bli_sdcopys( (x), (y) ); \
	bli_sscopys( (w), (x) ); \
}
#define bli_ddswaps( x, y ) \
{ \
	double   w; \
	bli_ddcopys( (y), (w) ); \
	bli_ddcopys( (x), (y) ); \
	bli_ddcopys( (w), (x) ); \
}
#define bli_cdswaps( x, y ) \
{ \
	scomplex w; \
	bli_dccopys( (y), (w) ); \
	bli_cdcopys( (x), (y) ); \
	bli_cccopys( (w), (x) ); \
}
#define bli_zdswaps( x, y ) \
{ \
	dcomplex w; \
	bli_dzcopys( (y), (w) ); \
	bli_zdcopys( (x), (y) ); \
	bli_zzcopys( (w), (x) ); \
}


#define bli_scswaps( x, y ) \
{ \
	float    w; \
	bli_cscopys( (y), (w) ); \
	bli_sccopys( (x), (y) ); \
	bli_sscopys( (w), (x) ); \
}
#define bli_dcswaps( x, y ) \
{ \
	double   w; \
	bli_cdcopys( (y), (w) ); \
	bli_dccopys( (x), (y) ); \
	bli_ddcopys( (w), (x) ); \
}
#define bli_ccswaps( x, y ) \
{ \
	scomplex w; \
	bli_cccopys( (y), (w) ); \
	bli_cccopys( (x), (y) ); \
	bli_cccopys( (w), (x) ); \
}
#define bli_zcswaps( x, y ) \
{ \
	dcomplex w; \
	bli_czcopys( (y), (w) ); \
	bli_zccopys( (x), (y) ); \
	bli_zzcopys( (w), (x) ); \
}


#define bli_szswaps( x, y ) \
{ \
	float    w; \
	bli_zscopys( (y), (w) ); \
	bli_szcopys( (x), (y) ); \
	bli_sscopys( (w), (x) ); \
}
#define bli_dzswaps( x, y ) \
{ \
	double   w; \
	bli_zdcopys( (y), (w) ); \
	bli_dzcopys( (x), (y) ); \
	bli_ddcopys( (w), (x) ); \
}
#define bli_czswaps( x, y ) \
{ \
	scomplex w; \
	bli_zccopys( (y), (w) ); \
	bli_czcopys( (x), (y) ); \
	bli_cccopys( (w), (x) ); \
}
#define bli_zzswaps( x, y ) \
{ \
	dcomplex w; \
	bli_zzcopys( (y), (w) ); \
	bli_zzcopys( (x), (y) ); \
	bli_zzcopys( (w), (x) ); \
}


#define bli_sswaps( x, y )  bli_ssswaps( x, y )
#define bli_dswaps( x, y )  bli_ddswaps( x, y )
#define bli_cswaps( x, y )  bli_ccswaps( x, y )
#define bli_zswaps( x, y )  bli_zzswaps( x, y )


#endif
// end bli_swaps.h
#line 191 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_xpbys.h
#line 1 "./frame/include/level0//bli_xpbys.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_XPBYS_H
#define BLIS_XPBYS_H

// xpbys

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of b.
// - The third char encodes the type of y.

// -- (xby) = (??s) ------------------------------------------------------------

#define bli_sssxpbys( x, b, y )  bli_rxxpbyris( bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_dssxpbys( x, b, y )  bli_rxxpbyris( bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_cssxpbys( x, b, y )  bli_rxxpbyris( bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_zssxpbys( x, b, y )  bli_rxxpbyris( bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )

#define bli_sdsxpbys( x, b, y )  bli_rxxpbyris( bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_ddsxpbys( x, b, y )  bli_rxxpbyris( bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_cdsxpbys( x, b, y )  bli_rxxpbyris( bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zdsxpbys( x, b, y )  bli_rxxpbyris( bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )

#define bli_scsxpbys( x, b, y )  bli_rxxpbyris( bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_dcsxpbys( x, b, y )  bli_rxxpbyris( bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_ccsxpbys( x, b, y )  bli_rxxpbyris( bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zcsxpbys( x, b, y )  bli_rxxpbyris( bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )

#define bli_szsxpbys( x, b, y )  bli_rxxpbyris( bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_dzsxpbys( x, b, y )  bli_rxxpbyris( bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_czsxpbys( x, b, y )  bli_rxxpbyris( bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zzsxpbys( x, b, y )  bli_rxxpbyris( bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )

// -- (xby) = (??d) ------------------------------------------------------------

#define bli_ssdxpbys( x, b, y )  bli_rxxpbyris( bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dsdxpbys( x, b, y )  bli_rxxpbyris( bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_csdxpbys( x, b, y )  bli_rxxpbyris( bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zsdxpbys( x, b, y )  bli_rxxpbyris( bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )

#define bli_sddxpbys( x, b, y )  bli_rxxpbyris( bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dddxpbys( x, b, y )  bli_rxxpbyris( bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_cddxpbys( x, b, y )  bli_rxxpbyris( bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zddxpbys( x, b, y )  bli_rxxpbyris( bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )

#define bli_scdxpbys( x, b, y )  bli_rxxpbyris( bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dcdxpbys( x, b, y )  bli_rxxpbyris( bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_ccdxpbys( x, b, y )  bli_rxxpbyris( bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zcdxpbys( x, b, y )  bli_rxxpbyris( bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )

#define bli_szdxpbys( x, b, y )  bli_rxxpbyris( bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dzdxpbys( x, b, y )  bli_rxxpbyris( bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_czdxpbys( x, b, y )  bli_rxxpbyris( bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zzdxpbys( x, b, y )  bli_rxxpbyris( bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )

#ifndef BLIS_ENABLE_C99_COMPLEX

// -- (xby) = (??c) ------------------------------------------------------------

#define bli_sscxpbys( x, b, y )  bli_rxxpbyris( bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_dscxpbys( x, b, y )  bli_rxxpbyris( bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_cscxpbys( x, b, y )  bli_crxpbyris( bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_zscxpbys( x, b, y )  bli_crxpbyris( bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )

#define bli_sdcxpbys( x, b, y )  bli_rxxpbyris( bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_ddcxpbys( x, b, y )  bli_rxxpbyris( bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_cdcxpbys( x, b, y )  bli_crxpbyris( bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zdcxpbys( x, b, y )  bli_crxpbyris( bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )

#define bli_sccxpbys( x, b, y )  bli_cxxpbyris( bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_dccxpbys( x, b, y )  bli_cxxpbyris( bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_cccxpbys( x, b, y )  bli_cxxpbyris( bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zccxpbys( x, b, y )  bli_cxxpbyris( bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )

#define bli_szcxpbys( x, b, y )  bli_cxxpbyris( bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_dzcxpbys( x, b, y )  bli_cxxpbyris( bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_czcxpbys( x, b, y )  bli_cxxpbyris( bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zzcxpbys( x, b, y )  bli_cxxpbyris( bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )

// -- (xby) = (??z) ------------------------------------------------------------

#define bli_sszxpbys( x, b, y )  bli_rxxpbyris( bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dszxpbys( x, b, y )  bli_rxxpbyris( bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_cszxpbys( x, b, y )  bli_crxpbyris( bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zszxpbys( x, b, y )  bli_crxpbyris( bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )

#define bli_sdzxpbys( x, b, y )  bli_rxxpbyris( bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_ddzxpbys( x, b, y )  bli_rxxpbyris( bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_cdzxpbys( x, b, y )  bli_crxpbyris( bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zdzxpbys( x, b, y )  bli_crxpbyris( bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )

#define bli_sczxpbys( x, b, y )  bli_cxxpbyris( bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dczxpbys( x, b, y )  bli_cxxpbyris( bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_cczxpbys( x, b, y )  bli_cxxpbyris( bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zczxpbys( x, b, y )  bli_cxxpbyris( bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )

#define bli_szzxpbys( x, b, y )  bli_cxxpbyris( bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dzzxpbys( x, b, y )  bli_cxxpbyris( bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_czzxpbys( x, b, y )  bli_cxxpbyris( bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zzzxpbys( x, b, y )  bli_cxxpbyris( bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

// -- (xby) = (??c) ------------------------------------------------------------

#define bli_sscxpbys( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_dscxpbys( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_cscxpbys( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_zscxpbys( x, b, y )  { (y) = (x) + (b) * (y); }

#define bli_sdcxpbys( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_ddcxpbys( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_cdcxpbys( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_zdcxpbys( x, b, y )  { (y) = (x) + (b) * (y); }

#define bli_sccxpbys( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_dccxpbys( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_cccxpbys( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_zccxpbys( x, b, y )  { (y) = (x) + (b) * (y); }

#define bli_szcxpbys( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_dzcxpbys( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_czcxpbys( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_zzcxpbys( x, b, y )  { (y) = (x) + (b) * (y); }

// -- (xby) = (??z) ------------------------------------------------------------

#define bli_sszxpbys( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_dszxpbys( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_cszxpbys( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_zszxpbys( x, b, y )  { (y) = (x) + (b) * (y); }

#define bli_sdzxpbys( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_ddzxpbys( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_cdzxpbys( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_zdzxpbys( x, b, y )  { (y) = (x) + (b) * (y); }

#define bli_sczxpbys( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_dczxpbys( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_cczxpbys( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_zczxpbys( x, b, y )  { (y) = (x) + (b) * (y); }

#define bli_szzxpbys( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_dzzxpbys( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_czzxpbys( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_zzzxpbys( x, b, y )  { (y) = (x) + (b) * (y); }

#endif // BLIS_ENABLE_C99_COMPLEX


#define bli_sxpbys( x, b, y )  bli_sssxpbys( x, b, y )
#define bli_dxpbys( x, b, y )  bli_dddxpbys( x, b, y )
#define bli_cxpbys( x, b, y )  bli_cccxpbys( x, b, y )
#define bli_zxpbys( x, b, y )  bli_zzzxpbys( x, b, y )


#endif

// end bli_xpbys.h
#line 193 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_xpbyjs.h
#line 1 "./frame/include/level0//bli_xpbyjs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_XPBYJS_H
#define BLIS_XPBYJS_H

// xpbyjs

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of b.
// - The third char encodes the type of y.

// -- (xby) = (??s) ------------------------------------------------------------

#define bli_sssxpbyjs( x, b, y )  bli_rxxpbyjris( bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_dssxpbyjs( x, b, y )  bli_rxxpbyjris( bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_cssxpbyjs( x, b, y )  bli_rxxpbyjris( bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )
#define bli_zssxpbyjs( x, b, y )  bli_rxxpbyjris( bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_sreal(y), bli_simag(y) )

#define bli_sdsxpbyjs( x, b, y )  bli_rxxpbyjris( bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_ddsxpbyjs( x, b, y )  bli_rxxpbyjris( bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_cdsxpbyjs( x, b, y )  bli_rxxpbyjris( bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zdsxpbyjs( x, b, y )  bli_rxxpbyjris( bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_sreal(y), bli_simag(y) )

#define bli_scsxpbyjs( x, b, y )  bli_rxxpbyjris( bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_dcsxpbyjs( x, b, y )  bli_rxxpbyjris( bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_ccsxpbyjs( x, b, y )  bli_rxxpbyjris( bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zcsxpbyjs( x, b, y )  bli_rxxpbyjris( bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_sreal(y), bli_simag(y) )

#define bli_szsxpbyjs( x, b, y )  bli_rxxpbyjris( bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_dzsxpbyjs( x, b, y )  bli_rxxpbyjris( bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_czsxpbyjs( x, b, y )  bli_rxxpbyjris( bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )
#define bli_zzsxpbyjs( x, b, y )  bli_rxxpbyjris( bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_sreal(y), bli_simag(y) )

// -- (xby) = (??d) ------------------------------------------------------------

#define bli_ssdxpbyjs( x, b, y )  bli_rxxpbyjris( bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dsdxpbyjs( x, b, y )  bli_rxxpbyjris( bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_csdxpbyjs( x, b, y )  bli_rxxpbyjris( bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zsdxpbyjs( x, b, y )  bli_rxxpbyjris( bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_dreal(y), bli_dimag(y) )

#define bli_sddxpbyjs( x, b, y )  bli_rxxpbyjris( bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dddxpbyjs( x, b, y )  bli_rxxpbyjris( bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_cddxpbyjs( x, b, y )  bli_rxxpbyjris( bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zddxpbyjs( x, b, y )  bli_rxxpbyjris( bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_dreal(y), bli_dimag(y) )

#define bli_scdxpbyjs( x, b, y )  bli_rxxpbyjris( bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dcdxpbyjs( x, b, y )  bli_rxxpbyjris( bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_ccdxpbyjs( x, b, y )  bli_rxxpbyjris( bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zcdxpbyjs( x, b, y )  bli_rxxpbyjris( bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_dreal(y), bli_dimag(y) )

#define bli_szdxpbyjs( x, b, y )  bli_rxxpbyjris( bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_dzdxpbyjs( x, b, y )  bli_rxxpbyjris( bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_czdxpbyjs( x, b, y )  bli_rxxpbyjris( bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )
#define bli_zzdxpbyjs( x, b, y )  bli_rxxpbyjris( bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_dreal(y), bli_dimag(y) )

#ifndef BLIS_ENABLE_C99_COMPLEX

// -- (xby) = (??c) ------------------------------------------------------------

#define bli_sscxpbyjs( x, b, y )  bli_rxxpbyjris( bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_dscxpbyjs( x, b, y )  bli_rxxpbyjris( bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_cscxpbyjs( x, b, y )  bli_crxpbyjris( bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )
#define bli_zscxpbyjs( x, b, y )  bli_crxpbyjris( bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_creal(y), bli_cimag(y) )

#define bli_sdcxpbyjs( x, b, y )  bli_rxxpbyjris( bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_ddcxpbyjs( x, b, y )  bli_rxxpbyjris( bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_cdcxpbyjs( x, b, y )  bli_crxpbyjris( bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zdcxpbyjs( x, b, y )  bli_crxpbyjris( bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_creal(y), bli_cimag(y) )

#define bli_sccxpbyjs( x, b, y )  bli_cxxpbyjris( bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_dccxpbyjs( x, b, y )  bli_cxxpbyjris( bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_cccxpbyjs( x, b, y )  bli_cxxpbyjris( bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zccxpbyjs( x, b, y )  bli_cxxpbyjris( bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_creal(y), bli_cimag(y) )

#define bli_szcxpbyjs( x, b, y )  bli_cxxpbyjris( bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_dzcxpbyjs( x, b, y )  bli_cxxpbyjris( bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_czcxpbyjs( x, b, y )  bli_cxxpbyjris( bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )
#define bli_zzcxpbyjs( x, b, y )  bli_cxxpbyjris( bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_creal(y), bli_cimag(y) )

// -- (xby) = (??z) ------------------------------------------------------------

#define bli_sszxpbyjs( x, b, y )  bli_rxxpbyjris( bli_sreal(x), bli_simag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dszxpbyjs( x, b, y )  bli_rxxpbyjris( bli_dreal(x), bli_dimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_cszxpbyjs( x, b, y )  bli_crxpbyjris( bli_creal(x), bli_cimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zszxpbyjs( x, b, y )  bli_crxpbyjris( bli_zreal(x), bli_zimag(x), bli_sreal(b), bli_simag(b), bli_zreal(y), bli_zimag(y) )

#define bli_sdzxpbyjs( x, b, y )  bli_rxxpbyjris( bli_sreal(x), bli_simag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_ddzxpbyjs( x, b, y )  bli_rxxpbyjris( bli_dreal(x), bli_dimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_cdzxpbyjs( x, b, y )  bli_crxpbyjris( bli_creal(x), bli_cimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zdzxpbyjs( x, b, y )  bli_crxpbyjris( bli_zreal(x), bli_zimag(x), bli_dreal(b), bli_dimag(b), bli_zreal(y), bli_zimag(y) )

#define bli_sczxpbyjs( x, b, y )  bli_cxxpbyjris( bli_sreal(x), bli_simag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dczxpbyjs( x, b, y )  bli_cxxpbyjris( bli_dreal(x), bli_dimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_cczxpbyjs( x, b, y )  bli_cxxpbyjris( bli_creal(x), bli_cimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zczxpbyjs( x, b, y )  bli_cxxpbyjris( bli_zreal(x), bli_zimag(x), bli_creal(b), bli_cimag(b), bli_zreal(y), bli_zimag(y) )

#define bli_szzxpbyjs( x, b, y )  bli_cxxpbyjris( bli_sreal(x), bli_simag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_dzzxpbyjs( x, b, y )  bli_cxxpbyjris( bli_dreal(x), bli_dimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_czzxpbyjs( x, b, y )  bli_cxxpbyjris( bli_creal(x), bli_cimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )
#define bli_zzzxpbyjs( x, b, y )  bli_cxxpbyjris( bli_zreal(x), bli_zimag(x), bli_zreal(b), bli_zimag(b), bli_zreal(y), bli_zimag(y) )

#else // ifdef BLIS_ENABLE_C99_COMPLEX

// -- (xby) = (??c) ------------------------------------------------------------

#define bli_sscxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_dscxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_cscxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_zscxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }

#define bli_sdcxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_ddcxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_cdcxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_zdcxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }

#define bli_sccxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_dccxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_cccxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_zccxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }

#define bli_szcxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_dzcxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_czcxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_zzcxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }

// -- (xby) = (??z) ------------------------------------------------------------

#define bli_sszxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_dszxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_cszxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_zszxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }

#define bli_sdzxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_ddzxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_cdzxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_zdzxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }

#define bli_sczxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_dczxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_cczxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_zczxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }

#define bli_szzxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_dzzxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_czzxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }
#define bli_zzzxpbyjs( x, b, y )  { (y) = (x) + (b) * (y); }

#endif // BLIS_ENABLE_C99_COMPLEX


#define bli_sxpbyjs( x, b, y )  bli_sssxpbyjs( x, b, y )
#define bli_dxpbyjs( x, b, y )  bli_dddxpbyjs( x, b, y )
#define bli_cxpbyjs( x, b, y )  bli_cccxpbyjs( x, b, y )
#define bli_zxpbyjs( x, b, y )  bli_zzzxpbyjs( x, b, y )


#endif

// end bli_xpbyjs.h
#line 194 "./frame/include//bli_scalar_macro_defs.h"


// Inlined scalar macros in loops
// begin bli_adds_mxn.h
#line 1 "./frame/include/level0//bli_adds_mxn.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_ADDS_MXN_H
#define BLIS_ADDS_MXN_H

// adds_mxn

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of y.


// xy = ?s

BLIS_INLINE void bli_ssadds_mxn( const dim_t m, const dim_t n, float*    restrict x, const inc_t rs_x, const inc_t cs_x,
                                                          float*    restrict y, const inc_t rs_y, const inc_t cs_y )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_ssadds( *(x + ii + jj*cs_x),
		            *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_ssadds( *(x + ii*rs_x + jj),
		            *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_ssadds( *(x + ii*rs_x + jj*cs_x),
		            *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_dsadds_mxn( const dim_t m, const dim_t n, double*   restrict x, const inc_t rs_x, const inc_t cs_x,
                                                          float*    restrict y, const inc_t rs_y, const inc_t cs_y )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_dsadds( *(x + ii + jj*cs_x),
		            *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_dsadds( *(x + ii*rs_x + jj),
		            *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_dsadds( *(x + ii*rs_x + jj*cs_x),
		            *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_csadds_mxn( const dim_t m, const dim_t n, scomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
                                                          float*    restrict y, const inc_t rs_y, const inc_t cs_y )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_csadds( *(x + ii + jj*cs_x),
		            *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_csadds( *(x + ii*rs_x + jj),
		            *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_csadds( *(x + ii*rs_x + jj*cs_x),
		            *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_zsadds_mxn( const dim_t m, const dim_t n, dcomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
                                                          float*    restrict y, const inc_t rs_y, const inc_t cs_y )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_zsadds( *(x + ii + jj*cs_x),
		            *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_zsadds( *(x + ii*rs_x + jj),
		            *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_zsadds( *(x + ii*rs_x + jj*cs_x),
		            *(y + ii*rs_y + jj*cs_y) );
	}
}

// xy = ?d

BLIS_INLINE void bli_sdadds_mxn( const dim_t m, const dim_t n, float*    restrict x, const inc_t rs_x, const inc_t cs_x,
                                                          double*   restrict y, const inc_t rs_y, const inc_t cs_y )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_sdadds( *(x + ii + jj*cs_x),
		            *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_sdadds( *(x + ii*rs_x + jj),
		            *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_sdadds( *(x + ii*rs_x + jj*cs_x),
		            *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_ddadds_mxn( const dim_t m, const dim_t n, double*   restrict x, const inc_t rs_x, const inc_t cs_x,
                                                          double*   restrict y, const inc_t rs_y, const inc_t cs_y )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_ddadds( *(x + ii + jj*cs_x),
		            *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_ddadds( *(x + ii*rs_x + jj),
		            *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_ddadds( *(x + ii*rs_x + jj*cs_x),
		            *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_cdadds_mxn( const dim_t m, const dim_t n, scomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
                                                          double*   restrict y, const inc_t rs_y, const inc_t cs_y )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_cdadds( *(x + ii + jj*cs_x),
		            *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_cdadds( *(x + ii*rs_x + jj),
		            *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_cdadds( *(x + ii*rs_x + jj*cs_x),
		            *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_zdadds_mxn( const dim_t m, const dim_t n, dcomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
                                                          double*   restrict y, const inc_t rs_y, const inc_t cs_y )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_zdadds( *(x + ii + jj*cs_x),
		            *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_zdadds( *(x + ii*rs_x + jj),
		            *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_zdadds( *(x + ii*rs_x + jj*cs_x),
		            *(y + ii*rs_y + jj*cs_y) );
	}
}

// xy = ?c

BLIS_INLINE void bli_scadds_mxn( const dim_t m, const dim_t n, float*    restrict x, const inc_t rs_x, const inc_t cs_x,
                                                          scomplex* restrict y, const inc_t rs_y, const inc_t cs_y )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_scadds( *(x + ii + jj*cs_x),
		            *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_scadds( *(x + ii*rs_x + jj),
		            *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_scadds( *(x + ii*rs_x + jj*cs_x),
		            *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_dcadds_mxn( const dim_t m, const dim_t n, double*   restrict x, const inc_t rs_x, const inc_t cs_x,
                                                          scomplex* restrict y, const inc_t rs_y, const inc_t cs_y )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_dcadds( *(x + ii + jj*cs_x),
		            *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_dcadds( *(x + ii*rs_x + jj),
		            *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_dcadds( *(x + ii*rs_x + jj*cs_x),
		            *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_ccadds_mxn( const dim_t m, const dim_t n, scomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
                                                          scomplex* restrict y, const inc_t rs_y, const inc_t cs_y )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_ccadds( *(x + ii + jj*cs_x),
		            *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_ccadds( *(x + ii*rs_x + jj),
		            *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_ccadds( *(x + ii*rs_x + jj*cs_x),
		            *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_zcadds_mxn( const dim_t m, const dim_t n, dcomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
                                                          scomplex* restrict y, const inc_t rs_y, const inc_t cs_y )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_zcadds( *(x + ii + jj*cs_x),
		            *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_zcadds( *(x + ii*rs_x + jj),
		            *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_zcadds( *(x + ii*rs_x + jj*cs_x),
		            *(y + ii*rs_y + jj*cs_y) );
	}
}

// xy = ?z

BLIS_INLINE void bli_szadds_mxn( const dim_t m, const dim_t n, float*    restrict x, const inc_t rs_x, const inc_t cs_x,
                                                          dcomplex* restrict y, const inc_t rs_y, const inc_t cs_y )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_szadds( *(x + ii + jj*cs_x),
		            *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_szadds( *(x + ii*rs_x + jj),
		            *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_szadds( *(x + ii*rs_x + jj*cs_x),
		            *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_dzadds_mxn( const dim_t m, const dim_t n, double*   restrict x, const inc_t rs_x, const inc_t cs_x,
                                                          dcomplex* restrict y, const inc_t rs_y, const inc_t cs_y )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_dzadds( *(x + ii + jj*cs_x),
		            *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_dzadds( *(x + ii*rs_x + jj),
		            *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_dzadds( *(x + ii*rs_x + jj*cs_x),
		            *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_czadds_mxn( const dim_t m, const dim_t n, scomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
                                                          dcomplex* restrict y, const inc_t rs_y, const inc_t cs_y )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_czadds( *(x + ii + jj*cs_x),
		            *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_czadds( *(x + ii*rs_x + jj),
		            *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_czadds( *(x + ii*rs_x + jj*cs_x),
		            *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_zzadds_mxn( const dim_t m, const dim_t n, dcomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
                                                          dcomplex* restrict y, const inc_t rs_y, const inc_t cs_y )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_zzadds( *(x + ii + jj*cs_x),
		            *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_zzadds( *(x + ii*rs_x + jj),
		            *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_zzadds( *(x + ii*rs_x + jj*cs_x),
		            *(y + ii*rs_y + jj*cs_y) );
	}
}



BLIS_INLINE void bli_sadds_mxn( const dim_t m, const dim_t n, float*    restrict x, const inc_t rs_x, const inc_t cs_x,
                                                         float*    restrict y, const inc_t rs_y, const inc_t cs_y )
{
	bli_ssadds_mxn( m, n, x, rs_x, cs_x, y, rs_y, cs_y );
}
BLIS_INLINE void bli_dadds_mxn( const dim_t m, const dim_t n, double*   restrict x, const inc_t rs_x, const inc_t cs_x,
                                                         double*   restrict y, const inc_t rs_y, const inc_t cs_y )
{
	bli_ddadds_mxn( m, n, x, rs_x, cs_x, y, rs_y, cs_y );
}
BLIS_INLINE void bli_cadds_mxn( const dim_t m, const dim_t n, scomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
                                                         scomplex* restrict y, const inc_t rs_y, const inc_t cs_y )
{
	bli_ccadds_mxn( m, n, x, rs_x, cs_x, y, rs_y, cs_y );
}
BLIS_INLINE void bli_zadds_mxn( const dim_t m, const dim_t n, dcomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
                                                         dcomplex* restrict y, const inc_t rs_y, const inc_t cs_y )
{
	bli_zzadds_mxn( m, n, x, rs_x, cs_x, y, rs_y, cs_y );
}


#endif
// end bli_adds_mxn.h
#line 197 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_adds_mxn_uplo.h
#line 1 "./frame/include/level0//bli_adds_mxn_uplo.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_ADDS_MXN_UPLO_H
#define BLIS_ADDS_MXN_UPLO_H

// adds_mxn_u

#define bli_ssadds_mxn_u( diagoff, m, n, x, rs_x, cs_x, y, rs_y, cs_y ) \
{ \
	dim_t _i, _j; \
\
	for ( _j = 0; _j < n; ++_j ) \
	{ \
		for ( _i = 0; _i < m; ++_i ) \
		{ \
			if ( (doff_t)_j - (doff_t)_i >= diagoff ) \
			{ \
				bli_ssadds( *(x + _i*rs_x + _j*cs_x), \
				            *(y + _i*rs_y + _j*cs_y) ); \
			} \
		} \
	} \
}

#define bli_ddadds_mxn_u( diagoff, m, n, x, rs_x, cs_x, y, rs_y, cs_y ) \
{ \
	dim_t _i, _j; \
\
	for ( _j = 0; _j < n; ++_j ) \
	{ \
		for ( _i = 0; _i < m; ++_i ) \
		{ \
			if ( (doff_t)_j - (doff_t)_i >= diagoff ) \
			{ \
				bli_ddadds( *(x + _i*rs_x + _j*cs_x), \
				            *(y + _i*rs_y + _j*cs_y) ); \
			} \
		} \
	} \
}

#define bli_ccadds_mxn_u( diagoff, m, n, x, rs_x, cs_x, y, rs_y, cs_y ) \
{ \
	dim_t _i, _j; \
\
	for ( _j = 0; _j < n; ++_j ) \
	{ \
		for ( _i = 0; _i < m; ++_i ) \
		{ \
			if ( (doff_t)_j - (doff_t)_i >= diagoff ) \
			{ \
				bli_ccadds( *(x + _i*rs_x + _j*cs_x), \
				            *(y + _i*rs_y + _j*cs_y) ); \
			} \
		} \
	} \
}

#define bli_zzadds_mxn_u( diagoff, m, n, x, rs_x, cs_x, y, rs_y, cs_y ) \
{ \
	dim_t _i, _j; \
\
	for ( _j = 0; _j < n; ++_j ) \
	{ \
		for ( _i = 0; _i < m; ++_i ) \
		{ \
			if ( (doff_t)_j - (doff_t)_i >= diagoff ) \
			{ \
				bli_zzadds( *(x + _i*rs_x + _j*cs_x), \
				            *(y + _i*rs_y + _j*cs_y) ); \
			} \
		} \
	} \
}

// adds_mxn_l

#define bli_ssadds_mxn_l( diagoff, m, n, x, rs_x, cs_x, y, rs_y, cs_y ) \
{ \
	dim_t _i, _j; \
\
	for ( _j = 0; _j < n; ++_j ) \
	{ \
		for ( _i = 0; _i < m; ++_i ) \
		{ \
			if ( (doff_t)_j - (doff_t)_i <= diagoff ) \
			{ \
				bli_ssadds( *(x + _i*rs_x + _j*cs_x), \
				            *(y + _i*rs_y + _j*cs_y) ); \
			} \
		} \
	} \
}

#define bli_ddadds_mxn_l( diagoff, m, n, x, rs_x, cs_x, y, rs_y, cs_y ) \
{ \
	dim_t _i, _j; \
\
	for ( _j = 0; _j < n; ++_j ) \
	{ \
		for ( _i = 0; _i < m; ++_i ) \
		{ \
			if ( (doff_t)_j - (doff_t)_i <= diagoff ) \
			{ \
				bli_ddadds( *(x + _i*rs_x + _j*cs_x), \
				            *(y + _i*rs_y + _j*cs_y) ); \
			} \
		} \
	} \
}

#define bli_ccadds_mxn_l( diagoff, m, n, x, rs_x, cs_x, y, rs_y, cs_y ) \
{ \
	dim_t _i, _j; \
\
	for ( _j = 0; _j < n; ++_j ) \
	{ \
		for ( _i = 0; _i < m; ++_i ) \
		{ \
			if ( (doff_t)_j - (doff_t)_i <= diagoff ) \
			{ \
				bli_ccadds( *(x + _i*rs_x + _j*cs_x), \
				            *(y + _i*rs_y + _j*cs_y) ); \
			} \
		} \
	} \
}

#define bli_zzadds_mxn_l( diagoff, m, n, x, rs_x, cs_x, y, rs_y, cs_y ) \
{ \
	dim_t _i, _j; \
\
	for ( _j = 0; _j < n; ++_j ) \
	{ \
		for ( _i = 0; _i < m; ++_i ) \
		{ \
			if ( (doff_t)_j - (doff_t)_i <= diagoff ) \
			{ \
				bli_zzadds( *(x + _i*rs_x + _j*cs_x), \
				            *(y + _i*rs_y + _j*cs_y) ); \
			} \
		} \
	} \
}


#define bli_sadds_mxn_u( diagoff, m, n, x, rs_x, cs_x, y, rs_y, cs_y ) \
{ \
	bli_ssadds_mxn_u( diagoff, m, n, x, rs_x, cs_x, y, rs_y, cs_y ); \
}
#define bli_dadds_mxn_u( diagoff, m, n, x, rs_x, cs_x, y, rs_y, cs_y ) \
{ \
	bli_ddadds_mxn_u( diagoff, m, n, x, rs_x, cs_x, y, rs_y, cs_y ); \
}
#define bli_cadds_mxn_u( diagoff, m, n, x, rs_x, cs_x, y, rs_y, cs_y ) \
{ \
	bli_ccadds_mxn_u( diagoff, m, n, x, rs_x, cs_x, y, rs_y, cs_y ); \
}
#define bli_zadds_mxn_u( diagoff, m, n, x, rs_x, cs_x, y, rs_y, cs_y ) \
{ \
	bli_zzadds_mxn_u( diagoff, m, n, x, rs_x, cs_x, y, rs_y, cs_y ); \
}
#define bli_sadds_mxn_l( diagoff, m, n, x, rs_x, cs_x, y, rs_y, cs_y ) \
{ \
	bli_ssadds_mxn_l( diagoff, m, n, x, rs_x, cs_x, y, rs_y, cs_y ); \
}
#define bli_dadds_mxn_l( diagoff, m, n, x, rs_x, cs_x, y, rs_y, cs_y ) \
{ \
	bli_ddadds_mxn_l( diagoff, m, n, x, rs_x, cs_x, y, rs_y, cs_y ); \
}
#define bli_cadds_mxn_l( diagoff, m, n, x, rs_x, cs_x, y, rs_y, cs_y ) \
{ \
	bli_ccadds_mxn_l( diagoff, m, n, x, rs_x, cs_x, y, rs_y, cs_y ); \
}
#define bli_zadds_mxn_l( diagoff, m, n, x, rs_x, cs_x, y, rs_y, cs_y ) \
{ \
	bli_zzadds_mxn_l( diagoff, m, n, x, rs_x, cs_x, y, rs_y, cs_y ); \
}

#endif
// end bli_adds_mxn_uplo.h
#line 198 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_set0s_mxn.h
#line 1 "./frame/include/level0//bli_set0s_mxn.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SET0S_MXN_H
#define BLIS_SET0S_MXN_H

// set0s_mxn

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of y.

BLIS_INLINE void bli_sset0s_mxn( const dim_t m, const dim_t n,
                            float*    restrict y, const inc_t rs_y, const inc_t cs_y )
{
	for ( dim_t j = 0; j < n; ++j )
	for ( dim_t i = 0; i < m; ++i )
	bli_sset0s( *(y + i*rs_y + j*cs_y) );
}

BLIS_INLINE void bli_dset0s_mxn( const dim_t m, const dim_t n,
                            double*   restrict y, const inc_t rs_y, const inc_t cs_y )
{
	for ( dim_t j = 0; j < n; ++j )
	for ( dim_t i = 0; i < m; ++i )
	bli_dset0s( *(y + i*rs_y + j*cs_y) );
}

BLIS_INLINE void bli_cset0s_mxn( const dim_t m, const dim_t n,
                            scomplex* restrict y, const inc_t rs_y, const inc_t cs_y )
{
	for ( dim_t j = 0; j < n; ++j )
	for ( dim_t i = 0; i < m; ++i )
	bli_cset0s( *(y + i*rs_y + j*cs_y) );
}

BLIS_INLINE void bli_zset0s_mxn( const dim_t m, const dim_t n,
                            dcomplex* restrict y, const inc_t rs_y, const inc_t cs_y )
{
	for ( dim_t j = 0; j < n; ++j )
	for ( dim_t i = 0; i < m; ++i )
	bli_zset0s( *(y + i*rs_y + j*cs_y) );
}

#endif
// end bli_set0s_mxn.h
#line 199 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_set0s_edge.h
#line 1 "./frame/include/level0//bli_set0s_edge.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SET0S_EDGE_H
#define BLIS_SET0S_EDGE_H

// set0s_mxn

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of y.

#define GENTFUNC(ctype,ch,op) \
\
BLIS_INLINE void PASTEMAC(ch,op) \
     ( \
       const dim_t     i, \
       const dim_t     m, \
       const dim_t     j, \
       const dim_t     n, \
       ctype* restrict p, \
       const inc_t     ldp \
     ) \
{ \
	if ( i < m ) \
	{ \
		PASTEMAC(ch,set0s_mxn) \
		( \
		  m - i, \
		  j, \
		  p + i*1, 1, ldp \
		); \
	} \
\
	if ( j < n ) \
	{ \
		PASTEMAC(ch,set0s_mxn) \
		( \
		  m, \
		  n - j, \
		  p + j*ldp, 1, ldp \
		); \
	} \
}

INSERT_GENTFUNC_BASIC(set0s_edge)

#endif
// end bli_set0s_edge.h
#line 200 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_copys_mxn.h
#line 1 "./frame/include/level0//bli_copys_mxn.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_COPYS_MXN_H
#define BLIS_COPYS_MXN_H

// copys_mxn

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of y.

#undef  BLIS_ENABLE_CR_CASES
#define BLIS_ENABLE_CR_CASES 0

// -- bli_??copys_mxn --

#undef  GENTFUNC2
#define GENTFUNC2( ctypex, ctypey, chx, chy, opname, kername ) \
\
BLIS_INLINE void PASTEMAC2(chx,chy,opname) \
     ( \
       const dim_t   m, \
       const dim_t   n, \
       const ctypex* x, inc_t rs_x, inc_t cs_x, \
             ctypey* y, inc_t rs_y, inc_t cs_y  \
     ) \
{ \
	if      ( BLIS_ENABLE_CR_CASES && rs_x == 1 && rs_y == 1 ) \
	{ \
		for ( dim_t jj = 0; jj < n; ++jj ) \
		for ( dim_t ii = 0; ii < m; ++ii ) \
		PASTEMAC2(chx,chy,kername)( *(x + ii + jj*cs_x), \
		                            *(y + ii + jj*cs_y) ); \
	} \
	else if ( BLIS_ENABLE_CR_CASES && cs_x == 1 && cs_y == 1 ) \
	{ \
		for ( dim_t ii = 0; ii < m; ++ii ) \
		for ( dim_t jj = 0; jj < n; ++jj ) \
		PASTEMAC2(chx,chy,kername)( *(x + ii*rs_x + jj), \
		                            *(y + ii*rs_y + jj) ); \
	} \
	else \
	{ \
		for ( dim_t jj = 0; jj < n; ++jj ) \
		for ( dim_t ii = 0; ii < m; ++ii ) \
		PASTEMAC2(chx,chy,kername)( *(x + ii*rs_x + jj*cs_x), \
		                            *(y + ii*rs_y + jj*cs_y) ); \
	} \
}

INSERT_GENTFUNC2_BASIC ( copys_mxn, copys )
INSERT_GENTFUNC2_MIX_DP( copys_mxn, copys )


// -- bli_?copys_mxn --

#undef  GENTFUNC
#define GENTFUNC( ctype, ch, opname ) \
\
BLIS_INLINE void PASTEMAC(ch,opname) \
     ( \
       const dim_t  m, \
       const dim_t  n, \
       const ctype* x, inc_t rs_x, inc_t cs_x, \
             ctype* y, inc_t rs_y, inc_t cs_y  \
     ) \
{ \
	PASTEMAC2(ch,ch,opname)( m, n, x, rs_x, cs_x, y, rs_y, cs_y ); \
}

INSERT_GENTFUNC_BASIC( copys_mxn )




#if 0

// xy = ?s

BLIS_INLINE void bli_sscopys_mxn
     (
       const dim_t m,
       const dim_t n,
       const float*    restrict x, const inc_t rs_x, const inc_t cs_x,
             float*    restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_sscopys( *(x + ii + jj*cs_x),
		             *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_sscopys( *(x + ii*rs_x + jj),
		             *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_sscopys( *(x + ii*rs_x + jj*cs_x),
		             *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_dscopys_mxn
     (
       const dim_t m,
       const dim_t n,
       const double*   restrict x, const inc_t rs_x, const inc_t cs_x,
             float*    restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_dscopys( *(x + ii + jj*cs_x),
		             *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_dscopys( *(x + ii*rs_x + jj),
		             *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_dscopys( *(x + ii*rs_x + jj*cs_x),
		             *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_cscopys_mxn
     (
       const dim_t m,
       const dim_t n,
       const scomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
             float*    restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_cscopys( *(x + ii + jj*cs_x),
		             *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_cscopys( *(x + ii*rs_x + jj),
		             *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_cscopys( *(x + ii*rs_x + jj*cs_x),
		             *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_zscopys_mxn
     (
       const dim_t m,
       const dim_t n,
       const dcomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
             float*    restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_zscopys( *(x + ii + jj*cs_x),
		             *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_zscopys( *(x + ii*rs_x + jj),
		             *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_zscopys( *(x + ii*rs_x + jj*cs_x),
		             *(y + ii*rs_y + jj*cs_y) );
	}
}

// xy = ?d

BLIS_INLINE void bli_sdcopys_mxn
     (
       const dim_t m,
       const dim_t n,
       const float*    restrict x, const inc_t rs_x, const inc_t cs_x,
             double*   restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_sdcopys( *(x + ii + jj*cs_x),
		             *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_sdcopys( *(x + ii*rs_x + jj),
		             *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_sdcopys( *(x + ii*rs_x + jj*cs_x),
		             *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_ddcopys_mxn
     (
       const dim_t m,
       const dim_t n,
       const double*   restrict x, const inc_t rs_x, const inc_t cs_x,
             double*   restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_ddcopys( *(x + ii + jj*cs_x),
		             *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_ddcopys( *(x + ii*rs_x + jj),
		             *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_ddcopys( *(x + ii*rs_x + jj*cs_x),
		             *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_cdcopys_mxn
     (
       const dim_t m,
       const dim_t n,
       const scomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
             double*   restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_cdcopys( *(x + ii + jj*cs_x),
		             *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_cdcopys( *(x + ii*rs_x + jj),
		             *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_cdcopys( *(x + ii*rs_x + jj*cs_x),
		             *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_zdcopys_mxn
     (
       const dim_t m,
       const dim_t n,
       const dcomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
             double*   restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_zdcopys( *(x + ii + jj*cs_x),
		             *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_zdcopys( *(x + ii*rs_x + jj),
		             *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_zdcopys( *(x + ii*rs_x + jj*cs_x),
		             *(y + ii*rs_y + jj*cs_y) );
	}
}

// xy = ?c

BLIS_INLINE void bli_sccopys_mxn
     (
       const dim_t m,
       const dim_t n,
       const float*    restrict x, const inc_t rs_x, const inc_t cs_x,
             scomplex* restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_sccopys( *(x + ii + jj*cs_x),
		             *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_sccopys( *(x + ii*rs_x + jj),
		             *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_sccopys( *(x + ii*rs_x + jj*cs_x),
		             *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_dccopys_mxn
     (
       const dim_t m,
       const dim_t n,
       const double*   restrict x, const inc_t rs_x, const inc_t cs_x,
             scomplex* restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_dccopys( *(x + ii + jj*cs_x),
		             *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_dccopys( *(x + ii*rs_x + jj),
		             *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_dccopys( *(x + ii*rs_x + jj*cs_x),
		             *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_cccopys_mxn
     (
       const dim_t m,
       const dim_t n,
       const scomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
             scomplex* restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_cccopys( *(x + ii + jj*cs_x),
		             *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_cccopys( *(x + ii*rs_x + jj),
		             *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_cccopys( *(x + ii*rs_x + jj*cs_x),
		             *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_zccopys_mxn
     (
       const dim_t m,
       const dim_t n,
       const dcomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
             scomplex* restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_zccopys( *(x + ii + jj*cs_x),
		             *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_zccopys( *(x + ii*rs_x + jj),
		             *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_zccopys( *(x + ii*rs_x + jj*cs_x),
		             *(y + ii*rs_y + jj*cs_y) );
	}
}

// xy = ?c

BLIS_INLINE void bli_szcopys_mxn
     (
       const dim_t m,
       const dim_t n,
       const float*    restrict x, const inc_t rs_x, const inc_t cs_x,
             dcomplex* restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_szcopys( *(x + ii + jj*cs_x),
		             *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_szcopys( *(x + ii*rs_x + jj),
		             *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_szcopys( *(x + ii*rs_x + jj*cs_x),
		             *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_dzcopys_mxn
     (
       const dim_t m,
       const dim_t n,
       const double*   restrict x, const inc_t rs_x, const inc_t cs_x,
             dcomplex* restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_dzcopys( *(x + ii + jj*cs_x),
		             *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_dzcopys( *(x + ii*rs_x + jj),
		             *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_dzcopys( *(x + ii*rs_x + jj*cs_x),
		             *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_czcopys_mxn
     (
       const dim_t m,
       const dim_t n,
       const scomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
             dcomplex* restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_czcopys( *(x + ii + jj*cs_x),
		             *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_czcopys( *(x + ii*rs_x + jj),
		             *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_czcopys( *(x + ii*rs_x + jj*cs_x),
		             *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_zzcopys_mxn
     (
       const dim_t m,
       const dim_t n,
       const dcomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
             dcomplex* restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_zzcopys( *(x + ii + jj*cs_x),
		             *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_zzcopys( *(x + ii*rs_x + jj),
		             *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_zzcopys( *(x + ii*rs_x + jj*cs_x),
		             *(y + ii*rs_y + jj*cs_y) );
	}
}

BLIS_INLINE void bli_scopys_mxn
     (
       const dim_t m,
       const dim_t n,
       const float*    restrict x, const inc_t rs_x, const inc_t cs_x,
             float*    restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
	bli_sscopys_mxn( m, n, x, rs_x, cs_x, y, rs_y, cs_y );
}
BLIS_INLINE void bli_dcopys_mxn
     (
       const dim_t m,
       const dim_t n,
       const double*   restrict x, const inc_t rs_x, const inc_t cs_x,
             double*   restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
	bli_ddcopys_mxn( m, n, x, rs_x, cs_x, y, rs_y, cs_y );
}
BLIS_INLINE void bli_ccopys_mxn
     (
       const dim_t m,
       const dim_t n,
       const scomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
             scomplex* restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
	bli_cccopys_mxn( m, n, x, rs_x, cs_x, y, rs_y, cs_y );
}
BLIS_INLINE void bli_zcopys_mxn
     (
       const dim_t m,
       const dim_t n,
       const dcomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
             dcomplex* restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
	bli_zzcopys_mxn( m, n, x, rs_x, cs_x, y, rs_y, cs_y );
}
#endif

#endif
// end bli_copys_mxn.h
#line 201 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_scal2s_mxn.h
#line 1 "./frame/include/level0//bli_scal2s_mxn.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SCAL2S_MXN_H
#define BLIS_SCAL2S_MXN_H

// scal2s_mxn

#undef  GENTFUNC
#define GENTFUNC( ctype, ch, opname ) \
\
BLIS_INLINE void PASTEMAC(ch,opname) \
     ( \
       const conj_t       conjx, \
       const dim_t        m, \
       const dim_t        n, \
       ctype*    restrict alpha, \
       ctype*    restrict x, const inc_t rs_x, const inc_t cs_x, \
       ctype*    restrict y, const inc_t rs_y, const inc_t cs_y  \
     ) \
{ \
	if ( bli_is_conj( conjx ) ) \
	{ \
		for ( dim_t j = 0; j < n; ++j ) \
		{ \
			ctype* restrict xj = x + j*cs_x; \
			ctype* restrict yj = y + j*cs_y; \
\
			for ( dim_t i = 0; i < m; ++i ) \
			{ \
				ctype* restrict xij = xj + i*rs_x; \
				ctype* restrict yij = yj + i*rs_y; \
\
				PASTEMAC(ch,scal2js)( *alpha, *xij, *yij ); \
			} \
		} \
	} \
	else /* if ( bli_is_noconj( conjx ) ) */ \
	{ \
		for ( dim_t j = 0; j < n; ++j ) \
		{ \
			ctype* restrict xj = x + j*cs_x; \
			ctype* restrict yj = y + j*cs_y; \
\
			for ( dim_t i = 0; i < m; ++i ) \
			{ \
				ctype* restrict xij = xj + i*rs_x; \
				ctype* restrict yij = yj + i*rs_y; \
\
				PASTEMAC(ch,scal2s)( *alpha, *xij, *yij ); \
			} \
		} \
	} \
}

INSERT_GENTFUNC_BASIC( scal2s_mxn )

#endif
// end bli_scal2s_mxn.h
#line 202 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_xpbys_mxn.h
#line 1 "./frame/include/level0//bli_xpbys_mxn.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_XPBYS_MXN_H
#define BLIS_XPBYS_MXN_H

// xpbys_mxn

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of b.
// - The third char encodes the type of y.
// - We only implement cases where typeof(b) == typeof(y).

#undef  BLIS_ENABLE_CR_CASES
#define BLIS_ENABLE_CR_CASES 0

// -- bli_???xpbys_mxn --

#undef  GENTFUNC2
#define GENTFUNC2( ctypex, ctypey, chx, chy, opname, kername ) \
\
BLIS_INLINE void PASTEMAC3(chx,chy,chy,opname) \
     ( \
       const dim_t   m, \
       const dim_t   n, \
       const ctypex* x, inc_t rs_x, inc_t cs_x, \
       const ctypey* beta, \
             ctypey* y, inc_t rs_y, inc_t cs_y  \
     ) \
{ \
	/* If beta is zero, overwrite y with x (in case y has infs or NaNs). */ \
	if ( PASTEMAC(chy,eq0)( *beta ) ) \
	{ \
		PASTEMAC2(chx,chy,copys_mxn)( m, n, x, rs_x, cs_x, y, rs_y, cs_y ); \
		return; \
	} \
\
	if      ( BLIS_ENABLE_CR_CASES && rs_x == 1 && rs_y == 1 ) \
	{ \
		for ( dim_t jj = 0; jj < n; ++jj ) \
		for ( dim_t ii = 0; ii < m; ++ii ) \
		PASTEMAC3(chx,chy,chy,kername) \
		( \
		  *(x + ii + jj*cs_x), *beta, \
		  *(y + ii + jj*cs_y) \
		); \
	} \
	else if ( BLIS_ENABLE_CR_CASES && cs_x == 1 && cs_y == 1 ) \
	{ \
		for ( dim_t ii = 0; ii < m; ++ii ) \
		for ( dim_t jj = 0; jj < n; ++jj ) \
		PASTEMAC3(chx,chy,chy,kername) \
		( \
		  *(x + ii*rs_x + jj), *beta, \
		  *(y + ii*rs_y + jj) \
		); \
	} \
	else \
	{ \
		for ( dim_t jj = 0; jj < n; ++jj ) \
		for ( dim_t ii = 0; ii < m; ++ii ) \
		PASTEMAC3(chx,chy,chy,kername) \
		( \
		  *(x + ii*rs_x + jj*cs_x), *beta, \
		  *(y + ii*rs_y + jj*cs_y) \
		); \
	} \
}

INSERT_GENTFUNC2_BASIC ( xpbys_mxn, xpbys )
INSERT_GENTFUNC2_MIX_DP( xpbys_mxn, xpbys )


// -- bli_?xpbys_mxn --

#undef  GENTFUNC
#define GENTFUNC( ctype, ch, opname ) \
\
BLIS_INLINE void PASTEMAC(ch,opname) \
     ( \
       const dim_t  m, \
       const dim_t  n, \
       const ctype* x, inc_t rs_x, inc_t cs_x, \
       const ctype* beta, \
             ctype* y, inc_t rs_y, inc_t cs_y  \
     ) \
{ \
    PASTEMAC3(ch,ch,ch,opname)( m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y ); \
}

INSERT_GENTFUNC_BASIC( xpbys_mxn )



#if 0
// -- (xby) = (?ss) ------------------------------------------------------------

BLIS_INLINE void bli_sssxpbys_mxn
     (
       const dim_t m,
       const dim_t n,
       const float*    restrict x, const inc_t rs_x, const inc_t cs_x,
       const float*    restrict beta,
             float*    restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
	// If beta is zero, overwrite y with x (in case y has infs or NaNs).
	if ( bli_seq0( *beta ) )
	{
		bli_sscopys_mxn( m, n, x, rs_x, cs_x, y, rs_y, cs_y );
		return;
	}

#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_sssxpbys( *(x + ii + jj*cs_x), *beta,
		              *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_sssxpbys( *(x + ii*rs_x + jj), *beta,
		              *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_sssxpbys( *(x + ii*rs_x + jj*cs_x), *beta,
		              *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_dssxpbys_mxn
     (
       const dim_t m,
       const dim_t n,
       const double*   restrict x, const inc_t rs_x, const inc_t cs_x,
       const float*    restrict beta,
             float*    restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
	// If beta is zero, overwrite y with x (in case y has infs or NaNs).
	if ( bli_seq0( *beta ) )
	{
		bli_dscopys_mxn( m, n, x, rs_x, cs_x, y, rs_y, cs_y );
		return;
	}

#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_dssxpbys( *(x + ii + jj*cs_x), *beta,
		              *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_dssxpbys( *(x + ii*rs_x + jj), *beta,
		              *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_dssxpbys( *(x + ii*rs_x + jj*cs_x), *beta,
		              *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_cssxpbys_mxn
     (
       const dim_t m,
       const dim_t n,
       const scomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
       const float*    restrict beta,
             float*    restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
	// If beta is zero, overwrite y with x (in case y has infs or NaNs).
	if ( bli_seq0( *beta ) )
	{
		bli_cscopys_mxn( m, n, x, rs_x, cs_x, y, rs_y, cs_y );
		return;
	}

#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_cssxpbys( *(x + ii + jj*cs_x), *beta,
		              *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_cssxpbys( *(x + ii*rs_x + jj), *beta,
		              *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_cssxpbys( *(x + ii*rs_x + jj*cs_x), *beta,
		              *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_zssxpbys_mxn
     (
       const dim_t m,
       const dim_t n,
       const dcomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
       const float*    restrict beta,
             float*    restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
	// If beta is zero, overwrite y with x (in case y has infs or NaNs).
	if ( bli_seq0( *beta ) )
	{
		bli_zscopys_mxn( m, n, x, rs_x, cs_x, y, rs_y, cs_y );
		return;
	}

#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_zssxpbys( *(x + ii + jj*cs_x), *beta,
		              *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_zssxpbys( *(x + ii*rs_x + jj), *beta,
		              *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_zssxpbys( *(x + ii*rs_x + jj*cs_x), *beta,
		              *(y + ii*rs_y + jj*cs_y) );
	}
}

// -- (xby) = (?dd) ------------------------------------------------------------

BLIS_INLINE void bli_sddxpbys_mxn
     (
       const dim_t m,
       const dim_t n,
       const float*    restrict x, const inc_t rs_x, const inc_t cs_x,
       const double*   restrict beta,
             double*   restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
	// If beta is zero, overwrite y with x (in case y has infs or NaNs).
	if ( bli_deq0( *beta ) )
	{
		bli_sdcopys_mxn( m, n, x, rs_x, cs_x, y, rs_y, cs_y );
		return;
	}

#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_sddxpbys( *(x + ii + jj*cs_x), *beta,
		              *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_sddxpbys( *(x + ii*rs_x + jj), *beta,
		              *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_sddxpbys( *(x + ii*rs_x + jj*cs_x), *beta,
		              *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_dddxpbys_mxn
     (
       const dim_t m,
       const dim_t n,
       const double*   restrict x, const inc_t rs_x, const inc_t cs_x,
       const double*   restrict beta,
             double*   restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
	// If beta is zero, overwrite y with x (in case y has infs or NaNs).
	if ( bli_deq0( *beta ) )
	{
		bli_ddcopys_mxn( m, n, x, rs_x, cs_x, y, rs_y, cs_y );
		return;
	}

#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_dddxpbys( *(x + ii + jj*cs_x), *beta,
		              *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_dddxpbys( *(x + ii*rs_x + jj), *beta,
		              *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_dddxpbys( *(x + ii*rs_x + jj*cs_x), *beta,
		              *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_cddxpbys_mxn
     (
       const dim_t m,
       const dim_t n,
       const scomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
       const double*   restrict beta,
             double*   restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
	// If beta is zero, overwrite y with x (in case y has infs or NaNs).
	if ( bli_deq0( *beta ) )
	{
		bli_cdcopys_mxn( m, n, x, rs_x, cs_x, y, rs_y, cs_y );
		return;
	}

#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_cddxpbys( *(x + ii + jj*cs_x), *beta,
		              *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_cddxpbys( *(x + ii*rs_x + jj), *beta,
		              *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_cddxpbys( *(x + ii*rs_x + jj*cs_x), *beta,
		              *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_zddxpbys_mxn
     (
       const dim_t m,
       const dim_t n,
       const dcomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
       const double*   restrict beta,
             double*   restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
	// If beta is zero, overwrite y with x (in case y has infs or NaNs).
	if ( bli_deq0( *beta ) )
	{
		bli_zdcopys_mxn( m, n, x, rs_x, cs_x, y, rs_y, cs_y );
		return;
	}

#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_zddxpbys( *(x + ii + jj*cs_x), *beta,
		              *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_zddxpbys( *(x + ii*rs_x + jj), *beta,
		              *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_zddxpbys( *(x + ii*rs_x + jj*cs_x), *beta,
		              *(y + ii*rs_y + jj*cs_y) );
	}
}

// -- (xby) = (?cc) ------------------------------------------------------------

BLIS_INLINE void bli_sccxpbys_mxn
     (
       const dim_t m,
       const dim_t n,
       const float*    restrict x, const inc_t rs_x, const inc_t cs_x,
       const scomplex* restrict beta,
             scomplex* restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
	// If beta is zero, overwrite y with x (in case y has infs or NaNs).
	if ( bli_ceq0( *beta ) )
	{
		bli_sccopys_mxn( m, n, x, rs_x, cs_x, y, rs_y, cs_y );
		return;
	}

#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_sccxpbys( *(x + ii + jj*cs_x), *beta,
		              *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_sccxpbys( *(x + ii*rs_x + jj), *beta,
		              *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_sccxpbys( *(x + ii*rs_x + jj*cs_x), *beta,
		              *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_dccxpbys_mxn
     (
       const dim_t m,
       const dim_t n,
       const double*   restrict x, const inc_t rs_x, const inc_t cs_x,
       const scomplex* restrict beta,
             scomplex* restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
	// If beta is zero, overwrite y with x (in case y has infs or NaNs).
	if ( bli_ceq0( *beta ) )
	{
		bli_dccopys_mxn( m, n, x, rs_x, cs_x, y, rs_y, cs_y );
		return;
	}

#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_dccxpbys( *(x + ii + jj*cs_x), *beta,
		              *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_dccxpbys( *(x + ii*rs_x + jj), *beta,
		              *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_dccxpbys( *(x + ii*rs_x + jj*cs_x), *beta,
		              *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_cccxpbys_mxn
     (
       const dim_t m,
       const dim_t n,
       const scomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
       const scomplex* restrict beta,
             scomplex* restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
	// If beta is zero, overwrite y with x (in case y has infs or NaNs).
	if ( bli_ceq0( *beta ) )
	{
		bli_cccopys_mxn( m, n, x, rs_x, cs_x, y, rs_y, cs_y );
		return;
	}

#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_cccxpbys( *(x + ii + jj*cs_x), *beta,
		              *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_cccxpbys( *(x + ii*rs_x + jj), *beta,
		              *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_cccxpbys( *(x + ii*rs_x + jj*cs_x), *beta,
		              *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_zccxpbys_mxn
     (
       const dim_t m,
       const dim_t n,
       const dcomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
       const scomplex* restrict beta,
             scomplex* restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
	// If beta is zero, overwrite y with x (in case y has infs or NaNs).
	if ( bli_ceq0( *beta ) )
	{
		bli_zccopys_mxn( m, n, x, rs_x, cs_x, y, rs_y, cs_y );
		return;
	}

#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_zccxpbys( *(x + ii + jj*cs_x), *beta,
		              *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_zccxpbys( *(x + ii*rs_x + jj), *beta,
		              *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_zccxpbys( *(x + ii*rs_x + jj*cs_x), *beta,
		              *(y + ii*rs_y + jj*cs_y) );
	}
}

// -- (xby) = (?zz) ------------------------------------------------------------

BLIS_INLINE void bli_szzxpbys_mxn
     (
       const dim_t m,
       const dim_t n,
       const float*    restrict x, const inc_t rs_x, const inc_t cs_x,
       const dcomplex* restrict beta,
             dcomplex* restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
	// If beta is zero, overwrite y with x (in case y has infs or NaNs).
	if ( bli_zeq0( *beta ) )
	{
		bli_szcopys_mxn( m, n, x, rs_x, cs_x, y, rs_y, cs_y );
		return;
	}

#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_szzxpbys( *(x + ii + jj*cs_x), *beta,
		              *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_szzxpbys( *(x + ii*rs_x + jj), *beta,
		              *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_szzxpbys( *(x + ii*rs_x + jj*cs_x), *beta,
		              *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_dzzxpbys_mxn
     (
       const dim_t m,
       const dim_t n,
       const double*   restrict x, const inc_t rs_x, const inc_t cs_x,
       const dcomplex* restrict beta,
             dcomplex* restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
	// If beta is zero, overwrite y with x (in case y has infs or NaNs).
	if ( bli_zeq0( *beta ) )
	{
		bli_dzcopys_mxn( m, n, x, rs_x, cs_x, y, rs_y, cs_y );
		return;
	}

#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_dzzxpbys( *(x + ii + jj*cs_x), *beta,
		              *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_dzzxpbys( *(x + ii*rs_x + jj), *beta,
		              *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_dzzxpbys( *(x + ii*rs_x + jj*cs_x), *beta,
		              *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_czzxpbys_mxn
     (
       const dim_t m,
       const dim_t n,
       const scomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
       const dcomplex* restrict beta,
             dcomplex* restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
	// If beta is zero, overwrite y with x (in case y has infs or NaNs).
	if ( bli_zeq0( *beta ) )
	{
		bli_czcopys_mxn( m, n, x, rs_x, cs_x, y, rs_y, cs_y );
		return;
	}

#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_czzxpbys( *(x + ii + jj*cs_x), *beta,
		              *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_czzxpbys( *(x + ii*rs_x + jj), *beta,
		              *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_czzxpbys( *(x + ii*rs_x + jj*cs_x), *beta,
		              *(y + ii*rs_y + jj*cs_y) );
	}
}
BLIS_INLINE void bli_zzzxpbys_mxn
     (
       const dim_t m,
       const dim_t n,
       const dcomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
       const dcomplex* restrict beta,
             dcomplex* restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
	// If beta is zero, overwrite y with x (in case y has infs or NaNs).
	if ( bli_zeq0( *beta ) )
	{
		bli_zzcopys_mxn( m, n, x, rs_x, cs_x, y, rs_y, cs_y );
		return;
	}

#ifdef BLIS_ENABLE_CR_CASES
	if ( rs_x == 1 && rs_y == 1 )
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_zzzxpbys( *(x + ii + jj*cs_x), *beta,
		              *(y + ii + jj*cs_y) );
	}
	else if ( cs_x == 1 && cs_y == 1 )
	{
		for ( dim_t ii = 0; ii < m; ++ii )
		for ( dim_t jj = 0; jj < n; ++jj )
		bli_zzzxpbys( *(x + ii*rs_x + jj), *beta,
		              *(y + ii*rs_y + jj) );
	}
	else
#endif
	{
		for ( dim_t jj = 0; jj < n; ++jj )
		for ( dim_t ii = 0; ii < m; ++ii )
		bli_zzzxpbys( *(x + ii*rs_x + jj*cs_x), *beta,
		              *(y + ii*rs_y + jj*cs_y) );
	}
}




BLIS_INLINE void bli_sxpbys_mxn
     (
       const dim_t m,
       const dim_t n,
       const float*    restrict x, const inc_t rs_x, const inc_t cs_x,
       const float*    restrict beta,
             float*    restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
	bli_sssxpbys_mxn( m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y );
}
BLIS_INLINE void bli_dxpbys_mxn
     (
       const dim_t m,
       const dim_t n,
       const double*   restrict x, const inc_t rs_x, const inc_t cs_x,
       const double*   restrict beta,
             double*   restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
	bli_dddxpbys_mxn( m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y );
}
BLIS_INLINE void bli_cxpbys_mxn
     (
       const dim_t m,
       const dim_t n,
       const scomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
       const scomplex* restrict beta,
             scomplex* restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
	bli_cccxpbys_mxn( m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y );
}
BLIS_INLINE void bli_zxpbys_mxn
     (
       const dim_t m,
       const dim_t n,
       const dcomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
       const dcomplex* restrict beta,
             dcomplex* restrict y, const inc_t rs_y, const inc_t cs_y
     )
{
	bli_zzzxpbys_mxn( m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y );
}
#endif


#endif
// end bli_xpbys_mxn.h
#line 203 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_xpbys_mxn_uplo.h
#line 1 "./frame/include/level0//bli_xpbys_mxn_uplo.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_XPBYS_MXN_UPLO_H
#define BLIS_XPBYS_MXN_UPLO_H

// xpbys_mxn_u

#define bli_sssxpbys_mxn_u( diagoff, m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y ) \
{ \
	dim_t _i, _j; \
\
	/* If beta is zero, overwrite y with x (in case y has infs or NaNs). */ \
	if ( bli_seq0( *beta ) ) \
	{ \
		for ( _j = 0; _j < n; ++_j ) \
		for ( _i = 0; _i < m; ++_i ) \
		if ( (doff_t)_j - (doff_t)_i >= diagoff ) \
		{ \
			bli_sscopys( *(x + _i*rs_x + _j*cs_x), \
			             *(y + _i*rs_y + _j*cs_y) ); \
		} \
	} \
	else \
	{ \
		for ( _j = 0; _j < n; ++_j ) \
		for ( _i = 0; _i < m; ++_i ) \
		if ( (doff_t)_j - (doff_t)_i >= diagoff ) \
		{ \
			bli_sssxpbys( *(x + _i*rs_x + _j*cs_x), \
			              *(beta), \
			              *(y + _i*rs_y + _j*cs_y) ); \
		} \
	} \
}

#define bli_dddxpbys_mxn_u( diagoff, m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y ) \
{ \
	dim_t _i, _j; \
\
	/* If beta is zero, overwrite y with x (in case y has infs or NaNs). */ \
	if ( bli_deq0( *beta ) ) \
	{ \
		for ( _j = 0; _j < n; ++_j ) \
		for ( _i = 0; _i < m; ++_i ) \
		if ( (doff_t)_j - (doff_t)_i >= diagoff ) \
		{ \
			bli_ddcopys( *(x + _i*rs_x + _j*cs_x), \
			             *(y + _i*rs_y + _j*cs_y) ); \
		} \
	} \
	else \
	{ \
		for ( _j = 0; _j < n; ++_j ) \
		for ( _i = 0; _i < m; ++_i ) \
		if ( (doff_t)_j - (doff_t)_i >= diagoff ) \
		{ \
			bli_dddxpbys( *(x + _i*rs_x + _j*cs_x), \
			              *(beta), \
			              *(y + _i*rs_y + _j*cs_y) ); \
		} \
	} \
}

#define bli_cccxpbys_mxn_u( diagoff, m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y ) \
{ \
	dim_t _i, _j; \
\
	/* If beta is zero, overwrite y with x (in case y has infs or NaNs). */ \
	if ( bli_ceq0( *beta ) ) \
	{ \
		for ( _j = 0; _j < n; ++_j ) \
		for ( _i = 0; _i < m; ++_i ) \
		if ( (doff_t)_j - (doff_t)_i >= diagoff ) \
		{ \
			bli_cccopys( *(x + _i*rs_x + _j*cs_x), \
			             *(y + _i*rs_y + _j*cs_y) ); \
		} \
	} \
	else \
	{ \
		for ( _j = 0; _j < n; ++_j ) \
		for ( _i = 0; _i < m; ++_i ) \
		if ( (doff_t)_j - (doff_t)_i >= diagoff ) \
		{ \
			bli_cccxpbys( *(x + _i*rs_x + _j*cs_x), \
			              *(beta), \
			              *(y + _i*rs_y + _j*cs_y) ); \
		} \
	} \
}

#define bli_zzzxpbys_mxn_u( diagoff, m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y ) \
{ \
	dim_t _i, _j; \
\
	/* If beta is zero, overwrite y with x (in case y has infs or NaNs). */ \
	if ( bli_zeq0( *beta ) ) \
	{ \
		for ( _j = 0; _j < n; ++_j ) \
		for ( _i = 0; _i < m; ++_i ) \
		if ( (doff_t)_j - (doff_t)_i >= diagoff ) \
		{ \
			bli_zzcopys( *(x + _i*rs_x + _j*cs_x), \
			             *(y + _i*rs_y + _j*cs_y) ); \
		} \
	} \
	else \
	{ \
		for ( _j = 0; _j < n; ++_j ) \
		for ( _i = 0; _i < m; ++_i ) \
		if ( (doff_t)_j - (doff_t)_i >= diagoff ) \
		{ \
			bli_zzzxpbys( *(x + _i*rs_x + _j*cs_x), \
			              *(beta), \
			              *(y + _i*rs_y + _j*cs_y) ); \
		} \
	} \
}

// xpbys_mxn_l

#define bli_sssxpbys_mxn_l( diagoff, m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y ) \
{ \
	dim_t _i, _j; \
\
	/* If beta is zero, overwrite y with x (in case y has infs or NaNs). */ \
	if ( bli_seq0( *beta ) ) \
	{ \
		for ( _j = 0; _j < n; ++_j ) \
		for ( _i = 0; _i < m; ++_i ) \
		if ( (doff_t)_j - (doff_t)_i <= diagoff ) \
		{ \
			bli_sscopys( *(x + _i*rs_x + _j*cs_x), \
			             *(y + _i*rs_y + _j*cs_y) ); \
		} \
	} \
	else \
	{ \
		for ( _j = 0; _j < n; ++_j ) \
		for ( _i = 0; _i < m; ++_i ) \
		if ( (doff_t)_j - (doff_t)_i <= diagoff ) \
		{ \
			bli_sssxpbys( *(x + _i*rs_x + _j*cs_x), \
			              *(beta), \
			              *(y + _i*rs_y + _j*cs_y) ); \
		} \
	} \
}

#define bli_dddxpbys_mxn_l( diagoff, m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y ) \
{ \
	dim_t _i, _j; \
\
	/* If beta is zero, overwrite y with x (in case y has infs or NaNs). */ \
	if ( bli_deq0( *beta ) ) \
	{ \
		for ( _j = 0; _j < n; ++_j ) \
		for ( _i = 0; _i < m; ++_i ) \
		if ( (doff_t)_j - (doff_t)_i <= diagoff ) \
		{ \
			bli_ddcopys( *(x + _i*rs_x + _j*cs_x), \
			             *(y + _i*rs_y + _j*cs_y) ); \
		} \
	} \
	else \
	{ \
		for ( _j = 0; _j < n; ++_j ) \
		for ( _i = 0; _i < m; ++_i ) \
		if ( (doff_t)_j - (doff_t)_i <= diagoff ) \
		{ \
			bli_dddxpbys( *(x + _i*rs_x + _j*cs_x), \
			              *(beta), \
			              *(y + _i*rs_y + _j*cs_y) ); \
		} \
	} \
}

#define bli_cccxpbys_mxn_l( diagoff, m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y ) \
{ \
	dim_t _i, _j; \
\
	/* If beta is zero, overwrite y with x (in case y has infs or NaNs). */ \
	if ( bli_ceq0( *beta ) ) \
	{ \
		for ( _j = 0; _j < n; ++_j ) \
		for ( _i = 0; _i < m; ++_i ) \
		if ( (doff_t)_j - (doff_t)_i <= diagoff ) \
		{ \
			bli_cccopys( *(x + _i*rs_x + _j*cs_x), \
			             *(y + _i*rs_y + _j*cs_y) ); \
		} \
	} \
	else \
	{ \
		for ( _j = 0; _j < n; ++_j ) \
		for ( _i = 0; _i < m; ++_i ) \
		if ( (doff_t)_j - (doff_t)_i <= diagoff ) \
		{ \
			bli_cccxpbys( *(x + _i*rs_x + _j*cs_x), \
			              *(beta), \
			              *(y + _i*rs_y + _j*cs_y) ); \
		} \
	} \
}

#define bli_zzzxpbys_mxn_l( diagoff, m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y ) \
{ \
	dim_t _i, _j; \
\
	/* If beta is zero, overwrite y with x (in case y has infs or NaNs). */ \
	if ( bli_zeq0( *beta ) ) \
	{ \
		for ( _j = 0; _j < n; ++_j ) \
		for ( _i = 0; _i < m; ++_i ) \
		if ( (doff_t)_j - (doff_t)_i <= diagoff ) \
		{ \
			bli_zzcopys( *(x + _i*rs_x + _j*cs_x), \
			             *(y + _i*rs_y + _j*cs_y) ); \
		} \
	} \
	else \
	{ \
		for ( _j = 0; _j < n; ++_j ) \
		for ( _i = 0; _i < m; ++_i ) \
		if ( (doff_t)_j - (doff_t)_i <= diagoff ) \
		{ \
			bli_zzzxpbys( *(x + _i*rs_x + _j*cs_x), \
			              *(beta), \
			              *(y + _i*rs_y + _j*cs_y) ); \
		} \
	} \
}


#define bli_sxpbys_mxn_u( diagoff, m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y ) \
{\
	bli_sssxpbys_mxn_u( diagoff, m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y ); \
}
#define bli_dxpbys_mxn_u( diagoff, m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y ) \
{\
	bli_dddxpbys_mxn_u( diagoff, m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y ); \
}
#define bli_cxpbys_mxn_u( diagoff, m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y ) \
{\
	bli_cccxpbys_mxn_u( diagoff, m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y ); \
}
#define bli_zxpbys_mxn_u( diagoff, m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y ) \
{\
	bli_zzzxpbys_mxn_u( diagoff, m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y ); \
}
#define bli_sxpbys_mxn_l( diagoff, m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y ) \
{\
	bli_sssxpbys_mxn_l( diagoff, m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y ); \
}
#define bli_dxpbys_mxn_l( diagoff, m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y ) \
{\
	bli_dddxpbys_mxn_l( diagoff, m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y ); \
}
#define bli_cxpbys_mxn_l( diagoff, m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y ) \
{\
	bli_cccxpbys_mxn_l( diagoff, m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y ); \
}
#define bli_zxpbys_mxn_l( diagoff, m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y ) \
{\
	bli_zzzxpbys_mxn_l( diagoff, m, n, x, rs_x, cs_x, beta, y, rs_y, cs_y ); \
}

#endif
// end bli_xpbys_mxn_uplo.h
#line 204 "./frame/include//bli_scalar_macro_defs.h"


// -- "broadcast B" scalar macros --

// begin bli_bcastbbs_mxn.h
#line 1 "./frame/include/level0/bb//bli_bcastbbs_mxn.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_BCASTBBS_MXN_H
#define BLIS_BCASTBBS_MXN_H

// bcastbbs_mxn

#undef  GENTFUNC
#define GENTFUNC( ctype, ch, opname ) \
\
BLIS_INLINE void PASTEMAC(ch,opname) \
     ( \
       const dim_t        m, \
       const dim_t        n, \
       ctype*    restrict y, const inc_t incy, const inc_t ldy  \
     ) \
{ \
	/* Assume that the duplication factor is the column stride of y. */ \
	const dim_t d    = ldy; \
	const dim_t ds_y = 1; \
\
	for ( dim_t i = 0; i < m; ++i ) \
	{ \
		ctype* restrict yi = y + i*incy; \
\
		for ( dim_t j = 0; j < n; ++j ) \
		{ \
			ctype* restrict yij = yi + j*ldy; \
\
			for ( dim_t p = 1; p < d; ++p ) \
			{ \
				ctype* restrict yijd = yij + p*ds_y; \
\
				PASTEMAC(ch,copys)( *yij, *yijd ); \
			} \
		} \
	} \
}

INSERT_GENTFUNC_BASIC( bcastbbs_mxn )

#endif
// end bli_bcastbbs_mxn.h
#line 208 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_scal2bbs_mxn.h
#line 1 "./frame/include/level0/bb//bli_scal2bbs_mxn.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SCAL2BBS_MXN_H
#define BLIS_SCAL2BBS_MXN_H

// scal2bbs_mxn

#undef  GENTFUNCRO
#define GENTFUNCRO( ctype, ch, opname ) \
\
BLIS_INLINE void PASTEMAC(ch,opname) \
     ( \
       const conj_t       conjx, \
       const dim_t        m, \
       const dim_t        n, \
       ctype*    restrict alpha, \
       ctype*    restrict x, const inc_t incx, const inc_t ldx, \
       ctype*    restrict y, const inc_t incy, const inc_t ldy  \
     ) \
{ \
	/* Assume that the duplication factor is the row stride of y. */ \
	const dim_t d    = incy; \
	const dim_t ds_y = 1; \
\
	if ( bli_is_conj( conjx ) ) \
	{ \
		for ( dim_t j = 0; j < n; ++j ) \
		{ \
			ctype* restrict xj = x + j*ldx; \
			ctype* restrict yj = y + j*ldy; \
\
			for ( dim_t i = 0; i < m; ++i ) \
			{ \
				ctype* restrict xij = xj + i*incx; \
				ctype* restrict yij = yj + i*incy; \
\
				PASTEMAC(ch,scal2js)( *alpha, *xij, *yij ); \
\
				for ( dim_t p = 1; p < d; ++p ) \
				{ \
					ctype* restrict yijd = yij + p*ds_y; \
\
					PASTEMAC(ch,copys)( *yij, *yijd ); \
				} \
			} \
		} \
	} \
	else /* if ( bli_is_noconj( conjx ) ) */ \
	{ \
		for ( dim_t j = 0; j < n; ++j ) \
		{ \
			ctype* restrict xj = x + j*ldx; \
			ctype* restrict yj = y + j*ldy; \
\
			for ( dim_t i = 0; i < m; ++i ) \
			{ \
				ctype* restrict xij = xj + i*incx; \
				ctype* restrict yij = yj + i*incy; \
\
				PASTEMAC(ch,scal2s)( *alpha, *xij, *yij ); \
\
				for ( dim_t p = 1; p < d; ++p ) \
				{ \
					ctype* restrict yijd = yij + p*ds_y; \
\
					PASTEMAC(ch,copys)( *yij, *yijd ); \
				} \
			} \
		} \
	} \
}

INSERT_GENTFUNCRO_BASIC( scal2bbs_mxn )


#undef  GENTFUNCCO
#define GENTFUNCCO( ctype, ctype_r, ch, chr, opname ) \
\
BLIS_INLINE void PASTEMAC(ch,opname) \
     ( \
       const conj_t       conjx, \
       const dim_t        m, \
       const dim_t        n, \
       ctype*    restrict alpha, \
       ctype*    restrict x, const inc_t incx, const inc_t ldx, \
       ctype*    restrict y, const inc_t incy, const inc_t ldy  \
     ) \
{ \
	/* Assume that the duplication factor is the row stride of y. */ \
	const dim_t       d          = incy; \
	const dim_t       ds_y       = 1; \
\
	const inc_t       incx2      = 2 * incx; \
	const inc_t       ldx2       = 2 * ldx; \
\
	const inc_t       incy2      = 2 * incy; \
	const inc_t       ldy2       = 2 * ldy; \
\
	ctype_r* restrict alpha_r    = ( ctype_r* )alpha; \
	ctype_r* restrict alpha_i    = ( ctype_r* )alpha + 1; \
	ctype_r* restrict chi_r      = ( ctype_r* )x; \
	ctype_r* restrict chi_i      = ( ctype_r* )x + 1; \
	ctype_r* restrict psi_r      = ( ctype_r* )y; \
	ctype_r* restrict psi_i      = ( ctype_r* )y + 1*d; \
\
	if ( bli_is_conj( conjx ) ) \
	{ \
		for ( dim_t j = 0; j < n; ++j ) \
		{ \
			ctype_r* restrict chij_r = chi_r + j*ldx2; \
			ctype_r* restrict chij_i = chi_i + j*ldx2; \
			ctype_r* restrict psij_r = psi_r + j*ldy2; \
			ctype_r* restrict psij_i = psi_i + j*ldy2; \
\
			for ( dim_t i = 0; i < m; ++i ) \
			{ \
				ctype_r* restrict chiij_r = chij_r + i*incx2; \
				ctype_r* restrict chiij_i = chij_i + i*incx2; \
				ctype_r* restrict psiij_r = psij_r + i*incy2; \
				ctype_r* restrict psiij_i = psij_i + i*incy2; \
\
				PASTEMAC(ch,scal2jris)( *alpha_r, *alpha_i, \
				                        *chiij_r, *chiij_i, \
				                        *psiij_r, *psiij_i ); \
\
				for ( dim_t p = 1; p < d; ++p ) \
				{ \
					ctype_r* restrict psiijd_r = psiij_r + p*ds_y; \
					ctype_r* restrict psiijd_i = psiij_i + p*ds_y; \
\
					PASTEMAC(ch,copyris)( *psiij_r,  *psiij_i, \
					                      *psiijd_r, *psiijd_i ); \
				} \
			} \
		} \
	} \
	else /* if ( bli_is_noconj( conjx ) ) */ \
	{ \
		for ( dim_t j = 0; j < n; ++j ) \
		{ \
			ctype_r* restrict chij_r = chi_r + j*ldx2; \
			ctype_r* restrict chij_i = chi_i + j*ldx2; \
			ctype_r* restrict psij_r = psi_r + j*ldy2; \
			ctype_r* restrict psij_i = psi_i + j*ldy2; \
\
			for ( dim_t i = 0; i < m; ++i ) \
			{ \
				ctype_r* restrict chiij_r = chij_r + i*incx2; \
				ctype_r* restrict chiij_i = chij_i + i*incx2; \
				ctype_r* restrict psiij_r = psij_r + i*incy2; \
				ctype_r* restrict psiij_i = psij_i + i*incy2; \
\
				PASTEMAC(ch,scal2ris)( *alpha_r, *alpha_i, \
				                       *chiij_r, *chiij_i, \
				                       *psiij_r, *psiij_i ); \
\
				for ( dim_t p = 1; p < d; ++p ) \
				{ \
					ctype_r* restrict psiijd_r = psiij_r + p*ds_y; \
					ctype_r* restrict psiijd_i = psiij_i + p*ds_y; \
\
					PASTEMAC(ch,copyris)( *psiij_r,  *psiij_i, \
					                      *psiijd_r, *psiijd_i ); \
				} \
			} \
		} \
	} \
}

INSERT_GENTFUNCCO( scal2bbs_mxn )

#endif
// end bli_scal2bbs_mxn.h
#line 209 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_set0bbs_mxn.h
#line 1 "./frame/include/level0/bb//bli_set0bbs_mxn.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SET0BBS_MXN_H
#define BLIS_SET0BBS_MXN_H

// set0bbs_mxn

#undef  GENTFUNC
#define GENTFUNC( ctype, ch, opname ) \
\
BLIS_INLINE void PASTEMAC(ch,opname) \
     ( \
       const dim_t        m, \
       const dim_t        n, \
       ctype*    restrict y, const inc_t incy, const inc_t ldy  \
     ) \
{ \
	/* Assume that the duplication factor is the row stride of y. */ \
	const dim_t d    = incy; \
	const dim_t ds_y = 1; \
\
	for ( dim_t j = 0; j < n; ++j ) \
	{ \
		ctype* restrict yj = y + j*ldy; \
\
		for ( dim_t i = 0; i < m; ++i ) \
		{ \
			ctype* restrict yij = yj + i*incy; \
\
			for ( dim_t p = 0; p < d; ++p ) \
			{ \
				ctype* restrict yijd = yij + p*ds_y; \
\
				PASTEMAC(ch,set0s)( *yijd ); \
			} \
		} \
	} \
}

INSERT_GENTFUNC_BASIC( set0bbs_mxn )

#endif
// end bli_set0bbs_mxn.h
#line 210 "./frame/include//bli_scalar_macro_defs.h"



// -- 1m-specific scalar macros --

// 1e
// begin bli_copy1es.h
#line 1 "./frame/include/level0/1e//bli_copy1es.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_COPY1ES_H
#define BLIS_COPY1ES_H

// copy1es

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of y.

#define bli_sscopy1es( a, bri, bir ) { ( void )a; ( void )bri; ( void )bir; }
#define bli_dscopy1es( a, bri, bir ) { ( void )a; ( void )bri; ( void )bir; }
#define bli_cscopy1es( a, bri, bir ) { ( void )a; ( void )bri; ( void )bir; }
#define bli_zscopy1es( a, bri, bir ) { ( void )a; ( void )bri; ( void )bir; }

#define bli_sdcopy1es( a, bri, bir ) { ( void )a; ( void )bri; ( void )bir; }
#define bli_ddcopy1es( a, bri, bir ) { ( void )a; ( void )bri; ( void )bir; }
#define bli_cdcopy1es( a, bri, bir ) { ( void )a; ( void )bri; ( void )bir; }
#define bli_zdcopy1es( a, bri, bir ) { ( void )a; ( void )bri; ( void )bir; }

#define bli_sccopy1es( a, bri, bir ) { ( void )a; ( void )bri; ( void )bir; }
#define bli_dccopy1es( a, bri, bir ) { ( void )a; ( void )bri; ( void )bir; }
#define bli_cccopy1es( a, bri, bir ) \
{ \
	bli_cccopyris(  bli_creal(a), bli_cimag(a), bli_creal(bri), bli_cimag(bri) ); \
	bli_cccopyris( -bli_cimag(a), bli_creal(a), bli_creal(bir), bli_cimag(bir) ); \
}
#define bli_zccopy1es( a, bri, bir ) \
{ \
	bli_zccopyris(  bli_zreal(a), bli_zimag(a), bli_creal(bri), bli_cimag(bri) ); \
	bli_zccopyris( -bli_zimag(a), bli_zreal(a), bli_creal(bir), bli_cimag(bir) ); \
}

#define bli_szcopy1es( a, bri, bir ) { ( void )a; ( void )bri; ( void )bir; }
#define bli_dzcopy1es( a, bri, bir ) { ( void )a; ( void )bri; ( void )bir; }
#define bli_czcopy1es( a, bri, bir ) \
{ \
	bli_czcopyris(  bli_creal(a), bli_cimag(a), bli_zreal(bri), bli_zimag(bri) ); \
	bli_czcopyris( -bli_cimag(a), bli_creal(a), bli_zreal(bir), bli_zimag(bir) ); \
}
#define bli_zzcopy1es( a, bri, bir ) \
{ \
	bli_zzcopyris(  bli_zreal(a), bli_zimag(a), bli_zreal(bri), bli_zimag(bri) ); \
	bli_zzcopyris( -bli_zimag(a), bli_zreal(a), bli_zreal(bir), bli_zimag(bir) ); \
}


#define bli_ccopy1es( a, bri, bir ) bli_cccopy1es( a, bri, bir )
#define bli_zcopy1es( a, bri, bir ) bli_zzcopy1es( a, bri, bir )

#endif

// end bli_copy1es.h
#line 216 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_copyj1es.h
#line 1 "./frame/include/level0/1e//bli_copyj1es.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_COPYJ1ES_H
#define BLIS_COPYJ1ES_H

// copyj1es

// Notes:
// - The first char encodes the type of x.
// - The second char encodes the type of y.

#define bli_sscopyj1es( a, bri, bir ) { ( void )a; ( void )bri; ( void )bir; }
#define bli_dscopyj1es( a, bri, bir ) { ( void )a; ( void )bri; ( void )bir; }
#define bli_cscopyj1es( a, bri, bir ) { ( void )a; ( void )bri; ( void )bir; }
#define bli_zscopyj1es( a, bri, bir ) { ( void )a; ( void )bri; ( void )bir; }

#define bli_sdcopyj1es( a, bri, bir ) { ( void )a; ( void )bri; ( void )bir; }
#define bli_ddcopyj1es( a, bri, bir ) { ( void )a; ( void )bri; ( void )bir; }
#define bli_cdcopyj1es( a, bri, bir ) { ( void )a; ( void )bri; ( void )bir; }
#define bli_zdcopyj1es( a, bri, bir ) { ( void )a; ( void )bri; ( void )bir; }

#define bli_sccopyj1es( a, bri, bir ) { ( void )a; ( void )bri; ( void )bir; }
#define bli_dccopyj1es( a, bri, bir ) { ( void )a; ( void )bri; ( void )bir; }
#define bli_cccopyj1es( a, bri, bir ) \
{ \
	bli_cccopyris( bli_creal(a), -bli_cimag(a), bli_creal(bri), bli_cimag(bri) ); \
	bli_cccopyris( bli_cimag(a),  bli_creal(a), bli_creal(bir), bli_cimag(bir) ); \
}
#define bli_zccopyj1es( a, bri, bir ) \
{ \
	bli_zccopyris( bli_zreal(a), -bli_zimag(a), bli_creal(bri), bli_cimag(bri) ); \
	bli_zccopyris( bli_zimag(a),  bli_zreal(a), bli_creal(bir), bli_cimag(bir) ); \
}

#define bli_szcopyj1es( a, bri, bir ) { ( void )a; ( void )bri; ( void )bir; }
#define bli_dzcopyj1es( a, bri, bir ) { ( void )a; ( void )bri; ( void )bir; }
#define bli_czcopyj1es( a, bri, bir ) \
{ \
	bli_czcopyris( bli_creal(a), -bli_cimag(a), bli_zreal(bri), bli_zimag(bri) ); \
	bli_czcopyris( bli_cimag(a),  bli_creal(a), bli_zreal(bir), bli_zimag(bir) ); \
}
#define bli_zzcopyj1es( a, bri, bir ) \
{ \
	bli_zzcopyris( bli_zreal(a), -bli_zimag(a), bli_zreal(bri), bli_zimag(bri) ); \
	bli_zzcopyris( bli_zimag(a),  bli_zreal(a), bli_zreal(bir), bli_zimag(bir) ); \
}


#define bli_ccopyj1es( a, bri, bir ) bli_cccopyj1es( a, bri, bir )
#define bli_zcopyj1es( a, bri, bir ) bli_zzcopyj1es( a, bri, bir )

#endif

// end bli_copyj1es.h
#line 217 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_invert1es.h
#line 1 "./frame/include/level0/1e//bli_invert1es.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_INVERT1ES_H
#define BLIS_INVERT1ES_H

// invert1es

#define bli_cinvert1es( bri, bir ) \
{ \
	bli_cinvertris( bli_creal(bri), bli_cimag(bri) ); \
	bli_ccopyris( bli_creal(bri), -bli_cimag(bri), bli_cimag(bir), bli_creal(bir) ); \
}

#define bli_zinvert1es( bri, bir ) \
{ \
	bli_zinvertris( bli_zreal(bri), bli_zimag(bri) ); \
	bli_zcopyris( bli_zreal(bri), -bli_zimag(bri), bli_zimag(bir), bli_zreal(bir) ); \
}

#endif

// end bli_invert1es.h
#line 219 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_scal1es.h
#line 1 "./frame/include/level0/1e//bli_scal1es.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SCAL1ES_H
#define BLIS_SCAL1ES_H

// scal1es

#define bli_cscal1es( a, yri, yir ) \
{ \
	bli_cscalris(  bli_creal(a),   bli_cimag(a),   bli_creal(yri), bli_cimag(yri) ); \
	bli_ccopyris( -bli_cimag(yri), bli_creal(yri), bli_creal(yir), bli_cimag(yir) ); \
}

#define bli_zscal1es( a, yri, yir ) \
{ \
	bli_zscalris(  bli_zreal(a),   bli_zimag(a),   bli_zreal(yri), bli_zimag(yri) ); \
	bli_zcopyris( -bli_zimag(yri), bli_zreal(yri), bli_zreal(yir), bli_zimag(yir) ); \
}

#endif

// end bli_scal1es.h
#line 221 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_scal21es.h
#line 1 "./frame/include/level0/1e//bli_scal21es.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SCAL21ES_H
#define BLIS_SCAL21ES_H

// scal21es

// Notes:
// - The first char encodes the type of a.
// - The second char encodes the type of x.
// - The third char encodes the type of y.

// -- (axy) = (??s) ------------------------------------------------------------

#define bli_sssscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_sdsscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_scsscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_szsscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }

#define bli_dssscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_ddsscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_dcsscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_dzsscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }

#define bli_cssscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_cdsscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_ccsscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_czsscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }

#define bli_zssscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_zdsscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_zcsscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_zzsscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }

// -- (axy) = (??d) ------------------------------------------------------------

#define bli_ssdscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_sddscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_scdscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_szdscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }

#define bli_dsdscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_dddscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_dcdscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_dzdscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }

#define bli_csdscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_cddscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_ccdscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_czdscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }

#define bli_zsdscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_zddscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_zcdscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_zzdscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }

// -- (axy) = (??c) ------------------------------------------------------------

#define bli_sscscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_sdcscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_sccscal21es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_sreal(a), bli_simag(a),  bli_creal(x), bli_cimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_sreal(a), bli_simag(a), -bli_cimag(x), bli_creal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_szcscal21es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_sreal(a), bli_simag(a),  bli_zreal(x), bli_zimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_sreal(a), bli_simag(a), -bli_zimag(x), bli_zreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}

#define bli_dscscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_ddcscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_dccscal21es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_dreal(a), bli_dimag(a),  bli_creal(x), bli_cimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_dreal(a), bli_dimag(a), -bli_cimag(x), bli_creal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_dzcscal21es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_dreal(a), bli_dimag(a),  bli_zreal(x), bli_zimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_dreal(a), bli_dimag(a), -bli_zimag(x), bli_zreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}

#define bli_cscscal21es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a),  bli_sreal(x), bli_simag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a), -bli_simag(x), bli_sreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_cdcscal21es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a),  bli_dreal(x), bli_dimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a), -bli_dimag(x), bli_dreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_cccscal21es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a),  bli_creal(x), bli_cimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a), -bli_cimag(x), bli_creal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_czcscal21es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a),  bli_zreal(x), bli_zimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a), -bli_zimag(x), bli_zreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}

#define bli_zscscal21es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a),  bli_sreal(x), bli_simag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a), -bli_simag(x), bli_sreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_zdcscal21es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a),  bli_dreal(x), bli_dimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a), -bli_dimag(x), bli_dreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_zccscal21es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a),  bli_creal(x), bli_cimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a), -bli_cimag(x), bli_creal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_zzcscal21es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a),  bli_zreal(x), bli_zimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a), -bli_zimag(x), bli_zreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}

// -- (axy) = (??z) ------------------------------------------------------------

#define bli_sszscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_sdzscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_sczscal21es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_sreal(a), bli_simag(a),  bli_creal(x), bli_cimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_sreal(a), bli_simag(a), -bli_cimag(x), bli_creal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_szzscal21es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_sreal(a), bli_simag(a),  bli_zreal(x), bli_zimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_sreal(a), bli_simag(a), -bli_zimag(x), bli_zreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}

#define bli_dszscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_ddzscal21es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_dczscal21es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_dreal(a), bli_dimag(a),  bli_creal(x), bli_cimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_dreal(a), bli_dimag(a), -bli_cimag(x), bli_creal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_dzzscal21es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_dreal(a), bli_dimag(a),  bli_zreal(x), bli_zimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_dreal(a), bli_dimag(a), -bli_zimag(x), bli_zreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}

#define bli_cszscal21es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a),  bli_sreal(x), bli_simag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a), -bli_simag(x), bli_sreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_cdzscal21es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a),  bli_dreal(x), bli_dimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a), -bli_dimag(x), bli_dreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_cczscal21es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a),  bli_creal(x), bli_cimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a), -bli_cimag(x), bli_creal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_czzscal21es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a),  bli_zreal(x), bli_zimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a), -bli_zimag(x), bli_zreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}

#define bli_zszscal21es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a),  bli_sreal(x), bli_simag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a), -bli_simag(x), bli_sreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_zdzscal21es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a),  bli_dreal(x), bli_dimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a), -bli_dimag(x), bli_dreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_zczscal21es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a),  bli_creal(x), bli_cimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a), -bli_cimag(x), bli_creal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_zzzscal21es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a),  bli_zreal(x), bli_zimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a), -bli_zimag(x), bli_zreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}



#define bli_cscal21es( a, x, yri, yir ) bli_cccscal21es( a, x, yri, yir )
#define bli_zscal21es( a, x, yri, yir ) bli_zzzscal21es( a, x, yri, yir )

#endif

// end bli_scal21es.h
#line 223 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_scal2j1es.h
#line 1 "./frame/include/level0/1e//bli_scal2j1es.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SCAL2J1ES_H
#define BLIS_SCAL2J1ES_H

// scal2j1es

// Notes:
// - The first char encodes the type of a.
// - The second char encodes the type of x.
// - The third char encodes the type of y.

// -- (axy) = (??s) ------------------------------------------------------------

#define bli_sssscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_sdsscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_scsscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_szsscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }

#define bli_dssscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_ddsscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_dcsscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_dzsscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }

#define bli_cssscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_cdsscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_ccsscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_czsscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }

#define bli_zssscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_zdsscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_zcsscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_zzsscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }

// -- (axy) = (??d) ------------------------------------------------------------

#define bli_ssdscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_sddscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_scdscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_szdscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }

#define bli_dsdscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_dddscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_dcdscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_dzdscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }

#define bli_csdscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_cddscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_ccdscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_czdscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }

#define bli_zsdscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_zddscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_zcdscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_zzdscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }

// -- (axy) = (??c) ------------------------------------------------------------

#define bli_sscscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_sdcscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_sccscal2j1es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_sreal(a), bli_simag(a), bli_creal(x), -bli_cimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_sreal(a), bli_simag(a), bli_cimag(x),  bli_creal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_szcscal2j1es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_sreal(a), bli_simag(a), bli_zreal(x), -bli_zimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_sreal(a), bli_simag(a), bli_zimag(x),  bli_zreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}

#define bli_dscscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_ddcscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_dccscal2j1es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_dreal(a), bli_dimag(a), bli_creal(x), -bli_cimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_dreal(a), bli_dimag(a), bli_cimag(x),  bli_creal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_dzcscal2j1es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_dreal(a), bli_dimag(a), bli_zreal(x), -bli_zimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_dreal(a), bli_dimag(a), bli_zimag(x),  bli_zreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}

#define bli_cscscal2j1es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a), bli_sreal(x), -bli_simag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a), bli_simag(x),  bli_sreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_cdcscal2j1es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a), bli_dreal(x), -bli_dimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a), bli_dimag(x),  bli_dreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_cccscal2j1es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a), bli_creal(x), -bli_cimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a), bli_cimag(x),  bli_creal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_czcscal2j1es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a), bli_zreal(x), -bli_zimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a), bli_zimag(x),  bli_zreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}

#define bli_zscscal2j1es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a), bli_sreal(x), -bli_simag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a), bli_simag(x),  bli_sreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_zdcscal2j1es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a), bli_dreal(x), -bli_dimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a), bli_dimag(x),  bli_dreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_zccscal2j1es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a), bli_creal(x), -bli_cimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a), bli_cimag(x),  bli_creal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_zzcscal2j1es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a), bli_zreal(x), -bli_zimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a), bli_zimag(x),  bli_zreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}

// -- (axy) = (??z) ------------------------------------------------------------

#define bli_sszscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_sdzscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_sczscal2j1es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_sreal(a), bli_simag(a), bli_creal(x), -bli_cimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_sreal(a), bli_simag(a), bli_cimag(x),  bli_creal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_szzscal2j1es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_sreal(a), bli_simag(a), bli_zreal(x), -bli_zimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_sreal(a), bli_simag(a), bli_zimag(x),  bli_zreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}

#define bli_dszscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_ddzscal2j1es( a, x, yri, yir ) { ( void )a; ( void )x; ( void )yri; ( void )yir; }
#define bli_dczscal2j1es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_dreal(a), bli_dimag(a), bli_creal(x), -bli_cimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_dreal(a), bli_dimag(a), bli_cimag(x),  bli_creal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_dzzscal2j1es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_dreal(a), bli_dimag(a), bli_zreal(x), -bli_zimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_dreal(a), bli_dimag(a), bli_zimag(x),  bli_zreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}

#define bli_cszscal2j1es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a), bli_sreal(x), -bli_simag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a), bli_simag(x),  bli_sreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_cdzscal2j1es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a), bli_dreal(x), -bli_dimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a), bli_dimag(x),  bli_dreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_cczscal2j1es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a), bli_creal(x), -bli_cimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a), bli_cimag(x),  bli_creal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_czzscal2j1es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a), bli_zreal(x), -bli_zimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a), bli_zimag(x),  bli_zreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}

#define bli_zszscal2j1es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a), bli_sreal(x), -bli_simag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a), bli_simag(x),  bli_sreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_zdzscal2j1es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a), bli_dreal(x), -bli_dimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a), bli_dimag(x),  bli_dreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_zczscal2j1es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a), bli_creal(x), -bli_cimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a), bli_cimag(x),  bli_creal(x), bli_zreal(yir), bli_zimag(yir) ); \
}
#define bli_zzzscal2j1es( a, x, yri, yir ) \
{ \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a), bli_zreal(x), -bli_zimag(x), bli_zreal(yri), bli_zimag(yri) ); \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a), bli_zimag(x),  bli_zreal(x), bli_zreal(yir), bli_zimag(yir) ); \
}



#define bli_cscal2j1es( a, x, yri, yir ) bli_cccscal2j1es( a, x, yri, yir )
#define bli_zscal2j1es( a, x, yri, yir ) bli_zzzscal2j1es( a, x, yri, yir )

#endif

// end bli_scal2j1es.h
#line 224 "./frame/include//bli_scalar_macro_defs.h"


// 1r
// begin bli_copy1rs.h
#line 1 "./frame/include/level0/1r//bli_copy1rs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_COPY1RS_H
#define BLIS_COPY1RS_H

// copy1rs

#define bli_ccopy1rs( a, br, bi ) \
{ \
	bli_ccopyris( bli_creal(a), bli_cimag(a), br, bi ); \
}

#define bli_zcopy1rs( a, br, bi ) \
{ \
	bli_zcopyris( bli_zreal(a), bli_zimag(a), br, bi ); \
}

#endif

// end bli_copy1rs.h
#line 227 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_copyj1rs.h
#line 1 "./frame/include/level0/1r//bli_copyj1rs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_COPYJ1RS_H
#define BLIS_COPYJ1RS_H

// copyj1rs

#define bli_ccopyj1rs( a, br, bi ) \
{ \
	bli_ccopyjris( bli_creal(a), bli_cimag(a), br, bi ); \
}

#define bli_zcopyj1rs( a, br, bi ) \
{ \
	bli_zcopyjris( bli_zreal(a), bli_zimag(a), br, bi ); \
}

#endif

// end bli_copyj1rs.h
#line 228 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_invert1rs.h
#line 1 "./frame/include/level0/1r//bli_invert1rs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_INVERT1RS_H
#define BLIS_INVERT1RS_H

// invert1rs

#define bli_cinvert1rs( xr, xi )  bli_cinvertris( xr, xi )
#define bli_zinvert1rs( xr, xi )  bli_zinvertris( xr, xi )

#endif
// end bli_invert1rs.h
#line 230 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_scal1rs.h
#line 1 "./frame/include/level0/1r//bli_scal1rs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SCAL1RS_H
#define BLIS_SCAL1RS_H

// scal1rs

#define bli_cscal1rs( a, yr, yi ) \
{ \
	bli_cscalris( bli_creal(a), bli_cimag(a), yr, yi ); \
}

#define bli_zscal1rs( a, yr, yi ) \
{ \
	bli_zscalris( bli_zreal(a), bli_zimag(a), yr, yi ); \
}

#define bli_scscal1rs( a, yr, yi ) \
{ \
	bli_scscalris( bli_sreal(a), bli_simag(a), yr, yi ); \
}

#define bli_dzscal1rs( a, yr, yi ) \
{ \
	bli_dzscalris( bli_dreal(a), bli_dimag(a), yr, yi ); \
}

#endif

// end bli_scal1rs.h
#line 232 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_scal21rs.h
#line 1 "./frame/include/level0/1r//bli_scal21rs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SCAL21RS_H
#define BLIS_SCAL21RS_H

// scal21rs

#define bli_cscscal21rs( a, x, yr, yi ) \
{ \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), yr, yi ); \
}

#define bli_cccscal21rs( a, x, yr, yi ) \
{ \
	bli_cxscal2ris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), yr, yi ); \
}

#define bli_zdzscal21rs( a, x, yr, yi ) \
{ \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), yr, yi ); \
}

#define bli_zzzscal21rs( a, x, yr, yi ) \
{ \
	bli_cxscal2ris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), yr, yi ); \
}


#define bli_cscal21rs( a, x, yr, yi ) bli_cccscal21rs( a, x, yr, yi )
#define bli_zscal21rs( a, x, yr, yi ) bli_zzzscal21rs( a, x, yr, yi )

#endif

// end bli_scal21rs.h
#line 234 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_scal2j1rs.h
#line 1 "./frame/include/level0/1r//bli_scal2j1rs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SCAL2J1RS_H
#define BLIS_SCAL2J1RS_H

// scal2j1rs

#define bli_cscscal2j1rs( a, x, yr, yi ) \
{ \
	bli_cscal2jris( bli_creal(a), bli_cimag(a), bli_sreal(x), bli_simag(x), yr, yi ); \
}

#define bli_cccscal2j1rs( a, x, yr, yi ) \
{ \
	bli_cscal2jris( bli_creal(a), bli_cimag(a), bli_creal(x), bli_cimag(x), yr, yi ); \
}

#define bli_zdzscal2j1rs( a, x, yr, yi ) \
{ \
	bli_zscal2jris( bli_zreal(a), bli_zimag(a), bli_dreal(x), bli_dimag(x), yr, yi ); \
}

#define bli_zzzscal2j1rs( a, x, yr, yi ) \
{ \
	bli_zscal2jris( bli_zreal(a), bli_zimag(a), bli_zreal(x), bli_zimag(x), yr, yi ); \
}


#define bli_cscal2j1rs( a, x, yr, yi ) bli_cccscal2j1rs( a, x, yr, yi )
#define bli_zscal2j1rs( a, x, yr, yi ) bli_zzzscal2j1rs( a, x, yr, yi )

#endif

// end bli_scal2j1rs.h
#line 235 "./frame/include//bli_scalar_macro_defs.h"


// 1m (1e or 1r)
// begin bli_invert1ms_mxn_diag.h
#line 1 "./frame/include/level0/1m//bli_invert1ms_mxn_diag.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_INVERT1MS_MXN_DIAG_H
#define BLIS_INVERT1MS_MXN_DIAG_H

// invert1ms_mxn_diag

#define bli_cinvert1ms_mxn_diag( schema, offm, offn, m, n, y, rs_y, cs_y, ld_y ) \
{ \
	dim_t min_m_n = bli_min( m, n ); \
	dim_t i; \
\
	/* Handle 1e and 1r separately. */ \
	if ( bli_is_1e_packed( schema ) ) \
	{ \
		scomplex* restrict y_off_ri = y + (offm  )*rs_y \
		                                + (offn  )*cs_y; \
		scomplex* restrict y_off_ir = y + (offm  )*rs_y \
		                                + (offn  )*cs_y + ld_y/2; \
\
		for ( i = 0; i < min_m_n; ++i ) \
		{ \
			bli_cinvert1es( *(y_off_ri + i*rs_y + i*cs_y), \
			                *(y_off_ir + i*rs_y + i*cs_y) ); \
		} \
	} \
	else /* if ( bli_is_1r_packed( schema ) ) */ \
	{ \
		inc_t rs_y2 = rs_y; \
		inc_t cs_y2 = cs_y; \
\
		/* Scale the non-unit stride by two for the 1r loop, which steps
		   in units of real (not complex) values. */ \
		if         ( rs_y2 == 1 )    { cs_y2 *= 2; } \
		else /* if ( cs_y2 == 1 ) */ { rs_y2 *= 2; } \
\
		float*    restrict y_cast  = ( float* )y; \
		float*    restrict y_off_r = y_cast + (offm  )*rs_y2 \
		                                    + (offn  )*cs_y2; \
		float*    restrict y_off_i = y_cast + (offm  )*rs_y2 \
		                                    + (offn  )*cs_y2 + ld_y; \
\
		for ( i = 0; i < min_m_n; ++i ) \
		{ \
			bli_cinvert1rs( *(y_off_r + i*rs_y2 + i*cs_y2), \
			                *(y_off_i + i*rs_y2 + i*cs_y2) ); \
		} \
	} \
}

#define bli_zinvert1ms_mxn_diag( schema, offm, offn, m, n, y, rs_y, cs_y, ld_y ) \
{ \
	dim_t min_m_n = bli_min( m, n ); \
	dim_t i; \
\
	/* Handle 1e and 1r separately. */ \
	if ( bli_is_1e_packed( schema ) ) \
	{ \
		dcomplex* restrict y_off_ri = y + (offm  )*rs_y \
		                                + (offn  )*cs_y; \
		dcomplex* restrict y_off_ir = y + (offm  )*rs_y \
		                                + (offn  )*cs_y + ld_y/2; \
\
		for ( i = 0; i < min_m_n; ++i ) \
		{ \
			bli_zinvert1es( *(y_off_ri + i*rs_y + i*cs_y), \
			                *(y_off_ir + i*rs_y + i*cs_y) ); \
		} \
	} \
	else /* if ( bli_is_1r_packed( schema ) ) */ \
	{ \
		inc_t rs_y2 = rs_y; \
		inc_t cs_y2 = cs_y; \
\
		/* Scale the non-unit stride by two for the 1r loop, which steps
		   in units of real (not complex) values. */ \
		if         ( rs_y2 == 1 )    { cs_y2 *= 2; } \
		else /* if ( cs_y2 == 1 ) */ { rs_y2 *= 2; } \
\
		double*   restrict y_cast  = ( double* )y; \
		double*   restrict y_off_r = y_cast + (offm  )*rs_y2 \
		                                    + (offn  )*cs_y2; \
		double*   restrict y_off_i = y_cast + (offm  )*rs_y2 \
		                                    + (offn  )*cs_y2 + ld_y; \
\
		for ( i = 0; i < min_m_n; ++i ) \
		{ \
			bli_zinvert1rs( *(y_off_r + i*rs_y2 + i*cs_y2), \
			                *(y_off_i + i*rs_y2 + i*cs_y2) ); \
		} \
	} \
}

#endif
// end bli_invert1ms_mxn_diag.h
#line 238 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_scal1ms_mxn.h
#line 1 "./frame/include/level0/1m//bli_scal1ms_mxn.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SCAL1MS_MXN_H
#define BLIS_SCAL1MS_MXN_H

// scal1ms_mxn

#define bli_cscal1ms_mxn( schema, m, n, a, y, rs_y, cs_y, ld_y ) \
{ \
	dim_t i, j; \
\
	/* Handle 1e and 1r separately. */ \
	if ( bli_is_1e_packed( schema ) ) \
	{ \
		scomplex* restrict y_ri = y; \
		scomplex* restrict y_ir = y + ld_y/2; \
\
		for ( j = 0; j < n; ++j ) \
		for ( i = 0; i < m; ++i ) \
		{ \
			bli_cscal1es( *(a), \
			              *(y_ri + i*rs_y + j*cs_y), \
			              *(y_ir + i*rs_y + j*cs_y) ); \
		} \
	} \
	else /* if ( bli_is_1r_packed( schema ) ) */ \
	{ \
		inc_t rs_y2 = rs_y; \
		inc_t cs_y2 = cs_y; \
\
		/* Scale the non-unit stride by two for the 1r loop, which steps
		   in units of real (not complex) values. */ \
		if         ( rs_y2 == 1 )    { cs_y2 *= 2; } \
		else /* if ( cs_y2 == 1 ) */ { rs_y2 *= 2; } \
\
		float*    restrict y_cast = ( float* )y; \
		float*    restrict y_r    = y_cast; \
		float*    restrict y_i    = y_cast + ld_y; \
\
		for ( j = 0; j < n; ++j ) \
		for ( i = 0; i < m; ++i ) \
		{ \
			bli_cscal1rs( *(a), \
			              *(y_r + i*rs_y2 + j*cs_y2), \
			              *(y_i + i*rs_y2 + j*cs_y2) ); \
		} \
	} \
}

#define bli_zscal1ms_mxn( schema, m, n, a, y, rs_y, cs_y, ld_y ) \
{ \
	dim_t i, j; \
\
	/* Handle 1e and 1r separately. */ \
	if ( bli_is_1e_packed( schema ) ) \
	{ \
		dcomplex* restrict y_ri = y; \
		dcomplex* restrict y_ir = y + ld_y/2; \
\
		for ( j = 0; j < n; ++j ) \
		for ( i = 0; i < m; ++i ) \
		{ \
			bli_zscal1es( *(a), \
			              *(y_ri + i*rs_y + j*cs_y), \
			              *(y_ir + i*rs_y + j*cs_y) ); \
		} \
	} \
	else /* if ( bli_is_1r_packed( schema ) ) */ \
	{ \
		inc_t rs_y2 = rs_y; \
		inc_t cs_y2 = cs_y; \
\
		/* Scale the non-unit stride by two for the 1r loop,
		   which steps in units of real (not complex) values. */ \
		if         ( rs_y2 == 1 )    { cs_y2 *= 2; } \
		else /* if ( cs_y2 == 1 ) */ { rs_y2 *= 2; } \
\
		double*   restrict y_cast = ( double* )y; \
		double*   restrict y_r    = y_cast; \
		double*   restrict y_i    = y_cast + ld_y; \
\
		for ( j = 0; j < n; ++j ) \
		for ( i = 0; i < m; ++i ) \
		{ \
			bli_zscal1rs( *(a), \
			              *(y_r + i*rs_y2 + j*cs_y2), \
			              *(y_i + i*rs_y2 + j*cs_y2) ); \
		} \
	} \
}

#endif
// end bli_scal1ms_mxn.h
#line 240 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_scal21ms_mxn.h
#line 1 "./frame/include/level0/1m//bli_scal21ms_mxn.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SCAL21MS_MXN_H
#define BLIS_SCAL21MS_MXN_H

// scal21ms_mxn

BLIS_INLINE void bli_cscal21ms_mxn
     (
       const pack_t       schema,
       const conj_t       conjx,
       const dim_t        m,
       const dim_t        n,
       scomplex* restrict alpha,
       scomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
       scomplex* restrict y, const inc_t rs_y, const inc_t cs_y, const inc_t ld_y
     )
{
	dim_t i, j;

	/* Handle 1e and 1r separately. */
	if ( bli_is_1e_packed( schema ) )
	{
		scomplex* restrict y_ri = y;
		scomplex* restrict y_ir = y + ld_y/2;

		if ( bli_is_conj( conjx ) )
		{
			for ( j = 0; j < n; ++j )
			for ( i = 0; i < m; ++i )
			{
				bli_cscal2j1es( *(alpha),
				                *(x    + i*rs_x + j*cs_x),
				                *(y_ri + i*rs_y + j*cs_y),
				                *(y_ir + i*rs_y + j*cs_y) );
			}
		}
		else /* if ( bli_is_noconj( conjx ) ) */
		{
			for ( j = 0; j < n; ++j )
			for ( i = 0; i < m; ++i )
			{
				bli_cscal21es( *(alpha),
				               *(x    + i*rs_x + j*cs_x),
				               *(y_ri + i*rs_y + j*cs_y),
				               *(y_ir + i*rs_y + j*cs_y) );
			}
		}
	}
	else /* if ( bli_is_1r_packed( schema ) ) */
	{
		inc_t rs_y2 = rs_y;
		inc_t cs_y2 = cs_y;

		/* Scale the non-unit stride by two for the 1r loop, which steps
		   in units of real (not complex) values. */
		if         ( rs_y2 == 1 )    { cs_y2 *= 2; }
		else /* if ( cs_y2 == 1 ) */ { rs_y2 *= 2; }

		float*  restrict y_cast = ( float* )y;
		float*  restrict y_r    = y_cast;
		float*  restrict y_i    = y_cast + ld_y;

		if ( bli_is_conj( conjx ) )
		{
			for ( j = 0; j < n; ++j )
			for ( i = 0; i < m; ++i )
			{
				bli_cscal2j1rs( *(alpha),
				                *(x   + i*rs_x  + j*cs_x ),
				                *(y_r + i*rs_y2 + j*cs_y2),
				                *(y_i + i*rs_y2 + j*cs_y2) );
			}
		}
		else /* if ( bli_is_noconj( conjx ) ) */
		{
			for ( j = 0; j < n; ++j )
			for ( i = 0; i < m; ++i )
			{
				bli_cscal21rs( *(alpha),
				               *(x   + i*rs_x  + j*cs_x ),
				               *(y_r + i*rs_y2 + j*cs_y2),
				               *(y_i + i*rs_y2 + j*cs_y2) );
			}
		}
	}
}

BLIS_INLINE void bli_zscal21ms_mxn
     (
       const pack_t       schema,
       const conj_t       conjx,
       const dim_t        m,
       const dim_t        n,
       dcomplex* restrict alpha,
       dcomplex* restrict x, const inc_t rs_x, const inc_t cs_x,
       dcomplex* restrict y, const inc_t rs_y, const inc_t cs_y, const inc_t ld_y
     )
{
	dim_t i, j;

	/* Handle 1e and 1r separately. */
	if ( bli_is_1e_packed( schema ) )
	{
		dcomplex* restrict y_ri = y;
		dcomplex* restrict y_ir = y + ld_y/2;

		if ( bli_is_conj( conjx ) )
		{
			for ( j = 0; j < n; ++j )
			for ( i = 0; i < m; ++i )
			{
				bli_zscal2j1es( *(alpha),
				                *(x    + i*rs_x + j*cs_x),
				                *(y_ri + i*rs_y + j*cs_y),
				                *(y_ir + i*rs_y + j*cs_y) );
			}
		}
		else /* if ( bli_is_noconj( conjx ) ) */
		{
			for ( j = 0; j < n; ++j )
			for ( i = 0; i < m; ++i )
			{
				bli_zscal21es( *(alpha),
				               *(x    + i*rs_x + j*cs_x),
				               *(y_ri + i*rs_y + j*cs_y),
				               *(y_ir + i*rs_y + j*cs_y) );
			}
		}
	}
	else /* if ( bli_is_1r_packed( schema ) ) */
	{
		inc_t rs_y2 = rs_y;
		inc_t cs_y2 = cs_y;

		/* Scale the non-unit stride by two for the 1r loop, which steps
		   in units of real (not complex) values. */
		if         ( rs_y2 == 1 )    { cs_y2 *= 2; }
		else /* if ( cs_y2 == 1 ) */ { rs_y2 *= 2; }

		double* restrict y_cast = ( double* )y;
		double* restrict y_r    = y_cast;
		double* restrict y_i    = y_cast + ld_y;

		if ( bli_is_conj( conjx ) )
		{
			for ( j = 0; j < n; ++j )
			for ( i = 0; i < m; ++i )
			{
				bli_zscal2j1rs( *(alpha),
				                *(x   + i*rs_x  + j*cs_x ),
				                *(y_r + i*rs_y2 + j*cs_y2),
				                *(y_i + i*rs_y2 + j*cs_y2) );
			}
		}
		else /* if ( bli_is_noconj( conjx ) ) */
		{
			for ( j = 0; j < n; ++j )
			for ( i = 0; i < m; ++i )
			{
				bli_zscal21rs( *(alpha),
				               *(x   + i*rs_x  + j*cs_x ),
				               *(y_r + i*rs_y2 + j*cs_y2),
				               *(y_i + i*rs_y2 + j*cs_y2) );
			}
		}
	}
}

#endif
// end bli_scal21ms_mxn.h
#line 242 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_scal21ms_mxn_diag.h
#line 1 "./frame/include/level0/1m//bli_scal21ms_mxn_diag.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SCAL21MS_MXN_DIAG_H
#define BLIS_SCAL21MS_MXN_DIAG_H

// scal21ms_mxn_diag

#define bli_cscscal21ms_mxn_diag( schema, m, n, a, x, rs_x, cs_x, y, rs_y, cs_y, ld_y ) \
{ \
	dim_t min_m_n = bli_min( m, n ); \
	dim_t i; \
\
	/* Handle 1e and 1r separately. */ \
	if ( bli_is_1e_packed( schema ) ) \
	{ \
		scomplex* restrict y_off_ri = y; \
		scomplex* restrict y_off_ir = y + ld_y/2; \
\
		for ( i = 0; i < min_m_n; ++i ) \
		{ \
			bli_cscscal21es( *(a), \
			                 *(x        + i*rs_x + i*cs_x), \
			                 *(y_off_ri + i*rs_y + i*cs_y), \
			                 *(y_off_ir + i*rs_y + i*cs_y) ); \
		} \
	} \
	else /* if ( bli_is_1r_packed( schema ) ) */ \
	{ \
		inc_t rs_y2 = rs_y; \
		inc_t cs_y2 = cs_y; \
\
		/* Scale the non-unit stride by two for the 1r loop, which steps
		   in units of real (not complex) values. */ \
		if         ( rs_y2 == 1 )    { cs_y2 *= 2; } \
		else /* if ( cs_y2 == 1 ) */ { rs_y2 *= 2; } \
\
		float*    restrict y_cast  = ( float* )y; \
		float*    restrict y_off_r = y_cast; \
		float*    restrict y_off_i = y_cast + ld_y; \
\
		for ( i = 0; i < min_m_n; ++i ) \
		{ \
			bli_cscscal21rs( *(a), \
			                 *(x       + i*rs_x  + i*cs_x ), \
			                 *(y_off_r + i*rs_y2 + i*cs_y2), \
			                 *(y_off_i + i*rs_y2 + i*cs_y2) ); \
		} \
	} \
}

#define bli_zdzscal21ms_mxn_diag( schema, m, n, a, x, rs_x, cs_x, y, rs_y, cs_y, ld_y ) \
{ \
	dim_t min_m_n = bli_min( m, n ); \
	dim_t i; \
\
	/* Handle 1e and 1r separately. */ \
	if ( bli_is_1e_packed( schema ) ) \
	{ \
		dcomplex* restrict y_off_ri = y; \
		dcomplex* restrict y_off_ir = y + ld_y/2; \
\
		for ( i = 0; i < min_m_n; ++i ) \
		{ \
			bli_zdzscal21es( *(a), \
			                 *(x        + i*rs_x + i*cs_x), \
			                 *(y_off_ri + i*rs_y + i*cs_y), \
			                 *(y_off_ir + i*rs_y + i*cs_y) ); \
		} \
	} \
	else /* if ( bli_is_1r_packed( schema ) ) */ \
	{ \
		inc_t rs_y2 = rs_y; \
		inc_t cs_y2 = cs_y; \
\
		/* Scale the non-unit stride by two for the 1r loop, which steps
		   in units of real (not complex) values. */ \
		if         ( rs_y2 == 1 )    { cs_y2 *= 2; } \
		else /* if ( cs_y2 == 1 ) */ { rs_y2 *= 2; } \
\
		double*   restrict y_cast  = ( double* )y; \
		double*   restrict y_off_r = y_cast; \
		double*   restrict y_off_i = y_cast + ld_y; \
\
		for ( i = 0; i < min_m_n; ++i ) \
		{ \
			bli_zdzscal21rs( *(a), \
			                 *(x       + i*rs_x  + i*cs_x ), \
			                 *(y_off_r + i*rs_y2 + i*cs_y2), \
			                 *(y_off_i + i*rs_y2 + i*cs_y2) ); \
		} \
	} \
}

#endif
// end bli_scal21ms_mxn_diag.h
#line 243 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_scal21ms_mxn_uplo.h
#line 1 "./frame/include/level0/1m//bli_scal21ms_mxn_uplo.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SCAL21MS_MXN_UPLO_H
#define BLIS_SCAL21MS_MXN_UPLO_H

// scal21ms_mxn_uplo

#define bli_cscal21ms_mxn_uplo( schema, uplo, conjx, m, a, x, rs_x, cs_x, y, rs_y, cs_y, ld_y ) \
{ \
	dim_t i, j; \
\
	/* Handle 1e and 1r separately. */ \
	if ( bli_is_1e_packed( schema ) ) \
	{ \
		scomplex* restrict y_ri = y; \
		scomplex* restrict y_ir = y + ld_y/2; \
\
		if ( bli_is_lower( uplo ) ) \
		{ \
			if ( bli_is_conj( conjx ) ) \
			{ \
				for ( j = 0; j < m; ++j ) \
				for ( i = j; i < m; ++i ) \
				{ \
					bli_cscal2j1es( *(a), \
					                *(x    + i*rs_x + j*cs_x), \
					                *(y_ri + i*rs_y + j*cs_y), \
					                *(y_ir + i*rs_y + j*cs_y) ); \
				} \
			} \
			else /* if ( bli_is_noconj( conjx ) ) */ \
			{ \
				for ( j = 0; j < m; ++j ) \
				for ( i = j; i < m; ++i ) \
				{ \
					bli_cscal21es( *(a), \
					               *(x    + i*rs_x + j*cs_x), \
					               *(y_ri + i*rs_y + j*cs_y), \
					               *(y_ir + i*rs_y + j*cs_y) ); \
				} \
			} \
		} \
		else /* if ( bli_is_upper( uplo ) ) */ \
		{ \
			if ( bli_is_conj( conjx ) ) \
			{ \
				for ( j = 0; j < m; ++j ) \
				for ( i = 0; i < j + 1; ++i ) \
				{ \
					bli_cscal2j1es( *(a), \
					                *(x    + i*rs_x + j*cs_x), \
					                *(y_ri + i*rs_y + j*cs_y), \
					                *(y_ir + i*rs_y + j*cs_y) ); \
				} \
			} \
			else /* if ( bli_is_noconj( conjx ) ) */ \
			{ \
				for ( j = 0; j < m; ++j ) \
				for ( i = 0; i < j + 1; ++i ) \
				{ \
					bli_cscal21es( *(a), \
					               *(x    + i*rs_x + j*cs_x), \
					               *(y_ri + i*rs_y + j*cs_y), \
					               *(y_ir + i*rs_y + j*cs_y) ); \
				} \
			} \
		} \
	} \
	else /* if ( bli_is_1r_packed( schema ) ) */ \
	{ \
		inc_t rs_y2 = rs_y; \
		inc_t cs_y2 = cs_y; \
\
		/* Scale the non-unit stride by two for the 1r loop, which steps
		   in units of real (not complex) values. */ \
		if         ( rs_y2 == 1 )    { cs_y2 *= 2; } \
		else /* if ( cs_y2 == 1 ) */ { rs_y2 *= 2; } \
\
		float*    restrict y_cast = ( float* )y; \
		float*    restrict y_r    = y_cast; \
		float*    restrict y_i    = y_cast + ld_y; \
\
		if ( bli_is_lower( uplo ) ) \
		{ \
			if ( bli_is_conj( conjx ) ) \
			{ \
				for ( j = 0; j < m; ++j ) \
				for ( i = j; i < m; ++i ) \
				{ \
					bli_cscal2j1rs( *(a), \
					                *(x   + i*rs_x  + j*cs_x ), \
					                *(y_r + i*rs_y2 + j*cs_y2), \
					                *(y_i + i*rs_y2 + j*cs_y2) ); \
				} \
			} \
			else /* if ( bli_is_noconj( conjx ) ) */ \
			{ \
				for ( j = 0; j < m; ++j ) \
				for ( i = j; i < m; ++i ) \
				{ \
					bli_cscal21rs( *(a), \
					               *(x   + i*rs_x  + j*cs_x ), \
					               *(y_r + i*rs_y2 + j*cs_y2), \
					               *(y_i + i*rs_y2 + j*cs_y2) ); \
				} \
			} \
		} \
		else /* if ( bli_is_upper( uplo ) ) */ \
		{ \
			if ( bli_is_conj( conjx ) ) \
			{ \
				for ( j = 0; j < m; ++j ) \
				for ( i = 0; i < j + 1; ++i ) \
				{ \
					bli_cscal2j1rs( *(a), \
					                *(x   + i*rs_x  + j*cs_x ), \
					                *(y_r + i*rs_y2 + j*cs_y2), \
					                *(y_i + i*rs_y2 + j*cs_y2) ); \
				} \
			} \
			else /* if ( bli_is_noconj( conjx ) ) */ \
			{ \
				for ( j = 0; j < m; ++j ) \
				for ( i = 0; i < j + 1; ++i ) \
				{ \
					bli_cscal21rs( *(a), \
					               *(x   + i*rs_x  + j*cs_x ), \
					               *(y_r + i*rs_y2 + j*cs_y2), \
					               *(y_i + i*rs_y2 + j*cs_y2) ); \
				} \
			} \
		} \
	} \
}

#define bli_zscal21ms_mxn_uplo( schema, uplo, conjx, m, a, x, rs_x, cs_x, y, rs_y, cs_y, ld_y ) \
{ \
	dim_t i, j; \
\
	/* Handle 1e and 1r separately. */ \
	if ( bli_is_1e_packed( schema ) ) \
	{ \
		dcomplex* restrict y_ri = y; \
		dcomplex* restrict y_ir = y + ld_y/2; \
\
		if ( bli_is_lower( uplo ) ) \
		{ \
			if ( bli_is_conj( conjx ) ) \
			{ \
				for ( j = 0; j < m; ++j ) \
				for ( i = j; i < m; ++i ) \
				{ \
					bli_zscal2j1es( *(a), \
					                *(x    + i*rs_x + j*cs_x), \
					                *(y_ri + i*rs_y + j*cs_y), \
					                *(y_ir + i*rs_y + j*cs_y) ); \
				} \
			} \
			else /* if ( bli_is_noconj( conjx ) ) */ \
			{ \
				for ( j = 0; j < m; ++j ) \
				for ( i = j; i < m; ++i ) \
				{ \
					bli_zscal21es( *(a), \
					               *(x    + i*rs_x + j*cs_x), \
					               *(y_ri + i*rs_y + j*cs_y), \
					               *(y_ir + i*rs_y + j*cs_y) ); \
				} \
			} \
		} \
		else /* if ( bli_is_upper( uplo ) ) */ \
		{ \
			if ( bli_is_conj( conjx ) ) \
			{ \
				for ( j = 0; j < m; ++j ) \
				for ( i = 0; i < j + 1; ++i ) \
				{ \
					bli_zscal2j1es( *(a), \
					                *(x    + i*rs_x + j*cs_x), \
					                *(y_ri + i*rs_y + j*cs_y), \
					                *(y_ir + i*rs_y + j*cs_y) ); \
				} \
			} \
			else /* if ( bli_is_noconj( conjx ) ) */ \
			{ \
				for ( j = 0; j < m; ++j ) \
				for ( i = 0; i < j + 1; ++i ) \
				{ \
					bli_zscal21es( *(a), \
					               *(x    + i*rs_x + j*cs_x), \
					               *(y_ri + i*rs_y + j*cs_y), \
					               *(y_ir + i*rs_y + j*cs_y) ); \
				} \
			} \
		} \
	} \
	else /* if ( bli_is_1r_packed( schema ) ) */ \
	{ \
		inc_t rs_y2 = rs_y; \
		inc_t cs_y2 = cs_y; \
\
		/* Scale the non-unit stride by two for the 1r loop, which steps
		   in units of real (not complex) values. */ \
		if         ( rs_y2 == 1 )    { cs_y2 *= 2; } \
		else /* if ( cs_y2 == 1 ) */ { rs_y2 *= 2; } \
\
		double*   restrict y_cast = ( double* )y; \
		double*   restrict y_r    = y_cast; \
		double*   restrict y_i    = y_cast + ld_y; \
\
		if ( bli_is_lower( uplo ) ) \
		{ \
			if ( bli_is_conj( conjx ) ) \
			{ \
				for ( j = 0; j < m; ++j ) \
				for ( i = j; i < m; ++i ) \
				{ \
					bli_zscal2j1rs( *(a), \
					                *(x   + i*rs_x  + j*cs_x ), \
					                *(y_r + i*rs_y2 + j*cs_y2), \
					                *(y_i + i*rs_y2 + j*cs_y2) ); \
				} \
			} \
			else /* if ( bli_is_noconj( conjx ) ) */ \
			{ \
				for ( j = 0; j < m; ++j ) \
				for ( i = j; i < m; ++i ) \
				{ \
					bli_zscal21rs( *(a), \
					               *(x   + i*rs_x  + j*cs_x ), \
					               *(y_r + i*rs_y2 + j*cs_y2), \
					               *(y_i + i*rs_y2 + j*cs_y2) ); \
				} \
			} \
		} \
		else /* if ( bli_is_upper( uplo ) ) */ \
		{ \
			if ( bli_is_conj( conjx ) ) \
			{ \
				for ( j = 0; j < m; ++j ) \
				for ( i = 0; i < j + 1; ++i ) \
				{ \
					bli_zscal2j1rs( *(a), \
					                *(x   + i*rs_x  + j*cs_x ), \
					                *(y_r + i*rs_y2 + j*cs_y2), \
					                *(y_i + i*rs_y2 + j*cs_y2) ); \
				} \
			} \
			else /* if ( bli_is_noconj( conjx ) ) */ \
			{ \
				for ( j = 0; j < m; ++j ) \
				for ( i = 0; i < j + 1; ++i ) \
				{ \
					bli_zscal21rs( *(a), \
					               *(x   + i*rs_x  + j*cs_x ), \
					               *(y_r + i*rs_y2 + j*cs_y2), \
					               *(y_i + i*rs_y2 + j*cs_y2) ); \
				} \
			} \
		} \
	} \
}

#endif
// end bli_scal21ms_mxn_uplo.h
#line 244 "./frame/include//bli_scalar_macro_defs.h"


// begin bli_set1ms_mxn.h
#line 1 "./frame/include/level0/1m//bli_set1ms_mxn.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SET1MS_MXN_H
#define BLIS_SET1MS_MXN_H

// set1ms_mxn

#define bli_sset1ms_mxn( schema, offm, offn, m, n, a, y, rs_y, cs_y, ld_y ) \
{ \
	/* Include real domain version to facilitate macro-izing mixed-datatype
	   components of packm. */ \
}

#define bli_dset1ms_mxn( schema, offm, offn, m, n, a, y, rs_y, cs_y, ld_y ) \
{ \
	/* Include real domain version to facilitate macro-izing mixed-datatype
	   components of packm. */ \
}

BLIS_INLINE void bli_cset1ms_mxn
     (
       const pack_t       schema,
       const dim_t        offm,
       const dim_t        offn,
       const dim_t        m,
       const dim_t        n,
       scomplex* restrict alpha,
       scomplex* restrict y, const inc_t rs_y, const inc_t cs_y, const inc_t ld_y
     )
{
	inc_t offm_local = offm;
	inc_t offn_local = offn;
	dim_t m_local    = m;
	dim_t n_local    = n;
	inc_t rs_y1      = rs_y;
	inc_t cs_y1      = cs_y;
	inc_t rs_y2      = rs_y;
	inc_t cs_y2      = cs_y;
	dim_t i, j;

	/* Optimization: The loops walk through y with unit stride if y is
	   column-stored. If y is row-stored, swap the dimensions and strides
	   to preserve unit stride movement. */
	if ( cs_y == 1 )
	{
		bli_swap_incs( &offm_local, &offn_local );
		bli_swap_dims( &m_local, &n_local );
		bli_swap_incs( &rs_y1, &cs_y1 );
		bli_swap_incs( &rs_y2, &cs_y2 );
	}

	/* Handle 1e and 1r separately. */
	if ( bli_is_1e_packed( schema ) )
	{
		scomplex* restrict y_off_ri = y + (offm_local  )*rs_y1
		                                + (offn_local  )*cs_y1;
		scomplex* restrict y_off_ir = y + (offm_local  )*rs_y1
		                                + (offn_local  )*cs_y1 + ld_y/2;

		for ( j = 0; j < n_local; ++j )
		for ( i = 0; i < m_local; ++i )
		{
			bli_ccopy1es( *(alpha),
			              *(y_off_ri + i*rs_y1 + j*cs_y1),
			              *(y_off_ir + i*rs_y1 + j*cs_y1) );
		}
	}
	else /* if ( bli_is_1r_packed( schema ) ) */
	{
		/* Scale the non-unit stride by two for the 1r loop, which steps
		   in units of real (not complex) values. */
		if         ( rs_y2 == 1 )    { cs_y2 *= 2; }
		else /* if ( cs_y2 == 1 ) */ { rs_y2 *= 2; }

		float*    restrict y_cast  = ( float* )y;
		float*    restrict y_off_r = y_cast + (offm_local  )*rs_y2
		                                    + (offn_local  )*cs_y2;
		float*    restrict y_off_i = y_cast + (offm_local  )*rs_y2
		                                    + (offn_local  )*cs_y2 + ld_y;

		for ( j = 0; j < n_local; ++j )
		for ( i = 0; i < m_local; ++i )
		{
			bli_ccopy1rs( *(alpha),
			              *(y_off_r + i*rs_y2 + j*cs_y2),
			              *(y_off_i + i*rs_y2 + j*cs_y2) );
		}
	}
}

BLIS_INLINE void bli_zset1ms_mxn
     (
       const pack_t       schema,
       const dim_t        offm,
       const dim_t        offn,
       const dim_t        m,
       const dim_t        n,
       dcomplex* restrict alpha,
       dcomplex* restrict y, const inc_t rs_y, const inc_t cs_y, const inc_t ld_y
     )
{
	inc_t offm_local = offm;
	inc_t offn_local = offn;
	dim_t m_local    = m;
	dim_t n_local    = n;
	inc_t rs_y1      = rs_y;
	inc_t cs_y1      = cs_y;
	inc_t rs_y2      = rs_y;
	inc_t cs_y2      = cs_y;
	dim_t i, j;

	/* Optimization: The loops walk through y with unit stride if y is
	   column-stored. If y is row-stored, swap the dimensions and strides
	   to preserve unit stride movement. */
	if ( cs_y == 1 )
	{
		bli_swap_incs( &offm_local, &offn_local );
		bli_swap_dims( &m_local, &n_local );
		bli_swap_incs( &rs_y1, &cs_y1 );
		bli_swap_incs( &rs_y2, &cs_y2 );
	}

	/* Handle 1e and 1r separately. */
	if ( bli_is_1e_packed( schema ) )
	{
		dcomplex* restrict y_off_ri = y + (offm_local  )*rs_y1
		                                + (offn_local  )*cs_y1;
		dcomplex* restrict y_off_ir = y + (offm_local  )*rs_y1
		                                + (offn_local  )*cs_y1 + ld_y/2;

		for ( j = 0; j < n_local; ++j )
		for ( i = 0; i < m_local; ++i )
		{
			bli_zcopy1es( *(alpha),
			              *(y_off_ri + i*rs_y1 + j*cs_y1),
			              *(y_off_ir + i*rs_y1 + j*cs_y1) );
		}
	}
	else /* if ( bli_is_1r_packed( schema ) ) */
	{
		/* Scale the non-unit stride by two for the 1r loop, which steps
		   in units of real (not complex) values. */
		if         ( rs_y2 == 1 )    { cs_y2 *= 2; }
		else /* if ( cs_y2 == 1 ) */ { rs_y2 *= 2; }

		double*   restrict y_cast  = ( double* )y;
		double*   restrict y_off_r = y_cast + (offm_local  )*rs_y2
		                                    + (offn_local  )*cs_y2;
		double*   restrict y_off_i = y_cast + (offm_local  )*rs_y2
		                                    + (offn_local  )*cs_y2 + ld_y;

		for ( j = 0; j < n_local; ++j )
		for ( i = 0; i < m_local; ++i )
		{
			bli_zcopy1rs( *(alpha),
			              *(y_off_r + i*rs_y2 + j*cs_y2),
			              *(y_off_i + i*rs_y2 + j*cs_y2) );
		}
	}
}

#endif
// end bli_set1ms_mxn.h
#line 246 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_set1ms_mxn_diag.h
#line 1 "./frame/include/level0/1m//bli_set1ms_mxn_diag.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SET1MS_MXN_DIAG_H
#define BLIS_SET1MS_MXN_DIAG_H

// set1ms_mxn_diag

#define bli_cset1ms_mxn_diag( schema, offm, offn, m, n, a, y, rs_y, cs_y, ld_y ) \
{ \
	dim_t min_m_n = bli_min( m, n ); \
	dim_t i; \
\
	/* Handle 1e and 1r separately. */ \
	if ( bli_is_1e_packed( schema ) ) \
	{ \
		scomplex* restrict y_off_ri = y + (offm  )*rs_y \
		                                + (offn  )*cs_y; \
		scomplex* restrict y_off_ir = y + (offm  )*rs_y \
		                                + (offn  )*cs_y + ld_y/2; \
\
		for ( i = 0; i < min_m_n; ++i ) \
		{ \
			bli_ccopy1es( *(a), \
			              *(y_off_ri + i*rs_y + i*cs_y), \
			              *(y_off_ir + i*rs_y + i*cs_y) ); \
		} \
	} \
	else /* if ( bli_is_1r_packed( schema ) ) */ \
	{ \
		inc_t rs_y2 = rs_y; \
		inc_t cs_y2 = cs_y; \
\
		/* Scale the non-unit stride by two for the 1r loop, which steps
		   in units of real (not complex) values. */ \
		if         ( rs_y2 == 1 )    { cs_y2 *= 2; } \
		else /* if ( cs_y2 == 1 ) */ { rs_y2 *= 2; } \
\
		float*    restrict y_cast  = ( float* )y; \
		float*    restrict y_off_r = y_cast + (offm  )*rs_y2 \
		                                    + (offn  )*cs_y2; \
		float*    restrict y_off_i = y_cast + (offm  )*rs_y2 \
		                                    + (offn  )*cs_y2 + ld_y; \
\
		for ( i = 0; i < min_m_n; ++i ) \
		{ \
			bli_ccopy1rs( *(a), \
			              *(y_off_r + i*rs_y2 + i*cs_y2), \
			              *(y_off_i + i*rs_y2 + i*cs_y2) ); \
		} \
	} \
}

#define bli_zset1ms_mxn_diag( schema, offm, offn, m, n, a, y, rs_y, cs_y, ld_y ) \
{ \
	dim_t min_m_n = bli_min( m, n ); \
	dim_t i; \
\
	/* Handle 1e and 1r separately. */ \
	if ( bli_is_1e_packed( schema ) ) \
	{ \
		dcomplex* restrict y_off_ri = y + (offm  )*rs_y \
		                                + (offn  )*cs_y; \
		dcomplex* restrict y_off_ir = y + (offm  )*rs_y \
		                                + (offn  )*cs_y + ld_y/2; \
\
		for ( i = 0; i < min_m_n; ++i ) \
		{ \
			bli_zcopy1es( *(a), \
			              *(y_off_ri + i*rs_y + i*cs_y), \
			              *(y_off_ir + i*rs_y + i*cs_y) ); \
		} \
	} \
	else /* if ( bli_is_1r_packed( schema ) ) */ \
	{ \
		inc_t rs_y2 = rs_y; \
		inc_t cs_y2 = cs_y; \
\
		/* Scale the non-unit stride by two for the 1r loop, which steps
		   in units of real (not complex) values. */ \
		if         ( rs_y2 == 1 )    { cs_y2 *= 2; } \
		else /* if ( cs_y2 == 1 ) */ { rs_y2 *= 2; } \
\
		double*   restrict y_cast  = ( double* )y; \
		double*   restrict y_off_r = y_cast + (offm  )*rs_y2 \
		                                    + (offn  )*cs_y2; \
		double*   restrict y_off_i = y_cast + (offm  )*rs_y2 \
		                                    + (offn  )*cs_y2 + ld_y; \
\
		for ( i = 0; i < min_m_n; ++i ) \
		{ \
			bli_zcopy1rs( *(a), \
			              *(y_off_r + i*rs_y2 + i*cs_y2), \
			              *(y_off_i + i*rs_y2 + i*cs_y2) ); \
		} \
	} \
}

#endif
// end bli_set1ms_mxn_diag.h
#line 247 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_set1ms_mxn_uplo.h
#line 1 "./frame/include/level0/1m//bli_set1ms_mxn_uplo.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SET1MS_MXN_UPLO_H
#define BLIS_SET1MS_MXN_UPLO_H

// set1ms_mxn_uplo

#define bli_cset1ms_mxn_uplo( schema, diagoff, uplo, m, n, a, y, rs_y, cs_y, ld_y ) \
{ \
	doff_t diagoff_abs = bli_abs( diagoff ); \
	inc_t  offdiag_inc; \
	dim_t  i, j; \
\
	/* Handle 1e and 1r separately. */ \
	if ( bli_is_1e_packed( schema ) ) \
	{ \
		/* Set the off-diagonal increment. */ \
		if         ( diagoff > 0 )    offdiag_inc = cs_y; \
		else /* if ( diagoff < 0 ) */ offdiag_inc = rs_y; \
\
		scomplex* restrict y0   = y + (diagoff_abs  )*offdiag_inc; \
		scomplex* restrict y_ri = y0; \
		scomplex* restrict y_ir = y0 + ld_y/2; \
\
		if ( bli_is_lower( uplo ) ) \
		{ \
			for ( j = 0; j < n; ++j ) \
			for ( i = j; i < m; ++i ) \
			{ \
				bli_ccopy1es( *(a), \
				              *(y_ri + i*rs_y + j*cs_y), \
				              *(y_ir + i*rs_y + j*cs_y) ); \
			} \
		} \
		else /* if ( bli_is_upper( uplo ) ) */ \
		{ \
			for ( j = 0; j < n; ++j ) \
			for ( i = 0; i < j + 1; ++i ) \
			{ \
				bli_ccopy1es( *(a), \
				              *(y_ri + i*rs_y + j*cs_y), \
				              *(y_ir + i*rs_y + j*cs_y) ); \
			} \
		} \
	} \
	else /* if ( bli_is_1r_packed( schema ) ) */ \
	{ \
		inc_t rs_y2 = rs_y; \
		inc_t cs_y2 = cs_y; \
\
		/* Scale the non-unit stride by two for the 1r loop, which steps
		   in units of real (not complex) values. */ \
		if         ( rs_y2 == 1 )    { cs_y2 *= 2; } \
		else /* if ( cs_y2 == 1 ) */ { rs_y2 *= 2; } \
\
		/* Set the off-diagonal increment. */ \
		if         ( diagoff > 0 )    offdiag_inc = cs_y2; \
		else /* if ( diagoff < 0 ) */ offdiag_inc = rs_y2; \
\
		float*    restrict y0  = ( float* )y + (diagoff_abs  )*offdiag_inc; \
		float*    restrict y_r = y0; \
		float*    restrict y_i = y0 + ld_y; \
\
		if ( bli_is_lower( uplo ) ) \
		{ \
			for ( j = 0; j < n; ++j ) \
			for ( i = j; i < m; ++i ) \
			{ \
				bli_ccopy1rs( *(a), \
				              *(y_r + i*rs_y2 + j*cs_y2), \
				              *(y_i + i*rs_y2 + j*cs_y2) ); \
			} \
		} \
		else /* if ( bli_is_upper( uplo ) ) */ \
		{ \
			for ( j = 0; j < n; ++j ) \
			for ( i = 0; i < j + 1; ++i ) \
			{ \
				bli_ccopy1rs( *(a), \
				              *(y_r + i*rs_y2 + j*cs_y2), \
				              *(y_i + i*rs_y2 + j*cs_y2) ); \
			} \
		} \
	} \
}

#define bli_zset1ms_mxn_uplo( schema, diagoff, uplo, m, n, a, y, rs_y, cs_y, ld_y ) \
{ \
	doff_t diagoff_abs = bli_abs( diagoff ); \
	inc_t  offdiag_inc; \
	dim_t  i, j; \
\
	/* Handle 1e and 1r separately. */ \
	if ( bli_is_1e_packed( schema ) ) \
	{ \
		/* Set the off-diagonal increment. */ \
		if         ( diagoff > 0 )    offdiag_inc = cs_y; \
		else /* if ( diagoff < 0 ) */ offdiag_inc = rs_y; \
\
		dcomplex* restrict y0   = y + (diagoff_abs  )*offdiag_inc; \
		dcomplex* restrict y_ri = y0; \
		dcomplex* restrict y_ir = y0 + ld_y/2; \
\
		if ( bli_is_lower( uplo ) ) \
		{ \
			for ( j = 0; j < n; ++j ) \
			for ( i = j; i < m; ++i ) \
			{ \
				bli_zcopy1es( *(a), \
				              *(y_ri + i*rs_y + j*cs_y), \
				              *(y_ir + i*rs_y + j*cs_y) ); \
			} \
		} \
		else /* if ( bli_is_upper( uplo ) ) */ \
		{ \
			for ( j = 0; j < n; ++j ) \
			for ( i = 0; i < j + 1; ++i ) \
			{ \
				bli_zcopy1es( *(a), \
				              *(y_ri + i*rs_y + j*cs_y), \
				              *(y_ir + i*rs_y + j*cs_y) ); \
			} \
		} \
	} \
	else /* if ( bli_is_1r_packed( schema ) ) */ \
	{ \
		inc_t rs_y2 = rs_y; \
		inc_t cs_y2 = cs_y; \
\
		/* Scale the non-unit stride by two for the 1r loop, which steps
		   in units of real (not complex) values. */ \
		if         ( rs_y2 == 1 )    { cs_y2 *= 2; } \
		else /* if ( cs_y2 == 1 ) */ { rs_y2 *= 2; } \
\
		/* Set the off-diagonal increment. */ \
		if         ( diagoff > 0 )    offdiag_inc = cs_y2; \
		else /* if ( diagoff < 0 ) */ offdiag_inc = rs_y2; \
\
		double*   restrict y0  = ( double* )y + (diagoff_abs  )*offdiag_inc; \
		double*   restrict y_r = y0; \
		double*   restrict y_i = y0 + ld_y; \
\
		if ( bli_is_lower( uplo ) ) \
		{ \
			for ( j = 0; j < n; ++j ) \
			for ( i = j; i < m; ++i ) \
			{ \
				bli_zcopy1rs( *(a), \
				              *(y_r + i*rs_y2 + j*cs_y2), \
				              *(y_i + i*rs_y2 + j*cs_y2) ); \
			} \
		} \
		else /* if ( bli_is_upper( uplo ) ) */ \
		{ \
			for ( j = 0; j < n; ++j ) \
			for ( i = 0; i < j + 1; ++i ) \
			{ \
				bli_zcopy1rs( *(a), \
				              *(y_r + i*rs_y2 + j*cs_y2), \
				              *(y_i + i*rs_y2 + j*cs_y2) ); \
			} \
		} \
	} \
}

#endif
// end bli_set1ms_mxn_uplo.h
#line 248 "./frame/include//bli_scalar_macro_defs.h"

// begin bli_seti01ms_mxn_diag.h
#line 1 "./frame/include/level0/1m//bli_seti01ms_mxn_diag.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_SETI01MS_MXN_DIAG_H
#define BLIS_SETI01MS_MXN_DIAG_H

// seti01ms_mxn_diag

#define bli_cseti01ms_mxn_diag( schema, m, n, y, rs_y, cs_y, ld_y ) \
{ \
	dim_t min_m_n = bli_min( m, n ); \
	dim_t i; \
\
	/* Handle 1e and 1r separately. */ \
	if ( bli_is_1e_packed( schema ) ) \
	{ \
		scomplex* restrict y_off_ri = y; \
		scomplex* restrict y_off_ir = y + ld_y/2; \
\
		for ( i = 0; i < min_m_n; ++i ) \
		{ \
			bli_cseti0s( *(y_off_ri + i*rs_y + i*cs_y) ); \
			bli_csetr0s( *(y_off_ir + i*rs_y + i*cs_y) ); \
		} \
	} \
	else /* if ( bli_is_1r_packed( schema ) ) */ \
	{ \
		inc_t rs_y2 = rs_y; \
		inc_t cs_y2 = cs_y; \
\
		/* Scale the non-unit stride by two for the 1r loop, which steps
		   in units of real (not complex) values. */ \
		if         ( rs_y2 == 1 )    { cs_y2 *= 2; } \
		else /* if ( cs_y2 == 1 ) */ { rs_y2 *= 2; } \
\
		float*    restrict y_cast  = ( float* )y; \
		float*    restrict y_off_i = y_cast + ld_y; \
\
		for ( i = 0; i < min_m_n; ++i ) \
		{ \
			bli_sset0s( *(y_off_i + i*rs_y2 + i*cs_y2) ); \
		} \
	} \
}

#define bli_zseti01ms_mxn_diag( schema, m, n, y, rs_y, cs_y, ld_y ) \
{ \
	dim_t min_m_n = bli_min( m, n ); \
	dim_t i; \
\
	/* Handle 1e and 1r separately. */ \
	if ( bli_is_1e_packed( schema ) ) \
	{ \
		dcomplex* restrict y_off_ri = y; \
		dcomplex* restrict y_off_ir = y + ld_y/2; \
\
		for ( i = 0; i < min_m_n; ++i ) \
		{ \
			bli_zseti0s( *(y_off_ri + i*rs_y + i*cs_y) ); \
			bli_zsetr0s( *(y_off_ir + i*rs_y + i*cs_y) ); \
		} \
	} \
	else /* if ( bli_is_1r_packed( schema ) ) */ \
	{ \
		inc_t rs_y2 = rs_y; \
		inc_t cs_y2 = cs_y; \
\
		/* Scale the non-unit stride by two for the 1r loop, which steps
		   in units of real (not complex) values. */ \
		if         ( rs_y2 == 1 )    { cs_y2 *= 2; } \
		else /* if ( cs_y2 == 1 ) */ { rs_y2 *= 2; } \
\
		double*   restrict y_cast  = ( double* )y; \
		double*   restrict y_off_i = y_cast + ld_y; \
\
		for ( i = 0; i < min_m_n; ++i ) \
		{ \
			bli_dset0s( *(y_off_i + i*rs_y2 + i*cs_y2) ); \
		} \
	} \
}

#endif
// end bli_seti01ms_mxn_diag.h
#line 249 "./frame/include//bli_scalar_macro_defs.h"



#endif
// end bli_scalar_macro_defs.h
#line 106 "./frame/include//bli_macro_defs.h"

// begin bli_error_macro_defs.h
#line 1 "./frame/include//bli_error_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_ERROR_MACRO_DEFS_H
#define BLIS_ERROR_MACRO_DEFS_H

// Used to insert filenames and line numbers into error-checking code.
#define bli_check_error_code( code ) \
        bli_check_error_code_helper( code, __FILE__, __LINE__ )


#endif

// end bli_error_macro_defs.h
#line 107 "./frame/include//bli_macro_defs.h"

// begin bli_blas_macro_defs.h
#line 1 "./frame/include//bli_blas_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_BLAS_MACRO_DEFS_H
#define BLIS_BLAS_MACRO_DEFS_H

// -- Various Fortran compatibility macros --

// Macro to treat negative dimensions as zero.

#define bli_convert_blas_dim1( n_blas, n_blis )\
{ \
	if ( n_blas < 0 ) n_blis = ( dim_t )0; \
	else              n_blis = ( dim_t )n_blas; \
}

// Macro to flip signs of increments if input increments are negative.

#define bli_convert_blas_incv( n, x_blas, incx_blas, \
                                  x_blis, incx_blis ) \
{ \
	if ( incx_blas < 0 ) \
	{ \
		/* The semantics of negative stride in BLAS are that the vector
		   operand be traversed in reverse order. (Another way to think
		   of this is that negative strides effectively reverse the order
		   of the vector, but without any explicit data movements.) This
		   is also how BLIS interprets negative strides. The differences
		   is that with BLAS, the caller *always* passes in the 0th (i.e.,
		   top-most or left-most) element of the vector, even when the
		   stride is negative. By contrast, in BLIS, negative strides are
		   used *relative* to the vector address as it is given. Thus, in
		   BLIS, if this backwards traversal is desired, the caller *must*
		   pass in the address to the (n-1)th (i.e., the bottom-most or
		   right-most) element along with a negative stride. */ \
		x_blis    = (x_blas) + (n-1)*(-incx_blas); \
		incx_blis = ( inc_t )(incx_blas); \
	} \
	else \
	{ \
		x_blis    = (x_blas); \
		incx_blis = ( inc_t )(incx_blas); \
	} \
}



#endif

// end bli_blas_macro_defs.h
#line 108 "./frame/include//bli_macro_defs.h"

// begin bli_builtin_macro_defs.h
#line 1 "./frame/include//bli_builtin_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2019, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_BUILTIN_MACRO_DEFS_H
#define BLIS_BUILTIN_MACRO_DEFS_H

#if   defined(__ICC) || defined(__INTEL_COMPILER)

  // icc

  #define bli_prefetch( addr, rw, loc )

#elif defined(__clang__)

  // clang

  #define bli_prefetch( addr, rw, loc )

#elif defined(__GNUC__)

  // gcc

  #define bli_prefetch( addr, rw, loc ) __builtin_prefetch( addr, rw, loc );

#endif


#endif
// end bli_builtin_macro_defs.h
#line 109 "./frame/include//bli_macro_defs.h"


// begin bli_oapi_macro_defs.h
#line 1 "./frame/include//bli_oapi_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// Define the suffix to add to object API function names that include
// additional "expert" parameters.
#define BLIS_OAPI_EX_SUF _ex

// end bli_oapi_macro_defs.h
#line 111 "./frame/include//bli_macro_defs.h"

// begin bli_tapi_macro_defs.h
#line 1 "./frame/include//bli_tapi_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

// Define the suffix to add to typed API function names that include
// additional "expert" parameters.
#define BLIS_TAPI_EX_SUF _ex

// end bli_tapi_macro_defs.h
#line 112 "./frame/include//bli_macro_defs.h"



#endif
// end bli_macro_defs.h
#line 75 "./frame/include/blis.h"



// -- pragma definitions --

// begin bli_pragma_macro_defs.h
#line 1 "./frame/include//bli_pragma_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2019, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

/*
  NOTE: The following code is based on [1].

  [1] https://github.com/jeffhammond/nwchem-tce-triples-kernels/blob/master/src/pragma_vendor.h
*/

#ifndef BLIS_PRAGMA_MACRO_DEFS_H
#define BLIS_PRAGMA_MACRO_DEFS_H

// Generally speaking, if BLIS_ENABLE_PRAGMA_OMP_SIMD is set, then we define
// all instances of PRAGMA_SIMD as _Pragma("omp simd").

#ifdef BLIS_ENABLE_PRAGMA_OMP_SIMD
  #define PRAGMA_OMP_SIMD _Pragma("omp simd")
#else
  #define PRAGMA_OMP_SIMD
#endif

// Require ISO C99 or later for SIMD-related pragmas.
#if (( __STDC_VERSION__ >= 199901L ))

  #define GEN_PRAGMA(x) _Pragma(#x)

  #if   defined(__ICC) || defined(__INTEL_COMPILER)

    // Intel icc.
    //#define PRAGMA_SIMD  GEN_PRAGMA(simd)
    #define PRAGMA_SIMD  PRAGMA_OMP_SIMD

  #elif defined(__clang__)

    // clang/llvm.
    #define PRAGMA_SIMD  PRAGMA_OMP_SIMD

  #elif defined(__GNUC__)

    // GNU gcc.
    #define PRAGMA_SIMD  PRAGMA_OMP_SIMD

  #else

    // Unknown compiler.
    #define PRAGMA_SIMD

  #endif
#endif

#endif
// end bli_pragma_macro_defs.h
#line 80 "./frame/include/blis.h"



// -- BLIS architecture/kernel definitions --

// begin bli_pre_ker_params.h
#line 1 "./frame/include//bli_pre_ker_params.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2023, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_PRE_KER_PARAMS_H
#define BLIS_PRE_KER_PARAMS_H

// These macros are used in bli_*_ker_prot.h and bli_*_ker_ft.h to make it
// easy to update them in the future, if needed.

#define BLIS_AUXINFO_PARAM        auxinfo_t* data
#define BLIS_CNTX_PARAM     const cntx_t*    cntx


#endif
// end bli_pre_ker_params.h
#line 85 "./frame/include/blis.h"

// begin bli_l1v_ker_params.h
#line 1 "./frame/1//bli_l1v_ker_params.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_L1V_KER_PARAMS_H
#define BLIS_L1V_KER_PARAMS_H


#define addv_params \
\
             conj_t  conjx, \
             dim_t   n, \
       const void*   x, inc_t incx, \
             void*   y, inc_t incy

#define amaxv_params \
\
             dim_t   n, \
       const void*   x, inc_t incx, \
             dim_t*  index

#define axpbyv_params \
\
             conj_t  conjx, \
             dim_t   n, \
       const void*   alpha, \
       const void*   x, inc_t incx, \
       const void*   beta, \
             void*   y, inc_t incy

#define axpyv_params \
\
             conj_t  conjx, \
             dim_t   n, \
       const void*   alpha, \
       const void*   x, inc_t incx, \
             void*   y, inc_t incy

#define copyv_params \
\
             conj_t  conjx, \
             dim_t   n, \
       const void*   x, inc_t incx, \
             void*   y, inc_t incy

#define dotv_params \
\
             conj_t  conjx, \
             conj_t  conjy, \
             dim_t   n, \
       const void*   x, inc_t incx, \
       const void*   y, inc_t incy, \
             void*   rho

#define dotxv_params \
\
             conj_t  conjx, \
             conj_t  conjy, \
             dim_t   n, \
       const void*   alpha, \
       const void*   x, inc_t incx, \
       const void*   y, inc_t incy, \
       const void*   beta, \
             void*   rho

#define invertv_params \
\
             dim_t   n, \
             void*   x, inc_t incx

#define invscalv_params \
\
             conj_t  conjalpha, \
             dim_t   n, \
       const void*   alpha, \
             void*   x, inc_t incx

#define scalv_params \
\
             conj_t  conjalpha, \
             dim_t   n, \
       const void*   alpha, \
             void*   x, inc_t incx

#define scal2v_params \
\
             conj_t  conjx, \
             dim_t   n, \
       const void*   alpha, \
       const void*   x, inc_t incx, \
             void*   y, inc_t incy

#define setv_params \
\
             conj_t  conjalpha, \
             dim_t   n, \
       const void*   alpha, \
             void*   x, inc_t incx

#define subv_params \
\
             conj_t  conjx, \
             dim_t   n, \
       const void*   x, inc_t incx, \
             void*   y, inc_t incy

#define swapv_params \
\
             dim_t   n, \
             void*   x, inc_t incx, \
             void*   y, inc_t incy

#define xpbyv_params \
\
             conj_t  conjx, \
             dim_t   n, \
       const void*   x, inc_t incx, \
       const void*   beta, \
             void*   y, inc_t incy

#endif

// end bli_l1v_ker_params.h
#line 86 "./frame/include/blis.h"

// begin bli_l1f_ker_params.h
#line 1 "./frame/1f//bli_l1f_ker_params.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_L1F_KER_PARAMS_H
#define BLIS_L1F_KER_PARAMS_H


#define axpy2v_params \
\
             conj_t  conjx, \
             conj_t  conjy, \
             dim_t   n, \
       const void*   alphax, \
       const void*   alphay, \
       const void*   x, inc_t incx, \
       const void*   y, inc_t incy, \
             void*   z, inc_t incz

#define axpyf_params \
\
             conj_t  conja, \
             conj_t  conjx, \
             dim_t   m, \
             dim_t   b_n, \
       const void*   alpha, \
       const void*   a, inc_t inca, inc_t lda, \
       const void*   x, inc_t incx, \
             void*   y, inc_t incy

#define dotaxpyv_params \
\
             conj_t  conjxt, \
             conj_t  conjx, \
             conj_t  conjy, \
             dim_t   m, \
       const void*   alpha, \
       const void*   x, inc_t incx, \
       const void*   y, inc_t incy, \
             void*   rho, \
             void*   z, inc_t incz

#define dotxaxpyf_params \
\
             conj_t  conjat, \
             conj_t  conja, \
             conj_t  conjw, \
             conj_t  conjx, \
             dim_t   m, \
             dim_t   b_n, \
       const void*   alpha, \
       const void*   a, inc_t inca, inc_t lda, \
       const void*   w, inc_t incw, \
       const void*   x, inc_t incx, \
       const void*   beta, \
             void*   y, inc_t incy, \
             void*   z, inc_t incz

#define dotxf_params \
\
             conj_t  conjat, \
             conj_t  conjx, \
             dim_t   m, \
             dim_t   b_n, \
       const void*   alpha, \
       const void*   a, inc_t inca, inc_t lda, \
       const void*   x, inc_t incx, \
       const void*   beta, \
             void*   y, inc_t incy

#endif

// end bli_l1f_ker_params.h
#line 87 "./frame/include/blis.h"

// begin bli_l1m_ker_params.h
#line 1 "./frame/1m//bli_l1m_ker_params.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_L1M_KER_PARAMS_H
#define BLIS_L1M_KER_PARAMS_H


//
// -- Level-1m kernel function types -------------------------------------------
//

// packm

// NOTE: This is the function type for the structure-aware "kernel".

#define packm_params \
\
             struc_t strucc, \
             diag_t  diagc, \
             uplo_t  uploc, \
             conj_t  conjc, \
             pack_t  schema, \
             bool    invdiag, \
             dim_t   panel_dim, \
             dim_t   panel_len, \
             dim_t   panel_dim_max, \
             dim_t   panel_len_max, \
             dim_t   panel_dim_off, \
             dim_t   panel_len_off, \
       const void*   kappa, \
       const void*   c, inc_t incc, inc_t ldc, \
             void*   p,             inc_t ldp, \
                        inc_t is_p, \
       const void*   params  \


// packm_cxk (packm microkernel)

#define packm_cxk_params \
\
             conj_t  conja, \
             pack_t  schema, \
             dim_t   cdim, \
             dim_t   n, \
             dim_t   n_max, \
       const void*   kappa, \
       const void*   a, inc_t inca, inc_t lda, \
             void*   p,             inc_t ldp  \


// unpackm_cxk kernel

#define unpackm_cxk_params \
\
             conj_t  conja, \
             pack_t  schema, \
             dim_t   cdim, \
             dim_t   n, \
       const void*   kappa, \
       const void*   p,             inc_t ldp, \
             void*   a, inc_t inca, inc_t lda  \


// packm_cxc_diag kernel

#define packm_cxc_diag_params \
\
             struc_t struca, \
             diag_t  diaga, \
             uplo_t  uploa, \
             conj_t  conja, \
             pack_t  schema, \
             bool    invdiag, \
             dim_t   cdim, \
             dim_t   n_max, \
       const void*   kappa, \
       const void*   a, inc_t inca, inc_t lda, \
             void*   p,             inc_t ldp  \


#endif

// end bli_l1m_ker_params.h
#line 88 "./frame/include/blis.h"

// begin bli_l3_ukr_params.h
#line 1 "./frame/3//bli_l3_ukr_params.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_L3_UKR_PARAMS_H
#define BLIS_L3_UKR_PARAMS_H


#define gemm_params \
\
             dim_t  m, \
             dim_t  n, \
             dim_t  k, \
       const void*  alpha, \
       const void*  a, \
       const void*  b, \
       const void*  beta, \
             void*  c, inc_t rs_c, inc_t cs_c

#define gemmtrsm_params \
\
             dim_t  m, \
             dim_t  n, \
             dim_t  k, \
       const void*  alpha, \
       const void*  a1x, \
       const void*  a11, \
       const void*  bx1, \
             void*  b11, \
             void*  c11, inc_t rs_c, inc_t cs_c

#define trsm_params \
\
       const void*  a, \
             void*  b, \
             void*  c, inc_t rs_c, inc_t cs_c


#endif

// end bli_l3_ukr_params.h
#line 89 "./frame/include/blis.h"

// begin bli_l3_sup_ker_params.h
#line 1 "./frame/3//bli_l3_sup_ker_params.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_L3_SUP_KER_PARAMS_H
#define BLIS_L3_SUP_KER_PARAMS_H


#define gemmsup_params \
\
             conj_t conja, \
             conj_t conjb, \
             dim_t  m, \
             dim_t  n, \
             dim_t  k, \
       const void*  alpha, \
       const void*  a, inc_t rs_a, inc_t cs_a, \
       const void*  b, inc_t rs_b, inc_t cs_b, \
       const void*  beta, \
             void*  c, inc_t rs_c, inc_t cs_c


#endif

// end bli_l3_sup_ker_params.h
#line 90 "./frame/include/blis.h"


// begin bli_l1v_ker_prot.h
#line 1 "./frame/1//bli_l1v_ker_prot.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_L1V_KER_PROT_H
#define BLIS_L1V_KER_PROT_H

//
// Define template prototypes for level-1v kernels.
//

#undef  L1VTPROT
#define L1VTPROT( ctype, ch, funcname, opname ) \
\
void PASTEMAC(ch,funcname) \
     ( \
       PASTECH(opname,_params), \
       BLIS_CNTX_PARAM  \
     );

#define ADDV_KER_PROT(     ctype, ch, fn )  L1VTPROT( ctype, ch, fn, addv );
#define AMAXV_KER_PROT(    ctype, ch, fn )  L1VTPROT( ctype, ch, fn, amaxv );
#define AXPBYV_KER_PROT(   ctype, ch, fn )  L1VTPROT( ctype, ch, fn, axpbyv );
#define AXPYV_KER_PROT(    ctype, ch, fn )  L1VTPROT( ctype, ch, fn, axpyv );
#define COPYV_KER_PROT(    ctype, ch, fn )  L1VTPROT( ctype, ch, fn, copyv );
#define DOTV_KER_PROT(     ctype, ch, fn )  L1VTPROT( ctype, ch, fn, dotv );
#define DOTXV_KER_PROT(    ctype, ch, fn )  L1VTPROT( ctype, ch, fn, dotxv );
#define INVERTV_KER_PROT(  ctype, ch, fn )  L1VTPROT( ctype, ch, fn, invertv );
#define INVSCALV_KER_PROT( ctype, ch, fn )  L1VTPROT( ctype, ch, fn, invscalv );
#define SCALV_KER_PROT(    ctype, ch, fn )  L1VTPROT( ctype, ch, fn, scalv );
#define SCAL2V_KER_PROT(   ctype, ch, fn )  L1VTPROT( ctype, ch, fn, scal2v );
#define SETV_KER_PROT(     ctype, ch, fn )  L1VTPROT( ctype, ch, fn, setv );
#define SUBV_KER_PROT(     ctype, ch, fn )  L1VTPROT( ctype, ch, fn, subv );
#define SWAPV_KER_PROT(    ctype, ch, fn )  L1VTPROT( ctype, ch, fn, swapv );
#define XPBYV_KER_PROT(    ctype, ch, fn )  L1VTPROT( ctype, ch, fn, xpbyv );


#endif

// end bli_l1v_ker_prot.h
#line 92 "./frame/include/blis.h"

// begin bli_l1f_ker_prot.h
#line 1 "./frame/1f//bli_l1f_ker_prot.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_L1F_KER_PROT_H
#define BLIS_L1F_KER_PROT_H

//
// Define template prototypes for level-1f kernels.
//

#undef  L1FTPROT
#define L1FTPROT( ctype, ch, funcname, opname ) \
\
void PASTEMAC(ch,funcname) \
     ( \
       PASTECH(opname,_params), \
       BLIS_CNTX_PARAM  \
     );

#define AXPY2V_KER_PROT(    ctype, ch, fn )  L1FTPROT( ctype, ch, fn, axpy2v );
#define AXPYF_KER_PROT(     ctype, ch, fn )  L1FTPROT( ctype, ch, fn, axpyf );
#define DOTAXPYV_KER_PROT(  ctype, ch, fn )  L1FTPROT( ctype, ch, fn, dotaxpyv );
#define DOTXAXPYF_KER_PROT( ctype, ch, fn )  L1FTPROT( ctype, ch, fn, dotxaxpyf );
#define DOTXF_KER_PROT(     ctype, ch, fn )  L1FTPROT( ctype, ch, fn, dotxf );


#endif

// end bli_l1f_ker_prot.h
#line 93 "./frame/include/blis.h"

// begin bli_l1m_ker_prot.h
#line 1 "./frame/1m//bli_l1m_ker_prot.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_L1M_KER_PROT_H
#define BLIS_L1M_KER_PROT_H

//
// Define template prototypes for level-1m kernels.
//

#undef  L1MTPROT
#define L1MTPROT( ctype, ch, funcname, opname ) \
\
void PASTEMAC(ch,funcname) \
     ( \
       PASTECH(opname,_params), \
       BLIS_CNTX_PARAM  \
     );

#define PACKM_KER_PROT(      ctype, ch, fn )  L1MTPROT( ctype, ch, fn, packm_cxk );
#define UNPACKM_KER_PROT(    ctype, ch, fn )  L1MTPROT( ctype, ch, fn, unpackm_cxk );
#define PACKM_DIAG_KER_PROT( ctype, ch, fn )  L1MTPROT( ctype, ch, fn, packm_cxc_diag );


#endif

// end bli_l1m_ker_prot.h
#line 94 "./frame/include/blis.h"

// begin bli_l3_ukr_prot.h
#line 1 "./frame/3//bli_l3_ukr_prot.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_L3_UKR_PROT_H
#define BLIS_L3_UKR_PROT_H

//
// Define template prototypes for level-3 micro-kernels.
//

#undef  L3TPROT
#define L3TPROT( ctype, ch, funcname, opname ) \
\
void PASTEMAC(ch,funcname) \
     ( \
       PASTECH(opname,_params), \
       BLIS_AUXINFO_PARAM, \
       BLIS_CNTX_PARAM  \
     );

#define GEMM_UKR_PROT(     ctype, ch, fn )  L3TPROT( ctype, ch, fn, gemm );
#define GEMMTRSM_UKR_PROT( ctype, ch, fn )  L3TPROT( ctype, ch, fn, gemmtrsm );
#define TRSM_UKR_PROT(     ctype, ch, fn )  L3TPROT( ctype, ch, fn, trsm );


#endif

// end bli_l3_ukr_prot.h
#line 95 "./frame/include/blis.h"

// begin bli_l3_sup_ker_prot.h
#line 1 "./frame/3//bli_l3_sup_ker_prot.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_L3_SUP_KER_PROT_H
#define BLIS_L3_SUP_KER_PROT_H

//
// Define template prototypes for level-3 kernels on small/unpacked matrices.
//

#undef  SUPTPROT
#define SUPTPROT( ctype, ch, funcname, opname ) \
\
void PASTEMAC(ch,funcname) \
     ( \
       PASTECH(opname,_params), \
       BLIS_AUXINFO_PARAM, \
       BLIS_CNTX_PARAM  \
     );

#define GEMMSUP_KER_PROT( ctype, ch, fn )  SUPTPROT( ctype, ch, fn, gemmsup );


#endif

// end bli_l3_sup_ker_prot.h
#line 96 "./frame/include/blis.h"


// begin bli_arch_config_pre.h
#line 1 "./frame/include//bli_arch_config_pre.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_ARCH_CONFIG_PRE_H
#define BLIS_ARCH_CONFIG_PRE_H


// -- Naming-related kernel definitions ----------------------------------------

// The default suffix appended to reference kernels.
#define BLIS_REF_SUFFIX  _ref

// A suffix used for labeling certain induced method aware functions.
#define BLIS_IND_SUFFIX  _ind

// Add an underscore to the BLIS kernel set string, if it was defined.
#ifdef  BLIS_CNAME
#define BLIS_CNAME_INFIX  PASTECH(_,BLIS_CNAME)
#endif

// Combine the CNAME and _ref for convenience to the code that defines
// reference kernels.
//#define BLIS_CNAME_REF_SUFFIX  PASTECH2(_,BLIS_CNAME,BLIS_REF_SUFFIX)

// -- Prototype-generating macro definitions -----------------------------------

// Prototype-generating macro for bli_cntx_init_<arch>*() functions.
#define CNTX_INIT_PROTS( archname ) \
\
void PASTEMAC(cntx_init_,archname) \
     ( \
       cntx_t* cntx \
     ); \
void PASTEMAC2(cntx_init_,archname,BLIS_REF_SUFFIX) \
     ( \
       cntx_t* cntx \
     ); \
void PASTEMAC2(cntx_init_,archname,BLIS_IND_SUFFIX) \
     ( \
       ind_t   method, \
       cntx_t* cntx \
     );


#endif

// end bli_arch_config_pre.h
#line 98 "./frame/include/blis.h"

// begin bli_arch_config.h
#line 1 "./frame/include//bli_arch_config.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2016, Hewlett Packard Enterprise Development LP
   Copyright (C) 2019 - 2020, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_ARCH_CONFIG_H
#define BLIS_ARCH_CONFIG_H

//
// -- Context initialization prototypes ----------------------------------------
//

// -- Intel64 architectures --

#ifdef BLIS_CONFIG_SKX
CNTX_INIT_PROTS( skx )
#endif
#ifdef BLIS_CONFIG_KNL
CNTX_INIT_PROTS( knl )
#endif
#ifdef BLIS_CONFIG_KNC
CNTX_INIT_PROTS( knc )
#endif
#ifdef BLIS_CONFIG_HASWELL
CNTX_INIT_PROTS( haswell )
#endif
#ifdef BLIS_CONFIG_SANDYBRIDGE
CNTX_INIT_PROTS( sandybridge )
#endif
#ifdef BLIS_CONFIG_PENRYN
CNTX_INIT_PROTS( penryn )
#endif

// -- AMD64 architectures --

#ifdef BLIS_CONFIG_ZEN3
CNTX_INIT_PROTS( zen3 )
#endif
#ifdef BLIS_CONFIG_ZEN2
CNTX_INIT_PROTS( zen2 )
#endif
#ifdef BLIS_CONFIG_ZEN
CNTX_INIT_PROTS( zen )
#endif
#ifdef BLIS_CONFIG_EXCAVATOR
CNTX_INIT_PROTS( excavator )
#endif
#ifdef BLIS_CONFIG_STEAMROLLER
CNTX_INIT_PROTS( steamroller )
#endif
#ifdef BLIS_CONFIG_PILEDRIVER
CNTX_INIT_PROTS( piledriver )
#endif
#ifdef BLIS_CONFIG_BULLDOZER
CNTX_INIT_PROTS( bulldozer )
#endif

// -- ARM architectures --

// ARM-SVE
#ifdef BLIS_CONFIG_ARMSVE
CNTX_INIT_PROTS( armsve )
#endif
#ifdef BLIS_CONFIG_A64FX
CNTX_INIT_PROTS( a64fx )
#endif
// ARM-NEON (4x128)
#ifdef BLIS_CONFIG_ALTRAMAX
CNTX_INIT_PROTS( altramax )
#endif
#ifdef BLIS_CONFIG_ALTRA
CNTX_INIT_PROTS( altra )
#endif
#ifdef BLIS_CONFIG_FIRESTORM
CNTX_INIT_PROTS( firestorm )
#endif
// ARM-NEON (2x128)
#ifdef BLIS_CONFIG_THUNDERX2
CNTX_INIT_PROTS( thunderx2 )
#endif
#ifdef BLIS_CONFIG_CORTEXA57
CNTX_INIT_PROTS( cortexa57 )
#endif
#ifdef BLIS_CONFIG_CORTEXA53
CNTX_INIT_PROTS( cortexa53 )
#endif
// ARM 32-bit (vintage)
#ifdef BLIS_CONFIG_CORTEXA15
CNTX_INIT_PROTS( cortexa15 )
#endif
#ifdef BLIS_CONFIG_CORTEXA9
CNTX_INIT_PROTS( cortexa9 )
#endif

// -- IBM Power --

#ifdef BLIS_CONFIG_POWER10
CNTX_INIT_PROTS( power10 )
#endif
#ifdef BLIS_CONFIG_POWER9
CNTX_INIT_PROTS( power9 )
#endif
#ifdef BLIS_CONFIG_POWER7
CNTX_INIT_PROTS( power7 )
#endif

// -- IBM BG/Q --

#ifdef BLIS_CONFIG_BGQ
CNTX_INIT_PROTS( bgq )
#endif

// -- RISC-V --

#ifdef BLIS_CONFIG_RV32I
CNTX_INIT_PROTS( rv32i )
#endif
#ifdef BLIS_CONFIG_RV64I
CNTX_INIT_PROTS( rv64i )
#endif
#ifdef BLIS_CONFIG_RV32IV
CNTX_INIT_PROTS( rv32iv )
#endif
#ifdef BLIS_CONFIG_RV64IV
CNTX_INIT_PROTS( rv64iv )
#endif

// -- SiFive architectures --

#ifdef BLIS_CONFIG_SIFIVE_X280
CNTX_INIT_PROTS( sifive_x280 )
#endif

// -- Generic --

#ifdef BLIS_CONFIG_GENERIC
CNTX_INIT_PROTS( generic )
#endif


//
// -- Architecture family-specific headers -------------------------------------
//

// -- x86_64 families --

#ifdef BLIS_FAMILY_INTEL64
#include "bli_family_intel64.h" // skipped
#endif
#ifdef BLIS_FAMILY_AMD64
#include "bli_family_amd64.h" // skipped
#endif
#ifdef BLIS_FAMILY_AMD64_LEGACY
#include "bli_family_amd64_legacy.h" // skipped
#endif
#ifdef BLIS_FAMILY_X86_64
#include "bli_family_x86_64.h" // skipped
#endif

// -- Intel64 architectures --

#ifdef BLIS_FAMILY_SKX
#include "bli_family_skx.h" // skipped
#endif
#ifdef BLIS_FAMILY_KNL
#include "bli_family_knl.h" // skipped
#endif
#ifdef BLIS_FAMILY_KNC
#include "bli_family_knc.h" // skipped
#endif
#ifdef BLIS_FAMILY_HASWELL
#include "bli_family_haswell.h" // skipped
#endif
#ifdef BLIS_FAMILY_SANDYBRIDGE
#include "bli_family_sandybridge.h" // skipped
#endif
#ifdef BLIS_FAMILY_PENRYN
#include "bli_family_penryn.h" // skipped
#endif

// -- AMD64 architectures --

#ifdef BLIS_FAMILY_ZEN3
#include "bli_family_zen3.h" // skipped
#endif
#ifdef BLIS_FAMILY_ZEN2
#include "bli_family_zen2.h" // skipped
#endif
#ifdef BLIS_FAMILY_ZEN
#include "bli_family_zen.h" // skipped
#endif
#ifdef BLIS_FAMILY_EXCAVATOR
#include "bli_family_excavator.h" // skipped
#endif
#ifdef BLIS_FAMILY_STEAMROLLER
#include "bli_family_steamroller.h" // skipped
#endif
#ifdef BLIS_FAMILY_PILEDRIVER
#include "bli_family_piledriver.h" // skipped
#endif
#ifdef BLIS_FAMILY_BULLDOZER
#include "bli_family_bulldozer.h" // skipped
#endif

// -- ARM families --
#ifdef BLIS_FAMILY_ARM64
#include "bli_family_arm64.h" // skipped
#endif
#ifdef BLIS_FAMILY_ARM32
#include "bli_family_arm32.h" // skipped
#endif

// -- ARM architectures --

// ARM-SVE
#ifdef BLIS_FAMILY_ARMSVE
#include "bli_family_armsve.h" // skipped
#endif
#ifdef BLIS_FAMILY_A64FX
#include "bli_family_a64fx.h" // skipped
#endif
// ARM-NEON (4x128)
#ifdef BLIS_FAMILY_ALTRAMAX
#include "bli_family_altramax.h" // skipped
#endif
#ifdef BLIS_FAMILY_ALTRA
#include "bli_family_altra.h" // skipped
#endif
#ifdef BLIS_FAMILY_FIRESTORM
#include "bli_family_firestorm.h" // skipped
#endif
// ARM-NEON (2x128)
#ifdef BLIS_FAMILY_THUNDERX2
#include "bli_family_thunderx2.h" // skipped
#endif
#ifdef BLIS_FAMILY_CORTEXA57
#include "bli_family_cortexa57.h" // skipped
#endif
#ifdef BLIS_FAMILY_CORTEXA53
#include "bli_family_cortexa53.h" // skipped
#endif
// ARM 32-bit (vintage)
#ifdef BLIS_FAMILY_CORTEXA15
#include "bli_family_cortexa15.h" // skipped
#endif
#ifdef BLIS_FAMILY_CORTEXA9
#include "bli_family_cortexa9.h" // skipped
#endif

// -- IBM Power families --
#ifdef BLIS_FAMILY_POWER
#include "bli_family_power.h" // skipped
#endif

// -- IBM Power architectures --

#ifdef BLIS_FAMILY_POWER10
#include "bli_family_power10.h" // skipped
#endif
#ifdef BLIS_FAMILY_POWER9
#include "bli_family_power9.h" // skipped
#endif
#ifdef BLIS_FAMILY_POWER7
#include "bli_family_power7.h" // skipped
#endif

// -- IBM BG/Q --

#ifdef BLIS_FAMILY_BGQ
#include "bli_family_bgq.h" // skipped
#endif

// -- SiFive families --

#ifdef BLIS_FAMILY_SIFIVE_X280
#include "bli_family_sifive_x280.h" // skipped
#endif

// -- Generic --

#ifdef BLIS_FAMILY_GENERIC
// begin bli_family_generic.h
#line 1 "./config/generic//bli_family_generic.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

//#ifndef BLIS_FAMILY_H
//#define BLIS_FAMILY_H




//#endif

// end bli_family_generic.h
#line 314 "./frame/include//bli_arch_config.h"

#endif


//
// -- kernel set prototypes ----------------------------------------------------
//

// -- Intel64 architectures --
#ifdef BLIS_KERNELS_SKX
#include "bli_kernels_skx.h" // skipped
#endif
#ifdef BLIS_KERNELS_KNL
#include "bli_kernels_knl.h" // skipped
#endif
#ifdef BLIS_KERNELS_KNC
#include "bli_kernels_knc.h" // skipped
#endif
#ifdef BLIS_KERNELS_HASWELL
#include "bli_kernels_haswell.h" // skipped
#endif
#ifdef BLIS_KERNELS_SANDYBRIDGE
#include "bli_kernels_sandybridge.h" // skipped
#endif
#ifdef BLIS_KERNELS_PENRYN
#include "bli_kernels_penryn.h" // skipped
#endif

// -- AMD64 architectures --

#ifdef BLIS_KERNELS_ZEN2
#include "bli_kernels_zen2.h" // skipped
#endif
#ifdef BLIS_KERNELS_ZEN
#include "bli_kernels_zen.h" // skipped
#endif
//#ifdef BLIS_KERNELS_EXCAVATOR
//#include "bli_kernels_excavator.h"
//#endif
//#ifdef BLIS_KERNELS_STEAMROLLER
//#include "bli_kernels_steamroller.h"
//#endif
#ifdef BLIS_KERNELS_PILEDRIVER
#include "bli_kernels_piledriver.h" // skipped
#endif
#ifdef BLIS_KERNELS_BULLDOZER
#include "bli_kernels_bulldozer.h" // skipped
#endif

// -- ARM architectures --

#ifdef BLIS_KERNELS_ARMSVE
#include "bli_kernels_armsve.h" // skipped
#endif
#ifdef BLIS_KERNELS_ARMV8A
#include "bli_kernels_armv8a.h" // skipped
#endif
#ifdef BLIS_KERNELS_ARMV7A
#include "bli_kernels_armv7a.h" // skipped
#endif

// -- IBM Power --

#ifdef BLIS_KERNELS_POWER10
#include "bli_kernels_power10.h" // skipped
#endif
#ifdef BLIS_KERNELS_POWER9
#include "bli_kernels_power9.h" // skipped
#endif
#ifdef BLIS_KERNELS_POWER7
#include "bli_kernels_power7.h" // skipped
#endif

// -- IBM BG/Q --

#ifdef BLIS_KERNELS_BGQ
#include "bli_kernels_bgq.h" // skipped
#endif

// -- RISC-V --

#ifdef BLIS_KERNELS_RVI
#include "bli_kernels_rvi.h" // skipped
#endif
#ifdef BLIS_KERNELS_RVIV
#include "bli_kernels_rviv.h" // skipped
#endif

// -- SiFive RISC-V architectures --

#ifdef BLIS_KERNELS_SIFIVE_X280
#include "bli_kernels_sifive_x280.h" // skipped
#endif


#endif

// end bli_arch_config.h
#line 99 "./frame/include/blis.h"


// begin bli_kernel_macro_defs.h
#line 1 "./frame/include//bli_kernel_macro_defs.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_KERNEL_MACRO_DEFS_H
#define BLIS_KERNEL_MACRO_DEFS_H


// -- Define default threading parameters --------------------------------------

// -- Conventional (large code path) values --

// These BLIS_THREAD_RATIO_? macros distort the amount of work in the m and n
// dimensions for the purposes of factorizing the total number of threads into
// ways of parallelism in the ic and jc loops. See bli_rntm.c to see how these
// macros are used.
#ifndef BLIS_THREAD_RATIO_M
#define BLIS_THREAD_RATIO_M     1
#endif

#ifndef BLIS_THREAD_RATIO_N
#define BLIS_THREAD_RATIO_N     1
#endif

// These BLIS_THREAD_MAX_?R macros place a ceiling on the maximum amount of
// parallelism allowed when performing automatic factorization. See bli_rntm.c
// to see how these macros are used.
#ifndef BLIS_THREAD_MAX_IR
#define BLIS_THREAD_MAX_IR      1
#endif

#ifndef BLIS_THREAD_MAX_JR
#define BLIS_THREAD_MAX_JR      4
#endif

#if 0
// -- Skinny/small possibly-unpacked (sup code path) values --

#ifndef BLIS_THREAD_SUP_RATIO_M
#define BLIS_THREAD_SUP_RATIO_M   1
#endif

#ifndef BLIS_THREAD_SUP_RATIO_N
#define BLIS_THREAD_SUP_RATIO_N   2
#endif

#ifndef BLIS_THREAD_SUP_MAX_IR
#define BLIS_THREAD_SUP_MAX_IR    1
#endif

#ifndef BLIS_THREAD_SUP_MAX_JR
#define BLIS_THREAD_SUP_MAX_JR    8
#endif
#endif


// -- Memory allocation --------------------------------------------------------

// hbwmalloc.h provides hbw_malloc() and hbw_free() on systems with
// libmemkind. But disable use of libmemkind if BLIS_DISABLE_MEMKIND
// was explicitly defined.
#ifdef BLIS_DISABLE_MEMKIND
  #undef BLIS_ENABLE_MEMKIND
#endif
#ifdef BLIS_ENABLE_MEMKIND
#include <hbwmalloc.h> // skipped
#endif

// Memory allocation functions. These macros define the three types of
// malloc()-style functions, and their free() counterparts: one for each
// type of memory to be allocated.
// NOTE: ANY ALTERNATIVE TO malloc()/free() USED FOR ANY OF THE FOLLOWING
// THREE PAIRS OF MACROS MUST USE THE SAME FUNCTION PROTOTYPE AS malloc()
// and free():
//
//   void* malloc( size_t size );
//   void  free( void* p );
//

// This allocation function is called to allocate memory for blocks within
// BLIS's internal memory pools.
#ifndef BLIS_MALLOC_POOL
  // If use of libmemkind was enabled at configure-time, the default
  // memory allocation function for memory pools should be hbw_malloc()
  // instead of malloc().
  #ifdef  BLIS_ENABLE_MEMKIND
  #define BLIS_MALLOC_POOL               hbw_malloc
  #else
  #define BLIS_MALLOC_POOL               malloc
  #endif
#endif

#ifndef BLIS_FREE_POOL
  // If use of libmemkind was enabled at configure-time, the default
  // memory deallocation function for memory pools should be hbw_free()
  // instead of free().
  #ifdef  BLIS_ENABLE_MEMKIND
  #define BLIS_FREE_POOL                 hbw_free
  #else
  #define BLIS_FREE_POOL                 free
  #endif
#endif

// This allocation function is called to allocate memory for internally-
// used objects and structures, such as control tree nodes.
#ifndef BLIS_MALLOC_INTL
#define BLIS_MALLOC_INTL                 malloc
#endif

#ifndef BLIS_FREE_INTL
#define BLIS_FREE_INTL                   free
#endif

// This allocation function is called to allocate memory for objects
// created by user-level API functions, such as bli_obj_create().
#ifndef BLIS_MALLOC_USER
#define BLIS_MALLOC_USER                 malloc
#endif

#ifndef BLIS_FREE_USER
#define BLIS_FREE_USER                   free
#endif


// -- Other system-related definitions -----------------------------------------

// Size of a virtual memory page. This is used to align blocks within the
// memory pools.
#ifndef BLIS_PAGE_SIZE
#define BLIS_PAGE_SIZE                   4096
#endif

// The maximum number of named SIMD vector registers available for use.
// When configuring with umbrella configuration families, this should be
// set to the maximum number of registers across all sub-configurations in
// the family.
#ifndef BLIS_SIMD_MAX_NUM_REGISTERS
#define BLIS_SIMD_MAX_NUM_REGISTERS      32
#endif

// The maximum size (in bytes) of each SIMD vector.
// When configuring with umbrella configuration families, this should be
// set to the maximum SIMD size across all sub-configurations in the family.
#ifndef BLIS_SIMD_MAX_SIZE
#define BLIS_SIMD_MAX_SIZE               64
#endif

// Alignment size (in bytes) needed by the instruction set for aligned
// SIMD/vector instructions.
#ifndef BLIS_SIMD_ALIGN_SIZE
#define BLIS_SIMD_ALIGN_SIZE             BLIS_SIMD_MAX_SIZE
#endif

// The maximum size in bytes of local stack buffers within macro-kernel
// functions. These buffers are usually used to store a temporary copy
// of a single microtile. The reason we multiply by 2 is to handle induced
// methods, where we use real domain register blocksizes in units of
// complex elements. Specifically, the macro-kernels will need this larger
// micro-tile footprint, even though the virtual micro-kernels will only
// ever be writing to half (real or imaginary part) at a time.
#ifndef BLIS_STACK_BUF_MAX_SIZE
#define BLIS_STACK_BUF_MAX_SIZE          ( BLIS_SIMD_MAX_NUM_REGISTERS * \
                                           BLIS_SIMD_MAX_SIZE * 2 )
#endif

// Alignment size used to align local stack buffers within macro-kernel
// functions.
#ifndef BLIS_STACK_BUF_ALIGN_SIZE
#define BLIS_STACK_BUF_ALIGN_SIZE        BLIS_SIMD_ALIGN_SIZE
#endif

// Alignment size used when allocating memory via BLIS_MALLOC_USER.
// To disable heap alignment, set this to 1.
#ifndef BLIS_HEAP_ADDR_ALIGN_SIZE
#define BLIS_HEAP_ADDR_ALIGN_SIZE        BLIS_SIMD_ALIGN_SIZE
#endif

// Alignment size used when sizing leading dimensions of memory allocated
// via BLIS_MALLOC_USER.
#ifndef BLIS_HEAP_STRIDE_ALIGN_SIZE
#define BLIS_HEAP_STRIDE_ALIGN_SIZE      BLIS_SIMD_ALIGN_SIZE
#endif

// Alignment sizes used when allocating blocks to the internal memory
// pool, via BLIS_MALLOC_POOL.
#ifndef BLIS_POOL_ADDR_ALIGN_SIZE_A
#define BLIS_POOL_ADDR_ALIGN_SIZE_A      BLIS_PAGE_SIZE
#endif

#ifndef BLIS_POOL_ADDR_ALIGN_SIZE_B
#define BLIS_POOL_ADDR_ALIGN_SIZE_B      BLIS_PAGE_SIZE
#endif

#ifndef BLIS_POOL_ADDR_ALIGN_SIZE_C
#define BLIS_POOL_ADDR_ALIGN_SIZE_C      BLIS_PAGE_SIZE
#endif

#ifndef BLIS_POOL_ADDR_ALIGN_SIZE_GEN
#define BLIS_POOL_ADDR_ALIGN_SIZE_GEN    BLIS_PAGE_SIZE
#endif

// Offsets from alignment specified by BLIS_POOL_ADDR_ALIGN_SIZE_*.
#ifndef BLIS_POOL_ADDR_OFFSET_SIZE_A
#define BLIS_POOL_ADDR_OFFSET_SIZE_A     0
#endif

#ifndef BLIS_POOL_ADDR_OFFSET_SIZE_B
#define BLIS_POOL_ADDR_OFFSET_SIZE_B     0
#endif

#ifndef BLIS_POOL_ADDR_OFFSET_SIZE_C
#define BLIS_POOL_ADDR_OFFSET_SIZE_C     0
#endif

#ifndef BLIS_POOL_ADDR_OFFSET_SIZE_GEN
#define BLIS_POOL_ADDR_OFFSET_SIZE_GEN   0
#endif


// -- MR and NR blocksizes (only for reference kernels) ------------------------

// The build system defines BLIS_IN_REF_KERNEL, but only when compiling
// reference kernels. By using compile-time constants for MR and NR, the
// compiler can perform certain optimizations, such as unrolling and
// vectorization, that would not be otherwise be possible.
#ifdef BLIS_IN_REF_KERNEL

#ifndef BLIS_MR_s
#define BLIS_MR_s 4
#endif

#ifndef BLIS_MR_d
#define BLIS_MR_d 4
#endif

#ifndef BLIS_MR_c
#define BLIS_MR_c 4
#endif

#ifndef BLIS_MR_z
#define BLIS_MR_z 4
#endif

#ifndef BLIS_NR_s
#define BLIS_NR_s 16
#endif

#ifndef BLIS_NR_d
#define BLIS_NR_d 8
#endif

#ifndef BLIS_NR_c
#define BLIS_NR_c 8
#endif

#ifndef BLIS_NR_z
#define BLIS_NR_z 4
#endif

#ifndef BLIS_BBM_s
#define BLIS_BBM_s 1
#endif

#ifndef BLIS_BBM_d
#define BLIS_BBM_d 1
#endif

#ifndef BLIS_BBM_c
#define BLIS_BBM_c 1
#endif

#ifndef BLIS_BBM_z
#define BLIS_BBM_z 1
#endif

#ifndef BLIS_BBN_s
#define BLIS_BBN_s 1
#endif

#ifndef BLIS_BBN_d
#define BLIS_BBN_d 1
#endif

#ifndef BLIS_BBN_c
#define BLIS_BBN_c 1
#endif

#ifndef BLIS_BBN_z
#define BLIS_BBN_z 1
#endif

#ifndef BLIS_PACKMR_s
#define BLIS_PACKMR_s (BLIS_MR_s*BLIS_BBM_s)
#endif

#ifndef BLIS_PACKMR_d
#define BLIS_PACKMR_d (BLIS_MR_d*BLIS_BBM_d)
#endif

#ifndef BLIS_PACKMR_c
#define BLIS_PACKMR_c (BLIS_MR_c*BLIS_BBM_c)
#endif

#ifndef BLIS_PACKMR_z
#define BLIS_PACKMR_z (BLIS_MR_z*BLIS_BBM_z)
#endif

#ifndef BLIS_PACKNR_s
#define BLIS_PACKNR_s (BLIS_NR_s*BLIS_BBN_s)
#endif

#ifndef BLIS_PACKNR_d
#define BLIS_PACKNR_d (BLIS_NR_d*BLIS_BBN_d)
#endif

#ifndef BLIS_PACKNR_c
#define BLIS_PACKNR_c (BLIS_NR_c*BLIS_BBN_c)
#endif

#ifndef BLIS_PACKNR_z
#define BLIS_PACKNR_z (BLIS_NR_z*BLIS_BBN_z)
#endif

#endif

#endif

// end bli_kernel_macro_defs.h
#line 101 "./frame/include/blis.h"



// -- Threading definitions --

// begin bli_thread.h
#line 1 "./frame/thread//bli_thread.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2016, Hewlett Packard Enterprise Development LP
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_THREAD_H
#define BLIS_THREAD_H

// Include thread communicator (thrcomm_t) object definitions and prototypes.
// begin bli_thrcomm.h
#line 1 "./frame/thread//bli_thrcomm.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_THRCOMM_H
#define BLIS_THRCOMM_H

// Define barrier_t, which is specific to the tree barrier in the OpenMP
// implementation. This needs to be done first since it is (potentially)
// used within the definition of thrcomm_t below.

#ifdef BLIS_ENABLE_OPENMP
#ifdef BLIS_TREE_BARRIER
struct barrier_s
{
	int               arity;
	struct barrier_s* dad;

	// We insert a cache line of padding here to eliminate false sharing between
	// the fields above and fields below.
	char   padding1[ BLIS_CACHE_LINE_SIZE ];

	dim_t             count;

	// We insert a cache line of padding here to eliminate false sharing between
	// the fields above and fields below.
	char   padding2[ BLIS_CACHE_LINE_SIZE ];

	gint_t            signal;

	// We insert a cache line of padding here to eliminate false sharing between
	// this struct and the next one.
	char   padding3[ BLIS_CACHE_LINE_SIZE ];
};
typedef struct barrier_s barrier_t;
#endif
#endif

// Define hpx_barrier_t, which is specific to the barrier used in HPX
// implementation. This needs to be done first since it is (potentially)
// used within the definition of thrcomm_t below.

#ifdef BLIS_ENABLE_HPX
typedef struct hpx_barrier_t
{
	void* handle;
} hpx_barrier_t;
#endif

// Define the thrcomm_t structure, which will be common to all threading
// implementations.

typedef struct thrcomm_s
{
	// -- Fields common to all threading implementations --

	void*       sent_object;
	dim_t       n_threads;
	timpl_t     ti;

	// We insert a cache line of padding here to eliminate false sharing between
	// the fields above and fields below.
	char   padding1[ BLIS_CACHE_LINE_SIZE ];

	// NOTE: barrier_sense was originally a gint_t-based bool_t, but upon
	// redefining bool_t as bool we discovered that some gcc __atomic built-ins
	// don't allow the use of bool for the variables being operated upon.
	// (Specifically, this was observed of __atomic_fetch_xor(), but it likely
	// applies to all other related built-ins.) Thus, we get around this by
	// redefining barrier_sense as a gint_t.
	//volatile gint_t  barrier_sense;
	gint_t barrier_sense;

	// We insert a cache line of padding here to eliminate false sharing between
	// the fields above and fields below.
	char   padding2[ BLIS_CACHE_LINE_SIZE ];

	dim_t  barrier_threads_arrived;

	// We insert a cache line of padding here to eliminate false sharing between
	// the fields above and whatever data structures follow.
	char   padding3[ BLIS_CACHE_LINE_SIZE ];

	// -- Fields specific to OpenMP --

	#ifdef BLIS_ENABLE_OPENMP
	#ifdef BLIS_TREE_BARRIER
	// This field is only needed if the tree barrier implementation is being
	// compiled. The non-tree barrier code does not use it.
	barrier_t** barriers;
	#endif
	#endif

	// -- Fields specific to pthreads --

	#ifdef BLIS_ENABLE_PTHREADS
	#ifdef BLIS_USE_PTHREAD_BARRIER
	// This field is only needed if the pthread_barrier_t implementation is
	// being compiled. The non-pthread_barrier_t code does not use it.
	bli_pthread_barrier_t barrier;
	#endif
	#endif

	// -- Fields specific to HPX --

	#ifdef BLIS_ENABLE_HPX
	hpx_barrier_t barrier;
	#endif

} thrcomm_t;





// Include definitions (mostly thrcomm_t) specific to the method of
// multithreading.
// begin bli_thrcomm_single.h
#line 1 "./frame/thread//bli_thrcomm_single.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_THRCOMM_SINGLE_H
#define BLIS_THRCOMM_SINGLE_H

// Always define these prototypes since disabling multithreading is always
// an option.

// Sequential-specific function prototypes.
void bli_thrcomm_init_single( dim_t nt, thrcomm_t* comm );
void bli_thrcomm_cleanup_single( thrcomm_t* comm );
void bli_thrcomm_barrier_single( dim_t tid, thrcomm_t* comm );

#endif

// end bli_thrcomm_single.h
#line 150 "./frame/thread//bli_thrcomm.h"

// begin bli_thrcomm_openmp.h
#line 1 "./frame/thread//bli_thrcomm_openmp.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_THRCOMM_OPENMP_H
#define BLIS_THRCOMM_OPENMP_H

// Define these prototypes for situations when OpenMP multithreading is
// enabled.
#ifdef BLIS_ENABLE_OPENMP

#include <omp.h> // skipped

// OpenMP-specific function prototypes.
void bli_thrcomm_init_openmp( dim_t nt, thrcomm_t* comm );
void bli_thrcomm_cleanup_openmp( thrcomm_t* comm );
void bli_thrcomm_barrier_openmp( dim_t tid, thrcomm_t* comm );

// Prototypes specific to the OpenMP tree barrier implementation.
#ifdef BLIS_TREE_BARRIER
barrier_t* bli_thrcomm_tree_barrier_create( int num_threads, int arity, barrier_t** leaves, int leaf_index );
void       bli_thrcomm_tree_barrier_free( barrier_t* barrier );
void       bli_thrcomm_tree_barrier( barrier_t* barack );
#endif

#endif

#endif

// end bli_thrcomm_openmp.h
#line 151 "./frame/thread//bli_thrcomm.h"

// begin bli_thrcomm_pthreads.h
#line 1 "./frame/thread//bli_thrcomm_pthreads.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_THRCOMM_PTHREADS_H
#define BLIS_THRCOMM_PTHREADS_H

// Define these prototypes for situations when POSIX multithreading is enabled.
#ifdef BLIS_ENABLE_PTHREADS 

// pthreads-specific function prototypes.
void bli_thrcomm_init_pthreads( dim_t nt, thrcomm_t* comm );
void bli_thrcomm_cleanup_pthreads( thrcomm_t* comm );
void bli_thrcomm_barrier_pthreads( dim_t tid, thrcomm_t* comm );

#endif

#endif

// end bli_thrcomm_pthreads.h
#line 152 "./frame/thread//bli_thrcomm.h"

// begin bli_thrcomm_hpx.h
#line 1 "./frame/thread//bli_thrcomm_hpx.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2022 Tactical Computing Laboratories, LLC

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_THRCOMM_HPX_H
#define BLIS_THRCOMM_HPX_H

// Define these prototypes for situations when HPX multithreading is enabled.
#ifdef BLIS_ENABLE_HPX

void bli_thrcomm_init_hpx( dim_t nt, thrcomm_t* comm );
void bli_thrcomm_cleanup_hpx( thrcomm_t* comm );
void bli_thrcomm_barrier_hpx( dim_t tid, thrcomm_t* comm );

#endif

#endif

// end bli_thrcomm_hpx.h
#line 153 "./frame/thread//bli_thrcomm.h"


// Define a function pointer type for each of the functions that are
// "overloaded" by each method of multithreading.
typedef void (*thrcomm_init_ft)( dim_t nt, thrcomm_t* comm );
typedef void (*thrcomm_cleanup_ft)( thrcomm_t* comm );
typedef void (*thrcomm_barrier_ft)( dim_t tid, thrcomm_t* comm );


// thrcomm_t query (field only)

BLIS_INLINE dim_t bli_thrcomm_num_threads( thrcomm_t* comm )
{
	return comm->n_threads;
}

BLIS_INLINE timpl_t bli_thrcomm_thread_impl( thrcomm_t* comm )
{
	return comm->ti;
}


// Threading method-agnostic function prototypes.
thrcomm_t* bli_thrcomm_create( timpl_t ti, pool_t* sba_pool, dim_t n_threads );
void       bli_thrcomm_free( pool_t* sba_pool, thrcomm_t* comm );

// Threading method-specific function prototypes.
// NOTE: These are the prototypes to the dispatcher functions and thus they
// require the timpl_t as an argument. The threading-specific functions can
// (and do) omit the timpl_t from their function signatures since their
// threading implementation is intrinsically known.
void                   bli_thrcomm_init( timpl_t ti, dim_t n_threads, thrcomm_t* comm );
void                   bli_thrcomm_cleanup( thrcomm_t* comm );
BLIS_EXPORT_BLIS void  bli_thrcomm_barrier( dim_t thread_id, thrcomm_t* comm );

// Other function prototypes.
BLIS_EXPORT_BLIS void* bli_thrcomm_bcast( dim_t inside_id, void* to_send, thrcomm_t* comm );
void                   bli_thrcomm_barrier_atomic( dim_t thread_id, thrcomm_t* comm );

#endif

// end bli_thrcomm.h
#line 42 "./frame/thread//bli_thread.h"


// Include thread info (thrinfo_t) object definitions and prototypes.
// begin bli_thrinfo.h
#line 1 "./frame/thread//bli_thrinfo.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin
   Copyright (C) 2018 - 2019, Advanced Micro Devices, Inc.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#ifndef BLIS_THRINFO_H
#define BLIS_THRINFO_H

// Thread info structure definition
struct thrinfo_s
{
	// The thread communicator for the other threads sharing the same work
	// at this level.
	thrcomm_t*         comm;

	// Our thread id within the thread communicator.
	dim_t              thread_id;

	// The number of communicators which are "siblings" of our communicator.
	dim_t              n_way;

	// An id to identify what we're working on. This is the same for all threads
	// in the same communicator, and 0 <= work_id < n_way.
	dim_t              work_id;

	// When freeing, should the communicators in this node be freed? Usually,
	// this is field is true, but when nodes are created that share the same
	// communicators as other nodes (such as with packm nodes), this is set
	// to false.
	bool               free_comm;

	// The small block pool.
	pool_t*            sba_pool;

	// The packing block allocator.
	pba_t*             pba;

	// Storage for allocated memory obtained from the packing block allocator.
	mem_t              mem;

	struct thrinfo_s*  sub_prenode;
	struct thrinfo_s*  sub_node;
};
typedef struct thrinfo_s thrinfo_t;

//
// thrinfo_t functions
//

// thrinfo_t query (field only)

BLIS_INLINE dim_t bli_thrinfo_num_threads( const thrinfo_t* t )
{
	return (t->comm)->n_threads;
}

BLIS_INLINE dim_t bli_thrinfo_thread_id( const thrinfo_t* t )
{
	return t->thread_id;
}

BLIS_INLINE dim_t bli_thrinfo_n_way( const thrinfo_t* t )
{
	return t->n_way;
}

BLIS_INLINE dim_t bli_thrinfo_work_id( const thrinfo_t* t )
{
	return t->work_id;
}

BLIS_INLINE thrcomm_t* bli_thrinfo_comm( const thrinfo_t* t )
{
	return t->comm;
}

BLIS_INLINE bool bli_thrinfo_needs_free_comm( const thrinfo_t* t )
{
	return t->free_comm;
}

BLIS_INLINE pool_t* bli_thrinfo_sba_pool( const thrinfo_t* t )
{
	return t->sba_pool;
}

BLIS_INLINE pba_t* bli_thrinfo_pba( const thrinfo_t* t )
{
	return t->pba;
}

BLIS_INLINE mem_t* bli_thrinfo_mem( thrinfo_t* t )
{
	return &t->mem;
}

BLIS_INLINE thrinfo_t* bli_thrinfo_sub_node( const thrinfo_t* t )
{
	return t->sub_node;
}

BLIS_INLINE thrinfo_t* bli_thrinfo_sub_prenode( const thrinfo_t* t )
{
	return t->sub_prenode;
}

// thrinfo_t query (complex)

BLIS_INLINE bool bli_thrinfo_am_chief( const thrinfo_t* t )
{
	return t->thread_id == 0;
}

// thrinfo_t modification

BLIS_INLINE void bli_thrinfo_set_comm( thrcomm_t* comm, thrinfo_t* t )
{
	t->comm = comm;
}

BLIS_INLINE void bli_thrinfo_set_thread_id( dim_t thread_id, thrinfo_t* t )
{
	t->thread_id = thread_id;
}

BLIS_INLINE void bli_thrinfo_set_n_way( dim_t n_way, thrinfo_t* t )
{
	t->n_way = n_way;
}

BLIS_INLINE void bli_thrinfo_set_work_id( dim_t work_id, thrinfo_t* t )
{
	t->work_id = work_id;
}

BLIS_INLINE void bli_thrinfo_set_free_comm( bool free_comm, thrinfo_t* t )
{
	t->free_comm = free_comm;
}

BLIS_INLINE void bli_thrinfo_set_sba_pool( pool_t* sba_pool, thrinfo_t* t )
{
	t->sba_pool = sba_pool;
}

BLIS_INLINE void bli_thrinfo_set_pba( pba_t* pba, thrinfo_t* t )
{
	t->pba = pba;
}

BLIS_INLINE void bli_thrinfo_set_sub_node( thrinfo_t* sub_node, thrinfo_t* t )
{
	t->sub_node = sub_node;
}

BLIS_INLINE void bli_thrinfo_set_sub_prenode( thrinfo_t* sub_prenode, thrinfo_t* t )
{
	t->sub_prenode = sub_prenode;
}

// other thrinfo_t-related functions

BLIS_INLINE void* bli_thrinfo_broadcast( const thrinfo_t* t, void* p )
{
	return bli_thrcomm_bcast( t->thread_id, p, t->comm );
}

BLIS_INLINE void bli_thrinfo_barrier( const thrinfo_t* t )
{
	bli_thrcomm_barrier( t->thread_id, t->comm );
}


//
// Prototypes for level-3 thrinfo functions not specific to any operation.
//

thrinfo_t* bli_thrinfo_create_root
     (
       thrcomm_t* comm,
       dim_t      thread_id,
       pool_t*    sba_pool,
       pba_t*     pba
     );

thrinfo_t* bli_thrinfo_create
     (
       thrcomm_t* comm,
       dim_t      thread_id,
       dim_t      n_way,
       dim_t      work_id,
       bool       free_comm,
       pool_t*    sba_pool,
       pba_t*     pba
     );

BLIS_EXPORT_BLIS void bli_thrinfo_free
     (
       thrinfo_t* thread
     );

// -----------------------------------------------------------------------------

thrinfo_t* bli_thrinfo_split
     (
       dim_t      n_way,
       thrinfo_t* thread_par
     );

void bli_thrinfo_print
     (
       thrinfo_t* thread
     );

void bli_thrinfo_print_sub
     (
       thrinfo_t* thread,
       gint_t     level
     );

#endif
// end bli_thrinfo.h
#line 45 "./frame/thread//bli_thread.h"


// Thread lanuch prototypes. Must go before including implementation headers.
typedef void (*thread_func_t)( thrcomm_t* gl_comm, dim_t tid, const void* params );

// Include threading implementations.
// begin bli_thread_openmp.h
#line 1 "./frame/thread//bli_thread_openmp.h"

/*

   BLIS
   An object-based framework for developing high-performance BLAS-like
   libraries.

   Copyright (C) 2014, The University of Texas at Austin

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:
    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    - Neither the name(s) of the copyright holder(s) nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LO