#		U T I L  . T C L
#
# Module Name:
#	util.tcl
#
# Function:
#       Utilites used by configure which were originally in dtclu.
#
# Description:
#
#	DtcluCreateMain   - Create a main windoW
#	DtcluCommand      - Popup a window into to enter Tcl commands in.
#	MsgOut  	  - Output the specified string into the message area.
#	ErsOut            - Output the specified string as an error.
#       DtcluFDialog      - Create a file dialog.
#
#	DtcluBindEntry	   - Sets up bindings for Entry widgets.
#	DtcluBindText	   - Sets up bindings for Text widgets.
#	DtcluBindEntryInt  - Sets up bindings for Integer Entry widgets 
#	DtcluBindEntryReal - Sets up bindings for Real number entry widgets
#	DtcluBindEntryPos  - Sets up bindings for Telescope Position entry 
#					widgets
#       DtcluDialogPosSet  - Centre a dialog on the screen
#       DtcluDialogPosReset - Ensure a dialog is viewable.
#
#    This routines are used by the Bind routines.
#	DtcluInsertInt	  - Used to verify insertion for an integer text 
#			    field insertion.
#	DtcluInsertReal	  - Used to verify insertion for a real number text 
#			    field insertion.
#	DtcluIntertPos	  - Used to verify insertion for Sky positions in
#				RA and DEc.
#
# Languge: TCL
# 
# Author: Tony Farrell
#
#
# Sccs Id:     util.tcl, Release 1.11, 04/27/00
#
# History:
#	03-Nov-1998 - TJF- Original Version, extracted from DTLCU.TCL
#                          to allow the configure program to not be dependent
#                          on DRAMA.
#       18-Dec-1998 - TJF - Update DtcluFDialog to work with tcl 8.0.3
#                          Make proper use of bgerror to report errors
#                          in the DtcluFDialog use command and ensure
#                          we put the results of ErsFlush in the message.
#       21-Jan-1999 - TJF - Add Sccs Id: line.
#       22-Oct-1999 - TJF - Incorporate Gavin's requested change
#			    to ErsOut such that is can be run in
#			    a mode where the program continues
#			    whilst an Error dialog is up.  See
#			    email subject "Configure vesion R3.3" of
#			    21st Oct, 1999.
#       17-Mar-2000 - TJF - DtcluFDialog now changes default directory
#                           tothe location of the last file opened/saved.
#       20-Mar-2000 - TJF - Grab a copy of DtclTkDialog and use it to
#                           implement ErsOut.  Can not drop stuff from
#                            22-Oct change.
#       05-Apr-2000 - TJf - Grab a copy of DtclError and use it as tkerror,
#                           which allows use to continue whilst errors are up.
#       27-Apr-2000 - TJF - Replace use of Bell by bell.
#       28-May-2001 - KS  - Increased wrap width of DtclTkDialog from 3" to
#                           6", changed text in error messages to "Error"
#                           instead of "Programming error"
#
#
global Dtclu
set Dtclu(Ecount) 0

global utils_font_1           ;# Font used for the various dialogues.
global utils_font_2           ;# Font used for the various dialogues.
global tcl_platform           ;# Tcl array giving platform-dependent info. In
                              ;# this case used to pick font names.

#  Set the font used for the various dialogues.
                              
if { ($tcl_platform(platform) == "windows") ||
           ($tcl_platform(platform) == "macintosh")} {
    set utils_font_1 {Helv 14 }
    set utils_font_1 {Times 14 }
} else {
    set utils_font_1 "-Adobe-Times-Medium-R-Normal-*-180-*"
    set utils_font_2 "-Adobe-Helvetica-Medium-R-Normal-*-120-*"
    set utils_font_1 {Helv 14 }
    set utils_font_2 {Times 14 }
}
 


#
# Ensure we have loaded the bgerror command.  Note, this only works
# for later versions of Tcl/Tk.  Not sure as yet of what we should
# do for older versions.
#
# We must force the load here due to Tcl's lazy load feature, which means
# we don't get the command until we use it.
#
if {[info commands bgerror] == "" } {
   if { [file exists $tk_library/bgerror.tcl] } {
	    source $tk_library/bgerror.tcl
   }	
}
#
# Now rename bgerror.
#
rename bgerror Dtclu_origbgerror

#
# Our version of bgerror.  This has to ensure ErsFlush is invoked and
# pick up the values returned by ErsFlush.
#
proc bgerror err {
    global Dtclu
    # ErsFlush will result in ErsOut being invoked.  We want to catch the
    # value passed to ErsOut.  We do this by renamed ErsOut and creating
    # our own version.  That version just saves any text in Dtclu(SavedMessage)
    rename ErsOut OldErsOut
    set Dtclu(SavedMessage) ""
    proc ErsOut { mess args } {
        global Dtclu
        set Dtclu(SavedMessage) $mess
    }
    ErsFlush

    # Delete our version of ErsOut and restore the original version.
    rename ErsOut ""
    rename OldErsOut ErsOut

    # If we flushed anything, then append it to the error report
    if { $Dtclu(SavedMessage) != "" } {
        Dtclu_origbgerror "$Dtclu(SavedMessage)\n\n$err"
    } else {
        Dtclu_origbgerror $err
    }
}

# Must work out if the -validate option has been built into Tk
# We need a window to do this. In addition, if we do have this
# option, we must have a version which accepts the"key" argument
# to -validate.  To check this, we get the default value, which
# should be none.
entry .dtclu_eee
if { [string first validate [.dtclu_eee configure]] == -1 } {
    set Dtclu(HaveValidate) 0
} else {
    if { [.dtclu_eee cget -validate] == "none" } {
        set Dtclu(HaveValidate) 1
    } else {
        set Dtclu(HaveValidate) 0
    }
}
destroy .dtclu_eee


###############################################################################
#		D t c l u C r e a t e M a i n
#+
# Function Name:
#	DtcluCreateMain
#
# Function:
#	Create a main window
#
# Description:
#	Creates a main window containing three 
#		frames - .menubar, .mainframe and .messframe.  
#
#	Into .menubar will be packed three menu buttons -
#		[.file.menu] containing an Exit buttons
#		[.commands.menu] containing nothing
#		[.options.menu] containing nothing.
#
#
#	The Exit button will create a pop-up prompting for confirmation of
#	exit.  If confirmation is received, a task, the name of which is found
#	in the global variable supplied to this routine,  will be sent
#	an Exit command, if possible.  The confirmation dialog also allows
#	for a "Dirty Shutdown", which is simply a "destroy ." command.
#
#	Ers and MsgOut message handlers are defined which will send messages 
#       to a scrolling region packed into .messframe.
#
#	The routine Disconnect, which must be supplied by the user, will be 
#	invoked when a task disconnects.
#
#	Entry and Text widgets are given default key and mouse bindings which
#	make them work like similar Motif/Athena widgets.
#
#	The user can add items to .mainframe , the menus and to the menu bar.  
#	All menu bar and menu items should have a character underlined to 
#	enable keyboard traversal.
#
# Call:
#	DtcluCreateMain	taskvar
#
# Parameters: 
#	(>) taskvar	(string) The name of a global variable containing the 
#			task name.  Only used by the Exit button.
#
# Options:
#	[-messarea boolean]  Set true if the scrolling message area is to 
#			be enabled.  Set false to disable it.  Default is true.
#			If false, then the procedure MsgOut (the default 
#			MessageHandler procedure) outputs messages to stdout 
#			instead of a scolling message area. If set false, then
#			-expandmain (below) is set true.
#	[-expandmain boolean]  Determines if the main packing area is packed
#			with expansion enabled.  By default, this is false.
#			In this case, if the window is resized by the
#			user, the main packing area is not resized.  This
#			allows all the resizing to be taken up by the scrolling
#			message area, the result normally required.  If true
#			then the main window and message scrolling area will
#			share the results of the resizing.  If the message
#			area is disabled using "-messarea false", then this
#			option is set true.  It can in this case be cleared 
#			again by putting this option after -messarea on the 
#			command	line.
#	[-messheight integer] Sets the height of the scrolling message 
#			area (the number of messages displayed).  This value 
#			defaults to 10 (reason unknown).  See the -height 
#			option of the listbox widget for more details.
#	[messwidth integer] Sets the width of the scrolling message area in 
#			characters.  This value defaults to 20 but the actual
#			width of the message area will be no less then the
#			width of the main window.  See the -width option of the
#			listbox widget for more details.
#				
#
#
# Support: Tony Farrell, AAO
#-
#
proc DtcluCreateMain { args} {

    global utils_font_1     ;# Font used in dialogues
   
    set MessageAreaOn 1
    set MainExpand 0
    set SetMessHeight 0
    set SetMessWidth  0
    
    if { $args != "" } {
        set len [llength $args]
        set len2 [expr $len - 1]
        
        set i 0
        
        while { $i <= $len2 } {
            # option name
            set option [lindex $args $i]
	    incr i
	    
	    # Option value
	    set value [lindex $args $i]
	    incr i
	    
            switch -- $option {
                "-messarea" { 
                	if { $value == "" } {
                	    error "Invalid value for option \"$option\""
                	}
                	set MessageAreaOn $value 
                	if { ! $value } {
                	    set MainExpand 1
                	}
                }
                "-mainexpand" { 
                	if { $value == "" } {
                	    error "Invalid value for option \"$option\""
                	}
                	set MainExpand $value 
                }
                 "-messheight" { 
                	if { $value == "" } {
                	    error "Invalid value for option \"$option\""
                	}
                	set SetMessHeight 1
                	set MessHeight $value 
                }
                 "-messwidth" { 
                	if { $value == "" } {
                	    error "Invalid value for option \"$option\""
                	}
                	set SetMessWidth 1
                	set MessWidth $value 
                }
                default { error "unknown option \"$option\""}
            }
	}            
    }

    global tk_version
    wm minsize . 5 5
    wm maxsize . 1000 800

    if { [winfo depth .] > 1 } {
        option add *activeForeground "red"
        option add *selectForeground "Black"
    }

#
# We create to frames, one for the menu bar, the main area and the message area
#
    frame .menubar -relief raised -borderwidth 1
    frame .mainframe -relief raised -borderwidth 1 
    pack .menubar    -side top -fill both
    pack .mainframe  -side top -fill both -expand $MainExpand
    if { $MessageAreaOn } {
        frame .messframe -relief raised -borderwidth 1
        pack .messframe  -side top -fill both -expand 1
    }  

#   Create the file menu.
    menubutton .file -text File -menu .file.menu   -underline 0

    menu .file.menu
#    .file.menu add command -label "Exit" \
    	-command "Dtclu___Exit \"$taskvar\"" -underline 0



#   Create the commands menu
    menubutton .commands -text Commands -menu .commands.menu -underline 0
    menu .commands.menu 


#   Create the options menu
    menubutton .options -text Options -menu .options.menu  -underline 0
    menu .options.menu

    pack .file .commands .options -side left -padx 7 -pady 7  -in .menubar
#
#   Message area.  (Note, text message area not tested under Tk 3.6)
#
    if { $MessageAreaOn } {
        label .messagetitle -text "Messages" -borderwidth 3 
	scrollbar .vscroll -orient vertical   -command {.messagearea yview}
        scrollbar .hscroll -orient horizontal -command {.messagearea xview}
        text .messagearea -relief groove  -borderwidth 3 -setgrid 1\
   	    -yscrollcommand {.vscroll set} \
	    -xscrollcommand {.hscroll set} \
	    -height 10 -width 0 -state disabled -wrap none \
	    -background [.vscroll cget -troughcolor] \
	    -selectbackground black \
	    -selectforeground [.vscroll cget -troughcolor]


        pack .messagetitle -side top -fill x -in .messframe
        pack .vscroll -side right -fill y -in .messframe
        pack .messagearea -fill both -expand 1 -in .messframe
        pack .hscroll -fill x -in .messframe

# 	Highlighting text tags for MsgOut
	.messagearea tag configure error -foreground black -background red \
		-font $utils_font_1
#		-font "-*-helvetica-bold-r-normal-*-14-*-*-*-*-*-*-*"

	.messagearea tag configure highlight -foreground black \
		-font $utils_font_1 \
			-background  skyblue
#		-font "-*-helvetica-bold-r-normal-*-14-*-*-*-*-*-*-*" \

#	Make sure the tags are of lower priority then sel, so that we can
#	always see the selection.

	.messagearea tag lower error sel
	.messagearea tag lower highlight sel

        bind .messagearea <Destroy> Dtclu___MessageAreaDestory
        if { $tk_version >= 4.0 } {
#	    Message area should not take focus.
	    .messagearea configure -takefocus 0
	    
	    if { $SetMessHeight } {
	        .messagearea configure -height $MessHeight
	    }
	    if { $SetMessWidth } {
	        .messagearea configure -width $MessWidth
	    }
        }
    }


#
#   Setup keyboard to operate menu bar and default bindings for Text
#   widgets.  These are not necessary after 4.0 when all we need to do
#   is make Delete work as per BackSpace.
#

    if { $tk_version < 4.0 } {
        tk_menuBar .menubar .file .commands .options
        tk_bindForTraversal . 
    	bind . <Enter> { focus . }

        DtcluBindText Text
        bind Text <Any-KeyPress> {Dtclu___TextSel %W 1 {
            if {"%A" != ""} {
                %W insert insert %A
            }
          }
        }

#	Entry bindings.
        DtcluBindEntry Entry 
        bind Entry <Any-KeyPress> {Dtclu___EntrySel %W 1 {
            if {"%A" != ""} {
                %W insert insert %A
            }
     
          }
        }
    } else {
#	Tk 4 or greater.  
        DtcluBindText Text
	DtcluBindEntry Entry
	bind all <Mod1-KeyPress> {
	   tkTraverseToMenu %W %A
	}
    }
    if { $tk_version >= 8.0 } {
	bind Button <Return> {
    		tkButtonInvoke %W
	}
    }



}
###############################################################################
#	Internal Procedures
#
#
#
# Dtclu___MessageAreaDestory - invoked when the message area is destory,
# which is probably when the main window (.) is being destory.
#
# We redefine MsgOut, ErsOut  such the messages are written to
# stdout and stderr instead of trying to use the now destoryed window system.

proc Dtclu___MessageAreaDestory {} {
   proc MsgOut { msg {tag "" } } {
	    puts "MsgOut:$msg"
   }
   proc ErsOut { msg args } {
	    puts stderr "ErsOut:$msg"
   }
}

#
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
# This stuff, which handles selections and insertions in Text and Entry
# widgets is not necessary for Tk 4 and later.
global tk_version
if { $tk_version < 4.0 } {
#
#  Internal procedure used to move about text widgets.
#
proc Dtclu___MoveChar { w lines cols } {
#  Work cout if moves are negative or positive
   set xlines $lines
   set xline_flag "+"
   if {$xlines<1} {
       set xlines [expr $xlines*-1]
       set xline_flag "-"
   }
   set xcols $cols
   set xcol_flag "+"
   if {$xcols<1} {
       set xcols [expr $xcols*-1]
       set xcol_flag "-"
   }

#  Get the new position, incremented by the move
   set xposition [$w index \
	"insert $xline_flag $xlines lines $xcol_flag $xcols chars"]

#  Set the new position
   $w mark set insert $xposition 
   $w yview -pickplace insert
}
#
# Internal procedures used to handle selection within Text widgets
#
# window is the window in which the event has occured
# flag   is 1 to do cmd anyway and 0 to only do if there is no selection
# cmd	 is a command to execute after select is handled
proc Dtclu___TextSel { window flag cmd } {

    set SelExists 0

#   If the selection exists, delete it.
#
    if { [$window tag ranges sel] != ""} {
	set SelExists 1
	$window delete sel.first sel.last
    }
#
#   IF either the selection did not exist, or the flag is true, execute
#	the command.
#
    if { ($SelExists == 0) || ($flag)} {
	eval $cmd
    }
    $window yview -pickplace insert
}
# Internal procedure used to move Entry widget cursor positions
proc Dtclu___EntryMoveChar { window incr } {

    set current [$window index insert]
    set new [expr ($current+$incr) ]
    $window icursor $new
    tk_entrySeeCaret $window
}
#
# Internal procedure used to handle selection within Entry widgets
#
# window is the window in which the event has occured
# flag   is 1 to do cmd anyway and 0 to only do if there is no selection
# cmd	 is a command to execute after select is handled
proc Dtclu___EntrySel { window flag cmd } {

#   If the selection exists, delete it.
#
    set NoSelection [catch "$window index sel.first"]
    if { $NoSelection == 0 } {
	$window delete sel.first sel.last
    }

#
#   IF either the selection did not exist, or the flag is true, execute
#	the command.
#
    if { ($NoSelection) || ($flag)} {
	eval $cmd
    }
    tk_entrySeeCaret $window
}


# Close bracket for tk_version < 4.0
}

#
###############################################################################
#		D t c l u C o m m a n d
#+
# Function Name:
#	DtcluCommand
# Function:
#	Pop up a dialog into which a Tcl command can be entered.
#
# Description:
#	A simple dialog window will be created into which the user can
#	enter any tcl commands.
#
#	The commands will be executed at global level.
#
#	If invoked when a previous invocation's window is iconized, then
#	the window will be deiconized and raised.  If a previous invocation's 
#	window is displayed, then a new window will be created.
#
# Call:
#	DtcluCommand
#
# Support: Tony Farrell, AAO
#-
proc DtcluCommand {} {
    set i 1

    while { 1 } { 
      set win ".dtcluCommand$i"
      set NotCreated [catch "wm state $win" state]
      if {$NotCreated} {
        toplevel $win -class DtcluCommand
        wm title $win "Configure Tcl Command Window $i"
        wm iconname $win TclCommand
        wm minsize  $win 5 5
        wm maxsize  $win 1000 800
        wm iconbitmap $win [wm iconbitmap .]
 
        frame $win.mainframe -relief raised -borderwidth 1
        frame $win.buttons   -relief raised -borderwidth 1
        pack  $win.mainframe -side top -expand 1 -fill both
        pack  $win.buttons -side bottom -fill both

        text  $win.text -wrap none -height 5 -width 60 -setgrid 1
        pack  $win.text -side top -expand 1 -fill both -in $win.mainframe

        button $win.ok -text Ok -command "Dtclu___CommandApply 1 $win" -pady 0
        button $win.apply -text Apply  -command "Dtclu___CommandApply 0 $win" -pady 0
        button $win.clear -text Clear  -command "$win.text delete 1.0 end" -pady 0
        button $win.cancel -text Cancel -command "wm withdraw $win" -pady 0
        button $win.new -text New -command DtcluCommand -pady 0

	bind $win.text <Shift-Return> {
		set parent [winfo parent %W]
		$parent.ok flash
		Dtclu___CommandApply 0 $parent	
		break
	}

	bind $win.text <Control-Return> {
		set parent [winfo parent %W]
		$parent.ok flash
		Dtclu___CommandApply 1 $parent	
		break
	}

        pack $win.ok $win.apply $win.clear $win.cancel $win.new -side left \
	  -padx 5 -pady 5 -expand 1 -fill x -in $win.buttons

        DtcluDialogPosSet $win
	global tk_version
	if { $tk_version >= 4.0 } {
	    focus $win.text
	}
	break

      } elseif { ($state=="withdrawn")||($state=="iconic") } {
        DtcluDialogPosReset $win
	global tk_version
	if { $tk_version >= 4.0 } {
	    focus $win.text
	}
	break
      } else {
	incr i
      }
    }
}


#  Internal procedure to actually do the command.
proc Dtclu___CommandApply {remove win} {
    global errorInfo
    set text [$win.text get 1.0 end]
    set Caught [catch {uplevel #0 "$text"} message]
    set savedInfo $errorInfo
    if { $Caught } {
        bgerror $message
    } else {
        if { "$message" != "" } {
	    MsgOut $message 
	}
        if { $remove } {
	    wm withdraw $win
        }
    }
}
###############################################################################
#		M s g O u t
#+
# Function Name:
#	MsgOut
# Function:
#	Write a string into the Dtclu message area.
#
# Description:
#	The specified string is written into the Dtclu message area.
#
# Call:
#	MsgOut message [tag]
#
# Parameters:
#	(>) string (message) The message to be output
#	(>) tag	   (text tag) A tag to be assocaited with the message.  This
#			is one of the text tags in the .messagearea widget.
#			Two are provided by default, error and highlight.
#
# Support: Tony Farrell, AAO
#-
proc MsgOut {s {tag ""} } {

  
#   If the message area does not exist, output to standard output and return.
    if { [info commands .messagearea]  == "" } {
	puts stdout $s
	return ""
    }

#
#   enable user interaction and allow us to see the end of the list before
#   inserting the message into the list.
#
    .messagearea configure -state normal
    .messagearea see end
    if {[info commands clock] == "clock" } {
        .messagearea insert end "[clock format [clock seconds] -format %T] $s\n" $tag
    } else {
        .messagearea insert end "$s\n" $tag
    }

#
#   Get the number of elements
    scan [.messagearea index end] "%d.%d" NumElements junk
    incr NumElements -2
#
#   We don't let the list get any bigger then about 200 elements since it
#   will take up a lot of memory and the response will get bad.  So every time
#   we get 20 elements above 200, we delete 20.  We insert a message at the
#   top of the list indicating what we have done.
    if {( $NumElements>2000) && ( $NumElements%20 == 0)} {
        .messagearea delete 1.0 20.0 
        .messagearea insert 1.0 "###PREVIOUS MESSAGES DELETED###\n"  
	.messagearea see end
    }
#
#   Prohibit user interaction, other then selection.

    .messagearea configure -state disabled

}


###############################################################################
#		E r s O u t
#+
# Function Name:
#	ErsOut
# Function:
#	Report an error.
#
# Description:
#	The specified string is written into the Dtclu message area and
#	an error popup is created to alert the user.
#
# Call:
#	ErsOut message
#
# Parameters:
#	(>) string (message) The message to be output
#
# Support: Tony Farrell, AAO
#
#-

global DtcluSupportAddress DtcluProgramDescr
set DtcluSupportAddress ks@aaoepp.aao.gov.au
set DtcluProgramDescr "2dF Configure Program"


proc ErsOut {s args } {
    global Dtclu

    set list [split $s "\n"]

#   Output to the message area, where logging will ocurrs if enabled 
    set count 1
    foreach i $list  {
        if { $count == 1 } {
            MsgOut "# $i" error
        } else {
            MsgOut "##$i" error
        }
        incr count
    }
#   Create the pop-up and display the message.

    incr Dtclu(Ecount)
    set w .ew$Dtclu(Ecount)

    set cmd [list Dtclu___ErsOutAck $Dtclu(Ecount) $w $s]
    DtclTkDialog !ErsOutDialog$w \
        "Error (Configure) $Dtclu(Ecount)" \
        $s error 0 $cmd OK "Clear All" 

}


proc Dtclu___ErsOutAck {count win text button} {
    if { $button == 2 } {
#       Use the NotifySupport stuff to setup a notification dialog.
        global DtcluProgramDescr
        Dtclu___ErrorNotifySupport \
            "Error in $DtcluProgramDescr" \
            "Error message triggered by NotifySupport button in ErsOut" \
            $text .ersNotify$count $win

    } elseif { $button == 1 } {
    	global Dtclu
        for { set i 1 } { $i <= $Dtclu(Ecount) } { incr i } {
            if [winfo exists .ew$i ] {
                destroy .ew$i
            }
        }	
    } 
}
#
#  Dtclu___ErrorNotifySupport.
#       Pops up a dialog box to enable the user to send a message to the
#       email address specified in DtcluSupportAddress.
#  Arguments:
#       subject - The subject line for the message
#       intro   - The introduction text for the message
#       errortext - The relevant error message text
#       textwin  - Name to use for the text window
#       errwin  - Error window name (destroyed on success)
#
#
#
proc Dtclu___ErrorNotifySupport { subject intro errortext textwin errwin } {

    toplevel $textwin -class NotifySupport
    wm title $textwin "Configure Error Notify Text Window"
    wm iconname $textwin ErrorNotify
    wm minsize  $textwin 5 5
    wm maxsize  $textwin 1000 800
    wm iconbitmap $textwin [wm iconbitmap .]


    frame $textwin.mainframe -relief raised -borderwidth 1
    frame $textwin.buttons   -relief raised -borderwidth 1
    pack  $textwin.mainframe -side top -expand 1 -fill both
    pack  $textwin.buttons -side bottom -fill both


    global DtcluSupportAddress
    message $textwin.msg  -text "\
Enter a message for support staff ($DtcluSupportAddress).  The actual error \
message text will be appended automatically to what you type.  You should \
try to describe what it was you did which appeared to trigger the error.\
\n\nIf the account you are using does not identify you (e.g. a general user\
account), please give your name and email address.\
\n\nWhen finished, hit the \"send\" button"\
            -relief ridge -border 2 -width 400

#            -font -Adobe-Times-Medium-R-Normal-*-180-* \
 


    text  $textwin.text -wrap none -height 5 -width 60 -setgrid 1

    frame $textwin.cc -borderwidth 2 -relief ridge
    frame $textwin.username -borderwidth 2 -relief ridge

    pack  $textwin.msg $textwin.text $textwin.cc $textwin.username \
                 -side top -expand 1 -fill both \
                -in $textwin.mainframe

    label $textwin.cc.lab -text "Copy messages to:"
    entry $textwin.cc.ent -width 30

    pack $textwin.cc.lab $textwin.cc.ent -fill x -expand 1 -side left

    label $textwin.username.lab -text "Confirm your email address:"
    entry $textwin.username.ent -width 30
    pack $textwin.username.lab $textwin.username.ent \
        -fill x -expand 1 -side left

    $textwin.username.ent insert 0 [TranslateName USER]

    set cmd \
        [list Dtclu___NotifySend $subject $intro $errortext $textwin $errwin]

    button $textwin.send -text Send -command $cmd
    button $textwin.cancel -text Cancel -command "destroy $textwin"

    pack $textwin.send  $textwin.cancel -side left \
          -padx 7 -pady 7 -expand 1 -fill x -in $textwin.buttons

   focus  $textwin.text
}
#  Callback for Dtclu___ErrorNotifySupport, same arguments.
#
#  This function actually sends the message.
#
proc Dtclu___NotifySend { subject intro errortext textwin errwin } {
    global DtcluSupportAddress DtcluProgramDescr


    set user [$textwin.username.ent get]
    if { $user == "" } {
        set user [TranslateName USER]
    }
    set cc [$textwin.cc.ent get]
 
    if { $cc != "" } {
        set cmd  [list "|mailx" -s "$subject"  -c "$cc" $DtcluSupportAddress]
    } else {
        set cmd  [list "|mailx" -s "$subject" $DtcluSupportAddress]
    }
#   Create a pipeline to the mail command.
#
    set pipe [open $cmd w]

#   Output text.

    set text [$textwin.text get 1.0 end]
    puts $pipe "$intro"
    puts $pipe ""
    puts $pipe "Details"
    puts $pipe "    Program Descr: $DtcluProgramDescr"

    if {[info commands clock] == "clock" } {
        set date [clock format [clock seconds]]
    } else {
        set date [exec date]
    }
    puts $pipe "    Time         : $date"
    puts $pipe "    User         : $user"
    puts $pipe "    Node         : [TranslateName HOST -default unknown]"
    global argv0 argv
    if { [info exists argv0] } {
        puts $pipe "    Command      : [concat $argv0 $argv]"
    }
    puts $pipe ""
    puts $pipe "User supplied notes"
    puts $pipe "    $text"
    puts $pipe ""
    puts $pipe "Error Info"
    puts $pipe "    $errortext"
    puts $pipe ""

#  Close pipe (sends message)
    close $pipe

#   Destroy windows. The error window could have already have gone so
#   catch any errors

    destroy $textwin
    catch {destroy $errwin}
}


###############################################################################
#^L             D t c l u I n s e r t I n t
#+
# Function Name:
#       DtcluInsertInt
#
# Function:
#   This function inserts the specified character into the specified window
#   iff the result is a valid integer number between the specified ranges
#
#   Note, a min greater then zero or a max less then zero are not handled.
#
# Description:
#
# Call:
#       DtcluInsertInt window min max char
#
# Parameters: 
#	(>) window	(tk window) The window.  Should be able to accept
#			the "path insert insert" command (.e.g. entry fields).
#	(>) min		(integer) The min acceptable number. This is only
#			used if it is less then zero since we cannot do
#			partial checks on a minium size > 0.
#	(>) max		(integer) The maximum acceptable number.  Only used
#			is greater then zero since we cannot do partial
#			checks on a maximum size less then zero.
#	(>) char	(string) The character to insert.
#
# Returned Value:
#	Returns 1 if character was handled.  (In Tk 4.0 or higher, a return
#	of 1 indicates binding should not do Entry level binding
#
# Support: Tony Farrell, AAO
#-
#
#
set Dtclu(InsertCount) 0
set Dtclu(InsertWindow) ""
proc DtcluInsertInt { window min max {char ""} } {
    if { ("$char" == "") || ("$char" == "{}") } {
    } elseif { "$char" < " "}  {
	global tk_version
	if { $tk_version < 4.0 } bell
    } else {
      global Dtclu
      set curtext [$window get]
      set end [$window index end]
      set insert [$window index insert]

      if { $insert < $end } {
	set prefix [string range $curtext 0 [expr ($insert-1)]]
	set suffix [string range $curtext $insert end]
	set proposed [join "$prefix $char $suffix" ""]
      } else {
        set proposed [ join "$curtext $char" "" ]
      }
      if { $Dtclu(InsertWindow) == "$window" } {
          incr Dtclu(InsertCount)
      } else {
	  set Dtclu(InsertCount) 1
	  set Dtclu(InsertWindow) $window
      }
      if { ("$proposed" == "-") && ($min < 0) } {
	    $window insert insert $char
	    set Dtclu(InsertCount) 0
      } elseif { [regexp {^-?[0-9]+$} "$proposed"] } {
	if { ($min < 0) && ($proposed < $min)} { 
	    bell
	} elseif { ($max > 0) && ($proposed > $max) } {
	    bell
	} else {
	    $window insert insert $char
	    set Dtclu(InsertCount) 0
	}
      } else {
	 bell
      }
      if { $Dtclu(InsertCount) >= 2 } { 
          ErsOut "This Field requires an integer between $min and $max."
      }
      return 1
    }
    return 0
}

###############################################################################
#^L             D t c l u I n s e r t R e a l
#+
# Function Name:
#       DtcluInsertReal
#
# Function:
#   This function inserts the specified character into the specified window
#   iff the character is part of a valid real number.
#
# Description:
#
# Call:
#       DtcluInsertReal window flag char
#
# Parameters: 
#	(>) window	(tk window) The window.  Should be able to accept
#			the "path insert insert" command (.e.g. entry fields).
#	(>) flag	(string) If non-neg, disallow negative numbers.  
#	(>) char	(string) The character to insert.
#
# Returned Value:
#	Returns 1 if character was handled.  (In Tk 4.0 or higher, a return
#	of 1 indicates binding should not do Entry level binding
#
# Support: Tony Farrell, AAO
#-
#
#
proc DtcluInsertReal { window flag {char ""} } {
    if { ("$char" == "") || ("$char" == "{}") } {
    } elseif { "$char" < " " }  {
	global tk_version
	if { $tk_version < 4.0 } bell
    } else {
        global Dtclu
        set curtext [$window get]
        set end [$window index end]
        set insert [$window index insert]

        if { $insert < $end } {
	    set prefix [string range $curtext 0 [expr ($insert-1)]]
	    set suffix [string range $curtext $insert end]
	    set proposed [join "$prefix $char $suffix" ""]
        } else {
            set proposed [ join "$curtext $char" "" ]
        }


        if { $Dtclu(InsertWindow) == "$window" } {
          incr Dtclu(InsertCount)
        } else {
	  set Dtclu(InsertCount) 1
	  set Dtclu(InsertWindow) $window
        }
	if { ($flag == "non-neg")&&([string index $proposed 0] == "-") } {
	    bell
	} elseif { [Dtclu___PartialParse $proposed] } {
	    $window insert insert $char
	    set Dtclu(InsertCount) 0
	} else {
	    bell
	}

        if { $Dtclu(InsertCount) >= 2 } { 
	    if { $flag == "non-neg" } {
               ErsOut "This Field requires a non-negative real number."
	    } else {
               ErsOut "This Field requires a real number."
	    }
        }
	return 1
    }
    return 0
}
#
#   Dtclu___PartialParse check to see if the specified string represents
#   the begining of a real number.  It returns 1 if yes, 0 if not.
#
proc Dtclu___PartialParse { s } {
    set index 0
    set current [string index $s $index]

#
#   gobble plus/minus sign.
#
    if {($current == "-")||($current == "+")} {
	incr index
        set current [string index $s $index]
    }
#
#   Gobble any numbers
#    
    while { [regexp {^[0-9]$} $current] } {
	incr index
        set current [string index $s $index]
    }
#
#   If we have the end of the string, then we have a valid partical parse
#
    if { $current == "" } {
	return 1
    }

#
#  Now, the only other thing we expect is a decimal point
#
    if { $current != "." } {
	return 0
    } else {
	incr index
        set current [string index $s $index]
    }
#
#   Now gobble up the rest of the numbers after the decimal point
#
    while { [regexp {^[0-9]$} $current] } {
	incr index
        set current [string index $s $index]
    }
    if { $current == "" } {
	return 1
    }

#
#  Next should be a e or E
#
    if { ($current == "e") || ($current == "E")} {
	incr index
        set current [string index $s $index]
    } else {
	return 0
    }
#
#  gobble plus/minus sign.
#
    if {($current == "-")||($current == "+")} {
	incr index
        set current [string index $s $index]
    }
#
#  Gobble up the remaining digits
#
    while { [regexp {^[0-9]$} $current] } {
	incr index
        set current [string index $s $index]
    }
#
#  Should be nothing left
#
    if { $current == "" } {
	return 1
    } else {
	return 0
   }
}
###############################################################################
#^L             D t c l u I n s e r t P o s
#+
# Function Name:
#       DtcluInsertPos
#
# Function:
#   This function inserts the specified character into the specified window
#   iff the result is a valid position in ra or dec
#
#
# Description:
#	Note, if a space is to be accepted as a the separator, then normally
# 	the call must it in quots.
#
# Call:
#       DtcluInsertPos window flag char 
#
# Parameters: 
#	(>) window	(tk window) The window.  Should be able to accept
#			the "path insert insert" command (.e.g. entry fields).
#       (>) flag	(string) either ra or dec.
#	(>) char	(string) The character to insert.
#
# Returned Value:
#	Returns 1 if character was handled.  (In Tk 4.0 or higher, a return
#	of 1 indicates binding should not do Entry level binding
#
# Support: Tony Farrell, AAO
#-
#
#
proc DtcluInsertPos { window flag {char ""} } {

    if {( "$char" == "") || ("$char" == "{}") } {
    } elseif { "$char" < " "}  {
	global tk_version
	if { $tk_version < 4.0 } bell
    } else {
      global Dtclu

	
      set curtext [list [$window get]]
      set end [$window index end]
      set insert [$window index insert]

      if { $insert < $end } {
	set prefix [string range $curtext 0 [expr ($insert-1)]]
	set suffix [string range $curtext $insert end]
	set proposed [join "$prefix $char $suffix" ""]
      } else {
        set proposed [ join "$curtext $char" "" ]
      }

      if { $Dtclu(InsertWindow) == "$window" } {
          incr Dtclu(InsertCount)
      } else {
	  set Dtclu(InsertCount) 1
	  set Dtclu(InsertWindow) $window
      }
      
      set mode "-s2$flag"
      if { ($proposed == "-") && ($flag == "dec") } {
          $window insert insert $char
	  set Dtclu(InsertCount) 0
       } elseif { [catch "Translate \"$proposed\" $mode"] } {
	  bell
      } else {
          $window insert insert $char
	  set Dtclu(InsertCount) 0
      }
      if { $Dtclu(InsertCount) >= 2 } { 
	  if { $flag == "dec" } {
              ErsOut \
"This Field requires a $flag value of the form \"dd mm ss.##.\"  You can use any non-numeric character as the field separator."
	  } else {
              ErsOut \
"This Field requires a $flag value of the form \"hh mm ss.##.\"  You can use any non-numeric character as the field separator."
	  }
      }
      return 1
    }
    return 0
}
###############################################################################
#^L             D t c l u B i n d  E n t r y
#+
# Function Name:
#       DtcluBindEntry
#
# Function:
#	Bind a specified Entry widget to the standard bindings used by Dtclu
#
# Description:
#	This call sets up the default bindings used by DtcluCreateMain
#	for Entry widgets for a specific entry widget, except the 
#	<Any-KeyPress> binding.  It is required if the Entry has redefined
#	<Any-KeyPress> but wants the normal Dtclu behaviour to work otherwise.
#
#	It can also be used by programs not invoking DtcluCreateMain
#	which wish to get the standard bindings for Entry widgets.
#
#	For Tk version less then 4, Dtclu also adds the following binding 
#	for <Any-KeyPress>.
#
#v
#v      bind Entry <Any-KeyPress> {Dtclu___EntrySel %W 1 {
#v          if {"%A" != ""} {
#v              %W insert insert %A
#v          }     
#v       }
#v     }
#
#	Normally you would define Any-KeyPress and any other bindings before
#	callings this function.
#
#
# Call:
#       DtcluBindEntry window 
#
# Parameters: 
#	(>) window	(tk window) The window or "Entry" for all entry widgets.
#
# Support: Tony Farrell, AAO
#-
#
proc DtcluBindEntry { win } {

    global tk_version tk_patchLevel Dtclu

    if { $tk_version < 4.0 } {
        bind $win <Return>    "bell"
        bind $win <Tab>       "bell"
        bind $win <Left>  { Dtclu___EntryMoveChar %W  -1 }
        bind $win <Right> { Dtclu___EntryMoveChar %W   1  }
        bind $win <BackSpace>   \
                         {Dtclu___EntrySel %W 0 "tk_entryBackspace %W"}
        bind $win <Delete>       \
                        {Dtclu___EntrySel %W 0 "tk_entryBackspace %W"}
        bind $win <Control-h>    \
                        {Dtclu___EntrySel %W 0 "tk_entryBackspace %W"}

#       Make button 1 clear the selection before doing the normal tk.tcl stuff.
# 
        bind $win <1> {
            %W select clear
            %W icursor @%x
            %W select from @%x
            if {[lindex [%W config -state] 4] == "normal"} {focus %W}
        }
#       Make button one motion select the selection and reposition the cursor
        bind $win <B1-Motion> {
            %W select to @%x
            %W icursor @%x 
        }

    } elseif { "$win" == "Entry" } {
#	Tk 4 or greater.  
#	Force Entry bindings to take effect 
#	tkEntryBind ""  ;#No longer needed of possible
#	rebind Delete to the same as BackSpace. 
	bind Entry <Delete> [bind  Entry <BackSpace>]
    } elseif { $Dtclu(HaveValidate) == 0 } {
    
#	Set the binding tags such that the window specific bindings are hit
#	before the class bindings so we can avoid doing the Class bindings 
#	if our binding is done.

	bindtags $win "all [winfo toplevel $win] $win Entry"

#	The Mod1 and Alt modified keypresses should be ignored and 
#	Control and Delete should just be passed though to Entry level bindings.
        bind $win <Mod1-KeyPress>  { break}
        bind $win <Alt-KeyPress>   { break }
        bind $win <Control-KeyPress>  { #Just do default stuff }
        bind $win <Delete> { #Just do default stuff }    
#	bind $win <KeyPress-Return> { puts "Return hit" }
    }


}



###############################################################################
#^L             D t c l u B i n d  T e x t
#+
# Function Name:
#       DtcluBindText
#
# Function:
#	Bind a specified Text widget to the standard bindings used by Dtclu
#
# Description:
#	This call sets up the default bindings used by DtcluCreateMain
#	for Text widgets for a specific text widget, except the 
#	<Any-KeyPress> binding.  It is required if the text widget redefined
#	<Any-KeyPress> but wants the normal Dtclu behaviour to work otherwise.
#
#	It can also be used by programs not invoking DtcluCreateMain
#	which wish to get the standard bindings for  Text widgets.
#
#	For Tk version less then 4, Dtclu also adds the following binding 
#	for <Any-KeyPress>.
#
#V
#V       bind Text <Any-KeyPress> {Dtclu___TextSel %W 1 {
#V           if {"%A" != ""} {
#V               %W insert insert %A
#V           }
#V        }
#V      }
#
#	Normally you would define Any-KeyPress and any other bindings before
#	callings this function.
#
#
# Call:
#       DtcluBindText window 
#
# Parameters: 
#	(>) window	(tk window) The window or "Text" for all text widgets
#
# Support: Tony Farrell, AAO
#-
#
proc DtcluBindText { win } {

    global tk_version 
    if { $tk_version < 4.0 } {
        bind $win <Left>  { Dtclu___MoveChar %W 0 -1 }
        bind $win <Right> { Dtclu___MoveChar %W 0 1  }
        bind $win <Down>  { Dtclu___MoveChar %W 1 0  }
        bind $win <Up>    { Dtclu___MoveChar %W -1 0 }
        bind $win <Return>       {Dtclu___TextSel %W 1 {%W insert insert \n}}
        bind $win <BackSpace>    {Dtclu___TextSel %W 0 "tk_textBackspace %W"}
        bind $win <Delete>       {Dtclu___TextSel %W 0 "tk_textBackspace %W"}
        bind $win <Control-h>    {Dtclu___TextSel %W 0 "tk_textBackspace %W"}
   

#
#       Make button 2 insert the select in Text widgets
#
        bind $win <2>	 { 
    	    catch {%W insert insert [selection get STRING]}
	    %W yview -pickplace insert
        }
#
#       Make Shift-button 2 clear the selection
#
        bind $win <Shift-2>  { catch {selection clear %W} }

#       Make button one clear the selection before doing the normal tk.tcl
#       stuff.  Do the same for entry widets.
        bind $win <1> {
            %W tag remove sel 0.0 end
            set tk_priv(selectMode) char
            %W mark set insert @%x,%y
            %W mark set anchor insert
           if {[lindex [%W config -state] 4] == "normal"} {focus %W}
        }

    } elseif { "$win" == "Text" } {
#	Tk 4 or greater.  

#	Force Text bindings to take effect 
#	tkTextBind ""	;# No longer needed for possible
#	rebind Delete to the same as BackSpace.  
	bind Text <Delete> [bind  Text <BackSpace>]
    }
}



###############################################################################
#^L             D t c l u B i n d  E n t r y I n t
#+
# Function Name:
#       DtcluBindEntryInt
#
# Function:
#	Bind a specified Integer Entry widget to the standard Dtclu bindings.
#
# Description:
#	Sets up bindings for an Entry widget intended to only accept integers.
#	First sets a binding such that key presses are vetted by 
#	DtcluInsertInt.    It then ensures the standard Dtclu bindings are
#	restored by calling DtcluBindEntry.
#	
#
# Call:
#       DtcluBindEntryInt window  min max
#
# Parameters: 
#	(>) window	(tk window) The window name.
#	(>) min		(integer) The minimum acceptable number. This is only
#			used if it is less then zero since we cannot do
#			partial checks on a minimum size > 0.  Otherwise it is
#			only used in error messages.
#	(>) max		(integer) The maximum acceptable number.  Only used
#			is greater then zero since we cannot do partial
#			checks on a maximum size less then zero.  Otherwise it
#			is only used in error messages.
##
# Support: Tony Farrell, AAO
#-
#
proc DtcluBindEntryInt { win min max } {

    global tk_version  tk_patchLevel Dtclu

    if { $tk_version < 4.0 } {
        bind $win <Any-KeyPress> \
            "Dtclu___EntrySel %W 1 \"DtcluInsertInt  %W $min $max %A\""

        bind $win <2> \
 	" DtcluInsertInt %W $min $max \[selection get STRING\] ;\
	  tk_entrySeeCaret %W "
        DtcluBindEntry $win
    } elseif { $Dtclu(HaveValidate) == 0 } {

#	First, save the min and max values.
	global Dtclu
	set Dtclu($win-min) $min
	set Dtclu($win-max) $max

#	Default bindings
	DtcluBindEntry $win

#	Keypresses must be veted by DtcluInsertInt.
        bind $win <KeyPress>  {
	    if {[DtcluInsertInt  %W $Dtclu(%W-min) $Dtclu(%W-max) "%A"]} {
		 break
	    }
	}

#	Pasting the selection. We must to some extend copy the default
#	<ButtonRelease-2> binding before veting the results using
#	DtcluInsertInt.

	bind $win <ButtonRelease-2> {	
	    if !$tkPriv(mouseMoved) { 	
		if {[catch {set selection [selection get -displayof %W]}]== 0} {
	            if {[DtcluInsertInt %W $Dtclu(%W-min)  \
					$Dtclu(%W-max) $selection]} {
	                tkEntrySeeInsert %W
	                break;
		    }
	        }
	    }
	}

#	Pasting the clipboard. As per above but use Selection CLIPBOARD.
#	Also we don't have to worry about mouse movement.

	bind $win <F18> {	
		if {[catch {set selection [selection get -displayof %W \
			-selection CLIPBOARD ]}]== 0} {
	            if {[DtcluInsertInt %W $Dtclu(%W-min)  \
					$Dtclu(%W-max) $selection]} {
	                tkEntrySeeInsert %W
	                break;
		    }
	        }
	}

    } else {
    	$win configure -validate key -validatecommand "Dtclu___ValidateInt %W $min $max %P"
    }
} 

###############################################################################
#^L             D t c l u B i n d  E n t r y R e a l
#+
# Function Name:
#       DtcluBindEntryReal
#
# Function:
#      Bind a specified Real Number Entry widget to the standard Dtclu bindings.
#
# Description:
#	Sets up bindings for an Entry widget intended to only accept real 
#	numbers.
#	First sets a binding such that key presses are vetted by 
#	DtcluInsertReal.    It then ensures the standard Dtclu bindings are
#	restored by calling DtcluBindEntry.
#	
#
# Call:
#       DtcluBindEntryReal window flag
#
# Parameters: 
#	(>) window	(tk window) The window name.
#	(>) flag	(string) If non-neg, disallow negative numbers.  
##
# Support: Tony Farrell, AAO
#-
#
proc DtcluBindEntryReal { win {flag ""}} {


    global tk_version tk_patchLevel Dtclu
#   NOTE, WE currently assume that from tk 4.0, patch level 2, the 
#   -validatecommand option is available to entry widgets.

    if { $tk_version < 4.0 } {
        bind $win <Any-KeyPress> \
            "Dtclu___EntrySel %W 1 \"DtcluInsertReal  %W $flag %A\""

        bind $win <2> \
 	    " DtcluInsertReal %W $flag \[selection get STRING\] ; \
  	    tk_entrySeeCaret %W "
        DtcluBindEntry $win
    } elseif { $Dtclu(HaveValidate) == 0 } {
#	First, save the min and max values.
	global Dtclu
	set Dtclu($win-flag) $flag

#	Default bindings
	DtcluBindEntry $win


#	Keypresses must be veted by DtcluInsertReal. 
        bind $win <KeyPress>  {
	    if {[DtcluInsertReal  %W $Dtclu(%W-flag) %A]} {
		 break
	    }
	}

#	Pasting the selection. We must to some extend copy the default
#	<ButtonRelease-2> binding before veting the results using
#	DtcluInsertInt.

	bind $win <ButtonRelease-2> {	
	    if !$tkPriv(mouseMoved) { 	
		if {[catch {set selection [selection get -displayof %W]}]== 0} {
	            if {[DtcluInsertReal %W $Dtclu(%W-flag) $selection]} {
	                tkEntrySeeInsert %W
	                break;
		    }
	        }
	    }
	}

#	Pasting the clipboard. As per above but use Selection CLIPBOARD.
#	Also we don't have to worry about mouse movement.

	bind $win <F18> {	
		if {[catch {set selection [selection get -displayof %W \
			-selection CLIPBOARD ]}]== 0} {
	            if {[DtcluInsertReal %W $Dtclu(%W-flag) $selection]} {
	                tkEntrySeeInsert %W
	                break;
		    }
	        }
	}
    } else {
    	$win configure -validate key -validatecommand "Dtclu___ValidateReal %W $flag %P"
    }
}
###############################################################################

#^L             D t c l u B i n d  E n t r y P o s
#+
# Function Name:
#       DtcluBindEntryPos
#
# Function:
#      Bind a specified Position Entry widget to the standard Dtclu bindings.
#
# Description:
#	Sets up bindings for an Entry widget intended to only accept 
#	positions in RA (hh:mm:ss.xx) or Dec (dd:mm:ss.xx)
#
#	First sets a binding such that key presses are vetted by 
#	DtcluInsertPos.    It then ensures the standard Dtclu bindings are
#	restored by calling DtcluBindEntry.
#
#
# Call:
#       DtcluBindEntryPos window flag
#
# Parameters: 
#	(>) window	(tk window) The window name.
#	(>) flag	(string) Either ra or dec.
##
# Support: Tony Farrell, AAO
#-
#
proc DtcluBindEntryPos { win flag } {

    global tk_version tk_patchLevel Dtclu

    if { $tk_version < 4.0 } {

        if { $flag == "ra" } {
            bind $win <Any-KeyPress> \
                "Dtclu___EntrySel %W 1  { DtcluInsertPos  %W ra \"%A\"}"
        } else {
            bind $win <Any-KeyPress> \
                "Dtclu___EntrySel %W 1  { DtcluInsertPos  %W dec \"%A\"}"
        }


        bind $win <2> \
	   "DtcluInsertPos %W $flag \[selection get STRING\] ; \
	   tk_entrySeeCaret %W "

        DtcluBindEntry $win
    } elseif { "$Dtclu(HaveValidate)" == 0 } {
#	First, save the min and max values.
	global Dtclu
	set Dtclu($win-flag) $flag

#	Default bindings
	DtcluBindEntry $win


#	Keypresses must be veted by DtcluInsertPos"
        bind $win <KeyPress>  {
	    #puts "Pos key press, keycode = %K, key = \"%A\""
            if {[DtcluInsertPos  %W $Dtclu(%W-flag) "%A"]} {
                 break
            }
	}
	
	
#	Pasting the election. We must to some extend copy the default
#	<ButtonRelease-2> binding before veting the results using
#	DtcluInsertInt.
	
	bind $win <ButtonRelease-2> {	
	    if !$tkPriv(mouseMoved) { 	
		if {[catch {set selection [selection get -displayof %W]}]== 0} {
	            if {[DtcluInsertPos %W $Dtclu(%W-flag) "$selection"]} {
	                tkEntrySeeInsert %W
	                break;
		    }
	        }
	    }
	}

#	Pasting the clipboard. As per above but use Selection CLIPBOARD.
#	Also we don't have to worry about mouse movement.

	bind $win <F18> {	
		if {[catch {set selection [selection get -displayof %W \
			-selection CLIPBOARD ]}]== 0} {
	            if {[DtcluInsertPos %W $Dtclu(%W-flag) "$selection"]} {
	                tkEntrySeeInsert %W
	                break;
		    }
	        }
	}
    } else {
        if { $flag == "ra" } {
      	    $win configure -validate key -validatecommand "Dtclu___ValidateTime %W %P"
      	} else  {
      	    $win configure -validate key -validatecommand "Dtclu___ValidateDegrees %W %P"
      	}
    }
} 





#
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # 
#  Validation routines.  These routines are used to validate Entry field
#  in the version of Tk where we have the -validatecommand option available.
#
#  Versions are
#	Dtclu___ValidateInt	-> Validates an integer field
#	Dtclu___ValidateRead	-> Validates a real number field
#	Dtclu___ValidateTime	-> Validates a time field
#	Dtclu___ValidateDegrees -> Validates a degrees field
#
#
#
proc Dtclu___ValidateInt {window min max val} {

# Handle technqiue of putting up message on second error in same window.
  global Dtclu
  if { $Dtclu(InsertWindow) == "$window" } {
      incr Dtclu(InsertCount)
  } else {
      set Dtclu(InsertCount) 1
      set Dtclu(InsertWindow) $window
  }
  if { $val == "" } {
  	return 1
  } elseif { (($min < 0) && ([regexp -nocase {^[-+]?[0-9]*$} $val] ==  1)) ||
       ([regexp -nocase {^[0-9]*$} $val] ==  1) } {
      if { ($val != "-") && ($min < 0) && ($val < $min)} {
         # Value too small
      } elseif { ($val != "-") && ($val != "+")&& ($max > 0) && ($val > $max)} {
         # Value too large
      } else {
          set Dtclu(InsertCount) 0
          return 1
      }
  }

  if { $Dtclu(InsertCount) >= 2 } {
      ErsOut "This field requires an integer value between $min and $max"
  }
  bell
  return 0
}

proc Dtclu___ValidateReal {window flag val} {
  global Dtclu
  if { $Dtclu(InsertWindow) == "$window" } {
      incr Dtclu(InsertCount)
  } else {
      set Dtclu(InsertCount) 1
      set Dtclu(InsertWindow) $window
  }
  if {($flag == "non-neg")&&([string index $val 0] == "-")} {
      # non-neg but negative
  } elseif {[regexp -nocase {^[-+]?[0-9]*\.?[0-9]*([0-9]\.?e[-+]?[0-9]*)?$} $val] == 1 } {
        set Dtclu(InsertCount) 0
        return 1
  }
  if { $Dtclu(InsertCount) >= 2 } {
      if { $flag == "non-neg" } {
          ErsOut "This field requires a non-negative real value."
      } else {
          ErsOut "This field requires an real value."
      }
  }
  bell
  return 0
}

# Time field validation.
proc Dtclu___ValidateTime {window val} {
 
  global Dtclu
  if { $Dtclu(InsertWindow) == "$window" } {
      incr Dtclu(InsertCount)
  } else {
      set Dtclu(InsertCount) 1
      set Dtclu(InsertWindow) $window
  }
  if { ($val == "") || ([catch "Translate \"$val\" -s2ra"] == 0) } {
      set Dtclu(InsertCount) 0
      return 1
  } elseif { $Dtclu(InsertCount) >= 2 } {
      ErsOut "This field requires a time value of the form \"hh mm ss.##.\"  You can use any non-numeric character as the field separator."
  }
  bell
  return 0
}

# Degrees field.
proc Dtclu___ValidateDegrees {window val} {
 
  global Dtclu
  if { $Dtclu(InsertWindow) == "$window" } {
      incr Dtclu(InsertCount)
  } else {
      set Dtclu(InsertCount) 1
      set Dtclu(InsertWindow) $window
  }
  if { $val == "-" } {
      set Dtclu(InsertCount) 0
      return 1
  }
  if {  ($val == "") || ([catch "Translate \"$val\" -s2dec"] == 0) } {
      set Dtclu(InsertCount) 0
      return 1
  } elseif { $Dtclu(InsertCount) >= 2 } {
      ErsOut "This field requires a degree value of the form \"ddd mm ss.##.\"  You can use any non-numeric character as the field separator."
  }
  bell
  return 0
}



###############################################################################
#^L             D t c l u F D i a l o g
#+
# Function Name:
#       DtcluFDialog
#
# Function:
#	Create a file dialog
#
# Description:
#	This command creates a File dialog box.  The dialog is actually a
#	copy of the Unix Tk file dialog (not the Motif) one, which is itself
#	a clone of the Windows file dialog.
#
#	The reason for the existance of this command is the requirement in
#	DRAMA applications to avoid use of tkwait.  Instead of doing grab
#	and blocking until the user provides input, this command allows
#	you to provide a command to be invoked when the use selects the file.
#
#	The implementation of this command uses many procedures defined by
#	the Unix Tk Tcl library, in the file tkfbox.tcl  IF that changes
#	significatly, this command will require modification.
#
#	This command adds ballon help to the dialog if Tix is available.
#
# Call:
#       DtcluFDialog type command [options...]
#
# Parameters: 
#	(>) type (string)  Specify "open" to open an existing file or
#				"save" to save a file.  The later mode will
#				automatically prompt for confirmation is you
#				are going to overwrite an existing file.
#	(>) command (string) Command to be executed when the user has selected
#				a file.  The name of the file will be appended.
#				if no file was selected, the command is
#				never executed.
#
# Options:
#	See the Tk manual entry getOpenFile for details of options.
##
# Support: Tony Farrell, AAO
#-
#
proc DtcluFDialog {type cmd args} {
    global tkPriv
    set w __tk_filedialog
    upvar #0 $w data

    tkFDialog_Config $w $type $args

    if {![string compare $data(-parent) .]} {
        set w .$w
    } else {
        set w $data(-parent).$w
    }

    # (re)create the dialog box if necessary
    #
    if {![winfo exists $w]} {
	tkFDialog_Create $w
    } elseif {[string compare [winfo class $w] TkFDialog]} {
	destroy $w
	tkFDialog_Create $w
    } else {
	set data(dirMenuBtn) $w.f1.menu
	set data(dirMenu) $w.f1.menu.menu
	set data(upBtn) $w.f1.up
	set data(icons) $w.icons
	set data(ent) $w.f2.ent
	set data(typeMenuLab) $w.f3.lab
	set data(typeMenuBtn) $w.f3.menu
	set data(typeMenu) $data(typeMenuBtn).m
	set data(okBtn) $w.f2.ok
	set data(cancelBtn) $w.f3.cancel
    }
    wm transient $w $data(-parent)

    # 5. Initialize the file types menu
    #
    if {$data(-filetypes) != {}} {
	$data(typeMenu) delete 0 end
	foreach type $data(-filetypes) {
	    set title  [lindex $type 0]
	    set filter [lindex $type 1]
	    $data(typeMenu) add command -label $title \
		-command [list tkFDialog_SetFilter $w $type]
	}
	tkFDialog_SetFilter $w [lindex $data(-filetypes) 0]
	$data(typeMenuBtn) config -state normal
	$data(typeMenuLab) config -state normal
    } else {
	set data(filter) "*"
	$data(typeMenuBtn) config -state disabled -takefocus 0
	$data(typeMenuLab) config -state disabled
    }

    tkFDialog_UpdateWhenIdle $w

    # 6. Withdraw the window, then update all the geometry information
    # so we know how big it wants to be, then center the window in the
    # display and de-iconify it.

    wm withdraw $w
    update idletasks
    set x [expr {[winfo screenwidth $w]/2 - [winfo reqwidth $w]/2 \
	    - [winfo vrootx [winfo parent $w]]}]
    set y [expr {[winfo screenheight $w]/2 - [winfo reqheight $w]/2 \
	    - [winfo vrooty [winfo parent $w]]}]
    wm geom $w [winfo reqwidth $w]x[winfo reqheight $w]+$x+$y
    wm deiconify $w
    wm title $w $data(-title)

    # 7. Claim the focus too.

    focus $data(ent)
    $data(ent) delete 0 end
    $data(ent) insert 0 $data(selectFile)
    $data(ent) select from 0
    $data(ent) select to   end
    $data(ent) icursor end


   # 8. Trace the variable change.  When the variable has changed,
   #    the trace routine will invoke the user's command.   
   trace variable tkPriv(selectFilePath) w  [list Dtclu__FDialogTrace $cmd]
}

#
# Old version of above, which clearly prior to 8.0.3.  Kepts as I am not
# not sure if the above version works prior to 8.0.3.
#
if { 0 } {
proc DtcluFDialog {type cmd args} {
    global tkPriv
    set w .__tk_filedialog
    upvar #0 $w data

    tkFDialog_Config $w $type $args

    # (re)create the dialog box if necessary
    #
    set created 0
    if {![winfo exists $w]} {
	tkFDialog_Create $w
    	set created 1
    } elseif {[string compare [winfo class $w] TkFDialog]} {
	destroy $w
	tkFDialog_Create $w
    	set created 1
    }
    
    
    wm transient $w $data(-parent)

    # 5. Initialize the file types menu
    #
    if {$data(-filetypes) != {}} {
	$data(typeMenu) delete 0 end
	foreach type $data(-filetypes) {
	    set title  [lindex $type 0]
	    set filter [lindex $type 1]
	    $data(typeMenu) add command -label $title \
		-command [list tkFDialog_SetFilter $w $type]
	}
	tkFDialog_SetFilter $w [lindex $data(-filetypes) 0]
	$data(typeMenuBtn) config -state normal
	$data(typeMenuLab) config -fg [$data(typeMenuBtn) cget -fg]
    } else {
	set data(filter) "*"
	$data(typeMenuBtn) config -state disabled
	$data(typeMenuLab) config -fg \
	    [$data(typeMenuBtn) cget -disabledforeground]
    }

    tkFDialog_UpdateWhenIdle $w

    # 6. Withdraw the window, then update all the geometry information
    # so we know how big it wants to be, then center the window in the
    # display and de-iconify it.

    wm withdraw $w
    update idletasks
    set x [expr [winfo screenwidth $w]/2 - [winfo reqwidth $w]/2 \
	    - [winfo vrootx [winfo parent $w]]]
    set y [expr [winfo screenheight $w]/2 - [winfo reqheight $w]/2 \
	    - [winfo vrooty [winfo parent $w]]]
    wm geom $w [winfo reqwidth $w]x[winfo reqheight $w]+$x+$y
    wm deiconify $w
    wm title $w $data(-title)

    # 7. claim the focus.

    focus $data(ent)
    $data(ent) delete 0 end
    $data(ent) insert 0 $data(selectFile)
    $data(ent) select from 0
    $data(ent) select to   end
    $data(ent) icursor end

   # 8. Trace the variable change.  When the variable has changed,
   #    the trace routine will invoke the user's command.   
   trace variable tkPriv(selectFilePath) w  [list Dtclu__FDialogTrace $cmd]
}
}

#
# File dialog variable trace routine.
#
proc Dtclu__FDialogTrace { command name1 name2 op } {

    global tkPriv
    
#   Remove the trace.
    trace vdelete tkPriv(selectFilePath) w  [list Dtclu__FDialogTrace $command]


#   Withdraw the window and invoke the user's command.

    wm withdraw .__tk_filedialog

    if { $tkPriv(selectFilePath) != "" } {
        set filedir [file dirname $tkPriv(selectFilePath)]
        cd $filedir
                     
        global errorInfo
        set Caught [catch {uplevel #0 "$command $tkPriv(selectFilePath)"} message]
        set savedInfo $errorInfo
        if { $Caught } {
            bgerror $message 
	}
    }
}



###############################################################################
#
#+                      D t c l u D i a l o g P o s S e t
#
# Function name:
#    DtcluDialogPosSet
#
# Function:
#      Set the initial position of a dialog.
#
# Description:
#	Given a created dialog, this command will set the position of
#	the dialog such that it is centred on the screen.
#      
# Language:
#    TCL
#
# Call:
#   DtcluDialogPosSet window
#
# Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
#	(>) window (tk_window) The name of the dialog top-level window.
#      
#
# Support: Tony Farrell, AAO
#
#-
#
# History:
#    03-May-1995 - TJF - Original version
#
#

global Dtclu

# Save screen size and centre for future use.  Note, we drop 40 of the
# size, since this is the value we actually want.
#
set Dtclu(Screen,x) [expr [winfo screenwidth .] - 40]
set Dtclu(Screen,y) [expr [winfo screenheight .] - 40]
set Dtclu(ScreenCentre,x) [expr $Dtclu(Screen,x)/2]
set Dtclu(ScreenCentre,y) [expr $Dtclu(Screen,y)/2]

proc DtcluDialogPosSet { w } {

#   Withdraw the window and force an update so that the sizes are valid.
    wm withdraw $w
    update idletasks
    global Dtclu
#   Workout the x and y.
    set x [expr $Dtclu(ScreenCentre,x) -  [winfo reqwidth $w]/2]
    set y [expr $Dtclu(ScreenCentre,y) -  [winfo reqheight $w]/2]

#   Set position and make window appear.
    wm geom $w +$x+$y
    wm deiconify $w
    raise $w

}


#
#+                      D t c l u D i a l o g P o s R e s e t
#
# Function name:
#    DtcluDialogPosReset
#
# Function:
#	Ensure a dialog is on the screen.
#
# Description:
#	Given a created dialog, this command will ensure the dialog
#	position allows it to be viewed.  If it is not viewable, then
#	the dialog is centred.  The dialog is deiconified if necessary
#	and raised to the top of the stack.
#      
# Language:
#    TCL
#
# Call:
#   DtcluDialogPosReset window
#
# Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
#	(>) window (tk_window) The name of the dialog top-level window.
#      
#
# Support: Tony Farrell, AAO
#
#-
#
# History:
#    03-May-1995 - TJF - Original version
#
#

proc DtcluDialogPosReset { w } {

#   Get current geometry of the window.
    set dialog_geom   [winfo geometry $w]
    scan $dialog_geom "%dx%d+%d+%d" dialog_width dialog_height dialog_x dialog_y

#   Workout the bottom right position
    set dialog_bottom [expr $dialog_height + $dialog_y ]
    set dialog_right  [expr $dialog_width + $dialog_x ]
    
    global Dtclu        
#   IF outside the screen (or only 40 pixels inside), then reset position.
    if { ($dialog_bottom < 40) || ($dialog_right < 40) ||
         ($dialog_x > $Dtclu(Screen,x)) || ($dialog_y > $Dtclu(Screen,y)) }  {
	DtcluDialogPosSet $w	
    } else {
#	Deiconify and raise the window.
        wm deiconify $w
	raise $w
    }

}



# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # 
#		D t c l T k D i a l o g
#+
# Function Name:
#	DtclTkDialog
#
# Function:
#    Creates a dialog box containing a bitmap, message and one or more buttons
#
# Description:
#	This procedure is similar to tk_dialog, but is optimized for use
#	in Dtcl/Dtk programs.   In particular, is does not use tk_wait, which
#	can cause problems.  Instead, you must supply a command which will
#	be invoked when a button is pressed.
#
#
# Call:
#	DtclTkDialog w title text [bitmap] default command arg1 [args...]
#
# Parameters: 
# 	(>) w		(tk window name) Window to use for dialog top-level.
#			If the window name is prefixed with !, then the format
#			is !class.w where class is the class name to be given
#			to the dialog and .w is the window name.
# 	(>) title 	(string) Title to display in dialog's decorative frame.
# 	(>) text	(string) Message to display in dialog.
# 	(>) bitmap 	(tk bitmap name) Bitmap to display in dialog. An empty 
#			string means none.
# 	(>) default 	(integer) Index of button that is to display the 
#			default ring.  Use -1 to indicate no default.
# 	(>) command 	(string) Command to be invoked when a button is pressed.
#			It will have button number appended.
# 	(>) args 	(strings) One or more strings to display in buttons 
#			across the bottom of the dialog box.  By default,
#			the dialog is destroyed when the button is pressed.
#			If button name starts with a dash (-), then the dash 
#			is removed and the dialog is not destroyed when the 
#			button is pressed.  
#
# See Also: DTCL manual, tk_dialog(n).
#
# Support: Tony Farrell, AAO
#-
#
proc DtclTkDialog {w title text bitmap default command args} {

    global utils_font_2     ;# Font used in dialogues

#   First, check if user has specified a class name for this window.
    if {[string range $w 0 0 ] == "!" } {
	set period [string first "." $w]
	if { $period == -1 } {
	    error "Invalid window name $w"
	}
        set class [string range $w 1 [expr {$period-1}]]
	set w     [string range $w $period end]
    } else {
	set class Dialog
    }

    # 1. Create the top-level window and divide it into top
    # and bottom parts.

    catch {destroy $w}
    toplevel $w -class $class
    wm title $w $title
    wm iconname $w Dialog
    wm iconbitmap $w [wm iconbitmap .]

    frame $w.top -relief raised -bd 1
    pack $w.top -side top -fill both
    frame $w.bot -relief raised -bd 1
    pack $w.bot -side bottom -fill both

    # 2. Fill the top part with bitmap and message.

    global tk_version
    if { $tk_version < 4.0 } {
        message $w.msg -width 4i -text $text \
	    -font $utils_font_2
#	    -font -Adobe-Times-Medium-R-Normal-*-180-*
        pack $w.msg -in $w.top -side right -expand 1 -fill both -padx 5m -pady 5m
        set pad1 3m
    } else {
        label $w.msg -wraplength 6i -justify left -text $text \
            -font $utils_font_2
#            -font -Adobe-Times-Medium-R-Normal--*-180-*-*-*-*-*-*
        pack $w.msg -in $w.top -side right -expand 1 -fill both -padx 3m -pady 3m
        set pad1 5m
    }
    if {$bitmap != ""} {
	label $w.bitmap -bitmap $bitmap
	pack $w.bitmap -in $w.top -side left -padx $pad1 -pady $pad1
    }

    # 3. Create a row of buttons at the bottom of the dialog.

    set i 0
    foreach but $args {
	set firstchar [string range $but 0 0]
	if { $firstchar == "-" } {
	    set but [string range $but 1 end]
	    button $w.button$i -text $but -command "$command $i" \
                        -padx 0 -pady 0
	} else {
	    button $w.button$i -text $but -command "destroy $w ; $command $i" \
                        -padx 0 -pady 0
        }
	if {$i == $default} {
	    frame $w.default -relief sunken -bd 1
	    raise $w.button$i $w.default
	    pack $w.default -in $w.bot -side left -expand 1 -padx 3m -pady 2m
	    pack $w.button$i -in $w.default -padx 2m -pady 2m \
		    -ipadx 2m -ipady 1m
            if { $tk_version < 4.0 } {
	       bind $w <Return> "$w.button$i flash ; destroy $w ; $command $i"
	    }
	} else {
	    pack $w.button$i -in $w.bot -side left -expand 1 \
		    -padx 1m -pady 1m -ipadx 2m -ipady 1m
	}
	incr i
    }

    # 4. Withdraw the window, then update all the geometry information
    # so we know how big it wants to be, then center the window in the
    # display and de-iconify it.

    wm withdraw $w
    update idletasks
    set x [expr [winfo screenwidth $w]/2 - [winfo reqwidth $w]/2 \
	    - [winfo vrootx [winfo parent $w]]]
    set y [expr [winfo screenheight $w]/2 - [winfo reqheight $w]/2 \
	    - [winfo vrooty [winfo parent $w]]]

#   The above may come out negative.  wm gemo will accept -x-y but not
#   +-x and +-y  (at least, it won't any more, it may have in the past).
#   There, we must set up the x and y values correctly 

    if { $x >= 0 } { set x "+[set x]" }
    if { $y >= 0 } { set y "+[set y]" }
    wm geom $w $x$y
    wm deiconify $w

    # I think we could get into trouble if some other window has the
    # grab at this point.  So try and release it.
    if { $tk_version >= 4.0 } {
        set cur_grab [grab current .]
        if { $cur_grab != "" } {
            puts "DtclTkDialog releasing current grab \"$cur_grab\""
           grab release $cur_grab
        }
    }

    if { ($tk_version >= 4.0) && ($default >= 0) } {
    	focus $w.button$default
    } else {
        focus $w
    }

}


#
# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # 
#		t k e r r o r
#
# Function Name:
#	tkerror
#
# Function:
#	An implementation of a background error handler
#
# Description:
#	This routine will be invoked when a background Dtcl error occurs.
#	It pops up a dialog with the error message and allows the user to
#	display a stack dump.  
#  
#       Unlike the standard tkerror producure, this one does not force
#       a response before allowing the program to continue.
#
# Call:
#	trror errmsg
#
# Parameters: 
#	(>) errmsg	(string) The error message.
#
# Support: Tony Farrell, AAO
#
#
global DtclErrorCnt
set DtclErrorCnt 0

proc tkerror err {

    global DtclErrorCnt
    incr DtclErrorCnt

    global errorInfo

#   Store the additional error information in a global variable for passing
#   to the stack display.

    global DtclErrorInfo$DtclErrorCnt
    set  DtclErrorInfo$DtclErrorCnt $errorInfo

#
#   We use DtclTkDialog to display the message.  The procedure below, 
#   Dtcl__ErrorAck is used to handle button presses.  Note that it is possible
#   for this function to be invoked before the X-window associated with
#   the . has been created, so we check.
#

    if { [winfo id .] != 0 } {
        DtclTkDialog !ProgErrorDialog.dtclerrorDialog$DtclErrorCnt \
	     "Error ($DtclErrorCnt)" \
   	        "Error: $err" error 0 \
	        "Dtcl___ErrorAck \"$DtclErrorCnt\"" OK \
	        "See Stack Trace" "Clear All" "-Help"
   } else {
	    puts stderr "Error:$err"
	    puts stderr "     :$errorInfo"
   }
}


#
#  Invoked when a button is pressed.  The count indicates the variable
#  the stack dump is in while button indicates which button.  Only button
#  1 produces a stack dump.  Button 2  pops up help.
#

proc Dtcl___ErrorAck { count button } {

    if { $button == 3 } {
	Dtcl___ErrorHelpDialog $count
	return ""	
    } elseif { $button == 2 } { 
	Dtcl___ErrorClearAll 
    } elseif { $button == 1 } { 
	Dtcl___ErrorStack $count
	return ""
    }
}
proc Dtcl___ErrorClearAll {} {
    global DtclErrorCnt
    for { set i 1 } { $i <= $DtclErrorCnt } { incr i } {
	if [winfo exists .dtclerrorDialog$i ] {
	    destroy .dtclerrorDialog$i
        }
    }
}

proc Dtcl___ErrorStack { count } {
#    global DtclErrorInfo$count
    upvar #0 DtclErrorInfo$count info
 

    set w .dtclerrorTrace$count
    catch {destroy $w}
    toplevel $w -class ErrorTrace
    wm minsize $w 1 1
    wm title $w "Stack Trace for Error ($count)"
    wm iconname $w "Stack Trace"
    wm iconbitmap $w [wm iconbitmap .]

    button $w.ok -text OK -command "destroy $w"
    text $w.text -relief raised -bd 2 -yscrollcommand "$w.scroll set" \
	    -setgrid true -width 40 -height 10
    scrollbar $w.scroll -relief flat -command "$w.text yview"
    pack $w.ok -side bottom -padx 3m -pady 3m -ipadx 2m -ipady 1m
    pack $w.scroll -side right -fill y
    pack $w.text -side left -expand yes -fill both
    $w.text insert 0.0 $info

    $w.text mark set insert 0.0

    # Center the window on the screen.

    wm withdraw $w
    update idletasks
    set x [expr [winfo screenwidth $w]/2 - [winfo reqwidth $w]/2 \
	    - [winfo vrootx [winfo parent $w]]]
    set y [expr [winfo screenheight $w]/2 - [winfo reqheight $w]/2 \
	    - [winfo vrooty [winfo parent $w]]]

#   The above may come out negative.  wm gemo will accept -x-y but not
#   +-x and +-y  (at least, it won't any more, it may have in the past).
#   There, we must set up the x and y values correctly 

    if { $x >= 0 } { set x "+[set x]" }
    if { $y >= 0 } { set y "+[set y]" }
    wm geom $w $x$y
    wm deiconify $w

    # I think we could get into trouble if some other window has the
    # grab at this point.  So try and release it.
    global tk_version
    if { $tk_version >= 4.0 } {
        set cur_grab [grab current .]
        if { $cur_grab != "" } {
            puts "DtclTkDialog releasing current grab \"$cur_grab\""
           grab release $cur_grab
        }
    }
 

}


global Dtcl___HelpMessage
set Dtcl___HelpMessage "This dialog appears when a background error occurs. \
    This type of error normally indicates a programming error in the \
    application and support staff should be notified.  \n\nPress \"OK\" to \
    continue or \"See Stack Trace\" if you want the full details (which \
    are probably only of use to programmers).  Press \"Clear All\" to remove \
    all such dialogs from this application."


proc Dtcl___ErrorHelpDialog { count } {

    global Dtcl___HelpMessage
    DtclTkHelpDialog .dtclerrorDialogHelp$count \
	"Help on Dialog" $Dtcl___HelpMessage
    return ""
}


proc DtclTkHelpDialog {w title text } {
    DtclTkDialog !HelpDialog$w $title $text info 0 "#" Close
}


