#*******************************************************************************
# E.S.O. - VLT project
#
# "@(#) $Id: mimic.tcl 264656 2015-02-07 21:50:36Z pbaksai $"
#
# who       when        what
# --------  ----------  --------------------------------------------------------
# pbaksai   2015/02/07  CCB-000755: Support for 64 bits.
#                       - Fixed font XLFD names.
# rsc/cme   2010-12-27  PPRS-38081: VLT_GuideMinMag=11 (previous=13).
#                       VLT_GuideMaxMag=9  (previous=11).
# rschmutz  2004-07-15  PPRS13442: VLT_GuideMaxMag=11 (previous=9).
# rschmutz  28/01/03    PPRS8649: DrawAFibre seems to be called too soon
#                       when option -f is used: ignore errors.
# rschmutz  28/01/03    this header created - previous history see below.
#
#
#                           M i m i c . t c l
#   Filename:
#       mimic.tcl
#
#   Function:
#       Tcl script for the AAO fibre instrument configuration mimic window.
#
#   Description:
#        This file contains the tcl procedures that support the mimic display
#        window used by the AAO fibre configuration program 'configure'. It
#        is loaded by configure.tcl and works in conjunction with the 
#        procedures defined in that file. The split between the two files
#        is a little hazy in places and the two work quite closely together -
#        there has been no attempt to make this file a set of self-contained
#        routines that properly encapsulate the mimic window. Many of the
#        global variables used by this file are also used by configure.tcl.
#
#   Authors:
#       Jeremy Bailey, AAO. (JAB)
#       Gavin Dalton, University of Oxford (GBD)
#       Tony Farrell, AAO (TJF)
#       Keith Shortridge, AAO (KS)  
#
#   Copyright (c)  Anglo-Australian Telescope Board, 1995 - 2001.
#   Permission granted for use for non-commercial purposes.
#
# Sccs Id:     mimic.tcl, Release 1.18, 12/19/02
#
# Based on POS subsystem mimic.tcl SCCS version Release 1.20, 01/21/99.
#
# Merged with GBD's modified configure version on 22-Jan-1999.  
#
# History:
#     The original version of this program (only supporting 2dF) dates from
#     1993. The history section of the code that dates from 1993 to 2001
#     is long and - given the substantial rewriting to support multiple
#     instruments such as 6dF and FLAMES - somewhat irrelevant. It has been
#     removed from this file and saved in the file configure.history, which
#     can be consulted as a historical document. The following section
#     begins afresh with the first proper OzPoz release.
#
#     26-Sep-2001 -  KS - First release with proper OzPoz support. Commenting
#                         revised.
#     30-Oct-2001 -  KS - Changed ShowFibre dialogue. Apply button removed,
#                         type of allocated object now shown.
#      1-Nov-2001 -  KS - VLT guide probe code now tests for target outside the
#                         allowed circle, and for objects outside the suggested
#                         magnitude limits. These limits all controlled by the
#                         VLT_Guidexxx global variables.
#      5-Nov-2001 -  KS - Now intercepts attempts to close the mimic window.
#      5-Dec-2001 -  KS - Fixed the spelling mistake - 'shaddowed' - that 
#                         appeared in the basic sequence dialogue. The same
#                         mistake appears throughout the code, but it shouldn't
#                         be seen by the user!
#      5-Feb-2002 -  KS - Introduced MimicClicked global to indicate that the
#                         user has clicked on the mimic display. Also introduced
#                         the HighlightClick global.
#     12-Feb-2002 -  KS - Removed all use of DtcluFDialog - used the standard
#                         tk_getSaveFile instead.
#     11-Apr-2002 -  KS - Now reports when fibres are not in use as part of
#                         the OverFibre{} routine.
#      5-Sep-2002 -  KS - Added new global SelectedGuideName to record name of
#                         selected VLT guide star - enables it to be identified
#                         properly by the PAF output code. Introduced current
#                         directory into all open/save file dialogues.
#     16-Sep-2002 -  KS - Added comment in ShowHighlighting{} about unhighlit
#                         fibres being filled black. Made printer dialogue
#                         prompt for the entire print command to be used.
#     25-Sep-2002 -  KS - Introduced separate input and output directories.
#     19-Dec-2002 -  KS - Corrected code in DisableShaddowedPivots{} that 
#                         calculated the number of unallocated object entries.
#                         In the course of this, spotted and fixed a bug in
#                         DrawFiducials{} involving the syntax of an array
#                         reference.
#     19-Dec-2002 - RSC - Merged with previous version.
#                         

#------------------------------------------------------------------------------ 
#
#                          G l o b a l s
#
#  This section should list all the global variables used in this file. This 
#  section isn't necessary for the program to work, but it provides a 
#  convenient place to document (briefly) the use to which these variables are
#  put. At present this list may be incomplete.

global ArrowHighlighting   ;# Flag set if arrow highlighting is enabled
global Cfid                ;# Sds Id for the top level of the current
                           ;# configuration structure. 
                           ;# This is the structure containing the "fieldData",
                           ;# "unallocGuide", "unallocObject" and "objects" 
                           ;# sub-structures.
global CurrentOutDirectory ;# Current directory for saving files
global ExpertMode          ;# True if the system has been set to 'expert mode'.
global FibreCombination    ;# Fibre combination selected for use. This is an
                           ;# index value. -1 indicates that what has been
                           ;# selected is not a known configuration. Values from
                           ;# 0 up are index values that must match the order
                           ;# returned by [FibreCombos].
global HighlightClick      ;# True if clicked position is to be highlit.
global LastClickRa         ;# The mean RA of the point last clicked upon.
global LastClickDec        ;# The mean Dec of the point last clicked upon.
                           ;# If no point has been clicked upon, these are zero.
global MimicClicked        ;# Indicates that mimic display was clicked upon.
global MimicShowFont       ;# Font used by mimic for 'show fibre' and 'show
                           ;# object' dialogues.
global ObjectError         ;# Describes any error associated with object.
                           ;# Used only as a textvariable by ShowObject{}.
global PlotLabel           ;# Canvas Id for text used to label a hardcopy plot.
                           ;# This is normally blank, and is just set to the
                           ;# label text as the plot is produced - otherwise
                           ;# it clutters the display.
global PrinterCommand      ;# Command used to print a postscript file
global SelectedFibre       ;# Currently selected fibre (pivot) number - starts
                           ;# from 1 - zero => no fibre selected.
global SelectedGuideDescr  ;# Describes guide star selection as a text string
global SelectedGuideName   ;# Name of selected guide star.
global SelectedGuidePivot  ;# Indicates which of the two possible guide pivot
                           ;# positions has been selected.
                           ;# Zero if no pivot selected, 1 or 2 when a
                           ;# selection has been made.
global SelectedGuidePosn   ;# Position of guider with selected pivot. 
                           ;# A string of four numbers with the pivot
                           ;# position (x,y) followed by the guide star
                           ;# position (x,y).
global SelectedGuideObject ;# Which guide object is selected ( 0=> none).
                           ;# This is the index (starting from 1) of the
                           ;# selected target in the "unallocGuide"
                           ;# section of the overall SDS structure whose SDS id
                           ;# is held in Cfid.
global SelectedGuideOrient ;# Guide probe orientation - "POS" or "NEG".
global SelectedGuideState  ;# State of guider position selection and display.
                           ;# Zero => no guider selected or just selected but
                           ;# not yet displayed, 1 => Selected and
                           ;# displayed as an opaque stipple, 2 => selected
                           ;# and displayed as an outline. Clicking on a guide
                           ;# pivot position toggles between the last two 
                           ;# states.
global ShowEmptyFlag       ;# True if empty fibre positions are to be shown.
global SmallShowFont       ;# A small font used for the status bar
global TargetScale         ;# Scale factor used to magnify targets.
global tcl_platform        ;# Tcl array giving platform-dependent info. In
                           ;# this case used to pick font names.
global TkAvailable         ;# Set if Tk operations are permitted.
global VLT_GuideCircleDia  ;# Diameter in arcmin in which guider can operate.
                           ;# Any guide target must be within this circle. This
                           ;# was originally set to 30 arcmin, then to 28. The
                           ;# code now always uses a numeric value taken from
                           ;# this global variable, but a lot of the variable
                           ;# names in the code still refer to '28'.
global VLT_GuideMinMag     ;# Minimum magnitude for VLT guider targets.
global VLT_GuideMaxMag     ;# Maximum magnitude for VLT guider targets.
                           ;# Note that VLT_GuideMinMag is a larger numeric
                           ;# value than VLT_GuideMaxMag.
global XClickPosition      ;# Last X position in mimic display clicked upon.
global YClickPosition      ;# Last Y position in mimic display clicked upon.
                           ;# The X and Y click positions are fractions of the
                           ;# scroll region to the left of or above the 
                           ;# selected point. The intent is that zooming takes
                           ;# place about the selected point.
global XClickScrollStart   ;# The X scrollbar start fraction when clicked.
global YClickScrollStart   ;# The Y scrollbar start fraction when clicked.
                           ;# These two are used to record the scrollbar setting
                           ;# when the X and Y click positions were set. This
                           ;# allows the code to tell if the display has been
                           ;# scrolled since the click positions were set.
global zoom                ;# Current zoom factor - 1 is normal.

#  -----------------------------------------------------------------------------
#
#                    I n i t i a l i s a t i o n s
#
#  Initialisations - performed as the file is loaded.
#
#  Set the font used for the 'show fibre' and 'show object' dialogues.
#  We have to make sure we use a name that is understood and works on the
#  underlying platform.
                              
if { ($tcl_platform(platform) == "windows") ||
           ($tcl_platform(platform) == "macintosh")} {
    set MimicShowFont {Helvetica 12 }
    set SmallShowFont {Helvetica 9 }
} else {
    set MimicShowFont {Helvetica 12 }
    set SmallShowFont {Helvetica 9 }
}

#  Set the globals connected with guider selection. These all are set to the
#  values that indicate that no guider position has been selected as yet.

set SelectedGuidePivot 0
set SelectedGuideState 0
set SelectedGuideObject 0
set SelectedGuideName "None"
set SelectedGuideOrient "POS"
set SelectedGuideDescr "Not selected"

#  Set the limits used for the VLT guide star selection. These values are
#  subject to change, and if they are to be changed, this is the place to 
#  make those changes.

set VLT_GuideCircleDia 28.0
set VLT_GuideMinMag 11.0
set VLT_GuideMaxMag  9.0

#  Set the 'last clicked upon' position variables to a null value

set XClickPosition 0.5
set YClickPosition 0.5
set XClickScrollStart 0.0
set YClickScrollStart 0.0

set LastClickRa 0.0
set LastClickDec 0.0
set MimicClicked 0
set HighlightClick 0

#  -----------------------------------------------------------------------------
#
#                          C r e a t e  M i m i c
#
#   Creates the mimic display used by the configure program. The top level
#   for this is .mimic, which is created by the Setup{} procedure (found in
#   configure.tcl). The display consists of a canvas (.mimic.scv.f1.field) and
#   two scrollbars (.mimic.scv.f1.hscroll  and .mimic.scv.vscroll), and a
#   status line (.mimic.info) which is a frame containing three labels  
#   associated with text variables.

proc CreateMimic {} {

    global zoom             ;# Current zoom factor - 1 is normal.
    global SelectedFibre    ;# Currently selected fibre (pivot) number - starts
                            ;# from 1 - zero => no fibre selected.
    global SmallShowFont    ;# A small font used for the status bar
    global CursorOverText   ;# Holds the description of whatever the cursor
                            ;# is currently positioned over.
    global PrinterCommand   ;# Command used to print a postscript file
    global SelObjectText    ;# Describes the currently selected object
    global SelFibreText     ;# Describes the currently selected fibre
    global TkAvailable      ;# Set if Tk operations are permitted.
    global XClickPosition   ;# Last X position in mimic display clicked upon.
    global YClickPosition   ;# Last Y position in mimic display clicked upon.
    global XClickScrollStart;# The X scrollbar start fraction when clicked.
    global YClickScrollStart;# The Y scrollbar start fraction when clicked.
                            
    if { !$TkAvailable } return
    
    set WindowSize [Mimic WindowSize]
    
    #  We create a small status frame at the top of the mimic display that
    #  we use to show what the cursor is over and the selected fibre and
    #  object (if any)
    
    set CursorOverText ""
    set SelObjectText ""
    set SelFibreText ""
    frame .mimic.info
    label .mimic.info.over -width 30 -textvariable CursorOverText \
                                                          -font $SmallShowFont
    label .mimic.info.selobj -width 15 -textvariable SelObjectText \
                                          -foreground red -font $SmallShowFont
    label .mimic.info.selfib -width 10 -textvariable SelFibreText \
                                          -foreground red -font $SmallShowFont
    pack .mimic.info.over -side left -fill both
    pack .mimic.info.selobj -side left -fill both
    pack .mimic.info.selfib -side left -fill both
    pack .mimic.info -fill both
    
    frame .mimic.scv
    
    frame .mimic.scv.f1

#  Create a canvas widget, size determined by [Mimic WindowSize] pixels
#  (Mimic command is implemented by C code, which determines the window
#   size based on the screen size)

    canvas .mimic.scv.f1.field -height $WindowSize -width $WindowSize \
        -yscrollcommand {.mimic.scv.vscroll set} \
        -xscrollcommand {.mimic.scv.f1.hscroll set} -relief sunken \
        -scrollregion "0 0 $WindowSize $WindowSize" \
        -xscrollincrement 20 -yscrollincrement 20 -background white

#  Link horizontal and vertical scroll bars to it

    scrollbar .mimic.scv.f1.hscroll -orient horizontal \
        -command {.mimic.scv.f1.field xview}
    pack .mimic.scv.f1.hscroll -side bottom -fill x
    pack .mimic.scv.f1.field -side top -expand 1
    scrollbar .mimic.scv.vscroll -orient vertical \
                                        -command {.mimic.scv.f1.field yview}
    pack .mimic.scv.f1 -side left -expand 1 -fill both
    pack .mimic.scv.vscroll -side right -fill y
    pack .mimic.scv -fill both
    
    focus .mimic.scv.f1.field
#  Set initial zoom factor

    set zoom 1

#   Set default printer command used

    set PrinterCommand "lpr"
    
    set SelectedFibre 0
    
    DrawFrame
    
    #This is just a test to see if the %x %y work properly
    bind .mimic.scv.f1.field <ButtonPress-1> {
        SetClickPosition %x %y
    }

    bind .mimic.scv.f1.field <ButtonPress-3> {
        SetClickPosition %x %y
        ZoomMimic  [expr {$zoom * 2.0}]
    }
    bind .mimic.scv.f1.field <Control-ButtonPress-3> {
        SetClickPosition %x %y
        ZoomMimic [expr {$zoom / 2.0}]
    }
    
    bind .mimic.scv.f1.field <Configure> {
        ResizeMimic
    }
    
    #  Don't allow the user to close the window
    
    wm protocol .mimic WM_DELETE_WINDOW NullAction
    
    #  Set the 'last clicked upon' variables to indicate the center of
    #  the display.
    
    set XClickPosition 0.5
    set YClickPosition 0.5
    set XClickScrollStart 0.0
    set YClickScrollStart 0.0
    
}

#  -----------------------------------------------------------------------------
#
#                        R e s i z e  M i m i c
#
#   At the moment, this is a null routine, but the intention is that it will
#   be used to allow the mimic to be redrawn sensibly when the window is
#   resized by the user. It is invoked by such a resize, being bound to the
#   "Configure" event for the window.

proc ResizeMimic {} {
}

#  -----------------------------------------------------------------------------
#
#                   S e t  C l i c k  P o s i t i o n
#
#   Records the given position as the 'last clicked on' position. This
#   is the last position in the mimic display that was clicked on, and
#   so is a sensible place to zoom around. The X and Y positions passed
#   should be in canvas pixel coordinates. This means they will be from
#   0.0 to the window size in pixels in X and Y. (At the moment, this
#   is the same in X and Y, and is the value returned by [Mimic WindowSize]).
#   If the HighlightClick global variable is true, the clicked position is
#   indicated by a red arrow.

proc SetClickPosition { XPosn YPosn } {

    global Cfid             ;# Sds Id for the top level of the current
                            ;# configuration structure.
    global HighlightClick   ;# True if clicked position is to be highlit.
    global LastClickRa      ;# The mean RA of the point last clicked upon.
    global LastClickDec     ;# The mean Dec of the point last clicked upon.
    global MimicClicked     ;# Indicates that mimic display was clicked upon.
    global XClickPosition   ;# Last X position in mimic display clicked upon.
    global YClickPosition   ;# Last Y position in mimic display clicked upon.
    global XClickScrollStart;# The X scrollbar start fraction when clicked.
    global YClickScrollStart;# The Y scrollbar start fraction when clicked.
    global zoom             ;# Current zoom factor - 1 is normal.
    
    #  The canvas xview command returns two fractions. The canvas displays
    #  a portion of the scroll region associated with it. The first fraction
    #  is the fraction of the scroll region to the left of the displayed
    #  area, and the second is the fraction of the scroll region to the
    #  left of the right edge of the display. That is, the two fractions give
    #  the display limits in X expressed as fractions of the whole scroll
    #  region.
    #
    #  The display is XWindowSize pixels by YWindowSize pixels (at the moment,
    #  both of these are the same and are obtained from [Mimic WindowSize]).
    #  We make sure these are in floating point, so calculations are all
    #  done in floating point.
    #
    #  We determine the start and end fractions of the scroll region currently
    #  displayed, and calculate the fraction of the scroll region actually
    #  shown in the display.
    
    set XWindowSize [expr [Mimic WindowSize] * 1.0]
    set YWindowSize [expr [Mimic WindowSize] * 1.0]
    set XViewDescr [.mimic.scv.f1.field xview]
    set XStartFraction [lindex $XViewDescr 0]
    set XEndFraction [lindex $XViewDescr 1]
    set YViewDescr [.mimic.scv.f1.field yview]
    set YStartFraction [lindex $YViewDescr 0]
    set YEndFraction [lindex $YViewDescr 1]
    
    set XDisplayedFraction [expr $XEndFraction - $XStartFraction]
    set YDisplayedFraction [expr $YEndFraction - $YStartFraction]
    
    #  From that we can calculate the X and Y Click positions as fractions
    #  of the scroll region.
    
    set XClickPosition [expr (($XPosn / $XWindowSize) * $XDisplayedFraction) \
                                                        + $XStartFraction]
    set YClickPosition [expr (($YPosn / $YWindowSize) * $YDisplayedFraction) \
                                                        + $YStartFraction]
    
    #  And, just to be sure, we check they're in the range 0..1
    
    if { $XClickPosition < 0.0 } { set XClickPosition 0.0 }
    if { $XClickPosition > 1.0 } { set XClickPosition 1.0 }
    if { $YClickPosition < 0.0 } { set YClickPosition 0.0 }
    if { $YClickPosition > 1.0 } { set YClickPosition 1.0 }
    
    #  Set the global variables that record the scrollbar setting when
    #  this routine was called.
    
    set XClickScrollStart $XStartFraction
    set YClickScrollStart $YStartFraction
    
    #  We also convert this position to a mean RA,Dec position, if we can.
    #  First, we need the pixel position in the scroll region. We can then
    #  convert that to a plate X,Y position in microns.
    
    set XPixelPosn [expr ($XWindowSize * $zoom * $XStartFraction) + $XPosn]
    set YPixelPosn [expr ($YWindowSize * $zoom * $YStartFraction) + $YPosn]
    set XPlatePosn [DispXtoPlateX $XPixelPosn $zoom]
    set YPlatePosn [DispYtoPlateY $YPixelPosn $zoom]
    
    #  If clicked positions are being highlit, do so.
    
    if { $HighlightClick } {
        eval .mimic.scv.f1.field create bitmap $XPixelPosn $YPixelPosn \
                     -bitmap arrow -anchor sw -foreground red -tag redArrow
    }
                     
    #  If we have a field open, we can convert that plate position in microns
    #  to a mean Ra,Dec position, and set the global variables LastClickRa
    #  and LastClickDec to that.
    
    if { $Cfid } {
        PlateXYtoRaDec $Cfid $XPlatePosn $YPlatePosn Ra Dec
        set LastClickRa $Ra
        set LastClickDec $Dec
        set MimicClicked 1
    }
}

#  -----------------------------------------------------------------------------
#
#                R e m o v e  C l i c k  H i g h l i g h t s
#
#  This routine removes any red arrow highlights that have been used to
#  indicate positions clicked upon in the mimic display.

proc RemoveClickHighlights {} {

    global TkAvailable        ;# Set if Tk operations are permitted.

    if { !$TkAvailable } return
    
    #  Delete the red arrows used for highlighting.

   .mimic.scv.f1.field delete redArrow
}

#  -----------------------------------------------------------------------------
#
#                            D o  Z o o m
#
#  DoZoom is the core routine for drawing on the mimic. It clears the whole
#  display and redraws it using the zoom factor given by the global variable
#  zoom. The center of the display is determined by the values of the
#  parameters xcc and ycc. These specify the required center as the fractional
#  coordinates of the center in terms of the scroll region - the same 
#  convention as used by the global variables XClickPosition and 
#  YClickPosition, and indeed these are usually the values used for xcc 
#  and ycc.
#

proc DoZoom {xcc ycc} {

    global zoom               ;# Current zoom factor - 1 is normal.
    global SelectedGuidePivot ;# Indicates which of the two possible guide pivot
                              ;# positions has been selected.
    global TkAvailable        ;# Set if Tk operations are permitted.

    if { !$TkAvailable } return
    
    #  Delete the current fibres, guide pivots, guide shaddow 
    #  and frame from the display. Delete everything, in fact.

   .mimic.scv.f1.field delete fibres
   .mimic.scv.f1.field delete frame
   .mimic.scv.f1.field delete fiducials
   .mimic.scv.f1.field delete guidepivot1
   .mimic.scv.f1.field delete guidepivot2
   .mimic.scv.f1.field delete guideshaddow
   .mimic.scv.f1.field delete argus
   .mimic.scv.f1.field delete 20arcmin
   .mimic.scv.f1.field delete 224arcmin
   .mimic.scv.f1.field delete 28arcmin
   .mimic.scv.f1.field delete fieldedge
   .mimic.scv.f1.field delete pivotcircle
   .mimic.scv.f1.field delete blueArrow
   .mimic.scv.f1.field delete redArrow

   #  Get window size and force it to be a real number

   set WindowSize [expr [Mimic WindowSize] * 1.0]

   set xc [expr $xcc - (1.0 / ($zoom * 2.0))]
   set yc [expr $ycc - (1.0 / ($zoom * 2.0))]

   #  Reset the size of the scrolling region of the canvas widget. This is
   #  the important bit as far as the zooming goes. When the display is
   #  zoomed in, the scroll region covers a larger range in pixel 
   #  coordinates than there are pixels in the display - and so only a
   #  fraction of the scroll region is displayed. 

   set z [expr {$zoom * $WindowSize}]
   .mimic.scv.f1.field configure -scrollregion "0 0 $z $z"

   #  Set the current scroll position to give the required centre

   .mimic.scv.f1.field xview moveto $xc
   .mimic.scv.f1.field yview moveto $yc

   #  Draw the frame and the fibres

   DrawFrame
   DrawObjects
   DrawFiducials
   DrawFibres

   # If we have a Select Guide Pivot (currently OzPoz only, but no harm
   # is done by doing it all the time) select it.

   if { $SelectedGuidePivot > 0 } {
      ShowGuidePivot
   }
}

#  -----------------------------------------------------------------------------
#
#                      D r a w  F i d u c i a l s

proc DrawFiducials {} {

  global FidId
  global zoom

  if {[info exists FidId] == 1} {
    if {$FidId != 0} {
      set id [SdsFind $FidId "xf"]
      SdsInfo $id name code ndims dims
      set nfids $dims(0)

      set n 0
      while {$n < $nfids} {
          set text [GetFid $FidId $n $zoom]
         eval .mimic.scv.f1.field create $text -tags fiducials
         incr n
      }
      SdsFreeId $Id
    }
  }
}

#  -----------------------------------------------------------------------------
#
#                      S e t  W i n d o w  S i z e
#
#  Resizes the window used for the mimic display. 

proc SetWindowSize {size} {

    .mimic.scv.f1.field  configure -height $size -width $size \
        -yscrollcommand {.mimic.scv.vscroll set} \
        -xscrollcommand {.mimic.scv.f1.hscroll set} -relief sunken \
        -scrollregion "0 0 $size $size" \
        -xscrollincrement 20 -yscrollincrement 20

    #  Note that DoZoom{} will override the scrollregion settings used.
    #  The scrollregion only matches the window size when the whole 
    #  of the mimic field is visible.
}

#  -----------------------------------------------------------------------------
#
#                         Z o o m  M i m i c
#
#   Procedure to zoom display to any size. The parameter newZoomFactor
#   gives the new setting for the zoom factor (1 is normal, 2 is zoomed
#   in a factor two, 0.5 is zoomed out a factor 2, etc). The center of
#   the new display depends on whether or not the display has been panned
#   using the scrollbars since a click was made in the displayed region.
#   We assume the user will want to zoom about the last clicked position,
#   unless they have panned the display since, in which case they surely
#   want the display zoomed about where they panned to. 

proc ZoomMimic {newZoomFactor} {

    global tcl_platform     ;# Tcl array giving platform-dependent info.
    global TkAvailable      ;# Set if Tk operations are permitted.
    global zoom             ;# Current zoom factor - 1 is normal.
    global XClickPosition   ;# Last X position in mimic display clicked upon.
    global YClickPosition   ;# Last Y position in mimic display clicked upon.
    global XClickScrollStart;# The X scrollbar start fraction when clicked.
    global YClickScrollStart;# The Y scrollbar start fraction when clicked.
    
    if { !$TkAvailable } return
    
    #  If we are going to change the size of the actual window, note that
    #  fact now.
    
    set Changed 0
    if { $zoom != $newZoomFactor } {
        if { ($newZoomFactor < 1.0) || \
                ($newZoomFactor >= 1.0 && ($zoom < 1.0)) } {
            set Changed 1
        }
    }
    
    set zoom $newZoomFactor

    #  Get the displayed region limits. We have to do this now, because
    #  SetWindowSize{} will reset these.
    
    set XViewDescr [.mimic.scv.f1.field xview]
    set XStartFraction [lindex $XViewDescr 0]
    set YViewDescr [.mimic.scv.f1.field yview]
    set YStartFraction [lindex $YViewDescr 0]
    set XEndFraction [lindex $XViewDescr 1]
    set YEndFraction [lindex $YViewDescr 1]
    
    # If we are zooming out relative to the original size, then
    # we want to reduce the window size.  Otherwise, we use the
    # standard size.
    
    if {$zoom < 1} {
        SetWindowSize [expr {[Mimic WindowSize]*$zoom}]
    } else {
        SetWindowSize [Mimic WindowSize]
    }
    
    #  If the display size has changed, reposition it - this gets round
    #  a problem where it was possible to get the window off the top of
    #  a screen with no way of getting it back. (For some reason, this
    #  doesn't work on a Macintosh - the window just vanishes..)
    
    if { $Changed } {
        if { ($tcl_platform(platform) != "macintosh") } {
           wm geometry .mimic -0+30
        }
    }
    
    #  See if the display limits had moved since the X and Y click position
    #  variables were set.
    
    if { ($XClickScrollStart == $XStartFraction) && \
                           ($YClickScrollStart == $YStartFraction) } {
        
        #  Limits unchanged, so zoom about the last clicked upon position.
        
        set XCenter $XClickPosition       
        set YCenter $YClickPosition
               
    } else {
    
        #  Limits have changed, so we zoom about the center of the current
        #  display.
        
        set XCenter [ expr ($XStartFraction + $XEndFraction) * 0.5 ]
        set YCenter [ expr ($YStartFraction + $YEndFraction) * 0.5 ]
    }
    DoZoom $XCenter $YCenter    
}

#  -----------------------------------------------------------------------------
#
#                         D r a w  F r a m e 
#
#   Procedure to draw the circle marking the outer edge of the field plate
#

proc DrawFrame {} {

   global TkAvailable         ;# Set if Tk operations are permitted.
   global VLT_GuideCircleDia  ;# Diameter in arcmin in which guider can operate.
   global zoom                ;# Current zoom factor - 1 is normal.
   
   if { !$TkAvailable } return

#  Get details of the circle from the C code.
   set CircleOffset [Mimic CircleOffset]
   set CircleRadius [Mimic CircleRadius]


#  We are drawing a cirle of radius CircleRadius within a square, centered
#  at CircleOffset,CircleOffset (both x and y of center are at CircleOffset).
#
#  We need the left most point, the right most point, the top and bottom
#  points for the Tk canvas create oval command. 

   set left_top     [expr {$zoom*($CircleOffset - $CircleRadius)}]
   set right_bottom [expr {$zoom*($CircleOffset + $CircleRadius)}]

#  Create the circle.
   .mimic.scv.f1.field create oval $left_top $left_top $right_bottom \
       $right_bottom -tags frame -tags pivotcircle
   .mimic.scv.f1.field bind pivotcircle <Enter> "OverItem pivotcircle"
   .mimic.scv.f1.field bind pivotcircle <Leave> NotOverFibre


   # Draw a circle indicating the actual field
   #  Get details of the field circle
   set CircleRadius [Mimic FieldRadius]
   
   set left_top     [expr {$zoom*($CircleOffset - $CircleRadius)}]
   set right_bottom [expr {$zoom*($CircleOffset + $CircleRadius)}]
   
   #  Create the circle.
   if { [UseGrey] } {
       .mimic.scv.f1.field create oval $left_top $left_top $right_bottom \
           $right_bottom -tags frame -tags fieldedge -outline grey35
   } else {
       .mimic.scv.f1.field create oval $left_top $left_top $right_bottom \
           $right_bottom -tags frame -tags fieldedge -outline blue
   } 
   .mimic.scv.f1.field bind fieldedge <Enter> "OverItem edge"
   .mimic.scv.f1.field bind fieldedge <Leave> NotOverFibre


   #  Special code for the OzPoz (FLAMES) instrument.   We need 20 and 25
   #  arc min ciricles to show the guider and a 14mm circle on the plate
   #  to show where Argus is.  This special-case code based on one instrument
   #  should really be abstracted in some way.
     
   if { [Instrument] == "FLAMES"} {

       #  These draw the 28 and 20 arc min circles for OzPoz (FLAMES) - assuming
       #  the main circle shows the 25 arc min full field. Note that this
       #  value of 25 arc minutes has to match the definitions for 
       #  fpcolINST_FIELD_RAD and fpcolINST_FIELD_ON_SKY in the OzPoz file
       #  fpcolInstDescr.h.
       
       set CircleRadius28 [expr $CircleRadius*($VLT_GuideCircleDia/25.0)]

       set left_top     [expr {$zoom*($CircleOffset - $CircleRadius28)}]
       set right_bottom [expr {$zoom*($CircleOffset + $CircleRadius28)}]

       #  Create the circle.
       if { [UseGrey] } {
           .mimic.scv.f1.field create oval $left_top $left_top $right_bottom \
               $right_bottom -tags frame -tags 28arcmin -outline grey20
       } else {
           .mimic.scv.f1.field create oval $left_top $left_top $right_bottom \
               $right_bottom -tags frame -tags 28arcmin -outline red
       }
       .mimic.scv.f1.field bind 28arcmin <Enter> "OverItem 28"
       .mimic.scv.f1.field bind 28arcmin <Leave> NotOverFibre

       set CircleRadius20 [expr $CircleRadius*(20.0/25.0)]

       set left_top     [expr {$zoom*($CircleOffset - $CircleRadius20)}]
       set right_bottom [expr {$zoom*($CircleOffset + $CircleRadius20)}]

       #  Create the circle.
       if { [UseGrey] } {
           .mimic.scv.f1.field create oval $left_top $left_top $right_bottom \
               $right_bottom -tags frame -tags 20arcmin -outline grey50
       } else {
           .mimic.scv.f1.field create oval $left_top $left_top $right_bottom \
               $right_bottom -tags frame -tags 20arcmin -outline yellow
       }
       .mimic.scv.f1.field bind 20arcmin <Enter> "OverItem 20"
       .mimic.scv.f1.field bind 20arcmin <Leave> NotOverFibre

       # Circle with radius 11.2"
       set CircleRadius224 [expr $CircleRadius*(22.4/25.0)]

       set left_top     [expr {$zoom*($CircleOffset - $CircleRadius224)}]
       set right_bottom [expr {$zoom*($CircleOffset + $CircleRadius224)}]

       #  Create the circle.
       if { [UseGrey] } {
           .mimic.scv.f1.field create oval $left_top $left_top $right_bottom \
               $right_bottom -tags frame -tags 224arcmin -outline grey80
       } else {
           .mimic.scv.f1.field create oval $left_top $left_top $right_bottom \
               $right_bottom -tags frame -tags 224arcmin -outline red
       }
       .mimic.scv.f1.field bind 224arcmin <Enter> "OverItem 224"
       .mimic.scv.f1.field bind 224arcmin <Leave> NotOverFibre

       #  Argus. This is 14mm in diameter whilst the CircleRadius represents
       #  460 mm.  Note that this 460 figure has to match the value returned
       #  by FpilGetFieldRadiusIncPark(), which at the moment is 460.
       
       set CircleRadiusArgus [expr $CircleRadius*(14.0/460.0)]

       set left_top     [expr {$zoom*($CircleOffset - $CircleRadiusArgus)}]
       set right_bottom [expr {$zoom*($CircleOffset + $CircleRadiusArgus)}]

       #  Create the circle.
       
       .mimic.scv.f1.field create oval $left_top $left_top $right_bottom \
           $right_bottom -tags frame -tags argus -outline black -fill black
       .mimic.scv.f1.field bind argus <Enter> "OverItem argus"
       .mimic.scv.f1.field bind argus <Leave> NotOverFibre


   }

   DrawFrameDecorations $zoom
   
}

#  -----------------------------------------------------------------------------
#
#                          O v e r  I t e m
#
#   Invoked when the cursor passes over some of the frame decorations.
#   The argument is used to indicate which - there are a number of fixed
#   possibilities, which isn't very elegant. The description is put into
#   the status line of the display.
#
proc OverItem { item } {

    global CursorOverText     ;# Holds the description of whatever the cursor
                              ;# is currently positioned over.
    global VLT_GuideCircleDia ;# Diameter in arcmin in which guider can operate.
    
    if { $item == "20" } {
        set CursorOverText "20 arcmin diameter"
    } elseif { $item == "224" } {
        set CursorOverText "No go zone 22.4 arcmin diameter"
    } elseif { $item == "28" } {
        set CursorOverText "$VLT_GuideCircleDia arcmin diameter"
    } elseif { $item == "argus" } {                     
        set CursorOverText "ARGUS"
    } elseif { $item == "edge" } {                     
        set CursorOverText "Usable field limit (25.0 arcmin diameter)"
    } elseif { $item == "pivotcircle" } {                     
        set CursorOverText "Pivot circle"
    }
}

#  -----------------------------------------------------------------------------
#
#                  D r a w  F r a m e  D e c o r a t i o n s
#
#   Once the basic Mimic window has been drawn, this routine adds some
#   'decorations'. These are the compass in the top right hand corner
#   that shows the positions of North and East, the label at top left
#   that describes the instrument being configured. The routine also sets
#   up (but does not use) the text field at the bottom of the window that
#   is used to label the plot when a hardcopy is made. The distinction
#   between this routine and DrawFrame{} is pretty loose, and this is 
#   really just the second half of DrawFrame{}, which otherwise would be
#   a bit long and awkward.
#

proc DrawFrameDecorations { zoom } {

    global Cfid            ;# Sds Id for the top level of the current
                           ;# configuration structure.
    global MimicShowFont
    global PlotLabelId     ;# Canvas Id of text used to label a hardcopy plot.
    global TkAvailable     ;# Set if Tk operations are permitted.

    if { !$TkAvailable } return
    
    #  We start by working out a few useful sizes for the display. Most of
    #  what we do is in the corners of the display, so what we're really
    #  doing here is working out the size of the corner sections we plan
    #  to use.  The 'area' is really the area in the corner outside of the
    #  main circle.
    
    set CircleOffset [expr $zoom * [Mimic CircleOffset]]
    set CircleRadius [expr $zoom * [Mimic CircleRadius]]
    set WindowSize   [expr $zoom * [Mimic WindowSize]]

    #  The radus of the circle is CircleRadius and the center CircleOffset
    #  So the lower left point of the area of interest is defined by
    #  the following, where .707 is the sin and cosine of 40 degrees

    if { [Instrument] == "FLAMES"} {
       set x [expr $CircleRadius * (30.0 / 25.0) * 0.707 + $CircleOffset]
    } else {
       set x [expr $CircleRadius * 0.707 + $CircleOffset]
    }

    #  This is the size of the area we are dealing with, and we also work out
    #  some useful fractions based on that size.
    
    set AreaSize [expr $WindowSize - $x]

    set ThirdAreaSize [expr $AreaSize / 3.0]
    set SixthAreaSize [expr $AreaSize / 6.0]
    set ForthAreaSize [expr $AreaSize / 4.0]
    
    #  First, we draw the compass. We do this properly, using the various
    #  astrometric routines to work out just which direction North and East
    #  really are on our display. This is the only correct way to do it, and
    #  it also acts as a useful check that the astrometric code is OK. Since
    #  the astrometric routines need the firld information, we can't use
    #  them until we have the field information in the main data structure
    #  (accessed through Cfid).

    if { $Cfid } {
    
        #  Determine the center point of our compass. This is where the
        #  North and East arrows will be drawn from.

        set CompassStartX [expr $x + $ForthAreaSize]
        set CompassStartY [expr $AreaSize - $ForthAreaSize]

        #  There are some complications to deal with. Basically, we're
        #  going to get the RA,Dec of the center of our compass, add a
        #  fraction in Dec, convert that back into display coords and
        #  draw our North arrow to that point and label it. Then more or
        #  less the same for East. But this doesn't quite work, because
        #  if we do this we'll find that out at the edge of the display the
        #  distortion can result in non-orthogonal arrows, which looks 
        #  messy. So we actually do the calculation based on the center of
        #  the field, and then offset the results to the edge of the display.
        #  So the ArrowStartX/Y variables are the start point for our arrows
        #  if they were drawn at the center - ie they are the display center,
        #  and ArrowEndX/Y are the end points for those arrows if drawn
        #  from the center. CompassEndX/Y are the end points for the arrows
        #  displaced so they start at CompassStartX/Y.
        
        set ArrowStartX $CircleOffset
        set ArrowStartY $CircleOffset
        
        #  We also have to work out a suitable offset in Dec and Ra to get
        #  decent size arrows. We calculate this by working out the difference
        #  in Ra Dec between the display center and the outside (using
        #  ArrowStartX/Y for the former and CompassStartX/Y for the latter),
        #  and applying a more or less arbitrary scale factor of 0.25, and
        #  use the larget of the two in each case.
        
        set XPlatePosn [DispXtoPlateX $ArrowStartX $zoom]
        set YPlatePosn [DispYtoPlateY $ArrowStartY $zoom]
        PlateXYtoRaDec $Cfid $XPlatePosn $YPlatePosn RaCent DecCent
        set XPlatePosn [DispXtoPlateX $CompassStartX $zoom]
        set YPlatePosn [DispYtoPlateY $CompassStartY $zoom]
        PlateXYtoRaDec $Cfid $XPlatePosn $YPlatePosn Ra Dec
        set RaOffset [expr ($Ra - $RaCent) * 0.2]
        set DecOffset [expr ($Dec - $DecCent) * 0.2]
        if { $RaOffset < 0.0 } { set RaOffset [expr 0 - $RaOffset] }
        if { $DecOffset < 0.0 } { set DecOffset [expr 0 - $DecOffset] }
        if { ($RaOffset > $DecOffset) } {
            set Offset $RaOffset
        } else {
            set Offset $DecOffset
        }
                
        #  Now we work out the display position of that point just a bit
        #  to the north of the display center.
        
        set XPlatePosn [DispXtoPlateX $ArrowStartX $zoom]
        set YPlatePosn [DispYtoPlateY $ArrowStartY $zoom]
        PlateXYtoRaDec $Cfid $XPlatePosn $YPlatePosn Ra Dec
        set Dec [expr $Dec + $Offset]
        RaDecToPlateXY $Cfid $Ra $Dec XPlatePosn YPlatePosn
        set ArrowEndX [PlateXtoDispX $XPlatePosn $zoom]
        set ArrowEndY [PlateYtoDispY $YPlatePosn $zoom]
        
        #  And then offset this over to the corner. We also work out the
        #  best place for the 'N' label, which has to be away from the center
        #  in the direction of the arrow.
        
        set CompassEndX [expr $ArrowEndX - $ArrowStartX + $CompassStartX]
        set CompassEndY [expr $ArrowEndY - $ArrowStartY + $CompassStartY]
        .mimic.scv.f1.field create line $CompassStartX $CompassStartY \
           $CompassEndX $CompassEndY  -arrow last -tags frame
        set TextStartX [expr $CompassEndX + \
                                     (0.1 * ($CompassEndX - $CompassStartX))]
        set TextStartY [expr $CompassEndY +  \
                                     (0.1 * ($CompassEndY - $CompassStartY))]
        .mimic.scv.f1.field create text  $TextStartX $TextStartY -text "N" \
           -tags frame
        
        
        #  And now we do the same thing for the East arrow.
        
        set XPlatePosn [DispXtoPlateX $ArrowStartX $zoom]
        set YPlatePosn [DispYtoPlateY $ArrowStartY $zoom]
        PlateXYtoRaDec $Cfid $XPlatePosn $YPlatePosn Ra Dec
        set Ra [expr $Ra + $Offset]
        RaDecToPlateXY $Cfid $Ra $Dec XPlatePosn YPlatePosn
        set ArrowEndX [PlateXtoDispX $XPlatePosn $zoom]
        set ArrowEndY [PlateYtoDispY $YPlatePosn $zoom]
        set CompassEndX [expr $ArrowEndX - $ArrowStartX + $CompassStartX]
        set CompassEndY [expr $ArrowEndY - $ArrowStartY + $CompassStartY]
        .mimic.scv.f1.field create line $CompassStartX $CompassStartY \
           $CompassEndX $CompassEndY  -arrow last -tags frame
        set TextStartX [expr $CompassEndX + \
                                     (0.1 * ($CompassEndX - $CompassStartX))]
        set TextStartY [expr $CompassEndY +  \
                                     (0.1 * ($CompassEndY - $CompassStartY))]
        .mimic.scv.f1.field create text  $TextStartX $TextStartY -text "E" \
           -tags frame
    }
       
    #  Now in the top left hand corner, we want to put a text field
    #  giving information.

    #  If telescope equals VLT2 (the slalib name of VLT UT2)
    #  translate name to "VLT UT2"

    set telescope [Telescope]
    if { $telescope == "VLT2" } {
        set telescope "VLT UT2"
    }

    if { $zoom < 0.8 } {
        set text "[Instrument] ($telescope)"
    } else {
        set text "Configuring [Instrument] ($telescope)"
    }
    .mimic.scv.f1.field create text [expr $SixthAreaSize] $ThirdAreaSize \
        -text $text -tags frame -anchor w
        
    #  In the bottom left hand corner, we place a text field that will be used
    #  as a label for the postscript file generated by the 'Print' command.
    #  It's convenient to locate it in this routine, but only to use it in the
    #  plotting code.

    set PlotLabelId [.mimic.scv.f1.field create text [expr $WindowSize / 2] \
              [expr $WindowSize - $SixthAreaSize] \
                         -text "" -tags frame -anchor c]
   
}

#  -----------------------------------------------------------------------------
#
#                         D r a w  F i b r e s
#
#   Procedure to draw the fibres on the mimic display
#

proc DrawFibres {} {

    global TkAvailable     ;# Set if Tk operations are permitted.

    if { !$TkAvailable } return
    
    .mimic.scv.f1.field delete fibres
    DrawNFibres [NumPivots] 20

}

#  -----------------------------------------------------------------------------
#
#                         D r a w  N  F i b r e s
#
#   This procedure draws num fibres on the mimic display in blocks of
#   m, rescheduling after each m fibres are drawn to allow the display
#   to be updated and other events to be processed.
#
#   The rescheduling is achieved by using the "after" command to 
#   schedule another call to the procedure after a delay of 10ms to
#   draw the next block. Note that the use of an update command in the
#   loop is not possible, because this procedure is called from Drama
#   event handlers and this would potentially set up a recursive call
#   to the Drama event processing which is not allowed.
#
#   Using a delay of much less than 10ms does not seem to allow all the
#   pending events to get in.
#

proc DrawNFibres {num m} {


    set i $num
    set n $m

#   Loop over fibres

    while {$i > 0} {
        
        DrawAFibre $i
        
        incr i -1 
        
        #   If the block is finished, schedule another call in 10ms
        
        if {$n == 0} {
            after 10 "DrawNFibres $i $m"
            break        
        }
        incr n -1
    }
}

#  -----------------------------------------------------------------------------
#
#                    C h e c k   F i b r e 

proc CheckFibre { pivot zoom line_list } {
    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    set radius [expr $zoom*[Mimic WindowSize]]
    foreach item $line_list {
        if { $item > $radius } {
            puts "Fibre $pivot invalid position - $line_list"
            puts "Zoom = $zoom, WindowSize = [Mimic WindowSize]"
            puts "Item = $item, radius = $radius"
            SdsWrite $Cfid CheckFibreFail.sds
            InvokeDebugger
            break
        }
    }
}

#  -----------------------------------------------------------------------------
#
#                    D r a w  A  F i b r e 
#
# Draws a single fibre and button, with the specified pivot 
# number (1 to [NumPivots]).
#
proc DrawAFibre {i} {


    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global FibreCombination  ;# Fibre combination selected for use.
    global SelectedFibre     ;# Currently selected fibre (pivot) number - starts
                             ;# from 1 - zero => no fibre selected.
    global ShowEmptyFlag     ;# True if empty fibre positions are to be shown.
    global zoom              ;# Current zoom factor - 1 is normal.
                             
    if { $Cfid != 0 } {
    
        #  If the fibre doesn't really exist, we generally aren't
        #  interested in drawing it.
        
        if { $ShowEmptyFlag == 0 } {
           if { [ReallyExists [PivotsId] $i] == 0 } return
        }

        #  If the fibre is disabled,  we don't draw it, usually.
   
        set colour [GetType [PivotsId] $i $FibreCombination]
        if { $colour == "Blank" } {
           if { $ShowEmptyFlag == 0 } return
           set colour Gray
        }
        
        #  Draw the line representing the fibre
	#
	# @@@ If -f is used, then this seems to be called too soon ...
	# @@@ For the moment, just ignore the error and return.
	# @@@ This should be checked...
	if {[catch {
            set LineList [GetButton $Cfid [PivotsId] $i $i $zoom \
                         button_graphic1 button_graphic2]
            eval .mimic.scv.f1.field create line $LineList -tags lin$i -width 0.25
	}]} {
	    return
	}
        CheckFibre $i $zoom $LineList

        
        #  Draw the polygon representing the button. 
        
        eval .mimic.scv.f1.field create $button_graphic1 -tags fib$i \
                                                                 -fill $colour
        if { [string length $button_graphic2] > 0 } {
            eval .mimic.scv.f1.field create $button_graphic2 -tags fib$i \
                                                                 -fill $colour
        }

        #  Add the fibres tag (which allows all the fibres to be deleted)

        .mimic.scv.f1.field addtag fibres withtag fib$i
        .mimic.scv.f1.field addtag fibres withtag lin$i

        #  Bind a single mouse click to the SelectFibre command, and a double
        #  mouse click to the ShowFibre command

        .mimic.scv.f1.field bind lin$i <Button-1> " SelectFibre $i "
        .mimic.scv.f1.field bind lin$i <Double-Button-1> ShowFibre
        .mimic.scv.f1.field bind fib$i <Button-1> " SelectFibre $i "
        .mimic.scv.f1.field bind fib$i <Double-Button-1> ShowFibre

        .mimic.scv.f1.field bind lin$i <Enter> "OverFibre  $i"
        .mimic.scv.f1.field bind fib$i <Enter> "OverFibre  $i"
    
        .mimic.scv.f1.field bind lin$i <Leave> NotOverFibre
        .mimic.scv.f1.field bind fib$i <Leave> NotOverFibre

        if { $SelectedFibre == $i } {
            RemoveSelectedFibreDetails
        }
    }

}

#  -----------------------------------------------------------------------------
#
#                         S e l e c t  F i b r e
#
#   Procedure to make fibre number n the currently selected fibre
#

proc SelectFibre {n} {

    global FibreCombination ;# Fibre combination selected for use.
    global SelectedFibre    ;# Currently selected fibre (pivot) number - starts
                            ;# from 1 - zero => no fibre selected.
    global SelFibreText     ;# Describes the currently selected fibre
    
#   Check that the fibre number is valid and put up an error dialog
#   if it is not

    if {($n > [NumPivots]) || ($n < 1)} {
        error_dialog .er Error \
            "Illegal Pivot Number $n \nMust be between 1 and [NumPivots]" error ""
        return 1
    } else {

#    Deselect the current selected fibre by setting it to the normal
#    colour and deleting the selected tag
        
        .mimic.scv.f1.field itemconfigure selected \
            -fill [GetType [PivotsId] $SelectedFibre $FibreCombination]
        .mimic.scv.f1.field dtag selected selected

#    Select fibre number n by changing its colour to blue, adding the
#    selected tag to it, and setting the SelectedFibre global variable

        if { [UseGrey] } {
           .mimic.scv.f1.field itemconfigure fib$n -fill black
        } else {  
           .mimic.scv.f1.field itemconfigure fib$n -fill blue
        }  
        .mimic.scv.f1.field addtag selected withtag fib$n
        set SelectedFibre $n

#    
        set SelFibreText "Fibre $n"

        
        return 0
   }
}

#  -----------------------------------------------------------------------------
#
#                      P r i n t  F i e l d
#  
#   Procedure called as a result of the "Print..." selection from the
#   File menu - Create a postscript version of the mimic display and either
#   send it to the printer or save as a file.
#

proc PrintField {} {

    global ConfigFile      ;# Currently displayed configuration file
    global CurrentOutDirectory;# Current directory for saving files
    global PrinterCommand  ;# Command used to print a postscript file
    global PlotLabelId     ;# Canvas id for text used to label a hardcopy plot.
    global TkAvailable     ;# Set if Tk operations are permitted.

    if { !$TkAvailable } return
    
#  Put up a "working" dialog and ensure the screen display is up to date

    error_dialog .postscript Working "Creating Postscript File" hourglass ""
    update
    
#  For the moment, label the plot - otherwise all plots tend to look the
#  same. Once the plot is made, we clear it.

    set Date ""
    catch {set Date [exec date]}
    .mimic.scv.f1.field dchars $PlotLabelId 0 end
    .mimic.scv.f1.field insert $PlotLabelId end "$ConfigFile $Date"

#  Create the postscript file as .pos.ps and then destroy the dialog

    .mimic.scv.f1.field postscript -file ".pos.ps" -pagewidth 180m -colormode color
    destroy .postscript
    .mimic.scv.f1.field dchars $PlotLabelId 0 end
    .mimic.scv.f1.field insert $PlotLabelId end ""

#  Put up a prompt dialog to get the printer name (also has a button to
#      request saving the file

    set button [prompt_dialog .printer Print \
                    "Enter command to print a postscript file" PrinterCommand \
                    0 Print "Save File" Cancel]

#  If the print button is selected print the file and then delete it. The
#  messing about with lists and list elements is because exec expects to be
#  passed a list of separate arguments, not just a string with arguments
#  separated by spaces. (Just "exec $PrinterCommand .pos.ps" doesn't work,
#  because it will take the whole of $PrinterCommand as the first argument,
#  ie the program to execute. See chapter 10 of Welch's Tcl book on eval for
#  more details.)

    if {$button == 0} {
        set CommandList [split $PrinterCommand]
        set PrintCommand [lindex $CommandList 0]
        set CommandArgs [lreplace $CommandList 0 0]
        eval {exec $PrintCommand} $CommandArgs .pos.ps
        exec rm .pos.ps
    } else { 

#  If the "save file" button is pressed put up a file selection dialog to
#  get the file name and rename .pos.ps to the selected name

        if {$button == 1} {
            set l1 [list "Postscript" "*.ps"]
            set l2 [list "All Files" *]
            set types [list $l1 $l2]

            set File [tk_getSaveFile    \
                -filetypes $types \
                -initialdir $CurrentOutDirectory \
                -title "Save Postscript File as..."]
            if {$File != ""} {
                set CurrentOutDirectory [file dirname $File]
                DoSaveField $File
            }
        } else {
            
            #  Otherwise the Cancel button was selected so delete the file
            
            exec rm .pos.ps
        }
    }
}

#  -----------------------------------------------------------------------------
#
#                         D o  S a v e  F i e l d

proc DoSaveField {file} {

   exec mv -f .pos.ps $file
}

#  -----------------------------------------------------------------------------

#                         S h o w  F i b r e
#
#   Procedure to display the information about a fibre - invoked in response
#   to the Show Fibre menu entry, or by double clicking on a fibre on the
#   mimic display.
#

proc ShowFibre {} {

   global SelectedFibre     ;# Currently selected fibre (pivot) number - starts
                            ;# from 1 - zero => no fibre selected.
   global Cfid              ;# Sds Id for the top level of the current
                            ;# configuration structure.
   global TkAvailable      ;# Set if Tk operations are permitted.

#  These shouldn't need to be global. They are because of the bizarre way
#  that ShowFibreDialog makes use of the arguments it is passed, using them
#  in a -textvariable context that requires that they be global. This could
#  be tidied up!

   global ShowX
   global ShowY
   global ShowTheta
   global ShowPivot
   global ShowObjectType
   global ShowFibreType
   global ShowObjName
   global ShowPrio
   global ShowPivotX
   global ShowPivotY
   global ShowParkX
   global ShowParkY
   
    if { !$TkAvailable } return
    
#   Get the fibre information - See comments to ConfGetFibre() in configure.c
#   to see the information returned in the information array.

   set InfoArray [GetFibre $Cfid $SelectedFibre ]
   set ShowX [lindex $InfoArray 0]
   set ShowY [lindex $InfoArray 1]
   set ShowTheta [lindex $InfoArray 2]
   set ShowPivot $SelectedFibre
   set ShowObjectType [lindex $InfoArray 3]
   set ShowPrio [lindex $InfoArray 5]
   if {$ShowPrio != 0} {
       set ShowObjName [lindex $InfoArray 6]
   } else {
       set ShowObjName " "
   }
   # Get the fibre type name
   set ShowFibreType [lindex $InfoArray 7]
   
#    And the park and pivot positions

   set ShowPivotX [lindex $InfoArray 9]
   set ShowPivotY [lindex $InfoArray 10]
   set ShowParkX [lindex $InfoArray 11]
   set ShowParkY [lindex $InfoArray 12]

#   Put up a dialog displaying it

   ShowFibreDialog .show "Show Fibre" ShowPivot \
                   ShowX ShowY ShowTheta ShowFibreType ShowObjectType \
                   ShowPrio ShowObjName ShowPivotX ShowPivotY \
                   ShowParkX ShowParkY

}

#  -----------------------------------------------------------------------------

#                    S h o w  F i b r e  D i a l o g
#
# ShowFibreDialog:
#
# Arguments:
# w -		Window to use for dialog top-level.
# title -       Title for window
# Pivot -       Variable to recieve pivot number.
# X     -       Variable to receive X position.
# Y     -       Variable to receive Y position.
# Theta -       Variable to receive Theta value.
# FibreType  -  Fibre type for this pivot
# ObjectType -  Type of object allocated to the fibre, or the unallocated code
# Prio  -       If allocated the priority of the object
# Name  -       If allocated the name of the object
# XPark -       Variable for XPark position.
# YPark -       Variable for YPark position.
# XPivot -      Variable for XPivot position.
# YPivot -      Variable for YPivot position.
#
#  Note that the arguments in each case are the names of global variables
#  that contain the information in question. They are not the values of
#  those variables. This is bizarre - as is the way the routine waits at
#  the end for the tk_priv(button) variable to change and then returns
#  the new value. It would be better to have the values passed to it and
#  have it return the pivot number entered by the user.  It would also be
#  better to make this a permanent dialogue and just hide and reveal it
#  when a fibre is selected. All in all, this routine needs a little work.
#
#  Originally, the pivot number was displayed in an entry box, and there was
#  an 'apply' button that would show details of the entered button when
#  clicked on. This proved confusing, since this was inconsistent with the
#  rest of the user interface, and it has been dropped. The pivot is now
#  shown in a label and the 'apply' button has been dropped.

proc ShowFibreDialog {w title pPivot pX pY pTheta pFibreType pObjectType 
                             oPrio oName pXp pYp pXpk pYpk} {
    global tk_priv
    global ExpertMode       ;# True if the system has been set to 'expert mode'.
    global MimicShowFont    ;# Font used by mimic for 'show fibre' and 'show
                            ;# object' dialogues.
    
    #  The use of upvar stinks. But since what is passed is the name of the
    #  global variable, not the value, this is the easy way to get the value.
    
    upvar #1 $pObjectType ObjectType
    upvar #1 $oPrio Prio
    
    # Create the top-level window and divide it into top
    # and bottom parts.

    catch {destroy $w}
    toplevel $w -class Dialog
    wm title $w $title
    wm iconname $w $title
    wm geom $w +0+450
    frame $w.top 
    pack $w.top -side top -fill both -pady 4m
    frame $w.mid -relief raised -bd 1
    pack $w.mid -side top -fill both
    frame $w.bot -relief raised -bd 1
    pack $w.bot -side bottom -fill both

    # Fill the top part with message and entry field.

    frame $w.top.l1
    frame $w.top.l2

    label $w.msg1 -width 10 -text "Pivot:" -anchor e -font $MimicShowFont
    pack $w.msg1 -in $w.top.l1 -side left -padx 2m -pady 2m
    label $w.ent1 -width 4 -textvariable $pPivot -borderwidth 2 \
	                             -relief sunken -font $MimicShowFont 
    pack $w.ent1 -in $w.top.l1 -side left -ipadx 2m -padx 2m -pady 2m

    label $w.msg5 -width 13 -text "Fibre type:" -font $MimicShowFont
    pack $w.msg5 -in $w.top.l1 -side left  -padx 2m -pady 2m
    label $w.ent5 -width 10 -textvariable $pFibreType -font $MimicShowFont \
                                                              -relief sunken 
    pack $w.ent5 -in $w.top.l1 -side left -ipadx 2m -padx 0m -pady 2m

    if {$ObjectType == [UnallocatedType]} {   
       label $w.msg6 -width 12 -text "Unallocated" -font $MimicShowFont \
                                                              -relief sunken 
    } else {
       label $w.msg6 -width 12 -text "Allocated" -font $MimicShowFont \
                                                              -relief sunken 
    }
    pack $w.msg6 -in $w.top.l1 -side left  -padx 2m -pady 2m

    pack $w.top.l1 -side top -anchor w
    
    if { $ExpertMode } {
    
        label $w.msg2 -width 10 -text "X:" -anchor e -font $MimicShowFont
        pack $w.msg2 -in $w.top.l2 -side left -padx 2m -pady 2m
        label $w.ent2 -width 8 -textvariable $pX -font $MimicShowFont \
                                                              -relief sunken 
        pack $w.ent2 -in $w.top.l2 -side left -ipadx 2m -padx 0m -pady 2m

        label $w.msg3 -width 7 -text "Y:" -anchor e -font $MimicShowFont
        pack $w.msg3 -in $w.top.l2 -side left  -padx 2m -pady 2m 
        label $w.ent3 -width 8 -textvariable $pY -font $MimicShowFont \
                                                              -relief sunken 
        pack $w.ent3 -in $w.top.l2 -side left -ipadx 2m -padx 0m -pady 2m

   
        label $w.msg4 -width 6 -text "Theta:" -anchor e -font $MimicShowFont
        pack $w.msg4 -in $w.top.l2 -side left  -padx 2m -pady 2m
        label $w.ent4 -width 8 -textvariable $pTheta -font $MimicShowFont \
                                                              -relief sunken 
        pack $w.ent4 -in $w.top.l2 -side left -ipadx 2m -padx 0m -pady 2m

        pack $w.top.l2 -side top -anchor w
       
        frame $w.top.l3
        frame $w.top.l4
    
        label $w.msg51 -width 10 -text "Xpivot:" -anchor e -font $MimicShowFont
        pack $w.msg51 -in $w.top.l3 -side left  -padx 2m -pady 2m
        label $w.ent51 -width 8 -textvariable $pXp -font $MimicShowFont \
                                                              -relief sunken 
        pack $w.ent51 -in $w.top.l3 -side left -ipadx 2m -padx 0m -pady 2m
        label $w.msg52 -width 7 -text "Ypivot:" -anchor e -font $MimicShowFont
        pack $w.msg52 -in $w.top.l3 -side left  -padx 2m -pady 2m
        label $w.ent52 -width 8 -textvariable $pYp -font $MimicShowFont \
                                                              -relief sunken 
        pack $w.ent52 -in $w.top.l3 -side left -ipadx 2m -padx 0m -pady 2m

        label $w.msg61 -width 10 -text "Xpark:" -anchor e -font $MimicShowFont
        pack $w.msg61 -in $w.top.l4 -side left  -padx 2m -pady 2m
        label $w.ent61 -width 8 -textvariable $pXpk -font $MimicShowFont \
                                                              -relief sunken 
        pack $w.ent61 -in $w.top.l4 -side left -ipadx 2m -padx 0m -pady 2m
        label $w.msg62 -width 7 -text "Ypark:" -anchor e -font $MimicShowFont
        pack $w.msg62 -in $w.top.l4 -side left  -padx 2m -pady 2m
        label $w.ent62 -width 8 -textvariable $pYpk -font $MimicShowFont \
                                                              -relief sunken 
        pack $w.ent62 -in $w.top.l4 -side left -ipadx 2m -padx 0m -pady 2m

        pack $w.top.l3 -side top -anchor w
        pack $w.top.l4 -side top -anchor w
    }

    if {$ObjectType != [UnallocatedType]} {
	    if {$Prio != 0} {
	       frame $w.top.l5
	       label $w.msg7 -width 10 -text "Object:" -anchor e \
              -font $MimicShowFont
	       pack $w.msg7 -in $w.top.l5 -side left  -padx 2m -pady 2m 
	       label $w.ent7 -width 14 -textvariable $oName \
		        -font $MimicShowFont -relief sunken                                                              -relief sunken 
	       pack $w.ent7 -in $w.top.l5 -side left -ipadx 2m -padx 0m -pady 2m
	    
	       label $w.msg8 -width 18 -text "Priority:" -anchor e \
		        -font $MimicShowFont
	       pack $w.msg8 -in $w.top.l5 -side left  -padx 2m -pady 2m
	       label $w.ent8 -width 8 -textvariable $oPrio \
		        -font $MimicShowFont -relief sunken 
	       pack $w.ent8 -in $w.top.l5 -side left -ipadx 2m -padx 0m -pady 2m
	       pack $w.top.l5 -side top -anchor w
	       frame $w.top.l6
	       label $w.msg9 -width 10 -text "Type:" -anchor e \
              -font $MimicShowFont
	       pack $w.msg9 -in $w.top.l6 -side left  -padx 2m -pady 2m 
          set TargetType [TargetDescr $ObjectType]
	       label $w.ent9 -width 20 -text $TargetType \
		        -font $MimicShowFont -relief sunken                                                              -relief sunken 
	       pack $w.ent9 -in $w.top.l6 -side left -ipadx 2m -padx 0m -pady 2m
	       pack $w.top.l6 -side top -anchor w
	    }
    }
    
    # Set up the Cancel buton 

    frame $w.buttons
    button $w.buttons.cancel -text Cancel -relief raised -command "destroy $w"
    pack $w.buttons.cancel -pady 4m
    pack $w.buttons -side top

    # Set up the window and wait for response

    wm geom $w +0+450
    update idletasks
    wm deiconify $w
}

#  -----------------------------------------------------------------------------
#
#                          O v e r  F i b r e
#
#   This is bound to the enter action for the drawing of each fibre, and so is
#   invoked whenever the cursor is placed over a fibre (either the fibre itself
#   or the button). It displays the fibre number in the text field at the
#   bottom left of the mimic display.

proc OverFibre { pivot } {

    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global CursorOverText    ;# Holds the description of whatever the cursor
                             ;# is currently positioned over.
    
    set InfoArray [GetFibre $Cfid $pivot ]
    set FibreType [lindex $InfoArray 7]
    set InUse [PivotInUse [PivotsId] $pivot]
    
    if { $InUse } {
       set CursorOverText "Fibre $pivot ($FibreType)"
    } else { 
       set CursorOverText "Fibre $pivot ($FibreType - not in use)"
    }
}

#  -----------------------------------------------------------------------------
#
#                          N o t  O v e r  F i b r e
#
#   This is bound to the leave action for the drawing of each fibre, and so is
#   invoked whenever the cursor is moved away from a fibre (either the fibre 
#   itself or the button). It clears the fibre number in the text field in the
#   status line of the mimic display.

proc NotOverFibre { } {

    global CursorOverText   ;# Holds the description of whatever the cursor
                            ;# is currently positioned over.
                            
    set CursorOverText " "
}
#  -----------------------------------------------------------------------------
#
#                          O v e r  T a r g e t
#
#   This is bound to the enter action for the drawing of each target, and so is
#   invoked whenever the cursor is placed over a target.

proc OverTarget { tag i } {

    global Cfid            ;# Sds Id for the top level of the current
                           ;# configuration structure. 
    global CursorOverText  ;# Holds the description of whatever the cursor
                           ;# is currently positioned over.
    
    if {$tag == "uguide"} {
        set id [SdsFind $Cfid unallocGuide]
    } else {
        if {$tag == "uobj"} {
           set id [SdsFind $Cfid unallocObject]
        } else {
           set id [SdsFind $Cfid objects]
        } 
    }
    GetObjectData $id $i obj_name obj_ra obj_dec type spect \
        obj_prio obj_mag obj_pid obj_comment obj_text obj_x obj_y

    set CursorOverText "$obj_name ($obj_text)"
}

#  -----------------------------------------------------------------------------
#
#                          C o n e
#
#  This is a diagnostic procedure that was made use of briefly to try to
#  clarify some aspects of the 'Oxford' allocation algorithm, by showing
#  the cone defined by a specified target. It never got properly polished,
#  but has been left here for the moment in case this sort of thing needs
#  to be done again.

proc Cone { T R } {

   .mimic.scv.f1.field delete cone
   set CircleRadius [Mimic CircleRadius]
   set CircleOffset [Mimic CircleOffset]
   set MimicR [expr {$R * 0.001} * $CircleRadius / 460.0]
   set X [expr {($MimicR * cos($T)) + $CircleOffset}]
   set Y [expr {0.0 - ($MimicR * sin($T)) + $CircleOffset}]
   .mimic.scv.f1.field create line $CircleOffset $CircleOffset $X $Y -tags cone
}

#  -----------------------------------------------------------------------------
#
#                          N o t  O v e r  T a r g e t
#
#   This is bound to the leave action for the drawing of each target, and so is
#   invoked whenever the cursor is moved away from a target. It clears the 
#   target information in the text field in the status line above the mimic 
#   display.

proc NotOverTarget { } {

    global CursorOverText   ;# Holds the description of whatever the cursor
                            ;# is currently positioned over.
    
    set CursorOverText " "
}

# ------------------------------------------------------------------------------

#         R e m o v e  S e l e c t e d  F i b r e  D e t a i l s
#
#  Clears the field displaying the number of the currently selected fibre in
#  the status line above the mimic display.
  
proc RemoveSelectedFibreDetails { } {

    global SelFibreText     ;# Describes the currently selected fibre
    
    set SelFibreText ""
    
}

# ------------------------------------------------------------------------------

#                 D i s p l a y  S e l e c t e d  O b j e c t
#
#  Displays the name of the currently selected object - identified by the 
#  global variables SelectedObject and SelectedTag - at the bottom left of
#  the mimic display.
  
proc DisplaySelectedObject { } {

    global SelectedObject    ;# Index of selected target object - starts at one.
    global SelectedTag       ;# Tag associated with selected target object.
    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global SelObjectText     ;# Describes the currently selected object

    if {$SelectedTag == "uguide"} {
        set id [SdsFind $Cfid unallocGuide]
    } else {
        if {$SelectedTag == "uobj"} {
           set id [SdsFind $Cfid unallocObject]
        } else {
           set id [SdsFind $Cfid objects]
        } 
    }
    GetObjectData $id $SelectedObject obj_name obj_ra obj_dec type spect \
        obj_prio obj_mag obj_pid obj_comment obj_text obj_x obj_y

    set SelObjectText "$obj_name"
    
    SdsFreeId $id
    
}

# ------------------------------------------------------------------------------

#         R e m o v e  S e l e c t e d  O b j e c t  D e t a i l s
#
#  Clears the name of the currently selected object at the bottom left of
#  the mimic display.
  
proc RemoveSelectedObjectDetails { } {
                             
    global SelObjectText    ;# Describes the currently selected object
    
    set SelObjectText ""

}

#------------------------------------------------------------------------------
#
#                          D r a w  O b j e c t s

proc DrawObjects {} {

   global Cfid             ;# Sds Id for the top level of the current
                           ;# configuration structure.
   global TkAvailable      ;# Set if Tk operations are permitted.

   if { !$TkAvailable } return
   
   if {$Cfid != 0} {

      .mimic.scv.f1.field delete uobj
      .mimic.scv.f1.field delete uguide
      .mimic.scv.f1.field delete obj
      if { [UseGrey] } {
          .mimic.scv.f1.field bind objects <Enter> \
                           "%W itemconfigure current -fill black"
          .mimic.scv.f1.field bind objects <Leave> \
                  "ShowHighlighting; %W itemconfigure oselected -fill black"
      } else {
          .mimic.scv.f1.field bind objects <Enter> \
                           "%W itemconfigure current -fill blue"
          .mimic.scv.f1.field bind objects <Leave> \
                  "ShowHighlighting; %W itemconfigure oselected -fill blue"
      }
      set UgFid [SdsFind $Cfid unallocGuide]
      set UgRA [SdsFind $UgFid type]
      SdsInfo $UgRA name code ndims dims
      set UgSize $dims(0)


      DrawNObjects $UgFid $UgSize 20 uguide


      SdsFreeId $UgFid
      SdsFreeId $UgRA

      set UoFid [SdsFind $Cfid unallocObject]
      set UoRA [SdsFind $UoFid ra]
      SdsInfo $UoRA name code ndims dims
      set UoSize $dims(0)
   
      DrawNObjects $UoFid $UoSize 20 uobj

      SdsFreeId $UoFid
      SdsFreeId $UoRA

      set status [catch {SdsFind $Cfid objects} oFid]
      if {$status == 0} {
          set oRA [SdsFind $oFid ra]
          SdsInfo $oRA name code ndims dims
          set oSize $dims(0)
          
          DrawNObjects $oFid $oSize 20 obj

          SdsFreeId $oFid
          SdsFreeId $oRA
      }
      StartUpdate $Cfid [PivotsId]
      
      HighlightByCriteria
   }
}

#-------------------------------------------------------------------------------
#
#                          D r a w  N  O b j e c t s
#
#   This procedure draws num objects on the mimic display in blocks of
#   m, rescheduling after each m objects are drawn to allow the display
#   to be updated and other events to be processed.
#
#   The rescheduling is achieved by using the "after" command to 
#   schedule another call to the procedure after a delay of 10ms to
#   draw the next block. 
#
#   Using a delay of much less than 10ms does not seem to allow all the
#   pending events to get in.
#

proc DrawNObjects {id num m tag} {
    
    set i $num
    set n $m
    
    #  Loop over objects
    
    while {$i > 0} {
        
        #  Draw the object
        
        DrawOneObject $id $i $tag
        
        
        incr i -1 
        
        #  If the block is finished, schedule another call in 10ms

        if {$n == 0} {
            #     after 50 "DrawNObjects $id $i $m $tag"
            update
            #     break 
            set n $m       
        }
        incr n -1
    }
}

#------------------------------------------------------------------------------ 
#
#                   R e d r a w  O n  A l l o c a t e

proc RedrawOnAllocate {fib obj tag} {

    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global TkAvailable       ;# Set if Tk operations are permitted.


    if { !$TkAvailable } return
    
    .mimic.scv.f1.field delete $tag$obj
    set id [ SdsFind $Cfid objects]
    DrawOneObject $id $fib obj
    SdsFreeId $id
}
     
#------------------------------------------------------------------------------
#
#                 R e d r a w  O n  D e a l l o c a t e

proc RedrawOnDeallocate {obj n tag} {

   global Cfid               ;# Sds Id for the top level of the current
                             ;# configuration structure.
   global TkAvailable        ;# Set if Tk operations are permitted.

   if { !$TkAvailable } return
    
   .mimic.scv.f1.field delete obj$obj
   if {$tag == "uobj"} {
     set id [ SdsFind $Cfid unallocObject]
   } else {
     set id [ SdsFind $Cfid unallocGuide]
   }
   DrawOneObject $id $n $tag
   SdsFreeId $id
}

# ------------------------------------------------------------------------------

#                       D r a w  O n e  O b j e c t
#
#   Draws one target object in the mimic display. id is the SDS id of the
#   sub-structure that contains the target object, i is the index of the
#   target object within that sub-structure (numbers from 1 up), and tag
#   is a string identifying the type of target object to be drawn - it can
#   be one of 'obj', 'uobj' or 'uguide'. 

proc DrawOneObject {id i tag} {

    global zoom             ;# Current zoom factor - 1 is normal.
    global SelectedObject   ;# Index of selected target object - starts at one.
    global TargetScale      ;# Scale factor used to magnify targets.

    set text [GetObject $id $i $zoom $TargetScale]
    if {$text != "unused"} {
        eval .mimic.scv.f1.field create $text -tags $tag$i
        
        #  Add the objects tag (which allows all the objects to be deleted)
        #  and the unhighlit tag (which may be replaced by the highlit tag
        #  if the object is selected for highlighting).

        .mimic.scv.f1.field addtag $tag withtag $tag$i
        .mimic.scv.f1.field addtag objects withtag $tag$i
        .mimic.scv.f1.field addtag unhighlit withtag $tag$i
        
        #  Bind a single mouse click to the SelectObject command, and a double
        #  mouse click to the ShowFibre command

        .mimic.scv.f1.field bind $tag$i <Button-1> " SelectObject $tag $i "
        .mimic.scv.f1.field bind $tag$i <Double-Button-1> ShowObject
        .mimic.scv.f1.field bind $tag$i <Button-2> \
                                      " SelectObject $tag $i; AllocateFib "

        .mimic.scv.f1.field bind $tag$i <Enter> "OverTarget $tag $i"   
        .mimic.scv.f1.field bind $tag$i <Leave> NotOverTarget
        
        if [ info exists SelectedObject ] {
            if { $SelectedObject == $id } {
                RemoveSelectedObjectDetails
            }
        }
    }
}

# ------------------------------------------------------------------------------

#                          S e l e c t  O b j e c t
#
#  This procedure is invoked when a target is single-clicked on.  It sets
#  the globals 'SelectedObject' and 'SelectedTag' so that they relate to the
#  object in question. The values for SelectedTag and SelectedObject are passed
#  as the arguments 'tag' and 'n'.
#

proc SelectObject {tag n} {

    #  Globals set by this procedure:
    
    global SelectedObject    ;# Index of selected target object - starts at one.
    global SelectedTag       ;# Tag associated with selected target object

    #  Deselect the current selected object by setting it to the normal
    #  colour and deleting the oselected tag

    .mimic.scv.f1.field itemconfigure oselected -fill ""
    .mimic.scv.f1.field dtag oselected oselected

    #  Select object number n by changing its colour to blue, adding the
    #  oselected tag to it, and setting the SelectedObject global variable

    if { [UseGrey] } {
        .mimic.scv.f1.field itemconfigure $tag$n -fill black
    } else {  
        .mimic.scv.f1.field itemconfigure $tag$n -fill blue
    }
    .mimic.scv.f1.field addtag oselected withtag $tag$n
    set SelectedObject $n
    set SelectedTag $tag

    #  Update the selected object name at the bottom left of the screen - note
    #  that DisplaySelectedObject uses the globals SelectedObject and 
    #  SelectedTag.

    DisplaySelectedObject

    return 0
}

# ------------------------------------------------------------------------------

#                   H i g h l i g h t  B y  C r i t e r i a
#
#   This routine is called when the 'apply' button in the select targets
#   dialogue is clicked on. It uses the TargetXxxx global variables to 
#   see what criteria are being used to select objects, then runs through
#   all the objects in the current configuration, highlighting those that
#   meet the selected criteria and clearing all the rest.

proc HighlightByCriteria {} {

    #  Global variables used by this procedure (input globals) :

    global ArrowHighlighting  ;# Flag set if arrow highlighting is enabled
    global Cfid               ;# Sds Id for the top level of the current
                              ;# configuration structure.
    global TargetCriteriaFlags;# Flags set to show target selection criteria.
    global TargetHighPriority ;# Entry variable for selected top priority
    global TargetLowPriority  ;# Entry variable for selected low priority
    global TargetByAllocation ;# Entry variable used to select by allocation
    global TargetNumCriteria  ;# No. of instrument-dependent selection criteria
    global TkAvailable        ;# Set if Tk operations are permitted.
    global UnallocGui         ;# The number of unallocated guide targets.
    global UnallocObj         ;# The number of unallocated non-guide, non-sky,
                              ;# targets.
    global UnallocSky         ;# The number of unallocated sky targets.
    global zoom               ;# Current zoom factor - 1 is normal.

    #  Global variables set by this procedure (output globals):
    
    global TargetResult       ;# Number of selected targets
    
    if { !$TkAvailable } return
    
    #  Clear out any blue arrow highlighting we might have, and see if we
    #  are still using arrow highlighting.
    
    .mimic.scv.f1.field delete blueArrow
   
    #  First, see how many of the instrument-specific criteria were selected.
    #  We assume that these all relate to fibre-type, so if any of these
    #  have been selected, the user wants to limit the selection to some
    #  subset of the fibre types.

    set SpecificCriteriaSet 0
    for { set iCrit 0 } { ($iCrit < $TargetNumCriteria) } { incr iCrit } {
        if { $TargetCriteriaFlags($iCrit) } {
            set SpecificCriteriaSet 1
        }
    }
     
    #  PrioFlagIndex is the index of the ByPriority flag in the flags array.
    #  Alloc FlagIndex is the index of the ByAllocation flag in the flags array.
    
    set AllocFlagIndex [expr $TargetNumCriteria + 1]
    set PrioFlagIndex $TargetNumCriteria
       
    #  Note what we are going to do for each object. We treat the selection
    #  criteria in three separate sets. One set is the instrument-specific
    #  criteria, and we bundle these all together. The second is the priority
    #  range and the third is the allocation criterion. If any of these
    #  criteria are selected (ie if any of the instrument-specific criteria
    #  have been selected, or if the 'by priority' button has been set, or
    #  if the 'by allocation' button has been set) then a target must meet
    #  that criterion to be displayed. If the criterion has not been selected,
    #  then any target is taken as meeting it. The exception is if no
    #  criteria at all have been selected (the 'clear' condition), and in
    #  that case no targets will be highlit at all.
     
    set AllocCriteriaSet 0
    if { $TargetCriteriaFlags($AllocFlagIndex) } {
        set AllocCriteriaSet 1
    }
    set PrioCriteriaSet 0
    if { $TargetCriteriaFlags($PrioFlagIndex) } {
        set PrioCriteriaSet 1
    }
    set ClearAll 0
    if { ($AllocCriteriaSet == 0) && ($PrioCriteriaSet == 0) && \
                                      ($SpecificCriteriaSet == 0) } {
        set ClearAll 1
    }
    
    #  Collect some values connected with priorities. PrioFlagIndex is the
    #  index of the ByPriority flag in the flags array. LowPriority and
    #  HighPriority are the priority range limits, and we allow the user
    #  to be confused as to which is the highest..
    
    set LowPriority $TargetLowPriority
    set HighPriority $TargetHighPriority
    if { $HighPriority < $LowPriority } {
        set LowPriority $TargetHighPriority
        set HighPriority $TargetLowPriority
    }
    
    #  Clear the selected objects count.
    
    set TargetResult 0

    #  Now we work through all the objects in the unallocated objects section
    #  of the current structure. 
    
    set id [SdsFind $Cfid unallocObject]
    set n 0
    while {$n < $UnallocObj+$UnallocSky} {
        incr n
        
        #  Handle this object.  If we're clearing everything, we clearly
        #  don't highlight it. Otherwise, get its details and see how
        #  they fit against our criteria.
        
        set HighlightThis 1
        if { $ClearAll } {
            set HighlightThis 0
        } else {
        
            GetObjectData $id $n obj_name obj_ra obj_dec type spect \
                 obj_prio obj_mag obj_pid obj_comment obj_text obj_x obj_y
         
            #  If instrument specific criteria are being used, see if
            #  it meets any of them.  If not, we don't highlight it.
            
            if { $SpecificCriteriaSet } {
                if { [CheckTargetCriteria TargetCriteriaFlags \
                               $TargetNumCriteria $type $spect 1] <= 0 } {
                    set HighlightThis 0
                }
            }
            
            #  If the priority criterion is in use, see if the object falls
            #  within the specified range. If not, don't highlight it.
            
            if { $PrioCriteriaSet } {
                if { ($obj_prio > $HighPriority) || \
                                        ($obj_prio < $LowPriority) } {
                    set HighlightThis 0
                }
            }
            
            #  And ditto the allocation criterion.  Note that for the
            #  unallocated objects, clearly none are allocated, so if
            #  the user is asking to see allocated objects, this isn't
            #  one of them.
            
            if { $AllocCriteriaSet } {
                if { $TargetByAllocation != 0 } {
                    set HighlightThis 0
                }
            }
            
            
        }
        
        #  Now, either highlight the object or clear it. Note that we highlight
        #  the selected targets indirectly, by giving them the 'highlit' or 
        #  'unhighlit' tag. Doing it with tags means that we can redraw the
        #  highlighting easily after something happens to change the way a 
        #  target is displayed - for example when the cursor is moved over a
        #  target, the binding of the 'enter' event causes it to be highlit and
        #  we need to be sure that the 'leave' event can be bound to something
        #  (in this case the procedure ShowHighlighting{}) that can restore 
        #  its previous state.  Arrow highlighting is handled separately.
        
        if { $HighlightThis } {
            .mimic.scv.f1.field addtag highlit withtag uobj$n
            .mimic.scv.f1.field dtag uobj$n unhighlit
            incr TargetResult
            
            if { $ArrowHighlighting } {    
                set DispX [PlateXtoDispX $obj_x $zoom]
                set DispY [PlateYtoDispY $obj_y $zoom]
                eval .mimic.scv.f1.field create bitmap $DispX $DispY \
                     -bitmap arrow -anchor sw -foreground blue -tag blueArrow
            }
                                 
        } else {
            .mimic.scv.f1.field addtag unhighlit withtag uobj$n
            .mimic.scv.f1.field dtag uobj$n highlit
        }
    }
    SdsFreeId $id

    #  Now repeat the process with the unallocated guide stars. This code
    #  is almost identical to the unallocated objects case above, so see
    #  that for comments. The tags used in the highlighting use 'uguide'
    #  instead of 'uobj'.
    
    set id [SdsFind $Cfid unallocGuide]
    set n 0
    while {$n < $UnallocGui} {
        incr n
        set HighlightThis 1
        if { $ClearAll } {
            set HighlightThis 0
        } else {
            GetObjectData $id $n obj_name obj_ra obj_dec type spect \
                 obj_prio obj_mag obj_pid obj_comment obj_text obj_x obj_y
            if { $SpecificCriteriaSet } {
                if { [CheckTargetCriteria TargetCriteriaFlags \
                               $TargetNumCriteria $type $spect 1] <= 0 } {
                    set HighlightThis 0
                }
            }
            if { $PrioCriteriaSet } {
                if { ($obj_prio > $HighPriority) || \
                                        ($obj_prio < $LowPriority) } {
                    set HighlightThis 0
                }
            }
            if { $AllocCriteriaSet } {
                if { $TargetByAllocation != 0 } {
                    set HighlightThis 0
                }
            }
        }
        if { $HighlightThis } {
            .mimic.scv.f1.field addtag highlit withtag uguide$n
            .mimic.scv.f1.field dtag uguide$n unhighlit
            incr TargetResult
            
            if { $ArrowHighlighting } {    
                set DispX [PlateXtoDispX $obj_x $zoom]
                set DispY [PlateYtoDispY $obj_y $zoom]
                eval .mimic.scv.f1.field create bitmap $DispX $DispY \
                     -bitmap arrow -anchor sw -foreground blue -tag blueArrow
            }
                                 
        } else {
            .mimic.scv.f1.field addtag unhighlit withtag uguide$n
            .mimic.scv.f1.field dtag uguide$n highlit
        }
    }
    SdsFreeId $id

    #  Now repeat the process with the allocated objects. Note that in this
    #  case there is an entry in the structure for each pivot, with the
    #  unallocated pivots having the 'unallocated' value set for the object
    #  type. We ignore these.
    
    set id [SdsFind $Cfid objects]
    set n 0
    while {$n < [NumPivots]} {
        incr n
        GetObjectData $id $n obj_name obj_ra obj_dec type spect \
            obj_prio obj_mag obj_pid obj_comment obj_text obj_x obj_y
            
        #  Ignore pivots that haven't been allocated
        
        if { $type != [UnallocatedType] } {
        
            #  The code that decides whether or not to highlight an object
            #  is the same as in the two previous cases, except that these
            #  objects are allocated, so the allocation test is reversed
            #  compared to the test used for the unallocated objects.
            
            set HighlightThis 1
            if { $ClearAll } {
                set HighlightThis 0
            } else {
                if { $SpecificCriteriaSet } {
                    if { [CheckTargetCriteria TargetCriteriaFlags \
                               $TargetNumCriteria $type $spect 1] <= 0 } {
                        set HighlightThis 0
                    }
                }
                if { $PrioCriteriaSet } {
                    if { ($obj_prio > $HighPriority) || \
                                        ($obj_prio < $LowPriority) } {
                        set HighlightThis 0
                    }
                }
                if { $AllocCriteriaSet } {
                    if { $TargetByAllocation == 0 } {
                        set HighlightThis 0
                    }
                }
            }
                  
            #  If the target meets the selection criteria we need to highlight
            #  it, but since it is covered by a fibre in the mimic display
            #  this won't be enough. We also change the colour of the fibre
            #  allocated to it, using the 'highlitfib' and 'unhighlitfib'
            #  tags.
            
            if { $HighlightThis } {
                .mimic.scv.f1.field addtag highlit withtag obj$n
                .mimic.scv.f1.field dtag obj$n unhighlit
                .mimic.scv.f1.field addtag highlitfib withtag fib$n
                .mimic.scv.f1.field dtag fib$n unhighlitfib
                incr TargetResult
                
                if { $ArrowHighlighting } {
                    set DispX [PlateXtoDispX $obj_x $zoom]
                    set DispY [PlateYtoDispY $obj_y $zoom]
                    eval .mimic.scv.f1.field create bitmap $DispX $DispY \
                      -bitmap arrow -anchor sw -foreground blue -tag blueArrow
                }
            } else {
                .mimic.scv.f1.field addtag unhighlit withtag obj$n
                .mimic.scv.f1.field dtag obj$n highlit
                .mimic.scv.f1.field addtag unhighlitfib withtag fib$n
                .mimic.scv.f1.field dtag fib$n highlitfib
            }
        }
    }
    SdsFreeId $id
    
    ShowHighlighting
}

# ------------------------------------------------------------------------------

#                     S h o w  H i g h l i g h t i n g
#
#  This procedure highlights those objects that have the 'highlit' or
#  'highlitfib' tags, and clears any highlighting on those that have the
#  'unhighlit' or 'unhighlitfib' tags. This allows one command to restore
#  any highlighting, for example when the cursor moves off a target.

proc ShowHighlighting {} {

    global TkAvailable     ;# Set if Tk operations are permitted.

    if { !$TkAvailable } return
    
    #  Note that this isn't quite satisfactory - it fills the fibres that
    #  are not allocated to highlit targets with black. It ought to use the
    #  normal colour for each fibre - unfortunately, this differs with fibre
    #  type. We really need a more complex scheme for this, probably 
    #  involving an explicit redrawing of fibres when highlighting changes.
    
    .mimic.scv.f1.field itemconfigure unhighlit -fill ""
    .mimic.scv.f1.field itemconfigure unhighlitfib -fill black
    if { [UseGrey] } {  
        .mimic.scv.f1.field itemconfigure highlit -fill black
        .mimic.scv.f1.field itemconfigure highlitfib -fill black
    } else {
        .mimic.scv.f1.field itemconfigure highlit -fill blue
        .mimic.scv.f1.field itemconfigure highlitfib -fill blue 
    } 
}


# ------------------------------------------------------------------------------

#                          S h o w  O b j e c t
#
#  This procedure is invoked when a target is double-clicked on.  It assumes
#  that the globals 'SelectedObject' and 'SelectedTag' have already been set
#  (these are set by SelectObject, invoked when a target is single-clicked on),
#  and given these provides additional information in a dialogue box about
#  the object. In addition, when the target selected is a telescope guider
#  target, it invokes additional processing connected with such objects.

proc ShowObject {} {

   #  Globals used by this procedure:
    
   global Cfid               ;# Sds Id for the top level of the current
                             ;# configuration structure.
   global ExpertMode         ;# True if the system has been set to expert mode
   global MimicShowFont      ;# Font used by mimic for 'show fibre' and 'show
                             ;# object' dialogues.
   global ObjectError        ;# Describes any error associated with object
   global SelectedGuideObject;# Which guide object is selected ( 0=> none).
   global SelectedObject     ;# Index of selected target object - starts at one.
   global SelectedTag        ;# Tag associated with selected target object
   global VLT_GuideCircleDia ;# Diameter in arcmin in which guider can operate.
   global VLT_GuideMinMag    ;# Minimum magnitude for VLT guider targets.
   global VLT_GuideMaxMag    ;# Maximum magnitude for VLT guider targets.
   global zoom               ;# Current zoom factor

   #  Get all the relevant information about the target object from the
   #  relevant SDS structure.

   if {$SelectedTag == "uguide"} {
       set id [SdsFind $Cfid unallocGuide]
   } else {
       if {$SelectedTag == "uobj"} {
           set id [SdsFind $Cfid unallocObject]
       } else {
           set id [SdsFind $Cfid objects]
       } 
   }
   
   GetObjectData $id $SelectedObject obj_name obj_ra_rad obj_dec_rad \
       type spect obj_prio obj_mag obj_pid obj_comment obj_type obj_x obj_y
   set obj_ra [Ra2string $obj_ra_rad]
   set obj_dec [Dec2string $obj_dec_rad]
           
   #  Delete any existing version of the dialogue box.
   
   catch {destroy .object}
  
   #  Set up the dialogue box with the data about the object. Although
   #  we call it a dialogue box, and although it uses sunken fields for
   #  the data, these aren't something that the user can change - it just
   #  looks neater to have them sunken. In fact, all the user can do is
   #  dismiss the dialogue using the cancel button.
   
   toplevel .object -class Dialog
   wm title .object "Object Data"
   wm geom .object +0-0
   frame .object.l1 
   pack .object.l1 -side top -anchor w
   label .object.l1.lab -width 10 -text "Name:" -anchor e -font $MimicShowFont
   pack .object.l1.lab -side left -padx 2m -pady 2m
   label .object.l1.txt -width 20 -text $obj_name -relief sunken \
       -font $MimicShowFont
   pack .object.l1.txt -side left -padx 2m -pady 2m

   frame .object.l2
   pack .object.l2 -side top -anchor w
   label .object.l2.lab -width 10 -text "RA:" -anchor e -font $MimicShowFont
   pack .object.l2.lab -side left -padx 2m -pady 2m
   label .object.l2.txt -width 12 -text $obj_ra -relief sunken \
       -font $MimicShowFont
   pack .object.l2.txt -side left -padx 2m -pady 2m
   label .object.l2.lab2 -width 5 -text "Dec:" -anchor e \
       -font $MimicShowFont
   pack .object.l2.lab2 -side left -padx 2m -pady 2m
   label .object.l2.txt2 -width 12 -text $obj_dec -relief sunken \
       -font $MimicShowFont
   pack .object.l2.txt2 -side left -padx 2m -pady 2m

   if { $ExpertMode } {
       frame .object.l21
       pack .object.l21 -side top -anchor w
       label .object.l21.lab -width 10 -text "RA (rad):" -anchor e \
           -font $MimicShowFont
       pack .object.l21.lab -side left -padx 2m -pady 2m
       label .object.l21.txt -width 12 -text $obj_ra_rad -relief sunken \
           -font $MimicShowFont
       pack .object.l21.txt -side left -padx 2m -pady 2m
       label .object.l21.lab2 -width 5 -text "Dec (rad):" -anchor e \
           -font $MimicShowFont
       pack .object.l21.lab2 -side left -padx 2m -pady 2m
       label .object.l21.txt2 -width 12 -text $obj_dec_rad -relief sunken \
           -font $MimicShowFont
       pack .object.l21.txt2 -side left -padx 2m -pady 2m
   
       frame .object.l22
       pack .object.l22 -side top -anchor w
       label .object.l22.lab -width 10 -text "X:" -anchor e \
           -font $MimicShowFont
       pack .object.l22.lab -side left -padx 2m -pady 2m
       label .object.l22.txt -width 12 -text $obj_x -relief sunken \
           -font $MimicShowFont
       pack .object.l22.txt -side left -padx 2m -pady 2m
       label .object.l22.lab2 -width 5 -text "Y:" -anchor e \
           -font $MimicShowFont
       pack .object.l22.lab2 -side left -padx 2m -pady 2m
       label .object.l22.txt2 -width 12 -text $obj_y -relief sunken \
           -font $MimicShowFont
       pack .object.l22.txt2 -side left -padx 2m -pady 2m
   }
   
   frame .object.l3
   pack .object.l3 -side top -anchor w
   label .object.l3.lab -width 10 -text "Magnitude:" -anchor e \
       -font $MimicShowFont
   pack .object.l3.lab -side left -padx 2m -pady 2m
   label .object.l3.txt -width 6 -text $obj_mag -relief sunken \
       -font $MimicShowFont
   pack .object.l3.txt -side left -padx 2m -pady 2m
   set ObjectError ""
   label .object.l3.err -width 24 -textvariable ObjectError \
                                      -foreground red -font $MimicShowFont
   pack .object.l3.err -side left -padx 2m -pady 2m

   frame .object.l4
   pack .object.l4 -side top -anchor w
   label .object.l4.lab -width 10 -text "Type:" -anchor e -font $MimicShowFont
   pack .object.l4.lab -side left -padx 2m -pady 2m
   label .object.l4.txt -width 32 -text $obj_type -relief sunken \
       -font $MimicShowFont
   pack .object.l4.txt -side left -padx 2m -pady 2m
   
   frame .object.l5
   pack .object.l5 -side top -anchor w
   label .object.l5.lab -width 10 -text "Priority:" -anchor e \
       -font $MimicShowFont
   pack .object.l5.lab -side left -padx 2m -pady 2m
   label .object.l5.txt -width 3 -text $obj_prio -relief sunken \
       -font $MimicShowFont
   pack .object.l5.txt -side left -padx 2m -pady 2m
   label .object.l5.lab2 -width 10 -text "Program id:" -anchor e \
       -font $MimicShowFont
   pack .object.l5.lab2 -side left -padx 2m -pady 2m
   label .object.l5.txt2 -width 12 -text $obj_pid -relief sunken \
       -font $MimicShowFont
   pack .object.l5.txt2 -side left -padx 2m -pady 2m

   frame .object.l6
   pack .object.l6 -side top -anchor w
   label .object.l6.lab -width 10 -text "Comment:" -anchor e \
       -font $MimicShowFont
   pack .object.l6.lab -side left -padx 2m -pady 2m
   label .object.l6.txt -width 40 -text $obj_comment -relief sunken \
       -font $MimicShowFont
   pack .object.l6.txt -side left -padx 2m -pady 2m

   frame .object.buttons
   button .object.buttons.cancel -text Cancel -relief raised \
       -command {destroy .object}
   pack .object.buttons.cancel -pady 4m
   pack .object.buttons -side top

   wm withdraw .object
   update idletasks
   set x [expr [winfo screenwidth .object]/2 - [winfo reqwidth .object]/2 \
        - [winfo vrootx [winfo parent .object]]]
   set y [expr [winfo screenheight .object]/2 - [winfo reqheight .object]/2 \
        - [winfo vrooty [winfo parent .object]]]
   wm deiconify .object

   if {[TypeIsTelGuide $type]} {
          
       #  The object in question is a VLT guide star. If it is within the
       #  VLT guide limit, we calculate the guide probe pivot positions
       #  for it and display them.
       
       set Allowed 0
       if { $Cfid } {
    
           #  This code works out - relatively messily - the distance in
           #  microns from the plate center for a target exactly on the 
           #  VLT limit, and the distance from the center of this target.
           #  Comparing the two tells us if this target is in the allowed
           #  circle.  (Actually, this test should really be added to 
           #  VLTGuider, but here will do for the moment.) 
            
           set fid [SdsFind $Cfid fieldData]
           set CenterRa [ArgGet $fid "cenRa"]
           set CenterDec [ArgGet $fid "cenDec"]
           set LimitDiameter [expr $VLT_GuideCircleDia * 0.5]
           set NewDec \
                [expr ($CenterDec + $LimitDiameter * (3.14159 / (60.0 * 180)))]
           RaDecToPlateXY $Cfid $CenterRa $NewDec XPNewCenter YPNewCenter
           
           #  We square quite large numbers here, so we make sure the 
           #  calculations are done in floating point.
           
           set XPFNewCenter [expr $XPNewCenter * 1.0]
           set YPFNewCenter [expr $YPNewCenter * 1.0]
           set DistLimitSq [expr ($XPFNewCenter * $XPFNewCenter) + \
                                               ($YPFNewCenter * $YPFNewCenter)]
           set ObjF_x [expr $obj_x * 1.0]
           set ObjF_y [expr $obj_y * 1.0]
           set ObjDistSq [expr ($ObjF_x * $ObjF_x) + ($ObjF_y * $ObjF_y)]
           if { $ObjDistSq < $DistLimitSq } { set Allowed 1 }
            
       }
       if { $Allowed } {
       
           #  Warn if the magnitude of the object is out of range for the
           #  VLT guider, but don't prevent it being used.
           
           if { $obj_mag < $VLT_GuideMaxMag } {
              set ObjectError "Target may be too bright"
           } elseif { $obj_mag > $VLT_GuideMinMag } {
              set ObjectError "Target may be too faint"
           }
           .mimic.scv.f1.field delete guidepivot1
           .mimic.scv.f1.field delete guidepivot2
           .mimic.scv.f1.field delete guideshaddow
           VLTGuider $id $SelectedObject posn1 posn2 pospos
           VLTDrawGuider $posn1 $zoom pivot oval polygon polygon_crack
           eval .mimic.scv.f1.field create oval $pivot \
                                              -fill blue -tag guidepivot1 
           VLTDrawGuider $posn2 $zoom pivot oval polygon polygon_crack
           eval .mimic.scv.f1.field create oval $pivot \
                                              -fill blue -tag guidepivot2
           .mimic.scv.f1.field bind guidepivot1 <Button-1> \
                                           " SetGuidePivot 1 $SelectedObject "
           .mimic.scv.f1.field bind guidepivot2 <Button-1> \
                                           " SetGuidePivot 2 $SelectedObject "
       } else {
           set ObjectError "Too far out to be used"
       }
   }
   SdsFreeId $id

}

#  -----------------------------------------------------------------------------

#                       S e t  G u i d e  P i v o t
#
#   This procedure is called whenever a guide pivot position - either one of
#   the blue potential positions drawn by SelectObject when a guider target
#   is selected, or one of the selected positions drawn by ShowGuidePivot -
#   is clicked upon. It makes the selected position the current guider position
#   disables and enables targets depending on this guider position, and
#   displays the guider in the selected position. PivNo should be either 1
#   or 2.

proc SetGuidePivot { PivNo GuideObject } {

    #  Globals used by this procedure:
    
    global Cfid               ;# Sds Id for the top level of the current
                              ;# configuration structure
                              
    #  Globals modified by this procedure:
    
    global SelectedGuideName  ;# Name of selected guide star.
    global SelectedGuidePivot ;# Indicates which of the two possible guide pivot
                              ;# positions has been selected.
    global SelectedGuidePosn  ;# Position of guider with selected pivot.
    global SelectedGuideState ;# State of guider position selection and display.
    global SelectedGuideObject;# Which guide object is selected
    global SelectedGuideOrient;# Guide probe orientation - "POS" or "NEG".

    #  If this is being invoked for a new guider position, we do the hard work
    #  of setting the new position, handling the target enabling/disabling etc.
    #  We also tick 'guide star selected' in the basic sequence dialogue
    #  If it is the same position as before, all we do is redisplay, allowing
    #  the display to be toggled.
    
    if {($PivNo != $SelectedGuidePivot) || 
                  ($GuideObject != $SelectedGuideObject)} {
                  
        #  If this will invalidate an allocation, check that we want to
        #  continue.
        
        if { [InvalidateAlloc] == 0 } {
        
            #  OK to go ahead
            
            set SelectedGuidePivot $PivNo
            set SelectedGuideObject $GuideObject
            set id [SdsFind $Cfid unallocGuide]
            VLTGuider $id $GuideObject posn1 posn2 pospos
            if { $PivNo == 1 } {
                set SelectedGuidePosn $posn1
            } else {
                set SelectedGuidePosn $posn2
            }
            if { $PivNo == $pospos } {
                set SelectedGuideOrient "POS"
            } else {
                set SelectedGuideOrient "NEG"
            }
            
            #  We need the object name to identify the guide object 
            #  unambiguously - the index number will change after an
            #  allocation, the position (which was used by earlier versions
            #  of the code) can change if the observation date or other
            #  astrometric quantities are changed.
            
            GetObjectData $id $GuideObject obj_name obj_ra obj_dec type \
              spect  obj_prio obj_mag obj_pid obj_comment obj_text obj_x obj_y
            set SelectedGuideName $obj_name
            
            DisableShaddowedPivots
            set outline 0
            set SelectedGuideState 0

            SetTick GuideStar
        }
    }
   
    ShowGuidePivot
}

#  -----------------------------------------------------------------------------

#                       S h o w  G u i d e  P i v o t
#
#   This procedure is called whenever the display is redrawn, or the guider
#   selection is changed, or when a guide pivot is clicked on.  It displays
#   the guider at its selected position, toggling between an outline or
#   a stippled display depending on the value of SelectedGuideState. It
#   uses SelectedGuideState internally to get the toggling effect, but it can
#   also be explicitly set by a calling routine (setting it to zero forces
#   the stippled display used when a guider position is first selected).

proc ShowGuidePivot { } {

    #  Globals used by this procedure:
    
    global zoom               ;# Current zoom factor
    global SelectedGuidePosn  ;# Position of guider with selected pivot. 
    global SelectedGuidePivot ;# Indicates which of the two possible guide pivot
                              ;# positions has been selected.
    global SelectedGuideObject;# Which guide object is selected
    global TkAvailable        ;# Set if Tk operations are permitted.
   
    #  Globals modified  by this procedure:
    
    global SelectedGuideState ;# State of guider position selection and display.
    
    if { !$TkAvailable } return
    
    #  Delete any current guide shaddow and pivot drawn on the canvas. We
    #  leave blue circles indicating the two possible pivot positions (tagged
    #  as guidepivot1 and guidepivot2) - one of them will be obscured by the
    #  blue guide pivot drawn by this procedure, but will be revealed again
    #  if the guideshaddow and guidepivot are deleted. 
    
    .mimic.scv.f1.field delete guideshaddow
    .mimic.scv.f1.field delete guidepivot
    
    #  The type of display we use depends on SelectedGuideState. Usually, zero
    #  indicates this is the first time this routine has been called for this
    #  guider position. Values of 1 and 2 are set by this routine to get the
    #  toggling effect.
    
    if {($SelectedGuideState == 0) || ($SelectedGuideState == 2)} {
    
        #  Draw the stippled display - this shows the effect of the guider
        #  better, but prevents obscured objects from being clicked upon.
        
        VLTDrawGuider $SelectedGuidePosn $zoom pivot oval polygon polygon_crack
        eval .mimic.scv.f1.field create oval $oval -fill black -stipple gray12 \
                                                   -tag guideshaddow
        eval .mimic.scv.f1.field create polygon $polygon -stipple gray12 \
                                                   -tag guideshaddow     
        eval .mimic.scv.f1.field create polygon $polygon_crack -stipple gray12 \
                                                   -tag guideshaddow     
        eval .mimic.scv.f1.field create oval $pivot -fill red -tag guidepivot
        set SelectedGuideState 1
        
    } else { 
    
        #  Draw the outline display. This is less clear, but allows the
        #  obscured objects to be selected using the mouse.
           
        VLTDrawGuiderOutline $SelectedGuidePosn $zoom pivot lines arc lines_crack
        eval .mimic.scv.f1.field create line $lines -fill red \
                                                   -tag guideshaddow
        eval .mimic.scv.f1.field create line $lines_crack -fill red \
                                                   -tag guideshaddow
        eval .mimic.scv.f1.field create arc \
                         $arc -style arc -outline red -tag guideshaddow
        eval .mimic.scv.f1.field create oval $pivot -fill red -tag guidepivot
        set SelectedGuideState 2
    }
    
    #  Whichever display was used, make sure clicking on the pivot will
    #  have the effect of selecting the guide position again - which will
    #  cause SetGuidePivot to do little more than invoke this routine to
    #  toggle the display.
    
    .mimic.scv.f1.field bind guidepivot <Button-1> \
                     " SetGuidePivot $SelectedGuidePivot $SelectedGuideObject "
}

#  -----------------------------------------------------------------------------

#               D i s a b l e  S h a d d o w e d  P i v o t s
#
#   This procedure should be run whenever the guide shaddowing is changed.
#   It works through all the currently unallocated targets, disabling those
#   that are shaddowed given the current guider position, and enabling
#   all those that are not.  

proc DisableShaddowedPivots {} {

    #  Globals used by this procedure:

    global Cfid              ;# Sds Id for the top level of the current
                             ;# configuration structure.
    global SelectedGuidePivot;# Indicates which of the two possible guide pivot
                             ;# positions has been selected.
    global SelectedGuidePosn ;# Position of guider with selected pivot. 
    global UnallocGui        ;# The number of unallocated guide targets.
    global UnallocObj        ;# The number of unallocated non-guide, non-sky,
                             ;# targets.
    global UnallocSky        ;# The number of unallocated sky targets.
    global SelectedGuideDescr;# Describes guide star selection
    
    #  We want to be able to call this even if no guide position has been 
    #  selected. If SelectedGuidePivot is zero, then this is the case. We
    #  trap this and set SelectedGuidePosn to a null set of values, which
    #  VLTGuideShaddow knows should be ignored.
    
    if { $SelectedGuidePivot == 0 } {
       set SelectedGuidePosn "0 0 0 0"
    }
    
    #  First those targets in the "unallocObject"  sub-structure. These are
    #  all the sky and object targets that are unallocated. We get the number
    #  of targets from the size of one of the arrays it contains, for which
    #  'ra' is as good as any. (Earlier versions of this code assumed that
    #  the number of targets in unallocObject is $UnallocObj + $UnallocSky,
    #  but this is not true after an allocation, when the arrays have been
    #  re-ordered.
   
    set id [SdsFind $Cfid unallocObject]
    set raid [SdsFind $id ra]
    SdsInfo $raid name code ndims dims
    set n 0
    set Count 0
    while {$n < $dims(0)} {
        incr n
       
        #  VLTGuideShaddow returns "OK" or "shaddowed" depending on whether
        #  or not the target is shaddowed. The important thing is that
        #  VLTGuideShaddow changes the 'type' of each target that is shaddowed
        #  to the appropriate 'disabled' type, which means they won't be
        #  allocated to a fibre. It also enables all unshaddowed pivots.
       
        set shaddow [VLTGuideShaddow $id $n $SelectedGuidePosn]
        if { $shaddow == "shaddowed" } { incr Count }
    }
    SdsFreeId $raid
    SdsFreeId $id
   
    #  Then those targets that are in the "unallocGuide" structure. These
    #  are all the guide stars that are unallocated.
   
    set id [SdsFind $Cfid unallocGuide]
    set raid [SdsFind $id ra]
    SdsInfo $raid name code ndims dims
    set n 0
    while {$n < $dims(0)} {
        incr n
        set shaddow [VLTGuideShaddow $id $n $SelectedGuidePosn]
        if { $shaddow == "shaddowed" } { incr Count }
    }
    set SelectedGuideDescr "Selected - $Count targets shadowed"
    SdsFreeId $id
    
}

#  -----------------------------------------------------------------------------

#                   D e s e l e c t  G u i d e  P i v o t
#
#   This procedure is called to remove any settings relating to the selected
#   guide pivot. It resets things to their state before any guide star was 
#   even selected, clearing the relevant globals and deleting any objects 
#   drawn in the display that relate to the guide pivot.

proc DeselectGuidePivot { } {
   
    #  Globals set  by this procedure:
    
    global SelectedGuideState ;# State of guider position selection and display.
    global SelectedGuidePivot ;# Indicates which of the two possible guide pivot
                              ;# positions has been selected.
    global SelectedGuideObject;# Which guide object is selected
    global SelectedGuideDescr ;# Describes guide star selection
    global SelectedGuideName  ;# Name of selected guide star.
    global TkAvailable        ;# Set if Tk operations are permitted.
    
    if { !$TkAvailable } return
    
    #  Delete any current guide shaddow and pivot drawn on the canvas. Also
    #  delete the blue circles indicating the two possible pivot positions
    #  (tagged as guidepivot1 and guidepivot2). 
    
    .mimic.scv.f1.field delete guideshaddow
    .mimic.scv.f1.field delete guidepivot
    .mimic.scv.f1.field delete guidepivot1
    .mimic.scv.f1.field delete guidepivot2
    
    #  Now reset all the relevant global variables. SelectedGuidePosn is
    #  relevant but is only used if SelectedGuidePivot is non-zero, so we
    #  leave it alone.
    
    set SelectedGuidePivot 0
    set SelectedGuideState 0
    set SelectedGuideObject 0
    set SelectedGuideName "None"
    set SelectedGuideDescr "Not selected"
      
}


#        1         2         3         4         5         6         7         8
#2345678901234567890123456789012345678901234567890123456789012345678901234567890

