#if !defined  HAVE_ASCENT_RGS_H__
#define       HAVE_ASCENT_RGS_H__
// This file is part of the FXT library.
// Copyright (C) 2012, 2013, 2014, 2019, 2023, 2024 Joerg Arndt
// License: GNU General Public License version 3 or later,
// see the file COPYING.txt in the main directory.

#include "comb/is-ascent-rgs.h"

#include "comb/comb-print.h"

#include "fxttypes.h"


// whether to use arrays instead of pointers:
#define ASCENT_RGS_FIXARRAYS  // default on
// GCC 4.5.0: slight slowdown
// GCC 4.8.0: speedup
// GCC 8.3.0: speedup


class ascent_rgs
// Ascent sequences (restricted growth strings, RGS), lexicographic order.
// An ascent sequence is a sequence [d(1), d(2), ..., d(n)] where d(1)=0, d(k)>=0,
//   and d(k) <= 1 + asc([d(1), d(2), ..., d(k-1)]) and asc(.) counts the
//   number of ascents of its argument.
// See OEIS sequence A022493.
{
public:
#if !defined ASCENT_RGS_FIXARRAYS
    ulong *A;  // digits of the RGS: A[k] <= M[k-1] + 1
    ulong *M;  // m[k] = number of ascents in prefix a[0,1,..,k]
#else
    ulong A[32];  // > 2^98 ascent sequences
    ulong M[32];
#endif
    ulong n_;   // Number of digits

    ascent_rgs(const ascent_rgs&) = delete;
    ascent_rgs & operator = (const ascent_rgs&) = delete;

public:
    explicit ascent_rgs(ulong n)
    {
        n_ = (n > 0 ? n : 1);  // shall work for n==0
#if !defined ASCENT_RGS_FIXARRAYS
        A = new ulong[n_];
        M = new ulong[n_];
#endif
        first();
    }

    ~ascent_rgs()
    {
#if !defined ASCENT_RGS_FIXARRAYS
        delete [] A;
        delete [] M;
#endif
    }

    const ulong * data()  const  { return A; }

    void first()
    {
        for (ulong k=0; k<n_; ++k)  A[k] = 0;
        for (ulong k=0; k<n_; ++k)  M[k] = 0;
    }

    void last()
    {
        for (ulong k=0; k<n_; ++k)  A[k] = k;
        for (ulong k=0; k<n_; ++k)  M[k] = k;
    }

    ulong next()
    // Return position of leftmost change, zero with last.
    {
        ulong j = n_ - 1;
        while ( j != 0 )
        {
            if ( A[j] <= M[j-1] )  break;  // can increment
            A[j] = 0;
            --j;
        }

        if ( j==0 )  return 0;  // current is last

        ++A[j];

        const ulong na = M[j-1] + ( A[j] > A[j-1] );  // ascents: A022493
        for (ulong z=j; z<n_; ++z)  M[z] = na;

//        ulong na = M[j-1] + ( A[j] != A[j-1] );  // changes: A000522
//        M[j] = na;
//        for (ulong z=j+1; z<n_; ++z) { na+=(A[z-1]!=A[z]);  M[z] = na; }

//        ulong na = M[j-1] + ( A[j] < A[j-1] );  // descents: A225588
//        M[j] = na;
//        na += 1;
//        for (ulong z=j+1; z<n_; ++z)  M[z] = na;

        return j;
    }

    ulong prev()
    // Return position of leftmost change, zero with first.
    {
        ulong j = n_ - 1;
        while ( j != 0 )
        {
            if ( A[j] != 0 )  break;  // can decrement
            --j;
        }

        if ( j==0 )  return 0;  // current is first

        --A[j];
        M[j] = M[j-1] + ( A[j] > A[j-1] );

        ulong i = j;
        while ( ++i < n_ )
        {
            const ulong na = M[i-1] + 1;
            A[i] = na;
            M[i] = na;
        }
        return j;
    }

    ulong num_ascents()  const
    // Return number of ascents.
    { return  M[n_-1]; }

    void print(const char *bla, bool dfz=true)  const
    { print_vec(bla, data(), n_, dfz); }

    bool OK()  const
    {
        // ascent stats in m[] OK?
        if ( M[0] != 0 )  return false;
        ulong na = 0;
        for (ulong j=1; j<n_; ++j)
        {
            na += ( A[j] > A[j-1] );
            if ( M[j] != na )  return false;
        }

        return  is_ascent_rgs(A, n_);
    }
};
// -------------------------


#endif  // !defined HAVE_ASCENT_RGS_H__
