/* $Id: gic_structs.h,v 1.6 2005/07/30 16:57:37 cegger Exp $
******************************************************************************

   Generic Input Configuration library for GII.

   Copyright (C) 1999 Andreas Beck	[becka@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#ifndef _GGI_GIC_STRUCTS_H
#define _GGI_GIC_STRUCTS_H

#include <stdio.h>
#include <stdlib.h>
#include <ggi/gii.h>

typedef int32_t gic_state;
#define GIC_STATE_MIN (0)
#define GIC_STATE_MAX (0x7fffffff)
#define GIC_STATE_MIDDLE ((GIC_STATE_MIN+GIC_STATE_MAX)/2)
#define GIC_NOACTION  (-1)

typedef uint32_t gic_flag;
#define GIC_FLAG_MUSTKNOWMASK	0x0000ffff
#define GIC_FLAG_PULSE		0x00000001	/* This event is "pulsed" */

/* We support several "levels" of conflict.
 */
enum gic_conflict {

	/* Both recognizers are compatible, as far as we can tell. 
	 */
	GIC_C_NOCONFLICT=0,

	/* There might be a conflict, but that information is only due to 
	 * a heuristic and thus unreliable.
	 */
	GIC_C_MAYBECONFLICT	=0x0100,

	/* Both recognizers cannot be active at the same time. This is usually
	 * desireable for things like forward/backward, but e.g. not for
	 * turn-left/fire. You might want to warn the user.
	 */
	GIC_C_NOTATSAMETIME	=0x0200,

	/* One of the checked recognizers implies the other. That is, when
	 * the "mightier" one gets active, the other gets active as well.
	 * This is usually the level of error you might want to use for
	 * auto-detaching, if multi-binding is not desired.
	 */
	GIC_C_ALWAYSATSAMETIME	=0x0300,

	/* Both recognizers are equivalent.
	 */
	GIC_C_ISSAME		=0x0400,

	/* Just a const to detect bogus values or newer protocols.
	 */
	GIC_C_LAST
};

struct gic_recognizer;
struct gic_feature;	/* forward declaration */

typedef struct gic_recognizerdriver {

	const  char *drivername;	/* DLL responsible for this control */
	int	  (*check)         (gic_handle_t hand, struct gic_recognizer *ctrl,gii_event *event, struct gic_feature *feature,int recnum);
	int	  (*get_name)      (gic_handle_t hand, struct gic_recognizer *ctrl,char *string, size_t maxlen);

	int	  (*write_pvtdata) (gic_handle_t hand, struct gic_recognizer *ctrl,char *string,int maxlen);
	int	  (*read_pvtdata)  (gic_handle_t hand, struct gic_recognizer *ctrl,const char *string);
	void	  (*free_pvtdata)  (gic_handle_t hand, struct gic_recognizer *ctrl);

	int	  (*train)         (gic_handle_t hand, struct gic_recognizer **ctrl,gii_event *event);
	int	  (*check_conflict)(gic_handle_t hand, struct gic_recognizer *ctrl,struct gic_recognizer *otherctrl);
	int	  (*get_opposite)  (gic_handle_t hand, struct gic_recognizer *recognizer,struct gic_recognizer **opposite);

} gic_recognizerdriver;

typedef struct gic_recognizer {
	struct gic_recognizer	*next;
	gic_recognizerdriver	*driver;
	void *privdata;
	gic_state	confidence;	/* For training */
} gic_recognizer;

typedef struct gic_actionlist {
	struct gic_actionlist	*next;
	const char *name;
	void (*action)(gic_handle_t hand, struct gic_actionlist *action,
			struct gic_feature *feature, gic_state newstate,gic_flag flag,int recnum);
	void *privdata;
} gic_actionlist;

/*
 * Stage 4: Features.
 * Features are individual "moves" that control something. They can have multiple
 * actions and recognizers attached.
 */

typedef struct gic_feature {
	char name[65];				/* name of the feature */
	char shortname[5];			/* name of the feature */
	gic_recognizer	*recognizers;		/* list of attached recognizers */
	gic_actionlist	*actions;		/* list of attached actions */
} gic_feature;

typedef struct gic_featurelist {
	struct gic_featurelist	*next;
	gic_feature	*feature;
} gic_featurelist;

/*
 * Stage 3: Controls.
 * Controls contain all "Features" that make up a given set like all things
 * you might need to do for navigating a menu.
 */

typedef struct gic_control {
	char 	name[65];			/* name of the control */
	char 	shortname[5];			/* name of the control */
	gic_featurelist		*features;	/* list of attached features */
} gic_control;

typedef struct gic_controllist {
	struct gic_controllist	*next;
	gic_control	*control;
} gic_controllist;

/*
 * Stage 2: Contexts.
 * Contexts describe _all_ "Controls" that are applicable in a given state of
 * an application.
 */

typedef struct gic_context {
	struct gic_context	*next;	/* linked list of all available contexts for auto-destroy */
	char 	name[65];			/* name of the context */
	gic_controllist		*controls;	/* list of attached controls */
} gic_context;

typedef struct gic_contextlist {
	struct gic_contextlist	*next;
	gic_context	*context;
} gic_contextlist;

/*
 * Stage 1: Heads.
 * This is a convenience structure that can be used to group contexts.
 * There are some convenience functions that will load/restore complete heads
 * allowing to save configuration with a single call.
 */

typedef struct gic_head {
	char 		 name[65];	/* name of the context */
	gic_contextlist	*contexts;	/* list of attached contexts */
} gic_head;

#endif /* _GGI_GIC_STRUCTS_H */
