# -*- coding: utf-8 -*-
"""
ctypeslib2 package

ctypeslib2 helps generate Python-friendly interfaces to C libraries, such as automatically generating Python classes
that represent C data structures, simplifying the process of wrapping C functions with Python functions,
and providing tools for handling errors and exceptions that may occur when calling C functions.

"""

import ctypes
import os
import re
import sys
import warnings
from ctypes.util import find_library

from pkg_resources import get_distribution, DistributionNotFound

try:
    __dist = get_distribution('ctypeslib2')
    # Normalize case for Windows systems
    # if you are in a virtualenv, ./local/* are aliases to ./*
    __dist_loc = os.path.normcase(os.path.realpath(__dist.location))
    __here = os.path.normcase(os.path.realpath(__file__))
    if not __here.startswith(os.path.join(__dist_loc, 'ctypeslib')):
        # not installed, but there is another version that *is*
        raise DistributionNotFound
except DistributionNotFound:
    __version__ = 'Please install the latest version of this python package'
else:
    __version__ = __dist.version


def clang_version():
    """Pull the clang C library version from the API"""
    # avoid loading the cindex API (cindex.conf.lib) to avoid version conflicts
    get_version = cindex.conf.get_cindex_library().clang_getClangVersion
    get_version.restype = ctypes.c_char_p
    version_string = get_version()
    version = 'Unknown'
    if version_string and len(version_string) > 0:
        version_groups = re.match(br'.+version ((\d+\.)?(\d+\.)?(\*|\d+))', version_string)
        if version_groups and len(version_groups.groups()) > 0:
            version = version_groups.group(1).decode()
    return version


def clang_py_version():
    """Return the python clang package version"""
    return __clang_py_version__


def __configure_clang_cindex():
    # The (MacPorts) installed clang module already knows (and depends upon)
    # the appropriate libclang; don't go searching for something else.
    return cindex.Config().get_filename()


# check which clang python module is available
# check which clang library is available
try:
    from clang import cindex
    from ctypeslib.codegen.codegenerator import translate, translate_files

    __clang_py_version__ = get_distribution('clang').version
    __clang_library_filename = __configure_clang_cindex()
    if __clang_library_filename is None:
        warnings.warn("Could not find the clang library. please install llvm libclang", RuntimeWarning)
        # do not fail - maybe the user has a plan
    else:
        # set a warning if major versions differs.
        if clang_version().split('.')[0] != clang_py_version().split('.')[0]:
            clang_major = clang_version().split('.')[0]
            warnings.warn(f"Version of python-clang ({clang_py_version()}) and "
                          f"clang C library ({clang_version()}) are different. "
                          f"Did you try pip install clang=={clang_major}.*", RuntimeWarning)
except ImportError as e:
    __clang_py_version__ = None
    warnings.warn("Could not find a version of python-clang installed. "
                  "Please pip install clang==<version>.*", RuntimeWarning)
    raise e


__all__ = ['translate', 'translate_files', 'clang_version', 'clang_py_version']
