/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2020      Evan Nemerson <evan@nemerson.com>
 *   2020      Himanshi Mathur <himanshi18037@iiitd.ac.in>
 *   2020      Hidayat Khan <huk2209@gmail.com>
 */

#define SIMDE_TEST_X86_AVX512_INSN div

#include <test/x86/avx512/test-avx512.h>
#include <simde/x86/avx512/set.h>
#include <simde/x86/avx512/div.h>

static int
test_simde_mm512_div_ps(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512 a;
    simde__m512 b;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   653.62), SIMDE_FLOAT32_C(   981.74), SIMDE_FLOAT32_C(   780.10), SIMDE_FLOAT32_C(    59.38),
                         SIMDE_FLOAT32_C(  -795.11), SIMDE_FLOAT32_C(   923.87), SIMDE_FLOAT32_C(  -270.01), SIMDE_FLOAT32_C(  -411.99),
                         SIMDE_FLOAT32_C(   -97.83), SIMDE_FLOAT32_C(  -393.82), SIMDE_FLOAT32_C(   934.81), SIMDE_FLOAT32_C(    74.53),
                         SIMDE_FLOAT32_C(   843.79), SIMDE_FLOAT32_C(   465.05), SIMDE_FLOAT32_C(   -42.07), SIMDE_FLOAT32_C(  -685.83)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   596.54), SIMDE_FLOAT32_C(  -116.40), SIMDE_FLOAT32_C(  -989.77), SIMDE_FLOAT32_C(  -794.40),
                         SIMDE_FLOAT32_C(   183.38), SIMDE_FLOAT32_C(  -185.75), SIMDE_FLOAT32_C(   429.70), SIMDE_FLOAT32_C(   664.04),
                         SIMDE_FLOAT32_C(   296.78), SIMDE_FLOAT32_C(  -698.78), SIMDE_FLOAT32_C(   908.33), SIMDE_FLOAT32_C(   181.85),
                         SIMDE_FLOAT32_C(  -397.89), SIMDE_FLOAT32_C(  -586.75), SIMDE_FLOAT32_C(   904.99), SIMDE_FLOAT32_C(  -321.15)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     1.10), SIMDE_FLOAT32_C(    -8.43), SIMDE_FLOAT32_C(    -0.79), SIMDE_FLOAT32_C(    -0.07),
                         SIMDE_FLOAT32_C(    -4.34), SIMDE_FLOAT32_C(    -4.97), SIMDE_FLOAT32_C(    -0.63), SIMDE_FLOAT32_C(    -0.62),
                         SIMDE_FLOAT32_C(    -0.33), SIMDE_FLOAT32_C(     0.56), SIMDE_FLOAT32_C(     1.03), SIMDE_FLOAT32_C(     0.41),
                         SIMDE_FLOAT32_C(    -2.12), SIMDE_FLOAT32_C(    -0.79), SIMDE_FLOAT32_C(    -0.05), SIMDE_FLOAT32_C(     2.14)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   729.63), SIMDE_FLOAT32_C(  -908.06), SIMDE_FLOAT32_C(  -769.77), SIMDE_FLOAT32_C(   -70.66),
                         SIMDE_FLOAT32_C(   482.71), SIMDE_FLOAT32_C(   244.66), SIMDE_FLOAT32_C(  -615.83), SIMDE_FLOAT32_C(   841.42),
                         SIMDE_FLOAT32_C(  -571.10), SIMDE_FLOAT32_C(   971.96), SIMDE_FLOAT32_C(   149.38), SIMDE_FLOAT32_C(   497.71),
                         SIMDE_FLOAT32_C(   988.69), SIMDE_FLOAT32_C(   479.68), SIMDE_FLOAT32_C(  -128.24), SIMDE_FLOAT32_C(   585.28)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   359.65), SIMDE_FLOAT32_C(  -730.08), SIMDE_FLOAT32_C(   977.98), SIMDE_FLOAT32_C(  -215.53),
                         SIMDE_FLOAT32_C(  -315.50), SIMDE_FLOAT32_C(    80.64), SIMDE_FLOAT32_C(  -996.10), SIMDE_FLOAT32_C(  -556.83),
                         SIMDE_FLOAT32_C(  -628.68), SIMDE_FLOAT32_C(   938.60), SIMDE_FLOAT32_C(  -147.98), SIMDE_FLOAT32_C(   378.31),
                         SIMDE_FLOAT32_C(   246.47), SIMDE_FLOAT32_C(   109.18), SIMDE_FLOAT32_C(  -575.64), SIMDE_FLOAT32_C(  -426.86)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     2.03), SIMDE_FLOAT32_C(     1.24), SIMDE_FLOAT32_C(    -0.79), SIMDE_FLOAT32_C(     0.33),
                         SIMDE_FLOAT32_C(    -1.53), SIMDE_FLOAT32_C(     3.03), SIMDE_FLOAT32_C(     0.62), SIMDE_FLOAT32_C(    -1.51),
                         SIMDE_FLOAT32_C(     0.91), SIMDE_FLOAT32_C(     1.04), SIMDE_FLOAT32_C(    -1.01), SIMDE_FLOAT32_C(     1.32),
                         SIMDE_FLOAT32_C(     4.01), SIMDE_FLOAT32_C(     4.39), SIMDE_FLOAT32_C(     0.22), SIMDE_FLOAT32_C(    -1.37)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -148.70), SIMDE_FLOAT32_C(  -327.17), SIMDE_FLOAT32_C(  -310.14), SIMDE_FLOAT32_C(  -718.80),
                         SIMDE_FLOAT32_C(   382.69), SIMDE_FLOAT32_C(  -181.61), SIMDE_FLOAT32_C(  -214.09), SIMDE_FLOAT32_C(    55.72),
                         SIMDE_FLOAT32_C(   438.03), SIMDE_FLOAT32_C(  -458.01), SIMDE_FLOAT32_C(   144.59), SIMDE_FLOAT32_C(   165.00),
                         SIMDE_FLOAT32_C(  -331.04), SIMDE_FLOAT32_C(   406.96), SIMDE_FLOAT32_C(  -326.43), SIMDE_FLOAT32_C(   373.82)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   791.83), SIMDE_FLOAT32_C(   191.69), SIMDE_FLOAT32_C(  -460.58), SIMDE_FLOAT32_C(  -915.08),
                         SIMDE_FLOAT32_C(  -877.38), SIMDE_FLOAT32_C(  -915.27), SIMDE_FLOAT32_C(   207.85), SIMDE_FLOAT32_C(   567.35),
                         SIMDE_FLOAT32_C(   304.30), SIMDE_FLOAT32_C(  -777.07), SIMDE_FLOAT32_C(  -683.73), SIMDE_FLOAT32_C(  -113.32),
                         SIMDE_FLOAT32_C(  -701.16), SIMDE_FLOAT32_C(  -942.92), SIMDE_FLOAT32_C(  -489.97), SIMDE_FLOAT32_C(   911.34)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    -0.19), SIMDE_FLOAT32_C(    -1.71), SIMDE_FLOAT32_C(     0.67), SIMDE_FLOAT32_C(     0.79),
                         SIMDE_FLOAT32_C(    -0.44), SIMDE_FLOAT32_C(     0.20), SIMDE_FLOAT32_C(    -1.03), SIMDE_FLOAT32_C(     0.10),
                         SIMDE_FLOAT32_C(     1.44), SIMDE_FLOAT32_C(     0.59), SIMDE_FLOAT32_C(    -0.21), SIMDE_FLOAT32_C(    -1.46),
                         SIMDE_FLOAT32_C(     0.47), SIMDE_FLOAT32_C(    -0.43), SIMDE_FLOAT32_C(     0.67), SIMDE_FLOAT32_C(     0.41)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -869.58), SIMDE_FLOAT32_C(   763.75), SIMDE_FLOAT32_C(  -558.93), SIMDE_FLOAT32_C(   756.19),
                         SIMDE_FLOAT32_C(   509.82), SIMDE_FLOAT32_C(  -855.71), SIMDE_FLOAT32_C(  -965.40), SIMDE_FLOAT32_C(  -279.29),
                         SIMDE_FLOAT32_C(  -798.08), SIMDE_FLOAT32_C(   256.40), SIMDE_FLOAT32_C(   739.89), SIMDE_FLOAT32_C(  -903.46),
                         SIMDE_FLOAT32_C(  -771.75), SIMDE_FLOAT32_C(   -54.77), SIMDE_FLOAT32_C(   397.04), SIMDE_FLOAT32_C(   925.94)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -355.51), SIMDE_FLOAT32_C(   136.73), SIMDE_FLOAT32_C(   586.70), SIMDE_FLOAT32_C(   712.56),
                         SIMDE_FLOAT32_C(   135.88), SIMDE_FLOAT32_C(  -693.91), SIMDE_FLOAT32_C(  -131.33), SIMDE_FLOAT32_C(  -933.79),
                         SIMDE_FLOAT32_C(   864.29), SIMDE_FLOAT32_C(  -834.00), SIMDE_FLOAT32_C(   475.52), SIMDE_FLOAT32_C(   502.31),
                         SIMDE_FLOAT32_C(  -746.87), SIMDE_FLOAT32_C(  -364.10), SIMDE_FLOAT32_C(  -995.18), SIMDE_FLOAT32_C(   683.54)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     2.45), SIMDE_FLOAT32_C(     5.59), SIMDE_FLOAT32_C(    -0.95), SIMDE_FLOAT32_C(     1.06),
                         SIMDE_FLOAT32_C(     3.75), SIMDE_FLOAT32_C(     1.23), SIMDE_FLOAT32_C(     7.35), SIMDE_FLOAT32_C(     0.30),
                         SIMDE_FLOAT32_C(    -0.92), SIMDE_FLOAT32_C(    -0.31), SIMDE_FLOAT32_C(     1.56), SIMDE_FLOAT32_C(    -1.80),
                         SIMDE_FLOAT32_C(     1.03), SIMDE_FLOAT32_C(     0.15), SIMDE_FLOAT32_C(    -0.40), SIMDE_FLOAT32_C(     1.35)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   119.21), SIMDE_FLOAT32_C(   360.54), SIMDE_FLOAT32_C(   885.26), SIMDE_FLOAT32_C(  -618.98),
                         SIMDE_FLOAT32_C(    -8.97), SIMDE_FLOAT32_C(  -881.58), SIMDE_FLOAT32_C(   -89.25), SIMDE_FLOAT32_C(  -937.64),
                         SIMDE_FLOAT32_C(  -660.18), SIMDE_FLOAT32_C(  -649.17), SIMDE_FLOAT32_C(  -279.52), SIMDE_FLOAT32_C(   812.95),
                         SIMDE_FLOAT32_C(  -471.80), SIMDE_FLOAT32_C(   805.98), SIMDE_FLOAT32_C(   532.44), SIMDE_FLOAT32_C(   126.30)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   944.81), SIMDE_FLOAT32_C(   946.29), SIMDE_FLOAT32_C(   161.37), SIMDE_FLOAT32_C(  -637.11),
                         SIMDE_FLOAT32_C(    16.54), SIMDE_FLOAT32_C(   417.79), SIMDE_FLOAT32_C(   257.34), SIMDE_FLOAT32_C(  -857.05),
                         SIMDE_FLOAT32_C(   770.17), SIMDE_FLOAT32_C(  -559.67), SIMDE_FLOAT32_C(  -862.75), SIMDE_FLOAT32_C(  -541.96),
                         SIMDE_FLOAT32_C(   412.30), SIMDE_FLOAT32_C(  -147.64), SIMDE_FLOAT32_C(   553.94), SIMDE_FLOAT32_C(  -736.63)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.13), SIMDE_FLOAT32_C(     0.38), SIMDE_FLOAT32_C(     5.49), SIMDE_FLOAT32_C(     0.97),
                         SIMDE_FLOAT32_C(    -0.54), SIMDE_FLOAT32_C(    -2.11), SIMDE_FLOAT32_C(    -0.35), SIMDE_FLOAT32_C(     1.09),
                         SIMDE_FLOAT32_C(    -0.86), SIMDE_FLOAT32_C(     1.16), SIMDE_FLOAT32_C(     0.32), SIMDE_FLOAT32_C(    -1.50),
                         SIMDE_FLOAT32_C(    -1.14), SIMDE_FLOAT32_C(    -5.46), SIMDE_FLOAT32_C(     0.96), SIMDE_FLOAT32_C(    -0.17)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   -81.24), SIMDE_FLOAT32_C(  -934.88), SIMDE_FLOAT32_C(   -84.21), SIMDE_FLOAT32_C(  -265.16),
                         SIMDE_FLOAT32_C(  -978.34), SIMDE_FLOAT32_C(  -425.47), SIMDE_FLOAT32_C(   792.31), SIMDE_FLOAT32_C(  -306.03),
                         SIMDE_FLOAT32_C(   911.07), SIMDE_FLOAT32_C(   992.01), SIMDE_FLOAT32_C(   172.45), SIMDE_FLOAT32_C(  -135.31),
                         SIMDE_FLOAT32_C(   652.11), SIMDE_FLOAT32_C(  -529.15), SIMDE_FLOAT32_C(    -0.58), SIMDE_FLOAT32_C(   883.05)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -110.89), SIMDE_FLOAT32_C(  -325.07), SIMDE_FLOAT32_C(   834.96), SIMDE_FLOAT32_C(  -681.06),
                         SIMDE_FLOAT32_C(  -877.63), SIMDE_FLOAT32_C(  -653.45), SIMDE_FLOAT32_C(    40.48), SIMDE_FLOAT32_C(  -644.02),
                         SIMDE_FLOAT32_C(  -687.76), SIMDE_FLOAT32_C(  -660.68), SIMDE_FLOAT32_C(   802.46), SIMDE_FLOAT32_C(  -477.95),
                         SIMDE_FLOAT32_C(  -125.80), SIMDE_FLOAT32_C(  -475.50), SIMDE_FLOAT32_C(  -806.50), SIMDE_FLOAT32_C(  -778.62)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.73), SIMDE_FLOAT32_C(     2.88), SIMDE_FLOAT32_C(    -0.10), SIMDE_FLOAT32_C(     0.39),
                         SIMDE_FLOAT32_C(     1.11), SIMDE_FLOAT32_C(     0.65), SIMDE_FLOAT32_C(    19.57), SIMDE_FLOAT32_C(     0.48),
                         SIMDE_FLOAT32_C(    -1.32), SIMDE_FLOAT32_C(    -1.50), SIMDE_FLOAT32_C(     0.21), SIMDE_FLOAT32_C(     0.28),
                         SIMDE_FLOAT32_C(    -5.18), SIMDE_FLOAT32_C(     1.11), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -1.13)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -387.95), SIMDE_FLOAT32_C(   255.55), SIMDE_FLOAT32_C(   948.01), SIMDE_FLOAT32_C(   520.84),
                         SIMDE_FLOAT32_C(   310.00), SIMDE_FLOAT32_C(  -412.39), SIMDE_FLOAT32_C(   412.17), SIMDE_FLOAT32_C(  -913.22),
                         SIMDE_FLOAT32_C(   810.06), SIMDE_FLOAT32_C(  -696.65), SIMDE_FLOAT32_C(   807.84), SIMDE_FLOAT32_C(    63.85),
                         SIMDE_FLOAT32_C(    -2.75), SIMDE_FLOAT32_C(  -763.61), SIMDE_FLOAT32_C(  -850.85), SIMDE_FLOAT32_C(   913.88)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -915.78), SIMDE_FLOAT32_C(   471.39), SIMDE_FLOAT32_C(  -324.79), SIMDE_FLOAT32_C(  -855.69),
                         SIMDE_FLOAT32_C(   966.81), SIMDE_FLOAT32_C(   668.44), SIMDE_FLOAT32_C(   925.33), SIMDE_FLOAT32_C(   564.88),
                         SIMDE_FLOAT32_C(  -130.24), SIMDE_FLOAT32_C(   360.71), SIMDE_FLOAT32_C(   966.21), SIMDE_FLOAT32_C(  -919.67),
                         SIMDE_FLOAT32_C(   198.47), SIMDE_FLOAT32_C(  -796.49), SIMDE_FLOAT32_C(   428.08), SIMDE_FLOAT32_C(   264.02)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.42), SIMDE_FLOAT32_C(     0.54), SIMDE_FLOAT32_C(    -2.92), SIMDE_FLOAT32_C(    -0.61),
                         SIMDE_FLOAT32_C(     0.32), SIMDE_FLOAT32_C(    -0.62), SIMDE_FLOAT32_C(     0.45), SIMDE_FLOAT32_C(    -1.62),
                         SIMDE_FLOAT32_C(    -6.22), SIMDE_FLOAT32_C(    -1.93), SIMDE_FLOAT32_C(     0.84), SIMDE_FLOAT32_C(    -0.07),
                         SIMDE_FLOAT32_C(    -0.01), SIMDE_FLOAT32_C(     0.96), SIMDE_FLOAT32_C(    -1.99), SIMDE_FLOAT32_C(     3.46)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   534.55), SIMDE_FLOAT32_C(  -263.46), SIMDE_FLOAT32_C(  -958.21), SIMDE_FLOAT32_C(   927.39),
                         SIMDE_FLOAT32_C(   830.49), SIMDE_FLOAT32_C(  -394.19), SIMDE_FLOAT32_C(  -755.65), SIMDE_FLOAT32_C(  -594.24),
                         SIMDE_FLOAT32_C(  -371.00), SIMDE_FLOAT32_C(   623.04), SIMDE_FLOAT32_C(   879.76), SIMDE_FLOAT32_C(   838.28),
                         SIMDE_FLOAT32_C(  -100.77), SIMDE_FLOAT32_C(  -708.14), SIMDE_FLOAT32_C(  -206.06), SIMDE_FLOAT32_C(  -203.03)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    65.94), SIMDE_FLOAT32_C(   158.39), SIMDE_FLOAT32_C(   532.17), SIMDE_FLOAT32_C(    -1.61),
                         SIMDE_FLOAT32_C(  -802.21), SIMDE_FLOAT32_C(  -782.13), SIMDE_FLOAT32_C(   831.96), SIMDE_FLOAT32_C(  -692.14),
                         SIMDE_FLOAT32_C(   581.38), SIMDE_FLOAT32_C(   943.65), SIMDE_FLOAT32_C(   585.87), SIMDE_FLOAT32_C(   329.94),
                         SIMDE_FLOAT32_C(  -747.39), SIMDE_FLOAT32_C(   976.32), SIMDE_FLOAT32_C(   362.23), SIMDE_FLOAT32_C(  -137.03)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     8.11), SIMDE_FLOAT32_C(    -1.66), SIMDE_FLOAT32_C(    -1.80), SIMDE_FLOAT32_C(  -576.02),
                         SIMDE_FLOAT32_C(    -1.04), SIMDE_FLOAT32_C(     0.50), SIMDE_FLOAT32_C(    -0.91), SIMDE_FLOAT32_C(     0.86),
                         SIMDE_FLOAT32_C(    -0.64), SIMDE_FLOAT32_C(     0.66), SIMDE_FLOAT32_C(     1.50), SIMDE_FLOAT32_C(     2.54),
                         SIMDE_FLOAT32_C(     0.13), SIMDE_FLOAT32_C(    -0.73), SIMDE_FLOAT32_C(    -0.57), SIMDE_FLOAT32_C(     1.48)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_div_ps(test_vec[i].a, test_vec[i].b);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_mask_div_ps(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512 src;
    simde__mmask16 k;
    simde__m512 a;
    simde__m512 b;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -745.89), SIMDE_FLOAT32_C(   663.97), SIMDE_FLOAT32_C(   886.69), SIMDE_FLOAT32_C(  -271.39),
                         SIMDE_FLOAT32_C(   845.36), SIMDE_FLOAT32_C(  -391.34), SIMDE_FLOAT32_C(  -606.86), SIMDE_FLOAT32_C(   818.59),
                         SIMDE_FLOAT32_C(   953.36), SIMDE_FLOAT32_C(   863.40), SIMDE_FLOAT32_C(   241.85), SIMDE_FLOAT32_C(  -815.86),
                         SIMDE_FLOAT32_C(   460.12), SIMDE_FLOAT32_C(  -674.64), SIMDE_FLOAT32_C(   868.62), SIMDE_FLOAT32_C(  -710.40)),
      UINT16_C( 9207),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -956.83), SIMDE_FLOAT32_C(  -855.01), SIMDE_FLOAT32_C(  -219.41), SIMDE_FLOAT32_C(    94.89),
                         SIMDE_FLOAT32_C(  -270.85), SIMDE_FLOAT32_C(   356.85), SIMDE_FLOAT32_C(   872.24), SIMDE_FLOAT32_C(   100.53),
                         SIMDE_FLOAT32_C(   234.39), SIMDE_FLOAT32_C(  -639.13), SIMDE_FLOAT32_C(   981.49), SIMDE_FLOAT32_C(   706.62),
                         SIMDE_FLOAT32_C(  -983.90), SIMDE_FLOAT32_C(   124.15), SIMDE_FLOAT32_C(  -820.87), SIMDE_FLOAT32_C(  -946.81)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   -17.46), SIMDE_FLOAT32_C(   371.67), SIMDE_FLOAT32_C(   390.77), SIMDE_FLOAT32_C(  -132.40),
                         SIMDE_FLOAT32_C(   276.69), SIMDE_FLOAT32_C(  -338.80), SIMDE_FLOAT32_C(   359.09), SIMDE_FLOAT32_C(  -631.66),
                         SIMDE_FLOAT32_C(  -455.96), SIMDE_FLOAT32_C(    16.63), SIMDE_FLOAT32_C(   194.96), SIMDE_FLOAT32_C(  -407.18),
                         SIMDE_FLOAT32_C(  -447.59), SIMDE_FLOAT32_C(  -276.48), SIMDE_FLOAT32_C(   631.98), SIMDE_FLOAT32_C(   430.67)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -745.89), SIMDE_FLOAT32_C(   663.97), SIMDE_FLOAT32_C(    -0.56), SIMDE_FLOAT32_C(  -271.39),
                         SIMDE_FLOAT32_C(   845.36), SIMDE_FLOAT32_C(  -391.34), SIMDE_FLOAT32_C(     2.43), SIMDE_FLOAT32_C(    -0.16),
                         SIMDE_FLOAT32_C(    -0.51), SIMDE_FLOAT32_C(   -38.43), SIMDE_FLOAT32_C(     5.03), SIMDE_FLOAT32_C(    -1.74),
                         SIMDE_FLOAT32_C(   460.12), SIMDE_FLOAT32_C(    -0.45), SIMDE_FLOAT32_C(    -1.30), SIMDE_FLOAT32_C(    -2.20)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   769.85), SIMDE_FLOAT32_C(   -75.51), SIMDE_FLOAT32_C(   417.80), SIMDE_FLOAT32_C(   327.87),
                         SIMDE_FLOAT32_C(   287.24), SIMDE_FLOAT32_C(  -627.46), SIMDE_FLOAT32_C(   540.48), SIMDE_FLOAT32_C(  -625.88),
                         SIMDE_FLOAT32_C(  -108.88), SIMDE_FLOAT32_C(   663.67), SIMDE_FLOAT32_C(  -412.74), SIMDE_FLOAT32_C(  -226.36),
                         SIMDE_FLOAT32_C(   757.77), SIMDE_FLOAT32_C(  -897.40), SIMDE_FLOAT32_C(    27.15), SIMDE_FLOAT32_C(  -443.34)),
      UINT16_C(26651),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -896.67), SIMDE_FLOAT32_C(  -181.49), SIMDE_FLOAT32_C(  -338.89), SIMDE_FLOAT32_C(   -19.28),
                         SIMDE_FLOAT32_C(   886.35), SIMDE_FLOAT32_C(  -662.07), SIMDE_FLOAT32_C(   925.60), SIMDE_FLOAT32_C(   651.41),
                         SIMDE_FLOAT32_C(   597.16), SIMDE_FLOAT32_C(    67.32), SIMDE_FLOAT32_C(  -911.68), SIMDE_FLOAT32_C(   202.35),
                         SIMDE_FLOAT32_C(   208.06), SIMDE_FLOAT32_C(   747.61), SIMDE_FLOAT32_C(    81.71), SIMDE_FLOAT32_C(    40.88)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   429.04), SIMDE_FLOAT32_C(    13.69), SIMDE_FLOAT32_C(   491.03), SIMDE_FLOAT32_C(   366.42),
                         SIMDE_FLOAT32_C(  -264.60), SIMDE_FLOAT32_C(   201.75), SIMDE_FLOAT32_C(  -598.58), SIMDE_FLOAT32_C(  -939.94),
                         SIMDE_FLOAT32_C(   118.06), SIMDE_FLOAT32_C(   355.92), SIMDE_FLOAT32_C(   978.59), SIMDE_FLOAT32_C(  -224.11),
                         SIMDE_FLOAT32_C(   -71.37), SIMDE_FLOAT32_C(   333.99), SIMDE_FLOAT32_C(  -515.40), SIMDE_FLOAT32_C(   -38.06)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   769.85), SIMDE_FLOAT32_C(   -13.26), SIMDE_FLOAT32_C(    -0.69), SIMDE_FLOAT32_C(   327.87),
                         SIMDE_FLOAT32_C(    -3.35), SIMDE_FLOAT32_C(  -627.46), SIMDE_FLOAT32_C(   540.48), SIMDE_FLOAT32_C(  -625.88),
                         SIMDE_FLOAT32_C(  -108.88), SIMDE_FLOAT32_C(   663.67), SIMDE_FLOAT32_C(  -412.74), SIMDE_FLOAT32_C(    -0.90),
                         SIMDE_FLOAT32_C(    -2.92), SIMDE_FLOAT32_C(  -897.40), SIMDE_FLOAT32_C(    -0.16), SIMDE_FLOAT32_C(    -1.07)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -301.18), SIMDE_FLOAT32_C(  -952.56), SIMDE_FLOAT32_C(   361.18), SIMDE_FLOAT32_C(    53.08),
                         SIMDE_FLOAT32_C(   179.94), SIMDE_FLOAT32_C(  -914.68), SIMDE_FLOAT32_C(  -695.32), SIMDE_FLOAT32_C(  -492.39),
                         SIMDE_FLOAT32_C(   -86.02), SIMDE_FLOAT32_C(   123.88), SIMDE_FLOAT32_C(   274.86), SIMDE_FLOAT32_C(   554.74),
                         SIMDE_FLOAT32_C(  -845.80), SIMDE_FLOAT32_C(  -156.28), SIMDE_FLOAT32_C(  -737.31), SIMDE_FLOAT32_C(   590.88)),
      UINT16_C(31164),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   694.79), SIMDE_FLOAT32_C(   687.99), SIMDE_FLOAT32_C(  -648.58), SIMDE_FLOAT32_C(  -272.14),
                         SIMDE_FLOAT32_C(   -52.85), SIMDE_FLOAT32_C(  -298.63), SIMDE_FLOAT32_C(   917.21), SIMDE_FLOAT32_C(  -876.76),
                         SIMDE_FLOAT32_C(   677.44), SIMDE_FLOAT32_C(  -857.42), SIMDE_FLOAT32_C(   -56.60), SIMDE_FLOAT32_C(   488.58),
                         SIMDE_FLOAT32_C(   876.79), SIMDE_FLOAT32_C(  -578.18), SIMDE_FLOAT32_C(  -335.03), SIMDE_FLOAT32_C(   980.62)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   247.15), SIMDE_FLOAT32_C(   126.77), SIMDE_FLOAT32_C(   867.78), SIMDE_FLOAT32_C(  -450.16),
                         SIMDE_FLOAT32_C(    94.73), SIMDE_FLOAT32_C(  -587.88), SIMDE_FLOAT32_C(   776.58), SIMDE_FLOAT32_C(  -595.96),
                         SIMDE_FLOAT32_C(   345.82), SIMDE_FLOAT32_C(  -768.91), SIMDE_FLOAT32_C(   -31.17), SIMDE_FLOAT32_C(    -4.10),
                         SIMDE_FLOAT32_C(  -234.58), SIMDE_FLOAT32_C(   278.53), SIMDE_FLOAT32_C(  -336.24), SIMDE_FLOAT32_C(  -974.01)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -301.18), SIMDE_FLOAT32_C(     5.43), SIMDE_FLOAT32_C(    -0.75), SIMDE_FLOAT32_C(     0.60),
                         SIMDE_FLOAT32_C(    -0.56), SIMDE_FLOAT32_C(  -914.68), SIMDE_FLOAT32_C(  -695.32), SIMDE_FLOAT32_C(     1.47),
                         SIMDE_FLOAT32_C(     1.96), SIMDE_FLOAT32_C(   123.88), SIMDE_FLOAT32_C(     1.82), SIMDE_FLOAT32_C(  -119.17),
                         SIMDE_FLOAT32_C(    -3.74), SIMDE_FLOAT32_C(    -2.08), SIMDE_FLOAT32_C(  -737.31), SIMDE_FLOAT32_C(   590.88)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -412.81), SIMDE_FLOAT32_C(  -265.44), SIMDE_FLOAT32_C(  -550.71), SIMDE_FLOAT32_C(  -725.27),
                         SIMDE_FLOAT32_C(  -302.34), SIMDE_FLOAT32_C(  -375.85), SIMDE_FLOAT32_C(   423.25), SIMDE_FLOAT32_C(   778.83),
                         SIMDE_FLOAT32_C(   480.23), SIMDE_FLOAT32_C(  -401.59), SIMDE_FLOAT32_C(   489.09), SIMDE_FLOAT32_C(   775.60),
                         SIMDE_FLOAT32_C(  -569.06), SIMDE_FLOAT32_C(  -632.55), SIMDE_FLOAT32_C(  -156.10), SIMDE_FLOAT32_C(   658.93)),
      UINT16_C( 3671),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -474.43), SIMDE_FLOAT32_C(  -465.91), SIMDE_FLOAT32_C(   545.15), SIMDE_FLOAT32_C(   262.71),
                         SIMDE_FLOAT32_C(   599.41), SIMDE_FLOAT32_C(  -408.02), SIMDE_FLOAT32_C(  -664.44), SIMDE_FLOAT32_C(   446.87),
                         SIMDE_FLOAT32_C(   816.32), SIMDE_FLOAT32_C(   622.16), SIMDE_FLOAT32_C(    40.97), SIMDE_FLOAT32_C(  -230.30),
                         SIMDE_FLOAT32_C(   122.84), SIMDE_FLOAT32_C(   457.98), SIMDE_FLOAT32_C(  -118.87), SIMDE_FLOAT32_C(  -211.46)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   271.75), SIMDE_FLOAT32_C(   598.59), SIMDE_FLOAT32_C(  -130.09), SIMDE_FLOAT32_C(   474.47),
                         SIMDE_FLOAT32_C(   -94.60), SIMDE_FLOAT32_C(   846.28), SIMDE_FLOAT32_C(   108.99), SIMDE_FLOAT32_C(  -793.22),
                         SIMDE_FLOAT32_C(   -12.05), SIMDE_FLOAT32_C(  -325.70), SIMDE_FLOAT32_C(  -510.95), SIMDE_FLOAT32_C(   213.60),
                         SIMDE_FLOAT32_C(  -818.29), SIMDE_FLOAT32_C(  -431.12), SIMDE_FLOAT32_C(  -186.49), SIMDE_FLOAT32_C(    53.27)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -412.81), SIMDE_FLOAT32_C(  -265.44), SIMDE_FLOAT32_C(  -550.71), SIMDE_FLOAT32_C(  -725.27),
                         SIMDE_FLOAT32_C(    -6.34), SIMDE_FLOAT32_C(    -0.48), SIMDE_FLOAT32_C(    -6.10), SIMDE_FLOAT32_C(   778.83),
                         SIMDE_FLOAT32_C(   480.23), SIMDE_FLOAT32_C(    -1.91), SIMDE_FLOAT32_C(   489.09), SIMDE_FLOAT32_C(    -1.08),
                         SIMDE_FLOAT32_C(  -569.06), SIMDE_FLOAT32_C(    -1.06), SIMDE_FLOAT32_C(     0.64), SIMDE_FLOAT32_C(    -3.97)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -254.94), SIMDE_FLOAT32_C(   234.45), SIMDE_FLOAT32_C(   235.56), SIMDE_FLOAT32_C(   930.35),
                         SIMDE_FLOAT32_C(   137.93), SIMDE_FLOAT32_C(   979.46), SIMDE_FLOAT32_C(   688.15), SIMDE_FLOAT32_C(   707.95),
                         SIMDE_FLOAT32_C(    35.42), SIMDE_FLOAT32_C(   748.55), SIMDE_FLOAT32_C(   649.98), SIMDE_FLOAT32_C(   702.04),
                         SIMDE_FLOAT32_C(   443.56), SIMDE_FLOAT32_C(  -944.39), SIMDE_FLOAT32_C(   717.51), SIMDE_FLOAT32_C(   716.62)),
      UINT16_C(24144),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -982.71), SIMDE_FLOAT32_C(   639.75), SIMDE_FLOAT32_C(   842.03), SIMDE_FLOAT32_C(   717.68),
                         SIMDE_FLOAT32_C(   294.25), SIMDE_FLOAT32_C(  -411.52), SIMDE_FLOAT32_C(   632.28), SIMDE_FLOAT32_C(   531.91),
                         SIMDE_FLOAT32_C(  -198.66), SIMDE_FLOAT32_C(   722.92), SIMDE_FLOAT32_C(  -890.25), SIMDE_FLOAT32_C(   -36.77),
                         SIMDE_FLOAT32_C(  -651.17), SIMDE_FLOAT32_C(   559.24), SIMDE_FLOAT32_C(   496.39), SIMDE_FLOAT32_C(  -143.68)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -779.19), SIMDE_FLOAT32_C(   587.24), SIMDE_FLOAT32_C(   850.25), SIMDE_FLOAT32_C(   172.75),
                         SIMDE_FLOAT32_C(   237.73), SIMDE_FLOAT32_C(   792.79), SIMDE_FLOAT32_C(  -225.26), SIMDE_FLOAT32_C(   810.16),
                         SIMDE_FLOAT32_C(   235.61), SIMDE_FLOAT32_C(   123.68), SIMDE_FLOAT32_C(  -869.51), SIMDE_FLOAT32_C(   811.23),
                         SIMDE_FLOAT32_C(   292.28), SIMDE_FLOAT32_C(   158.60), SIMDE_FLOAT32_C(  -861.10), SIMDE_FLOAT32_C(   297.31)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -254.94), SIMDE_FLOAT32_C(     1.09), SIMDE_FLOAT32_C(   235.56), SIMDE_FLOAT32_C(     4.15),
                         SIMDE_FLOAT32_C(     1.24), SIMDE_FLOAT32_C(    -0.52), SIMDE_FLOAT32_C(    -2.81), SIMDE_FLOAT32_C(   707.95),
                         SIMDE_FLOAT32_C(    35.42), SIMDE_FLOAT32_C(     5.85), SIMDE_FLOAT32_C(   649.98), SIMDE_FLOAT32_C(    -0.05),
                         SIMDE_FLOAT32_C(   443.56), SIMDE_FLOAT32_C(  -944.39), SIMDE_FLOAT32_C(   717.51), SIMDE_FLOAT32_C(   716.62)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   711.46), SIMDE_FLOAT32_C(  -417.42), SIMDE_FLOAT32_C(  -736.14), SIMDE_FLOAT32_C(  -654.73),
                         SIMDE_FLOAT32_C(  -297.59), SIMDE_FLOAT32_C(   899.88), SIMDE_FLOAT32_C(   819.21), SIMDE_FLOAT32_C(  -451.55),
                         SIMDE_FLOAT32_C(   831.09), SIMDE_FLOAT32_C(   694.55), SIMDE_FLOAT32_C(  -231.88), SIMDE_FLOAT32_C(  -711.25),
                         SIMDE_FLOAT32_C(  -213.96), SIMDE_FLOAT32_C(  -411.84), SIMDE_FLOAT32_C(  -325.79), SIMDE_FLOAT32_C(  -424.22)),
      UINT16_C( 4465),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   985.56), SIMDE_FLOAT32_C(  -969.44), SIMDE_FLOAT32_C(   -91.63), SIMDE_FLOAT32_C(  -416.19),
                         SIMDE_FLOAT32_C(   716.00), SIMDE_FLOAT32_C(   579.33), SIMDE_FLOAT32_C(   678.78), SIMDE_FLOAT32_C(   650.46),
                         SIMDE_FLOAT32_C(  -988.30), SIMDE_FLOAT32_C(   206.47), SIMDE_FLOAT32_C(   214.00), SIMDE_FLOAT32_C(  -226.18),
                         SIMDE_FLOAT32_C(  -410.63), SIMDE_FLOAT32_C(  -238.02), SIMDE_FLOAT32_C(   520.82), SIMDE_FLOAT32_C(  -882.63)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   373.48), SIMDE_FLOAT32_C(  -376.14), SIMDE_FLOAT32_C(   103.99), SIMDE_FLOAT32_C(   900.82),
                         SIMDE_FLOAT32_C(   827.14), SIMDE_FLOAT32_C(   -50.15), SIMDE_FLOAT32_C(   675.06), SIMDE_FLOAT32_C(   239.90),
                         SIMDE_FLOAT32_C(   531.97), SIMDE_FLOAT32_C(    52.69), SIMDE_FLOAT32_C(  -376.06), SIMDE_FLOAT32_C(  -290.42),
                         SIMDE_FLOAT32_C(  -325.12), SIMDE_FLOAT32_C(  -471.17), SIMDE_FLOAT32_C(  -511.21), SIMDE_FLOAT32_C(   -90.11)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   711.46), SIMDE_FLOAT32_C(  -417.42), SIMDE_FLOAT32_C(  -736.14), SIMDE_FLOAT32_C(    -0.46),
                         SIMDE_FLOAT32_C(  -297.59), SIMDE_FLOAT32_C(   899.88), SIMDE_FLOAT32_C(   819.21), SIMDE_FLOAT32_C(     2.71),
                         SIMDE_FLOAT32_C(   831.09), SIMDE_FLOAT32_C(     3.92), SIMDE_FLOAT32_C(    -0.57), SIMDE_FLOAT32_C(     0.78),
                         SIMDE_FLOAT32_C(  -213.96), SIMDE_FLOAT32_C(  -411.84), SIMDE_FLOAT32_C(  -325.79), SIMDE_FLOAT32_C(     9.80)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -788.99), SIMDE_FLOAT32_C(   888.94), SIMDE_FLOAT32_C(   861.99), SIMDE_FLOAT32_C(  -655.94),
                         SIMDE_FLOAT32_C(  -815.78), SIMDE_FLOAT32_C(   460.30), SIMDE_FLOAT32_C(  -596.09), SIMDE_FLOAT32_C(   480.08),
                         SIMDE_FLOAT32_C(  -800.23), SIMDE_FLOAT32_C(  -511.53), SIMDE_FLOAT32_C(   235.71), SIMDE_FLOAT32_C(   833.52),
                         SIMDE_FLOAT32_C(   343.49), SIMDE_FLOAT32_C(   413.97), SIMDE_FLOAT32_C(   264.73), SIMDE_FLOAT32_C(   769.22)),
      UINT16_C(57880),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -782.73), SIMDE_FLOAT32_C(   -41.33), SIMDE_FLOAT32_C(   183.64), SIMDE_FLOAT32_C(     0.86),
                         SIMDE_FLOAT32_C(  -449.70), SIMDE_FLOAT32_C(   153.64), SIMDE_FLOAT32_C(   543.55), SIMDE_FLOAT32_C(  -321.17),
                         SIMDE_FLOAT32_C(   944.46), SIMDE_FLOAT32_C(  -863.15), SIMDE_FLOAT32_C(   155.57), SIMDE_FLOAT32_C(   671.09),
                         SIMDE_FLOAT32_C(   138.46), SIMDE_FLOAT32_C(   937.90), SIMDE_FLOAT32_C(   367.36), SIMDE_FLOAT32_C(  -187.79)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -177.92), SIMDE_FLOAT32_C(   382.46), SIMDE_FLOAT32_C(  -344.53), SIMDE_FLOAT32_C(   306.51),
                         SIMDE_FLOAT32_C(   804.79), SIMDE_FLOAT32_C(    74.50), SIMDE_FLOAT32_C(  -171.92), SIMDE_FLOAT32_C(  -865.07),
                         SIMDE_FLOAT32_C(   788.06), SIMDE_FLOAT32_C(  -723.82), SIMDE_FLOAT32_C(    43.98), SIMDE_FLOAT32_C(  -303.25),
                         SIMDE_FLOAT32_C(  -511.21), SIMDE_FLOAT32_C(   460.56), SIMDE_FLOAT32_C(   217.57), SIMDE_FLOAT32_C(  -900.02)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     4.40), SIMDE_FLOAT32_C(    -0.11), SIMDE_FLOAT32_C(    -0.53), SIMDE_FLOAT32_C(  -655.94),
                         SIMDE_FLOAT32_C(  -815.78), SIMDE_FLOAT32_C(   460.30), SIMDE_FLOAT32_C(    -3.16), SIMDE_FLOAT32_C(   480.08),
                         SIMDE_FLOAT32_C(  -800.23), SIMDE_FLOAT32_C(  -511.53), SIMDE_FLOAT32_C(   235.71), SIMDE_FLOAT32_C(    -2.21),
                         SIMDE_FLOAT32_C(    -0.27), SIMDE_FLOAT32_C(   413.97), SIMDE_FLOAT32_C(   264.73), SIMDE_FLOAT32_C(   769.22)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(    54.65), SIMDE_FLOAT32_C(   276.51), SIMDE_FLOAT32_C(   227.98), SIMDE_FLOAT32_C(  -128.31),
                         SIMDE_FLOAT32_C(  -191.48), SIMDE_FLOAT32_C(   348.63), SIMDE_FLOAT32_C(   444.48), SIMDE_FLOAT32_C(   206.11),
                         SIMDE_FLOAT32_C(  -692.44), SIMDE_FLOAT32_C(  -865.72), SIMDE_FLOAT32_C(   763.64), SIMDE_FLOAT32_C(  -849.66),
                         SIMDE_FLOAT32_C(   804.26), SIMDE_FLOAT32_C(   570.08), SIMDE_FLOAT32_C(   125.91), SIMDE_FLOAT32_C(   149.60)),
      UINT16_C(24771),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   747.34), SIMDE_FLOAT32_C(   607.83), SIMDE_FLOAT32_C(    25.24), SIMDE_FLOAT32_C(  -542.52),
                         SIMDE_FLOAT32_C(   568.70), SIMDE_FLOAT32_C(   899.42), SIMDE_FLOAT32_C(   120.86), SIMDE_FLOAT32_C(  -424.59),
                         SIMDE_FLOAT32_C(   377.13), SIMDE_FLOAT32_C(   761.91), SIMDE_FLOAT32_C(  -902.23), SIMDE_FLOAT32_C(  -759.84),
                         SIMDE_FLOAT32_C(   430.99), SIMDE_FLOAT32_C(   555.32), SIMDE_FLOAT32_C(  -397.14), SIMDE_FLOAT32_C(   608.52)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -615.94), SIMDE_FLOAT32_C(   113.16), SIMDE_FLOAT32_C(    26.04), SIMDE_FLOAT32_C(  -142.02),
                         SIMDE_FLOAT32_C(   273.49), SIMDE_FLOAT32_C(   374.88), SIMDE_FLOAT32_C(   453.99), SIMDE_FLOAT32_C(  -241.36),
                         SIMDE_FLOAT32_C(   181.97), SIMDE_FLOAT32_C(   143.35), SIMDE_FLOAT32_C(   400.04), SIMDE_FLOAT32_C(   610.27),
                         SIMDE_FLOAT32_C(  -726.06), SIMDE_FLOAT32_C(  -819.96), SIMDE_FLOAT32_C(   674.91), SIMDE_FLOAT32_C(   406.86)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    54.65), SIMDE_FLOAT32_C(     5.37), SIMDE_FLOAT32_C(     0.97), SIMDE_FLOAT32_C(  -128.31),
                         SIMDE_FLOAT32_C(  -191.48), SIMDE_FLOAT32_C(   348.63), SIMDE_FLOAT32_C(   444.48), SIMDE_FLOAT32_C(   206.11),
                         SIMDE_FLOAT32_C(     2.07), SIMDE_FLOAT32_C(     5.32), SIMDE_FLOAT32_C(   763.64), SIMDE_FLOAT32_C(  -849.66),
                         SIMDE_FLOAT32_C(   804.26), SIMDE_FLOAT32_C(   570.08), SIMDE_FLOAT32_C(    -0.59), SIMDE_FLOAT32_C(     1.50)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_mask_div_ps(test_vec[i].src, test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_maskz_div_ps(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask16 k;
    simde__m512 a;
    simde__m512 b;
    simde__m512 r;
  } test_vec[8] = {
    { UINT16_C(32824),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   745.69), SIMDE_FLOAT32_C(  -258.59), SIMDE_FLOAT32_C(  -549.06), SIMDE_FLOAT32_C(   646.98),
                         SIMDE_FLOAT32_C(   925.86), SIMDE_FLOAT32_C(   378.90), SIMDE_FLOAT32_C(  -524.10), SIMDE_FLOAT32_C(  -563.31),
                         SIMDE_FLOAT32_C(   112.08), SIMDE_FLOAT32_C(   712.48), SIMDE_FLOAT32_C(  -754.71), SIMDE_FLOAT32_C(   256.61),
                         SIMDE_FLOAT32_C(   768.73), SIMDE_FLOAT32_C(   227.99), SIMDE_FLOAT32_C(   174.97), SIMDE_FLOAT32_C(   338.39)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   841.82), SIMDE_FLOAT32_C(  -330.14), SIMDE_FLOAT32_C(  -498.05), SIMDE_FLOAT32_C(  -706.46),
                         SIMDE_FLOAT32_C(  -284.71), SIMDE_FLOAT32_C(  -940.98), SIMDE_FLOAT32_C(  -491.84), SIMDE_FLOAT32_C(    52.49),
                         SIMDE_FLOAT32_C(   759.92), SIMDE_FLOAT32_C(   629.58), SIMDE_FLOAT32_C(    23.76), SIMDE_FLOAT32_C(   980.95),
                         SIMDE_FLOAT32_C(   224.97), SIMDE_FLOAT32_C(   818.07), SIMDE_FLOAT32_C(  -531.75), SIMDE_FLOAT32_C(  -531.67)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.89), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   -31.76), SIMDE_FLOAT32_C(     0.26),
                         SIMDE_FLOAT32_C(     3.42), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C( 4283),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   838.22), SIMDE_FLOAT32_C(   464.78), SIMDE_FLOAT32_C(  -248.37), SIMDE_FLOAT32_C(    28.49),
                         SIMDE_FLOAT32_C(  -176.67), SIMDE_FLOAT32_C(  -468.39), SIMDE_FLOAT32_C(  -893.30), SIMDE_FLOAT32_C(   771.96),
                         SIMDE_FLOAT32_C(  -167.30), SIMDE_FLOAT32_C(  -738.71), SIMDE_FLOAT32_C(  -816.67), SIMDE_FLOAT32_C(    43.31),
                         SIMDE_FLOAT32_C(   -98.40), SIMDE_FLOAT32_C(   217.89), SIMDE_FLOAT32_C(   626.98), SIMDE_FLOAT32_C(  -409.09)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -156.59), SIMDE_FLOAT32_C(   153.69), SIMDE_FLOAT32_C(   895.38), SIMDE_FLOAT32_C(  -242.63),
                         SIMDE_FLOAT32_C(   994.17), SIMDE_FLOAT32_C(  -265.23), SIMDE_FLOAT32_C(   -57.91), SIMDE_FLOAT32_C(  -586.11),
                         SIMDE_FLOAT32_C(  -443.71), SIMDE_FLOAT32_C(  -786.78), SIMDE_FLOAT32_C(   -92.41), SIMDE_FLOAT32_C(  -378.62),
                         SIMDE_FLOAT32_C(   632.49), SIMDE_FLOAT32_C(  -867.20), SIMDE_FLOAT32_C(   977.79), SIMDE_FLOAT32_C(  -788.71)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -0.12),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.38), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     8.84), SIMDE_FLOAT32_C(    -0.11),
                         SIMDE_FLOAT32_C(    -0.16), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.64), SIMDE_FLOAT32_C(     0.52)) },
    { UINT16_C(27708),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -309.30), SIMDE_FLOAT32_C(  -478.69), SIMDE_FLOAT32_C(  -499.66), SIMDE_FLOAT32_C(  -834.97),
                         SIMDE_FLOAT32_C(  -926.76), SIMDE_FLOAT32_C(   306.74), SIMDE_FLOAT32_C(   350.68), SIMDE_FLOAT32_C(   698.74),
                         SIMDE_FLOAT32_C(  -748.23), SIMDE_FLOAT32_C(   960.31), SIMDE_FLOAT32_C(   -52.56), SIMDE_FLOAT32_C(   -18.49),
                         SIMDE_FLOAT32_C(  -174.79), SIMDE_FLOAT32_C(  -875.70), SIMDE_FLOAT32_C(   270.45), SIMDE_FLOAT32_C(   571.57)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -891.46), SIMDE_FLOAT32_C(   298.88), SIMDE_FLOAT32_C(   907.92), SIMDE_FLOAT32_C(   585.94),
                         SIMDE_FLOAT32_C(   976.00), SIMDE_FLOAT32_C(   860.60), SIMDE_FLOAT32_C(  -807.57), SIMDE_FLOAT32_C(  -501.53),
                         SIMDE_FLOAT32_C(   887.26), SIMDE_FLOAT32_C(  -380.63), SIMDE_FLOAT32_C(   603.15), SIMDE_FLOAT32_C(   906.17),
                         SIMDE_FLOAT32_C(  -446.90), SIMDE_FLOAT32_C(   518.96), SIMDE_FLOAT32_C(   325.09), SIMDE_FLOAT32_C(   394.29)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -1.60), SIMDE_FLOAT32_C(    -0.55), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(    -0.95), SIMDE_FLOAT32_C(     0.36), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -0.09), SIMDE_FLOAT32_C(    -0.02),
                         SIMDE_FLOAT32_C(     0.39), SIMDE_FLOAT32_C(    -1.69), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(21979),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -722.04), SIMDE_FLOAT32_C(  -251.19), SIMDE_FLOAT32_C(   885.20), SIMDE_FLOAT32_C(  -718.95),
                         SIMDE_FLOAT32_C(  -995.18), SIMDE_FLOAT32_C(   316.41), SIMDE_FLOAT32_C(   425.49), SIMDE_FLOAT32_C(  -889.60),
                         SIMDE_FLOAT32_C(  -764.37), SIMDE_FLOAT32_C(  -698.84), SIMDE_FLOAT32_C(   111.54), SIMDE_FLOAT32_C(   627.05),
                         SIMDE_FLOAT32_C(   619.20), SIMDE_FLOAT32_C(   107.79), SIMDE_FLOAT32_C(   830.07), SIMDE_FLOAT32_C(  -991.50)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    -9.32), SIMDE_FLOAT32_C(   588.15), SIMDE_FLOAT32_C(   740.36), SIMDE_FLOAT32_C(   589.64),
                         SIMDE_FLOAT32_C(  -146.10), SIMDE_FLOAT32_C(   771.62), SIMDE_FLOAT32_C(  -975.31), SIMDE_FLOAT32_C(   550.04),
                         SIMDE_FLOAT32_C(   902.97), SIMDE_FLOAT32_C(  -970.67), SIMDE_FLOAT32_C(  -396.71), SIMDE_FLOAT32_C(   740.42),
                         SIMDE_FLOAT32_C(  -740.07), SIMDE_FLOAT32_C(   691.95), SIMDE_FLOAT32_C(  -434.89), SIMDE_FLOAT32_C(   270.74)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -0.43), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -1.22),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.41), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -1.62),
                         SIMDE_FLOAT32_C(    -0.85), SIMDE_FLOAT32_C(     0.72), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.85),
                         SIMDE_FLOAT32_C(    -0.84), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -1.91), SIMDE_FLOAT32_C(    -3.66)) },
    { UINT16_C( 1193),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   347.59), SIMDE_FLOAT32_C(   162.65), SIMDE_FLOAT32_C(   724.10), SIMDE_FLOAT32_C(   124.00),
                         SIMDE_FLOAT32_C(  -823.97), SIMDE_FLOAT32_C(  -185.15), SIMDE_FLOAT32_C(    33.85), SIMDE_FLOAT32_C(  -430.54),
                         SIMDE_FLOAT32_C(  -534.02), SIMDE_FLOAT32_C(   815.29), SIMDE_FLOAT32_C(   942.25), SIMDE_FLOAT32_C(  -825.08),
                         SIMDE_FLOAT32_C(   638.03), SIMDE_FLOAT32_C(   599.07), SIMDE_FLOAT32_C(   164.45), SIMDE_FLOAT32_C(   429.94)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -817.35), SIMDE_FLOAT32_C(  -889.89), SIMDE_FLOAT32_C(   528.79), SIMDE_FLOAT32_C(  -600.85),
                         SIMDE_FLOAT32_C(  -168.12), SIMDE_FLOAT32_C(  -798.12), SIMDE_FLOAT32_C(  -637.75), SIMDE_FLOAT32_C(  -580.73),
                         SIMDE_FLOAT32_C(   697.23), SIMDE_FLOAT32_C(   654.25), SIMDE_FLOAT32_C(  -236.09), SIMDE_FLOAT32_C(   234.13),
                         SIMDE_FLOAT32_C(  -696.60), SIMDE_FLOAT32_C(  -486.03), SIMDE_FLOAT32_C(    69.79), SIMDE_FLOAT32_C(   435.18)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.23), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(    -0.77), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -3.99), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(    -0.92), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.99)) },
    { UINT16_C(47777),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    96.65), SIMDE_FLOAT32_C(   -38.76), SIMDE_FLOAT32_C(   585.22), SIMDE_FLOAT32_C(  -683.52),
                         SIMDE_FLOAT32_C(   268.64), SIMDE_FLOAT32_C(  -393.28), SIMDE_FLOAT32_C(   102.94), SIMDE_FLOAT32_C(   786.82),
                         SIMDE_FLOAT32_C(   138.90), SIMDE_FLOAT32_C(   225.78), SIMDE_FLOAT32_C(   449.88), SIMDE_FLOAT32_C(   347.32),
                         SIMDE_FLOAT32_C(    33.80), SIMDE_FLOAT32_C(  -559.14), SIMDE_FLOAT32_C(  -159.05), SIMDE_FLOAT32_C(  -491.42)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   -55.95), SIMDE_FLOAT32_C(  -837.50), SIMDE_FLOAT32_C(  -575.23), SIMDE_FLOAT32_C(   248.03),
                         SIMDE_FLOAT32_C(   907.04), SIMDE_FLOAT32_C(   -74.96), SIMDE_FLOAT32_C(  -821.80), SIMDE_FLOAT32_C(  -847.93),
                         SIMDE_FLOAT32_C(  -925.94), SIMDE_FLOAT32_C(   664.01), SIMDE_FLOAT32_C(  -745.59), SIMDE_FLOAT32_C(  -301.31),
                         SIMDE_FLOAT32_C(   146.53), SIMDE_FLOAT32_C(  -440.81), SIMDE_FLOAT32_C(   427.27), SIMDE_FLOAT32_C(  -219.59)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    -1.73), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -1.02), SIMDE_FLOAT32_C(    -2.76),
                         SIMDE_FLOAT32_C(     0.30), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -0.13), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(    -0.15), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -0.60), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     2.24)) },
    { UINT16_C(50336),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -724.66), SIMDE_FLOAT32_C(  -778.29), SIMDE_FLOAT32_C(  -888.11), SIMDE_FLOAT32_C(  -623.31),
                         SIMDE_FLOAT32_C(   617.08), SIMDE_FLOAT32_C(    42.91), SIMDE_FLOAT32_C(   907.40), SIMDE_FLOAT32_C(  -402.88),
                         SIMDE_FLOAT32_C(  -278.23), SIMDE_FLOAT32_C(  -640.08), SIMDE_FLOAT32_C(   108.85), SIMDE_FLOAT32_C(  -527.72),
                         SIMDE_FLOAT32_C(  -791.82), SIMDE_FLOAT32_C(  -207.31), SIMDE_FLOAT32_C(  -642.88), SIMDE_FLOAT32_C(   536.44)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   766.31), SIMDE_FLOAT32_C(   592.01), SIMDE_FLOAT32_C(   324.90), SIMDE_FLOAT32_C(    55.55),
                         SIMDE_FLOAT32_C(   -34.13), SIMDE_FLOAT32_C(  -588.88), SIMDE_FLOAT32_C(   991.78), SIMDE_FLOAT32_C(  -468.91),
                         SIMDE_FLOAT32_C(    78.86), SIMDE_FLOAT32_C(    18.25), SIMDE_FLOAT32_C(   295.51), SIMDE_FLOAT32_C(  -293.26),
                         SIMDE_FLOAT32_C(  -877.24), SIMDE_FLOAT32_C(   952.33), SIMDE_FLOAT32_C(  -274.18), SIMDE_FLOAT32_C(   654.17)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    -0.95), SIMDE_FLOAT32_C(    -1.31), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -0.07), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(    -3.53), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.37), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(  740),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -222.38), SIMDE_FLOAT32_C(  -847.72), SIMDE_FLOAT32_C(  -497.04), SIMDE_FLOAT32_C(   862.38),
                         SIMDE_FLOAT32_C(  -840.40), SIMDE_FLOAT32_C(   998.10), SIMDE_FLOAT32_C(  -257.93), SIMDE_FLOAT32_C(  -204.46),
                         SIMDE_FLOAT32_C(  -373.11), SIMDE_FLOAT32_C(  -912.42), SIMDE_FLOAT32_C(   207.13), SIMDE_FLOAT32_C(   784.69),
                         SIMDE_FLOAT32_C(    82.66), SIMDE_FLOAT32_C(   123.09), SIMDE_FLOAT32_C(  -384.17), SIMDE_FLOAT32_C(  -845.08)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   839.49), SIMDE_FLOAT32_C(  -285.39), SIMDE_FLOAT32_C(  -548.76), SIMDE_FLOAT32_C(   -35.10),
                         SIMDE_FLOAT32_C(  -295.04), SIMDE_FLOAT32_C(   738.77), SIMDE_FLOAT32_C(   340.04), SIMDE_FLOAT32_C(  -585.87),
                         SIMDE_FLOAT32_C(  -711.46), SIMDE_FLOAT32_C(   926.37), SIMDE_FLOAT32_C(   696.23), SIMDE_FLOAT32_C(   766.17),
                         SIMDE_FLOAT32_C(  -330.24), SIMDE_FLOAT32_C(   369.18), SIMDE_FLOAT32_C(  -498.71), SIMDE_FLOAT32_C(  -288.61)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -0.76), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.52), SIMDE_FLOAT32_C(    -0.98), SIMDE_FLOAT32_C(     0.30), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.33), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_maskz_div_ps(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_div_pd(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512d a;
    simde__m512d b;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  -97.83), SIMDE_FLOAT64_C( -393.82),
                         SIMDE_FLOAT64_C(  934.81), SIMDE_FLOAT64_C(   74.53),
                         SIMDE_FLOAT64_C(  843.79), SIMDE_FLOAT64_C(  465.05),
                         SIMDE_FLOAT64_C(  -42.07), SIMDE_FLOAT64_C( -685.83)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  296.78), SIMDE_FLOAT64_C( -698.78),
                         SIMDE_FLOAT64_C(  908.33), SIMDE_FLOAT64_C(  181.85),
                         SIMDE_FLOAT64_C( -397.89), SIMDE_FLOAT64_C( -586.75),
                         SIMDE_FLOAT64_C(  904.99), SIMDE_FLOAT64_C( -321.15)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   -0.33), SIMDE_FLOAT64_C(    0.56),
                         SIMDE_FLOAT64_C(    1.03), SIMDE_FLOAT64_C(    0.41),
                         SIMDE_FLOAT64_C(   -2.12), SIMDE_FLOAT64_C(   -0.79),
                         SIMDE_FLOAT64_C(   -0.05), SIMDE_FLOAT64_C(    2.14)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  653.62), SIMDE_FLOAT64_C(  981.74),
                         SIMDE_FLOAT64_C(  780.10), SIMDE_FLOAT64_C(   59.38),
                         SIMDE_FLOAT64_C( -795.11), SIMDE_FLOAT64_C(  923.87),
                         SIMDE_FLOAT64_C( -270.01), SIMDE_FLOAT64_C( -411.99)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  596.54), SIMDE_FLOAT64_C( -116.40),
                         SIMDE_FLOAT64_C( -989.77), SIMDE_FLOAT64_C( -794.40),
                         SIMDE_FLOAT64_C(  183.38), SIMDE_FLOAT64_C( -185.75),
                         SIMDE_FLOAT64_C(  429.70), SIMDE_FLOAT64_C(  664.04)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    1.10), SIMDE_FLOAT64_C(   -8.43),
                         SIMDE_FLOAT64_C(   -0.79), SIMDE_FLOAT64_C(   -0.07),
                         SIMDE_FLOAT64_C(   -4.34), SIMDE_FLOAT64_C(   -4.97),
                         SIMDE_FLOAT64_C(   -0.63), SIMDE_FLOAT64_C(   -0.62)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -571.10), SIMDE_FLOAT64_C(  971.96),
                         SIMDE_FLOAT64_C(  149.38), SIMDE_FLOAT64_C(  497.71),
                         SIMDE_FLOAT64_C(  988.69), SIMDE_FLOAT64_C(  479.68),
                         SIMDE_FLOAT64_C( -128.24), SIMDE_FLOAT64_C(  585.28)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -628.68), SIMDE_FLOAT64_C(  938.60),
                         SIMDE_FLOAT64_C( -147.98), SIMDE_FLOAT64_C(  378.31),
                         SIMDE_FLOAT64_C(  246.47), SIMDE_FLOAT64_C(  109.18),
                         SIMDE_FLOAT64_C( -575.64), SIMDE_FLOAT64_C( -426.86)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.91), SIMDE_FLOAT64_C(    1.04),
                         SIMDE_FLOAT64_C(   -1.01), SIMDE_FLOAT64_C(    1.32),
                         SIMDE_FLOAT64_C(    4.01), SIMDE_FLOAT64_C(    4.39),
                         SIMDE_FLOAT64_C(    0.22), SIMDE_FLOAT64_C(   -1.37)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  729.63), SIMDE_FLOAT64_C( -908.06),
                         SIMDE_FLOAT64_C( -769.77), SIMDE_FLOAT64_C(  -70.66),
                         SIMDE_FLOAT64_C(  482.71), SIMDE_FLOAT64_C(  244.66),
                         SIMDE_FLOAT64_C( -615.83), SIMDE_FLOAT64_C(  841.42)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  359.65), SIMDE_FLOAT64_C( -730.08),
                         SIMDE_FLOAT64_C(  977.98), SIMDE_FLOAT64_C( -215.53),
                         SIMDE_FLOAT64_C( -315.50), SIMDE_FLOAT64_C(   80.64),
                         SIMDE_FLOAT64_C( -996.10), SIMDE_FLOAT64_C( -556.83)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    2.03), SIMDE_FLOAT64_C(    1.24),
                         SIMDE_FLOAT64_C(   -0.79), SIMDE_FLOAT64_C(    0.33),
                         SIMDE_FLOAT64_C(   -1.53), SIMDE_FLOAT64_C(    3.03),
                         SIMDE_FLOAT64_C(    0.62), SIMDE_FLOAT64_C(   -1.51)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  438.03), SIMDE_FLOAT64_C( -458.01),
                         SIMDE_FLOAT64_C(  144.59), SIMDE_FLOAT64_C(  165.00),
                         SIMDE_FLOAT64_C( -331.04), SIMDE_FLOAT64_C(  406.96),
                         SIMDE_FLOAT64_C( -326.43), SIMDE_FLOAT64_C(  373.82)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  304.30), SIMDE_FLOAT64_C( -777.07),
                         SIMDE_FLOAT64_C( -683.73), SIMDE_FLOAT64_C( -113.32),
                         SIMDE_FLOAT64_C( -701.16), SIMDE_FLOAT64_C( -942.92),
                         SIMDE_FLOAT64_C( -489.97), SIMDE_FLOAT64_C(  911.34)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    1.44), SIMDE_FLOAT64_C(    0.59),
                         SIMDE_FLOAT64_C(   -0.21), SIMDE_FLOAT64_C(   -1.46),
                         SIMDE_FLOAT64_C(    0.47), SIMDE_FLOAT64_C(   -0.43),
                         SIMDE_FLOAT64_C(    0.67), SIMDE_FLOAT64_C(    0.41)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -148.70), SIMDE_FLOAT64_C( -327.17),
                         SIMDE_FLOAT64_C( -310.14), SIMDE_FLOAT64_C( -718.80),
                         SIMDE_FLOAT64_C(  382.69), SIMDE_FLOAT64_C( -181.61),
                         SIMDE_FLOAT64_C( -214.09), SIMDE_FLOAT64_C(   55.72)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  791.83), SIMDE_FLOAT64_C(  191.69),
                         SIMDE_FLOAT64_C( -460.58), SIMDE_FLOAT64_C( -915.08),
                         SIMDE_FLOAT64_C( -877.38), SIMDE_FLOAT64_C( -915.27),
                         SIMDE_FLOAT64_C(  207.85), SIMDE_FLOAT64_C(  567.35)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   -0.19), SIMDE_FLOAT64_C(   -1.71),
                         SIMDE_FLOAT64_C(    0.67), SIMDE_FLOAT64_C(    0.79),
                         SIMDE_FLOAT64_C(   -0.44), SIMDE_FLOAT64_C(    0.20),
                         SIMDE_FLOAT64_C(   -1.03), SIMDE_FLOAT64_C(    0.10)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -798.08), SIMDE_FLOAT64_C(  256.40),
                         SIMDE_FLOAT64_C(  739.89), SIMDE_FLOAT64_C( -903.46),
                         SIMDE_FLOAT64_C( -771.75), SIMDE_FLOAT64_C(  -54.77),
                         SIMDE_FLOAT64_C(  397.04), SIMDE_FLOAT64_C(  925.94)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  864.29), SIMDE_FLOAT64_C( -834.00),
                         SIMDE_FLOAT64_C(  475.52), SIMDE_FLOAT64_C(  502.31),
                         SIMDE_FLOAT64_C( -746.87), SIMDE_FLOAT64_C( -364.10),
                         SIMDE_FLOAT64_C( -995.18), SIMDE_FLOAT64_C(  683.54)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   -0.92), SIMDE_FLOAT64_C(   -0.31),
                         SIMDE_FLOAT64_C(    1.56), SIMDE_FLOAT64_C(   -1.80),
                         SIMDE_FLOAT64_C(    1.03), SIMDE_FLOAT64_C(    0.15),
                         SIMDE_FLOAT64_C(   -0.40), SIMDE_FLOAT64_C(    1.35)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -869.58), SIMDE_FLOAT64_C(  763.75),
                         SIMDE_FLOAT64_C( -558.93), SIMDE_FLOAT64_C(  756.19),
                         SIMDE_FLOAT64_C(  509.82), SIMDE_FLOAT64_C( -855.71),
                         SIMDE_FLOAT64_C( -965.40), SIMDE_FLOAT64_C( -279.29)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -355.51), SIMDE_FLOAT64_C(  136.73),
                         SIMDE_FLOAT64_C(  586.70), SIMDE_FLOAT64_C(  712.56),
                         SIMDE_FLOAT64_C(  135.88), SIMDE_FLOAT64_C( -693.91),
                         SIMDE_FLOAT64_C( -131.33), SIMDE_FLOAT64_C( -933.79)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    2.45), SIMDE_FLOAT64_C(    5.59),
                         SIMDE_FLOAT64_C(   -0.95), SIMDE_FLOAT64_C(    1.06),
                         SIMDE_FLOAT64_C(    3.75), SIMDE_FLOAT64_C(    1.23),
                         SIMDE_FLOAT64_C(    7.35), SIMDE_FLOAT64_C(    0.30)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_div_pd(test_vec[i].a, test_vec[i].b);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_mask_div_pd(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512d src;
    simde__mmask8 k;
    simde__m512d a;
    simde__m512d b;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -962.94), SIMDE_FLOAT64_C(  989.45),
                         SIMDE_FLOAT64_C( -190.71), SIMDE_FLOAT64_C(  -80.90),
                         SIMDE_FLOAT64_C( -820.03), SIMDE_FLOAT64_C(  710.84),
                         SIMDE_FLOAT64_C(  742.77), SIMDE_FLOAT64_C( -124.19)),
      UINT8_C( 62),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  764.73), SIMDE_FLOAT64_C( -738.72),
                         SIMDE_FLOAT64_C(  462.89), SIMDE_FLOAT64_C( -909.36),
                         SIMDE_FLOAT64_C(  920.77), SIMDE_FLOAT64_C(  830.94),
                         SIMDE_FLOAT64_C( -436.90), SIMDE_FLOAT64_C( -984.49)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  318.55), SIMDE_FLOAT64_C( -958.54),
                         SIMDE_FLOAT64_C( -878.41), SIMDE_FLOAT64_C(  198.47),
                         SIMDE_FLOAT64_C(  585.51), SIMDE_FLOAT64_C(  -97.52),
                         SIMDE_FLOAT64_C( -112.08), SIMDE_FLOAT64_C( -145.20)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -962.94), SIMDE_FLOAT64_C(  989.45),
                         SIMDE_FLOAT64_C(   -0.53), SIMDE_FLOAT64_C(   -4.58),
                         SIMDE_FLOAT64_C(    1.57), SIMDE_FLOAT64_C(   -8.52),
                         SIMDE_FLOAT64_C(    3.90), SIMDE_FLOAT64_C( -124.19)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  267.17), SIMDE_FLOAT64_C( -878.34),
                         SIMDE_FLOAT64_C(  132.07), SIMDE_FLOAT64_C(  827.87),
                         SIMDE_FLOAT64_C(  178.51), SIMDE_FLOAT64_C(  362.39),
                         SIMDE_FLOAT64_C(  200.13), SIMDE_FLOAT64_C( -407.98)),
      UINT8_C( 51),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -126.54), SIMDE_FLOAT64_C( -164.31),
                         SIMDE_FLOAT64_C( -971.32), SIMDE_FLOAT64_C(  611.23),
                         SIMDE_FLOAT64_C(  591.83), SIMDE_FLOAT64_C(  793.58),
                         SIMDE_FLOAT64_C(  171.77), SIMDE_FLOAT64_C(  109.83)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  225.35), SIMDE_FLOAT64_C( -734.84),
                         SIMDE_FLOAT64_C(  728.29), SIMDE_FLOAT64_C( -721.11),
                         SIMDE_FLOAT64_C( -448.10), SIMDE_FLOAT64_C(  310.61),
                         SIMDE_FLOAT64_C( -362.27), SIMDE_FLOAT64_C( -413.07)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  267.17), SIMDE_FLOAT64_C( -878.34),
                         SIMDE_FLOAT64_C(   -1.33), SIMDE_FLOAT64_C(   -0.85),
                         SIMDE_FLOAT64_C(  178.51), SIMDE_FLOAT64_C(  362.39),
                         SIMDE_FLOAT64_C(   -0.47), SIMDE_FLOAT64_C(   -0.27)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  439.30), SIMDE_FLOAT64_C(  935.39),
                         SIMDE_FLOAT64_C(   20.00), SIMDE_FLOAT64_C( -941.65),
                         SIMDE_FLOAT64_C(  988.79), SIMDE_FLOAT64_C(  773.96),
                         SIMDE_FLOAT64_C( -788.78), SIMDE_FLOAT64_C( -311.91)),
      UINT8_C(178),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -374.30), SIMDE_FLOAT64_C(  599.21),
                         SIMDE_FLOAT64_C(  966.83), SIMDE_FLOAT64_C(  775.18),
                         SIMDE_FLOAT64_C(  846.32), SIMDE_FLOAT64_C(  124.04),
                         SIMDE_FLOAT64_C( -883.36), SIMDE_FLOAT64_C( -405.09)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  467.70), SIMDE_FLOAT64_C( -626.02),
                         SIMDE_FLOAT64_C(  355.93), SIMDE_FLOAT64_C(  294.34),
                         SIMDE_FLOAT64_C( -575.79), SIMDE_FLOAT64_C( -504.82),
                         SIMDE_FLOAT64_C(  854.52), SIMDE_FLOAT64_C( -173.82)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   -0.80), SIMDE_FLOAT64_C(  935.39),
                         SIMDE_FLOAT64_C(    2.72), SIMDE_FLOAT64_C(    2.63),
                         SIMDE_FLOAT64_C(  988.79), SIMDE_FLOAT64_C(  773.96),
                         SIMDE_FLOAT64_C(   -1.03), SIMDE_FLOAT64_C( -311.91)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -274.81), SIMDE_FLOAT64_C(  196.78),
                         SIMDE_FLOAT64_C( -805.22), SIMDE_FLOAT64_C(  855.89),
                         SIMDE_FLOAT64_C( -996.67), SIMDE_FLOAT64_C(  424.78),
                         SIMDE_FLOAT64_C(  489.73), SIMDE_FLOAT64_C(  635.35)),
      UINT8_C( 38),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   79.19), SIMDE_FLOAT64_C( -114.25),
                         SIMDE_FLOAT64_C(  983.59), SIMDE_FLOAT64_C(  645.66),
                         SIMDE_FLOAT64_C(  982.80), SIMDE_FLOAT64_C( -683.73),
                         SIMDE_FLOAT64_C(  259.13), SIMDE_FLOAT64_C(  186.09)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  665.49), SIMDE_FLOAT64_C( -110.92),
                         SIMDE_FLOAT64_C(  978.65), SIMDE_FLOAT64_C(  104.45),
                         SIMDE_FLOAT64_C(  903.68), SIMDE_FLOAT64_C( -580.74),
                         SIMDE_FLOAT64_C(  776.44), SIMDE_FLOAT64_C(  571.14)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -274.81), SIMDE_FLOAT64_C(  196.78),
                         SIMDE_FLOAT64_C(    1.01), SIMDE_FLOAT64_C(  855.89),
                         SIMDE_FLOAT64_C( -996.67), SIMDE_FLOAT64_C(    1.18),
                         SIMDE_FLOAT64_C(    0.33), SIMDE_FLOAT64_C(  635.35)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  105.93), SIMDE_FLOAT64_C( -486.73),
                         SIMDE_FLOAT64_C(  293.04), SIMDE_FLOAT64_C(  328.58),
                         SIMDE_FLOAT64_C( -725.03), SIMDE_FLOAT64_C(    3.53),
                         SIMDE_FLOAT64_C(  663.75), SIMDE_FLOAT64_C(  -59.32)),
      UINT8_C( 67),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  917.98), SIMDE_FLOAT64_C( -430.92),
                         SIMDE_FLOAT64_C(  839.77), SIMDE_FLOAT64_C( -412.68),
                         SIMDE_FLOAT64_C( -397.37), SIMDE_FLOAT64_C( -947.31),
                         SIMDE_FLOAT64_C(  584.59), SIMDE_FLOAT64_C( -352.12)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  781.61), SIMDE_FLOAT64_C(  978.32),
                         SIMDE_FLOAT64_C(  374.68), SIMDE_FLOAT64_C( -857.00),
                         SIMDE_FLOAT64_C(  821.72), SIMDE_FLOAT64_C(  -88.08),
                         SIMDE_FLOAT64_C(  243.00), SIMDE_FLOAT64_C( -640.77)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  105.93), SIMDE_FLOAT64_C(   -0.44),
                         SIMDE_FLOAT64_C(  293.04), SIMDE_FLOAT64_C(  328.58),
                         SIMDE_FLOAT64_C( -725.03), SIMDE_FLOAT64_C(    3.53),
                         SIMDE_FLOAT64_C(    2.41), SIMDE_FLOAT64_C(    0.55)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -591.91), SIMDE_FLOAT64_C(  615.29),
                         SIMDE_FLOAT64_C( -726.07), SIMDE_FLOAT64_C(  857.36),
                         SIMDE_FLOAT64_C(  636.31), SIMDE_FLOAT64_C(  104.40),
                         SIMDE_FLOAT64_C( -167.77), SIMDE_FLOAT64_C( -372.65)),
      UINT8_C( 15),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  411.16), SIMDE_FLOAT64_C(  928.95),
                         SIMDE_FLOAT64_C(  110.13), SIMDE_FLOAT64_C(  933.76),
                         SIMDE_FLOAT64_C(  836.76), SIMDE_FLOAT64_C(  628.60),
                         SIMDE_FLOAT64_C( -586.52), SIMDE_FLOAT64_C(  293.24)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -457.28), SIMDE_FLOAT64_C(  705.56),
                         SIMDE_FLOAT64_C( -798.08), SIMDE_FLOAT64_C(  773.61),
                         SIMDE_FLOAT64_C( -590.48), SIMDE_FLOAT64_C( -291.69),
                         SIMDE_FLOAT64_C(  654.27), SIMDE_FLOAT64_C( -537.59)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -591.91), SIMDE_FLOAT64_C(  615.29),
                         SIMDE_FLOAT64_C( -726.07), SIMDE_FLOAT64_C(  857.36),
                         SIMDE_FLOAT64_C(   -1.42), SIMDE_FLOAT64_C(   -2.16),
                         SIMDE_FLOAT64_C(   -0.90), SIMDE_FLOAT64_C(   -0.55)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  270.92), SIMDE_FLOAT64_C( -517.94),
                         SIMDE_FLOAT64_C(   36.22), SIMDE_FLOAT64_C(  204.54),
                         SIMDE_FLOAT64_C(  579.30), SIMDE_FLOAT64_C(  257.34),
                         SIMDE_FLOAT64_C( -998.24), SIMDE_FLOAT64_C( -146.41)),
      UINT8_C(152),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  268.93), SIMDE_FLOAT64_C( -893.46),
                         SIMDE_FLOAT64_C( -476.89), SIMDE_FLOAT64_C( -696.00),
                         SIMDE_FLOAT64_C( -817.69), SIMDE_FLOAT64_C(  127.75),
                         SIMDE_FLOAT64_C( -366.34), SIMDE_FLOAT64_C( -437.04)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -372.16), SIMDE_FLOAT64_C(  900.88),
                         SIMDE_FLOAT64_C( -550.65), SIMDE_FLOAT64_C(  567.85),
                         SIMDE_FLOAT64_C(  968.56), SIMDE_FLOAT64_C( -695.12),
                         SIMDE_FLOAT64_C(  555.56), SIMDE_FLOAT64_C(  952.92)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   -0.72), SIMDE_FLOAT64_C( -517.94),
                         SIMDE_FLOAT64_C(   36.22), SIMDE_FLOAT64_C(   -1.23),
                         SIMDE_FLOAT64_C(   -0.84), SIMDE_FLOAT64_C(  257.34),
                         SIMDE_FLOAT64_C( -998.24), SIMDE_FLOAT64_C( -146.41)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -468.36), SIMDE_FLOAT64_C(  186.16),
                         SIMDE_FLOAT64_C( -910.43), SIMDE_FLOAT64_C( -280.07),
                         SIMDE_FLOAT64_C(  -96.94), SIMDE_FLOAT64_C(  387.95),
                         SIMDE_FLOAT64_C(  198.14), SIMDE_FLOAT64_C( -504.51)),
      UINT8_C( 21),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  573.90), SIMDE_FLOAT64_C(  496.67),
                         SIMDE_FLOAT64_C( -823.61), SIMDE_FLOAT64_C(  204.56),
                         SIMDE_FLOAT64_C( -856.87), SIMDE_FLOAT64_C( -449.28),
                         SIMDE_FLOAT64_C(    9.73), SIMDE_FLOAT64_C( -739.12)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -471.24), SIMDE_FLOAT64_C( -711.43),
                         SIMDE_FLOAT64_C( -281.50), SIMDE_FLOAT64_C(  493.76),
                         SIMDE_FLOAT64_C(  103.01), SIMDE_FLOAT64_C( -996.35),
                         SIMDE_FLOAT64_C(  670.04), SIMDE_FLOAT64_C( -895.53)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -468.36), SIMDE_FLOAT64_C(  186.16),
                         SIMDE_FLOAT64_C( -910.43), SIMDE_FLOAT64_C(    0.41),
                         SIMDE_FLOAT64_C(  -96.94), SIMDE_FLOAT64_C(    0.45),
                         SIMDE_FLOAT64_C(  198.14), SIMDE_FLOAT64_C(    0.83)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_mask_div_pd(test_vec[i].src, test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_maskz_div_pd(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask8 k;
    simde__m512d a;
    simde__m512d b;
    simde__m512d r;
  } test_vec[8] = {
    { UINT8_C(113),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  112.08), SIMDE_FLOAT64_C(  712.48),
                         SIMDE_FLOAT64_C( -754.71), SIMDE_FLOAT64_C(  256.61),
                         SIMDE_FLOAT64_C(  768.73), SIMDE_FLOAT64_C(  227.99),
                         SIMDE_FLOAT64_C(  174.97), SIMDE_FLOAT64_C(  338.39)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  759.92), SIMDE_FLOAT64_C(  629.58),
                         SIMDE_FLOAT64_C(   23.76), SIMDE_FLOAT64_C(  980.95),
                         SIMDE_FLOAT64_C(  224.97), SIMDE_FLOAT64_C(  818.07),
                         SIMDE_FLOAT64_C( -531.75), SIMDE_FLOAT64_C( -531.67)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    1.13),
                         SIMDE_FLOAT64_C(  -31.76), SIMDE_FLOAT64_C(    0.26),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(   -0.64)) },
    { UINT8_C( 88),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  841.82), SIMDE_FLOAT64_C( -330.14),
                         SIMDE_FLOAT64_C( -498.05), SIMDE_FLOAT64_C( -706.46),
                         SIMDE_FLOAT64_C( -284.71), SIMDE_FLOAT64_C( -940.98),
                         SIMDE_FLOAT64_C( -491.84), SIMDE_FLOAT64_C(   52.49)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  259.38), SIMDE_FLOAT64_C(  745.69),
                         SIMDE_FLOAT64_C( -258.59), SIMDE_FLOAT64_C( -549.06),
                         SIMDE_FLOAT64_C(  646.98), SIMDE_FLOAT64_C(  925.86),
                         SIMDE_FLOAT64_C(  378.90), SIMDE_FLOAT64_C( -524.10)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(   -0.44),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    1.29),
                         SIMDE_FLOAT64_C(   -0.44), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(184),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -167.30), SIMDE_FLOAT64_C( -738.71),
                         SIMDE_FLOAT64_C( -816.67), SIMDE_FLOAT64_C(   43.31),
                         SIMDE_FLOAT64_C(  -98.40), SIMDE_FLOAT64_C(  217.89),
                         SIMDE_FLOAT64_C(  626.98), SIMDE_FLOAT64_C( -409.09)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -443.71), SIMDE_FLOAT64_C( -786.78),
                         SIMDE_FLOAT64_C(  -92.41), SIMDE_FLOAT64_C( -378.62),
                         SIMDE_FLOAT64_C(  632.49), SIMDE_FLOAT64_C( -867.20),
                         SIMDE_FLOAT64_C(  977.79), SIMDE_FLOAT64_C( -788.71)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.38), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    8.84), SIMDE_FLOAT64_C(   -0.11),
                         SIMDE_FLOAT64_C(   -0.16), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(119),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -156.59), SIMDE_FLOAT64_C(  153.69),
                         SIMDE_FLOAT64_C(  895.38), SIMDE_FLOAT64_C( -242.63),
                         SIMDE_FLOAT64_C(  994.17), SIMDE_FLOAT64_C( -265.23),
                         SIMDE_FLOAT64_C(  -57.91), SIMDE_FLOAT64_C( -586.11)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -851.62), SIMDE_FLOAT64_C(  838.22),
                         SIMDE_FLOAT64_C(  464.78), SIMDE_FLOAT64_C( -248.37),
                         SIMDE_FLOAT64_C(   28.49), SIMDE_FLOAT64_C( -176.67),
                         SIMDE_FLOAT64_C( -468.39), SIMDE_FLOAT64_C( -893.30)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.18),
                         SIMDE_FLOAT64_C(    1.93), SIMDE_FLOAT64_C(    0.98),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    1.50),
                         SIMDE_FLOAT64_C(    0.12), SIMDE_FLOAT64_C(    0.66)) },
    { UINT8_C(181),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -748.23), SIMDE_FLOAT64_C(  960.31),
                         SIMDE_FLOAT64_C(  -52.56), SIMDE_FLOAT64_C(  -18.49),
                         SIMDE_FLOAT64_C( -174.79), SIMDE_FLOAT64_C( -875.70),
                         SIMDE_FLOAT64_C(  270.45), SIMDE_FLOAT64_C(  571.57)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  887.26), SIMDE_FLOAT64_C( -380.63),
                         SIMDE_FLOAT64_C(  603.15), SIMDE_FLOAT64_C(  906.17),
                         SIMDE_FLOAT64_C( -446.90), SIMDE_FLOAT64_C(  518.96),
                         SIMDE_FLOAT64_C(  325.09), SIMDE_FLOAT64_C(  394.29)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   -0.84), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(   -0.09), SIMDE_FLOAT64_C(   -0.02),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(   -1.69),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    1.45)) },
    { UINT8_C(108),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -891.46), SIMDE_FLOAT64_C(  298.88),
                         SIMDE_FLOAT64_C(  907.92), SIMDE_FLOAT64_C(  585.94),
                         SIMDE_FLOAT64_C(  976.00), SIMDE_FLOAT64_C(  860.60),
                         SIMDE_FLOAT64_C( -807.57), SIMDE_FLOAT64_C( -501.53)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -634.78), SIMDE_FLOAT64_C( -309.30),
                         SIMDE_FLOAT64_C( -478.69), SIMDE_FLOAT64_C( -499.66),
                         SIMDE_FLOAT64_C( -834.97), SIMDE_FLOAT64_C( -926.76),
                         SIMDE_FLOAT64_C(  306.74), SIMDE_FLOAT64_C(  350.68)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(   -0.97),
                         SIMDE_FLOAT64_C(   -1.90), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(   -1.17), SIMDE_FLOAT64_C(   -0.93),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(  5),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -764.37), SIMDE_FLOAT64_C( -698.84),
                         SIMDE_FLOAT64_C(  111.54), SIMDE_FLOAT64_C(  627.05),
                         SIMDE_FLOAT64_C(  619.20), SIMDE_FLOAT64_C(  107.79),
                         SIMDE_FLOAT64_C(  830.07), SIMDE_FLOAT64_C( -991.50)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  902.97), SIMDE_FLOAT64_C( -970.67),
                         SIMDE_FLOAT64_C( -396.71), SIMDE_FLOAT64_C(  740.42),
                         SIMDE_FLOAT64_C( -740.07), SIMDE_FLOAT64_C(  691.95),
                         SIMDE_FLOAT64_C( -434.89), SIMDE_FLOAT64_C(  270.74)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.16),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(   -3.66)) },
    { UINT8_C( 94),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   -9.32), SIMDE_FLOAT64_C(  588.15),
                         SIMDE_FLOAT64_C(  740.36), SIMDE_FLOAT64_C(  589.64),
                         SIMDE_FLOAT64_C( -146.10), SIMDE_FLOAT64_C(  771.62),
                         SIMDE_FLOAT64_C( -975.31), SIMDE_FLOAT64_C(  550.04)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  380.47), SIMDE_FLOAT64_C( -722.04),
                         SIMDE_FLOAT64_C( -251.19), SIMDE_FLOAT64_C(  885.20),
                         SIMDE_FLOAT64_C( -718.95), SIMDE_FLOAT64_C( -995.18),
                         SIMDE_FLOAT64_C(  316.41), SIMDE_FLOAT64_C(  425.49)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(   -0.81),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.67),
                         SIMDE_FLOAT64_C(    0.20), SIMDE_FLOAT64_C(   -0.78),
                         SIMDE_FLOAT64_C(   -3.08), SIMDE_FLOAT64_C(    0.00)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_maskz_div_pd(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

SIMDE_TEST_FUNC_LIST_BEGIN
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_div_ps)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_div_ps)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_div_ps)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_div_pd)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_div_pd)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_div_pd)
SIMDE_TEST_FUNC_LIST_END

#include <test/x86/avx512/test-avx512-footer.h>
