/***************************************************************
 * Name:      event.cpp
 * Author:    David Vachulka (arch_dvx@users.sourceforge.net)
 * Copyright: 2013
 * License:   GPL3
 **************************************************************/

#include "event.h"
#include "engine.h"
#include "dxdefs.h"
#include "dxsettings.h"
#include "dxutils.h"

Event::Event()
: m_id(-1), m_event(wxEmptyString), m_descr(wxEmptyString), m_date(INFINITY_DATE), m_reminder(0), m_reminded(true), m_day(0), m_month(0),
  m_hidden(false), m_recurrence(R_ONCE), m_minutes(0), m_hours(0), m_days(0), m_months(0), m_monthlyday(0), m_monthlyweek(0), m_actualreminder(0),
  m_until(INFINITY_DATE), m_always(false), m_color(0)
{
}

Event::Event(wxInt64 id, const wxString &event, const wxString &descr, const wxDateTime &date, wxInt64 reminder, bool reminded, int day, int month, bool hidden,
             int recurrence, int minutes, int hours, int days, int months, int monthlyday, int monthlyweek, wxInt64 actualreminder, const wxDateTime &until, bool always, int color)
: m_id(id), m_event(event), m_descr(descr), m_date(date), m_reminder(reminder), m_reminded(reminded), m_day(day), m_month(month), m_hidden(hidden),
  m_recurrence(recurrence), m_minutes(minutes), m_hours(hours), m_days(days), m_months(months), m_monthlyday(monthlyday), m_monthlyweek(monthlyweek), m_actualreminder(actualreminder),
  m_until(until), m_always(always), m_color(color)
{
    if(m_recurrence == R_MINUTES && m_minutes==0)
    {
        m_minutes = 1;
    }
    if(m_recurrence == R_HOURLY && m_hours==0)
    {
        m_hours = 1;
    }
    if(m_recurrence == R_OWN && m_minutes==0 && m_hours==0 && m_days==0 && m_months==0)
    {
        m_days = 1;
    }
    if((m_recurrence == R_MONTHLYPLUSBUSINESSDAY || m_recurrence == R_MONTHLYGIVENBUSINESSDAY || m_recurrence == R_QUARTERLYPLUSBUSINESSDAY || m_recurrence == R_QUARTERLYGIVENBUSINESSDAY) && m_monthlyday == 0)
    {
        m_monthlyday = 1;
    }
    if(m_recurrence == R_WEEKLYGIVENBUSINESSDAY || m_recurrence == R_2WEEKLYGIVENBUSINESSDAY)
    {
        checkWeeklyGivenBusinessDay();
    }
    if(m_recurrence == R_WEEKLYPLUSBUSINESSDAY || m_recurrence == R_2WEEKLYPLUSBUSINESSDAY)
    {
        checkWeeklyPlusBusinessDay();
    }
    if(m_recurrence == R_WEEKLYBUSINESS || m_recurrence == R_2WEEKLYBUSINESS)
    {
        checkWeeklyBusinessDay();
    }
}

Event::~Event()
{
}

void Event::setEvent(const wxString &event)
{
    m_event = event;
}

void Event::setDescr(const wxString &descr)
{
    m_descr = descr;
}

wxDateTime Event::realEventDate()
{
    if(m_recurrence == R_WEEKLYPLUSBUSINESSDAY)
    {
        wxDateTime date = iengine->plusBusinessDay(m_date, m_monthlyday);
        if(m_date.GetWeekOfYear() != date.GetWeekOfYear())
        {
            m_date = m_date-wxDateSpan::Days(7);
            m_date = iengine->recurrentEventDate(*this, m_date+wxDateSpan::Days(14));
            date = iengine->plusBusinessDay(m_date, m_monthlyday);
            iengine->editEvent(*this);
        }
        return date;
    }
    if(m_recurrence == R_2WEEKLYPLUSBUSINESSDAY)
    {
        wxDateTime date = iengine->plusBusinessDay(m_date, m_monthlyday);
        if(m_date.GetWeekOfYear() != date.GetWeekOfYear())
        {
            m_date = m_date-wxDateSpan::Days(14);
            m_date = iengine->recurrentEventDate(*this, m_date+wxDateSpan::Days(28));
            date = iengine->plusBusinessDay(m_date, m_monthlyday);
            iengine->editEvent(*this);
        }
        return date;
    }
    if(m_recurrence == R_MONTHLYPLUSBUSINESSDAY || m_recurrence == R_QUARTERLYPLUSBUSINESSDAY)
    {
        return iengine->plusBusinessDay(m_date, m_monthlyday);
    }
    if(m_recurrence == R_WEEKLYGIVENBUSINESSDAY)
    {
        wxDateTime date = iengine->weeklyAtGivenBusinessDay(m_date, m_monthlyday);
        if(m_date.GetWeekOfYear() != date.GetWeekOfYear())
        {
            m_date = m_date-wxDateSpan::Days(7);
            m_date = iengine->recurrentEventDate(*this, m_date+wxDateSpan::Days(14));
            date = iengine->weeklyAtGivenBusinessDay(m_date, m_monthlyday);
            iengine->editEvent(*this);
        }
        return date;
    }
    if(m_recurrence == R_2WEEKLYGIVENBUSINESSDAY)
    {
        wxDateTime date = iengine->weeklyAtGivenBusinessDay(m_date, m_monthlyday);
        if(m_date.GetWeekOfYear() != date.GetWeekOfYear())
        {
            m_date = m_date-wxDateSpan::Days(14);
            m_date = iengine->recurrentEventDate(*this, m_date+wxDateSpan::Days(28));
            date = iengine->weeklyAtGivenBusinessDay(m_date, m_monthlyday);
            iengine->editEvent(*this);
        }
        return date;
    }
    if(m_recurrence == R_MONTHLYGIVENBUSINESSDAY || m_recurrence == R_QUARTERLYGIVENBUSINESSDAY)
    {
        wxDateTime date = m_date;
        date.SetDay(1);
        date = iengine->montlyAtGivenBusinessDay(date, m_monthlyday);
        return date;
    }
    if(m_recurrence == R_WEEKLYBUSINESS)
    {
        if(iengine->isNonworkingDay(m_date))
        {
            wxDateTime date = iengine->nextWorkingDay(m_date);
            if(m_date.GetWeekOfYear() != date.GetWeekOfYear())
            {

                m_date = m_date-wxDateSpan::Days(7);
                m_date = iengine->recurrentEventDate(*this, m_date+wxDateSpan::Days(14));
                iengine->editEvent(*this);
                date = m_date;
            }
            return date;
        }
    }
    if(m_recurrence == R_2WEEKLYBUSINESS)
    {
        if(iengine->isNonworkingDay(m_date))
        {
            wxDateTime date = iengine->nextWorkingDay(m_date);
            if(m_date.GetWeekOfYear() != date.GetWeekOfYear())
            {
                m_date = m_date-wxDateSpan::Days(14);
                m_date = iengine->recurrentEventDate(*this, m_date+wxDateSpan::Days(28));
                iengine->editEvent(*this);
                date = m_date;
            }
            return date;
        }
    }
    if(m_recurrence == R_DAILYBUSINESS || m_recurrence == R_MONTHLYBUSINESS || m_recurrence == R_QUARTERLYBUSINESS)
    {
        if(iengine->isNonworkingDay(m_date)) return iengine->nextWorkingDay(m_date);
    }
    if(m_actualreminder < 0)
    {
        return wxDateTime(m_date-wxTimeSpan::Minutes(m_actualreminder));
    }
    return m_date;
}

void Event::setDate(wxDateTime date)
{
    m_date = date;
}

void Event::setReminded(bool reminded)
{
    m_reminded = reminded;
}

int Event::day() const
{
    return m_day;
}

void Event::setDay(int day)
{
    m_day = day;
}

bool Event::hidden() const
{
    return m_hidden;
}

void Event::setHidden(bool hidden)
{
    m_hidden = hidden;
}

int Event::month() const
{
    return m_month;
}

void Event::setMonth(int month)
{
    m_month = month;
}

int Event::recurrence() const
{
    return m_recurrence;
}

void Event::setRecurrence(int recurrence)
{
    m_recurrence = recurrence;
}

int Event::minutes() const
{
    return m_minutes;
}

void Event::setMinutes(int minutes)
{
    m_minutes = minutes;
}

int Event::hours() const
{
    return m_hours;
}

void Event::setHours(int hours)
{
    m_hours = hours;
}

int Event::days() const
{
    return m_days;
}

void Event::setDays(int days)
{
    m_days = days;
}

int Event::months() const
{
    return m_months;
}

void Event::setMonths(int months)
{
    m_months = months;
}

int Event::monthlyday() const
{
    return m_monthlyday;
}

void Event::setMonthlyday(int monthlyday)
{
    m_monthlyday = monthlyday;
}

int Event::monthlyweek() const
{
    return m_monthlyweek;
}

void Event::setMonthlyweek(int monthlyweek)
{
    m_monthlyweek = monthlyweek;
}

void Event::checkWeeklyGivenBusinessDay()
{
    if(m_monthlyday > 5)
    {
        m_monthlyday = 5;
    }
}

void Event::checkWeeklyPlusBusinessDay()
{
    if(static_cast<wxDateTime::WeekDay>(m_day) == wxDateTime::Sat)
    {
        m_day = wxDateTime::Fri;
        m_date -= wxDateSpan::Day();
    }
    if(m_date.GetWeekDay() == wxDateTime::Sat)
    {
        m_date -= wxDateSpan::Day();
    }
    if(static_cast<wxDateTime::WeekDay>(m_day) == wxDateTime::Sun)
    {
        m_day = wxDateTime::Fri;
        m_date -= wxDateSpan::Days(2);
    }
    if(m_date.GetWeekDay() == wxDateTime::Sun)
    {
        m_date -= wxDateSpan::Days(2);
    }
    if(m_monthlyday+m_day > 5)
    {
        m_monthlyday = 5 - m_day;
    }
}

void Event::checkWeeklyBusinessDay()
{
    if(static_cast<wxDateTime::WeekDay>(m_day) == wxDateTime::Sat)
    {
        m_day = wxDateTime::Fri;
        m_date -= wxDateSpan::Day();
    }
    if(m_date.GetWeekDay() == wxDateTime::Sat)
    {
        m_date -= wxDateSpan::Day();
    }
    if(static_cast<wxDateTime::WeekDay>(m_day) == wxDateTime::Sun)
    {
        m_day = wxDateTime::Fri;
        m_date -= wxDateSpan::Days(2);
    }
    if(m_date.GetWeekDay() == wxDateTime::Sun)
    {
        m_date -= wxDateSpan::Days(2);
    }
}

wxString Event::tipText(bool withrecurrence)
{
    wxString text;
    text << m_event;
    text << "\n";
    text << _("Date:");
    text << " ";
    text << dxutils::formatDateTime(realEventDate(), dxsettings.dateFormat(), dxsettings.timeFormat(), dxsettings.dateLayout());
    text << "\n";
    text << _("Reminder:");
    text << " ";
    text << reminderText();
    if(withrecurrence)
    {
        text << "\n";
        text << _("Recurrence:");
        text << " ";
        text << recurrenceText();
        if(!m_until.IsSameDate(INFINITY_DATE))
        {
            text << "\n";
            text << _("Until:");
            text << " ";
            text << dxutils::formatDate(m_until, dxsettings.dateFormat());
        }
    }
    return text;
}

wxString Event::tipTextRecurrent()
{
    wxString tip = m_event;
    tip << "\n";
    tip << _("Date:");
    tip << " ";
    tip << dxutils::formatDateTime(realEventDate(), dxsettings.dateFormat(), dxsettings.timeFormat(), dxsettings.dateLayout());
    tip << "\n";
    tip << _("Recurrence:");
    tip << " ";
    tip << recurrenceText();
    if(!m_until.IsSameDate(INFINITY_DATE))
    {
        tip << "\n";
        tip << _("Until:");
        tip << " ";
        tip << dxutils::formatDate(m_until, dxsettings.dateFormat());
    }
    return tip;
}

wxString Event::reminderText() const
{
    if(m_actualreminder < 0)
    {
        return _("At set time");
    }
    switch(m_actualreminder)
    {
        case 0: return _("At set time"); //0 mins
        case 5: return _("5 minutes before"); //5 mins
        case 15: return _("15 minutes before"); //15 mins
        case 60: return _("1 hour before"); //1 hours
        case 120: return _("2 hours before"); //2 hours
        case 1440: return _("1 day before"); //1 day
        case 2880: return _("2 days before"); //2 days
        default:
        {
            wxInt64 mins = m_actualreminder;
            wxInt64 days = mins/(60*24);
            mins -= days*60*24;
            wxInt64 hours = mins/60;
            mins -= hours*60;
            wxString text = _("Custom");
            if(mins || hours || days)
            {
                text << " (";
            }
            if(days)
            {
                text << wxString::Format(wxPLURAL("%lld day","%lld days",days),days);
                if(mins || hours) text << ", ";
            }
            if(hours)
            {
                text << wxString::Format(wxPLURAL("%lld hour","%lld hours",hours),hours);
                if(mins) text << ", ";
            }
            if(mins)
            {
                text << wxString::Format(wxPLURAL("%lld minute","%lld minutes",mins),mins);
            }
            if(mins || hours || days)
            {
                text << ")";
            }
            return text;
        }
    }
}

wxString Event::recurrenceText() const
{
    switch(m_recurrence) {
    case R_ONCENOTDELETE: return _("Once (Do not delete)");
    case R_DAILY: return _("Daily");
    case R_DAILYBUSINESS: return _("Daily at business day");
    case R_WEEKLY: return _("Weekly");
    case R_2WEEKLY: return _("Fortnightly");
    case R_WEEKLYBUSINESS: return _("Weekly at business day");
    case R_WEEKLYPLUSBUSINESSDAY: return wxString::Format(wxPLURAL("Weekly plus %d business day","Weekly plus %d business days",m_monthlyday), m_monthlyday);
    case R_WEEKLYGIVENBUSINESSDAY: return wxString::Format(_("Weekly at %d business day"), m_monthlyday);
    case R_2WEEKLYBUSINESS: return _("Fortnightly at business day");
    case R_2WEEKLYPLUSBUSINESSDAY: return wxString::Format(wxPLURAL("Fortnightly plus %d business day","Fortnightly plus %d business days",m_monthlyday), m_monthlyday);
    case R_2WEEKLYGIVENBUSINESSDAY: return wxString::Format(_("Fortnightly at %d business day"), m_monthlyday);
    case R_MONTHLY: return _("Monthly");
    case R_MONTHLYBUSINESS: return _("Monthly at business day");
    case R_QUARTERLY: return _("Quarterly");
    case R_QUARTERLYBUSINESS: return _("Quarterly at business day");
    case R_YEARLY: return _("Yearly");
    case R_MONTHLYPLUSBUSINESSDAY:
    {
        return wxString::Format(wxPLURAL("Monthly plus %d business day","Monthly plus %d business days",m_monthlyday),m_monthlyday);
    }
    case R_MONTHLYGIVENBUSINESSDAY:
    {
        return wxString::Format(_("Monthly at %d business day"),m_monthlyday);
    }
    case R_QUARTERLYPLUSBUSINESSDAY:
    {
        return wxString::Format(wxPLURAL("Quarterly plus %d business day","Quarterly plus %d business days",m_monthlyday),m_monthlyday);
    }
    case R_QUARTERLYGIVENBUSINESSDAY:
    {
        return wxString::Format(_("Quarterly at %d business day"),m_monthlyday);
    }
    case R_MONTHLYATDAY:
    {
        return wxString::Format(_("Monthly every %d. %s of month"), m_monthlyweek, wxDateTime::GetWeekDayName(static_cast<wxDateTime::WeekDay>(wxMin(6,m_monthlyday))));
    }
    case R_OWN:
    {
        wxString text = _("Custom");
        if(m_minutes || m_hours || m_days || m_months)
        {
            text << " (";
        }
        if(m_months)
        {
            text << wxString::Format(wxPLURAL("%d month","%d months",m_months),m_months);
            if(m_minutes || m_hours || m_days) text << ", ";
        }
        if(m_days)
        {
            text << wxString::Format(wxPLURAL("%d day","%d days",m_days),m_days);
            if(m_minutes || m_hours) text << ", ";
        }
        if(m_hours)
        {
            text << wxString::Format(wxPLURAL("%d hour","%d hours",m_hours),m_hours);
            if(m_minutes) text << ", ";
        }
        if(m_minutes)
        {
            text << wxString::Format(wxPLURAL("%d minute","%d minutes",m_minutes),m_minutes);
        }
        if(m_minutes || m_hours || m_days || m_months)
        {
            text << ")";
        }
        return text;
    }
    case R_HOURLY:
    {
        return wxString::Format(wxPLURAL("Hourly every %d hour","Hourly every %d hours",m_hours), m_hours);
    }
    case R_MINUTES:
    {
        return wxString::Format(wxPLURAL("Every %d minute","Every %d minutes",m_minutes), m_minutes);
    }
    case R_NONE:
    {
        return _("None (Do not delete)");
    }
    default: return _("Once");
    }
}

wxInt64 Event::id() const
{
    return m_id;
}

wxInt64 Event::reminder() const
{
    return m_reminder;
}

void Event::setReminder(const wxInt64 &reminder)
{
    m_reminder = reminder;
}

wxInt64 Event::actualreminder() const
{
    return m_actualreminder;
}

void Event::setActualreminder(const wxInt64 &actualreminder)
{
    m_actualreminder = actualreminder;
}

wxDateTime Event::until() const
{
    return m_until;
}

void Event::setUntil(const wxDateTime &until)
{
    m_until = until;
}

bool Event::always() const
{
    return m_always;
}

void Event::setAlways(bool newAlways)
{
    m_always = newAlways;
}

int Event::color() const
{
    return m_color;
}

void Event::setColor(int newColor)
{
    m_color = newColor;
}

EventView::EventView()
    : m_id(-1), m_name(wxEmptyString), m_reminder(wxEmptyString), m_recurrence(wxEmptyString), m_date(INFINITY_DATE), m_highlight(0), m_recurrencetype(R_ONCE), m_always(false)
{
}

EventView::EventView(wxInt64 id, const wxString &name, const wxDateTime &date, const wxString &reminder, const wxString &recurrence, int highlight, int recurrencetype, bool always)
    : m_id(id), m_name(name), m_reminder(reminder), m_recurrence(recurrence), m_date(date), m_highlight(highlight), m_recurrencetype(recurrencetype), m_always(always)
{
}

wxString EventView::name() const
{
    return m_name;
}

void EventView::setName(const wxString &name)
{
    m_name = name;
}

wxString EventView::reminder() const
{
    return m_reminder;
}

void EventView::setReminder(const wxString &reminder)
{
    m_reminder = reminder;
}

wxString EventView::recurrence() const
{
    return m_recurrence;
}

void EventView::setRecurrence(const wxString &recurrence)
{
    m_recurrence = recurrence;
}

wxDateTime EventView::date() const
{
    return m_date;
}

void EventView::setDate(const wxDateTime &date)
{
    m_date = date;
}

wxInt64 EventView::id() const
{
    return m_id;
}

void EventView::setId(const wxInt64 &id)
{
    m_id = id;
}

int EventView::highlight() const
{
    return m_highlight;
}

void EventView::setHighlight(int highlight)
{
    m_highlight = highlight;
}

int EventView::recurrencetype() const
{
    return m_recurrencetype;
}

void EventView::setRecurrencetype(int recurrencetype)
{
    m_recurrencetype = recurrencetype;
}

bool EventView::always() const
{
    return m_always;
}

void EventView::setAlways(bool newAlways)
{
    m_always = newAlways;
}
