"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WORKFLOW_STEP_TYPE = exports.WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP = exports.WORKFLOW_STATE = exports.WORKFLOW_RESOURCE_TYPE = exports.MODEL_STATE = exports.MODEL_ALGORITHM = void 0;
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
/**
 ********** WORKFLOW TYPES/INTERFACES **********
 */
/**
 ********** MODAL SUB-FORM TYPES/INTERFACES **********
 We persist sub-forms in the form modals s.t. data is only
 saved back to the parent form if the user explicitly saves.

 We define the structure of the forms here.
 */
// Ingest docs modal
// Search request modal
// Configure template modal
// Configure expression modal
// Configure multi-expression modal
/**
 ********** WORKSPACE TYPES/INTERFACES **********
 */
/**
 * An input field for a component. Specifies enough configuration for the
 * UI node to render it properly (help text, links, etc.)
 */
/**
 * The base interface the components will implement.
 */
/**
 * We need to include some extra instance-specific data to the ReactFlow component
 * to perform extra functionality, such as deleting the node from the ReactFlowInstance.
 */
/**
 ********** USE CASE TEMPLATE TYPES/INTERFACES **********
 */
// A stateless template of a workflow
// An instance of a workflow based on a workflow template
/**
 ********** ML PLUGIN TYPES/INTERFACES **********
 */
// Based off of https://github.com/opensearch-project/ml-commons/blob/main/common/src/main/java/org/opensearch/ml/common/model/MLModelState.java
let MODEL_STATE = exports.MODEL_STATE = /*#__PURE__*/function (MODEL_STATE) {
  MODEL_STATE["REGISTERED"] = "Registered";
  MODEL_STATE["REGISTERING"] = "Registering";
  MODEL_STATE["DEPLOYING"] = "Deploying";
  MODEL_STATE["DEPLOYED"] = "Deployed";
  MODEL_STATE["PARTIALLY_DEPLOYED"] = "Partially deployed";
  MODEL_STATE["UNDEPLOYED"] = "Undeployed";
  MODEL_STATE["DEPLOY_FAILED"] = "Deploy failed";
  return MODEL_STATE;
}({}); // Based off of https://github.com/opensearch-project/ml-commons/blob/main/common/src/main/java/org/opensearch/ml/common/FunctionName.java
let MODEL_ALGORITHM = exports.MODEL_ALGORITHM = /*#__PURE__*/function (MODEL_ALGORITHM) {
  MODEL_ALGORITHM["LINEAR_REGRESSION"] = "Linear regression";
  MODEL_ALGORITHM["KMEANS"] = "K-means";
  MODEL_ALGORITHM["AD_LIBSVM"] = "AD LIBSVM";
  MODEL_ALGORITHM["SAMPLE_ALGO"] = "Sample algorithm";
  MODEL_ALGORITHM["LOCAL_SAMPLE_CALCULATOR"] = "Local sample calculator";
  MODEL_ALGORITHM["FIT_RCF"] = "Fit RCF";
  MODEL_ALGORITHM["BATCH_RCF"] = "Batch RCF";
  MODEL_ALGORITHM["ANOMALY_LOCALIZATION"] = "Anomaly localization";
  MODEL_ALGORITHM["RCF_SUMMARIZE"] = "RCF summarize";
  MODEL_ALGORITHM["LOGISTIC_REGRESSION"] = "Logistic regression";
  MODEL_ALGORITHM["TEXT_EMBEDDING"] = "Text embedding";
  MODEL_ALGORITHM["METRICS_CORRELATION"] = "Metrics correlation";
  MODEL_ALGORITHM["REMOTE"] = "Remote";
  MODEL_ALGORITHM["SPARSE_ENCODING"] = "Sparse encoding";
  MODEL_ALGORITHM["SPARSE_TOKENIZE"] = "Sparse tokenize";
  MODEL_ALGORITHM["TEXT_SIMILARITY"] = "Text similarity";
  MODEL_ALGORITHM["QUESTION_ANSWERING"] = "Question answering";
  MODEL_ALGORITHM["AGENT"] = "Agent";
  return MODEL_ALGORITHM;
}({}); // Based off of JSONSchema. For more info, see https://json-schema.org/understanding-json-schema/reference/type
// For rendering options, we extract the name (the key in the input/output obj) and combine into a single obj.
// Also persist an optional field to dynamically run / not run validation on the form later on
/**
 ********** MISC TYPES/INTERFACES ************
 */
// Based off of https://github.com/opensearch-project/flow-framework/blob/main/src/main/java/org/opensearch/flowframework/model/State.java
let WORKFLOW_STATE = exports.WORKFLOW_STATE = /*#__PURE__*/function (WORKFLOW_STATE) {
  WORKFLOW_STATE["NOT_STARTED"] = "Not started";
  WORKFLOW_STATE["PROVISIONING"] = "Provisioning";
  WORKFLOW_STATE["FAILED"] = "Failed";
  WORKFLOW_STATE["COMPLETED"] = "Completed";
  return WORKFLOW_STATE;
}({});
// Based off of https://github.com/opensearch-project/flow-framework/blob/main/src/main/java/org/opensearch/flowframework/common/WorkflowResources.java
let WORKFLOW_RESOURCE_TYPE = exports.WORKFLOW_RESOURCE_TYPE = /*#__PURE__*/function (WORKFLOW_RESOURCE_TYPE) {
  WORKFLOW_RESOURCE_TYPE["PIPELINE_ID"] = "Ingest pipeline";
  WORKFLOW_RESOURCE_TYPE["INDEX_NAME"] = "Index";
  WORKFLOW_RESOURCE_TYPE["MODEL_ID"] = "Model";
  WORKFLOW_RESOURCE_TYPE["MODEL_GROUP_ID"] = "Model group";
  WORKFLOW_RESOURCE_TYPE["CONNECTOR_ID"] = "Connector";
  return WORKFLOW_RESOURCE_TYPE;
}({});
let WORKFLOW_STEP_TYPE = exports.WORKFLOW_STEP_TYPE = /*#__PURE__*/function (WORKFLOW_STEP_TYPE) {
  WORKFLOW_STEP_TYPE["CREATE_INGEST_PIPELINE_STEP_TYPE"] = "create_ingest_pipeline";
  WORKFLOW_STEP_TYPE["CREATE_SEARCH_PIPELINE_STEP_TYPE"] = "create_search_pipeline";
  WORKFLOW_STEP_TYPE["CREATE_INDEX_STEP_TYPE"] = "create_index";
  return WORKFLOW_STEP_TYPE;
}({}); // We cannot disambiguate ingest vs. search pipelines based on workflow resource type. To work around
// that, we maintain this map from workflow step type -> formatted type
let WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP = exports.WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP = /*#__PURE__*/function (WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP) {
  WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP["create_ingest_pipeline"] = "Ingest pipeline";
  WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP["create_search_pipeline"] = "Search pipeline";
  WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP["create_index"] = "Index";
  return WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP;
}({});
/**
 ********** OPENSEARCH TYPES/INTERFACES ************
 */
// from https://opensearch.org/docs/latest/ingest-pipelines/simulate-ingest/#example-specify-a-pipeline-in-the-path
// from https://opensearch.org/docs/latest/ingest-pipelines/simulate-ingest/#example-specify-a-pipeline-in-the-path
// from https://opensearch.org/docs/latest/ingest-pipelines/simulate-ingest/#example-specify-a-pipeline-in-the-path
// verbose mode
// from https://opensearch.org/docs/latest/ingest-pipelines/simulate-ingest/#query-parameters
// verbose mode
// from https://opensearch.org/docs/latest/ingest-pipelines/simulate-ingest/#query-parameters
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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