"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerText2VizRoutes = registerText2VizRoutes;
var _configSchema = require("@osd/config-schema");
var _llm = require("../../common/constants/llm");
var _error_handler = require("./error_handler");
var _style_single_metric = require("../utils/style_single_metric");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const inputSchema = _configSchema.schema.string({
  maxLength: _llm.TEXT2VEGA_INPUT_SIZE_LIMIT,
  validate(value) {
    if (!value || value.trim().length === 0) {
      return "can't be empty or blank.";
    }
  }
});
function registerText2VizRoutes(router, assistantService) {
  router.post({
    path: _llm.TEXT2VIZ_API.TEXT2VEGA,
    validate: {
      body: _configSchema.schema.object({
        input_question: inputSchema,
        input_instruction: _configSchema.schema.maybe(_configSchema.schema.string({
          maxLength: _llm.TEXT2VEGA_INPUT_SIZE_LIMIT
        })),
        ppl: _configSchema.schema.string(),
        dataSchema: _configSchema.schema.string(),
        sampleData: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, router.handleLegacyErrors(async (context, req, res) => {
    const assistantClient = assistantService.getScopedClient(req, context);
    try {
      const agentConfigName = req.body.input_instruction ? _llm.TEXT2VEGA_WITH_INSTRUCTIONS_AGENT_CONFIG_ID : _llm.TEXT2VEGA_RULE_BASED_AGENT_CONFIG_ID;
      const response = await assistantClient.executeAgentByConfigName(agentConfigName, {
        input_question: req.body.input_question,
        input_instruction: req.body.input_instruction,
        ppl: req.body.ppl,
        dataSchema: req.body.dataSchema,
        sampleData: req.body.sampleData
      });
      let textContent = response.body.inference_results[0].output[0].result;
      // Check if the visualization is single value:
      // it should have exactly 1 metric and no dimensions.
      let ifSingleMetric = (0, _style_single_metric.checkSingleMetric)(textContent);

      // extra content between tag <vega-lite></vega-lite>
      const startTag = '<vega-lite>';
      const endTag = '</vega-lite>';
      const startIndex = textContent.indexOf(startTag);
      const endIndex = textContent.indexOf(endTag);
      if (startIndex !== -1 && endIndex !== -1 && startIndex < endIndex) {
        // Extract the content between the tags
        textContent = textContent.substring(startIndex + startTag.length, endIndex).trim();
      }
      // extract json object
      const jsonMatch = textContent.match(/\{.*\}/s);
      if (jsonMatch) {
        let result = JSON.parse(jsonMatch[0]);
        // sometimes llm returns {response: <schema>} instead of <schema>
        if (result.response) {
          result = JSON.parse(result.response);
        }
        // Sometimes the response contains width and height which is not needed, here delete the these fields
        delete result.width;
        delete result.height;
        if (ifSingleMetric) {
          result = (0, _style_single_metric.addTitleTextLayer)(result);
        }
        // make sure $schema field always been added, sometimes, LLM 'forgot' to add this field
        result.$schema = 'https://vega.github.io/schema/vega-lite/v5.json';
        return res.ok({
          body: result
        });
      }
      return res.badRequest();
    } catch (e) {
      return (0, _error_handler.handleError)(e, res, context.assistant_plugin.logger);
    }
  }));
  router.post({
    path: _llm.TEXT2VIZ_API.TEXT2PPL,
    validate: {
      body: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        question: inputSchema
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, router.handleLegacyErrors(async (context, req, res) => {
    const assistantClient = assistantService.getScopedClient(req, context);
    try {
      const response = await assistantClient.executeAgentByConfigName(_llm.TEXT2PPL_AGENT_CONFIG_ID, {
        question: req.body.question,
        index: req.body.index
      });
      const result = JSON.parse(response.body.inference_results[0].output[0].result);
      return res.ok({
        body: result
      });
    } catch (e) {
      return (0, _error_handler.handleError)(e, res, context.assistant_plugin.logger);
    }
  }));
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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