"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AgentFrameworkStorageService = void 0;
var _message_parser_runner = require("../../utils/message_parser_runner");
var _constants = require("../../utils/constants");
var _format = require("../../utils/format");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

class AgentFrameworkStorageService {
  constructor(clientTransport, messageParsers = []) {
    this.clientTransport = clientTransport;
    this.messageParsers = messageParsers;
  }
  async getConversation(conversationId) {
    const [interactionsResp, conversation] = await Promise.all([this.clientTransport.request({
      method: 'GET',
      path: `${_constants.ML_COMMONS_BASE_API}/memory/${encodeURIComponent(conversationId)}/messages?max_results=1000`
    }), this.clientTransport.request({
      method: 'GET',
      path: `${_constants.ML_COMMONS_BASE_API}/memory/${encodeURIComponent(conversationId)}`
    })]);
    const finalInteractions = interactionsResp.body.messages.map(item => (0, _format.formatInteractionFromBackend)(item));
    return {
      title: conversation.body.name,
      createdTimeMs: +new Date(conversation.body.create_time),
      updatedTimeMs: +new Date(conversation.body.updated_time),
      messages: await this.getMessagesFromInteractions(finalInteractions),
      interactions: finalInteractions
    };
  }
  async getConversations(query) {
    let sortField = '';
    if (query.sortField === 'updatedTimeMs') {
      sortField = 'updated_time';
    } else if (query.sortField === 'createTimeMs') {
      sortField = 'create_time';
    }
    let searchFields = [];
    if (query.search && query.searchFields) {
      if (typeof query.searchFields === 'string') {
        searchFields = [...searchFields, query.searchFields.replace('title', 'name')];
      } else {
        searchFields = query.searchFields.map(item => item.replace('title', 'name'));
      }
    }
    const requestParams = {
      from: (query.page - 1) * query.perPage,
      size: query.perPage,
      ...(searchFields.length > 0 && {
        query: {
          multi_match: {
            query: query.search,
            fields: searchFields
          }
        }
      }),
      ...(searchFields.length === 0 && {
        query: {
          match_all: {}
        }
      }),
      ...(sortField && query.sortOrder && {
        sort: [{
          [sortField]: query.sortOrder
        }]
      })
    };
    const conversations = await this.clientTransport.request({
      method: 'GET',
      path: `${_constants.ML_COMMONS_BASE_API}/memory/_search`,
      body: requestParams
    });
    return {
      objects: conversations.body.hits.hits.filter(hit => hit._source !== null && hit._source !== undefined).map(item => ({
        id: item._id,
        title: item._source.name,
        version: 1,
        createdTimeMs: Date.parse(item._source.create_time),
        updatedTimeMs: Date.parse(item._source.updated_time),
        messages: []
      })),
      total: typeof conversations.body.hits.total === 'number' ? conversations.body.hits.total : conversations.body.hits.total.value
    };
  }
  async saveMessages(title, conversationId, messages) {
    throw new Error('Method is not needed');
  }
  async deleteConversation(conversationId) {
    const response = await this.clientTransport.request({
      method: 'DELETE',
      path: `${_constants.ML_COMMONS_BASE_API}/memory/${encodeURIComponent(conversationId)}`
    });
    if (response.statusCode === 200) {
      return {
        success: true
      };
    } else {
      return {
        success: false,
        statusCode: response.statusCode,
        message: JSON.stringify(response.body)
      };
    }
  }
  async updateConversation(conversationId, title) {
    const response = await this.clientTransport.request({
      method: 'PUT',
      path: `${_constants.ML_COMMONS_BASE_API}/memory/${encodeURIComponent(conversationId)}`,
      body: {
        name: title
      }
    });
    if (response.statusCode === 200) {
      return {
        success: true
      };
    } else {
      return {
        success: false,
        statusCode: response.statusCode,
        message: JSON.stringify(response.body)
      };
    }
  }
  async getTraces(interactionId) {
    const response = await this.clientTransport.request({
      method: 'GET',
      path: `${_constants.ML_COMMONS_BASE_API}/memory/message/${encodeURIComponent(interactionId)}/traces`
    });
    return response.body.traces.map(item => ({
      interactionId: item.message_id,
      input: item.input,
      output: item.response,
      createTime: item.create_time,
      origin: item.origin,
      traceNumber: item.trace_number
    }));
  }
  async updateInteraction(interactionId, additionalInfo) {
    const response = await this.clientTransport.request({
      method: 'PUT',
      path: `${_constants.ML_COMMONS_BASE_API}/memory/message/${encodeURIComponent(interactionId)}`,
      body: {
        additional_info: additionalInfo
      }
    });
    if (response.statusCode === 200) {
      return {
        success: true
      };
    } else {
      return {
        success: false,
        statusCode: response.statusCode,
        message: JSON.stringify(response.body)
      };
    }
  }
  async getMessagesFromInteractions(interactions) {
    const messageParserRunner = new _message_parser_runner.MessageParserRunner(this.messageParsers);
    const finalInteractions = [...interactions];
    let finalMessages = [];
    for (const interaction of finalInteractions) {
      finalMessages = [...finalMessages, ...(await messageParserRunner.run(interaction, {
        interactions: [...(finalInteractions || [])]
      }))];
    }
    return finalMessages;
  }
  async getInteraction(conversationId, interactionId) {
    if (!conversationId) {
      throw new Error('conversationId is required');
    }
    if (!interactionId) {
      throw new Error('interactionId is required');
    }
    const interactionsResp = await this.clientTransport.request({
      method: 'GET',
      path: `${_constants.ML_COMMONS_BASE_API}/memory/message/${encodeURIComponent(interactionId)}`
    });
    return (0, _format.formatInteractionFromBackend)(interactionsResp.body);
  }
}
exports.AgentFrameworkStorageService = AgentFrameworkStorageService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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