"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.handleWithCallback = void 0;
exports.registerIntegrationsRoute = registerIntegrationsRoute;
exports.serveStaticImage = void 0;
var _configSchema = require("@osd/config-schema");
var mime = _interopRequireWildcard(require("mime"));
var _isomorphicDompurify = _interopRequireDefault(require("isomorphic-dompurify"));
var _sanitizeFilename = _interopRequireDefault(require("sanitize-filename"));
var _shared = require("../../../common/constants/shared");
var _integrations_manager = require("../../adaptors/integrations/integrations_manager");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

/**
 * Handle an `OpenSearchDashboardsRequest` using the provided `callback` function.
 * This is a convenience method that handles common error handling and response formatting.
 * The callback must accept a `IntegrationsManager` as its first argument.
 *
 * If the callback throws an error,
 * the `OpenSearchDashboardsResponse` will be formatted using the error's `statusCode` and `message` properties.
 * Otherwise, the callback's return value will be formatted in a JSON object under the `data` field.
 *
 * @param {IntegrationsManager} manager The integration manager instance to use for making requests.
 * @param {OpenSearchDashboardsResponseFactory} response The factory to use for creating responses.
 * @callback callback A callback that will invoke a request on a provided adaptor.
 * @returns {Promise<OpenSearchDashboardsResponse>} An `OpenSearchDashboardsResponse` with the return data from the callback.
 */
const handleWithCallback = async (manager, response, callback) => {
  try {
    const data = await callback(manager);
    return response.ok({
      body: {
        data
      }
    });
  } catch (e) {
    const err = e;
    console.error(`handleWithCallback: callback failed with error "${err.message}"`);
    return response.custom({
      statusCode: err.statusCode || 500,
      body: err.message
    });
  }
};
exports.handleWithCallback = handleWithCallback;
const serveStaticImage = (path, content, response) => {
  const mtype = mime.getType(path);
  switch (mime.getType(path)) {
    case 'image/gif':
    case 'image/jpeg':
    case 'image/png':
    case 'image/tiff':
    case 'image/webp':
    case 'image/avif':
      return response.ok({
        headers: {
          'Content-Type': mtype
        },
        body: content
      });
    case 'image/svg+xml':
      return response.ok({
        headers: {
          'Content-Type': mtype
        },
        body: _isomorphicDompurify.default.sanitize(content.toString('utf8'))
      });
    default:
      return response.custom({
        body: `not a supported image type: ${mtype}`,
        statusCode: 400
      });
  }
};
exports.serveStaticImage = serveStaticImage;
function registerIntegrationsRoute(router) {
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository`,
    validate: false
  }, async (context, request, response) => {
    const adaptor = new _integrations_manager.IntegrationsManager(context.core.savedObjects.client);
    return handleWithCallback(adaptor, response, async a => a.getIntegrationTemplates());
  });
  router.post({
    path: `${_shared.INTEGRATIONS_BASE}/store/{templateName}`,
    validate: {
      params: _configSchema.schema.object({
        templateName: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({
        dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        })),
        dataSourceMDSLabel: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        })),
        name: _configSchema.schema.string(),
        indexPattern: _configSchema.schema.string(),
        workflows: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
        dataSource: _configSchema.schema.maybe(_configSchema.schema.string()),
        tableName: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const adaptor = new _integrations_manager.IntegrationsManager(context.core.savedObjects.client);
    return handleWithCallback(adaptor, response, async a => {
      return a.loadIntegrationInstance(request.params.templateName, request.body.name, request.body.indexPattern, request.body.dataSourceMDSId, request.body.dataSourceMDSLabel, request.body.workflows, request.body.dataSource, request.body.tableName);
    });
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{name}`,
    validate: {
      params: _configSchema.schema.object({
        name: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = new _integrations_manager.IntegrationsManager(context.core.savedObjects.client);
    return handleWithCallback(adaptor, response, async a => (await a.getIntegrationTemplates({
      name: request.params.name
    })).hits[0]);
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{id}/static/{path}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string(),
        path: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = new _integrations_manager.IntegrationsManager(context.core.savedObjects.client);
    try {
      const requestPath = (0, _sanitizeFilename.default)(request.params.path);
      const result = await adaptor.getStatic(request.params.id, requestPath);
      return serveStaticImage(request.params.path, result, response);
    } catch (e) {
      const err = e;
      return response.custom({
        statusCode: err.statusCode ? err.statusCode : 500,
        body: err.message
      });
    }
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{id}/schema`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = new _integrations_manager.IntegrationsManager(context.core.savedObjects.client);
    return handleWithCallback(adaptor, response, async a => a.getSchemas(request.params.id));
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{id}/assets`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = new _integrations_manager.IntegrationsManager(context.core.savedObjects.client);
    return handleWithCallback(adaptor, response, async a => a.getAssets(request.params.id));
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{id}/data`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = new _integrations_manager.IntegrationsManager(context.core.savedObjects.client);
    return handleWithCallback(adaptor, response, async a => a.getSampleData(request.params.id));
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/store`,
    validate: false
  }, async (context, request, response) => {
    const adaptor = new _integrations_manager.IntegrationsManager(context.core.savedObjects.client);
    return handleWithCallback(adaptor, response, async a => a.getIntegrationInstances({}));
  });
  router.delete({
    path: `${_shared.INTEGRATIONS_BASE}/store/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = new _integrations_manager.IntegrationsManager(context.core.savedObjects.client);
    return handleWithCallback(adaptor, response, async a => a.deleteIntegrationInstance(request.params.id));
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/store/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = new _integrations_manager.IntegrationsManager(context.core.savedObjects.client);
    return handleWithCallback(adaptor, response, async a => a.getIntegrationInstance({
      id: request.params.id
    }));
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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