/**************************************************************************
* Waveform version 1.0: generation of POVray code for time-based triangle *
*                       meshes of propagating wave fronts                 *
*                                                                         *
*   Copyright (C) 1997  Aaron Gage                                        *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the Free Software           *
*   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.             *
**************************************************************************/

#include <stdio.h>
#include <math.h>

#define MAX(a, b)	(a > b? a : b)
#define MIN(a, b)	(a < b? a : b)

#define	HEIGHT	200	/* Z direction from 0 to HEIGHT	*/
#define WIDTH	200	/* X direction from 0 to WIDTH	*/

#define V	2.5	/* rate of propagation of wave (units/time)	*/
#define b	0.025	/* loss of energy per unit distance	*/
#define OMEGA	0.375	/* frequency of wave			*/
#define N	1	/* number of waves to send from source	*/
#define AMP	32000	/* amplitude of wave (-32000 - 32000)	*/

struct vertex_data
{
	int value;
	float normal[3];
};

float normalize(float normal[])
{
   float length;

   length = sqrt(pow(normal[0], 2) + pow(normal[1], 2) + pow(normal[2], 2));
   normal[0] /= length;
   normal[1] /= length;
   normal[2] /= length;
   return length;
}

int main(int argc, char **argv)
{
   struct vertex_data Matrix[WIDTH][HEIGHT];
   float dist = 0;
   int deltaY = 0;
   int temp;
   int sources = 0;
   int max_sources = 0;
   int row;
   int column;
   int X0, Z0;
   float tau;
   int time;
   int i, j;
   int V1[3], V2[3], V3[3];
   float length;
   float I, J, K;
   int A, B, C, D, E, F;
   int leg1[3], leg2[3];
   FILE *output;

   if(argc < 2 || (argc - 1)%3)
	{
	fprintf(stderr, "Incorrect number of arguments given.\n\r");
	fprintf(stderr, "Usage: %s <time elapsed> <X0> <Z0> ...\n\r", argv[0]);
	fprintf(stderr, "\tX0 and Z0 are (X, Z) coordinates where the source of the wave is\n\r\tlocated on the XZ plane.\n\r");
	fprintf(stderr, "\tThe time is how long ago this source began.\n\r");
	fprintf(stderr, "\tMore sources can be given in the same form at different\n\r\tpoints and origins.\n\r");
	return 1;
	}

   for(i = 0; i < HEIGHT; i++)
	for(j = 0; j < WIDTH; j++)
		{
		Matrix[j][i].value = 0;
		Matrix[j][i].normal[0] = 0;
		Matrix[j][i].normal[1] = 0;
		Matrix[j][i].normal[2] = 0;
		}

   max_sources = (argc - 1)/3;

   for(sources = 0; sources < max_sources; sources++)
	{
	X0 = atoi(argv[sources*3 +2]);
	Z0 = atoi(argv[sources*3 +3]);
	time = atoi(argv[sources*3 +1]);
	for(row = 0; row < HEIGHT; row++)
		{
		for(column = 0; column < WIDTH; column++)
			{
			/* calculate distance from origin to current point */
			dist = sqrt(pow(column - X0, 2) + pow(row - Z0, 2));

			/* tau is the time delay between this point and the
			   origin, based on wave velocity */
			tau = dist/(1.0 * V);

			/* if the wave has not come this far, no change */
			if(tau > time)
				continue;

			/* if enough waves have passed here, no change */
			if(OMEGA*(time - tau) >= N*2*M_PI)
				continue;

			deltaY = (1-MIN(b*dist, 1))*AMP*sin(OMEGA*(time - tau));

			/* bounds checking */
			temp = Matrix[column][row].value + deltaY;
			/* temp = MIN(temp, 127);
			temp = MAX(temp, -128); */

			temp = MIN(temp, 32000);
			temp = MAX(temp, -32000);

			Matrix[column][row].value = temp;
			}
		}
	}

/* The shape of the triangle mesh will be as follows:

	x_______x_______x_______x_______x
	|      /|      /|      /|      /|
	|    /  |    /  |    /  |    /  |
	|  /    |  /    |  /    |  /    |
	x/______x/______x/______x/______x
	|      /|      /|      /|      /|
	|    /  |    /  |    /  |    /  |
	|  /    |  /    |  /    |  /    |
	x/______x/______x/______x/______x
	|      /|      /|      /|      /|
	|    /  |    /  |    /  |    /  |
	|  /    |  /    |  /    |  /    |
	x/______x/______x/______x/______x

	Each vertex, marked as 'x', is shared by exactly six
	triangles (except for the corners, which are special
	cases).  Each of triangles gives 1/6 of its normalized
	vector to the normal for the vertex (for the Phong
	interpolation).

*/

   /* traverse triangles and perform normal calculations */

   for(row = 0; row < HEIGHT - 1; row++)
	{
	for(column = 0; column < WIDTH - 1; column++)
		{
		V1[0] = column;
		V1[1] = Matrix[column][row].value;
		V1[2] = row;

		V2[0] = column;
		V2[1] = Matrix[column][row+1].value;
		V2[2] = row + 1;

		V3[0] = column + 1;
		V3[1] = Matrix[column + 1][row].value;
		V3[2] = row;

		/* MUST use order of leg1 X leg2 */

		leg1[0] = V2[0] - V1[0];
		leg1[1] = V2[1] - V1[1];
		leg1[2] = V2[2] - V1[2];

		leg2[0] = V3[0] - V1[0];
		leg2[1] = V3[1] - V1[1];
		leg2[2] = V3[2] - V1[2];

   		A = leg1[0];
		B = leg1[1];
		C = leg1[2];

		D = leg2[0];
		E = leg2[1];
		F = leg2[2];

		I = F*B-E*C;
		J = C*D-F*A;
		K = A*E-D*B;

		/* I, J, and K are components of normal vector */
		/* now to normalize */

		length = sqrt(pow(I, 2) + pow(J, 2) + pow(K, 2));
		if(length < 0.0001)
			{
			fprintf(stderr, "Numeric instability.  Aborting.\n\r");
			exit(3);
			}

		Matrix[column][row].normal[0] += (1/6.0)*(I/length);
		Matrix[column][row].normal[1] += (1/6.0)*(J/length);
		Matrix[column][row].normal[2] += (1/6.0)*(K/length);

		Matrix[column][row + 1].normal[0] += (1/6.0)*(I/length);
		Matrix[column][row + 1].normal[1] += (1/6.0)*(J/length);
		Matrix[column][row + 1].normal[2] += (1/6.0)*(K/length);

		Matrix[column + 1][row].normal[0] += (1/6.0)*(I/length);
		Matrix[column + 1][row].normal[1] += (1/6.0)*(J/length);
		Matrix[column + 1][row].normal[2] += (1/6.0)*(K/length);

		/* Perform same actions on lower triangle */

                V1[0] = column;
                V1[1] = Matrix[column][row + 1].value;
                V1[2] = row + 1;

                V2[0] = column + 1;
                V2[1] = Matrix[column+1][row+1].value;
                V2[2] = row + 1;

                V3[0] = column + 1;
                V3[1] = Matrix[column + 1][row].value;
                V3[2] = row;

                /* MUST use order of leg1 X leg2 */

                leg1[0] = V2[0] - V1[0];
                leg1[1] = V2[1] - V1[1];
                leg1[2] = V2[2] - V1[2];

                leg2[0] = V3[0] - V1[0];
                leg2[1] = V3[1] - V1[1];
                leg2[2] = V3[2] - V1[2];

                A = leg1[0];
                B = leg1[1];
                C = leg1[2];

                D = leg2[0];
                E = leg2[1];
                F = leg2[2];

                I = F*B-E*C;
                J = C*D-F*A;
                K = A*E-D*B;

                /* I, J, and K are components of normal vector */
                /* now to normalize */

                length = sqrt(pow(I, 2) + pow(J, 2) + pow(K, 2));
		if(length < 0.0001)
			{
			fprintf(stderr, "Numeric instability.  Aborting.\n\r");
			exit(3);
			}

                Matrix[column][row + 1].normal[0] += (1/6.0)*(I/length);
                Matrix[column][row + 1].normal[1] += (1/6.0)*(J/length);
                Matrix[column][row + 1].normal[2] += (1/6.0)*(K/length);

                Matrix[column + 1][row + 1].normal[0] += (1/6.0)*(I/length);
                Matrix[column + 1][row + 1].normal[1] += (1/6.0)*(J/length);
                Matrix[column + 1][row + 1].normal[2] += (1/6.0)*(K/length);

                Matrix[column + 1][row].normal[0] += (1/6.0)*(I/length);
                Matrix[column + 1][row].normal[1] += (1/6.0)*(J/length);
                Matrix[column + 1][row].normal[2] += (1/6.0)*(K/length);

		}
	}

/* Now write the entire thing out in a POV format */

   output = fopen("waveform.inc", "w+t");
   if(!output)
	{
	fprintf(stderr, "Critical error!  Cannot modify file waveform.inc\n\r");
	return 2;
	}
   fprintf(output, "/* Time-dependent wave mesh file for POVRAY */\n");
   fprintf(output, "/* Generated by Waveform 1.0 */\n");
   fprintf(output, "#declare WaveSurface = mesh {\n");

   for(row = 0; row < HEIGHT-1; row++)
	{
	for(column = 0; column < WIDTH-1; column++)
		{
		fprintf(output, "   smooth_triangle {\n");
		normalize(Matrix[column][row].normal);
		fprintf(output, "\t<%d, %d, %d>, <%1.8f, %1.8f, %1.8f>\n",
			column, Matrix[column][row].value, row,
			Matrix[column][row].normal[0],
			Matrix[column][row].normal[1],
			Matrix[column][row].normal[2]);
		normalize(Matrix[column][row+1].normal);
		fprintf(output, "\t<%d, %d, %d>, <%1.8f, %1.8f, %1.8f>\n",
			column, Matrix[column][row+1].value, row + 1,
			Matrix[column][row+1].normal[0],
			Matrix[column][row+1].normal[1],
			Matrix[column][row+1].normal[2]);
		normalize(Matrix[column+1][row].normal);
		fprintf(output, "\t<%d, %d, %d>, <%1.8f, %1.8f, %1.8f> }\n",
			column+1, Matrix[column+1][row].value, row,
			Matrix[column+1][row].normal[0],
			Matrix[column+1][row].normal[1],
			Matrix[column+1][row].normal[2]);
		fprintf(output, "   smooth_triangle {\n");
		fprintf(output, "\t<%d, %d, %d>, <%1.8f, %1.8f, %1.8f>\n",
			column, Matrix[column][row+1].value, row+1,
			Matrix[column][row+1].normal[0],
			Matrix[column][row+1].normal[1],
			Matrix[column][row+1].normal[2]);
		normalize(Matrix[column+1][row+1].normal);
		fprintf(output, "\t<%d, %d, %d>, <%1.8f, %1.8f, %1.8f>\n",
			column+1, Matrix[column+1][row+1].value, row + 1,
			Matrix[column+1][row+1].normal[0],
			Matrix[column+1][row+1].normal[1],
			Matrix[column+1][row+1].normal[2]);
		fprintf(output, "\t<%d, %d, %d>, <%1.8f, %1.8f, %1.8f> }\n",
			column+1, Matrix[column+1][row].value, row,
			Matrix[column+1][row].normal[0],
			Matrix[column+1][row].normal[1],
			Matrix[column+1][row].normal[2]);
		}
	}
	fprintf(output, "   translate <%1.8f, 0, %1.8f> /* to center */\n",
		-WIDTH/2.0, -HEIGHT/2.0);
	fprintf(output, "   scale <%1.8f, %1.8f, %1.8f> /* To fit in unit box */\n",
		2/(WIDTH*1.0), 1/32000.0, 2/(HEIGHT*1.0));
	fprintf(output, "}  /* end WaveSurface mesh object */\n");
	fclose(output);
}
