#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>

// Include Povray "vector.h",after sufficient defines.
#define DBL double
#define X 0
#define Y 1
#define Z 2
typedef double v3[3];
#define VECTOR v3
#include "vector.h"
DBL VTemp;

class Vector {
public:
    Vector() { v[0]=v[1]=v[2]=0; }
    Vector(double x, double y, double z) { v[0]=x;v[1]=y;v[2]=z; }

    double x() const { return v[X]; }
    double y() const { return v[Y]; }
    double z() const { return v[Z]; }
    double& x() { return v[X]; }
    double& y() { return v[Y]; }
    double& z() { return v[Z]; }

    Vector operator +(const Vector& o) const { Vector r; VAdd(r.v,v,o.v); return r; }
    const Vector& operator +=(const Vector& o) { VAddEq(v,o.v); return *this; }
    Vector operator -(const Vector& o) const { Vector r; VSub(r.v,v,o.v); return r; }
    const Vector& operator -=(const Vector& o) { VSubEq(v,o.v); return *this; }
    Vector cross(const Vector& o) const { Vector r; VCross(r.v,v,o.v); return r; }
    double dot(const Vector& o) const { double r; VDot(r,v,o.v); return r; }
    double length() const { double r; VLength(r,v); return r; }
    bool isZero() const { return !v[0]&&!v[1]&&!v[2]; };
    Vector operator *(double k) const { Vector r; VScale(r.v,v,k); return r; }
    Vector operator /(double k) const { Vector r; VScale(r.v,v,1/k); return r; }
    const Vector& operator *=(double k) { VScaleEq(v,k); return *this; }

    void rotateY(double r)
    {
	// WHICH DIRECTION? + or -?
	double x = v[X];
	double z = v[Z];
	double a = atan2(z,x);
	double l = sqrt(x*x+z*z);
	v[X] = cos(a+r)*l;
	v[Z] = sin(a+r)*l;
    }

    void rotateZ(double r)
    {
	// WHICH DIRECTION? + or -?
	double x = v[X];
	double y = v[Y];
	double a = atan2(y,x);
	double l = sqrt(x*x+y*y);
	v[X] = cos(a+r)*l;
	v[Y] = sin(a+r)*l;
    }

private:
    VECTOR v;
};
typedef Vector Point;

// Control variables - see also pingpong.inc
//
// Official measures taken from Table Tennis FAQ (in meters)
static const double hw=1.525/2; // half width of table
static const double th=0.76; // height of table (floor is at -th)
static const double l=2.74; // length of table
static const double net=0.1525; // height of net
static const double bouncefficiency=0.770; // friction
// Universal
static const double grav=-9.8; // grav const
// Whimsical
static const int hvolleys=20; // half-volleys - each robot does this many
static const double baseh=0.50; // height of robot base.
static const double basex=0.50; // backward offset of pivot point
static const double robot_arm[3]={1.6,0.9,0.5}; // arm lengths
static const Point robotbase[2] = { Point(0,baseh-th,0-0.5),
				    Point(0,baseh-th,l+0.5) };
static const Point robothome[2] = { Point(0,0.2,0+0.1),
				    Point(0,0.2,l-0.1) };
static const double mind=-.1; // minimum hit-back point
static const double maxd=l/2.1; // maximum hit-back point
static const double minhitheight=0.1; // leave room for bat
static const double hitspace=0.1; // leave room for bat
static const Point initball(0,0.5,0);

#define withintable(p) (p.x()>-hw && p.x()<+hw && p.z()>0 && p.z()<l)
#define withintablerange(p,r) (p.x()>-hw-r && p.x()<+hw+r && p.z()>0-r && p.z()<l+r)

Point ballPos(double t,double bounce_time,Point ball,Vector ballv)
{
    Point p;
    if ( t > bounce_time ) {
	p = ballPos(bounce_time,bounce_time,ball,ballv);
	Vector bouncev = ballv;
	bouncev.y() = -(ballv.y()+grav*bounce_time)*bouncefficiency;
	return ballPos(t-bounce_time,t,p,bouncev);
    } else {
	p = ball+ballv*t;
	p.y()+=0.5*grav*t*t;
    }
    return p;
}


Vector findLegalHit(Point ball, bool reverse, int maxtries, double& bounce_time, bool to_initball)
{
    const int debug=0;

    if ( reverse )
	ball.z()=l-ball.z();

    Vector ballv;

    bounce_time=0;

    while (!bounce_time && --maxtries) {
	// make up a random hit, then see what would happen

	// Random new direction for ball
	if ( to_initball ) {
	    // Aim towards reversed initball
	    double m=drand48();
	    double x=(initball.x()-ball.x())*m;
	    double z=((l-initball.z())-ball.z())*m;
	    ballv = Vector(x,drand48()-.5,z);
	    ballv *= 8; // give it more room to find a solution
	} else {
	    double x = drand48()*2-1;
	    if ( drand48() > 0.5 )
		x = -x;
	    ballv = Vector(x,drand48()*5-1.5,drand48()*5);
	}

	if ( debug) fprintf(stderr,"Try hit from (%g,%g,%g), dir (%g,%g,%g)\n",
		ball.x(),ball.y(),ball.z(),
		ballv.x(),ballv.y(),ballv.z());

	// When (in the future) will it pass through the plane of the table?
	const double a = 0.5*grav;
	const double b = ballv.y();
	const double c = ball.y();
	const double det = b*b-4*a*c;
	if ( det >= 0 ) {
	    const double sqdet = sqrt(det);
	    for ( int sol=-1; sol<=1 && !bounce_time; sol+=2) {
		const double t = (-b + sol*sqdet)/(2*a);
		if ( t > 0 ) {
		    Point p = ball+ballv*t; // actually only x,z valid
		    double vy = ballv.y()+t*grav;
		    if (withintable(p) && vy<0) {
			// Will hit table.  But will it clear the net?
			// When will it pass the net?
			double n=(l/2-ball.z())/ballv.z();
			double ny=ball.y()+ballv.y()*n+0.5*grav*n*n;
			if ( ny > net ) {
			    bounce_time=t;
			    if ( to_initball ) {
				// When will it be level with initball?
				Point rib=initball;
				rib.z()=(l-rib.z());
				double n=(rib.z()-ball.z())/ballv.z();
				Point p = ballPos(n,t,ball,ballv);
				double miss=(rib-p).length();
				if ( miss > 0.001 ) {
				    bounce_time=0;
				    if ( debug) fprintf(stderr,"  miss initball by %g\n",miss);
				} else {
				    if ( debug) fprintf(stderr,"  succeeds\n");
				}
			    } else {
				if ( debug) fprintf(stderr,"  succeeds\n");
			    }
			} else {
			    if ( debug) fprintf(stderr,"  fails for %d - hits net at height %g\n",sol,ny);
			}
		    } else {
			if ( debug) fprintf(stderr,"  fails for %d - off table at (%g,%g,%g) v(%g,%g,%g)\n",
			    sol, p.x(),0.0,p.z(),ballv.x(),vy,ballv.y());
		    }
		} else {
		    if ( debug) fprintf(stderr,"  fails for %d - hit time <= 0 (%g)\n",sol,t);
		}
	    }
	} else {
	    if ( debug) fprintf(stderr,"  fails - det < 0\n");
	}
    }
//fprintf(stderr,"%d\n",maxtries);
    if ( !bounce_time )
	ballv=Vector();
    if ( reverse )
	ballv.z() = -ballv.z();

    return ballv;
}

struct RobotRec {
    Point ball;
    Vector ballv;
    double ptime;
    double time;
    double t;
    double var[5];
};

main(int argc, char* argv[])
{
    if ( argc != 5 ) {
	fprintf(stderr,"Usage: calc ball static realtime robots\n");
	exit(1);
    }

    // All calculation is for 0-radius ball
    // We compenstate by making the net shorter by ball-radius
    //  and the table lower by ball-radius.

    // At time=0:
    //
    //  - bat0 is just about to decided (instantly) how to hit the ball,
    //		then do so.
    //  - the ball will then instantly travel to somewhere on the other
    //		side of the table sufficient to be hit by bat1.
    //  - bat1 will move instantly to the location of the ball.
    //
    // Thus the cycle repeats, with bat0 and bat1 in opposite situations.
    //
    const int MAXREC=2048;
    RobotRec robotrecs[2][2][MAXREC];

    RobotRec robotrec[2][MAXREC];

    for ( int attempt=1; attempt<1000000; attempt++ ) {
	FILE* f_ball     ;
	FILE* f_static   ;
	FILE* f_realtime ;
	FILE* f_robots   ;
	srand48(1234+13+attempt);
	//srand48(12345+attempt);

	f_ball     = fopen(argv[1],"w");
	f_static   = fopen(argv[2],"w");
	f_realtime = fopen(argv[3],"w");
	f_robots   = fopen(argv[4],"w");

	if ( !f_ball || !f_static || !f_realtime || !f_robots ) {
	    fprintf(stderr,"I/O error");
	    exit(1);
	}

	Point ball=initball;
	double prealtime=0.0;
	double realtime=0.0;
	double bounce_time;
	Vector initballv=findLegalHit(ball,false,0,bounce_time,false);
	Vector ballv=initballv;

	fprintf(f_static,"  sphere { <%g,%g,%g>,0.02 pigment { color Blue } }\n",
	    ball.x(),ball.y(),ball.z());

	const int nvolley=hvolleys*2;
	int volley;
	for (volley=0; volley<nvolley; volley++) {
	    bool far_hitter=!(volley&1);
	    bool last_volley_but1=volley==nvolley-2;
	    bool last_volley=volley==nvolley-1;

	    // Move hitter to somewhere on path of ball
	    double t=0;

	    while (1) {
		Point p;
		int tr=0;
		do {
		    // pick a position
		    double hitz=mind+drand48()*(maxd-mind);
		    if ( last_volley )
			hitz=initball.z();
		    if ( far_hitter )
			hitz=l-hitz;
		    // When will ball be there?
		    t=(hitz-ball.z())/ballv.z();
		    // Where will ball be?
		    p = ballPos(t,bounce_time,ball,ballv);
		    if ( tr++ > 500 )
			goto failure;
		} while ( p.y() < minhitheight && withintablerange(p,hitspace) );

		double next_bounce_time=1.0;
		Vector v = last_volley
		    ? initballv
		    : findLegalHit(p,far_hitter,last_volley_but1 ? 100000 : 150000,next_bounce_time,last_volley_but1);
		if ( v.isZero() )
		    goto failure;
		if ( next_bounce_time ) {
		    for ( double tt = 0; tt<t; tt+=0.004 ) {
			// Where will ball be?
			Point q = ballPos(tt,bounce_time,ball,ballv);
			fprintf(f_static,"sphere { <%g,%g,%g>,0.006 pigment { color Yellow } }\n",
				q.x(),q.y(),q.z());
		    }

		    Point q1 = ball;
		    Point q2 = ballPos(bounce_time,bounce_time,ball,ballv);
		    Vector bouncev = ballv;
		    bouncev.y() = -(ballv.y()+grav*bounce_time)*bouncefficiency;
		    
		    fprintf(f_ball," #if (realtime >= %g & realtime < %g)\n",realtime,realtime+t);
		    fprintf(f_ball,"  #declare T = realtime - %g\n",realtime);
		    if ( t > bounce_time ) {
			fprintf(f_ball,"  #if (T < %g)\n",bounce_time);
			fprintf(f_ball,"   #declare S=<%g,%g,%g>\n",q1.x(),q1.y(),q1.z());
			fprintf(f_ball,"   #declare U=<%g,%g,%g>\n",ballv.x(),ballv.y(),ballv.z());
			fprintf(f_ball,"  #else\n");
			fprintf(f_ball,"   #declare S=<%g,%g,%g>\n",q2.x(),q2.y(),q2.z());
			fprintf(f_ball,"   #declare U=<%g,%g,%g>\n",bouncev.x(),bouncev.y(),bouncev.z());
			fprintf(f_ball,"   #declare T = T - %g\n",bounce_time);
			fprintf(f_ball,"  #end\n");
		    } else {
			fprintf(f_ball,"  #declare S=<%g,%g,%g>\n",q1.x(),q1.y(),q1.z());
			fprintf(f_ball,"  #declare U=<%g,%g,%g>\n",ballv.x(),ballv.y(),ballv.z());
		    }
		    fprintf(f_ball,"  #declare BallPos = S+U*T+y*0.5*%g*T*T\n",grav);
		    fprintf(f_ball," #end\n");

		    bounce_time = next_bounce_time;
		    ball = p;
		    ballv = v;

		    fprintf(f_static,"  sphere { <%g,%g,%g>,0.01 pigment { color Red } }\n"
			   "   cone { <%g,%g,%g> 0.01 <%g,%g,%g> 0 pigment { color White } }\n",
			p.x(),p.y(),p.z(),
			p.x(),p.y(),p.z(),
			p.x()+v.x()/30,
			p.y()+v.y()/30,
			p.z()+v.z()/30);
		    
		    break;
		}
	    }

	    RobotRec& rr = robotrec[far_hitter][volley/2];
	    rr.ball = ball;
	    rr.ballv = ballv;
	    rr.ptime = prealtime;
	    rr.time = realtime+t;
	    rr.t = t;

	    prealtime=realtime;
	    realtime+=t;
	}
	fprintf(f_realtime,"#declare maxrealtime = %g\n",realtime);
	fprintf(f_realtime,"#declare realtime = clock*maxrealtime\n");

        robotrecs[0] = robotrec; 

	{
	// Output the robots...

	fprintf(f_robots,"#declare BaseY=%g\n\n",baseh);
	fprintf(f_robots,"#declare BaseX=%g\n\n",basex);
	for (int i=0; i<3; i++)
	    fprintf(f_robots,"#declare Arm%d=%g\n\n",i+1,robot_arm[i]);

	int varset=0;

	// Add embellishment positions
	for (int w=0; w<2; w++) {
	    int j=0;
	    Point home = robothome[w];
	    for (int i=0; i<hvolleys; i++) {
		RobotRec& ra = robotrecs[varset][w][i];
		RobotRec& rb = robotrecs[varset][w][(i+hvolleys-1)%hvolleys];
		RobotRec& ro = robotrecs[1-varset][w][j];
		RobotRec& rn = robotrecs[1-varset][w][j+1];

		ro.ptime = ra.ptime;
		ro.time = (ra.ptime+ra.time)/2;
		ro.t = ro.time - ro.ptime;

		if ( ro.time - ro.ptime < 0.5 ) {
		    // Not much time - just average
		    ro.ball = (ra.ball+rb.ball)/2;
		    ro.ballv = (ra.ballv+rb.ballv)/2;
		} else {
		    // Plenty of time, back off
		    ro.ball = (ra.ball+rb.ball+home*3)/5;
		    ro.ballv = (ra.ballv*2+rb.ballv)/3;
		}

		rn = ra;
		rn.ptime = ro.time;
		rn.t = rn.time - rn.ptime;

		j+=2;
	    }
	}
	int last = hvolleys * 2;
	varset=1-varset;

	// Generate points in robot paths...
	//
	for (int w=0; w<2; w++) {
	    for (int i=0; i<last; i++) {
		RobotRec& rr = robotrecs[varset][w][i];
		Point ball = rr.ball;
		Vector ballv = rr.ballv;
		double prealtime = rr.ptime;
		double realtime = rr.time;
		double *newrobot = rr.var;

		// Move robot's bat to target point, facing
		// direction targetv at time t.
		//
		Point target = ball; 
		Vector targetv = ballv; 

		Vector d = target-robotbase[w];
		Vector v = targetv;
		newrobot[0] = atan(d.x()/d.z());
		double a=w ? M_PI/2 : -M_PI/2;
		d.rotateY(newrobot[0]+a); // zero z coord
		v.rotateY(newrobot[0]+a);
		Vector pz = v;
		v.z()=0;

		// Base offset
		d.x() += basex;

		Vector v1 = v/v.length();
		v1.rotateZ(M_PI/2);
		Point dp=d+v1*robot_arm[2];
		double dpl=dp.length();
		newrobot[2]=acos((dpl*dpl-robot_arm[0]*robot_arm[0]-robot_arm[1]*robot_arm[1])/
				 -(2*robot_arm[0]*robot_arm[1])); // law of cosines
		double b2=acos((robot_arm[1]*robot_arm[1]-dpl*dpl-robot_arm[0]*robot_arm[0])/
				 -(2*dpl*robot_arm[0])); // law of cosines
		newrobot[1]=atan(dp.y()/dp.x())+b2;
		Point cp = Point(robot_arm[0],0,0);
		cp.rotateZ(newrobot[1]);
		double ld=(d-cp).length();
		newrobot[3]=acos((ld*ld-robot_arm[2]*robot_arm[2]-robot_arm[1]*robot_arm[1])/
				    -(2*robot_arm[2]*robot_arm[1])); // law of cosines
		// Can need to bend the other way - see which is closest
		Vector tst = (d-dp).cross(cp-dp);
		if ( tst.z() > 0 ) {
		    double alt = 2*M_PI-newrobot[3];
		    newrobot[3] = alt;
		}

		// Angle between pz and z is E variable
		Vector vcpz = v.cross(pz);
		newrobot[4] = asin(vcpz.length()/v.length()/pz.length());
		if ( (d-dp+vcpz).length() > (d-dp-vcpz).length() ) {
		    // Other way
		    newrobot[4] = -newrobot[4];
		}

		for (int vn=0; vn<5; vn++) {
		    if ( isnan(newrobot[vn]) ) {
			fprintf(stderr,"Value is Not a Number... Robot is Not a Contortionist (variable %c, position %d, robot %d).\n",
			    'A'+vn,i,w);
			goto failure;
		    }
		}
	    }
	}

	// Append an extra record to robot 1, to equalize total timespans
	//
	RobotRec& brr = robotrecs[varset][1][0];
	RobotRec& err = robotrecs[varset][0][last-1];
	double end0 = err.time;
	double end1 = err.time - err.t;
	int nrobotvar[2][2];
	nrobotvar[varset][0] = last;
	nrobotvar[varset][1] = last+1;
	for (int v=0; v<5; v++) {
	    robotrecs[varset][1][last].ptime = robotrecs[varset][1][last-1].time;
	    robotrecs[varset][1][last].time = err.time;
	    robotrecs[varset][1][last].t = brr.t+err.t;
	    double val =
	     (brr.var[v] * brr.t + err.var[v] * err.t) / (brr.t+err.t);
	    robotrecs[varset][1][last].var[v] = val;
	}


	// Only .var[], .time, and .ptime are relevant now...

	// Output paths...
	//
	for (int w=0; w<2; w++) {
	    int nv = nrobotvar[varset][w];
	    for (int i=0; i<nv; i++) {
		RobotRec& rr = robotrecs[varset][w][i];
		double prealtime = rr.ptime;
		double realtime = rr.time;

		double *robot=robotrecs[varset][w][(i+nv-1)%nv].var;
		double *newrobot=robotrecs[varset][w][i].var;

		fprintf(f_robots," #if (realtime >= %g & realtime < %g)\n",prealtime,realtime);
		fprintf(f_robots,"  #declare T=realtime-%g\n",prealtime);
		fprintf(f_robots,"  #declare M=%g\n", (realtime-prealtime));
		fprintf(f_robots,"  #declare R = %d\n",w);

		for ( int v=0; v<5; v++ ) {
		    double dif1 = newrobot[v]-robot[v];
		    double dif2 = newrobot[v]-robot[v]-M_PI*2;
		    double dif;
		    if ( 1/*only angles*/ ) {
			dif = fabs(dif1) < fabs(dif2) ? dif1 : dif2;
		    } else {
			dif = dif1;
		    }
		    fprintf(f_robots,"  #declare %c = %g+T*%g/M // %0.0f to %0.0fdeg\n",
			'A'+v, robot[v], dif,
			robot[v]*180/M_PI,newrobot[v]*180/M_PI);
		}
		fprintf(f_robots,"  #include \"robot.inc\"\n");
		fprintf(f_robots,"  object { Robot translate <%g,%g,%g> }\n",
		    robotbase[w]);
		fprintf(f_robots," #end\n");
	    }
	}
	}

	fclose(f_ball);
	fclose(f_static);
	fclose(f_realtime);
	fclose(f_robots);
	break;

	failure:
	    fclose(f_ball);
	    fclose(f_static);
	    fclose(f_realtime);
	    fclose(f_robots);
	    printf("Failed attempt %d after %d%%\n",attempt,volley*100/nvolley);
    }
}


