// Persistence of Vision Ray Tracer Scene Description File
// File: r4ev.pov
// Vers: 3.1
// Desc: Top level scene file for the "Robot Factory" animation.
// Date: 01/14/00
//
// COPYRIGHT: I SUBMIT TO THE STANDARD RAYTRACING COMPETITION COPYRIGHT.
// Copyright (c) Dan Lauer, 2000                               
//
// mailto:dlauer@optonline.net
//
// This is the top level scene description file for my "Robot Factory" Animation.
// It was created specifically for entry into the January 15, 2000 round of the
// Internet Ray Tracing Competition.  See http://www.irtc.org/.
//
// The code in this and the accompanying files may be used freely for non-commercial
// purposes without permission.  However, I would love to here from anyone with questions
// or comments.  Commercial uses other than those described in the STANDARD RAYTRACING
// COMPETITION COPYRIGHT require permission from the author.  Please contact me via e-mail.
//
//

#version 3.1;

//
// Film set up variables
                       
#declare LED_frame_rate = 5;
#declare cycles_per_film = 12.1;
#declare first_cycle_offset = -0.45;  //0-1 as a percent of a cycle
#declare use_full_lights = yes;
#declare ci_offset = 0;
//#declare debug_robot = true; 

#declare belt_center = 8;
#declare aisle_a = 4.5;
#declare aisle_b = 11.5;
#declare last_station = 15;
#declare station_dist = 5;
#include "colors.inc"       
#include "miscutil.inc"

// ********* Define some useful macros ***************************************
#include "MrRobot.inc"   

#declare r2d = 180/pi; //Radians to Degrees
#declare d2r = pi/180; //Degrees to Radians
                                                                              
#macro Interpolate(T,T1,T2,P1,P2)
   (P1+((T-T1)/(T2-T1))*(P2-P1))
#end                                    
#macro hypot(A,B)
  sqrt(pow(A,2)+pow(B,2))
#end        

#macro array_transform(index)
    translate <robot_info[index][e_pre_x_pos], robot_info[index][e_pre_y_pos], robot_info[index][e_pre_z_pos]>
    rotate    <robot_info[index][e_x_rot], robot_info[index][e_y_rot], robot_info[index][e_z_rot]>
    translate <robot_info[index][e_x_pos], robot_info[index][e_y_pos], robot_info[index][e_z_pos]>
#end

#macro compute_arms(index, rx, ry, rz, lx, ly, lz)
  #if(abs(rx) < 5)
    #local sw_d  = hypot(rz, rx);
    #local sw_a  = asin(rx/sw_d);
    #local ua_a  = sw_a - acos( sw_d/3);  //3 = 2*upper_arm_length
    #local la_a  = 2 * (sw_a - ua_a);
    #declare robot_info[index][e_y_r_shoulder] = ua_a * r2d;                             
    #declare robot_info[index][e_x_r_elbow] = -atan2(ry, 1.5) * r2d;
    #declare robot_info[index][e_y_r_elbow] = la_a * r2d;                             
    #declare robot_info[index][e_len_r_arm] = hypot(ry, 1.5);
    #declare robot_info[index][e_x_r_wrist] = atan2(ry,1.5) * r2d;
    #declare robot_info[index][e_y_r_wrist] = 90 - ((la_a+ua_a) * r2d);                             
  #end
  #if(abs(lx) < 5)
    #local sw_d  = hypot(lz, lx);
    #local sw_a  = asin(lx/sw_d);
    #local ua_a  = sw_a - acos( sw_d/3);  //3 = 2*upper_arm_length
    #local la_a  = 2 * (sw_a - ua_a);
    #declare robot_info[index][e_y_l_shoulder] = ua_a * r2d;                             
    #declare robot_info[index][e_x_l_elbow] = atan2(ly, 1.5) * r2d;                             
    #declare robot_info[index][e_y_l_elbow] = la_a * r2d;                             
    #declare robot_info[index][e_len_l_arm] = hypot(ly, 1.5);
    #declare robot_info[index][e_x_l_wrist] = -atan2(ly,1.5) * r2d;
    #declare robot_info[index][e_y_l_wrist] = 90 - ((la_a+ua_a) * r2d);                             
  #end
#end 

//*********   Declare the clock variables  **********************************************
#declare A_clock = mod(clock * cycles_per_film - first_cycle_offset+1, 1);
#declare B_clock = mod(clock * cycles_per_film - first_cycle_offset + 0.5, 1);
#declare temp_Clock = mod(A_clock * 2,1);     
#if (A_clock >= 1) 
  #declare A_clock = 0; 
#end     
#if (B_clock >= 1) 
  #declare B_clock = 0; 
#end     

#declare A_cycle_count = floor(clock * cycles_per_film - first_cycle_offset);
#declare B_cycle_count = floor(clock * cycles_per_film + 0.5 - first_cycle_offset);
#declare belt_cycle_count = A_cycle_count + B_cycle_count; //floor((clock-first_cycle_offset) * cycles_per_film * 2);
#switch (temp_Clock)
  #range (0.0, 0.10)
    #declare belt_clock = 0;
    #break
  #range (0.10, 0.90)
    #declare belt_clock = Interpolate(temp_Clock, 0.10, 0.90, 0.0, 1.0);
    #break
  #range (0.90, 1.01)
    #declare belt_clock = 1;
    #break;
#end

//Include this here because it uses the clock variables.
#include "Factory.pov"          

#render concat("Clock is    ", str(clock, 8, 6), "\n")
#render concat("A Clock is  ", str(A_clock, 8, 6), " on cycle ",str(A_cycle_count, 2, 1), "\n")
#render concat("B Clock is  ", str(B_clock, 8, 6), " on cycle ",str(B_cycle_count, 2, 1), "\n")
#render concat("Belt clock= ", str(belt_clock, 8, 6), " on cycle ",str(belt_cycle_count, 2, 1), "\n")

// Robot_info array enumeration constants
#declare e_exist        = 0;
#declare e_x_pos        = 1;
#declare e_y_pos        = 2;
#declare e_z_pos        = 3;
#declare e_x_rot        = 4;
#declare e_y_rot        = 5;
#declare e_z_rot        = 6;
#declare e_power        = 7;
#declare e_z_head       = 8;
#declare e_z_eye        = 9;
#declare e_leg_length   = 10;
#declare e_y_leg_angle  = 11;
#declare e_z_leg_rot    = 12;
#declare e_ankle_angle  = 13;
#declare e_x_r_shoulder = 14;
#declare e_y_r_shoulder = 15;
#declare e_x_r_elbow    = 16;
#declare e_y_r_elbow    = 17;
#declare e_len_r_arm    = 18;
#declare e_x_r_wrist    = 19;
#declare e_y_r_wrist    = 20;
#declare e_z_r_wrist    = 21;
#declare e_r_thumb      = 22;
#declare e_x_l_shoulder = 23;
#declare e_y_l_shoulder = 24;
#declare e_x_l_elbow    = 25;
#declare e_y_l_elbow    = 26;
#declare e_len_l_arm    = 27;
#declare e_x_l_wrist    = 28;
#declare e_y_l_wrist    = 29;
#declare e_z_l_wrist    = 30;
#declare e_l_thumb      = 31;
#declare e_pre_x_pos    = 32;
#declare e_pre_y_pos    = 33;
#declare e_pre_z_pos    = 34;
#declare e_color_index  = 35;

#declare robot_cnt = 22;
#declare param_cnt = 36;

#declare robot_info = array[robot_cnt][param_cnt]                         
#declare r_pos = array[robot_cnt]                             
                             
// Initialize the Robot Array
#declare i = 0;
#while (i < robot_cnt)
  #declare j = 0;
  #while (j < param_cnt)
    #declare robot_info[i][j] = 0;
    #declare j = j + 1;
  #end
  #declare robot_info[i][e_leg_length] = 2.8;
  #declare robot_info[i][e_len_r_arm] = 1.5;
  #declare robot_info[i][e_len_l_arm] = 1.5;
  #declare i = i + 1;
#end
// Finish initialization                                

//robot 1  this is the first appearence of the torso
#declare i = 1;
#declare robot_info[i][e_exist] = true;
#declare pivot_point_y = last_station + (5 * station_dist);
#declare robot_info[i][e_y_rot] = 90;
#declare robot_info[i][e_color_index] = mod(ci_offset+A_cycle_count+12,12);

#switch(A_clock)
  #range (0, 0.499) //belt clock doesn't quite go to 1 ** come up elevator
    #declare robot_info[i][e_z_rot] = -90;
    #declare robot_info[i][e_x_pos] = (aisle_a - 3.3);
    #declare robot_info[i][e_y_pos] = pivot_point_y;
    #declare robot_info[i][e_z_pos] = Interpolate(belt_clock, 0, 1, 0.5, 3.5);
    #break
  #range (0.499, 0.525)   // get picked up
    #declare robot_info[i][e_z_rot] = -90;
    #declare robot_info[i][e_x_pos] = Interpolate(A_clock, 0.5, 0.525, (aisle_a - 3.3) , (aisle_a - 3) );
    #declare robot_info[i][e_y_pos] = pivot_point_y;
    #declare robot_info[i][e_z_pos] = Interpolate(A_clock, 0.5, 0.525, 3.5, 3.7);
    #break
  #range (0.525, 0.975)  // spin around
    #declare robot_info[i][e_z_rot] = Interpolate(A_clock, 0.525, 0.975, 270, 90);
    #declare robot_info[i][e_x_pos] = aisle_a + (3*sin(Interpolate(A_clock, 0.525, 0.975, (-pi/2), (pi/2))));
    #declare robot_info[i][e_y_pos] = pivot_point_y + (3*sin(Interpolate(A_clock, 0.525, 0.975, 0, pi)));
     #declare robot_info[i][e_z_pos] = 3.7;
    #break
  #range (0.975, 1.01)   // get put down
    #declare robot_info[i][e_z_rot] = 90;
    #declare robot_info[i][e_x_pos] = Interpolate(A_clock, 0.975, 1.0, (aisle_a + 3) , belt_center);
    #declare robot_info[i][e_y_pos] = pivot_point_y;
    #declare robot_info[i][e_z_pos] = Interpolate(A_clock, 0.975, 1.0, 3.7, 3.5);
    #break                    
#end
//This is the elevator for the torso
#switch(A_clock)
  #range (0, 0.499999)
    object {
      Elevator( 2.8, 3.4, 3.0, belt_clock )
      translate <1.4  ,pivot_point_y, 0>
    }
    #break
  #range (0.499999, 1.0)
    object {
      Elevator( 2.8, 3.4, 3.0, 1-belt_clock )
      translate <1.4 ,pivot_point_y, 0>
    }
    #break
#end
       
//robot 2  this is the first appearence of the leg 
// There is some special handling here for the hip joint
#declare i = 2;
#declare robot_info[i][e_exist] = true;
#declare pivot_point_y = last_station + (4 * station_dist) - 1.2 - 1.25;
#declare pivot_point_x = aisle_b - 0.5;
#declare robot_info[i][e_y_rot] = 90;
#declare robot_info[i][e_pre_z_pos] = 1.2 + 1.25;
#declare robot_info[i][e_pre_y_pos] = 0;
#declare robot_info[i][e_y_leg_angle] = 6.2;
#declare robot_info[i][e_ankle_angle] = -6.2;
#declare robot_info[i][e_color_index] = mod(ci_offset+B_cycle_count+11,12);
#switch(B_clock)
  #range (0, 0.499) //belt clock doesn't quite go to 1
    #declare robot_info[i][e_z_rot] = 270;   
    #declare robot_info[i][e_x_pos] = (pivot_point_x + 3.0 );
    #declare robot_info[i][e_y_pos] = pivot_point_y;
    #declare robot_info[i][e_z_pos] = Interpolate(belt_clock, 0, 1, 0.5, 3.5);
    //HipJoint stuff
    #declare robot_info[i][e_pre_y_pos] = robot_info[i][e_pre_y_pos] + 0.3;
      object {
        HipJoint
        array_transform(i)
      }
    #declare robot_info[i][e_pre_y_pos] = robot_info[i][e_pre_y_pos] - 0.3;
    #break
  #range (0.499, 0.53)
    #declare robot_info[i][e_z_rot] = 270;
    #declare robot_info[i][e_x_pos] = (pivot_point_x + 3.0);
    #declare robot_info[i][e_y_pos] = pivot_point_y;
    #declare robot_info[i][e_z_pos] = Interpolate(B_clock, 0.5, 0.53, 3.5, 3.6);
    //HipJoint stuff
    #declare robot_info[i][e_pre_y_pos] = robot_info[i][e_pre_y_pos] + Interpolate(B_clock, 0.53, 0.73, 0.3, 0.2);
      object {
        HipJoint
        array_transform(i)
      }
    #declare robot_info[i][e_pre_y_pos] = robot_info[i][e_pre_y_pos] - Interpolate(B_clock, 0.53, 0.73, 0.3, 0.2);
    #break
  #range (0.53, 0.93)
    #declare robot_info[i][e_z_rot] = Interpolate(B_clock, 0.53, 0.93, 270, 90);
    #declare robot_info[i][e_x_pos] = pivot_point_x - 3.0*sin(Interpolate(B_clock, 0.53, 0.93, (-pi/2), (pi/2)));
    #declare robot_info[i][e_y_pos] = pivot_point_y - 3.0*sin(Interpolate(B_clock, 0.53, 0.93, 0, pi));
    #declare robot_info[i][e_z_pos] = 3.6;
    //HipJoint stuff
    #declare robot_info[i][e_pre_y_pos] = robot_info[i][e_pre_y_pos] + Interpolate(B_clock, 0.53, 0.93, 0.2, -1.7);
      object {
        HipJoint
        array_transform(i)
      }
    #declare robot_info[i][e_pre_y_pos] = robot_info[i][e_pre_y_pos] - Interpolate(B_clock, 0.53, 0.93, 0.2, -1.7);
    #break
  #range (0.93, 0.95)
    #declare robot_info[i][e_z_rot] = 90;
    #declare robot_info[i][e_x_pos] = (pivot_point_x - 3.0 );
    #declare robot_info[i][e_y_pos] = pivot_point_y;
    #declare robot_info[i][e_z_pos] = Interpolate(B_clock, 0.93, 0.95, 3.6, 3.5);
    //HipJoint stuff
    #declare robot_info[i][e_pre_y_pos] = robot_info[i][e_pre_y_pos] - 1.7;
      object {
        HipJoint
        array_transform(i)
      }
    #declare robot_info[i][e_pre_y_pos] = robot_info[i][e_pre_y_pos] + 1.7;
    #break                    
  #range (0.95, 1.01)
    #declare robot_info[i][e_z_rot] = 90;
    #declare robot_info[i][e_x_pos] = (pivot_point_x - 3.0 );
    #declare robot_info[i][e_y_pos] = pivot_point_y;
    #declare robot_info[i][e_z_pos] = 3.5;
    //HipJoint stuff
    #declare robot_info[i][e_pre_y_pos] = robot_info[i][e_pre_y_pos] + Interpolate(B_clock, 0.95, 1.0, -1.7, 0.0);
      object {
        HipJoint
        array_transform(i)
      }
    #declare robot_info[i][e_pre_y_pos] = robot_info[i][e_pre_y_pos] - Interpolate(B_clock, 0.95, 1.0, -1.7, 0.0);
    #break                    
#end
//This is the elevator for the leg
#switch(B_clock)
  #range (0, 0.499999)
    object {
      Elevator( 3.2, 3.8, 3.2, belt_clock )
      translate <aisle_b+2.9  ,pivot_point_y, 0>
    }
    #break
  #range (0.499999, 1.0)
    object {
      Elevator( 3.2, 3.8, 3.2, 1 - belt_clock )
      translate <aisle_b+2.9,pivot_point_y, 0>
    }
    #break
#end

//robot 3  this is the first appearence of the right arm
#declare i = 3;
#declare robot_info[i][e_exist] = true;
#declare pivot_point_y = last_station + (3 * station_dist) - 0.5;
#declare pivot_point_x = aisle_a - 0.5;
#declare robot_info[i][e_y_rot] = 90;
#declare robot_info[i][e_pre_z_pos] = 0.5;
#declare robot_info[i][e_pre_y_pos] = -1.25;
#declare robot_info[i][e_color_index] = mod(ci_offset+A_cycle_count+11,12);
#switch(A_clock)
  #range (0, 0.499) //belt clock doesn't quite go to 1
    #declare robot_info[i][e_z_rot] = 270;   
    #declare robot_info[i][e_x_pos] = (pivot_point_x - 4 + 1.25 );
    #declare robot_info[i][e_y_pos] = pivot_point_y;
    #declare robot_info[i][e_z_pos] = Interpolate(belt_clock, 0, 1, 0.5, 3.5);
    #break
  #range (0.499, 0.58)
    #declare robot_info[i][e_z_rot] = 270;
    #declare robot_info[i][e_x_pos] = Interpolate(A_clock, 0.5, 0.58, (pivot_point_x - 4 + 1.25) , (pivot_point_x - 3 + 1.25) );
    #declare robot_info[i][e_y_pos] = pivot_point_y;
    #declare robot_info[i][e_z_pos] = Interpolate(A_clock, 0.5, 0.58, 3.5, 3.7);
    #break
  #range (0.58, 0.95)
    #declare robot_info[i][e_z_rot] = Interpolate(A_clock, 0.58, 0.95, 270, 90);
    #declare robot_info[i][e_x_pos] = pivot_point_x + ((3-1.25)*sin(Interpolate(A_clock, 0.58, 0.95, (-pi/2), (pi/2))));
    #declare robot_info[i][e_y_pos] = pivot_point_y + ((3-1.25)*sin(Interpolate(A_clock, 0.58, 0.95, 0, pi)));
    #declare robot_info[i][e_z_pos] = Interpolate(A_clock, 0.58, 0.95, 3.7, 3.5);
    #break
  #range (0.95, 1.01)
    #declare robot_info[i][e_z_rot] = 90;
    #declare robot_info[i][e_x_pos] = Interpolate(A_clock, 0.95, 1.0, (pivot_point_x + 3 - 1.25 ) , belt_center - 1.25);
    #declare robot_info[i][e_y_pos] = pivot_point_y;
    #declare robot_info[i][e_z_pos] = 3.5;
    #break                    
#end
//This is the elevator for the right arm
#switch(A_clock)
  #range (0, 0.499999)
    object {
      Elevator( 2.0, 4.4, 3.35, belt_clock )
      translate <1.0  ,pivot_point_y+0.4, 0>
    }
    #break
  #range (0.499999, 1.0)
    object {
      Elevator( 2.0, 4.4, 3.35, 1-belt_clock )
      translate <1.0 ,pivot_point_y+0.4, 0>
    }
    #break
#end
       

//robot 4  this is the first appearence of the left arm
#declare i = 4;
#declare robot_info[i][e_exist] = true;
#declare pivot_point_y = last_station + (2 * station_dist) - 0.5;
#declare pivot_point_x = aisle_b + 0.5;
#declare robot_info[i][e_y_rot] = 90;
#declare robot_info[i][e_pre_z_pos] = 0.5;
#declare robot_info[i][e_pre_y_pos] = 1.25;
#declare robot_info[i][e_color_index] = mod(ci_offset+B_cycle_count+10,12);
#switch(B_clock)
  #range (0, 0.49999) //belt clock doesn't quite go to 1
    #declare robot_info[i][e_z_rot] = 270;   
    #declare robot_info[i][e_x_pos] = (pivot_point_x + 4 - 1.25 );
    #declare robot_info[i][e_y_pos] = pivot_point_y;
    #declare robot_info[i][e_z_pos] = Interpolate(belt_clock, 0, 1, 0.5, 3.5);
    #break
  #range (0.49999, 0.55)
    #declare robot_info[i][e_z_rot] = 270;
    #declare robot_info[i][e_x_pos] = Interpolate(B_clock, 0.5, 0.55, (pivot_point_x + 4 - 1.25) , (pivot_point_x + 3 - 1.25) );
    #declare robot_info[i][e_y_pos] = pivot_point_y;
    #declare robot_info[i][e_z_pos] = Interpolate(B_clock, 0.5, 0.55, 3.5, 3.7);
    #break
  #range (0.55, 0.95)
    #declare robot_info[i][e_z_rot] = Interpolate(B_clock, 0.55, 0.95, 270, 90);
    #declare robot_info[i][e_x_pos] = pivot_point_x - ((3-1.25)*sin(Interpolate(B_clock, 0.55, 0.95, (-pi/2), (pi/2))));
    #declare robot_info[i][e_y_pos] = pivot_point_y - ((3-1.25)*sin(Interpolate(B_clock, 0.55, 0.95, 0, pi)));
    #declare robot_info[i][e_z_pos] = Interpolate(B_clock, 0.55, 0.95, 3.7, 3.5);
    #break
  #range (0.95, 1.01)
    #declare robot_info[i][e_z_rot] = 90;
    #declare robot_info[i][e_x_pos] = Interpolate(B_clock, 0.95, 1.0, (pivot_point_x - 3 + 1.25 ) , belt_center + 1.25);
    #declare robot_info[i][e_y_pos] = pivot_point_y;
    #declare robot_info[i][e_z_pos] = 3.5;
    #break                    
#end
//This is the elevator for the left arm
#switch(B_clock)
  #range (0, 0.499999)
    object {
      Elevator( 2.0, 4.4, 3.35, belt_clock )
      translate <aisle_b+3.5  ,pivot_point_y+0.04, 0>
    }
    #break
  #range (0.499999, 1.0)
    object {
      Elevator( 2.0, 4.4, 3.35, 1 - belt_clock )
      translate <aisle_b+3.5,pivot_point_y+0.04, 0>
    }
    #break
#end

//robot 5  this is the first appearence of the head
#declare i = 5;
#declare robot_info[i][e_exist] = true;
#declare pivot_point_y = last_station + (1 * station_dist) +1.7 - 1.25 + 0.3;
#declare pivot_point_x = aisle_a + 0.3;
#declare robot_info[i][e_y_rot] = 90;
#declare robot_info[i][e_pre_z_pos] = -1.7 + 1.25 - 0.3;
#declare robot_info[i][e_color_index] = mod(ci_offset+A_cycle_count+10,12);
#switch(A_clock)
  #range (0, 0.499) //belt clock doesn't quite go to 1
    #declare robot_info[i][e_z_rot] = 270;   
    #declare robot_info[i][e_x_pos] = (pivot_point_x - 3.2 );
    #declare robot_info[i][e_y_pos] = pivot_point_y;
    #declare robot_info[i][e_z_pos] = Interpolate(belt_clock, 0, 1, 0.5, 3.5);
    #break
  #range (0.499, 0.55)
    #declare robot_info[i][e_z_rot] = 270;
    #declare robot_info[i][e_x_pos] = Interpolate(A_clock, 0.5, 0.55, (pivot_point_x - 3.2) , (pivot_point_x - 2.7) );
    #declare robot_info[i][e_y_pos] = pivot_point_y;
    #declare robot_info[i][e_z_pos] = Interpolate(A_clock, 0.5, 0.55, 3.5, 3.7);
    #break
  #range (0.55, 1.0)
    #declare robot_info[i][e_z_rot] = Interpolate(A_clock, 0.55, 1.0, 270, 90);
    #declare tempa = (270 - robot_info[i][e_z_rot]) * d2r;
    #declare tempd = Interpolate(A_clock, 0.55, 1.0, 2.7, 3.2);
    #declare robot_info[i][e_x_pos] = pivot_point_x - tempd * cos(tempa);
    #declare robot_info[i][e_y_pos] = pivot_point_y + tempd * sin(tempa);
    #declare robot_info[i][e_z_pos] = Interpolate(A_clock, 0.55, 1.0, 3.7, 3.5);
//    #declare robot_info[i][e_x_pos] = pivot_point_x + 2.7*sin(Interpolate(A_clock, 0.55, 0.95, (-pi/2), (pi/2)));
//    #declare robot_info[i][e_y_pos] = pivot_point_y + 2.7*sin(Interpolate(A_clock, 0.55, 0.95, 0, pi));
//    #declare robot_info[i][e_z_pos] = 3.7;
    #break
//  #range (0.95, 1.01)
//    #declare robot_info[i][e_z_rot] = 90;
//    #declare robot_info[i][e_x_pos] = Interpolate(A_clock, 0.95, 1.0, (pivot_point_x + 2.7 ) , belt_center);
//    #declare robot_info[i][e_y_pos] = pivot_point_y;
//    #declare robot_info[i][e_z_pos] = Interpolate(A_clock, 0.95, 1.0, 3.7, 3.5);
    #break                    
#end
//This is the elevator for the head
#switch(A_clock)
  #range (0, 0.499999)
    object {
      Elevator( 2.6, 2.0, 3.0, belt_clock )
      translate <1.3  ,pivot_point_y - 0.9, 0>
    }
    #break
  #range (0.499999, 1.0)
    object {
      Elevator( 2.6, 2.0, 3.0, 1-belt_clock )
      translate <1.3 ,pivot_point_y - 0.9, 0>
    }
    #break
#end

//robot 6  this is the first appearence of the front panel
#declare i = 6;
#declare robot_info[i][e_exist] = true;
#declare pivot_point_y = last_station + (0 * station_dist);
#declare pivot_point_x = aisle_b;
#declare robot_info[i][e_x_rot] = -90;
#declare robot_info[i][e_pre_x_pos] = 0.5;
#declare robot_info[i][e_pre_y_pos] = 1.20;
#switch(B_clock)
  #range (0, 0.4999999) //belt clock doesn't quite go to 1
    #declare robot_info[i][e_y_rot] = 90;
    #declare robot_info[i][e_z_rot] = 180;
    #declare robot_info[i][e_x_pos] = (pivot_point_x + 3.5 -1.2);
    #declare robot_info[i][e_y_pos] = pivot_point_y;
    #declare robot_info[i][e_z_pos] = 0.5+Interpolate(belt_clock, 0, 1, 0.5, 3.5);
    #break
  #range (0.4999999, 0.55)
    #declare robot_info[i][e_y_rot] = Interpolate(B_clock, 0.5, 0.55, 90, 135);
    #declare robot_info[i][e_z_rot] = 180;
    #declare robot_info[i][e_x_pos] = pivot_point_x +  Interpolate(B_clock, 0.5, 0.55, 2.3, 2.0);
    #declare robot_info[i][e_y_pos] = pivot_point_y;
    #declare robot_info[i][e_z_pos] = 0.5+Interpolate(B_clock, 0.5, 0.55, 3.5, 3.7);
    #break
  #range (0.55, 0.935)
    #declare robot_info[i][e_y_rot] = 135;
    #declare robot_info[i][e_z_rot] = Interpolate(B_clock, 0.55, 0.935, 180, 0.0);
    #declare robot_info[i][e_x_pos] = pivot_point_x - 2.0*sin(Interpolate(B_clock, 0.55, 0.935, (-pi/2), (pi/2)));
    #declare robot_info[i][e_y_pos] = pivot_point_y - 2.0*sin(Interpolate(B_clock, 0.55, 0.935, 0, pi));
    #declare robot_info[i][e_z_pos] = 0.5+3.7;
    #break
  #range (0.935, 0.98)
    #declare robot_info[i][e_y_rot] = Interpolate(B_clock, 0.935, 0.98, 135, 90);
    #declare robot_info[i][e_x_pos] = Interpolate(B_clock, 0.935, 0.98, pivot_point_x-2.0, belt_center + 1.2);
    #declare robot_info[i][e_y_pos] = pivot_point_y;
    #declare robot_info[i][e_z_pos] = 0.5+3.7;
    #break                    
  #range (0.98, 0.99)
    #declare robot_info[i][e_y_rot] = 90;
    #declare robot_info[i][e_x_pos] = belt_center + 1.2;
    #declare robot_info[i][e_y_pos] = pivot_point_y;
    #declare robot_info[i][e_z_pos] = 0.5+3.7;
    #break                    
  #range (0.99, 1.00)
    #declare robot_info[i][e_y_rot] = 90;
    #declare robot_info[i][e_x_pos] = belt_center + 1.2;
    #declare robot_info[i][e_y_pos] = pivot_point_y;
    #declare robot_info[i][e_z_pos] = 0.5+Interpolate(B_clock, 0.985, 1.0, 3.7, 3.5);
    #break                    
#end
//This is the elevator for the front panel
#switch(B_clock)
  #range (0, 0.499999)
    object {
      Elevator( 2.3, 2.3, 3.35, belt_clock )
      translate <aisle_b+3.35  ,pivot_point_y, 0>
    }
    #break
  #range (0.499999, 1.0)
    object {
      Elevator( 2.3, 2.3, 3.35, 1 - belt_clock )
      translate <aisle_b+3.35,pivot_point_y, 0>
    }
    #break
#end

//robot 7  This is the torso on the belt     
#if (A_clock <= 0.5)
  #declare i = 7;
  #declare start_y = last_station + (5 * station_dist);
  #declare end_y = last_station + (4 * station_dist);
  #declare robot_info[i][e_exist] = true;
  #declare robot_info[i][e_x_pos] = belt_center;
  #declare robot_info[i][e_y_pos] = Interpolate(belt_clock, 0, 1, start_y, end_y);
  #declare robot_info[i][e_z_pos] = 3.5;
  #declare robot_info[i][e_y_rot] = 90;
  #declare robot_info[i][e_z_rot] = 90;
  #declare robot_info[i][e_color_index] = mod(ci_offset+A_cycle_count+11,12);
#end

//robot 8   This is the torso and leg on the belt     
#if (B_clock < 0.5)
  #declare i = 8;
  #declare start_y = last_station + (4 * station_dist);
  #declare end_y = last_station + (3 * station_dist);
  #declare robot_info[i][e_exist] = true;
  #declare robot_info[i][e_x_pos] = belt_center;
  #declare robot_info[i][e_y_pos] = Interpolate(belt_clock, 0, 1, start_y, end_y);
  #declare robot_info[i][e_z_pos] = 3.5;
  #declare robot_info[i][e_y_rot] = 90;
  #declare robot_info[i][e_z_rot] = 90;
  #declare robot_info[i][e_y_leg_angle] = 6.2;
  #declare robot_info[i][e_ankle_angle] = -6.2;
  #declare robot_info[i][e_color_index] = mod(ci_offset+B_cycle_count+10,12);
#end

//robot 9     This is the torso, leg and right arm on the belt
#if (A_clock < 0.5)
  #declare i = 9;
  #declare start_y = last_station + (3 * station_dist);
  #declare end_y = last_station + (2 * station_dist);
  #declare robot_info[i][e_exist] = true;
  #declare robot_info[i][e_x_pos] = belt_center;
  #declare robot_info[i][e_y_pos] = Interpolate(belt_clock, 0, 1, start_y, end_y);
  #declare robot_info[i][e_z_pos] = 3.5;
  #declare robot_info[i][e_y_rot] = 90;
  #declare robot_info[i][e_z_rot] = 90;
  #declare robot_info[i][e_y_leg_angle] = 6.2;
  #declare robot_info[i][e_ankle_angle] = -6.2;
  #declare robot_info[i][e_color_index] = mod(ci_offset+A_cycle_count+10,12);
#end

//robot 10     This is the torso, right arm, left arm, and leg on the belt     
#if (B_clock < 0.5)
  #declare i = 10;
  #declare start_y = last_station + (2 * station_dist);
  #declare end_y = last_station + (1 * station_dist);
  #declare robot_info[i][e_exist] = true;
  #declare robot_info[i][e_x_pos] = belt_center;
  #declare robot_info[i][e_y_pos] = Interpolate(belt_clock, 0, 1, start_y, end_y);
  #declare robot_info[i][e_z_pos] = 3.5;
  #declare robot_info[i][e_y_rot] = 90;
  #declare robot_info[i][e_z_rot] = 90;
  #declare robot_info[i][e_y_leg_angle] = 6.2;
  #declare robot_info[i][e_ankle_angle] = -6.2;
  #declare robot_info[i][e_color_index] = mod(ci_offset+B_cycle_count+9,12);
#end

//robot 11     This is the torso, right arm, left arm, leg and head on the belt     
#if (A_clock < 0.5)
  #declare i = 11;
  #declare start_y = last_station + (1 * station_dist);
  #declare end_y = last_station + (0 * station_dist);
  #declare robot_info[i][e_exist] = true;
  #declare robot_info[i][e_x_pos] = belt_center;
  #declare robot_info[i][e_y_pos] = Interpolate(belt_clock, 0, 1, start_y, end_y);
  #declare robot_info[i][e_z_pos] = 3.5;
  #declare robot_info[i][e_y_rot] = 90;
  #declare robot_info[i][e_z_rot] = 90;
  #declare robot_info[i][e_y_leg_angle] = 6.2;
  #declare robot_info[i][e_ankle_angle] = -6.2;  
  #declare robot_info[i][e_color_index] = mod(ci_offset+A_cycle_count+9,12);
  #switch (belt_clock)
    #range (0.9, 1.0)
      #declare robot_info[i][e_y_leg_angle] = Interpolate(belt_clock, 0.9, 0.98, 6.2, -5);
      #break
  #end
#end

//robot 12     This is the whole robot on the belt
#declare i = 12;
#declare start_y = last_station + (0 * station_dist);
#declare end_y = last_station + (-1 * station_dist); //not really
#declare robot_info[i][e_exist] = true;
#declare robot_info[i][e_x_pos] = belt_center;
#declare robot_info[i][e_y_rot] = 90;
#declare robot_info[i][e_z_rot] = 90;
#declare robot_info[i][e_z_pos] = 3.5;
#declare robot_info[i][e_y_pos] = Interpolate(belt_clock, 0, 1, start_y, end_y);   
#declare robot_info[i][e_color_index] = mod(ci_offset+B_cycle_count+8,12);
// These points are used for unloading off of the belt
#declare PA = <0, 12, 2.5>; //belt end rotation point
#declare PB = PA + <0, -cos(45*d2r), +sin(45*d2r)>; //Center of body after rotation
#declare PC1 = PB + <0, -1.2*cos(45*d2r), -1.2*sin(45*d2r)>; //Hip location, after rotation, before standing up
#declare PD = PC1 + <0, (2.425-0.5)*cos(75*d2r), -(2.425-0.5)*sin(75*d2r)>; //Ankle location while standing up
#switch (B_clock)
  #range ( 0.0, 0.15) //Just move down the belt, position based on belt_clock above
    #declare robot_info[i][e_y_leg_angle] = -5;
    #declare robot_info[i][e_ankle_angle] = -6.2;
    #break
  #range ( 0.15, 0.3) //Continue down belt, but lower the leg.
    #declare robot_info[i][e_y_leg_angle] = Interpolate(B_clock, 0.14, 0.3, -5, -60);
    #declare robot_info[i][e_ankle_angle] = -6.2;
    #break
  #range ( 0.3, 0.35) //Roll off the end and hit the floor      
    #declare robot_info[i][e_y_pos] = 12; //Interpolate(/*belt_clock*/ 0.625, 0, 1, start_y, end_y);
    #declare robot_info[i][e_pre_x_pos] = -1;
    #declare robot_info[i][e_z_pos] = 2.5;
    #declare robot_info[i][e_y_leg_angle] = -60;
    #declare robot_info[i][e_y_r_shoulder] = Interpolate(B_clock, 0.3, 0.35, 0.0, 22);
    #declare robot_info[i][e_y_l_shoulder] = Interpolate(B_clock, 0.3, 0.35, 0.0, 22);
    #declare robot_info[i][e_y_r_elbow] = Interpolate(B_clock, 0.3, 0.35, 0.0, 22);
    #declare robot_info[i][e_y_l_elbow] = Interpolate(B_clock, 0.3, 0.35, 0.0, 22);
    #declare robot_info[i][e_ankle_angle] = Interpolate(B_clock, 0.3, 0.35, -6.2, 15);
    #declare robot_info[i][e_y_rot] = Interpolate(B_clock, 0.3, 0.35, 90, 45);
    #declare robot_info[i][e_leg_length] = Interpolate(B_clock, 0.3, 0.35, 2.8, 2.425);
    #break
  #range ( 0.35, 0.4)  //Stand up
    #declare start_t = 0.35;
    #declare end_t =  0.4;      
    #declare robot_info[i][e_pre_z_pos] = 1.2;
    #declare robot_info[i][e_y_pos] = Interpolate(B_clock, start_t, end_t, (PC1.y), (PD.y));
    #declare robot_info[i][e_z_pos] = Interpolate(B_clock, start_t, end_t, (PC1.z), (PD.z+2.3));
    #declare robot_info[i][e_y_r_shoulder] = Interpolate(B_clock, start_t, end_t, 22, 10);
    #declare robot_info[i][e_y_l_shoulder] = Interpolate(B_clock, start_t, end_t, 22, 10);
    #declare robot_info[i][e_y_r_elbow] = Interpolate(B_clock, start_t, end_t, 22, 100);
    #declare robot_info[i][e_y_l_elbow] = Interpolate(B_clock, start_t, end_t, 22, 100);
    #declare robot_info[i][e_ankle_angle] = 90 - r2d*atan2(((robot_info[i][e_z_pos]-PD.z)),(PD.y-robot_info[i][e_y_pos]));
    #declare robot_info[i][e_y_rot] = Interpolate(B_clock, start_t, end_t, 45, 0);
    #declare robot_info[i][e_y_leg_angle] = -robot_info[i][e_ankle_angle] - robot_info[i][e_y_rot];
    #declare robot_info[i][e_leg_length] = 0.5 + hypot((robot_info[i][e_z_pos]-PD.z),(robot_info[i][e_y_pos]-PD.y));
    #break
  #range ( 0.4, 0.60)  //Start up
    #declare start_t = 0.4;
    #declare end_t =  0.60;      
    #declare robot_info[i][e_power] = true;
    #declare robot_info[i][e_z_pos] = 4.0;
    #declare robot_info[i][e_y_pos] = Interpolate(pow((B_clock-start_t)/(end_t-start_t),2), 0, 1, (PD.y), 10.0); //accelerate
    #declare robot_info[i][e_y_rot] = 0;
    #declare robot_info[i][e_z_rot] = 90;
    #declare robot_info[i][e_y_r_shoulder] = Interpolate(B_clock, start_t, end_t, 10, 0);
    #declare robot_info[i][e_y_l_shoulder] = Interpolate(B_clock, start_t, end_t, 10, 0);
    #declare robot_info[i][e_x_r_shoulder] = 25*sin(Interpolate(B_clock, start_t, end_t, 0, pi));
    #declare robot_info[i][e_x_l_shoulder] = -25*sin(Interpolate(B_clock, start_t, end_t, 0, pi));
    #declare robot_info[i][e_y_r_elbow] = Interpolate(B_clock, start_t, end_t, 100, 90);
    #declare robot_info[i][e_y_l_elbow] = Interpolate(B_clock, start_t, end_t, 100, 90);
    #break                                                                                                               
  #range ( 0.60, 1.0)  //Start moving around the corner
    #declare start_t = 0.60;
    #declare end_t =  1.0;
    #declare turn_start_t = 0.6;      
    #declare turn_end_t = 1.05;      
    #declare pivot_y = 10;
    #declare robot_x_start = belt_center;
    #declare robot_x_end   = aisle_a + 0.3;
    #declare turn_radius = (robot_x_start-robot_x_end)/2;
    #declare pivot_x = robot_x_end+turn_radius;
    #declare robot_info[i][e_power] = true;
    #declare robot_info[i][e_y_rot] = 0;
    #declare robot_info[i][e_z_rot] = Interpolate(B_clock, turn_start_t, turn_end_t, 90, -90);
    #declare robot_info[i][e_x_pos] = pivot_x + turn_radius*cos(Interpolate(B_clock, turn_start_t, turn_end_t, 0, pi));
    #declare robot_info[i][e_y_pos] = pivot_y - turn_radius*sin(Interpolate(B_clock, turn_start_t, turn_end_t, 0, pi));
    #declare robot_info[i][e_z_pos] = 4.0;
    #declare robot_info[i][e_y_r_elbow] = 90;
    #declare robot_info[i][e_y_l_elbow] = 90;
    #break                                                                                                               
  #else
    #error "Out of Range";
#end

//robot 13 This guy move from assemblee to assembler
#declare i = 13;
#declare robot_info[i][e_exist] = true;
#declare robot_info[i][e_power] = true;
#declare robot_x_start = belt_center;
#declare robot_x_end   = aisle_a + 0.3;
#declare robot_y_start = 10;
#declare robot_y_end   = last_station + (1 * station_dist) + 0.75;
#declare robot_info[i][e_z_pos] = 4;
#declare x_dist = robot_x_end - robot_x_start;
#declare y_dist = robot_y_end - robot_y_start;
#declare se_dir = atan2 (x_dist, y_dist);
#declare robot_dir = 270 - se_dir * (cos(Interpolate(B_clock, 0.52, 0.96, 0, 2*pi))- 1);
#declare robot_info[i][e_color_index] = mod(ci_offset+B_cycle_count+7,12);

#switch(B_clock)
  #range ( 0.0, 0.05)  ///Continue moving around the corner
    #declare start_t = 0.0;
    #declare end_t =  0.3;
    #declare turn_start_t = 0.6;      
    #declare turn_end_t = 1.05;      
    #declare pivot_y = 10;
    #declare robot_x_start = belt_center;
    #declare robot_x_end   = aisle_a + 0.3;
    #declare turn_radius = (robot_x_start-robot_x_end)/2;
    #declare pivot_x = robot_x_end+turn_radius;
    #declare robot_info[i][e_power] = true;
    #declare robot_info[i][e_y_rot] = 0;
    #declare robot_info[i][e_z_rot] = Interpolate((1+B_clock), turn_start_t, turn_end_t, 90, -90);
    #declare robot_info[i][e_x_pos] = pivot_x + turn_radius*cos(Interpolate((1+B_clock), turn_start_t, turn_end_t, 0, pi));
    #declare robot_info[i][e_y_pos] = pivot_y - turn_radius*sin(Interpolate((1+B_clock), turn_start_t, turn_end_t, 0, pi));
    #declare robot_info[i][e_z_pos] = 4.0;
    #declare robot_info[i][e_y_r_elbow] = 90;
    #declare robot_info[i][e_y_l_elbow] = 90;
    #break                                                                                                               
  #range (0.05, 0.8) // Cruise down aisle A
    #declare start_t = 0.05;
    #declare end_t =  0.80;
    compute_arms(
      i,             
      10, //skip right
      0.45,
      (4.7 -3.5),
      1.5,
      0.0,  
      (4.7 - 3.2)
    ) 
    #declare robot_info[i][e_y_r_elbow] = Interpolate(B_clock, start_t, end_t, 90, 0);                          
    #declare robot_info[i][e_x_pos] = robot_x_end;
    #declare robot_info[i][e_y_pos] = Interpolate(B_clock, start_t, end_t, robot_y_start, robot_y_end - 2);
    #declare robot_info[i][e_z_rot] = robot_dir;
    #declare robot_info[i][e_l_thumb] = 30;
    #declare robot_info[i][e_r_thumb] = 30;
    #break        
  #range (0.80, 0.95) //Slow dow, turn into position
    #declare start_t = 0.80;
    #declare end_t =  0.95;
    compute_arms(
      i,             
      10, //skip right
      0.0,
      1.5,
      1.5,
      0.0,  
      (4.7 - Interpolate(B_clock, start_t, end_t, 3.2, 3.5))
    )                           
    #declare robot_info[i][e_x_pos] = robot_x_end;
    #declare robot_info[i][e_y_pos] = robot_y_end - 2*pow(Interpolate(B_clock, start_t, end_t, 1, 0), 2);
    #declare robot_info[i][e_z_rot] = Interpolate(B_clock, start_t, end_t, 270, 360);
    #declare robot_info[i][e_l_thumb] = 30;
    #declare robot_info[i][e_r_thumb] = 30;
    #declare robot_info[i][e_z_leg_rot] = 270 - robot_info[i][e_z_rot];
    #break        
  #range (0.95, 0.99) //extend arms
    compute_arms(
      i,             
      10, //skip right
      0.0,
      1.5,
      Interpolate(B_clock, 0.95, 0.99, 1.5, 1.75),
      0.0,  
      (4.7 - 3.5)
    )                           
    // implied #declare robot_info[i][e_z_rot] = 360;
    #declare robot_info[i][e_z_leg_rot] = -90;
    #declare robot_info[i][e_l_thumb] = Interpolate(B_clock, 0.95, 0.99, 30, 0);
    #declare robot_info[i][e_r_thumb] = Interpolate(B_clock, 0.95, 0.99, 30, 0);
    #declare robot_info[i][e_x_pos] = robot_x_end;
    #declare robot_info[i][e_y_pos] = robot_y_end;
  #break        
  #range (0.99, 1.00) //pause
    compute_arms(
      i,             
      10, //skip right
      0.0,
      1.5,
      1.75,
      0.0,  
      (4.7 - 3.5)
    )                           
    // implied #declare robot_info[i][e_z_rot] = 360;
    #declare robot_info[i][e_z_leg_rot] = -90;
    #declare robot_info[i][e_l_thumb] = 0;
    #declare robot_info[i][e_r_thumb] = 0;
    #declare robot_info[i][e_x_pos] = robot_x_end;
    #declare robot_info[i][e_y_pos] = robot_y_end;
  #break        
  #else
    #error "Out of Range";
#end

//robot 14 This guy adds the head
#declare i = 14;
#declare robot_info[i][e_exist] = true;
#declare robot_info[i][e_power] = true;
#declare robot_x_start = aisle_a + 0.3;
#declare robot_x_end   = aisle_a - 0.5;
#declare robot_y_start = last_station + (1 * station_dist) +1.7 - 1.25 + 0.3;
#declare robot_y_end   = last_station + (3 * station_dist) - 0.5;
#declare robot_info[i][e_z_pos] = 4;
#declare x_dist = robot_x_end - robot_x_start;
#declare y_dist = robot_y_end - robot_y_start;
#declare se_dir = atan2 (x_dist, y_dist);
#declare robot_dir = 270 - se_dir * (cos(Interpolate(B_clock, 0.52, 0.96, 0, 2*pi))- 1);
#declare robot_info[i][e_color_index] = mod(ci_offset+B_cycle_count+6,12);

#switch(B_clock)
  #range (0, 0.05) // Pull the head off of the elevator   
    compute_arms(
      i,
      10, //skip right
      0.0,
      1.5,
      (Interpolate(B_clock, 0, 0.05, 2.25, 1.75)),
      0.0,
      (4.7 - Interpolate(B_clock, 0, 0.05, 3.5, 3.7))
    )
    #declare robot_info[i][e_z_rot] = 0;
    #declare robot_info[i][e_z_leg_rot] = -90;
    #declare robot_info[i][e_x_pos] = robot_x_start;
    #declare robot_info[i][e_y_pos] = robot_y_start;
    #break
  #range (0.05, 0.5) //Spin around, lowering arm as we go
    compute_arms(
      i,
      10, //skip right
      0.0,
      1.5,
      Interpolate(B_clock, 0.05, 0.5, 1.75, 2.25),
      0.0,
      (4.7 - Interpolate(B_clock, 0.05, 0.5, 3.7, 3.5))
    )
    #declare robot_info[i][e_z_rot] = Interpolate(B_clock, 0.05, 0.5, 360, 180);
    #declare robot_info[i][e_z_leg_rot] = Interpolate(B_clock, 0.05, 0.5, -90, 90);
    #declare robot_info[i][e_x_pos] = robot_x_start;
    #declare robot_info[i][e_y_pos] = robot_y_start;
    #break
//  #range (0.45, 0.50)  //Shove the head into the torso  
//    compute_arms(
//      i,             
//      10, //skip right
//      0.0,
//      1.5,
//      Interpolate(B_clock, 0.45, 0.5, 1.75, 2.25),
//      0.0,  
//      (4.7 - Interpolate(B_clock, 0.45, 0.5, 3.7, 3.5))
//    )
//    #declare robot_info[i][e_z_rot] = 180;
//    #declare robot_info[i][e_z_leg_rot] = 90;
//    #declare robot_info[i][e_x_pos] = robot_x_start;
//    #declare robot_info[i][e_y_pos] = robot_y_start;
//    #break
  #range (0.50, 0.53) // Pull arms back
    compute_arms(
      i,             
      10, //skip right
      0.15,  
      (4.7 -3.5),
      Interpolate(B_clock, 0.5, 0.53, 2.25, 1.75),
      0.0,  
      (4.7 - 3.5)
    )                           
    #declare robot_info[i][e_l_thumb] = Interpolate(B_clock, 0.5, 0.53, 0, 30);
    #declare robot_info[i][e_r_thumb] = Interpolate(B_clock, 0.5, 0.53, 0, 30);
    #declare robot_info[i][e_z_head] = Interpolate(B_clock, 0.5, 0.53, 0, 20);
    #declare robot_info[i][e_z_eye] = Interpolate(B_clock, 0.5, 0.53, 0, 20);
    #declare robot_info[i][e_z_rot] = 180;
    #declare robot_info[i][e_z_leg_rot] = 90;
    #declare robot_info[i][e_x_pos] = robot_x_start;
    #declare robot_info[i][e_y_pos] = robot_y_start;
    #break
  #range (0.53, 0.62) //Turn and start down aisle A
    compute_arms(
      i,             
      10, //skip right
      Interpolate(B_clock, 0.53, 0.62, 0.15, 0.45),
      (4.7 -3.5),
      Interpolate(B_clock, 0.53, 0.62, 1.75, 1.5),
      0.0,  
      (4.7 - 3.5)
    )                           
    #declare robot_info[i][e_z_rot] = Interpolate(B_clock, 0.53, 0.62, 180, robot_dir);
    #declare robot_info[i][e_z_leg_rot] = robot_dir - robot_info[i][e_z_rot];
    #declare robot_info[i][e_z_head] = Interpolate(B_clock, 0.53, 0.62, 20, 0);
    #declare robot_info[i][e_z_eye] = Interpolate(B_clock, 0.53, 0.62, 20, 0);
    #declare robot_info[i][e_l_thumb] = 30;
    #declare robot_info[i][e_r_thumb] = 30;
    #declare robot_info[i][e_x_pos] = robot_x_start + (x_dist * (0.5+ sin(Interpolate(B_clock, 0.53, 0.95, -pi/2, pi/2))/2));
    #declare robot_info[i][e_y_pos] = robot_y_start + (y_dist * (0.5+ sin(Interpolate(B_clock, 0.53, 0.95, -pi/2, pi/2))/2));
    #break        
  #range (0.62, 0.86) // Cruise down aisle A
    compute_arms(
      i,             
      10, //skip right
      0.45,
      (4.7 -3.5),
      1.5,
      0.0,  
      (4.7 - 3.5)
    ) 
    #declare robot_info[i][e_y_r_elbow] = Interpolate(B_clock, 0.62, 0.86, 0, 90);                          
    #declare robot_info[i][e_x_pos] = robot_x_start + (x_dist * (0.5+ sin(Interpolate(B_clock, 0.53, 0.95, -pi/2, pi/2))/2));
    #declare robot_info[i][e_y_pos] = robot_y_start + (y_dist * (0.5+ sin(Interpolate(B_clock, 0.53, 0.95, -pi/2, pi/2))/2));
    #declare robot_info[i][e_z_rot] = robot_dir;
    #declare robot_info[i][e_l_thumb] = 30;
    #declare robot_info[i][e_r_thumb] = 30;
    #break        
  #range (0.86, 0.95) //Slow dow, turn into position
    compute_arms(
      i,             
      Interpolate(B_clock, 0.86, 0.95, 1.5, 1.6),
      Interpolate(B_clock, 0.86, 0.95, 0, 0.15),
      Interpolate(B_clock, 0.86, 0.95, 1.5, (4.7 -3.5)),
      Interpolate(B_clock, 0.86, 0.95, 1.5, 1.6),
      Interpolate(B_clock, 0.86, 0.95, 0.0, 0.25),
      (4.7 - 3.5)
    )                           
    #declare robot_info[i][e_x_pos] = robot_x_start + (x_dist * (0.5+ sin(Interpolate(B_clock, 0.53, 0.95, -pi/2, pi/2))/2));
    #declare robot_info[i][e_y_pos] = robot_y_start + (y_dist * (0.5+ sin(Interpolate(B_clock, 0.53, 0.95, -pi/2, pi/2))/2));
    #declare robot_info[i][e_z_rot] = Interpolate(B_clock, 0.86, 0.95, robot_dir, 360);
    #declare robot_info[i][e_l_thumb] = 30;
    #declare robot_info[i][e_r_thumb] = 30;
    #declare robot_info[i][e_z_leg_rot] = robot_dir - robot_info[i][e_z_rot];
    #break        
  #range (0.95, 0.99) //extend arms
    compute_arms(
      i,             
      Interpolate(B_clock, 0.95, 0.99, 1.6, 2.1),
      0.15,
      (4.7 -3.5),
      Interpolate(B_clock, 0.95, 0.99, 1.6, 2.2),
      0.25,  
      (4.7 - 3.5)
    )                           
    // implied #declare robot_info[i][e_z_rot] = 360;
    #declare robot_info[i][e_z_leg_rot] = -90;
    #declare robot_info[i][e_l_thumb] = Interpolate(B_clock, 0.95, 0.99, 30, 0);
    #declare robot_info[i][e_r_thumb] = Interpolate(B_clock, 0.95, 0.99, 30, 0);
    #declare robot_info[i][e_x_pos] = robot_x_end;
    #declare robot_info[i][e_y_pos] = robot_y_end;
  #break        
  #range (0.99, 1.00) //pause
    compute_arms(
      i,             
      2.1,
      0.15,
      (4.7 -3.5),
      2.2,
      0.25,  
      (4.7 - 3.5)
    )                           
    // implied #declare robot_info[i][e_z_rot] = 360;
    #declare robot_info[i][e_z_leg_rot] = -90;
    #declare robot_info[i][e_l_thumb] = 0;
    #declare robot_info[i][e_r_thumb] = 0;
    #declare robot_info[i][e_x_pos] = robot_x_end;
    #declare robot_info[i][e_y_pos] = robot_y_end;
  #break        
  #else
    #error "Out of Range";
#end

//robot 15 This guy adds the right arm
#declare i = 15;
#declare robot_info[i][e_exist] = true;
#declare robot_info[i][e_power] = true;
#declare robot_x_start = aisle_a - 0.5;
#declare robot_x_end   = aisle_a;
#declare robot_y_start = last_station + (3 * station_dist) - 0.5; 
#declare robot_y_end   = last_station + (5 * station_dist);
#declare x_dist = robot_x_end - robot_x_start;
#declare y_dist = robot_y_end - robot_y_start;
#declare se_dir = atan2 (x_dist, y_dist);
#declare robot_dir = 270 - se_dir * (cos(Interpolate(B_clock, 0.52, 0.96, 0, 2*pi))- 1);
#declare robot_info[i][e_z_pos] = 4;
#declare robot_info[i][e_color_index] = mod(ci_offset+B_cycle_count+5,12);

#switch(B_clock)
  #range (0, 0.08) // Pull the arm off of the elevator   
    compute_arms(
      i,
      (Interpolate(B_clock, 0, 0.08, 2.1, 1.1)),
      0.15,
      (4.7 - Interpolate(B_clock, 0, 0.08, 3.5, 3.7)),
      (Interpolate(B_clock, 0, 0.08, 2.2, 1.2)),
      0.25,
      (4.7 - Interpolate(B_clock, 0, 0.08, 3.5, 3.7))
    )
    #declare robot_info[i][e_z_rot] = 0;
    #declare robot_info[i][e_z_leg_rot] = -90;
    #declare robot_info[i][e_x_pos] = robot_x_start;
    #declare robot_info[i][e_y_pos] = robot_y_start;
    #break
  #range (0.08, 0.45) //Spin around, lowering arm as we go
    compute_arms(
      i,
      1.1,
      0.15,
      (4.7- Interpolate(B_clock, 0.08, 0.45, 3.7, 3.5)),
      1.2,
      0.25,
      (4.7 - Interpolate(B_clock, 0.08, 0.45, 3.7, 3.5))
    )
    #declare robot_info[i][e_z_rot] = Interpolate(B_clock, 0.08, 0.45, 360, 180);
    #declare robot_info[i][e_z_leg_rot] = Interpolate(B_clock, 0.08, 0.45, -90, 90);
    #declare robot_info[i][e_x_pos] = robot_x_start;
    #declare robot_info[i][e_y_pos] = robot_y_start;
    #break
  #range (0.45, 0.50)  //Shove the arm into the torso  
    compute_arms(
      i,             
      Interpolate(B_clock, 0.45, 0.5, 1.1, 2.1),
      0.15,  
      (4.7 - 3.5),
      Interpolate(B_clock, 0.45, 0.5, 1.2, 2.2),
      0.25,  
      (4.7 - 3.5)
    )
    #declare robot_info[i][e_z_rot] = 180;
    #declare robot_info[i][e_z_leg_rot] = 90;
    #declare robot_info[i][e_x_pos] = robot_x_start;
    #declare robot_info[i][e_y_pos] = robot_y_start;
    #break
  #range (0.50, 0.52) // Pull arms back
    compute_arms(
      i,             
      Interpolate(B_clock, 0.5, 0.52, 2.1, 1.6),
      0.15,  
      (4.7 -3.5),
      Interpolate(B_clock, 0.5, 0.52, 2.1, 1.6),
      0.25,  
      (4.7 - 3.5)
    )                           
    #declare robot_info[i][e_l_thumb] = Interpolate(B_clock, 0.5, 0.52, 0, 30);
    #declare robot_info[i][e_r_thumb] = Interpolate(B_clock, 0.5, 0.52, 0, 30);
    #declare robot_info[i][e_z_head] = Interpolate(B_clock, 0.5, 0.52, 0, 20);
    #declare robot_info[i][e_z_eye] = Interpolate(B_clock, 0.5, 0.52, 0, 20);
    #declare robot_info[i][e_z_rot] = 180;
    #declare robot_info[i][e_z_leg_rot] = 90;
    #declare robot_info[i][e_x_pos] = robot_x_start;
    #declare robot_info[i][e_y_pos] = robot_y_start;
    #break
  #range (0.52, 0.62) //Turn and start down aisle A
    compute_arms(
      i,             
      1.6,
      Interpolate(B_clock, 0.52, 0.62, 0.15, 0.45),
      (4.7 -3.5),
      1.6,
      0.25,  
      (4.7 - 3.5)
    )                           
    #declare robot_info[i][e_z_rot] = Interpolate(B_clock, 0.52, 0.62, 180, robot_dir);
    #declare robot_info[i][e_z_leg_rot] = robot_dir - robot_info[i][e_z_rot];
    #declare robot_info[i][e_z_head] = Interpolate(B_clock, 0.52, 0.62, 20, 0);
    #declare robot_info[i][e_z_eye] = Interpolate(B_clock, 0.52, 0.62, 20, 0);
    #declare robot_info[i][e_l_thumb] = 30;
    #declare robot_info[i][e_r_thumb] = 30;
    #declare robot_info[i][e_x_pos] = robot_x_start + (x_dist * (0.5+ sin(Interpolate(B_clock, 0.52, 0.96, -pi/2, pi/2))/2));
    #declare robot_info[i][e_y_pos] = robot_y_start + (y_dist * (0.5+ sin(Interpolate(B_clock, 0.52, 0.96, -pi/2, pi/2))/2));
    #break        
  #range (0.62, 0.86) // Cruise down aisle A
    compute_arms(
      i,             
      1.6
      0.45,
      (4.7 -3.5),
      1.6,
      0.25,  
      (4.7 - 3.5)
    )                           
    #declare robot_info[i][e_l_thumb] = 30;
    #declare robot_info[i][e_r_thumb] = 30;
    #declare robot_info[i][e_x_pos] = robot_x_start + (x_dist * (0.5+ sin(Interpolate(B_clock, 0.52, 0.96, -pi/2, pi/2))/2));
    #declare robot_info[i][e_y_pos] = robot_y_start + (y_dist * (0.5+ sin(Interpolate(B_clock, 0.52, 0.96, -pi/2, pi/2))/2));
    #declare robot_info[i][e_z_rot] = robot_dir;
    #break        
  #range (0.86, 0.96) //Slow dow, turn into position
    compute_arms(
      i,             
      1.6
      0.45,
      (4.7 -3.5),
      1.6,
      0.25,  
      (4.7 - 3.5)
    )                           
    #declare robot_info[i][e_l_thumb] = 30;
    #declare robot_info[i][e_r_thumb] = 30;
    #declare robot_info[i][e_x_pos] = robot_x_start + (x_dist * (0.5+ sin(Interpolate(B_clock, 0.52, 0.96, -pi/2, pi/2))/2));
    #declare robot_info[i][e_y_pos] = robot_y_start + (y_dist * (0.5+ sin(Interpolate(B_clock, 0.52, 0.96, -pi/2, pi/2))/2));
    #declare robot_info[i][e_z_rot] = Interpolate(B_clock, 0.86, 0.96, robot_dir, 360);
    #declare robot_info[i][e_z_leg_rot] = robot_dir - robot_info[i][e_z_rot];
    #break        
  #range (0.96, 0.99) //extend arms
    compute_arms(
      i,             
      Interpolate(B_clock, 0.96, 0.99, 1.6, 1.9),
      0.45,
      (4.7 -3.5),
      Interpolate(B_clock, 0.96, 0.99, 1.6, 1.9),
      0.25,  
      (4.7 - 3.5)
    )                           
    // implied #declare robot_info[i][e_z_rot] = 360;
    #declare robot_info[i][e_z_leg_rot] = -90;
    #declare robot_info[i][e_l_thumb] = Interpolate(B_clock, 0.96, 0.99, 30, 0);
    #declare robot_info[i][e_r_thumb] = Interpolate(B_clock, 0.96, 0.99, 30, 0);
    #declare robot_info[i][e_x_pos] = robot_x_end;
    #declare robot_info[i][e_y_pos] = robot_y_end;
  #break        
  #range (0.99, 1.00) //extend arms
    compute_arms(
      i,             
      1.9,
      0.45,
      (4.7 -3.5),
      1.9,
      0.25,  
      (4.7 - 3.5)
    )                           
    // implied #declare robot_info[i][e_z_rot] = 360;
    #declare robot_info[i][e_z_leg_rot] = -90;
    #declare robot_info[i][e_l_thumb] = 0;
    #declare robot_info[i][e_r_thumb] = 0;
    #declare robot_info[i][e_x_pos] = robot_x_end;
    #declare robot_info[i][e_y_pos] = robot_y_end;
  #break        
  #else
    #error "Out of Range";
#end

//robot 16    This is the one that loads the torso on to the belt       
#declare i = 16;

#declare robot_info[i][e_exist] = true;
#declare robot_info[i][e_power] = true;
#declare robot_info[i][e_x_pos] = aisle_a;
#declare robot_info[i][e_y_pos] = last_station + (5 * station_dist);
#declare robot_y_start = last_station + (5 * station_dist);
#declare robot_info[i][e_z_pos] = 4;
#declare robot_info[i][e_color_index] = mod(ci_offset+B_cycle_count+4,12);

#switch(B_clock)
  #range (0, 0.025) 
    compute_arms( //pick it up
      i,   
      Interpolate(B_clock, 0, 0.025, 1.9, 1.6),
      0.45
      (4.7 - Interpolate(B_clock, 0, 0.025, 3.5, 3.7)),
      Interpolate(B_clock, 0, 0.025, 1.9, 1.6),
      0.25
      (4.7 - Interpolate(B_clock, 0, 0.025, 3.5, 3.7))
    )
    #declare robot_info[i][e_z_rot] = 0;
    #declare robot_info[i][e_z_leg_rot] = -90;
    #break
  #range (0.025, 0.475)
    compute_arms(
      i,   
      1.6,
      0.45
      (4.7 - 3.7),
      1.6,
      0.25
      (4.7 - 3.7)
    )
    #declare robot_info[i][e_z_rot] = Interpolate(B_clock, 0.025, 0.475, 360, 180);
    #declare robot_info[i][e_z_leg_rot] = Interpolate(B_clock, 0.025, 0.475, -90, 90);
    #break
  #range (0.475, 0.50)    
    compute_arms(
      i,   
      Interpolate(B_clock, 0.475, 0.5, 1.6, 2.1),
      0.45
      (4.7 - Interpolate(B_clock, 0.475, 0.5, 3.7, 3.5)),
      Interpolate(B_clock, 0.475, 0.5, 1.6, 2.1),
      0.25
      (4.7 - Interpolate(B_clock, 0.475, 0.5, 3.7, 3.5))
    )
    #declare robot_info[i][e_z_rot] = 180;
    #declare robot_info[i][e_z_leg_rot] = 90;
    #break
  #range (0.50, 0.54) // Pull arms back
    compute_arms(
      i,             
      Interpolate(B_clock, 0.5, 0.54, 2.1, 1.6),
      Interpolate(B_clock, 0.5, 0.54, 0.45, 0.35), 
      (4.7 -3.5),
      Interpolate(B_clock, 0.5, 0.54, 2.2, 1.6),
      0.25,  
      (4.7 - 3.5)
    )                           
    #declare robot_info[i][e_l_thumb] = Interpolate(B_clock, 0.5, 0.54, 0, 30);
    #declare robot_info[i][e_r_thumb] = Interpolate(B_clock, 0.5, 0.54, 0, 30);
    #declare robot_info[i][e_z_head] = Interpolate(B_clock, 0.5, 0.54, 0, 20);
    #declare robot_info[i][e_z_eye] = Interpolate(B_clock, 0.5, 0.54, 0, 20);
    #declare robot_info[i][e_z_rot] = 180;
    #declare robot_info[i][e_z_leg_rot] = 90;
    #break
  #range (0.54, 0.712) //Accelerate and head for the corner
    compute_arms(
      i,             
      1.6,
      Interpolate(B_clock, 0.54, 0.712, 0.35, 0.2),
      (4.7 -3.5),
      1.6,
      0.25,  
      (4.7 - 3.5)
    )                           
    #declare robot_info[i][e_z_rot] = Interpolate(B_clock, 0.54, 0.712, 180, robot_dir);
    #declare robot_info[i][e_z_leg_rot] = robot_dir - robot_info[i][e_z_rot];
    #declare robot_info[i][e_z_head] = Interpolate(B_clock, 0.54, 0.712, 20, 0);
    #declare robot_info[i][e_z_eye] = Interpolate(B_clock, 0.54, 0.712, 20, 0);
    #declare robot_info[i][e_l_thumb] = 30;
    #declare robot_info[i][e_r_thumb] = 30;
    #declare robot_info[i][e_y_pos] = robot_y_start + 2.5*(pow(Interpolate(B_clock, 0.54, 0.712, 0, 1),2));
    #break        
  #range (0.712, 0.846) //round the aisle A corner
    compute_arms(
      i,             
      1.6
      0.45,
      (4.7 -3.5),
      1.6,
      0.25,  
      (4.7 - 3.5)
    )                           
    #declare robot_info[i][e_x_pos] = aisle_a + 2.5 * (1.0 - cos(Interpolate(B_clock, 0.712, 0.846, 0.0, pi/2)));
    #declare robot_info[i][e_y_pos] = robot_y_start + 2.5 +(2.5 * sin(Interpolate(B_clock, 0.712, 0.846, 0.0, pi/2)));
    #declare robot_info[i][e_z_rot] = Interpolate(B_clock, 0.712, 0.846, 270, 180);
    #declare robot_info[i][e_l_thumb] = 30;
    #declare robot_info[i][e_r_thumb] = 30;
    #break        
  #range (0.846, 0.898)  //cruise over to aisle B
    compute_arms(
      i,             
      1.6
      0.45,
      (4.7 -3.5),
      1.6,
      0.25,  
      (4.7 - 3.5)
    )                           
    #declare robot_info[i][e_x_pos] = Interpolate(B_clock, 0.846, 0.898, (aisle_a + 2.5), (aisle_b - 3));
    #declare robot_info[i][e_y_pos] = robot_y_start + 5;
    #declare robot_info[i][e_z_rot] = 180;
    #declare robot_info[i][e_l_thumb] = 30;
    #declare robot_info[i][e_r_thumb] = 30;
    #break        
  #range (0.898, 1.0) //Go part way around the corner
    compute_arms(
      i,             
      1.6
      0.45,
      (4.7 -3.5),
      1.6,
      0.25,  
      (4.7 - 3.5)
    )                           
    #declare robot_info[i][e_x_pos] = aisle_b - 3.0 + (2.5 * (sin(Interpolate(B_clock, 0.898, 1.033, 0.0, pi/2))));
    #declare robot_info[i][e_y_pos] = robot_y_start + 2.5 +(2.5 * cos(Interpolate(B_clock, 0.898, 1.033, 0.0, pi/2)));
    #declare robot_info[i][e_z_rot] = Interpolate(B_clock, 0.898, 1.033, 180, 90);
    #declare robot_info[i][e_l_thumb] = 30;
    #declare robot_info[i][e_r_thumb] = 30;
    #break        
  #else
    #error "Out of Range";
#end

//robot 17    This is the one rolls around the corner, connects 16 to 18.
#declare i = 17;

#declare robot_info[i][e_exist] = true;
#declare robot_info[i][e_power] = true;
#declare robot_info[i][e_x_pos] = aisle_b - 0.5;
#declare robot_y_end = last_station + (4 * station_dist) - 1.2 - 1.25;
#declare robot_info[i][e_z_pos] = 4;
#declare robot_info[i][e_color_index] = mod(ci_offset+B_cycle_count+3,12);

#switch(B_clock)
  #range (0, 0.033) //Go the rest of the way around the corner
    compute_arms(
      i,             
      1.6
      0.45,
      (4.7 -3.5),
      1.6,
      0.25,  
      (4.7 - 3.5)
    )                           
    #declare robot_info[i][e_x_pos] = aisle_b - 3.0 + (2.5 * (sin(Interpolate((B_clock+1), 0.898, 1.033, 0.0, pi/2))));
    #declare robot_info[i][e_y_pos] = robot_y_end + 9.95 +(2.5 * cos(Interpolate((B_clock+1), 0.898, 1.033, 0.0, pi/2)));
    #declare robot_info[i][e_z_rot] = Interpolate((B_clock+1), 0.898, 1.033, 180, 90);
    #declare robot_info[i][e_l_thumb] = 30;
    #declare robot_info[i][e_r_thumb] = 30;
    #break        
  #range (0.033, 0.288) //Cruise up Aisle B
    compute_arms(
      i,             
      1.6,
      0.45,
      (4.7 -3.5),
      1.6,
      0.25,  
      (4.7 - 3.5)
    )                           
    #declare robot_info[i][e_y_pos] = robot_y_end + Interpolate(B_clock, 0.033, 0.288, 9.95, 2.5);
    #declare robot_info[i][e_z_rot] = 90;
    #declare robot_info[i][e_l_thumb] = 30;
    #declare robot_info[i][e_r_thumb] = 30;
    #break        
  #range (0.288, 0.46)  //Cruise up Aisle B, slowing down and turning   
    compute_arms(
      i,   
      1.6,
      Interpolate(B_clock, 0.288, 0.46, 0.45, 0.25), 
      (4.7 - 3.5),                     
      Interpolate(B_clock, 0.288, 0.46, 1.6, 1.1), 
      Interpolate(B_clock, 0.288, 0.46, 0.25, 0.15),
      (4.7 - 3.5)
    )
    #declare robot_info[i][e_y_pos] = robot_y_end + 2.5 * pow(Interpolate(B_clock, 0.288, 0.46, 1, 0),2);
    #declare robot_info[i][e_l_thumb] = 30;
    #declare robot_info[i][e_r_thumb] = 30;
    #declare robot_info[i][e_z_rot] = Interpolate(B_clock, 0.288, 0.46, 90, 180);
    #declare robot_info[i][e_z_leg_rot] = Interpolate(B_clock, 0.288, 0.46, 0, -90);
    #break
  #range (0.46, 0.5) // extend arms
    compute_arms(
      i,             
      Interpolate(B_clock, 0.46, 0.5, 1.6, 2.0),
      Interpolate(B_clock, 0.46, 0.5, 0.25, 0.0), 
      (4.7 -3.5),
      Interpolate(B_clock, 0.46, 0.5, 1.1, 2.3),
      Interpolate(B_clock, 0.46, 0.5, 0.15, 1.0),
      (4.7 - 3.5)
    )                           
    #declare robot_info[i][e_y_pos] = robot_y_end;
    #declare robot_info[i][e_l_thumb] = Interpolate(B_clock, 0.46, 0.5, 30, 0);
    #declare robot_info[i][e_r_thumb] = Interpolate(B_clock, 0.46, 0.5, 30, 0);
    #declare robot_info[i][e_z_rot] = 180;
    #declare robot_info[i][e_z_leg_rot] = -90;
    #break
  #range (0.5, 1.0) // Dissappear
    #declare robot_info[i][e_exist] = false;
    #break        
  #else
    #error "Out of Range";
#end

//robot 18 This guy adds the leg
#declare i = 18;
#declare robot_info[i][e_exist] = true;
#declare robot_info[i][e_power] = true;
#declare robot_x_start = aisle_b - 0.5;
#declare robot_x_end   = aisle_b + 0.5;
#declare robot_y_start = last_station + (4 * station_dist) -1.2 - 1.25;
#declare robot_y_end   = last_station + (2 * station_dist) -0.5;
#declare x_dist = robot_x_end - robot_x_start;
#declare y_dist = robot_y_end - robot_y_start;
#declare se_dir = atan2 (x_dist, y_dist);
#declare robot_dir = 90 - se_dir * (cos(Interpolate(B_clock, 0.52, 0.96, 0, 2*pi))- 1);
#declare robot_info[i][e_x_pos] = robot_x_start;
#declare robot_info[i][e_y_pos] = robot_y_start;
#declare robot_info[i][e_z_pos] = 4;
//#declare robot_info[i][e_z_rot] = 180;
#declare extra_z = 1.5*sin(6.2/r2d);
#declare robot_info[i][e_color_index] = mod(ci_offset+A_cycle_count+3,12);

#switch(A_clock)
  #range (0, 0.03)    // pick up the leg
    compute_arms(
      i,
      Interpolate(A_clock, 0.0, 0.03, 2.0, 1.9),
      0.0,
      (4.7 - Interpolate(A_clock, 0, 0.03, 3.5, 3.6)),
      2.3,
      1.0,
      (4.7 - Interpolate(A_clock, 0, 0.03, 3.5, 3.6)) - extra_z 
    )
    #declare robot_info[i][e_z_rot] = 180;
    #declare robot_info[i][e_z_leg_rot] = -90;
    #break
  #range (0.03, 0.43)  // spin around
    compute_arms(
      i,
      Interpolate(A_clock, 0.03, 0.43, 1.9, 0.0),
      0.0,
      (4.7 - 3.6),
      2.3,
      1.0,
      (4.7 - 3.6 - extra_z )
    )
    #declare robot_info[i][e_z_rot] = Interpolate(A_clock, 0.03, 0.43, 180, 0);
    #declare robot_info[i][e_z_leg_rot] = Interpolate(A_clock, 0.03, 0.43, -90, 90);
    #break
  #range (0.43, 0.45)  // lower leg
    compute_arms(
      i,
      0.0,
      0.0,
      (4.7 - Interpolate(A_clock, 0.43, 0.45, 3.6, 3.5)),
      2.3,
      1.0,
      (4.7 - Interpolate(A_clock, 0.43, 0.45, 3.6, 3.5) - extra_z )
    )
    #declare robot_info[i][e_z_rot] = 0;
    #declare robot_info[i][e_z_leg_rot] = 90;
    #break
  #range (0.45, 0.50)  // insert hip joint
    compute_arms(
      i,
      Interpolate(A_clock, 0.45, 0.50, 0.0, 1.7),
      0.0,
      (4.7 - 3.5),
      2.3,
      1.0,
      (4.7 - 3.5 - extra_z )
    )
    #declare robot_info[i][e_z_rot] = 0;
    #declare robot_info[i][e_z_leg_rot] = 90;
    #break
  #range (0.50, 0.52) // Pull arms back
    compute_arms(
      i,             
      Interpolate(A_clock, 0.5, 0.52, 1.7, 1.2),
      0.0,  
      (4.7 -3.5),
      Interpolate(A_clock, 0.5, 0.52, 2.3, 1.8),
      1.0,  
      (4.7 - 3.5)
    )                           
    #declare robot_info[i][e_l_thumb] = Interpolate(A_clock, 0.5, 0.52, 0, 30);
    #declare robot_info[i][e_r_thumb] = Interpolate(A_clock, 0.5, 0.52, 0, 30);
    #declare robot_info[i][e_z_head] = Interpolate(A_clock, 0.5, 0.52, 0, 20);
    #declare robot_info[i][e_z_eye] = Interpolate(A_clock, 0.5, 0.52, 0, 20);
    #declare robot_info[i][e_z_rot] = 0;
    #declare robot_info[i][e_z_leg_rot] = 90;
    #declare robot_info[i][e_x_pos] = robot_x_start;
    #declare robot_info[i][e_y_pos] = robot_y_start;
    #break
  #range (0.52, 0.62) //Turn and start down aisle B
    compute_arms(
      i,             
      1.2,
      Interpolate(A_clock, 0.52, 0.62, 0.0, 0.45),
      (4.7 -3.5),
      1.8,
      Interpolate(A_clock, 0.52, 0.62, 1.0, 0.25),  
      (4.7 - 3.5)
    )                           
    #declare robot_info[i][e_z_rot] = Interpolate(A_clock, 0.52, 0.62, 0, robot_dir);
    #declare robot_info[i][e_z_leg_rot] = robot_dir - robot_info[i][e_z_rot];
    #declare robot_info[i][e_z_head] = Interpolate(A_clock, 0.52, 0.62, 20, 0);
    #declare robot_info[i][e_z_eye] = Interpolate(A_clock, 0.52, 0.62, 20, 0);
    #declare robot_info[i][e_l_thumb] = 30;
    #declare robot_info[i][e_r_thumb] = 30;
    #declare robot_info[i][e_x_pos] = robot_x_start + (x_dist * (0.5+ sin(Interpolate(A_clock, 0.52, 0.96, -pi/2, pi/2))/2));
    #declare robot_info[i][e_y_pos] = robot_y_start + (y_dist * (0.5+ sin(Interpolate(A_clock, 0.52, 0.96, -pi/2, pi/2))/2));
    #break        
  #range (0.62, 0.86) // Cruise down aisle B
    compute_arms(
      i,             
      Interpolate(A_clock, 0.62, 0.86, 1.2, 1.6)
      0.45,
      (4.7 -3.5),
      Interpolate(A_clock, 0.62, 0.86, 1.8, 1.6),
      0.25,  
      (4.7 - 3.5)
    )                           
    #declare robot_info[i][e_x_pos] = robot_x_start + (x_dist * (0.5+ sin(Interpolate(A_clock, 0.52, 0.96, -pi/2, pi/2))/2));
    #declare robot_info[i][e_y_pos] = robot_y_start + (y_dist * (0.5+ sin(Interpolate(A_clock, 0.52, 0.96, -pi/2, pi/2))/2));
    #declare robot_info[i][e_z_rot] = robot_dir;
    #declare robot_info[i][e_l_thumb] = 30;
    #declare robot_info[i][e_r_thumb] = 30;
    #break        
  #range (0.86, 0.96) //Slow dow, turn into position
    compute_arms(
      i,             
      1.6,
      Interpolate(A_clock, 0.86, 0.96, 0.45, 0.25),
      (4.7 -3.5),
      Interpolate(A_clock, 0.86, 0.96, 1.6, 1.2),
      Interpolate(A_clock, 0.86, 0.96, 0.25, 0.15), 
      (4.7 - 3.5)
    )                           
    #declare robot_info[i][e_x_pos] = robot_x_start + (x_dist * (0.5+ sin(Interpolate(A_clock, 0.52, 0.96, -pi/2, pi/2))/2));
    #declare robot_info[i][e_y_pos] = robot_y_start + (y_dist * (0.5+ sin(Interpolate(A_clock, 0.52, 0.96, -pi/2, pi/2))/2));
    #declare robot_info[i][e_z_rot] = Interpolate(A_clock, 0.86, 0.96, robot_dir, 180);
    #declare robot_info[i][e_z_leg_rot] = robot_dir - robot_info[i][e_z_rot];
    #declare robot_info[i][e_l_thumb] = 30;
    #declare robot_info[i][e_r_thumb] = 30;
    #break        
  #range (0.96, 1.00) //extend arms
    compute_arms(
      i,             
      Interpolate(A_clock, 0.96, 1.0, 1.6, 2.1),
      0.25,
      (4.7 - 3.5),
      Interpolate(A_clock, 0.96, 1.0, 1.2, 2.2),
      0.15,  
      (4.7 - 3.5)
    )                           
    #declare robot_info[i][e_z_rot] = 180;
    #declare robot_info[i][e_z_leg_rot] = -90;
    #declare robot_info[i][e_l_thumb] = Interpolate(A_clock, 0.96, 1.0, 30, 0);
    #declare robot_info[i][e_r_thumb] = Interpolate(A_clock, 0.96, 1.0, 30, 0);
    #declare robot_info[i][e_x_pos] = robot_x_end;
    #declare robot_info[i][e_y_pos] = robot_y_end;
  #break        
  #else
    #error "Out of Range";
#end


//robot 19 This guy adds the left arm
#declare i = 19;
#declare robot_info[i][e_exist] = true;
#declare robot_info[i][e_power] = true;
#declare robot_x_start = aisle_b + 0.5;
#declare robot_x_end   = aisle_b;
#declare robot_y_start = last_station + (2 * station_dist) - 0.5;
#declare robot_y_end   = last_station + (0 * station_dist);
#declare x_dist = robot_x_end - robot_x_start;
#declare y_dist = robot_y_end - robot_y_start;
#declare se_dir = atan2 (x_dist, y_dist);
#declare robot_dir = 90 - se_dir * (cos(Interpolate(B_clock, 0.52, 0.96, 0, 2*pi))- 1);
#declare robot_info[i][e_x_pos] = robot_x_start;
#declare robot_info[i][e_y_pos] = robot_y_start;
#declare robot_info[i][e_z_pos] = 4;
#declare robot_info[i][e_color_index] = mod(ci_offset+A_cycle_count+2,12);

#switch(A_clock)
  #range (0, 0.05)    // pick up the left arm
    compute_arms(
      i,
      (Interpolate(A_clock, 0, 0.05, 2.1, 1.1)),
      0.25,
      (4.7 - Interpolate(A_clock, 0, 0.05, 3.5, 3.7)),
      (Interpolate(A_clock, 0, 0.05, 2.2, 1.2)),
      0.15,
      (4.7 - Interpolate(A_clock, 0, 0.05, 3.5, 3.7))
    )
    #declare robot_info[i][e_z_rot] = 180;
    #declare robot_info[i][e_z_leg_rot] = -90;
    #break
  #range (0.05, 0.45)  // spin around
    compute_arms(
      i,
      1.1,
      0.25,
      (4.7- Interpolate(A_clock, 0.05, 0.45, 3.7, 3.5)),
      1.2,
      0.15,
      (4.7 - Interpolate(A_clock, 0.05, 0.45, 3.7, 3.5))
    )
    #declare robot_info[i][e_z_rot] = Interpolate(A_clock, 0.05, 0.45, 180, 0);
    #declare robot_info[i][e_z_leg_rot] = Interpolate(A_clock, 0.05, 0.45, -90, 90);
    #break
  #range (0.45, 0.50)  // insert left arm into torso    
    compute_arms(
      i,             
      Interpolate(A_clock, 0.45, 0.5, 1.1, 2.1),
      0.25,  
      (4.7 - Interpolate(A_clock, 0.45, 0.5, 3.7, 3.5)),
      Interpolate(A_clock, 0.45, 0.5, 1.2, 2.2),
      0.15,  
      (4.7 - Interpolate(A_clock, 0.45, 0.5, 3.7, 3.5))
    )
    #declare robot_info[i][e_z_rot] = 0;
    #declare robot_info[i][e_z_leg_rot] = 90;
    #break
  #range (0.50, 0.52) // Pull arms back
    compute_arms(
      i,             
      Interpolate(A_clock, 0.5, 0.52, 2.1, 1.6),
      0.15,  
      (4.7 -3.5),
      Interpolate(A_clock, 0.5, 0.52, 2.1, 1.6),
      0.25,  
      (4.7 - 3.5)
    )                           
    #declare robot_info[i][e_l_thumb] = Interpolate(A_clock, 0.5, 0.52, 0, 30);
    #declare robot_info[i][e_r_thumb] = Interpolate(A_clock, 0.5, 0.52, 0, 30);
    #declare robot_info[i][e_z_head] = Interpolate(A_clock, 0.5, 0.52, 0, 20);
    #declare robot_info[i][e_z_eye] = Interpolate(A_clock, 0.5, 0.52, 0, 20);
    #declare robot_info[i][e_z_rot] = 0;
    #declare robot_info[i][e_z_leg_rot] = 90;
    #declare robot_info[i][e_x_pos] = robot_x_start;
    #declare robot_info[i][e_y_pos] = robot_y_start;
    #break
  #range (0.52, 0.62) //Turn and start down aisle B
    compute_arms(
      i,             
      1.6,
      Interpolate(A_clock, 0.52, 0.62, 0.15, 0.45),
      (4.7 -3.5),
      1.6,
      0.25,  
      (4.7 - 3.5)
    )                           
    #declare robot_info[i][e_z_rot] = Interpolate(A_clock, 0.52, 0.62, 0, robot_dir);
    #declare robot_info[i][e_z_leg_rot] = robot_dir - robot_info[i][e_z_rot];
    #declare robot_info[i][e_z_head] = Interpolate(A_clock, 0.52, 0.62, 20, 0);
    #declare robot_info[i][e_z_eye] = Interpolate(A_clock, 0.52, 0.62, 20, 0);
    #declare robot_info[i][e_l_thumb] = 30;
    #declare robot_info[i][e_r_thumb] = 30;
    #declare robot_info[i][e_x_pos] = robot_x_start + (x_dist * (0.5+ sin(Interpolate(A_clock, 0.52, 0.96, -pi/2, pi/2))/2));
    #declare robot_info[i][e_y_pos] = robot_y_start + (y_dist * (0.5+ sin(Interpolate(A_clock, 0.52, 0.96, -pi/2, pi/2))/2));
    #break        
  #range (0.62, 0.86) // Cruise down aisle B
    compute_arms(
      i,             
      1.6
      0.45,
      (4.7 -Interpolate(A_clock, 0.62, 0.86, 3.5, 3.975)),
      1.6,
      0.25,  
      (4.7 -Interpolate(A_clock, 0.62, 0.86, 3.5, 3.975))
    )                           
    #declare robot_info[i][e_x_pos] = robot_x_start + (x_dist * (0.5+ sin(Interpolate(A_clock, 0.52, 0.96, -pi/2, pi/2))/2));
    #declare robot_info[i][e_y_pos] = robot_y_start + (y_dist * (0.5+ sin(Interpolate(A_clock, 0.52, 0.96, -pi/2, pi/2))/2));
    #declare robot_info[i][e_l_thumb] = 30;
    #declare robot_info[i][e_r_thumb] = 30;
    #declare robot_info[i][e_z_rot] = robot_dir;
    #break        
  #range (0.86, 0.96) //Slow dow, turn into position
    compute_arms(
      i,             
      1.6
      0.45,
      (4.7 -3.975),
      1.6,
      0.25,  
      (4.7 - 3.975)
    )                           
    #declare robot_info[i][e_x_pos] = robot_x_start + (x_dist * (0.5+ sin(Interpolate(A_clock, 0.52, 0.96, -pi/2, pi/2))/2));
    #declare robot_info[i][e_y_pos] = robot_y_start + (y_dist * (0.5+ sin(Interpolate(A_clock, 0.52, 0.96, -pi/2, pi/2))/2));
    #declare robot_info[i][e_z_rot] = Interpolate(A_clock, 0.86, 0.96, robot_dir, 180);
    #declare robot_info[i][e_z_leg_rot] = robot_dir - robot_info[i][e_z_rot];
    #declare robot_info[i][e_l_thumb] = 30;
    #declare robot_info[i][e_r_thumb] = 30;
    #break        
  #range (0.96, 1.00) //extend arms
    compute_arms(
      i,             
      Interpolate(A_clock, 0.96, 1.0, 1.6, 2.4),
      Interpolate(A_clock, 0.96, 1.0, 0.45, 0.4),
      (4.7 -3.975),
      Interpolate(A_clock, 0.96, 1.0, 1.6, 2.4),
      Interpolate(A_clock, 0.96, 1.0, 0.25, 0.4), 
      (4.7 - 3.975)
    )                           
    #declare robot_info[i][e_z_rot] = 180;
    #declare robot_info[i][e_z_leg_rot] = -90;
    #declare robot_info[i][e_l_thumb] = Interpolate(A_clock, 0.96, 1.0, 30, 0);
    #declare robot_info[i][e_r_thumb] = Interpolate(A_clock, 0.96, 1.0, 30, 0);
    #declare robot_info[i][e_x_pos] = robot_x_end;
    #declare robot_info[i][e_y_pos] = robot_y_end;
  #break        
  #else
    #error "Out of Range";
#end

//robot 20 This guy adds the front panel
#declare i = 20;
#declare robot_info[i][e_exist] = true;
#declare robot_info[i][e_power] = true;
#declare robot_x_start = aisle_b;
#declare robot_y_start = last_station + (0 * station_dist);
#declare robot_info[i][e_x_pos] = robot_x_start;
#declare robot_info[i][e_y_pos] = robot_y_start;
#declare robot_info[i][e_z_pos] = 4;
#declare robot_info[i][e_color_index] = mod(ci_offset+A_cycle_count+1,12);
#ifdef (debug_robot)
  #declare spline_file = "Robot20a.spl"
  #declare spline_smoothness = 25;
    #declare spline_object = sphere {<0, 0, 0>, 0.1
      pigment {rgb <0, 0, 1>} finish {ambient 1 diffuse 0}
      no_shadow}
  
  object {#include "ShowSpl.inc"}
  #declare spline_file = "Robot20b.spl"
  #declare spline_smoothness = 25;
    #declare spline_object = sphere {<0, 0, 0>, 0.1
      pigment {rgb <1, 0, 1>} finish {ambient 1 diffuse 0}
      no_shadow}
  
  object {#include "ShowSpl.inc"}
#end
#switch(A_clock)
  #range (0, 0.05)    // pick up the panel
    compute_arms(
      i,
      (Interpolate(A_clock, 0, 0.05, 2.4, 2.1)),
      0.4,
      (4.7 - 0.475 - Interpolate(A_clock, 0, 0.05, 3.5, 3.7)),
      (Interpolate(A_clock, 0, 0.05, 2.4, 2.1)),
      0.4,
      (4.7 - 0.475 - Interpolate(A_clock, 0, 0.05, 3.5, 3.7))
    )
    #declare robot_info[i][e_y_r_wrist] = robot_info[i][e_y_r_wrist] + Interpolate(A_clock, 0, 0.05, 0, 45);
    #declare robot_info[i][e_y_l_wrist] = robot_info[i][e_y_l_wrist] + Interpolate(A_clock, 0, 0.05, 0, 45);
    #declare robot_info[i][e_z_rot] = 180;
    #declare robot_info[i][e_z_leg_rot] = -90;
    #break
  #range (0.05, 0.435)  // spin around
    compute_arms(
      i,
      2.1,
      0.4,
      (4.7 - 0.475 - 3.7),
      2.1,
      0.4,
      (4.7 - 0.475 - 3.7),
    )
    #declare robot_info[i][e_y_r_wrist] = robot_info[i][e_y_r_wrist] + 45;
    #declare robot_info[i][e_y_l_wrist] = robot_info[i][e_y_l_wrist] + 45;
    #declare robot_info[i][e_z_rot] = Interpolate(A_clock, 0.05, 0.435, 180, 0);
    #declare robot_info[i][e_z_leg_rot] = Interpolate(A_clock, 0.05, 0.435, -90, 90);
    #break
  #range (0.435, 0.48)  // lower
    compute_arms(
      i,             
      Interpolate(A_clock, 0.435, 0.48, 2.1, 2.4),
      0.4,  
      (4.7 - 0.475 - 3.7),
      Interpolate(A_clock, 0.435, 0.48, 2.1, 2.4),
      0.4,  
      (4.7 - 0.475 - 3.7)
    )
    #declare robot_info[i][e_y_r_wrist] = robot_info[i][e_y_r_wrist] + Interpolate(A_clock, 0.435, 0.48, 45, 0);
    #declare robot_info[i][e_y_l_wrist] = robot_info[i][e_y_l_wrist] + Interpolate(A_clock, 0.435, 0.48, 45, 0);
    #declare robot_info[i][e_z_rot] = 0;
    #declare robot_info[i][e_z_leg_rot] = 90;
    #break
  #range (0.48, 0.50) // Pull arms back (Note: the panel drops at 0.49)
    compute_arms(
      i,             
      Interpolate(A_clock, 0.48, 0.50, 2.4, 2.0),
      0.4,  
      (4.7 - 0.475-3.7),
      Interpolate(A_clock, 0.48, 0.50, 2.4, 2.0),
      0.4,  
      (4.7 - 0.475-3.7)
    )                           
    #declare robot_info[i][e_l_thumb] = Interpolate(A_clock, 0.48, 0.50, 0, 30);
    #declare robot_info[i][e_r_thumb] = Interpolate(A_clock, 0.48, 0.50, 0, 30);
    #declare robot_info[i][e_z_head] = Interpolate(A_clock, 0.48, 0.50, 0, 20);
    #declare robot_info[i][e_z_eye] = Interpolate(A_clock, 0.48, 0.50, 0, 20);
    #declare robot_info[i][e_z_rot] = 0;
    #declare robot_info[i][e_z_leg_rot] = 90;
    #break
  #range (0.50, 0.98) //Turn and start down aisle B 
    #declare spline_clock = 0.5 + sin(Interpolate(A_clock, 0.50, 0.98, -pi/2, pi/2))/2;
    //#render concat("Spline Clock is ",str(spline_clock, 7,4),"\n")
    #include "Robot20a.spl"
    //#render concat("Spline Position is ",vectToStr(spline_pos, 7,4),"\n")
    #declare robot_info[i][e_x_pos] = spline_pos.x;
    #declare robot_info[i][e_y_pos] = spline_pos.y;
    #declare robot_info[i][e_z_pos] = spline_pos.z;
    #if (spline_clock < 0.99)
      #declare spline_clock = spline_clock + 0.01; 
      #include "Robot20a.spl"
      #declare go_dir = spline_pos;
    #else  
      #declare go_dir = <11.0,    5,  4>;
    #end
    //#render concat("go dir = ",vectToStr(go_dir, 7,4),"\n")
    #declare robot_dir = 90 + r2d*atan2((go_dir.x-robot_info[i][e_x_pos]), (robot_info[i][e_y_pos]-go_dir.y));
    #switch (A_clock)
      #range (0.50, 0.62) //Turn and start down aisle B 
        compute_arms(
          i,             
          Interpolate(A_clock, 0.50, 0.62, 2.0, 1.5),
          Interpolate(A_clock, 0.50, 0.62, 0.4, 0.0),
          Interpolate(A_clock, 0.50, 0.62, (4.7 - 0.475-3.7), 1.5),
          Interpolate(A_clock, 0.50, 0.62, 2.0, 1.5),
          Interpolate(A_clock, 0.50, 0.62, 0.4, 0.0),
          Interpolate(A_clock, 0.50, 0.62, (4.7 - 0.475-3.7), 1.5)
        )                           
        #declare robot_info[i][e_z_rot] = Interpolate(A_clock, 0.50, 0.62, 0, robot_dir);
        #declare robot_info[i][e_z_leg_rot] = robot_dir - robot_info[i][e_z_rot];
        #declare robot_info[i][e_z_head] = Interpolate(A_clock, 0.50, 0.62, 20, 0);
        #declare robot_info[i][e_z_eye] = Interpolate(A_clock, 0.50, 0.62, 20, 0);
        #declare robot_info[i][e_l_thumb] = Interpolate(A_clock, 0.50, 0.62, 30, 0);
        #declare robot_info[i][e_r_thumb] = Interpolate(A_clock, 0.50, 0.62, 30, 0);
        #break        
      #range (0.62, 0.98) // Cruise down aisle B
        #declare robot_info[i][e_y_r_elbow] = 90;
        #declare robot_info[i][e_y_l_elbow] = 90;
        #declare robot_info[i][e_z_rot] = robot_dir;
        #break        
    #end
    #break
  #range (0.98, 1.00) //back into the box
    #declare spline_clock = 0.5 + sin(Interpolate(A_clock, 0.98, 1.20, -pi/2, pi/2))/2;
    //#render concat("Spline Clock is ",str(spline_clock, 7,4),"\n")
    #include "Robot20b.spl"
    //#render concat("Spline Position is ",vectToStr(spline_pos, 7,4),"\n")
    #declare robot_info[i][e_x_pos] = spline_pos.x;
    #declare robot_info[i][e_y_pos] = spline_pos.y;
    #declare robot_info[i][e_z_pos] = spline_pos.z;
    #if (spline_clock > 0.01)
      #declare spline_clock = spline_clock - 0.01; 
      #include "Robot20b.spl"
      #declare go_dir = spline_pos;
    #else  
      #declare go_dir = <11.0,    5,  4>;
    #end
    //#render concat("go dir = ",vectToStr(go_dir, 7,4),"\n")
    #declare robot_info[i][e_z_rot] = 90 + r2d*atan2((go_dir.x-robot_info[i][e_x_pos]), (robot_info[i][e_y_pos]-go_dir.y));
    #declare robot_info[i][e_x_r_elbow] = Interpolate(A_clock, 0.98, 1.20, 0, -20);
    #declare robot_info[i][e_x_l_elbow] = Interpolate(A_clock, 0.98, 1.20, 0, 20);
    #declare robot_info[i][e_y_r_elbow] = 90;
    #declare robot_info[i][e_y_l_elbow] = 90;
    #if (robot_info[i][e_x_pos] > 17.5)
      #declare robot_info[i][e_z_pos] = 4.0625;
      #declare robot_info[i][e_ankle_angle] = -7;
    #end
    #if (robot_info[i][e_x_pos] > 18.5)
      #declare robot_info[i][e_z_pos] = 4.125;
      #declare robot_info[i][e_ankle_angle] = 0;
    #end
  #break
  #else
    #error "Out of Range";
#end

//robot 21 This robot closes the box and then gets in
#declare i = 21;
#declare robot_info[i][e_exist] = true;
#declare robot_info[i][e_power] = true;
#declare robot_info[i][e_x_pos] = 19;
#declare robot_info[i][e_y_pos] = 7;
#declare robot_info[i][e_z_pos] = 4.125;
#declare robot_info[i][e_color_index] = mod(ci_offset+A_cycle_count+0,12);
#switch (A_clock)
  #range (0.0, 0.2) //Back up
    #declare robot_info[i][e_x_r_elbow] = Interpolate(A_clock, -0.02, 0.20, 0, -20);
    #declare robot_info[i][e_x_l_elbow] = Interpolate(A_clock, -0.02, 0.20, 0, 20);
    #declare robot_info[i][e_y_r_elbow] = 90;
    #declare robot_info[i][e_y_l_elbow] = 90;
    #declare spline_clock = 0.5 + sin(Interpolate(A_clock, -0.02, 0.20, -pi/2, pi/2))/2;
    //#render concat("Spline Clock is ",str(spline_clock, 7,4),"\n")
    #include "Robot20b.spl"
    //#render concat("Spline Position is ",vectToStr(spline_pos, 7,4),"\n")
    #declare robot_info[i][e_x_pos] = spline_pos.x;
    #declare robot_info[i][e_y_pos] = spline_pos.y;
    #declare robot_info[i][e_z_pos] = spline_pos.z;
    #if (spline_clock > 0.01)
      #declare spline_clock = spline_clock - 0.01; 
      #include "Robot20b.spl"
      #declare go_dir = spline_pos;
    #else  
      #declare go_dir = <11.0,    5,  4>;
    #end
    //#render concat("go dir = ",vectToStr(go_dir, 7,4),"\n")
    #declare robot_info[i][e_z_rot] = 90 + r2d*atan2((go_dir.x-robot_info[i][e_x_pos]), (robot_info[i][e_y_pos]-go_dir.y));
    #switch (robot_info[i][e_x_pos])
      #range(16.0, 16.5)  //hit the ramp
      #declare robot_info[i][e_z_pos] = 4 + Interpolate(robot_info[i][e_x_pos],16.5, 17.0, 8/128, 9/128);
      #declare robot_info[i][e_ankle_angle] = Interpolate(robot_info[i][e_x_pos],16.5, 17.0, -1.5, -2.25);
      #break
      #range(16.5, 17.0)
      #declare robot_info[i][e_z_pos] = 4 + Interpolate(robot_info[i][e_x_pos],16.5, 17.0, 9/128, 11/128);
      #declare robot_info[i][e_ankle_angle] = Interpolate(robot_info[i][e_x_pos],16.5, 17.0, -2, -3);
      #break
      #range(17.0, 17.5)
      #declare robot_info[i][e_z_pos] = 4 + Interpolate(robot_info[i][e_x_pos],17.0, 17.5, 11/128, 13/128);
      #declare robot_info[i][e_ankle_angle] = -3;
      #break
      #range(17.5, 18.0)
      #declare robot_info[i][e_z_pos] = 4 + Interpolate(robot_info[i][e_x_pos],17.5, 18.0, 13/128, 15/128);
      #declare robot_info[i][e_ankle_angle] = Interpolate(robot_info[i][e_x_pos],17.5, 18.0, -3, -1.5);
      #break
      #range(18.0, 18.5)
      #declare robot_info[i][e_z_pos] = 4 + Interpolate(robot_info[i][e_x_pos],18.0, 18.5, 15/128, 16/128);
      #declare robot_info[i][e_ankle_angle] = Interpolate(robot_info[i][e_x_pos],18.0, 18.5, -3, 0);
      #break
      #range(18.5, 21.5)
      #declare robot_info[i][e_z_pos] = 4 + 1/8;
      #declare robot_info[i][e_ankle_angle] = 0;
    #end
  #break      
  #range (0.20, 0.215) //Tap box doors
    #declare start_t = 0.20;
    #declare end_t =  0.215;      
    #declare robot_info[i][e_x_r_elbow] = Interpolate(A_clock, start_t, end_t, -20,  4);
    #declare robot_info[i][e_x_l_elbow] = Interpolate(A_clock, start_t, end_t,  20, -4);
    #declare robot_info[i][e_y_r_elbow] = 90;
    #declare robot_info[i][e_y_l_elbow] = 90;
    #break
  #range (0.215, 0.22) // Retract arms
    #declare start_t = 0.215;
    #declare end_t =  0.22;      
    #declare robot_info[i][e_x_r_elbow] = Interpolate(A_clock, start_t, end_t,  4, 0);
    #declare robot_info[i][e_x_l_elbow] = Interpolate(A_clock, start_t, end_t, -4, 0);
    #declare robot_info[i][e_y_r_elbow] = Interpolate(A_clock, start_t, 0.28,  90, 0);
    #declare robot_info[i][e_y_l_elbow] = Interpolate(A_clock, start_t, 0.28,  90, 0);
    #break
  #range (0.22, 0.28) // Retract arms
    #declare start_t = 0.22;
    #declare end_t =  0.28;      
    #declare robot_info[i][e_y_r_elbow] = Interpolate(A_clock, 0.215, end_t,  90, 0);
    #declare robot_info[i][e_y_l_elbow] = Interpolate(A_clock, 0.215, end_t,  90, 0);
    #if (A_clock > 0.24)
      #declare robot_info[i][e_power] = false;
    #end
    #break
  #range (0.28, 1.0) //Power off
    #declare robot_info[i][e_power] = false;
    #declare robot_info[i][e_exist] = false;
    #break
  #else
    #error "Out of Range";
#end

//Box 1
#declare box_start = 15;
#declare box_end = 11;
#declare box_dist = box_start - box_end;
#switch (A_clock)
  #range (0.0, 0.31)
    #declare box_y = box_start;
    #break
  #range (0.31, 0.7)
    #declare box_y = box_start - box_dist*(sin(Interpolate(A_clock, 0.31, 0.7, -pi/2, pi/2))+1)/2;
    #break
  #range (0.7, 1.0)
    #declare box_y = box_end;
    #break
#end
object {
  RobotBox (90, 75, 93)
  translate <19, box_y ,0>
}

//Box 2
#declare box_start = 11;
#declare box_end = 7;
#declare box_dist = box_start - box_end;
#switch (A_clock)
  #range (0.0, 0.31)
    #declare box_y = box_start;
    #break
  #range (0.31, 0.7)
    #declare box_y = box_start - box_dist*(sin(Interpolate(A_clock, 0.31, 0.7, -pi/2, pi/2))+1)/2;
    #break
  #range (0.7, 1.0)
    #declare box_y = box_end;
    #break
#end
object {
  RobotBox (90, 75, 93)
  translate <19, box_y ,0>
}

//Box 3
#declare box_start = 7;
#declare box_end = 3;
#declare box_dist = box_start - box_end;
#switch (A_clock)
  #range (0.0, 0.215)
    #declare box_y = box_start;
    #declare side_flap = 90;
    #declare top_flap = 75;
    #declare bottom_flap = 93;
    #break
  #range (0.215, 0.26)
    #declare start_t = 0.215;
    #declare end_t =  0.26;      
    #declare box_y = box_start;
    #declare side_flap = 90 + max(0, (20*sin(Interpolate(A_clock, start_t, 0.235, 0, pi))));
    #declare top_flap = Interpolate(A_clock, start_t, end_t, 75, 0 );
    #declare bottom_flap = Interpolate(A_clock, start_t, end_t, 93, 0);
    #break
  #range (0.26, 0.30)
    #declare start_t = 0.26;
    #declare end_t =  0.30;      
    #declare box_y = box_start;
    #declare side_flap = Interpolate(A_clock, start_t, end_t, 90, 0);
    #declare top_flap = 0;
    #declare bottom_flap = 0;
    #break
  #range (0.30, 0.31)
    #declare start_t = 0.30;
    #declare end_t =  0.31;      
    #declare box_y = box_start;
    #declare side_flap = 0;
    #declare top_flap = 0;
    #declare bottom_flap = 0;
    #break
  #range (0.31, 0.7)
    #declare box_y = box_start - box_dist*(sin(Interpolate(A_clock, 0.31, 0.7, -pi/2, pi/2))+1)/2;
    #declare side_flap = 0;
    #declare top_flap = 0;
    #declare bottom_flap = 0;
    #break
  #range (0.7, 1.0)
    #declare box_y = box_end;
    #declare side_flap = 0;
    #declare top_flap = 0;
    #declare bottom_flap = 0;
    #break
#end
object {
  RobotBox (side_flap, top_flap, bottom_flap)
  translate <19, box_y ,0>
}

//Box 4
#declare box_start = 3;
#declare box_end = -1;
#declare box_dist = box_start - box_end;
#switch (A_clock)
  #range (0.0, 0.31)
    #declare box_y = box_start;
    #break
  #range (0.31, 0.7)
    #declare box_y = box_start - box_dist*(sin(Interpolate(A_clock, 0.31, 0.7, -pi/2, pi/2))+1)/2;
    #break
  #range (0.7, 1.0)
    #declare box_y = box_end;
    #break
#end
object {
  RobotBox (0, 0, 0)
  translate <19, box_y ,0>
}



// Create the Robots
#declare i = 0;
#while (i < robot_cnt)
  #declare r_pos[i] = <robot_info[i][e_x_pos], robot_info[i][e_y_pos], robot_info[i][e_z_pos]>;
  #if (robot_info[i][e_exist])   
    #render concat("Creating Robot number ",str(i, 2, 0)," at ", vectToStr(r_pos[i], 7 ,4), 
                    " z_rot= ", str(robot_info[i][e_z_rot], 8, 4), " color ", str(robot_info[i][e_color_index], 2, 0), "\n")
    #switch (i)
    #case (1)
      object {
        Torso(robot_info[i][e_color_index])
        array_transform(i)
      }
    #break
    #case (2)
      object {
        Leg(   robot_info[i][e_leg_length],                                                             //leg length
               robot_info[i][e_y_leg_angle],                                                            //hip angle
               <0,                            robot_info[i][e_ankle_angle], robot_info[i][e_z_leg_rot]>,   //lower leg rotation
               robot_info[i][e_color_index]
        )
        array_transform(i)
      }
    #break
    #case (3)
      object {
        RightArm(
               <robot_info[i][e_x_r_shoulder], robot_info[i][e_y_r_shoulder], 0>,                         //r_shoulder_vec,
               <robot_info[i][e_x_r_elbow],   robot_info[i][e_y_r_elbow],   0>,                         //r_elbow_vec,
               robot_info[i][e_len_r_arm],                                                              //r_length,
               <robot_info[i][e_x_r_wrist],   robot_info[i][e_y_r_wrist],   robot_info[i][e_z_r_wrist]>,//r_hand_vec,     
               robot_info[i][e_r_thumb],                                                                 //r_thumb_angle,
               robot_info[i][e_color_index]
        )
        array_transform(i)
      }
    #break
    #case (4)
      object {    
        LeftArm(
               <robot_info[i][e_x_l_shoulder], robot_info[i][e_y_l_shoulder], 0>,                         //l_shoulder_vec,
               <robot_info[i][e_x_l_elbow],   robot_info[i][e_y_l_elbow],   0>,                         //l_elbow_vec,
               robot_info[i][e_len_l_arm],                                                              //l_length,
               <robot_info[i][e_x_l_wrist],   robot_info[i][e_y_l_wrist],   robot_info[i][e_z_l_wrist]>,//l_hand_vec,     
               robot_info[i][e_l_thumb],                                                                 //l_thumb_angle,
               robot_info[i][e_color_index]
        )
        array_transform(i)
      }
    #break
    #case (5)
      object {
        Head(  robot_info[i][e_power],                                                                  //power
               robot_info[i][e_z_head],                                                                 //head rotation
               robot_info[i][e_z_eye],                                                                  //eye rotation
               robot_info[i][e_color_index]                                                                  //eye rotation
        ) 
        array_transform(i)
      }
    #break
    #case (6)
      object {
        FrontPanel(robot_info[i][e_power])
        array_transform(i)
      }
     #break

    #case (11)
      object {
        Head(  robot_info[i][e_power],                                                                  //power
               robot_info[i][e_z_head],                                                                 //head rotation
               robot_info[i][e_z_eye],                                                                  //eye rotation
               robot_info[i][e_color_index]
        ) 
        array_transform(i)
      }
    // No #break
    #case (10)
      object {    
        LeftArm(
               <robot_info[i][e_x_l_shoulder], robot_info[i][e_y_l_shoulder], 0>,                         //l_shoulder_vec,
               <robot_info[i][e_x_l_elbow],   robot_info[i][e_y_l_elbow],   0>,                         //l_elbow_vec,
               robot_info[i][e_len_l_arm],                                                              //l_length,
               <robot_info[i][e_x_l_wrist],   robot_info[i][e_y_l_wrist],   robot_info[i][e_z_l_wrist]>,//l_hand_vec,     
               robot_info[i][e_l_thumb],                                                                 //l_thumb_angle,
               robot_info[i][e_color_index]
        )
        array_transform(i)
      }
    // No #break
    #case (9)
      object {
        RightArm(
               <robot_info[i][e_x_r_shoulder], robot_info[i][e_y_r_shoulder], 0>,                         //r_shoulder_vec,
               <robot_info[i][e_x_r_elbow],   robot_info[i][e_y_r_elbow],   0>,                         //r_elbow_vec,
               robot_info[i][e_len_r_arm],                                                              //r_length,
               <robot_info[i][e_x_r_wrist],   robot_info[i][e_y_r_wrist],   robot_info[i][e_z_r_wrist]>,//r_hand_vec,     
               robot_info[i][e_r_thumb],                                                                 //r_thumb_angle,
               robot_info[i][e_color_index]
        )
        array_transform(i)
      }
    // No #break
    #case (8)
      object {
        Leg(   robot_info[i][e_leg_length],                                                             //leg length
               robot_info[i][e_y_leg_angle],                                                            //hip angle
               <0,                            robot_info[i][e_ankle_angle], robot_info[i][e_z_leg_rot]>,   //lower leg rotation
               robot_info[i][e_color_index]
        )
        array_transform(i)
      }
      object {
        HipJoint
        array_transform(i)
      }
    // No #break
    #case (7)
      object {
        Torso(robot_info[i][e_color_index])
        array_transform(i)
      }
    #break

    #else
      object {
      MrRobot( robot_info[i][e_power],                                                                  //power
               robot_info[i][e_z_head],                                                                 //head rotation
               robot_info[i][e_z_eye],                                                                  //eye rotation
               robot_info[i][e_leg_length],                                                             //leg length
               robot_info[i][e_y_leg_angle],                                                            //hip angle
               <0,                            robot_info[i][e_ankle_angle], robot_info[i][e_z_leg_rot]>,   //lower leg rotation
               <robot_info[i][e_x_r_shoulder], robot_info[i][e_y_r_shoulder], 0>,                         //r_shoulder_vec,
               <robot_info[i][e_x_r_elbow],   robot_info[i][e_y_r_elbow],   0>,                         //r_elbow_vec,
               robot_info[i][e_len_r_arm],                                                              //r_length,
               <robot_info[i][e_x_r_wrist],   robot_info[i][e_y_r_wrist],   robot_info[i][e_z_r_wrist]>,//r_hand_vec,     
               robot_info[i][e_r_thumb],                                                                //r_thumb_angle,
               <robot_info[i][e_x_l_shoulder], robot_info[i][e_y_l_shoulder], 0>,                         //l_shoulder_vec,
               <robot_info[i][e_x_l_elbow],   robot_info[i][e_y_l_elbow],   0>,                         //l_elbow_vec,
               robot_info[i][e_len_l_arm],                                                              //l_length,
               <robot_info[i][e_x_l_wrist],   robot_info[i][e_y_l_wrist],   robot_info[i][e_z_l_wrist]>,//l_hand_vec,     
               robot_info[i][e_l_thumb],                                                                //l_thumb_angle,
               robot_info[i][e_color_index])
        array_transform(i)
      }  
    #end
  #end
  #ifdef (debug_robot) 
    union {
      cylinder { -1*x,  1*x,  0.033
        pigment {red 1}
      }
      cylinder { -1*y,  1*y,  0.033
        pigment {green 1}
      }
      cylinder { -1*z,  1*z,  0.033
        pigment {blue 1}
      }
      rotate    <robot_info[i][e_x_rot], robot_info[i][e_y_rot], robot_info[i][e_z_rot]>
      translate <robot_info[i][e_x_pos], robot_info[i][e_y_pos], robot_info[i][e_z_pos]>
    }              
  #end
  #declare i = i + 1;
#end
        
// Set up the camera

#declare spline_segments = 1;
#declare spline_loop = false;
#switch (belt_cycle_count)
  #case (0)  //S->B: Torso on the elevator
    #declare cam_loc = <8, 33, 5.5>;
    #declare cam_look_at = <r_pos[1].x, r_pos[1].y, 3.5>;
    #declare cam_angle = 18;
    #break
  #case (1)  //B->A: Torso goes from elvator to belt
    #declare cam_loc = <belt_center, Interpolate(belt_clock, 0, 1, 33, 31), 5.5>;
    #declare cam_look_at = r_pos[1];
    #declare cam_angle = Interpolate(B_clock, 0, 0.5, 18, 30);
    #break
  #case (2)  //A->B: Torso move down belt as leg comes from elevator
    #declare spline_clock = A_clock *2;
    #declare point0 = <8,   32, 5.5>;
    #declare point1 = <8,   31, 5.5>;
    #declare point2 = <7, 27.5, 6.5>;
    #declare point3 = <7, 26.5, 6.6>;
    #include "Spline.inc"
    #declare cam_loc = spline_pos;

    #declare point0 = <8,  41, 3.5>;
    #declare point1 = <8,  40, 3.5>;
    #declare point2 = <10,  35, 3.6>;
    #declare point3 = <10,  34, 3.6>;
    #include "Spline.inc"
    #declare cam_look_at = spline_pos;

    #declare cam_angle = Interpolate(A_clock, 0, 0.5, 30, 40);
    #break
  #case (3)  //B->A: T&L move down belt, RA comes over
    #declare spline_clock = B_clock *2;
    #declare point0 = <7,  28.5, 6.4>;
    #declare point1 = <7,  27.5, 6.5>;
    #declare point2 = <9,  24,   7.0>;
    #declare point3 = <9,  23,   7.0>;
    #include "Spline.inc"
    #declare cam_loc = spline_pos;

    #declare point0 = <10,  36, 3.6>;
    #declare point1 = <10,  35, 3.6>;
    #declare point2 = < 6,  30, 3.9>;
    #declare point3 = < 6,  29, 3.9>;
    #include "Spline.inc"
    #declare cam_look_at = spline_pos;

    #declare cam_angle = 40;
    #break
  #case (4)  //A->B: T,L&RA move down belt, LA comes over
    #declare spline_clock = A_clock *2;
    #declare point0 = <9,  25, 7.0>;
    #declare point1 = <9,  24, 7.0>;
    #declare point2 = <7,  19, 7.0>;
    #declare point3 = <7,  18, 7.0>;
    #include "Spline.inc"
    #declare cam_loc = spline_pos;

    #declare point0 = < 6,  31, 3.9>;
    #declare point1 = < 6,  30, 3.9>;
    #declare point2 = <10,  25, 4.0>;
    #declare point3 = <10,  24, 4.0>;
    #include "Spline.inc"
    #declare cam_look_at = spline_pos;

    #declare cam_angle = Interpolate(A_clock, 0, 0.5, 40, 42.5);
    #break
  #case (5)  //B->A: T,L,RA & LA move down belt. H comes over
    #declare spline_clock = B_clock *2;
    #declare point0 = <7,  20, 7.0>;
    #declare point1 = <7,  19, 7.0>;
    #declare point2 = <9,  15.5, 8>;
    #declare point3 = <9,  14.5, 8>;
    #include "Spline.inc"
    #declare cam_loc = spline_pos;

    #declare spline_segments = 1;
    #declare point0 = <10,  26, 4.0>;
    #declare point1 = <10,  25, 4.0>;
    #declare point2 = < 6.1,  20.5, 3.5>;
    #declare point3 = < 6.2,  19.5, 3.5>;
    #include "Spline.inc"
    #declare cam_look_at = spline_pos;

    #declare cam_angle = Interpolate(B_clock, 0, 0.5, 42.5, 45);
    #break
  #case (6)  //A->B: T,L,RA,LA move down belt. FP comes over
    #declare spline_clock = A_clock *2;
    #declare point0 = <9,  16.5, 8.0>;
    #declare point1 = <9,  15.5, 8.0>;
    #declare point2 = <5,  10, 7.0>;
    #declare point3 = <4.9,   9, 7.0>;
    #include "Spline.inc"
    #declare cam_loc = spline_pos;

    #declare point0 = < 6.0,  21.5, 3.5>;
    #declare point1 = < 6.1,  20.5, 3.5>;
    #declare point2 = <10,  15, 4.2>;
    #declare point3 = <10,  14, 4.2>;
    #include "Spline.inc"
    #declare cam_look_at = spline_pos;

    #declare cam_angle = 45;
    #break
  #case (7)  //B->A: Roll off the belt.
    #declare spline_clock = B_clock *2;
    #declare spline_segments = 2;
    #declare point0 = <5.2,  11, 7.0>;
    #declare point1 = <5.0,  10, 7.0>;
    #declare point2 = <5.0,   5, 7.5>;
    #declare point3 = <5.1,  4.5, 7.55>;
    #include "Spline.inc"
    #declare point2 = <7.3,  1.5, 7.0>;
    #declare point3 = <7.5,  1.0, 7.0>;
    #include "Spline.inc"
    #declare cam_loc = spline_pos;

    #declare spline_segments = 2;
    #declare point0 = < 10, 16, 4.2>;
    #declare point1 = < 10, 15, 4.2>;
    #declare point2 = < 8.5, 13, 3.0>;
    #declare point3 = < 8.5, 12, 3.0>;
    #include "Spline.inc"
    #declare point2 = < 7.8,  10, 3.5>;
    #declare point3 = < 7.8,   9, 3.5>;
    #include "Spline.inc"
    #declare cam_look_at = spline_pos;

    #declare cam_angle = 45;
    #break
  #case (8)  //A->B: Follow the Blue Guy(12) to the head
    #declare spline_clock = A_clock; //traverse this spline in a whole cycle this time
    #declare spline_segments = 2;
    #declare point0 = <7.3,  2, 7.0>;
    #declare point1 = <7.3,  1.5, 7.0>;
    #declare point2 = <7.85, 0.2, 7.5>;
    #declare point3 = <7.90, 0.2, 7.5>;
    #include "Spline.inc"
    #declare point2 = <10,    18.0, 7.0>;
    #declare point3 = <10.05, 19.0, 7.0>;
    #include "Spline.inc"
    #declare cam_loc = spline_pos;
    
    #declare cam_look_at = Interpolate(A_clock, 0, 0.5, < 7.8,  10, 3.5>, (r_pos[12] + 0.5*z));
    
    #declare cam_angle = 45;
    #break
  #case (9)  //B->A: Follow the Blue Guy(13) to the head
    #declare spline_clock = A_clock;  //same as previous
    #declare spline_segments = 2;
    #declare point0 = <7.3,  2, 7.0>;
    #declare point1 = <7.3,  1.5, 7.0>;
    #declare point2 = <7.85, 0.2, 7.5>;
    #declare point3 = <7.90, 0.2, 7.5>;
    #include "Spline.inc"
    #declare point2 = <10,    18.0, 7.0>;
    #declare point3 = <10.05, 19.0, 7.0>;
    #include "Spline.inc"
    #declare cam_loc = spline_pos;

    #declare cam_look_at = r_pos[13] + 0.5*z;

    #declare cam_angle = 45;
    #break

  #case (10) //A->B:           
    //This spline is used for the camera for the next 5 half cycles
    #declare spline_clock = A_clock / 2.5;
    #include "Cam10_14.spl"
    #declare cam_loc = spline_pos;
    #declare cam_look_at = r_pos[13] + Interpolate(A_clock, 0, 0.5, 0.5*z, 0);
    #declare cam_angle = 45;
    #break
  #case (11) //B->A:           
    //This spline is used for the camera for the next 4 half cycles
    #declare spline_clock = A_clock / 2.5;
    #include "Cam10_14.spl"
    #declare cam_loc = spline_pos;
    #declare cam_look_at = r_pos[14] + Interpolate(B_clock, 0, 0.5, 0, -0.5*z);
    #declare cam_angle = 45;
    #break
  #case (12) //A->B:           
    //This spline is used for the camera for the next 3 half cycles
    #declare spline_clock = 0.4 + (A_clock / 2.5);
    #include "Cam10_14.spl"
    #declare cam_loc = spline_pos;
    #declare cam_look_at = r_pos[14] - 0.5*z;
    #declare cam_angle = 45;
    #break
  #case (13) //B->A:           
    //This spline is used for the camera for the next 2 half cycles
    #declare spline_clock =0.4 + (A_clock / 2.5);
    #include "Cam10_14.spl"
    #declare cam_loc = spline_pos;
    #declare cam_look_at = r_pos[15] - 0.5*z;
    #declare cam_angle = 45;
    #break
  #case (14) //A->B:           
    //This spline is used for the camera for the next 1 half cycles
    #declare spline_clock = 0.8 + (A_clock / 2.5);
    #include "Cam10_14.spl"
    #declare cam_loc = spline_pos;
    #declare cam_look_at = r_pos[15] - 0.5*z;
    #declare cam_angle = 45;
    #break
  #case (15) //A->B:           
    //This spline is used for the camera for the next 1 half cycles
    #declare cam_loc = <10, 46, 6.8>;
    #declare cam_look_at = r_pos[16] - 0.5*z;
    #declare cam_angle = 45;
    #break
  #case (16) //A->B:           
    //This spline is used for the camera for the next 3 half cycles
    #declare spline_clock = A_clock/0.9;
    #include "Cam16_17.spl"
    #declare cam_loc = spline_pos;
    #declare cam_look_at = r_pos[16] + z*(-0.5 + sin(Interpolate(spline_clock, 0, 1, 0, pi)));
    #declare cam_angle = Interpolate(A_clock, 0, 0.5, 45, 50);
    #break
  #case (17) //B->A:           
    //This spline is used for the camera for the next 2 half cycles
    #declare spline_clock = min(1.0,A_clock/0.9);
    #include "Cam16_17.spl"
    #declare cam_loc = spline_pos;
    #declare cam_look_at = r_pos[17] + z*(-0.5 + sin(Interpolate(spline_clock, 0, 1, 0, pi)));
    #declare cam_angle = 50;
    #break
  #case (18) //A->B: 
     //This spline is used for the camera for the next 1 half cycles
    #declare spline_clock = (A_clock / 2.5);
    #include "Cam18_22.spl"
    #declare cam_loc = spline_pos;
    #declare cam_look_at = r_pos[18] -0.5*z;
    #declare cam_angle = 50;
    #break
  #case (19) //B->A: 
    //This spline is used for the camera for the next 1 half cycles
    #declare spline_clock = (A_clock / 2.5);
    #include "Cam18_22.spl"
    #declare cam_loc = spline_pos;
    #declare cam_look_at = r_pos[18] -0.5*z;
    #declare cam_angle = 50;
    #break
  #case (20) //A->B: 
    //This spline is used for the camera for the next 1 half cycles
    #declare spline_clock = 0.4 + (A_clock / 2.5);
    #include "Cam18_22.spl"
    #declare cam_loc = spline_pos;
    #declare cam_look_at = r_pos[19] -0.5*z;
    #declare cam_angle = 50;
    #break
  #case (21) //B->A: 
    //This spline is used for the camera for the next 1 half cycles
    #declare spline_clock = 0.4 + (A_clock / 2.5);
    #include "Cam18_22.spl"
    #declare cam_loc = spline_pos;
    #declare cam_look_at = r_pos[19] -0.5*z;
    #declare cam_angle = 50;
    #break
  #case (22) //A->B: 
    //This spline is used for the camera for the next 1 half cycles
    #declare spline_clock = 0.8 + (A_clock / 2.5);
    #include "Cam18_22.spl"
    #declare cam_loc = spline_pos;
    #declare cam_look_at = r_pos[20] -0.5*z;
    #declare cam_angle = Interpolate(A_clock, 0, 0.5, 50, 45);
    #break
  #case (23)  //B->A: Head for the box
    #declare spline_clock = B_clock *2;
    #declare spline_segments = 1;
    #declare point0 = <6.0,  8.0, 7.3>;
    #declare point1 = <6.0,  7.0, 7.3>;
    #declare point2 = <3.0,  3.0, 5.0>;
    #declare point3 = <2.9,  2.9, 4.9>;
    #include "Spline.inc"
    #declare cam_loc = spline_pos;
    #declare cam_look_at = r_pos[20] -0.5*z;
    #declare cam_angle = Interpolate(B_clock, 0, 0.5, 45, 40);
    #break
  #case (24)  //A->B: get in and roll away
  #case (25)  //B->A:   Zoom in and fade to credits.
    #declare cam_loc = <  3,  3,   5>;
    #switch (A_clock)
      #range (0.0, 0.2)
        #declare cam_look_at = r_pos[21] -0.5*z;
        #declare cam_angle = Interpolate(A_clock, 0, 0.2, 40, 38);
        #break
      #range (0.2, 0.31)
        #declare cam_look_at = Interpolate(A_clock, 0.2, 0.31, (r_pos[21] -0.5*z), < 18,   7,   3.6>);
        #declare cam_angle = 38;
        #break
      #range (0.31, 1.0)    
        #declare start_look_at = < 18,   7,   3.6>;
        #declare final_look_at = < 18, (7.655 -box_dist*(sin(Interpolate(A_clock, 0.31, 0.7, -pi/2, pi/2))+1)/2), 4.87>;
        //sphere { final_look_at, 0.05 
        //  pigment {Red}                         
        //}
        #switch (A_clock)
          #range(0.31, 0.44)
            #declare cam_look_at = start_look_at;
            #declare cam_angle = 38;
            #break
          #range(0.44, 0.46)
            #declare cam_look_at = Interpolate(A_clock, 0.44, 0.52, start_look_at, final_look_at);
            #declare cam_angle = 38;
            #break
          #range(0.46, 0.52)
            #declare cam_look_at = Interpolate(A_clock, 0.44, 0.52, start_look_at, final_look_at);
            #declare cam_angle = Interpolate(A_clock, 0.46, 0.532, 38, 3.95);
            #break
          #range(0.52, 0.532)
            #declare cam_look_at = final_look_at;
            #declare cam_angle = Interpolate(A_clock, 0.46, 0.532, 38, 3.95);
            #break
          #else
            #declare cam_look_at = final_look_at;
            #declare cam_angle = 3.95;
        #end
        #break
    #end
    #break 
  #else
    #error "Out of Range"
#end

/* Camera coordinate for testing.  Comment in or out as needed. */
//#declare cam_loc     = < 2,     1,   7   >;
//#declare cam_look_at = <17.50, 11.5, 2.75>;
//#declare cam_angle   = 40;

#render concat("Camera at ",vectToStr(cam_loc, 6 ,2), ," looking at ", 
                vectToStr(cam_look_at, 6 ,2), " angle ", str(cam_angle, 4, 1), "\n")
// The Camera
camera {  //  Camera StdCam
  location  cam_loc
  direction <        0.0,      1.8317, 0>
  sky       <    0.00000,     0.00000,     1.00000> // Use right handed-system 
  up        <        0.0,         0.0,         1.0> // Where Z is up
  right     <    1.33333,         0.0,         0.0> // Aspect ratio = 4/3
  angle     cam_angle
  look_at   cam_look_at
}




