
// =================================================
/*

  POV-Ray 3.5 Source

  Created by Richard Webster irtc_mail@yahoo.co.uk

  for the Internet Ray Tracing Competition

  April 2003 Animation Round www.irtc.org

*/
// =================================================

#include "poses.inc"
#include "transforms.inc"

// -------------------------------------------------------------------------------

// position and align an object aligned along axes

#macro Place_Object( ob, pos, v_dir, v_side )

  // initial direction <1,0,0>, initial side <0,1,0>

  // check direction alignment
  #local dot1 = vdot(<1,0,0>,v_dir);
  #local current_v_side = <0,1,0>;

  #if(abs(dot1) < 1)
     // object needs rotating
    #local angle1 = degrees(acos(dot1));
    #local axis1 = vcross(<1,0,0>,v_dir);
    #local ob_transform1 = Axis_Rotate_Trans(axis1, angle1);
    #local current_v_side = vnormalize(vaxis_rotate(current_v_side, axis1, angle1 ));
  #else
    // object already aligned along x-axis
    // but check for <-1,0,0> alignment and flip if so
    #if(dot1 < 0)
       #local ob_transform1 = Axis_Rotate_Trans(<0,1,0>, 180);
    #else
       // alignment is <1,0,0> so no flip required
       #local ob_transform1 = transform{};
    #end
  #end

  // align current side vector with required side vector
  #local dot2 = vdot(current_v_side, v_side);
  // rotation about v_dir required
  #if(abs(dot2) < 1)
    #local angle2 = degrees(acos(dot2));
    #local axis2 = vcross(current_v_side, v_side);
    #local ob_transform2 = Axis_Rotate_Trans(axis2, angle2);
  #else
    // current side vector already aligned along v_side
    // but check for negative alignment and flip if so
    #if(dot2 < 0)
      #local angle2 = 180;
      #local axis2 = v_dir;
      #local ob_transform2 = Axis_Rotate_Trans(axis2, angle2);
    #else
      #local ob_transform2 = transform{};
    #end

  #end

  object{ob
         transform ob_transform1
         transform ob_transform2 
         translate pos
  }
#end


// ------------------------------------------------------------------------
// rotate 3 vectors around V_axis by angle Theta in degrees

#macro rot( V0, V1, V2, V_axis, Theta )
  #if (abs(Theta) > 1e-6 )
    #declare V0 = vnormalize(vaxis_rotate(V0, V_axis, Theta ));
    #declare V1 = vnormalize(vaxis_rotate(V1, V_axis, Theta ));
    #declare V2 = vnormalize(vaxis_rotate(V2, V_axis, Theta ));
  #end
#end


// ----------------------------------------------------------------

// indices for pose parameters array

#declare n_pose_params = 26;

#declare i_head_turn = 0;
#declare i_head_swing = 1;
#declare i_head_tilt = 2;

#declare i_upper_back_bend = 3;
#declare i_upper_back_twist = 4;
#declare i_upper_back_tilt = 5;

#declare i_lower_back_bend = 6;
#declare i_lower_back_twist = 7;
#declare i_lower_back_tilt = 8;

#declare i_left_arm_swing = 9;
#declare i_left_arm_tilt = 10;
#declare i_left_arm_twist = 11;
#declare i_left_elbow_bend = 12;

#declare i_right_arm_swing = 13;
#declare i_right_arm_tilt = 14;
#declare i_right_arm_twist = 15;
#declare i_right_elbow_bend = 16;

#declare i_left_leg_swing = 17;
#declare i_left_leg_tilt = 18;
#declare i_left_knee_bend = 19;
#declare i_left_ankle_bend = 20;

#declare i_right_leg_swing = 21;
#declare i_right_leg_tilt = 22;
#declare i_right_knee_bend = 23;
#declare i_right_ankle_bend = 24;

#declare i_height_offset = 25;


// pose arrays

#declare pose = array[n_pose_params];
#declare pose1 = array[n_pose_params];
#declare pose2 = array[n_pose_params];
#declare pose3 = array[n_pose_params];
#declare pose4 = array[n_pose_params];

// ----------------------------------------------------------------

// make a posed figure

#macro make_figure( p, v_dir, v_side, v_up, fig_pose )

  // set dimensions

  #local upper_back_length = 0.45;
  #local lower_back_length = 0.2;
  #local shoulder_width = 0.4;
  #local neck_length =  0.08;
  #local lower_arm_length = 0.35;
  #local upper_arm_length = 0.35;
  #local hip_width = 0.17;
  #local upper_leg_length = 0.45;
  #local lower_leg_length = 0.45;

  // get pose parameters

  #local head_turn = fig_pose[i_head_turn];
  #local head_swing = fig_pose[i_head_swing];
  #local head_tilt = fig_pose[i_head_tilt];
  #local upper_back_bend = fig_pose[i_upper_back_bend];
  #local upper_back_twist = fig_pose[i_upper_back_twist];
  #local upper_back_tilt = fig_pose[i_upper_back_tilt];
  #local lower_back_bend = fig_pose[i_lower_back_bend];
  #local lower_back_twist = fig_pose[i_lower_back_twist];
  #local lower_back_tilt = fig_pose[i_lower_back_tilt];
  #local left_arm_swing = fig_pose[i_left_arm_swing];
  #local left_arm_tilt = fig_pose[i_left_arm_tilt];
  #local left_arm_twist = fig_pose[i_left_arm_twist];
  #local left_elbow_bend = fig_pose[i_left_elbow_bend];
  #local right_arm_swing = fig_pose[i_right_arm_swing];
  #local right_arm_tilt = fig_pose[i_right_arm_tilt];
  #local right_arm_twist = fig_pose[i_right_arm_twist];
  #local right_elbow_bend = fig_pose[i_right_elbow_bend];
  #local left_leg_swing = fig_pose[i_left_leg_swing];
  #local left_leg_tilt = fig_pose[i_left_leg_tilt];
  #local left_knee_bend = fig_pose[i_left_knee_bend];
  #local left_ankle_bend = fig_pose[i_left_ankle_bend];
  #local right_leg_swing = fig_pose[i_right_leg_swing];
  #local right_leg_tilt = fig_pose[i_right_leg_tilt];
  #local right_knee_bend = fig_pose[i_right_knee_bend];
  #local right_ankle_bend = fig_pose[i_right_ankle_bend];
  #local height_offset = fig_pose[i_height_offset];
 
  // ----------------------- vectors --------------------------

  // overall orientation
  #local v_dir_figure = v_dir;
  #local v_side_figure = v_side;
  #local v_up_figure = v_up;

  // upper back vectors
  #local v_dir_upper_back = v_dir_figure;
  #local v_side_upper_back = v_side_figure;
  #local v_up_upper_back = v_up_figure;
  rot(v_dir_upper_back, v_side_upper_back, v_up_upper_back, v_side_upper_back, upper_back_bend )
  rot(v_dir_upper_back, v_side_upper_back, v_up_upper_back, v_dir_upper_back, upper_back_tilt )
  rot(v_dir_upper_back, v_side_upper_back, v_up_upper_back, v_up_upper_back, upper_back_twist )

  // lower back vectors
  #local v_dir_lower_back = v_dir_figure;
  #local v_side_lower_back = v_side_figure;
  #local v_up_lower_back = v_up_figure;
  rot(v_dir_lower_back, v_side_lower_back, v_up_lower_back, v_side_lower_back, -lower_back_bend )
  rot(v_dir_lower_back, v_side_lower_back, v_up_lower_back, v_dir_lower_back, -lower_back_tilt )
  rot(v_dir_lower_back, v_side_lower_back, v_up_lower_back, v_up_lower_back, lower_back_twist )

  // head vectors
  #local v_dir_head = v_dir_upper_back;
  #local v_side_head = v_side_upper_back;
  #local v_up_head = v_up_upper_back;
  rot(v_dir_head, v_side_head, v_up_head, v_up_head, head_turn )
  rot(v_dir_head, v_side_head, v_up_head, v_side_head, head_swing )
  rot(v_dir_head, v_side_head, v_up_head, v_dir_head, head_tilt )

  // upper left arm vectors
  #local v_dir_left_upper_arm =  v_dir_upper_back;
  #local v_side_left_upper_arm = v_side_upper_back;
  #local v_up_left_upper_arm = v_up_upper_back;
  rot(v_dir_left_upper_arm, v_side_left_upper_arm, v_up_left_upper_arm,
      v_side_left_upper_arm, -left_arm_swing )
  rot(v_dir_left_upper_arm, v_side_left_upper_arm, v_up_left_upper_arm,
      v_dir_left_upper_arm, left_arm_tilt )
  rot(v_dir_left_upper_arm, v_side_left_upper_arm, v_up_left_upper_arm,
      v_up_left_upper_arm, left_arm_twist )

  // lower left arm vectors
  #local v_dir_left_lower_arm = v_dir_left_upper_arm;
  #local v_side_left_lower_arm = v_side_left_upper_arm;
  #local v_up_left_lower_arm = v_up_left_upper_arm;
  rot(v_dir_left_lower_arm, v_side_left_lower_arm, v_up_left_lower_arm,
      v_side_left_upper_arm, -left_elbow_bend )

  // upper right arm vectors
  #local v_dir_right_upper_arm = v_dir_upper_back; 
  #local v_side_right_upper_arm = v_side_upper_back;
  #local v_up_right_upper_arm = v_up_upper_back;
  rot(v_dir_right_upper_arm, v_side_right_upper_arm, v_up_right_upper_arm,
      v_side_right_upper_arm, -right_arm_swing )
  rot(v_dir_right_upper_arm, v_side_right_upper_arm, v_up_right_upper_arm,
      v_dir_right_upper_arm, -right_arm_tilt )
  rot(v_dir_right_upper_arm, v_side_right_upper_arm, v_up_right_upper_arm,
      v_up_right_upper_arm, right_arm_twist )
 
  // lower right arm vectors
  #local v_dir_right_lower_arm = v_dir_right_upper_arm;
  #local v_side_right_lower_arm = v_side_right_upper_arm;
  #local v_up_right_lower_arm = v_up_right_upper_arm;
  rot(v_dir_right_lower_arm, v_side_right_lower_arm, v_up_right_lower_arm,
      v_side_right_upper_arm, -right_elbow_bend )

  // left upper leg vectors
  #local v_dir_left_upper_leg = v_dir_lower_back;
  #local v_side_left_upper_leg = v_side_lower_back;
  #local v_up_left_upper_leg = v_up_lower_back;;
  rot(v_dir_left_upper_leg, v_side_left_upper_leg, v_up_left_upper_leg,
      v_side_left_upper_leg, -left_leg_swing )
  rot(v_dir_left_upper_leg, v_side_left_upper_leg, v_up_left_upper_leg,
      v_dir_left_upper_leg, left_leg_tilt )

  // left lower leg vectors
  #local v_dir_left_lower_leg = v_dir_left_upper_leg;
  #local v_side_left_lower_leg = v_side_left_upper_leg;
  #local v_up_left_lower_leg = v_up_left_upper_leg;
  rot(v_dir_left_lower_leg, v_side_left_lower_leg, v_up_left_lower_leg,
      v_side_left_lower_leg, left_knee_bend )

  // left foot vectors
  #local v_dir_left_foot = v_dir_left_lower_leg;
  #local v_side_left_foot = v_side_left_lower_leg;
  #local v_up_left_foot = v_up_left_lower_leg;
  rot(v_dir_left_foot, v_side_left_foot, v_up_left_foot, v_side_left_foot, left_ankle_bend )

  // right upper leg vectors
  #local v_dir_right_upper_leg = v_dir_lower_back;
  #local v_side_right_upper_leg = v_side_lower_back;
  #local v_up_right_upper_leg = v_up_lower_back;
  rot(v_dir_right_upper_leg, v_side_right_upper_leg, v_up_right_upper_leg,
      v_side_right_upper_leg, -right_leg_swing )
  rot(v_dir_right_upper_leg, v_side_right_upper_leg, v_up_right_upper_leg,
      v_dir_right_upper_leg, -right_leg_tilt )

  // right lower leg vectors
  #local v_dir_right_lower_leg = v_dir_right_upper_leg;
  #local v_side_right_lower_leg = v_side_right_upper_leg;
  #local v_up_right_lower_leg = v_up_right_upper_leg;
  rot(v_dir_right_lower_leg, v_side_right_lower_leg, v_up_right_lower_leg,
      v_side_right_lower_leg, right_knee_bend )

  // right foot vectors
  #local v_dir_right_foot = v_dir_right_lower_leg;
  #local v_side_right_foot = v_side_right_lower_leg;
  #local v_up_right_foot = v_up_right_lower_leg;
  rot(v_dir_right_foot, v_side_right_foot, v_up_right_foot, v_side_right_foot, right_ankle_bend )

  // ------------------- points ----------------------

  #local p_middle_back = p + <0,0,height_offset>;
  #local p_collar = p_middle_back + upper_back_length * v_up_upper_back;
  #local p_pelvis = p_middle_back - lower_back_length * v_up_lower_back;

  #local p_left_shoulder  = p_collar + 0.5*shoulder_width*v_side_upper_back;
  #local p_right_shoulder = p_collar - 0.5*shoulder_width*v_side_upper_back;

  #local p_head = p_collar + neck_length*v_up_upper_back;

  #local p_left_elbow = p_left_shoulder - upper_arm_length*v_up_left_upper_arm;
  #local p_left_wrist = p_left_elbow - lower_arm_length*v_up_left_lower_arm;

  #local p_right_elbow = p_right_shoulder - upper_arm_length*v_up_right_upper_arm;
  #local p_right_wrist = p_right_elbow - lower_arm_length*v_up_right_lower_arm;

  #local p_left_hip = p_pelvis + 0.5*hip_width*v_side_lower_back;
  #local p_left_knee = p_left_hip - upper_leg_length*v_up_left_upper_leg;
  #local p_left_ankle = p_left_knee - lower_leg_length*v_up_left_lower_leg;

  #local p_right_hip = p_pelvis - 0.5*hip_width*v_side_lower_back;
  #local p_right_knee = p_right_hip - upper_leg_length*v_up_right_upper_leg;
  #local p_right_ankle = p_right_knee - lower_leg_length*v_up_right_lower_leg;

  
  // ---------------------- human figure --------------------------


  // textures

  #declare P_suit = pigment{color rgb <0.8,0.0,0.0>};
  #declare P_strip = pigment{color rgb <0.0,0.0,0.0>};
  #declare F_suit = finish{ambient 0.4 diffuse 0.8 phong 1 phong_size 4};
  
  #declare T_strip = texture{
    pigment{planar scale 0.015 rotate z*0
            pigment_map{[0 P_suit][0.001 P_suit][0.002 P_strip][1 P_strip]}}
    finish{F_suit}
  };
  
  #declare T_suit = texture{pigment{P_suit}finish{F_suit}};


  // diver figure

  union{

  // upper back

  #declare upper_back = union{

      union{
        cone{<0,0,0>, 0.1 <0,0,0.9*upper_back_length>, 0.13}
        sphere{<0,0,0.9*upper_back_length>, 0.13}
        scale<1,1.5,1> 
        texture{T_strip}
      }

      // tanks
      union{
        cylinder{<-0.2,-0.08,0><-0.2,-0.08,0.5>,0.08}
        cylinder{<-0.2, 0.08,0><-0.2, 0.08,0.5>,0.08}
        sphere{<-0.2,-0.08,0.5>,0.08}
        sphere{<-0.2, 0.08,0.5>,0.08}
        texture{pigment{color rgb 0.5}
                finish{ambient 0.3 phong 1 phong_size 20 reflection 0.25
         }}
      }
    };

  Place_Object(upper_back, p_middle_back, v_dir_upper_back, v_side_upper_back)

  // middle

  #declare middle = sphere{<0,0,0>, 0.1 scale<1,1.5,1> texture{T_suit}}
  Place_Object(middle, p_middle_back, v_dir_figure, v_side_figure)


  // lower back

   #declare lower_back = union{
      cone{<0,0,0>, 0.1 <0,0,-0.14>, 0.12 scale<1,1.5,1>}
      sphere{<0,0,0>,0.12 scale<1.0,1.5,1.3> clipped_by{box{<-0.15,-0.15,-0.15>,<0.15,0.15,0.0>}}
             translate -0.14*z}
      texture{T_suit}
   };
  
  Place_Object(lower_back, p_middle_back, v_dir_lower_back, v_side_lower_back)
 
  // shoulder
  #declare shoulder = cylinder{<0,0,0> <0,-shoulder_width,0>, 0.06 texture{T_suit}}
  Place_Object(shoulder, p_left_shoulder, v_dir_upper_back, v_side_upper_back)


 // head and neck
  #declare head = union{

    // head
    blob{threshold 0.5
      sphere{<0,0,0.2>,2, 1  }
      sphere{<0.35,0,-0.5>,0.9, 1}
      scale <0.1,0.1,0.12>
      texture{T_suit}
    }

   // neck
   cylinder{<0,0,0> <0,0,-0.3>, 0.06 texture{T_suit}}


    // goggles
    union{

       // body
       cylinder{<0,0,0><0.12,0,0>, 0.08 texture{T_suit}}

       // glass
       cylinder{<0.12,0,0><0.121,0,0>, 0.08
                texture{pigment{color rgb 0}finish{ambient 1 reflection 0.5}}}

       // frame
       torus{0.08, 0.01 rotate 90*z translate<0.12,0,0>
             texture{pigment{color rgb 0}finish{F_suit}}}

       scale<1,1.2,0.75>
       translate<0,0,0.04>
    }

   // breathing gear
   union{

     cylinder{<0,-0.05,0><0,0.05,0>, 0.02 texture{pigment{color rgb 0}finish{F_suit}}}
     cylinder{<0,0,0><0.022,0,0>, 0.025 texture{pigment{color rgb 0.6}finish{F_suit}}}
     cylinder{<0,0,0><0.025,0,0>, 0.015 texture{pigment{color rgb 0}finish{F_suit}}}
     translate<0.11,0,-0.06>
   }

    translate<0,0,0.15>
   };
   

  Place_Object(head, p_head, v_dir_head, v_side_head)

  // left upper arm
  #declare left_upper_arm = union{
     sphere{<0,0,0>, 0.06}
     cone{<0,0,0>,0.06 <0,0,-upper_arm_length>, 0.04}
     texture{T_strip}};
  Place_Object(left_upper_arm, p_left_shoulder, v_dir_left_upper_arm, v_side_left_upper_arm)

  // left lower arm
  #declare left_lower_arm = union{
     sphere{<0,0,0>, 0.04}
     cone{<0,0,0>,0.04 <0,0,-lower_arm_length>, 0.03}

   // watch
   torus{0.03, 0.01 rotate 90*x scale<1,1,3> translate<0,0,-lower_arm_length*0.7>
             texture{pigment{color rgb 0}finish{F_suit}}}

     texture{T_strip}};
  Place_Object(left_lower_arm, p_left_elbow, v_dir_left_lower_arm, v_side_left_lower_arm)
  

  // left hand
  #declare left_hand = union{
     sphere{<-0.01,0,0>, 0.035}
     cylinder{<-0.01,0,0><0.01,0,0>, 0.035}
     sphere{<0.01,0,0>, 0.035}
     scale<1,1,1.2>
     texture{T_suit}};
  Place_Object(left_hand, p_left_wrist, v_dir_left_lower_arm, v_side_left_lower_arm)


  // right upper arm
  #declare right_upper_arm = union{
     sphere{<0,0,0>, 0.06}
     cone{<0,0,0>,0.06 <0,0,-upper_arm_length>, 0.04}
     texture{T_strip}};
  Place_Object(right_upper_arm, p_right_shoulder, v_dir_right_upper_arm, v_side_right_upper_arm)
  
  // right lower arm
  #declare right_lower_arm = union{
     sphere{<0,0,0>, 0.04}
     cone{<0,0,0>,0.04 <0,0,-lower_arm_length>, 0.03}
     texture{T_strip}};
  Place_Object(right_lower_arm, p_right_elbow, v_dir_right_lower_arm, v_side_right_lower_arm)

  // right hand
  #declare right_hand = union{
     sphere{<-0.01,0,0>, 0.035}
     cylinder{<-0.01,0,0><0.01,0,0>, 0.035}
     sphere{<0.01,0,0>, 0.035}
     scale<1,1,1.2>
     texture{T_suit}};
  Place_Object(right_hand, p_right_wrist, v_dir_right_lower_arm, v_side_right_lower_arm)

  // left upper leg
  #declare left_upper_leg = union{
    sphere{<0,0,0> 0.1}
    cone{<0,0,0>, 0.1 <0,0,-upper_leg_length>, 0.06 }
    texture{T_strip}};
  Place_Object(left_upper_leg, p_left_hip, v_dir_left_upper_leg, v_side_left_upper_leg)

  // left lower leg
  #declare left_lower_leg = union{
    sphere{<0,0,0>, 0.06}
    cone{<0,0,0>, 0.06 <0,0,-lower_leg_length>, 0.04 }
    sphere{<0,0,-lower_leg_length>, 0.04}
    sphere{<-0.02,0,-0.05>,0.06 scale<1,1,2.5>}    
    texture{T_strip}};
  Place_Object(left_lower_leg, p_left_knee, v_dir_left_lower_leg, v_side_left_lower_leg)

 #declare flipper = union{
      cylinder{<-0.03,0,0><-0.03,0,0.04> 0.03}
     sphere{<0,0,0>, 0.04 scale<3.5,1,1> translate <0.08,0,0>
             clipped_by{plane{-z, 0}}}
     cone{<-0.01,0,0>, 0.02 <0.3,0,0>, 0.09 scale<1,1,0.1>}
    translate<0,0,-0.05>
    texture{T_suit}};

  #declare left_foot = object{flipper};
  Place_Object(left_foot, p_left_ankle, v_dir_left_foot, v_side_left_foot)


  // right upper leg
  #declare right_upper_leg = union{
    sphere{<0,0,0> 0.1}
    cone{<0,0,0>, 0.1 <0,0,-upper_leg_length>, 0.06 }
    texture{T_strip}};
  Place_Object(right_upper_leg, p_right_hip, v_dir_right_upper_leg, v_side_right_upper_leg)

  // right lower leg
  #declare right_lower_leg = union{
    sphere{<0,0,0>, 0.06}
    cone{<0,0,0>, 0.06 <0,0,-lower_leg_length>, 0.04 }
    sphere{<0,0,-lower_leg_length>, 0.04}
     sphere{<-0.02,0,-0.05>,0.06 scale<1,1,2.5>}
    texture{T_strip}};
  Place_Object(right_lower_leg, p_right_knee, v_dir_right_lower_leg, v_side_right_lower_leg)
 
  // right foot
  #declare right_foot = object{flipper};
  Place_Object(right_foot, p_right_ankle, v_dir_right_foot, v_side_right_foot)

  } // end union


#end // end figure macro

