/*
   This file is a translation of the POV-Ray standard include file
   TEXTURES.INC into Polyray syntax.  They will look similar in almost
   all cases to what you would get from POV-Ray.  They will not be
   exactly the same, due to the differences in the way texturing is
   performed in the two renderers.
   
*/
/* Basic surface types */
define matte
surface {
   ambient 0.1
   diffuse 0.7
   }

define shiny
surface {
   ambient 0.1
   diffuse 0.6
   specular white, 0.6
   microfacet Phong 7
   }

// Functions that simulate POV-Ray internal texturing routines
define agate_fn ((sin(6.2832*(1.1*z+2.6*noise(P,<2.8,0.5,4>)))+1)/2)^0.77
define bozo_fn noise(P, 1)
define granite_fn noise(7*P, 5)
define leopard_fn ((sin(x)+sin(y)+sin(z))/3)^2
define marble_fn sawtooth(x + 3 * noise(P, 5))
define onion_fn fmod(|P|, 1)
define wood_fn sawtooth(sqrt(x*x+y*y))

// Colors used by POV-Ray that Polyray may not understand
define Brass       <0.71, 0.65, 0.26>
define BrightGold  <0.85, 0.85, 0.10>
define Bronze      <0.55, 0.47, 0.14>
define Copper      <0.72, 0.45, 0.20>
define DustyRose   <0.52, 0.39, 0.39>
define Feldspar    <0.82, 0.57, 0.46>
define Quartz      <0.85, 0.85, 0.95>
define Mica        black
define NeonPink    <1.00, 0.43, 0.78>
define OldGold     <0.81, 0.71, 0.23>
define Silver      <0.90, 0.91, 0.98>
define SummerSky   <0.22, 0.69, 0.87>
define YellowGreen <0.60, 0.80, 0.20>

//*****************************************************************************
//                      STONE PIGMENTS
//*****************************************************************************

// Drew Wells' superb Jade.  Color map works nicely with other textures, too.
define jade_color_map
   color_map([0, 0.8, <1, 0.6, 0.1>, <0, 0.3, 0>]
             [0.8, 1, <0.1, 0.5, 0.1>, <0, 0, 0.3>])
define Jade
texture {
   special shiny {
      color jade_color_map[marble_fn]
      position P + 1.8 * dnoise(P)
      }
   }

// Classic white marble with red veins.  Over-worked, like checkers.
define red_marble_map
   color_map([0, 0.8, <1, 0.8, 0.8>, <1, 0.4, 0.4>]
             [0.8, 1, <1, 0.4, 0.4>, <1, 0.1, 0.1>])
define Red_Marble
texture {
   special shiny {
      color red_marble_map[marble_fn]
      position P + dnoise(P)
      }
   }

// White marble with black veins.
define white_marble_map
   color_map([0, 0.8, 0.9*white, 0.5*white]
             [0.8, 1, 0.5*white, 0.2*white])
define White_Marble
texture {
   special shiny {
      color white_marble_map[marble_fn]
      position P + dnoise(P)
      }
   }

// Light blue and black marble with a thin red vein 
// Try changing LBlue and Vein below to modify the marble.
define LBlue <0.0, 0.6, 0.6>
define Vein <0.6, 0.0, 0.0>
define blood_marble_map
   color_map([0, 0.8, black, LBlue]
             [0.8, 0.9, LBlue, Vein]
             [0.8, 1, Vein, black])
define Blood_Marble
texture {
   special shiny {
      color blood_marble_map[marble_fn]
      position P + 2.3 * dnoise(P)
      }
   }

// a grey blue agate -- kind of purplish.
define blue_agate_map
   color_map(
     [0.0, 0.5,  <0.30, 0.30, 0.50>, <0.30, 0.30, 0.50>]
     [0.5, 0.55, <0.30, 0.30, 0.50>, <0.20, 0.20, 0.30>]
     [0.55, 0.6, <0.20, 0.20, 0.30>, <0.25, 0.25, 0.35>]
     [0.6, 0.7,  <0.25, 0.25, 0.35>, <0.15, 0.15, 0.26>]
     [0.7, 0.8,  <0.15, 0.15, 0.26>, <0.10, 0.10, 0.20>]
     [0.8, 0.9,  <0.10, 0.10, 0.20>, <0.30, 0.30, 0.50>]
     [0.9, 1.001,<0.30, 0.30, 0.50>, <0.10,0.10, 0.20>])
define Blue_Agate
texture { special shiny { color blue_agate_map[agate_fn] } }

// Deep blue agate -- almost glows.
define sapphire_agate_map
      color_map([0.0, 0.3, <0, 0, 0.9>, <0, 0, 0.8>]
                [0.3, 1,   <0, 0, 0.8>, <0, 0, 0.4>])
define Sapphire_Agate
texture { special shiny { color blue_agate_map[agate_fn] } }

// Brown and white agate -- very pretty.
define brown_agate_map
      color_map([0.0, 0.5, <0.9, 0.7, 0.6>, <0.9, 0.7, 0.6>]
                [0.5, 0.6, <0.9, 0.7, 0.6>, <0.9, 0.7, 0.4>]
                [0.6, 1.0, <0.9, 0.7, 0.4>, <0.7, 0.4, 0.2>])
define Brown_Agate
texture { special shiny { color brown_agate_map[agate_fn] } }

define pink_granite_map
      color_map([0.0, 0.4, black, black]
                [0.4, 0.45, Quartz, Quartz]
                [0.45, 0.5, Quartz, gray]
                [0.5, 0.55, gray, Feldspar]
                [0.55, 0.8, Feldspar, Feldspar]
                [0.8, 1.0, Feldspar, orange])
define Pink_Granite
texture { special shiny { color pink_granite_map[granite_fn] } }

//*****************************************************************************
//                      STONE TEXTURES
//*****************************************************************************

// Gray-pink alabaster or marble.  Layers are scaled for a unit object
// and relative to each other.
define pink_alabaster_map1
      color_map([0.0, 1.0, <0.9,0.75,0.75>, <0.6,0.6,0.6>])
define pink_alabaster_map2
     color_map(
       [0.0, 0.9, DustyRose, 1, DustyRose, 0.5]
       [0.9, 1.0, dark_slate_gray, dark_slate_gray])
define PinkAlabaster
texture {
   layered
      // Second layer texture has some filter values, yet a fair amount of color
      // Viening is kept quite thin in color map and by the largish scale.
      texture {
         special shiny {
            color pink_alabaster_map2[granite_fn]
            }
         scale <2,2,2>
         },

      // Underlying surface is very subtly mottled with bozo
      texture {
         special matte {
            color pink_alabaster_map1[bozo_fn]
            ambient 0.25
            position P + 0.25 * dnoise(P)
            }
         scale <0.4,0.4,0.4>
         }
   }

//*****************************************************************************
//                           SKY PIGMENTS
//*****************************************************************************

// Basic Blue Sky w/ clouds.
define blue_sky_map
   color_map([0.0, 0.5, <0.25, 0.25, 0.5>, <0.25, 0.25, 0.5>]
             [0.5, 0.6, <0.25, 0.25, 0.5>, 0.7*white]
             [0.6, 1.0, 0.7*white, 0.3*white])
define Blue_Sky
texture {
   special matte {
      color blue_sky_map[bozo_fn]
      position P + 0.3 * dnoise(P)
      }
   }

// Bright Blue Sky w/ very white clouds.
define bright_blue_sky_map
   color_map([0.0, 0.5, <0.5, 0.5, 1>, <0.5, 0.5, 1>]
             [0.5, 0.6, <0.5, 0.5, 1>, <1, 1, 1>]
             [0.6, 1.0, <1, 1, 1>, <0.5, 0.5, 0.5>])
define Bright_Blue_Sky
texture {
   special matte {
      color bright_blue_sky_map[noise(P)]
      position P + 0.3 * dnoise(P)
      }
   }

// Another sky
define blue_sky2_map
   color_map([0.0, 0.3, blue, blue]
             [0.3, 1.0, blue, white])
define Blue_Sky2
texture { special matte { color blue_sky2_map[agate_fn] }
          scale <0.75, 0.75, 0.75> }

// Small puffs of white clouds
define Blue_Sky3
texture {
   special matte {
      color blue_sky2_map[granite_fn]
      position P + 0.1 * dnoise(P)
      }
   scale <0.75, 0.75, 0.75>
   }

// Red sky w/ yellow clouds -- very surreal.
define blood_sky_map
   color_map(
     [0.0, 0.5, <0.9, 0.7, 0>, <0.3, 0.2, 0>]
     [0.5, 0.6, <0.6, 0.025, 0>, <0.9, 0.7, 0>]
     [0.6, 1.0, <0.6, 0.025, 0>, <0.6, 0.025, 0>])
define Blood_Sky
texture {
   special matte {
      color blood_sky_map[bozo_fn]
      position P + 0.5 * dnoise(P)
      }
   }

// Black sky with red and purple clouds 
// Try adding turbulence values from 0.1 - 5.0 -- CdW
define apocalypse_map
   color_map(
     [0.0, 0.4, <0.8, 0, 0>, <0.4, 0, 0.4>]
     [0.4, 0.6, <0.4, 0, 0.4>, <0, 0, 0.2>]
     [0.6, 1.0, <0, 0, 0.2>, black])
define Apocalypse
texture { special matte { color apocalypse_map[bozo_fn] } }

// White clouds w/ transparent sky.
define clouds_map
   color_map(
     [0.0, 0.1, 0.8*white, 0.8*white]
     [0.1, 0.5, 0.8*white, 0, black, 1]
     [0.5, 1.0, black, 1, black, 1])
define Clouds
texture { special matte { color clouds_map[bozo_fn] } }

//*****************************************************************************
//                           WOODEN PIGMENTS
//*****************************************************************************

// Several wooden pigments by Tom Price:
// A light reddish wood.
define cherry_wood_map
   color_map(
     [0.0, 0.8, <0.666, 0.312, 0.2>, <0.666, 0.312, 0.2>]
     [0.8, 1.0, <0.4, 0.133, 0.066>, <0.2, 0.065, 0.033>])
define Cherry_Wood
texture {
   special shiny {
      color cherry_wood_map[wood_fn]
      position P + 0.3 * dnoise(P)
      }
   }

// A light tan wood with greenish rings.
define pine_wood_map
   color_map(
     [0.0, 0.8, <1.0, 0.72, 0.25>, <1.0, 0.72, 0.25>]
     [0.8, 1.0, <0.5, 0.5, 0.066>, <0.4, 0.4, 0.033>])
define Pine_Wood
texture {
   special shiny {
      color pine_wood_map[wood_fn]
      position P + 0.2 * dnoise(P)
      }
   }

// Dark wood with a greenish hue to it.
define dark_wood_map
   color_map(
     [0.0, 0.8, <0.43, 0.24, 0.05>, <0.43, 0.24, 0.05>]
     [0.8, 1.0, <0.4, 0.33, 0.066>, <0.2, 0.033, 0.033>])
define Dark_Wood
texture {
   special shiny {
      color dark_wood_map[wood_fn]
      position P + 0.2 * dnoise(P)
      }
   }

// Light tan wood with brown rings.
define tan_wood_map
   color_map(
     [0.0, 0.8, <0.89, 0.6, 0.3>, <0.89, 0.6, 0.3>]
     [0.8, 1.0, <0.6, 0.4, 0.2>, <0.4, 0.3, 0.2>])
define Tan_Wood
texture {
   special shiny {
      color tan_wood_map[wood_fn]
      position P + 0.1 * dnoise(P)
      }
   }

// A very pale wood with tan rings -- kind of balsa-ish.
define white_wood_map
   color_map(
     [0.0, 0.8, <0.93, 0.71, 0.53>, <0.98, 0.81, 0.6>]
     [0.8, 1.0, <0.6, 0.33, 0.27>, <0.7, 0.6, 0.23>])
define White_Wood
texture {
   special shiny {
      color white_wood_map[wood_fn]
      position P + 0.6 * dnoise(P)
      }
   }

// Brown wood - looks stained.
define tom_wood_map
   color_map(
     [0.0, 0.8, <0.7, 0.3, 0.0>, <0.7, 0.3, 0.0>]
     [0.8, 1.0, <0.5, 0.2, 0.0>, <0.4, 0.1, 0.0>])
define Tom_Wood
texture {
   special shiny {
      color tom_wood_map[wood_fn]
      position P + 0.31 * dnoise(P)
      }
   }

// The scaling in these definitions is relative to a unit-sized object
// (radius 1).  Note that woods are functionally equivilent to a log lying
// along the z axis.  For best results, think like a woodcutter trying to
// extract the nicest board out of that log.  A little tilt along the x axis
// will give elliptical rings of grain like you'd expect to find on most
// boards.  Experiment.
// (The first five came from DODEC2.POV in the POV Scenefile Library.)
define DMFWood1_map
   color_map(
      [0.00, 0.90, <0.6, 0.3, 0.18>, <0.6, 0.15, 0.09>]
      [0.90, 1.00, <0.3, 0.15, 0.09>, <0.3, 0.15, 0.09>])
define DMFWood1
texture {
   special surface {
      color DMFWood1_map[wood_fn]
      position P + 0.04 * dnoise(P, 3)
      }
   scale <0.05, 0.05, 1>
   }

define DMFWood2_map
   color_map(
      [0.00, 0.10, <0.52, 0.37, 0.26>, <0.52, 0.37, 0.26>]
      [0.10, 0.90, <0.52, 0.37, 0.26>, <0.42, 0.26, 0.15>]
      [0.90, 1.00, <0.42, 0.26, 0.15>, <0.42, 0.26, 0.15>])
define DMFWood2
texture {
   special surface {
      color DMFWood2_map[wood_fn]
      position P + 0.03 * dnoise(P, 4)
      }
   scale <0.05, 0.05, 1>
   }

define DMFWood3_map
   color_map(
      [0.00, 0.10, <0.4, 0.13, 0.07>, <0.4, 0.13, 0.07>]
      [0.10, 0.90, <0.4, 0.13, 0.07>, <0.2, 0.07, 0.03>]
      [0.90, 1.00, <0.2, 0.07, 0.03>, <0.2, 0.07, 0.03>])
define DMFWood3
texture {
   special surface {
      color DMFWood3_map[wood_fn]
      position P + 0.05 * dnoise(P, 2)
      }
   scale <0.05, 0.05, 1>
   }
  
define DMFWood4_map
   color_map(
      [0.00, 0.10, <0.89, 0.6, 0.3>, <0.89, 0.6, 0.3>]
      [0.10, 0.90, <0.89, 0.6, 0.3>, <0.6, 0.4, 0.2>]
      [0.90, 1.00, <0.6, 0.4, 0.2>, <0.6, 0.4, 0.2>])
define DMFWood4
texture {
   special surface {
      color DMFWood4_map[wood_fn]
      position P + 0.04 * dnoise(P, 3)
      }
   scale <0.05, 0.05, 1>
   }

// Is this really oak?  I dunno.  Quite light, maybe more like spruce.
define DMFLight_Oak_map
   color_map(
      [0.00, 0.10, <0.42, 0.26, 0.15>, <0.42, 0.26, 0.15>]
      [0.10, 0.90, <0.42, 0.26, 0.15>, <0.52, 0.37, 0.26>]
      [0.90, 1.00, <0.52, 0.37, 0.26>, <0.52, 0.37, 0.26>])
define DMFLightOak
texture {
   special surface {
      color DMFLight_Oak_map[wood_fn]
      position P + 0.05 * dnoise(P)
      }
   scale <0.2, 0.2, 1>
   }

// Looks like old desk oak if used correctly.
define DMFDark_Oak_map
   color_map(
      [0.00, 0.10, <0.6, 0.3, 0.18>, <0.6, 0.3, 0.18>]
      [0.10, 0.90, <0.6, 0.3, 0.18>, <0.3, 0.15, 0.09>]
      [0.90, 1.00, <0.3, 0.15, 0.09>, <0.3, 0.15, 0.09>])
define DMFDarkOak
texture {
   special surface {
      color DMFDark_Oak_map[wood_fn]
      position P + 0.04 * dnoise(P, 3)
      }
   scale <0.2, 0.2, 1>
   }

// Wood by Eric Barish
define EMBWood1_map1
   color_map(
      [0.00, 0.34, <0.58, 0.45, 0.23>, <0.65, 0.45, 0.25>]
      [0.34, 0.40, <0.65, 0.45, 0.25>, <0.33, 0.23, 0.13>]
      [0.40, 0.47, <0.33, 0.23, 0.13>, <0.6, 0.4, 0.2>]
      [0.47, 1.00, <0.6, 0.4, 0.2>, <0.25, 0.15, 0.05>])
define EMBWood1_map2
   color_map(
      [0.00, 0.80, black, 1, <0, 0.1, 0.2>, 0.8]
      [0.80, 1.00, <0, 0.1, 0.2>, 0.8, <0.7, 0.8, 0.9>, 0.4])
define EMBWood1
texture {
   layered
      // top layer, adds small dark spots
      texture {
         special surface {
            color EMBWood1_map2[bozo_fn]
            }
         scale <0.25, 0.25, 0.25>
         },
      // Bottom wood-grain layer
      texture {
         special surface {
            color EMBWood1_map1[wood_fn]
            position P + 0.05 * dnoise(P)
            normal N + 0.05 * dnoise(P)
            ambient 0.32
            diffuse 0.63
            specular 0.2
            microfacet Phong 10
            }
         scale <0.2, 0.2, 1>
         }
   }


//   Doug Otwell woods
//   Yellow pine, close grained
//
define Yellow_Pine_map1
   color_map(
      [0.000, 0.22, <0.81,  0.67,  0.25>, <0.81,  0.67, 0.25>]
      [0.222, 0.34, <0.81,  0.67,  0.25>, <0.60,  0.35, 0.04>]
      [0.342, 0.39, <0.60,  0.35,  0.04>, <0.81,  0.67, 0.25>]
      [0.393, 0.71, <0.81,  0.67,  0.25>, <0.81,  0.67, 0.25>]
      [0.709, 0.82, <0.81,  0.67,  0.25>, <0.53,  0.30, 0.03>]
      [0.821, 1.00, <0.53,  0.30,  0.03>, <0.81,  0.671, 0.25>])
define Yellow_Pine_map2
   color_map(
      [0.00, 0.12, <1.00, 1.00, 1.00>, 1.00, <0.70, 0.41, 0.12>, 0.61]
      [0.12, 0.23, <0.70, 0.41, 0.12>, 0.61, <0.70, 0.47, 0.12>, 0.61]
      [0.23, 0.50, <0.70, 0.47, 0.12>, 0.61, <1.00, 1.00, 1.00>, 1.00]
      [0.50, 0.70, <1.00, 1.00, 1.00>, 1.00, <1.00, 1.00, 1.00>, 1.00]
      [0.70, 0.83, <1.00, 1.00, 1.00>, 1.00, <0.70, 0.47, 0.12>, 0.61]
      [0.83, 1.00, <0.70, 0.47, 0.12>, 0.61, <1.00, 1.00, 1.00>, 1.00])
define Yellow_Pine
texture {
   layered
      // Yellow_Pine layer 2
      texture {
         special shiny {
            color Yellow_Pine_map2[wood_fn]
            position P + 0.01 * dnoise(P)
            }
         scale <0.5,0.5,0.5>
         translate <10, 0, 0>
         },
      // Yellow_Pine layer 1
      texture {
         special shiny {
            color Yellow_Pine_map1[wood_fn]
            position P + 0.02 * dnoise(P)
            }
         scale <0.1,0.1,0.1>
         translate <10, 0, 0>
         }
   }

//
//   Rosewood
//
define Rosewood_map1
   color_map(
      [0.000, 0.26, <0.204, 0.110, 0.078>,  0, <0.231, 0.125, 0.090>,  0]
      [0.256, 0.39, <0.231, 0.125, 0.090>,  0, <0.247, 0.133, 0.090>,  0]
      [0.393, 0.59, <0.247, 0.133, 0.090>,  0, <0.204, 0.110, 0.075>,  0]
      [0.581, 0.73, <0.204, 0.110, 0.075>,  0, <0.259, 0.122, 0.102>,  0]
      [0.726, 0.98, <0.259, 0.122, 0.102>,  0, <0.231, 0.125, 0.086>,  0]
      [0.983, 1.00, <0.231, 0.125, 0.086>,  0, <0.204, 0.110, 0.078>,  0])
define Rosewood_map2
   color_map(
      [0.000, 0.14, <0.55, 0.35, 0.25>, 1.000, <0.00, 0.00, 0.00>, 0.004]
      [0.139, 0.15, <0.00, 0.00, 0.00>, 0.004, <0.00, 0.00, 0.00>, 0.004]
      [0.148, 0.29, <0.00, 0.00, 0.00>, 0.004, <0.55, 0.35, 0.25>, 1.000]
      [0.287, 0.44, <0.55, 0.35, 0.25>, 1.000, <0.55, 0.35, 0.25>, 1.000]
      [0.443, 0.63, <0.55, 0.35, 0.25>, 1.000, <0.00, 0.00, 0.00>, 0.004]
      [0.626, 0.64, <0.00, 0.00, 0.00>, 0.004, <0.00, 0.00, 0.00>, 0.004]
      [0.635, 0.84, <0.00, 0.00, 0.00>, 0.004, <0.55, 0.35, 0.25>, 1.000]
      [0.843, 1.00, <0.55, 0.35, 0.25>, 1.000, <0.55, 0.35, 0.25>, 1.000])
define Rosewood
texture {
   layered
      // Rosewood layer 2
      texture {
         special matte {
            color Rosewood_map2[wood_fn]
            position P + 0.04 * dnoise(P)
            }
         scale <0.5, 0.5, 1>
         translate <10, 0, 0>
         },
      // Rosewood layer 1
      texture {
         special matte {
            color Rosewood_map1[bozo_fn]
            ambient 0.5
            diffuse 0.8
            position P + 0.04 * dnoise(P)
            }
         scale <0.5, 0.5, 1>
         translate <10, 0, 0>
         }
   }

//
//   Sandalwood ( makes a great burled maple, too)
//
define Sandalwood_map1
   color_map(
      [0.00, 0.17,  <0.725, 0.659, 0.455>, 0, <0.682, 0.549, 0.420>, 0]
      [0.17, 0.27,  <0.682, 0.549, 0.420>, 0, <0.557, 0.451, 0.322>, 0]
      [0.27, 0.39,  <0.557, 0.451, 0.322>, 0, <0.725, 0.659, 0.455>, 0]
      [0.39, 0.56,  <0.725, 0.659, 0.455>, 0, <0.682, 0.549, 0.420>, 0]
      [0.56, 0.70,  <0.682, 0.549, 0.420>, 0, <0.482, 0.392, 0.278>, 0]
      [0.70, 1.00,  <0.482, 0.392, 0.278>, 0, <0.725, 0.659, 0.455>, 0])
define Sandalwood_map2
   color_map(
      [0.00, 0.09, <0.68, 0.60, 0.38>, 1.00, <0.76, 0.69, 0.60>, 0.02]
      [0.09, 0.23, <0.76, 0.69, 0.60>, 0.02, <0.64, 0.55, 0.33>, 1.00]
      [0.23, 0.35, <0.64, 0.55, 0.33>, 1.00, <0.76, 0.69, 0.60>, 0.02]
      [0.35, 0.50, <0.76, 0.69, 0.60>, 0.02, <0.68, 0.60, 0.38>, 1.00]
      [0.50, 0.57, <0.68, 0.60, 0.38>, 1.00, <0.76, 0.69, 0.60>, 0.02]
      [0.57, 0.66, <0.76, 0.69, 0.60>, 0.02, <0.68, 0.60, 0.38>, 1.00]
      [0.66, 0.84, <0.68, 0.60, 0.38>, 1.00, <0.76, 0.69, 0.60>, 0.02]
      [0.84, 1.00, <0.76, 0.69, 0.60>, 0.02, <0.68, 0.60, 0.38>, 1.00])
define Sandalwood
texture {
   layered
      // Sandalwood layer 2
      texture {
         special shiny {
            color Sandalwood_map2[bozo_fn]
            position P + 0.8 * dnoise(P)
            }
         scale <0.4, 0.4, 0.4>
         },
      // Sandalwood layer 1
      texture {
         special shiny {
            color Sandalwood_map1[bozo_fn]
            position P + 0.2 * dnoise(P)
            }
         scale <0.4, 0.4, 2>
         }
   }

//*****************************************************************************
//                     SURFACE FINISHES
//*****************************************************************************


// Dull creates a large, soft highlight on the object's surface
define Dull surface { specular 0.5 microfacet Cook 20 }

// Shiny creates a small, tight highlight on the object's surface
define Shiny surface { specular 1 microfacet Cook 3 }

// Phong highlights are less "realistic" than specular, but useful
// for different effects.
// Dull creates a large, soft highlight on the object's surface
define Phong_Dull surface { specular 0.5 microfacet Phong 20 }

// Shiny creates a small, tight highlight on the object's surface
define Phong_Shiny surface { specular 1 microfacet Phong 3 }

// Very shiny with very tight highlights and a fair amount of reflection
define Glossy surface { specular 1 microfacet Phong 3 reflection 0.13 }
define Phong_Glossy surface { specular 1 microfacet Phong 3 reflection 0.13 }

// Luminous for shadowless skies and light_sources.
define Luminous surface { ambient 1  diffuse 0 specular 0 reflection 0 }

// a perfectly mirrored finish with no highlights
define Mirror surface { ambient 1  diffuse 0 specular 0 reflection 1 }

//*****************************************************************************
//                           GLASS TEXTURES
//*****************************************************************************


define Glass_Finish_surf
Shiny {
   ambient 0
   diffuse 0
   reflection white, 0.1
   transmission white, 1, 1.5
   }

define Glass
texture {
   Glass_Finish_surf {
      color black
      transmission white, 0.7, 1.5
      }
   }

// Probably more of a "Plexiglas" than glass
define Glass2_surf
surface {
   color white
   ambient 0
   diffuse 0
   reflection 0.5
   transmission 0.85, 1.0
   specular 0.3
   microfacet Phong 5 // phong_size = 60
   }
define Glass2 texture { Glass2_surf }

// An excellent lead crystal glass!
define Glass3_surf
surface {
   color <0.98, 0.98, 0.98>
   ambient 0.1
   diffuse 0.1
   specular white, 0.8
   transmission white, 0.9, 1.45
   microfacet Phong 4
   }
define Glass3 texture { Glass3_surf }

define Green_Glass texture { Glass3_surf { color <0.8, 1, 0.95> } }

//*****************************************************************************
//                           METAL FINISHES
//*****************************************************************************

define Metal
surface {
   ambient 0.2
   diffuse 0.7
   brilliance 6
   reflection white, 0.25
   specular 0.75
   microfacet Phong 5 // phong_size 80
   }

//*****************************************************************************
//                           METAL TEXTURES
//*****************************************************************************


// Good looking "metal" textures 
// IMPORTANT: They REQUIRE that colors.inc be included BEFORE textures.inc!

define Chrome_Texture
texture {
   surface {
      color LightGray
      ambient 0.3
      diffuse 0.7
      reflection white, 0.15
      // brilliance 8            // Not valid before Polyray v1.8
      specular LightGray, 0.8
      microfacet Cook 5          // roughness 0.1
      }
   }

// A series of metallic textures using the Metal texture:
define Brass_Texture
texture { Metal { color Brass specular Brass, 0.75 } }
define Bronze_Texture
texture { Metal { color Bronze specular Bronze, 0.75 } }
define Copper_Texture
texture { Metal { color Copper specular Copper, 0.75 } }
define Gold_Texture
texture { Metal { color BrightGold specular BrightGold, 0.75 } }
define Silver_Texture
texture { Metal { color Silver specular Silver, 0.75 } }

define OldGoldMetal
texture { Metal { color OldGold specular OldGold, 0.75 } }

// In the future, please refer to Chrome_Texture by this name.  I'd like
// to scrap the old name someday. Ditto with other "_Texture" names!
define Chrome_Metal Chrome_Texture
define Brass_Metal  Brass_Texture
define Bronze_Metal Bronze_Texture
define Gold_Metal   Gold_Texture
define Bronze_Metal Bronze_Texture
define Copper_Metal Copper_Texture

// A couple highly reflective metal textures.
define Polished_Chrome
texture {
   surface {
      color 0.2 * white
      ambient 0.1
      diffuse 0.7
      reflection white, 0.6
      // brilliance 6            // Not valid before Polyray v1.8
      specular 0.2 * white, 0.8
      microfacet Phong 3         // phong_size 120
      }
   }

define Polished_Brass
texture {
   surface {
      color <0.578, 0.422, 0.195>
      ambient 0.1
      diffuse 0.8
      reflection white, 0.4
      // brilliance 6            // Not valid before Polyray v1.8
      specular <0.578, 0.422, 0.195>, 0.8
      microfacet Phong 3         // phong_size 120
      }
   }


// Beautiful military brass texture!
define New_Brass_surf
surface {
   color <0.7, 0.56, 0.37>
   ambient 0.35
   diffuse 1
   // brilliance 15            // Not valid before Polyray v1.8
   specular white, 0.41
   microfacet Phong 15         // phong_size 5
   }
define New_Brass texture { New_Brass_surf }

// Spun Brass texture for cymbals & such
define Spun_Brass
texture {
   noise New_Brass_surf {
      frequency 200
      normal 2
      bump_scale 0.35
      }
   }

define SilverFinish_surf
surface {
   ambient 0.25
   diffuse 0.65
   reflection white, 0.45
   // brilliance 6
   microfacet Phong 3 // phong_size 100
   }
define SilverFinish
texture { SilverFinish_surf }

// Each of these looks good.  Slightly bluer as you go down
define Silver1_Colour <0.94, 0.93, 0.83>
define Silver2_Colour <0.94, 0.93, 0.86>
define Silver3_Colour <0.94, 0.93, 0.90>

define Silver1
texture {
   SilverFinish_surf {
      color Silver1_Colour
      specular Silver1_Colour, 1
      }
   }

define Silver2
texture {
   SilverFinish_surf {
      color Silver2_Colour
      specular Silver2_Colour, 1
      }
   }

define Silver3
texture {
   SilverFinish_surf {
      color Silver3_Colour
      specular Silver3_Colour, 1
      }
   }

// Interesting texture -- Give it a try.
// Sort of a "Black Hills Gold", black, white, and orange specks or splotches.
define Brass_Valley_map
   color_map(
       [0.0, 0.3, Feldspar, Feldspar]
       [0.3, 0.6, Mica, Quartz]
       [0.6, 1.0, Feldspar, Quartz])
define Brass_Valley
texture {
   special surface {
      color Brass_Valley_map[granite_fn]
      specular Brass_Valley_map[granite_fn], 0.75
      // brilliance 6
      reflection white, 0.75
      microfacet Phong 10
      }
   }

define Rusty_Iron_map
   color_map(
     [0.0, 0.5, <0.21, 0.1, 0.1>, <0.25, 0.25, 0.01>]
     [0.5, 0.6, <0.25, 0.25, 0.01>, <0.3, 0.1, 0.1>]
     [0.6, 1.0, <0.15, 0.1, 0.1>, <0.15, 0.1, 0.1>])
define Rusty_Iron
texture {
   special surface {
      color Rusty_Iron_map[agate_fn]
      }
   }

define Rust_map
   color_map(
     [0.0, 0.4, <0.89, 0.51, 0.28>, <0.70, 0.13, 0>]
     [0.4, 0.5, <0.70, 0.13, 0>, <0.69, 0.41, 0.08>]
     [0.5, 0.6, <0.69, 0.41, 0.08>, <0.49, 0.31, 0.28>]
     [0.6, 1.0, <0.49, 0.31, 0.28>, <0.89, 0.51, 0.28>])
define Rust
texture {
   special surface {
      color Rust_map[bozo_fn]
      }
   }

//*****************************************************************************
//                    SPECIAL EFFECTS
//*****************************************************************************

// Red & white stripes - Looks best on a y axis Cylinder
// It "spirals" because it's gradient on two axis
define Candy_Cane_map
   color_map(
     [0.00, 0.25, red, red]
     [0.25, 0.75, white, white]
     [0.75, 1.00, red , red])
define Candy_Cane_surf
surface {
   color Candy_Cane_map[(x + y) - floor(x+y)]
   }
define Candy_Cane
texture {
   special Candy_Cane_surf {
      ambient 0.1
      diffuse 0.6
      specular 0.3
      microfacet Cook 10
      }
   }

// Orange and Clear stripes spiral around the texture
// to make an object look like it was "Peeled"
// Now, you too can be M.C. Escher
define Peel_map
   color_map(
      [0.00, 0.25, orange, orange]
      [0.25, 0.75, black, 1, black, 1]
      [0.75, 1.00, orange, orange])
define Peel_surf
surface {
   color Peel_map[(x + y) - floor(x + y)]
   }
define Peel
texture {
   special Peel_surf {
      ambient 0.2
      diffuse 0.6
      specular white, 0.4
      microfacet Cook 10
      }
   }

define Y_Gradient_map
   color_map(
     [0.00, 0.33, red, blue]
     [0.33, 0.66, blue, green]
     [0.66, 1.00, green, red])
define Y_Gradient
texture {
   special shiny {
      color Y_Gradient_map[y]
      }
   }

define X_Gradient_map
   color_map(
     [0.00, 0.33, red, blue]
     [0.33, 0.66, blue, white]
     [0.66, 1.00, white, white])
define X_Gradient
texture {
   special shiny {
      color X_Gradient_map[x]
      }
   }

// A good wavy water example.
// Requires a sub-plane, and may require scaling to fit your scene.
define Water
texture {
   noise surface {
      color Blue
      reflection white, 0.3
      transmission white, 0.5, 1.33
      frequency 10
      normal 2
      bump_scale 0.75
      }
   }

define Cork_map
    color_map(
      [0.0, 0.6, <0.93, 0.71, 0.532>, <0.98, 0.81, 0.60>]
      [0.6, 0.65, <0.50, 0.30, 0.20>, <0.50, 0.30, 0.20>]
      [0.65, 1.0, <0.80, 0.53, 0.46>, <0.85, 0.75, 0.35>])
define Cork
texture {
   special surface {
      color Cork_map[granite_fn]
      specular white, 0.1
      microfacet Cook 30 // roughness 0.5
      }
   scale <0.25, 0.25, 0.25>
   }
