//////////////////////////////////////////////////////////////////////////
// BodyMotion.inc  1.0  KEA  Copyright(c) 2003, K.E. Ayers Designs      //
//////////////////////////////////////////////////////////////////////////
// Controls the motion of our character's body. It outputs the          //
// following global values:                                             //
//                                                                      //
//  BodyPosition    A vector that specifies the center of Ball Boy's    //
//                  body.                                               //
//  BodyXRotate     The body's rotation about the X-axis (usually 0).   //
//  BodyYRotate     The body's rotation about the X-axis (orientation). //
//  BodyZRotate     The body's rotation about the Z-axis (roll).        //
//  BodyScale       Although Ball Boy is basically spherical, his body  //
//                  will sometimes squash or stretch, as he collides    //
//                  with other objects or accelerates.                  //
//////////////////////////////////////////////////////////////////////////

#ifndef (_BodyMotion_Inc_)
#declare _BodyMotion_Inc_ = true;

#include "SceneConstants.inc"
#include "SceneTiming.inc"


// A macro to smoothly roll the character's body from a starting position,
// to an ending position, by the fractional amount.
//
// NOTE:    BodyXRot and BodyYRot must be set outside of this macro.
//          Only BodyZRotate is set to reflect the rolling motion.
#macro RollTo(StartPos, EndPos, Fraction)
    #local Delta    = EndPos - StartPos;
    #local Dist     = sqrt(Delta.x * Delta.x + Delta.z * Delta.z);
    #local NTurns   = Dist / TurnDist;

    #declare BodyPosition   = StartPos + Delta * Fraction;
    #declare BodyZRotate    = -NTurns * 360 * Fraction;
#end

// A macro to roll the character's body from a starting position, to an
// ending position, by the fractional amount, with acceleration.
//
// NOTE:    BodyXRot and BodyYRot must be set outside of this macro.
//          Only BodyZRotate is set to reflect the rolling motion.
#macro AccelerateTo(StartPos, EndPos, Fraction)
    // SAME AS ROLL-TO ... FOR NOW!
    #local Delta    = EndPos - StartPos;
    #local Dist     = sqrt(Delta.x * Delta.x + Delta.z * Delta.z);
    #local NTurns   = Dist / TurnDist;
    #local AccFract = 1 - sin((Fraction * pi / 2) + (pi / 2));

    #declare BodyPosition   = StartPos + Delta * AccFract;
    #declare BodyZRotate    = -NTurns * 360 * AccFract;
#end

// A macro to smoothly roll the character's body bacwards from a
// starting position, to an ending position, by the fractional amount.
//
// NOTE:    BodyXRot and BodyYRot must be set outside of this macro.
//          Only BodyZRotate is set to reflect the rolling motion.
#macro RollBackTo(StartPos, EndPos, Fraction)
    #local Delta    = EndPos - StartPos;
    #local Dist     = sqrt(Delta.x * Delta.x + Delta.z * Delta.z);
    #local NTurns   = Dist / TurnDist;

    #declare BodyPosition   = StartPos + Delta * Fraction;
    #declare BodyZRotate    = NTurns * 360 * Fraction;
#end


// A macro to smoothly turn the character's body from a starting Y-axis
// rotation angle, to an ending angle, by the fractional amount. The
// body will also lean a little into the turn (X-rotation).
#macro TurnTo(StartYRot, EndYRot, Fraction)
    #declare BodyYRotate = StartYRot + (EndYRot - StartYRot) * Fraction;

    #local MaxLean = 30;
    #local LeanAmt = ((Fraction <= 0.5)
                        ? (MaxLean * Fraction * 2)
                        : (MaxLean * (1 - Fraction) * 2));
                        
    #declare BodyXRotate =
        ((EndYRot < StartYRot)
            ?  LeanAmt      // Counterclockwise: Lean left.
            : -LeanAmt);    // Clockwise: Lean right.
#end


// A macro to "squash" the character's body height (Y-axis), to the
// specified percentage value, through the specified cycle clock.
// As the body compresses along the Y-axis, it will be expanded
// proportionally along the X- and Z-axis (an animation "law" to
// preserve the illusion of constant body mass).
//
// In addition, the body's Y-position will be adjusted to keep its
// bottom anchored; so, BodyPosition must be set before invoking
// this macro.
//
// Finally, this is a four-phase squash/stretch cycle. The fractional
// value is expected to be a "cycle clock" that ranges from 0.0 to 1.0.
// Thus the cycle is:
//
//  Cycle Clock
//  ===========
//  0.0 - 0.25      Squash the body down to its minimum height
//  0.25 - 0.5      Body returns to normal scale
//  0.5 - 0.75      Body stretches to maximum height
//  0.75 - 1.0      Body returns to normal scale
#macro SquashHeight(percentSquash, cycleClock, includeStretch)
    #local numPhases  = ((includeStretch) ? 4 : 2);
    #local phaseClock = cycleClock * numPhases;

    #switch (phaseClock)
        #range (0, 1)
            // Body is squashing down.
            #local YScl  = 1 - (phaseClock * percentSquash);
            #local XZScl = 1 + (phaseClock * percentSquash * 0.3);
        #break

        #range (1, 2)
            // Body is returning to normal after squashing down.
            #local YScl  = 1 - ((2 - phaseClock) * percentSquash);
            #local XZScl = 1 + ((2 - phaseClock) * percentSquash * 0.3);
        #break

        #range (2, 3)
            // Body is stretching up.
            #local YScl  = 1 + ((phaseClock - 2) * percentSquash * 0.5);
            #local XZScl = 1 - ((phaseClock - 2) * percentSquash * 0.25);
        #break

        #else
            // Body is returning to normal after stretching up.
            #local YScl  = 1 + ((4 - phaseClock) * percentSquash * 0.5);
            #local XZScl = 1 - ((4 - phaseClock) * percentSquash * 0.25);
    #end

    #declare BodyPosition   = < BodyPosition.x,
                                BodyPosition.y
                                    + ((YScl - 1) * BallBoyRad),
                                BodyPosition.z >;
    #declare BodyScale      = < XZScl, YScl, XZScl >;

#end    // #macro SquashHeight


// A macro to "squash" the character's body thickness (X-axis), to the
// specified percentage value, through the specified cycle clock.
// As the body compresses along the X-axis, it will be expanded
// proportionally along the Y- and Z-axis (an animation "law" to
// preserve the illusion of constant body mass).
//
// In addition, the body's X-position will be adjusted in the specified
// direction (-1 or +1) to keep its position anchored; so, BodyPosition
// must be set before invoking this macro.
//
// Finally, this is a four-phase squash/stretch cycle. The fractional
// value is expected to be a "cycle clock" that ranges from 0.0 to 1.0.
// Thus the cycle is:
//
//  Cycle Clock
//  ===========
//  0.0 - 0.25      Squash the body down to its minimum thickness
//  0.25 - 0.5      Body returns to normal scale
//  0.5 - 0.75      Body stretches to maximum thickness
//  0.75 - 1.0      Body returns to normal scale
#macro SquashThickness(percentSquash, cycleClock, xDir, includeStretch)
    #local numPhases  = ((includeStretch) ? 4 : 2);
    #local phaseClock = cycleClock * numPhases;

    #switch (phaseClock)
        #range (0, 1)
            // Body is squashing down.
            #local XScl  = 1 - (phaseClock * percentSquash);
            #local YZScl = 1 + (phaseClock * percentSquash * 0.5);
        #break

        #range (1, 2)
            // Body is returning to normal after squashing down.
            #local XScl  = 1 - ((2 - phaseClock) * percentSquash);
            #local YZScl = 1 + ((2 - phaseClock) * percentSquash * 0.5);
        #break

        #range (2, 3)
            // Body is stretching up.
            #local XScl  = 1 + ((phaseClock - 2) * percentSquash);
            #local YZScl = 1 - ((phaseClock - 2) * percentSquash * 0.5);
        #break

        #else
            // Body is returning to normal after stretching up.
            #local XScl  = 1 + ((4 - phaseClock) * percentSquash);
            #local YZScl = 1 - ((4 - phaseClock) * percentSquash * 0.5);
    #end

    #declare BodyPosition   = < BodyPosition.x
                                    + ((XScl - 1) * BallBoyRad * xDir),
                                BodyPosition.y
                                    + ((YZScl - 1) * BallBoyRad),
                                BodyPosition.z >;
    #declare BodyScale      = < XScl, YZScl, YZScl >;

#end    // #macro SquashHeight


// A macro to make the character leap from a starting position to
// an ending position, with the specified maximum height, over
// the specified "leap clock". The leap clock is expected to range
// from 0.0 to 1.0 over the duration of the leap.
//
// The leaps consists of two phases:
//
// Leap Clock
// ==========
// 0.00 - 0.20  The body squashes down by 30%, preparing to leap.
// 0.20 - 1.00  The actual leap in a sinusoidal arc.
#macro LeapTo(startPos, endPos, maxHgt, leapClock)
    #local launchPhaseLen   = 0.20;
    #local endLaunchPhase   = launchPhaseLen;
    #local peakStretchPhase = endLaunchPhase + 0.20;
    #local endStretchPhase  = 0.50;
    #local startDownStretch = 0.6;
    #local maxDownStretch   = 0.8;
    #local inAirPhaseLen    = 1 - endLaunchPhase;
    #local maxStretch       = 0.25;
    
    #switch (leapClock)

        #range (0.0, launchPhaseLen)
            // Squash the body down in preparation for the launch.
            #declare BodyPosition = startPos;
            SquashHeight(0.35,
                         leapClock / endLaunchPhase,
                         false)
        #break

        #else
            // He's in the air! Normalize the range to 0.0 - 1.0.
            #local airClock = ((leapClock - launchPhaseLen) / inAirPhaseLen);
            #local airDelta = endPos - startPos;

            #declare BodyPosition =
                < startPos.x + (airDelta.x * airClock),
                  startPos.y + sin(airClock * pi) * maxHgt,
                  startPos.z + (airDelta.z * airClock) >; 

            // As he's ascending, Ball Boy stretches.
            #switch (leapClock)
                #range (endLaunchPhase, peakStretchPhase)
                    #local stretchAmt =
                        ((leapClock - endLaunchPhase)
                            / (peakStretchPhase - endLaunchPhase))
                                    * maxStretch;
                #break
                
                #range (peakStretchPhase, endStretchPhase)
                    #local stretchAmt =
                        -((leapClock - endStretchPhase)
                            / (endStretchPhase - peakStretchPhase))
                                    * maxStretch;
                #break

                #range(startDownStretch, maxDownStretch)
                    #local stretchAmt =
                        ((leapClock - startDownStretch)
                            / (maxDownStretch - startDownStretch))
                                    * maxStretch;
                #break

                #range(maxDownStretch, 1.0)
                    #local stretchAmt =
                        -((leapClock - 1)
                            / (1 - maxDownStretch))
                                    * maxStretch;
                #break

                #else
                    #local stretchAmt = 0;                
            #end
            
            #if (leapClock > endStretchPhase)
                #local leanClk      =
                    sin(((leapClock - endStretchPhase)
                            / (1 - endStretchPhase)) * pi);
                            
                #declare CharacterRotate = < -40 * leanClk, 0, 0 >;
            #end

            #declare BodyScale  = < 1 - stretchAmt * 0.5,
                                    1 + stretchAmt,
                                    1 - stretchAmt * 0.5 >;
                                    
    #end    //#switch (leapClock)

#end    // #macro LeapTo

//////////////////////////////////////////////////////////////////////////
// These are the various positions at which the character will end up,  //
// and the corresponding (Y-axis) orientations and deformations.        //

// Here's where the character starts from.
#local PosStart         = < EastWallPos - EastMarkOfs,
                            BallBoyRad,
                            SouthWallPos + SouthMarkOfs >;
#local RotStart         = -180 * y;
#local SclStart         = < 1, 1, 1 >;

// Here's where he stops at the end of the hall.
#local PosEndHall       = < WestWallPos + WestMarkOfs,
                            BallBoyRad,
                            SouthWallPos + SouthMarkOfs >;

// Here's the stopping position along the west wall of the room.
#local PosWestRoom      = < WestWallPos + WestMarkOfs,
                            BallBoyRad,
                            NorthWallPos - NorthMarkOfs >;

// ... along the east wall of the room.
#local PosEastRoom      = < EastWallPos - EastMarkOfs,
                            BallBoyRad,
                            NorthWallPos - NorthMarkOfs >;

// Here's where he collides with the east and west walls.
#local PosEastWall      = < EastWallPos - BallBoyRad * 1.7,
                            BallBoyRad,
                            NorthWallPos - NorthMarkOfs >;
#local PosWestWall      = < WestWallPos + BallBoyRad * 1.7,
                            BallBoyRad,
                            NorthWallPos - NorthMarkOfs >;

// Here's the center of the room.
#local PosRoomCenter    = < RoomCenterXOfs,
                            BallBoyRad,
                            NorthWallPos - NorthMarkOfs >;

// His first leap lands him just slightly forward from the
// center of the room.
#local PosLeap1Land     = < RoomCenterXOfs,
                            BallBoyRad,
                            NorthWallPos - NorthMarkOfs * 0.8 >;

// His second leap lands him closer to the wall.
#local PosLeap2Land     = < RoomCenterXOfs,
                            BallBoyRad,
                            NorthWallPos - NorthMarkOfs * 0.5 >;

// After that, he's over the wall and on to four joyous escape bounces.
#local PosOverWallLand  = < RoomCenterXOfs,
                            BallBoyRad,
                            NorthWallPos + NorthMarkOfs * 0.5 >;
#local PosEscLeap1Land  = < RoomCenterXOfs + BallBoyRad * 2,
                            BallBoyRad,
                            NorthWallPos
                                + NorthMarkOfs * 0.5
                                + TurnDist >;
#local PosEscLeap2Land  = < RoomCenterXOfs - BallBoyRad * 2,
                            BallBoyRad,
                            NorthWallPos
                                + NorthMarkOfs * 0.5
                                + TurnDist * 2 >;
#local PosEscLeap3Land  = < RoomCenterXOfs + BallBoyRad * 2,
                            BallBoyRad,
                            NorthWallPos
                                + NorthMarkOfs * 0.5
                                + TurnDist * 5 >;
#local PosEscLeap4Land  = < RoomCenterXOfs - BallBoyRad * 2,
                            BallBoyRad,
                            NorthWallPos
                                + NorthMarkOfs * 0.5
                                + TurnDist * 8 >;

//////////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////////
// Here's where we perform the scene-specific actions.                  //

// As a rule, that compisite character will have no rotation applied.
#declare CharacterRotate = < 0, 0, 0 >;

#local LeanBackAngle = -30;

#switch (ActiveScene)

    #case (SCN_Intro)
    #case (SCN_FadingIn)
    #case (SCN_CameraPansUp)
        #declare BodyPosition   = PosStart;
        #declare BodyXRotate    = RotStart.x;
        #declare BodyYRotate    = RotStart.y;
        #declare BodyZRotate    = RotStart.z;
        #declare BodyScale      = SclStart;
    #break

    #case (SCN_RollsToEndOfHall)
        #declare BodyXRotate    = RotStart.x;
        #declare BodyYRotate    = RotStart.y;
        RollTo(PosStart, PosEndHall, SceneClock)
        #declare BodyScale      = SclStart;
    #break

    #case (SCN_TurnsToFaceNorth)
        #declare BodyPosition   = PosEndHall;
        #declare BodyZRotate    = 0;
        TurnTo(-180, -90, SceneClock)
        #declare BodyScale      = SclStart;
    #break

    #case (SCN_RollsIntoRoom)
        #declare BodyXRotate    = RotStart.x;
        #declare BodyYRotate    = -90;
        RollTo(PosEndHall, PosWestRoom, SceneClock)
        #declare BodyScale      = SclStart;
    #break

    #case (SCN_TurnsToFaceEast)
        #declare BodyPosition   = PosWestRoom;
        #declare BodyZRotate    = 0;
        TurnTo(-90, 0, SceneClock)
        #declare BodyScale      = SclStart;
    #break

    #case (SCN_LooksAtEastWall)
    #case (SCN_LooksDetermined)
        #declare BodyPosition   = PosWestRoom;
        #declare BodyXRotate    = 0;
        #declare BodyYRotate    = 0;
        #declare BodyZRotate    = 0;
        #declare BodyScale      = SclStart;

        // TO DO:   Maybe add a slight stretch near the end of
        //          SCN_LooksDetermined, as he prepares to roll.
    #break

    #case (SCN_ChargesEastWall)
        #declare BodyXRotate    = RotStart.x;
        #declare BodyYRotate    = 0;
        #declare BodyScale      = SclStart;
        AccelerateTo(PosWestRoom, PosEastWall, SceneClock)

        #local leanClk = 0.5;
        #local maxLean = -20;
                
        #if (SceneClock < leanClk)
            #declare CharacterRotate =
                < 0, 0, (1 - (leanClk - SceneClock) / leanClk) * maxLean >;
        #else
            #declare CharacterRotate = < 0, 0, maxLean >;
        #end        
    #break

    #case (SCN_CollidesWithEastWall)
        #declare BodyPosition   = PosEastWall;
        #declare BodyXRotate    = 0;
        #declare BodyYRotate    = 0;
        #declare BodyZRotate    = 0;
        SquashThickness(0.30, SceneClock, 1, false)
    #break

    #range (SCN_RollsBackFromEastWall, SCN_LooksFrustrated)
        #declare BodyXRotate    = 0;
        #declare BodyZRotate    = 0;

        #switch (ActiveScene)
            #case (SCN_RollsBackFromEastWall)
                #declare BodyYRotate = 0;
                RollBackTo(PosEastWall, PosEastRoom, SceneClock)
            #break

            #case (SCN_EastWallStarsAppear)
                #declare BodyYRotate    = 0;
                #declare BodyPosition   = PosEastRoom;
            #break
            
            #case (SCN_TurnsToFaceWest)
                #declare BodyPosition   = PosEastRoom;
                TurnTo(0, 180, SceneClock)
            #break

            #else            
                #declare BodyPosition   = PosEastRoom;
                #declare BodyYRotate    = -180;
        #end
        
        // Ball Boy is seeing stars after the collision!
        #include "SpinningStars.inc"

        #local starClk = MultiSceneClock(SCN_RollsBackFromEastWall,
                                         SCN_ChargesWestWall);        
        #declare CharacterRotate =
            < sin(8 * pi * starClk) * 10, 0, 0 >;
        #local charScale = sin(4 * pi * starClk) * 0.10;
        
        #declare BodyScale  = < 1 - charScale * 0.5,
                                1 + charScale,
                                1 - charScale * 0.5 >;
            
        object {
            SpinningStars(starClk)
            rotate 720 * starClk * y
            translate < BodyPosition.x,
                        BodyPosition.y + BallBoyRad * 1.5,
                        BodyPosition.z >
        }
    #break

    #case (SCN_ChargesWestWall)
        #declare BodyXRotate    = 0;
        #declare BodyYRotate    = -180;
        #declare BodyScale      = SclStart;
        AccelerateTo(PosEastRoom, PosWestWall, SceneClock)

        #local leanClk = 0.5;
        #local maxLean = 20;
                
        #if (SceneClock < leanClk)
            #declare CharacterRotate =
                < 0, 0, (1 - (leanClk - SceneClock) / leanClk) * maxLean >;
        #else
            #declare CharacterRotate = < 0, 0, maxLean >;
        #end        
    #break

    #case (SCN_CollidesWithWestWall)
        #declare BodyPosition   = PosWestWall;
        #declare BodyXRotate    = 0;
        #declare BodyYRotate    = -180;
        #declare BodyZRotate    = 0;
        #declare BodyScale      = SclStart;
        SquashThickness(0.30, SceneClock, -1, false)
    #break

    #range (SCN_RollsBackFromWestWall, SCN_TurnsToFaceNorthWall)
        #declare BodyXRotate    = 0;
        #declare BodyZRotate    = 0;
        #declare BodyScale      = SclStart;

        #switch (ActiveScene)
            #case (SCN_RollsBackFromWestWall)
                #declare BodyYRotate = -180;
                RollBackTo(PosWestWall, PosWestRoom, SceneClock)
            #break

            #case (SCN_WestWallStarsAppear)
                #declare BodyYRotate    = -180;
                #declare BodyPosition   = PosWestRoom;
            #break
            
            #case (SCN_TurnsToFaceCenter)
                #declare BodyPosition   = PosWestRoom;
                TurnTo(180, 0, SceneClock)
            #break

            #case (SCN_RollsToCenterOfRoom)
                #declare BodyYRotate    = 0;
                RollTo(PosWestRoom, PosRoomCenter, SceneClock)
            #break
        
            #case (SCN_TurnsToFaceNorthWall)
                #declare BodyPosition   = PosRoomCenter;
                TurnTo(0, -90, SceneClock)
            #break

            #else            
                #declare BodyPosition   = PosWestRoom;
                #declare BodyYRotate    = 0;
        #end
        
        // Ball Boy is seeing stars after the collision!
        #include "SpinningStars.inc"

        #local starClk = MultiSceneClock(SCN_RollsBackFromWestWall,
                                         SCN_LooksUp);        
        #declare CharacterRotate =
            < sin(8 * pi * starClk) * 10, 0, 0 >;
        #local charScale = sin(4 * pi * starClk) * 0.10;
        
        #declare BodyScale  = < 1 - charScale * 0.5,
                                1 + charScale,
                                1 - charScale * 0.5 >;
            
        object {
            SpinningStars(starClk)
            rotate 720 * starClk * y
            translate < BodyPosition.x,
                        BodyPosition.y + BallBoyRad * 1.5,
                        BodyPosition.z >
        }
    #break

    #case (SCN_LooksUp)
        #declare BodyPosition   = PosRoomCenter;
        #declare BodyXRotate    = 0;
        #declare BodyYRotate    = -90;
        #declare BodyZRotate    = 0;
        #declare BodyScale      = SclStart;

        #declare CharacterRotate = < LeanBackAngle * SceneClock, 0, 0 >

        // TO DO:   In SCN_LooksFrustrated will have to add
        //          some body gesture motion.
    #break
    
    #case (SCN_SeesTopOfWall)
        #declare BodyPosition   = PosRoomCenter;
        #declare BodyXRotate    = 0;
        #declare BodyYRotate    = -90;
        #declare BodyZRotate    = 0;
        #declare BodyScale      = SclStart;
                
        #if (SceneClock < 0.5)
            #declare CharacterRotate = < LeanBackAngle, 0, 0 >;
        #else
            #local rotClk = 1 - ((SceneClock - 0.5) * 2);
            #declare CharacterRotate = < LeanBackAngle * rotClk, 0, 0 >;
        #end

        #include "LightBulb.inc"
        object {
            LightBulb(SceneClock)
            translate BodyPosition + BallBoyRad * 3 * y
        }
    #break

    #case (SCN_Leaps)
        #declare BodyXRotate    = 0;
        #declare BodyYRotate    = -90;
        #declare BodyZRotate    = 0;
        #declare BodyScale      = SclStart;
        LeapTo(PosRoomCenter, PosLeap1Land, PenWallHgt * 0.8, SceneClock)
    #break

    #case (SCN_LeapsHigher)
        #declare BodyXRotate    = 0;
        #declare BodyYRotate    = -90;
        #declare BodyZRotate    = 0;
        #declare BodyScale      = SclStart;
        LeapTo(PosLeap1Land, PosLeap2Land, PenWallHgt * 1.0, SceneClock)
    #break

    #case (SCN_LeapsOverWall)
        #declare BodyXRotate    = 0;
        #declare BodyYRotate    = -90;
        #declare BodyZRotate    = 0;
        #declare BodyScale      = SclStart;
        LeapTo(PosLeap2Land, PosOverWallLand, PenWallHgt * 2, SceneClock)
    #break

    #case (SCN_EscapeLeap1)
        #declare BodyXRotate    = 0;
        #declare BodyZRotate    = 0;
        #declare BodyScale      = SclStart;
        LeapTo(PosOverWallLand, PosEscLeap1Land, PenWallHgt * 2, SceneClock)
        TurnTo(-90, -270, MultiSceneClock(SCN_EscapeLeap1, SCN_EscapeLeap3))
    #break

    #case (SCN_EscapeLeap2)
        #declare BodyXRotate    = 0;
        #declare BodyZRotate    = 0;
        #declare BodyScale      = SclStart;
        LeapTo(PosEscLeap1Land, PosEscLeap2Land, PenWallHgt * 1, SceneClock)
        TurnTo(-90, -270, MultiSceneClock(SCN_EscapeLeap1, SCN_EscapeLeap3))
    #break

    #case (SCN_EscapeLeap3)
        #declare BodyXRotate    = 0;
        #declare BodyYRotate    = -270;
        #declare BodyZRotate    = 0;
        #declare BodyScale      = SclStart;
        LeapTo(PosEscLeap2Land, PosEscLeap3Land, PenWallHgt * 1, SceneClock)
    #break

    #case (SCN_EscapeLeap4)
        #declare BodyXRotate    = 0;
        #declare BodyYRotate    = -270;
        #declare BodyZRotate    = 0;
        #declare BodyScale      = SclStart;
        LeapTo(PosEscLeap3Land, PosEscLeap4Land, PenWallHgt * 1, SceneClock)
    #break

#end    // #switch (ActiveScene)


#end    // #ifndef (_BodyMotion_Inc_)

