//////////////////////////////////////////////////////////////////////////
// SceneTiming.inc  1.0  KEA  Copyright(c) 2003, K.E. Ayers Designs     //
//////////////////////////////////////////////////////////////////////////
// Slices and dices the animation clock to conform to a scene-oriented  //
// view. Defines the following scene-related global variables:          //
//                                                                      //
// SCN_xxxxx            The numeric identies of the various scenes.     //
// AnimationLength      The total length of the animation, in seconds.  //
// AnimationClock       The "clock" value, scaled to the animation's    //
//                      total length, in seconds.                       //
// AnimationClockDelta  Basically: 1/FrameRate.                         //
// ActiveScene          The numeric identity of the scene currently     //
//                      being rendered ... the scene's start time.      //
// SceneName            A string with the name of the current scene.    //
// SceneLength          The length of the current scene, in seconds.    //
// SceneClock           A clock value normalized to the length of the   //
//                      scene currently being rendered.  Runs from 0.0  //
//                      at the start of the scene, to 1.0 at the end.   //
//////////////////////////////////////////////////////////////////////////

#ifndef (_SceneTiming_Inc_)
#declare _SceneTiming_Inc_ = true;


// Accumulates our scene start times.
#declare STime = 0;
// Counts the number of scenes defined for the animation.
#declare NumScenes = 0;

// The following arrays hold the start time, length, and name for
// each scene. Elements will be filled in by the "DefineScene" macro.
#declare SceneStarts    = array[100]
#declare SceneLengths   = array[100]
#declare SceneNames     = array[100]

// A macro to define a scene's start time, its length, and its name.
#macro DefineScene(sceneLen, sceneName)
    // Record the scene parameters.
    #declare SceneStarts[NumScenes]  = STime;
    #declare SceneLengths[NumScenes] = sceneLen;
    #declare SceneNames[NumScenes]   = sceneName
    // Update the scene count and the start time for the next scene.
    #declare NumScenes = NumScenes + 1;
    #declare STime     = STime + sceneLen;

#end    // #macro SetSceneLength

// A macro to define a composite "scene clock" that spans
// multiple scenes. The "nextScene" argument specifies the
// AnimationClock value for the start of scene that follows
// the scenes defined as the "composite scene".
#macro MultiSceneClock(startScene, nextScene)
    (AnimationClock - startScene) / (nextScene - startScene)
#end


//////////////////////////////////////////////////////////////////////////
// Set the numeric identities of the various scenes.  Basically, a      //
// scene's identity is the time in seconds at which it starts. For each //
// scene, we'll invoke the "DefineScene" macro to record its start      //
// time, length, and name. These values will subsequently be used when  //
// decoding the clock to determine the scene currently being rendered.  //

// Display the animation's title and credits.
#declare SCN_Intro = STime;                 DefineScene(1.5, "Intro")
// Fade-in to look at the wall of the pen.
#declare SCN_FadingIn = STime;              DefineScene(1.5, "FadingIn")
// The camera pans up and looks over the wall.
// Ball Boy, just sitting there!
#declare SCN_CameraPansUp = STime;          DefineScene(3, "CameraPansUp")
// Ball Boy rolls to the end of the dividing wall.
#declare SCN_RollsToEndOfHall = STime;      DefineScene(4, "RollsToEndOfHall")
// Ball Boy turns right 90-degrees.
#declare SCN_TurnsToFaceNorth = STime;      DefineScene(0.75, "TurnsToFaceNorth")
// Ball Boy rolls to the center of the room, along the west wall.
#declare SCN_RollsIntoRoom = STime;         DefineScene(2, "RollsIntoRoom")
// Ball Boy turns right to face the east wall.
#declare SCN_TurnsToFaceEast = STime;       DefineScene(0.75, "TurnsToFaceEast")
// He pauses to look at the wall.
#declare SCN_LooksAtEastWall = STime;       DefineScene(0.75, "LooksAtEastWall")
// His eyes squint as he gets a "determined" look.
#declare SCN_LooksDetermined = STime;       DefineScene(0.75, "LooksDetermined")
// Ball Boy rolls toward the east wall, gathering speed.
#declare SCN_ChargesEastWall = STime;       DefineScene(3, "ChargesEastWall")
// Ball Boy collides with the east wall.
#declare SCN_CollidesWithEastWall = STime;  DefineScene(0.75, "CollidesWithEastWall")
// He rolls back a turn, facing the east wall.
#declare SCN_RollsBackFromEastWall = STime; DefineScene(1, "RollsBackFromEastWall")
// He sees stars!
#declare SCN_EastWallStarsAppear = STime;   DefineScene(1.5, "EastWallStarsAppear")
// He turns around to face west.
#declare SCN_TurnsToFaceWest = STime;       DefineScene(1, "TurnsToFaceWest")
// His eyes squint as he gets a "frustrated" look.
#declare SCN_LooksFrustrated = STime;       DefineScene(0.75, "LooksFrustrated")
// Ball Boy rolls toward the west wall, gathering speed.
#declare SCN_ChargesWestWall = STime;       DefineScene(3, "ChargesWestWall")
// Ball Boy collides with the west wall.
#declare SCN_CollidesWithWestWall = STime;  DefineScene(0.75, "CollidesWithWestWall")
// He rolls back a turn, facing the west wall.
#declare SCN_RollsBackFromWestWall = STime; DefineScene(1, "RollsBackFromWestWall")
// He sees stars ... again!
#declare SCN_WestWallStarsAppear = STime;   DefineScene(1.5, "WestWallStarsAppear")
// He turns around to face the center of the room.
#declare SCN_TurnsToFaceCenter = STime;     DefineScene(1, "TurnsToFaceCenter")
// He gets a "sad" look.
#declare SCN_LooksSad = STime;              DefineScene(0.75, "LooksSad")
// Ball Boy rolls slowly to the center of the room.
#declare SCN_RollsToCenterOfRoom = STime;   DefineScene(2.5, "RollsToCenterOfRoom")
// He turns to face the north wall.
#declare SCN_TurnsToFaceNorthWall = STime;  DefineScene(1.5, "TurnsToFaceNorthWall")
// Ball Boy "leans" back.
#declare SCN_LooksUp = STime;               DefineScene(1, "LooksUp")
// He looks surprised as he sees the top of the wall.
#declare SCN_SeesTopOfWall = STime;         DefineScene(2, "SeesTopOfWall")
// He leaps into the air, not quite to the top of the wall.
#declare SCN_Leaps = STime;                 DefineScene(1, "Leaps")
// He leaps higher, clearing the top of the wall.
#declare SCN_LeapsHigher = STime;           DefineScene(1.5, "LeapsHigher")
// Ball Boy leaps over the wall.
#declare SCN_LeapsOverWall = STime;         DefineScene(1.5, "LeapsOverWall")
// Ball Boy has escaped; he bounds away erratically.
#declare SCN_EscapeLeap1 = STime;           DefineScene(1, "EscapeLeap1")
#declare SCN_EscapeLeap2 = STime;           DefineScene(1, "EscapeLeap2")
#declare SCN_EscapeLeap3 = STime;           DefineScene(1, "EscapeLeap3")
#declare SCN_EscapeLeap4 = STime;           DefineScene(2, "EscapeLeap4")

//////////////////////////////////////////////////////////////////////////

// Remember the index of the last scene that was defined.
#declare LastSceneIndex = NumScenes - 1;
// This is the total accumulated length of all scenes.
#declare AnimationLength = STime;
// The is the clock, normalized to the animation's total length.
#declare AnimationClock = clock * AnimationLength;

// Normalize the animation clock's frame-to-frame delta, to the
// length of the animation.
#declare AnimationClockDelta = clock_delta * AnimationLength;


// The fade-out begins two seconds from the end.
#declare STime = AnimationLength - 2;
#declare SCN_FadingOut = STime; DefineScene(2, "FadingOut")


//////////////////////////////////////////////////////////////////////////
// Decode the scene currently being rendered, based on the clock value. //

#declare ActiveScene    = -1;
#declare SceneName      = ""
#declare SceneLength    = -1;
#declare SceneClock     = -1;

#local sceneNum = ((clock < 1) ? 0 : LastSceneIndex);

#while ((sceneNum <= LastSceneIndex) & (ActiveScene < 0))
    #local sStart = SceneStarts[sceneNum];
    #local sLen   = SceneLengths[sceneNum];
    #local sEnd   = sStart + sLen;

    #if ((clock = 1) | ((AnimationClock >= sStart) & (AnimationClock <= sEnd)))
        // Found the current scene ... activate it!
        #declare ActiveScene = sStart;
        #declare SceneName   = SceneNames[sceneNum]
        #declare SceneLength = sLen;
        #declare SceneClock  = (AnimationClock - sStart) / (sEnd - sStart);
    #else
        #local sceneNum = sceneNum + 1;
    #end
#end

#if (ActiveScene < 0)
    #debug "\n\n"
    #debug "***************************************\n"
    #debug "** Cannot determine the active scene **\n"
    #debug "***************************************\n"
    #debug concat("Clock:          ",str(clock,1,8),"\n") 
    #debug concat("AnimationClock: ",str(AnimationClock,1,8),"\n") 
    #debug concat("sStart:         ",str(sStart,1,8),"\n") 
    #debug concat("sEnd:           ",str(sEnd,1,8),"\n") 
    #debug concat("sLen:           ",str(sLen,1,8),"\n") 
    #debug "***************************************\n"
    #error "\n\n"
#end


//////////////////////////////////////////////////////////////////////////

// If enabled, generate a report containing information about
// each scene: The scene's name, its starting time, its length,
// and the starting frames numbers for 15, 24 and 30 FPS rates.
#if (GenerateSceneReport)
    #local hdrUnderline =
        "==========================  ======  ======  ======  ======  ======\n"
    #fopen SceneReportFile SceneReportFilename write

    #write
    (
        SceneReportFile,
        "SCENE TIMING REPORT\n",
        "  Animation title:  ", AnimationTitle, "\n",
        "  Animation length: ", str(AnimationLength,1,1), " sec\n",
        "  Number of scenes: ", str(NumScenes,1,0), "\n",
        "                            Start   Scene   Starting Frame Nuumber\n",
        "Scene Name                  Time    Length  15 FPS  24 FPS  30 FPS\n",
        hdrUnderline
    )

    //          1111111111222222222233333333334444444444555555555566666666667 //
    // 1234567890123456789012345678901234567890123456789012345678901234567890 //
    // Scene Name                  Start   Length  15 FPS  24 FPS  30 FPS      //
    // ==========================  ======  ======  ======  ======  ======      //
    // abcdefghijklmnopqrstuvwxyz  1234.0  1234.0  123456  123456  123456      //

    #local namePad  = "                          "
    #local sceneNum = 0;

    #while (sceneNum < NumScenes)
        #local sName  = SceneNames[sceneNum]
        #local sStart = SceneStarts[sceneNum];
        #local sLen   = SceneLengths[sceneNum];
        #local nPad   = strlen(namePad) - strlen(sName);

        #if (nPad > 0)
            #local sName = concat(sName, substr(namePad, 1, nPad))
        #end

        #write
        (
            SceneReportFile,
            sName, "  ",
            str(sStart,6,1), "  ",
            str(sLen,6,1), "  ",
            str(sStart * 15 + 1,6,0), "  ",
            str(sStart * 24 + 1,6,0), "  ",
            str(sStart * 30 + 1,6,0),
            "\n"
        )

        #local sceneNum = sceneNum + 1;
    #end

    #write
    (
        SceneReportFile,
        hdrUnderline,
        "                    Total:          ",
        str(AnimationLength,6,1), "  ",
        str(AnimationLength * 15,6,0), "  ",
        str(AnimationLength * 24,6,0), "  ",
        str(AnimationLength * 30,6,0),
        "\n"
    )
    #fclose SceneReportFile
#end


#end    // #ifndef (_SceneTiming_Inc_)

