/* FLAG.INC
 * An animatable flag (sine wave surface) for POVRAY 3.x
 *
 * Created by Guy Rauscher, July 1997.
 * Inspired by "manta" by Heino Ruescher
 *
 * Overview:
 * This file creates a thin surface in the x-y plane made of small
 * triangles. The surface extrudes in the z directions according to
 * two sine waves that move along the x axis and the y axis. If no
 * parameters are provided the surface is created from <0,0,0> to
 * <1,1,0> and contains two waves with the length of 1 unit. In one
 * clock cycle the two waves move 1 unit in the positive direction.
 * The triangle mesh is made of a 10x10 triangle grid. The extrusion
 * amplitude is 0.25 and the normal maximum angle is 45 degrees.
 *
 * Usage:
 * 1. Explicit inclusion - just creates the object:
 *
 *   #include "flag.inc"
 *
 * 2. Inclusion within an object statment - use this if you want to
 *    transform the flag:
 *
 *   object { #include "flag.inc" rotate y*50 translate y*7 }
 *
 * 3. Declaration - use this method if you want to use many instances
 *    of the flag or just for style:
 *
 *   #declare Flag = #include "flag.inc"
 *   object { Flag }
 *
 * Options:
 * flag_xsize - The maximum x cordinate of the surface.
 * flag_ysize - The maximum y cordinate of the surface.
 * flag_xsteps - The number of steps to take along the x axis. Each step
 *   creates two triangles.
 * flag_ysteps - The number of steps to take along the y axis. Each step
 *   creates two triangles.
 * flag_xwave - The number of full sine wave lengths along the x direction.
 * flag_ywave - The number of full sine wave lengths along the y direction.
 * flag_xanim - The number of times to translate the wave flag_xsize units
 *   along the x direction in one clock cycle.
 * flag_yanim - The number of times to translate the wave flag_ysize units
 *   along the y direction in one clock cycle.
 * flag_amp - The maximum amount to extrude the surface in the z directions.
 * flag_angle - The maximum angle to use when calculating normals to simulate
 *   smoothness with smooth_triangles.
 * flag_clock - use this to replace the clock variable.
 */

// Check variables and assign defaults:
#ifndef(flag_xsize) #declare flag_xsize = 1 #end
#ifndef(flag_xsteps) #declare flag_xsteps = 10 #end
#ifndef(flag_xwave) #declare flag_xwave = 1 #end
#ifndef(flag_xanim) #declare flag_xanim = 1 #end
#ifndef(flag_ysize) #declare flag_ysize = 1 #end
#ifndef(flag_ysteps) #declare flag_ysteps = 10 #end
#ifndef(flag_ywave) #declare flag_ywave = 1 #end
#ifndef(flag_yanim) #declare flag_yanim = 1 #end
#ifndef(flag_amp) #declare flag_amp = 0.25 ; #end
#ifndef(flag_angle) #declare flag_angle = 45 #end
#ifndef(flag_clock) #declare flag_clock = clock ; #end

// Create surface:
mesh {
#declare X0 = 0;
#while(X0 < flag_xsize)
   #declare X1 = X0 + flag_xsize/flag_xsteps;
   #declare Y0 = 0;
   #while(Y0 < flag_ysize)
      #declare Y1 = Y0 + flag_ysize/flag_ysteps;
         #declare Z0 =(sin((X0/flag_xsize-flag_xanim*flag_clock)*flag_xwave*2*pi)+sin((Y0/flag_ysize-flag_yanim*flag_clock)*flag_ywave*2*pi))*0.5*flag_amp;
         #declare N0 = vrotate(<0,0,1>,<0,-flag_angle*cos((X0/flag_xsize-flag_xanim*flag_clock)*flag_xwave*2*pi),0>) +vrotate(<0,0,1>,<flag_angle*cos((Y0/flag_ysize-flag_yanim*flag_clock)*flag_ywave*2*pi),0,0>);
         #declare Z1 =(sin((X1/flag_xsize-flag_xanim*flag_clock)*flag_xwave*2*pi)+sin((Y0/flag_ysize-flag_yanim*flag_clock)*flag_ywave*2*pi))*0.5*flag_amp;
         #declare N1 = vrotate(<0,0,1>,<0,-flag_angle*cos((X1/flag_xsize-flag_xanim*flag_clock)*flag_xwave*2*pi),0>)+vrotate(<0,0,1>,<flag_angle*cos((Y0/flag_ysize-flag_yanim*flag_clock)*flag_ywave*2*pi),0,0>);
         #declare Z2 =(sin((X0/flag_xsize-flag_xanim*flag_clock)*flag_xwave*2*pi)+sin((Y1/flag_ysize-flag_yanim*flag_clock)*flag_ywave*2*pi))*0.5*flag_amp;
         #declare N2 = vrotate(<0,0,1>,<0,-flag_angle*cos((X0/flag_xsize-flag_xanim*flag_clock)*flag_xwave*2*pi),0>) +vrotate(<0,0,1>,<flag_angle*cos((Y1/flag_ysize-flag_yanim*flag_clock)*flag_ywave*2*pi),0,0>);
         #declare Z3 =(sin((X1/flag_xsize-flag_xanim*flag_clock)*flag_xwave*2*pi)+sin((Y1/flag_ysize-flag_yanim*flag_clock)*flag_ywave*2*pi))*0.5*flag_amp;
         #declare N3 = vrotate(<0,0,1>,<0,-flag_angle*cos((X1/flag_xsize-flag_xanim*flag_clock)*flag_xwave*2*pi),0>)+vrotate(<0,0,1>,<flag_angle*cos((Y1/flag_ysize-flag_yanim*flag_clock)*flag_ywave*2*pi),0,0>);
         smooth_triangle { <X0,Y0,Z0>,N0, <X1,Y0,Z1>,N1, <X1,Y1,Z3>,N3 }
         smooth_triangle { <X0,Y0,Z0>,N0, <X0,Y1,Z2>,N2, <X1,Y1,Z3>,N3 }
      #declare Y0 = Y1;
   #end
   #declare X0 = X1;
#end
}
